/***************************************************************************/
/* Written 1994++ by Peter Boesecke                                        */
/* Copyright (C) 2011 European Synchrotron Radiation Facility              */
/*                       Grenoble, France                                  */
/*                                                                         */
/*    Principal authors: Peter Boesecke  (boesecke@esrf.eu)                */
/*                                                                         */
/*    This program is free software: you can redistribute it and/or modify */
/*    it under the terms of the GNU General Public License as published by */
/*    the Free Software Foundation, either version 3 of the License, or    */
/*    (at your option) any later version.                                  */
/*                                                                         */
/*    This program is distributed in the hope that it will be useful,      */
/*    but WITHOUT ANY WARRANTY; without even the implied warranty of       */
/*    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the        */
/*    GNU General Public License for more details.                         */
/*                                                                         */
/*    You should have received a copy of the GNU General Public License    */
/*    along with this program.  If not, see <http://www.gnu.org/licenses/>.*/
/***************************************************************************/
/*+++
1 saxs_tiff
  Transformation of an edf image sequence to 1byte/2byte/4byte tiff format

2 PURPOSE
  Transformation of an edf image sequence to 1byte/2byte/4byte tiff format
  (default 2 byte tiff)

  Arguments:
  saxs_tiff [options] 
            <i1nam> <tiff file prefix> <i1fst> <i1lst> <i1inc> 
            <odum> <odim1> <odim2> <8|16|32-bits> <orientation>
            <auto> [<min> <max>]
  Defaults:
  <input file name>   : input.edf 
  <tiff file prefix>  : input.tif (default conversion from input filename)
  <first image>       : <first image number in input file>
  <last image>                              : <last image number in input file>
     if argument list ends with first image : <first image>
  <increment>         : 1
  <dummy>             : <dummy value in first image of input file>
  <dimension 1>       : <horizontal dimension of first image in input file>
  <dimension 2>       : <vertical dimension of first image in input file>
  <bps>               : 16 (number of bits per pixel in output file) 
  <orientation>       : orientation of output image (default: 3)
  <auto>              : autoscaling on/off (default on)
  <min>               : 0 (minimum of mapping region)
  <max>               : 2^bps-1 (maximum of mapping region)

  Specific arguments:
  -pref 'prefix'      : specify the prefix of the ouptput file

  Wildcard:
  A "=" in the argument list chooses the default.

2 HISTORY
  1999-10-31  PB from saxs_bsl V3.1
  1999-11-12  PB V3.0
  1999-12-21  PB V3.01 output file name without number 
                       if only 1 image is written
  1999-12-25  PB V3.02 default output file name without path
  2000-08-04  PB %f->%g
  2001-07-07  PB V3.03 IO_DontReadWrite
  2001-07-08  PB V3.04 FFirst
  2001-07-09  PB V3.05 (pcb->ib)[0].FileNumber
  2001-07-09  PB V4.00 new loop, no repetition
  2001-08-30  PB V4.01 
  2002-06-01  PB V4.50 ave
  2003-02-03  PB V4.51 num[BlockNum] 
  2003-02-09  PB V4.52 line args for Dummy, DDummy, Dim
  2003-05-30  PB V4.53 ImageLoop: Function call changed
  2003-08-14  PB V4.54 option_define, all option numbers removed
  2003-11-30  PB V4.55 BinSiz
  2004-10-31  PB V4.56 MAKE_FUNCTION
  2005-07-15  PB V4.57 ReportSaxsStatus parameter list updated
  2005-08-04  PB V4.58 PrintImageHeaderBlock paramters updated
  2007-06-11  PB V4.59 ImageLoop parameters updated
  2012-01-31  PB V4.60 ImageLoop parameters updated
  2013-01-18  PB V4.61 Dummy and output dimensions are skipped
  2013-01-25  PB V4.62 Saxs2TiffFilenames according to ArgvBlocks
  2016-02-17  PB V4.63 remove .gz suffix when creating output filename
  2017-10-19  PB V4.64 No key value replacement in user_io

---*/
# define Version  "saxs_tiff 4.64 2017-10-19, Peter Boesecke"

# include <errno.h>
# include <stdio.h>
# include <unistd.h>
# include <fcntl.h>

# include "SaxsPrograms.h"
# include "edftiff.h"

/* Flg */
# define SAuto   "auto"
/* Lin */
# define SPrefix "tnam"
/* Num */
# define SBps    "bps"
# define SRori   "rori"

/* Val */
# define SMin    "min"
# define SMax    "max"

# define Usage "[options] \n\
                <i1nam> <tiff file name -tnam> <i1fst> <i1lst> <i1inc>\n\
                <odum> <odim1> <odim2> <8|16|32-bps> <1-8-rori>\n\
                <auto> [<min> <max>]\n\n\
  Transformation of an edf image sequence to 1byte/2byte/4byte tiff images\n\
  -bps <8|16|32> : 8, 16 and 32 bits in tiff image (default 16)\n\
  -rori          : orientation of tiff image\n\
                   1: (x,y)\n\
                   2: (-x,y)  x-axis inverted\n\
                   3: (x,-y)  y-axis inverted (default)\n\
                   4: (-x,-y) both axes inverted\n\
                   5: (y,x)   axes swapped\n\
                   6: (y,-x)  axes swapped and x-axis inverted\n\
                   7: (-y,x)  axes swapped and y-axis inverted\n\
                   8: (-y,-x) axes swapped and both axes inverted\n\
  +auto          : automatic scaling between minimum and maximum,\n\
                   output dummy is set to upper range value,\n\
                   minimum and maximum are calculated, if not set\n\
                   (default +auto)\n\
  -min <minimum of mapping range>, -max <maximum of mapping range>\n"

# define BlockNum 2       /* 1 input sequence + 1 output sequence */

/*---------------------------------------------------------------------------
NAME

   tiffnewstr --- allocate memory and copy a character string to it

SYNOPSIS

   char * tiffnewstr( const char * string );

DESCRIPTION
  Allocates strlen(string)+1 bytes of memory and copies string to it.
  In case of success the pointer to the allocated memory is returned. The
  null pointer is returned in case of an error.
  If string is the NULL pointer the NULL pointer is returned.

RETURN VALUE
  Returns the pointer to the allocated string or (char *) NULL in case
  of an error.
---------------------------------------------------------------------------*/
char * tiffnewstr( const char * string )
{ char * newstring;

  if (!string) return( (char *) NULL );
  if (!(newstring = (char *) malloc(strlen(string)+1))) return((char *) NULL);
  (void) strcpy(newstring, string);

  return( newstring );

} /* tiffnewstr */

/*---------------------------------------------------------------------------
NAME
  serialfilename --- create a file name with prefix, number and extension.

DESCRIPTION
  Combines a file name from prefix, number and extension according to
  <prefix><num><extension>

SYNOPSIS
  char * serialfilename(char * buffer, const char * prefix,
                long num, const char * extension );

ARGUMENTS
  char       * buffer    : buffer, must be sufficiently long
  const char * prefix    : prefix string
  long         num       : serial number
  const char * extension : extension string including separator, e.g. dot

RETURN VALUE
  char * filename
---------------------------------------------------------------------------*/
char * serialfilename(char * buffer, const char * prefix,
                long num, const char * extension )
{      if (num<=-1000)sprintf( buffer,   "%s-%4u%s", prefix,-num, extension );
  else if (num<=-100) sprintf( buffer,  "%s-0%3u%s", prefix,-num, extension );
  else if (num<=-10 ) sprintf( buffer, "%s-00%2u%s", prefix,-num, extension );
  else if (num<=-1  ) sprintf( buffer,"%s-000%1u%s", prefix,-num, extension );
  else if (num>=1000) sprintf( buffer,   "%s+%4u%s", prefix, num, extension );
  else if (num>=100 ) sprintf( buffer,  "%s+0%3u%s", prefix, num, extension );
  else if (num>=10  ) sprintf( buffer, "%s+00%2u%s", prefix, num, extension );
  else                sprintf( buffer,"%s+000%1u%s", prefix, num, extension );

  return( buffer );
} /* serialfilename */

/*---------------------------------------------------------------------------
NAME
  getprefix --- return file name prefix

DESCRIPTION
  returns the string before the last occurrence of .. If filename does
  not contain a . the full filename is returned as prefix. Memory
  is allocated for prefix. It must be released with free().

SYNOPSIS
  char * getprefix( const char * filename );

ARGUMENTS
  const char * filename : string to be analysed (prefix.extension)

RETURN VALUE
  char * prefix  (NULL, if filename too long)
---------------------------------------------------------------------------*/
char * getprefix( const char * filename )
{ char buffer[IO_size];
  size_t str_len;
  char *ps;

  str_len = strlen(filename);
  if (str_len>=IO_size) return( (char *) NULL );

  if (ps = strrchr( filename, (int) '.' )) {
    memcpy( buffer, filename, (ps-filename) );
    buffer[(ps-filename)]='\0';
    } else {
    memcpy( buffer, filename, (str_len+1) );
    }

  return( tiffnewstr(buffer) );

} /* getprefix */

/*---------------------------------------------------------------------------
NAME
  getextension --- return file name extension

DESCRIPTION
  Returns the string from the last occurrence of .. If filename does
  not contain a . an empty string is returned. Memory is allocated for
  extension. It must be released with free().

SYNOPSIS
  char * getextension( const char * filename );

ARGUMENTS
  const char * filename : string to be analysed (prefix.extension)

RETURN VALUE
  char * extension (NULL, if filename too long)
---------------------------------------------------------------------------*/
char * getextension( const char * filename )
{ char buffer[IO_size];
  size_t str_len;
  char *ps;

  str_len = strlen(filename);
  if (str_len>=IO_size) return( (char *) NULL );

  if (ps = strrchr( filename, (int) '.' )) {
    memcpy( buffer, ps, (str_len-(ps-filename)) );
    buffer[(str_len-(ps-filename))]='\0';
    } else {
    buffer[0]='\0';
    }

  return( tiffnewstr(buffer) );

} /* getextension */

/*---------------------------------------------------------------------------
Saxs2TiffFilenames

Ask for all file names in the following way:
  input file 1 [input.edf] :
  input file 2 [<input file 1>] :
  ...
  output file [output.edf] :

  first image [<minimum image number>] :
  last image [<maximum image number>] :
  increment [1] :

  output dummy [<dummy in first image>] :

The routine must be used for all images at once.
---------------------------------------------------------------------------*/
void Saxs2TiffFilenames ( CmdBlk * pcb, ImgBlk ib[], ImgHeadBlk ihb[],
                     int block_1st, int block_lst, int * pstatus)
{ int blkno, first_input, last_input;
  long image_1st, image_lst;
  char linebuffer[IO_size];
  char prefix[IO_size];
  long num[BlockNum];
  IO_line *pPrefix;

  first_input = MAX2(1,block_1st);
  last_input  = block_lst;

  /* ---  File names and open modes --- */

  for (blkno = first_input;blkno <= last_input;blkno++) {

    /*--- Open mode ib[blkno].OpenMode of input files */
    if (!ib[blkno].OpenMode.I) {
      ib[blkno].OpenMode.V = IO_Old | IO_FileProtect;
      ib[blkno].OpenMode.I = TRUE;
      }

    /*--- File name ib[blkno].Name of input files */
    (void) sprintf(linebuffer,"Input sequence %d",blkno);
    if (blkno==1)
      argv_filename( pcb,linebuffer, ib, 1, DefaultInput, pstatus);
    else argv_filename( pcb,linebuffer,ib,blkno,ib[blkno-1].Name.V,pstatus);
    /* Exit in all cases after an abort from the first argument */
    if (*pstatus!=Success) return;

    /* No key value replacement in user_io -> ihb=NULL */
    OpenImageFile( pcb,ib,blkno,NULL,
                   ib[blkno].Name.V,ib[1].FFirst.V,
                   IO_Old|IO_FileProtect,pstatus);
    if (*pstatus!=Success) return;

    if (blkno==1) {
      /* Search for minimum and maximum image number in 1st input sequence */
      (void) SearchMinMaxImage ( pcb, ib, 1, &image_1st, &image_lst, pstatus);
      if (*pstatus!=Success) return;
    }

  } /* for input files */

  if (block_1st<=0) {
    /*--- OpenMode of output file */
    if (!ib[0].OpenMode.I) {
      ib[0].OpenMode.V = IO_ImageProtect | IO_DontReadWrite | IO_DontOpen;
      ib[0].OpenMode.I = TRUE;
      }

    /*--- File name ib[0].Name of output (not open) */
    if (!ib[0].Name.I) {
      (void) sprintf(ib[0].Name.V,"%s",DefaultOutput);
      }

    /*--- Argument  : output tiff file name */
    pPrefix = (IO_line*) option_parameter_search( SPrefix, pcb, pstatus );
    if (*pstatus!=Success) return;

    if (!pPrefix->I) {
      char Suffix[IO_size];

      /* Determine prefix from file names */
      if (ib[0].Name.I) {
       filename_body(prefix,IO_size-5,ib[0].Name.V);
      } else {
       filename_body(prefix,IO_size-5,filename_name(prefix,IO_size,ib[1].Name.V));
      }

      /* Check for .gz extension (use Suffix as temporary buffer) */
      filename_extension ( Suffix, IO_size, prefix );
      if (!strcmp(Suffix,".gz")) {
        // Remove .gz extension
        filename_body ( prefix, IO_size, prefix ); // remove .gz
      } 
 
      (void) sprintf(pPrefix->V,"%s%s", prefix, ".tif");
      pPrefix->I = TRUE;
    }

    argv_line( pcb,"Output tiff file",pPrefix,pPrefix->V,pstatus);
    if (*pstatus!=Success) return;

    /* extract output filenumbers from TIFF-file name */
    extract_filenumbers( &ib[0], pPrefix, TRUE, pstatus );
    if (*pstatus!=Success) return;

    /*--- Take prefix as default for output filename */
    if (!ib[0].Name.I) {
      (void) sprintf(ib[0].Name.V,"%s",pPrefix->V);
      }

  } /* output file */

  /* ---  Image numbers --- */

  if (first_input==1) {

    /*--- Argument : number of first image */
    argv_long( pcb,"First image of input sequence 1", &ib[1].First, image_1st,
               pstatus);
    if (*pstatus!=Success) return;

    /*--- Argument : number of last image */
    if (pcb->argc==*pcb->parg_no+1) image_lst = ib[1].First.V;
    argv_long( pcb,"Last image of input sequence 1", &ib[1].Last, image_lst,
               pstatus);
    if (*pstatus!=Success) return;

    /*--- Argument : increment (default = 1) */
    argv_long( pcb,"Increment of input sequence 1", &ib[1].Inc, 1 , pstatus);
    if (*pstatus!=Success) return;

    num[1] = ib[1].First.V;
    (void) ReadImageHeader( pcb, ib, 1, num, ihb, pstatus);
    if (*pstatus!=Success) return;
    if (pcb->TestBit) PrintImageHeaderBlock(&ihb[1]);

  } /* first input file */

  for (blkno = MAX2(2,first_input);blkno<=last_input;blkno++) {
    /*--- image numbers of all other images */
    (void) sprintf(linebuffer,"First image of input sequence %d",blkno);
    argv_long( pcb,linebuffer, &ib[blkno].First, ib[1].First.V, pstatus);
    if (*pstatus!=Success) return;

  } /* all other input files */

  if (block_1st <= 0) {

    if (ib[0].First.I) num[0] = ib[0].First.V;
    else num[0] = image_1st;

    if ( (pcb->ImgBlkLen>1) && (ihb[1].I) ) {
      NewImageHeader(pcb,ib,0,num,ihb,&(ihb[1]),NULL,pstatus);
    } else {
      NewImageHeader(pcb,ib,0,num,ihb,NULL,NULL,pstatus);
    }
    if (*pstatus!=Success) return;

    /*--- Argument : output dummy */
    sprintf(linebuffer,"Output dummy, e.g. %g",ihb[0].Dummy.V);
    argv_flexp( pcb, linebuffer, &ib[0].Dummy, "", pstatus);
    if (*pstatus!=Success) return;

    /*--- Argument : output dimension 1 */
    sprintf(linebuffer,"Output dimension 1, e.g. %ld",ihb[0].Dim[1]);
    argv_lvexp(pcb,linebuffer,
      &ib[0].Dim[1],ib[0].Dim[1].I?ib[0].Dim[1].V:"",pstatus);
    if (*pstatus!=Success) return;

    /*--- Argument : output dimension 2 */
    sprintf(linebuffer,"Output dimension 2, e.g. %ld",ihb[0].Dim[2]);
    argv_lvexp(pcb,linebuffer,
      &ib[0].Dim[2],ib[0].Dim[2].I?ib[0].Dim[2].V:"", pstatus);
    if (*pstatus!=Success) return;

  }

  for (blkno = MAX2(1,first_input);blkno<=last_input;blkno++) {
    CloseImageFile( pcb, ib, blkno, pstatus) ;
    if (*pstatus!=Success) return;
  } /* close all input files */

} /* Saxs2TiffFilenames */

/*---------------------------------------------------------------------------
1 saxs_tiff

2 PURPOSE
  Multiplicates an image with a factor and adds a constant
  output to tiff file 
---------------------------------------------------------------------------*/
void saxs_tiff (CmdBlk * pcb, long num[], ImgHeadBlk ihb[], int * pstatus)
{ int i,imax;
  int j;

  float *I0Data;
  int   I0Dim_1,I0Dim_2;
  float *pI0Data;
  float I0Dummy, I0DDummy;

  int   I1Ave;
  float *I1Data;
  float *pI1Data;
  int   I1Dim_1,I1Dim_2;
  float I1Dummy, I1DDummy;
  float I1Value, I1Sum, I1Weight;

  float Value;

  int i_1, i_2;
  float f_1[BlockNum], f_2[BlockNum];
 
  float Off_1[BlockNum], Off_2[BlockNum];
  float Ps_1[BlockNum], Ps_2[BlockNum];
 
  float f1_1[BlockNum], f3_1[BlockNum], Df_1[BlockNum];
  float f1_2[BlockNum], f3_2[BlockNum], Df_2[BlockNum];
 
  int Imin_1, Imin_2, Imax_1, Imax_2;
 
  float Wmin_1, Wmax_1, W_1, DW_1;
  float Wmin_2, Wmax_2, W_2, DW_2; 

  IO_long Auto;
  IO_line Prefix;
  IO_long Bps, Rori;
  IO_float Min, Max;

  char *prefix, *extension;
  char OutputBinaryName[IO_size];
  FILE *OutputBinaryId;
  size_t t_length;
  char fname_full[IO_size], Temp[IO_size];

  short bps;
  int   orientation;
  int   autorange, automin, automax;
  float vmin, vmax;
  long  xdim, ydim;
  short compression=1;
  char  *title, *description=(char *) NULL;
  char  comment[IO_size];
  float xdpcm, ydpcm;

  void *tiffdata;

  *pstatus = Success;

  imax = pcb->ImgBlkLen;
  printf("\nCalculating ihb[0,% d] = Function(",ihb[0].ImNum);
  for(i=1;i<imax;i++) {
    printf("ihb[% d,% d] ", i, ihb[i].ImNum); }

  printf(")\n\n");

  for(i=1;i<imax;i++) {
    if (pcb->TestBit) PrintImageHeaderBlock(&ihb[i]);}

 /* Check the number of images */
  if (pcb->ImgBlkLen!=2) {
     printf("%d images found, 1 input and 1 output image required\n",
             pcb->ImgBlkLen); *pstatus=Failed; return; }

  GetReferenceParameters( pcb, ihb, 0, imax-1,
                          Off_1, Off_2, Ps_1,  Ps_2, pstatus );
  if (*pstatus!=Success) return;

  UpdateImageParameters ( pcb, ihb, 1, imax-1,
                          Off_1, Off_2, Ps_1,  Ps_2, pstatus );
  if (*pstatus!=Success) return;

 /* --- Get option values */
  Auto = option_flag ( SAuto, pcb, pstatus );
  if (*pstatus!=Success) return;
  Prefix = option_line ( SPrefix, pcb, ihb, num, 1, pstatus );
  if (*pstatus!=Success) return;
  Bps = option_long ( SBps, pcb, ihb, num, 1, pstatus );
  if (*pstatus!=Success) return;
  Rori = option_long ( SRori, pcb, ihb, num, 1, pstatus );
  if (*pstatus!=Success) return;
  Min = option_float ( SMin, pcb, ihb, num, 1, pstatus );
  if (*pstatus!=Success) return;
  Max = option_float ( SMax, pcb, ihb, num, 1, pstatus );
  if (*pstatus!=Success) return;

 /* 1 input, 1 output */
  I0Data   = ihb[0].Data;
  I0Dummy  = ihb[0].Dummy.V;
  I0DDummy = ihb[0].DDummy.V;
  I0Dim_1  = (int) ihb[0].Dim[1];
  I0Dim_2  = (int) ihb[0].Dim[2];

  I1Ave    = pcb->ib[1].Ave.V;
  I1Data   = ihb[1].Data;
  I1Dummy  = ihb[1].Dummy.V;
  I1DDummy = ihb[1].DDummy.V;
  I1Dim_1  = (int) ihb[1].Dim[1];
  I1Dim_2  = (int) ihb[1].Dim[2];

  GetImageRange         ( pcb, ihb, 1, imax-1,
                          Off_1, Off_2, Ps_1,  Ps_2,
                          f1_1, f3_1, Df_1, f1_2, f3_2, Df_2,
                          &Imin_1, &Imax_1, &Imin_2, &Imax_2,
                          &Wmin_1, &Wmax_1, &DW_1, &Wmin_2, &Wmax_2, &DW_2, 1);

 /* special parameters */
  bps         = (short) Bps.V;
  orientation = (int) Rori.V;
  autorange   = (int) Auto.V;
  automin     = (int) (!Min.I); 
  automax     = (int) (!Max.I); 
  vmin        = Min.V;
  vmax        = Max.V;
  xdim        = I0Dim_1;
  ydim        = I0Dim_2;
  if (ihb[0].Title.I)
    title = ihb[0].Title.V; // output file title
   else title = pcb->ib[1].Name.V; 
  if (ihb->PixSiz[1].I)
    xdpcm = 0.01/ihb->PixSiz[1].V;  // dots per cm
   else xdpcm = 1.0;
  if (ihb->PixSiz[2].I)
    ydpcm = 0.01/ihb->PixSiz[2].V;  // dots per cm
   else ydpcm = 1.0;

  if (pcb->TestBit) {
    printf("         bps = %d\n",bps);
    printf("   autorange = %d\n",autorange);
    if (automin) 
      printf("        vmin = auto\n"); 
     else printf("        vmin = %g\n",vmin);
    if (automax)
      printf("        vmax = auto\n");
     else printf("        vmax = %g\n",vmax);
    printf("        xdim = %d\n",xdim);
    printf("        ydim = %d\n",ydim);
    printf(" compression = %d\n",compression);
    printf("       title = %s\n",title);
    printf("       xdpcm = %g\n",xdpcm);
    printf("       ydpcm = %g\n",xdpcm);
    }

  /* loop over the output array */
  W_2 = Wmin_2; for (i=0;i<imax;i++) f_2[i]=f1_2[i];
  for (i_2=Imin_2;i_2<=Imax_2;i_2++) {
 
    pI0Data = ABSPTR(I0Data,I0Dim_1,I0Dim_2,Imin_1,i_2);
    W_1 = Wmin_1; for (i=0;i<imax;i++) f_1[i]=f1_1[i];
    for (i_1=Imin_1;i_1<=Imax_1;i_1++) {
 
      if (Isum2ldw (I1Data,I1Dim_1,I1Dim_2,I1Dummy,I1DDummy,
            f_1[1], f_2[1], f_1[1]+Df_1[1], f_2[1]+Df_2[1], &I1Sum, &I1Weight)) {
 
        I1Value = I1Sum; if (I1Ave) I1Value /= I1Weight;
 
        UPDATE( *pI0Data, I1Value, I0Dummy, I0DDummy );
 
      } /* if (Isum2ldw ... */
 
      pI0Data++;
      W_1+=DW_1; for (i=0;i<imax;i++) { f_1[i]+=Df_1[i]; }
    } /* for i_1 ... */
 
    W_2+=DW_2; for (i=0;i<imax;i++) { f_2[i]+=Df_2[i]; }
  } /* for i_2 ... */

  if (pcb->TestBit) printf("Output to multiple tiff files\n");

  // Generate tiff filename
  filename_parameter (fname_full,IO_size,Prefix.V,0);
  filename_pattern (fname_full,IO_size,fname_full,(pcb->ib)[0].FileNumber);
  filename_full (fname_full,IO_size,fname_full);

  /* Output to multiple tiff files */
   if ( (pcb->ib[0].Inc.V==0) || (pcb->ib[0].First.V==pcb->ib[0].Last.V) ) {
    // fixed output file name
    strncpy(OutputBinaryName,fname_full,IO_size);
    } else {
    // numbered output file name
    prefix = getprefix(fname_full);
    extension = getextension(fname_full);
    serialfilename(OutputBinaryName, prefix, ihb[0].ImNum, extension );
    free(prefix); free(extension);
    }
  if (pcb->TestBit) printf("OutputBinaryName = %s\n",OutputBinaryName);

  /* convert to tiff file */
  tiffdata = float2unsigned (I0Data, I0Dummy, I0DDummy, I0Dim_1, I0Dim_2, 
                             autorange,
                             automin, vmin, automax, vmax, bps, comment, 
                             pstatus);
  if (*pstatus!=Success) return;

  /* write description */
  description = (char *) malloc(strlen(comment)+strlen(title)+3);
  sprintf(description,"%s\r\n%s",title,comment);

  /* write tiff file */
  write_tiff ( OutputBinaryName, xdim, ydim, bps,
               compression, tiffdata, orientation,
               description, xdpcm, ydpcm, pstatus );
  free(description);
  if (*pstatus!=Success) return;

  if(tiffdata); free(tiffdata);

  printf("\n");

  printf("Input file         : % s\n",pcb->ib[1].FileName);
  printf("Input image number : % d\n",ihb[1].ImNum);
  printf("Output tiff file   : % s\n",OutputBinaryName);

 } /* saxs_tiff*/

/*---------------------------------------------------------------------------
user_io
Do all the keyboard io and return cb, and ib
---------------------------------------------------------------------------*/

void user_io(CmdBlk * pcb, ImgBlk * ib, int * pstatus)
{
  char buffer[IO_size];
  ImgHeadBlk ihb[BlockNum];
  float tmp;

  float ROff_1, RPs_1, UOff_1, UPs_1;
  float ROff_2, RPs_2, UOff_2, UPs_2;

  IO_long  *pAuto;
  IO_line  *pPrefix;
  IO_lvexp *pBps, *pRori;
  IO_flexp *pMin, *pMax;

  IO_long Bps;

  filename_name ( buffer, IO_size,  pcb->argv[0] );

  /* show usage if no arguments are given */
  if (pcb->argc<=1) printf("Usage: %s %s\n", buffer, Usage);

  /*--- Write name of program ---*/
  printf("\n");
  printf("   %s %s\n",buffer,Version);
  printf("\n");

  Saxs2TiffFilenames ( pcb, ib, ihb, 0, BlockNum-1, pstatus);
  if (*pstatus!=Success) return;
  GetReference(pcb->RSys.V,1,ihb,&ROff_1,&ROff_2,&RPs_1,&RPs_2,pstatus );
  if (*pstatus!=Success) return;
  GetReference(pcb->USys.V,1,ihb,&UOff_1,&UOff_2,&UPs_1,&UPs_2,pstatus );
  if (*pstatus!=Success) return;

  /* --- Argument  : Bits per sample */
  pBps = (IO_lvexp*) option_parameter_search( SBps, pcb, pstatus );
  if (*pstatus!=Success) return;
  argv_lvexp( pcb,"Bits per sample (8, 16, 32)", pBps, pBps->V, pstatus);
  if (*pstatus!=Success) return;

  /* --- Argument  : Rasterorientation */
  pRori = (IO_lvexp*) option_parameter_search( SRori, pcb, pstatus );
  if (*pstatus!=Success) return;
  argv_lvexp( pcb,"Orientation of output image (1-8)", pRori,pRori->V, pstatus);
  if (*pstatus!=Success) return;

  /* --- Argument  : Autorange */
  pAuto = (IO_long*) option_parameter_search( SAuto, pcb, pstatus );
  if (*pstatus!=Success) return;
  argv_flag( pcb,"Range output image with minimum and maximum?", 
             pAuto,pAuto->V, pstatus);
  if (*pstatus!=Success) return;

  pMin = (IO_flexp*) option_parameter_search( SMin, pcb, pstatus );
  if (*pstatus!=Success) return;
  pMax = (IO_flexp*) option_parameter_search( SMax, pcb, pstatus );
  if (*pstatus!=Success) return;

  if (!(pAuto->V)) {
    /* --- Argument  : Minimum value */
    if (!(pMin->I)) sprintf(pMin->V,"0.0");
    argv_flexp( pcb,"Minimum value", pMin, pMin->V, pstatus);
    if (*pstatus!=Success) return;

    /* --- Argument  : Maximum value */
    if (!(pMax->I)) {
      sprintf(buffer,"pow(2,%s)-1.0",pBps->V);
      tmp=floatexpr( buffer, pstatus );
      if (*pstatus!=Success) CopyLine(buffer,pMax->V,IO_size,0);
      else sprintf(pMax->V,"%g",tmp); 
    }
    argv_flexp( pcb,"Maximum value", pMax, pMax->V, pstatus);
    if (*pstatus!=Success) return;
    }

  pPrefix = (IO_line*) option_parameter_search( SPrefix, pcb, pstatus );
  if (*pstatus!=Success) return;

  printf("\n");
  if (ib[1].Name.I)    printf("input file         : %s\n",ib[1].Name.V);
  if (pPrefix->I)      printf("output prefix      : %s\n",pPrefix->V);
  if (ib[1].First.I)   printf("first image        : %d\n",ib[1].First.V);
  if (ib[1].Last.I)    printf("last image         : %d\n",ib[1].Last.V);
  if (ib[1].Inc.I)     printf("increment          : %d\n",ib[1].Inc.V);
  if (ib[0].Dummy.I)   printf("output dummy       : %s\n",ib[0].Dummy.V);
  if (ib[0].Dim[1].I)  printf("output dimension 1 : %s\n",ib[0].Dim[1].V);
  if (ib[0].Dim[2].I)  printf("output dimension 2 : %s\n",ib[0].Dim[2].V);
  if (pBps->I)         printf("bits per sample    : %s\n",pBps->V);
  if (pAuto->I)        printf("autorange          : %s\n",pAuto->V?"yes":"no");
  if (!(pAuto->V)) {
    if (pMin->I)       printf("minimum value      : %s\n",pMin->V);
    if (pMax->I)       printf("maximum value      : %s\n",pMax->V);
    }
  printf("\n");

  if (pcb->TestBit) {
    PrintBlocks ( pcb, ib );
    printf("\n"); }

  return;
} /* user_io */

/*---------------------------------------------------------------------------
main
---------------------------------------------------------------------------*/

#if MAKE_FUNCTION
# define MAIN main_saxs_tiff
#else
# define MAIN main
#endif

int MAIN (int argc, char *argv[])
{
  CmdBlk cb;                /* command block  */
  ImgBlk ib[BlockNum];      /* image blocks */

  int status;
  int arg_no = 0;

 /* Init options, control block and image blocks */
  InitOptions( Usage, Version, NULL, TRUE, &cb, ib, BlockNum );

  option_define ( IO_tpflag,  SAuto,   "TRUE", &cb );
  option_define ( IO_tpline,  SPrefix, "datafile", &cb );
  option_define ( IO_tplvexp, SBps,    "16", &cb );
  option_define ( IO_tplvexp, SRori,   "3", &cb );

  option_define ( IO_tpflexp, SMin,    "-inf", &cb );
  option_define ( IO_tpflexp, SMax,    "+inf", &cb );

 /* Read options from argument list */
  ReadOptions( argv, &arg_no, &cb, ib, &status);

  /* USER KEYBOARD I/O */
  if (status==Success) {
    argv_start ( &cb, 1 );
    user_io( &cb, ib, &status);
    argv_end( &cb ); /* must be called after user_io */
  }

  /* SEQUENCE CALCULATION */
  if (status==Success) 
    IMAGELOOP( &cb, ib, saxs_tiff, NULL, NULL, TRUE, &status );

  return( ReportSaxsStatus( status, 0 ) );

} /* MAIN */

