/***************************************************************************/
/* Written 1994++ by Peter Boesecke                                        */
/* Copyright (C) 2011 European Synchrotron Radiation Facility              */
/*                       Grenoble, France                                  */
/*                                                                         */
/*    Principal authors: Peter Boesecke  (boesecke@esrf.eu)                */
/*                                                                         */
/*    This program is free software: you can redistribute it and/or modify */
/*    it under the terms of the GNU General Public License as published by */
/*    the Free Software Foundation, either version 3 of the License, or    */
/*    (at your option) any later version.                                  */
/*                                                                         */
/*    This program is distributed in the hope that it will be useful,      */
/*    but WITHOUT ANY WARRANTY; without even the implied warranty of       */
/*    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the        */
/*    GNU General Public License for more details.                         */
/*                                                                         */
/*    You should have received a copy of the GNU General Public License    */
/*    along with this program.  If not, see <http://www.gnu.org/licenses/>.*/
/***************************************************************************/
/*+++
1 saxs_stat
  Calculate some statistics on the image input file (esrf format) and
  write them to the output text file. The statistics calculated are:
                inum    Image number
                dim1    Dimension 1
                dim2    Dimension 2
                inf1    inferior limit 1
                sup1    superior limit 1
                inf2    inferior limit 2
                sup2    superior limit 2
                ntot    Number of data points
                nval    Number of valid data points
                ndum    Number of dummies
                max     Maximum Value
                min     Minimum Value
                amax    Maximum absolute value
                amin    Minimum absolute value
                dum     Dummy value
                sum     Sum of data values
                ssum    Sum of squared data values
                mean    Mean value = sum / nval
                smean   Mean squared value = ssum / nval
                rms     Root mean square value = sqrt( smean )
                devi    Standard deviation value = sqrt( smean - mean*mean )
                gc1     Center of mass 1
                gc2     Center of mass 2
                gd1     Standard deviation 1 of center of mass 
                gd2     Standard deviation 2 of center of mass
                [key]   Header key value

2 PURPOSE

  Arguments:
  saxs_stat [options] <i1nam> <output prefix> <i1fst> <i1lst> <i1inc> <select>
  defaults:
  <input file name>   : input.edf 
  <output prefix>     : input 
  <first image>       : <first image number in input file>
  <last image>                              : <last image number in input file>
     if argument list ends with first image : <first image>
  <increment>         : 1
  <selection>         : all

  Specific options: 
  -pref <output file prefix> : specify the name of the ascii ouput file in which
                               results wil be stored.
  -sel <column name>         : select the column that should be
                               written to the output file
  -key <key>                 : selects the key(s) which values are written
                               to a separate column of the output file.
                               Several keys must be separated with commas.

  Specific options:
  +pref <output file prefix>    : prefix of the output file name
  +ext  <output file extension> : extension of the output file name
  +fnam <output file name>      : output file name
  +-show                        : show results

  The values are written to the file <prefix>_<keyword><extension> .

  Wildcard:
  A "=" in the argument list chooses the default.

2 HISTORY
  21-Jun-1995 PJE 
  04-Dec-1995 PB  (3.0)
  15-Dec-1995 PB  center of mass deviation changed to standard deviation (3.1)
  29-Mar-1996 PB  center of mass now also for negative mean
  26-Jun-1999 PB  DevSeparator, DirSeparator, ExtSeparator defined in SaxsInput
  23-Jul-1999 PB  V3.12 double float values during calculations, 
                  normalization of array with MeanZ before COM calculation 
  04-Oct-1999 PB  V3.13 wrong results for AbsMeanZ = 0
  09-Nov-1999 PB  V3.14 DUMMYDEFINED updated
  26-Nov-1999 PB  V3.15 cc on dec alpha:
                        underflow for eps = 1e-40
                        eps initialized with FLT_MIN (<float.h> included)
  16-Apr-2000 PB  V3.16 \r\n
  2000-08-04  PB  %f->%g
  2001-07-03  PB  V3.17 filename_full
  2001-07-08  PB  V3.18 FFirst
  2001-07-09  PB  V3.19 (pcb->ib)[0].FileNumber
  2001-07-09  PB  V4.00 new loop, no repetition
  2001-09-16  PB  V4.01 min abs value
  2001-10-27  PB  V4.02 ImageIsFirst, %%%.edf -> to single txt file possible
  2002-06-01  PB  V4.50 ave
  2003-02-03  PB  V4.51 num[BlockNum]
  2003-02-09  PB  V4.52 line args for Dummy, DDummy, Dim
  2003-05-30  PB  V4.53 ImageLoop: Function call changed
  2003-08-14  PB  V4.54 option_define, all option numbers removed
  2003-11-30  PB  V4.55 BinSiz
  2004-09-13  PB  V4.56 Missing label 'KeysMean' added, option Hist added
  2004-10-31  PB  V4.57 MAKE_FUNCTION
  2005-07-15  PB  V4.58 ReportSaxsStatus parameter list updated,
                        SetSaxsErrorExternal added
  2005-08-04  PB  V4.59 PrintImageHeaderBlock paramters updated
  2007-06-03  PB  V4.60 Option 'key' added
  2007-06-11  PB  V4.61 ImageLoop parameters updated
  2008-12-05  PB  V4.62 open output files as binary files to avoid
                        translation of \r\n to \n when compiled on windows,
                        output in spec file format
  2011-05-14  PB  V4.63 split.h -> strlib.h
  2012-01-31  PB  V4.64 ImageLoop parameters updated
  2016-02-17  PB  V4.65 remove .gz suffix when creating output filename
  2016-10-02  PB  V4.66 NUsage corrected
  2016-10-07  PB  V4.67 filename_full expansion of Prefix option removed
  2017-10-20  PB  V4.68 No key value replacement in user_io,
                        ReadHeaderLine arguments updated
  2018-02-07  PB  V4.69 MaxAbsZ

---*/
# define Version  "saxs_stat V4.69 2018-02-07, Peter Boesecke"

# include <errno.h>
# include <stdio.h>
# include <unistd.h>
# include <fcntl.h>
# include <float.h>

# include "SaxsPrograms.h"
# include "strlib.h"

/* Option definitions */
# define SSelect           "sel"  /* string */
# define SKey              "key"  /* header key to write */
# define SPrefix           "pref"
# define SExtension        "ext"    
# define SFName            "fnam"
# define SSpacer           "spc"
# define SShow             "show" /* flag */
# define SHist             "hist" /* write history */
# define SSpec             "spec" /* write spec file */

# define Default_select    SelectAll 

# define NUsage "[options] \n\
                <i1nam> <output prefix> <i1fst> <i1lst> <i1inc> \n\
                <selection key>\n\n\
                specific options: -sel   <selection key (default all)> \n\
                                  -key   <header key>{,<header key>}> \n\
                                  -pref  <output prefix> \n\
                                  -ext   <output extension (default .txt)> \n\
                                  -fnam  <output file name> \n\
                                  -spc   <spacer (default \"  \\t\")> \n\
                                  +-spec write spec header (default)\n\
                                  +-show show results\n\n\
                Calculate statistical parameters of an image.\n\n"

/* keywords for the colunm selection*/
# define SelectAll "all"
# define KeyiNum "inum"
# define KeyDim1 "dim1"
# define KeyDim2 "dim2"
# define KeyMin1 "inf1"
# define KeyMax1 "sup1"
# define KeyMin2 "inf2"
# define KeyMax2 "sup2"
# define KeynTot "ntot"
# define KeynVal "nval"
# define KeynDum "ndum"
# define KeyMax  "max"
# define KeyMin  "min"
# define KeyaMax "amax"
# define KeyaMin "amin"
# define KeyDum  "dum"
# define KeySum  "sum"
# define KeysSum "ssum"
# define KeyMean "mean"
# define KeysMean "smean"
# define KeyRMS  "rms"
# define KeyDevi "devi"
# define KeyGC1  "gc1"
# define KeyGC2  "gc2"
# define KeyGD1  "gd1"
# define KeyGD2  "gd2"

/* suit of the help */
# define h00 "\t\tselection keys:\n"
# define h01 "\t\t"SelectAll"\tfor all\n"
# define h02 "\t\t"KeyiNum"\tImage number\n"
# define h03 "\t\t"KeyDim1"\tDimension 1\n"
# define h04 "\t\t"KeyDim2"\tDimension 2\n"
# define h05 "\t\t"KeyMin1"\tinferior limit 1\n"
# define h06 "\t\t"KeyMax1"\tsuperior limit 1\n"
# define h07 "\t\t"KeyMin2"\tinferior limit 2\n"
# define h08 "\t\t"KeyMax2"\tsuperior limit 2\n"
# define h09 "\t\t"KeynTot"\tNumber of data points\n"
# define h10 "\t\t"KeynVal"\tNumber of valid data points\n"
# define h11 "\t\t"KeynDum"\tNumber of dummies\n"
# define h12 "\t\t"KeyMax"\tMaximum Value\n"
# define h13 "\t\t"KeyMin"\tMinimum Value\n"
# define h14 "\t\t"KeyaMax"\tMaximum absolute value\n"
# define h15 "\t\t"KeyaMin"\tMinimum absolute value\n"
# define h16 "\t\t"KeyDum"\tDummy value\n"
# define h17 "\t\t"KeySum"\tSum of data values\n"
# define h18 "\t\t"KeysSum"\tSum of squared data values\n"
# define h19 "\t\t"KeyMean"\tMean value = sum / nval\n"
# define h20 "\t\t"KeysMean"\tMean square value = ssum / nval\n"
# define h21 "\t\t"KeyRMS"\tRoot mean square value = sqrt( smean )\n"
# define h22 \
  "\t\t"KeyDevi"\tStandard deviation value = sqrt( smean - mean*mean )\n"
# define h23 "\t\t"KeyGC1"\tCenter of mass 1\n"
# define h24 "\t\t"KeyGC2"\tCenter of mass 2\n"
# define h25 "\t\t"KeyGD1"\tStandard deviation 1 of center of mass\n"
# define h26 "\t\t"KeyGD2"\tStandard deviation 2 of center of mass\n"
# define Usage NUsage \
h00 h01 h02 h03 h04 h05 h06 h07 h08 h09 h10 h11 h12 h13 \
h14 h15 h16 h17 h18 h19 h20 h21 h22 h23 h24 h25 h26

# define BlockNum 2       /* 1 input sequence + 1 output sequence */

# define SIGN( X ) ( (X) < 0 ) ? -1 : 1

/*---------------------------------------------------------------------------
SaxsScalArgvFilenames

Ask for all file names in the following way:
  input file 1 [input.edf] :
  input file 2 [<input file 1>] :
  ...
  output file [output.edf] :

  first image [<minimum image number>] :
  last image [<maximum image number>] :
  increment [1] :

  output dummy [<dummy in first image>] :

If blkno >= 1 the header block of the first image is read in ihb[1].
The routine must be used for all images at once.
---------------------------------------------------------------------------*/
void SaxsScalArgvFilenames ( CmdBlk * pcb, ImgBlk ib[], ImgHeadBlk ihb[],
                     int block_1st, int block_lst, int * pstatus)
{ int blkno, first_block;
  long image_1st, image_lst;
  char linebuffer[1000];
  int nameblk;
  long num[BlockNum];
  IO_line *pPrefix;

  first_block = MAX2(1,block_1st);

  /* ---  File names and open modes --- */

  for (blkno = first_block;blkno <= block_lst;blkno++) {

    /*--- Open mode ib[blkno].OpenMode of input files */
    if (!ib[blkno].OpenMode.I) {
      ib[blkno].OpenMode.V = IO_Old | IO_FileProtect;
      ib[blkno].OpenMode.I = TRUE;
      }

    /*--- File name ib[blkno].Name of input files */
    (void) sprintf(linebuffer,"Input sequence %d",blkno);
    if (blkno==1)
      argv_filename( pcb,linebuffer, ib, 1, DefaultInput, pstatus);
    else argv_filename( pcb,linebuffer,ib,blkno,ib[blkno-1].Name.V,pstatus);
    /* Exit in all cases after an abort from the first argument */
    if (*pstatus!=Success) return;

    /* No key value replacement in user_io -> ihb=>NULL */
    OpenImageFile( pcb,ib,blkno,NULL,
                   ib[blkno].Name.V,ib[1].FFirst.V,
                   IO_Old|IO_FileProtect,pstatus);
    if (*pstatus!=Success) return;

    if (blkno==1) {
      /* Search for minimum and maximum image number in 1st input sequence */
      (void) SearchMinMaxImage ( pcb, ib, 1, &image_1st, &image_lst, pstatus);
      if (*pstatus!=Success) return;
    }

  } /* for input files */

  if (block_1st<=0) {

    /*--- OpenMode of output file */
    if (!ib[0].OpenMode.I) {
      ib[0].OpenMode.V = IO_ImageProtect | IO_DontReadWrite | IO_DontOpen;
      ib[0].OpenMode.I = TRUE;
      }

    /*--- Argument  : text file prefix */
    if (ib[0].Name.I) nameblk = 0; else nameblk = 1;
    pPrefix = (IO_line*) option_parameter_search( SPrefix, pcb, pstatus );
    if (*pstatus!=Success) return;
 
    if (!pPrefix->I) {
      /* Determine prefix from input file name */
      /* Check for .gz extension (use pPrefix->V as temporary buffer) */
      filename_extension ( pPrefix->V, IO_size, ib[nameblk].Name.V );
      if (!strcmp(pPrefix->V,".gz")) {
        // Remove .gz extension
        filename_body ( pPrefix->V, IO_size, ib[nameblk].Name.V ); // remove .gz
        filename_body ( pPrefix->V, IO_size, pPrefix->V ); // remove ext.
      } else {
        /* Generate filename of ASCII file */
        filename_body ( pPrefix->V, IO_size, ib[nameblk].Name.V ); // remove ext.
      }
      filename_name ( pPrefix->V, IO_size, pPrefix->V );
      pPrefix->I = TRUE;
    }

    /*--- Prefix of output file name */
    argv_line( pcb,"Output file prefix",pPrefix,pPrefix->V,
               pstatus);
    if (*pstatus!=Success) return;

    /* extract output filenumbers from ASCII file name */
    extract_filenumbers( &ib[0], pPrefix, TRUE, pstatus );
    if (*pstatus!=Success) return;
 
    /* Prefix cannot be expanded with filename_full 
       before replacing all keyvalues */
       
    /*--- Take prefix as default for output filename */
    if (!ib[0].Name.I) {
      (void) sprintf(ib[0].Name.V,"%s",pPrefix->V);
    }
   
  } /* output file */

  /* ---  Image numbers --- */

  if (first_block==1) {

    /*--- Argument : number of first image */
    argv_long( pcb,"First image of input sequence 1", &ib[1].First, image_1st,
               pstatus);
    if (*pstatus!=Success) return;

    /*--- Argument : number of last image */
    if (pcb->argc==*pcb->parg_no+1) image_lst = ib[1].First.V;
    argv_long( pcb,"Last image of input sequence 1", &ib[1].Last, image_lst,
               pstatus);
    if (*pstatus!=Success) return;

    /*--- Argument : increment (default = 1) */
    argv_long( pcb,"Increment of input sequence 1", &ib[1].Inc, 1 , pstatus);
    if (*pstatus!=Success) return;

    num[1] = ib[1].First.V;
    (void) ReadImageHeader( pcb, ib, 1, num, ihb, pstatus);
    if (*pstatus!=Success) return;
    if (pcb->TestBit) PrintImageHeaderBlock(&ihb[1]);

    /* Defaults for output file */
    if (!ib[0].Dummy.I) sprintf(ib[0].Dummy.V,"%g",ihb[1].Dummy.V);
    if (!ib[0].Dim[1].I) sprintf(ib[0].Dim[1].V,"%d",ihb[1].Dim[1]);
    if (!ib[0].Dim[2].I) sprintf(ib[0].Dim[2].V,"%d",ihb[1].Dim[2]);

  } /* first input file */

  for (blkno = MAX2(2,first_block);blkno<=block_lst;blkno++) {
    /*--- image numbers of all other images */
    (void) sprintf(linebuffer,"First image of input sequence %d",blkno);
    argv_long( pcb,linebuffer, &ib[blkno].First, ib[1].First.V, pstatus);
    if (*pstatus!=Success) return;

  } /* all other input files */

  for (blkno = MAX2(1,first_block);blkno<=block_lst;blkno++) {
    CloseImageFile( pcb, ib, blkno, pstatus) ;
    if (*pstatus!=Success) return;
  } /* close all input files */

} /* SaxsScalArgvFilenames */

/*----------------------------------------------------------------------------
NAME
  WriteHistory

DESCRIPTION
  Writes the full history to the output file 'out'

RETURN VALUE
  Number of written lines

HISTORY
  2004-09-13 PB from saxs_ascii and modified
----------------------------------------------------------------------------*/
long WriteHistory( FILE * out, CmdBlk * pcb, long num[], ImgHeadBlk ihb[], 
                   int * pstatus )
{
  long cnt=0;
  char * hdmk = "# ";

  const char * header_key="header_tmp";
  const char * key, * value;
 
  int ErrorValue;

  *pstatus = Failed;

  /* write history to a separate header */

  if ( !edf_new_header( header_key ) ) return( cnt );
  edf_history_write_header ( header_key, SaxsHistoryKey,
                             &ErrorValue, pstatus );
  SetSaxsErrorExternal( ErrorValue, ReportSaxsImageError );
  if (*pstatus!=Success) { edf_free_header( header_key ); return( cnt ); }

  edf_first_header_element ( header_key, &key, &value, &ErrorValue, pstatus );
  SetSaxsErrorExternal( ErrorValue, ReportSaxsImageError );
  if (*pstatus!=Success) { edf_free_header( header_key ); return( cnt ); }

  while (key) {
    fprintf(out,"%s%s = %s \r\n", hdmk, key, value);
    cnt++;
    edf_next_header_element ( header_key, &key, &value, &ErrorValue, pstatus );
    SetSaxsErrorExternal( ErrorValue, ReportSaxsImageError );
    if (*pstatus!=Success) { edf_free_header( header_key ); return( cnt ); }
  }

  if ( !edf_free_header( header_key ) ) return( cnt );

  *pstatus = Success;
  return(cnt);

} /* WriteHistory */

/*----------------------------------------------------------------------------
NAME
  write_spechead

DESCRIPTION
  Writes spec header values of block 0

RETURN VALUE
  Number of written lines

HISTORY
  2008-11-30 PB
----------------------------------------------------------------------------*/
long write_spechead( FILE * out, CmdBlk *pcb, long num[], ImgHeadBlk ihb[],
                    int *pstatus )
{ long cnt=0;
  int blkno=0;
  fprintf(out,"\r\n");cnt++;
  fprintf(out,"#S %d %s %s \r\n", ihb[blkno].ImNum,
    ihb[blkno].Time.I?ihb[blkno].Time.V:"",
    ihb[blkno].Title.I?ihb[blkno].Title.V:"");cnt++;
  *pstatus = Success;
  return(cnt);
} // write_spechead

/*---------------------------------------------------------------------------
1 saxs_stat

2 PURPOSE
  Statistics 
---------------------------------------------------------------------------*/
void saxs_stat (CmdBlk * pcb, long num[], ImgHeadBlk ihb[], int * pstatus)
{ //const float eps = 1e-40;
  const float eps = FLT_MIN;
  int i,imax;
  int j;
 
  float *I0Data;
  int   I0Dim_1,I0Dim_2;
  float *pI0Data;
  float I0Dummy, I0DDummy;

  int   I1Ave;
  float *I1Data;
  float *pI1Data;
  int   I1Dim_1,I1Dim_2;
  float I1Dummy, I1DDummy;
  float I1Value, I1Sum, I1Weight; 

  float Value;
  double DValue;

  int i_1, i_2;
  float f_1[BlockNum], f_2[BlockNum];
 
  float Off_1[BlockNum], Off_2[BlockNum];
  float Ps_1[BlockNum], Ps_2[BlockNum];
 
  float f1_1[BlockNum], f3_1[BlockNum], Df_1[BlockNum];
  float f1_2[BlockNum], f3_2[BlockNum], Df_2[BlockNum];
 
  int Imin_1, Imin_2, Imax_1, Imax_2;
 
  float Wmin_1, Wmax_1, W_1, DW_1;
  float Wmin_2, Wmax_2, W_2, DW_2;

  /* for the statistics */
  float Emin_1, Emax_1;
  float Emin_2, Emax_2;

  long NbDumZ, NbGoodZ, NbPixZ;
  double  MaxZ, MinZ, MaxAbsZ, MinAbsZ;
  double MeanZ = 0.0, MeanZ2 = 0.0, AbsMeanZ = 0.0, AbsMeanZ2 = 0.0;
  double SZ, SZ2;
  double SW1Z, SW2Z, SSW1Z, SSW2Z;
  double GC1, GC2, GD1, GD2;

  int   ZI=FALSE;
  int   GI=FALSE;

  char Sel[IO_size];

  char Key[IO_size], KeyValue[IO_size];

  int parno, cols;

  long  show;

  IO_line Select, KeyList, FName, Spacer, Extension, Prefix;
  IO_long Show, Hist, Spec;

  /* output file */
  FILE *OutputFileId;
  char fname_full[IO_size];

  *pstatus = Success;

  imax = pcb->ImgBlkLen;
  printf("Calculating statistics on ihb[0,% d] = Function(",ihb[0].ImNum);
  for(i=1;i<imax;i++) {
    printf("ihb[% d,% d] ", i, ihb[i].ImNum); }

  printf(")\n");

  if (pcb->TestBit)
    for(i=1;i<imax;i++)
      PrintImageHeaderBlock(&ihb[i]);

  /* Check the number of images */
  if (pcb->ImgBlkLen!=2) {
    fprintf(ERROUT,"%d images found, 1 input and 1 output image required\n",
            pcb->ImgBlkLen); *pstatus=Failed; return; 
  }

  GetReferenceParameters( pcb, ihb, 0, imax-1,
                          Off_1, Off_2, Ps_1,  Ps_2, pstatus );
  if (*pstatus!=Success) return;

  UpdateImageParameters ( pcb, ihb, 1, imax-1,
                          Off_1, Off_2, Ps_1,  Ps_2, pstatus );
  if (*pstatus!=Success) return;

  /* --- Get option values */
  Select = option_line ( SSelect, pcb, ihb, num, 1, pstatus );
  if (*pstatus!=Success) return;
  KeyList = option_line ( SKey, pcb, ihb, num, 1, pstatus );
  if (*pstatus!=Success) return;
  Extension = option_line ( SExtension, pcb, ihb, num, 1, pstatus );
  if (*pstatus!=Success) return;
  Prefix = option_line ( SPrefix, pcb, ihb, num, 1, pstatus );
  if (*pstatus!=Success) return;
  FName = option_line ( SFName, pcb, ihb, num, 1, pstatus );
  if (*pstatus!=Success) return;
  Spacer = option_line ( SSpacer, pcb, ihb, num, 1, pstatus );
  if (*pstatus!=Success) return;
  Show = option_flag ( SShow, pcb, pstatus );
  if (*pstatus!=Success) return;
  Hist = option_flag ( SHist, pcb, pstatus );
  if (*pstatus!=Success) return;
  Spec = option_flag ( SSpec, pcb, pstatus );
  if (*pstatus!=Success) return;

  /* 1 input, 1 output */

  I0Data   = ihb[0].Data;
  I0Dummy  = ihb[0].Dummy.V;
  I0DDummy = ihb[0].DDummy.V;
  I0Dim_1  = (int) ihb[0].Dim[1];
  I0Dim_2  = (int) ihb[0].Dim[2];

  I1Ave    = pcb->ib[1].Ave.V;
  I1Data   = ihb[1].Data;
  I1Dummy  = ihb[1].Dummy.V;
  I1DDummy = ihb[1].DDummy.V;
  I1Dim_1  = (int) ihb[1].Dim[1];
  I1Dim_2  = (int) ihb[1].Dim[2];

  GetImageRange         ( pcb, ihb, 1, imax-1,
                          Off_1, Off_2, Ps_1,  Ps_2,
                          f1_1, f3_1, Df_1, f1_2, f3_2, Df_2,
                          &Imin_1, &Imax_1, &Imin_2, &Imax_2,
                          &Wmin_1, &Wmax_1, &DW_1, &Wmin_2, &Wmax_2, &DW_2, 1);

  /* --- STATISTICS */
    /* Edge coordinates */
    Emin_1 = WORLD(INDEXSTART+LOWERBORDER,Off_1[0],Ps_1[0]);
    Emin_2 = WORLD(INDEXSTART+LOWERBORDER,Off_2[0],Ps_2[0]);
    Emax_1 = WORLD(INDEXSTART+I0Dim_1+LOWERBORDER,Off_1[0],Ps_1[0]);
    Emax_2 = WORLD(INDEXSTART+I0Dim_2+LOWERBORDER,Off_2[0],Ps_2[0]);

    SZ    = 0.0; SZ2   = 0.0; 
    SW1Z  = 0.0; SW2Z  = 0.0;
    SSW1Z = 0.0; SSW2Z = 0.0;
    GC1   = 0.0; GC2   = 0.0;
    GD1   = 0.0; GD2   = 0.0;

    NbGoodZ = 0;

  /* copy input image to output image */
  W_2 = Wmin_2; for (i=0;i<imax;i++) f_2[i]=f1_2[i];
  for (i_2=Imin_2;i_2<=Imax_2;i_2++) {
 
    pI0Data = ABSPTR(I0Data,I0Dim_1,I0Dim_2,Imin_1,i_2);
    W_1 = Wmin_1; for (i=0;i<imax;i++) f_1[i]=f1_1[i];
    for (i_1=Imin_1;i_1<=Imax_1;i_1++) {
 
      if (Isum2ldw (I1Data,I1Dim_1,I1Dim_2,I1Dummy,I1DDummy,
        f_1[1], f_2[1], f_1[1]+Df_1[1], f_2[1]+Df_2[1], &I1Sum, &I1Weight)) {
 
        I1Value = I1Sum; if (I1Ave) I1Value /= I1Weight;

        *pI0Data = I1Value;

        /* then do something with the data */
        DValue = (double) I1Value;
        SZ += DValue;
        SZ2 += DValue*DValue;
        NbGoodZ++; 
 
      } /* if (Isum2ldw ... */
 
      pI0Data++;
      W_1+=DW_1; for (i=0;i<imax;i++) { f_1[i]+=Df_1[i]; }
    } /* for i_1 ... */
 
    W_2+=DW_2; for (i=0;i<imax;i++) { f_2[i]+=Df_2[i]; }
  } /* for i_2 ... */

  /* all data points are valid, if no output dummy is defined */
  NbPixZ=I0Dim_1*I0Dim_2;
  if (NODUMMYDEFINED( I0Dummy, I0DDummy )) NbGoodZ = NbPixZ;  
  NbDumZ = NbPixZ - NbGoodZ;

  if (NbGoodZ>0) { 
    MeanZ  = SZ  / (double) NbGoodZ;  
    MeanZ2 = SZ2 / (double) NbGoodZ; 
    AbsMeanZ = fabs(MeanZ);
    AbsMeanZ2 = fabs(MeanZ2);
    ZI = TRUE;
    } else AbsMeanZ = 0.0; 

   if (pcb->TestBit) {
     printf("[Emin_1..Emax_1]       = [%g..%g]\n",Emin_1,Emax_1);
     printf("[Emin_2..Emax_2]       = [%g..%g]\n",Emin_2,Emax_2);
     printf("Number of pixels       = %d\n",NbPixZ);
     printf("Number of valid pixels = %d\n",NbGoodZ);
     printf("Number of dummies      = %d\n",NbDumZ);
     if (ZI) {
       printf("Mean value             = %g\n",MeanZ);
       printf("Mean square value      = %g\n",MeanZ2);
       } else  printf("Mean values            = undefined\n");
     }

  /* calculate the statistic for the WHOLE output image if it
     contains valid data */

  if (NbGoodZ>0) {
   if (AbsMeanZ>eps) { 
     /* normalize array with AbsMeanZ */
     MaxZ = SIGN(MeanZ); MinZ = SIGN(MeanZ); MaxAbsZ = 1.0; MinAbsZ = 1.0;
   } else {
     /* no normalization of data with AbsMeanZ */
     MaxZ = MeanZ; MinZ = MeanZ; MaxAbsZ = AbsMeanZ; MinAbsZ = AbsMeanZ;
     AbsMeanZ = 1.0;
   }

    Wmin_1 = WORLD(INDEXSTART,Off_1[0],Ps_1[0]);   
    Wmin_2 = WORLD(INDEXSTART,Off_2[0],Ps_2[0]);  
    Wmax_1 = WORLD(INDEXSTART+I0Dim_1-1,Off_1[0],Ps_1[0]);
    Wmax_2 = WORLD(INDEXSTART+I0Dim_2-1,Off_2[0],Ps_2[0]);
    pI0Data = ABSPTR(I0Data,I0Dim_1,I0Dim_2,INDEXSTART,INDEXSTART);

    /* loop over the output array and calculation of the statistics */
    W_2 = Wmin_2;
    for (i_2=INDEXSTART;i_2<I0Dim_2;i_2++) {
      W_1 = Wmin_1;
      for (i_1=INDEXSTART;i_1<I0Dim_1;i_1++) {
          Value = *pI0Data++; 
          if (NODUMMY( Value, I0Dummy, I0DDummy )) {
            Value /= AbsMeanZ;
            if (Value>MaxZ) MaxZ = Value;
            if (Value<MinZ) MinZ = Value;
            if (fabs(Value)<MinAbsZ) MinAbsZ = (double) fabs(Value);
            if (fabs(Value)>MaxAbsZ) MaxAbsZ = (double) fabs(Value);
                        
            SW1Z+=W_1*Value;
            SW2Z+=W_2*Value;
            SSW1Z+=W_1*W_1*Value;
            SSW2Z+=W_2*W_2*Value;
            }
          W_1+=DW_1; 
          } /* for i_1 ... */
        W_2+=DW_2; 
      } /* for i_2 ... */

    if (pcb->TestBit) {
      printf(" AbsMeanZ         = %g\n",AbsMeanZ);
      printf(" MaxZ/AbsMeanZ    = %g\n",MaxZ);
      printf(" MinZ/AbsMeanZ    = %g\n",MinZ);
      printf(" MaxAbsZ/AbsMeanZ = %g\n",MaxAbsZ);
      printf(" MinAbsZ/AbsMeanZ = %g\n",MinAbsZ);
    }
   
    SW1Z    *= AbsMeanZ;
    SW2Z    *= AbsMeanZ;
    SSW1Z   *= AbsMeanZ;
    SSW2Z   *= AbsMeanZ;
    MaxZ    *= AbsMeanZ;
    MinZ    *= AbsMeanZ;
    MaxAbsZ *= AbsMeanZ;
    MinAbsZ *= AbsMeanZ;
    
    /* Calculate center of mass values */
    if (fabs(SZ)>eps) {
      GC1 = SW1Z/SZ;
      GC2 = SW2Z/SZ;
      GD1 = SSW1Z/SZ-GC1*GC1;
      GD2 = SSW2Z/SZ-GC2*GC2;
      GI  = TRUE;
      }

  } /* if (NbGoodZ>0) */

  /*--- Get selection */
  sprintf(Sel,"%s",Select.V);

  /* --- Show results  */
  /*  The results are shown if only one single image is analyzed */
  if ((pcb->ib[1].First.V==pcb->ib[1].Last.V) || (pcb->ib[1].Inc.V==0))
    show = TRUE; else show = FALSE;
  if (Show.I) show = Show.V;  

  if (show) {
    printf("\n");
    printf("  reference system           = %s\n", reftostr(pcb->RSys.V));
    if ((strcmp(Sel,KeyiNum)==0) || (strcmp(Sel,SelectAll)==0))
               printf("  image number               = %d\n",ihb[0].ImNum);
    if ((strcmp(Sel,KeyDim1)==0) || (strcmp(Sel,SelectAll)==0))
               printf("  dimension 1                = %d\n", ihb[0].Dim[1]);
    if ((strcmp(Sel,KeyDim2)==0) || (strcmp(Sel,SelectAll)==0))
               printf("  dimension 2                = %d\n", ihb[0].Dim[2]);
    if ((strcmp(Sel,KeyMin1)==0) || (strcmp(Sel,SelectAll)==0))
               printf("  inferior edge coordinate 1 = %g\n", Emin_1);
    if ((strcmp(Sel,KeyMax1)==0) || (strcmp(Sel,SelectAll)==0))
               printf("  superior edge coordinate 1 = %g\n", Emax_1);
    if ((strcmp(Sel,KeyMin2)==0) || (strcmp(Sel,SelectAll)==0))
               printf("  inferior edge coordinate 2 = %g\n", Emin_2);
    if ((strcmp(Sel,KeyMax2)==0) || (strcmp(Sel,SelectAll)==0))
               printf("  superior edge coordinate 2 = %g\n", Emax_2);
    if ((strcmp(Sel,KeynTot)==0) || (strcmp(Sel,SelectAll)==0))
               printf("  pixels                     = %d\n", NbPixZ);
    if ((strcmp(Sel,KeynVal)==0) || (strcmp(Sel,SelectAll)==0))
               printf("  valid pixels               = %d\n", NbGoodZ);
    if ((strcmp(Sel,KeynDum)==0) || (strcmp(Sel,SelectAll)==0))
               printf("  dummy pixels               = %d\n", NbDumZ);
    if ((strcmp(Sel,KeyMax)==0)  || (strcmp(Sel,SelectAll)==0))
       if (ZI) printf("  maximum                    = %g\n", MaxZ);
          else printf("  maximum                    = undefined\n");
    if ((strcmp(Sel,KeyMin)==0)  || (strcmp(Sel,SelectAll)==0))
       if (ZI) printf("  minimum                    = %g\n", MinZ);
          else printf("  minimum                    = undefined\n");
    if ((strcmp(Sel,KeyaMax)==0) || (strcmp(Sel,SelectAll)==0))
       if (ZI) printf("  max abs value              = %g\n", MaxAbsZ);
          else printf("  max abs value              = undefined\n");
    if ((strcmp(Sel,KeyaMin)==0) || (strcmp(Sel,SelectAll)==0))
       if (ZI) printf("  min abs value              = %g\n", MinAbsZ);
          else printf("  min abs value              = undefined\n");
    if ((strcmp(Sel,KeyDum)==0)  || (strcmp(Sel,SelectAll)==0))
               printf("  dummy value                = %g\n", I0Dummy);
    if ((strcmp(Sel,KeySum)==0)  || (strcmp(Sel,SelectAll)==0))
       if (ZI) printf("  sum                        = %g\n", SZ);
          else printf("  sum                        = undefined\n");
    if ((strcmp(Sel,KeysSum)==0) || (strcmp(Sel,SelectAll)==0))
       if (ZI) printf("  sum square                 = %g\n", SZ2);
          else printf("  sum square                 = undefined\n");
    if ((strcmp(Sel,KeyMean)==0) || (strcmp(Sel,SelectAll)==0))
       if (ZI) printf("  mean                       = %g\n", MeanZ);
          else printf("  mean                       = undefined\n");
    if ((strcmp(Sel,KeysMean)==0)|| (strcmp(Sel,SelectAll)==0))
       if (ZI) printf("  mean sqare                 = %g\n", MeanZ2);
          else printf("  mean sqare                 = undefined\n");
    if ((strcmp(Sel,KeyRMS)==0)  || (strcmp(Sel,SelectAll)==0))
       if (ZI) printf("  root mean square           = %g\n",sqrt(MeanZ2));
          else printf("  root mean square           = undefined\n");
    if ((strcmp(Sel,KeyDevi)==0) || (strcmp(Sel,SelectAll)==0))
       if (ZI) printf("  standard deviation         = %g\n",
                 (float)sqrt(MeanZ2-MeanZ*MeanZ));
          else printf("  standard deviation         = undefined\n");
    if ((strcmp(Sel,KeyGC1)==0)  || (strcmp(Sel,SelectAll)==0))
       if (GI) printf("  center of mass 1 (COM 1)   = %g\n" ,GC1);
          else printf("  center of mass 1 (COM 1)   = undefined\n");
    if ((strcmp(Sel,KeyGC2)==0)  || (strcmp(Sel,SelectAll)==0))
       if (GI) printf("  center of mass 2 (COM 2)   = %g\n" ,GC2);
          else printf("  center of mass 2 (COM 2)   = undefined\n");
    if ((strcmp(Sel,KeyGD1)==0)  || (strcmp(Sel,SelectAll)==0))
       if (GI) printf("  standard dev. of COM 1     = %g\n", (float) sqrt(GD1));
          else printf("  standard dev. of COM 1     = undefined\n");
    if ((strcmp(Sel,KeyGD2)==0)  || (strcmp(Sel,SelectAll)==0))
       if (GI) printf("  standard dev. of COM 2     = %g\n", (float) sqrt(GD2));
          else printf("  standard dev. of COM 2     = undefined\n");

    if (KeyList.I) {
      parno = 1;
      while (strlib_param ( Key, IO_size, KeyList.V, ',', parno++ )) {
        if (strlen(Key)) {
          ReadHeaderLine          ( pcb, 1, num, ihb, Key, KeyValue, pstatus );
          if (*pstatus) printf("%24s     = %s\n",Key,"(not found)");
          else printf("%24s     = %s\n",Key,KeyValue);
        }
      }
    }

    printf("\n");
  } /* if (show) */

  /*--- Get output filename */
  if (!FName.I) {
    sprintf(FName.V,"%s_%s%s",Prefix.V,Select.V,Extension.V);
  }

  /* Generate filename of ASCII file */
  filename_parameter (fname_full,IO_size,FName.V,0);
  filename_pattern (fname_full,IO_size,fname_full,(pcb->ib)[0].FileNumber);
  filename_full (fname_full,IO_size,fname_full);

  /*--- Open output file
    If this is the first image of the sequence, an existing file
    with the same name must be deleted (open mode = "w"),
    in all other cases it must be open to append a line (open mode "a"). */

  if (pcb->TestBit) printf("Opening output file %s\n",fname_full);
  *pstatus = FileOpeningError;
  if (ImageIsFirst( pcb, 0, ihb )) {
    /* 1st image: new write */
    if ((OutputFileId=fopen(fname_full,"wb")) == NULL) return;
    } else {
    /* appending */
    if ((OutputFileId=fopen(fname_full,"ab")) == NULL) return;
    }

  /*--- Write to the output file
    If this is the first image of the sequence, a header line is written */

  if (ImageIsFirst( pcb, 0, ihb )) {
    /* 1st image: write header line */

    printf("Writing to file   : %s, image %d\n", fname_full,ihb[0].ImNum);


    if (Spec.V) {
      write_spechead( OutputFileId, pcb, num, ihb, pstatus );
      if (*pstatus!=Success) return;
    }

    fprintf(OutputFileId,"# Input file       : %s\r\n",pcb->ib[1].FileName);
    fprintf(OutputFileId,"# Reference system : %s\r\n", reftostr(pcb->RSys.V));
    if (Hist.V) {
      WriteHistory( OutputFileId, pcb, num, ihb, pstatus );
      if (*pstatus!=Success) return;
    }

    // count key list values 
    cols=0;
    if (KeyList.I) {
      parno = 1;
      while (strlib_param ( Key, IO_size, KeyList.V, ',', parno++ )) {
        if (strlen(Key)) cols++;
      }
    }

    // Write all labels
    if (strcmp(Sel,SelectAll)==0) {
      cols+=24;
      if (Spec.V) {
        fprintf(OutputFileId,"#N %d \r\n",cols);
        fprintf(OutputFileId,"#L %s",KeyiNum);
      } else fprintf(OutputFileId,"# %s",KeyiNum); // 1

      fprintf(OutputFileId,"%s%s",Spacer.V,KeyDim1); // 2
      fprintf(OutputFileId,"%s%s",Spacer.V,KeyDim2); // 3
      fprintf(OutputFileId,"%s%s",Spacer.V,KeyMin1); // 4
      fprintf(OutputFileId,"%s%s",Spacer.V,KeyMax1); // 5
      fprintf(OutputFileId,"%s%s",Spacer.V,KeyMin2); // 6
      fprintf(OutputFileId,"%s%s",Spacer.V,KeyMax2); // 7
      fprintf(OutputFileId,"%s%s",Spacer.V,KeynTot); // 8
      fprintf(OutputFileId,"%s%s",Spacer.V,KeynVal); // 9
      fprintf(OutputFileId,"%s%s",Spacer.V,KeynDum); // 10
      fprintf(OutputFileId,"%s%s",Spacer.V,KeyMax); // 11
      fprintf(OutputFileId,"%s%s",Spacer.V,KeyMin); // 12
      fprintf(OutputFileId,"%s%s",Spacer.V,KeyaMax); // 13
      fprintf(OutputFileId,"%s%s",Spacer.V,KeyaMin); // 14
      fprintf(OutputFileId,"%s%s",Spacer.V,KeyDum); // 15
      fprintf(OutputFileId,"%s%s",Spacer.V,KeySum); // 16
      fprintf(OutputFileId,"%s%s",Spacer.V,KeysSum); // 17
      fprintf(OutputFileId,"%s%s",Spacer.V,KeyMean); // 18
      fprintf(OutputFileId,"%s%s",Spacer.V,KeysMean); // 19
      fprintf(OutputFileId,"%s%s",Spacer.V,KeyRMS); // 20
      fprintf(OutputFileId,"%s%s",Spacer.V,KeyDevi); // 21
      fprintf(OutputFileId,"%s%s",Spacer.V,KeyGC1); // 22
      fprintf(OutputFileId,"%s%s",Spacer.V,KeyGC2); // 23
      fprintf(OutputFileId,"%s%s",Spacer.V,KeyGD1); // 24
      fprintf(OutputFileId,"%s%s",Spacer.V,KeyGD2); // 25
/*++++++++++++
%s\t%s\t%s\t%s\t%s\t%s\t%s\t%s\t\
%s\t%s\t%s\t%s\t%s\t%s\t\%s\t%s\t%s\t\
%s\t%s\t%s\t%s\t%s\t%s\t%s\t%s",
        KeyiNum,KeyDim1,KeyDim2,KeyMin1,KeyMax1,KeyMin2,KeyMax2,KeynTot,
        KeynVal,KeynDum,KeyMax,KeyMin,KeyaMax,KeyaMin,KeyDum,KeySum,KeysSum,
        KeyMean,KeysMean,KeyRMS,KeyDevi,KeyGC1,KeyGC2,KeyGD1,KeyGD2);
+++++++++++ */
    } else { 
      cols+=2;
      if (Spec.V) {
        fprintf(OutputFileId,"#N %d \r\n",cols);
        fprintf(OutputFileId,"#L %s",KImage);
      } else fprintf(OutputFileId,"# %s",KImage); // 1
      fprintf(OutputFileId,"%s%s",Spacer.V, Sel); // 2
    }

    if (KeyList.I) {
      parno = 1;
      while (strlib_param ( Key, IO_size, KeyList.V, ',', parno++ )) {
        if (strlen(Key)) fprintf(OutputFileId,"%s%s",Spacer.V, Key);
      }
    }

    fprintf(OutputFileId,"\r\n");

  } else printf("Appending to file : %s, image %d\n",
                 fname_full,ihb[1].ImNum);

  /*   Write statistics to the output file */

    fprintf(OutputFileId, "%d",ihb[0].ImNum);
  if ((strcmp(Sel,KeyDim1)==0) || (strcmp(Sel,SelectAll)==0))
    fprintf(OutputFileId, "%s%d", Spacer.V, ihb[0].Dim[1]);
  if ((strcmp(Sel,KeyDim2)==0) || (strcmp(Sel,SelectAll)==0))
    fprintf(OutputFileId, "%s%d", Spacer.V, ihb[0].Dim[2]);
  if ((strcmp(Sel,KeyMin1)==0) || (strcmp(Sel,SelectAll)==0))
    fprintf(OutputFileId, "%s%g", Spacer.V, Emin_1);
  if ((strcmp(Sel,KeyMax1)==0) || (strcmp(Sel,SelectAll)==0))
    fprintf(OutputFileId, "%s%g", Spacer.V, Emax_1);
  if ((strcmp(Sel,KeyMin2)==0) || (strcmp(Sel,SelectAll)==0))
    fprintf(OutputFileId, "%s%g", Spacer.V, Emin_2);
  if ((strcmp(Sel,KeyMax2)==0) || (strcmp(Sel,SelectAll)==0))
    fprintf(OutputFileId, "%s%g", Spacer.V, Emax_2);
  if ((strcmp(Sel,KeynTot)==0) || (strcmp(Sel,SelectAll)==0))
    fprintf(OutputFileId, "%s%d", Spacer.V, NbPixZ);
  if ((strcmp(Sel,KeynVal)==0) || (strcmp(Sel,SelectAll)==0))
    fprintf(OutputFileId, "%s%d", Spacer.V, NbGoodZ);
  if ((strcmp(Sel,KeynDum)==0) || (strcmp(Sel,SelectAll)==0))
    fprintf(OutputFileId, "%s%d", Spacer.V, NbDumZ);
  if ((strcmp(Sel,KeyMax)==0)  || (strcmp(Sel,SelectAll)==0))
    if (ZI) fprintf(OutputFileId, "%s%g", Spacer.V, MaxZ);
       else fprintf(OutputFileId, "%s*", Spacer.V);
  if ((strcmp(Sel,KeyMin)==0)  || (strcmp(Sel,SelectAll)==0))
    if (ZI) fprintf(OutputFileId, "%s%g", Spacer.V, MinZ);
       else fprintf(OutputFileId, "%s*", Spacer.V);
  if ((strcmp(Sel,KeyaMax)==0) || (strcmp(Sel,SelectAll)==0))
    if (ZI) fprintf(OutputFileId, "%s%g", Spacer.V,MaxAbsZ);
       else fprintf(OutputFileId, "%s*", Spacer.V);
  if ((strcmp(Sel,KeyaMin)==0) || (strcmp(Sel,SelectAll)==0))
    if (ZI) fprintf(OutputFileId, "%s%g", Spacer.V,MinAbsZ);
       else fprintf(OutputFileId, "%s*", Spacer.V);
  if ((strcmp(Sel,KeyDum)==0)  || (strcmp(Sel,SelectAll)==0))
    fprintf(OutputFileId, "%s%g", Spacer.V, I0Dummy);
  if ((strcmp(Sel,KeySum)==0)  || (strcmp(Sel,SelectAll)==0))
    if (ZI) fprintf(OutputFileId, "%s%g", Spacer.V,SZ);
       else fprintf(OutputFileId, "%s*", Spacer.V);
  if ((strcmp(Sel,KeysSum)==0) || (strcmp(Sel,SelectAll)==0))
    if (ZI) fprintf(OutputFileId, "%s%g", Spacer.V,SZ2);
        else fprintf(OutputFileId, "%s*", Spacer.V);
  if ((strcmp(Sel,KeyMean)==0) || (strcmp(Sel,SelectAll)==0))
    if (ZI) fprintf(OutputFileId, "%s%g", Spacer.V,MeanZ);
        else fprintf(OutputFileId, "%s*", Spacer.V);
  if ((strcmp(Sel,KeysMean)==0)|| (strcmp(Sel,SelectAll)==0))
    if (ZI) fprintf(OutputFileId, "%s%g", Spacer.V,MeanZ2);
        else fprintf(OutputFileId, "%s*", Spacer.V);
  if ((strcmp(Sel,KeyRMS)==0)  || (strcmp(Sel,SelectAll)==0))
    if (ZI) fprintf(OutputFileId, "%s%g", Spacer.V, (float)sqrt(MeanZ2));
        else fprintf(OutputFileId, "%s*", Spacer.V);
  if ((strcmp(Sel,KeyDevi)==0) || (strcmp(Sel,SelectAll)==0))
    if (ZI) fprintf(OutputFileId, "%s%g", Spacer.V, (float)sqrt(MeanZ2-MeanZ*MeanZ));
        else fprintf(OutputFileId, "%s*", Spacer.V);
  if ((strcmp(Sel,KeyGC1)==0)  || (strcmp(Sel,SelectAll)==0))
    if (GI) fprintf(OutputFileId, "%s%g", Spacer.V,GC1);
        else fprintf(OutputFileId, "%s*", Spacer.V);
  if ((strcmp(Sel,KeyGC2)==0)  || (strcmp(Sel,SelectAll)==0))
    if (GI) fprintf(OutputFileId, "%s%g", Spacer.V,GC2);
        else fprintf(OutputFileId, "%s*", Spacer.V);
  if ((strcmp(Sel,KeyGD1)==0)  || (strcmp(Sel,SelectAll)==0))
    if (GI) fprintf(OutputFileId, "%s%g", Spacer.V,(float) sqrt(GD1));
        else fprintf(OutputFileId, "%s*", Spacer.V);
  if ((strcmp(Sel,KeyGD2)==0)  || (strcmp(Sel,SelectAll)==0))
    if (GI) fprintf(OutputFileId, "%s%g", Spacer.V,(float) sqrt(GD2));
        else fprintf(OutputFileId, "%s*", Spacer.V);

  if (KeyList.I) {
    parno = 1;
    while (strlib_param ( Key, IO_size, KeyList.V, ',', parno++ )) {
      if (strlen(Key)) {
        ReadHeaderLine          ( pcb, 1, num, ihb, Key, KeyValue, pstatus );
        if (*pstatus) fprintf(OutputFileId,"%s%s", Spacer.V,"*");
        else fprintf(OutputFileId,"%s%s", Spacer.V,KeyValue);
      }
    }
  }

  fprintf(OutputFileId,"\r\n");

  /*--- Close output file */

  if (pcb->TestBit) printf("Closing file %s\n",fname_full);
  *pstatus = FileClosingError;
  if (fclose(OutputFileId)!=0) return;

  *pstatus = Success;

} /* saxs_stat*/

/*---------------------------------------------------------------------------
user_io
Do all the keyboard io and return cb, and ib
---------------------------------------------------------------------------*/

void user_io(CmdBlk * pcb, ImgBlk * ib, int * pstatus)
{
  char  progfn[IO_size];
  ImgHeadBlk ihb[BlockNum];

  float ROff_1, RPs_1, UOff_1, UPs_1;
  float ROff_2, RPs_2, UOff_2, UPs_2;

  IO_line *pSelect, *pPrefix;

 /* Determine program name without directory */
   (void) RightFR((char *) pcb->argv[0],DirSeparator,progfn,
           IO_size);

  /* show usage if no arguments are given */
  if (pcb->argc<=1) printf("Usage: %s %s\n",progfn,Usage);

  /*--- Write name of program ---*/
  printf("\n");
  printf("   %s %s\n",progfn,Version);
  printf("\n");

  SaxsScalArgvFilenames ( pcb, ib, ihb, 0, BlockNum-1, pstatus);
  if (*pstatus!=Success) return;
  GetReference(pcb->RSys.V,1,ihb,&ROff_1,&ROff_2,&RPs_1,&RPs_2,pstatus );
  if (*pstatus!=Success) return;
  GetReference(pcb->USys.V,1,ihb,&UOff_1,&UOff_2,&UPs_1,&UPs_2,pstatus );
  if (*pstatus!=Success) return;

  /*--- Argument  : selection */
  pSelect = (IO_line*) option_parameter_search( SSelect, pcb, pstatus );
  if (*pstatus!=Success) return;
  argv_line(pcb,"selection", pSelect,pSelect->V,pstatus);
  if (*pstatus!=Success) return;

  pPrefix = (IO_line*) option_parameter_search( SPrefix, pcb, pstatus );
  if (*pstatus!=Success) return;

  printf("\n");
  if (ib[1].Name.I) printf("i/p file            : %s\n",ib[1].Name.V);
  if (pPrefix->I)   printf("output file prefix  : %s\n",pPrefix->V);
  if (ib[1].First.I)printf("first image         : %d\n",ib[1].First.V);
  if (ib[1].Last.I) printf("last image          : %d\n",ib[1].Last.V);
  if (ib[1].Inc.I)  printf("increment           : %d\n",ib[1].Inc.V);
  if (pSelect->I)   printf("selected statistics : %s\n",pSelect->V);
  printf("\n");

  if (pcb->TestBit) {
    PrintBlocks ( pcb, ib );
    printf("\n"); }

  return;
} /* user_io */

/*---------------------------------------------------------------------------
main
---------------------------------------------------------------------------*/

#if MAKE_FUNCTION
# define MAIN main_saxs_stat
#else
# define MAIN main
#endif

int MAIN (int argc, char *argv[])
{
  CmdBlk cb;                /* command block  */
  ImgBlk ib[BlockNum];      /* image blocks */

  int status;
  int arg_no = 0;

 /* Init options, control block and image blocks */
  InitOptions( Usage, Version, NULL, TRUE, &cb, ib, BlockNum );

  option_define ( IO_tpline, SSelect,Default_select, &cb );
  option_define ( IO_tpline, SKey,"", &cb );
  option_define ( IO_tpline, SPrefix,"input", &cb );
  option_define ( IO_tpline, SExtension, ".txt", &cb );
  option_define ( IO_tpline, SFName,"output.txt", &cb );
  option_define ( IO_tpline, SSpacer, "  \t", &cb ); /* default spc,spc,tab */
  option_define ( IO_tpflag, SShow,"FALSE", &cb );
  option_define ( IO_tpflag, SHist, "TRUE", &cb );
  option_define ( IO_tpflag, SSpec, "TRUE", &cb );

 /* Read options from argument list */
  ReadOptions( argv, &arg_no, &cb, ib, &status);

  /* USER KEYBOARD I/O */
  if (status==Success) {
    argv_start ( &cb, 1 );
    user_io( &cb, ib, &status);
    argv_end( &cb ); /* must be called after user_io */
  }

  /* SEQUENCE CALCULATION */
  if (status==Success) 
    IMAGELOOP( &cb, ib, saxs_stat, NULL, NULL, TRUE, &status );

  return( ReportSaxsStatus( status, 0 ) );

} /* MAIN */
