/***************************************************************************/
/* Written 1994++ by Peter Boesecke                                        */
/* Copyright (C) 2011 European Synchrotron Radiation Facility              */
/*                       Grenoble, France                                  */
/*                                                                         */
/*    Principal authors: Peter Boesecke  (boesecke@esrf.eu)                */
/*                                                                         */
/*    This program is free software: you can redistribute it and/or modify */
/*    it under the terms of the GNU General Public License as published by */
/*    the Free Software Foundation, either version 3 of the License, or    */
/*    (at your option) any later version.                                  */
/*                                                                         */
/*    This program is distributed in the hope that it will be useful,      */
/*    but WITHOUT ANY WARRANTY; without even the implied warranty of       */
/*    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the        */
/*    GNU General Public License for more details.                         */
/*                                                                         */
/*    You should have received a copy of the GNU General Public License    */
/*    along with this program.  If not, see <http://www.gnu.org/licenses/>.*/
/***************************************************************************/
/*+++
1 saxs_rot
  Rotation of an image sequence

2 PURPOSE
  Rotation of an image sequence

  Arguments:
  saxs_rot [options] <i1nam> <onam> <i1fst> <i1lst> <i1inc> 
             <odum> <odim1> <odim2> <alfa[0.0_deg]>

  Defaults:
  <input file name>   : input.edf 
  <output file name>  : output.edf
  <first image>       : <first image number in input file>
  <last image>                              : <last image number in input file>
     if argument list ends with first image : <first image>
  <increment>         : 1
  <dummy>             : <dummy value in first image of input file>
  <dimension 1>       : <horizontal dimension of first image in input file>
  <dimension 2>       : <vertical dimension of first image in input file>
  <alfa>              : 0.0_deg

  Wildcard:
  A "=" in the argument list chooses the default.

2 HISTORY
  1999-12-31  V3.0 PB from saxs_aff V3.0
  2000-08-04  PB %f->%g
  2001-07-09  PB V4.00 new loop, no repetition
  2003-02-09  PB V4.01 line args for Dummy, DDummy, Dim
  2003-05-30  PB V4.02 ImageLoop: Function call changed
  2003-08-14  PB V4.03 option_define, all option numbers removed
  2004-10-02  PB V4.03 cosmetics
  2004-10-14  PB V4.04 alfa in radian or with '_' and unit
  2004-10-31  PB V4.05 MAKE_FUNCTION
  2005-07-15  PB V4.06 ReportSaxsStatus parameter list updated
  2005-08-30  PB V4.07 VarDat, call to Affine updated
  2007-06-11  PB V4.08 ImageLoop parameters updated
  2012-01-31  PB V4.09 ImageLoop parameters updated

---*/
# define Version  "saxs_rot V4.09 2012-01-31, Peter Boesecke"

# include <errno.h>
# include <stdio.h>
# include <unistd.h>
# include <fcntl.h>

# include "SaxsPrograms.h"

# define Usage "[options] \n\
                <i1nam> <onam> <i1fst> <i1lst> <i1inc> \n\
                <odum> <odim1> <odim2> <alfa[0.0_deg]>\n\
            ---   Rotation of an image sequence by alfa --- "

# define BlockNum 2       /* 1 input sequence + 1 output sequence */

/* Float Options */
# define SAlpha "alfa"

/*---------------------------------------------------------------------------
1 saxs_rot

2 PURPOSE
  Multiplicates an image with a factor and adds a constant
---------------------------------------------------------------------------*/
void saxs_rot (CmdBlk * pcb, long num[], ImgHeadBlk ihb[], int * pstatus)
{ int i,j,k;
  int imax = pcb->ImgBlkLen;
  double t[3][2];
  float alpha_rad;
  IO_float Alpha;

  *pstatus = Success;

  printf("\n Calculating ihb[0,% d] = Function(",ihb[0].ImNum);
  for(i=1;i<imax;i++) printf("ihb[% d,% d] ", i, ihb[i].ImNum); 
  printf(")\n\n");

 /* Check the number of images */
  if (imax!=2) {
     printf("%d images found, 1 input and 1 output image required\n", imax);
     *pstatus=Failed; return; }

 /* Get option values */
  Alpha = option_float ( SAlpha, pcb, ihb, num, 1, pstatus );
  if (*pstatus!=Success) return;

  /* generate transformation matrix */
  alpha_rad  = Alpha.V;
  t[0][0]    = (double) cos(alpha_rad);
  t[1][0]    = (double) -sin(alpha_rad);
  t[2][0]    = (double) 0.0;
  t[0][1]    = (double) sin(alpha_rad);
  t[1][1]    = (double) cos(alpha_rad);
  t[2][1]    = (double) 0.0;

  if (pcb->TestBit) {
    for (k=0;k<2;k++) for (j=0;j<3;j++)
        printf("t[%d][%d] = %g\n",j,k,t[j][k]);
    }

  Affine (  pcb, ihb, 1, 0, t, 0, pstatus );
  if (*pstatus!=Success) return;

 } /* saxs_rot*/

/*---------------------------------------------------------------------------
user_io
Do all the keyboard io and return cb, and ib
---------------------------------------------------------------------------*/

void user_io(CmdBlk * pcb, ImgBlk * ib, int * pstatus)
{
  char  progfn[IO_size];
  ImgHeadBlk ihb[BlockNum];

  float ROff_1, RPs_1, UOff_1, UPs_1;
  float ROff_2, RPs_2, UOff_2, UPs_2;

  IO_flexp *pAlpha;

 /* Determine program name without directory */
   (void) RightFR((char *) pcb->argv[0],DirSeparator,progfn,IO_size);

  /* show usage if no arguments are given */
  if (pcb->argc<=1) printf("Usage: %s %s\n",progfn,Usage);

  /*--- Write name of program ---*/
  printf("\n");
  printf("   %s %s\n",progfn,Version);
  printf("\n");

  ArgvFilenames ( pcb, ib, ihb, 0, BlockNum-1, pstatus);
  if (*pstatus!=Success) return;
  GetReference(pcb->RSys.V,1,ihb,&ROff_1,&ROff_2,&RPs_1,&RPs_2,pstatus );
  if (*pstatus!=Success) return;
  GetReference(pcb->USys.V,1,ihb,&UOff_1,&UOff_2,&UPs_1,&UPs_2,pstatus );
  if (*pstatus!=Success) return;

  pAlpha = (IO_flexp*) option_parameter_search( SAlpha, pcb, pstatus );
  if (*pstatus!=Success) return;
  argv_flexp(pcb,"Rotation angle Alpha (ccw)", 
                       pAlpha,pAlpha->V,pstatus);
  if (*pstatus!=Success) return;

  printf("\n");
  if (ib[1].Name.I)    printf("i/p file           : %s\n",ib[1].Name.V);
  if (ib[0].Name.I)    printf("o/p file           : %s\n",ib[0].Name.V);
  if (ib[1].First.I)   printf("first image        : %d\n",ib[1].First.V);
  if (ib[1].Last.I)    printf("last image         : %d\n",ib[1].Last.V);
  if (ib[1].Inc.I)     printf("increment          : %d\n",ib[1].Inc.V);
  if (ib[0].Dummy.I)   printf("output dummy       : %s\n",ib[0].Dummy.V);
  if (ib[0].Dim[1].I)  printf("output dimension 1 : %s\n",ib[0].Dim[1].V);
  if (ib[0].Dim[2].I)  printf("output dimension 2 : %s\n",ib[0].Dim[2].V);
  if (pAlpha->I)       printf("Alpha (ccw)        : %s\n",pAlpha->V);
  printf("\n");

  if (pcb->TestBit) {
    PrintBlocks ( pcb, ib );
    printf("\n"); }

  return;
}

/*---------------------------------------------------------------------------
main
---------------------------------------------------------------------------*/

#if MAKE_FUNCTION
# define MAIN main_saxs_rot
#else
# define MAIN main
#endif

int MAIN (int argc, char *argv[])
{
  CmdBlk cb;                /* command block  */
  ImgBlk ib[BlockNum];      /* image blocks */

  int status;
  int arg_no = 0;

 /* Init options, control block and image blocks */
  InitOptions( Usage, Version, NULL, TRUE, &cb, ib, BlockNum );

 /* define special options */
  option_define ( IO_tpflexp, SAlpha, "0.0_deg", &cb );

 /* Read options from argument list */
  ReadOptions( argv, &arg_no, &cb, ib, &status);

  /* USER KEYBOARD I/O */
  if (status==Success) {
    argv_start ( &cb, 1 );
    user_io( &cb, ib, &status);
    argv_end( &cb ); /* must be called after user_io */
  }

  /* SEQUENCE CALCULATION */
  if (status==Success) 
    IMAGELOOP( &cb, ib, saxs_rot, NULL, NULL, TRUE, &status );

  return( ReportSaxsStatus( status, 0 ) );

} /* MAIN */

