/***************************************************************************/
/* Written 1994++ by Peter Boesecke                                        */
/* Copyright (C) 2011 European Synchrotron Radiation Facility              */
/*                       Grenoble, France                                  */
/*                                                                         */
/*    Principal authors: Peter Boesecke  (boesecke@esrf.eu)                */
/*                                                                         */
/*    This program is free software: you can redistribute it and/or modify */
/*    it under the terms of the GNU General Public License as published by */
/*    the Free Software Foundation, either version 3 of the License, or    */
/*    (at your option) any later version.                                  */
/*                                                                         */
/*    This program is distributed in the hope that it will be useful,      */
/*    but WITHOUT ANY WARRANTY; without even the implied warranty of       */
/*    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the        */
/*    GNU General Public License for more details.                         */
/*                                                                         */
/*    You should have received a copy of the GNU General Public License    */
/*    along with this program.  If not, see <http://www.gnu.org/licenses/>.*/
/***************************************************************************/
/*+++
1 saxs_normn
  Normalization of an image sequence

2 PURPOSE
  Normalization of an image sequence

  Calculation:       I0 = F(I1,factor,IF) = (I1/IF) * factor
  Error propagation: V0 = V1*(df/dI1)^2+VF*(df/dIF)^2
                        = (V1*(1/IF)^2+VF*((I1/IF)/IF)^2)*factor^2

  Defaults:
  <input file>                     : input.edf
  <scaler file>                    : <input file>
  <output file name>               : output.edf
  <first image in input file>      : <first image number in first input file>
  <last image in input file>       : <last image number in input file>
     if argument list ends with first image : <first image>
  <increment>                      : 1
  <first image in scaler file>     : 1
  <increment in scaler file>       : <increment of 1st input file>
  <output dummy>          : <dummy value in 1st chosen image of 1st input file>
  <output dimension 1>    : <horizontal dimension of first chosen image>
  <output dimension 2>    : <horizontal dimension of first chosen image>
  <transmission>                   : 1 (-trm)
  <scaler number for I0 monitor>   : <HSI0 value of 1st chosen image> (-mons)
  <scaler number for exposure time>: <HSTime value of 1st chosen image (-tims)
  <scaler number for anode counts> : <HSAnode value of 1st chosen image (-anos)
  <scaler for 2nd I0 monitor>      : <HSI0S value of 1st chosen image> (-mo2s)
  <scaler for 2nd exposure time>   : <HSTimeS value of 1st chosen image (-ti2s)
  <scaler for 2nd anode counts>    : <HSAnodeS value of 1st chosen image (-an2s)

  Wildcard:
  A "=" in the argument list chooses the default.

2 METHOD
  The normalization is done in the following way:

        Iin          (anoc*anof)           rd       rd^2        1      1
 Iout = --- * -------------------------- * --- * ----------- * --- * -----
        sum   (monc-monz*timc*timf)*monf   dis    pix1*pix2    trm   Iflat

       with rd^2 = dis^2 +
                     INDEX2R(i1,off1-cen1,pix1)^2 +
                     INDEX2R(i2,off2-cen2,pix2)^2

       with trm  = (trmc-trmz*timc*timf)*trmf / (monc-monz*timc*timf)*monf
        or  trm  = <option>

  sum    : sum of input image                        [event]
  timc   : time counts                               [ct]
  timf   : time factor                               [s]
  timv   : time value                                [s]
  anoc   : anode counts                              [ct]
  anof   : anode factor                              [event/ct]
  anoz   : anode zero                                [ct/s]
  monc   : monitor counts                            [ct]
  monf   : monitor factor                            [photon/ct]
  monz   : monitor zero                              [ct/s]
  monv   : monitor value                             [photon]
  trmc   : transmission monitor counts               [ct]
  trmf   : transmission monitor factor               [photon/ct]
  trmz   : transmission monitor zero                 [ct/s]
  trmv   : transmission monitor value                [photon]
  trm    : tranmission
  rd     : distance sample-pixel                     [m]
  dis    : distance sample-detector (SampleDistance) [m]
  cen1   : horizontal beam center                    [pixel]
  cen2   : vertical beam center                      [pixel]
  off1   : horizontal offset                         [pixel]
  off2   : vertical offset                           [pixel]
  pix1   : horizontal pixel size (PixSiz_1)          [m/pixel]
  pix2   : vertical pixel size (PixSiz_2)            [m/pixel]
  re     : classical electron radius (RELECTRON)   = 2.81794e-15 m

  Iin    : input image                               [event/pixel]
  Iflat  : flatfield image                           [event/photon]
  Iout   : output image = (1/Vol) * dSigma/dOmega    [1/m]
 
  In case that the -flat option is set the result is not divided by the
  flatfield image (default +flat).

  The values of timc, timf etc. are stored as keywords in the header
  of the raw data file.
  To convert the output image to absolute values it has to be devided
  by the flat field file [event/photon].

  The second scalers are only used if (count2*calib2>max_count*calib1)
  where max_count is defined as maxcount = 2^scalerdepth. If the number
  of the second scaler is set to 0 it will not be used.
  ATTENTION: Only the counts and the factors of the second scalers are 
  read. Only the zero values of the first scalers are read.  

2 HISTORY
  18-Aug-1995 PB      creation from saxs_normm program (PB)
  22-Aug-1995 PB      output is (1/V) * dSigma/dOmega [1/m]
  21-Sep-1995 PB      too long strings changed for SIG
  28-Oct-1995 PB      scalerdepth changed from integer to float
  07-Dec-1995 PB 3.0
  08-Dec-1995 PB      Calculation of accepted corrected (AnoCount changed 
                      to anoval)
  09-Dec-1995 PB      The binning correction factor cancels out. It is only
		      needed to calculate the accepted number of counts
  15-Dec-1995 PB 3.1  a) read KHScalerLen
  26-Dec-1995 PB 3.11 a) Missing \n\ in Usage added,
                      b) in user_io: Usage replaced by pcb->MainHelp
                                     Version replaced by pcb->MainVersion
                      c) RoutineSucceeded --> Success
  22-Apr-1996 PB 3.12 Title written before calculation
  18-May-1996 PB 3.2  Option rdis (relative distance, RelDis) added. 
                      If RelDis is set, the value of SampleDistance is 
                      ignored and replaced by the sum of detectorposition 
                      plus RelDis in input sequence 1 and 2 (i1 and i2) and 
                      in the output sequence (o). The value of
                      detectorposition is read from input sequence 2.
  02-Oct-1996 PB 3.3  Calculation and display of maximum pixel value
  20-Jul-1997 PB 3.4  Option +ccd: no dead time correction for ccd
  27-Apr-1998 PB 3.5  Option +tron: read transmission value from argument 
				    on/off (1: read from arguments and
				    use transmission 1 as default,
				    default 1)
  26-Jun-1999 PB DevSeparator, DirSeparator, ExtSeparator defined in SaxsInput
  29-Mar-2000 PB 3.6 AnoZero AnoZero2
  31-Mar-2000 PB 3.7 option -anoz <anode zero>
  2000-08-04  PB %f->%g
  2000-08-31  PB 3.8 flg option +/-scal, float option -monv, -trmv, anov, timv 
                    +scal: monval,trmval,timval,anoval are calculated from 
                           scalers counts
                    -scal: monval,trmval,timval,anoval are read from keywords
                           KValueI0, KValueI1, KValueAnode, KValueTime

                    Default is +scal. If the keyword KHScalerI0 is missing -scal
                    is used.

                    Values given with options (-monv, -trmv, -anov, -timv)
                    overwrite scaler and keyword values
                    -tron is default (instead of +tron)
                    -flat is default (instead of +flat)
  2000-11-03  PB 3.9 initialization of monname, timname etc.
  2000-11-06  PB 3.10 mons is no longer overwritten by scaleron test
                      trmv sets value of transmitted photons
  2001-05-22  PB 3.11 rd/dis = 1/cos(2Theta) (only wrong in displayed formula)
  2001-07-07  PB 3.12 do not use second image block to get image overlap
  2001-07-08  PB 3.13 FFirst
  2001-07-09  PB V4.00 new loop, no repetition
  2002-03-30  PB V4.01 two new switches to choose  
                       +/-ccd rejection (anode) correction
                       +/-mon intensity (monitor) normalization
                       +/-dis spherical angle (distance) normalization
  2002-06-01  PB V4.50 ave
  2003-02-03  PB V4.51 num[BlockNum]
  2003-02-07  PB V4.52 ib[].Bin -> ihb[].Bin
  2003-02-09  PB V4.53 line args for Dummy, DDummy, Dim
  2003-04-13  PB V4.54 Do not ask for anode values if 
                       rejectioncorrection is TRUE (FALSE!!!)
  2003-05-30  PB V4.55 ImageLoop: Function call changed
  2003-08-13  PB V4.56 option_define, all option numbers removed
  2003-11-30  PB V4.57 BinSiz
  2004-10-31  PB V4.58 MAKE_FUNCTION
  2005-07-09  PB V4.59 all exits removed, ERROUT
  2005-07-15  PB V4.60 ReportSaxsStatus parameter list updated
  2005-08-04  PB V4.61 PrintImageHeaderBlock paramters updated
  2005-09-17  PB V4.62 VarDat 
  2007-06-02  PB V4.63 Description updated 
  2007-06-11  PB V4.64 ImageLoop parameters updated
  2012-01-31  PB V4.65 ImageLoop parameters updated
  2015-10-04  PB V4.66 OpenImageFile parameter blkno changed 
                       from 3 to INflat,
                       Default_ipfn3 => Default_ipfnflat
  2017-10-20  PB V4.67 No key value replacement in user_io
                       ReadHeader... arguments updated 
---*/
 
# define Version  "saxs_normn V4.67 2017-10-20, Peter Boesecke"

# include <errno.h>
# include <stdio.h>
# include <unistd.h>
# include <fcntl.h>

# include "SaxsPrograms.h"

# define Usage "[options]\n\
    <i/p file> <scaler file> [<flatfield file>] <o/p file>\n\
    <first image> <last image> <inc>\n\
    <first scaler header> <inc scaler header>\n\
    [<flatfield image>]\n\
    <o/p dummy> <o/p dim1> <o/p dim2>\n\
    <transmission>\n\
    <scaler I0 mon.> <scaler exposure time> <scaler anode counts>\n\
    <2nd scaler I0 mon.> <2nd scaler exposure time> <2nd scaler anode counts>\n\
Purpose: Normalization of 2D small-angle scattering data\n\
Options:  -trm  <transmission>, -bcf <binning correction factor>,\n\
          +-flat, +-ccd, +-tron\n\
         The following options are replacing header values:\n\
          -mons <scaler I0 mon.>,       -mo2s <second scaler I0 mon.>,\n\
          -tims <scaler exposure time>, -ti2s <second scaler exposure time>,\n\
          -anos <scaler anode counts>,  -an2s <second scaler anode counts>,\n\
          -trms <scaler I1 mon.>,       -tr2s <second scaler I1 mon.>,\n\
          -dpth <eff. scaler depth in bits>, -slen <scaler length>,\n\
         Additional options to change individual scaler values (see +h)" 

# define Default_ipfn1     DefaultInput
# define Default_ipfn2     Default_ipfn1
# define Default_ipfnflat  "flatfield"DefaultExtension
# define Default_opfn      DefaultOutput
# define Default_inc       1
 
# define Default_flat      "FALSE"
# define Default_ccd       "FALSE"
# define Default_trmon     "FALSE"
# define Default_scalon    "TRUE"
# define Default_monon     "TRUE"
# define Default_dison     "TRUE"
# define Default_slen      "32"
# define Default_depth     "24"
# define Default_mons      "4"
# define Default_trms      "8"
# define Default_tims      "17"
# define Default_anos      "19"
# define Default_mo2s      "0"
# define Default_ti2s      "0"
# define Default_an2s      "0"
# define Default_tr2s      "0"
# define Default_trm       "1.0"

/* Special options */
# define SFlat     "flat"  /* use flatfield file */
# define SCcd      "ccd"   /* no rejection correction */
# define STrmOn    "tron"  /* prompt for trm, default trm=1 */
# define SScalOn   "scal"  /* use scaler file */
# define SMonOn    "mon"   /* intensity (monitor) normalization */
# define SDisOn    "dis"   /* spherical angle (distance) normalization */

# define SMonSca   "mons"  /* integer */
# define STimSca   "tims"
# define SAnoSca   "anos"
# define STrmSca   "trms" 
# define SMonScaS  "mo2s"
# define STimScaS  "ti2s"
# define SAnoScaS  "an2s"
# define STrmScaS  "tr2s"
# define SScaLen   "slen"

# define STrm       "trm"  /* float */
# define SScaDepth  "dpth"
# define SMonCount  "monc"
# define SMonZero   "monz"
# define SMonFac    "monf"
# define SMonVal    "monv"
# define STimCount  "timc"
# define STimFac    "timf"
# define STimVal    "timv"
# define SAnoCount  "anoc"
# define SAnoZero   "anoz"
# define SAnoFac    "anof"
# define SAnoVal    "anov"
# define STrmCount  "trmc"
# define STrmZero   "trmz"
# define STrmFac    "trmf"
# define STrmVal    "trmv"
# define SMonCountS "mo2c"
# define SMonFacS   "mo2f"
# define STimCountS "ti2c"
# define STimFacS   "ti2f"
# define SAnoCountS "an2c"
# define SAnoFacS   "an2f"
# define STrmCountS "tr2c"
# define STrmFacS   "tr2f"
# define SI1sum     "sum"
# define SBinCorFac "bcf"
# define SRelDis    "rdis"

# define BlockNum 4          /* 2 input + 1 output */

# define PRIVATE static        /* used to declare variables of private type */

/***************************************************************************
 * Global Initialised Variables                                            *
 ***************************************************************************/

PRIVATE char ScalerKeyNameBuffer[IO_size];

/*---------------------------------------------------------------------------
ScalerKey
return the full keyword of scaler number
---------------------------------------------------------------------------*/

char * ScalerKey ( const char * kh , const int scalernum )
{ if (scalernum<10)
     sprintf(ScalerKeyNameBuffer,"%s%s0%1u",KHScaler,kh,scalernum); else
     sprintf(ScalerKeyNameBuffer,"%s%s%2u",KHScaler,kh,scalernum);
  return ( ScalerKeyNameBuffer );
} /* ScalerKey */

/*---------------------------------------------------------------------------
user_io
Do all the keyboard io and return cb, and ib
---------------------------------------------------------------------------*/

void user_io( CmdBlk * pcb, ImgBlk * ib, int * pstatus )
{
  char  progfn[IO_size];

  long  image_1st;
  long  image_lst;

  int   next_block;
  int   INflat; /* image block number of flat field file */

  char  MonitorName[IO_size];
  char  TimeName[IO_size];
  char  AnodeName[IO_size];

  char  MonitorNameS[IO_size];
  char  TimeNameS[IO_size];
  char  AnodeNameS[IO_size];

  ImgHeadBlk ihb[BlockNum];
  long num[BlockNum];

  float ROff_1, RPs_1, UOff_1, UPs_1;
  float ROff_2, RPs_2, UOff_2, UPs_2;

  IO_long Flat, Ccd, TrmOn;
  IO_lvexp *pMonSca, *pTimSca, *pAnoSca, *pMonScaS, *pTimScaS, *pAnoScaS;
  IO_flexp *pTrm, *pBinCorFac;
  IO_long MonSca, TimSca, AnoSca, MonScaS, TimScaS, AnoScaS;

 /* Determine program name without directory */
   (void) RightFR((char *) pcb->argv[0],DirSeparator,progfn, IO_size);

  /* show usage if no arguments are given */
  if (pcb->argc<=1) printf("Usage: %s %s\n",progfn,pcb->MainHelp);

  /*--- Write name of program ---*/
  printf("\n");
  printf("   %s %s\n",progfn,pcb->MainVersion);
  printf("\n");

  Flat = option_flag ( SFlat, pcb, pstatus );
  if (*pstatus!=Success) return;
  Ccd = option_flag ( SCcd, pcb, pstatus );
  if (*pstatus!=Success) return;
  TrmOn = option_flag ( STrmOn, pcb, pstatus );
  if (*pstatus!=Success) return;

  /*--- Reduce total number of block, if no flatfield correction requested */
  next_block = 3;
  if (!Flat.V) {
    INflat = -1; pcb->ImgBlkLen = BlockNum-1;
  } else { 
    INflat = next_block++; 
  }

  /*--- OpenMode of first input file */
  if (!ib[1].OpenMode.I) {
    ib[1].OpenMode.V = IO_Old | IO_FileProtect;
    ib[1].OpenMode.I = TRUE;
  }

  /*--- Argument  : input file name */
  argv_filename( pcb, "Input file name", ib, 1, Default_ipfn1, pstatus);

  /* Exit in all cases after an abort from the first argument */
  if (*pstatus!=Success) return;

  /* No key value replacement in user_io -> ihb=NULL */
  OpenImageFile( pcb,ib,1,NULL,
                 ib[1].Name.V,ib[1].FFirst.V,
                 IO_Old|IO_FileProtect,pstatus);
  if (*pstatus!=Success) return;

  /*--- OpenMode of second input file */
  if (!ib[2].OpenMode.I) {
    ib[2].OpenMode.V = IO_Old | IO_FileProtect | IO_DontReadData;
    ib[2].OpenMode.I = TRUE;
  }

  /*--- Argument  : name of second input file (scaler file)*/
  argv_filename( pcb, "Scaler file name", ib, 2, ib[1].Name.V, pstatus);
  if (*pstatus!=Success) return;

  /* No key value replacement in user_io -> ihb=NULL */
  OpenImageFile( pcb,ib,2, NULL,
                 ib[2].Name.V,ib[2].FFirst.V,
                 IO_Old|IO_FileProtect,pstatus);
  if (*pstatus!=Success) return;

  /*--- Flatfield file */
  if (Flat.V) {
    /*--- OpenMode of third input file (flatfield file) */
    if (!ib[INflat].OpenMode.I) {
      ib[INflat].OpenMode.V = IO_Old | IO_FileProtect;
      ib[INflat].OpenMode.I = TRUE;
    }

    /*--- Argument  : name of flat field file */
    argv_filename(pcb,"Flatfield file name",ib,INflat,Default_ipfnflat, pstatus);
    if (*pstatus!=Success) return;

    /* No key value replacement in user_io -> ihb=NULL */
    OpenImageFile( pcb,ib,INflat,NULL,
                   ib[INflat].Name.V,ib[INflat].FFirst.V,
                   IO_Old|IO_FileProtect,pstatus);
    if (*pstatus!=Success) return;
  } /* Flatfield file */

  /*--- OpenMode of output file */
  if (!ib[0].OpenMode.I) {
    ib[0].OpenMode.V = IO_ImageProtect;
    ib[0].OpenMode.I = TRUE;
  }

  /*--- Argument  : output file name */
  argv_filename(pcb,"Output file name",ib,0,Default_opfn, pstatus);
  if (*pstatus!=Success) return;

  /* Search for minimum and maximum image number */
  (void) SearchMinMaxImage ( pcb, ib, 1, &image_1st, &image_lst, pstatus);
  if (*pstatus!=Success) return;

  /*--- Argument : number of first image */
  argv_long( pcb,"First image of input file", &ib[1].First, image_1st, pstatus);
  if (*pstatus!=Success) return;

  /*--- Argument : number of last image */
  if (pcb->argc==*pcb->parg_no+1) image_lst = ib[1].First.V;
  argv_long( pcb,"Last image of input file", &ib[1].Last, image_lst, pstatus);
  if (*pstatus!=Success) return;

  /*--- Argument : increment */
  argv_long( pcb,"Increment of input file", &ib[1].Inc, Default_inc, pstatus);
  if (*pstatus!=Success) return;

  num[1] = ib[1].First.V;
  (void) ReadImageHeader( pcb, ib, 1, num, ihb, pstatus);
  if (*pstatus!=Success) return;
  if (pcb->TestBit) PrintImageHeaderBlock(&ihb[1]);

  /*--- Argument : number of first image */
  argv_long( pcb,"First scaler image", &ib[2].First, ib[1].First.V,
             pstatus);
  if (*pstatus!=Success) return;

  /*--- Argument : increment */
  argv_long( pcb,"Increment of scaler image", &ib[2].Inc, ib[1].Inc.V, pstatus);
  if (*pstatus!=Success) return;

  num[2] = ib[2].First.V;
  (void) ReadImageHeader( pcb, ib, 2, num, ihb, pstatus);
  if (*pstatus!=Success) return;
  if (pcb->TestBit) PrintImageHeaderBlock(&ihb[2]);

  /*--- Flatfield file */
  if (Flat.V) {
    /* Search for minimum and maximum image number */
    (void) SearchMinMaxImage ( pcb, ib, 3, &image_1st, &image_lst, pstatus);
    if (*pstatus!=Success) return;

    /*--- Argument : number of first image */
    argv_long( pcb,"Flatfield image", &ib[INflat].First, image_1st, pstatus);
    if (*pstatus!=Success) return;

  } /* Flatfield file */

  /*--- Argument  : output dummy */
  if (!ib[0].Dummy.I) sprintf(ib[0].Dummy.V,"%g",ihb[1].Dummy.V);
  argv_flexp( pcb,"Output dummy", &ib[0].Dummy,ib[0].Dummy.V, pstatus);
  if (*pstatus!=Success) return;

  /*--- Argument  : output dimension 1 */
  if (!ib[0].Dim[1].I) sprintf(ib[0].Dim[1].V,"%d",ihb[1].Dim[1]);
  argv_lvexp( pcb,"Output dimension 1", &ib[0].Dim[1],ib[0].Dim[1].V, pstatus);
  if (*pstatus!=Success) return;

  /*--- Argument  : output dimension 2 */
  if (!ib[0].Dim[2].I) sprintf(ib[0].Dim[2].V,"%d",ihb[1].Dim[2]);
  argv_lvexp( pcb,"Output dimension 2", &ib[0].Dim[2],ib[0].Dim[2].V, pstatus);
  if (*pstatus!=Success) return;

 /*--- Argument  : Transmission */
  pTrm = (IO_flexp*) option_parameter_search( STrm, pcb, pstatus );
  if (*pstatus!=Success) return;
  if ( TrmOn.V ) {
    argv_flexp( pcb, "Transmission", pTrm, pTrm->V, pstatus);
    if (*pstatus!=Success) return;
  }

 /*--- Argument  : scaler number of I0 monitor */
  printf("\nThe following normalization will be done:\n");
  printf( "\n\
         Iin   rd       rd^2       1           (anoc*anof)              1\n\
  Iout = --- * ---- * --------- * --- * --------------------------  * -----\n\
         sum   dis    pix1*pix2   trm   (monc-monz*timc*timf)*monf    Iflat\n\
");

  printf( "\n\
         ^^^^^^^^^^^^^^^^^^^^^^^^^^^^   ^^^^^^^^^^^^^^^^^^^^^^^^^^    ^^^^^\n\
            first input file (i1)         scaler input file (i2)       (i3)\n\
");

  printf( "\n\
         with rd^2   = SampleDistance^2 + <radial distance on detector>^2\n\
              rd/dis = 1/cos(2*Theta)");
  if (!pTrm->I)
    printf( "\n\n\
                       (trmc-trmz*timc*timf)*trmf\n\
              trm    = --------------------------\n\
                       (monc-monz*timc*timf)*monf)");
    else printf( "\n\
              trm    = Transmission");
  printf( "\n\
              Iin  (detector raw data) : [event/pixel]\n\
              Iout (normalized data)   : [1/m]\n\n");
  if (Flat.V) printf( "\n\
	      Iflat (flatfield image)  : [event/photon]\n");
  printf( "\n");
  if (Flat.V) printf("  Division by flatfield\n\n"); 
  if (Ccd.V)  printf("  No rejection correction\n\n");

  /* -------------------------------------------------- */
  /* --- Argument: monitor scaler number (pMonSca) --- */
  /* -------------------------------------------------- */

    // Get default value of monitor scaler (MonSca) from blknum=2 
    pMonSca = (IO_lvexp*) option_parameter_search( SMonSca, pcb, pstatus );
    if (*pstatus!=Success) return;
    if (!pMonSca->I) {
      if (ReadHeaderLine( pcb,2,num,ihb,KHScalerI0,pMonSca->V,pstatus)) {
        if (*pstatus!=Success) return;
      }
    }
    // Ask for monitor scaler number
    argv_lvexp( pcb,"Number of I0 monitor scaler", pMonSca,pMonSca->V, pstatus);
    if (*pstatus!=Success) return;
   
    // Get monitor scaler name as info
    MonSca = option_long ( SMonSca, pcb, ihb, num, 2, pstatus );
    if (*pstatus!=Success) return;
    if (ReadHeaderLine( pcb,2,num,ihb,ScalerKey(KHN,MonSca.V),\
         MonitorName,pstatus)) {
      if (*pstatus!=Success) return;
    } else MonitorName[0] = (char) 0;


  /* -------------------------------------------------- */
  /* --- Argument: time scaler number (pTimSca) --- */
  /* -------------------------------------------------- */

    // Get default value of time scaler (TimSca) from blknum=2
    pTimSca = (IO_lvexp*) option_parameter_search( STimSca, pcb, pstatus );
    if (*pstatus!=Success) return;
    if (!pTimSca->I) {
      if (ReadHeaderLine( pcb,2,num,ihb,KHScalerTime,pTimSca->V,pstatus)) {
        if (*pstatus!=Success) return;
      }
    }
    // Ask for time scaler number
    argv_lvexp( pcb,"Number of time scaler", pTimSca,pTimSca->V, pstatus);
    if (*pstatus!=Success) return;
 
    // Get time scaler name as info
    TimSca = option_long ( STimSca, pcb, ihb, num, 2, pstatus );
    if (*pstatus!=Success) return;
    if (ReadHeaderLine( pcb,2,num,ihb,ScalerKey(KHN,TimSca.V),\
         TimeName,pstatus)) {
      if (*pstatus!=Success) return;
    } else TimeName[0] = (char) 0;

  /* -------------------------------------------------- */
  /* --- Argument: anode scaler number (pAnoSca) --- */
  /* -------------------------------------------------- */
 
    // Get default value of anode scaler (AnoSca) from blknum=2
    pAnoSca = (IO_lvexp*) option_parameter_search( SAnoSca, pcb, pstatus );
    if (*pstatus!=Success) return;
    if (!pAnoSca->I) {
      if (ReadHeaderLine( pcb,2,num,ihb,KHScalerAnode,pAnoSca->V,pstatus)) {
        if (*pstatus!=Success) return;
      }
    }
    // Ask for anode scaler number
    argv_lvexp( pcb,"Number of anode scaler", 
                pAnoSca, pAnoSca->V, pstatus);
    if (*pstatus!=Success) return;
 
    // Get anode scaler name as info
    AnoSca = option_long ( SAnoSca, pcb, ihb, num, 2, pstatus );
    if (*pstatus!=Success) return;
    if (ReadHeaderLine( pcb,2,num,ihb,ScalerKey(KHN,AnoSca.V),\
         AnodeName,pstatus)) {
      if (*pstatus!=Success) return;
    } else AnodeName[0] = (char) 0;

  printf("The second scalers are used in case of an overflow.\n");
  printf("Type 0 if you do not want to use a second scaler.\n");

  /* -------------------------------------------------------------- */
  /* --- Argument:  number of 2nd I0 monitor scaler (pMonScaS) --- */
  /* -------------------------------------------------------------- */
 
    // Get default value of 2nd I0 monitor scaler (pMonScaS) from blknum=2
    pMonScaS = (IO_lvexp*) option_parameter_search( SMonScaS, pcb, pstatus );
    if (*pstatus!=Success) return;
    if (!pMonScaS->I)
      if (ReadHeaderLine( pcb,2,num,ihb,KHScalerI0S,pMonScaS->V,pstatus)) {
        if (*pstatus!=Success) return;
      } 

    // Ask for number of 2nd I0 monitor scaler
    argv_lvexp( pcb,"Number of 2nd I0 monitor scaler",
                pMonScaS, pMonScaS->V, pstatus);
    if (*pstatus!=Success) return;

    // Get number of 2nd I0 monitor scaler name as info
    MonScaS = option_long ( SMonScaS, pcb, ihb, num, 2, pstatus );
    if (*pstatus!=Success) return;
    if (ReadHeaderLine( pcb,2,num,ihb,ScalerKey(KHN,MonScaS.V),\
         MonitorNameS,pstatus)) {
      if (*pstatus!=Success) return;
    } else MonitorNameS[0] = (char) 0;

  /* --------------------------------------------------------- */
  /* --- Argument:  number of 2nd time scaler (pTimScaS) --- */
  /* --------------------------------------------------------- */
 
    // Get default value of 2nd time scaler (pTimScaS) from blknum=2
    pTimScaS = (IO_lvexp*) option_parameter_search( STimScaS, pcb, pstatus );
    if (*pstatus!=Success) return;
    if (!pTimScaS->I)
      if (ReadHeaderLine( pcb,2,num,ihb,KHScalerTimeS,pTimScaS->V,pstatus)) {
        if (*pstatus!=Success) return;
      }
 
    // Ask for number of 2nd time scaler
    argv_lvexp( pcb,"Number of 2nd time scaler",
                pTimScaS, pTimScaS->V, pstatus);
    if (*pstatus!=Success) return;
 
    // Get number of 2nd time scaler name as info
    TimScaS = option_long ( STimScaS, pcb, ihb, num, 2, pstatus );
    if (*pstatus!=Success) return;
    if (ReadHeaderLine( pcb,2,num,ihb,ScalerKey(KHN,TimScaS.V),\
         TimeNameS,pstatus)) {
      if (*pstatus!=Success) return;
    } else TimeNameS[0] = (char) 0;

  /* ------------------------------------------------------- */
  /* --- Argument: number of 2nd anode scaler (pAnoScaS) --- */
  /* ------------------------------------------------------- */
 
    // Get default value of 2nd anode scaler (AnoScaS) from blknum=2
    pAnoScaS = (IO_lvexp*) option_parameter_search( SAnoScaS, pcb, pstatus );
    if (*pstatus!=Success) return;
    if (!pAnoScaS->I) {
      if (ReadHeaderLine( pcb,2,num,ihb,KHScalerAnodeS,pAnoScaS->V,pstatus)) {
        if (*pstatus!=Success) return;
      }
    }
    // Ask for 2nd anode scaler number
    argv_lvexp( pcb,"Number of 2nd anode scaler",
                pAnoScaS, pAnoScaS->V, pstatus);
    if (*pstatus!=Success) return;
 
    // Get 2nd anode scaler name as info
    AnoScaS = option_long ( SAnoScaS, pcb, ihb, num, 2, pstatus );
    if (*pstatus!=Success) return;
    if (ReadHeaderLine( pcb,2,num,ihb,ScalerKey(KHN,AnoScaS.V),\
         AnodeNameS,pstatus)) {
      if (*pstatus!=Success) return;
    } else AnodeNameS[0] = (char) 0;

 /* Binning correction factor (only for terminal info) */
  pBinCorFac = (IO_flexp*) option_parameter_search( SBinCorFac, pcb, pstatus );
  if (*pstatus!=Success) return;
  if (!pBinCorFac->I) { 
    sprintf( pBinCorFac->V,"%g",
      (float) ( MAX2(1,ihb[1].Bin[1].V) * MAX2(1,ihb[1].Bin[2].V) ) );
    pBinCorFac->I = TRUE; 
  }

 /* Close image files */
  if (Flat.V) {
    CloseImageFile( pcb, ib, 3, pstatus) ;
    if (*pstatus!=Success) return;
  }
  CloseImageFile( pcb, ib, 2, pstatus) ;
  if (*pstatus!=Success) return;
  CloseImageFile( pcb, ib, 1, pstatus) ;
  if (*pstatus!=Success) return;

  printf("\n");
  if (ib[1].Name.I)
  printf("Input file            : %s\n",ib[1].Name.V);
  if (ib[2].Name.I)
  printf("Scaler file           : %s\n",ib[2].Name.V);
/*  if ( (Flat.V) && (ib[INflat].Name.I) ) */
  if (INflat>=0)
    printf("Flatfield file        : %s\n",ib[INflat].Name.V);
  if (ib[0].Name.I)
  printf("Output file           : %s\n",ib[0].Name.V);
  if (ib[1].First.I)
  printf("First image           : %d\n",ib[1].First.V);
  if (ib[1].Last.I)
  printf("Last image            : %d\n",ib[1].Last.V);
  if (ib[1].Inc.I)   
  printf("Increment             : %d\n",ib[1].Inc.V);
  if (ib[2].First.I)
  printf("First scaler image    : %d\n",ib[2].First.V);
  if (ib[2].Inc.I)
  printf("Inc. of scaler image  : %d\n",ib[2].First.V);
/*  if ( (Flat.V) && (ib[INflat].First.I) ) */
  if (INflat>=0)
    printf("Flatfield image       : %d\n",ib[INflat].First.V);
  if (ib[0].Dummy.I)
  printf("Output dummy          : %s\n",ib[0].Dummy.V);
  if (ib[0].Dim[1].I)
  printf("Output dimension 1    : %s\n",ib[0].Dim[1].V);
  if (ib[0].Dim[2].I)
  printf("Output dimension 2    : %s\n",ib[0].Dim[2].V);
  if (pMonSca->I)
  printf("I0 monitor scaler     : %s (%s)\n",pMonSca->V,MonitorName);
  if (pTimSca->I)
  printf("Time scaler           : %s (%s)\n",pTimSca->V,TimeName);
  if (pAnoSca->I)
  printf("Anode scaler          : %s (%s)\n",pAnoSca->V,AnodeName);
  if (pMonScaS->I)
  printf("2nd I0 monitor scaler : %s (%s)\n",pMonScaS->V,MonitorNameS);
  if (pTimScaS->I)
  printf("2nd time scaler       : %s (%s)\n",pTimScaS->V,TimeNameS);
  if (pAnoScaS->I)
  printf("2nd anode scaler      : %s (%s)\n",pAnoScaS->V,AnodeNameS);
  if (pTrm->I)
  printf("Transmission          : %s \n",pTrm->V);
  printf("\n");

  if (pcb->TestBit) { PrintBlocks ( pcb, ib ); printf("\n"); }

} /* user_io */

/*---------------------------------------------------------------------------
1 saxs_normn

2 PURPOSE
  Normalization of an image
  The normalization is done in the following way:

                 (anoc*anof)/sum                 rd^3         1      1
 Iout = Iin * -------------------------- * --------------- * --- * -----
              (monc-monz*timc*timf)*monf    dis*pix1*pix2    trm   Iflat

       with rd^2 = dis^2 +
                     INDEX2R(i1,off1-cen1,pix1)^2 +
                     INDEX2R(i2,off2-cen2,pix2)^2

            trm = ((trmc-trmz*timc*timf)*trmf)/((monc-monz*timc*timf)*monf)
         or trm = "Option"

  If the normalization is not possible, e.g. division by zero, the output
  image contains only dummies. 
  The division by Iflat is omitted if -flat option is set
  If +ccd is set no rejection correction is done: 
       anoval = i1total * BinCorFac
---------------------------------------------------------------------------*/
void saxs_normn (CmdBlk * pcb, long num[], ImgHeadBlk ihb[], int * pstatus)
{ int NumberOfImages;
  const float eps=1e-30;
  int i,imax;
  int j;
  int normok, flatfieldcorrection;
  int rejectioncorrection, monitornormalization, distancenormalization;
  int scaleron=TRUE;
  float divider;
  int ErrorValue;

  /* output image information */
  float *I0Data, *E0Data;
  int   I0Dim_1,I0Dim_2;
  float *pI0Data, *pE0Data;
  float I0Dummy, I0DDummy;
  float VarDDummy=DDSET( VarDummy );
  float E0Value;

  /* input image information */
  int   I1Ave;
  float *I1Data,*E1Data;
  int   I1Dim_1,I1Dim_2;
  float *pI1Data;
  float I1Dummy, I1DDummy;
  float I1Value, I1Sum, I1Weight;
  float E1Value, E1Sum, E1Weight;

  /* scaler image information */

  /* flatfield image information */
  int   IFAve;
  int   INflat; /* number of flat field block */
  float *IFData,*EFData;
  int   IFDim_1,IFDim_2;
  float *pIFData;
  float IFDummy, IFDDummy;
  float IFValue, IFSum, IFWeight;
  float EFValue, EFSum, EFWeight;
  float InvIFValue2, I1InvIFValue2;
  
  int   scalerlen;
  float scalerdepth, maxcount;
  float moncount2, monzero2, monfac2;
  float timcount2, timfac2;
  float anocount2, anozero2, anofac2;
  float trmcount2, trmzero2, trmfac2;
  int   monscaler2, timscaler2, anoscaler2, trmscaler2;
  float moncount, monzero, monfac;
  float timcount, timfac;
  float anocount, anozero, anofac;
  float trmcount, trmzero, trmfac;
  float monval, trmval, anoval, timval;
  int   monscaler, timscaler, anoscaler, trmscaler;
  float detectorposition;
  float off1, off2, rd2, rd20, rd3, dd2, pix1, pix2, cen1, cen2, dis;
  float re2 = RELECTRON*RELECTRON;
  IO_long tmp;

  char  monname[IO_size];
  char  timname[IO_size];
  char  anoname[IO_size];
  char  trmname[IO_size];

  int   monI=FALSE, timI=FALSE, anoI=FALSE, trmI=FALSE;

  float i1total, i1max, bincorfac, normfac, accepted, effcountrate, trm;
  float maxpixrate, maxpixval, detval;
  float monvaltrm, dispix1pix2;
  float factor, factor2;

  IO_long Flat, Ccd, TrmOn, ScalOn, MonOn, DisOn;
  IO_long MonSca, TimSca, AnoSca, TrmSca, ScaLen;
  IO_long MonScaS, TimScaS, AnoScaS, TrmScaS;
  IO_float Trm, ScaDepth, MonCount, MonZero, MonFac, MonVal;
  IO_float TimCount, TimFac, TimVal;
  IO_float AnoCount, AnoZero, AnoFac, AnoVal;
  IO_float TrmCount, TrmZero, TrmFac, TrmVal;
  IO_float MonCountS, MonFacS, TimCountS, TimFacS;
  IO_float AnoCountS, AnoFacS, TrmCountS, TrmFacS;
  IO_float I1sum, BinCorFac, RelDis;

  /* Do not change the following definitions,
     they are automatically changed with BlockNum */
  int i_1, i_2;
  float f_1[BlockNum], f_2[BlockNum];
 
  float Off_1[BlockNum], Off_2[BlockNum];
  float Ps_1[BlockNum], Ps_2[BlockNum];
 
  float f1_1[BlockNum], f3_1[BlockNum], Df_1[BlockNum];
  float f1_2[BlockNum], f3_2[BlockNum], Df_2[BlockNum];
 
  int Imin_1, Imin_2, Imax_1, Imax_2;
 
  float Wmin_1, Wmax_1, W_1, DW_1;
  float Wmin_2, Wmax_2, W_2, DW_2;

  *pstatus = Success;

 /* Get special flag options */
  Flat = option_flag ( SFlat, pcb, pstatus );
  if (*pstatus!=Success) return;
  Ccd = option_flag ( SCcd, pcb, pstatus );
  if (*pstatus!=Success) return;
  TrmOn = option_flag ( STrmOn, pcb, pstatus );
  if (*pstatus!=Success) return;
  ScalOn = option_flag ( SScalOn, pcb, pstatus );
  if (*pstatus!=Success) return;
  MonOn = option_flag ( SMonOn, pcb, pstatus );
  if (*pstatus!=Success) return;
  DisOn = option_flag ( SDisOn, pcb, pstatus );
  if (*pstatus!=Success) return;

  imax = pcb->ImgBlkLen;
/*  printf("\n Calculating ihb[0,% d] = Function(",ihb[0].ImNum);
    for(i=1;i<imax;i++) {
      printf("ihb[% d,% d] ", i, ihb[i].ImNum); }
    printf(")\n\n");
*/
  for(i=1;i<imax;i++) {
    if (pcb->TestBit) PrintImageHeaderBlock(&ihb[i]);}

 /* Check the number of images */
  NumberOfImages = 3;
  INflat = NumberOfImages; /* must be an existing block */
  if (Flat.V) {
    flatfieldcorrection=TRUE; NumberOfImages += 1; 
    } else { flatfieldcorrection=FALSE; }

  if (pcb->ImgBlkLen!=NumberOfImages) {
     printf("%d images found, %d input and 1 output image required\n",
             pcb->ImgBlkLen, NumberOfImages-1); *pstatus=Failed; return; }

 /* Get special long int options */
  MonSca = option_long ( SMonSca, pcb, ihb, num, 1, pstatus );
  if (*pstatus!=Success) return;
  TimSca = option_long ( STimSca, pcb, ihb, num, 1, pstatus );
  if (*pstatus!=Success) return;
  AnoSca = option_long ( SAnoSca, pcb, ihb, num, 1, pstatus );
  if (*pstatus!=Success) return;
  TrmSca = option_long ( STrmSca, pcb, ihb, num, 1, pstatus );
  if (*pstatus!=Success) return;
  MonScaS = option_long ( SMonScaS, pcb, ihb, num, 1, pstatus );
  if (*pstatus!=Success) return;
  TimScaS = option_long ( STimScaS, pcb, ihb, num, 1, pstatus );
  if (*pstatus!=Success) return;
  AnoScaS = option_long ( SAnoScaS, pcb, ihb, num, 1, pstatus );
  if (*pstatus!=Success) return;
  TrmScaS = option_long ( STrmScaS, pcb, ihb, num, 1, pstatus );
  if (*pstatus!=Success) return;
  ScaLen = option_long ( SScaLen, pcb, ihb, num, 1, pstatus );
  if (*pstatus!=Success) return;

 /* Get special float options */
  Trm = option_float ( STrm, pcb, ihb, num, 1, pstatus );
  if (*pstatus!=Success) return;
  ScaDepth = option_float ( SScaDepth, pcb, ihb, num, 1, pstatus );
  if (*pstatus!=Success) return;
  MonCount = option_float ( SMonCount, pcb, ihb, num, 1, pstatus );
  if (*pstatus!=Success) return;
  MonZero = option_float ( SMonZero, pcb, ihb, num, 1, pstatus );
  if (*pstatus!=Success) return;
  MonFac = option_float ( SMonFac, pcb, ihb, num, 1, pstatus );
  if (*pstatus!=Success) return;
  MonVal = option_float ( SMonVal, pcb, ihb, num, 1, pstatus );
  if (*pstatus!=Success) return;
  TimCount = option_float ( STimCount, pcb, ihb, num, 1, pstatus );
  if (*pstatus!=Success) return;
  TimFac = option_float ( STimFac, pcb, ihb, num, 1, pstatus );
  if (*pstatus!=Success) return;
  TimVal = option_float ( STimVal, pcb, ihb, num, 1, pstatus );
  if (*pstatus!=Success) return;
  AnoCount = option_float ( SAnoCount, pcb, ihb, num, 1, pstatus );
  if (*pstatus!=Success) return;
  AnoZero = option_float ( SAnoZero, pcb, ihb, num, 1, pstatus );
  if (*pstatus!=Success) return;
  AnoFac = option_float ( SAnoFac, pcb, ihb, num, 1, pstatus );
  if (*pstatus!=Success) return;
  AnoVal = option_float ( SAnoVal, pcb, ihb, num, 1, pstatus );
  if (*pstatus!=Success) return;
  TrmCount = option_float ( STrmCount, pcb, ihb, num, 1, pstatus );
  if (*pstatus!=Success) return;
  TrmZero = option_float ( STrmZero, pcb, ihb, num, 1, pstatus );
  if (*pstatus!=Success) return;
  TrmFac = option_float ( STrmFac, pcb, ihb, num, 1, pstatus );
  if (*pstatus!=Success) return;
  TrmVal = option_float ( STrmVal, pcb, ihb, num, 1, pstatus );
  if (*pstatus!=Success) return;
  MonCountS = option_float ( SMonCountS, pcb, ihb, num, 1, pstatus );
  if (*pstatus!=Success) return;
  MonFacS = option_float ( SMonFacS, pcb, ihb, num, 1, pstatus );
  if (*pstatus!=Success) return;
  TimCountS = option_float ( STimCountS, pcb, ihb, num, 1, pstatus );
  if (*pstatus!=Success) return;
  TimFacS = option_float ( STimFacS, pcb, ihb, num, 1, pstatus );
  if (*pstatus!=Success) return;
  AnoCountS = option_float ( SAnoCountS, pcb, ihb, num, 1, pstatus );
  if (*pstatus!=Success) return;
  AnoFacS = option_float ( SAnoFacS, pcb, ihb, num, 1, pstatus );
  if (*pstatus!=Success) return;
  TrmCountS = option_float ( STrmCountS, pcb, ihb, num, 1, pstatus );
  if (*pstatus!=Success) return;
  TrmFacS = option_float ( STrmFacS, pcb, ihb, num, 1, pstatus );
  if (*pstatus!=Success) return;
  I1sum = option_float ( SI1sum, pcb, ihb, num, 1, pstatus );
  if (*pstatus!=Success) return;
  BinCorFac = option_float ( SBinCorFac, pcb, ihb, num, 1, pstatus );
  if (*pstatus!=Success) return;
  RelDis = option_float ( SRelDis, pcb, ihb, num, 1, pstatus );
  if (*pstatus!=Success) return;

  /* Check, if rejeciton correction is requested */
  if (Ccd.V) rejectioncorrection=FALSE; 
    else rejectioncorrection=TRUE;

  /* Check, if monitor normalization is requested */
  if (MonOn.V) monitornormalization=TRUE;
    else monitornormalization=FALSE;

  /* Check, if distance normalization is requested */
  if (DisOn.V) distancenormalization=TRUE;
    else distancenormalization=FALSE;

  /* Check, if scalers should be used */
  /* scalers are not used when keyword KHScalerI0 is not found */
 
  if (ScalOn.I) 
     scaleron=(int)ScalOn.V; 
   else /* test, if KHScalerI0 can be read */
     scaleron=ReadHeaderLong(pcb,2,num,ihb,KHScalerI0,&tmp.V,pstatus);

  GetReferenceParameters( pcb, ihb, 0, imax-1,
                          Off_1, Off_2, Ps_1,  Ps_2, pstatus );
  if (*pstatus!=Success) return;

  UpdateImageParameters ( pcb, ihb, 1, imax-1,
                          Off_1, Off_2, Ps_1,  Ps_2, pstatus );
  if (*pstatus!=Success) return;

 /* 1 input for data (input file -> I1),
    1 input for header (scaler file -> I2, only used to read scalers),
    1 output -> I0*/

  I0Data  = ihb[0].Data;
  E0Data  = ihb[0].VarDat;
  I0Dummy = ihb[0].Dummy.V;
  I0DDummy = ihb[0].DDummy.V;
  I0Dim_1  = (int) ihb[0].Dim[1];
  I0Dim_2  = (int) ihb[0].Dim[2];

  I1Ave    = pcb->ib[1].Ave.V;
  I1Data   = ihb[1].Data;
  E1Data   = ihb[1].VarDat;
  I1Dummy  = ihb[1].Dummy.V;
  I1DDummy = ihb[1].DDummy.V;
  I1Dim_1  = (int) ihb[1].Dim[1];
  I1Dim_2  = (int) ihb[1].Dim[2];

  IFAve    = pcb->ib[INflat].Ave.V;
  IFData   = ihb[INflat].Data;
  EFData   = ihb[INflat].VarDat;
  IFDummy  = ihb[INflat].Dummy.V;
  IFDDummy = ihb[INflat].DDummy.V;
  IFDim_1  = (int) ihb[INflat].Dim[1];
  IFDim_2  = (int) ihb[INflat].Dim[2];

  DoNotScaleWith(ihb,2); /* Do not use header ihb[2] for scaling */
  GetImageRange         ( pcb, ihb, 1, imax-1,
                          Off_1, Off_2, Ps_1,  Ps_2,
                          f1_1, f3_1, Df_1, f1_2, f3_2, Df_2,
                          &Imin_1, &Imax_1, &Imin_2, &Imax_2,
                          &Wmin_1, &Wmax_1, &DW_1, &Wmin_2, &Wmax_2, &DW_2, 1);

  /* if RelDis is set, calculate SampleDistance from detectorposition+RelDis */
  if (RelDis.I) {
      ReadHeaderFloat(pcb,2,num,ihb,KDetectorPosition,&detectorposition,pstatus);
      if (*pstatus!=Success) return;
      ihb[0].SampleDistance.V = detectorposition + RelDis.V;
      ihb[1].SampleDistance.V = ihb[0].SampleDistance.V;
      ihb[2].SampleDistance.V = ihb[1].SampleDistance.V;
      ihb[0].SampleDistance.I = TRUE;
      ihb[1].SampleDistance.I = TRUE;
      ihb[2].SampleDistance.I = TRUE;
      if (pcb->TestBit) {
        printf("RelativeDistance   = %9.3g m\n",RelDis.V);
        printf("DetectorPosition   = %9.3g m\n",detectorposition);
        printf("New SampleDistance = %9.3g m\n",ihb[0].SampleDistance.V);
        }
      } 
    
  /* get geometrical parameters from the header */
  if (ihb[1].Offset[1].I) off1 = ihb[1].Offset[1].V;
    else {fprintf(ERROUT,"Missing header information (Offset[1])\n");
          *pstatus=KeywordMissing; return;}
  if (ihb[1].Offset[2].I) off2 = ihb[1].Offset[2].V;
    else {fprintf(ERROUT,"Missing header information (Offset[2])\n");
          *pstatus=KeywordMissing; return;}
  if (pcb->TestBit) {
     printf("off1 = %9.3g pixel  , off2 = %9.3g pixel\n",off1,off2);
     printf("re^2 = %9.3g m^2\n",re2);
    }

  if (distancenormalization) {
    if (ihb[1].PixSiz[1].I) pix1 = ihb[1].PixSiz[1].V;
      else {fprintf(ERROUT,"Missing header information (PixSiz[1])\n");
          *pstatus=KeywordMissing; return;}
    if (ihb[1].PixSiz[2].I) pix2 = ihb[1].PixSiz[2].V;
      else {fprintf(ERROUT,"Missing header information (PixSiz[2])\n");
          *pstatus=KeywordMissing; return;}
    if (ihb[1].Center[1].I) cen1 = ihb[1].Center[1].V;
      else {fprintf(ERROUT,"Missing header information (Center[1])\n");
          *pstatus=KeywordMissing; return;}
    if (ihb[1].Center[2].I) cen2 = ihb[1].Center[2].V;
      else {fprintf(ERROUT,"Missing header information (Center[2])\n");
          *pstatus=KeywordMissing; return;}
    if (ihb[1].SampleDistance.I) dis = ihb[1].SampleDistance.V;
      else {fprintf(ERROUT,"Missing header information (SampleDistance)\n");
          *pstatus=KeywordMissing; return;}

    if (pcb->TestBit) {
      printf("cen1 = %9.3g pixel  , cen2 = %9.3g pixel\n",cen1,cen2);
      printf("pix1 = %9.3g m/pixel, pix2 = %9.3g m/pixel\n",pix1,pix2);
      printf("dis  = %9.3g m\n",dis);
    }

    if (fabs(pix1*pix2)<eps) {
      fprintf(ERROUT,"Pixel size too small (pix1*pix2 = %g)\n",pix1*pix2); 
          *pstatus=KeywordMissing; return;
    }
    if (fabs(dis)<eps) {
      fprintf(ERROUT,"Sample distance too small (dis = %g)\n",dis); 
          *pstatus=KeywordMissing; return;
    }
  } // distancenormalization

  /* --- Calculate sum of input image 1 i1total for rejection correction */
  if (!I1sum.I) {
    i1total=0.0;
    pI1Data=I1Data;
    for (j=0;j<I1Dim_1*I1Dim_2;j++) {
      if (NODUMMY(*pI1Data,I1Dummy,I1DDummy)) i1total+=*pI1Data;
      pI1Data++;
      }
    } else i1total=I1sum.V;

  /* --- Binning correction factor (only for info) */
  if (BinCorFac.I) bincorfac = BinCorFac.V;
    else bincorfac = 1.0;

  /* --- Maximum value (only for info) */
  pI1Data=I1Data;
  /* skip all dummies */
  for (j=0;(j<I1Dim_1*I1Dim_2)&&(DUMMY(*pI1Data,I1Dummy,I1DDummy));j++) {
    pI1Data++;
    }
  if (j<I1Dim_1*I1Dim_2) i1max=*pI1Data; else i1max=I1Dummy;
  /* find maximum */
  for (j=j;j<I1Dim_1*I1Dim_2;j++) {
    if (NODUMMY(*pI1Data,I1Dummy,I1DDummy)) i1max=MAX2(i1max,*pI1Data);
    pI1Data++;
    }

  if (pcb->TestBit)
    printf("i1max    = %g\n",i1max);

  /* Initialize scaler names */
  monname[0] = (char) 0;
  timname[0] = (char) 0;
  anoname[0] = (char) 0;
  trmname[0] = (char) 0;

  if (scaleron) { /* read counter values from file */

    /* --- maxcount */
    if (!ScaDepth.I) 
      if (ReadHeaderFloat(pcb,2,num,ihb,KHScalerDepth,&ScaDepth.V,
          pstatus)) {
        if (*pstatus!=Success) return;
        } 

    scalerdepth = ScaDepth.V;
    maxcount = exp(scalerdepth*log(2));

    if (pcb->TestBit) {
      printf("scalerdepth = %g\n", scalerdepth);
      printf("maxcount = %g\n", maxcount);
      }

    /* --- scalerlen */
    if (!ScaLen.I) {
      if (ReadHeaderLong(pcb,2,num,ihb,KHScalerLen, &ScaLen.V,
          pstatus)) { if (*pstatus!=Success) return;
        } 
      }
    scalerlen = (int) ScaLen.V;
    if (pcb->TestBit) printf("scalerlen = %d\n", scalerlen);

    /* --- Monitor */
    if (!MonVal.I) {

    monI = FALSE;
    /*     monscaler1 */
    if (!MonSca.I) {
      if (!ReadHeaderLong(pcb,2,num,ihb,KHScalerI0,&MonSca.V,
          pstatus)) { *pstatus = KeywordMissing;
        fprintf(ERROUT,"\nMissing keyword for monscaler1 %s\n",KHScalerI0); } 
      if (*pstatus!=Success) return;
      }
    monscaler = (int) MonSca.V;
    if (pcb->TestBit) printf("monscaler = %d\n", monscaler);

    if ((1<=monscaler) && (monscaler<=scalerlen)) {
      /*     moncount1 */
      if (!MonCount.I) {
        if (!ReadHeaderFloat(pcb,2,num,ihb,ScalerKey(KHC,monscaler),&moncount,
            pstatus)) { *pstatus = KeywordMissing;
          fprintf(ERROUT,"\nMissing keyword for moncount1 %s\n",
            ScalerKey(KHC,monscaler));}
        if (*pstatus!=Success) return;
        } else moncount = MonCount.V;

      /*      monfac1 */
      if (!MonFac.I) {
        if (!ReadHeaderFloat(pcb,2,num,ihb,ScalerKey(KHF,monscaler),&monfac,
            pstatus)) { *pstatus = KeywordMissing;
          fprintf(ERROUT,"\nMissing keyword for monfac1 %s\n",
            ScalerKey(KHF,monscaler)); }
        if (*pstatus!=Success) return;
        } else monfac = MonFac.V;
  
      /*     monzero1 */
      if (!MonZero.I) {
        if (!ReadHeaderFloat(pcb,2,num,ihb,ScalerKey(KHZ,monscaler),&monzero,
            pstatus)) { *pstatus = KeywordMissing;
          fprintf(ERROUT,"\nMissing keyword for monzero1 %s\n",
            ScalerKey(KHZ,monscaler)); }
        if (*pstatus!=Success) return;
        } else monzero = MonZero.V;
      monI = TRUE;
      } else { 
      moncount = 0.0; monfac = 1.0; monzero = 0.0; 
      } /* if (monscaler ... */

    /*     monscaler2 */
    if (!MonScaS.I) {
      if (ReadHeaderLong(pcb,2,num,ihb,KHScalerI0S, &MonScaS.V,
          pstatus)) { if (*pstatus!=Success) return; 
        } 
      }
    monscaler2 = (int) MonScaS.V;
  
    if ((1<=monscaler2) && (monscaler2<=scalerlen)) {
      /*     moncount2 */
      if (!MonCountS.I) {
        if (!ReadHeaderFloat(pcb,2,num,ihb,ScalerKey(KHC,monscaler2),&moncount2,
            pstatus)) { *pstatus = KeywordMissing;
         fprintf(ERROUT,"\nMissing keyword for moncount2 %s\n",
           ScalerKey(KHC,monscaler2));}
        if (*pstatus!=Success) return;
        } else moncount2 = MonCountS.V;
        
      /*      monfac2 */
      if (!MonFacS.I) {
        if (!ReadHeaderFloat(pcb,2,num,ihb,ScalerKey(KHF,monscaler2),&monfac2,
            pstatus)) { *pstatus = KeywordMissing;
          fprintf(ERROUT,"\nMissing keyword for monfac2 %s\n",
            ScalerKey(KHF,monscaler2)); }
        if (*pstatus!=Success) return;
        } else monfac2 = MonFacS.V;
 
      /*     monzero2  (no separate option) */
      if (!MonZero.I) {
        if (!ReadHeaderFloat(pcb,2,num,ihb,ScalerKey(KHZ,monscaler2),&monzero2,
            pstatus)) { *pstatus = KeywordMissing;
          fprintf(ERROUT,"\nMissing keyword for monzero2 %s\n",
            ScalerKey(KHZ,monscaler2));}
        if (*pstatus!=Success) return;
        } else monzero2 = MonZero.V;
      monI = TRUE;

      /*      Find valid scaler */
      if (moncount2*monfac2>maxcount*monfac) {
        monscaler = monscaler2;
        moncount  = moncount2;
        monfac    = monfac2;
        monzero   = monzero2;
        if (pcb->TestBit) printf("Using alternativescaler for monitor.\n");
        }
      } /* if (monscaler2 ... */

    /*     monname */
    if (ReadHeaderLine(pcb,2,num,ihb,ScalerKey(KHN,monscaler),monname, 
        pstatus)) { if (*pstatus!=Success) return; 
      } else monname[0] = (char) 0;

    if (pcb->TestBit) {
      printf("monscaler = %d, monname = %s\n", monscaler, monname);
      printf("moncount = %g, monzero = %g, monfac = %g\n", \
              moncount, monzero, monfac);
      }
      } /* if (!MonVal.I) */ 

    /* --- Time */
    if (!TimVal.I) {

    timI = FALSE;
    /*     timscaler1 */
    if (!TimSca.I) {
      if (!ReadHeaderLong(pcb,2,num,ihb,KHScalerTime, &TimSca.V,
          pstatus)) { *pstatus = KeywordMissing;
        fprintf(ERROUT,"\nMissing keyword for timscaler1 %s\n",KHScalerTime); }
      if (*pstatus!=Success) return;
      }
  
    timscaler = (int) TimSca.V;
 
    if ((1<=timscaler) && (timscaler<=scalerlen)) {
      /*     timcount1 */
      if (!TimCount.I) {
        if (!ReadHeaderFloat(pcb,2,num,ihb,ScalerKey(KHC,timscaler),&timcount, 
            pstatus)) { *pstatus = KeywordMissing;
          fprintf(ERROUT,"\nMissing keyword for timcount1 %s\n",
            ScalerKey(KHC,timscaler));}
        if (*pstatus!=Success) return;
        } else timcount = TimCount.V;
  
      /*      timfac1 */
      if (!TimFac.I) {
        if (!ReadHeaderFloat(pcb,2,num,ihb,ScalerKey(KHF,timscaler), &timfac,
            pstatus)) { *pstatus = KeywordMissing;
          fprintf(ERROUT,"\nMissing keyword for timfac1 %s\n",
            ScalerKey(KHF,timscaler)); }
        if (*pstatus!=Success) return; 
        } else timfac = TimFac.V;
      timI = TRUE;
      } else {
      timcount = 0.0; timfac = 1.0; 
      }

    /*     timscaler2 */
    if (!TimScaS.I) {
      if (ReadHeaderLong(pcb,2,num,ihb,KHScalerTimeS,&TimScaS.V,
          pstatus)) { if (*pstatus!=Success) return;
        }
      }
    timscaler2 = (int) TimScaS.V;

    if ((1<=timscaler2) && (timscaler2<=scalerlen)) {
      /*     timcount2 */
      if (!TimCountS.I) {
        if (!ReadHeaderFloat(pcb,2,num,ihb,ScalerKey(KHC,timscaler2),&timcount2, 
            pstatus)) { *pstatus = KeywordMissing;
         fprintf(ERROUT,"\nMissing keyword for timcount2 %s\n",
           ScalerKey(KHC,timscaler2));}
        if (*pstatus!=Success) return;
        } else timcount2 = TimCountS.V;
  
      /*      timfac2 */
      if (!TimFacS.I) {
        if (!ReadHeaderFloat(pcb,2,num,ihb,ScalerKey(KHF,timscaler2), &timfac2, 
            pstatus)) { *pstatus = KeywordMissing;
          fprintf(ERROUT,"\nMissing keyword for timfac2 %s\n",
            ScalerKey(KHF,timscaler2));}
        if (*pstatus!=Success) return;
        } else timfac2 = TimFacS.V;
      timI = TRUE;

      /*      Find valid scaler */
      if (timcount2*timfac2>maxcount*timfac) {
        timscaler = timscaler2;
        timcount  = timcount2;
        timfac    = timfac2;
        if (pcb->TestBit) printf("Using alternative scaler for time.\n");
        }
      } /* if (1<=timscaler2 ... */
  
    /*     timname */
    if (ReadHeaderLine(pcb,2,num,ihb,ScalerKey(KHN,timscaler),timname, 
        pstatus)) { if (*pstatus!=Success) return;
      } else timname[0] = (char) 0;
  
    if (pcb->TestBit) {
      printf("timscaler = %d, timname = %s\n", timscaler, timname);
      printf("timcount = %g, timfac = %g\n", timcount, timfac);
      }
      } /* if (TimVal.I) */
  
    /* --- Anode */
    if ((rejectioncorrection)&&(!AnoVal.I)) {

    anoI = FALSE;
    /*     anoscaler1 */
     if (!AnoSca.I) {
      if (!ReadHeaderLong(pcb,2,num,ihb,KHScalerAnode, &AnoSca.V,
          pstatus)) { *pstatus = KeywordMissing;
        fprintf(ERROUT,"\nMissing keyword for anoscaler1 %s\n",KHScalerAnode); }
      if (*pstatus!=Success) return;
      }
    anoscaler = (int) AnoSca.V;
 
    if ((1<=anoscaler) && (anoscaler<=scalerlen)) {
      /*     anocount1 */
      if (!AnoCount.I) {
        if (!ReadHeaderFloat(pcb,2,num,ihb,ScalerKey(KHC,anoscaler),&anocount,
            pstatus)) { *pstatus = KeywordMissing;
          fprintf(ERROUT,"\nMissing keyword for anocount1 %s\n",
            ScalerKey(KHC,anoscaler));}
        if (*pstatus!=Success) return;
        } else anocount = AnoCount.V;

      /*      anofac1 */
      if (!AnoFac.I) {
        if (!ReadHeaderFloat(pcb,2,num,ihb,ScalerKey(KHF,anoscaler), &anofac,
            pstatus)) { *pstatus = KeywordMissing;
          fprintf(ERROUT,"\nMissing keyword for anofac1 %s\n",
            ScalerKey(KHF,anoscaler)); }
        if (*pstatus!=Success) return;
        } else anofac = AnoFac.V;
  
      /*     anozero1 */
      if (!AnoZero.I) {
        if (!ReadHeaderFloat(pcb,2,num,ihb,ScalerKey(KHZ,anoscaler),&anozero,
            pstatus)) { *pstatus = KeywordMissing;
          fprintf(ERROUT,"\nMissing keyword for anozero1 %s\n",
            ScalerKey(KHZ,anoscaler)); }
        if (*pstatus!=Success) return;
        } else anozero = AnoZero.V;
   
      anoI = TRUE;
      } else {
      anocount = 0.0; anofac = 1.0;
      } /* if (anoscaler ... */

    /*     anoscaler2 */
   if (!AnoScaS.I) {
      if (ReadHeaderLong(pcb,2,num,ihb,KHScalerAnodeS, &AnoScaS.V,
          pstatus)) { if (*pstatus!=Success) return;
        }
      }
    anoscaler2 = (int) AnoScaS.V;
 
    if ((1<=anoscaler2) && (anoscaler2<=scalerlen)) {
      /*     anocount2 */
      if (!AnoCountS.I) {
        if (!ReadHeaderFloat(pcb,2,num,ihb,ScalerKey(KHC,anoscaler2), &anocount2, 
            pstatus)) { *pstatus = KeywordMissing;
         fprintf(ERROUT,"\nMissing keyword for anocount2 %s\n",
           ScalerKey(KHC,anoscaler2));}
        if (*pstatus!=Success) return;
        } else anocount2 = AnoCountS.V;

      /*      anofac2 */
      if (!AnoFacS.I) {
        if (!ReadHeaderFloat(pcb,2,num,ihb,ScalerKey(KHF,anoscaler2),&anofac2,
            pstatus)) { *pstatus = KeywordMissing;
          fprintf(ERROUT,"\nMissing keyword for anofac2 %s\n",
            ScalerKey(KHF,anoscaler2)); }
        if (*pstatus!=Success) return;
        } else anofac2 = AnoFacS.V;

      /*     anozero2  (no separate option) */
      if (!AnoZero.I) {
        if (!ReadHeaderFloat(pcb,2,num,ihb,ScalerKey(KHZ,anoscaler2),&anozero2,
            pstatus)) { *pstatus = KeywordMissing;
          fprintf(ERROUT,"\nMissing keyword for anozero2 %s\n",
            ScalerKey(KHZ,anoscaler2));
          }
        if (*pstatus!=Success) return;
        } else anozero2 = AnoZero.V; 

      anoI = TRUE;

      /*      Find valid scaler */
      if (anocount2*anofac2>maxcount*anofac) {
        anoscaler = anoscaler2;
        anocount  = anocount2;
        anofac    = anofac2;
        anozero   = anozero2;
        if (pcb->TestBit) printf("Using alternate scaler for anode.\n");
        }
      } /* if (anoscaler2 ... */

    /*     anoname */
    if (ReadHeaderLine(pcb,2,num,ihb,ScalerKey(KHN,anoscaler), anoname, 
        pstatus)) { if (*pstatus!=Success) return;
      } else anoname[0] = (char) 0;

    if (pcb->TestBit) {
      printf("anoscaler = %d, anoname = %s\n", anoscaler, anoname);
      printf("anocount = %g, anozero = %g, anofac = %g\n", 
         anocount, anozero, anofac);
      }
    } /* if (AnoVal.I) */

    /* --- Transmission monitor */
    if (!TrmVal.I) { 

    trmI = FALSE;
    /*     trmscaler1 */
    if (!TrmSca.I) {
      if (!ReadHeaderLong(pcb,2,num,ihb,KHScalerI1,&TrmSca.V,
          pstatus)) { *pstatus = KeywordMissing;
        fprintf(ERROUT,"\nMissing keyword for trmscaler1 %s\n",KHScalerI1); }
      if (*pstatus!=Success) return;
      }
    trmscaler = (int) TrmSca.V;

    if ((1<=trmscaler) && (trmscaler<=scalerlen)) {
      /*     trmcount1 */
      if (!TrmCount.I) {
        if (!ReadHeaderFloat(pcb,2,num,ihb,ScalerKey(KHC,trmscaler),&trmcount,
            pstatus)) { *pstatus = KeywordMissing;
          fprintf(ERROUT,"\nMissing keyword for trmcount1 %s\n",
            ScalerKey(KHC,trmscaler));}
        if (*pstatus!=Success) return;
        } else trmcount = TrmCount.V;

      /*      trmfac1 */
      if (!TrmFac.I) {
        if (!ReadHeaderFloat(pcb,2,num,ihb,ScalerKey(KHF,trmscaler),&trmfac,
            pstatus)) { *pstatus = KeywordMissing;
          fprintf(ERROUT,"\nMissing keyword for trmfac1 %s\n",
            ScalerKey(KHF,trmscaler)); }
         if (*pstatus!=Success) return;
        } else trmfac = TrmFac.V;

      /*     trmzero1 */
      if (!TrmZero.I) {
        if (!ReadHeaderFloat(pcb,2,num,ihb,ScalerKey(KHZ,trmscaler),&trmzero,
            pstatus)) { *pstatus = KeywordMissing;
          fprintf(ERROUT,"\nMissing keyword for trmzero1 %s\n",
            ScalerKey(KHZ,trmscaler));}
        if (*pstatus!=Success) return;
        } else trmzero = TrmZero.V;
      trmI = TRUE;
      } else {
      trmcount = 0.0; trmfac = 1.0; trmzero = 0.0;
      }

    /*     trmscaler2 */
    if (!TrmScaS.I) {
      if (ReadHeaderLong(pcb,2,num,ihb,KHScalerI1S, &TrmScaS.V,
          pstatus)) { if (*pstatus!=Success) return;
        }
      }
    trmscaler2 = (int) TrmScaS.V;
  
    if ((1<=trmscaler2) && (trmscaler2<=scalerlen)) {
      /*     trmcount2 */
      if (!TrmCountS.I) {
        if (!ReadHeaderFloat(pcb,2,num,ihb,ScalerKey(KHC,trmscaler2),&trmcount2,
            pstatus)) { *pstatus = KeywordMissing;
            fprintf(ERROUT,"\nMissing keyword for trmcount2 %s\n",
            ScalerKey(KHC,trmscaler2));}
        if (*pstatus!=Success) return;
        } else trmcount2 = TrmCountS.V;

      /*      trmfac2 */
      if (!TrmFacS.I) {
        if (!ReadHeaderFloat(pcb,2,num,ihb,ScalerKey(KHF,trmscaler2),&trmfac2,
            pstatus)) { *pstatus = KeywordMissing;
          fprintf(ERROUT,"\nMissing keyword for trmfac2 %s\n",
            ScalerKey(KHF,trmscaler2));}
        if (*pstatus!=Success) return;
        } else trmfac2 = TrmFacS.V;

      /*     trmzero2  (no separate option) */
      if (!TrmZero.I) {
        if (!ReadHeaderFloat(pcb,2,num,ihb,ScalerKey(KHZ,trmscaler2),&trmzero2,
            pstatus)) { *pstatus = KeywordMissing;
         fprintf(ERROUT,"\nMissing keyword for trmzero2 %s\n",
           ScalerKey(KHZ,trmscaler2));}
        if (*pstatus!=Success) return;
        } else trmzero2 = TrmZero.V;
      trmI = TRUE;

      /*      Find valid scaler */
      if (trmcount2*trmfac2>maxcount*trmfac) {
        trmscaler = trmscaler2;
        trmcount  = trmcount2;
        trmfac    = trmfac2;
        trmzero   = trmzero2;
        if (pcb->TestBit) 
          printf("Using alternative scaler for transmission.\n");
        }
      } /* if (trmscaler2 ... */

      /*     trmname */
      if (ReadHeaderLine(pcb,2,num,ihb,ScalerKey(KHN,trmscaler),trmname,
          pstatus)) { if (*pstatus!=Success) return;
         } else trmname[0] = (char) 0;

    if (pcb->TestBit) {
      printf("trmscaler = %d, trmname = %s\n", trmscaler, trmname);
      printf("trmcount = %g, trmzero = %g, trmfac = %g\n", \
              trmcount, trmzero, trmfac);
      }
      } /* if (TrmVal.I) */

    /* --- Calculation of the values */
    if (TimVal.I) { timval = TimVal.V; timI = TRUE; }
      else timval = timcount*timfac; /* time in seconds */
    if (MonVal.I) { monval = MonVal.V; monI = TRUE; }
      else monval = (moncount-monzero*timval) * monfac; /*#incident photons*/
    if (TrmVal.I) { trmval = TrmVal.V; trmI = TRUE; }
      else trmval = (trmcount-trmzero*timval) * trmfac; /*#transmitted photons*/
    /* --- Set anoval to number of detected photons */
    if (rejectioncorrection)
      if (AnoVal.I) { anoval = AnoVal.V; anoI = TRUE; }
        else anoval = (anocount-anozero*timval)*anofac; /*#detected photons*/
      else anoval = i1total*bincorfac;

    } else {

    /* monval */
    if (monitornormalization) {
      if (MonVal.I) monval = MonVal.V;
        else {
        if (!ReadHeaderFloat(pcb,2,num,ihb,KValueI0,&monval, pstatus)) { 
          *pstatus = KeywordMissing;
          fprintf(ERROUT,"\nMissing keyword for monnal %s\n",KValueI0); }
        if (*pstatus!=Success) return;
        }
      sprintf(monname,"%s",KValueI0);
      monI = TRUE;
    }

    /* trmval */
    if (monitornormalization) {
      if (TrmVal.I) trmval = TrmVal.V;
        else {
        if (!ReadHeaderFloat(pcb,2,num,ihb,KValueI1,&trmval, pstatus)) { 
          *pstatus = KeywordMissing;
          fprintf(ERROUT,"\nMissing keyword for trmval %s\n",KValueI1); }
        if (*pstatus!=Success) return;
        }
      sprintf(trmname,"%s",KValueI1);
      trmI = TRUE;
    }

    /* timval */
    if (monitornormalization) {
      if (TimVal.I) timval = TimVal.V;
        else {
        if (!ReadHeaderFloat(pcb,2,num,ihb,KValueTime,&timval, pstatus)) {
          *pstatus = KeywordMissing;
          fprintf(ERROUT,"\nMissing keyword for timval %s\n",KValueTime); }
        if (*pstatus!=Success) return;
        }
      sprintf(timname,"%s",KValueTime);
      timI = TRUE;
    }

    /* --- Set anoval to number of detected counts */
    if (rejectioncorrection) {
      if (AnoVal.I) anoval = AnoVal.V;
        else {       
        if (!ReadHeaderFloat(pcb,2,num,ihb,KValueAnode,&anoval, pstatus)) { 
          *pstatus = KeywordMissing; 
          fprintf(ERROUT,"\nMissing keyword for anoval %s\n",KValueAnode); } 
        if (*pstatus!=Success) return;
        }
      anoI = TRUE;
      } else anoval = i1total*bincorfac;
    sprintf(anoname,"%s",KValueAnode);

    } /* if (scaleron) */

  *pstatus = Success;

  /* --- Transmission */
  if (monitornormalization) {
    if (Trm.I) {
      trm = Trm.V;
      trmval = monval * trm;
      sprintf(trmname,"trm*mon");
      } else 
      if (trmI) {
        if (fabs(monval)>eps) trm = trmval/monval; else trm = 0.0;   
        } else {  
        trm = 1.0;
        trmval = monval;
        sprintf(trmname,"%s",monname);
        }
  } else {
     trm = 1.0;
     trmval = 1.0;
     sprintf(trmname,"%s","none");
  }

    if (pcb->TestBit) {
      printf("monitornormalization %s\n",monitornormalization?"ON":"OFF");
      printf("distancenormalization %s\n",distancenormalization?"ON":"OFF");
      printf("rejectioncorrection %s\n",rejectioncorrection?"ON":"OFF");
      if (rejectioncorrection)
        printf("anoval = %g\n", anoval);
      if (monitornormalization) {
        printf("timval = %g\n", timval);
        printf("monval = %g, trmval = %g, trm = %g\n", monval, trmval, trm);
      }
    }

  /* --- Calculation of the normalization factor */

  normok = FALSE;
  if (monitornormalization) monvaltrm=monval*trm; else monvaltrm=1.0;
  if (distancenormalization) dispix1pix2=dis*pix1*pix2; else dispix1pix2=1.0;
  divider = i1total*monvaltrm*dispix1pix2;
  if (ABS(divider)>eps) {
    normfac  = (anoval/divider);
    normok = TRUE;
  }

  if (pcb->TestBit) 
    if (distancenormalization)
      printf("i1total  = %g, normfac = %g * rd3\n",i1total, normfac);
    else
      printf("i1total  = %g, normfac = %g\n",i1total, normfac);

  /* --- Write info */

  detval     = i1total*bincorfac;

  if ( NODUMMY(i1max,I1Dummy,I1DDummy) ) {
    maxpixval = i1max*bincorfac;
    } else maxpixval = -1;

  if ( (ABS(timval)>eps) && (NODUMMY(i1max,I1Dummy,I1DDummy)) ) {
    maxpixrate = i1max*bincorfac/timval;
    } else maxpixrate = -1;

  if (ABS(timval)>eps) effcountrate = detval/timval;
    else effcountrate = -1;

  if (ABS(anoval)>eps) accepted = detval/anoval;
    else accepted = -1.0;

  if (pcb->TestBit)  {
    printf("Detector counts = %g, bincorfac = %g\n", detval,bincorfac);
    printf("accepted        = %g\n", accepted);
    if (monitornormalization)
      printf("effcountrate    = %g\n", effcountrate);
    printf("maxpixval       = %g\n", maxpixval);
    if (monitornormalization)
      printf("maxpixrate      = %g\n", maxpixrate);
    }

  if (ihb[1].Title.I) printf("\n%s\n",ihb[1].Title.V);

  if (normok) {
    printf("\n Calculating ihb[0,% d] = %9.4g * (",ihb[0].ImNum,normfac); 
      printf("ihb[% d,% d]", 1, ihb[1].ImNum);
      printf(")\n");
  } else printf("\n Calculating ihb[0,% d] = DUMMY\n",ihb[0].ImNum);
  if (monitornormalization) {
    printf(" Total number of incident photons (%12s)               = %9.3g ph\n",\
             monname, monval);
    printf(" Total number of transmitted photons (%12s)            = %9.3g ph\n",\
             trmname, trmval);
    printf(" Counting time (%12s) = %9.3g s, transmission      = %9.3g\n",\
             timname, timval, trm);
  }
  if (rejectioncorrection)
    printf(" Anode counts (%12s)  = %9.3g  , Detector counts   = %9.3g\n",\
             anoname, anoval, i1total*bincorfac); 
  else
    printf("               %12s     %9s    Detector counts   = %9.3g\n",\
             " "," ", i1total*bincorfac);
  printf(" Maximum detector counts      = %9.3g ct/pix\n",\
           maxpixval);
  if (monitornormalization)
    printf(" Maximum detector rate        = %9.3g ct/s/pix\n",\
             maxpixrate);
  if (rejectioncorrection)
    if (monitornormalization) {
      printf(" Rejection                 = %9.3g %%,\
   Effective count rate = %9.3g /s\n", (1.0-accepted)*100, effcountrate); 
    } else {
      printf(" Rejection                 = %9.3g %%\n",(1.0-accepted)*100);
    }
  else
    if (monitornormalization) {
      printf("                             %9s    Effective count rate = %9.3g /s\
   \n", " ", effcountrate); 
    }
  if (flatfieldcorrection) printf(" Division by flatfield\n");
  printf("\n");

  if (normok) {
  /* loop over the output array */
    W_2 = Wmin_2; for (i=0;i<imax;i++) f_2[i]=f1_2[i];
    for (i_2=Imin_2;i_2<=Imax_2;i_2++) {
 
      pI0Data = ABSPTR(I0Data,I0Dim_1,I0Dim_2,Imin_1,i_2);
      pE0Data = E0Data-I0Data+pI0Data;
      W_1 = Wmin_1; for (i=0;i<imax;i++) f_1[i]=f1_1[i];
      if (distancenormalization) {
        dd2  = INDEX2R(i_2,off2-cen2,pix2);
        rd20 = dis*dis + dd2*dd2;
      }
      for (i_1=Imin_1;i_1<=Imax_1;i_1++) {

        if (distancenormalization) {
          dd2 = INDEX2R(i_1,off1-cen1,pix1);
          rd2 = rd20 + dd2*dd2;
          rd3 = rd2*sqrt(rd2);
        } else rd3 = 1.0;

        factor = normfac * rd3;
        factor2 = factor*factor;

        if (flatfieldcorrection) {
          /* normalization and flatfield correction */
          // I0 = (I1/IF) * factor

          if ( E0Data ) {
            // V0 = (V1*(1/IF)^2+VF*((I1/IF)/IF)^2)*factor^2
            if ( Isum2ldwE (I1Data,E1Data,I1Dim_1,I1Dim_2,I1Dummy,I1DDummy,
                   f_1[1],f_2[1],f_1[1]+Df_1[1],f_2[1]+Df_2[1],
                   &I1Sum,&I1Weight,&E1Sum,&E1Weight) &&
                 Isum2ldwE (IFData,EFData, IFDim_1,IFDim_2,IFDummy,IFDDummy,
                   f_1[INflat], f_2[INflat], 
                   f_1[INflat]+Df_1[INflat], f_2[INflat]+Df_2[INflat], 
                   &IFSum, &IFWeight, &EFSum, &EFWeight) ) {

              /* then normalize and divide with error propagation */

              I1Value = I1Sum; if (I1Ave) I1Value /= I1Weight; 
              IFValue = IFSum; if (IFAve) IFValue /= IFWeight; 
 
              if (IFValue>eps)
                UPDATE(*pI0Data,(I1Value/IFValue)*factor,I0Dummy,I0DDummy);

              if ( (E1Sum >= 0.0)&&(EFSum >= 0.0) ) {
                E1Value = E1Sum; if (I1Ave) E1Value /= E1Weight*E1Weight;
                EFValue = EFSum; if (IFAve) EFValue /= EFWeight*EFWeight;

                if (IFValue>eps) {
                  // V0 = (V1*(1/IF)^2+VF*((I1/IF)/IF)^2)*factor^2
                  InvIFValue2 = 1/IFValue/IFValue;
                  I1InvIFValue2 = I1Value*InvIFValue2;
                  E0Value = factor2 * ( E1Value*InvIFValue2 + 
                    EFValue * I1InvIFValue2 * I1InvIFValue2 );
                  UPDATE( *pE0Data, E0Value, VarDummy, VarDDummy );
                }
              } // if E1Value ... E2Value
            } // if ISum ...
          } else {
            // no error propagation
            if ( Isum2ldw (I1Data,I1Dim_1,I1Dim_2,I1Dummy,I1DDummy,
                   f_1[1],f_2[1],f_1[1]+Df_1[1],f_2[1]+Df_2[1],
                   &I1Sum,&I1Weight) &&
                 Isum2ldw (IFData,IFDim_1,IFDim_2,IFDummy,IFDDummy,
                   f_1[INflat], f_2[INflat], f_1[INflat]+Df_1[INflat], 
                   f_2[INflat]+Df_2[INflat], 
                   &IFSum, &IFWeight) ) {

              /* then normalize and divide */
 
              I1Value = I1Sum; if (I1Ave) I1Value /= I1Weight; 
              IFValue = IFSum; if (IFAve) IFValue /= IFWeight; 
 
              if (IFValue>eps)
                UPDATE(*pI0Data,(I1Value/IFValue)*factor,I0Dummy,I0DDummy);

            } /* if Isum ... */  
          } // if E0Data ...
        } else {
          /* no flatfieldcorrection, normalization only */
          // I0 = I1 * factor

          if ( E0Data ) {
            // V0 = V1*factor^2
            if ( Isum2ldwE (I1Data,E1Data,I1Dim_1,I1Dim_2,I1Dummy,I1DDummy,
                   f_1[1],f_2[1],f_1[1]+Df_1[1],f_2[1]+Df_2[1],
                   &I1Sum,&I1Weight,&E1Sum,&E1Weight) ) {

              /* then normalize */
              I1Value = I1Sum; if (I1Ave) I1Value /= I1Weight;
              UPDATE( *pI0Data, I1Value * factor, I0Dummy, I0DDummy );

              if (E1Sum >= 0.0) {
                // V0 = V1*factor^2
                E1Value = E1Sum; if (I1Ave) E1Value /= E1Weight;
                E0Value = E1Value * factor2;
                UPDATE( *pE0Data, E0Value, VarDummy, VarDDummy );
              }
            } // if Isum ...
          } else {
            if ( Isum2ldw (I1Data,I1Dim_1,I1Dim_2,I1Dummy,I1DDummy,
                   f_1[1],f_2[1],f_1[1]+Df_1[1],f_2[1]+Df_2[1],
                   &I1Sum,&I1Weight) ) {
 
              /* then normalize */
              I1Value = I1Sum; if (I1Ave) I1Value /= I1Weight;
              UPDATE( *pI0Data, I1Value * factor, I0Dummy, I0DDummy );
 
            } /* if (Isum2ldw ... */
          }

        } /* if (flatfieldcorrection) */
 
        pI0Data++;
        pE0Data++;
        W_1+=DW_1; for (i=0;i<imax;i++) { f_1[i]+=Df_1[i]; }
      } /* for i_1 ... */
 
      W_2+=DW_2; for (i=0;i<imax;i++) { f_2[i]+=Df_2[i]; }
    } /* for i_2 ... */
  }

} /* saxs_normn*/

/*---------------------------------------------------------------------------
main
---------------------------------------------------------------------------*/

#if MAKE_FUNCTION
# define MAIN main_saxs_normn
#else
# define MAIN main
#endif

int MAIN (int argc, char *argv[])
{
  CmdBlk cb;                /* command block  */
  ImgBlk ib[BlockNum];      /* image blocks */

  int status;
  int arg_no = 0;

 /* Init options, control block and image blocks */
  InitOptions( Usage, Version, NULL, TRUE, &cb, ib, BlockNum );

 /* Define special options */
  option_define ( IO_tpflag,  SFlat,       Default_flat, &cb );
  option_define ( IO_tpflag,  SCcd,        Default_ccd,  &cb );
  option_define ( IO_tpflag,  STrmOn,      Default_trmon, &cb );
  option_define ( IO_tpflag,  SScalOn,     Default_scalon, &cb );
  option_define ( IO_tpflag,  SMonOn,      Default_monon, &cb );
  option_define ( IO_tpflag,  SDisOn,      Default_dison, &cb );

  option_define ( IO_tplvexp, SMonSca,     Default_mons, &cb );
  option_define ( IO_tplvexp, STimSca,     Default_tims, &cb );
  option_define ( IO_tplvexp, SAnoSca,     Default_anos, &cb );
  option_define ( IO_tplvexp, STrmSca,     Default_trms, &cb );
  option_define ( IO_tplvexp, SMonScaS,    Default_mo2s, &cb );
  option_define ( IO_tplvexp, STimScaS,    Default_ti2s, &cb );
  option_define ( IO_tplvexp, SAnoScaS,    Default_an2s, &cb );
  option_define ( IO_tplvexp, STrmScaS,    Default_tr2s, &cb );
  option_define ( IO_tplvexp, SScaLen,     Default_slen, &cb );

  option_define ( IO_tpflexp, STrm,        Default_trm, &cb );
  option_define ( IO_tpflexp, SScaDepth,   Default_depth , &cb );
 
  option_define ( IO_tpflexp, SMonCount,   "0.0", &cb );
  option_define ( IO_tpflexp, SMonZero,    "0.0", &cb );
  option_define ( IO_tpflexp, SMonFac,     "1.0", &cb );
  option_define ( IO_tpflexp, SMonVal,     "0.0", &cb );
  
  option_define ( IO_tpflexp, STimCount,   "0.0", &cb );
  option_define ( IO_tpflexp, STimFac,     "1.0", &cb );
  option_define ( IO_tpflexp, STimVal,     "0.0", &cb );

  option_define ( IO_tpflexp, SAnoCount,   "0.0", &cb );
  option_define ( IO_tpflexp, SAnoZero,    "0.0", &cb );
  option_define ( IO_tpflexp, SAnoFac,     "1.0", &cb );
  option_define ( IO_tpflexp, SAnoVal,     "0.0", &cb );

  option_define ( IO_tpflexp, STrmCount,   "0.0", &cb );
  option_define ( IO_tpflexp, STrmZero,    "0.0", &cb );
  option_define ( IO_tpflexp, STrmFac,     "1.0", &cb );
  option_define ( IO_tpflexp, STrmVal,     "0.0", &cb );

  option_define ( IO_tpflexp, SMonCountS,  "0.0", &cb );
  option_define ( IO_tpflexp, SMonFacS,    "1.0", &cb );

  option_define ( IO_tpflexp, STimCountS,  "0.0", &cb );
  option_define ( IO_tpflexp, STimFacS,    "1.0", &cb );

  option_define ( IO_tpflexp, SAnoCountS,  "0.0", &cb );
  option_define ( IO_tpflexp, SAnoFacS,    "1.0", &cb );

  option_define ( IO_tpflexp, STrmCountS,  "0.0", &cb );
  option_define ( IO_tpflexp, STrmFacS,    "1.0", &cb );

  option_define ( IO_tpflexp, SI1sum,      "0.0", &cb );
  option_define ( IO_tpflexp, SBinCorFac,  "1.0", &cb );
  option_define ( IO_tpflexp, SRelDis,     "0.0", &cb );

 /* Read options from argument list */
  ReadOptions( argv, &arg_no, &cb, ib, &status);

  /* USER KEYBOARD I/O */
  if (status==Success) {
    argv_start ( &cb, 1 );
    user_io( &cb, ib, &status);
    argv_end( &cb ); /* must be called after user_io */
  }

  /* SEQUENCE CALCULATION */
  if (status==Success) 
    IMAGELOOP( &cb, ib, saxs_normn, NULL, NULL, TRUE, &status );

  return( ReportSaxsStatus( status, 0 ) );

} /* MAIN */

