/***************************************************************************/
/* Written 1994++ by Peter Boesecke                                        */
/* Copyright (C) 2011 European Synchrotron Radiation Facility              */
/*                       Grenoble, France                                  */
/*                                                                         */
/*    Principal authors: Peter Boesecke  (boesecke@esrf.eu)                */
/*                                                                         */
/*    This program is free software: you can redistribute it and/or modify */
/*    it under the terms of the GNU General Public License as published by */
/*    the Free Software Foundation, either version 3 of the License, or    */
/*    (at your option) any later version.                                  */
/*                                                                         */
/*    This program is distributed in the hope that it will be useful,      */
/*    but WITHOUT ANY WARRANTY; without even the implied warranty of       */
/*    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the        */
/*    GNU General Public License for more details.                         */
/*                                                                         */
/*    You should have received a copy of the GNU General Public License    */
/*    along with this program.  If not, see <http://www.gnu.org/licenses/>.*/
/***************************************************************************/
/*+++
1 saxs_i1otxt
  Reads from a saxs input sequence and writes to a text file (template)

2 PURPOSE
  Reads from a saxs input sequence and writes to a text file (template)
  To construct the name of the output text file the saxs-file extension 
  is replaced by ".txt"

  Calculation:       I0 = I1
  Error propagation: V0 = V1

  Arguments:
  saxs_i1otxt [options] <i1nam> <onam> <i1fst> <i1lst> <i1inc> 
             <odum> <odim1> <odim2> <ofac> <ocon>"

  Defaults:
  <input file name>   : input.edf 
  <output file name>  : output.edf
  <first image>       : <first image number in input file>
  <last image>                              : <last image number in input file>
     if argument list ends with first image : <first image>
  <increment>         : 1
  <dummy>             : <dummy value in first image of input file>
  <dimension 1>       : <horizontal dimension of first image in input file>
  <dimension 2>       : <vertical dimension of first image in input file>
  <output const>      : 0.0
  <output factor>     : 1.0

  Wildcard:
  A "=" in the argument list chooses the default.

2 HISTORY
  2003-06-16  PB  V4.50 
  2003-11-30  PB  V4.51 BinSiz
  2004-10-31  PB  V4.52 MAKE_FUNCTION
  2005-07-15  PB  V4.57 ReportSaxsStatus parameter list updated
  2005-09-17  PB  V4.58 VarDat
  2007-06-11  PB  V4.59 ImageLoop parameters updated
  2008-12-05  PB  V4.60 open all files as binary file to avoid
                        translation of \r\n to \n when compiled on windows.
  2012-01-31  PB  V4.61 ImageLoop parameters updated

---*/
# define Version  "saxs_i1otxt V4.61 2012-01-31, Peter Boesecke"

# include <errno.h>
# include <stdio.h>
# include <unistd.h>
# include <fcntl.h>

# include "SaxsPrograms.h"

/* Defaults */
# define EXTENSION  ".txt"

# define Usage "[options] \n\
                <i1nam> <onam> <i1fst> <i1lst> <i1inc> \n\
                <odum> <odim1> <odim2> <ofac> <ocon>"

# define BlockNum 2       /* 1 input sequence + 1 output sequence */

/*---------------------------------------------------------------------------
1 saxs_i1otxt

2 PURPOSE
  Multiplicates an image with a factor and adds a constant
---------------------------------------------------------------------------*/
void saxs_i1otxt (CmdBlk * pcb, long num[], ImgHeadBlk ihb[], int * pstatus)
{ const float eps=1e-32;
  int i,imax;
  int j;

  float *I0Data, *E0Data; 
  int   I0Dim_1, I0Dim_2;
  float *pI0Data, *pE0Data;
  float I0Dummy, I0DDummy;
  float VarDDummy=DDSET(VarDummy);

  int   I1Ave;
  float *I1Data, *E1Data;
  int   I1Dim_1, I1Dim_2;
  float I1Dummy, I1DDummy;
  float I1Value, I1Sum, I1Weight;
  float E1Value, E1Sum, E1Weight;

  int i_1, i_2;
  float f_1[BlockNum], f_2[BlockNum];

  float Off_1[BlockNum], Off_2[BlockNum];
  float Ps_1[BlockNum], Ps_2[BlockNum];


  float f1_1[BlockNum], f3_1[BlockNum], Df_1[BlockNum];
  float f1_2[BlockNum], f3_2[BlockNum], Df_2[BlockNum]; 

  int Imin_1, Imin_2, Imax_1, Imax_2;
 
  float Wmin_1, Wmax_1, DW_1;
  float Wmin_2, Wmax_2, DW_2;

  /* output file */
  char fname_full[IO_size];
  FILE *OutputFileId;

  *pstatus = Success;

  imax = pcb->ImgBlkLen;
  printf("\n Calculating ihb[0,% d] = Function(",ihb[0].ImNum);
  for(i=1;i<imax;i++) {
    printf("ihb[% d,% d] ", i, ihb[i].ImNum); }

  printf(")\n\n");

 /* Check the number of images */
  if (pcb->ImgBlkLen!=2) {
     printf("%d images found, 1 input and 1 output image required\n",
             pcb->ImgBlkLen); *pstatus=Failed; return; }

  GetReferenceParameters( pcb, ihb, 0, imax-1,
                          Off_1, Off_2, Ps_1,  Ps_2, pstatus );
  if (*pstatus!=Success) return;

  UpdateImageParameters ( pcb, ihb, 1, imax-1,
                          Off_1, Off_2, Ps_1,  Ps_2, pstatus );
  if (*pstatus!=Success) return;

  GetImageRange         ( pcb, ihb, 1, imax-1,
                          Off_1, Off_2, Ps_1,  Ps_2,
                          f1_1, f3_1, Df_1, f1_2, f3_2, Df_2,
                          &Imin_1, &Imax_1, &Imin_2, &Imax_2,
                          &Wmin_1, &Wmax_1, &DW_1, &Wmin_2, &Wmax_2, &DW_2, 1);

 /* 1 input, 1 output */
  I0Data  = ihb[0].Data;
  E0Data = ihb[0].VarDat;
  I0Dummy = ihb[0].Dummy.V;
  I0DDummy = ihb[0].DDummy.V;
  I0Dim_1  = (int) ihb[0].Dim[1];
  I0Dim_2  = (int) ihb[0].Dim[2];

  I1Ave   = pcb->ib[1].Ave.V;
  I1Data  = ihb[1].Data;
  E1Data  = ihb[1].VarDat;
  I1Dummy = ihb[1].Dummy.V;
  I1DDummy = ihb[1].DDummy.V;
  I1Dim_1  = (int) ihb[1].Dim[1];
  I1Dim_2  = (int) ihb[1].Dim[2];

  /* loop over the output array  */
  for (i=0;i<imax;i++) f_2[i]=f1_2[i];
  for (i_2=Imin_2;i_2<=Imax_2;i_2++) {

    pI0Data = ABSPTR(I0Data,I0Dim_1,I0Dim_2,Imin_1,i_2);
    pE0Data = E0Data-I0Data+pI0Data;
    for (i=0;i<imax;i++) f_1[i]=f1_1[i];
    for (i_1=Imin_1;i_1<=Imax_1;i_1++) {
      // I0 = I1

      if ( E0Data ) {
        // V0 = V1
        if ( Isum2ldwE(I1Data,E1Data,I1Dim_1,I1Dim_2,I1Dummy,I1DDummy,
               f_1[1], f_2[1], f_1[1]+Df_1[1], f_2[1]+Df_2[1],
               &I1Sum, &I1Weight, &E1Sum, &E1Weight) ) {
          /* then do something with the data */
          I1Value = I1Sum; if (I1Ave) I1Value /= I1Weight;
          UPDATE( *pI0Data, I1Value, I0Dummy, I0DDummy );
 
          if (E1Sum >= 0.0) {
            E1Value = E1Sum; if (I1Ave) E1Value /= E1Weight*E1Weight;
            UPDATE( *pE0Data, E1Value, VarDummy, VarDDummy );
          }
        }
      } else {
        if ( Isum2ldw (I1Data,I1Dim_1,I1Dim_2,I1Dummy,I1DDummy,
               f_1[1], f_2[1], f_1[1]+Df_1[1], f_2[1]+Df_2[1],
               &I1Sum, &I1Weight) ) {
          /* then do something with the data */
          I1Value = I1Sum; if (I1Ave) I1Value /= I1Weight;
          UPDATE( *pI0Data, I1Value, I0Dummy, I0DDummy );
        }
      }
      pI0Data++;
      pE0Data++;
      for (i=0;i<imax;i++) f_1[i]+=Df_1[i];

    } /* for i_1 ... */

    for (i=0;i<imax;i++) f_2[i]+=Df_2[i];
  } /* for i_2 ... */

// open output text file

  /* Generate filename of ASCII file */
  filename_body ( fname_full, IO_size, pcb->ib[0].FileName ); // remove extension
  if ( filename_compare ( fname_full, DUMMYFILENAME ) )
    sprintf( fname_full,"%s%s",fname_full,EXTENSION); // add extension

  /*--- Open output file
    If this is the first image of the sequence, an existing file
    with the same name must be deleted (open mode = "w"),
    in all other cases it must be open to append a line (open mode "a"). */

  if (pcb->TestBit) printf("Opening output file %s\n",fname_full);
  *pstatus = FileOpeningError;
  if (ImageIsFirst( pcb, 0, ihb )) {
    /* 1st image: new write */
    if ((OutputFileId=fopen(fname_full,"wb")) == NULL) return;
    } else {
    /* appending */
    if ((OutputFileId=fopen(fname_full,"ab")) == NULL) return;
    }

// process image start

   /* HINTS
      The subroutine should be called with at least the following parameters:

      subroutine ( FILE * OutputFileId, 
                   float data[], float vardat[], int dim_1, int dim_2, 
                   float off_1, float off_2, float ps_1, float ps_2,
                   float dummy, float ddummy, int * pstatus);

      inside subroutine:

      - pointer to data[i,j]: ABSPTR(I0Data,I0Dim_1,I0Dim_2,i,j); (0<=i<dim)

      - pointer to vardat[i,j]: ABSPTR(E0Data,I0Dim_1,I0Dim_2,i,j); (0<=i<dim)
        It must be checked that vardat!=NULL, otherwise vardat must be ignored.

      - check for dummy with: if (DUMMY( value, I0Dummy, I0DDummy)) then skip it

      - for interpolation and consistency, the output data should refer to 
        pixel coordinates, e.g.
        data[i,j]: lower left corner  : coordinate (i,j), 
                   pixel center       : coordinate (i+0.5, j+0.5)
                   upper right corner : coordinate (i+1, j+1)
        The pixel coordinate range of an dim_1 X dim_2 image is
          (0..dim_1,0..dim_2)

      For consistency, the macro WORLD should be used to output coordinates
      (offsets are taken into account), e.g. 
      WORLD(i,off_1,ps_1) to output horizontal coordinate (index i)
      WORLD(j,off_2,ps_2) to output vertical coordinate (index j)

      ATTENTION:
      Pixel numbers (i,j) are meaningless because they depend on the
      programming language. Therefore, do not output pixel numbers i 
      and/or j directly without conversion to pixel coordinates!

      CALL
      subroutine( OutputFileId, 
                  I0Data, E0Data, I0Dim_1, I0Dim_2,
                  Off_1[0], Off_2[0], Ps_1[1], Ps_2[0],
                  I0Dummy, I0DDummy, pstatus );
              
   */

// proces image end

  /*--- Write to the output file */
  if (ImageIsFirst( pcb, 0, ihb )) {
    /* 1st image: write header line */
    fprintf(OutputFileId,"First line\r\n");
  }

  /*--- Write data line */
  printf("Writing data to %s\n",fname_full);
  fprintf(OutputFileId,"line\r\n");

// close output text file
  if (pcb->TestBit) printf("Closing file %s\n",fname_full);
  *pstatus = FileClosingError;
  if (fclose(OutputFileId)!=0) return;

  *pstatus = Success;

} /* saxs_i1otxt*/

/*---------------------------------------------------------------------------
user_io
Do all the keyboard io and return cb, and ib
---------------------------------------------------------------------------*/
void user_io(CmdBlk * pcb, ImgBlk * ib, int * pstatus)
{
  char  progfn[IO_size];
  ImgHeadBlk ihb[BlockNum];

  float ROff_1, RPs_1, UOff_1, UPs_1;
  float ROff_2, RPs_2, UOff_2, UPs_2;

 /* Determine program name without directory */
   (void) RightFR((char *) pcb->argv[0],DirSeparator,progfn,IO_size);

  /* show usage if no arguments are given */
  if (pcb->argc<=1) printf("Usage: %s %s\n",progfn,Usage);

  /*--- Write name of program ---*/
  printf("\n");
  printf("   %s %s\n",progfn,Version);
  printf("\n");

  ArgvFilenames ( pcb, ib, ihb, 0, BlockNum-1, pstatus);
  if (*pstatus!=Success) return;
  GetReference(pcb->RSys.V,1,ihb,&ROff_1,&ROff_2,&RPs_1,&RPs_2,pstatus );
  if (*pstatus!=Success) return;
  GetReference(pcb->USys.V,1,ihb,&UOff_1,&UOff_2,&UPs_1,&UPs_2,pstatus );
  if (*pstatus!=Success) return;

  /*--- Argument  : multiplication factor */
  printf("<output image> = <input image> * Factor + Const\n");
  argv_flexp(pcb,"Multiplication factor",&ib[0].Factor,"1.0",pstatus);
  if (*pstatus!=Success) return;

  /*--- Argument  : additive constant */
  printf("<output image> = <input image> * Factor + Const\n");
  argv_flexp(pcb,"Addition constant",&ib[0].Const,"0.0",pstatus);
  if (*pstatus!=Success) return;

  printf("\n");
  if (ib[1].Name.I)    printf("i/p file           : %s\n",ib[1].Name.V);
  if (ib[0].Name.I)    printf("o/p file           : %s\n",ib[0].Name.V);
  if (ib[1].First.I)   printf("first image        : %d\n",ib[1].First.V);
  if (ib[1].Last.I)    printf("last image         : %d\n",ib[1].Last.V);
  if (ib[1].Inc.I)     printf("increment          : %d\n",ib[1].Inc.V);
  if (ib[0].Dummy.I)   printf("output dummy       : %s\n",ib[0].Dummy.V);
  if (ib[0].Dim[1].I)  printf("output dimension 1 : %s\n",ib[0].Dim[1].V);
  if (ib[0].Dim[2].I)  printf("output dimension 2 : %s\n",ib[0].Dim[2].V);
  if (ib[0].Factor.I)  printf("factor             : %s\n",ib[0].Factor.V);
  if (ib[0].Const.I)   printf("constant           : %s\n",ib[0].Const.V);
  printf("\n");

  if (pcb->TestBit) {
    PrintBlocks ( pcb, ib );
    printf("\n"); }

  return;
}

/*---------------------------------------------------------------------------
main
---------------------------------------------------------------------------*/

#if MAKE_FUNCTION
# define MAIN main_saxs_i1otxt
#else
# define MAIN main
#endif

int MAIN (int argc, char *argv[])
{
  CmdBlk cb;                /* command block  */
  ImgBlk ib[BlockNum];      /* image blocks */

  int status;
  int arg_no = 0;

 /* Init options, control block and image blocks */
  InitOptions( Usage, Version, NULL, TRUE, &cb, ib, BlockNum );

 /* Read options from argument list */
  ReadOptions( argv, &arg_no, &cb, ib, &status);

  /*--- OpenMode of output file */
  if (!ib[0].OpenMode.I) {
    ib[0].OpenMode.V = IO_ImageProtect | IO_DontReadWrite | IO_DontOpen;
    ib[0].OpenMode.I = TRUE;
  }

  /* USER KEYBOARD I/O */
  if (status==Success) {
    argv_start ( &cb, 1 );
    user_io( &cb, ib, &status);
    argv_end( &cb ); /* must be called after user_io */
  }

  /* SEQUENCE CALCULATION */
  if (status==Success) 
    IMAGELOOP( &cb, ib, saxs_i1otxt, NULL, NULL, TRUE, &status );

  return( ReportSaxsStatus( status, 0 ) );

} /* MAIN */

