/***************************************************************************/
/* Written 1994++ by Peter Boesecke                                        */
/* Copyright (C) 2011 European Synchrotron Radiation Facility              */
/*                       Grenoble, France                                  */
/*                                                                         */
/*    Principal authors: Peter Boesecke  (boesecke@esrf.eu)                */
/*                                                                         */
/*    This program is free software: you can redistribute it and/or modify */
/*    it under the terms of the GNU General Public License as published by */
/*    the Free Software Foundation, either version 3 of the License, or    */
/*    (at your option) any later version.                                  */
/*                                                                         */
/*    This program is distributed in the hope that it will be useful,      */
/*    but WITHOUT ANY WARRANTY; without even the implied warranty of       */
/*    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the        */
/*    GNU General Public License for more details.                         */
/*                                                                         */
/*    You should have received a copy of the GNU General Public License    */
/*    along with this program.  If not, see <http://www.gnu.org/licenses/>.*/
/***************************************************************************/
/*+++
1 saxs_filter
  Removes spurious noise from detector data.

2 PURPOSE
  Removes spurious noise from detector data.

  Calculation:       I0 = I1
  Error propagation: V0 = V1   

  Arguments:
  saxs_filter [options] <i1nam> <onam> <i1fst> <i1lst> <i1inc> 
             <odum> <odim1> <odim2> <ofac> <ocon>"

  Defaults:
  <input file name>   : input.edf 
  <output file name>  : output.edf
  <first image>       : <first image number in input file>
  <last image>                              : <last image number in input file>
     if argument list ends with first image : <first image>
  <increment>         : 1
  <dummy>             : <dummy value in first image of input file>
  <dimension 1>       : <horizontal dimension of first image in input file>
  <dimension 2>       : <vertical dimension of first image in input file>
  <output const>      : 0.0
  <output factor>     : 1.0

  Wildcard:
  A "=" in the argument list chooses the default.

2 HISTORY
  2007-06-28 PB V4.61 from saxs_mac V4.60 and VPF_PEAKFILTER 27-JUL-1990
  2007-06-29 PB V4.62 Option max added + debugging
  2012-01-31 PB V4.63 ImageLoop parameters updated

---*/
# define Version  "saxs_filter V4.63 2012-01-31, Peter Boesecke"

# include <errno.h>
# include <stdio.h>
# include <unistd.h>
# include <fcntl.h>

# include "SaxsPrograms.h"

# define Usage "[options] \n\
                <i1nam> <onam> <i1fst> <i1lst> <i1inc> \n\
                <odum> <odim1> <odim2> <w> <wa> <a> <rmsk> <chi2> <max>"

# define BlockNum 2       /* 1 input sequence + 1 output sequence */

/* num options */
# define SMrad  "rmsk"  // radius of mask
# define SMode  "mode"  // mode 0: set to dummy, 1: set to average

/* line options */
# define SMname  "mnam"  // mask file name

/* flag options */
# define SMfile  "msk"  // mask file flag 

/* float options */
# define SChi2   "chi2"  // noise threshold
# define SMax    "max"   // maximum difference between pixel and mask value 
# define SW      "w"     // gaussian width
# define SWa     "wa"    // dip width
# define SA      "a"     // a

/*---------------------------------------------------------------------------
NAME

  gen_tring_mask

PURPOSE

  Calculates a two dimensional gaussian with fwhm-width multiplied 
  by the function (1.0-a*pow(2,(-16.0*(r^4/widtha^4)) which is zero in the
  center if a=1. The sum of the generated data points is normalized to 1.
  A should be positive.

  (1/mean) * exp(-r^2 * /(2*sigma^2)) * (1-a*2^(-16*r^4/widtha^4)

    where 1/(2*sigma^2) = 4*log(2)/width^2

  The sum of the result is normalized to 1.

ARGUMENTS

   float buffer[] : pointer to a buffer with dimension dim * dim
   long dim       : linear dimension
   float width    : fwhm width of the gaussian
   float a        : weight factor (default 1)
   float widtha   :

   return value   : pointer to modified buffer
---------------------------------------------------------------------------*/
float *gen_tring_mask( float buffer[], long dim_1, long dim_2,
                       float width, float a, float widtha )
{ float eps=1e-32;
  float *msk;

  long i_1, i_2, i, k_1, k_2, r2;
  float yfac, h, sum;
  float ha;

  if (!buffer) return (NULL);

  if ( (dim_1==1) && (dim_2==1) ) {
    msk = ABSPTR(buffer,dim_1,dim_2,0,0);
    *msk=1.0;
  } else {
    sum=0.0;

    h=width*width;
    h=4.0*log(2.0)/h;
    ha=widtha*widtha*widtha*widtha;
    ha=16.0*log(2.0)/ha;

    k_1=dim_1/2;
    k_2=dim_2/2;

    for (i_2=0;i_2<dim_2;i_2++) {
      for (i_1=0;i_1<dim_1;i_1++) {
        msk = ABSPTR(buffer,dim_1,dim_2,i_1,i_2);
        r2 = (k_2-i_2)*(k_2-i_2)+(k_1-i_1)*(k_1-i_1); 
        *msk = exp(-r2*h)*(1.0-a*exp(-r2*r2*ha));
        sum += *msk;
      }
    }

    msk = ABSPTR(buffer,dim_1,dim_2,0,0);
    if (fabs(sum)>eps) {
      yfac=(dim_1*dim_2)/sum;
      for (i_2=0;i_2<dim_2;i_2++) {
        for (i_1=0;i_1<dim_1;i_1++) {
          *msk++ *= yfac;
        }
      }
    } 
  }

  return ( buffer );

} // gen_tring_mask

/*---------------------------------------------------------------------------
NAME

  peakfilter
  
PURPOSE

ARGUMENTS

  float out[]         : output array
  float in[]          : input array
  long dim_1, dim_2   : array dimensions of in and out
  float dummy
  float msk[]
  long mdim_1, mdim_2
  float chi2          : OK, if (<I>-I)^2/(<I^2>-<I>^2) <= chi2
  float max           : OK, if fabs(<I>-I) <= max
                        Pixel is rejected if none of the two conditions
                        are fulfilled
  long mode           : 0: set to dummy, 1: set to average
  long debug          : debug flag
  float *changed      : returns number of changed pixels

  return value float* : pointer to output array

---------------------------------------------------------------------------*/
float *peakfilter( float out[], float in[], long dim_1, long dim_2,
                   float dummy, float ddummy,
                   float msk[], long mdim_1, long mdim_2,
                   float chi2, float max, int mode, int debug, long *pchanged )
{
  long i_1, i_2, mcen_1, mcen_2, j_1, j_2;
  float sumi, sumi2, sumw, j12_value, msk_value, xi, xi2;
  float mwi, mwi2, diffwi, diffwi2, diffmwi2, in_value, value;
  float eps=1e-32;

  float *pout, *pin, *pmsk;

  long changed=0;

  if (pchanged) *pchanged=changed; 

  if ((!out)||(!in)) return(NULL);

  mcen_1=mdim_1/2;
  mcen_2=mdim_2/2;

  for (i_2=0;i_2<dim_2;i_2++) {
    for (i_1=0;i_1<dim_1;i_1++) {

      sumw=0.0;
      sumi=0.0;
      sumi2=0.0;

      pin = ABSPTR(in,dim_1,dim_2,i_1,i_2);
      in_value = *pin;

      // preset output value with in_value
      value = in_value;

      if ( ( NODUMMY(in_value, dummy, ddummy) )&&( msk ) ) {
        pmsk = ABSPTR(msk,mdim_1,mdim_2,0,0);

        for (j_2=i_2-mcen_2;j_2<=i_2+mcen_2;j_2++) {
          for (j_1=i_1-mcen_1;j_1<=i_1+mcen_1;j_1++) {

            if ((j_1>=0)&&(j_1<dim_1)&&(j_2>=0)&&(j_2<dim_2)) {
              pin = ABSPTR(in,dim_1,dim_2,j_1,j_2);
              j12_value=*pin;
              msk_value=*pmsk;

              if ( NODUMMY(j12_value, dummy, ddummy) ) {
                sumi+=j12_value*msk_value;
                sumi2+=j12_value*j12_value*msk_value;
                sumw+=msk_value;
              }
            } // if ...
            pmsk++;
          } // j_1
        } // j_2

        if (sumw>=eps) {

          // after dark level subtraction
          // a better measure could be (mwi-I)^2/mwi^2 < thresh^2

          mwi = sumi/sumw;
          mwi2 = sumi2/sumw;
          diffmwi2 = mwi2-mwi*mwi;
          diffwi = mwi-in_value;
          diffwi2 = diffwi*diffwi;

          if ((fabs(diffwi2)>eps)&&(fabs(diffmwi2)>eps)) {
            xi2 = diffwi2/diffmwi2;
          } else {
            xi2 = 0;
          }

          if ((xi2>chi2)&&(fabs(diffwi)>max)) {

            // subtract bad pixel value from sum
            pmsk = ABSPTR(msk,mdim_1,mdim_2,mcen_1,mcen_2);
            msk_value = *pmsk;
            sumi-=msk_value*in_value;
            sumw-=msk_value;

            if (mode==1) {
              if (sumw>=eps) { value=sumi/sumw; } else { value=dummy; }
            } else value=dummy;

            if (debug>4) {
              printf("BAD VALUE DETECTED IN [%d,%d]: XI2=%g, I=%g, (<I>-I)^2=%g,  (<I^2>-<I>*<I>)=%g\n",
                  i_1,i_2,xi2,in_value,diffwi2,diffmwi2);
            }
            changed++;
          }
        } 
      } // if (msk)
      pout = ABSPTR(out,dim_1,dim_2,i_1,i_2);
      *pout = value;
    }
  }

  if (pchanged) *pchanged=changed;

  return ( out );

} // peakfilter

/*---------------------------------------------------------------------------
1 saxs_filter

2 PURPOSE
  Multiplicates an image with a factor and adds a constant
---------------------------------------------------------------------------*/
void saxs_filter (CmdBlk * pcb, long num[], ImgHeadBlk ihb[], int * pstatus)
{ const float eps=1e-32;
  int i,imax;
  int j;

  float *I0Data, *E0Data;
  int   I0Dim_1, I0Dim_2;
  float *pI0Data, *pE0Data;
  float I0Dummy, I0DDummy;
  float VarDDummy=DDSET(VarDummy);

  int   I1Ave;
  float *I1Data, *E1Data;
  int   I1Dim_1, I1Dim_2;
  float I1Dummy, I1DDummy;
  float I1Value, I1Sum, I1Weight;
  float E1Value, E1Sum, E1Weight;

  int i_1, i_2;
  float f_1[BlockNum], f_2[BlockNum];

  float Off_1[BlockNum], Off_2[BlockNum];
  float Ps_1[BlockNum], Ps_2[BlockNum];

  float f1_1[BlockNum], f3_1[BlockNum], Df_1[BlockNum];
  float f1_2[BlockNum], f3_2[BlockNum], Df_2[BlockNum]; 

  int Imin_1, Imin_2, Imax_1, Imax_2;
 
  float Wmin_1, Wmax_1, DW_1;
  float Wmin_2, Wmax_2, DW_2;

  IO_line Mname;
  IO_long Mfile, Mrad, Mode;
  IO_float Chi2, Max, W, Wa, A;

  long *mdim=NULL;
  size_t buflen, mbuflen;
  float *buffer=NULL, *mbuffer=NULL, *msk=NULL;

  int mskstream;
  int ErrorValue;

  long changed;

  *pstatus = Success;

  imax = pcb->ImgBlkLen;
  printf("\n Calculating ihb[0,% d] = Function(",ihb[0].ImNum);
  for(i=1;i<imax;i++) {
    printf("ihb[% d,% d] ", i, ihb[i].ImNum); }

  printf(")\n");

 /* Check the number of images */
  if (pcb->ImgBlkLen!=2) {
     fprintf(ERROUT,"%d images found, 1 input and 1 output image required\n",
             pcb->ImgBlkLen); *pstatus=Failed; return; }

  GetReferenceParameters( pcb, ihb, 0, imax-1,
                          Off_1, Off_2, Ps_1,  Ps_2, pstatus );
  if (*pstatus!=Success) return;

  UpdateImageParameters ( pcb, ihb, 1, imax-1,
                          Off_1, Off_2, Ps_1,  Ps_2, pstatus );
  if (*pstatus!=Success) return;

  GetImageRange         ( pcb, ihb, 1, imax-1,
                          Off_1, Off_2, Ps_1,  Ps_2,
                          f1_1, f3_1, Df_1, f1_2, f3_2, Df_2,
                          &Imin_1, &Imax_1, &Imin_2, &Imax_2,
                          &Wmin_1, &Wmax_1, &DW_1, &Wmin_2, &Wmax_2, &DW_2, 1);

 /* --- Get option values */
  Mfile = option_flag ( SMfile, pcb, pstatus );
  if (*pstatus!=Success) return;
  Mname = option_line ( SMname, pcb, ihb, num, 1, pstatus );
  if (*pstatus!=Success) return;

  Mrad = option_long ( SMrad, pcb, ihb, num, 1, pstatus );
  if (*pstatus!=Success) return;
  Mode = option_long ( SMode, pcb, ihb, num, 1, pstatus );
  if (*pstatus!=Success) return;
  Chi2 = option_float ( SChi2, pcb, ihb, num, 1, pstatus );
  if (*pstatus!=Success) return;
  Max = option_float ( SMax, pcb, ihb, num, 1, pstatus );
  if (*pstatus!=Success) return;
  W = option_float ( SW, pcb, ihb, num, 1, pstatus );
  if (*pstatus!=Success) return;
  Wa = option_float ( SWa, pcb, ihb, num, 1, pstatus );
  if (*pstatus!=Success) return;
  A = option_float ( SA, pcb, ihb, num, 1, pstatus );
  if (*pstatus!=Success) return;

 /* generate mask */
  mdim = (long*) malloc( sizeof(long)*3 );
  mdim[0] = 2;
  mdim[1] = 2*fabs(Mrad.V)+1; 
  mdim[2] = 2*fabs(Mrad.V)+1;
  mbuflen = sizeof(float)*mdim[1]*mdim[2];
  if (mbuflen>0) {
    mbuffer = (float*) malloc( mbuflen );
    msk = gen_tring_mask( mbuffer, mdim[1], mdim[2], W.V, A.V, Wa.V );
  }

  if (Mfile.V) {
    mskstream = edf_open_data_file ( Mname.V, "new", &ErrorValue, pstatus );
    SetSaxsErrorExternal( ErrorValue, ReportSaxsImageError );
    if (*pstatus!=Success) { 
      if (mdim) free (mdim);
      if (mbuffer) free ( mbuffer ); 
      return; 
    }
    edf_write_data( mskstream,ihb[0].ImNum,1l,mdim,msk,MFloat,
                    &ErrorValue, pstatus );
    SetSaxsErrorExternal( ErrorValue, ReportSaxsImageError );
    if (*pstatus!=Success) { 
      if (mdim) free (mdim);
      if (mbuffer) free ( mbuffer ); 
      return; 
    }
    edf_close_data_file ( mskstream, &ErrorValue, pstatus );
    SetSaxsErrorExternal( ErrorValue, ReportSaxsImageError );
    if (*pstatus!=Success) { 
      if (mdim) free (mdim);
      if (mbuffer) free ( mbuffer ); 
      return; 
    }
  }

 /* allocate and preset temporary output array */
  buflen = sizeof(float)*ihb[0].Dim[1]*ihb[0].Dim[2];
  buffer = (float*) malloc( buflen );
  pI0Data = buffer;
  for (i_2=0;i_2<ihb[0].Dim[2];i_2++)
    for (i_1=0;i_1<ihb[0].Dim[1];i_1++)
      *pI0Data++ = ihb[0].Dummy.V; 

 /* 1 input, 1 output */
  I0Data  = buffer;
  E0Data = ihb[0].VarDat;
  I0Dummy = ihb[0].Dummy.V;
  I0DDummy = ihb[0].DDummy.V;
  I0Dim_1  = (int) ihb[0].Dim[1];
  I0Dim_2  = (int) ihb[0].Dim[2];

  I1Ave   = pcb->ib[1].Ave.V;
  I1Data  = ihb[1].Data;
  E1Data  = ihb[1].VarDat;
  I1Dummy = ihb[1].Dummy.V;
  I1DDummy = ihb[1].DDummy.V;
  I1Dim_1  = (int) ihb[1].Dim[1];
  I1Dim_2  = (int) ihb[1].Dim[2];

  /* loop over the output array  */
  for (i=0;i<imax;i++) f_2[i]=f1_2[i];
  for (i_2=Imin_2;i_2<=Imax_2;i_2++) {

    pI0Data = ABSPTR(I0Data,I0Dim_1,I0Dim_2,Imin_1,i_2);
    pE0Data = E0Data-I0Data+pI0Data;
    for (i=0;i<imax;i++) f_1[i]=f1_1[i];
    for (i_1=Imin_1;i_1<=Imax_1;i_1++) {

      if ( E0Data ) {
        // V0 = V1
        if ( Isum2ldwE(I1Data,E1Data,I1Dim_1,I1Dim_2,I1Dummy,I1DDummy,
               f_1[1], f_2[1], f_1[1]+Df_1[1], f_2[1]+Df_2[1], 
               &I1Sum, &I1Weight, &E1Sum, &E1Weight) ) {
          /* then do something with the data */
          I1Value = I1Sum; if (I1Ave) I1Value /= I1Weight;
          UPDATE( *pI0Data, I1Value, I0Dummy, I0DDummy );

          if (E1Sum >= 0.0) {
            E1Value = E1Sum; if (I1Ave) E1Value /= E1Weight*E1Weight;
            UPDATE( *pE0Data, E1Value, VarDummy, VarDDummy );
          }
        }
      } else {
        if ( Isum2ldw (I1Data,I1Dim_1,I1Dim_2,I1Dummy,I1DDummy,
               f_1[1], f_2[1], f_1[1]+Df_1[1], f_2[1]+Df_2[1], 
               &I1Sum, &I1Weight) ) {
          /* then do something with the data */
          I1Value = I1Sum; if (I1Ave) I1Value /= I1Weight;
          UPDATE( *pI0Data, I1Value, I0Dummy, I0DDummy );
        }
      }

      pI0Data++;
      pE0Data++;
      for (i=0;i<imax;i++) f_1[i]+=Df_1[i];

    } /* for i_1 ... */

    for (i=0;i<imax;i++) f_2[i]+=Df_2[i];
  } /* for i_2 ... */

  // filter temporary array and write result to output array
  // to accelerate Imin_1, Imax_1, Imin_2, Imax_2 should be included
  peakfilter( ihb[0].Data, I0Data, I0Dim_1, I0Dim_2, I0Dummy, I0DDummy,
              msk, mdim[1], mdim[2], Chi2.V, Max.V, Mode.V, pcb->TestBit, &changed );

  printf(" (%ld pixels changed)\n",changed);

  if (buffer) free ( buffer );
  if (mbuffer) free ( mbuffer );
  if (mdim) free (mdim);

} /* saxs_filter*/

/*---------------------------------------------------------------------------
user_io
Do all the keyboard io and return cb, and ib
---------------------------------------------------------------------------*/
void user_io(CmdBlk * pcb, ImgBlk * ib, int * pstatus)
{
  char  progfn[IO_size];
  ImgHeadBlk ihb[BlockNum];

  float ROff_1, RPs_1, UOff_1, UPs_1;
  float ROff_2, RPs_2, UOff_2, UPs_2;

  IO_lvexp *pMrad, *pMode;
  IO_flexp *pW, *pWa, *pA, *pChi2, *pMax;

 /* Determine program name without directory */
   (void) RightFR((char *) pcb->argv[0],DirSeparator,progfn,IO_size);

  /* show usage if no arguments are given */
  if (pcb->argc<=1) printf("Usage: %s %s\n",progfn,Usage);

  /*--- Write name of program ---*/
  printf("\n");
  printf("   %s %s\n",progfn,Version);
  printf("\n");

  ArgvFilenames ( pcb, ib, ihb, 0, BlockNum-1, pstatus);

  /*--- Argument  : width */
  pW = (IO_flexp*) option_parameter_search( SW, pcb, pstatus );
  if (*pstatus!=Success) return;
  argv_flexp(pcb,"Width (w) [pixel]", pW,pW->V,pstatus);
  if (*pstatus!=Success) return;

  /*--- Argument  : widtha */
  pWa = (IO_flexp*) option_parameter_search( SWa, pcb, pstatus );
  if (*pstatus!=Success) return;
  argv_flexp(pcb,"Dip width (wa) [pixel]", pWa,pWa->V,pstatus);
  if (*pstatus!=Success) return;

  /*--- Argument  : a */
  pA = (IO_flexp*) option_parameter_search( SA, pcb, pstatus );
  if (*pstatus!=Success) return;
  argv_flexp(pcb,"Relative deepness (a) (0..1)", pA,pA->V,pstatus);
  if (*pstatus!=Success) return;

  /*--- Argument  : msk */
  pMrad = (IO_lvexp*) option_parameter_search( SMrad, pcb, pstatus );
  if (*pstatus!=Success) return;
  argv_lvexp(pcb,"Mask radius in pixels (rmsk)", pMrad,pMrad->V,pstatus);
  if (*pstatus!=Success) return;

  /*--- Argument  : threshould */
  pChi2 = (IO_flexp*) option_parameter_search( SChi2, pcb, pstatus );
  if (*pstatus!=Success) return;
  argv_flexp(pcb,"Noise threshold (chi2)", pChi2,pChi2->V,pstatus);
  if (*pstatus!=Success) return;

  /*--- Argument  : maximum difference */
  pMax = (IO_flexp*) option_parameter_search( SMax, pcb, pstatus );
  if (*pstatus!=Success) return;
  argv_flexp(pcb,"Absolute threshold (max)", pMax,pMax->V,pstatus);
  if (*pstatus!=Success) return;

  printf("\n");
  if (ib[1].Name.I)   printf("i/p file                  : %s\n",ib[1].Name.V);
  if (ib[0].Name.I)   printf("o/p file                  : %s\n",ib[0].Name.V);
  if (ib[1].First.I)  printf("first image               : %d\n",ib[1].First.V);
  if (ib[1].Last.I)   printf("last image                : %d\n",ib[1].Last.V);
  if (ib[1].Inc.I)    printf("increment                 : %d\n",ib[1].Inc.V);
  if (ib[0].Dummy.I)  printf("output dummy              : %s\n",ib[0].Dummy.V);
  if (ib[0].Dim[1].I) printf("output dimension 1        : %s\n",ib[0].Dim[1].V);
  if (ib[0].Dim[2].I) printf("output dimension 2        : %s\n",ib[0].Dim[2].V);
  if (pW->I)          printf("width (w)                 : %s pixel\n",pW->V);
  if (pWa->I)         printf("dip width (wa)            : %s pixel\n",pWa->V);
  if (pA->I)          printf("relative deepness (a)     : %s\n",pA->V);
  if (pMrad->I)       printf("mask radius (rmsk)        : %s pixels\n",pMrad->V);
  if (pChi2->I)       printf("noise threshold (chi2)    : %s\n",pChi2->V);
  if (pMax->I)        printf("absolute threshold (max)  : %s\n",pMax->V);
  printf("\n");

  if (pcb->TestBit) {
    PrintBlocks ( pcb, ib );
    printf("\n"); }

  return;
}

/*---------------------------------------------------------------------------
main
---------------------------------------------------------------------------*/

#if MAKE_FUNCTION
# define MAIN main_saxs_filter
#else
# define MAIN main
#endif

int MAIN (int argc, char *argv[])
{
  CmdBlk cb;                /* command block  */
  ImgBlk ib[BlockNum];      /* image blocks */

  int status;
  int arg_no = 0;

 /* Init options, control block and image blocks */
  InitOptions( Usage, Version, NULL, TRUE, &cb, ib, BlockNum );

 /* Define special options */
  option_define ( IO_tpline,  SMname,   "msk.edf", &cb );
  option_define ( IO_tpflag,  SMfile,     "FALSE", &cb );

  option_define ( IO_tplvexp,  SMrad,         "4", &cb );
  option_define ( IO_tplvexp,  SMode,         "0", &cb );
  option_define ( IO_tpflexp,  SChi2,       "5.0", &cb );
  option_define ( IO_tpflexp,  SMax,        "0.0", &cb );
  option_define ( IO_tpflexp,  SW,          "4.0", &cb );
  option_define ( IO_tpflexp,  SWa,         "4.0", &cb );
  option_define ( IO_tpflexp,  SA,          "1.0", &cb );

 /* Read options from argument list */
  ReadOptions( argv, &arg_no, &cb, ib, &status);

  /* USER KEYBOARD I/O */
  if (status==Success) {
    argv_start ( &cb, 1 );
    user_io( &cb, ib, &status);
    argv_end( &cb ); /* must be called after user_io */
  }

  /* SEQUENCE CALCULATION */
  if (status==Success) 
    IMAGELOOP( &cb, ib, saxs_filter, NULL, NULL, TRUE, &status );

  return( ReportSaxsStatus( status, 0 ) );

} /* MAIN */
