/***************************************************************************/
/* Written 1994++ by Peter Boesecke                                        */
/* Copyright (C) 2011 European Synchrotron Radiation Facility              */
/*                       Grenoble, France                                  */
/*                                                                         */
/*    Principal authors: Peter Boesecke  (boesecke@esrf.eu)                */
/*                                                                         */
/*    This program is free software: you can redistribute it and/or modify */
/*    it under the terms of the GNU General Public License as published by */
/*    the Free Software Foundation, either version 3 of the License, or    */
/*    (at your option) any later version.                                  */
/*                                                                         */
/*    This program is distributed in the hope that it will be useful,      */
/*    but WITHOUT ANY WARRANTY; without even the implied warranty of       */
/*    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the        */
/*    GNU General Public License for more details.                         */
/*                                                                         */
/*    You should have received a copy of the GNU General Public License    */
/*    along with this program.  If not, see <http://www.gnu.org/licenses/>.*/
/***************************************************************************/
/*+++
1 saxs_ave
  Averaging of two image sequences

2 PURPOSE
  Averaging of two image sequences

  Calculation:       I0 = Mean(I1+I2)
  Error propagation: V0 = Mean(V1+V2)

  Arguments:
  saxs_ave [options] <i1nam> <onam> <i1fst> <i1lst> <i1inc> <i2fst> 
             <odum> <odim1> <odim2> <i1con> <i1fac> <i2con> <i2fac> 
             <ocon> <ofac> 

  Defaults:
  <input sequence 1>  : input.edf 
  <input sequence 2>  : <input sequence 1>
  <output sequence>   : output.edf
  <first image>       : <first image number in sequence 1>
  <last image>                              : <last image number in input file>
     if argument list ends with first image : <first image>
  <increment>         : 1
  <first image>       : <first image number in sequence 2>
  <output dummy>      : <dummy value in first image of input file>
  <output dimension 1>: <horizontal dimension of first image in input file>
  <output dimension 2>: <vertical dimension of first image in input file>
  <i1fac>             : 1.0 input 1 multiplication factor
  <i1con>             : 0.0 input 1 addition constant
  <i2fac>             : 1.0
  <i2con>             : 0.0
  <ofac>              : 1.0
  <ocon>              : 0.0

  Wildcard:
  A "=" in the argument list chooses the default.

2 HISTORY
  28-Nov-1995 PB V3.0 
  2000-08-04  PB V3.0  %f->%g
  2001-07-09  PB V4.00 new loop, no repetition
  2002-06-01  PB V4.50 ave 
  2003-02-05  PB V4.51 line args for Factor and Const
  2003-02-08  PB V4.52 Dummy, DDummy, Dim -> IO_*exp
  2003-05-30  PB V4.53 ImageLoop: Function call changed
  2003-11-30  PB V4.54 BinSiz
  2004-04-17  PB V4.55 weighting changed: 
                       To calculate a weighted average of two images the
                       options -i1ave -i2ave +oave must be set (default).

                       -i1ave -i2ave +i0ave: The sum values of image I1 and I2
                                             are averaged (weighted average)
                                             I0 = (<I1>+<I2>)/(weight1+weigth2)
                       -i1ave -i2ave -i0ave: The sum values of image I1 and I2
                                             are added
                                             I0 = <I1>+<I2>
                       +i1ave +i2ave +i0ave: The average values of 
                                             image I1 and I2 are averaged
                                             I0 = (<I1>+<I2>)/2
                       +i1ave +i2ave -i0ave: The average values of 
                                             image I1 and I2 are added 
                                             I0 = <I1>+<I2>
  2004-10-31  PB V4.56 MAKE_FUNCTION
  2005-07-15  PB V4.57 ReportSaxsStatus parameter list updated
  2005-08-04  PB V4.58 PrintImageHeaderBlock paramters updated
  2005-09-17  PB V4.59 VarDat
  2007-06-11  PB V4.60 ImageLoop parameters updated
  2012-01-31  PB V4.61 ImageLoop parameters updated
  2016-05-18  PB V4.62 Option +/-dumo added.
                       If +dumo is set, the values of image 2 are
                       only used for replacing dummies of image 1
                       (default -dumo, average all)
---*/
# define Version  "saxs_ave V4.62 2016-05-18, Peter Boesecke"

# include <errno.h>
# include <stdio.h>
# include <unistd.h>
# include <fcntl.h>

# include "SaxsPrograms.h"

# define DevSeparator ":"
# define DirSeparator "/"    
# define ExtSeparator "."

# define SOnly "dumo"

# define Usage "[options] \n\
             <i1nam> <onam> <i1fst> <i1lst> <i1inc> <i2fst>\n\
             <odum> <odim1> <odim2> <i1con> <i1fac> <i2con> <i2fac>\n\
             <ocon> <ofac> "

# define BlockNum 3       /* 2 input sequences + 1 output sequence */

/*---------------------------------------------------------------------------
1 saxs_ave

2 PURPOSE
  Average of two images
---------------------------------------------------------------------------*/

void saxs_ave (CmdBlk * pcb, long num[], ImgHeadBlk ihb[], int * pstatus)
{ const int NumberOfImages=3;
  int i,imax;
  int j;

  int   I0Ave;
  float *I0Data, *E0Data;
  int   I0Dim_1,I0Dim_2;
  float *pI0Data, *pE0Data;
  float I0Dummy;
  float I0DDummy;

  int   I1Ave; 
  float *I1Data, *E1Data;
  int   I1Dim_1,I1Dim_2;
  float I1Dummy, I1DDummy;
  float I1Value, I1Sum, I1Weight;
  float E1Value, E1Sum, E1Weight;

  int   I2Ave; 
  float *I2Data, *E2Data;
  int   I2Dim_1,I2Dim_2;
  float I2Dummy, I2DDummy;
  float I2Value, I2Sum, I2Weight;
  float E2Value, E2Sum, E2Weight;

  /* Do not change the following definitions,
     they are automatically changed with BlockNum */
  int i_1, i_2;
  float f_1[BlockNum], f_2[BlockNum];
 
  float Off_1[BlockNum], Off_2[BlockNum];
  float Ps_1[BlockNum], Ps_2[BlockNum];
 
  float f1_1[BlockNum], f3_1[BlockNum], Df_1[BlockNum];
  float f1_2[BlockNum], f3_2[BlockNum], Df_2[BlockNum];
 
  int Imin_1, Imin_2, Imax_1, Imax_2;
 
  float Wmin_1, Wmax_1, W_1, DW_1;
  float Wmin_2, Wmax_2, W_2, DW_2;                                                                                                            
  int cnt1, cnt2;
  float Value, E0Value;

  IO_long Only;

  *pstatus = Success;

  imax = pcb->ImgBlkLen;
  printf("\n Calculating ihb[0,% d] = Function(",ihb[0].ImNum);
  for(i=1;i<imax;i++) {
    printf("ihb[% d,% d] ", i, ihb[i].ImNum); }

  printf(")\n\n");

  for(i=1;i<imax;i++) {
    if (pcb->TestBit) PrintImageHeaderBlock(&ihb[i]);}

 /* Check the number of images */
  if (pcb->ImgBlkLen!=3) {
     printf("%d images found, %d input and 1 output image required\n",
             pcb->ImgBlkLen,2);
       *pstatus=Failed; return; }

 /* Copy first image to output array */

  GetReferenceParameters( pcb, ihb, 0, imax-1,
                          Off_1, Off_2, Ps_1,  Ps_2, pstatus ); 
  if (*pstatus!=Success) return;

  UpdateImageParameters ( pcb, ihb, 1, imax-1,
                          Off_1, Off_2, Ps_1,  Ps_2, pstatus );
  if (*pstatus!=Success) return;

  // Get large range (mode=2)
  GetImageRange         ( pcb, ihb, 1, imax-1,
                          Off_1, Off_2, Ps_1,  Ps_2,
                          f1_1, f3_1, Df_1, f1_2, f3_2, Df_2,
                          &Imin_1, &Imax_1, &Imin_2, &Imax_2,
                          &Wmin_1, &Wmax_1, &DW_1, &Wmin_2, &Wmax_2, &DW_2, 2);

  Only = option_flag ( SOnly, pcb, pstatus );
  if (*pstatus!=Success) return;

 /* 1 input, 1 output */
  I0Ave   = pcb->ib[0].Ave.V;
  I0Data  = ihb[0].Data;
  E0Data  = ihb[0].VarDat;
  I0Dummy = ihb[0].Dummy.V;
  I0DDummy = ihb[0].DDummy.V;
  I0Dim_1  = (int) ihb[0].Dim[1];
  I0Dim_2  = (int) ihb[0].Dim[2];

  I1Ave   = pcb->ib[1].Ave.V;
  I1Data  = ihb[1].Data;
  E1Data  = ihb[1].VarDat;
  I1Dummy = ihb[1].Dummy.V;
  I1DDummy = ihb[1].DDummy.V;
  I1Dim_1  = (int) ihb[1].Dim[1];
  I1Dim_2  = (int) ihb[1].Dim[2];

  I2Ave   = pcb->ib[2].Ave.V;
  I2Data   = ihb[2].Data;
  E2Data   = ihb[2].VarDat;
  I2Dummy  = ihb[2].Dummy.V;
  I2DDummy = ihb[2].DDummy.V;
  I2Dim_1  = (int) ihb[2].Dim[1];
  I2Dim_2  = (int) ihb[2].Dim[2];

  /* loop over the output array */
  W_2 = Wmin_2; for (i=0;i<imax;i++) f_2[i]=f1_2[i];
  for (i_2=Imin_2;i_2<=Imax_2;i_2++) {
 
    pI0Data = ABSPTR(I0Data,I0Dim_1,I0Dim_2,Imin_1,i_2);
    pE0Data = E0Data-I0Data+pI0Data;
    W_1 = Wmin_1; for (i=0;i<imax;i++) f_1[i]=f1_1[i];
    for (i_1=Imin_1;i_1<=Imax_1;i_1++) {

      if ( E0Data ) {

        cnt1 = Isum2ldwE (I1Data,E1Data,I1Dim_1,I1Dim_2,I1Dummy,I1DDummy,
          f_1[1], f_2[1], f_1[1]+Df_1[1], f_2[1]+Df_2[1], 
          &I1Sum, &I1Weight,&E1Sum,&E1Weight);
 
        if ((cnt1)&&(Only.V)) {
          // Do not use data of second image 
          cnt2 = 0; I2Sum=0.0; I2Weight=0.0; E2Sum=-1; E2Weight=0.0;
        } else {
          cnt2 = Isum2ldwE (I2Data,E1Data,I2Dim_1,I2Dim_2,I2Dummy,I2DDummy,
            f_1[2], f_2[2], f_1[2]+Df_1[2], f_2[2]+Df_2[2], 
            &I2Sum, &I2Weight,&E2Sum,&E2Weight);
        }

        if ( cnt1 || cnt2 ) {

          // I0 = (I1+I2)/(I1Weight+I2Weight)
          I1Value=I1Sum; if ((cnt1)&&(I1Ave)) {I1Value/=I1Weight;I1Weight=1.0;}
          I2Value=I2Sum; if ((cnt2)&&(I2Ave)) {I2Value/=I2Weight;I2Weight=1.0;}
          Value = I1Value + I2Value;
          if (I0Ave) Value /= I1Weight + I2Weight;
          UPDATE( *pI0Data, Value, I0Dummy, I0DDummy );

          // V0 = (V1+V2)/( (E1Weight+E2Weight)*(E1Weight+E2Weight) )
          if (E1Sum>=0.0) {
            E1Value=E1Sum;
            if ((cnt1)&&(I1Ave)) {
              E1Value/=E1Weight*E1Weight;E1Weight=1.0;
            }
          } else { E1Value = 0.0; E1Weight = 0.0; }
          if (E2Sum>=0.0) {
            E2Value=E2Sum;
            if ((cnt2)&&(I2Ave)) {
              E2Value/=E2Weight*E2Weight;E2Weight=1.0;
            }
          } else { E2Value = 0.0; E2Weight = 0.0; }
          E0Value = E1Value + E2Value;
          if (I0Ave) E0Value /= (E1Weight + E2Weight)*(E1Weight + E2Weight);
          UPDATE( *pE0Data, E0Value, VarDummy, DDSET(VarDummy) );
 
        } /* if (Ipol2d ... */

      } else {
 
        cnt1 = Isum2ldw (I1Data,I1Dim_1,I1Dim_2,I1Dummy,I1DDummy,
          f_1[1], f_2[1], f_1[1]+Df_1[1], f_2[1]+Df_2[1], &I1Sum, &I1Weight);

        if ((cnt1)&&(Only.V)) {
          // Do not use data of second image 
          cnt2 = 0; I2Sum=0.0; I2Weight=0.0;
        } else {
          cnt2 = Isum2ldw (I2Data,I2Dim_1,I2Dim_2,I2Dummy,I2DDummy,
            f_1[2], f_2[2], f_1[2]+Df_1[2], f_2[2]+Df_2[2], &I2Sum, &I2Weight);
        }

        if ( cnt1 || cnt2 ) {

          I1Value=I1Sum; if ((cnt1)&&(I1Ave)) {I1Value/=I1Weight;I1Weight=1.0;}
          I2Value=I2Sum; if ((cnt2)&&(I2Ave)) {I2Value/=I2Weight;I2Weight=1.0;}

          Value = I1Value + I2Value;
          if (I0Ave) Value /= I1Weight + I2Weight;

          UPDATE( *pI0Data, Value, I0Dummy, I0DDummy );
 
        } /* if (Ipol2d ... */
      }
 
      pI0Data++;
      pE0Data++;
      W_1+=DW_1; for (i=0;i<imax;i++) { f_1[i]+=Df_1[i]; }
    } /* for i_1 ... */
 
    W_2+=DW_2; for (i=0;i<imax;i++) { f_2[i]+=Df_2[i]; }
  } /* for i_2 ... */

} /* saxs_ave*/

/*---------------------------------------------------------------------------
user_io
Do all the keyboard io and return cb, and ib
---------------------------------------------------------------------------*/

void user_io(CmdBlk * pcb, ImgBlk * ib, int * pstatus)
{
  char  progfn[IO_size];
  ImgHeadBlk ihb[BlockNum];

  float ROff_1, RPs_1, UOff_1, UPs_1;
  float ROff_2, RPs_2, UOff_2, UPs_2;

 /* Determine program name without directory */
   (void) RightFR((char *) pcb->argv[0],DirSeparator,progfn,IO_size);

  /* show usage if no arguments are given */
  if (pcb->argc<=1) printf("Usage: %s %s\n",progfn,Usage);

  /*--- Write name of program ---*/
  printf("\n");
  printf("   %s %s\n",progfn,Version);
  printf("\n");

  ArgvFilenames ( pcb, ib, ihb, 0, BlockNum-1, pstatus);
  if (*pstatus!=Success) return;
  GetReference(pcb->RSys.V,1,ihb,&ROff_1,&ROff_2,&RPs_1,&RPs_2,pstatus );
  if (*pstatus!=Success) return;
  GetReference(pcb->USys.V,1,ihb,&UOff_1,&UOff_2,&UPs_1,&UPs_2,pstatus );
  if (*pstatus!=Success) return;

  /*--- Argument  : multiplication factor */
  printf("<1st sequence> = <read 1st sequence> * Factor + Const\n");
  argv_flexp(pcb,"Multiplication factor",&ib[1].Factor,"1.0",pstatus);
  if (*pstatus!=Success) return;
 
  /*--- Argument  : additive constant */
  argv_flexp(pcb,"Addition constant",&ib[1].Const,"0.0",pstatus);
  if (*pstatus!=Success) return;
 
  /*--- Argument  : multiplication factor */
  printf("<2nd sequence> = <read 2nd sequence> * Factor + Const\n");
  argv_flexp(pcb,"Multiplication factor",&ib[2].Factor,"1.0",pstatus);
  if (*pstatus!=Success) return;
 
  /*--- Argument  : additive constant */
  argv_flexp(pcb,"Addition constant",&ib[2].Const,"0.0",pstatus);
  if (*pstatus!=Success) return;
 
  /*--- Argument  : multiplication factor */
  printf("<output sequence> = (<1st sequence>+<2nd sequence>)*Factor+Const\n");
  argv_flexp(pcb,"Multiplication factor",&ib[0].Factor,"1.0",pstatus);
  if (*pstatus!=Success) return;
 
  /*--- Argument  : additive constant */
  argv_flexp(pcb,"Addition constant",&ib[0].Const,"0.0",pstatus);
  if (*pstatus!=Success) return;

  printf("\n");
  if (ib[1].Name.I)    printf("i/p file 1         : %s\n",ib[1].Name.V);
  if (ib[2].Name.I)    printf("i/p file 2         : %s\n",ib[2].Name.V);
  if (ib[0].Name.I)    printf("o/p file           : %s\n",ib[0].Name.V);
  if (ib[1].First.I)   printf("first image in 2   : %d\n",ib[1].First.V);
  if (ib[1].Last.I)    printf("last image         : %d\n",ib[1].Last.V);
  if (ib[1].Inc.I)     printf("increment          : %d\n",ib[1].Inc.V);
  if (ib[2].First.I)   printf("image in 2         : %d\n",ib[2].First.V);
  if (ib[0].Dummy.I)   printf("output dummy       : %s\n",ib[0].Dummy.V);
  if (ib[0].Dim[1].I)  printf("output dimension 1 : %s\n",ib[0].Dim[1].V);
  if (ib[0].Dim[2].I)  printf("output dimension 2 : %s\n",ib[0].Dim[2].V);
  if (ib[1].Factor.I)  printf("i/p 1 factor       : %s\n",ib[1].Factor.V);
  if (ib[1].Const.I)   printf("i/p 1 constant     : %s\n",ib[1].Const.V);
  if (ib[2].Factor.I)  printf("i/p 2 factor       : %s\n",ib[2].Factor.V);
  if (ib[2].Const.I)   printf("i/p 2 constant     : %s\n",ib[2].Const.V);
  if (ib[0].Factor.I)  printf("output factor      : %s\n",ib[0].Factor.V);
  if (ib[0].Const.I)   printf("output constant    : %s\n",ib[0].Const.V);
  printf("\n");

  if (pcb->TestBit) {
    PrintBlocks ( pcb, ib );
    printf("\n"); }

  return;
} /* user_io */

/*---------------------------------------------------------------------------
main
---------------------------------------------------------------------------*/

#if MAKE_FUNCTION
# define MAIN main_saxs_ave
#else
# define MAIN main
#endif

int MAIN (int argc, char *argv[])
{
  CmdBlk cb;                /* command block  */
  ImgBlk ib[BlockNum];      /* image blocks */

  int status;
  int arg_no = 0;

 /* Init options, control block and image blocks */
  InitOptions( Usage, Version, NULL, TRUE, &cb, ib, BlockNum );

 /* Defaults */
  ib[0].Ave.V = TRUE;    /* average output image */
  ib[1].Ave.V = FALSE;   /* use sum of values of input images */
  ib[2].Ave.V = FALSE;

 /* Define special options */
  option_define ( IO_tpflag,  SOnly,     "FALSE", &cb ); /* -dumo */

 /* Read options from argument list */
  ReadOptions( argv, &arg_no, &cb, ib, &status);

  /* USER KEYBOARD I/O */
  if (status==Success) {
    argv_start ( &cb, 1 );
    user_io( &cb, ib, &status);
    argv_end( &cb ); /* must be called after user_io */
  }

  /* SEQUENCE CALCULATION */
  if (status==Success)
    IMAGELOOP( &cb, ib, saxs_ave, NULL, NULL, TRUE, &status );

  return( ReportSaxsStatus( status, 0 ) );

} /* MAIN */

