/***************************************************************************/
/* Written 1994++ by Peter Boesecke                                        */
/* Copyright (C) 2011 European Synchrotron Radiation Facility              */
/*                       Grenoble, France                                  */
/*                                                                         */
/*    Principal authors: Peter Boesecke  (boesecke@esrf.eu)                */
/*                                                                         */
/*    This program is free software: you can redistribute it and/or modify */
/*    it under the terms of the GNU General Public License as published by */
/*    the Free Software Foundation, either version 3 of the License, or    */
/*    (at your option) any later version.                                  */
/*                                                                         */
/*    This program is distributed in the hope that it will be useful,      */
/*    but WITHOUT ANY WARRANTY; without even the implied warranty of       */
/*    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the        */
/*    GNU General Public License for more details.                         */
/*                                                                         */
/*    You should have received a copy of the GNU General Public License    */
/*    along with this program.  If not, see <http://www.gnu.org/licenses/>.*/
/***************************************************************************/
# define Version  "V1.7 2017-05-21 Peter Boesecke"
/*---------------------------------------------------------------------------
NAME
   roca - calibrate detector rotation angles

SYNOPSIS

  roca <options> <return> {<calibration data>} <Ctrl-D>

DESCRIPTION

  roca [<options>] <start parameters> <return>
    # parameters and start values (fixed parameters must be given in square
    # brackets. It is not possible to calibrate pix1 or pix2. This must be 
    # done with a square grid. Pix1 and pix2 must always be given in square 
    # brackets .
    # rot3 cannot be calibrated because the scattering pattern is rotationally
    # symmetric around the beam axis (rot3), but it can be supplied as start
    # parameter (no effect).
    [<pix1/m>] [<pix2/m>] <cen1> <cen2> <dis/m> <rot1/rad> <rot2/rad> [<rot3/rad>]<return>

    # input data (from calibration sample)
    # 2Theta-Angle == 0 is the position of the primary beam on the detector
    <2Theta-Angle/rad> <i1> <i2> <return>
    ...
    <Ctrl-D>

  In case of success the output line is
    <pix1> <pix2> <cen1> <cen2> <dis> <rot1> <rot2> <rot3>
  separated by spaces.

EXAMPLE

     The input data file can contain as many points as necessary. Each 
     line must contain the scattering angle (TwoTheta) in radian, 
     the i1-coordinate and the i2-coordinate (in image pixels), separated
     by spaces or tabs. A hash (#) is a comment marker. All characters 
     after a hash are ignored.

     The following example is based on a calculated input image from 
     which data points have been extraced manually (e.g. with edfplot, 
     fit2d).

     The correct parameters of the input image are: 

     5e-05 5e-05 400 500 0.1 -0.3927 0 0

     The data points have been manually extracted from this image
     and are therefore not exact. The calculation minimizes the 
     angular difference between the fit and the data points.
     The weights are set to 1 if omitted.

     file: points.dat 

     #TwoTheta_rad   i1_I   i2_I    [weight]
     #3 1 0
     0.19326         125.5  1013.5  1
     0.19326         241.6   937.0  1
     0.19326         401.9   903.7  1
     0.19326         567.2   940.7  1
     0.19326         681.9  1019.7  1
     #4 1 0
     0.21367          57.8  1021.8  1
     0.21367         160.4   932.1  1
     0.21367         401.9   861.4  1
     0.21367         650.5   940.0  1
     0.21367         742.7  1021.4  1
     #2 0 1
     0.2703           18.8   893.7  1
     0.2703          226.7   773.2  1
     0.2703          576.1   774.8  1
     0.2703          756.8   874.2  1
     0.2703          886.3  1014.2  1
     #1 1 1
     0.3059           11.0   805.0  1
     0.3059          139.8   728.8  1
     0.3059          400.4   673.6  1
     0.3059          821.6   829.8  1
     0.3059          980.9  1019.8  1

     > cat points.dat | roca  50_um 50_um 500 500 11_cm -22_deg 0 0 debug=0x8
     roca V0.8 2010-01-06 Peter Boesecke

          Name          Value         Refine
          pix1          5e-05             no
          pix2          5e-05             no
          cen1            500            yes
          cen2            500            yes
           dis           0.11            yes
          rot1      -0.383972            yes
          rot2              0            yes
          rot3              0             no

           Num       TwoTheta             i1             i2
             1        0.19326          125.5         1013.5
             2        0.19326          241.6            937
             3        0.19326          401.9          903.7
             4        0.19326          567.2          940.7
             5        0.19326          681.9         1019.7
             6        0.21367           57.8         1021.8
             7        0.21367          160.4          932.1
             8        0.21367          401.9          861.4
             9        0.21367          650.5            940
            10        0.21367          742.7         1021.4
            11         0.2703           18.8          893.7
            12         0.2703          226.7          773.2
            13         0.2703          576.1          774.8
            14         0.2703          756.8          874.2
            15         0.2703          886.3         1014.2
            16         0.3059             11            805
            17         0.3059          139.8          728.8
            18         0.3059          400.4          673.6
            19         0.3059          821.6          829.8
            20         0.3059          980.9         1019.8

          Name          Value         Refine
          pix1          5e-05             no
          pix2          5e-05             no
          cen1        385.956            yes
          cen2        502.483            yes
           dis      0.0999409            yes
          rot1      -0.391358            yes
          rot2     0.00657904            yes
          rot3              0             no

     5e-05 5e-05 385.956 502.483 0.0999409 -0.391358 0.00657904 0

     The standard deviation per point of the fit is 0.011_deg 
     (can be found with debug=9). For comparison here again the 
     correct parameters of the image:

     5e-05 5e-05 400 500 0.1 -0.3927 0 0

HISTORY
  V0.0 2009-12-07 Peter Boesecke start
  V0.8 2010-01-07 Peter Boesecke first version
  V0.9 2010-07-29 Peter Boesecke weights added
  V1.0 2010-07-30 Peter Boesecke output of spec file with all iteration steps
  V1.1 2010-08-06 Peter Boesecke more columns in spec file + tilts
  V1.2 2010-08-19 Peter Boesecke two output lines: 
                                 1st) fitted parameters
                                 2nd) estimated accuracy of fitted parameters
  V1.3 2010-09-16 Peter Boesecke third output line: statistics,
                                 RocaConvergenceError added. 
  V1.4 2011-06-15 Peter Boesecke tilt3d calculation corrected
  V1.5 2011-07-21 Peter Boesecke new makefile, is_white etc. replaced with
                                 strlib functions
  V1.6 2013-12-09 Peter Boesecke clean up
  V1.7 2017-05-21 Peter Boesecke ioalloc
---------------------------------------------------------------------------*/

/****************************************************************************
*  Include                                                                  *
****************************************************************************/

# include <stdio.h>
# include <stdlib.h>
# include <string.h>
# include <ctype.h>
# include <limits.h>
# include <errno.h>
# include <fcntl.h>
# include <math.h>
# include <float.h>

# include "strlib.h"
# include "numio.h"
# include "roca_base.h"
# include "roca_list.h"

/****************************************************************************
* Definitions                                                               *
****************************************************************************/
# define Usage "roca [-h] [<options>] [debug=0|1|...] [input=<filename>] <start values>\n\n" \
"  <start values> := <pix1/m> <pix2/m> <cen1/pixel> <cen2/pixel> <dis/m> \n" \
"                    <rot1/rad> <rot2/rad> <rot3/rad>\n" \
"    defaults: pix1 and pix2 will not be refined, rot3 cannot be refined due\n" \
"              to rotational symmetry,\n" \
"    mark parameters with \'~\' to refine, \'=\' to keep constant\n" \
"  <options> := maxiter=nnn (number of loops, default 100) | mindev=ddd (min stddev to be reached, \n"\
"               default 1e-7), maxdev=mmm (maximum accepted final deviation, default 1_deg)\n\n" \
"  Each line of the input file must contain three values separated by spaces:\n" \
"   TwoTheta in radian and two image pixel coordinates.\n" \
"  Optionally, a weight (>=0) can be appended to each line (default 1).\n"

# define Example "  Example: \n" \
"     file: points.dat\n\n" \
"     #TwoTheta_rad   i1_I   i2_I\n" \
"     #3 1 0\n" \
"     0.19326         125.5  1013.5\n" \
"     0.19326         241.6   937.0\n" \
"     ...\n\n" \
"          >cat points.dat | roca  50_um 50_um 500 500 11_cm -22_deg 0 0 debug=1\n" \
"           bin/LINUX64/roca V1.1 2010-08-06 Peter Boesecke\n\n" \
"           step=8, stddev=0.000194115, change=1.60406e-18, mindev=-2.59197e-06, mindev=0.000511031\n" \
"           pix1=[5e-05] pix2=[5e-05] cen1=385.956 cen2=502.483 dis=0.0999409 rot1=-0.391358 rot2=0.00657904 rot3=[0]\n" \
"               Par = 5e-05 5e-05   385.956  502.483     0.0999409  -0.391358    0.00657904  0\n"\
"            ParDev =     0     0   0.69859  0.474283    0.000184965 0.000232974 0.000324257 0\n"\
"           BeamPar = 5e-05 5e-05   400.182  1327.28     0.108118    3.12435     0.391411    0.00130407\n"\
"               5e-05 5e-05 385.956 502.483  0.0999409  -0.391358    0.00657904  0\n"\
"                   0     0 0.69859 0.474283 0.000184965 0.000232974 0.000324257 0\n"\
"          0.00019411 -2.59197e-06  0.000511 20          20          5\n\n"\
"     The 1st and 2nd output line contains, in the order of the start values, the optimized values\n"\
"     and their (partial) standard deviations. The 3rd line contains the statistics of the fitted\n"\
"     data points: stdev, mindev, maxdev, total, ndata, nopt (average standard deviation per data point, \n"\
"     minimum and maximum standard deviation, total weight, number of data points, number of optimized parameters)\n"\

# define RocaSuccess 0x0
# define RocaError   0x1
# define RocaInputError 0x10 
# define RocaOptimizationError 0x20 
# define RocaConvergenceError 0x40 

/*---------------------------------------------------------------------------
NAME 
    
   get_par_mode 
  
SYNOPSIS

   int get_par_mode( const char *value , const char **pvalue, int defval)

DESCRIPTION

   Test whether the value must be kept fixed (1) or must be optimized (-1).
   Parameter starts with
        '=' and '[' optionally closing ']': keep fixed, return 0, *pvalue=value+1
        '~': refine, return value 1, *pvalue=value+1
        everything else: return defval
   The pointer to value is incremented in *pvalue.

RETURN VALUE

   parameter mode 0 | 1 depending on first character, default if not specified.

---------------------------------------------------------------------------*/
int get_par_mode( const char *value , const char **pvalue, int defval)
{ int mode;

  switch (*value) {
    case '[':
    case '=':  mode=0; if (pvalue) *pvalue=value+1; break; // keep
    case '~':  mode=1; if (pvalue) *pvalue=value+1; break; // refine
    default: mode=defval; if (pvalue) *pvalue=value;
  }
  return( mode );

} // get_par_mode

int fprint_parameters_verbose( FILE *out, double Par[], int ParMode[] )
{
  int num;
  const char **RocaParamNames;

  RocaParamNames=RocaParamNamesArray();
  
  // show parameters
  fprintf(out,"%14s %14s %14s\n","Name","Value","Refine");

  for (num=0;num<NParams;num++)
    if (ParMode[num]>=0)
      fprintf(out,"%14s %14g %14s\n",RocaParamNames[num],Par[num],ParMode[num]?"yes":"no");
    else fprintf(out,"%14s %14s %14s\n",RocaParamNames[num],"missing","-");

  fprintf(out,"\n");

  return(0);

} // fprint_parameters_verbose

int fprint_input_data( FILE *out, size_t ndata, double TwoTheta[], 
                       double Coord1[], double Coord2[], double Weight[] )
{ size_t num;
  // show input data
  printf("%14s %14s %14s %14s %14s\n","Num","TwoTheta","i1","i2","Weight");
  for (num=0;num<ndata;num++) {
    printf("%14lu %14lg %14lg %14lg %14lg\n",
      num+1,TwoTheta[num],Coord1[num],Coord2[num],Weight[num]);
  }
  printf("\n");

  return(0);

} // fprint_input_data

int fprint_parameters( FILE *out, double Par[], const char * label )
{ char buffer[ROCABUFLEN];
  if (!label) label="";
  fprintf(out,"%12s%s\n",label,sprint_parameters( buffer, ROCABUFLEN, Par ));
  return(0);
} // fprint_parameters

int fprint_beam_parameters( FILE *out, double Par[], const char * label )
{ char buffer[ROCABUFLEN];
  if (!label) label="";
  fprintf(out,"%12s%s\n",
    label, sprint_beam_parameters( buffer, ROCABUFLEN, Par ));
  return(0);
} // fprint_beam_parameters

int fprint_statistics( FILE *out, double Stat[], const char * label )
{ char buffer[ROCABUFLEN];
  if (!label) label="";
  fprintf(out,"%12s%s\n",label,sprint_statistics( buffer, ROCABUFLEN, Stat ));
  return(0);
} // fprint_statistics


/*+++------------------------------------------------------------------------
main
---------------------------------------------------------------------------*/

#if MAKE_FUNCTION
# define MAIN main_roca
#else
# define MAIN main
#endif

int MAIN (int argc, char *argv[])
{ 
  char *nul = (char *) NULL;
  char **pargv;
  int status=0;
  int debug=0;
  char * string;
  char* finput=NULL;
  LList *list;
  LElement *element;
  const char *value;
  FILE * input;
  int maxiter=100; // number of loops (default)
  double mindev=1e-7; // minimum mean deviation per data point in radian to continue iteration (default)
  double maxdev=0.01745329252; // maximum accepted mean deviation per data point in radian (default 1_deg)

  const char **RocaParamNames;
  int *RocaParamDefault;

  int num, cnt;

  char buffer[ROCABUFLEN];
  char keybuf[ROCABUFLEN];

  const char *ps, *key;

  char errbuf[ROCABUFLEN];

  int errval;
  const char *str;

  // Get array with names of parameters
  RocaParamNames=RocaParamNamesArray();

  // Get array with defaults of parameters
  RocaParamDefault=RocaParamDefaultArray();

  // Tables of NParams start parameters defined in RocaParam 
  double *ParStart=NULL; // start values
  double *ParOpt=NULL; // optimized values
  double *ParDev=NULL; // deviations of optimized values
  int *ParMode=NULL;     // mode (0: keep, 1: refine)
  double *ParStat=NULL; // some statistics 

  // Tables of input data (2Theta, i1, i2)
  size_t ndata;                       // number of data points 
  double *TwoTheta=NULL;              // 2-Theta angle
  double *Coord1=NULL, *Coord2=NULL;  // detector image coordinates
  double *Weight=NULL;                // weight of data point (>=0.0)

  fprintf( stderr, "%s %s %s\n\n", argv[0], Version, roca_libversion() );

  // Usage
  if (argc<=1) { printf( "%s\n", Usage ); return(1); }


  /*****************************************************************************
  * Read arguments                                                             *
  *****************************************************************************/

  // create new list for start parameters
  if (new_list( "start", &list )) goto main_error; 

  // count options or help 
  pargv = &argv[1];

  num=0;
  for (;*pargv!=nul;pargv++) {
    if (( strncmp( *pargv,"-h",2 )==0 )||( strncmp( *pargv,"--help",6 )==0 )) {
      // help
      fprintf( stderr, "%s\n", Usage );
      fprintf( stderr, "  debug=(%s)\n\n", RocaUsage2str() );
      fprintf( stderr,"%s\n",Example);
      fprintf( stderr,"\n");
      return( 1 );
    } else if ( strncmp( *pargv,"debug=", 6 ) == 0 ) {
      // set debug level
      ps=*pargv+6;
      debug = (int) num_str2double ( ps, &ps, &errval);
      RocaDebugSet(debug);
      list_debug_set(debug&ROCA_LISTDEBUG);
    } else if ( strncmp( *pargv,"input=", 6 ) == 0 ) {
      // set input file
      finput=*pargv+6;
    } else if ( strncmp( *pargv,"maxiter=", 8 ) == 0 ) {
      // set upper limit of iterations
      ps=*pargv+8;
      maxiter = (int) num_str2double ( ps, &ps, &errval);
    } else if ( strncmp( *pargv,"mindev=", 7 ) == 0 ) {
      // set angular deviation minimum to stop iteration
      ps=*pargv+7;
      mindev = num_str2double ( ps, &ps, &errval);
    } else if ( strncmp( *pargv,"maxdev=", 7 ) == 0 ) {
      // set maximum of accepted angular deviation after iteration
      ps=*pargv+7;
      maxdev = num_str2double ( ps, &ps, &errval);
    } else {
      // write arguments to start list
      if ( num<NParams ) key=RocaParamNames[num];
      else {
        sprintf(keybuf,"%d",num); key=keybuf;
      }
      if (!is_element( list, key )) num++;
      if (write_element_value( list, key, *pargv)) goto main_error;
    }
  }

  if (ROCALevel()>0) 
    print_lists( stdout, ROCALevel()-1, ROCADebug()&ROCA_VERBOSE );

  /*****************************************************************************
  * Read data                                                                  *
  *****************************************************************************/

  cnt = cnt_params(list, RocaParamNames);

  if (cnt>=NParams) {

    // open input file if defined
    if (finput) {
      if (ROCADebug()&ROCA_DEBUG) fprintf( stdout, "opening >>%s<< for reading\n", finput );
      if ( !( input=fopen(finput,"r") ) ) { 
        sprintf(errbuf,"fopen(%s,\"r\")",finput);
        perror(errbuf); goto main_error; 
      }
    } else input=stdin;

    // create new list for input data
    if (new_list( "data", &list )) goto main_error;

    // read from standard input
    num=1;
    while ( !feof( input ) ) {
      string = fgets ( buffer, ROCABUFLEN, input );

      if ( string ) {
        if ( (!strncmp(string,"exit",4)) || (!strncmp(string,"quit",4)) )
          break;
        // read 2Theta-Angle, i1 and i2 and add to a list
        // remove comments, leading and trailing white spaces, collapse rest
        strlib_collapse( strlib_trim( strlib_uncomment( string ) ) );

        if (!strlib_is_empty( string )) {
          sprintf(keybuf,"%d",num);
          if (write_element_value( list, keybuf, string)) goto main_error;
          num++;
        }
      }
    }

    if (ROCADebug()&ROCA_VERBOSE) 
      print_lists( stdout, ROCALevel(), ROCADebug()&ROCA_VERBOSE );

    // fclose( input )
    if (finput) {
      if (ROCADebug()&ROCA_DEBUG) fprintf( stdout, "closing >>%s<<\n", finput );
      fclose( input );
    }
  } // cnt>=NParams

  /*****************************************************************************
  * Create data tables                                                         *
  *****************************************************************************/

  // create table with start parameters for  "pix1","pix2","cen1", etc.
  if ( search_list( "start", &list ) ) goto main_error;

  if ( !(ParStart=(double *) MALLOC (sizeof(double)*NParams)) ) goto main_error;
  if ( !(ParMode=(int *) MALLOC (sizeof(int)*NParams)) ) goto main_error;
  if ( !(ParOpt=(double *) MALLOC (sizeof(double)*NParams)) ) goto main_error;
  if ( !(ParDev=(double *) MALLOC (sizeof(double)*NParams)) ) goto main_error;
  if ( !(ParStat=(double *) MALLOC (sizeof(double)*NStats)) ) goto main_error;

  cnt=0;
  for ( num=0; num<NParams; num++ ) {
    key=RocaParamNames[num];
    if ( read_element_value( list, key, &value) ) goto main_error;
    if (value) {
      ParMode[num] = get_par_mode( value , &ps, RocaParamDefault[num]);
      ParStart[num] = num_str2double ( ps, &ps, &errval);
      if (errval) { 
         fprintf(stderr,"ERROR %d in numio reading %s\n",errval,key);
         fprintf(stderr,"from line >>%s<< when creating table with start parameters\n",ps);
         goto main_error;
      }
      cnt++;
    } else ParMode[num] = -1; // unset
  }

  if (cnt<NParams) {
    fprintf(stderr,"%s\nMissing %d start value%s\n\n",Usage,8-cnt,cnt<7?"s":"");
    fprint_parameters_verbose( stderr, ParStart, ParMode );
    return( 1 );
  }

  // import whole data list
  if ( search_list( "data", &list ) ) goto main_error;

  ndata = number_of_elements( list );

  if (ndata<1) {
    fprintf(stderr,"%s\nMissing input data\n\n",Usage);
    goto main_error;
  }

  if ( !(TwoTheta=(double *) MALLOC (sizeof(double)*ndata)) ) goto main_error;
  if ( !(Coord1=(double *) MALLOC (sizeof(double)*ndata)) ) goto main_error;
  if ( !(Coord2=(double *) MALLOC (sizeof(double)*ndata)) ) goto main_error;
  if ( !(Weight=(double *) MALLOC (sizeof(double)*ndata)) ) goto main_error;

  if ( first_element( list, &element ) ) goto main_error;

  num=0;
  while ( element ) {
    if ( get_element_value( element, &value ) ) goto main_error;
    str=value;
    TwoTheta[num] = num_str2double ( str, &str, &errval);
    if (errval) { 
       fprintf(stderr,"ERROR %d in numio reading TwoTheta[%d]\n",errval,num);
       fprintf(stderr,"from line >>%s<< when creating input data tables.\n",str); 
       goto main_error; 
    }
    Coord1[num] = num_str2double ( str, &str, &errval);
    if (errval) { 
       fprintf(stderr,"ERROR %d in numio reading Coord1[%d]\n",errval,num);
       fprintf(stderr,"from line >>%s<< when creating input data tables.\n",str); 
       goto main_error;
    }
    Coord2[num] = num_str2double ( str, &str, &errval);
    if (errval) {
       fprintf(stderr,"ERROR %d in numio reading Coord2[%d]\n",errval,num);
       fprintf(stderr,"from line >>%s<< when creating input data tables.\n",str); 
       goto main_error;
    }
    if (strlib_is_empty( str )) Weight[num]=1.0;
    else {
      Weight[num] = num_str2double ( str, &str, &errval);
      if (errval) {
         fprintf(stderr,"ERROR %d in numio reading Weight[%d]\n",errval,num);
         fprintf(stderr,"from line >>%s<< when creating input data tables.\n",str);
         goto main_error;
      } else {
        if (Weight[num]<0) {
          fprintf(stderr,"WARNING: negative Weight[%d]==%lg set to zero.\n",
            num,Weight[num]);
          Weight[num]=0.0;
        }
      }
    }

    if ( next_element( list, &element ) ) goto main_error;
    num++;
  }

  // show start parameters
  if (ROCADebug()&ROCA_SHOWDATA)
    fprint_parameters_verbose( stdout, ParStart, ParMode );

  // show input data
  if (ROCADebug()&ROCA_SHOWDATA)
    fprint_input_data( stdout, ndata, TwoTheta, Coord1, Coord2, Weight );

  /*****************************************************************************
  * Calculation                                                                *
  *****************************************************************************/
  if ( RocaOptimize( ParOpt, ParDev, ParStat, ParStart, ParMode,
                     ndata, TwoTheta, Coord1, Coord2, Weight,
                     mindev, maxiter ) ) {
    status=RocaOptimizationError;
    goto main_error;
  }

  // show optimized parameters
  if (ROCADebug()&ROCA_SHOWDATA)
    fprint_parameters_verbose( stdout, ParOpt, ParMode );

  // show parameters
  if (ROCADebug()&ROCA_VERBOSE)
    fprint_parameters( stdout, ParOpt, "    Par =" );

  // show parameters
  if (ROCADebug()&ROCA_VERBOSE)
    fprint_parameters( stdout, ParDev, " ParDev =" );

  // show beam parameters
  if (ROCADebug()&ROCA_VERBOSE)
    fprint_beam_parameters( stdout, ParOpt, "BeamPar =" );

  // show global statistics
  if (ROCADebug()&ROCA_VERBOSE)
    fprint_statistics( stdout, ParStat, "ParStat =" );

  /*****************************************************************************
  * Calculation finished                                                       *
  *****************************************************************************/

  // print results
  fprint_parameters( stdout, ParOpt, NULL );  // optimized parameters
  fprint_parameters( stdout, ParDev, NULL );  // deviations
  fprint_statistics( stdout, ParStat, NULL ); // statistics

  if ( fabs(ParStat[RStdDev] > maxdev ) ) {
    fprintf(stderr,"Standard deviation of optimized parameters exceeds maximum accepted deviation\n");
    fprintf(stderr,"The standard deviation is %lg rad, the maximum accepted deviation is %lg rad\n",
      ParStat[RStdDev],maxdev);
    fprintf(stderr,"The maximum accepted deviation can be changed with the option maxdev=<deviation>\n");
    status=RocaConvergenceError;
    goto main_error;
  }

  FREE(TwoTheta); FREE(Coord1); FREE(Coord2); FREE(Weight);
  FREE(ParStart); FREE(ParMode); FREE(ParOpt); FREE(ParDev);
  FREE(ParStat);

  free_lists();

  fprintf( stderr, "%s %s\n", argv[0], "succesfully completed" );

  return( status );

main_error:

  status |= 1;

  fprintf(stderr,"ERROR: status=%d\n",status);

  if ( status & RocaOptimizationError ) {
    fprintf(stderr,"The optimization loop did not terminate successfully.\n");
    fprintf(stderr,"Adjust the start parameters or modify the input data.\n");
  }

  FREE(TwoTheta); FREE(Coord1); FREE(Coord2); FREE(Weight);
  FREE(ParStart); FREE(ParMode); FREE(ParOpt); FREE(ParDev);
  FREE(ParStat);

  free_lists();

  return( status );

} /* MAIN */
