/***************************************************************************/
/* Written 1994++ by Peter Boesecke                                        */
/* Copyright (C) 2011 European Synchrotron Radiation Facility              */
/*                       Grenoble, France                                  */
/*                                                                         */
/*    Principal authors: Peter Boesecke  (boesecke@esrf.eu)                */
/*                                                                         */
/*    This program is free software: you can redistribute it and/or modify */
/*    it under the terms of the GNU General Public License as published by */
/*    the Free Software Foundation, either version 3 of the License, or    */
/*    (at your option) any later version.                                  */
/*                                                                         */
/*    This program is distributed in the hope that it will be useful,      */
/*    but WITHOUT ANY WARRANTY; without even the implied warranty of       */
/*    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the        */
/*    GNU General Public License for more details.                         */
/*                                                                         */
/*    You should have received a copy of the GNU General Public License    */
/*    along with this program.  If not, see <http://www.gnu.org/licenses/>.*/
/***************************************************************************/
/*+++
1 mca2saxs
  Input of mca data

2 PURPOSE
  Input of mca data

  Arguments:
  mca2saxs [options] <i1nam> <onam> <i1fst> <i1lst> <i1inc> 
             <odum> <odim1> <odim2>

  Defaults:
  <input file name>   : input.edf 
  <output file name>  : output.edf
  <first image>       : <first image number in input file>
  <last image>                              : <last image number in input file>
     if argument list ends with first image : <first image>
  <increment>         : 1
  <dummy>             : <dummy value in first image of input file>
  <dimension 1>       : <horizontal dimension of first image in input file>
  <dimension 2>       : <vertical dimension of first image in input file>
  <output const>      : 0.0
  <output factor>     : 1.0

  Wildcard:
  A "=" in the argument list chooses the default.

2 HISTORY
  2003-04-28  PB V4.50 from saxs_mac V4.53
  2003-05-25  PB V4.51 ib[].Offset.V -> flexp
  2003-05-30  PB V4.52 ImageLoop: new function call
  2003-06-04  PB V4.53 Skip, ihb[1].ImNum -> num[1]
  2003-08-11  PB V4.54 option_define, all option numbers removed
  2003-11-30  PB V4.55 BinSiz
  2003-12-04  PB V4.56 user_io: ib[1].FileNumber -> ib[1].FFirst.V
  2004-10-01  PB V4.57 WriteHeader
  2004-10-30  PB V4.58 MAKE_FUNCTION
  2005-07-11  PB V4.59 exit -> return
  2005-07-15  PB V4.60 ReportSaxsStatus parameter updated, 
                       SetSaxsErrorExternal added
  2005-10-22  PB V4.61 SetSaxsErrorMessage added
  2007-06-11  PB V4.62 ImageLoop parameters updated
  2012-01-31  PB V4.63 ImageLoop parameters updated
  2017-10-21  PB V4.64 *keyvalue parameters updated

---*/
# define Version  "mca2saxs V4.64 2017-10-19, Peter Boesecke"

# include <errno.h>
# include <stdio.h>
# include <unistd.h>
# include <fcntl.h>

# include "SaxsPrograms.h"
# include "mca.h"

# define Usage "[options] \n\
                <i1nam> <onam> <i1fst> <i1lst> <i1inc> \n\
                <odum> <odim1> <odim2>"

# define BlockNum 2       /* 1 input sequence + 1 output sequence */

//+++++++++++ MODIF START
 /* Define Int options */
# define SSkip "skip"
//+++++++++++ MODIF END

/*---------------------------------------------------------------------------
1 mca2saxs

2 PURPOSE
  Multiplicates an image with a factor and adds a constant
---------------------------------------------------------------------------*/
void mca2saxs (CmdBlk * pcb, long num[], ImgHeadBlk ihb[], int * pstatus)
{ const float eps=1e-32;
  int i,imax;
  int j;

  int ErrorValue;

  IO_long Skip;

  float *I0Data;
  int   I0Dim_1,I0Dim_2;
  float *pI0Data;
  float I0Dummy, I0DDummy;

  int   I1Ave;
  float *I1Data;
  int   I1Dim_1,I1Dim_2;
  float I1Dummy, I1DDummy;
  float I1Value, I1Sum, I1Weight;

  int i_1, i_2;
  float f_1[BlockNum], f_2[BlockNum];

  float Off_1[BlockNum], Off_2[BlockNum];
  float Ps_1[BlockNum], Ps_2[BlockNum];


  float f1_1[BlockNum], f3_1[BlockNum], Df_1[BlockNum];
  float f1_2[BlockNum], f3_2[BlockNum], Df_2[BlockNum]; 

  int Imin_1, Imin_2, Imax_1, Imax_2;
 
  float Wmin_1, Wmax_1, DW_1;
  float Wmin_2, Wmax_2, DW_2;

// +++++++++++ MODIF START

  char mcafilename[IO_size];
  McaData * mcadata;
  const char * chann;
  int dim, offset;
  long buflen;
// +++++++++++ MODIF END

  *pstatus = Success;

  imax = pcb->ImgBlkLen;
  printf("\n Calculating ihb[0,% d] = Function(",ihb[0].ImNum);
  for(i=1;i<imax;i++) {
    printf("ihb[% d,% d] ", i, ihb[i].ImNum); }

  printf(")\n\n");

 /* Check the number of images */
  if (pcb->ImgBlkLen!=2) {
     fprintf(ERROUT,"%d images found, 1 input and 1 output image required\n",
             pcb->ImgBlkLen); *pstatus=Failed; return; }

// +++++++++++ MODIF START
 /* --- read mca file */

  /* read data */
  buflen = longkeyvalue(pcb->ib[1].Dim[1].V,pcb,ihb,num,0,pstatus);
  if (*pstatus!=Success) return;

  if ( !(mcadata = mca_New( buflen )) ) {
    *pstatus = NotEnoughMemoryAvailable;
    return;
  }

  filename_parameter (mcafilename,IO_size,pcb->ib[1].Name.V,0);
  filename_pattern (mcafilename,IO_size,mcafilename, pcb->ib[1].FileNumber);
  Skip = option_long ( SSkip, pcb, ihb, num, 1, pstatus );
  if (*pstatus!=Success) return;
  printf("Reading %s scan number %d\n",mcafilename,ihb[1].ImNum);
  if ((mca_Read( mcadata, mcafilename, num[1], Skip.V, 1 ))) {
    mca_Free( mcadata );
    *pstatus = FileReadError;
    return;
  }

  if (pcb->TestBit) {
    printf("mcadata->dim = %d\n", mcadata->dim); 
    printf("mcadata->len = %d\n", mcadata->len);
    printf("mcadata->buflen = %d\n", mcadata->buflen);
  }

  ihb[1].Data = mcadata->data;
  mcadata->data = (float *) NULL; // unlink data from mca

  // read offset and dim 
  ihb[1].Dim[0] = 2;
  ihb[1].Dim[1] = mcadata->len;
  ihb[1].Dim[2] = 1;

  edf_search_header_element ( mcadata->header_key, "@CHANN", &chann,
                              &ErrorValue, pstatus );
  SetSaxsErrorExternal ( ErrorValue, ReportSaxsImageError );
  if (*pstatus!=Success) return;

  if (sscanf(chann,"%d%d",&dim,&offset)!=2) {
     // +++++++++++++++++
     fprintf(ERROUT,"WARNING: Conversion error in line \"%s\"\n",
       chann);
     ; // conversion error
  }
  if (dim!=mcadata->len) {
     fprintf(ERROUT,
       "WARNING: Number of declared (%d) and read data points (%d) differ\n", 
       dim, mcadata->len);
  }

  if (!ihb[1].Offset[1].I) { 
    ihb[1].Offset[1].V = offset; ihb[1].Offset[1].I=TRUE; 
  }
  if (!ihb[1].Offset[2].I) {
    ihb[1].Offset[2].V = 0; ihb[1].Offset[2].I=TRUE;
  }

  if (!ihb[0].Offset[1].I) {
    ihb[0].Offset[1].V = ihb[1].Offset[1].V;
    ihb[0].Offset[1].I = TRUE;
  }
  if (!ihb[0].Offset[2].I) {
    ihb[0].Offset[2].V = ihb[1].Offset[2].V;
    ihb[0].Offset[2].I = TRUE;
  }

// +++++++++++ MODIF END

  GetReferenceParameters( pcb, ihb, 0, imax-1,
                          Off_1, Off_2, Ps_1,  Ps_2, pstatus );
  if (*pstatus!=Success) return;

  UpdateImageParameters ( pcb, ihb, 1, imax-1,
                          Off_1, Off_2, Ps_1,  Ps_2, pstatus );
  if (*pstatus!=Success) return;

  GetImageRange         ( pcb, ihb, 1, imax-1,
                          Off_1, Off_2, Ps_1,  Ps_2,
                          f1_1, f3_1, Df_1, f1_2, f3_2, Df_2,
                          &Imin_1, &Imax_1, &Imin_2, &Imax_2,
                          &Wmin_1, &Wmax_1, &DW_1, &Wmin_2, &Wmax_2, &DW_2, 1);

 /* 1 input, 1 output */
  I0Data  = ihb[0].Data;
  I0Dummy = ihb[0].Dummy.V;
  I0DDummy = ihb[0].DDummy.V;
  I0Dim_1  = (int) ihb[0].Dim[1];
  I0Dim_2  = (int) ihb[0].Dim[2];

  I1Ave   = pcb->ib[1].Ave.V;
  I1Data  = ihb[1].Data;
  I1Dummy = ihb[1].Dummy.V;
  I1DDummy = ihb[1].DDummy.V;
  I1Dim_1  = (int) ihb[1].Dim[1];
  I1Dim_2  = (int) ihb[1].Dim[2];

  /* loop over the output array  */
  for (i=0;i<imax;i++) f_2[i]=f1_2[i];
  for (i_2=Imin_2;i_2<=Imax_2;i_2++) {

    pI0Data = ABSPTR(I0Data,I0Dim_1,I0Dim_2,Imin_1,i_2);
    for (i=0;i<imax;i++) f_1[i]=f1_1[i];
    for (i_1=Imin_1;i_1<=Imax_1;i_1++) {

      if (Isum2ldw (I1Data,I1Dim_1,I1Dim_2,I1Dummy,I1DDummy,
          f_1[1], f_2[1], f_1[1]+Df_1[1], f_2[1]+Df_2[1], &I1Sum, &I1Weight)) {
        /* then do something with the data */
 
        I1Value = I1Sum; if (I1Ave) I1Value /= I1Weight;

        UPDATE( *pI0Data, I1Value, I0Dummy, I0DDummy );

      }
      pI0Data++;
      for (i=0;i<imax;i++) f_1[i]+=Df_1[i];

    } /* for i_1 ... */

    for (i=0;i<imax;i++) f_2[i]+=Df_2[i];
  } /* for i_2 ... */

// +++++++++++ MODIF START
  if (pcb->PassHeader.V) {
    WriteHeader ( pcb, 0, ihb, mcadata->header_key, pstatus );
    if (*pstatus!=Success) return;
  }
// +++++++++++ MODIF END

// +++++++++++ MODIF START
  if ( (mca_Free( mcadata )) ) {
    *pstatus = FileClosingError;
    return;
  }
// +++++++++++ MODIF END

} /* mca2saxs*/

/*---------------------------------------------------------------------------
user_io
Do all the keyboard io and return cb, and ib
---------------------------------------------------------------------------*/
void user_io(CmdBlk * pcb, ImgBlk * ib, int * pstatus)
{
  char  progfn[IO_size];
  ImgHeadBlk ihb[BlockNum];
  long num[BlockNum];

  IO_long Skip;

  float ROff_1, RPs_1, UOff_1, UPs_1;
  float ROff_2, RPs_2, UOff_2, UPs_2;

// +++++++++++ MODIF START
  char mcafilename[IO_size];
  char fname_pattern[IO_size];
  char tmp_line[IO_size];
  McaData * mcadata;
  const char * chann;
  int dim, offset;
  int ErrorValue;
// +++++++++++ MODIF END

 /* Determine program name without directory */
   (void) RightFR((char *) pcb->argv[0],DirSeparator,progfn,IO_size);

  /* show usage if no arguments are given */
  if (pcb->argc<=1) printf("Usage: %s %s\n",progfn,Usage);

  /*--- Write name of program ---*/
  printf("\n");
  printf("   %s %s\n",progfn,Version);
  printf("\n");

//+++++++++++ MODIF START

  if (!ib[1].Dim[1].I) sprintf(ib[1].Dim[1].V,"8192");
  if (!ib[1].Dim[2].I) sprintf(ib[1].Dim[2].V,"1");

//+++++++++++ MODIF END 

//+++++++++++ MODIF START
  /*--- Argument  : input file name */
  argv_filename( pcb, "Input file name", ib, 1, "data_001.mca", pstatus);
  if (*pstatus!=Success) return;

  /*--- Argument  : output file name */
  argv_filename(pcb,"Output file name",ib,0,"output.edf",pstatus);
  if (*pstatus!=Success) return;

  /*--- OpenMode of input file */
  if (!ib[1].OpenMode.I) {
    ib[1].OpenMode.V = IO_Old | IO_FileProtect | IO_DontReadWrite | IO_DontOpen;
    ib[1].OpenMode.I = TRUE;
    }

  /* Under no circumstances read any data */
  ib[1].OpenMode.V = ib[1].OpenMode.V | IO_DontReadData;

  /*--- OpenMode of output file */
  if (!ib[0].OpenMode.I) {
    ib[0].OpenMode.V = IO_ImageProtect;
    ib[0].OpenMode.I = TRUE;
    }

 /* --- read mca file */

  /* Generate filename of first binary file and test, whether it exists */
  filename_pattern (fname_pattern,IO_size,ib[1].Name.V,ib[1].FFirst.V);
   if (!filename_exists(fname_pattern)) { 
     SetSaxsErrorMessage ( fname_pattern );
     *pstatus = FileNotFound; return; 
   }

  /* do not read data */
  if ( !(mcadata = mca_New( 0 )) ) {
    *pstatus = NotEnoughMemoryAvailable;
    return;
  }

  filename_parameter (mcafilename,IO_size,ib[1].Name.V,0);
  filename_pattern (mcafilename,IO_size,mcafilename, ib[1].FFirst.V);

  num[0] = ib[0].First.V; num[1] = ib[1].First.V;
  Skip = option_long ( SSkip, pcb, ihb, num, 1, pstatus );
  if (*pstatus!=Success) return;

  printf("Reading %s scan number %d\n",mcafilename,ib[1].First.V);
  if ((mca_Read( mcadata, mcafilename, ib[1].First.V, Skip.V, 0 ))) { //++++++++++++++++
    mca_Free( mcadata );
    SetSaxsErrorMessage ( mcafilename );
    *pstatus = FileReadError;
    return;
  }

  if (pcb->TestBit) {
    printf("mcadata->dim = %d\n", mcadata->dim);
    printf("mcadata->len = %d\n", mcadata->len);
    printf("mcadata->buflen = %d\n", mcadata->buflen);
  }

  // read offset and dim
  if (!ib[1].Dim[1].I) sprintf(ib[1].Dim[1].V,"%d",mcadata->dim);
  if (!ib[1].Dim[2].I) sprintf(ib[1].Dim[2].V,"1");

  edf_search_header_element ( mcadata->header_key, "@CHANN", &chann,
                              &ErrorValue, pstatus );
  SetSaxsErrorExternal ( ErrorValue, ReportSaxsImageError );
  if (*pstatus!=Success) return;

  if (sscanf(chann,"%d%d",&dim,&offset)<2) {
     // +++++++++++++++++
     fprintf(ERROUT,"WARNING: Conversion error in line \"%s\"\n",
       chann);
     ; // conversion error
  }
//  printf("dim = %d, offset = %d\n",dim,offset); //++++++++++

  if (dim!=mcadata->dim) {
     fprintf(ERROUT,
       "WARNING: Number of declared (%d) and read data points (%d) differ\n",
       dim, mcadata->dim);
  }

  if (!ib[1].Offset[1].I) sprintf(ib[1].Offset[1].V,"%d",offset); 
  ib[1].Offset[1].I=TRUE;
  if (!ib[1].Offset[2].I) sprintf(ib[1].Offset[2].V,"0"); 
  ib[1].Offset[2].I=TRUE;

  /*--- output dimension 1 */
  argv_lvexp( pcb,"Output dimension 1", &ib[0].Dim[1],ib[1].Dim[1].V, pstatus);
  if (*pstatus!=Success) return;
printf("ib[0].Dim[1].V = %s\n",ib[0].Dim[1].V);//+++++++++++++++

  /*--- output dimension 2 */
  argv_lvexp( pcb,"Output dimension 2", &ib[0].Dim[2],ib[1].Dim[2].V, pstatus);
  if (*pstatus!=Success) return;
printf("ib[0].Dim[2].V = %s\n",ib[0].Dim[2].V);//+++++++++++++++

//+++++++++++ MODIF END 

//+++++++++++ MODIF START
//  ArgvFilenames ( pcb, ib, ihb, 0, BlockNum-1, pstatus);
//  if (*pstatus!=Success) return;

//  GetReference(pcb->RSys.V,1,ihb,&ROff_1,&ROff_2,&RPs_1,&RPs_2,pstatus );
//  if (*pstatus!=Success) return;
//  GetReference(pcb->USys.V,1,ihb,&UOff_1,&UOff_2,&UPs_1,&UPs_2,pstatus );
//  if (*pstatus!=Success) return;

  /*--- Argument  : multiplication factor */
//  printf("<output image> = <input image> * Factor + Const\n");
//  argv_flexp(pcb,"Multiplication factor",&ib[0].Factor,"1.0",pstatus);
//  if (*pstatus!=Success) return;

  /*--- Argument  : additive constant */
//  printf("<output image> = <input image> * Factor + Const\n");
//  argv_flexp(pcb,"Addition constant",&ib[0].Const,"0.0",pstatus);
//  if (*pstatus!=Success) return;
//+++++++++++ MODIF END 

  printf("\n");
  if (ib[1].Name.I)    printf("i/p file           : %s\n",ib[1].Name.V);
  if (ib[0].Name.I)    printf("o/p file           : %s\n",ib[0].Name.V);
  if (ib[1].First.I)   printf("first image        : %d\n",ib[1].First.V);
  if (ib[1].Last.I)    printf("last image         : %d\n",ib[1].Last.V);
  if (ib[1].Inc.I)     printf("increment          : %d\n",ib[1].Inc.V);
//  if (ib[0].Dummy.I)   printf("output dummy       : %s\n",ib[0].Dummy.V);
  if (ib[0].Dim[1].I)  printf("output dimension 1 : %s\n",ib[0].Dim[1].V);
  if (ib[0].Dim[2].I)  printf("output dimension 2 : %s\n",ib[0].Dim[2].V);
//  if (ib[0].Factor.I)  printf("factor             : %s\n",ib[0].Factor.V);
//  if (ib[0].Const.I)   printf("constant           : %s\n",ib[0].Const.V);
  printf("\n");

  if (pcb->TestBit) {
    PrintBlocks ( pcb, ib );
    printf("\n"); }

  return;
}

/*---------------------------------------------------------------------------
main
---------------------------------------------------------------------------*/

#if MAKE_FUNCTION
# define MAIN main_mca2saxs
#else
# define MAIN main
#endif

int MAIN (int argc, char *argv[])
{
  CmdBlk cb;                /* command block  */
  ImgBlk ib[BlockNum];      /* image blocks */

  int status;
  int arg_no = 0;

 /* Init options, control block and image blocks */
  InitOptions( Usage, Version, NULL, TRUE, &cb, ib, BlockNum );

//+++++++++++ MODIF START
 /* Define options */
  option_define ( IO_tplvexp, SSkip, "0", &cb );
//+++++++++++ MODIF END

 /* Read options from argument list */
  ReadOptions( argv, &arg_no, &cb, ib, &status);

  /* USER KEYBOARD I/O */
  if (status==Success) {
    argv_start ( &cb, 1 );
    user_io( &cb, ib, &status);
    argv_end( &cb ); /* must be called after user_io */
  }

  /* SEQUENCE CALCULATION */
  if (status==Success)
    IMAGELOOP( &cb, ib, mca2saxs, NULL, NULL, TRUE, &status );

  return( ReportSaxsStatus( status, 0 ) );

} /* MAIN */

