/***************************************************************************/
/* Written 1994++ by Peter Boesecke                                        */
/* Copyright (C) 2011 European Synchrotron Radiation Facility              */
/*                       Grenoble, France                                  */
/*                                                                         */
/*    Principal authors: Peter Boesecke  (boesecke@esrf.eu)                */
/*                                                                         */
/*    This program is free software: you can redistribute it and/or modify */
/*    it under the terms of the GNU General Public License as published by */
/*    the Free Software Foundation, either version 3 of the License, or    */
/*    (at your option) any later version.                                  */
/*                                                                         */
/*    This program is distributed in the hope that it will be useful,      */
/*    but WITHOUT ANY WARRANTY; without even the implied warranty of       */
/*    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the        */
/*    GNU General Public License for more details.                         */
/*                                                                         */
/*    You should have received a copy of the GNU General Public License    */
/*    along with this program.  If not, see <http://www.gnu.org/licenses/>.*/
/***************************************************************************/
/* History
   2000-08-02 float -> double
   2000-08-04 %.20g -> %.15g
   2004-03-16 unit, ndigits 
   2004-05-30
   2004-06-19 debug
   2004-06-20 default of ndigits reduced to 10 (from 15)
   2004-10-30 MAKE_FUNCTION, all exits removed
   2005-02-12 ERROUT
   2006-08-10 pipe
   2007-03-08 ReportInputStatus replaced by ReportSaxsStatus, numio
   2008-05-21 format=hex using long2hex, should be replaced by double2hex
              because some number are too long for long int
   2008-05-22 double2hex
   2011-04-16 Use always standard input if no expression is given
              as argument, option pipe removed 
   2013-03-12 NULL_SPACE added to avoid segmentation fault in case of error
   2017-02-12 IODBG_NUMUNITS | IODBG_NUMCONSTANTS | IODBG_NUMPROGRAMS
   2017-02-28 edf_report_data_error updated using ifndef EDFIO - endif
*/
#	include <errno.h>
#	include <stdio.h>
#	include <fcntl.h>
#       include "SaxsError.h"
#	include "SaxsInput.h"
#       include "numio.h"

/* removes trailing line feed \n */
char * trimlf ( char * s )
{ size_t slen;
  if ( s ) {
    slen = strlen( s );
    if ( slen > 0 )
      if (s[slen-1]=='\n') s[slen-1]='\0';
  }
  return ( s );
} // trimlf

/* returns 1 if s is an empty string (only white space) otherwise 0. */
int isempty( const char *s )
{ const char *ps;
  int empty=1;

  if (s) {
    ps = s;
    while (*ps) { empty = empty && isspace(*ps); ps++; }
  }
  return( empty );

} //isempty

/*---------------------------------------------------------------------------
main
---------------------------------------------------------------------------*/
    
#if MAKE_FUNCTION
# define MAIN main_fcalc
#else
# define MAIN main
#endif
    
# define DECIMAL 1
# define HEXA 2

# define FCALC_VERSION "2017-02-12, Peter Boesecke"

# define FCALC_USAGE \
  "[units=<unit>] [digits=<digits>] [debug=<0|1>] [--help] <expression>"

# define NULL_SPACE( s) ( ( s)? ( s) : ("" ) )


int MAIN (int argc, char *argv[])
{ 
  int errval=0;
  int status=Success;
  int argcnt=1;
  double doublenumber = (double) 1.0;
  char * unit = "";
  int format = DECIMAL;
/*  int digits = 15; */
  int digits = -10; // if negative, positive hex values are normally written.
  int debug = 0;

  char *pstring[2];

  #define BUFLEN 1024
  char buffer[BUFLEN];

  // skip all options
  while (argc > argcnt) {
    if ( strncmp( argv[argcnt],"unit=", 5 ) == 0 ) {
      unit=&argv[argcnt][5];
      argcnt++; }
    else if ( strncmp( argv[argcnt],"format=", 7 ) == 0 ) {
       if ( strncmp( &argv[argcnt][7],"hex", 3 ) == 0 )
      format=HEXA;
      argcnt++; }
    else if ( strncmp( argv[argcnt],"digits=", 7 ) == 0 ) {
      sscanf(&argv[argcnt][7],"%d",&digits);
      argcnt++; }
    else if ( strncmp( argv[argcnt],"debug=", 6 ) == 0 ) {
      sscanf(&argv[argcnt][6],"%d",&debug);
      switch (debug) {
        case 0: numio_debug( 0l ); break;
        case 1: numio_debug( IODBG_NUMUNITS ); break;
        case 2: numio_debug( IODBG_NUMCONSTANTS ); break;
        case 3: numio_debug( IODBG_NUMPROGRAMS ); break;
        default:
          numio_debug( IODBG_NUMUNITS | IODBG_NUMCONSTANTS | IODBG_NUMPROGRAMS );
      }
      argcnt++; }
    else if ( strncmp( argv[argcnt],"--version", 9 ) == 0 ) {
      printf("\n %s version: %s\n\n", argv[0], FCALC_VERSION);
      printf(" %s\n",numio_version () );
      status=StoppedAfterHelp;
      argcnt++;
      break; }
    else if ( strncmp( argv[argcnt],"--help", 6 ) == 0 ) {
      printf("\n %s version: %s\n\n", argv[0], FCALC_VERSION);
      printf(" %s %s\n",argv[0],FCALC_USAGE);
      status=StoppedAfterHelp;
      argcnt++; 
      break; }
    else break;
  }

  if (status == Success) {
    if (argc>argcnt) {
      while (argc>argcnt) {
        doublenumber = doubleexpr(argv[argcnt],&status);
        if (status != Success) { ReportSaxsStatus(status,0); break; }
        if (format==HEXA) {
          printf( "%s\n",
            NULL_SPACE(num_double2hex(buffer,BUFLEN,doublenumber,digits,&errval)) );
        } else {
          printf( "%s\n",
            NULL_SPACE(num_double2str(buffer,BUFLEN,doublenumber,unit,digits,&errval)) );
        }
        if (errval) {
          fprintf(ERROUT,"ERROR: %s\n",num_errval2str( buffer,BUFLEN,errval ));
          status = Failed;
        }
        argcnt++;
      }
    } else {
      // read from standard input
      pstring[1] = (char *) NULL;
      while ( !feof( stdin ) ) {
        pstring[0] = fgets ( buffer, BUFLEN, stdin );
        pstring[0] = trimlf ( pstring[0] );
  
        if ( pstring[0] ) {
          if ( (!strncmp(pstring[0],"exit",4)) || (!strncmp(pstring[0],"quit",4)) )
            return ( status );
          // edfdump

          if (!isempty (pstring[0])) {
            doublenumber = doubleexpr(pstring[0],&status);
            if (status != Success) ReportSaxsStatus(status,0);
            else printf( "%s\n",
              NULL_SPACE(num_double2str(buffer,BUFLEN,doublenumber,unit,digits,&errval)) );
            if (errval) {
              fprintf(ERROUT,"ERROR: %s\n",num_errval2str( buffer,BUFLEN,errval ));
              status = Failed;
            }
          } else fprintf( stdout,"\n");
        }
      }
    }
  }
  return ( status );
} /* MAIN */

# ifndef EDFIO
/* empty definition, not used */
const char *edf_report_data_error     ( int ErrorValue )
{ return("ExternalError not available\n") ;}
# endif // EDFIO

