/***************************************************************************/
/* Written 1994++ by Peter Boesecke                                        */
/* Copyright (C) 2011 European Synchrotron Radiation Facility              */
/*                       Grenoble, France                                  */
/*                                                                         */
/*    Principal authors: Peter Boesecke  (boesecke@esrf.eu)                */
/*                                                                         */
/*    This program is free software: you can redistribute it and/or modify */
/*    it under the terms of the GNU General Public License as published by */
/*    the Free Software Foundation, either version 3 of the License, or    */
/*    (at your option) any later version.                                  */
/*                                                                         */
/*    This program is distributed in the hope that it will be useful,      */
/*    but WITHOUT ANY WARRANTY; without even the implied warranty of       */
/*    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the        */
/*    GNU General Public License for more details.                         */
/*                                                                         */
/*    You should have received a copy of the GNU General Public License    */
/*    along with this program.  If not, see <http://www.gnu.org/licenses/>.*/
/***************************************************************************/
/*+++
1 gas2saxs 

2 PURPOSE

2 HISTORY
  17-Feb-1995 PB  
  19-Feb-1995 PB 
  24-Feb-1995 PB new versions of FullFileName and FileExists
  06-Apr-1995 PB problem with scaler values  
  20-May-1995 PB modified for dld header version 1.0
  17-Aug-1995 PB HmScalerDepth added
  08-Oct-1995 PB Keyword SaxsDataVersion added (1.0)
  10-Oct-1995 PB Keywords followed by empty strings are suppressed.
  28-Oct-1995 PB HmScalerDepth and Scalars -> float (2.13)
  28-Nov-1995 PB V3.0
  08-Dec-1995 PB V3.1 for new dld data (V2.0)
  20-Jan-1996 PB V3.2 new GetDldFrame (V2.1)
  21-Jan-1996 PB V3.3 DetectorPosition (GetDldFrame V2.2)
  15-Mar-1998 PB V4.0 new edf routines
  18-Apr-1999 PB V4.01 new call to edf routines
  30-Apr-1999 PB V4.02 status set to Success before calling edf routines
  16-Jun-1999 PB V4.03 edfio_debug_set added
  26-Jun-1999 PB DevSeparator, DirSeparator, ExtSeparator defined in SaxsInput
  23-Jul-1999 PB V4.04 call to edf_write_data_2d_raw updated for edfio 
                 version 1.07, SaxsDataVersion = 2.0 
  22-Dec-1999 PB V4.05 SubTitle -> ExperimentInfo
  2000-08-04  PB %f -> %g
  2000-11-10  PB V4.10 changes for edfio V1.30
  2000-11-12  PB V4.11 changes for edfio V1.32
                 V4.12 changes for edfio V1.34
  2000-12-31  PB V4.13 Dim[3] not used any more
  2001-07-03  PB V4.14 filename_full
  2003-06-04  PB V4.15 DefExtenstion -> pExtension
  2003-08-11  PB V4.16 option_define, all option numbers removed
  2004-10-31  PB V4.17 MAKE_FUNCTION, user_io: status->*pstatus,
                       ReportStatus removed
  2005-07-15  PB V4.18 ReportSaxsStatus parameter updated
  2007-04-23  PB V4.19 write_data_raw: eps defined as float
  2008-12-05  PB V4.20 open all input files as binary file to avoid
                       translation of \r\n to \n when compiled on windows.
  2017-02-12  PB V4.21 update for iodbg 


---*/
# define Version  "V4.21 2017-02-12, Peter Boesecke"

# include <errno.h>
# include <unistd.h>
# include <stdio.h>
# include <fcntl.h>

# include "SaxsPrograms.h"
# include "GetDldFrame.h"
# include "edfio.h"

# define Wild         "="
# define Usage "\n<i/p file> <dummy> <first> <last> <inc> <o/p file> <o/p first> <i/p hm>"

# define Default_ipfn      "gasdet001"
# define Default_extension DefaultExtension 
# define Default_dummy     -1
# define Default_ininc     1
# define Hm_extension      "hm"

/* Option definitions */
# define SDefExt           "ext"

/* Error value definition, provisorically */
# ifndef GasFileNotFound
#  define GasFileNotFound CouldNotOpenFile
# endif
# ifndef GasImageProtected
#  define GasImageProtected WriteDataError
# endif
# ifndef GasFileProtected
#  define GasFileProtected FileIsNotWritable 
# endif

# ifndef TRUE
#   define TRUE 1
# endif
# ifndef FALSE
#   define FALSE 0
# endif

static int G2S_debug = 0;

/*+++------------------------------------------------------------------------
1 ImageNumberExists

2 DESCRIPTION
  Tests, whether an image exists already.

  return( int ) : (o)   TRUE    : image number found
                        FALSE   : error or image number not found

  int ImageNumber (i)           : image number of the header
  int *pErrorValue (o)          : error value

2 HISTORY
  17-Feb-1995   Peter Boesecke

  ---------------------------------------------------------------------------*/
int ImageNumberExists      ( int stream, int ImageNumber, int * pErrorValue )
{ int found = 0;
  int DataChain = 1;
  long *Dim, Tmp[4];
  size_t DataArraySize;
  int status = Success;

  if (G2S_debug) printf("ImageNumberExists BEGIN\n");

  Dim=Tmp; Dim[0] = 2l;
  found = edf_read_data_dimension ( stream, ImageNumber, DataChain,
                                    &Dim, &DataArraySize,
                                    pErrorValue, &status ); 

  if (G2S_debug) printf("ImageNumberExists END\n");
 
  return ( found );

} /* ImageNumberExists */

/*---------------------------------------------------------------------------
  write_data_raw
  Writes raw data to the block "<keyword> = <ImNum>"
  A raw data header is created. 
  return value TRUE if no Error
  SaxsDataVersion 2.0
  ---------------------------------------------------------------------------*/

void write_data_raw ( int stream,
                   char *HmDldVersion,
                   int ImNum, 
                   char *HmFile,
                   int HmFrame,
                   int DataLength, 
                   int Dim_1, int Dim_2, 
                   int Off_1, int Off_2, 
                   void *Start,
                   float Scaler[],
                   char * HmScalerName[], 
                   float HmScalerZero[], 
                   float HmScalerCalib[],
		   float HmScalerDepth,
                   int HmScalerI0, 
                   int HmScalerI1, 
                   int HmScalerAnode, 
                   int HmScalerTime,
                   int HmScalerI0S,
                   int HmScalerI1S,
                   int HmScalerAnodeS,
                   int HmScalerTimeS,
                   char * HmStartTime,
                   float HmDeltaTime,
                   float Dummy,
                   char * Title,
                   char * ExperimentInfo, 
                   float Center_1, float Center_2, int ICenter,
                   float PixSize_1, float PixSize_2, int IPixSize,
                   float WaveLength, int IWaveLength,
                   float SampleDistance, int ISampleDistance,
		   float DetectorPosition, int IDetectorPosition,
                   char * DetectorType, 
                   char * MachineInfo, 
                   char * OpticsInfo,
                   char * StationInfo, 
                   char * ProposalInfo,
                   int * pErrorValue )

  {
  long  Dim[4];
  long  DataNumber = (long) ImNum;
  long  DataChain = 1;
  int   status = Success;
  int   cnt;
  int   DataType;
  const float eps = 1e-32;

  const char * SaxsDataVersion = "2.0"; 
   
  const int Hm_ScalerLen  = HScalerLen;
  const char * counterkey[HScalerLen] = { 
    KHScaler""KHC"01", KHScaler""KHC"02", KHScaler""KHC"03", KHScaler""KHC"04", 
    KHScaler""KHC"05", KHScaler""KHC"06", KHScaler""KHC"07", KHScaler""KHC"08", 
    KHScaler""KHC"09", KHScaler""KHC"10", KHScaler""KHC"11", KHScaler""KHC"12", 
    KHScaler""KHC"13", KHScaler""KHC"14", KHScaler""KHC"15", KHScaler""KHC"16", 
    KHScaler""KHC"17", KHScaler""KHC"18", KHScaler""KHC"19", KHScaler""KHC"20", 
    KHScaler""KHC"21", KHScaler""KHC"22", KHScaler""KHC"23", KHScaler""KHC"24", 
    KHScaler""KHC"25", KHScaler""KHC"26", KHScaler""KHC"27", KHScaler""KHC"28", 
    KHScaler""KHC"29", KHScaler""KHC"30", KHScaler""KHC"31", KHScaler""KHC"32"}; 
  const char * namekey[HScalerLen] = {
    KHScaler""KHN"01", KHScaler""KHN"02", KHScaler""KHN"03", KHScaler""KHN"04",
    KHScaler""KHN"05", KHScaler""KHN"06", KHScaler""KHN"07", KHScaler""KHN"08",
    KHScaler""KHN"09", KHScaler""KHN"10", KHScaler""KHN"11", KHScaler""KHN"12",
    KHScaler""KHN"13", KHScaler""KHN"14", KHScaler""KHN"15", KHScaler""KHN"16",
    KHScaler""KHN"17", KHScaler""KHN"18", KHScaler""KHN"19", KHScaler""KHN"20",
    KHScaler""KHN"21", KHScaler""KHN"22", KHScaler""KHN"23", KHScaler""KHN"24",
    KHScaler""KHN"25", KHScaler""KHN"26", KHScaler""KHN"27", KHScaler""KHN"28",
    KHScaler""KHN"29", KHScaler""KHN"30", KHScaler""KHN"31", KHScaler""KHN"32"};

  const char * zerokey[HScalerLen] = {
    KHScaler""KHZ"01", KHScaler""KHZ"02", KHScaler""KHZ"03", KHScaler""KHZ"04",
    KHScaler""KHZ"05", KHScaler""KHZ"06", KHScaler""KHZ"07", KHScaler""KHZ"08",
    KHScaler""KHZ"09", KHScaler""KHZ"10", KHScaler""KHZ"11", KHScaler""KHZ"12",
    KHScaler""KHZ"13", KHScaler""KHZ"14", KHScaler""KHZ"15", KHScaler""KHZ"16",
    KHScaler""KHZ"17", KHScaler""KHZ"18", KHScaler""KHZ"19", KHScaler""KHZ"20",
    KHScaler""KHZ"21", KHScaler""KHZ"22", KHScaler""KHZ"23", KHScaler""KHZ"24",
    KHScaler""KHZ"25", KHScaler""KHZ"26", KHScaler""KHZ"27", KHScaler""KHZ"28",
    KHScaler""KHZ"29", KHScaler""KHZ"30", KHScaler""KHZ"31", KHScaler""KHZ"32"};

  const char * factorkey[HScalerLen] = {
    KHScaler""KHF"01", KHScaler""KHF"02", KHScaler""KHF"03", KHScaler""KHF"04",
    KHScaler""KHF"05", KHScaler""KHF"06", KHScaler""KHF"07", KHScaler""KHF"08",
    KHScaler""KHF"09", KHScaler""KHF"10", KHScaler""KHF"11", KHScaler""KHF"12",
    KHScaler""KHF"13", KHScaler""KHF"14", KHScaler""KHF"15", KHScaler""KHF"16",
    KHScaler""KHF"17", KHScaler""KHF"18", KHScaler""KHF"19", KHScaler""KHF"20",
    KHScaler""KHF"21", KHScaler""KHF"22", KHScaler""KHF"23", KHScaler""KHF"24",
    KHScaler""KHF"25", KHScaler""KHF"26", KHScaler""KHF"27", KHScaler""KHF"28",
    KHScaler""KHF"29", KHScaler""KHF"30", KHScaler""KHF"31", KHScaler""KHF"32"};

  register int i;

  if (G2S_debug) printf("write_data_raw BEGIN\n");
  if (G2S_debug>1)  edfio_debug_set(IODBG_DEBUG1); 

        *pErrorValue = RoutineSucceeded;


        /* SaxsDataVersion */
        edf_write_header_line ( stream, DataNumber, DataChain, 
                            KSaxsDataVersion, SaxsDataVersion, 
                            pErrorValue, &status );
        if (*pErrorValue!=RoutineSucceeded) return;

        /* HmFrame */
        edf_write_header_long ( stream, DataNumber, DataChain,
                            KHFrame, HmFrame,
                            pErrorValue, &status );
        if (*pErrorValue!=RoutineSucceeded) return;

        /* HmFile */
	if (strlen(HmFile)) {
          edf_write_header_line ( stream, DataNumber, DataChain,
                              KHFile, HmFile,
                              pErrorValue, &status );
          if (*pErrorValue!=RoutineSucceeded) return;
	  }

        /* Title */
	if ((Title!=(char *) NULL) && (strlen(Title))) {
          edf_write_header_line ( stream, DataNumber, DataChain,
                              KTitle, Title,
                              pErrorValue, &status );
          if (*pErrorValue!=RoutineSucceeded) return;
          }

        /* ExperimentInfo */
        if ((ExperimentInfo!=(char *) NULL) && (strlen(ExperimentInfo))) {
          edf_write_header_line ( stream, DataNumber, DataChain,
                              KExperimentInfo, ExperimentInfo,
                              pErrorValue, &status );
          if (*pErrorValue!=RoutineSucceeded) return;
          }

        /* HmStartTime */
        if (strlen(HmStartTime)>0) {
          edf_write_header_line ( stream, DataNumber, DataChain,
                              KHStartTime, HmStartTime,
                              pErrorValue, &status );
          if (*pErrorValue!=RoutineSucceeded) return;
          }

        /* HmDeltaTime */
        if (ABS(HmDeltaTime)>eps) {
          edf_write_header_float ( stream, DataNumber, DataChain,
                              KHDeltaTime, HmDeltaTime,
                              pErrorValue, &status );
          if (*pErrorValue!=RoutineSucceeded) return;
          }

        /* Dummy */
        edf_write_header_float( stream, DataNumber, DataChain,
                            KDummy, Dummy,
                            pErrorValue, &status );
        if (*pErrorValue!=RoutineSucceeded) return;

        /* ScalerLen */
        edf_write_header_long( stream, DataNumber, DataChain,
                            KHScalerLen, (long) Hm_ScalerLen,
                            pErrorValue, &status );
        if (*pErrorValue!=RoutineSucceeded) return;


	/* Scalers */
        for (i=0;i<Hm_ScalerLen;i++) {
          edf_write_header_float( stream, DataNumber, DataChain,
                              counterkey[i], Scaler[i],
                              pErrorValue, &status );
          if (*pErrorValue!=RoutineSucceeded) return;
        }

	/* Offsets */
        edf_write_header_long( stream, DataNumber, DataChain,
                            KOffset_1, (long) Off_1,
                            pErrorValue, &status );
        if (*pErrorValue!=RoutineSucceeded) return;

        edf_write_header_long( stream, DataNumber, DataChain,
                            KOffset_2, (long) Off_2,
                            pErrorValue, &status );
        if (*pErrorValue!=RoutineSucceeded) return;

        /* Centers */
        if (ICenter) {
          edf_write_header_float( stream, DataNumber, DataChain,
                              KCenter_1, Center_1,
                              pErrorValue, &status );
          if (*pErrorValue!=RoutineSucceeded) return;

          edf_write_header_float( stream, DataNumber, DataChain,
                              KCenter_2, Center_2,
                              pErrorValue, &status );
          if (*pErrorValue!=RoutineSucceeded) return;
          }

        if (IPixSize) {
          edf_write_header_float( stream, DataNumber, DataChain,
                              KPixSiz_1, PixSize_1,
                              pErrorValue, &status );
          if (*pErrorValue!=RoutineSucceeded) return;

          edf_write_header_float( stream, DataNumber, DataChain,
                              KPixSiz_2, PixSize_2,
                              pErrorValue, &status );
          if (*pErrorValue!=RoutineSucceeded) return;
          }

        if (IWaveLength) {
          edf_write_header_float( stream, DataNumber, DataChain,
                              KWaveLength, WaveLength,
                              pErrorValue, &status );
          if (*pErrorValue!=RoutineSucceeded) return;
          }

        if (ISampleDistance) {
          edf_write_header_float( stream, DataNumber, DataChain,
                              KSampleDistance, SampleDistance,
                              pErrorValue, &status );
          if (*pErrorValue!=RoutineSucceeded) return;
          }

        if (IDetectorPosition) {
          edf_write_header_float( stream, DataNumber, DataChain,
                              KDetectorPosition, DetectorPosition,
                              pErrorValue, &status );
          if (*pErrorValue!=RoutineSucceeded) return;
          }

        if ((DetectorType!=(char *) NULL) && (strlen(DetectorType))) {
          edf_write_header_line( stream, DataNumber, DataChain,
                              KDetectorType, DetectorType,
                              pErrorValue, &status );
          if (*pErrorValue!=RoutineSucceeded) return;
          }

        if ((MachineInfo!=(char *) NULL) && (strlen(MachineInfo))) {
          edf_write_header_line( stream, DataNumber, DataChain,
                              KMachineInfo, MachineInfo,
                              pErrorValue, &status );
          if (*pErrorValue!=RoutineSucceeded) return;
          }

        if ((OpticsInfo!=(char *) NULL) && (strlen(OpticsInfo))) {
          edf_write_header_line( stream, DataNumber, DataChain,
                              KOpticsInfo, OpticsInfo,
                              pErrorValue, &status );
          if (*pErrorValue!=RoutineSucceeded) return;
          }

        if ((StationInfo!=(char *) NULL) && (strlen(StationInfo))) {
          edf_write_header_line( stream, DataNumber, DataChain,
                              KStationInfo, StationInfo,
                              pErrorValue, &status );
          if (*pErrorValue!=RoutineSucceeded) return;
          }

        /* Keywords for Version >=1.0 */

      if (strcmp(HmDldVersion,"1.0")>=0) {
        for (i=0;i<Hm_ScalerLen;i++) {
          edf_write_header_line( stream, DataNumber, DataChain,
                             namekey[i], HmScalerName[i],
                             pErrorValue, &status );
          if (*pErrorValue!=RoutineSucceeded) return;

          edf_write_header_float( stream, DataNumber, DataChain,
                              zerokey[i], HmScalerZero[i],
                              pErrorValue, &status );
          if (*pErrorValue!=RoutineSucceeded) return;

          edf_write_header_float( stream, DataNumber, DataChain,
                              factorkey[i], HmScalerCalib[i],
                              pErrorValue, &status );
          if (*pErrorValue!=RoutineSucceeded) return;
 
          } /* for */

        /* scaler number for incident beam intensity I0 */
        edf_write_header_long( stream, DataNumber, DataChain,
                            KHScalerI0, (long) HmScalerI0,
                            pErrorValue, &status );
        if (*pErrorValue!=RoutineSucceeded) return;

        /* scaler number for transmitted beam intensity I1 */
        edf_write_header_long( stream, DataNumber, DataChain,
                            KHScalerI1, (long) HmScalerI1,
                            pErrorValue, &status );
        if (*pErrorValue!=RoutineSucceeded) return;

        /* scaler number for anode counts */
        edf_write_header_long( stream, DataNumber, DataChain,
                            KHScalerAnode, (long) HmScalerAnode,
                            pErrorValue, &status );
        if (*pErrorValue!=RoutineSucceeded) return;

        /* scaler number for time */
        edf_write_header_long( stream, DataNumber, DataChain,
                            KHScalerTime, (long) HmScalerTime,
                            pErrorValue, &status );
        if (*pErrorValue!=RoutineSucceeded) return;

        } /* if (strcmp(HmDldVersion,"1.0")>=0) */

      if (strcmp(HmDldVersion,"1.1")>=0) {

        /* scaler depth */
        edf_write_header_float( stream, DataNumber, DataChain,
                            KHScalerDepth, HmScalerDepth,
                            pErrorValue, &status );
        if (*pErrorValue!=RoutineSucceeded) return;

        /* 2nd scaler number for incident beam intensity I0 */
        edf_write_header_long ( stream, DataNumber, DataChain,
                            KHScalerI0S, (long) HmScalerI0S,
                            pErrorValue, &status );
        if (*pErrorValue!=RoutineSucceeded) return;

        /* 2nd scaler number for transmitted beam intensity I1 */
        edf_write_header_long ( stream, DataNumber, DataChain,
                            KHScalerI1S, (long) HmScalerI1S,
                            pErrorValue, &status );
        if (*pErrorValue!=RoutineSucceeded) return;

        /* 2nd scaler number for anode counts */
        edf_write_header_long ( stream, DataNumber, DataChain,
                            KHScalerAnodeS, (long) HmScalerAnodeS,
                            pErrorValue, &status );
        if (*pErrorValue!=RoutineSucceeded) return;

        /* 2nd scaler number for time */
        edf_write_header_long ( stream, DataNumber, DataChain,
                            KHScalerTimeS, (long) HmScalerTimeS,
                            pErrorValue, &status );
        if (*pErrorValue!=RoutineSucceeded) return;

        }  /* if (strcmp(HmDldVersion,"1.2")>=0) */

        switch (DataLength) {
          case 2: /* 2 bytes */
            DataType = Unsigned16;
            break;
          case 4: /* 4 bytes */
            DataType = Unsigned32;
            break;
          default: /* conversion error */
             *pErrorValue=BadDataBlock;
             return;
          } /* switch */

        /* Dim */
        Dim[0] = 2l; Dim[1] = (long) Dim_1; Dim[2] = (long) Dim_2;

        edf_write_data_raw ( stream, DataNumber, DataChain, Dim,
                             Start, DataType, 0l, HighByteFirst, 1l, 
                             pErrorValue, &status );
        if (*pErrorValue!=RoutineSucceeded) return;

        *pErrorValue = RoutineSucceeded;

  if (G2S_debug) printf("write_data_raw END\n");

  return;
} /* write_data_raw */

/*+++------------------------------------------------------------------------
Dld2Saxs
----------------------------------------------------------------------------*/
void Dld2Saxs(CmdBlk * pcb, FILE * InFstream, int In, int DataLength, 
              int Dim_1, int Dim_2, int Off_1, int Off_2,
              FILE * InHmstream, int OutStream, int Out, 
              char InName[], float Dummy, 
              int * pErrorValue) 
{ 
  int      status;
  void   * Start = (void *) NULL;
  float    HmScaler[HScalerLen];
  char   * HmScalerName[HScalerLen];
  float    HmScalerZero[HScalerLen];
  float    HmScalerCalib[HScalerLen];
  float    HmScalerDepth;
  int      HmScalerI0, HmScalerI1;
  int      HmScalerAnode, HmScalerTime;
  int      HmScalerI0S, HmScalerI1S;
  int      HmScalerAnodeS, HmScalerTimeS;
  float    HmDeltaTime;
  char     HmStartTime[IO_size];
  char   * HmDldVersion;

  float    Center_1, Center_2, PixSize_1, PixSize_2;
  float    WaveLength,SampleDistance,DetectorPosition; 
  int      ICenter, IPixSize, IWaveLength, ISampleDistance, IDetectorPosition;
  char   * Title = (char *) NULL;
  char   * ExperimentInfo = (char *) NULL;
  char   * DetectorType = (char *) NULL;
  char   * MachineInfo = (char *) NULL;
  char   * OpticsInfo = (char *) NULL;
  char   * StationInfo = (char *) NULL;
  char   * ProposalInfo = (char *) NULL;
  char   * HmFraming = (char *) NULL;

  register int i;

  *pErrorValue = EndOfFileDetected;

  if (G2S_debug) printf("Dld2Saxs BEGIN\n");
 
  /* reading gas file */
  /* --- read scaler information */
  if (G2S_debug) printf("ReadDldScalars\n");
  if (status=ReadDldScalars (InFstream, &HmDldVersion,
      HmScaler, HmScalerName, HmScalerZero, HmScalerCalib, &HmScalerDepth,
      &HmScalerI0, &HmScalerI1, &HmScalerAnode, &HmScalerTime,
      &HmScalerI0S, &HmScalerI1S, &HmScalerAnodeS, &HmScalerTimeS,
      &HmDeltaTime, HmStartTime, In)) {
     printf("ReadDldScalars return status = % d\n",status);return; }
  if (pcb->TestBit) {
    printf("DldVersion = % s\n",HmDldVersion);
    for (i=0;i<HScalerLen;i++) {
      printf("% sC%2u = % g\n", KHScaler, i+1, HmScaler[i]);       /* Channel */
      if (strcmp(HmDldVersion,"1.0")>=0) {
        printf("% sN%2u = % s\n", KHScaler, i+1, HmScalerName[i]);    /* Name */
        printf("% sZ%2u = % g\n", KHScaler, i+1, HmScalerZero[i]);    /* Zero */
        printf("% sS%2u = % g\n", KHScaler, i+1, HmScalerCalib[i]);  /* Shift */
        }
      }
    if (strcmp(HmDldVersion,"1.0")>=0) {
      printf("ScalerDepth = % g\n",HmScalerDepth);
      printf("ScalerI0    = % d\n",HmScalerI0); 
      printf("ScalerI1    = % d\n",HmScalerI1);
      printf("ScalerAnode = % d\n",HmScalerAnode);
      printf("ScalerTime  = % d\n",HmScalerTime);
      printf("DeltaTime   = % g\n",HmDeltaTime);
      printf("StartTime   = % s\n",HmStartTime);
      }
    } /* if (pcb->TestBit) ...*/

  /* --- read histogramming memory data */
  if (G2S_debug) printf("  Start = %p\n",Start);
  if (strcmp(HmDldVersion,"2.0")>=0) {
    if (G2S_debug) printf("  ReadDldData\n");
    if (status=ReadDldData (InHmstream, In, &Start)) {
       printf("ReadDldData return status = % d\n",status);return; }
    } else {
    if (G2S_debug) printf("  ReadDldDataV1\n");
    if (status=ReadDldDataV1 (InHmstream, In, &Start)) {
       printf("ReadDldDataV1 return status = % d\n",status);return; }
    } /* if (strcmp(HmDldVersion,"2.0")>=0) */
  if (G2S_debug) printf("  Start = %p\n",Start);

  /* --- read additional experiment information */ 
  if (G2S_debug) printf("  ReadDldExperiment\n");
  if (status=ReadDldExperiment(InFstream, 
         &Title, &ExperimentInfo, 
         &Center_1, &Center_2, &ICenter,
         &PixSize_1, &PixSize_2, &IPixSize,
         &WaveLength, &IWaveLength,
         &SampleDistance, &ISampleDistance,
	 &DetectorPosition, &IDetectorPosition) ) {
    printf("ReadDldExperiment return status = % d\n",status);return; }   

  if (pcb->TestBit) {
   if (Title!=(char *) NULL) printf("Title = % s\n",Title);
   if (ExperimentInfo!=(char *) NULL) 
      printf("ExperimentInfo = % s\n",ExperimentInfo);
   if (ICenter) printf("Center_1 = %g, Center_2 = %g\n",Center_1,Center_2);
   if (IPixSize) printf("PixSize_1 = %g, PixSize_2 = %g\n",PixSize_1,PixSize_2);
   if (IWaveLength) printf("WaveLength = %g\n",WaveLength);
   if (ISampleDistance) printf("SampleDistance = %g\n",SampleDistance);
   if (IDetectorPosition) printf("DetectorPosition = %g\n",DetectorPosition);
   } /* if (pcb->TestBit) ...*/

  if (G2S_debug) printf("  ReadDldInfo\n");
  if (status=ReadDldInfo(InFstream, &DetectorType, 
        &MachineInfo, &OpticsInfo, &StationInfo, &ProposalInfo)) {
    printf("ReadDldInfo return status = % d\n",status);return; }
  if (pcb->TestBit) {
   if (DetectorType!=(char *) NULL) printf("DetectorType = % s\n",DetectorType);
   if (MachineInfo!=(char *) NULL) printf("MachineInfo = % s\n",MachineInfo);
   if (OpticsInfo!=(char *) NULL) printf("OpticsInfo = % s\n",OpticsInfo);
   if (StationInfo!=(char *) NULL) printf("StationInfo = % s\n",StationInfo);
   if (ProposalInfo!=(char *) NULL) printf("ProposalInfo = % s\n",ProposalInfo);
   } /* if (pcb->TestBit) ...*/

  /* test whether output image exists already, else error */
  if (ImageNumberExists( OutStream, Out, pErrorValue ))
     {*pErrorValue=GasImageProtected; return; }

  /* write to esrf data file */
  write_data_raw ( OutStream,
                   HmDldVersion,
                   Out, 
                   InName,  In, 
                   DataLength, Dim_1, Dim_2, Off_1, Off_2, Start,
                   HmScaler,  
                   HmScalerName, HmScalerZero, HmScalerCalib,
		   HmScalerDepth,
                   HmScalerI0, HmScalerI1, HmScalerAnode, HmScalerTime,
                   HmScalerI0S, HmScalerI1S, HmScalerAnodeS, HmScalerTimeS,
                   HmStartTime, 
                   HmDeltaTime,
                   Dummy,
                   Title, ExperimentInfo, 
                   Center_1, Center_2, ICenter,
                   PixSize_1, PixSize_2, IPixSize,
                   WaveLength, IWaveLength,
                   SampleDistance, ISampleDistance,
		   DetectorPosition, IDetectorPosition,
                   DetectorType, MachineInfo, OpticsInfo,
                   StationInfo, ProposalInfo, 
                   pErrorValue );

  if (*pErrorValue!=RoutineSucceeded) return;

  /* free allocated memory */
  if (Title)          free( Title );
  if (ExperimentInfo) free( ExperimentInfo );
  if (DetectorType)   free( DetectorType );
  if (MachineInfo)    free( MachineInfo );
  if (OpticsInfo)     free( OpticsInfo );
  if (StationInfo)    free( StationInfo );
  if (ProposalInfo)   free( ProposalInfo );
  if (HmFraming)      free( HmFraming );

  if (G2S_debug) printf("Start = %p\n",Start);

  if (Start);          free( Start ); 

  if (G2S_debug) printf("Dld2Saxs END\n");

  return;

} /* Dld2Saxs */

/*---------------------------------------------------------------------------
user_io
Do simple keyboard io 
---------------------------------------------------------------------------*/

void user_io( int argc, char *argv[], int * parg_no, CmdBlk *pcb,
              char InName[], int * pInFirst, int * pInLast, int * pInInc,
	      char InHmName[], char OutName[], int * pOutFirst, 
              float * pDummy, int * pDataLength, int * pDim_1, int * pDim_2,
              int * pOff_1, int * pOff_2, int * pstatus)

{
  char  progfn[IO_size];
  char  InNamePre[IO_size];
  IO_line * pExtension;

  FILE * InFstream;
  int prompt;
  int opt_no;

  if (G2S_debug) printf("user_io BEGIN\n");

 /* Determine program name without directory name */
   (void) RightFR((char *) argv[0],DirSeparator,progfn,IO_size);

  /* switch to prompt mode if input file name is missing and no 
     instruction has been given */
  if (argc<(*parg_no+2)) prompt=TRUE; else prompt=FALSE;
  if (pcb->Prompt.I) prompt=(int) pcb->Prompt.V;
 
  /* show usage if no arguments are given */
  if (argc<=1) printf("Usage: %s %s\n",progfn,Usage);

  /*--- Write name of program ---*/
  printf("\n");
  printf("   %s %s\n",progfn,Version);
  printf("\n");

  /*--- Get default extension */
  pExtension = option_parameter_search  ( SDefExt, pcb, pstatus );
  if (*pstatus!=Success) return;

  /*--- Argument 1 : input file name */
  *parg_no += 1;
  (void) strcpy(InName,Default_ipfn);
  if ((argc>*parg_no) && (strcmp((char *) argv[*parg_no],Wild))) {
     (void) strcpy(InName,(char *) argv[*parg_no]);
     }
  if (prompt) {
     *pstatus = GetLine(stdin,InName,1,"Input file name",InName);
   /* Exit in all cases after an abort from the first argument */
   if (*pstatus!=Success) return;
    } 
  /* Expand InName to the full file name */ 
  if (!filename_full(InName,IO_size,InName)) {
     printf("% s: Input file name or path not allowed\n",progfn);
     *pstatus=FileNotFound; return;}    

  /* Determine directory and prefix ouf histogramming memory name (InHmName) */
  (void) LeftFL((char *) InName,DevSeparator,InHmName,IO_size);

  /* Determine prefix of input file name (InName) */
  (void) RightFR((char *) InName,DevSeparator,InNamePre,IO_size);
  (void) RightFR((char *) InNamePre,DirSeparator,InNamePre,IO_size);
  (void) LeftFL((char *) InNamePre,ExtSeparator,InNamePre,IO_size);

 /*--- Read header information of input file --- */

  if ((InFstream = fopen(InName, "rb")) == (FILE *) NULL) {
    *pstatus = FileOpeningError; return; }

  /* minimum and maximum */
  if (G2S_debug) printf("GetMaxDldNumber\n");
  *pInFirst=1;
  if (GetMaxDldNumber(InFstream,pInLast)) {
    *pstatus = CannotFindHeader; return; }

  /* DataLength, Dim_1, Dim_2 */
  if (G2S_debug) printf("ReadDldType\n");
  if (ReadDldType(InFstream,pDim_1,pDim_2,pDataLength,pOff_1,pOff_2)) {
    *pstatus = CannotFindHeader; return; }
  
  (void) fclose(InFstream); 

  /*--- Argument 2 : output dummy */
  *parg_no += 1;
  *pDummy = Default_dummy;
  if ((argc>*parg_no) && (strcmp((char *) argv[*parg_no],Wild))) {
     if (sscanf((char *) argv[*parg_no],"%g",pDummy)!=1)
       *pstatus = ScanError; return;
     }
  if (prompt) {
     *pstatus = GetReal(stdin,pDummy,1,"output dummy",*pDummy);
     if (*pstatus!=Success) return;
     }

  /*--- Argument 3 : number of first image */
  *parg_no += 1;
  if ((argc>*parg_no) && (strcmp((char *) argv[*parg_no],Wild))) {
     if (sscanf((char *) argv[*parg_no],"%d",pInFirst)!=1) {
         *pstatus = ScanError; return;
       }
     }
  if (prompt) {
     *pstatus = GetInteger(stdin,pInFirst,1,"First image",*pInFirst);
     if (*pstatus!=Success) return;
     }

  /*--- Argument 4 : number of last image */
  *parg_no += 1;
  if (argc==*parg_no)  /* exception: take First if it is last argument */
     *pInLast = *pInFirst;
  if ((argc>*parg_no) && (strcmp((char *) argv[*parg_no],Wild))) {
     if (sscanf((char *) argv[*parg_no],"%d",pInLast)!=1) {
        *pstatus = ScanError; return; }
     }
  if (prompt) {
     *pstatus = GetInteger(stdin,pInLast,1,"Last image",*pInLast);
     if (*pstatus!=Success) return; }

  /*--- Argument 5 : increment */
  *parg_no += 1;
  *pInInc = Default_ininc;
  if ((argc>*parg_no) && (strcmp((char *) argv[*parg_no],Wild))) {
     if (sscanf((char *) argv[*parg_no],"%d",pInInc)!=1) {
        *pstatus = ScanError; return; }
     }
  if (prompt) {
     *pstatus = GetInteger(stdin,pInInc,1,"Increment",*pInInc);
     if (*pstatus!=Success) return; }

  /*--- Argument 6 : output file name */
  *parg_no += 1;
  (void) sprintf(OutName,"%s%s",InNamePre,pExtension->V);
  if ((argc>*parg_no) && (strcmp((char *) argv[*parg_no],Wild))) {
     (void) strcpy(OutName, (char *) argv[*parg_no]);
     }
  if (prompt) {
     *pstatus = GetLine(stdin,OutName,1,"Output file name",OutName);
     if (*pstatus!=Success) return;
     }

  /* Expand OutName to the full file name */
  if (!filename_full(OutName,IO_size,OutName)) {
     printf("% s: Output file name or path not allowed\n",progfn);
     *pstatus = FileNameError; return; }

  /*--- Argument 7 : number of first output image */
  *parg_no += 1;
  *pOutFirst = *pInFirst;
  if ((argc>*parg_no) && (strcmp((char *) argv[*parg_no],Wild))) {
     if (sscanf((char *) argv[*parg_no],"%d",pOutFirst)!=1) {
        *pstatus = ScanError; return; }
       }
  if (prompt) {
     *pstatus = GetInteger(stdin,pOutFirst,1,"First output image",*pOutFirst);
     if (*pstatus!=Success) return;
     }

  /*--- Argument 8 : data length */
  *parg_no += 1;
  if ((argc>*parg_no) && (strcmp((char *) argv[*parg_no],Wild))) {
     if (sscanf((char *) argv[*parg_no],"%d",pDataLength)!=1) {
       *pstatus = ScanError; return; }
     }
  if (prompt) {
     *pstatus = GetInteger(stdin,pDataLength,1,"Data length in bytes",*pDataLength);
     if (*pstatus!=Success) return;
     }

  /*--- Argument 9 : number of columns */
  *parg_no += 1;
  if ((argc>*parg_no) && (strcmp((char *) argv[*parg_no],Wild))) {
     if (sscanf((char *) argv[*parg_no],"%d",pDim_1)!=1) {
         *pstatus = ScanError; return; }
     }
  if (prompt) {
     *pstatus = GetInteger(stdin,pDim_1,1,"Number of columns",*pDim_1);
     if (*pstatus!=Success) return;
     }

  /*--- Argument 10 : number of rows */
  *parg_no += 1;
  if ((argc>*parg_no) && (strcmp((char *) argv[*parg_no],Wild))) {
     if (sscanf((char *) argv[*parg_no],"%d",pDim_2)!=1) {
         *pstatus = ScanError; return; }
     }
  if (prompt) {
     *pstatus = GetInteger(stdin,pDim_2,1,"Number of rows",*pDim_2);
     if (*pstatus!=Success) return;
     }

  /*--- Argument 11 : name of histogramming memory file */
  *parg_no += 1;
  (void) sprintf(InHmName,"%s%s",InHmName,Hm_extension);
  if ((argc>*parg_no) && (strcmp((char *) argv[*parg_no],Wild))) {
     (void) strcpy(InHmName, (char *) argv[*parg_no]);
     }
  if (prompt) {
     *pstatus = GetLine(stdin,InHmName,1,"Histogramming memory file",InHmName);
     if (*pstatus!=Success) return;
     }

  /* Expand InHmName to the full file name */
  if (!filename_full(InHmName,IO_size,InHmName)) {
     printf("% s: Histogramming memory file or path not allowed\n", progfn);
     *pstatus = FileNameError; return; }

  /*--- End of argument list */
  *parg_no += 1;
  if (argc>*parg_no) { /* too many arguments */
     printf("%s : Too many arguments\n",progfn);
     printf("Usage: %s %s\n",progfn,Usage);
     if (!prompt) { *pstatus = StoppedAfterHelp; return; }
     }

  printf("\n");
  printf("i/p file           : %s\n",InName);
  printf("output dummy       : %g\n",*pDummy);
  printf("first image        : %d\n",*pInFirst);
  printf("last image         : %d\n",*pInLast);
  printf("increment          : %d\n",*pInInc);
  printf("o/p file           : %s\n",OutName);
  printf("first output image : %d\n",*pOutFirst);
  printf("data length        : %d bytes\n",*pDataLength); 
  printf("dimension 1        : %d\n",*pDim_1);
  printf("dimension 2        : %d\n",*pDim_2);   
  printf("i/p hm file        : %s\n",InHmName);
  printf("\n");

  if (G2S_debug) printf("user_io END\n");

  return;
} /* user_io */

/*---------------------------------------------------------------------------
main

RETURN VALUE:            0: SUCCESS
              other values: ERROR
---------------------------------------------------------------------------*/

#if MAKE_FUNCTION
# define MAIN main_PPP
#else
# define MAIN main
#endif

int MAIN (int argc, char *argv[])
{
  int status = Success, ErrorValue;
  int arg_no = 0;
  CmdBlk cb;  /* command block  */

  FILE  * InFstream, *InHmstream;
  char  InName[IO_size];
  char  InHmName[IO_size];
  int   InFirst, InLast, InInc;
  int   MinFrame, MaxFrame;
  int   OutStream;
  char  OutName[IO_size];
  int   OutFirst; 
  float Dummy;
  int   DataLength, Dim_1, Dim_2, Off_1, Off_2;

  int   loop, loopmax;
  int	In, Out;

    /* Init options, control block and image blocks */
     InitOptions( Usage, Version, NULL, TRUE, &cb, NULL, 0 );

     option_define ( IO_tpline, SDefExt, Default_extension, &cb ); 

    /* Read options from argument list */
     ReadOptions( argv, &arg_no, &cb, NULL, &status);
     if (status!=Success) return( ReportSaxsStatus( status, 0 ) );

     G2S_debug = cb.TestBit;

     user_io( argc, argv, &arg_no, &cb,
              InName, &InFirst, &InLast, &InInc,
              InHmName, OutName, &OutFirst, &Dummy,
              &DataLength, &Dim_1, &Dim_2, &Off_1, &Off_2,
              &status);  
     if (status!=Success) return( ReportSaxsStatus( status, 0 ) );

     /* loop */
     /* open header file */
     if (G2S_debug) printf("open header file\n");
     if ((InFstream = fopen(InName, "rb")) == (FILE *) NULL)
       return( ReportSaxsStatus( FileOpeningError, 0 ) );

     /* open histogramming memory file */
     if (G2S_debug) printf("open histogramming memory file\n");
     if ((InHmstream = fopen(InHmName, "rb")) == (FILE *) NULL)
       return( ReportSaxsStatus( FileOpeningError, 0 ) );

     /* minimum and maximum */
     MinFrame=1;
     if (G2S_debug) printf("GetMaxDldNumber\n");
     if (GetMaxDldNumber(InFstream,&MaxFrame))
       return( ReportSaxsStatus( CannotFindHeader, 0 ) );

     /* open esrf data file */
     if (G2S_debug) printf("open esrf data file\n");
     OutStream = edf_open_data_file ( OutName, "any", &ErrorValue, &status );
     if (ErrorValue!=RoutineSucceeded)
       return( ReportSaxsStatus( FileOpeningError, 0 ) );

    if (InInc!=0) {
       loopmax =  MAX2(1,1 + (InLast - InFirst)/InInc);
       } else { loopmax = 1; }

    In  = InFirst;
    Out = OutFirst;
    for (loop=1, In=InFirst, Out=OutFirst; 
         loop<=loopmax; loop++, In+=InInc, Out+=InInc ) {

        if ((MinFrame<=In) && (In<=MaxFrame)) {
	  printf(" Converting frame % d to image % d\n",In,Out);

	  Dld2Saxs(&cb, InFstream, 
		   In, DataLength, Dim_1, Dim_2, Off_1, Off_2,
                   InHmstream, OutStream, Out,  
                   InName, Dummy, &ErrorValue);
          if (ErrorValue!=RoutineSucceeded) {
             return(ReportSaxsStatus(RoutineFailed,0));}
          } else {
          printf(" Input frame % d does not exist.\n",In);
          } /* if ((MinFrame ... */

      } /* for (loop ... */

   /* close esrf data file */
   status = Success;
   edf_close_data_file        ( OutStream, &ErrorValue, &status );
   if (ErrorValue!=RoutineSucceeded)
      return(ReportSaxsStatus(FileClosingError,0));

   /* close histogramming memory file */
   (void) fclose(InFstream);

   /* close header file */
   (void) fclose(InFstream);

   printf("\n");
   printf("Input header file : % s\n",InName);
   printf("Input memory file : % s\n",InHmName);
   printf("Output file       : % s\n",OutName);

   printf("\nEnd of % s\n",argv[0]);

   return( 0 );

} /* MAIN */
