/***************************************************************************/
/* Written 1994++ by Peter Boesecke                                        */
/* Copyright (C) 2011 European Synchrotron Radiation Facility              */
/*                       Grenoble, France                                  */
/*                                                                         */
/*    Principal authors: Peter Boesecke  (boesecke@esrf.eu)                */
/*                                                                         */
/*    This program is free software: you can redistribute it and/or modify */
/*    it under the terms of the GNU General Public License as published by */
/*    the Free Software Foundation, either version 3 of the License, or    */
/*    (at your option) any later version.                                  */
/*                                                                         */
/*    This program is distributed in the hope that it will be useful,      */
/*    but WITHOUT ANY WARRANTY; without even the implied warranty of       */
/*    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the        */
/*    GNU General Public License for more details.                         */
/*                                                                         */
/*    You should have received a copy of the GNU General Public License    */
/*    along with this program.  If not, see <http://www.gnu.org/licenses/>.*/
/***************************************************************************/
# define Version  "epoch2isotime 1.2 2017-10-08, Peter Boesecke"
/*---------------------------------------------------------------------------
NAME
  epoch2isotime - convert epoch (seconds since 1970-01-01) to isotime 

SYNOPSIS
  epoch2isotime {<epoch>} 

DESCRIPTION
  Converts epoch string from argument list or from stdin to isotime and 
  returns the result to stdout.
  stdin is only read if no string is given in the argument list.

  epoch2isotime [options] 1149248916.000000

  epoch2isotime [options] < file

  The file must contain 1 epoch string per line. The conversion stops
  at eof, "quit" or "exit"

HISTORY
  V1.0 2006-06-02 Peter Boesecke
  V1.1 2012-10-28 PB shift added, mode added
  V1.2 2017-10-08 PB shift epoch function renamed to epoch_add
---------------------------------------------------------------------------*/

# include "isotime.h"

/* removes trailing line feed \n */
char * trimlf ( char * s )
{ size_t slen;
  if ( s ) { 
    slen = strlen( s );
    if ( slen > 0 )      
      if (s[slen-1]=='\n') s[slen-1]='\0';
  }
  return ( s );
} // trimlf

/* returns 1 if s is an empty string (only white space) otherwise 0. */
int isempty( const char *s )
{ const char *ps;
  int empty=1;
  
  if (s) {
    ps = s;
    while (*ps) { empty = empty && isspace(*ps); ps++; }
  }
  return( empty );

} //isempty

/*+++------------------------------------------------------------------------
main
---------------------------------------------------------------------------*/

#if MAKE_FUNCTION
# define MAIN main_epoch2isotime
#else
# define MAIN main
#endif

int MAIN (int argc, char *argv[])
{ char *nul = (char *) NULL;
  char **pargv;
  IsotimeEpoch epoch;
  int status=0;
  const char *ps, *offset="", *shift="", *mode="";
  long debug=0;
  float tmp=0.0;
  int argconv=0; // is set to 1 if an isotime argument is found
  char * string;

  #define BUFLEN 1024
  char buffer[BUFLEN];
  char isobuf[ISOTIMELEN];

  // count options or help or help
  pargv = &argv[1];

  for (;*pargv!=nul;*pargv++) {
    if (( strncmp( *pargv,"-h",2 )==0 )||( strncmp( *pargv,"--help",6 )==0 )) {
      // help
      fprintf( stderr, "%s\n", Version );
      fprintf( stderr, "%s\n", isotime_version() );
      fprintf( stderr, 
        "epoch2isotime [-h] [debug=0|1] [offset=+|-HhMm] [shift=<seconds>] [mode=nospace|space] {<epoch>}\n" );
      return( 1 );
    } else if ( strncmp( *pargv,"debug=", 6 ) == 0 ) {
      // set debug level
      ps=*pargv+6;
      sscanf(ps,"%ld",&debug);
      isotime_debug(debug);
    } else if ( strncmp( *pargv,"offset=", 7 ) == 0 ) {
      // set offset 
      offset=*pargv+7;
    } else if ( strncmp( *pargv,"shift=", 6 ) == 0 ) {
      // set shift
      shift=*pargv+6;
    } else if ( strncmp( *pargv,"mode=", 5 ) == 0 ) {
      // set shift
      mode=*pargv+5;
      // set isotime mode
      isotime_mode( isotime_str2mode( mode ) );
    } else {
      // convert epoch to isotime
      argconv=1;

      epoch = setoffset2epoch(string2epoch(*pargv),offset);
      epoch = epoch_add( epoch, string2epoch(shift) );
      if (epoch.status) {
        status=epoch.status;
        fprintf( stderr,"ERROR: Time read failed.\n");
      } else {
        fprintf( stdout,"%s\n", epoch2isotime( isobuf, ISOTIMELEN, epoch ) );
      }
    }
  }
  if (!argconv) {
    // read from standard input
    while ( !feof( stdin ) ) {
      string = fgets ( buffer, BUFLEN, stdin );

      if ( string ) {
        if ( (!strncmp(string,"exit",4)) || (!strncmp(string,"quit",4)) )
          return ( status );
        // convert epoch to isotime
        if (!isempty (string)) {
          epoch = setoffset2epoch(string2epoch(string),offset);
          epoch = epoch_add( epoch, string2epoch(shift) );
          if (epoch.status) {
            status=epoch.status;
            fprintf( stderr,"ERROR: Time read failed.\n");
          } else {
            fprintf( stdout,"%s\n", epoch2isotime(isobuf, ISOTIMELEN, epoch) );
          }
        } else fprintf( stdout,"\n");
      }
    }
  }
  return( status );
} /* MAIN */

