/***************************************************************************/
/* Written 1994++ by Peter Boesecke                                        */
/* Copyright (C) 2011 European Synchrotron Radiation Facility              */
/*                       Grenoble, France                                  */
/*                                                                         */
/*    Principal authors: Peter Boesecke  (boesecke@esrf.eu)                */
/*                                                                         */
/*    This program is free software: you can redistribute it and/or modify */
/*    it under the terms of the GNU General Public License as published by */
/*    the Free Software Foundation, either version 3 of the License, or    */
/*    (at your option) any later version.                                  */
/*                                                                         */
/*    This program is distributed in the hope that it will be useful,      */
/*    but WITHOUT ANY WARRANTY; without even the implied warranty of       */
/*    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the        */
/*    GNU General Public License for more details.                         */
/*                                                                         */
/*    You should have received a copy of the GNU General Public License    */
/*    along with this program.  If not, see <http://www.gnu.org/licenses/>.*/
/***************************************************************************/
/*+++
1 SaxsPrompt.inc

2 DESCRIPTION
  Private include file of "SaxsOption.c".
  See PUBLIC functions for detail.

2 CALL
  void argv_start( CmdBlk * pcb, int narg )
  void argv_line( CmdBlk * pcb,
                  char * promptline, IO_line * value, char * defval,
                  int * pstatus);
  void argv_float( CmdBlk * pcb,
                   char * promptline, IO_float * value, float defval,
                   int * pstatus);
  void argv_flexp( CmdBlk * pcb,
                  char * promptline, IO_flexp * value, char * defval,
                  int * pstatus);
  void argv_long( CmdBlk * pcb,
                  char * promptline, IO_long * value, long int defval,
                  int * pstatus);
  void argv_lvexp( CmdBlk * pcb,
                   char   * promptline, IO_lvexp * value, char * defval,
                   int    * pstatus);
  void argv_flag( CmdBlk * pcb,
                  char * promptline, IO_long * value, long int defval,
                  int * pstatus);
  void argv_coord( CmdBlk * pcb,
                   long int RSys, float ROff, float RPs,
                   long int USys, float UOff, float UPs,
                   char * promptline, IO_float * value, float defval,
                   int * pstatus);
  void argv_dist ( CmdBlk * pcb,
                   long int RSys, float ROff, float RPs,
                   long int USys, float UOff, float UPs,
                   char * promptline, IO_float * value, float defval,
                   int * pstatus);
  void argv_end( CmdBlk * pcb );

2 HISTORY
  16-Sep-1995 PB
  14-Nov-1995 PB argv_coord added, SkipToken in argument list
  28-Nov-1995 PB argv_dist added
  26-Dec-1995 PB string constants
  11-Oct-1996 PB PRIVATE, PUBLIC
  12-Oct-1996 PB "ReportSaxsStatus" replaced by "return"
  24-Jul-1999 PB argv_coord history corrected (%d -> %g) 
  08-Nov-1999 PB edf_history_skip and edf_history_take added
  09-Jul-2001 PB repeat mode removed
  13-Sep-2001 PB SaxsHistoryKey
  04-Feb-2003 PB argv_flexp
  07-Feb-2003 PB argv_lvexp
  01-Oct-2004 PB some cosmetics 
  04-Oct-2004 PB argv_flexp, argv_lvexp: empty defaults
  15-Jul-2005 PB SetSaxsError without seb, status set to Success
  19-Apr-2007 PB -Wall compiler warnings resolved
---*/

/****************************************************************************
*  Defines                                                                  *
****************************************************************************/

#ifndef PRIVATE
#  define PRIVATE static
#endif

#ifndef PUBLIC
#  define PUBLIC
#endif

/*--------------------------------------------------------------------------
 String constants
--------------------------------------------------------------------------*/
PRIVATE const char * InputSkipped = "... skipped";
PRIVATE const char * InputValue   = "Value =";
PRIVATE const char * InputRefSys  = "Input reference system is";

/*--------------------------------------------------------------------------
   show_argv : Show the current argument number
  --------------------------------------------------------------------------*/
void show_arg_no (const char * RoutineName, CmdBlk * pcb)
{ printf(" %s: argc = %d, arg_no = %d\n",RoutineName,pcb->argc,*pcb->parg_no);
} /* show_arg_no */

/*--------------------------------------------------------------------------
   argv_start : Prepare to read arguments.
                Switch to prompt mode if less than narg arguments are given.
  
   return value		   :	void
    CmdBlk * pcb           :    command block
     IO_long * pcb->Prompt :
       pcb->Prompt.I (i)   :    unchanged
       pcb->Prompt.V (o)   :    changed to TRUE,
                                  if less than narg arguments are given
     int * pcb->parg_no(i) :    argument number of previous argument
     int   pcb->argc (i)   :    number of arguments
   int narg          (i)   :    number of required arguments
  --------------------------------------------------------------------------*/
PUBLIC void argv_start( CmdBlk * pcb, int narg )
{
  if (!pcb->Prompt.I) {
    if (pcb->argc<(*pcb->parg_no+narg+1)) pcb->Prompt.V=TRUE;
      else pcb->Prompt.V=FALSE;
    }

} /* argv_start */

/*--------------------------------------------------------------------------
   argv_line : Read a char line from the argument list and from the input file
               A wild card in the argument list chooses the default.
               A skip token in the argument or in the input line returns
               the default according to the .I flag. The .I flag is not
               changed.
  
   return value		   :	void
    CmdBlk * pcb           :    command block
     FILE * pcb->infile(i) :    pointer to input file
     char * pcb->argv[](i) :    pointer to argument list
     int * pcb->parg_no(i) :    argument number of previous argument
                       (o) :    incremented to current argument number
     long pcb->Prompt.V(i) :    if set, prompt mode
     char * promptline (i) :    user prompt in prompt mode
     IO_line * value (i/o) :    read line
               value.I (i) :    if set, value.V is default,
                                if not set, defval is default
               value.I (o) :    set to TRUE, if input is not SkipToken
               value.V (i) :    used as default, if value.I is set
               value.V (o) :    read value
     char * defval     (i) :    default value
     int * pstatus     (o) :    exit status
  --------------------------------------------------------------------------*/
PUBLIC void argv_line( CmdBlk * pcb,
                char * promptline, IO_line * value, char * defval,
                int * pstatus)
{ int iset = TRUE;
  const char * RoutineName = "argv_line";

  SetSaxsError( SOName, RoutineName, NULL );

  *pstatus = Success;

  *pcb->parg_no += 1;
  if (pcb->TestBit) show_arg_no ( RoutineName, pcb );
  if (!value->I) CopyLine(defval,value->V,IO_size,0);
  if (pcb->argc>*pcb->parg_no) {
    if (strcmp(pcb->argv[*pcb->parg_no],SkipToken)) {
      if (strcmp(pcb->argv[*pcb->parg_no],WildCard))
        CopyLine(pcb->argv[*pcb->parg_no],value->V,IO_size,0);
      } else iset = FALSE;
    } else edf_history_skip( SaxsHistoryKey );

  if (pcb->Prompt.V) {
     iset = TRUE;
     *pstatus = GetLine(pcb->infile,value->V,TRUE,promptline,value->V);
     if (*pstatus==SkipInput) { iset = FALSE; *pstatus=Success; }
     if (*pstatus!=Success) return;
     edf_history_take( SaxsHistoryKey );
     }
  if (iset) value->I = TRUE; else printf("%s\n",InputSkipped);
  if ((pcb->TestBit) && (value->I)) printf("%s %s\n",InputValue,value->V);

  /* add value to argument history list */
  if (value->I) edf_history_argv( SaxsHistoryKey, value->V );
    else edf_history_argv( SaxsHistoryKey, SkipToken );

} /* argv_line */

/*--------------------------------------------------------------------------
   argv_float : Read a float expression from the argument list and
                from the input file. A wild card in the argument
                list chooses the default.
                A skip token in the argument or in the input line returns
                the default according to the .I flag. The .I flag is not
                changed.
   return value		   :	void
    CmdBlk * pcb           :    command block
     FILE * pcb->infile(i) :    pointer to input file
     char * pcb->argv[](i) :    pointer to argument list
     int * pcb->parg_no(i) :    argument number of previous argument
                       (o) :    incremented to current argument number
     long pcb->Prompt.V(i) :    if set, prompt mode
     char * promptline (i) :    user prompt in prompt mode
     IO_float * value(i/o) :    read float
               value.I (i) :    if set, value.V is default,
                                if not set, defval is default
               value.I (o) :    set to TRUE, if input is not SkipToken
               value.V (i) :    used as default, if value.I is set
               value.V (o) :    read value
     float * defval    (i) :    default value
     int * pstatus     (o) :    exit status
  --------------------------------------------------------------------------*/
PUBLIC void argv_float( CmdBlk * pcb,
                 char * promptline, IO_float * value, float defval,
                 int * pstatus)
{ int iset = TRUE;
  const char * RoutineName = "argv_float";
  char buffer[IO_size];

  SetSaxsError( SOName, RoutineName, NULL );

  *pstatus = Success;

  *pcb->parg_no += 1;
  if (pcb->TestBit) show_arg_no ( RoutineName, pcb );
  if (!value->I) value->V = defval;
  if (pcb->argc>*pcb->parg_no) {
    if (strcmp(pcb->argv[*pcb->parg_no],SkipToken)) {
      if (strcmp(pcb->argv[*pcb->parg_no],WildCard)) {
        value->V = floatexpr(pcb->argv[*pcb->parg_no],pstatus);
        if (*pstatus!=Success) {
          SetSaxsErrorMessage(pcb->argv[*pcb->parg_no]);
          return;
          }
        }
      } else iset = FALSE;
    } else edf_history_skip( SaxsHistoryKey );

  if (pcb->Prompt.V) {
     iset = TRUE;
     *pstatus = GetReal(pcb->infile,&value->V,TRUE,promptline,value->V);
     if (*pstatus==SkipInput) { iset = FALSE; *pstatus=Success; }
     if (*pstatus!=Success) return;
     edf_history_take( SaxsHistoryKey );
     }
  if (iset) value->I = TRUE; else printf("%s\n",InputSkipped);
  if ((pcb->TestBit) && (value->I)) printf("%s %g\n",InputValue,value->V);

  /* add value to argument history list */
  if (value->I) {
    sprintf(buffer,"%g",value->V);
    edf_history_argv( SaxsHistoryKey, buffer );
    } else edf_history_argv( SaxsHistoryKey, SkipToken );

} /* argv_float */

/*--------------------------------------------------------------------------
   argv_flexp : Read a float expression line from the argument list and from 
               the input file. A wild card in the argument list chooses the 
               default. A skip token in the argument or in the input line 
               returns the default according to the .I flag. The .I flag is 
               not changed.
               If the default string is empty, Getline is called without a
               default. If then no value is entered, the original value is
               returned with the I.flag set to FALSE.
  
   return value            :    void
    CmdBlk * pcb           :    command block
     FILE * pcb->infile(i) :    pointer to input file
     char * pcb->argv[](i) :    pointer to argument list
     int * pcb->parg_no(i) :    argument number of previous argument
                       (o) :    incremented to current argument number
     long pcb->Prompt.V(i) :    if set, prompt mode
     char * promptline (i) :    user prompt in prompt mode
     IO_flexp *value (i/o) :    read line
               value.I (i) :    if set, value.V is default,
                                if not set, defval is default
               value.I (o) :    set to TRUE, if input is not SkipToken
               value.V (i) :    used as default, if value.I is set
               value.V (o) :    read value
     char * defval     (i) :    default value
     int * pstatus     (o) :    exit status
  --------------------------------------------------------------------------*/
PUBLIC void argv_flexp( CmdBlk * pcb,
                        char * promptline, IO_flexp * value, char * defval,
                        int * pstatus)
{ int iset = TRUE;
  const char * RoutineName = "argv_flexp";
  char input_value[IO_size];
 
  SetSaxsError( SOName, RoutineName, NULL );

  *pstatus = Success;
 
  *pcb->parg_no += 1;
  if (pcb->TestBit) show_arg_no ( RoutineName, pcb );
  // backup input value
  CopyLine(value->V,input_value,IO_size,0);
  if (!value->I) CopyLine(defval,value->V,IO_size,0);
  if (pcb->argc>*pcb->parg_no) {
    if (strcmp(pcb->argv[*pcb->parg_no],SkipToken)) {
      if (strcmp(pcb->argv[*pcb->parg_no],WildCard))
        CopyLine(pcb->argv[*pcb->parg_no],value->V,IO_size,0);
      } else iset = FALSE;
    } else edf_history_skip( SaxsHistoryKey );
 
  if (pcb->Prompt.V) {
     iset = TRUE;
//++++++     *pstatus = GetLine(pcb->infile,value->V,TRUE,promptline,value->V);
     *pstatus = GetLine(pcb->infile,value->V,strlen(value->V),
                        promptline,value->V);
     if (*pstatus==SkipInput) { iset = FALSE; *pstatus=Success; }
//++++++ next line is new
     if (*pstatus==NoDefault) { iset = FALSE; *pstatus=Success; }
     if (*pstatus!=Success) return;
     edf_history_take( SaxsHistoryKey );
     }

  if (!strlen(value->V)) {
    CopyLine(input_value,value->V,IO_size,0);
    iset=FALSE;
  }
  if (iset) value->I = TRUE; else printf("%s\n",InputSkipped);
  if ((pcb->TestBit) && (value->I)) printf("%s %s\n",InputValue,value->V);
 
  /* add value to argument history list */
  if (value->I) edf_history_argv( SaxsHistoryKey, value->V );
    else edf_history_argv( SaxsHistoryKey, SkipToken );
 
} /* argv_flexp */

/*--------------------------------------------------------------------------
   argv_long : Read a long integer expression from the argument list and
               from the input file. A wild card in the argument list chooses
               the default.
               A skip token in the argument or in the input line returns
               the default according to the .I flag. The .I flag is not
               changed.
   return value		   :	void
    CmdBlk * pcb           :    command block
     FILE * pcb->infile(i) :    pointer to input file
     char * pcb->argv[](i) :    pointer to argument list
     int * pcb->parg_no(i) :    argument number of previous argument
                       (o) :    incremented to current argument number
     long pcb->prompt.V(i) :    if set, prompt mode
     char * promptline (i) :    user prompt in prompt mode
     IO_long * value (i/o) :    read long integer
               value.I (i) :    if set, value.V is default,
                                if not set, defval is default
               value.I (o) :    set to TRUE, if input is not SkipToken
               value.V (i) :    used as default, if value.I is set
               value.V (o) :    read value
     long * defval     (i) :    default value
     int * pstatus     (o) :    exit status
  --------------------------------------------------------------------------*/
PUBLIC void argv_long( CmdBlk * pcb,
                char * promptline, IO_long * value, long int defval,
                int * pstatus)
{ int iset = TRUE;
  const char * RoutineName = "argv_long";
  char buffer[IO_size];

  SetSaxsError( SOName, RoutineName, NULL );

  *pstatus = Success;

  *pcb->parg_no += 1;
  if (pcb->TestBit) show_arg_no ( RoutineName, pcb );
  if (!value->I) value->V = defval;
  if (pcb->argc>*pcb->parg_no) {
    if (strcmp(pcb->argv[*pcb->parg_no],SkipToken)) {
      if (strcmp(pcb->argv[*pcb->parg_no],WildCard)) {
        value->V = longexpr(pcb->argv[*pcb->parg_no],pstatus);
        if (*pstatus!=Success) {
          SetSaxsErrorMessage(pcb->argv[*pcb->parg_no]);
          return;
          }
        }
      } else iset = FALSE;
    } else edf_history_skip( SaxsHistoryKey );

  if (pcb->Prompt.V) {
     iset = TRUE;
     *pstatus = GetLong(pcb->infile,&value->V,TRUE,promptline,value->V);
     if (*pstatus==SkipInput) { iset = FALSE; *pstatus=Success; }
     if (*pstatus!=Success) return;
     edf_history_take( SaxsHistoryKey );
     }
  if (iset) value->I = TRUE; else printf("%s\n",InputSkipped);
  if ((pcb->TestBit) && (value->I)) printf("%s %ld\n",InputValue,value->V);

  /* add value to argument history list */
  if (value->I) {
    sprintf(buffer,"%ld",value->V);
    edf_history_argv( SaxsHistoryKey, buffer );
    } else edf_history_argv( SaxsHistoryKey, SkipToken );

} /* argv_long */

/*--------------------------------------------------------------------------
   argv_lvexp : Read a long expression line from the argument list and from
               the input file. A wild card in the argument list chooses the
               default. A skip token in the argument or in the input line
               returns the default according to the .I flag. The .I flag is
               not changed.
               If the default string is empty, Getline is called without a 
               default. If then no value is entered, the original value is
               returned with the I.flag set to FALSE.
  
   return value            :    void
    CmdBlk * pcb           :    command block
     FILE * pcb->infile(i) :    pointer to input file
     char * pcb->argv[](i) :    pointer to argument list
     int * pcb->parg_no(i) :    argument number of previous argument
                       (o) :    incremented to current argument number
     long pcb->Prompt.V(i) :    if set, prompt mode
     char * promptline (i) :    user prompt in prompt mode
     IO_lvexp *value (i/o) :    read line
               value.I (i) :    if set, value.V is default,
                                if not set, defval is default
               value.I (o) :    set to TRUE, if input is not SkipToken
               value.V (i) :    used as default, if value.I is set
               value.V (o) :    read value
     char * defval     (i) :    default value
     int * pstatus     (o) :    exit status
  --------------------------------------------------------------------------*/
PUBLIC void argv_lvexp( CmdBlk * pcb,
                char   * promptline, IO_lvexp * value, char * defval,
                int    * pstatus)
{ int iset = TRUE;
  const char * RoutineName = "argv_lvexp";
  char input_value[IO_size];
 
  SetSaxsError(SOName, RoutineName, NULL );

  *pstatus = Success;
 
  *pcb->parg_no += 1;
  if (pcb->TestBit) show_arg_no ( RoutineName, pcb );
  // backup input value
  CopyLine(value->V,input_value,IO_size,0);
  if (!value->I) CopyLine(defval,value->V,IO_size,0);
  if (pcb->argc>*pcb->parg_no) {
    if (strcmp(pcb->argv[*pcb->parg_no],SkipToken)) {
      if (strcmp(pcb->argv[*pcb->parg_no],WildCard))
        CopyLine(pcb->argv[*pcb->parg_no],value->V,IO_size,0);
      } else iset = FALSE;
    } else edf_history_skip( SaxsHistoryKey );
 
  if (pcb->Prompt.V) {
     iset = TRUE;
//++++++     *pstatus = GetLine(pcb->infile,value->V,TRUE,promptline,value->V);
     *pstatus = GetLine(pcb->infile,value->V,strlen(value->V),
                        promptline,value->V);
     if (*pstatus==SkipInput) { iset = FALSE; *pstatus=Success; }
//++++++ next line is new
     if (*pstatus==NoDefault) { iset = FALSE; *pstatus=Success; }
     if (*pstatus!=Success) return;
     edf_history_take( SaxsHistoryKey );
     }
  if (!strlen(value->V)) {
    CopyLine(input_value,value->V,IO_size,0);
    iset=FALSE;
  }
  if (iset) value->I = TRUE; else printf("%s\n",InputSkipped);
  if ((pcb->TestBit) && (value->I)) printf("%s %s\n",InputValue,value->V);
 
  /* add value to argument history list */
  if (value->I) edf_history_argv( SaxsHistoryKey, value->V );
    else edf_history_argv( SaxsHistoryKey, SkipToken );
 
} /* argv_lvexp */

/*--------------------------------------------------------------------------
   argv_flag : Read a flag expression from the argument list and
               from the input file. A wild card in the argument list chooses
               the default.
               A skip token in the argument or in the input line returns
               the default according to the .I flag. The .I flag is not
               changed.
   return value		   :	void
    CmdBlk * pcb           :    command block
     FILE * pcb->infile(i) :    pointer to input file
     char * pcb->argv[](i) :    pointer to argument list
     int * pcb->parg_no(i) :    argument number of previous argument
                       (o) :    incremented to current argument number
     long pcb->Prompt.V(i) :    if set, prompt mode
     char * promptline (i) :    user prompt in prompt mode
     IO_long * value (i/o) :    read flag
               value.I (i) :    if set, value.V is default,
                                if not set, defval is default
               value.I (o) :    set to TRUE, if input is not SkipToken
               value.V (i) :    used as default, if value.I is set
               value.V (o) :    long integer (FALSE or TRUE)
     long * defval     (i) :    default value
     int * pstatus     (o) :    exit status
  --------------------------------------------------------------------------*/
PUBLIC void argv_flag( CmdBlk * pcb,
                char * promptline, IO_long * value, long int defval,
                int * pstatus)
{ int iset = TRUE;
  char buffer[IO_size];
  const char * RoutineName = "argv_flag";

  SetSaxsError( SOName, RoutineName, NULL );

  *pstatus = Success;

  *pcb->parg_no += 1;
  if (pcb->TestBit) show_arg_no ( RoutineName, pcb );
  if (!value->I) value->V = defval;
  if (pcb->argc>*pcb->parg_no) {
    if (strcmp(pcb->argv[*pcb->parg_no],SkipToken)) {
      if (strcmp(pcb->argv[*pcb->parg_no],WildCard)) {
        value->V = longexpr(pcb->argv[*pcb->parg_no],pstatus);
        if (*pstatus!=Success) {
          SetSaxsErrorMessage(pcb->argv[*pcb->parg_no]);
          return;
          }
        }
      } else iset = FALSE;
    } else edf_history_skip( SaxsHistoryKey );

  if (pcb->Prompt.V) {
    iset = TRUE;
    /* allow also simple "y", "n", "t", "f" */
    if (value->V)
      *pstatus = GetLine(pcb->infile,buffer,TRUE,promptline,"TRUE");
      else *pstatus = GetLine(pcb->infile,buffer,TRUE,promptline,"FALSE");
    if (*pstatus==SkipInput) { iset = FALSE; *pstatus=Success; }
    if (*pstatus!=Success) return;
    EditLine(LowerCase | UnComment, buffer);
    if (strlen(buffer)==1)
      switch (buffer[0]) {
        case 'y' : value->V = TRUE;  break;
        case 'n' : value->V = FALSE; break;
        case 't' : value->V = TRUE;  break;
        case 'f' : value->V = FALSE; break;
        case '1' : value->V = TRUE;  break;
        case '0' : value->V = FALSE; break;
        default  : *pstatus=NoFlag;  break;
        }
      else value->V = longexpr(buffer,pstatus);;
    if (*pstatus!=Success) return;
    edf_history_take( SaxsHistoryKey );
    }
  if (value->V) value->V = TRUE;
  if (iset) value->I = TRUE; else printf("%s\n",InputSkipped);
  if ((pcb->TestBit) && (value->I)) printf("%s %ld\n",InputValue,value->V);

  /* add value to argument history list */
  if (value->I) {
    sprintf(buffer,"%ld",value->V);
    edf_history_argv( SaxsHistoryKey, buffer );
    } else edf_history_argv( SaxsHistoryKey, SkipToken );

} /* argv_flag */

/*--------------------------------------------------------------------------
   argv_coord : Read a pixel coordinate in the user system from the argument
                list and from the input file. Convert it into the reference
                system. A wild card in the argument list chooses the default.
                A skip token in the argument or in the input line returns
                the default according to the .I flag. The .I flag is not
                changed.
   return value		   :	void
    CmdBlk * pcb           :    command block
     FILE * pcb->infile(i) :    pointer to input file
     char * pcb->argv[](i) :    pointer to argument list
     int * pcb->parg_no(i) :    argument number of previous argument
                       (o) :    incremented to current argument number
     long  pcb->Prompt.V(i):    if set, prompt mode
     long  Rsys (i)        :    reference system
     float ROff            :    internal offset (reference)
     float RPs             :    internal pixel size (reference)
     long  Usys (i)        :    user system
     float UOff            :    internal offset (user)
     float UPs             :    internal pixel size (user)
     char * promptline (i) :    user prompt in prompt mode
     IO_float * value(i/o) :    read pixel coordinate (RSys)
               value.I (i) :    if set, value.V is default,
                                if not set, defval (conv. to RSys) is default
               value.I (o) :    set to TRUE, if input is not SkipToken
               value.V (i) :    used as default (RSys), if value.I is set
               value.V (o) :    read value (RSys)
     float * defval    (i) :    default value (RSys)
     int * pstatus     (o) :    exit status
  --------------------------------------------------------------------------*/
PUBLIC void argv_coord( CmdBlk * pcb,
                 long int RSys, float ROff, float RPs,
                 long int USys, float UOff, float UPs,
                 char * promptline, IO_float * value, float defval,
                 int * pstatus)
{ float valin;
  int iset = TRUE;
  const char * RoutineName = "argv_coord";
  char buffer[IO_size];

  SetSaxsError( SOName, RoutineName, NULL );

  *pstatus = Success;

  *pcb->parg_no += 1;
  if (pcb->TestBit) show_arg_no ( RoutineName, pcb );

  if (!value->I) value->V = defval;
  if (pcb->argc>*pcb->parg_no) {
    if(strcmp(pcb->argv[*pcb->parg_no],SkipToken)) {
      if (strcmp(pcb->argv[*pcb->parg_no],WildCard)) {
        valin    = floatexpr(pcb->argv[*pcb->parg_no],pstatus);
        if (*pstatus!=Success) {
          SetSaxsErrorMessage(pcb->argv[*pcb->parg_no]);
          return;
          }
        value->V = USER2REF(valin,ROff,RPs,UOff,UPs);
        }
      } else iset = FALSE;
    } else edf_history_skip( SaxsHistoryKey );

  if (pcb->Prompt.V) {
     iset = TRUE;
     valin    = REF2USER(value->V,ROff,RPs,UOff,UPs);
     printf("%s %s\n",InputRefSys,reftostr(USys));
     *pstatus = GetReal(pcb->infile,&valin,TRUE,promptline,valin);
     if (*pstatus==SkipInput) { iset = FALSE; *pstatus=Success; }
     if (*pstatus!=Success) return;
     value->V = USER2REF(valin,ROff,RPs,UOff,UPs);
     edf_history_take( SaxsHistoryKey );
     }
  if (iset) value->I = TRUE; else printf("%s\n",InputSkipped);
  if ((pcb->TestBit) && (value->I)) printf("%s %g\n",InputValue,value->V);

  /* add value to argument history list */
  if (value->I) {
    sprintf(buffer,"%g",value->V);
    edf_history_argv( SaxsHistoryKey, buffer );
    } else edf_history_argv( SaxsHistoryKey, SkipToken );

} /* argv_coord */

/*--------------------------------------------------------------------------
   argv_dist  : Read a distance in the user system from the argument
                list and from the input file. Convert it into the reference
                system. A wild card in the argument list chooses the default.
                A skip token in the argument or in the input line returns
                the default according to the .I flag. The .I flag is not
                changed.
   return value            :    void
    CmdBlk * pcb           :    command block
     FILE * pcb->infile(i) :    pointer to input file
     char * pcb->argv[](i) :    pointer to argument list
     int * pcb->parg_no(i) :    argument number of previous argument
                       (o) :    incremented to current argument number
     long  pcb->Prompt.V(i):    if set, prompt mode
     long  Rsys (i)        :    reference system
     float ROff            :    internal offset (reference)
     float RPs             :    internal pixel size (reference)
     long  Usys (i)        :    user system
     float UOff            :    internal offset (user)
     float UPs             :    internal pixel size (user)
     char * promptline (i) :    user prompt in prompt mode
     IO_float * value(i/o) :    read distance (RSys)
               value.I (i) :    if set, value.V is default,
                                if not set, defval (conv. to RSys) is default
               value.I (o) :    set to TRUE, if input is not SkipToken
               value.V (i) :    used as default (RSys), if value.I is set
               value.V (o) :    read value (RSys)
     float * defval    (i) :    default value (RSys)
     int * pstatus     (o) :    exit status
  --------------------------------------------------------------------------*/
PUBLIC void argv_dist ( CmdBlk * pcb,
                 long int RSys, float ROff, float RPs,
                 long int USys, float UOff, float UPs,
                 char * promptline, IO_float * value, float defval,
                 int * pstatus)
{ float valin;
  int iset = TRUE;
  const char * RoutineName = "argv_dist";
  char buffer[IO_size];

  SetSaxsError( SOName, RoutineName, NULL );

  *pstatus = Success;

  *pcb->parg_no += 1;
  if (pcb->TestBit) show_arg_no ( RoutineName, pcb );
  if (!value->I) value->V = defval;
  if (pcb->argc>*pcb->parg_no) {
    if(strcmp(pcb->argv[*pcb->parg_no],SkipToken)) {
      if (strcmp(pcb->argv[*pcb->parg_no],WildCard)) {
        valin    = floatexpr(pcb->argv[*pcb->parg_no],pstatus);
        if (*pstatus!=Success) {
          SetSaxsErrorMessage( pcb->argv[*pcb->parg_no]);
          return;
          }
        value->V = DUSER2DREF(valin,RPs,UPs);
        }
      } else iset = FALSE;
    } else edf_history_skip( SaxsHistoryKey );

  if (pcb->Prompt.V) {
     iset = TRUE;
     valin    = DREF2DUSER(value->V,RPs,UPs);
     printf("%s %s\n",InputRefSys,reftostr(USys));
     *pstatus = GetReal(pcb->infile,&valin,TRUE,promptline,valin);
     if (*pstatus==SkipInput) { iset = FALSE; *pstatus=Success; }
     if (*pstatus!=Success) return;
     value->V = DUSER2DREF(valin,RPs,UPs);
     edf_history_take( SaxsHistoryKey );
     }
  if (iset) value->I = TRUE; else printf("%s\n",InputSkipped);
  if ((pcb->TestBit) && (value->I)) printf("%s %g\n",InputValue,value->V);

  /* add value to argument history list */
  if (value->I) {
    sprintf(buffer,"%g",value->V);
    edf_history_argv( SaxsHistoryKey, buffer );
    } else edf_history_argv( SaxsHistoryKey, SkipToken );

} /* argv_dist */

/*--------------------------------------------------------------------------
   argv_end : Read to the end of the argument list. Exit, if too many
              arguments and if not in prompt mode.
  
   return value		   :	void
    CmdBlk * pcb           :    command block
     FILE * pcb->infile(i) :    pointer to input file
     char * pcb->argv[](i) :    pointer to argument list
     int * pcb->parg_no(i) :    argument number of previous argument
                       (o) :    set to argc
  --------------------------------------------------------------------------*/
PUBLIC void argv_end( CmdBlk * pcb )
{
  *pcb->parg_no += 1;
  if (pcb->argc>*pcb->parg_no) { /* too many arguments */
     printf("WARNING : %s : Too many arguments\n",pcb->MainName);
     printf("Usage: %s %s\n",pcb->MainName,pcb->MainHelp);
     /* if (!pcb->Prompt.V) exit(-1); */
     }
   *pcb->parg_no = pcb->argc;
} /* argv_end */

/****************************************************************************/
