/***************************************************************************/
/* Written 1994++ by Peter Boesecke                                        */
/* Copyright (C) 2011 European Synchrotron Radiation Facility              */
/*                       Grenoble, France                                  */
/*                                                                         */
/*    Principal authors: Peter Boesecke  (boesecke@esrf.eu)                */
/*                                                                         */
/*    This program is free software: you can redistribute it and/or modify */
/*    it under the terms of the GNU General Public License as published by */
/*    the Free Software Foundation, either version 3 of the License, or    */
/*    (at your option) any later version.                                  */
/*                                                                         */
/*    This program is distributed in the hope that it will be useful,      */
/*    but WITHOUT ANY WARRANTY; without even the implied warranty of       */
/*    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the        */
/*    GNU General Public License for more details.                         */
/*                                                                         */
/*    You should have received a copy of the GNU General Public License    */
/*    along with this program.  If not, see <http://www.gnu.org/licenses/>.*/
/***************************************************************************/
# define SAXSPAR_VERSION "SaxsParameter : V0.98 Peter Boesecke 2017-10-19"
/*+++------------------------------------------------------------------------
NAME

   SaxsParameter.c --- Option parameter routines 
 
SYNOPSIS
 
   # include SaxsParameter.h

DESCRIPTION
 
HISTORY
2003-05-29 V0.5  PB Peter Boesecke compilation OK
2003-05-30 V0.6  PB Peter Boesecke without SaxsImage.h 
2003-06-02 V0.7  PB optiontype -> parametertype -> option_type
2003-06-03 V0.8  PB some bugs removed, option_flag: int -> IO_long 
                    option_update now for all types
2003-08-03 V0.9  PB option_define for all types except IO_tpmode and IO_tprsys,
                    option_type2string
2003-08-11 V0.91 PB option_update called with const char * and int I
                    option_update  -> option_parameter_update
                    option_search  -> option_parameter_search
                    SaxsParameter.h
2003-08-12 V0.92 PB option_flag_update
2003-12-02 V0.93 PB new_command_option: semicolons added after empty labels
2005-07-15 V0.94 PB updated for modified SaxsError routine calls,
                    option_flag_update: wrong status check removed
                    option_parameter_search: status was not updated
2005-11-30 V0.95 PB option_line: parameter list of linekeyvalue updated
2007-04-19 V0.96 PB -Wall compiler warnings resolved
2007-06-17 V0.97 PB SaxsFilename.h -> filename.h
2017-10-19 V0.98 PB option_line, option_long, option_float:
                    argument changed: ImgBlk[] ib -> ImgHeadBlk ihb[]
                    floatkeyvalue, longkeyvalue, linekeyvalue: ib -> ihb
--------------------------------------------------------------------------*/
/***************************************************************************
* Private part                                                             *
***************************************************************************/
 
/***************************************************************************
* Include files                                                            *
***************************************************************************/
# include <stdio.h>
# include <stdlib.h>
# include <string.h>
# include <stdlib.h>
 
# include "SaxsDefinition.h"
# include "SaxsError.h"
# include "SaxsInput.h"
# include "SaxsOption.h"
# include "filename.h"
# include "edfio.h"                      /* data format specific routines */
 
# include "SaxsReplacekey.h"
# include "SaxsParameter.h"
 
#ifndef WIN32
# include <unistd.h>
#endif

/*----------------------------------------------------------------------------
 Constants
-----------------------------------------------------------------------------*/
const char * SPName = "SaxsParameter";

/*+++------------------------------------------------------------------------
NAME
 
   option_parameter_version --- return version string
 
SYNOPSIS
 
   const char *option_parameter_version ( void );
 
DESCRIPTION
 
RETURN VALUE
   Pointer to a constant character string containing the version of
   this module.
 
----------------------------------------------------------------------------*/
const char *option_parameter_version ( void )
{ return(SAXSPAR_VERSION);
} // option_parameter_version 

/*+++------------------------------------------------------------------------
NAME
 
  option_type2string --- convert option type to string
 
SYNOPSIS
 
  const char *option_type2string( int ptyp );
 
DESCRIPTION
  Converts option type to string,
 
RETURN VALUE
  option type as string
 
----------------------------------------------------------------------------*/
const char *option_type2string( int ptyp )
{ const char * value;
 
  switch (ptyp) {
    case IO_tpflag  : value="flag"; break;
    case IO_tpline  : value="line"; break;
    case IO_tplong  : value="long"; break;
    case IO_tplvexp : value="lvexp"; break;
    case IO_tpfloat : value="float"; break;
    case IO_tpflexp : value="flexp"; break;
    case IO_tpmode  : value="mode"; break;
    case IO_tprsys  : value="rsys"; break;
            default : value="none";
  }
 
  return( value );
 
} // option_type2string

/*+++------------------------------------------------------------------------
NAME
 
  search_command_option --- returns a pointer to the command option 
 
SYNOPSIS
 
  OptDsc * search_command_option(CmdBlk *pcb, const char *option);
 
DESCRIPTION
  Searches for option in pcb.co[] and returns its number (nct).
 
RETURN VALUE
  Returns the number of the command option or 0, if not found 
 
----------------------------------------------------------------------------*/
int search_command_option(CmdBlk *pcb, const char *option)
{ int value=0;
  int nct;

 if ((option[0]=='\0')||(option[0]=='#')) return(value);

 for (nct=1; nct<(pcb->co[0]).npar; nct++)
   if (strncmp(pcb->co[nct].code, option,IO_CodeLen-1)==0) {
     value = nct; 
     break;
   }

 return( value );
    
} // search_command_option

/*+++------------------------------------------------------------------------
NAME
 
  new_command_option --- returns the number of a free command option of type
 
SYNOPSIS
 
  int new_command_option(CmdBlk *pcb, int ptyp, const char *option)
 
DESCRIPTION
  Returns the number of a free command option of ptyp
 
RETURN VALUE
  Returns the number of a free command option or 0, if not available 
  The option ptyp and the option name are set.

----------------------------------------------------------------------------*/
int new_command_option(CmdBlk *pcb, int ptyp, const char *option)
{ int nct, value=0;
  int first=0, last=-1;

  switch (ptyp) {
    case IO_tpfloat : first = NArgFst;   last = NArgLst;   break;
    case IO_tpflexp : first = NFlexpFst; last = NFlexpLst; break;
    case IO_tplong  : first = NNumFst;   last = NNumLst;   break;
    case IO_tplvexp : first = NLvexpFst; last = NLvexpLst; break;
    case IO_tpline  : first = NLinFst;   last = NLinLst;   break;
    case IO_tpflag  : first = NFlgFst;   last = NFlgLst;   break;
    case IO_tpmode  : ;
    case IO_tprsys  : ;
    default: ;
  } 

  for (nct=first; nct<=last; nct++) {
    if ((pcb->co[nct]).code[0]=='\0') {
      value = nct;
      CopyLine(option,(pcb->co[value]).code,IO_CodeLen,-1);
      (pcb->co[value]).ptyp = ptyp;
      break;
    }
  }

  return( value );
 
} // new_command_option 

/*+++------------------------------------------------------------------------
NAME

  option_type --- return type of a command option 
 
SYNOPSIS

  int option_type ( const char *option, CmdBlk *pcb );
 
 
DESCRIPTION
  Returns 0 (IO_tpnone) if option is not defined, otherwise

        IO_tpflag,  IO_tpline,      IO_tplong,      IO_tplvexp,
        IO_tpfloat, IO_tpflexp,     IO_tpmode,      IO_tprsys

 
RETURN VALUE
  0 : option is not defined
 
----------------------------------------------------------------------------*/ 
int option_type     ( const char *option, CmdBlk *pcb )
{ int nct;
  int value=0;
 
  if (!pcb) return (value);

  nct = search_command_option( pcb, option );

  value = pcb->co[nct].ptyp;

  return(value);

} // option_type

/*+++------------------------------------------------------------------------
NAME
 
  option_parameter_search --- returns a pointer to the parameter option
 
SYNOPSIS
 
  void * option_parameter_search ( const char *option,
                                   CmdBlk *pcb, int * pstatus );
 
DESCRIPTION
  The return value is currently defined as void*. 
 
RETURN VALUE
  Pointer to the command line parameter option, NULL, if not found
 
----------------------------------------------------------------------------*/ 
void * option_parameter_search( const char *option,
                                CmdBlk *pcb, int * pstatus )
{ int n, nct;
  void * value = (void*) NULL;

  *pstatus = Failed;
 
  if (!pcb) return (value);

  SetSaxsError( SPName, "option_parameter_search", option );

  *pstatus = Success; 

  if (pcb->TestBit) {
    printf("option_parameter_search: option=%s\n", option);
  } 
 
  nct = search_command_option( pcb, option );
  if (!nct) { *pstatus = NoOption; return(value); }

  // Find values
  if ((NArgFst<=nct)&&(nct<=NArgLst)) {
    n=nct-NArgFst; value = (void *) &(pcb->Arg[n]);
  } else
    if ((NFlexpFst<=nct)&&(nct<=NFlexpLst)) {
      n=nct-NFlexpFst; value = (void * ) &(pcb->Flexp[n]);
  } else
    if ((NNumFst<=nct)&&(nct<=NNumLst)) {
      n=nct-NNumFst; value = (void * ) &(pcb->Num[n]);
  } else
    if ((NLvexpFst<=nct)&&(nct<=NLvexpLst)) {
      n=nct-NLvexpFst; value = (void * ) &(pcb->Lvexp[n]);
  } else
    if ((NLinFst<=nct)&&(nct<=NLinLst)) {
      n=nct-NLinFst; value = (void * ) &(pcb->Lin[n]);
  } else
    if ((NFlgFst<=nct)&&(nct<=NFlgLst)) {
      n=nct-NFlgFst; value = (void * ) &(pcb->Flg[n]);
  } else n=-1;
  if (n<0) { *pstatus = OutOfRange; return(value); }

  *pstatus = Success;
 
  return(value);

} // option_parameter_search 

/*+++------------------------------------------------------------------------
NAME
 
  option_parameter_update --- updates a command option parameter
 
SYNOPSIS

  void  *option_parameter_update ( const char *Value, int I, const char *option,
                                   CmdBlk *pcb, int * pstatus );
 
 
DESCRIPTION
  Updates the command option parameter with value. If the output type is 
  IO_float or IO_long the value string is converted to float or long using
  the functions floatexpression or longexpression.
 
RETURN VALUE
  Pointer to the updated command option, NULL, in case of an error
 
----------------------------------------------------------------------------*/
void  *option_parameter_update ( const char *Value, int I, const char *option,
                                 CmdBlk *pcb, int * pstatus )
{ int n, nct;

  void * value = (void *) NULL;
 
  *pstatus = Failed;

  if (!pcb) return (value);

  SetSaxsError( SPName, "option_parameter_update", option );

  *pstatus = Success;

  if (pcb->TestBit) {
    printf("option_parameter_update: option=%s\n", option);
  } 
 
  nct = search_command_option( pcb, option );
  if (!nct) { *pstatus = NoOption; return(value); }
  // Find values
  if ((NArgFst<=nct)&&(nct<=NArgLst)) {
    n=nct-NArgFst; 
    pcb->Arg[n].I = I;
    pcb->Arg[n].V = floatexpr( Value, pstatus );
    if (*pstatus!=Success) return( value );
    value = (void *) &(pcb->Arg[n]);
  } else
    if ((NFlexpFst<=nct)&&(nct<=NFlexpLst)) {
      n=nct-NFlexpFst; 
      pcb->Flexp[n].I = I;
      CopyLine(Value,pcb->Flexp[n].V,IO_size,0);
      value = (void *) &(pcb->Flexp[n]);
  } else
    if ((NNumFst<=nct)&&(nct<=NNumLst)) {
      n=nct-NNumFst; 
      pcb->Num[n].I = I;
      pcb->Num[n].V = longexpr( Value, pstatus );
      if (*pstatus!=Success) return( value );
      value = (void *) &(pcb->Num[n]);
  } else
    if ((NLvexpFst<=nct)&&(nct<=NLvexpLst)) {
      n=nct-NLvexpFst; 
      pcb->Lvexp[n].I = I;
      CopyLine(Value,pcb->Lvexp[n].V,IO_size,0);
      value = (void *) &(pcb->Lvexp[n]);
  } else
    if ((NLinFst<=nct)&&(nct<=NLinLst)) {
      n=nct-NLinFst; 
      pcb->Lin[n].I = I;
      CopyLine(Value,pcb->Lin[n].V,IO_size,0);
      value = (void *) &(pcb->Lin[n]);
  } else
    if ((NFlgFst<=nct)&&(nct<=NFlgLst)) {
      n=nct-NFlgFst; 
      pcb->Flg[n].I = I;
      pcb->Flg[n].V = longexpr( Value, pstatus );
      if (*pstatus!=Success) return( value );
      value = (void *) &(pcb->Flg[n]);
  } else n=-1;
  if (n<0) { *pstatus = OutOfRange; return(value); }

  *pstatus = Success;

  return( value );

} // option_parameter_update 

/*+++------------------------------------------------------------------------
NAME
 
  option_flag_update --- updates a flag option parameter
 
SYNOPSIS
 
  void  *option_flag_update ( long Value, int I, const char *option,
                              CmdBlk *pcb, int * pstatus );
 
 
DESCRIPTION
  Updates the flag command option parameter with value. 
 
RETURN VALUE
  Pointer to the updated flag command option, NULL, in case of an error
 
----------------------------------------------------------------------------*/ 
IO_long  *option_flag_update ( long Value, int I, const char *option,
                               CmdBlk *pcb, int * pstatus )
{ int n, nct;
 
  IO_long * value = (IO_long *) NULL;
 
  *pstatus = Failed;
 
  if (!pcb) return (value);
 
  SetSaxsError( SPName, "option_flag_update", option );

  *pstatus = Success;
 
  if (pcb->TestBit) {
    printf("option_flag_update: option=%s V=%ld I=%d\n", option, Value, I );
  }
 
  nct = search_command_option( pcb, option );
  if (!nct) { *pstatus = NoOption; return(value); }
  // Find values
  if ((NFlgFst<=nct)&&(nct<=NFlgLst)) {
    n=nct-NFlgFst;
    pcb->Flg[n].I = I;
    pcb->Flg[n].V = Value;
    value = &(pcb->Flg[n]);
  } else n=-1;
  if (n<0) { *pstatus = OutOfRange; return(value); }
 
  *pstatus = Success;
 
  return( value );
 
} // option_flag_update

/*+++------------------------------------------------------------------------
NAME
 
  option_float --- return parameter of a command option
 
SYNOPSIS
 
  IO_float option_float  ( const char *option,
                           CmdBlk *pcb, ImgHeadBlk ihb[], long num[],
                           long blkno, int *pstatus );
 
DESCRIPTION
  Writes value V and flag I of command option to *value.
  Keys are replaced and all expressions are evaluated.
  IO_float values are returned for float options (tp_float) and for
  float key options (tp_flexp)
 
RETURN VALUE

  IO_float value
 
----------------------------------------------------------------------------*/
IO_float option_float  ( const char *option,
                         CmdBlk *pcb, ImgHeadBlk ihb[], long num[],
                         long blkno, int *pstatus )
{ int nct, n;
  IO_float value;
 
  *pstatus = Failed;
  value.I = FALSE;
  value.V = 0.0;

  if (!pcb) return(value);
 
  SetSaxsError( SPName, "option_float", option );

  *pstatus = Success;

  nct = search_command_option( pcb, option );
  if (!nct) { *pstatus = NoOption; return(value); }

  // Find values
  if ((NArgFst<=nct)&&(nct<=NArgLst)) {
     n=nct-NArgFst; 
     value.I = pcb->Arg[n].I;
     value.V = pcb->Arg[n].V;
  } else {
    if ((NFlexpFst<=nct)&&(nct<=NFlexpLst)) {
      n=nct-NFlexpFst;
      value.I = pcb->Flexp[n].I;
      value.V = floatkeyvalue ( pcb->Flexp[n].V,
                     pcb, ihb, num, blkno, pstatus );
      if (*pstatus!=Success) return(value);
    } else {
    *pstatus = OutOfRange; return(value); 
    }
  }
 
  if (pcb->TestBit) {
    printf("option_float: %s.I=%d, %s.V=%g\n",
      option,value.I,option,value.V);
  }

  *pstatus=Success;
 
  return(value);
 
} // option_float

/*+++------------------------------------------------------------------------
NAME
 
  option_long --- return parameter of a command option
 
SYNOPSIS
 
  IO_long option_long ( const char *option,
                        CmdBlk *pcb, ImgHeadBlk ihb[], long num[],
                        long blkno, int *pstatus );
 
DESCRIPTION
  Writes value V and flag I of command option to *value.
  Keys are replaced and all expressions are evaluated.
  IO_long values are returned for long options (tp_long) and for
  long key options (tp_lvexp)
 
RETURN VALUE
   
  IO_long value
 
----------------------------------------------------------------------------*/
IO_long option_long ( const char *option,
                      CmdBlk *pcb, ImgHeadBlk ihb[], long num[],
                      long blkno, int *pstatus )
{ int nct, n;
  IO_long value;
 
  *pstatus = Failed;
  value.I = FALSE;
  value.V = 0l;
 
  if (!pcb) return(value);
 
  SetSaxsError( SPName, "option_long", option );

  *pstatus = Success;

  nct = search_command_option( pcb, option );
  if (!nct) { *pstatus = NoOption; return(value); }
 
  // Find values
  if ((NNumFst<=nct)&&(nct<=NNumLst)) {
     n=nct-NNumFst;
     value.I = pcb->Num[n].I;
     value.V = pcb->Num[n].V;
  } else {
    if ((NLvexpFst<=nct)&&(nct<=NLvexpLst)) {
      n=nct-NLvexpFst;
      value.I = pcb->Lvexp[n].I;
      value.V = longkeyvalue ( pcb->Lvexp[n].V,
                     pcb, ihb, num, blkno, pstatus );
      if (*pstatus!=Success) return(value);
    } else {
    *pstatus = OutOfRange; return(value);
    }
  } 

  if (pcb->TestBit) {
    printf("option_long: %s.I=%d, %s.V=%ld\n",
      option,value.I,option,value.V);
  }
 
  *pstatus=Success;
 
  return(value);
 
} // option_long 

/*+++------------------------------------------------------------------------
NAME
 
  option_line --- return parameter of a command option 
 
SYNOPSIS
 
  IO_line option_line ( const char *option,
                        CmdBlk *pcb, ImgHeadBlk ihb[], long num[],
                        long blkno, int *pstatus );

DESCRIPTION
  Writes value V and flag I of command option to *value. 
  Keys are replaced. 

ARGUMENTS
  ...
  long num[] : array with current image numbers, if num==NULL
               no replacements of keys
  ...
 
RETURN VALUE
 
  IO_long value

e---------------------------------------------------------------------------*/
IO_line option_line ( const char *option,
                      CmdBlk *pcb, ImgHeadBlk ihb[], long num[],
                      long blkno, int *pstatus )
{ int nct, n;

  IO_line value;

  *pstatus = Failed;
  value.I = FALSE;
  value.V[0] ='\0';
 
  if (!pcb) return(value);

  SetSaxsError( SPName, "option_line", option );
 
  *pstatus = Success;

  nct = search_command_option( pcb, option );
  if (!nct) { *pstatus = NoOption; return(value); }
 
  // Find values
  if ((NLinFst<=nct)&&(nct<=NLinLst)) n=nct-NLinFst; else n=-1;
  if (n<0) { *pstatus = OutOfRange; return(value); }

  value.I = pcb->Lin[n].I;
  if (num) {
    linekeyvalue ( value.V, IO_size, pcb->Lin[n].V, 0,
                   pcb, ihb, num, blkno, pstatus );
    if (*pstatus!=Success) return(value);
  } else CopyLine(pcb->Lin[n].V,value.V,IO_size,0);

  if (pcb->TestBit) {
    printf("option_line: %s.I=%d, %s.V=\"%s\"\n",
      option,value.I,option,value.V);
  }
 
  *pstatus=Success;

  return(value);

} // option_line

/*+++------------------------------------------------------------------------
NAME
 
  option_flag --- return parameter of a command option
 
SYNOPSIS
 
  IO_long option_flag ( const char *option, 
                        CmdBlk *pcb, int * pstatus );
 
DESCRIPTION
  Returns a flag option.
 
RETURN VALUE

  IO_long value
 
----------------------------------------------------------------------------*/
IO_long option_flag ( const char *option, 
                      CmdBlk *pcb, int * pstatus )
{ int nct, n;
  IO_long value;
 
  *pstatus = Failed;

  value.I = FALSE;
  value.V = 0l;
 
  if (!pcb) return(value);
 
  SetSaxsError( SPName, "option_flag", option );

  *pstatus = Success;

  nct = search_command_option( pcb, option );
  if (!nct) { *pstatus = NoOption; return(value); }
 
  // Find values
  if ((NFlgFst<=nct)&&(nct<=NFlgLst)) n=nct-NFlgFst; else n=-1;
  if (n<0) { *pstatus = OutOfRange; return(value); }
 
  value.V =  pcb->Flg[n].V;
  value.I =  pcb->Flg[n].I;

  if (pcb->TestBit) {
    printf("option_flag: %s.I=%d, %s.V=%ld\n",
      option,value.I,option,value.V);
  }
 
  *pstatus=Success;
 
  return(value);
 
} // option_flag
 
/*+++------------------------------------------------------------------------
NAME
 
  option_define --- define a new type option and a default value
 
SYNOPSIS
  
  void option_define ( int ptyp, const char *option, const char *defval,
                       CmdBlk *pcb );
 
 
DESCRIPTION
  Defines a new ptyp option and a default value. It replaces the functions 
  DefFloatOption, DefIntOption, DefLineOption and DefFlgOption. The option 
  number is not used any more. Depending on ptyp the options are written 
  to 
        ptyp IO_tpfloat : pcb->Arg[IO_MaxFloatOpt];
             IO_tpflexp : pcb->Flexp[IO_MaxFlexpOpt];
             IO_tplong  : pcb->Num[IO_MaxIntOpt];
             IO_tplvexp : pcb->Lvexp[IO_MaxLvexpOpt];
             IO_tpline  : pcb->Lin[IO_MaxLineOpt];
             IO_tpflag  : pcb->Flg[IO_MaxFlagOpt];
             IO_tpmode  : not implemented
             IO_tprsys  : not implemented

  If the definition fails the program stops with an error status.
 
----------------------------------------------------------------------------*/ 
void option_define ( int ptyp, const char *option, const char *defval,
                     CmdBlk *pcb )
{ int nct;
  int status = Success;

  if (pcb->TestBit) {
    printf("option_define: type=%s, default %s.V=%s\n", 
      option_type2string(ptyp), option, defval);
  }

  if( CheckOptionDuplication( pcb, option ) ) {
    fprintf(stderr,"%s-%s\n",ErrorOptionDuplicated,option);
    exit(-1);
  }
 
  nct = new_command_option(pcb, ptyp, option);
  if (nct==0) {
    fprintf(stderr,"%s%s\n", ErrorOptionDefnFailed, option); 
    exit(-1);
  }

  (pcb->co[nct]).npar = 1;

  /* Write default value */
  switch (ptyp) {
    case IO_tpfloat : 
      pcb->Arg[nct-NArgFst].V = floatexpr(defval,&status); 
      break;
    case IO_tpflexp : 
      CopyLine(defval,pcb->Flexp[nct-NFlexpFst].V,IO_size,0);
      break;
    case IO_tplong  : 
      pcb->Num[nct-NNumFst].V = longexpr(defval,&status);
      break;
    case IO_tplvexp : 
      CopyLine(defval,pcb->Lvexp[nct-NLvexpFst].V,IO_size,0);
      break;
    case IO_tpflag  : 
      pcb->Flg[nct-NFlgFst].V = longexpr(defval,&status);
      break;
    case IO_tpline  : 
      CopyLine(defval,pcb->Lin[nct-NLinFst].V,IO_size,0);
      break;
    default : status=Failed;
  }
  if (status!=Success) {
    fprintf(stderr,"%s%s: %s\n", ErrorOptionDefnFailed, option, defval);
    exit(-1);
  }

} // option_define


