/***************************************************************************/
/* Written 1994++ by Peter Boesecke                                        */
/* Copyright (C) 2011 European Synchrotron Radiation Facility              */
/*                       Grenoble, France                                  */
/*                                                                         */
/*    Principal authors: Peter Boesecke  (boesecke@esrf.eu)                */
/*                                                                         */
/*    This program is free software: you can redistribute it and/or modify */
/*    it under the terms of the GNU General Public License as published by */
/*    the Free Software Foundation, either version 3 of the License, or    */
/*    (at your option) any later version.                                  */
/*                                                                         */
/*    This program is distributed in the hope that it will be useful,      */
/*    but WITHOUT ANY WARRANTY; without even the implied warranty of       */
/*    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the        */
/*    GNU General Public License for more details.                         */
/*                                                                         */
/*    You should have received a copy of the GNU General Public License    */
/*    along with this program.  If not, see <http://www.gnu.org/licenses/>.*/
/***************************************************************************/
/* SaxsOption.h V6.80 by Peter Boesecke 2017-02-11                         */
/**************************************************************************/

#ifndef _SAXS_OPTION
# define _SAXS_OPTION

# include "SaxsError.h"
# include "SaxsExpression.h"
# include "SaxsInput.h"
# include "SaxsDefinition.h"
# include "numio.h"

/* History */
# define SaxsHistoryKey "i0" /* key to history */

/* Restrictions */
# define IO_CodeLen      5 /* maximum length of option strings */
# define IO_MaxFloatOpt 30 /* maximum number of definable float options */	
# define IO_MaxFlexpOpt 30 /* maximum number of definable flexp options */
# define IO_MaxIntOpt   10 /* maximum number of definable integer options */
# define IO_MaxLvexpOpt 10 /* maximum number of definable lvexp options */
# define IO_MaxLineOpt  10 /* maximum number of definable line options */
# define IO_MaxFlagOpt  10 /* maximum number of definable flags */

/* File opening modi */
# define IO_New             0x0001l   /* new file is open */
# define IO_Old             0x0002l   /* old file is open */
# define IO_FileProtect     0x0004l   /* file cannot be modified */
# define IO_ImageProtect    0x0008l   /* existing images cannot be modified */
# define IO_DontReadData    0x0010l   /* do not read/alloc binary data */
# define IO_DontReadWrite   0x0020l   /* do not rd/wr image header nor data */
# define IO_DontOpen        0x0040l   /* do not open file */
# define IO_Dummy           0x0080l   /* like IO_DontReadWrite, but create 
                                         dummy data */

/* Description
All files can be updated (read and write).

IO_New           : create a new output file, backup the old file, if possible.
IO_Old           : open an old file
IO_FileProtect   : file is protected (no write)
IO_ImageProtect  : all data blocks in the file are protected (no update of
                   existing data blocks, append only)
IO_DontReadData  : only the header will be read or created,
                   no binary data will be read or allocated
IO_DontReadWrite : file is not open,
                   no read, no write (e.g. for read and write with separate
                   routines, format conversion etc)
IO_DontOpen      : file is not open
IO_Dummy         : similar to IO_DontReadWrite, but reading routines 
                   create dummy data
*/

/* Block structures */
typedef struct { char     V[IO_size]; int I; } IO_line;
typedef struct { char     V[IO_size]; int I; } IO_lvexp;
typedef struct { long int V;         int I; } IO_long;
typedef struct { char     V[IO_size]; int I; } IO_flexp;
typedef struct { NumProg *V;         int I; } IO_prog;
typedef struct { float    V;         int I; } IO_float;

/* Option types */
enum { IO_tpnone=0,    IO_tpflag,    IO_tpline,      IO_tplong,
       IO_tplvexp,     IO_tpfloat,   IO_tpflexp,     IO_tpmode,
       IO_tprsys,      IO_tppro,     IO_tpaxis };

typedef struct OptionDescriptor {
	char	   code[IO_CodeLen];
	int        ptyp;   /* parameter type */
	int	   npar;   /* number of parameters */
	} OptDsc;

enum CmdOptions
{       NEmpty=0,        NIODebug,        NIOLevel,        NTestBit,        
        NHelp,           NPrompt,         NShift,          NRSys,
        NUSys,           NGeneralBlock,   NPassHeader,     NCleanHeader,
        NCreateVariance, NAntiAliased,    NVarianceWeight, NOutputDataType, 
        NOutputDataValueOffset, NOutputCompression,        NBslInputByteOrder,
        NAdd,            NIPMin,          NMaxLineWidth,   NMinHeaderSize,
        NCommonOrientation,               NBeam,
        NArgFst,          NArgLst=NArgFst+IO_MaxFloatOpt-1,
        NFlexpFst,        NFlexpLst=NFlexpFst+IO_MaxFlexpOpt-1,
        NNumFst,          NNumLst=NNumFst+IO_MaxIntOpt-1,
        NLvexpFst,        NLvexpLst=NLvexpFst+IO_MaxLvexpOpt-1,
        NLinFst,          NLinLst=NLinFst+IO_MaxLineOpt-1,
        NFlgFst,          NFlgLst=NFlgFst+IO_MaxFlagOpt-1,
        NCmdLast,         IO_CmdOptLen };
 
enum ImgOptions
{    /* NEmpty=0, */    NOpenMode=1,    NName,          NMemory,
        NFirst,         NLast,          NInc,           NMinClip,
        NMaxClip,       NConst,         NFactor,        NVal,
        NErrVal,        NBin,           NAve,           NGnaw,
        NTitle,         NTime,          NDim,           NDummy,
        NDDummy,        NOffset,        NBinSiz,        NPixSiz,
        NWaveLength,    NOrientation,   NIndexOrder,    NProjection,    
        NAxisType,      NCenter,        NSampleDistance,NDetRot,
        NBeamCenter,    NBeamDistance,  NDetTilt,       NSearchList,
        NImgLast,       IO_ImgOptLen };

typedef struct ImageBlock {
        int  blkno;                 /* block number */
        /* file control variables, internal use */
        int       Stream;           /* I/O stream used internally */
	long int  StreamOpenMode;   /* actual open mode used internally */
        long int  FileNumber;       /* file number of actually open file */
        long int  LoopNumber;       /* loop number of image */
        char      FileName[IO_size]; /* file name of actually open file */
        /* --- image option variables */
        IO_line   Name;          /* filename with pattern */
        IO_long   FFirst;        /* first file number extracted from Name */ 
        IO_long   FLast;         /* last file number extracted from Name */ 
        IO_long   FInc;          /* file number increment extracted from Name */
        IO_long   OpenMode;
        IO_long   Memory;
        IO_long   First;
        IO_long   Last;
        IO_long   Inc;
        IO_flexp  MinClip;
        IO_flexp  MaxClip;
        IO_flexp  Const;
        IO_flexp  Factor;
        IO_flexp  Val;
        IO_flexp  ErrVal;
        IO_lvexp  Bin[3];
        IO_long   Ave;
        IO_lvexp  Gnaw[3];

        /*     image header block variables */
        IO_line   Title;
        IO_line   Time;
        IO_lvexp  Dim[3];  // only 3 should be used (PB 2000-11-16) 
        IO_flexp  Dummy;
        IO_flexp  DDummy;
        IO_flexp  Offset[3];
        IO_flexp  Center[3];
        IO_flexp  BinSiz[3];
        IO_flexp  PixSiz[3];
        IO_flexp  SampleDistance;
        IO_flexp  WaveLength;
        IO_flexp  DetRot[4];
        IO_lvexp  Orientation;
        IO_lvexp  IndexOrder;
        IO_long   Projection;
        IO_long   AxisType[3];

        /*     alternative variables */
        IO_flexp  BeamCenter[3];
        IO_flexp  BeamDistance;
        IO_flexp  DetTilt[4];

        /* h5 open search list */
        IO_line   SearchList;

}       ImgBlk;

typedef struct CommandBlock {
	/* control variables, internal use */
	char *  MainName;    /* pointer to name of main program */
	const char * MainVersion; /* version string of main program */
	const char * MainHelp;    /* help string of main program */
        void *  MainData;    /* pointer to data needed by the main program */
        char    SaxsVersion[IO_size]; /* version string appended to history line */ 
	int     argc;        /* length of argument list */
	char ** argv;        /* pointer to argument list */
	int  *  parg_no;     /* pointer to argument number variable */
	FILE *  infile;      /* input file */
	OptDsc * co;         /* pointer to command option descriptor array,
                                NULL after call of ReadOption */
	OptDsc * io;	     /* pointer to image option descriptor array,
                                NULL after call of ReadOption */
	ImgBlk * ib;         /* pointer to image block array */
	int     ImgBlkLen;   /* number of defined image blocks */
	int     RejectNum;   /* if TRUE, numbers are rejected as options */
        long int  Cycles;    /* contains number of cycles before output image is written */
        /* --- command option variables */
	int     TestBit;     /* for program tests only: on/off */
	IO_long IODebug;     /* Debug settings of file i/o routines */
	IO_long IOLevel;     /* Debug level of file i/o routines */
	IO_long Help;        /* show help and exit: on */
	IO_long Prompt;      /* prompt mode: on/off */
	IO_long RSys;        /* reference system for calculation */
	IO_long USys;        /* reference system for user input */
	IO_long GeneralBlock;         /* write general block */ 
        IO_long PassHeader;           /* write the full input header into
                                         the output header */ 
        IO_long CleanHeader;          /* remove inconsistencies when reading headers */
        IO_long CreateVariance;       /* create variance arrays for error
                                         propagation calculations */
        IO_long AntiAliased;          /* interpolation mode */ 
        IO_long VarianceWeight;       /* data weighting mode */ 
        IO_long OutputDataType;       /* output data type */
        IO_long OutputDataValueOffset;/* output data value offset */
        IO_long OutputCompression;    /* compression type of output data */
        IO_long BslInputByteOrder;    /* byte order of bsl input files */
        IO_long MaxLineWidth;         /* maximum width of text lines */
        IO_long MinHeaderSize;        /* minimum header size */
	IO_long CommonOrientation;    /* common orientation of all images: on/off */
	IO_long Beam;                 /* write beam parameters to header: on/off */
        IO_long Add;                  /* number of output images to add */
        IO_float IPMin;               /* minimum pixel coverage */
        IO_long Ave;                  /* average/sum resized pixels */
	IO_flexp Shift[3];            /* shift of the output image origin */
	IO_float Arg[IO_MaxFloatOpt]; /* definable float options, free use */
        IO_flexp Flexp[IO_MaxFlexpOpt]; /* definable flexp options, free use */
	IO_long Num[IO_MaxIntOpt];    /* definable long options, free use */
	IO_lvexp Lvexp[IO_MaxLvexpOpt]; /* definable lvexp options, free use */
	IO_line Lin[IO_MaxLineOpt];   /* definable line options, free use */
	IO_long Flg[IO_MaxFlagOpt];   /* definable flags, free use */
}       CmdBlk;

extern void
    InitOptions    ( const char * mainhelp, const char * mainvers, 
                     void * maindata, int rjctnum,
		     CmdBlk * pcb, ImgBlk ib[], int ib_len ),
    DefFloatOption ( CmdBlk * pcb, char * name, int argopt_no ),
    DefIntOption   ( CmdBlk * pcb, char * name, int numopt_no ),
    DefLineOption  ( CmdBlk * pcb, char * name, int linopt_no ),
    DefFlgOption   ( CmdBlk * pcb, char * name, int flgopt_no ),
    ReadOptions    ( char * argv[], int * parg_no,
		     CmdBlk * pcb, ImgBlk ib[], int * status),
    PrintBlocks    ( CmdBlk * pcb, ImgBlk ib[] ),
    PrintImageBlock   ( ImgBlk * ib ),
    PrintCommandBlock ( CmdBlk * cb );

extern void
    argv_end       ( CmdBlk * pcb ),
    argv_start     ( CmdBlk * pcb, int narg ),
    argv_filename  ( CmdBlk * pcb,
                     char * promptline, ImgBlk ib[], int blkno,
                     char * defval, int * pstatus),
    argv_line      ( CmdBlk * pcb,
                     char   * promptline, IO_line  * value, char * defval,
                     int    * pstatus),
    argv_float     ( CmdBlk * pcb,
                     char   * promptline, IO_float * value, float  defval,
                     int    * pstatus),
    argv_flexp     ( CmdBlk * pcb,
                     char   * promptline, IO_flexp * value, char * defval,
                     int    * pstatus),
    argv_long      ( CmdBlk * pcb,
                     char   * promptline, IO_long  * value, long int defval,
                     int    * pstatus),
    argv_lvexp     ( CmdBlk * pcb,
                     char   * promptline, IO_lvexp * value, char * defval,
                     int    * pstatus),
    argv_flag      ( CmdBlk * pcb,
                     char   * promptline, IO_long  * value, long int defval,
                     int    * pstatus),
    argv_coord     ( CmdBlk * pcb,
                     long int RSys, float ROff, float RPs,
                     long int USys, float UOff, float UPs,
                     char * promptline, IO_float * value, float defval,
                     int * pstatus),
    argv_dist      ( CmdBlk * pcb,
                     long int RSys, float ROff, float RPs,
                     long int USys, float UOff, float UPs,
                     char * promptline, IO_float * value, float defval,
                     int * pstatus);
extern char
    *reftostr      ( long int refsys ),
    *protostr      ( long int protyp ),
    *axistostr     ( long int axistyp );

extern const char
    *saxs_libversion   ( void ),
    *saxs_progversion  ( CmdBlk * pcb ),
    *saxs_version      ( CmdBlk * pcb );

extern long int
    strtopro       ( const char * str ),
    strtoaxis      ( const char * str );

extern int
    CheckOptionDuplication( CmdBlk * pcb, const char * name ),
    IsOutputOption( const char * name ),
    IsInputOption( const char * name ),
    extract_filenumbers( ImgBlk *pib, IO_line *pName,
                         int opt_onoff, int * pstatus );


/*----------------------------------------------------------------------------
 String constants
----------------------------------------------------------------------------*/
extern const char 
  *ErrorOptionDuplicated, *ErrorOptionDefnFailed, *ErrorCmdArrayTooShort,
  *ErrorImgArrayTooShort, *ErrorCannotAllcMemory;

#endif /* _SAXS_OPTION */

/**************************************************************************/
/*                                                                        */
/**************************************************************************/
