/***************************************************************************/
/* Written 1994++ by Peter Boesecke                                        */
/* Copyright (C) 2011 European Synchrotron Radiation Facility              */
/*                       Grenoble, France                                  */
/*                                                                         */
/*    Principal authors: Peter Boesecke  (boesecke@esrf.eu)                */
/*                                                                         */
/*    This program is free software: you can redistribute it and/or modify */
/*    it under the terms of the GNU General Public License as published by */
/*    the Free Software Foundation, either version 3 of the License, or    */
/*    (at your option) any later version.                                  */
/*                                                                         */
/*    This program is distributed in the hope that it will be useful,      */
/*    but WITHOUT ANY WARRANTY; without even the implied warranty of       */
/*    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the        */
/*    GNU General Public License for more details.                         */
/*                                                                         */
/*    You should have received a copy of the GNU General Public License    */
/*    along with this program.  If not, see <http://www.gnu.org/licenses/>.*/
/***************************************************************************/
/*+++
1 SaxsOption

2 PURPOSE
Reading of options from the program parameter list and creation of command, 
variable and image blocks for image analysis programs.

2 HISTORY
04-Jan-1995 PB creation
06-Jan-1995 PB IO_Version 1.2
07-Jan-1995 PB IO_Version 1.4 rjctnum, version numbers in ShowHelp
10-Jan-1995 PB manually realigned for RED
11-Jan-1995 PB some output formatting changed
12-Jan-1995 PB arrays for Dim, Shift, Center etc. enlarged from Dim[2] to [3].
12-Jan-1995 PB Shift (float)
16-Jan-1995 PB V2.3 file opening modi in imageoption.h
28-Jan-1995 PB V2.4 Title, SampleDistance [m], WaveLength [m] added,
	       GetOption changed
29-Jan-1995 PB V2.42 CommandOption and ImageOption dynamically generated and
	       pointer in command block, after first call of ReadOption the
	       option descriptors are freed. ReadOption can only be used once!
	       DefineLineOption added, -rsys option added
29-Jan-1995 PB V2.5 final version
30-Jan-1995 PB V2.51 USys parameter for user reference system added
07-Feb-1995 PB V2.6  Open modes changed
11-Feb-1995 PB V2.7  image block with Stream
12-Feb-1995 PB V2.8  image block with FullName
26-Feb-1995 PB V2.9  reftostr added
05-Mar-1995 PB V2.91 IO_DontScale added
08-Mar-1995 PB V2.92 Help suppressed for ImgBlkLen <2 and <1.
08-Apr-1995 PB V3.0  pointer ib to image block in CmdBlk added.
08-Apr-1995 PB V3.1  IO_FlgOption added
04-Jun-1995 PB V4.0  Changed to 4.0 at the same time where the definition of
                     saxsref was changed in saxs_definition. No change in
                     imageoption. Programs with imageoption Version>4.0
                     are using the new saxsref definition.
19-Jun-1995 PB V4.1  for a corrected version of saxs_definitions
09-Sep-1995 PB V5    a) wild card for options: a wild card does neither
                        change the value (.V) nor the the i-flag (.I).
                        The option is IGNORED for the parameter, e.g.
                        " -opix = 0.4 +opix = = " sets only PixSiz[2]:
                        PixSiz[2].I = -1; PixSiz[2].V = 0.4.
                     b) options can be repeated, the last option is valid
                     c) It can be tested whether a "+" or "-"-option has
                        been used:
                        xxx.I > 0 if set by "+"-option
                        xxx.I == 0 if not set
                        xxx.I < 0 if set by "-"-option
                        The test of an option must be: if (xxx.I)  ...
                     d) pcb->Message pointer added for display of errors etc.
                     e) saxs_definitions.h included for display of SD_Version
                     f) floatexpr and longexpr used in ScanFloat and ScanLong
15-Sep-1995 PB V5.1  a) prompt routines
                     b) default for pcb->Prompt.V changed to TRUE
14-Nov-1995 PB V5.2  a) new prompt routines
                     b) default for pcb->USys.V is always pcb->RSys.V
                     c) new rebin option i<n>bin <bin_1> <bin_2>
                     d) pcb->ErrorValue added
24-Nov-1995 PB V5.3  a) In command block Message, ErrorValue and
			DataErrorValue replaced by SaxsErrBlk seb.
03-Dec-1995 PB V5.4  StreamOpenMode
25-Dec-1995 PB V5.41 int mode --> long mode
26-Dec-1995 PB V5.42 long --> long int, mhlp, mver --> const char *
29-Mar-1996 PB V5.6  clip function -i<N>min and -i<N>max
18-May-1996 PB V5.61 number of float options increased to 23
19-May-1996 PB V5.7  gnaw function -i<N>gna <d> <d>
06-Oct-1996 PB V5.71 constants of reftostr declared outside
10-Oct-1996 PB V5.72 FLT_MIN, FLT_MAX calculated from FLT_MIN_EXP etc.
                     if not defined
10-Oct-1996 PB V5.8  imageoption renamed to SaxsOption
14-Oct-1996 PB V5.9  SaxsOptionVersion
14-Nov-1996 PB V5.91 hidden options
17-Mar-1998 PB V6.00 GeneralBlock
01-Jun-1998 PB V6.01 BslInputByteOrder
24-Jun-1999 PB V6.1  edf_history (include edfio.h)
26-Jun-1999 PB V6.11 Time keyword 
21-Dec-1999 PB V6.12 IO_DontScale replaced with IO_Dummy
31-Dec-1999 PB V6.13 new reference system BEAM 
10-Jan-2000 PB V6.14 new reference systems CENTER and NORMAL,
                     NORMAL replaces BEAM
19-Mar-2000 PB V6.15 memory added, mem=0 excluded
31-Mar-2000 PB V6.16 MaxFloatOpt increased to 25 (from 23)
07-Apr-2000 PB V6.17 option -mlw <long> added (max_line_width)
01-Mai-2000 PB V6.18 ShowDataTypes()
30-Jul-2000 PB V6.19 Orientation
31-Aug-2000 PB V6.20 MaxFloatOptions increased from 25 to 30
15-Sep-2000 PB V6.21 MaxLineOptions increased from 4 to 5 
16-Nov-2000 PB V6.22 Image block: IO_long Dim[4] instead of Dim[3]
04-Dec-2000 PB V6.23 SaxsExpression.h
06-Jul-2001 PB V6.24 Stream default is -1
08-Jul-2001 PB V6.25 FFirst, FLast, FInc, argv_filename, FullName->FileName
09-Jul-2001 PB V6.26 ib[].FileNumber and ib[].FileName are set by OpenImageFile
09-Jul-2001 PB V6.27 repeat command +r -r removed
11-Jul-2001 PB V6.28 new command option: -add (number of output images to add) 
02-Sep-2001 PB V6.29 OutputDataType, OutputDataValueOffset
13-Sep-2001 PB V6.30 SaxsHistoryKey
16-Sep-2001 PB V6.31 PassHeader
2002-05-27  PB V6.32 new command option: +/-ave (average/sum resized/binned pixels)
2002-06-01  PB V6.33 command option ave changed to image option: 
                     +/-iNave (average/sum resized/binned pixels)
2002-06-09  PB V6.34 LoopNumber
2002-09-01  PB V6.35 extract_filenumbers: sets First, Last, Inc 
2003-02-04  PB V6.36 MinClip, MaxClip, Const, Factor IO_tpfloat -> IO_tpflexp 
                     (preparation for replacekey)
                     SaxsImage.h, SaxsPrompt.inc -> argv_flexp
2003-02-06  PB V6.37 Bin, Gnaw -> IO_tplong -> IO_tplvexp, 
                     SaxsPrompt.inc -> argv_lvexp
                     Bin, Gnaw -> IO_lvexp
2003-02-07  PB V6.38 IO_tpctrl renamed to IO_tpflag
                     Dummy, DDummy -> IO_flexp, Dim[4] -> Dim[3] -> IO_lvexp
2003-02-10  PB V6.39 Shift -> IO_flexp
2003-05-25  PB V6.40 Offset -> IO_flexp, Center -> IO_flexp, PixSiz -> IO_flexp,
                     SampleDistance -> IO_flexp, WaveLength -> IO_flexp
2003-05-29  PB V6.41 Orientation -> IO_lvexp, DoImgOptions: some breaks added,
2003-05-30  PB V6.42 IO-parameter types IO_tp... etc: value 0 is now IO_tpnone.
                     new include file: SaxsParameter.inc: floatoption, 
                     longoption, lineoption, flagoption
2003-05-31  PB V6.43 ReadOptions: does no longer free command options
2003-06-02           SaxsParameter.inc
2003-08-05  PB V6.44 option_flag: int -> IO_long *
                     Option types -> SaxsOption.h
                     new: IO_MaxFlexpOpt, IO_MaxLvexpOpt,
                          pcb->Flexp[IO_MaxFlexpOpt], pcb->Lvexp[IO_MaxLvexpOpt]
                          InitCommandOptions: IO_Flexp, IO_Lvexp
                     PrintOptions shows only defined options
                     const char *option_type2string( int ptyp );
                     DoCmdOptions: Flexp, Lvexp (corrected 2003-08-12)
                     SaxsParameter.inc -> SaxsParameter.h, .c
                     CmdOptions and ImgOptions -> SaxsOption.h
2003-11-30  PB V6.45 BinSiz, IO_Region 
2004-04-10  PB V6.46 numio included in ShowHelp
2004-07-17  PB V6.47 MinHeaderSize added, new option mhs
2004-09-27  PB V6.48 projection types IO_ProSaxs and IO_ProWaxs added,
                     new image option SProjection, new option type IO_tppro,
                     protostr, strtopro added, SaxsPrompt.inc
2004-10-01  PB V6.49 SaxsPrompt.inc 07-Feb-2003 restored with cosmetic changes
2004-10-04  PB V6.50 DetRot added, SaxsPrompt.inc
2004-10-30  PB V6.51 exit after ShowHelp replaced by return(StoppedAfterHelp),
                     only exits in init-functions and def-functions remain
2005-02-12  PB V6.52 ScanEmpty: To avoid reading over the end of the argument 
                     list the functions "Scan..." stop if the input parameter 
                     s1 is the NULL pointer.
2005-07-11  PB V6.53 Command option CreateVariance and image options 
                     IO_flexp Val and IO_flexp ErrVal added,
                     SetSaxsErrorMessage
2005-07-14  PB V6.54 seb removed from control block, also in SaxsPrompt.inc
2005-08-04  PB V6.55 pib->blkno added, InitImageBlock parameter blkno added
2005-08-05  PB V6.56 InitImageBlock updatee
2005-12-05  PB V6.57 AntiAliased added
2006-03-15  PB V6.58 CheckOptionDuplication is now public, because it is 
                     needed in SaxsParameter.c (name changed to const) 
                     PrintImageBlock, PrintCommandBlock are now public,
                     because they are needed in SaxsImage.c
2007-03-07  PB V6.59 ShowHelp: displays VVERSION if defined
2007-03-30  PB V6.60 sizeof is always first operand
2007-04-19           -Wall compiler warnings resolved
2007-06-17  PB V6.61 Variable containing the number of Cycles added.
                     SaxsFilename.h -> filename.h
2007-12-01  PB V6.62 Axis types
2008-05-25  PB V6.63 VarianceWeight added
2008-05-27  PB V6.64 Option pmin for Ipolmin added
2008-10-29  PB V6.65 MaxLinOpt increased from 5 to 10
2009-10-02  PB V6.66 ShowHelp: REFERENCE_VERSION added
2010-03-18  PB V6.67 Reference systems and projection types are 
                     now defined in project.h as preprocessor constants
2010-12-12  PB V6.68 ShowHelp: (int) NULL -> '\0'
2011-01-05  PB V6.69 output compression option: -cmpr <compression>
2011-05-14  PB V6.70 Axis types are now defined in reference.h as 
                     preprocessor constants
2011-06-05  PB V6.71 Option cori to use common orientation with output image
2011-07-24  PB V6.72 Reference System IO_Tangens added
2013-05-17  PB V6.73 modified: CheckOptionDuplication, 
                     added: IsOutputOption, IsInputOption
                     Names of input and output options must always be 
                     followed by a digit. This allows to use i and o as 
                     starting characters in user defined control options.
2013-05-28  PB V6.74 ShowHelp: ShowReplaceKeys added
2015-10-03  PB V6.74a cosmetics
2015-10-07  PB V6.75 ShowSaxsImage added
2015-10-08  PB V6.76 saxs_libversion() and saxs_progversion() added.
                     char cb.SaxsVersion[IO_line] added.
                     REMARK: InitOptions allocates memory that is 
                             never released! Remember to add a 
                             CloseOptions function!
2015-10-13  PB V6.77 IndexOrder, BeamCenter, BeamDistance, DetTilt
                     added to ImgOptions.
2015-11-17  PB V6.78 Beam added to ImgOptions,
                     CommonOrientation.V is initialized with 0
                     and not with 1. +cori does not set the common
                     orientation to 1 but uses the one from oori or 
                     i1ori.
2017-02-06 PB V6.79  num_strncasecmp => strlib_ncasecmp
2017-02-11 PB V6.80  Command Options IODebug and IOLevel added.
                     edfio_debug_set and edfio_level_set included.
2017-03-13 PB V6.81  Command Option SearchList added.
2017-10-17 PB V6.82  FindOption: due to changes in SaxsInput:
                     explicit stop, if next argument is NULL,
                     CleanHeader (default 1): filter input header
                     for consistency, e.g. 
                     Intensity0ShutCor -> Intensity0, 
                     try adding missing Time key.
2018-02-07 PB V6.83  ShowSearchLists added
                     
2 OPTION KEYWORDS
output file <N> = 0
input file  <N> > 0
ib<N>OpenMode                         -i<N>mod  <InputOpenMode>M
ib<N>Name                             -i<N>nam  <Name>s
ib<N>Memory                           -i<N>mem  <MemNum>l
ib<N>First 	                      -i<N>fst  <First>l
ib<N>Last	                      -i<N>lst  <Last>l
ib<N>Inc	                      -i<N>inc  <Increment>l
ib<N>Const                            -i<N>con  <Const>fe
ib<N>Factor                           -i<N>fac  <Factor>fe
ib<N>Val                              -i<N>val  <Value>fe
ib<N>ErrVal                           -i<N>err  <ErrVal>fe
ib<N>Bin[1], Bin[2]                   -i<N>bin  <Bin1>le <Bin2>le
ib<N>Ave                              -i<N>ave  <Ave>l
ib<N>MinClip                          -i<N>min  <MinClip>fe
ib<N>MaxClip                          -i<N>max  <MaxClip>fe
ib<N>Gnaw[1], Gnaw[2]                 -i<N>gna  <Gnaw1>le <Gnaw2>le

ib<N>Title                            -i<N>tit  <Title>se
ib<N>Time                             -i<N>time <Time>se
ib<N>Dim[1], Dim[2]                   -i<N>dim  <Dim1>le <Dim2>le
ib<N>Dummy                            -i<N>dum  <Dummy>fe
ib<N>DDummy                           -i<N>ddum <DDummy>fe
ib<N>Offset[1], Offset[2]             -i<N>off  <Offset1>fe <Offset2>fe
ib<N>Center[1], Center[2]             -i<N>cen  <Center1>fe <Center2>fe
ib<N>BinSiz[1], BinSiz[2]             -i<N>bis  <BinSiz1>fe <BinSiz2>fe
ib<N>PixSiz[1], PixSiz[2]             -i<N>pix  <PixSiz1>fe <PixSiz2>fe
ib<N>SampleDistance                   -i<N>dis  <SampleDistance>fe
ib<N>WaveLength                       -i<N>wvl  <WaveLength>fe
ib<N>DetRot[1], DetRot[2], DetRot[3]  -i<N>rot  <DetRot1>fe <DetRot2>fe 
                                                <DetRot2>fe
ib<N>SearchList                       -i<N>h5ls <SearchList>se

output file
ib<0>OpenMode                         -omod  <OutOpenMode>M
ib<0>Name                             -onam  <OutName>s
ib<0>Memory                           -omem  <MemNum>l
ib<0>First                            -ofst  <OutFirst>l
ib<0>Last                             -olst  <OutLast>l
ib<0>Inc                              -oinc  <OutInc>l
ib<0>Const                            -ocon  <Const>fe
ib<0>Factor                           -ofac  <Factor>fe
ib<0>Val                              -oval  <Value>fe
ib<0>ErrVal                           -oerr  <ErrVal>fe
ib<0>Bin[1], Bin[2]                   -obin  <Bin1>le <Bin2>le
ib<0>Ave                              -oave  <Ave>l
ib<0>MinClip                          -omin  <MinClip>fe
ib<0>MaxClip                          -omax  <MaxClip>fe
ib<0>Gnaw[1], Gnaw[2]                 -ogna  <Gnaw1>le <Gnaw2>le

ib<0>Title                            -otit  <Title>s
ib<0>Time                             -otime <Time>s
ib<0>Dim[1], Dim[2]                   -odim  <OutDim1>le <OutDim2>le
ib<0>Dummy                            -odum  <OutDummy>fe
ib<0>DDummy                           -oddum <OutDDummy>fe
ib<0>Offset[1], Offset[2]             -ooff  <OutOffset1>fe <OutOffset2>fe
ib<0>Center[1], Center[2]             -ocen  <Center1>fe <Center2>fe
ib<0>BinSiz[1], BinSiz[2]             -obis  <OutBinSiz1>fe <OutBinSiz2>fe
ib<0>PixSiz[1], PixSiz[2]             -opix  <OutPixSiz1>fe <OutPixSiz2>fe
ib<0>SampleDistance                   -odis  <SampleDistance>fe
ib<0>WaveLength                       -owvl  <WaveLength>fe
ib<0>DetRot[1], DetRot[2], DetRot[3]  -orot  <DetRot1>fe <DetRot2>fe
                                             <DetRot2>fe

Commands
Help      extended      +h
	  simple        -h
Promptf   on            +p
          off           -p
Test      on            +test
	  off           -test
cb.Shift[1]        cb.Shift[2]        -cshft <Shift1>fe <Shift2>fe
cb.Arg[0].V			      -<nameXXX> <value>f
...
cb.Arg[IO_MaxFloatOpt-1].V	      -<nameZZZ> <value>f
cb.Num[0].V			      -<nameAAA> <value>l
...
cb.Num[IO_MaxIntOpt-1].V	      -<nameCCC> <value>l
cb.Lin[0].V			      -<nameDDD> <value>s
...
cb.Lin[IO_MaxLinOpt-1].V	      -<nameFFF> <value>s

OpenMode
n+fp+ip new file created,
	old file write protected, all images write protected
n-fp+ip new file created,
	any old file deleted, all images write protected
n+fp-ip new file created,
	old file write protected, images can be deleted
n-fp-ip new file created, any old file deleted, images can be deleted
o+fp+ip old file open, all images write protected
o+fp-ip old file open, images can be deleted
a+ip     open either old or new, all images write protected
a-ip     open either old or new, images can be deleted

2	EXAMPLE
#	include <errno.h>
#	include <stdio.h>
#	include <fcntl.h>
#	include <stddef.h>
#	include "SaxsInput.h"
#	include "SaxsOption.h"
#	define ImageNum 2
main ( int argc, char * argv[] )
{
	const char * MyHelp    = "HELP: This is an example program";
	const char * MyVersion = "This is my private version";
	int status;
	int arg_no;
	const int RejectNum = 1;
	CmdBlk cb;
	ImgBlk ib[ImageNum];
	InitOptions( MyHelp, MyVersion, RejectNum, &cb, ib, ImageNum );
	arg_no = 0;
	ReadOptions(argv, &arg_no, &cb, ib, &status);
	if (status!=Success) ReportSaxsStatus( status, 1 );
	PrintBlocks ( &cb, ib, ImageNum );
	MainProgram(&cb,ib);
}
2	CHANGE OF OPTIONS AND BLOCKS
	To change image and command blocks the following manual procedure
	is necessary:
- Change of block definition
a) Change of structure definition (CmdBlk or ImgBlk).
b) Update of initialisation procedure (InitCommandBlock or InitImageBlock).
   The .I values have to be set to FALSE.
c) Update of print block procedure (PrintCommandBlock or PrintImageBlock).
- Add options
a) Add option string to preprocessor definitions of command options or
   image option strings, proceeded with capital S, e.g. # define SMySize 10.
b) Add a new enum constant preceeded with capital N to CmdOptions or
   ImgOptions, e.g. enum { ... , NMySize, ..., IO_CmdOptLen }. Use the same
   name as in a.
c) NO MANUAL CHANGE NECESSARY V2.4 PB 28-Jan-1995.
   Last constant in enum must be IO_CmdOptLen or IO_ImgOptLen respectively.
   The option arrays are dynamically generated in InitOptions and defined as:
   OptDsc CommandOption[IO_CmdOptLen]; OptDsc ImageOption[IO_ImgOptLen].
   The pointers to the option arrays are stored in the command block.
   After leaving ReadOption the corresponding memory is released and the
   pointers are not valid any more.
d) Update the initialisation procedure InitCommandOptions or InitImageOptions.
e) Add the corresponding option number in DoCmdOptions or for image options
   in DoImgOptions.
f) After compilation the new option should appear by calling the program with
   the option +test, e.g.  TestImageOption +test -<option string> ...
---*/

/****************************************************************************/
/* SaxsOption.c                                                             */
/****************************************************************************/
# define SO_Version "SaxsOption V6.83 Peter Boesecke 2018-02-07"

#	include <errno.h>
#	include <stdio.h>
#	include <fcntl.h>
#	include <stddef.h>
#	include <float.h>
#       include <limits.h>
#       include <ctype.h>

#       include "SaxsImage.h"
#       include "SaxsOption.h"
#       include "SaxsReplacekey.h"
#       include "filename.h"

# ifndef FLT_MAX
#   define FLT_MAX pow(10,FLT_MAX_EXP)
# endif
# ifndef FLT_MIN
#   define FLT_MIN pow(10,FLT_MIN_EXP)
# endif
#ifndef ROUND
# define ROUND( x ) floor( ( x ) + 0.5 )
#endif

# define DefaultFilename "datafile"DefaultExtension

/* Fixed command option strings, small characters only! */
# define STestBit	"test"
# define SIODebug	"dbg"
# define SIOLevel	"lvl"
# define SHelp		"h"
# define SPrompt	"p"
# define SShift		"shft"
# define SRSys		"rsys"
# define SUSys		"usys"
# define SGeneralBlock  "gblk"
# define SPassHeader    "pass"
# define SCleanHeader   "cln"
# define SCreateVariance "var"
# define SAntiAliased   "aa"
# define SVarianceWeight "vw"
# define SOutputDataType "type"
# define SOutputDataValueOffset "dvo"
# define SOutputCompression "cmpr"
# define SBslInputByteOrder "bibo"
# define SIPMin         "pmin" // minimum pixel coverage
# define SAdd           "add"
# define SMaxLineWidth  "mlw"
# define SMinHeaderSize "mhs"
# define SCommonOrientation "cori" // use common orientation
# define SBeam          "beam" // +beam write all parameters
#                              // -beam only beam parameters
/* Fixed image option strings, small characters only! */
# define OutChar	'o'
# define InChar		'i'
# define SOpenMode	"mod"
# define SName		"nam"
# define SMemory        "mem"
# define SFirst		"fst"
# define SLast		"lst"
# define SInc		"inc"
# define SMinClip       "min"
# define SMaxClip       "max"
# define SConst		"con"
# define SFactor	"fac"
# define SVal    	"val"
# define SErrVal 	"err"
# define SGnaw          "gna"
# define SBin           "bin"
# define SAve           "ave"
# define STitle		"tit"
# define STime          "time"
# define SDim		"dim"
# define SDummy		"dum"
# define SDDummy	"ddum"
# define SOffset	"off"
# define SBinSiz	"bis"
# define SPixSiz	"pix"
# define SWaveLength	"wvl"
# define SOrientation   "ori"
# define SIndexOrder    "kori"
# define SProjection    "pro"
# define SAxisType      "axis"

# define SCen           "cen"
# define SSampleDistance "dis"
# define SDetRot        "rot"

# define SBeamCen       "bcen"
# define SBeamDistance  "bdis"
# define SDetTilt       "tilt"

# define SSearchList    "h5ls"

/* Reference system code */
#	define SArray   "array"
#	define SImage   "image"
#       define SCenter  "center"
#	define SRegion  "region"
#	define SReal    "real"
#       define SNormal  "normal"
#       define STangens "tangens"
#	define SSaxs    "saxs"

/* File opening code */
#	define SNewIP     "n+ip"
  /* = new file created, images write protected */
#	define SNewFP     "n+fp"
  /* = new file created, file write protected */
#	define SNew       "n"
  /* = new file created,  no protection */
#	define SOldIP     "o+ip"
  /* = old file open, all images write protected */
#	define SOldFP     "o+fp"
  /* = old file open, file write protected */
#       define SOld       "o"
  /* = old file open, no protection */
#	define SAnyIP     "a+ip"
  /* = open either old or new, images write protected */
#       define SAnyFP     "a+fp"
  /* = open either old or new, , file write protected */
#	define SAny       "a"
  /* = open either old or new, no protection */

/*----------------------------------------------------------------------------
 String constants
----------------------------------------------------------------------------*/
const char * SOName                = "SaxsOption";
const char * ErrorOptionDuplicated = "ERROR: Option duplicated or reserved: ";
const char * ErrorOptionDefnFailed = "ERROR: Option definition failed: ";
const char * ErrorCmdArrayTooShort = "ERROR: Command option array too short";
const char * ErrorImgArrayTooShort = "ERROR: Image option array too short";
const char * ErrorCannotAllcMemory = "ERROR: Cannot allocate memory";

const char * ReferenceArray        = "Array";
const char * ReferenceImage        = "Image";
const char * ReferenceCenter       = "Center";
const char * ReferenceRegion       = "Region";
const char * ReferenceReal         = "Real";
const char * ReferenceNormal       = "Normal";
const char * ReferenceTangens      = "Tangens";
const char * ReferenceSaxs         = "Saxs";
const char * ReferenceUnknown      = "<unknown>";

const char * ProjectionUnknown     = "<unknown>";
const char * AxisTypeUnknown       = "<unknown>";

/*----------------------------------------------------------------------------
1 SaxsOptionVersion

2 DESCRIPTION
  Returns a pointer to the version string.
  char * SaxsOptionVersion ( void )

2 HISTORY
  14-Oct-1996 PB
----------------------------------------------------------------------------*/
char * SaxsOptionVersion ( void )
{ static char * Version = SO_Version;
  return ( Version );
} /* SaxsOptionVersion */

/*----------------------------------------------------------------------------
 Private include files
----------------------------------------------------------------------------*/
# include "edfio.h"        /* included for edf_history and edf_showdatatypes */
# include "SaxsPrompt.inc" /* included because file otherwise too long */

/*--------------------------------------------------------------------------
NAME
  scan_for_filenumber --- extract and convert a parameter 

SYNOPSIS
  int scan_for_filenumber( IO_long *pNumber, const char * fname, int parno,
                           int opt_onoff, int * pstatus );

DESCRIPTION
  Extract parameter parno from fname, scan it as float expression and
  convert it to IO_long *pNumber. If the parameter was found and could be
  converted the .I-flag is set to opt_onoff. The function returns the number of
  scanned parameters (0 or 1). In case of success the function *pstatus
  is set to success otherwise to the corresponding error status.

HISTORY
  2001-07-08 PB
--------------------------------------------------------------------------*/
int scan_for_filenumber( IO_long *pNumber, const char * fname, int parno,
                         int opt_onoff, int * pstatus )
{ char Number[IO_size];
  int scanned=0;

  *pstatus = Success;

  if ( filename_parameter ( Number, IO_size, fname, parno ) ) {
    scanned=1;
    if (strlen(Number)>0) {
      pNumber->I = opt_onoff;
      pNumber->V = (long) ROUND( doubleexpr( Number, pstatus) );
      }
  }

  return(scanned);

} // scan_for_filenumber

/*--------------------------------------------------------------------------
NAME
   extract_filenumbers --- extract FFirst, FLast, FInc

SYNOPSIS
   int extract_filenumbers( ImgBlk *pib, IO_line *pName,
                            int opt_onoff, int * pstatus );

DESCRIPTION
   Scans p->Name.V for pib->FFirst.V, pib->FLast.V and 
   pib->FInc.V. The return value is the number of found parameters.

HISTORY
  2001-07-08 PB
--------------------------------------------------------------------------*/
int extract_filenumbers( ImgBlk *pib, IO_line *pName,
                         int opt_onoff, int * pstatus )
{ int found=0;
  int fhp=0;

  if (!pName->I) return(found);

  fhp = filename_has_pattern ( pName->V ); 

  if (fhp) { 
    // scan Name for file numbers
    found+=scan_for_filenumber( &(pib->FFirst), pName->V, 1,
                               opt_onoff, pstatus );
    if (*pstatus!=Success) return(found);

    found+=scan_for_filenumber( &(pib->FLast), pName->V, 2,
                               opt_onoff, pstatus );
    if (*pstatus!=Success) return(found);

    found+=scan_for_filenumber( &(pib->FInc), pName->V, 3,
                                opt_onoff, pstatus );
    if (*pstatus!=Success) return(found);
  } else {
    // scan Name for image numbers
    found+=scan_for_filenumber( &(pib->First), pName->V, 1,
                               opt_onoff, pstatus );
    if (*pstatus!=Success) return(found);
 
    found+=scan_for_filenumber( &(pib->Last), pName->V, 2,
                               opt_onoff, pstatus );
    if (*pstatus!=Success) return(found);
 
    found+=scan_for_filenumber( &(pib->Inc), pName->V, 3,
                                opt_onoff, pstatus );
    if (*pstatus!=Success) return(found);
  }

  return(found);

} // extract_filenumbers

/*--------------------------------------------------------------------------
NAME
   argv_filename --- prompt for filename and filenumbers of ib[blkno]

SYNOPSIS
  void argv_filename ( CmdBlk * pcb,
                       const char * promptline, ImgBlk ib[], int blkno,  
                       const char * defval, int * pstatus);
DESCRIPTION
  Reads a filename with parameters from the argument list and from the 
input file using the functions argv_line and extract_filenumbers. 
If parameters are found it modifies ib[blkno].Name, ib[blkno].FFirst, 
ib[blkno].FLast and ib[blkno].FInc. 

HISTORY
  2001-07-08 PB
--------------------------------------------------------------------------*/
void argv_filename ( CmdBlk * pcb,
                     char * promptline, ImgBlk ib[], int blkno, 
                     char * defval, int * pstatus)
{ 
  argv_line( pcb,promptline,&ib[blkno].Name,defval,pstatus);
  if (*pstatus!=Success) return;

  extract_filenumbers( &ib[blkno], &(ib[blkno].Name), TRUE, pstatus );
  if (*pstatus!=Success) return;

} // argv_filename

/*---------------------------------------------------------------------------
1 reftostr

2 PURPOSE
  Returns a pointer to a string with the name of the reference system

2 CALL
  char *reftostr(long int refsys);

2 HISTORY
  Feb-1995 Peter Boesecke creation
---------------------------------------------------------------------------*/
char *reftostr(long int refsys)
{
  char *pcref;

  switch ((int) refsys) {
    case IO_Array   : pcref = (char *) ReferenceArray; break;
    case IO_Image   : pcref = (char *) ReferenceImage; break;
    case IO_Center  : pcref = (char *) ReferenceCenter; break;
    case IO_Region  : pcref = (char *) ReferenceRegion; break;
    case IO_Real    : pcref = (char *) ReferenceReal; break;
    case IO_Normal  : pcref = (char *) ReferenceNormal; break;
    case IO_Tangens : pcref = (char *) ReferenceTangens; break;
    case IO_Saxs    : pcref = (char *) ReferenceSaxs; break;
    default         : pcref = (char *) ReferenceUnknown;
    } /* switch */

  return(pcref);
} /* reftostr */

/*---------------------------------------------------------------------------
1 protostr

2 PURPOSE
  Returns a pointer to a string with the name of the projection type

2 CALL
  char *protostr(long int protyp);

2 HISTORY
  2004-09-25 Peter Boesecke 
---------------------------------------------------------------------------*/
char *protostr(long int protyp)
{
  char *pcpro;

  switch ((int) protyp) {
    case IO_ProSaxs  : pcpro = (char *) VProjectionSaxs; break;
    case IO_ProWaxs  : pcpro = (char *) VProjectionWaxs; break;
    default          : pcpro = (char *) ProjectionUnknown;
    } /* switch */

  return(pcpro);

} /* protostr */

/*---------------------------------------------------------------------------
1 strtopro

2 PURPOSE
  Converts the input string str to a projection type 

2 CALL
  long int strtopro ( char *str );

2 HISTORY
  2004-09-25 Peter Boesecke
---------------------------------------------------------------------------*/
long int strtopro ( const char *str )
{ long int pro;

  pro  = (long int) IO_NoPro;
  if (strlib_ncasecmp( str, VProjectionSaxs, IO_size) == 0) {
    pro = (long int) IO_ProSaxs; } else
  if (strlib_ncasecmp( str, VProjectionWaxs, IO_size) == 0) {
    pro = (long int) IO_ProWaxs; }
  return( pro );

} /* strtopro */

/*---------------------------------------------------------------------------
1 axisostr

2 PURPOSE
  Returns a pointer to a string with the name of the axis type

2 CALL
  char *axistostr(long int axistyp);

2 HISTORY
  2007-12-01 Peter Boesecke
---------------------------------------------------------------------------*/
char *axistostr(long int axistyp)
{
  char *pcaxis;

  switch ((int) axistyp) {
    case IO_AxisTypeDistance  : pcaxis = (char *) VAxisTypeDistance; break;
    case IO_AxisTypeAngle     : pcaxis = (char *) VAxisTypeAngle; break;
    case IO_AxisTypeNumerator : pcaxis = (char *) VAxisTypeNumerator; break;
    default                   : pcaxis = (char *) AxisTypeUnknown;
    } /* switch */

  return(pcaxis);

} /* axistostr */

/*---------------------------------------------------------------------------
1 strtoaxis

2 PURPOSE
  Converts the input string str to an axis type 
2 CALL
  long int strtoaxis ( char *str );

2 HISTORY
  2007-12-01 Peter Boesecke
---------------------------------------------------------------------------*/
long int strtoaxis ( const char *str )
{ long int axis;

  axis  = (long int) IO_NoAxisType;
  if (strlib_ncasecmp( str, VAxisTypeDistance, IO_size) == 0) {
    axis = (long int) IO_AxisTypeDistance; } else
  if (strlib_ncasecmp( str, VAxisTypeAngle, IO_size) == 0) {
    axis = (long int) IO_AxisTypeAngle; }
  if (strlib_ncasecmp( str, VAxisTypeNumerator, IO_size) == 0) {
    axis = (long int) IO_AxisTypeNumerator; }
  return( axis );

} /* strtoaxis */


/*--------------------------------------------------------------------------
   ShowSys : Print reference systems
  --------------------------------------------------------------------------*/
void ShowSys ( void )
{
  printf("\n");
  printf("Use -%s and -%s with one of the following reference systems : \n",\
         SRSys, SUSys);
  printf(" %8s, %8s, %8s, %8s, %8s, %8s, %8s, %8s\n",
         SArray, SImage, SCenter, SRegion, SReal, SNormal, STangens, SSaxs);
} /* ShowSys */

/*--------------------------------------------------------------------------
   ShowReplaceKeys : Print ReplaceKeys
  --------------------------------------------------------------------------*/
void ShowReplaceKeys ( void )
{
  printf("\n");
  printf("The following keys in square brackets are replaced by parameters:\n");
  printf("%s\n",replacekey_show());
} /* ShowReplaceKeys */

/*--------------------------------------------------------------------------
   ShowSaxsImage : Print Info about predefined variables
  --------------------------------------------------------------------------*/
void ShowSaxsImage ( void )
{
  printf("\n");
  printf("Predefined variables in expressions:\n");
  printf("%s\n",SaxsImageShow());
} /* ShowSaxsImage */

/*--------------------------------------------------------------------------
   ShowPro : Print projection types 
  --------------------------------------------------------------------------*/
void ShowPro ( void )
{
  printf("\n");
  printf("Use -%s with one of the following projection types : \n",SProjection);
  printf(" %8s, %8s\n", VProjectionSaxs, VProjectionWaxs);
} /* ShowPro */

/*--------------------------------------------------------------------------
   ShowAxis : Print axis types
  --------------------------------------------------------------------------*/
void ShowAxis ( void )
{
  printf("\n");
  printf("Use -%s with one of the following axis types : \n",SAxisType);
  printf(" %8s, %8s, %8s\n", 
         VAxisTypeDistance, VAxisTypeAngle, VAxisTypeNumerator);
} /* ShowAxis */

/*--------------------------------------------------------------------------
   ShowModes : Print File Opening Modes
  --------------------------------------------------------------------------*/
void ShowModes ( void )
{
  printf("Use -o%s with one of the following modes : \n",SOpenMode);
  printf(" %8s, %8s, %8s\n %8s, %8s, %8s\n %8s, %8s, %8s\n",
SNewFP, SNewIP, SNew, SOldFP, SOldIP, SOld, SAnyFP, SAnyIP, SAny );
}

void ShowSearchLists( void )
{
  printf("Use -o%s with one of the following search lists: \n", SSearchList);
  printf("%s\n",edf_searchlists());
}

/*--------------------------------------------------------------------------
   ShowOptions : Show defined options of dsc[]  ( for Help )
  --------------------------------------------------------------------------*/
void ShowOptions ( char * c, OptDsc dsc[] )
{
	char line[IO_size];
	const int NperLine = 4;
	register int k;
	int cnt;
	char tp[2];

	cnt=0;
	tp[1] = '\0';
	for (k=1;(k<dsc[0].npar);k++)
	  if ( (dsc[k].code[0]) && (dsc[k].code[0]!='#') ) {
	    if ((cnt % NperLine)==0) printf(" \n ");
	    switch (dsc[k].ptyp) {
	      case IO_tpflag: tp[0]='+'; break;
	      case IO_tplong: tp[0]='d'; break;
              case IO_tplvexp: tp[0]='D'; break;
	      case IO_tpfloat: tp[0]='f'; break;
	      case IO_tpflexp: tp[0]='F'; break;
	      case IO_tpline: tp[0]='s'; break;
	      case IO_tpmode: tp[0]='M'; break;
	      case IO_tprsys: tp[0]='R'; break;
	      case IO_tppro: tp[0]='P'; break;
	      case IO_tpaxis: tp[0]='A'; break;
	      default : tp[0]='?'; break;
	      }
            printf("%10s [% 1d, %s]", \
                   strcat(strcpy(line,c),dsc[k].code),dsc[k].npar,tp);
            cnt += 1;
            }
	printf("\n");
	
} /* ShowOptions */

/*--------------------------------------------------------------------------
   ShowDataTypes : Print Conversion Table of Supported Data Types 
  --------------------------------------------------------------------------*/
void ShowDataTypes ( void )
{
  printf("\n");
  printf("Conversion of file data types to internal data types\n");
  edf_showdatatypes( False );
}

/*--------------------------------------------------------------------------
   PrintOptions : Print Option Descriptors
  --------------------------------------------------------------------------*/
void PrintOptions ( char * dname, OptDsc dsc[], int dscl )
{
	register int i;
	printf("%20s		    code      ptyp	   npar      poff\n",
                dname);
	for (i=0;(i<dscl-1);i++)
          if (dsc[i].code[0])
            printf("%20s[% 4d] %9s %9d % 9d\n",
                   dname,i, dsc[i].code,dsc[i].ptyp,dsc[i].npar);
} /* PrintOptions */

/*--------------------------------------------------------------------------
  PrintImageBlock
  --------------------------------------------------------------------------*/
void PrintImageBlock ( ImgBlk * ib )
{ int i;
  char * VName = "ImageBlock";

  i = ib->blkno;
  printf("Image Block (blkno = %d)\n",i);
  if (ib->Name.I)      printf("%s[%d].Name.V      = %s\n",
    VName,i,ib->Name.V);
  if (ib->FFirst.I)    printf("%s[%d].FFirst.V    = %ld\n",
    VName,i,ib->FFirst.V);
  if (ib->FLast.I)     printf("%s[%d].FLast.V     = %ld\n",
    VName,i,ib->FLast.V);
  if (ib->FInc.I)      printf("%s[%d].FInc.V      = %ld\n",
    VName,i,ib->FInc.V);
  if (ib->OpenMode.I)  printf("%s[%d].OpenMode.V  = %#lx\n",
    VName,i,ib->OpenMode.V);
  if (ib->Memory.I)    printf("%s[%d].Memory.V    = %ld\n",
    VName,i,ib->Memory.V);
  if (ib->First.I)     printf("%s[%d].First.V     = %ld\n",
    VName,i,ib->First.V);
  if (ib->Last.I)	     printf("%s[%d].Last.V      = %ld\n",
    VName,i,ib->Last.V);
  if (ib->Inc.I)	     printf("%s[%d].Inc.V       = %ld\n",
    VName,i,ib->Inc.V);
  if (ib->MinClip.I)   printf("%s[%d].MinClip.V   = %s\n",
    VName,i,ib->MinClip.V);
  if (ib->MaxClip.I)   printf("%s[%d].MaxClip.V   = %s\n",
    VName,i,ib->MaxClip.V);
  if (ib->Const.I)     printf("%s[%d].Const.V     = %s\n",
    VName,i,ib->Const.V);
  if (ib->Factor.I)    printf("%s[%d] Factor.V    = %s\n",
    VName,i,ib->Factor.V);
  if (ib->Val.I)       printf("%s[%d] Val.V       = %s\n",
    VName,i,ib->Val.V);
  if (ib->ErrVal.I)    printf("%s[%d] ErrVal.V    = %s\n",
    VName,i,ib->ErrVal.V);
  if (ib->Bin[1].I)    printf("%s[%d].Bin[1].V    = %s\n",
    VName,i,ib->Bin[1].V);
  if (ib->Bin[2].I)    printf("%s[%d].Bin[2].V    = %s\n",
    VName,i,ib->Bin[2].V);
  if (ib->Ave.I)       printf("%s[%d].Ave.V       = %ld\n",
    VName,i,ib->Ave.V);
  if (ib->Gnaw[1].I)   printf("%s[%d].Gnaw[1].V   = %s\n",
    VName,i,ib->Gnaw[1].V);
  if (ib->Gnaw[2].I)   printf("%s[%d].Gnaw[2].V   = %s\n",
    VName,i,ib->Gnaw[2].V);
  if (ib->Title.I)     printf("%s[%d].Title.V     = %s\n",
    VName,i,ib->Title.V);
  if (ib->Time.I)      printf("%s[%d].Time.V      = %s\n",
    VName,i,ib->Time.V);
  if (ib->Dim[1].I)    printf("%s[%d].Dim[1].V    = %s\n",
    VName,i,ib->Dim[1].V);
  if (ib->Dim[2].I)    printf("%s[%d].Dim[2].V    = %s\n",
    VName,i,ib->Dim[2].V);
  if (ib->Dummy.I)     printf("%s[%d].Dummy.V     = %s\n",
    VName,i,ib->Dummy.V);
  if (ib->DDummy.I)    printf("%s[%d].DDummy.V    = %s\n",
    VName,i,ib->DDummy.V);
  if (ib->Offset[1].I) printf("%s[%d].Offset[1].V = %s\n",
    VName,i,ib->Offset[1].V);
  if (ib->Offset[2].I) printf("%s[%d].Offset[2].V = %s\n",
    VName,i,ib->Offset[2].V);
  if (ib->BinSiz[1].I) printf("%s[%d].BinSiz[1].V = %s\n",
    VName,i,ib->BinSiz[1].V);
  if (ib->BinSiz[2].I) printf("%s[%d].BinSiz[2].V = %s\n",
    VName,i,ib->BinSiz[2].V);
  if (ib->PixSiz[1].I) printf("%s[%d].PixSiz[1].V = %s\n",
    VName,i,ib->PixSiz[1].V);
  if (ib->PixSiz[2].I) printf("%s[%d].PixSiz[2].V = %s\n",
    VName,i,ib->PixSiz[2].V);
  if (ib->WaveLength.I) printf("%s[%d].WaveLength.V = %s\n",
    VName,i,ib->WaveLength.V);
  if (ib->Orientation.I)    printf("%s[%d].Orientation.V    = %s\n",
    VName,i,ib->Orientation.V);
  if (ib->IndexOrder.I)    printf("%s[%d].IndexOrder.V      = %s\n",
    VName,i,ib->IndexOrder.V);
  if (ib->Projection.I)     printf("%s[%d].Projection.V     = %ld\n",
    VName,i,ib->Projection.V);
  if (ib->AxisType[1].I)    printf("%s[%d].AxisType[1].V    = %ld\n",
    VName,i,ib->AxisType[1].V);
  if (ib->AxisType[2].I)    printf("%s[%d].AxisType[2].V    = %ld\n",
    VName,i,ib->AxisType[2].V);

  if (ib->Center[1].I) printf("%s[%d].Center[1].V = %s\n",
    VName,i,ib->Center[1].V);
  if (ib->Center[2].I) printf("%s[%d].Center[2].V = %s\n",
    VName,i,ib->Center[2].V);
  if (ib->SampleDistance.I) printf("%s[%d].SampleDistance.V = %s\n",
    VName,i,ib->SampleDistance.V);
  if (ib->DetRot[1].I) printf("%s[%d].DetRot[1].V = %s\n",
    VName,i,ib->DetRot[1].V);
  if (ib->DetRot[2].I) printf("%s[%d].DetRot[2].V = %s\n",
    VName,i,ib->DetRot[2].V);
  if (ib->DetRot[3].I) printf("%s[%d].DetRot[3].V = %s\n",
    VName,i,ib->DetRot[3].V);

  if (ib->BeamCenter[1].I) printf("%s[%d].BeamCenter[1].V = %s\n",
    VName,i,ib->BeamCenter[1].V);
  if (ib->BeamCenter[2].I) printf("%s[%d].BeamCenter[2].V = %s\n",
    VName,i,ib->BeamCenter[2].V);
  if (ib->BeamDistance.I) printf("%s[%d].BeamDistance.V = %s\n",
    VName,i,ib->BeamDistance.V);
  if (ib->DetTilt[1].I) printf("%s[%d].DetTilt[1].V = %s\n",
    VName,i,ib->DetTilt[1].V);
  if (ib->DetTilt[2].I) printf("%s[%d].DetTilt[2].V = %s\n",
    VName,i,ib->DetTilt[2].V);
  if (ib->DetTilt[3].I) printf("%s[%d].DetTilt[3].V = %s\n",
    VName,i,ib->DetTilt[3].V);

  if (ib->SearchList.I) printf("%s[%d].SearchList.V = %s\n",
    VName,i,ib->SearchList.V);

} /* PrintImageBlock */

/*-------------------------------------------------------------------------
   PrintCommandBlock
  -------------------------------------------------------------------------*/
void PrintCommandBlock ( CmdBlk * cb )
{
int i;
char * VName = "CommandBlock";

if (cb->Prompt.I)    printf("%s.Prompt.V                = %ld\n",
  VName,cb->Prompt.V);
if (cb->IODebug.I)   printf("%s.IODebug.V               = 0x%x\n",
  VName,(unsigned) cb->IODebug.V);
if (cb->IOLevel.I)   printf("%s.IOLevel.V               = %ld\n", 
  VName,cb->IOLevel.V);
if (cb->Help.I)	     printf("%s.Help.V                  = %ld\n",
  VName,cb->Help.V);
if (cb->Shift[1].I)  printf("%s.Shift[1].V              = %s\n",
  VName,cb->Shift[1].V);
if (cb->Shift[2].I)  printf("%s.Shift[2].V              = %s\n",
  VName,cb->Shift[2].V);
if (cb->RSys.I)      printf("%s.RSys.V                  = %ld\n",
  VName,cb->RSys.V);
if (cb->USys.I)      printf("%s.USys.V                  = %ld\n",
  VName,cb->USys.V);
if (cb->GeneralBlock.I)
                     printf("%s.GeneralBlock.V          = %ld\n",
  VName,cb->GeneralBlock.V);
if (cb->PassHeader.I)
                     printf("%s.PassHeader.V            = %ld\n",
  VName,cb->PassHeader.V);
if (cb->CleanHeader.I)
                     printf("%s.CleanHeader.V           = %ld\n",
  VName,cb->CleanHeader.V);
if (cb->CreateVariance.I)
                     printf("%s.CreateVariance.V        = %ld\n",
  VName,cb->CreateVariance.V);
if (cb->AntiAliased.I)
                     printf("%s.AntiAliased.V           = %ld\n",
  VName,cb->AntiAliased.V);
if (cb->VarianceWeight.I)
                     printf("%s.VarianceWeight.V        = %ld\n",
  VName,cb->VarianceWeight.V);
if (cb->OutputDataType.I)
                     printf("%s.OutputDataType.V        = %s\n",
  VName,edf_datatype2string(cb->OutputDataType.V));
if (cb->OutputDataValueOffset.I)
                     printf("%s.OutputDataValueOffset.V = %ld\n",
  VName,cb->OutputDataValueOffset.V);
if (cb->OutputCompression.I)
                     printf("%s.OutputCompression.V     = %s\n",
  VName,edf_compression2string(cb->OutputCompression.V));
if (cb->BslInputByteOrder.I)
                     printf("%s.BslInputByteOrder.V     = %ld\n",
  VName,cb->BslInputByteOrder.V);
if (cb->MaxLineWidth.I)
                     printf("%s.MaxLineWidth.V          = %ld\n",
  VName,cb->MaxLineWidth.V);
if (cb->MinHeaderSize.I)
                     printf("%s.MinHeaderSize.V         = %ld\n",
  VName,cb->MinHeaderSize.V);
if (cb->CommonOrientation.I)    
                     printf("%s.CommonOrientation.V     = %ld\n",
  VName,cb->CommonOrientation.V);
if (cb->Beam.I)    
                     printf("%s.Beam.V                  = %ld\n",
  VName,cb->Beam.V);
if (cb->Add.I)
                     printf("%s.Add.V                   = %ld\n",
  VName,cb->Add.V);
if (cb->IPMin.I)
                     printf("%s.IPMin.V                 = %g\n",
  VName,cb->IPMin.V);

for (i=0;i<IO_MaxFloatOpt;i++)
  if (cb->Arg[i].I)  printf("%s.Arg[%d].V	         = %e\n",
    VName,i,cb->Arg[i].V);
for (i=0;i<IO_MaxFlexpOpt;i++)
  if (cb->Flexp[i].I)  printf("%s.Flexp[%d].V          = %s\n",
    VName,i,cb->Flexp[i].V);
for (i=0;i<IO_MaxIntOpt;i++)
  if (cb->Num[i].I)  printf("%s.Num[%d].V	         = %ld\n",
    VName,i,cb->Num[i].V);
for (i=0;i<IO_MaxLvexpOpt;i++)
  if (cb->Lvexp[i].I)  printf("%s.Lvexp[%d].V          = %s\n",
    VName,i,cb->Lvexp[i].V);
for (i=0;i<IO_MaxLineOpt;i++)
  if (cb->Lin[i].I)  printf("%s.Lin[%d].V	         = %s\n",
    VName,i,cb->Lin[i].V);
for (i=0;i<IO_MaxFlagOpt;i++)
  if (cb->Flg[i].I)  printf("%s.Flg[%d].V              = %ld\n",
    VName,i,cb->Flg[i].V);
} /* PrintCommandBlock */

/*--------------------------------------------------------------------------
   PrintBlocks
  --------------------------------------------------------------------------*/
void PrintBlocks ( CmdBlk * pcb, ImgBlk ib[] )
{
	int ib_len;
	register int i;
	ib_len = pcb->ImgBlkLen;
	PrintCommandBlock( pcb );
	for (i=0;i<ib_len;i++) PrintImageBlock(&ib[i]);
} /* PrintBlocks */

/*+++------------------------------------------------------------------------
NAME

   saxs_libversion

SYNOPSIS

   const char *saxs_libversion ( void )

DESCRIPTION
   Returns a pointer to VVERSION, if defined, otherwise an empty string 

RETURN VALUE
   Pointer to the VVERSION string or an empty string
-------------------------------------------------------------------------+*/
const char *saxs_libversion   ( void )                               /*---*/
{
# ifdef VVERSION
  return ( VVERSION );
# else
  return ( "" );
# endif
} /* saxs_libversion */

/*+++------------------------------------------------------------------------
NAME

   saxs_progversion

SYNOPSIS

   const char *saxs_progversion ( void )

DESCRIPTION
   Returns a pointer to the version string of the main program.

RETURN VALUE
   Pointer to the version string of the main program or an empty string
-------------------------------------------------------------------------+*/
const char *saxs_progversion   ( CmdBlk * pcb )                      /*---*/
{ if (pcb&&pcb->MainVersion) {
    return( pcb->MainVersion );
  } else {
    return("");
  }
} /* saxs_progversion */

/*+++------------------------------------------------------------------------
NAME

   saxs_version

SYNOPSIS

   const char *saxs_version ( CmdBlk * pcb );

DESCRIPTION
   Returns a pointer to the general version string

RETURN VALUE
   Pointer to the general version string. 
-------------------------------------------------------------------------+*/
const char *saxs_version   ( CmdBlk * pcb )                      /*---*/
{ if (pcb) {
    return( (const char *) (pcb->SaxsVersion) );
  } else {
    return("");
  }
} /* saxs_version */

/*-------------------------------------------------------------------------
   ShowHelp
  -------------------------------------------------------------------------*/
/* The next modules are only included for display of version strings */
# include "SaxsImage.h"
# include "SaxsRoutine.h"
# include "edfio.h"
# include "numio.h"
void ShowHelp ( CmdBlk * pcb  )
{
  char c[8];
  printf("\nMain    : %s\n",pcb->MainVersion);
  printf(  "Usage   : %s %s\n",
				pcb->MainName, pcb->MainHelp);
  if (pcb->Help.V) {
# ifdef VVERSION
#   define _SD_VERSION SD_Version" ("VVERSION")"
# else
#   define _SD_VERSION SD_Version
# endif
    printf("\n%s\n",_SD_VERSION);
    printf(  "%s\n",REFERENCE_VERSION);
    printf(  "%s\n",SaxsErrorVersion());
    printf(  "%s\n",SaxsInputVersion());
    printf(  "%s\n",SaxsOptionVersion());
    printf(  "%s\n",SaxsImageVersion());
    printf(  "%s\n",SaxsRoutineVersion());
    printf(  "%s\n",numio_version());
    printf(  "%s\n",edfio_version());
    printf(  "E-mail : boesecke@esrf.fr\n");
    printf(  "\nDebug options (-dbg <bitmask>)\n%s\n",edfio_usage2str());

    ShowDataTypes();

    printf("\nList of options [<num of parameters>,<type>]\n");
    ShowOptions ( "-", pcb->co ); /* CommandOption */
    if (pcb->ImgBlkLen>0) {
      c[0] = '-'; c[1] = OutChar; c[2] = '\0';
        ShowOptions ( c, pcb->io ); /* ImageOption */
    }
    if (pcb->ImgBlkLen>1) {
      c[1] = InChar; strcat(c,"<n>");
      ShowOptions ( c, pcb->io );  /* ImageOption */
    }
    printf("\nNumber of image blocks            : %d\n",pcb->ImgBlkLen);
    if (pcb->ImgBlkLen>1)
    printf  ("Maximum number <n> of input blocks: %d\n",pcb->ImgBlkLen-1);
    if (pcb->ImgBlkLen>0) {
      printf  ("Dummy file name                   : %s\n",DUMMYFILENAME);
    }
    printf("\n");
    if (pcb->ImgBlkLen>0) ShowModes();
    if (pcb->ImgBlkLen>0) ShowSearchLists();
    ShowSys(); 
    ShowSaxsImage();
    ShowReplaceKeys();

  }
}
/*--------------------------------------------------------------------------
   InitCommandOptions : Initialisation of command options
  
   co[] (o) : command option array with length cl
   cl   (i) : lenght of array
  --------------------------------------------------------------------------*/
void InitCommandOptions( OptDsc co[], int cl )
{
	int nct;
	if (cl<IO_CmdOptLen)
	{printf("%s\n",ErrorCmdArrayTooShort); exit(0);}
	nct = NTestBit;
	CopyLine(STestBit,co[nct].code,IO_CodeLen,-1);
	co[nct].ptyp = IO_tpflag;
	co[nct].npar = 0;

	nct = NIODebug;
	CopyLine(SIODebug,co[nct].code,IO_CodeLen,-1); co[nct].ptyp = IO_tplong;
	co[nct].npar = 1;

	nct = NIOLevel;
	CopyLine(SIOLevel,co[nct].code,IO_CodeLen,-1); co[nct].ptyp = IO_tplong;
	co[nct].npar = 1;

	nct = NHelp;
	CopyLine(SHelp,co[nct].code,IO_CodeLen,-1); co[nct].ptyp = IO_tpflag;
	co[nct].npar = 0;

	nct = NPrompt;
	CopyLine(SPrompt,co[nct].code,IO_CodeLen,-1); co[nct].ptyp = IO_tpflag;
	co[nct].npar = 0;

	nct = NShift;
	CopyLine(SShift,co[nct].code,IO_CodeLen,-1); co[nct].ptyp = IO_tpflexp;
	co[nct].npar = 2;

	nct = NRSys;
	CopyLine(SRSys,co[nct].code,IO_CodeLen,-1); co[nct].ptyp = IO_tprsys;
	co[nct].npar = 1;

	nct = NUSys;
	CopyLine(SUSys,co[nct].code,IO_CodeLen,-1); co[nct].ptyp = IO_tprsys;
	co[nct].npar = 1;

        nct = NGeneralBlock;
        CopyLine(SGeneralBlock,co[nct].code,IO_CodeLen,-1); 
                                                    co[nct].ptyp = IO_tpflag;
        co[nct].npar = 0;

        nct = NPassHeader;
        CopyLine(SPassHeader,co[nct].code,IO_CodeLen,-1);
                                                    co[nct].ptyp = IO_tpflag;
        co[nct].npar = 0;

        nct = NCleanHeader;
        CopyLine(SCleanHeader,co[nct].code,IO_CodeLen,-1);
                                                    co[nct].ptyp = IO_tpflag;
        co[nct].npar = 0;

        nct = NCreateVariance;
        CopyLine(SCreateVariance,co[nct].code,IO_CodeLen,-1);
                                                    co[nct].ptyp = IO_tpflag;
        co[nct].npar = 0;

        nct = NAntiAliased;
        CopyLine(SAntiAliased,co[nct].code,IO_CodeLen,-1);
                                                    co[nct].ptyp = IO_tpflag;
        co[nct].npar = 0;

        nct = NVarianceWeight;
        CopyLine(SVarianceWeight,co[nct].code,IO_CodeLen,-1);
                                                    co[nct].ptyp = IO_tpflag;
        co[nct].npar = 0;

        nct = NOutputDataType;
        CopyLine(SOutputDataType,co[nct].code,IO_CodeLen,-1);
                                                    co[nct].ptyp = IO_tplong;
        co[nct].npar = 1; 

        nct = NOutputDataValueOffset;
        CopyLine(SOutputDataValueOffset,co[nct].code,IO_CodeLen,-1);
                                                    co[nct].ptyp = IO_tplong;
        co[nct].npar = 1;

        nct = NOutputCompression;
        CopyLine(SOutputCompression,co[nct].code,IO_CodeLen,-1); 
                                                    co[nct].ptyp = IO_tplong;
        co[nct].npar = 1;

        nct = NBslInputByteOrder;
        CopyLine(SBslInputByteOrder,co[nct].code,IO_CodeLen,-1);
                                                    co[nct].ptyp = IO_tplong;
        co[nct].npar = 1;

        nct = NAdd;
        CopyLine(SAdd,co[nct].code,IO_CodeLen,-1);
                                                    co[nct].ptyp = IO_tplong;
        co[nct].npar = 1;

        nct = NIPMin;
        CopyLine(SIPMin,co[nct].code,IO_CodeLen,-1);
                                                    co[nct].ptyp = IO_tpfloat;
        co[nct].npar = 1;

        nct = NMaxLineWidth;
        CopyLine(SMaxLineWidth,co[nct].code,IO_CodeLen,-1);
                                                    co[nct].ptyp = IO_tplong;
        co[nct].npar = 1;

        nct = NMinHeaderSize;
        CopyLine(SMinHeaderSize,co[nct].code,IO_CodeLen,-1);
                                                    co[nct].ptyp = IO_tplong;
        co[nct].npar = 1;

        nct = NCommonOrientation;
	CopyLine(SCommonOrientation,co[nct].code,IO_CodeLen,-1); 
                                                    co[nct].ptyp = IO_tpflag;
	co[nct].npar = 0;

        nct = NBeam;
	CopyLine(SBeam,co[nct].code,IO_CodeLen,-1); 
                                                    co[nct].ptyp = IO_tpflag;
	co[nct].npar = 0;

	for (nct=NArgFst;nct<=NArgLst;nct++) {	
	  co[nct].code[0] = '\0'; /*must be defined with DefFloatOpt*/
	  co[nct].ptyp    = IO_tpfloat;
	  co[nct].npar    = 1;
	  }

        for (nct=NFlexpFst;nct<=NFlexpLst;nct++) {
          co[nct].code[0] = '\0'; /*must be defined with option_define*/
          co[nct].ptyp    = IO_tpflexp;
          co[nct].npar    = 1;
          }

	for (nct=NNumFst;nct<=NNumLst;nct++) {	
	  co[nct].code[0] = '\0'; /*must be defined with DefIntOpt*/
	  co[nct].ptyp    = IO_tplong;
	  co[nct].npar    = 1;
	  }

        for (nct=NLvexpFst;nct<=NLvexpLst;nct++) {
          co[nct].code[0] = '\0'; /*must be defined with option_define*/
          co[nct].ptyp    = IO_tplvexp;
          co[nct].npar    = 1;
          }

	for (nct=NLinFst;nct<=NLinLst;nct++) {	
	  co[nct].code[0] = '\0'; /*must be defined with DefLineOpt*/
	  co[nct].ptyp    = IO_tpline;
	  co[nct].npar    = 1;
	  }

        for (nct=NFlgFst;nct<=NFlgLst;nct++) {
          co[nct].code[0] = '\0'; /*must be defined with DefFlgOpt*/
          co[nct].ptyp    = IO_tpflag;
          co[nct].npar    = 0;
          }

 /* stop with NULL in last field */
	nct = NCmdLast;
	co[nct].code[0] = '\0';
 /* write length to field 0 */
	CopyLine("Cmd",co[0].code,IO_CodeLen,0); co[0].ptyp = IO_tpflag;
	co[0].npar = nct;

} /* InitCommandOptions */
/*-------------------------------------------------------------------------
   InitImageOptions : Initialisation of image options
  -------------------------------------------------------------------------*/
void InitImageOptions( OptDsc io[], int il )
{
	int nct;
	if (il<IO_ImgOptLen)
          {printf("%s\n",ErrorImgArrayTooShort); exit(0);}

	nct = NName;
	CopyLine(SName,io[nct].code,IO_CodeLen,-1);
	io[nct].ptyp = IO_tpline;
	io[nct].npar = 1;

        nct = NMemory;
        CopyLine(SMemory,io[nct].code,IO_CodeLen,-1); io[nct].ptyp = IO_tplong;
        io[nct].npar = 1;

	nct = NFirst;
	CopyLine(SFirst,io[nct].code,IO_CodeLen,-1); io[nct].ptyp = IO_tplong;
	io[nct].npar = 1;

	nct = NLast;
	CopyLine(SLast,io[nct].code,IO_CodeLen,-1); io[nct].ptyp = IO_tplong;
	io[nct].npar = 1;

	nct = NInc;
	CopyLine(SInc,io[nct].code,IO_CodeLen,-1); io[nct].ptyp = IO_tplong;
	io[nct].npar = 1;

	nct = NTitle;
	CopyLine(STitle,io[nct].code,IO_CodeLen,-1);
	io[nct].ptyp = IO_tpline;
	io[nct].npar = 1;

        nct = NTime;
        CopyLine(STime,io[nct].code,IO_CodeLen,-1);
        io[nct].ptyp = IO_tpline;
        io[nct].npar = 1;

	nct = NDim;
	CopyLine(SDim,io[nct].code,IO_CodeLen,-1); io[nct].ptyp = IO_tplvexp;
	io[nct].npar = 2;

	nct = NDummy;
	CopyLine(SDummy,io[nct].code,IO_CodeLen,-1); io[nct].ptyp = IO_tpflexp;
	io[nct].npar = 1;

	nct = NDDummy;
	CopyLine(SDDummy,io[nct].code,IO_CodeLen,-1); io[nct].ptyp = IO_tpflexp;
	io[nct].npar = 1;

	nct = NOffset;
	CopyLine(SOffset,io[nct].code,IO_CodeLen,-1); io[nct].ptyp = IO_tpflexp;
	io[nct].npar = 2;

	nct = NBinSiz;
	CopyLine(SBinSiz,io[nct].code,IO_CodeLen,-1); io[nct].ptyp = IO_tpflexp;
	io[nct].npar = 2;

	nct = NPixSiz;
	CopyLine(SPixSiz,io[nct].code,IO_CodeLen,-1); io[nct].ptyp = IO_tpflexp;
	io[nct].npar = 2;

	nct = NWaveLength;
	CopyLine(SWaveLength,io[nct].code,IO_CodeLen,-1);
	io[nct].ptyp = IO_tpflexp;
	io[nct].npar = 1;

        nct = NOrientation;
        CopyLine(SOrientation,io[nct].code,IO_CodeLen,-1); 
        io[nct].ptyp = IO_tplvexp;
        io[nct].npar = 1;

        nct = NIndexOrder;
        CopyLine(SIndexOrder,io[nct].code,IO_CodeLen,-1);
        io[nct].ptyp = IO_tplvexp;
        io[nct].npar = 1;

        nct = NProjection;
        CopyLine(SProjection,io[nct].code,IO_CodeLen,-1);
        io[nct].ptyp = IO_tppro;
        io[nct].npar = 1;

        nct = NAxisType;
        CopyLine(SAxisType,io[nct].code,IO_CodeLen,-1);
        io[nct].ptyp = IO_tpaxis;
        io[nct].npar = 2;

        nct = NCenter;
        CopyLine(SCen,io[nct].code,IO_CodeLen,-1); io[nct].ptyp = IO_tpflexp;
        io[nct].npar = 2;

        nct = NSampleDistance;
        CopyLine(SSampleDistance,io[nct].code,IO_CodeLen,-1);
        io[nct].ptyp = IO_tpflexp;
        io[nct].npar = 1;

        nct = NDetRot;
        CopyLine(SDetRot,io[nct].code,IO_CodeLen,-1); io[nct].ptyp = IO_tpflexp;
        io[nct].npar = 3;

        nct = NBeamCenter;
        CopyLine(SBeamCen,io[nct].code,IO_CodeLen,-1); io[nct].ptyp = IO_tpflexp;
        io[nct].npar = 2;

        nct = NBeamDistance;
        CopyLine(SBeamDistance,io[nct].code,IO_CodeLen,-1);
        io[nct].ptyp = IO_tpflexp;
        io[nct].npar = 1;

        nct = NDetTilt;
        CopyLine(SDetTilt,io[nct].code,IO_CodeLen,-1); io[nct].ptyp = IO_tpflexp;
        io[nct].npar = 3;

	nct = NSearchList;
	CopyLine(SSearchList,io[nct].code,IO_CodeLen,-1);
	io[nct].ptyp = IO_tpline;
	io[nct].npar = 1;

	nct = NOpenMode;
	CopyLine(SOpenMode,io[nct].code,IO_CodeLen,-1);io[nct].ptyp = IO_tpmode;
	io[nct].npar = 1;

        nct = NMinClip;
        CopyLine(SMinClip,io[nct].code,IO_CodeLen,-1);io[nct].ptyp = IO_tpflexp;
        io[nct].npar = 1;

        nct = NMaxClip;
        CopyLine(SMaxClip,io[nct].code,IO_CodeLen,-1);io[nct].ptyp = IO_tpflexp;
        io[nct].npar = 1;

	nct = NConst;
	CopyLine(SConst,io[nct].code,IO_CodeLen,-1); io[nct].ptyp = IO_tpflexp;
	io[nct].npar = 1;

	nct = NFactor;
	CopyLine(SFactor,io[nct].code,IO_CodeLen,-1); io[nct].ptyp = IO_tpflexp;
	io[nct].npar = 1;

	nct = NVal;
	CopyLine(SVal,io[nct].code,IO_CodeLen,-1); io[nct].ptyp = IO_tpflexp;
	io[nct].npar = 1;

	nct = NErrVal;
	CopyLine(SErrVal,io[nct].code,IO_CodeLen,-1); io[nct].ptyp = IO_tpflexp;
	io[nct].npar = 1;

	nct = NBin;
	CopyLine(SBin,io[nct].code,IO_CodeLen,-1); io[nct].ptyp = IO_tplvexp;
	io[nct].npar = 2;

        nct = NAve;
        CopyLine(SAve,io[nct].code,IO_CodeLen,-1); io[nct].ptyp = IO_tpflag;
        io[nct].npar = 0;

        nct = NGnaw;
        CopyLine(SGnaw,io[nct].code,IO_CodeLen,-1); io[nct].ptyp = IO_tplvexp;
        io[nct].npar = 2;

 /* stop with NULL in last field */
	nct = NImgLast;
	io[nct].code[0] = '\0';

/* write length to field 0 */
	CopyLine("Cmd",io[0].code,IO_CodeLen,0); io[0].ptyp = IO_tpflag;
	io[0].npar = nct;
} /* InitImageOptions */

/*---------------------------------------------------------------------------
   InitCommandBlock : Initialisation of command block
  	   Sets init values of *pcb to FALSE
           And initializes default values
  ---------------------------------------------------------------------------*/
void InitCommandBlock ( const char * mhlp, const char * mvers, void * mdat,
			int rjctnum, ImgBlk ib[], int ib_len, CmdBlk * pcb )
{
	int i;

	pcb->MainName		= NULL;    /* set by ReadOptions, if NULL */
	pcb->MainVersion	= mvers;
	pcb->MainHelp		= mhlp;
	pcb->MainData		= mdat;    /* optional pointer to variables
                                              managed by the by main program, 
                                              must be allocated and released
                                              by the main programs */
        pcb->SaxsVersion[0]     = '\0';    /* terminating zero for string */
	pcb->argc		= 0;       /* set by ReadOptions, if 0    */
	pcb->argv		= NULL;    /* set by ReadOptions, if NULL */
	pcb->parg_no		= NULL;    /* set by ReadOptions, if NULL */
	pcb->infile		= stdin;
	pcb->RejectNum		= rjctnum;
        pcb->Cycles             = 0l; // contains number of cycles in saxs_image_loop
	pcb->ib			= ib;
	pcb->ImgBlkLen		= ib_len;

	pcb->co			= (OptDsc *) NULL;
	pcb->io			= (OptDsc *) NULL;
	pcb->TestBit		= FALSE;
	pcb->Prompt.I		= FALSE;
	pcb->IODebug.I          = FALSE;
	pcb->IOLevel.I          = FALSE;
	pcb->Help.I		= FALSE;
	pcb->Shift[0].I         = FALSE;
	pcb->Shift[1].I 	= FALSE;
	pcb->Shift[2].I 	= FALSE;
	pcb->RSys.I		= FALSE;
	pcb->USys.I		= FALSE;
        pcb->GeneralBlock.I     = FALSE; 
        pcb->PassHeader.I       = FALSE;
        pcb->CleanHeader.I      = FALSE;
        pcb->CreateVariance.I   = FALSE;
        pcb->AntiAliased.I      = FALSE;
        pcb->VarianceWeight.I   = FALSE;
        pcb->OutputDataType.I   = FALSE;
        pcb->OutputDataValueOffset.I = FALSE;
	pcb->OutputCompression.I= FALSE;
        pcb->BslInputByteOrder.I= FALSE;
        pcb->MaxLineWidth.I     = FALSE;
        pcb->MinHeaderSize.I    = FALSE;
	pcb->CommonOrientation.I= FALSE;
	pcb->Beam.I             = FALSE;
        pcb->Add.I              = FALSE;
        pcb->IPMin.I            = FALSE;
	for (i=0;i<IO_MaxFloatOpt;i++) pcb->Arg[i].I = FALSE;
	for (i=0;i<IO_MaxFlexpOpt;i++) pcb->Flexp[i].I = FALSE;
	for (i=0;i<IO_MaxIntOpt;i++)   pcb->Num[i].I = FALSE;
	for (i=0;i<IO_MaxLvexpOpt;i++) pcb->Lvexp[i].I = FALSE;
	for (i=0;i<IO_MaxLineOpt;i++)  pcb->Lin[i].I = FALSE;
	for (i=0;i<IO_MaxFlagOpt;i++)  pcb->Flg[i].I = FALSE;

/* Initialization of default values */

	pcb->Prompt.V		= TRUE;
	pcb->IODebug.V          = 0;
	pcb->IOLevel.V          = (long) INT_MAX; // used as int
	pcb->Help.V		= FALSE;
	CopyLine("2.0",pcb->Shift[0].V,IO_size,0);
	CopyLine("0.0",pcb->Shift[1].V,IO_size,0);
	CopyLine("0.0",pcb->Shift[2].V,IO_size,0);
	pcb->RSys.V		= IO_Image;
	pcb->USys.V		= pcb->RSys.V;
        pcb->GeneralBlock.V     = 0l;
        pcb->PassHeader.V       = 0l;
        pcb->CleanHeader.V      = 1l;
        pcb->CreateVariance.V   = 0l;
        pcb->AntiAliased.V      = 0l;
        pcb->VarianceWeight.V   = 0l;
        pcb->OutputDataType.V   = 0l;
        pcb->OutputDataValueOffset.V = 0l;
	pcb->OutputCompression.V= UnCompressed;
        pcb->BslInputByteOrder.V= 0l;
        pcb->MaxLineWidth.V     = 0l;
        pcb->MinHeaderSize.V    = 0l;
	pcb->CommonOrientation.V= 0l;
	pcb->Beam.V             = 0l;
        pcb->Add.V              = 1l;
        pcb->IPMin.V            = Ipolmin(-1); // get the current value
	for (i=0;i<IO_MaxFloatOpt;i++) pcb->Arg[i].V = 0.0;
	for (i=0;i<IO_MaxFlexpOpt;i++) pcb->Flexp[i].V[0] = '\0';
	for (i=0;i<IO_MaxIntOpt;i++)   pcb->Num[i].V = 0l;
	for (i=0;i<IO_MaxLvexpOpt;i++) pcb->Lvexp[i].V[0] = '\0';
	for (i=0;i<IO_MaxLineOpt;i++)  pcb->Lin[i].V[0] = '\0';
	for (i=0;i<IO_MaxFlagOpt;i++)  pcb->Flg[i].V = 0l;

} /* InitCommandBlock */

/*--------------------------------------------------------------------------
   InitImageBlock: Initialisation of image block with blkno
   Sets init values of *pib to FALSE
  --------------------------------------------------------------------------*/
void InitImageBlock ( ImgBlk ib[], int blkno )
{ 
        ImgBlk * pib;

        pib = &(ib[blkno]);

        pib->blkno              = blkno;
	pib->Name.I		= FALSE;
        pib->FFirst.I           = FALSE;
        pib->FLast.I            = FALSE;
        pib->FInc.I             = FALSE;
	pib->OpenMode.I		= FALSE;
        pib->Memory.I           = FALSE;
	pib->First.I		= FALSE;
	pib->Last.I		= FALSE;
	pib->Inc.I		= FALSE;
        pib->MinClip.I          = FALSE;
        pib->MaxClip.I          = FALSE;
	pib->Const.I		= FALSE;
	pib->Factor.I		= FALSE;
	pib->Val.I		= FALSE;
	pib->ErrVal.I		= FALSE;
	pib->Bin[0].I		= FALSE;
	pib->Bin[1].I		= FALSE;
	pib->Bin[2].I		= FALSE;
	pib->Ave.I		= FALSE;
	pib->Gnaw[0].I		= FALSE;
        pib->Gnaw[1].I          = FALSE;
        pib->Gnaw[2].I          = FALSE;
	pib->Title.I		= FALSE;
        pib->Time.I             = FALSE;
	pib->Dim[0].I           = FALSE;
	pib->Dim[1].I		= FALSE;
	pib->Dim[2].I		= FALSE;
	pib->Dummy.I		= FALSE;
	pib->DDummy.I		= FALSE;
	pib->Offset[0].I        = FALSE;
	pib->Offset[1].I 	= FALSE;
	pib->Offset[2].I 	= FALSE;
	pib->BinSiz[0].I        = FALSE;
	pib->BinSiz[1].I 	= FALSE;
	pib->BinSiz[2].I 	= FALSE;
	pib->PixSiz[0].I        = FALSE;
	pib->PixSiz[1].I 	= FALSE;
	pib->PixSiz[2].I 	= FALSE;
	pib->WaveLength.I	= FALSE;
        pib->Orientation.I      = FALSE;
        pib->IndexOrder.I       = FALSE;
        pib->Projection.I       = FALSE;
        pib->AxisType[0].I      = FALSE;
        pib->AxisType[1].I      = FALSE;
        pib->AxisType[2].I      = FALSE;

	pib->Center[0].I        = FALSE;
	pib->Center[1].I 	= FALSE;
	pib->Center[2].I 	= FALSE;
	pib->SampleDistance.I   = FALSE;
        pib->DetRot[0].I        = FALSE;
        pib->DetRot[1].I        = FALSE;
        pib->DetRot[2].I        = FALSE;
        pib->DetRot[3].I        = FALSE;

        pib->BeamCenter[0].I    = FALSE;
        pib->BeamCenter[1].I    = FALSE;
        pib->BeamCenter[2].I    = FALSE;
        pib->BeamDistance.I     = FALSE;
        pib->DetTilt[0].I       = FALSE;
        pib->DetTilt[1].I       = FALSE;
        pib->DetTilt[2].I       = FALSE;
        pib->DetTilt[3].I       = FALSE;

	pib->SearchList.I       = FALSE;

/* Initialization of default values */

        /* --- file control variables */
	pib->Stream		= -1; // default must be less than zero
	pib->StreamOpenMode	= 0l;
        pib->FileNumber         = 1l;
        pib->LoopNumber         = 1l;
	CopyLine("",pib->FileName,IO_size,0);
        /* --- image option variables */
        CopyLine(DefaultFilename,pib->Name.V,IO_size,0);
        pib->FFirst.V           = 1l;
        pib->FLast.V            = 1l;
        pib->FInc.V             = 1l;
        pib->OpenMode.V         = IO_ImageProtect;
        pib->Memory.V           = 1l;
        pib->First.V            = 1l;
        pib->Last.V             = 1l;
        pib->Inc.V              = 1l;
        CopyLine("-inf",pib->MinClip.V,IO_size,0);
        CopyLine("inf",pib->MaxClip.V,IO_size,0);
        CopyLine("0.0",pib->Const.V,IO_size,0);
        CopyLine("1.0",pib->Factor.V,IO_size,0);
        CopyLine("1.0",pib->Val.V,IO_size,0);
        CopyLine("0.0",pib->ErrVal.V,IO_size,0);
        CopyLine("2",pib->Bin[0].V,IO_size,0);
        CopyLine("1",pib->Bin[1].V,IO_size,0);
        CopyLine("1",pib->Bin[2].V,IO_size,0);
	pib->Ave.V		= 1l; // default +iNave
        CopyLine("2",pib->Gnaw[0].V,IO_size,0);
        CopyLine("0",pib->Gnaw[1].V,IO_size,0);
        CopyLine("0",pib->Gnaw[2].V,IO_size,0);
        CopyLine("",pib->Title.V,IO_size,0);
        CopyLine("",pib->Time.V,IO_size,0);
        CopyLine("2",pib->Dim[0].V,IO_size,0);
        CopyLine("1",pib->Dim[1].V,IO_size,0);
        CopyLine("1",pib->Dim[2].V,IO_size,0);
        CopyLine("0.0",pib->Dummy.V,IO_size,0);
        CopyLine("0.5",pib->DDummy.V,IO_size,0);
        CopyLine("2",pib->Offset[0].V,IO_size,0);
        CopyLine("0",pib->Offset[1].V,IO_size,0);
        CopyLine("0",pib->Offset[2].V,IO_size,0);
        CopyLine("2",pib->Center[0].V,IO_size,0);
        CopyLine("0",pib->Center[1].V,IO_size,0);
        CopyLine("0",pib->Center[2].V,IO_size,0);
        CopyLine("2",pib->BinSiz[0].V,IO_size,0);
        CopyLine("1",pib->BinSiz[1].V,IO_size,0);
        CopyLine("1",pib->BinSiz[2].V,IO_size,0);
        CopyLine("2",pib->PixSiz[0].V,IO_size,0);
        CopyLine("1",pib->PixSiz[1].V,IO_size,0);
        CopyLine("1",pib->PixSiz[2].V,IO_size,0);
        CopyLine("1",pib->SampleDistance.V,IO_size,0); /* 1 m */
        CopyLine("1e-10",pib->WaveLength.V,IO_size,0); /* 0.1 nm */ 
        CopyLine("3",pib->DetRot[0].V,IO_size,0);
        CopyLine("0",pib->DetRot[1].V,IO_size,0);
        CopyLine("0",pib->DetRot[2].V,IO_size,0);
        CopyLine("0",pib->DetRot[3].V,IO_size,0);
        CopyLine("",pib->SearchList.V,IO_size,0);
        CopyLine("1",pib->Orientation.V,IO_size,0); /* 1 */ 
	pib->Projection.V = IO_ProSaxs; // default -iNpro saxs 
	pib->AxisType[0].V = IO_NoAxisType; 
	pib->AxisType[1].V = IO_AxisTypeDistance; 
	pib->AxisType[2].V = IO_AxisTypeDistance; 

} /* InitImageBlock */

/*--------------------------------------------------------------------------
   InitOptions : Initialisation of options,
   initialisation of command block and all image blocks
   If ib_len<0 no image block is initialized
   return value  :	void
     *mainhelp	:	help string for main program
  			activated with -h, +h with listing of options
     *mainvers  :
     *maindata  :       optional pointer to a structure with data
                        managed by the main program, should be NULL, 
                        if not needed.
     *pcb	:	command block
     rjctnum	:	TRUE  : If the first letter after an
  			        option marker ('-' or '+') is a number
  			        it is assumed that the argument is a
  			        number and not an option.
  			FALSE : all signs after '-' or '+' are allowed
     ib[]	:	image block array
     ib_len	:	length of image block array
  			(ib[0] is always input)
  --------------------------------------------------------------------------*/
void InitOptions( const char * mainhelp, const char * mainvers, 
                  void * maindata, int rjctnum,
                  CmdBlk * pcb, ImgBlk ib[], int ib_len )
{
	register int i;
	
	InitCommandBlock( mainhelp, mainvers, maindata,
                          rjctnum, ib, ib_len, pcb );
	for (i=0;i<ib_len;i++) InitImageBlock( ib, i );

        pcb->co = (OptDsc *) malloc(sizeof(OptDsc)*IO_CmdOptLen);
	if (pcb->co == NULL)
	   {printf("%s for command options\n",
             ErrorCannotAllcMemory); exit(-1);}
        pcb->io = (OptDsc *) malloc(sizeof(OptDsc)*IO_ImgOptLen);
	if (pcb->io == NULL)
	   {printf("%s for image options\n",
             ErrorCannotAllcMemory); exit(-1);}

	InitCommandOptions ( pcb->co, IO_CmdOptLen );
	InitImageOptions ( pcb->io, IO_ImgOptLen );

        /* write general version string */
        snprintf(pcb->SaxsVersion,IO_size,
          "# %s %s %s",saxs_progversion( pcb ),saxs_libversion(),edf_libversion());
        pcb->SaxsVersion[IO_size-1]='\0'; // force termination

} /* InitOptions */

/*--------------------------------------------------------------------------
   CheckOptionDuplication : Test, whether an option is duplicated
   return value		:	TRUE  : duplicated
  			 	FALSE : not duplicated
     CmdBlk *pcb	:	command option block
     const char *name (i)	:	new command option name
   For options beginning with with 'OutChar' or 'InChar'<digit> the
   return value is always TRUE.
   The IO_CodeLen-1 first characters of name and pcb.co[k].code are compared.
   If they are identical the function returns TRUE, if not FALSE.
  --------------------------------------------------------------------------*/
int CheckOptionDuplication( CmdBlk * pcb, const char * name )
{ int k;
  int result = FALSE;

  if (name) {
    if (IsOutputOption(name)||IsInputOption(name)) {
      result = TRUE; 
    } else { /* Compare with all existing options */
      for (k=1;k<(pcb->co[0]).npar;k++) {
        if (strncmp(pcb->co[k].code, name,IO_CodeLen-1)==0) result = TRUE;
      }
    }
  }
  return( result);

} /* CheckOptionDuplication */

/*--------------------------------------------------------------------------
   IsOutputOption : Test, whether name is an output option name
   return value            :       TRUE  : output option
                                   FALSE : no output option 
     const char *name (i)  :       option name
   An output option starts with 'OutChar'.
   If name is an output option the return value is TRUE, otherwise FALSE
  --------------------------------------------------------------------------*/
int IsOutputOption( const char * name )
{ int result = FALSE;

  if ((name)&&(name[0]==OutChar)) {
      result = TRUE;
    }
  return( result );

} /* IsOutputOption */

/*--------------------------------------------------------------------------
   IsInputOption : Test, whether name is an input option name
   return value            :       TRUE  : input option
                                   FALSE : no input option 
     const char *name (i)  :       option name
   An input option starts with 'OutChar' and is followed by a digit.
   If name is an input option the return value is TRUE, otherwise FALSE
  --------------------------------------------------------------------------*/
int IsInputOption( const char * name )
{ int result = FALSE;

  if ((name)&&(name[0]==InChar)) {
    if ((strlen(name)>1)&&isdigit((int) name[1])) {
      result = TRUE;
    }
  }
  return( result );

} /* IsInputOption */

/*--------------------------------------------------------------------------
   DefFloatOption : Definition of float command option number argopt_no
   return value		:	void
     CmdBlk *pcb	:	command option block
     char *name	(i)	:	new command option name
     int argopt_no (i)	:	number of float argument [0..IO_MaxFloatOpt-1]
  --------------------------------------------------------------------------*/
void DefFloatOption ( CmdBlk * pcb, char * name, int argopt_no )
{
	int nct;

        if( CheckOptionDuplication( pcb, name ) )
	  {printf("%s-%s\n",ErrorOptionDuplicated,name);
           exit(-1);}

	nct = NArgFst + argopt_no;
	if ((nct<NArgFst) || (nct>NArgLst))
	  {printf("%s%s, argopt_no = %d \n", ErrorOptionDefnFailed,
                    name, argopt_no); exit(-1);}
	CopyLine(name,(pcb->co[nct]).code,IO_CodeLen,-1);
	(pcb->co[nct]).ptyp = IO_tpfloat; (pcb->co[nct]).npar = 1;
} /* DefFloatOption */

/*--------------------------------------------------------------------------
   DefIntOption : Definition of integer command option number numopt_no
   return value		:	void
     CmdBlk *pcb	:	command option block
     char *name	(i)	:	new command option name
     int numopt_no (i)	:	number of integer argument [0..IO_MaxIntOpt-1]
  --------------------------------------------------------------------------*/
void DefIntOption   ( CmdBlk * pcb, char * name, int numopt_no )
{
	int nct;

        if( CheckOptionDuplication( pcb, name ) )
	  {printf("%s-%s\n",ErrorOptionDuplicated,name);
	   exit(-1);}

	nct = NNumFst + numopt_no;
	if ((nct<NNumFst) || (nct>NNumLst))
	  {printf("%s%s, numopt_no = %d \n", ErrorOptionDefnFailed,
		   name, numopt_no); exit(-1);}
	CopyLine(name,(pcb->co[nct]).code,IO_CodeLen,-1);
	(pcb->co[nct]).ptyp = IO_tplong; (pcb->co)[nct].npar = 1;
} /* DefIntOption */

/*--------------------------------------------------------------------------
   DefLineOption : Definition of line command option number linopt_no
   return value		:	void
     CmdBlk *pcb	:	command option block
     char *name	(i)	:	new command option name
     int linopt_no (i)	:	number of line argument [0..IO_MaxIntOpt-1]
  --------------------------------------------------------------------------*/
void DefLineOption   ( CmdBlk * pcb, char * name, int linopt_no )
{
	int nct;

        if ( CheckOptionDuplication( pcb, name ) )
	  {printf("%s-%s\n",ErrorOptionDuplicated,name);
	   exit(-1);}

	nct = NLinFst + linopt_no;
	if ((nct<NLinFst) || (nct>NLinLst))
	  {printf("%s%s, linopt_no = %d \n", ErrorOptionDefnFailed,
		   name, linopt_no); exit(-1);}
	CopyLine(name,(pcb->co[nct]).code,IO_CodeLen,-1);
	(pcb->co[nct]).ptyp = IO_tpline; (pcb->co)[nct].npar = 1;
} /* DefLineOption */
/*--------------------------------------------------------------------------
   DefFlgOption : Definition of flag command option number flgopt_no
   return value         :       void
     CmdBlk *pcb        :       command option block
     char *name (i)     :       new command option name
     int flgopt_no (i)  :       number of flag argument [0..IO_MaxFlagOpt-1]
  --------------------------------------------------------------------------*/
void DefFlgOption   ( CmdBlk * pcb, char * name, int flgopt_no )
{
        int nct;

        if( CheckOptionDuplication( pcb, name ) )
          {printf("%s-%s\n",ErrorOptionDuplicated,name);
           exit(-1);}

        nct = NFlgFst + flgopt_no;
        if ((nct<NFlgFst) || (nct>NFlgLst))
          {printf("%s%s, flgopt_no = %d \n", ErrorOptionDefnFailed,
                   name, flgopt_no); exit(-1);}
        CopyLine(name,(pcb->co[nct]).code,IO_CodeLen,-1);
        (pcb->co[nct]).ptyp = IO_tpflag; (pcb->co)[nct].npar = 0;
} /* DefFlgOption */

/*--------------------------------------------------------------------------
   GetOption : Get option number from the string
   History : 28-Jan-1995 PB use of dsc[0].npar for loop
  
  	return value	(o) : void
  		   *ps	(i) : string with option
  		   dsc	(i) : option descriptor with length in dsc[0].npar
  	    *popt_no	(o) : option number
  				0 if not found
  	    *pstatus	(o) : Success or NoOption
  --------------------------------------------------------------------------*/
void GetOption ( char * ps, OptDsc dsc[], int * popt_no, int * pstatus )
{	int nct;
	*pstatus = NoOption;
	*popt_no = 0;
	nct = 1;
	while ( (nct < dsc[0].npar) && strcmp(ps,dsc[nct].code) ) nct++;
	if (dsc[nct].code[0]) { *popt_no = nct; *pstatus = Success; }
} /* GetOption */

/*-------------------------------------------------------------------------
   FindOption   : Read next argument and test whether it is an option.
                  If yes, specify it.
   It has no effect on *parg_no if argument string does not contain
   an option, i.e. if it does not contain a "-" or "+". In this case the
   return value is FALSE.
  
  	return value	(o) : TRUE if s contains an option
  				(known or unknown!)
  				FALSE if s does not contain an option
  	argv		(i) : pointer to argument string array
        *parg_no      (i/o) : i: field number of previous argument
  			      o: incremented, when return value is TRUE
        pcb->
  	RejectNum	(i) : TRUE  : reject numbers as options
  		              and return with FALSE and
  		              *pstatus = Success
  		      	      FALSE : accept numbers as options
        *pblk_no	(o) : block number of corresponding block,
  				-1: CommandBlock
  				 0: ImageBlock[0]
  				 n: ImageBlock[n]
        *popt_no	(o) : option number,
  				0 if option not found
        *popt_onoff	(o) : >0  if "+" option
  			      <0  if "-" option
        *pstatus	(o) : status : Success, ScanError, NoOption
  -------------------------------------------------------------------------*/
int FindOption(char * argv[], int * parg_no, CmdBlk * pcb,
		int * pblk_no, int * popt_no, int * popt_onoff, int * pstatus)
{
	char *  p1;
	char	c1[IO_size];
	int	i1;

	*pstatus    = Success;
	*popt_no    = 0;
	*popt_onoff = 0;
	*pblk_no    = -2;

        if (!(argv[*parg_no+1])) 
          /* no option */ return(FALSE);

	/* convert to small characters */
	CopyLine(argv[*parg_no+1],c1,IO_size,-1);
	/* Is argument an option? */
	if (c1[0] == '-') *popt_onoff = -1; else {
	if (c1[0] == '+') *popt_onoff = +1; else
	   /* no option */ return(FALSE); }
	if ( (pcb->RejectNum) && (0<=(c1[1]-'0')) && ((c1[1]-'0')<=9) )
	   /* no option */ return(FALSE);
	*parg_no += 1;   /* argument is an option */
	p1=&c1[1];   /* skip first character in string */
	/* What type of option? */
	*pstatus = ScanError;
	switch (*p1) {
		case 'o':
			/* output image option */
			p1++;
			*pblk_no = 0;
                        if (pcb->ImgBlkLen<1)
                           return(TRUE); /* error, no output block */
			break;
		case 'i':
			/* input image option */
			p1++;
			/* read number */
			i1 = (int) ((*p1++) - '0');
			if ((i1<=0) || (i1>9))
			   return(TRUE); /* error, no number */
			*pblk_no = i1;
			break;
		default:
			/* command option */
			*pblk_no = -1;
			break;
		}
	if (!(*p1)) return(TRUE); /* ScanError when rest string empty */
	*pstatus = NoOption;
	if (*pblk_no==-1) /* command option */
		GetOption ( p1, pcb->co, popt_no, pstatus ); else
	if (*pblk_no>=0)   /* image option */
		GetOption ( p1, pcb->io, popt_no, pstatus );
	return(TRUE);
} /* FindOption */

/*-------------------------------------------------------------------------
   ScanSys :  Read reference system
  	return value		(o) : 0 wild card, no rsys
  				      1 rsys found and returned
  	*s1			(i) : input string
  	*prsys			(o) : rsys
  	*pstatus		(o) : status : Success, ScanError
   The return value is always 0 in the case of an error.
  --------------------------------------------------------------------------*/
int ScanSys( char * s1, long int * prsys, int * pstatus )
{
  if (!s1) { *pstatus = ScanEmpty; return(0); }
  *pstatus = Success;
  if (strcmp(s1,WildCard) == 0) return(0); /* wild card */
  CopyLine(s1,s1,IO_size,-1);   /* convert to small characters */
  *prsys  = (long int) IO_NoRSys;
  if (strcmp( s1, SArray )  == 0) {
    *prsys = (long int) IO_Array; } else
  if (strcmp( s1, SImage )  == 0) {
    *prsys = (long int) IO_Image; } else
  if (strcmp( s1, SCenter )  == 0) {
    *prsys = (long int) IO_Center; } else
  if (strcmp( s1, SRegion  )  == 0) {
    *prsys = (long int) IO_Region; } else
  if (strcmp( s1, SReal  )  == 0) {
    *prsys = (long int) IO_Real; } else
  if (strcmp( s1, SNormal  )  == 0) {
    *prsys = (long int) IO_Normal; } else
  if (strcmp( s1, STangens  )  == 0) {
    *prsys = (long int) IO_Tangens; } else
  if (strcmp( s1, SSaxs  )  == 0) {
    *prsys = (long int) IO_Saxs; } else {
    /* unknown reference system*/
    *pstatus = ScanError; ShowSys(); return(0); }
  return(1);
} /* ScanSys */

/*-------------------------------------------------------------------------
   ScanPro :  Read projection type
        return value            (o) : 0 wild card, no pro 
                                      1 pro found and returned
        *s1                     (i) : input string
        *ppro                   (o) : pro 
        *pstatus                (o) : status : Success, ScanError
   The return value is always 0 in the case of an error.
  --------------------------------------------------------------------------*/
int ScanPro( char * s1, long int * ppro, int * pstatus )
{ 
  if (!s1) { *pstatus = ScanEmpty; return(0); }
  *pstatus = Success;
  if (strcmp(s1,WildCard) == 0) return(0); /* wild card */
  *ppro = strtopro ( s1 );
  if ( *ppro == IO_NoPro ) {
    /* unknown projection type */
    *pstatus = ScanError; ShowPro(); return(0); 
  }
  return(1);
} /* ScanPro */

/*-------------------------------------------------------------------------
   ScanAxis :  Read axis type
        return value            (o) : 0 wild card, no axis
                                      1 axis found and returned
        *s1                     (i) : input string
        *paxis                  (o) : axis 
        *pstatus                (o) : status : Success, ScanError
   The return value is always 0 in the case of an error.
  --------------------------------------------------------------------------*/
int ScanAxis( char * s1, long int * paxis, int * pstatus )
{
  if (!s1) { *pstatus = ScanEmpty; return(0); }
  *pstatus = Success;
  if (strcmp(s1,WildCard) == 0) return(0); /* wild card */
  *paxis = strtoaxis ( s1 );
  if ( *paxis == IO_NoAxisType ) {
    /* unknown axis type */
    *pstatus = ScanError; ShowAxis(); return(0);
  }
  return(1);
} /* ScanAxis */

/*-------------------------------------------------------------------------
   ScanMode :  Read file opening modus
  	return value		(o) : 0 : wild card, no modus
  				      1 : modus found and returned
  	*ps			(i) : input string
  	*pmode			(o) : output mode
  	*pstatus		(o) : status : Success, ScanError
   The return value is always 0 in the case of an error.
  --------------------------------------------------------------------------*/
int ScanMode( char * s1, long int * pmode, int * pstatus )
{ long int mode;

  if (!s1) { *pstatus = ScanEmpty; return(0); }
  *pstatus = Success;
  if (strcmp(s1,WildCard) == 0) return(0); /* wild card */
  CopyLine(s1,s1,IO_size,-1);   /* convert to small characters */
  *pmode   = 0l;
  mode = 0x0l;
  if (strcmp( s1, SNewFP )  == 0) {
    mode = mode | IO_New | IO_FileProtect;  }                  else
  if (strcmp( s1, SNewIP )  == 0) {
    mode = mode | IO_New | IO_ImageProtect; }                  else
  if (strcmp( s1, SNew )    == 0) {
    mode = mode | IO_New;                   }                  else
  if (strcmp( s1, SOldFP )  == 0) {
    mode = mode | IO_Old | IO_FileProtect;  }                  else
  if (strcmp( s1, SOldIP )  == 0) {
    mode = mode | IO_Old | IO_ImageProtect; }                  else
  if (strcmp( s1, SOld )    == 0) {
    mode = mode | IO_Old;                   }                  else
  if (strcmp( s1, SAnyIP )  == 0) {
    mode = mode | IO_ImageProtect;          }                  else
  if (strcmp( s1, SAnyFP )  == 0) {
    mode = mode | IO_FileProtect;           }                  else
  if (strcmp( s1, SAny	) == 0)   {
    mode = mode;                            }                  else {
    /* no possible mode */
    *pstatus = ScanError; ShowModes(); return(0); }
  *pmode = (long) mode;
  return(1);
} /* ScanMode */

/*-------------------------------------------------------------------------
   ScanLong :  Read long integer
  	return value		(o) : 0 wild card, no value
  				      1 value found and returned
  	*s1			(i) : input string
  	*pvalue			(o) : long value
  	*pstatus		(o) : status : Success, ScanError
   The return value is always 0 in the case of an error.
  --------------------------------------------------------------------------*/
int ScanLong( char * s1, long int * pvalue, int * pstatus )
{
  if (!s1) { *pstatus = ScanEmpty; return(0); }
  *pstatus = Success;
  if (strcmp(s1,WildCard) == 0) return(0); /* wild card */
  *pvalue = longexpr(s1,pstatus);
  if (*pstatus!=Success) return(0);
  return(1);

} /* ScanLong */

/*-------------------------------------------------------------------------
   ScanLvExp :  Read long expression
        return value            (o) : 0 wild card, no value
                                      1 value found and returned
        *s1                     (i) : input string
        *line                   (o) : long expression string
        *pstatus                (o) : status : Success, ScanError
   The return value is always 0 in the case of an error.
  --------------------------------------------------------------------------*/
int ScanLvExp( char * s1, char * line, int * pstatus )
{
  if (!s1) { *pstatus = ScanEmpty; return(0); }
  *pstatus = Success;
  if (strcmp(s1,WildCard) == 0) return(0); /* wild card */
  CopyLine(s1,line,IO_size,0);
  if (strlen(line)==0) {
    *pstatus = ScanError; return(0); }
  return(1);
 
} /* ScanLvExp */

/*-------------------------------------------------------------------------
   ScanFloat :  Read float
  	return value		(o) : 0 wild card, no value
  				      1 value found and returned
  	*s1			(i) : input string
  	*pvalue			(o) : float value
  	*pstatus		(o) : status : Success, ScanError
   The return value is always 0 in the case of an error.
  --------------------------------------------------------------------------*/
int ScanFloat( char * s1, float * pvalue, int * pstatus )
{
  if (!s1) { *pstatus = ScanEmpty; return(0); }
  *pstatus = Success;
  if (strcmp(s1,WildCard) == 0) return(0); /* wild card */
  *pvalue = floatexpr(s1,pstatus);
  if (*pstatus!=Success) return(0);
  return(1);

} /* ScanFloat */

/*-------------------------------------------------------------------------
   ScanFlExp :  Read float expression
        return value            (o) : 0 wild card, no value
                                      1 value found and returned
        *s1                     (i) : input string
        *line                   (o) : float expression string
        *pstatus                (o) : status : Success, ScanError
   The return value is always 0 in the case of an error.
  --------------------------------------------------------------------------*/
int ScanFlExp( char * s1, char * line, int * pstatus )
{
  if (!s1) { *pstatus = ScanEmpty; return(0); }
  *pstatus = Success;
  if (strcmp(s1,WildCard) == 0) return(0); /* wild card */
  CopyLine(s1,line,IO_size,0);
  if (strlen(line)==0) { *pstatus = ScanError; return(0); }
  return(1); 
 
} /* ScanFlExp */

/*-------------------------------------------------------------------------
   ScanLine :  Read line
  	return value		(o) : 0 wild card, no value
  				      1 value found and returned
  	*s1			(i) : input string
  	*line			(o) : line
  	*pstatus		(o) : status : Success, ScanError
   The return value is always 0 in the case of an error.
   The wild card string can be read if s1 starts with a blank, e.g.
   " =" for WildCard == "=". Leeding blanks are ignored.
  --------------------------------------------------------------------------*/
int ScanLine( char * s1, char * line, int * pstatus )
{ 
  if (!s1) { *pstatus = ScanEmpty; return(0); }
  *pstatus = Success;
  if (strcmp(s1,WildCard) == 0) return(0); /* wild card */
  CopyLine(s1,line,IO_size,0);
//  Empty strings are allowed. The next line seems to be an unnecessary
//  restriction. Therefore removed.
//  if (strlen(line)==0) { *pstatus = ScanError; return(0); }
  return(1);
} /* ScanLine */

/*-------------------------------------------------------------------------
   DoCmdOptions :  Write option values to command block
  	return value     (o) : void
  	ps	         (i) : pointer to argument string array
  	*pc	    (i/o) : i: field number of previous argument
                            o: incremened
  	opt_no	         (i) : option number > 0
  	opt_onoff        (i) : >0  if "+" option
  		               >0  if "-" option
  	*pcb	       (i/o) : command block
  	*pstatus           (o) : status : Success, ScanError, NoOption
  -------------------------------------------------------------------------*/
void DoCmdOption( char * ps[], int * pc, int opt_no, int opt_onoff,
	 	  CmdBlk * pcb, int * pstatus )
{
	int	 argopt_no;
	int	 numopt_no;
	int	 linopt_no;
	int	 flgopt_no;
	long int i1;
	float	 f1;
	char	 c1[IO_size];

	*pstatus = ScanError;

	switch (opt_no)
	{ case NTestBit:
		/* TestBit */
		if (opt_onoff>=0) pcb->TestBit += TRUE;
                  else pcb->TestBit = FALSE;
		break;
	  case NHelp:
		/* Help */
		if (opt_onoff>=0) pcb->Help.V = (long int) TRUE;
                  else pcb->Help.V = (long int) FALSE;
		pcb->Help.I = opt_onoff;
		ShowHelp(pcb);
                *pstatus = StoppedAfterHelp;
		return;   /* Help only */
          case NIODebug:
                /* IODebug */
                i1 = (long) doubleexpr( ps[++*pc], pstatus);
                if (*pstatus) {
                  return;
                } else {
                  pcb->IODebug.V = i1;
                  pcb->IODebug.I = opt_onoff;
                }
                if (opt_onoff) edfio_debug_set( i1 );
                break;
          case NIOLevel:
                /* IOLevel */
                i1 = longexpr( ps[++*pc], pstatus);
                if (*pstatus) {
                  return; 
                } else {
                  pcb->IOLevel.V = i1;
                  pcb->IOLevel.I = opt_onoff;
                }
                if (opt_onoff) edfio_level_set( i1 );
                break;
	  case NPrompt:
		/* Prompt */
		if (opt_onoff>=0) pcb->Prompt.V =(long int) TRUE;
                  else pcb->Prompt.V =(long int) FALSE;
                pcb->Prompt.I = opt_onoff;
		break;
	  case NShift:
		/* Shift1 */
                if (ScanFlExp(ps[++*pc], c1, pstatus)) {
                  CopyLine(c1,pcb->Shift[1].V,IO_size,0);
                  pcb->Shift[1].I = opt_onoff;
                } else if (*pstatus!=Success) return;
                /* Shift2 */
                if (ScanFlExp(ps[++*pc], c1, pstatus)) {
                  CopyLine(c1,pcb->Shift[2].V,IO_size,0);
                  pcb->Shift[2].I = opt_onoff;
                } else if (*pstatus!=Success) return;
                break;
	  case NRSys:
		/* reference system for calculation */
		if (ScanSys(ps[++*pc],&i1,pstatus)) {
		  pcb->RSys.V = i1;
		  pcb->RSys.I = opt_onoff;
                  /* RSys.V is default for USys.V */
                  if (!pcb->USys.I) pcb->USys.V = pcb->RSys.V;
		} else if (*pstatus!=Success) return;
                break;
	  case NUSys:
		/* reference system for user input */
		if (ScanSys(ps[++*pc],&i1,pstatus)) {
		  pcb->USys.V = i1;
		  pcb->USys.I = opt_onoff;
		} else if (*pstatus!=Success) return;
                break;
          case NGeneralBlock:
                /* GeneralBlock */
                if (opt_onoff>=0) pcb->GeneralBlock.V =(long int) TRUE;
                  else pcb->GeneralBlock.V =(long int) FALSE;
                pcb->GeneralBlock.I = opt_onoff;
                break;
          case NPassHeader:
                /* PassHeader */
                if (opt_onoff>=0) pcb->PassHeader.V =(long int) TRUE;
                  else pcb->PassHeader.V =(long int) FALSE;
                pcb->PassHeader.I = opt_onoff;
                break;
          case NCleanHeader:
                /* CleanHeader */
                if (opt_onoff>=0) pcb->CleanHeader.V =(long int) TRUE;
                  else pcb->CleanHeader.V =(long int) FALSE;
                pcb->CleanHeader.I = opt_onoff;
                break;
          case NCreateVariance:
                /* CreateVariance */
                if (opt_onoff>=0) pcb->CreateVariance.V =(long int) TRUE;
                  else pcb->CreateVariance.V =(long int) FALSE;
                pcb->CreateVariance.I = opt_onoff;
                break;
          case NAntiAliased:
                /* AntiAliased */
                if (opt_onoff>=0) pcb->AntiAliased.V =(long int) TRUE;
                  else pcb->AntiAliased.V =(long int) FALSE;
                pcb->AntiAliased.I = opt_onoff;
                break;
          case NVarianceWeight:
                /* VarianceWeight */
                if (opt_onoff>=0) pcb->VarianceWeight.V =(long int ) TRUE;
                  else pcb->VarianceWeight.V =(long int) FALSE;
                pcb->VarianceWeight.I = opt_onoff;
                break;
          case NOutputDataType:
                /* reference system for calculation */
                if ( (i1 = (long int) edf_string2datatype( ps[++*pc] )) ) {
                  pcb->OutputDataType.V = i1;
                  pcb->OutputDataType.I = opt_onoff;
                } else return;
                break;
          case NOutputDataValueOffset:
                /* number of output images to add */
                if (ScanLong(ps[++*pc], &i1, pstatus)) {
                  pcb->OutputDataValueOffset.V = i1;
                  pcb->OutputDataValueOffset.I = opt_onoff;
                }  else if (*pstatus!=Success) return;
                break;
          case NOutputCompression:
                /* output compression */
                if ( (i1 = (long int) edf_string2compression( ps[++*pc] )) ) {
                  pcb->OutputCompression.V = i1;
                  pcb->OutputCompression.I = opt_onoff;
                } else return;
                break;
          case NBslInputByteOrder:
                /* First */
                if (ScanLong(ps[++*pc], &i1, pstatus)) {
                  pcb->BslInputByteOrder.V = i1;
                  pcb->BslInputByteOrder.I = opt_onoff;
                }  else if (*pstatus!=Success) return;
                break;
          case NAdd:
                /* number of output images to add */
                if (ScanLong(ps[++*pc], &i1, pstatus)) {
                  pcb->Add.V = i1;
                  pcb->Add.I = opt_onoff;
                }  else if (*pstatus!=Success) return;
                break;
          case NIPMin:
                /* minimum coverage of output pixel */
                if (ScanFloat(ps[++*pc], &f1, pstatus)) {
                  pcb->IPMin.V = f1;
                  pcb->IPMin.I = opt_onoff;
                  // can only be set globally 
                  Ipolmin( pcb->IPMin.V ); 
                }  else if (*pstatus!=Success) return;
                break;
          case NMaxLineWidth:
                /* First */
                if (ScanLong(ps[++*pc], &i1, pstatus)) {
                  pcb->MaxLineWidth.V = i1;
                  pcb->MaxLineWidth.I = opt_onoff;
                }  else if (*pstatus!=Success) return;
                break;
          case NMinHeaderSize:
                /* Minimum header size */
                if (ScanLong(ps[++*pc], &i1, pstatus)) {
                  pcb->MinHeaderSize.V = i1;
                  pcb->MinHeaderSize.I = opt_onoff;
                }  else if (*pstatus!=Success) return;
                break;
	  case NCommonOrientation:
		/* Common orientation flag */
                /* pcb->CommonOrientation.V keeps dynamically the common orientation! */
                pcb->CommonOrientation.I = opt_onoff;
		break;
	  case NBeam:
                pcb->Beam.V = (long int) opt_onoff;
                pcb->Beam.I = opt_onoff;
		break;
	  default:
		if ( (opt_no>=NArgFst) && (opt_no<=NArgLst) ) {
		  /* Float Option */
		  argopt_no = opt_no-NArgFst;
		  if (ScanFloat(ps[++*pc], &f1, pstatus)) {
	 	    pcb->Arg[argopt_no].V = f1;
		    pcb->Arg[argopt_no].I = opt_onoff;
		    } else if (*pstatus!=Success) return;
		  break;
		  } /* Float */
                if ( (opt_no>=NFlexpFst) && (opt_no<=NFlexpLst) ) {
                  /* Flexp Option */
                  linopt_no = opt_no-NFlexpFst;
                  if (ScanLine(ps[++*pc], c1, pstatus)) {
                    CopyLine(c1,pcb->Flexp[linopt_no].V,IO_size,0);
                    pcb->Flexp[linopt_no].I = opt_onoff;
                    } else if (*pstatus!=Success) return;
                  break;
                  } /* Flexp */ 
		if ( (opt_no>=NNumFst) && (opt_no<=NNumLst) ) {
		  /* Integer Option */
		  numopt_no = opt_no-NNumFst;
		  if (ScanLong(ps[++*pc], &i1, pstatus)) {
	 	    pcb->Num[numopt_no].V = i1;
		    pcb->Num[numopt_no].I = opt_onoff;
		    } else if (*pstatus!=Success) return;
 		  break;
		  } /* Integer */
               if ( (opt_no>=NLvexpFst) && (opt_no<=NLvexpLst) ) {
                  /* Lvexp Option */
                  linopt_no = opt_no-NLvexpFst;
                  if (ScanLine(ps[++*pc], c1, pstatus)) {
                    CopyLine(c1,pcb->Lvexp[linopt_no].V,IO_size,0);
                    pcb->Lvexp[linopt_no].I = opt_onoff;
                    } else if (*pstatus!=Success) return;
                  break;
                  } /* Lvexp */
		if ( (opt_no>=NLinFst) && (opt_no<=NLinLst) ) {
		  /* Line Option */
		  linopt_no = opt_no-NLinFst;
		  if (ScanLine(ps[++*pc], c1, pstatus)) {
		    CopyLine(c1,pcb->Lin[linopt_no].V,IO_size,0);
		    pcb->Lin[linopt_no].I = opt_onoff;
		    } else if (*pstatus!=Success) return;
 		  break;
		  } /* Line */
                if ( (opt_no>=NFlgFst) && (opt_no<=NFlgLst) ) {
                  /* Flag Option */
                  flgopt_no = opt_no-NFlgFst;
                  if (opt_onoff>=0) pcb->Flg[flgopt_no].V = (long int) TRUE;
                    else pcb->Flg[flgopt_no].V = (long int) FALSE;
                  pcb->Flg[flgopt_no].I = opt_onoff;
                  break;
                  } /* Flag */
		*pstatus = NoOption;
		return;
		}
	*pstatus = Success;
} /* DoCmdOptions */

/*--------------------------------------------------------------------------
   DoImgOptions :  Write option values to image block
  	return value	(o) : void
  	ps		(i) : pointer to argument string array
  	*pc	       (i/o): i: field number of previous argument
  			 o: incremented
  	opt_no		(i) : option number > 0
  	opt_onoff	(i) : >0 if "+" option
  			      <0 if "-" option
  	*pib	       (i/o): image block
  	*pstatus	(o) : status : Success, ScanError, NoOption
  --------------------------------------------------------------------------*/
void DoImgOption( char * ps[], int * pc, int opt_no, int opt_onoff,
                  ImgBlk * pib, int * pstatus )
{
	long int	i1;
	char		c1[IO_size];

	*pstatus = ScanError;
	switch (opt_no)
	{ case NName:
		/* Name */
	  	if (ScanLine(ps[++*pc], c1, pstatus)) {
		  CopyLine(c1,pib->Name.V,IO_size,0);
		  pib->Name.I = opt_onoff;
                  /* scan Name for file numbers */
                  extract_filenumbers( pib, &(pib->Name), opt_onoff, pstatus );
                  if (*pstatus!=Success) return;
		} else if (*pstatus!=Success) return;
 		break;
	  case NOpenMode:
		/* OpenMode */
		if (ScanMode(ps[++*pc],&i1,pstatus)) {
		  pib->OpenMode.V = i1;
		  pib->OpenMode.I = opt_onoff;
		} else if (*pstatus!=Success) return;
                break;
          case NMemory:
                /* Memory */
                if (ScanLong(ps[++*pc], &i1, pstatus)) {
                  pib->Memory.V = i1;
                  pib->Memory.I = opt_onoff;
                }  else if (*pstatus!=Success) return;
                break;
	  case NFirst:
		/* First */
		if (ScanLong(ps[++*pc], &i1, pstatus)) {
		  pib->First.V = i1;
		  pib->First.I = opt_onoff;
		}  else if (*pstatus!=Success) return;
 		break;
	  case NLast:
		/* Last */
		if (ScanLong(ps[++*pc], &i1, pstatus)) {
		  pib->Last.V = i1;
		  pib->Last.I = opt_onoff;
		} else if (*pstatus!=Success) return;
 		break;
	  case NInc:
		/* Inc */
		if (ScanLong(ps[++*pc], &i1, pstatus)) {
		  pib->Inc.V = i1;
		  pib->Inc.I = opt_onoff;
		} else if (*pstatus!=Success) return;
 		break;
        case NMinClip:
                /* MinClip */
                if (ScanFlExp(ps[++*pc], c1, pstatus)) {
                  CopyLine(c1,pib->MinClip.V,IO_size,0);
                  pib->MinClip.I = opt_onoff;
                } else if (*pstatus!=Success) return;
                break;
        case NMaxClip:
                /* MaxClip */
                if (ScanFlExp(ps[++*pc], c1, pstatus)) {
                  CopyLine(c1,pib->MaxClip.V,IO_size,0);
                  pib->MaxClip.I = opt_onoff;
                } else if (*pstatus!=Success) return;
                break;
	case NConst:
		/* Const */
		if (ScanFlExp(ps[++*pc], c1, pstatus)) {
                  CopyLine(c1,pib->Const.V,IO_size,0);
		  pib->Const.I = opt_onoff;
		} else if (*pstatus!=Success) return;
 		break;
       case NFactor:
		/* Factor */
		if (ScanFlExp(ps[++*pc], c1, pstatus)) {
                  CopyLine(c1,pib->Factor.V,IO_size,0);
		  pib->Factor.I = opt_onoff;
		} else if (*pstatus!=Success) return;
 		break;
       case NVal:
		/* Val */
		if (ScanFlExp(ps[++*pc], c1, pstatus)) {
                  CopyLine(c1,pib->Val.V,IO_size,0);
		  pib->Val.I = opt_onoff;
		} else if (*pstatus!=Success) return;
 		break;
       case NErrVal:
		/* ErrVal */
		if (ScanFlExp(ps[++*pc], c1, pstatus)) {
                  CopyLine(c1,pib->ErrVal.V,IO_size,0);
		  pib->ErrVal.I = opt_onoff;
		} else if (*pstatus!=Success) return;
 		break;
	  case NBin:
		/* Bin1 */
		if (ScanLvExp(ps[++*pc], c1, pstatus)) {
                  CopyLine(c1,pib->Bin[1].V,IO_size,0);
		  pib->Bin[1].I = opt_onoff;
		} else if (*pstatus!=Success) return;
                /* Bin2 */
                if (ScanLvExp(ps[++*pc], c1, pstatus)) {
                  CopyLine(c1,pib->Bin[2].V,IO_size,0);
                  pib->Bin[2].I = opt_onoff;
                } else if (*pstatus!=Success) return;
 		break;
          case NAve:
                /* Ave */
                if (opt_onoff>=0) pib->Ave.V =(long int) TRUE;
                  else pib->Ave.V =(long int) FALSE;
                pib->Ave.I = opt_onoff;
                break;
          case NGnaw:
                /* Gnaw1 */
                if (ScanLvExp(ps[++*pc], c1, pstatus)) {
                  CopyLine(c1,pib->Gnaw[1].V,IO_size,0);
                  pib->Gnaw[1].I = opt_onoff;
                } else if (*pstatus!=Success) return;
                /* Gnaw2 */
                if (ScanLvExp(ps[++*pc], c1, pstatus)) {
                  CopyLine(c1,pib->Gnaw[2].V,IO_size,0);
                  pib->Gnaw[2].I = opt_onoff;
                } else if (*pstatus!=Success) return;
                break;
	  case NTitle:
		/* Title */
		if (ScanLine(ps[++*pc], c1, pstatus)) {
		  CopyLine(c1,pib->Title.V,IO_size,0);
		  pib->Title.I = opt_onoff;
		} else if (*pstatus!=Success) return;
 		break;
          case NTime:
                /* Time */
                if (ScanLine(ps[++*pc], c1, pstatus)) {
                  CopyLine(c1,pib->Time.V,IO_size,0);
                  pib->Time.I = opt_onoff;
                } else if (*pstatus!=Success) return;
                break;
	   case NDim:
                /* Dim1 */
                if (ScanLvExp(ps[++*pc], c1, pstatus)) {
                  CopyLine(c1,pib->Dim[1].V,IO_size,0);
                  pib->Dim[1].I = opt_onoff;
                } else if (*pstatus!=Success) return;
                /* Dim2 */
                if (ScanLvExp(ps[++*pc], c1, pstatus)) {
                  CopyLine(c1,pib->Dim[2].V,IO_size,0);
                  pib->Dim[2].I = opt_onoff;
                } else if (*pstatus!=Success) return;
                break;
	 case NDummy:
		/* Dummy */
                if (ScanFlExp(ps[++*pc], c1, pstatus)) {
                  CopyLine(c1,pib->Dummy.V,IO_size,0);
                  pib->Dummy.I = opt_onoff;
                } else if (*pstatus!=Success) return;
                break;
	case NDDummy:
		/* DDummy */
                if (ScanFlExp(ps[++*pc], c1, pstatus)) {
                  CopyLine(c1,pib->DDummy.V,IO_size,0);
                  pib->DDummy.I = opt_onoff;
                } else if (*pstatus!=Success) return;
                break;
	case NOffset:
		/* Offset1 */
                if (ScanFlExp(ps[++*pc], c1, pstatus)) {
                  CopyLine(c1,pib->Offset[1].V,IO_size,0);
                  pib->Offset[1].I = opt_onoff;
                } else if (*pstatus!=Success) return;
 		/* Offset2 */
                if (ScanFlExp(ps[++*pc], c1, pstatus)) {
                  CopyLine(c1,pib->Offset[2].V,IO_size,0);
                  pib->Offset[2].I = opt_onoff;
                } else if (*pstatus!=Success) return;
                break;
        case NBinSiz:
                /* BinSiz1 */
                if (ScanFlExp(ps[++*pc], c1, pstatus)) {
                  CopyLine(c1,pib->BinSiz[1].V,IO_size,0);
                  pib->BinSiz[1].I = opt_onoff;
                } else if (*pstatus!=Success) return;
                /* BinSiz2 */
                if (ScanFlExp(ps[++*pc], c1, pstatus)) {
                  CopyLine(c1,pib->BinSiz[2].V,IO_size,0);
                  pib->BinSiz[2].I = opt_onoff;
                } else if (*pstatus!=Success) return;
                break;
        case NPixSiz:
                /* PixSiz1 */
                if (ScanFlExp(ps[++*pc], c1, pstatus)) {
                  CopyLine(c1,pib->PixSiz[1].V,IO_size,0);
                  pib->PixSiz[1].I = opt_onoff;
                } else if (*pstatus!=Success) return;
                /* PixSiz2 */
                if (ScanFlExp(ps[++*pc], c1, pstatus)) {
                  CopyLine(c1,pib->PixSiz[2].V,IO_size,0);
                  pib->PixSiz[2].I = opt_onoff;
                } else if (*pstatus!=Success) return;
                break;
	case NWaveLength:
		/* WaveLength */
                if (ScanFlExp(ps[++*pc], c1, pstatus)) {
                  CopyLine(c1,pib->WaveLength.V,IO_size,0);
                  pib->WaveLength.I = opt_onoff;
                } else if (*pstatus!=Success) return;
                break;
        case NOrientation:
                /* Orientation */
                if (ScanLvExp(ps[++*pc], c1, pstatus)) {
                  CopyLine(c1,pib->Orientation.V,IO_size,0);
                  pib->Orientation.I = opt_onoff;
                } else if (*pstatus!=Success) return;
                break;
        case NIndexOrder:
                /* IndexOrder */
                if (ScanLvExp(ps[++*pc], c1, pstatus)) {
                  CopyLine(c1,pib->IndexOrder.V,IO_size,0);
                  pib->IndexOrder.I = opt_onoff;
                } else if (*pstatus!=Success) return;
                break;
        case NProjection:
              /* projection type */
              if (ScanPro(ps[++*pc],&i1,pstatus)) {
                pib->Projection.V = i1;
                pib->Projection.I = opt_onoff;
              } else if (*pstatus!=Success) return;
              break;
        case NAxisType:
              /* axis 1 type */
              if (ScanAxis(ps[++*pc],&i1,pstatus)) {
                pib->AxisType[1].V = i1;
                pib->AxisType[1].I = opt_onoff;
              } else if (*pstatus!=Success) return;
              /* axis 2 type */
              if (ScanAxis(ps[++*pc],&i1,pstatus)) {
                pib->AxisType[2].V = i1;
                pib->AxisType[2].I = opt_onoff;
              } else if (*pstatus!=Success) return;
              break;
	case NCenter:
		/* Center1 */
                if (ScanFlExp(ps[++*pc], c1, pstatus)) {
                  CopyLine(c1,pib->Center[1].V,IO_size,0);
                  pib->Center[1].I = opt_onoff;
                } else if (*pstatus!=Success) return;
 		/* Center2 */
                if (ScanFlExp(ps[++*pc], c1, pstatus)) {
                  CopyLine(c1,pib->Center[2].V,IO_size,0);
                  pib->Center[2].I = opt_onoff;
                } else if (*pstatus!=Success) return;
                break;
	case NSampleDistance:
		/* SampleDistance */
                if (ScanFlExp(ps[++*pc], c1, pstatus)) {
                  CopyLine(c1,pib->SampleDistance.V,IO_size,0);
                  pib->SampleDistance.I = opt_onoff;
                } else if (*pstatus!=Success) return;
                break;
        case NDetRot:
                /* DetRot1 */
                if (ScanFlExp(ps[++*pc], c1, pstatus)) {
                  CopyLine(c1,pib->DetRot[1].V,IO_size,0);
                  pib->DetRot[1].I = opt_onoff;
                } else if (*pstatus!=Success) return;
                /* DetRot2 */
                if (ScanFlExp(ps[++*pc], c1, pstatus)) {
                  CopyLine(c1,pib->DetRot[2].V,IO_size,0);
                  pib->DetRot[2].I = opt_onoff;
                } else if (*pstatus!=Success) return;
                /* DetRot3 */
                if (ScanFlExp(ps[++*pc], c1, pstatus)) {
                  CopyLine(c1,pib->DetRot[3].V,IO_size,0);
                  pib->DetRot[3].I = opt_onoff;
                } else if (*pstatus!=Success) return;
                break;
	case NBeamCenter:
		/* BeamCenter1 */
                if (ScanFlExp(ps[++*pc], c1, pstatus)) {
                  CopyLine(c1,pib->BeamCenter[1].V,IO_size,0);
                  pib->BeamCenter[1].I = opt_onoff;
                } else if (*pstatus!=Success) return;
 		/* BeamCenter2 */
                if (ScanFlExp(ps[++*pc], c1, pstatus)) {
                  CopyLine(c1,pib->BeamCenter[2].V,IO_size,0);
                  pib->BeamCenter[2].I = opt_onoff;
                } else if (*pstatus!=Success) return;
                break;
	case NBeamDistance:
		/* BeamDistance */
                if (ScanFlExp(ps[++*pc], c1, pstatus)) {
                  CopyLine(c1,pib->BeamDistance.V,IO_size,0);
                  pib->BeamDistance.I = opt_onoff;
                } else if (*pstatus!=Success) return;
                break;
        case NDetTilt:
                /* DetTilt1 */
                if (ScanFlExp(ps[++*pc], c1, pstatus)) {
                  CopyLine(c1,pib->DetTilt[1].V,IO_size,0);
                  pib->DetTilt[1].I = opt_onoff;
                } else if (*pstatus!=Success) return;
                /* DetTilt2 */
                if (ScanFlExp(ps[++*pc], c1, pstatus)) {
                  CopyLine(c1,pib->DetTilt[2].V,IO_size,0);
                  pib->DetTilt[2].I = opt_onoff;
                } else if (*pstatus!=Success) return;
                /* DetTilt3 */
                if (ScanFlExp(ps[++*pc], c1, pstatus)) {
                  CopyLine(c1,pib->DetTilt[3].V,IO_size,0);
                  pib->DetTilt[3].I = opt_onoff;
                } else if (*pstatus!=Success) return;
                break;
	  case NSearchList:
		/* SearchList */
		if (ScanLine(ps[++*pc], c1, pstatus)) {
		  CopyLine(c1,pib->SearchList.V,IO_size,0);
		  pib->SearchList.I = opt_onoff;
		} else if (*pstatus!=Success) return;
 		break;

	default:
		*pstatus = NoOption;
		return;
	}
	*pstatus = Success;
} /* DoImgOptions */

/*--------------------------------------------------------------------------
   DoOption :  Write option values to command and image blocks
  	return value	(o) : void
  	argv		(i) : pointer to argument string array
  	*parg_no      (i/o) : i: field number of previous argument
  			      o: incremented
  	blk_no		(i) : block number of corresponding block,
  			     -1  : CommandBlock
  			      0  : ImageBlock[0]
  			      n>0: ImageBlock[n]
  	opt_no		(i) : option number > 0
  	opt_onoff	(i) : >0 if "+" option
  			      <0 if "-" option
  	*pcb          (i/o) : command block
  	ib[]          (i/o) : image block array ib[ib_len]
        ib[0] : output image block
        ib[n] : input image n block
  	ib_len	(i) : dimension of ib
  	*pstatus	(o) : status : Success, ScanError, NoOption
  --------------------------------------------------------------------------*/
void DoOption ( char * argv[], int * parg_no,
		int blk_no, int opt_no, int opt_onoff,
		CmdBlk * pcb, ImgBlk ib[], int ib_len , int * pstatus )
{
  if (pcb->TestBit)
    printf("blk_no = %d, opt_no = %d, opt_onoff = %d\n",
           blk_no,opt_no,opt_onoff);


  if ((blk_no>=0) && (blk_no<ib_len))  /* image option */
    DoImgOption(argv, parg_no, opt_no, opt_onoff,
                &ib[blk_no], pstatus); 
  else if (blk_no==-1)   /* command option */
    DoCmdOption(argv, parg_no, opt_no, opt_onoff, pcb, pstatus); 
  else *pstatus = NoOption; /* unknown option block */

} /* DoOptions */

/*-------------------------------------------------------------------------
   ReadOptions : Read all options from the argument list
  -------------------------------------------------------------------------*/
void ReadOptions( char * argv[], int * parg_no,
CmdBlk * pcb, ImgBlk ib[], int * pstatus )
{
  char * msg;
  const char * filename;
  char ** pa;
  int ib_len;
  int blk_no;
  int opt_no;
  int opt_onoff;
  const char * RoutineName = "ReadOptions";
  int argno;

  if (pcb->TestBit) printf(" %s \n",RoutineName);

  /* Initialization */
  if (pcb->MainName == NULL) pcb->MainName = argv[0];
  if (pcb->argc == 0)
    while (argv[pcb->argc]) pcb->argc++;
  if (pcb->argv == NULL) pcb->argv = argv;
  if (pcb->parg_no == NULL) pcb->parg_no = parg_no;

  SetSaxsError( SOName, RoutineName, NULL );
  *pstatus = Success;

  /* init history lists */
  edf_history_new( SaxsHistoryKey );

  ib_len  = pcb->ImgBlkLen;
  while (FindOption( argv, parg_no, pcb,
         &blk_no, &opt_no, &opt_onoff, pstatus )) {
    DoOption( argv, parg_no, blk_no, opt_no, opt_onoff,
              pcb, ib, ib_len , pstatus );

    if (*pstatus!=Success) {
      /* Show the last two arguments, if they are accessible */

      if ((argv[*parg_no-1])) {
        if (!(argv[*parg_no])) {
          msg=(char *) malloc( 11 + strlen(argv[*parg_no-1]) );
          if ( (msg) != (char *) NULL )
            sprintf(msg,"    %s ... ", argv[*parg_no-1]);
          SetSaxsErrorMessage ( msg );
          if (msg) free(msg);
        } else {
          msg=(char *) malloc( 11 +
            strlen(argv[*parg_no-1])+strlen(argv[*parg_no]) );
          if ( (msg) != (char *) NULL )
            sprintf(msg,"    %s %s ... ",
                    argv[*parg_no-1],argv[*parg_no]);
          SetSaxsErrorMessage ( msg );
          if (msg) free(msg);
        }
      }
      free(pcb->co);free(pcb->io);return;
    } /* if !=Success */

  } /* while */	

  /* look for start of filename */
  if ( (filename = strrchr(argv[0],DirSeparator[0])) ) filename++; 
  else filename = argv[0];

  /* add all arguments from 0 to *parg_no to history */
  edf_history_argv( SaxsHistoryKey, filename );
  for (argno=1;argno<=*parg_no;argno++) {
    edf_history_argv( SaxsHistoryKey, argv[argno] );
  }

  if (pcb->TestBit) {
    /* write parameter list */
    pa = argv;
    printf("List :");
    while (*++pa!=NULL) printf(" %s", *pa);
    printf("\n");
    PrintOptions("CommandOption",pcb->co,IO_CmdOptLen);
    PrintOptions("ImageOption",pcb->io,IO_ImgOptLen);
    PrintBlocks ( pcb, ib );
  }
  /* Release memory for option definitions */
  free(pcb->io);
  // The command options must be available until the end of the program
  // PB 2003-05-31
  // free(pcb->co); 

} /* ReadOptions */

/*****************************************************************************/
