/***************************************************************************/
/* Written 1994++ by Peter Boesecke                                        */
/* Copyright (C) 2011 European Synchrotron Radiation Facility              */
/*                       Grenoble, France                                  */
/*                                                                         */
/*    Principal authors: Peter Boesecke  (boesecke@esrf.eu)                */
/*                                                                         */
/*    This program is free software: you can redistribute it and/or modify */
/*    it under the terms of the GNU General Public License as published by */
/*    the Free Software Foundation, either version 3 of the License, or    */
/*    (at your option) any later version.                                  */
/*                                                                         */
/*    This program is distributed in the hope that it will be useful,      */
/*    but WITHOUT ANY WARRANTY; without even the implied warranty of       */
/*    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the        */
/*    GNU General Public License for more details.                         */
/*                                                                         */
/*    You should have received a copy of the GNU General Public License    */
/*    along with this program.  If not, see <http://www.gnu.org/licenses/>.*/
/***************************************************************************/
/*+++
NAME

  SaxsError --- Error handling

SYNOPSIS

  char *SaxsErrorVersion( void );

  void SetSaxsError( const char * ModuleName,
                     const char * RoutineName, 
                     const char * Message ),
  void SetSaxsErrorMessage ( const char * Message );

  int ReportSaxsStatus(int Status, int ExitFlag);

DESCRIPTION

  SaxsErrorVersion    returns the version string of the module
  SetSaxsError        sets the error block variables
  SetSaxsErrorExternal sets ExternalError and the ReportError function
  SetSaxsErrorMessage sets only the error message
  ReportSaxsStatus    if status is set, it writes the corresponding
                      error message to ERROUT and reinitializes the 
                      error block

HISTORY

  1996-10-13 PB V1.0 Extracted from SaxsInput
  1996-10-15 PB V1.1 SaxsErrorVersion
  2000-01-07 PB V1.3 SizeMismatch, FileWriteError
  2000-03-19 PB V1.4 OutOfRange
  2001-01-01 PB V1.41 
  2001-07-06 PB V1.42 FileAlreadyOpen
  2001-07-10 PB V1.43 FileMustBeOpen
  2001-08-19 PB V1.44 HeaderNotInitialized
  2002-02-12 PB V1.45 UnsupportedDataType
  2002-05-25 PB V1.46 ERROUT
  2004-10-31 PB V1.47 TypeConversionFailed, StoppedAfterHelp,
                      CannotFindHeader, FileNameError,
                      ReportSaxsError, ReportSaxsStatus, ReportInputStatus
                      return status
  2004-12-11 PB V1.48 NotWaxsProjection 
  2005-02-12 PB V1.49 ScanEmpty
  2005-07-11 PB V1.50 SetSaxsErrorMessage added
                      SaxsErrorRoot added
  2005-07-14 PB V1.51 InitSaxsError replaced by SaxsError_init. 
                      It is automatically called.
                      Argument seb replaced by SaxsErrorBlock
                      ReportSaxsError removed, PrintSaxsError removed
                      ReportInputStatus only private function,
                      SetSaxsErrorExternal added,
                      SetSaxsError: status, ExternalError and ReportError 
                      removed from argument list. 
                      ExternalError and ReportError are now reset to 0,
                      status is removed from SaxsErrBlk.
  2005-08-04 PB V1.52 ERROUT set to stderr
  2010-03-14 PB V1.53 NotSaxsReference, NotNormalReference, 
                      ReferenceNotSupported
  2011-06-08 PB V1.54 InvalidOrientation
  2015-10-03 PB V1.55 AlreadyOpen, FileMustBeOpen
  2017-02-28 PB V1.56 SaxsErrBlk: char * => const char * and depending
                      functions: SetSaxsErrorExternal, SetSaxsErrorMessage
  2017-10-16 PB V1.57 HeaderNotRead
---*/

/***************************************************************************/
/* SaxsError.c                                                             */
/***************************************************************************/
# define SE_Version "SaxsError V1.56 Peter Boesecke 2017-02-28"

# include <errno.h>
# include <stdio.h>
# include <stdlib.h>
# include <string.h>
# include "SaxsError.h"
# include "edfio.h"

/***************************************************************************/
/* Private type definition                                                 */
/***************************************************************************/
typedef struct SE_Block {
  char * ModuleName;  /* pointer to module name */
  char * RoutineName; /* pointer to routine name */
  char * Message;     /* pointer to an info string */
  int  ExternalError; /* external error */
  const char *(*ReportError) (int ExternalError );
                      /* pointer to external error report function */
} SaxsErrBlk;

/***************************************************************************
* SaxsError Static Variables                                               *
***************************************************************************/
static int SAXSERROR_init = 0;                               /* init flag */
static SaxsErrBlk SaxsErrorBlock;                     /* saxs error block */

/*---------------------------------------------------------------------------
NAME

  SaxsError_init --- Initializes the module

SYNOPSIS

  int SaxsError_init ( void );

DESCRPTION

  Initializes the error block. It should be called by routines
  that access directly SaxsErrorBlock, if SAXSERROR_init is not set.

RETURN VALUE

  success:0, error:-1

---------------------------------------------------------------------------*/
int SaxsError_init ( void )
{
  SaxsErrorBlock.ModuleName    = (char *) NULL;
  SaxsErrorBlock.RoutineName   = (char *) NULL;
  SaxsErrorBlock.Message       = (char *) NULL;
  SaxsErrorBlock.ExternalError = 0;
  SaxsErrorBlock.ReportError   = NULL;

  SAXSERROR_init = 1;

  return(0);

} /* SaxsError_init */

/*---------------------------------------------------------------------------
NAME

   SaxsError_newstr --- allocate memory and copy a character string to it

SYNOPSIS

   char * SaxsError_newstr( const char * string );

DESCRIPTION
  Allocates strlen(string)+1 bytes of memory and copies string to it.
  In case of success the pointer to the allocated memory is returned. The
  null pointer is returned in case of an error.
  If string is the NULL pointer the NULL pointer is returned.

RETURN VALUE
  Returns the pointer to the allocated string or (char *) NULL in case
  of an error.
---------------------------------------------------------------------------*/
char * SaxsError_newstr( const char * string )
{ char * newstring;

  if (!string) return( (char *) NULL );
  if (!(newstring = (char *) malloc(strlen(string)+1))) return((char *) NULL);
  (void) strcpy(newstring,string);

  return( newstring );

} /* SaxsError_newstr */

/****************************************************************************
1 SaxsErrorVersion

2 DESCRIPTION
  Returns a pointer to the version string.
  char * SaxsErrorVersion ( void )

2 HISTORY
  14-Oct-1996 PB
****************************************************************************/
char * SaxsErrorVersion ( void )
{ static char * Version = SE_Version;
  return ( Version );
} /* SaxsErrorVersion */

/****************************************************************************
1 ReportInputStatus

2 DESCRIPTION
   int Status        (i) : SAXS status
   int ExitFlag      (i) : 1 : if (Status!=Success) exit;
		           0 : do not exit
2 HISTORY
  11-Oct-1996 PB
****************************************************************************/
int ReportInputStatus(int Status, int ExitFlag)
{
  const char * cout;

  switch (Status) {
        case Failed:
             cout = "Failed: Routine failed"; break;
	case Success:
             cout = "Success: Normal successful operation"; break;
	case Abort:
             cout = "Abort: Input aborted"; break;
	case SkipInput:
             cout = "SkipInput: Input skipped"; break;
	case ScanError:
             cout = "ScanError: Input conversion error"; break;
	case ScanEmpty:
             cout = "ScanEmpty: Unexpected end of argument list"; break;
	case NoDefault:
             cout = "NoDefault: No default possible"; break;
	case NoOption:
             cout = "NoOption: Unknown option"; break;
	case NoFloatNumber:
             cout = "NoFloatNumber: This is not a float number"; break;
	case NoIntegerNumber:
             cout = "NoIntegerNumber: This is not an integer number"; break;
	case InvalidOrientation:
             cout = "InvalidOrientation: Invalid orientation"; break;
	case BadParenthesis:
             cout = "BadParenthesis: Wrong number of parentheses"; break;
	case DivByZero:
             cout = "DivByZero: Division by zero"; break;
	case NoFlag:
             cout = "NoFlag: No flag (true|false|yes|no)"; break;
	case CommaExpected:
             cout = "CommaExpected"; break;
	case FloatingPointError:
             cout = "FloatingPointError"; break;
	case DomainError:
             cout = "DomainError"; break;
	case IntegerOverflow:
             cout = "IntegerOverflow"; break;
	case UnknownFloatFunction:
             cout = "UnknownFloatFunction"; break;
	case UnknownIntegerFunction:
             cout = "UnknownIntegerFunction"; break;
        case WrongImageNumber:
             cout = "WrongImageNumber"; break;
        case NotEnoughBlocks:
             cout = "NotEnoughBlocks"; break;
        case NotEnoughMemoryAvailable:
             cout = "NotEnoughMemoryAvailable"; break;
        case FileNotFound:
             cout = "FileNotFound"; break;
        case ImageProtected:
             cout = "ImageProtected"; break;
        case FileProtected:
             cout = "FileProtected"; break;
        case FileOpeningError:
             cout = "FileOpeningError"; break;
        case FileAlreadyOpen:
             cout = "FileAlreadyOpen"; break;
        case FileMustBeOpen:
             cout = "FileMustBeOpen"; break;
        case HeaderNotInitialized:
             cout = "HeaderNotInitialized"; break;
        case FilePositioningError:
             cout = "FilePositioningError"; break;
        case FileReadError:
             cout = "FileReadError"; break;
        case FileClosingError:
             cout = "FileClosingError"; break;
        case KeywordMissing:
             cout = "KeywordMissing"; break;
        case UnknownDataType:
             cout = "UnknownDataType"; break;
        case UnsupportedDataType:
             cout = "UnsupportedDataType"; break;
        case UnknownUnit:
             cout = "UnknownUnit"; break;
        case SizeMismatch:
             cout = "SizeMismatch"; break;
        case FileWriteError:
             cout = "FileWriteError"; break;
        case OutOfRange:
             cout = "OutOfRange"; break;
        case TypeConversionFailed:
             cout = "TypeConversionFailed"; break;
        case StoppedAfterHelp:
             cout = "StoppedAfterHelp"; break;
        case CannotFindHeader:
             cout = "CannotFindHeader"; break;
        case FileNameError:
             cout = "FileNameError"; break;
        case NotWaxsProjection:
             cout = "NotWaxsProjection"; break;
        case NotNormalReference:
             cout = "NotNormalReference"; break;
        case NotSaxsReference:
             cout = "NotSaxsReference"; break;
        case ReferenceNotSupported:
             cout = "ReferenceNotSupported"; break;
        case ExpressionWriteError:
             cout = "ExpressionWriteError"; break;
        case NoVariable:
             cout = "NoVariable"; break;
        case VariableError:
             cout = "VariableError"; break;
        case ProgramError:
             cout = "ProgramError"; break;
        case NoInstruction:
             cout = "NoInstruction"; break;
        case NoAccumulator:
             cout = "NoAccumulator"; break;
        case HeaderNotRead:
             cout = "HeaderNotRead"; break;
	case ExternalErrorReport:
             cout = "ExternalErrorReport"; break;

 	default: cout = "..."; break;
 	}
  fprintf (ERROUT,"Status = %d -- %s\n", Status, cout);
  if (ExitFlag && (Status!=Success)) exit ( Status );

  return( Status );

} /* ReportInputStatus */

/****************************************************************************
1 ReportSaxsStatus

2 DESCRIPTION

   If Status is not Success it writes a status report using Status and 
   SaxsErrorBlock and frees all allocated memories.

   int Status        (i) : SAXS status
   char * ModuleName (i) : name of module
   char * RoutineName(i) : name of routine
   char * Message    (i) : message string
   int status        (o) : status variable
   int ExternalError (i) : external error number, only valid and used when
                           Status==ExternalErrorReport.
   (const char *) (*ReportError) ( int ExternalError ) : error function
                           which returns an error message according to
                           ExternalError, only used when
                           Status==ExternalErrorReport.
   int ExitFlag      (i) : 1 : if (Status!=Success) exit;
		           0 : do not exit

2 HISTORY
  11-Oct-1996 PB
****************************************************************************/
int ReportSaxsStatus(int Status, int ExitFlag)
{ 
  if (!SAXSERROR_init) SaxsError_init();

  if (Status==Success) return( Status );

  fprintf(ERROUT,"\n");

  if (Status!=StoppedAfterHelp) {
    if (SaxsErrorBlock.ModuleName) {
      fprintf(ERROUT,"MODULE   : %s\n",SaxsErrorBlock.ModuleName);
      free(SaxsErrorBlock.ModuleName); SaxsErrorBlock.ModuleName = NULL;
    }
    if (SaxsErrorBlock.RoutineName) {
      fprintf(ERROUT,"ROUTINE  : %s\n",SaxsErrorBlock.RoutineName);
      free(SaxsErrorBlock.RoutineName); SaxsErrorBlock.RoutineName = NULL;
    }
    if (SaxsErrorBlock.Message) {
      fprintf(ERROUT,"%s\n",SaxsErrorBlock.Message);
      free(SaxsErrorBlock.Message); SaxsErrorBlock.Message = NULL;
    }
  }

  if (Status==ExternalErrorReport) {
    if (SaxsErrorBlock.ReportError)
      fprintf(ERROUT,"%s\n",
        SaxsErrorBlock.ReportError ( SaxsErrorBlock.ExternalError ));
    else fprintf(ERROUT,"%s\n",
           edf_report_data_error ( SaxsErrorBlock.ExternalError ));
  }

  Status = ReportInputStatus( Status, ExitFlag );

  SaxsError_init ( );
  
  return ( Status );

} /* ReportSaxsStatus */

/****************************************************************************
Name

  SetSaxsErrorExternal --- set external error

Synopsis
  void SetSaxsErrorExternal ( int ExternalError, 
                              char *  (*ReportError) (int ) ) 

  int ExternalError        (i) : external error number
  (const char *) (*ReportError) ( int ExternalError ) :
                            pointer to error report function
Description
  The routine sets the parameters ExternalError and ReportError in
  SaxsErrorBlock. All other parameter are left unchanged. It should be
  used after a call to a function that returns its own error code,
  e.g. 
    edf_ ... ( ..., &ErrorValue ,... ).
    SetSaxsErrorExternal ( ErrorValue, ReportError )
  ReportError is a function that returns a string with an error message. 


****************************************************************************/
void SetSaxsErrorExternal ( int ExternalError, 
                            const char *(*ReportError) (int ) ) 
{
  SaxsErrorBlock.ExternalError = ExternalError;
  SaxsErrorBlock.ReportError   = ReportError;

} /* SetSaxsErrorExternal */

/****************************************************************************
1 SetSaxsErrorMessage

2 DESCRIPTION 
  void SetSaxsErrorMessage ( const char * Message );

  Allocates memory, copies Message to it and links it with SaxsErrorBlock.Message.
  If the pointer SaxsErrorBlock.Memory is not NULL at entry, the specified memory is 
  freed before new memory is allocated.

   const char * Message     (i) : pointer to message

2 HISTORY
  11-Jul-2005 PB
****************************************************************************/
void SetSaxsErrorMessage ( const char * Message )
{ 
  if (!SAXSERROR_init) SaxsError_init();

  if (SaxsErrorBlock.Message) free(SaxsErrorBlock.Message);
    if (Message) SaxsErrorBlock.Message = SaxsError_newstr(Message);
    else SaxsErrorBlock.Message = NULL;

} /* SetSaxsErrorMessage */

/****************************************************************************
1 SetSaxsError

2 DESCRIPTION
  void SetSaxsError( const char * ModuleName,
                     const char * RoutineName, 
                     const char * Message );

   const char * ModuleName  (i) : pointer to module name
   const char * RoutineName (i) : pointer to routine name
   const char * Message     (i) : pointer to message

   ExternalError and ReportError are reset to 0.

2 HISTORY
  11-Oct-1996 PB
  14-Jul-2005 PB
****************************************************************************/
void SetSaxsError( const char * ModuleName,
                   const char * RoutineName, 
                   const char * Message )
{ 
  if (!SAXSERROR_init) SaxsError_init();

  SaxsErrorBlock.ModuleName    = SaxsError_newstr( ModuleName );
  SaxsErrorBlock.RoutineName   = SaxsError_newstr( RoutineName );
  SetSaxsErrorMessage( Message );
  SetSaxsErrorExternal ( 0, NULL );

} /* SetSaxsError */

/****************************************************************************/
