/***************************************************************************/
/* Written 1994++ by Peter Boesecke                                        */
/* Copyright (C) 2011 European Synchrotron Radiation Facility              */
/*                       Grenoble, France                                  */
/*                                                                         */
/*    Principal authors: Peter Boesecke  (boesecke@esrf.eu)                */
/*                                                                         */
/*    This program is free software: you can redistribute it and/or modify */
/*    it under the terms of the GNU General Public License as published by */
/*    the Free Software Foundation, either version 3 of the License, or    */
/*    (at your option) any later version.                                  */
/*                                                                         */
/*    This program is distributed in the hope that it will be useful,      */
/*    but WITHOUT ANY WARRANTY; without even the implied warranty of       */
/*    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the        */
/*    GNU General Public License for more details.                         */
/*                                                                         */
/*    You should have received a copy of the GNU General Public License    */
/*    along with this program.  If not, see <http://www.gnu.org/licenses/>.*/
/***************************************************************************/
/****************************************************************************
*                                                                           *
* Calculation of array indices                                              *
* Aij = A[i,j] = A[i*J+j] = A[i][j]                                         *
*                                                                           *
* double RocaA[NParams*ndata];                                              *
* double RocaATA[NParams*NParams];                                          *
* double RocaB[ndata];                                                      *
* double RocaATB[NParams];                                                  *
*                                                                           *
* Input                                                                     *
* Par[NParams],ParMode[NParams],Angle[ndata],I1[ndata],I2[ndata]            *
*                                                                           *
* Principle of Calculation                                                  *
*                                                                           *
* Par=ParStart;                                                             *
* Loop                                                                      *
*   Calculate Diff[ndata]=RocaB[ndata](Par,Angle,I1,I2)                     *
*   Approximate Par (according to ParMode)                                  *
*   RocaB[ndata] ~ RocaA[NParams*ndata] * DiffPar[NParams]                  *
*   RocaATB = RocaATA * DiffPar => DiffPar                                  *
*   Par'=Par+DiffPar                                                        *
*                                                                           *
* History                                                                   *
* 2010-07-28 PB transpose_mul_weight added (for weighted refinement)        *
*               RocaOptimize: Weight added and transpose_mul -> _mul_weight *
* 2010-08-05 PB Rotation array addressing adapted to agree with rot3d,      *
*               tilt3d and waxs                                             *
* 2010-08-18 PB calculate ParDev                                            *
* 2015-10-08 PB roca_libversion() added                                     *
* 2015-12-08 PB fprint_vector, roca_save_step:                              *
*               Reducing warnings in print statements when switching        *
*               between 32-bit and 64-bit system by transforming            *
*               size_t variables with SIZE_T to unsigned long               *
*               before printing as %lu. %lz is not known by all             *
*               compilers.                                                  *
* 2017-05-21 PB update for ioalloc                                          *
****************************************************************************/

/***************************************************************************
* Include                                                                  *
***************************************************************************/
# include "roca_base.h"

/***************************************************************************
* Internal                                                                 *
***************************************************************************/

/***************************************************************************
* Portables Print Format for size_t variable (to be printed with %lu)      *
***************************************************************************/
# define SIZE_T (unsigned long)

static char ROCA_Usage[ROCABUFLEN];

/****************************************************************************
* Private Structures                                                        *
****************************************************************************/
typedef struct RocaOrdered {
  int Index[NParams]; // index corresponding to RocaParam number
  const char* Names[NParams]; // RocaParamNames
  double Par[NParams]; // parameters
  double ParDev[NParams]; // parameter deviations
  const char* StatNames[NStats]; // RocaStatNames
  double Stat[NStats]; // statistics
  size_t nopt; // number of parameters to optimize
} RO;

/****************************************************************************
* Private Variables and Numbers                                             *
****************************************************************************/
static const char *RocaParamNames[] = {
  "pix1","pix2","cen1","cen2", "dis","rot1","rot2","rot3",
  (const char*) NULL }; // last element must be NULL

static int RocaParamDefault[] = {
   0,     0,     1,     1,      1,    1,     1,     0 }; // 1 == refine

static const char *RocaStatNames[] = {
  "stdev", "mindev", "maxdev", "total", "ndata", "nopt",
  (const char*) NULL }; // last element must be NULL

static int ROCA_level = 0;
static int ROCA_debug = 0;

/****************************************************************************
* Routines                                                                  *
****************************************************************************/

/*+++------------------------------------------------------------------------
NAME

   roca_libversion

SYNOPSIS

   const char *roca_libversion ( void )

DESCRIPTION
   Returns a pointer to RVERSION, if defined, otherwise an empty string

RETURN VALUE
   Pointer to the RVERSION string or an empty string 
-------------------------------------------------------------------------+*/
const char *roca_libversion   ( void )
{
# ifdef RVERSION
  return ( RVERSION );
# else
  return ( "" );
# endif
} /* roca_libversion */

int fprint_debug( FILE *out )
{ fprintf(out,"debug      = 0x%x\n", ROCA_debug);
  fprintf(out,"verbose    = %d\n", ROCA_debug&ROCA_VERBOSE?1:0);
  fprintf(out,"level      = %d\n", ROCA_level);
  fprintf(out,"showdata   = %d\n", ROCA_debug&ROCA_SHOWDATA?1:0);
  fprintf(out,"showtemp   = %d\n", ROCA_debug&ROCA_SHOWTEMP?1:0);
  fprintf(out,"rocadebug  = %d\n", ROCA_debug&ROCA_DEBUG?1:0);
  fprintf(out,"listdebug  = %d\n", ROCA_debug&ROCA_LISTDEBUG?1:0);
  fprintf(out,"savesteps   = %d\n", ROCA_debug&ROCA_SAVESTEPS?1:0);
  return(0);
} // fprint_debug

int roca_debug_set( int debug )
{ ROCA_debug = debug;
  ROCA_level = (ROCA_debug&ROCA_LEVEL)>>1;

  if (ROCA_debug&ROCA_DEBUG) fprint_debug( stdout );
  return(0);
} // roca_debug_set

const char *roca_usage2str( void )
{ sprintf(ROCA_Usage,"verbose=0x%x,level=0x%x,data=0x%x,temp=0x%x,roca=0x%x,list=0x%x,savesteps=0x%x",
    ROCA_VERBOSE, ROCA_LEVEL, ROCA_SHOWDATA,
    ROCA_SHOWTEMP, ROCA_DEBUG, ROCA_LISTDEBUG,
    ROCA_SAVESTEPS);
  return(ROCA_Usage);
} // roca_usage2str

int fprint_debug_usage( FILE *out )
{ fprintf(out,"verbose=0x%x,level=0x%x,data=0x%x,temp=0x%x,roca=0x%x,list=0x%x,savesteps=0x%x\n",
    ROCA_VERBOSE, ROCA_LEVEL, ROCA_SHOWDATA,
    ROCA_SHOWTEMP, ROCA_DEBUG, ROCA_LISTDEBUG,
    ROCA_SAVESTEPS);
  return(0);
} // fprint_debug_usage

int fprint_vector( FILE *out, size_t ndata, double V[], const char * label )
{ size_t num;
  // show data
  fprintf(out,"%14s %14s\n","Num",label);
  for (num=0;num<ndata;num++) {
    fprintf(out,"%14lu %14lg\n",SIZE_T num,V[num]);
  }
  fprintf(out,"\n");

  return(0);

} // fprint_input_data

int fprint_matrix( FILE *out, size_t nrows, size_t ncols, double M[], const char * label )
{ size_t row, col;
  // show data
  fprintf(out,"%14s \n",label);
  for (row=0;row<nrows;row++) {
    for (col=0;col<ncols;col++)
      fprintf(out,"%14g ",M[col*nrows+row]); // M[col][row]
    fprintf(out,"\n");
  }
  printf("\n");

  return(0);

} // fprint_matrix

int fprint_matrix_3d( FILE *out, double M[3][3], const char * label )
{ size_t nrows=3, ncols=3;
  return( fprint_matrix( out, nrows, ncols, &(M[0][0]), label ) );
} // fprint_matrix_3d

char * sprint_parameters( char * buffer, size_t bufsiz, double Par[] )
{ char *pb;
  size_t bs, bd;
  int num;

  pb=buffer;
  bs=bufsiz;

  for (num=0;num<NParams;num++) {
    bd=snprintf(pb,bs,"%12lg ",Par[num]);
    pb+=bd;
    bs-=bd;
  }

  // terminate in all cases
  pb=buffer+sizeof(char)*(bufsiz-1);
  *pb='\0';

  return(buffer);

} // sprint_parameters

char * sprint_statistics( char * buffer, size_t bufsiz, double Stat[] )
{ char *pb;
  size_t bs, bd;
  int num;

  pb=buffer;
  bs=bufsiz;

  for (num=0;num<NStats;num++) {
    bd=snprintf(pb,bs,"%12lg ",Stat[num]);
    pb+=bd;
    bs-=bd;
  }

  // terminate in all cases
  pb=buffer+sizeof(char)*(bufsiz-1);
  *pb='\0';

  return(buffer);

} // sprint_statistics

char * sprint_beam_parameters( char * buffer, size_t bufsiz, double Par[] )
{ double beam_center[2], beam_distance, tilt[3];
  int num;
  double *ParBeam=NULL;
  int *IParBeam=NULL;
  char *pb;
  size_t bs, bd;

  if ( !(ParBeam=(double *) MALLOC (sizeof(double)*NParams)) )
    goto sprint_beam_parameters_error;
  if ( !(IParBeam=(int *) MALLOC (sizeof(int)*NParams)) )
    goto sprint_beam_parameters_error;

  for (num=0;num<NParams;num++) IParBeam[num]=0;

  ParBeam[Pix1] = Par[Pix1]; IParBeam[Pix1] = 1;
  ParBeam[Pix2] = Par[Pix2]; IParBeam[Pix2] = 1;

  if ( !RocaBeamCenter( beam_center,  Par) ) {
    ParBeam[Cen1] = beam_center[0]; IParBeam[Cen1] = 1;
    ParBeam[Cen2] = beam_center[1]; IParBeam[Cen2] = 1;
  }

  if ( !RocaBeamDistance( &beam_distance, Par ) ) {
    ParBeam[Dis] = beam_distance; IParBeam[Dis] = 1;
  }

  if ( !RocaTilt(tilt, Par ) ) {
    ParBeam[Rot1] = tilt[0]; IParBeam[Rot1] = 1;
    ParBeam[Rot2] = tilt[1]; IParBeam[Rot2] = 1;
    ParBeam[Rot3] = tilt[2]; IParBeam[Rot3] = 1;
  }

  pb=buffer;
  bs=bufsiz;

  for (num=0;num<NParams;num++) {
    if (IParBeam[num]) bd=snprintf(pb,bs,"%12lg ",ParBeam[num]);
    else bd=snprintf(pb,bs,"%12s ","-");
    pb+=bd;
    bs-=bd;
  }

  // terminate in all cases
  pb=buffer+sizeof(char)*(bufsiz-1);
  *pb='\0';

  FREE(ParBeam);
  FREE(IParBeam);

  return(buffer);

sprint_beam_parameters_error:

  FREE(ParBeam);
  FREE(IParBeam);

  return(NULL);

} // sprint_beam_parameters

char * sprint_ordered_par( char *buffer, size_t bufsiz,  RO* Ordered )
{ int i,n;
  char *pb;
  size_t bs, bd;

  pb=buffer;
  bs=bufsiz;

  for (i=0;i<NParams;i++) {
    n=Ordered->Index[i];
    if (n<Ordered->nopt)
      bd=snprintf(pb,bs,"%s=%lg ",Ordered->Names[n],Ordered->Par[n]);
    else
      bd=snprintf(pb,bs,"%s=[%lg] ",Ordered->Names[n],Ordered->Par[n]);
    pb+=bd;
    bs-=bd;
  }

  // terminate in all cases
  pb=buffer+sizeof(char)*(bufsiz-1);
  *pb='\0';

  return(buffer);

} // sprint_ordered_par

char * sprint_ordered_dev( char *buffer, size_t bufsiz,  RO* Ordered )
{ int i,n;
  char *pb;
  size_t bs, bd=0;

  pb=buffer;
  bs=bufsiz;

  for (i=0;i<NParams;i++) {
    n=Ordered->Index[i];
    if (n<Ordered->nopt) {
      if (Ordered->ParDev[n]>=0) 
        bd=snprintf(pb,bs,"%sDev=%lg ",Ordered->Names[n],Ordered->ParDev[n]);
    } else {
      if (Ordered->ParDev[n]>=0) 
        bd=snprintf(pb,bs,"%sDev=[%lg] ",Ordered->Names[n],Ordered->ParDev[n]);
    }
    pb+=bd;
    bs-=bd;
  }

  // terminate in all cases
  pb=buffer+sizeof(char)*(bufsiz-1);
  *pb='\0';

  return(buffer);

} // sprint_ordered_dev

int fprint_ordered( FILE *out, RO* Ordered )
{ char buffer[ROCABUFLEN];
  char *str;

  fprintf(out,"%s\n",sprint_ordered_par( buffer, ROCABUFLEN,  Ordered ));
  str=sprint_ordered_dev( buffer, ROCABUFLEN,  Ordered );
  if (strlen(str)>0) fprintf(out,"%s\n",str);

  return(0);

} // fprint_ordered

/*+++------------------------------------------------------------------------
NAME

  order_Par --- Orders the parameters in such a way that the 
                first parameters are all optimized (ParMode!=0)
                and the following are not optimized (ParMode==0).


SYNOPSIS

  size_t order_Par ( RO* Ordered, double Par[NParams], int ParMode[NParams] );

DESCRIPTION

  If ParMode==NULL the parameters are just copied from Par to Ordered
  ParDev and Stat are just initialized.

RETURN VALUE

  Number of parameters to optimize

----------------------------------------------------------------------------*/
size_t order_Par ( RO* Ordered, double Par[NParams], int ParMode[NParams] )
{ int i, iparam, istat;

  if (ROCA_debug&ROCA_DEBUG) fprintf( stdout, "order_Par\n" );

  Ordered->nopt = 0;

  i=0;
  if (ParMode) { // sort
    // Sort first parameters to optimize
    for (iparam=0;iparam<NParams;iparam++) {
      if (ParMode[iparam]) {
        Ordered->Index[iparam]=i;
        Ordered->Names[i]=RocaParamNames[iparam];
        Ordered->Par[i]=RocaParamDefault[iparam]; // dummy line
        Ordered->Par[i]=Par[iparam];
        Ordered->ParDev[i]=-1.0;
        i++;
        Ordered->nopt = i;
      }
    }
    // Append fixed parameters to the list
    for (iparam=0;iparam<NParams;iparam++) {
      if (!ParMode[iparam]) {
        Ordered->Index[iparam]=i;
        Ordered->Names[i]=RocaParamNames[iparam];
        Ordered->Par[i]=Par[iparam];
        Ordered->ParDev[i]=-1.0;
        i++;
      }
    }
  } else { // do not sort, just copy
    // Copy all parameters to the list
    for (iparam=0;iparam<NParams;iparam++) {
      Ordered->Index[iparam]=i;
      Ordered->Names[i]=RocaParamNames[iparam];
      Ordered->Par[i]=Par[iparam];
      Ordered->ParDev[i]=-1.0;
      i++;
      Ordered->nopt = i;
    }
  } // if (ParMode)

  // Initialize Stat
  for (istat=0;istat<NStats;istat++) {
    Ordered->StatNames[i]=RocaStatNames[i];
    Ordered->Stat[istat]=-1.0;
  }

  return( Ordered->nopt );

} // order_Par

/*+++------------------------------------------------------------------------
NAME

  reorder_Par --- Redistributes Ordered->Par to Par


SYNOPSIS

  size_t reorder_Par ( RO* Ordered, 
                       double Par[NParams], double ParDev[NParams] );

DESCRIPTION

RETURN VALUE

  Number of parameters to optimize

----------------------------------------------------------------------------*/
size_t reorder_Par ( RO* Ordered, double Par[NParams], double ParDev[NParams], 
                     double Stat[NStats] )
{ int i;

  if (ROCA_debug&ROCA_DEBUG) fprintf( stdout, "reorder_Par\n" );
  for (i=0;i<NParams;i++) {
    if (Par) Par[i] = Ordered->Par[Ordered->Index[i]];
    if (ParDev) ParDev[i] = Ordered->ParDev[Ordered->Index[i]];
  }

  // Copy Stat
  if (Stat) for (i=0;i<NStats;i++) 
    Stat[i] = Ordered->Stat[i];
 
  return( Ordered->nopt );

} // reorder_Par

/*+++------------------------------------------------------------------------
NAME
  mat_mul --- product of two matrices

SYNOPSIS

  void mat_mul( double Out[L][N], double A[M][N], double B[L][M], 
                int L, int M, int N )

DESCRIPTION

  Out[L][N] = A[M][N]*B[L][M] | summed over M

RETURN VALUE
  none

----------------------------------------------------------------------------*/
void mat_mul ( double Out[], double A[], double B[], int L, int M, int N )
{ int l, m, n;
  if (ROCA_debug&ROCA_DEBUG) fprintf( stdout, "mat_mul\n" );
    for (n=0;n<N;n++)
      for (l=0;l<L;l++) {
        Out[l*N+n]=0.0; // Out[l][n] = 0.0
        for  (m=0;m<M;m++)
          Out[l*N+n]+=A[m*N+n]*B[l*M+m]; // Out[l][n] += A[m][n]*B[l][m]
      }
  return;
} // mat_mul 

/*+++------------------------------------------------------------------------
NAME
  transpose_mul --- product of two matrices, the first one is transposed

SYNOPSIS

  void transpose_mul( double Out[L][N], double A[N][M], double B[L][M], 
                      int L, int M, int N )

DESCRIPTION

  Out[L][N] = Transpose(A)[M][N]*B[L][M] | summed over M

RETURN VALUE
  none

----------------------------------------------------------------------------*/
void transpose_mul ( double Out[], double A[], double B[], int L, int M, int N )
{ int l, m, n;
  if (ROCA_debug&ROCA_DEBUG) fprintf( stdout, "transpose_mul\n" );
    for (n=0;n<N;n++)
      for (l=0;l<L;l++) {
        Out[l*N+n]=0.0; //  Out[l][n] = 0.0
        for  (m=0;m<M;m++)
          Out[l*N+n]+=A[n*M+m]*B[l*M+m]; // Out[l][n] += A[n][m]*B[l][m]
      }
  return;
} // transpose_mul 

/*+++------------------------------------------------------------------------
NAME
  transpose_mul_weight --- weighted product of matrices, first is transposed

SYNOPSIS

  void transpose_mul_weight( double Out[L][N], double A[N][M], double B[L][M],
                             double W[M], int L, int M, int N )

DESCRIPTION

  Out[L][N] = Transpose(A)[M][N]*B[L][M]*W[M] | summed over M

RETURN VALUE
  none

----------------------------------------------------------------------------*/
void transpose_mul_weight ( double Out[], double A[], double B[], double W[],
                           int L, int M, int N )
{ int l, m, n;
  if (ROCA_debug&ROCA_DEBUG) fprintf( stdout, "transpose_mul_weight\n" );
    for (n=0;n<N;n++)
      for (l=0;l<L;l++) {
        Out[l*N+n]=0.0; // Out[l][n] = 0.0
        for  (m=0;m<M;m++)
          Out[l*N+n]+=A[n*M+m]*B[l*M+m]*W[m]; // Out[l][n]+=A[n][m]*B[l][m]*W[m]
      }
  return;
} // transpose_mul_weight

/*+++------------------------------------------------------------------------
NAME

  vec_mul --- multiplication of a NxN matrix with a Nd vector

SYNOPSIS

  void vec_mul ( double VOut[N], double A[N][N], double V[N] )

DESCRIPTION

  VOut[N] = A[N][N]*V[N]


RETURN VALUE
  none

----------------------------------------------------------------------------*/
void vec_mul ( double VOut[], double A[], double V[], int N )
{
  if (ROCA_debug&ROCA_DEBUG) fprintf( stdout, "vec_mul\n" );
  mat_mul ( VOut, A, V,  1, N, N );
  return;

} // vec_mul

/*+++------------------------------------------------------------------------
NAME

  vec_add --- add two vectors of length N 

SYNOPSIS

  void vec_add ( double VOut[], double A[], double B[], int N )

DESCRIPTION

  VOut[N] = A[N]+B[N]

RETURN VALUE
  none

----------------------------------------------------------------------------*/
void vec_add ( double VOut[], double A[], double B[], int N )
{ int i;
  if (ROCA_debug&ROCA_DEBUG) fprintf( stdout, "vec_add\n" );

  for (i=0;i<N;i++) {
     VOut[i] = A[i] + B[i];
  }

} // vec_add

/*+++------------------------------------------------------------------------
NAME

  scalar_product --- scalar product of two n-dimensional vectors

SYNOPSIS

  double scalar_product ( double V[N], double W[N], int N );

DESCRIPTION
  Calculates the scalar product of V and W

RETURN VALUE
  Sum(V[i]*W[i], i=0,N-1)

----------------------------------------------------------------------------*/
double scalar_product ( double V[], double W[], int N )
{ double value;
  int i;
  if (ROCA_debug&ROCA_DEBUG) fprintf( stdout, "scalar_product\n" );
  value=0.0;
  for (i=0;i<N;i++)
    value += V[i] * W[i];
  return(value);
} // scalar_product

/*+++------------------------------------------------------------------------
NAME

  scalar_product_weight --- scalar product of two vectors and a Weight

SYNOPSIS

  double scalar_product_weight ( double V[N], double W[N], 
                                 double Weight[N], int N );

DESCRIPTION
  Calculates the weighted scalar product of V and W

RETURN VALUE
  Sum(V[i]*W[i]*Weight[i], i=0,N-1)

----------------------------------------------------------------------------*/
double scalar_product_weight ( double V[], double W[], 
                               double Weight[], int N )
{ double value;
  int i;
  if (ROCA_debug&ROCA_DEBUG) fprintf( stdout, "scalar_product_weight\n" );
  value=0.0;
  for (i=0;i<N;i++)
    value += V[i] * W[i] * Weight[i];
  return(value);
} // scalar_product_weight

/*+++------------------------------------------------------------------------
NAME

  scalar_components_sum --- sum of all vector components 

SYNOPSIS

  double scalar_components_sum ( double W[], int N )

DESCRIPTION
  Calculates the sum of all components of W 

RETURN VALUE
  Sum(W[i], i=0,N-1)

----------------------------------------------------------------------------*/
double scalar_components_sum ( double W[], int N )
{ double value;
  int i;
  if (ROCA_debug&ROCA_DEBUG) fprintf( stdout, "scalar_components_sum\n" );
  value=0.0;
  for (i=0;i<N;i++)
    value += W[i];
  return(value);
} // scalar_components_sum

/*+++------------------------------------------------------------------------
NAME
  
  scalar_absmin --- value of the absolute minimum of all components
  
SYNOPSIS

  double scalar_absmin ( double V[N], int N );

DESCRIPTION
  Returns the value of the absolute minimum of all components of
  the N-dimensional vector V.

RETURN VALUE
  Max(fabs(V[i]), i=0,N-1), V[i] of fabs(V[i]) is returned.

----------------------------------------------------------------------------*/
double scalar_absmin ( double V[], int N )
{ double value=0.0, minimum, tmpval, tmpabs;
  int i;
  if (ROCA_debug&ROCA_DEBUG) fprintf( stdout, "scalar_absmin\n" );
  if (N>0) {
    value=V[0]; minimum=fabs(value);
    for (i=1;i<N;i++) {
      tmpval=V[i]; tmpabs=fabs(tmpval);
      if (tmpabs<minimum) {
        value = tmpval;
        minimum=tmpabs;
      }
    } 
  } // if (N>0) 
  return(value);
} // scalar_absmin

/*+++------------------------------------------------------------------------
NAME
  
  scalar_absmax --- value of the absolute maximum of all components
  
SYNOPSIS

  double scalar_absmax ( double V[N], int N );

DESCRIPTION
  Returns the value of the absolute maximum of all components of
  the N-dimensional vector V.

RETURN VALUE
  Max(fabs(V[i]), i=0,N-1), V[i] of fabs(V[i]) is returned.

----------------------------------------------------------------------------*/
double scalar_absmax ( double V[], int N )
{ double value=0.0, maximum, tmpval, tmpabs;
  int i;
  if (ROCA_debug&ROCA_DEBUG) fprintf( stdout, "scalar_absmax\n" );
  if (N>0) {
    value=V[0]; maximum=fabs(value);
    for (i=1;i<N;i++) {
      tmpval=V[i]; tmpabs=fabs(tmpval);
      if (tmpabs>maximum) {
        value = tmpval;
        maximum=tmpabs;
      }
    } 
  } // if (N>0) 
  return(value);
} // scalar_absmax

/*+++------------------------------------------------------------------------
NAME
  rotation_matrix_3d --- calculates the 3-dimensional rotation matrix

SYNOPSIS

  void rotation_matrix_3d ( double R[3][3],
                            double rot1, double rot2, double rot3 );

DESCRIPTION

  Calculates the 3-dimensional rotation matrix R[3][3] for three 
  subsequent rotations angles rot1, rot2, rot3
  (in radian). 

  R[0][0] = Cos[rot2] Cos[rot3];
  R[1][0] = Cos[rot3] Sin[rot1] Sin[rot2] - Cos[rot1] Sin[rot3];
  R[2][0] = Cos[rot1] Cos[rot3] Sin[rot2] + Sin[rot1] Sin[rot3];

  R[0][1] = Cos[rot2] Sin[rot3]
  R[1][1] = Cos[rot1] Cos[rot3] + Sin[rot1] Sin[rot2] Sin[rot3]
  R[2][1] = -Cos[rot3] Sin[rot1] + Cos[rot1] Sin[rot2] Sin[rot3]

  R[0][2] = -Sin[rot2]
  R[1][2] = Cos[rot2] Sin[rot1]
  R[2][2] = Cos[rot1] Cos[rot2]

RETURN VALUE
  none

----------------------------------------------------------------------------*/
void rotation_matrix_3d ( double R[3][3],
                          double rot1, double rot2, double rot3 )
{ double c1, c2, c3;
  double s1, s2, s3;

  if (ROCA_debug&ROCA_DEBUG) fprintf( stdout, "rotation_matrix_3d\n" );

  c1 = cos(rot1); c2 = cos(rot2); c3 = cos(rot3);
  s1 = sin(rot1); s2 = sin(rot2); s3 = sin(rot3);

  R[0][0] =      c2 * c3;
  R[1][0] = s1 * s2 * c3 - c1      * s3;
  R[2][0] = c1 * s2 * c3 + s1      * s3;
  R[0][1] =      c2 * s3;
  R[1][1] = c1      * c3 + s1 * s2 * s3;
  R[2][1] = c1 * s2 * s3 - s1      * c3;
  R[0][2] =     -s2;
  R[1][2] = s1 * c2;
  R[2][2] = c1 * c2;


  return;

} // rotation_matrix_3d

/*+++------------------------------------------------------------------------
NAME

  i2normal_3d --- calculates the 3-dimensional normal vector from image coordinates 

SYNOPSIS

  void i2normal_3d( double VOut[3], double pix1, double pix2, 
                    double cen1, double cen2, double dis, double i1, double i2 );

DESCRIPTION

RETURN VALUE
  none

----------------------------------------------------------------------------*/
void i2normal_3d( double VOut[3], double pix1, double pix2, 
                  double cen1, double cen2, double dis, double i1, double i2 )
{
  if (ROCA_debug&ROCA_DEBUG) fprintf( stdout, "i2normal_3d\n" );
  if (VOut) {
    VOut[0] = (i1 - cen1) * pix1;
    VOut[1] = (i2 - cen2) * pix2;
    VOut[2] = -dis;
  }

} // i2normal_3d 

/*+++------------------------------------------------------------------------
NAME

  X_vector_3d --- calculates the lab vector X[3] from image coordinates

SYNOPSIS

  int X_vector_3d(double X[3], RO* Ordered, double i1, double i2)

DESCRIPTION

  X[1] = (-cen1 + i1) pix1 Cos[rot2] Cos[rot3] +
    (-cen2 + i2) pix2 (Cos[rot3] Sin[rot1] Sin[rot2] - Cos[rot1] Sin[rot3]) - 
    dis (Cos[rot1] Cos[rot3] Sin[rot2] + Sin[rot1] Sin[rot3]);
  X[2] = (-cen1 + i1) pix1 Cos[rot2] Sin[rot3] - 
    dis (-Cos[rot3] Sin[rot1] + Cos[rot1] Sin[rot2] Sin[rot3]) + 
    (-cen2 + i2) pix2 (Cos[rot1] Cos[rot3] + Sin[rot1] Sin[rot2] Sin[rot3]);
  X[3] = -dis Cos[rot1] Cos[rot2] + 
    (-cen2 + i2) pix2 Cos[rot2] Sin[rot1] - (-cen1 + i1) pix1 Sin[rot2];

RETURN VALUE
  pointer X 

----------------------------------------------------------------------------*/
double * X_vector_3d(double X[3], RO* Ordered, double i1, double i2)
{ const int M=3;
  double Normal[3], R[3][3];
  double pix1, pix2, cen1, cen2, dis, rot1, rot2, rot3;
  char tmp[ROCABUFLEN];

  if (ROCA_debug&ROCA_DEBUG) fprintf( stdout, "X_vector_3d\n" );

  pix1 = Ordered->Par[Ordered->Index[Pix1]];
  pix2 = Ordered->Par[Ordered->Index[Pix2]];
  cen1 = Ordered->Par[Ordered->Index[Cen1]];
  cen2 = Ordered->Par[Ordered->Index[Cen2]];
  dis  = Ordered->Par[Ordered->Index[Dis]];
  rot1 = Ordered->Par[Ordered->Index[Rot1]];
  rot2 = Ordered->Par[Ordered->Index[Rot2]];
  rot3 = Ordered->Par[Ordered->Index[Rot3]];
 
  i2normal_3d( Normal, pix1, pix2, cen1, cen2, dis, i1, i2);
  if (ROCA_debug&ROCA_SHOWTEMP) {
    sprintf(tmp,"Normal (i1=%lg, i2=%lg)",i1,i2);
    fprint_vector( stdout, M, Normal, tmp );
  }
  rotation_matrix_3d ( R, rot1, rot2, rot3 );
  if (ROCA_debug&ROCA_SHOWTEMP) fprint_matrix_3d( stdout, R, "R" );
  vec_mul ( X, &(R[0][0]), Normal, M );

  return( X );

} // X_vector_3d

/*+++------------------------------------------------------------------------
NAME

  DX_matrix_3d --- generates the partial derivates of the lab vector X[3] 

SYNOPSIS

  void DX_matrix_3d( double DX[NParams*3], RO* Ordered, double i1, double i2 );

DESCRIPTION
  Calculates the partial derivatives of the lab vector X at i1, i2.
  for ordered parameters.

RETURN VALUE
  N*3 matrix with N partial derivates of the lab vector X[3] with respect 
  to the ordered parameters Ordered->Index

----------------------------------------------------------------------------*/
void DX_matrix_3d( double DX[NParams*3], RO* Ordered, double i1, double i2 )
{ const int M=3;
  double center1, center2;
  double Cos_rot1, Cos_rot2, Cos_rot3;
  double Sin_rot1, Sin_rot2, Sin_rot3;

  int    IPix1, IPix2, ICen1, ICen2, IDis, IRot1, IRot2, IRot3;
  double pix1, pix2, cen1, cen2, dis, rot1, rot2, rot3;
  int    nopt;

  if (ROCA_debug&ROCA_DEBUG) fprintf( stdout, "DX_matrix_3d\n" );
  nopt = Ordered->nopt;

  IPix1 = Ordered->Index[Pix1];
  IPix2 = Ordered->Index[Pix2];
  ICen1 = Ordered->Index[Cen1];
  ICen2 = Ordered->Index[Cen2];
  IDis  = Ordered->Index[Dis];
  IRot1 = Ordered->Index[Rot1];
  IRot2 = Ordered->Index[Rot2];
  IRot3 = Ordered->Index[Rot3];

  pix1 = Ordered->Par[IPix1];
  pix2 = Ordered->Par[IPix2];
  cen1 = Ordered->Par[ICen1];
  cen2 = Ordered->Par[ICen2];
  dis  = Ordered->Par[IDis];
  rot1 = Ordered->Par[IRot1];
  rot2 = Ordered->Par[IRot2];
  rot3 = Ordered->Par[IRot3];

  if (ROCA_debug&ROCA_SHOWTEMP) {
    fprintf(stdout,"pix1=%lg, pix2=%lg, cen1=%lg, cen2=%lg, dis=%lg, rot1=%lg, rot2=%lg, rot3=%lg\n",
        pix1,pix2,cen1,cen2,dis,rot1,rot2,rot3);
    fprintf(stdout,"IPix1=%d, IPix2=%d, ICen1=%d, ICen2=%d, IDis=%d, IRot1=%d, IRot2=%d, IRot3=%d\n",
        IPix1,IPix2,ICen1,ICen2,IDis,IRot1,IRot2,IRot3);
  }

  Sin_rot1 = sin(rot1); Cos_rot1 = cos(rot1);
  Sin_rot2 = sin(rot2); Cos_rot2 = cos(rot2);
  Sin_rot3 = sin(rot3); Cos_rot3 = cos(rot3);

  center1 = (-cen1 + i1);
  center2 = (-cen2 + i2);

  if (DX) {
    //pix1
    if (IPix1<nopt) {
      DX[IPix1*M+0]=  center1*Cos_rot2*Cos_rot3;
      DX[IPix1*M+1]=  center1*Cos_rot2*Sin_rot3;
      DX[IPix1*M+2]= -center1*Sin_rot2;
    }

    //pix2
    if (IPix2<nopt) {
      DX[IPix2*M+0]=  center2*(Cos_rot3*Sin_rot1*Sin_rot2-Cos_rot1*Sin_rot3);
      DX[IPix2*M+1]=  center2*(Cos_rot1*Cos_rot3+Sin_rot1*Sin_rot2*Sin_rot3);
      DX[IPix2*M+2]=  center2*Cos_rot2*Sin_rot1;
    }

    //cen1
    if (ICen1<nopt) {
      DX[ICen1*M+0]= -pix1*Cos_rot2*Cos_rot3;
      DX[ICen1*M+1]= -pix1*Cos_rot2*Sin_rot3;
      DX[ICen1*M+2]=  pix1*Sin_rot2;
    }

    //cen2
    if (ICen2<nopt) {
      DX[ICen2*M+0]= -pix2*(Cos_rot3*Sin_rot1*Sin_rot2-Cos_rot1*Sin_rot3);
      DX[ICen2*M+1]= -pix2*(Cos_rot1*Cos_rot3+Sin_rot1*Sin_rot2*Sin_rot3);
      DX[ICen2*M+2]= -pix2*Cos_rot2*Sin_rot1;
    }

    //dis
    if (IDis<nopt) {
      DX[IDis*M+0] = -Cos_rot1*Cos_rot3*Sin_rot2-Sin_rot1*Sin_rot3;
      DX[IDis*M+1] =  Cos_rot3*Sin_rot1-Cos_rot1*Sin_rot2*Sin_rot3;
      DX[IDis*M+2] = -Cos_rot1*Cos_rot2;
    }

    //rot1
    if (IRot1<nopt) {
      DX[IRot1*M+0]=  dis*(Cos_rot3*Sin_rot1*Sin_rot2-Cos_rot1*Sin_rot3)+
                    center2*pix2*(Cos_rot1*Cos_rot3*Sin_rot2+Sin_rot1*Sin_rot3);
      DX[IRot1*M+1]=  dis*(Cos_rot1*Cos_rot3+Sin_rot1*Sin_rot2*Sin_rot3)-
                    center2*pix2*(Cos_rot3*Sin_rot1-Cos_rot1*Sin_rot2*Sin_rot3);
      DX[IRot1*M+2]=  Cos_rot2*(center2*pix2*Cos_rot1+dis*Sin_rot1);
    }

    //rot2
    if (IRot2<nopt) {
      DX[IRot2*M+0]=  Cos_rot3*(-dis*Cos_rot1*Cos_rot2+
                      center2*pix2*Cos_rot2*Sin_rot1-center1*pix1*Sin_rot2);
      DX[IRot2*M+1]=  (-dis*Cos_rot1*Cos_rot2+center2*pix2*Cos_rot2*Sin_rot1-
                      center1*pix1*Sin_rot2)*Sin_rot3;
      DX[IRot2*M+2]= -center1*pix1*Cos_rot2+
                     (dis*Cos_rot1-center2*pix2*Sin_rot1)*Sin_rot2;
    }

    //rot3
    if (IRot3<nopt) {
      DX[IRot3*M+0]= -dis*Cos_rot3*Sin_rot1-center1*pix1*Cos_rot2*Sin_rot3+
                    dis*Cos_rot1*Sin_rot2*Sin_rot3-
                    center2*pix2*(Cos_rot1*Cos_rot3+Sin_rot1*Sin_rot2*Sin_rot3);
      DX[IRot3*M+1]=  center1*pix1*Cos_rot2*Cos_rot3+
                    center2*pix2*(Cos_rot3*Sin_rot1*Sin_rot2-Cos_rot1*Sin_rot3)-
                    dis*(Cos_rot1*Cos_rot3*Sin_rot2+Sin_rot1*Sin_rot3);
      DX[IRot3*M+2]=  0;
    }
  }

} //DX_matrix_3d

/*+++------------------------------------------------------------------------
NAME

  Azimuth_3d --- calculates the azimuthal angle for a given X

SYNOPSIS

  double Azimuth_3d( double X[3] )

DESCRIPTION

  Calculates the azimuthal angle for a given X:

  azimuth = ((X[0]!=0)||(X[1]!=0))?atan2(X[1], X[0]):0.0;
   

RETURN VALUE
  azimuth angle

----------------------------------------------------------------------------*/
double Azimuth_3d( double X[3] )
{ double eps=0.0;
  double azimuth;

  if (ROCA_debug&ROCA_DEBUG) fprintf( stdout, "Azimuth_3d\n" );

  azimuth = ((fabs(X[0])>eps)||(fabs(X[1])>eps))?atan2(X[1],X[0]):0.0;

  return(azimuth);

} // Azimuth_3d

/*+++------------------------------------------------------------------------
NAME

  Angle_3d --- calculates the scattering angle for a given X

SYNOPSIS

  double Angle_3d( double X[3] )

DESCRIPTION

  Calculates the scattering angle for a given X:

     angle=atan2(sqrt(X[0]*X[0]+X[1]*X[1]),-X[2]);

RETURN VALUE
  scattering angle 

----------------------------------------------------------------------------*/
double Angle_3d( double X[3] )
{ double angle;

  if (ROCA_debug&ROCA_DEBUG) fprintf( stdout, "Angle_3d\n" );

  angle=atan2(sqrt(X[0]*X[0]+X[1]*X[1]),-X[2]);

  return(angle);

} // Angle_3d

/*+++------------------------------------------------------------------------
NAME

  DAngle_3d --- Partial derivative of Angle_3d with respect to Param 


SYNOPSIS

  double DAngle_3d(double X[3], double DX[nopt*3], int Param )

DESCRIPTION

  Calculates the partial derivative of the scattering angle Angle_3d
  at X with respect to the parameter Param.
  X is the coordinate of the reflection in lab coordinates calculated
  with X_vector_3d, DX are the partial derivatives of the X_vector_3d
  calculated with DX_matrix_3d. Param is the parameter number defined in
  RocaParam.

  Attention: The arguments of ArcTan in Mathematica and atan2 in C are swapped.
             The corresponing call of ArcTan[a,b] is atan2(b,a).


   D(atan2(B(x),A(x)))/Dx = (-A'(x)*B(x) + A(x)*B'(x))/(A(x)*A(x)+B(x)*B(x))

   D(atan2(sqrt(X(x)*X(x)+Y(x)*Y(x)), (-Z(x)), x)

     = ( -X1(x)*X3(x)*X1'(x)+X1(x)*X1(x)*X3'(x)+
          X2(x)*(-X3(x)*X2'(x)+ X2(x)*X3'(x)) ) /
          ( sqrt( X1(x)*X1(x)+X2(x)*X2(x) )*
            (X1(x)*X1(x)+X2(x)*X2(x)+X3(x)*X3(x)) )

     = DAngle(X1(x),X2(x),X3(x),X1'(x),X2'(x),X3'(x))

  If X[x] and Y[x] both are zero, the denominator becomes zero.
  In this case the value is:

     = -X2'(x)/X3(x)


RETURN VALUE
 derivative

----------------------------------------------------------------------------*/
double DAngle_3d(double X[3], double DX[], int Param )
{ const int M=3;
  double derivative, lenx1x2; 
  double eps=0.0;

  if (ROCA_debug&ROCA_DEBUG) fprintf( stdout, "DAngle_3d\n" );
  derivative = 0.0;
  if ((X)&&(DX)) {
    lenx1x2=sqrt(X[0]*X[0]+X[1]*X[1]);
    if ( (fabs(lenx1x2)>eps) )
      derivative = ( -X[0]*X[2]*DX[Param*M+0]+X[0]*X[0]*DX[Param*M+2]+
                      X[1]*(-X[2]*DX[Param*M+1]+X[1]*DX[Param*M+2]) ) /
                   ( lenx1x2*(X[0]*X[0]+X[1]*X[1]+X[2]*X[2]) );
    else
      derivative = -DX[Param*M+1]/X[2];
  }

  return( derivative );

} // DAngle_3d

/*+++------------------------------------------------------------------------
NAME

  DiffAngle --- Calculates difference between measured and calculated angle


SYNOPSIS

  double DiffAngle( RO* Ordered, double angle, double i1, double i2);

DESCRIPTION

  Calculates difference between measured and calculated angle.

RETURN VALUE

----------------------------------------------------------------------------*/
double DiffAngle( RO* Ordered, double angle, double i1, double i2)
{
  double X[3];
  double diffangle;

  if (ROCA_debug&ROCA_DEBUG) fprintf( stdout, "DiffAngle\n" );

  X_vector_3d(X, Ordered, i1, i2);
  if (ROCA_debug&ROCA_SHOWTEMP)
    fprint_vector( stdout, 3, X, "X" );

  diffangle = angle - Angle_3d( X );

  return( diffangle );

} // DiffAngle

/*+++------------------------------------------------------------------------
NAME

  RocaRMS --- Calculate the RMS per point of the input array.


SYNOPSIS

  double RocaRMS( size_t n, double B[], double Weight[] );

DESCRIPTION

  Calculates the RMS per point of the input array.

RETURN VALUE

  rms per data point or -1 in case of an error.

----------------------------------------------------------------------------*/
double RocaRMS( size_t n, double B[], double Weight[] )
{ double rms=-1.0, total;

  total=scalar_components_sum ( Weight, n );
  if (total>0) rms = sqrt ( scalar_product_weight ( B, B, Weight, n )/total );

  return ( rms );

} // RocaRMS

/*+++------------------------------------------------------------------------
NAME

  RocaStdDev --- Calculate the standard deviation per point of the input array.


SYNOPSIS

  double RocaStdDev( size_t n, double B[], double Weight[] );

DESCRIPTION

  Calculates the standard deviation per point of the input array.

RETURN VALUE

  stddev per data point or -1 in case of an error.

----------------------------------------------------------------------------*/
double RocaStdDev( size_t n, double B[], double Weight[] )
{ double stddev=-1.0, total;

  total=scalar_components_sum ( Weight, n );
  if (total>1.0) 
    stddev=sqrt(scalar_product_weight(B, B, Weight, n)/(total-1.0));

  return ( stddev );

} // RocaStdDev

/*+++------------------------------------------------------------------------
NAME

  RocaMinDev --- Calculates the minimum deviation of the input vector.


SYNOPSIS

  double RocaMinDev( size_t ndata, double B[] );

DESCRIPTION

  Calculates the minimum deviation of the input vector.

RETURN VALUE

  minimum deviation per data point or 0, if ndata is < 1.

----------------------------------------------------------------------------*/
double RocaMinDev( size_t n, double B[] )
{ double mindev=0.0;

  if (n>0) mindev = scalar_absmin ( B, n );

  return ( mindev );

} // RocaMinDev

/*+++------------------------------------------------------------------------
NAME

  RocaMaxDev --- Calculates the maximum deviation of the input vector.


SYNOPSIS

  double RocaMaxDev( size_t ndata, double B[] );

DESCRIPTION

  Calculates the maximum deviation of the input vector.

RETURN VALUE

  maximum deviation per data point or 0, if ndata is < 1.

----------------------------------------------------------------------------*/
double RocaMaxDev( size_t n, double B[] )
{ double maxdev=0.0;

  if (n>0) maxdev = scalar_absmax ( B, n );

  return ( maxdev );

} // RocaMaxDev

/*+++------------------------------------------------------------------------
NAME

  get_RocaB --- Calculates the B-Matrix


SYNOPSIS

  void get_RocaB( size_t ndata, double RocaB[ndata], RO *Ordered,
                  double Angle[ndata], double I1[ndata], double I2[ndata] )

DESCRIPTION

  Calculates the difference between all measured and calculated angles and
  writes them to RocaB.

RETURN VALUE

----------------------------------------------------------------------------*/
void get_RocaB( size_t ndata, double RocaB[], RO *Ordered,
                double Angle[], double I1[], double I2[] )
{ int i;
  if (ROCA_debug&ROCA_DEBUG) fprintf( stdout, "get_RocaB\n" );
  for (i=0;i<ndata;i++) 
    RocaB[i] = DiffAngle( Ordered, Angle[i], I1[i], I2[i] );
} // get_RocaB

/*+++------------------------------------------------------------------------
NAME

  get_RocaA --- Calculates the A-Matrix


SYNOPSIS

  int get_RocaA ( size_t ndata, double RocaA[nopt*ndata], RO *Ordered,
                  double Angle[ndata], double I1[ndata], double I2[ndata] );

DESCRIPTION

  Calculates the partial derivatives for all ndata data points with 
  respect to the first nopt parameters of Ordered.
  The result is returned in RocaA. The parameter number corresponding to
  each column i (Aij) is returned in Param[i]. The return value of 
  get_RocaA is the number of parameters for which partial derivatives
  have been calculated.

  The array RocaA must have at least nopt*ndata elements (nopt<=NParams).

RETURN VALUE

----------------------------------------------------------------------------*/
int get_RocaA ( size_t ndata, double RocaA[], RO *Ordered,
                double Angle[], double I1[], double I2[] )
{ double DX[NParams*3];
  double X[3];
  double i1, i2;
  int i, j;
  size_t nopt;

  if (ROCA_debug&ROCA_DEBUG) fprintf( stdout, "get_RocaA\n" );

  nopt = Ordered->nopt;

  for (j=0;j<ndata;j++) {
    i1 = I1[j]; i2=I2[j];
    X_vector_3d(X, Ordered, i1, i2);
    if (ROCA_debug&ROCA_SHOWTEMP)
      fprint_vector( stdout, 3, X, "X" );

    DX_matrix_3d(DX, Ordered, i1, i2);
    if (ROCA_debug&ROCA_SHOWTEMP)
      fprint_matrix( stdout, 3, nopt, DX, "DX" );

    for (i=0;i<nopt;i++) {
      RocaA[i*ndata+j] =  DAngle_3d(X, DX, i ); // Aij
      if (ROCA_debug&ROCA_SHOWTEMP)
        fprintf( stdout, "RocaA[%d][%d] = %lg = DAngle_3d(X,DX,%s)\n",
          i,j,RocaA[i*ndata+j],Ordered->Names[i] );
    }
  }

  return(nopt);

} // get_RocaA

/*+++------------------------------------------------------------------------
NAME

  Roca2Image_X3 --- calculates image coordinates I[2] for a lab vector X[3]

SYNOPSIS

  int Roca2Image_X3(double I[2], double X[3], double Par[NParams]);

DESCRIPTION

  Calcuates the image coordinates I[2] of the intersection point on the 
  detector between a line with direction X[3] and the detector surface.
  The BeamCenter is the intersection point of the vector X[]={0,0,-1}.

  I[0] = cen1+
         (dis*(X[2]*Sin_rot2-Cos_rot2*(X[0]*Cos_rot3+X[1]*Sin_rot3)))/
         (pix1*(Sin_rot1*(-X[1]*Cos_rot3+X[0]*Sin_rot3)+
          Cos_rot1*(X[2]*Cos_rot2+Sin_rot2*(X[0]*Cos_rot3+X[1]*Sin_rot3))))

  I[1] = cen2-
         (dis*(Cos_rot1*(X[1]*Cos_rot3-X[0]*Sin_rot3)+
          Sin_rot1*(X[2]*Cos_rot2+Sin_rot2*(X[0]*Cos_rot3+X[1]*Sin_rot3))))/
         (pix2*(Sin_rot1*(-X[1]*Cos_rot3+X[0]*Sin_rot3)+
          Cos_rot1*(X[2]*Cos_rot2+Sin_rot2*(X[0]*Cos_rot3+X[1]*Sin_rot3))))


RETURN VALUE
  0: Success
 -1: Failed, e.g. X is parallel to detector surface.

----------------------------------------------------------------------------*/
int Roca2Image_X3(double I[2], double X[3], double Par[NParams])
{ double eps=0.0;
  double pix1, pix2, cen1, cen2, dis, rot1, rot2, rot3;

  double Cos_rot1, Cos_rot2, Cos_rot3;
  double Sin_rot1, Sin_rot2, Sin_rot3;

  double numer1, numer2;
  double denom1, denom2;

  int status=0;

  pix1 = Par[Pix1];
  pix2 = Par[Pix2];
  cen1 = Par[Cen1];
  cen2 = Par[Cen2];
  dis  = Par[Dis];
  rot1 = Par[Rot1];
  rot2 = Par[Rot2];
  rot3 = Par[Rot3];

  Sin_rot1 = sin(rot1); Cos_rot1 = cos(rot1);
  Sin_rot2 = sin(rot2); Cos_rot2 = cos(rot2);
  Sin_rot3 = sin(rot3); Cos_rot3 = cos(rot3);

  denom1 = pix1*(Sin_rot1*(-X[1]*Cos_rot3+X[0]*Sin_rot3)+
          Cos_rot1*(X[2]*Cos_rot2+Sin_rot2*(X[0]*Cos_rot3+X[1]*Sin_rot3)));

  denom2 = pix2*(Sin_rot1*(-X[1]*Cos_rot3+X[0]*Sin_rot3)+
           Cos_rot1*(X[2]*Cos_rot2+Sin_rot2*(X[0]*Cos_rot3+X[1]*Sin_rot3)));

  if ( (fabs(denom1)>eps)&&(fabs(denom2)>eps) ) {

    numer1 = dis*(X[2]*Sin_rot2-Cos_rot2*(X[0]*Cos_rot3+X[1]*Sin_rot3));

    numer2 = dis*(Cos_rot1*(X[1]*Cos_rot3-X[0]*Sin_rot3)+
             Sin_rot1*(X[2]*Cos_rot2+Sin_rot2*(X[0]*Cos_rot3+X[1]*Sin_rot3))); 

    I[0] = cen1+numer1/denom1;
    I[1] = cen2-numer2/denom2;
  } else status=-1;

  return(status);

} // Roca2Image_X3

/*+++------------------------------------------------------------------------
NAME

  RocaBeamCenter --- calculates image coordinates I[2] of the beam center 

SYNOPSIS

  int RocaBeamCenter(double I[2], double Par[NParams]);

DESCRIPTION

  Calcuates the image coordinates I[2] of the beam center on the 
  detector.
  The BeamCenter is the intersection point of the lab vector {0,0,-1}.

RETURN VALUE
  0: Success
 -1: Failed, e.g. X is parallel to detector surface.

----------------------------------------------------------------------------*/
int RocaBeamCenter(double I[2], double Par[NParams])
{
  if (ROCA_debug&ROCA_DEBUG) fprintf( stdout, "RocaBeamCenter\n" );

  double X[3]={0.0,0.0,-1.0};

  return( Roca2Image_X3(I, X, Par) );

} // RocaBeamCenter

/*+++------------------------------------------------------------------------
NAME

  RocaBeamDistance --- calculates distance between beam center and sample

SYNOPSIS

  int RocaBeamDistance(double I[2], double Par[NParams]);

DESCRIPTION
  
  Calculates the distance between the beam center and the sample.

RETURN VALUE
  0: Success
 -1: Failed, e.g. X is parallel to detector surface.

----------------------------------------------------------------------------*/
int RocaBeamDistance(double *pdistance, double Par[NParams])
{ double I[2];
  double distance=0;
  double X[3];
  RO Ordered;
  int status=0;

  if (ROCA_debug&ROCA_DEBUG) fprintf( stdout, "RocaBeamDistance\n" );

  if ( (status=RocaBeamCenter(I, Par)) ) return(status);

  if (ROCA_debug&ROCA_SHOWTEMP)
    fprintf(stdout,"BeamCenter=(%lg,%lg)\n",I[0],I[1]);

  order_Par ( &Ordered, Par, NULL );

  if (ROCA_debug&ROCA_SHOWTEMP) fprint_ordered( stdout, &Ordered );

  X_vector_3d(X, &Ordered, I[0], I[1]);

  if (ROCA_debug&ROCA_SHOWTEMP)
    fprint_vector( stdout, 3, X, "X" );

  distance=-X[2]; 

  if (ROCA_debug&ROCA_SHOWTEMP)
    fprintf(stdout,"BeamDistance=%lg\n",distance);

  if (pdistance) *pdistance=distance;

  return( status );

} // RocaBeamDistance

/*+++------------------------------------------------------------------------
NAME

  RocaTilt --- calculates tilts from rotations 

SYNOPSIS

  int RocaTilt(double tilt[3], double Par[NParams]);

DESCRIPTION

  Calculates tilts from rotations.

RETURN VALUE
  0: Success
 -1: Failed, e.g. X is parallel to detector surface.

----------------------------------------------------------------------------*/
int RocaTilt(double tilt[3], double Par[NParams])
{ const int M=3;
  double R[3][3];
  double rot1, rot2, rot3;
  int status=-1;

  if (ROCA_debug&ROCA_DEBUG) fprintf( stdout, "RocaTilt\n" );

  rot1 = Par[Rot1];
  rot2 = Par[Rot2];
  rot3 = Par[Rot3];

  rotation_matrix_3d ( R, rot1, rot2, rot3 );

  if (ROCA_debug&ROCA_SHOWTEMP) fprint_matrix_3d( stdout, R, "R" );

  if ( tilt3d_angles( tilt, R ) ) goto RocaTilt_error;

  if (ROCA_debug&ROCA_SHOWTEMP)
    fprint_vector( stdout, M, tilt, "tilt" );

  status=0; // Success

  return( status );

RocaTilt_error:

  return( status );

} // RocaTilt

/*+++------------------------------------------------------------------------
NAME

  CholeskySolve --- Cholesky decomposition of the Matrix A 


SYNOPSIS

  int CholeskySolve( int n, double A[n*n], double B[n], double X[n] );

DESCRIPTION

  B[n] and X[n]  are vectors, A[n][n] is a positive definite Matrix. Then 
  it can be transformed in a triangular matrix G and its tranposed matrix
  GT=Transpose(G) so that 

  A = G*Transpose(G)

  The linear equation B=A*X can be solved by first solving

  B=G*Y => Y

  Y=Transpose(G)*X => X

  To reduce array space the values of G could immediately be written into A.
  This is not done here, a separate G matrix is allocated.

RETURN VALUE

  0 if OK, -1 in case of an error

----------------------------------------------------------------------------*/
int CholeskySolve( int n, double A[], double B[], double X[] ) 
{ 
  int i,j,k;
  double sum;
  
  double *G, *Y, *tmp=NULL;

  if (ROCA_debug&ROCA_DEBUG) fprintf( stdout, "CholeskySolve\n" );

  G = (double *) MALLOC( n * n * sizeof(double) );
  Y = (double *) MALLOC( n * sizeof(double) );

  if (ROCA_debug&ROCA_SHOWTEMP) {
    tmp = (double *) MALLOC( n * n * sizeof(double) ); // used for check
    for (i=0;i<n*n;i++) { G[i]=0.0; tmp[i]=0.0; }
    for (i=0;i<n;i++) Y[i]=0.0;
  }

 // G[n][n] is written as the upper right triangular matrix (j>=i)
  for (i=0;i<n;i++) {
    for (j=i;j<n;j++) {
      sum = A[i*n+j];
      for (k=i-1;k>=0;k--) {
        sum -= G[i*n+k]* G[j*n+k];
      }

      if (i==j) {
        if (sum<=0) goto CholeskySolveError; // not positive definite
        else G[j*n+i] = sqrt(sum);
      } else {
        G[j*n+i] = sum / G[i*n+i];
      } 
    }  // j
  } // i

  if (ROCA_debug&ROCA_SHOWTEMP) {
    fprint_matrix( stdout, n, n, G, "G" );
    transpose_mul ( tmp, G, G, n, n, n );
    fprint_matrix( stdout, n, n, tmp, "GTG" );
  }

  // Solve B=GT*Y for Y
  for (j=0;j<n;j++) {
    sum=0.0;
    for (i=0;i<j;i++) {
      sum+=G[j*n+i]*Y[i]; // sum+=GT[i][j]*Y[i];
    }
    Y[j] = ( B[j] - sum ) / G[j*n+j]; // Y[j] = ( B[j] - sum ) / GT[j][j];
  }

  // Solve Y=G*X for X
  for (j=n-1;j>=0;j--) {
    sum=0.0;
    for (i=n-1;i>j;i--) {
      sum+=G[i*n+j]*X[i]; // sum+=G[i][j]*X[i];
    }
    X[j] = ( Y[j] - sum ) / G[j*n+j]; // X[j] = ( Y[j] - sum ) / G[j][j];
  }

  FREE(G); FREE(Y); FREE(tmp);

  return(0);

CholeskySolveError:

  if (ROCA_debug&ROCA_SHOWTEMP)
    fprint_matrix( stdout, n, n, G, "G" );

  FREE(G); FREE(Y); FREE(tmp);

  return(-1);

} // CholeskySolve

/*+++------------------------------------------------------------------------
NAME

  roca_save_step --- Save current iteration step


SYNOPSIS

  int roca_save_step( int step, size_t ndata, RO *Ordered,
                 double Angle[ndata], double I1[ndata], double I2[ndata],
                 int truncate )


DESCRIPTION

  Calculates angle and azimuth for each data point and saves them as
  xy-plot with step as scan number in a spec file.

RETURN VALUE

  0 if OK, -1 in case of an error

----------------------------------------------------------------------------*/
int roca_save_step( int step, size_t ndata, RO *Ordered, 
                    double Angle[], double I1[], double I2[], double Weight[],
                    int truncate )
{ double X[3];
  double *angle_step=NULL, *azimuth_step=NULL;
  double *ParCur=NULL;
  size_t n;
  char buffer[ROCABUFLEN];
  FILE * out;
  int cnt;
  int status=-1;

  if (ROCA_debug&ROCA_DEBUG) fprintf( stdout, "roca_save_step\n" );

  // allocate matrices
  if (!( angle_step=(double *) MALLOC(ndata*sizeof(double)) ) )
    goto roca_save_step_error;
  if (!( azimuth_step=(double *) MALLOC(ndata*sizeof(double)) ) )
    goto roca_save_step_error;
  if (!( ParCur=(double *) MALLOC(NParams*sizeof(double)) ) )
    goto roca_save_step_error;

  // calculate iteration step
  for ( n=0; n<ndata; n++) {
    X_vector_3d(X, Ordered, I1[n], I2[n]);
    angle_step  [n] = Angle_3d( X );
    azimuth_step[n] = Azimuth_3d( X );
  }

  // reorder Order
  reorder_Par ( Ordered, ParCur, NULL, NULL );

  // write iteration step to ROCA_STEPFILE
  if ( truncate ) out = fopen( ROCA_STEPFILE, "wb+"); // truncate file 
  else out = fopen( ROCA_STEPFILE, "ab+"); // append to end of file 

  fprintf( out, "\r\n"); cnt=1;
  fprintf( out, "#S %d\r\n", step); cnt++;
  fprintf( out, "# Step%03d = %s\r\n",
    step, sprint_ordered_par( buffer, ROCABUFLEN, Ordered ) ); cnt++;
  fprintf( out, "# BeamPar = %s\r\n",
    sprint_beam_parameters( buffer, ROCABUFLEN, ParCur ) ); cnt++;
  fprintf( out, "#     Par = %s\r\n",
    sprint_parameters( buffer, ROCABUFLEN, ParCur ) ); cnt++;
  // start in line 27 or later
  for ( cnt=cnt; cnt<27; cnt++ ) fprintf( out, "\r\n");
  fprintf( out, "#N 6\r\n" );
  fprintf( out, "#L number\t  angle/rad\t  azimuth/rad\t  Weight\t  Target/rad\t  Difference/rad\r\n" );
  // write data
  for ( n=0; n<ndata; n++) {
    fprintf( out, "%lu\t  %lg\t  %lg\t  %lg\t  %lg\t  %lg\r\n",
      SIZE_T n, angle_step[n], azimuth_step[n], Weight[n], 
      Angle[n], Angle[n]-angle_step[n] );
  }

  fclose( out );

  FREE(ParCur);
  FREE(azimuth_step);
  FREE(angle_step);

  status=0;

  return( status );

roca_save_step_error:

  FREE(ParCur);
  FREE(azimuth_step);
  FREE(angle_step);

  return( status );

} // roca_save_step

/*+++------------------------------------------------------------------------
NAME

  roca_parts --- Calculate the average influence of parameter Param


SYNOPSIS

  double roca_parts( size_t n, double RocaA[nopt*n], double Weight[n], 
                     int Param );

  with 0<=Param<nopt

DESCRIPTION

  Average the square of all partial derivatives of the Angle with respect 
  to parameter Param over all input values. Return the square root of the
  result.

RETURN VALUE

  >=0.0 if OK, negative in case of an error

---------------------------------------------------------------------------*/
double roca_parts( size_t n, double RocaA[], double Weight[], int Param )
{ double pardev=-1.0, total;
  double *A;

  total=scalar_components_sum ( Weight, n );
  A = &(RocaA[Param*n]);
  if (total>1.0) 
    pardev=sqrt(scalar_product_weight(A, A, Weight, n)/(total-1.0));

  return(pardev);

} // roca_parts

int get_RocaParDev( size_t ndata, RO *Ordered,
                    double Angle[], double I1[], double I2[], double Weight[] )

{ double eps=1e-30;
  double *A=NULL, *B=NULL;
  double parts, stddev;
  double mindev, maxdev;
  double total;
  size_t nopt;
  int i;

  if (ROCA_debug&ROCA_DEBUG) fprintf( stdout, "get_RocaParDev\n" );

  nopt=Ordered->nopt;

  if (!( A=(double *) MALLOC(nopt*ndata*sizeof(double)) ) )
    goto get_RocaParDevError;
  if (!( B=(double *) MALLOC(ndata*sizeof(double)) ) )
    goto get_RocaParDevError;

  // get final matrix B
  get_RocaB( ndata, B, Ordered, Angle, I1, I2 );
  if (ROCA_debug&ROCA_SHOWTEMP)
    fprint_vector( stdout, ndata, B, "B" );

  // calculate stddev
  stddev = RocaStdDev( ndata, B, Weight );

  // calculate mindev and maxdev 
  mindev = RocaMinDev(ndata, B);
  maxdev = RocaMaxDev(ndata, B);

  // calculate sum of weight
  total = scalar_components_sum ( Weight, ndata );

  // get final matrix A
  get_RocaA( ndata, A, Ordered, Angle, I1, I2 );
  if (ROCA_debug&ROCA_SHOWTEMP)
    fprint_matrix( stdout, ndata, nopt, A, "A" );

  // calculate pardev
  for (i=0;i<nopt;i++) {
    parts=roca_parts( ndata, A, Weight, i );
    if ( fabs(parts) > eps ) 
      Ordered->ParDev[i] = stddev/parts;
    else Ordered->ParDev[i] = -1.0;
  }
  for (i=nopt;i<NParams;i++) 
    Ordered->ParDev[i] = 0.0;

  if (ROCA_debug&ROCA_SHOWTEMP) {
    printf("stddev=%lg, mindev=%lg, maxdev=%lg\n",
      stddev,mindev,maxdev);
    fprint_ordered( stdout, Ordered );
  }

  // RStdDev, RMinDev, RMaxDev, RTotal, RNData,  RNopt
  for (i=0;i<NStats;i++)
    Ordered->Stat[i] = 0.0;

  Ordered->Stat[RStdDev] = stddev;
  Ordered->Stat[RMinDev] = mindev;
  Ordered->Stat[RMaxDev] = maxdev;
  Ordered->Stat[RTotal]  = total;
  Ordered->Stat[RNData]  = ndata;
  Ordered->Stat[RNopt]   = nopt;

  FREE(A); FREE(B);

  return(0);

get_RocaParDevError:

  FREE(A); FREE(B);

  return(-1);

} // get_RocaParDev

/*+++------------------------------------------------------------------------
NAME
  
  RocaOptimize --- Optimizes the parameters Par[NParams] 


SYNOPSIS

  int RocaOptimize( double ParOpt[NParams], ParDev[NParams],
                    double Par[NParams], int ParMode[NParams],
                    size_t ndata,
                    double Angle[], double I1[], double I2[], double Weight[],
                    double maxdev, int maxstep  );

DESCRIPTION

  The parameters Par[NParams] for which ParMode[NParams] is 1 are optimized for 
  the data Angle[ndata], I1[ndata] and I2[ndata]. The result is written to 
  ParOpt[NParams]. The deviations of the optimized parameters are written to 
  ParDev[NParams].
  maxdev : maximum mean deviation of each data point from the fit
  maxstep: maximum number of iteration steps

RETURN VALUE

  0 if OK, -1 in case of an error

----------------------------------------------------------------------------*/
int RocaOptimize( double ParOpt[NParams], double ParDev[NParams], 
                  double ParStat[NStats],
                  double Par[NParams], int ParMode[NParams], 
                  size_t ndata, 
                  double Angle[], double I1[], double I2[], double Weight[],
                  double maxdev, int maxstep  )
{ RO Ordered;
  double *A=NULL, *B=NULL, *ATA=NULL, *ATB=NULL, *DOpt=NULL;
  int nopt;
  int status;
  double stddev, previous_stddev, diff_stddev=1, max_diff_stddev=1e-16;
  int step;  

  if (ROCA_debug&ROCA_DEBUG) fprintf( stdout, "RocaOptimize\n" );

  status = 0;

  // sort parameters
  nopt=order_Par ( &Ordered, Par, ParMode );

  // allocate matrices

  if (!( A=(double *) MALLOC(nopt*ndata*sizeof(double)) ) ) 
    goto RocaOptimizeError; 
  if (!( B=(double *) MALLOC(ndata*sizeof(double)) ) ) 
    goto RocaOptimizeError; 
  if (!( ATA=(double *) MALLOC(nopt*nopt*sizeof(double)) ) ) 
    goto RocaOptimizeError;  // ATA[nopt][nopt] -> L=nopt, N=nopt, M=ndata
  if (!( ATB=(double *) MALLOC(nopt*sizeof(double)) ) ) 
    goto RocaOptimizeError; // ATB[1][nopt] -> L=1, N=nopt, M=ndata
  if (!( DOpt=(double *) MALLOC(nopt*sizeof(double)) ) ) 
    goto RocaOptimizeError;

  // Optimization loop BEGIN

    // get first matrix B
     get_RocaB( ndata, B, &Ordered, Angle, I1, I2 );
     if (ROCA_debug&ROCA_SHOWTEMP)
       fprint_vector( stdout, ndata, B, "B" );

     stddev = RocaStdDev( ndata, B, Weight );
     step=0;

    // save iteration step
     if ( ROCA_debug&ROCA_SAVESTEPS ) {
       if (roca_save_step( step, ndata, &Ordered, Angle, I1, I2, Weight,1))
         goto RocaOptimizeError;
     }

     while ( ( stddev > maxdev )&&( step < maxstep )&&
             ( fabs(diff_stddev) > max_diff_stddev ) ) {

      // get matrix A
       get_RocaA( ndata, A, &Ordered, Angle, I1, I2 );
       if (ROCA_debug&ROCA_SHOWTEMP)
         fprint_matrix( stdout, ndata, nopt, A, "A" );

      // get weighted matrix ATB
       transpose_mul_weight ( ATB, A, B, Weight, 1, ndata, nopt );
       if (ROCA_debug&ROCA_SHOWTEMP)
         fprint_vector( stdout, nopt, ATB, "ATB" );

      // get weighted matrix ATA
       transpose_mul_weight ( ATA, A, A, Weight, nopt, ndata, nopt );
       if (ROCA_debug&ROCA_SHOWTEMP)
         fprint_matrix( stdout, nopt, nopt, ATA, "ATA" );

      // get matrix DOpt
       if ( CholeskySolve( nopt, ATA, ATB, DOpt ) ) goto RocaOptimizeError;
       if (ROCA_debug&ROCA_SHOWTEMP)
         fprint_vector( stdout, nopt, DOpt, "DOpt" );

      // add matrix DOpt to ordered parameters (matrix Ordered.Par)
       vec_add( Ordered.Par, Ordered.Par, DOpt, nopt );
       if (ROCA_debug&ROCA_SHOWTEMP)
         fprint_vector( stdout, NParams, Ordered.Par, "Ordered.Par" );

      // get next matrix B
       get_RocaB( ndata, B, &Ordered, Angle, I1, I2 );
       if (ROCA_debug&ROCA_SHOWTEMP)
         fprint_vector( stdout, ndata, B, "B" );

       previous_stddev = stddev;
       stddev = RocaStdDev( ndata, B, Weight );
       diff_stddev=stddev-previous_stddev;

       step++;

       if (ROCA_debug&ROCA_VERBOSE) {
         printf("step=%d, stddev=%lg, change=%lg, mindev=%lg, maxdev=%lg\n",
           step,stddev,diff_stddev,RocaMinDev(ndata, B),RocaMaxDev(ndata, B));
         fprint_ordered( stdout, &Ordered );
       }

      // save iteration step
       if ( ROCA_debug&ROCA_SAVESTEPS ) {
         if (roca_save_step(step,ndata, &Ordered, Angle, I1, I2, Weight,0))
           goto RocaOptimizeError;
       }

    } // while

  // Optimization loop END

  // get devisations of optimized parameters
  get_RocaParDev( ndata, &Ordered, Angle, I1, I2, Weight );

  // reorder Order
  reorder_Par ( &Ordered, ParOpt, ParDev, ParStat );

  FREE(A); FREE(B); FREE(ATA); FREE(ATB); FREE(DOpt);

  return(status);

RocaOptimizeError:

  FREE(A); FREE(B); FREE(ATA); FREE(ATB); FREE(DOpt);

  status |= 1;

  return(status);

} // RocaOptimize

int ROCALevel ( void )
{ return( ROCA_level );
} // ROCALevel

int ROCADebug ( void )
{ return( ROCA_debug );
} // ROCADebug

int RocaDebugSet( int debug ) 
{ return(roca_debug_set( debug ));
} // RocaDebug

const char *RocaUsage2str( void ) 
{ return( roca_usage2str() ); 
} // RocaUsage2str

const char **RocaParamNamesArray( void )
{ return(&RocaParamNames[0]);
} // RocaParamNamesArray

int *RocaParamDefaultArray( void )
{ return(&RocaParamDefault[0]);
} // RocaParamDefaultArray
