# !/usr/bin/env python
# -*- coding: utf-8 -*-
#
#  Project: The SPD Image correction and azimuthal regrouping
#                     http://forge.epn-campus.eu/projects/show/azimuthal
#
#  Copyright (C) 2005-2010 European Synchrotron Radiation Facility
#                          Grenoble, France
#
#  Principal authors: P. Boesecke (boesecke@esrf.fr)
#                     R. Wilcke (wilcke@esrf.fr)
#
#  This program is free software: you can redistribute it and/or modify
#  it under the terms of the GNU Lesser General Public License as published
#  by the Free Software Foundation, either version 3 of the License, or
#  (at your option) any later version.
#
#  This program is distributed in the hope that it will be useful,
#  but WITHOUT ANY WARRANTY; without even the implied warranty of
#  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#  GNU Lesser General Public License for more details.
#
#  You should have received a copy of the GNU General Public License
#  and the GNU Lesser General Public License  along with this program.
#  If not, see <http://www.gnu.org/licenses/>.
#

"""
SX geometry <-> Fit2D <-> Kieffer
Runs sxparams on the shell. The executable sxparams must be installed.
SxGeometry.__init__ creates a instance for a given set of parameters.
The input object is a dictionary containing all necessary keys.
SxGeometry().sxparams() calculates the corresponding parameters for different raster orientations (default is 1)
Depending on the argument allitems (default False), a dictionary with all items or only the calculated items is returned.

The rasterorientation (ori) is 1 (=='1,2,3') 
    when the fast changing index corresponds to the horizontal axis and the slow changing index to the vertical axis.

The rasterorientation (ori) is 13 (=='2,1,-3') 
    when the fast changing index corresponds to the vertical axis, the slow changing index to the horizontal axis and 
    when the direction of the third axis (vector product of the horizontal axis and the vertical axis) corresponds to 
    the direction of the primary beam.

Fit2D
tilt1=TiltPlanRot/deg, tilt2=Tilt/deg 

HISTORY:
2014-01-15 PB creation

EXAMPLE:

a)
ex_dict={'pix1':'4.842e-5','pix2':'4.684e-5','bcen1':'1027.526','bcen2':'513.285','bdis':'99.487_mm','tilt2':'0.921_deg','tilt1':'-89.551_deg'}
ex=sxparams.SxGeometry(ex_dict)


print(ex.sxparams())
{'rot2': '0.000125961', 'bcen1': '1027.53', 'bcen2': '513.285', 'cen2': '547.424', 'tilt3': '0', 'rot3': '1.01237e-06', 'cen1': '1027.27', 'pix1': '4.842e-05', 'pix2': '4.684e-05', 'tilt2': '0.0160745', 'rot1': '0.016074', 'tilt1': '-1.56296', 'ori': '1', 'bdis': '0.099487', 'dis': '0.0994741'}


print(ex.sxparams(5))
{'rot2': '-0.016074', 'bcen1': '513.285', 'bcen2': '1027.53', 'cen2': '1027.27', 'tilt3': '0', 'rot3': '1.0125e-06', 'cen1': '547.424', 'pix1': '4.684e-05', 'pix2': '4.842e-05', 'tilt2': '0.0160745', 'rot1': '-0.000125978', 'tilt1': '3.13376', 'ori': '5', 'bdis': '0.099487', 'dis': '0.0994741'}


b)
SxGeometry().sxparams(1,1)

Out[13]: {'pro': '-', 'dim2': '-', 'dim1': '-', 'axis1': '-', 'axis2': '-', 'rot1': '-', 'rot3': '-', 'rot2': '-', 'bis2': '-', 'bis1': '-', 'bdis': '-', 'off1': '-', 'off2': '-', 'bcen1': '-', 'bcen2': '-', 'pix1': '-', 'pix2': '-', 'wvl': '-', 'ori': '1', 'cen2': '-', 'cen1': '-', 'tilt2': '-', 'tilt3': '-', 'tilt1': '-', 'ras2': '-', 'ras1': '-', 'dis': '-'}

SxGeometry().sxparams(1)

Out[14]: {'ori': '1'}

"""

__author__ = "Peter Boesecke"
__contact__ = "boesecke@esrf.eu"
__license__ = "GPLv3+"
__copyright__ = "European Synchrotron Radiation Facility, Grenoble, France"
__date__ = "2014-01-22"
__status__ = "beta"
__docformat__ = 'restructuredtext'

import subprocess

class SxGeometry:
    """
    @param params_in: dictionary with the following items
        pro: projection type (1,2) Saxs=1, Waxs=2
        ori: orientation number (1-16) or tuple, e.g. '1,2,3'
        axis1: type of axis 1 (1,2,3)
        axis2: type of axis 2 (1,2,3)
        dim1:  dimension 1 of 2d array
        dim2:  dimension 2 of 2d array
        off1:  offset 1 of array coordinates
        off2:  offset 2 of array coordinates
        bis1:  binning size 1
        bis2:  binning size 2
        ras1:  raster region of axis 1
        ras2:  raster region of axis 2
        pix1:  pixel size 1 [m]
        pix2:  pixel size 2 [m]
        cen1:  PONI 1 (point of normal incidence)
        cen2:  PONI 2 (point of normal incidence)
        dis:   distance sample-PONI [m]
        rot1:  detector rotation 1 [rad] (         0 deg)
        rot2:  detector rotation 2 [rad] (        -0 deg)
        rot3:  detector rotation 3 [rad] (         0 deg)
        wvl:   wavelength [m]
       [bcen1: beam center 1
        bcen2: beam center 2
        bdis:  distance sample-bcen [m]
        tilt1: detector tilt 1 [rad] (       -90 deg)
        tilt2: detector tilt 2 [rad] (         0 deg)
        tilt3: detector tilt 3 [rad] (         0 deg)]

    Raster orientation 1 is the standard configuration. In this orientation
    the fast running index of the detector array corresponds to axis 1 (dim1) 
    and the slowly running index to axis 2 (dim2).  If all rotations are zero

    - axis 1 is supposed to be horizontal, to the right (dim1)

    - axis 2 is supposed to be vertical, upwards (dim2)

    It is supposed that the observer looks from the scattering center to the 
    detector. A third axis is necessary to define the orientation in space:

    - axis 3 is supposed to be the detector normal pointing to the observer.

    With these assumptions axis 1, 2 and 3 form a righthanded-orthonormal 
    coordinate system in the standard configuration (raster orientation 1). 

    In total 48 different raster orientations exist in three dimensional space, 
    half of them are right-handed. With the restriction that the detector axis 1 
    and 2 never swap with the third axis 16 orientation remain, 8 when looking 
    from the scattering center to the detector. This can be defined by tuples,
    where each position specifies the actual index and its direction with respect 
    to the standard orientation (horizontal(h),vertical(v),normal(hxv)).

    scattering center     detector
    -> detector           -> scattering center

    1: (1,2,+3),            9: (1,2,-3),
    2: (-1,2,+3),          10: (-1,2,-3),
    3: (1,-2,+3),          11: (1,-2,-3),
    4: (-1,-2,+3),         12: (-1,-2,-3),
    5: (2,1,+3),           13: (2,1,-3),
    6: (2,-1,+3),          14: (2,-1,-3),
    7: (-2,1,+3),          15: (-2,1,-3),
    8: (-2,-1,+3)          16: (-2,-1,-3)

    The geometrical parameters (offset, center, distance, rotations) are defined in 
    such a way that they can be applied without knowledge of the actual raster 
    orientation. However, any analysis that requires the knowledge of the absolute 
    orientation in space, e.g. a polarization correction, cannot be done without 
    specifiying the raster orientation.

    To make the observation direction independent of the raster orientation value 
    a parameter "observation direction" from(1)/to(2) scattering center is needed, 
    which would replace the third parameter of each tuple.

    

    """

    def __init__(self, params_in={}):
        '''The order of the arguments of sxparams and their names correspond exactly to the items of the following tuple'''
        self.items = ( 'pro', 'ori', 'axis1', 'axis2', 'dim1', 'dim2', 'off1', 'off2', 'bis1', 'bis2',
            'ras1', 'ras2', 'pix1', 'pix2', 'cen1', 'cen2', 'dis', 'rot1', 'rot2', 'rot3', 'wvl',
            'bcen1', 'bcen2', 'bdis', 'tilt1', 'tilt2', 'tilt3' )

        params_cp = dict(params_in)
        self.params = {}
        for item in self.items:
            if item in params_cp.keys():
                if (params_cp[item]=='None') or (params_cp[item]==None):
                    self.params[item] = '-'
                else:
                    self.params[item] = params_cp[item]
                # Remove value to check at the end that all items could be used
                params_cp[item] = None
            else:
                self.params[item] = '-'

        for item in params_cp.keys():
            if params_cp[item]!=None:
                print("WARNING: Input item {0}:{1} not recognized and ignored".format(item,params_cp[item]))

#       print( self.params )

    def sxparams(self,oori=1,allitems=False,options=('debug=0',)):
        """
        sxparams (<output orientation value (oori 1..16)>,<allitems>,<options_list (options)>)
        @param oori:     output orientation value (oori 1..16) or tuple, e.g. '1,2,3'
        @param allitems: include all items in returned dictionary (True or False)
        @param options:  list with options of sxparams, e.g. ('debug=8',)
        @return: dictionary params_out with all known or all items (depending on allitems)
        """

        command = 'sxparams'
        output_orientation = '{0}'.format(oori)

        Popen_in = [command]

        for item in options:
            Popen_in.append(item)

        Popen_in.append(output_orientation)

        for item in self.items:
            Popen_in.append(self.params[item])

#       print(Popen_in)

        child = subprocess.Popen(Popen_in,stdout=subprocess.PIPE,stderr=subprocess.PIPE)

        Popen_out = child.communicate()

        status = child.returncode

#       print(Popen_out)

        params_out = {}

        if status == 0:
        
            #new style (python3, works already in python V=2.6.6, but not in 2.5.2 
            out = Popen_out[0].decode("utf-8")
            err = Popen_out[1].decode("utf-8")

            #old style (python 2)
            #out = Popen_out[0]
            #err = Popen_out[1]

            #   Depending on the debug level additional output lines can be written.
            #   The line that starts with "#pro" lists the names of the output parameters.
            #   It is immediately followed by the line that lists the calculated parameters.
      
            # The part before # can contain debug output, the part afterwards the values
            values = out.split('#')

            # Print debug output when available
            if (len(values) > 0 ):
               if (len(values[0]) > 0 ):
                   print(values[0])

            value_line = values[-1].splitlines()

            key_list = value_line[0].split()
            value_list = value_line[1].split()

            for i in range(0,len(key_list)):
               if ( allitems != False ) or ( value_list[i] != '-' ):
                   params_out[key_list[i]]=value_list[i]

               # Verify the names of the returned arguments of sxparams
               if key_list[i] != self.items[i]:
                   print("ERROR: Name of sx parameter {0} '{1}' does not match returned item '{2}'".format(i,self.items[i],key_list[i]))
                   params_out = {}
                   status = -1 
        else:
            print("ERROR: {0} exit status = {1}".format(command,status))
            print( Popen_out )

        return(params_out)

    def sxitems(self):
        return(self.items)


# end 
