/*
 *   Project: The SPD Image correction and azimuthal regrouping
 *                      http://forge.epn-campus.eu/projects/show/azimuthal
 *
 *   Copyright (C) 2005-2010 European Synchrotron Radiation Facility
 *                           Grenoble, France
 *
 *   Principal authors: P. Boesecke (boesecke@esrf.fr)
 *                      R. Wilcke (wilcke@esrf.fr)
 *
 *   This program is free software: you can redistribute it and/or modify
 *   it under the terms of the GNU Lesser General Public License as published
 *   by the Free Software Foundation, either version 3 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU Lesser General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   and the GNU Lesser General Public License  along with this program.
 *   If not, see <http://www.gnu.org/licenses/>.
 */
# define STRLIB_VERSION    "strlib : V1.19 Peter Boesecke 2017-10-08"
/*---------------------------------------------------------------------------
NAME

   strlib.c --- string functions 

DESCRIPTION

   Collection of string functions. This module uses the ioalloc functions.
   Eventually allocated memory must be released with strlib_free() or iofree().

HISTORY

   2011-04-27 V1.0  Peter Boesecke
   2011-05-14 V1.01 PB functions renamed from str.. to strlib_..., 
                       strlib_ncasecmp, strlib_toupper, strlib_tolower added,
                       strlib_collapse collapses always to a single space.
   2011-05-25 V1.02 PB strlib_toupper, strlib_tolower: corrected
   2011-06-30 V1.03 PB strlib_concat added
   2011-10-22 V1.04 PB strlib_tok and strlib_tok_r added
   2011-12-12 V1.05 PB strlib_concat corrected for buffer==a
   2012-08-02 V1.06 PB strlib_nconcat, strlib_spn added
   2014-05-29 V1.07 PB strlib_is_crlf and strlib_trimcrlf added
   2015-12-06 V1.08 PB tolower and toupper replaced with
                       TOLOWER and TOUPPER
   2016-12-12 V1.09 PB strlib_casecmp added
   2017-01-09 V1.10 PB added: strlib_ncmp, strlib_cmp and strlib_numcmp
                       modified: strlib_uncomment, strlib_trim, 
                       strlib_trimcrlf, strlib_collapse, strlib_tolower 
                       return NULL if the input string is NULL. 
                       strlib_toupper, strlib_tolower are already doing it.
   2017-01-12 V1.11 PB strlib_numcasecmp added
   2017-03-07 V1.12 PB strlib_wildcmp added
   2017-03-13 V1.13 PB strlib_newstrn added,
                       strlib_ncasecmp code modified using strlib_newstrn.
   2017-03-30 V1.14 PB strlib_nconcat and strlib_concat: new case for
                       buffer==b.
                       strlib_c_count, strlib_is_list, strlib_freestringarray 
                       and strlib_list2stringarray added.
   2017-05-19 V1.15 PB ioalloc, removing NULL ptr checks before FREE
   2017-05-29 V1.16 PB strtok_r => strlib_tok_r
   2017-06-07 V1.17 PB strlib_free added
   2017-07-11 V1.18 PB strlib_concat, strlib_nconcat: NULL strings are
                       replaced with empty strings.
   2017-10-08 V1.19 PB strlib_numberstring added, including numio.h
    
---------------------------------------------------------------------------*/

/****************************************************************************
*  Include                                                                  *
****************************************************************************/
# include <errno.h>
# include <stdio.h>
# include <stdlib.h>
# include <stdarg.h>
# include <string.h>
# include <ctype.h>
# include <limits.h>

# include "strlib.h"

/***************************************************************************
 * Macros                                                                  *
 ***************************************************************************/
# define ISGRAPH(c) isgraph((int)(c))
# define ISSPACE(c) isspace((int)(c))
# define TOLOWER(s) (char)tolower((int)(s))
# define TOUPPER(s) (char)toupper((int)(s))
# define ISDIGIT(c) isdigit((int)(c))

/***************************************************************************
 * Defines                                                                 *
 ***************************************************************************/

# define BUFLEN EdfMaxLinLen+1

# ifndef MIN
#   define MIN( n1, n2) ( ( n1)<( n2) ? ( n1) : ( n2) )
# endif

# ifndef ABS
#   define ABS( n1) ( ( n1)<0 ? -( n1) : ( n1) )
# endif

/****************************************************************************
* Static Variables                                                          *
****************************************************************************/

PRIVATE char white_spaces[] = { ' ', '\t', '\r', '\n', '\f', '\v', '\0' };
PRIVATE char crlf[] = { '\r', '\n', '\0' };
  
/****************************************************************************
*  Functions                                                                *
****************************************************************************/

/*--------------------------------------------------------------------------
NAME

  strlib_version --- returns pointer to the version string

SYNOPSIS

  const char *strlib_version ( void );

DESCRPTION

  Returns pointer to the version string.

--------------------------------------------------------------------------*/
PUBLIC const char *strlib_version ( void )
{
  return ( STRLIB_VERSION );
} /* strlib_version */

/*---------------------------------------------------------------------------
NAME

    strlib_is_white

SYNOPSIS

    int strlib_is_white ( char c );

DESCRIPTION

Returns 1 if c is a white space, 0, if not.

HISTORY
18-Jan-1998 Peter Boesecke
---------------------------------------------------------------------------*/
PUBLIC int strlib_is_white ( char c )
{
   if (!c) return(0); // no white space
   if ( strchr( white_spaces, (int) c ) )
     return(1); // white space 
   else return(0); // no white space 

} /* strlib_is_white */

/*---------------------------------------------------------------------------
NAME

    strlib_is_crlf

SYNOPSIS

    int strlib_is_crlf ( char c );

DESCRIPTION

Returns 1 if c is a carriage return or line feed, 0, if not.

HISTORY
29-May-2014 Peter Boesecke
---------------------------------------------------------------------------*/
PUBLIC int strlib_is_crlf ( char c )
{
   if (!c) return(0); // no cr or lf
   if ( strchr( crlf, (int) c ) )
     return(1); // cr or lf
   else return(0); // no cr nor lf 

} /* strlib_is_crlf */

/*---------------------------------------------------------------------------
NAME

    strlib_is_no_skip --- returns 0 when the input string is a skip pattern. 

SYNOPSIS

   int strlib_is_no_skip( const char * s );

DESCRIPTION

   Checks, if the input string is a skip sign: "...", "-", ".".

RETURN VALUE

   Returns only 0, when the input string is a skip sign.

---------------------------------------------------------------------------*/
PUBLIC int strlib_is_no_skip( const char * s )
{ int value=0;

  value = strcmp ( s, "..." );
  if (value) value = strcmp ( s, "-" );
  if (value) value = strcmp ( s, "." );

  return ( value );
} // strlib_is_no_skip

/*---------------------------------------------------------------------------
NAME

    strlib_is_empty --- returns 1 if the input string is empty or not allocated

SYNOPSIS

   int strlib_is_empty( const char *s );

DESCRIPTION

   Rreturns 1 if the input string s contains only white space, if it
   has the length 0, or is the NULL pointer, otherwise 0.

RETURN VALUE

   1: if string is empty, 0: otherwise

---------------------------------------------------------------------------*/
PUBLIC int strlib_is_empty( const char *s )
{ const char *ps;
  int empty=1;

  if (s) {
    ps = s;
    while (*ps) { empty = empty&&strlib_is_white(*ps); ps++; }
  }
  return( empty );

} // strlib_is_empty

/*---------------------------------------------------------------------------
NAME

    strlib_spn --- search a string for a set of characters

SYNOPSIS

   size_t strlib_spn(const char *s, const char *accept);

DESCRIPTION

   The function calculates the length of the initial segment of s which 
   consists entirely of characters in accept.

RETURN VALUE
  
   Number of characters in the initial segment of s which consist 
   only of characters from accept.

---------------------------------------------------------------------------*/
PUBLIC size_t strlib_spn(const char *s, const char *accept)
{ const char *ps;
  size_t spnlen=0;

  if ( (s)&&(accept) )
    for ( ps=s; *ps; ps++ ) {
      // break, if character not found
      if ( !strchr(accept,*ps) ) break;
      spnlen++;
    }

  return( spnlen );

} // strlib_spn

/*---------------------------------------------------------------------------
NAME

    strlib_uncomment --- removes comments 

SYNOPSIS

   char *strlib_uncomment ( char *s );

DESCRIPTION

   Truncates the string at the comment char ('#').

RETURN VALUE

   Pointer to the modified input string.

---------------------------------------------------------------------------*/
PUBLIC char *strlib_uncomment ( char *s )
{ char *ps;

  if (s) {

    ps=s;

    // copy ps2 to ps1 until end of string or comment char
    while ( *ps ) {
      if (*ps == '#') break;
      ps++;
    }

    // terminate string
    *ps='\0';

  }

  return( s );

} // strlib_uncomment

/*---------------------------------------------------------------------------
NAME

    strlib_trim --- removes leading and trailing white spaces

SYNOPSIS

   char *strlib_trim ( char *s );

DESCRIPTION

   Removes leading and trailing white spaces from the input string.

RETURN VALUE

   Pointer to the modified input string.

---------------------------------------------------------------------------*/
PUBLIC char *strlib_trim ( char *s )
{ char *ps1, *ps2;

  if (s) {
    ps1=ps2=s;

    // search first non-white character (nul is not a white space)
    while ( strlib_is_white( *ps2 ) ) ps2++;

    // copy ps2 to ps1 until end of string
    while ( *ps2 ) {
      *ps1++=*ps2++;
    }

    // terminate string
    *ps1='\0';

    // remove trailing white spaces
    while ( --ps1>=s ) {
      if ( !strlib_is_white(*ps1) ) break;
      *ps1='\0';
    }
  }

  return( s );

} // strlib_trim

/*---------------------------------------------------------------------------
NAME

    strlib_trimcrlf --- removes leading and trailing linefeeds

SYNOPSIS

   char *strlib_trimcrlf ( char *s );

DESCRIPTION

   Removes leading and trailing carriage returns and line feed characters
   from the input string.

RETURN VALUE

   Pointer to the modified input string.

---------------------------------------------------------------------------*/
PUBLIC char *strlib_trimcrlf ( char *s )
{ char *ps1, *ps2;

  if (!s) return("");

  if (s) {
    ps1=ps2=s;

    // search first non-white character (nul is not a white space)
    while ( strlib_is_crlf( *ps2 ) ) ps2++;

    // copy ps2 to ps1 until end of string
    while ( *ps2 ) {
      *ps1++=*ps2++;
    }

    // terminate string
    *ps1='\0';

    // remove trailing white spaces
    while ( --ps1>=s ) {
      if ( !strlib_is_crlf(*ps1) ) break;
      *ps1='\0';
    }
  }

  return( s );

} // strlib_trimcrlf

/*---------------------------------------------------------------------------
NAME

    strlib_collapse --- collapses to single white spaces

SYNOPSIS

   char *strlib_collapse ( char *s );

DESCRIPTION

   Collapses multiple white spaces to a single space ' '.

RETURN VALUE

   Pointer to the modified input string.

---------------------------------------------------------------------------*/
PUBLIC char *strlib_collapse ( char *s )
{ 
  char *ps1, *ps2;

  if (!s) return("");

  if (s) {
    ps1=ps2=s;

    while (*ps2) {
      // copy ps2 to ps1 until next white space
      while ( (*ps2) && (!strlib_is_white( *ps2 )) ) {
        *ps1++=*ps2++;
      }

      // write a single white space
      if (*ps2) {
        *ps1++ = ' ';
        ps2++;
      }

      // search first non-white character
      while ( strlib_is_white( *ps2 ) ) ps2++;

    }

    // terminate string
    *ps1='\0';
  }

  return( s );

} // strlib_strcollapse

/*---------------------------------------------------------------------------
NAME

    strlib_split --- splits string at white spaces

SYNOPSIS

   int strlib_split ( char *sv[], int sc, char * s );

DESCRIPTION

   Splits s white spaces to substrings and writes the pointers to the
   first sc-1 substrings to subsequent cells of sv[]. All white spaces 
   in s are replaced with character '\0' as terminator.  The length 
   of sv is sc. The pointers in sv are valid as long as the string s exists.

RETURN VALUE

   Number of substring pointers in sv[].

---------------------------------------------------------------------------*/
PUBLIC int strlib_split ( char *sv[], int sc, char * s )
{ char *nul = (char *) NULL;
  int cnt=0;
  char *ps;

  if ((!s)||(!sv)||(!sc)) return(0);

  ps=s;

  cnt=0;

  // skip leading white spaces
  while (strlib_is_white( *ps )) ps++;

  while (*ps) {

    if (cnt+1<sc) sv[cnt++]=ps;

    // search next white space
    while ( (*ps) && (!strlib_is_white( *ps )) ) ps++;

    if ( strlib_is_white( *ps ) ) {
      *ps='\0';
      ps++;
    }

    // skip leading white spaces
    while (strlib_is_white( *ps )) ps++;

  }

  // terminate
  if (cnt<sc) sv[cnt]=nul;

  return( cnt );

} // strlib_split

/*---------------------------------------------------------------------------
 NAME

    strlib_nconcat --- return a pointer to the concatenation of a and b

 SYNOPSIS

    char *strlib_nconcat( char *buffer, size_t buflen, 
                          const char *a, const char *b, size_t n )


 DESCRIPTION

    If the pointer to b and to buffer are identical, the function makes
    a temporary copy of b.
    If the pointers to a and to buffer are different the function copies the 
    string a to the destination buffer and appends afterwards the string b.
    If the pointers to a and b are identical the string b is immediately
    appended to the destination buffer without copying a. Apart from that
    the destination buffer may not overlap with the input strings a and b.
    The output string is truncated at buflen-1. The output string is 
    terminated with zero. Not more than n bytes are copied from b.

 RETURN VALUE

    pointer to buffer with concatenated strings

---------------------------------------------------------------------------*/
char *strlib_nconcat( char *buffer, size_t buflen,
                      const char *a, const char *b, size_t n )
{ size_t rest, len;
  char *pbuf;
  char *tmpbuf=NULL;
  const char *pb;
  if ((!buffer)||(!buflen)) return( buffer );
  pbuf=buffer;
  rest=buflen-1;
  if (!a) a="";
  if (!b) b="";
  if (pbuf==(char*)b) {
    tmpbuf=strlib_newstr(b);
    pb=tmpbuf;
  } else {
    pb=b;
  }
  if (pbuf!=(char*)a) {
    strncpy( pbuf, a, rest );
    len=strlen(pbuf);
    pbuf+=len;
    rest-=len;
  } else {
    // set pbuf after last char in buffer
    len=strlen(buffer);
    pbuf = buffer+len;
    rest = buflen-1-len;
  }
  if (rest<n) n = rest;
  strncpy( pbuf, pb, n);
  // set pbuf after last char that could have been written 
  pbuf += n;
  *pbuf = '\0'; // terminating zero
  FREE(tmpbuf);
  return ( buffer );
} /* strlib_nconcat */

/*---------------------------------------------------------------------------
 NAME

    strlib_concat --- return a pointer to the concatenation of a and b

 SYNOPSIS

    char *strlib_concat( char *buffer, size_t buflen, 
                         const char *a, const char *b )


 DESCRIPTION

    If the pointer to b and to buffer are identical, the function makes
    a temporary copy of b.
    If the pointers to a and to buffer are different the function copies the 
    string a to the destination buffer and appends afterwards the string b.
    If the pointers to the buffer and a are identical the string b is 
    immediately appended to the destination buffer without copying a. 
    Apart from this exception, the destination buffer must not overlap 
    with one of the input strings a and b.
    NULL strings a or b are replaced with empty strings.
    The output string is truncated at buflen-1. The output string is 
    terminated with zero. 

 RETURN VALUE

    pointer to buffer with concatenated strings

---------------------------------------------------------------------------*/
char *strlib_concat( char *buffer, size_t buflen, 
                     const char *a, const char *b )
{ size_t rest, len;
  char *pbuf;
  const char *pb;
  char *tmpbuf=NULL;
  if ((!buffer)||(!buflen)) return( buffer );
  buffer[buflen-1] = '\0'; // terminating zero
  pbuf=buffer;
  rest=buflen-1;
  if (!a) a="";
  if (!b) b="";
  if (pbuf==(char*)b) {
    tmpbuf=strlib_newstr(b);
    pb=tmpbuf;
  } else {
    pb=b;
  }
  if (pbuf!=(char*)a) {
    strncpy( pbuf, a, rest );
    len=strlen(pbuf);
    pbuf+=len;
    rest-=len;
  } else {
   // set pbuf after last char in buffer
   len=strlen(buffer);
   pbuf = buffer+len;
   rest = buflen-1-len;
  }
  strncpy( pbuf, pb, rest);
  FREE(tmpbuf);
  return ( buffer );
} /* strlib_concat */

/*+++------------------------------------------------------------------------
NAME

   strlib_param --- extract parameter n from string 

SYNOPSIS

   char * strlib_param ( char * buffer, size_t buflen,
                         const char *s, char separator, int parno );

DESCRIPTION
   Extracts substrings from string that are separated by separator and copies 
   them to buffer. If the input string does not contain a comma the full 
   input string is copied. 
   If substring parno is not available (char *) NULL is returned. parno 0 
   returns always the full string.
   Different to the function filename_parameter, the substring before the 
   first comma has the number 1.

   <0> = "<1><separator><2><separator><3><separator>..."

RETURN VALUE
   Pointer to the filled output buffer or NULL if parameter string is not
   available. A parameter between two separators that immediately follow 
   each other is returned as an empty string.

----------------------------------------------------------------------------*/
PUBLIC char * strlib_param ( char * buffer, size_t buflen, 
                             const char *s, char separator, int parno )
{ size_t len;
  char *tmp=NULL;
  const char *str1, *str2, *strend;
  int cnt;

  if (!s) goto strlib_param_error; // missing input string 

  if (parno<0) goto strlib_param_error; // parameter is not available

  len = strlen(s);
  if ( !( tmp = (char*) MALLOC( sizeof(char)*(len+1) )) )
    goto strlib_param_error;

  // copy s to temporary buffer
  strcpy (tmp, s);

  strend = tmp+len;
  cnt = 0; str1 = tmp; str2 = str1-1;
  while ( cnt++ < parno ) { // start numbering before the first separator
    // set str1 to start of parameter
    if ( str2 < strend ) {
      str1 = ++str2; // skip separator
    } else {
      goto strlib_param_error; // no more parameter available
    }
    // search end of parameter
    if ( (str2 = strchr (str1, separator)) == (char *) NULL) {
      str2 = strend;
    } 
  }

  // copy parameter
  len = MIN(buflen,(size_t) (str2-str1+1))-1;
  strncpy (buffer, str1, len);
  buffer[len] = '\0';

  FREE( tmp );

  return( buffer );

strlib_param_error:

  FREE( tmp );

  return(  (char *) NULL );

} // strlib_parameter 

/*---------------------------------------------------------------------------
NAME

    strlib_toupper --- converts all characters to upper case 

SYNOPSIS

   char *strlib_toupper ( char *s );

DESCRIPTION

  Converts all characters to upper case and returns pointer to input string.

RETURN VALUE

  Pointer to input string 

---------------------------------------------------------------------------*/
PUBLIC char *strlib_toupper ( char *s )
{ char *ps;

  if(s) {
    ps=s;
    while ( *ps ) { *ps=TOUPPER(*ps); ps++; }
  }

  return(s);

} // strlib_toupper

/*---------------------------------------------------------------------------
NAME

    strlib_tolower --- converts all characters to lower case

SYNOPSIS

   char *strlib_tolower ( char *s );

DESCRIPTION

  Converts all characters to lower case and returns pointer to input string.
       
RETURN VALUE
       
  Pointer to input string
       
---------------------------------------------------------------------------*/
PUBLIC char *strlib_tolower ( char *s )
{
  if (s) {
    char *ps;
    ps=s;
    while ( *ps ) { *ps=TOLOWER(*ps); ps++; }
  }

  return(s);

} // strlib_tolower

/*---------------------------------------------------------------------------
NAME

        strlib_ncasecmp ---  strncasecmp

SYNOPSIS

       int strlib_ncasecmp(const char *s1, const char *s2, size_t n);


DESCRIPTION
       The function compares the first n characters ot the two
       strings s1  and s2,  ignoring  the  case of the characters.
       It returns an integer less than, equal to, or greater than
       zero if s1 is less  than, matches, or is greater than s2.
       It can be used instead of the function strncasecmp if this
       function is not available.

       NULL pointers are treated as empty strings.

RETURN VALUE
       The function returns an integer less than, equal to, or
       greater than  zero.

---------------------------------------------------------------------------*/
PUBLIC int strlib_ncasecmp(const char *s1, const char *s2, size_t n)
{ int value=0;
  size_t s1len, s2len;
  char *s1n=NULL, *s2n=NULL;
  char *ps1n, *ps2n;
  size_t i;

  if (!s1) s1="";
  if (!s2) s2="";

  s1n = strlib_newstrn(s1,n);
  s1len = strlen(s1n);

  s2n = strlib_newstrn(s2,n);
  s2len = strlen(s2n);

  if (s1&&s2) {
    ps1n=s1n;
    for (i=0;i<s1len;i++) { *ps1n=TOLOWER(*ps1n);ps1n++; }
    ps2n=s2n;
    for (i=0;i<s2len;i++) { *ps2n=TOLOWER(*ps2n);ps2n++; }

    value = strcmp(s1n,s2n);
  }

  FREE(s1n);
  FREE(s2n);

  return(value);

} /* strlib_ncasecmp */

/*---------------------------------------------------------------------------
NAME

        strlib_casecmp ---  strcasecmp

SYNOPSIS

       int strlib_casecmp(const char *s1, const char *s2);


DESCRIPTION
       The function compares the two strings s1  and s2,  ignoring  
       the  case of the characters. It returns an integer less than, 
       equal to, or greater than zero if s1 is less  than, matches, 
       or is greater than s2.
       It can be used instead of the function strcasecmp if this
       function is not available.

       NULL pointers are treated as empty strings.

RETURN VALUE
       The function returns an integer less than, equal to, or
       greater than  zero.

---------------------------------------------------------------------------*/
PUBLIC int strlib_casecmp(const char *s1, const char *s2)
{ size_t s1len, s2len;
  size_t n;
  int value=0;

  if (!s1) s1="";
  if (!s2) s2="";

  s1len = strlen(s1);
  s2len = strlen(s2);

  n=(s1len>s2len)?s1len:s2len;

  value = strlib_ncasecmp(s1, s2, n);

  return(value);

} /* strlib_casecmp */

/*---------------------------------------------------------------------------
NAME

        strlib_cmp ---  strcmp

SYNOPSIS

       int strlib_cmp(const char *s1, const char *s2)


DESCRIPTION
       The function compares the two strings s1  and s2 using strcmp.
       NULL pointers are treated as empty strings.

RETURN VALUE
       The function returns an integer less than, equal to, or
       greater than  zero.

---------------------------------------------------------------------------*/
PUBLIC int strlib_cmp(const char *s1, const char *s2)
{ 
  if (!s1) s1="";
  if (!s2) s2="";

  return(strcmp(s1, s2));

} /* strlib_cmp */

/*---------------------------------------------------------------------------
NAME

        strlib_ncmp ---  strncmp

SYNOPSIS

       int strlib_ncmp(const char *s1, const char *s2, size_t n)


DESCRIPTION
       The function compares the first n characters ot the two
       strings s1  and s2 using strncmp.
       It returns an integer less than, equal to, or greater than
       zero if s1 is less  than, matches, or is greater than s2.

       NULL pointers are treated as empty strings.

RETURN VALUE
       The function returns an integer less than, equal to, or
       greater than  zero.

---------------------------------------------------------------------------*/
PUBLIC int strlib_ncmp(const char *s1, const char *s2, size_t n)
{
  if (!s1) s1="";
  if (!s2) s2="";

  return(strncmp(s1, s2, n));

} /* strlib_ncmp */

/*---------------------------------------------------------------------------
NAME

  strlib_wildcmp --- compares a key with a wild card key

SYNOPSIS

  int strlib_wildcmp( const char *key, const char *wild );

DESCRIPTION
  The routines compares the characters of wild with the characters of key.
  If '*' is found in wild the comparison of the remaining characters in
  key is skipped until the first character after '*' is found in key. There,
  the comparison is continued. However, the function may not return the
  expected result if '*' is not the last character in wild and if key
  contains several times the character after '*', e.g.

  key="abcabc", wild = "abcabc" -> 0
  key="abcabc", wild = "abc*"   -> 0
  key="abcabc", wild = "*abc"   -> 1 comparison is stopped after the first
                                     occurrence of abc
  key="abcabc", wild = "*abc*"  -> 0
  key="abcabc", wild = "*cabc"  -> 0

RETURN VALUE
The routine returnes as result:

   input             output
   key < wild     -1
   key = wild      0
   key > wild     +1

TODO: improve!

---------------------------------------------------------------------------*/
int strlib_wildcmp( const char *key1, const char *wild )
{ const char *pc1, *pc2;
  int value=0;

  if ((!key1) || (!wild) ) {
    if (wild) return(-1); else { if (key1) return(1); else return(0); }
  }

  pc1 = key1; pc2 = wild;

  value=0;

  do {
    if ((*pc2)=='*') { // wild card
      do pc2++; while ((*pc2)=='*'); // get first character after wild card
      while ((*pc1)!=(*pc2)) {
        if (*pc1) pc1++;
        else break;
      }
    }
    if (*pc2) {
      if ( (*pc1)!=(*pc2) ) {
        if ( (*pc1)<(*pc2) ) value=-1; else value=1;
      }
    } else {
      if (*pc1) value=1;
    }
  } while ( (*pc1++) && (*pc2++) );

  return( value );

} // strlib_wildcmp

/*---------------------------------------------------------------------------
NAME

      truncatdigit --- terminate s at the first digit including sign

SYNOPSIS

      char *truncatdigit( char *s, long *value, char **tail );


DESCRIPTION

      The 'truncatdigit' is used to split a string in three parts, one
      before the first number, the value of the number and the tail after
      the number. If s contains a digit it will be modified.

      The `truncatdigit' function searches in s for the first substring
      starting with a digit character. If possible, it extends the
      substring to a preceeding sign character. Then it converts the
      substring to a long integer number. The tail after the converted
      integer substring is returned and s is modified in such a way that
      the substring before the number is terminated with '\0'.


      mode 0: do not include sign characters
           1: include sign characters

RETURN VALUE

      The pointer s is returned.

---------------------------------------------------------------------------*/
char *truncatdigit( char *s, long *value, char **tail, int mode )
{
  long number=0;
  char *rest=NULL;

  if (s) {
    char *ps;
    char *pre=NULL;

    for ( ps=s; *ps; ps++ ) {
      if ( ISDIGIT( *ps ) ) {
        if (mode) {
          if ( (pre) && ( (*pre=='-') || (*pre=='+') ) ) ps = pre;
        }
        break;
      }
      pre=ps;
    }

    if (*ps) {
      // convert number
      number = strtol(ps, &rest ,10);
      // terminate preceeding substring with '\0'
      *ps = '\0';
    }

  }

  if (value) *value = number;
  if (tail)  *tail  = rest;

  return(s);

} /* truncatdigit */

/*---------------------------------------------------------------------------
NAME

        _numcmp --- number string comparison

SYNOPSIS

       int _numcmp(const char *s1, const char *s2, int lowercase)


DESCRIPTION
       The function compares the two strings s1 and s2 using number search.
       If lowercase is 0 the strings are not converted to lowercase 
       before comparison.
       NULL pointers are treated as empty strings.

RETURN VALUE
       The function returns an integer less than, equal to, or
       greater than  zero.

---------------------------------------------------------------------------*/
int _numcmp(const char *s1, const char *s2, int lowercase )
{ int value=0, diff=0;
  long i1=0, i2=0;
  int mode=1;

  char *buf1=NULL, *start1=NULL, *tail1=NULL;
  char *buf2=NULL, *start2=NULL, *tail2=NULL;

  buf1 = strlib_trim( strlib_newstr( s1 ) );
  buf2 = strlib_trim( strlib_newstr( s2 ) );

  if (lowercase) {
    strlib_tolower( buf1 );
    strlib_tolower( buf2 );
  }

  start1 = buf1;
  start2 = buf2;

  while (start1&&start2) {

    truncatdigit( start1, &i1, &tail1, mode );
    truncatdigit( start2, &i2, &tail2, mode );

    mode=0; // next time do not include sign

    value = strcmp(start1,start2);
    if (value!=0) break;

    diff = i1 - i2;
    if (diff>INT_MAX) {
      value = (int) INT_MAX;
    } else if (diff<INT_MIN) {
      value = (int) INT_MIN;
    } else {
      value = (int) diff;
    }

    if (value!=0) break;

    start1 = tail1;
    start2 = tail2;

  }

  FREE( buf1 );
  FREE( buf2 );

  return(value);

} /* _numcmp */

/*---------------------------------------------------------------------------
NAME

        strlib_numcmp --- number string comparison

SYNOPSIS

       int strlib_numcmp(const char *s1, const char *s2)


DESCRIPTION
       The function compares the two strings s1 and s2 using number search.
       NULL pointers are treated as empty strings.

RETURN VALUE
       The function returns an integer less than, equal to, or
       greater than  zero.

---------------------------------------------------------------------------*/
PUBLIC int strlib_numcmp(const char *s1, const char *s2 ) 
{ return( _numcmp(s1, s2, 0 ) );
} /* strlib_numcmp */

/*---------------------------------------------------------------------------
NAME

        strlib_numcasecmp --- case insensitive number string comparison

SYNOPSIS

       int strlib_numcasecmp(const char *s1, const char *s2)


DESCRIPTION
       The function converts the strings s1 and s2 to lower case and 
       compares them using number search.
       NULL pointers are treated as empty strings.

RETURN VALUE
       The function returns an integer less than, equal to, or
       greater than  zero.

---------------------------------------------------------------------------*/
PUBLIC int strlib_numcasecmp(const char *s1, const char *s2 )
{ return( _numcmp(s1, s2, 1 ) );
} /* strlib_numcmp */

/*---------------------------------------------------------------------------
NAME

      strlib_tok_r --- split a string in tokens 

SYNOPSIS

      char *strlib_tok_r(char *s, const char *sep, char **lasts);


DESCRIPTION

      The `strlib_tok_r' function is used to isolate sequential tokens in a 
      NULL-terminated string `*s'. It replaces the string.h function 
      `strtok_r' when it is not available. 

      These tokens are delimited in the string by at least one of the 
      characters in `*sep'. The first time that `strlib_tok_r' is called, 
      `*s' must be specified; subsequent calls, wishing to obtain further 
      tokens from the same string, must pass a NULL pointer instead. The 
      separator string, `*sep', must be supplied each time and may change 
      between calls.

      The  `strlib_tok_r'  function  returns  a pointer to the beginning of 
      each subsequent token in the string, after replacing the separator 
      character itself with a null character. When no more tokens remain, a 
      NULL pointer is returned. The placeholder `*lasts' must be supplied by 
      the caller and must not be modified.

      The `strlib_tok_r' function has the same behavior as `strlib_tok', 
      except that a pointer to placeholder `*LASTS' must be supplied by 
      the caller.

RETURN VALUE

     `strlib_tok_r' returns a pointer to the next token, or `NULL' if no 
     more tokens can be found.

---------------------------------------------------------------------------*/
PUBLIC char *strlib_tok_r(char *s, const char *sep, char **lasts)
{ char *pstart,*ps,*retval=(char*) NULL;

  if (s) {
    /* skip starting separators */
    for (pstart=s;*pstart;pstart++) {
      if (!(strchr(sep,(int) *pstart))) {
        retval=pstart;
        break;
      }
    }
  } else {
    pstart=(lasts)?*lasts:(char*)NULL;
    retval=pstart;
  }
  
  *lasts=(char*)NULL;
  if (pstart) {
    for (ps=pstart;*ps;ps++) {
      if (strchr(sep,(int) *ps)) {
        *ps='\0';
        *lasts = ps+1;
        break;
      }
    }
  }

  return( retval );
  
} /* strlib_tok_r */

/*---------------------------------------------------------------------------
NAME

      strlib_tok --- split a string in tokens

SYNOPSIS

      char *strlib_tok(char *s, const char *sep);


DESCRIPTION

      The `strlib_tok_r' function is used to isolate sequential tokens in a
      NULL-terminated string `*s'. It replaces the string.h function `strtok'
      when it is not available.

      These tokens are delimited in the string by at least one of the
      characters in `*sep'. The first time that `strlib_tok_r' is called,
      `*s' must be specified; subsequent calls, wishing to obtain further
      tokens from the same string, must pass a NULL pointer instead. The
      separator string, `*sep', must be supplied each time and may change
      between calls.

      The  `strlib_tok'  function  returns  a pointer to the beginning of
      each subsequent token in the string, after replacing the separator
      character itself with a null character. When no more tokens remain, a
      NULL pointer is returned. 

      The `strlib_tok' function has the same behavior as `strlib_tok_r', 
      except that the placeholder `*LASTS' cannot be specified. It is kept 
      in an internal static buffer for subsequent calls and consequently the 
      function is not thread safe.

RETURN VALUE

     `strlib_tok' returns a pointer to the next token, or `NULL' if no
     more tokens can be found.

---------------------------------------------------------------------------*/
PUBLIC char *strlib_tok(char *s, const char *sep)
{ static char *lasts;

  return( strlib_tok_r(s, sep, &lasts) );

} /* strlib_tok */

/*---------------------------------------------------------------------------
NAME

     strlib_newstr --- copy input string to new allocated memory

SYNOPSIS

     char *strlib_newstr( const char *string );

DESCRIPTION

     Makes a copy of the input string to new allocated memory. The memory 
     must be released with FREE( ) when it is not needed any more.

     A NULL string is treated like an empty string.

RETURN VALUE

     Pointer to allocated string or NULL in case of error.

---------------------------------------------------------------------------*/
PUBLIC char *strlib_newstr( const char *string )
{ char *new;

  if (!string) string="";

  if ( (new=MALLOC( strlen(string)+1 )) )
    strcpy(new,string);
  return(new);

} /* strlib_newstr */

/*---------------------------------------------------------------------------
NAME

     strlib_free --- release memory of allocated string

SYNOPSIS

     void *strlib_free( void *string );

DESCRIPTION

     Frees the memory of a string that has been previously allocated
     in strlib, e.g. using strlib_newstr or strlib_newstrn.

RETURN VALUE

     Returns NULL.

---------------------------------------------------------------------------*/
PUBLIC void *strlib_free( void *string )
{
  FREE(string);
  return( NULL );

} /* strlib_free */

/*---------------------------------------------------------------------------
NAME

     strlib_newstrn --- copy input string to new allocated memory

SYNOPSIS

     char *strlib_newstrn( const char *string, size_t n );

DESCRIPTION

     Allocates an output string with n+1 0-bytes and copies the first n
     bytes (at most) of the input string to the allocated memory. 
     Copies the first n (at most) characters of the input string to the
     output string. The allocated memory of the returned string pointer
     must be released with FREE().

     A NULL string is converted to an empty string.

RETURN VALUE

     Pointer to allocated string or NULL in case of error.

---------------------------------------------------------------------------*/
PUBLIC char *strlib_newstrn( const char *string, size_t n )
{ char *new;

  if (!string) string="";

  // CALLOC(size_t nmemb, size_t size);
  if ( (new=CALLOC( n+1, 1 )) ) {
    strncpy(new,string,n);
  }
  return(new);

} /* strlib_newstrn */

/*---------------------------------------------------------------------------
NAME

  strlib_c_count

SYNOPSIS

  size_t strlib_c_count(const char *str, int c );

DESCRIPTION

  Returns the number of occurences of c in str.
  If str is NULL, 0 is returned.

RETURN VALUE

  Number of occurences of c in str.

---------------------------------------------------------------------------*/
PUBLIC size_t strlib_c_count(const char *str, int c )
{ size_t counts=0;

  if (str) {
    const char *ps=str;

    while (*ps) {
      if ((int) *ps==c) counts++;
      ps++;
    }
  }

  return(counts);

} /* strlib_c_count */

/*---------------------------------------------------------------------------
NAME

  strlib_is_list

SYNOPSIS

  int strlib_is_list( const char *str );

DESCRIPTION

  Checks whether is encapsulated in square brackets. If yes, 1 is returned,
  otherwise 0. Leading and trailing white space is removed.

RETURN VALUE
  0: str is not a list
  1: str is a list

---------------------------------------------------------------------------*/
PUBLIC int strlib_is_list( const char *str )
{ int is_list=0;

  if (str) {
    size_t len;
    len=strlen(str);
    if (len>1) {
      char *str_tmp=NULL;
      str_tmp = strlib_newstr(str);
      if (str_tmp) {
        strlib_trim(str_tmp);
        len = strlen(str_tmp);
        if ((str_tmp[0]=='[')&&(str_tmp[len-1]==']')) is_list=1;
        FREE(str_tmp);
      }
    }
  }

  return(is_list);
  
} /* strlib_is_list */

/*---------------------------------------------------------------------------
NAME

  strlib_freestringarray

SYNOPSIS

  const char **strlib_strlib_freestringarray( const char **stringarray, size_t nelements );

DESCRIPTION

  If nelements is negative, all non-zero elements are released until a NULL pointer
  is reached, otherwise nelements string elements of stringarray are released.
  Finally, stringarray is released.

---------------------------------------------------------------------------*/
PUBLIC const char **strlib_freestringarray( const char **stringarray, size_t nelements )
{
  if (stringarray) {
    char **pps;
    pps=(char **) stringarray;

    if (nelements>=0) {
      size_t i;
      for (i=0;i<nelements;i++) {
        FREE(*pps);
        pps++;
      }
    } else {
      while (*pps) {
        FREE(*pps);
        pps++; // stringarray is terminated with NULL
      }
    }
    FREE(stringarray);
  }

  return(NULL);

} /* strlib_freestringarray */

/*---------------------------------------------------------------------------
NAME

  strlib_list2stringarray

SYNOPSIS

  const char **strlib_list2stringarray( const char *str, size_t *nelements );

DESCRIPTION

  Converts a list str="[item0,item1,item2,...]" to an array array[0]="item0",
  array[1]="item1", array[2]="item2", ..., array[nelements-1] with nelements.

  The string str must start with '[' and end with ']'. The substring inside
  the brackets is split into comma separated substrings. Each of these 
  substring is copied to an element of the returned array.
  The memory of the returned array and the memory of each of its elements is 
  allocated and must be released with strlib_freestringarray for avoiding 
  resource leaks. The last array element is a NULL pointer.

RETURN VALUE

  If str is a list the returned value is an nelements array with strings,
  e.g. stringarray[0] = firstitem, stringarray[nelements-1]=lastitem.

  The last element, stringarray[nelements], is set to NULL as terminator.

  If str is NULL or not a list the returned value is NULL.

---------------------------------------------------------------------------*/
PUBLIC const char **strlib_list2stringarray( const char *str, size_t *nelements )
{
  size_t elements=-1;
  const char **stringarray=NULL;
  char *str_tmp=NULL;

  if (strlib_is_list(str)) {
    char *ps=NULL;

    str_tmp=strlib_newstr(str);
    // search position of '[' 
    ps=strchr(str_tmp,'[');
    if (ps) {
      char *pe=NULL;
      ps++; // start after '['
      // search position of ']' and set to '\0'
      pe=strrchr(ps,']');
      if (pe) {
        *pe='\0'; // replace ']' with '\0'

        // count commas in ps (elements=commas+1)
        elements = strlib_c_count(ps,',')+1;

        // initialize string_array elements with NULL and terminate with NULL
        stringarray = (const char**) CALLOC (elements+1,sizeof (char *));

        // copy substrings to stringarray
        char *saveptr=NULL;
        char *item;
        size_t i=0;

        // item = strtok_r(ps,",",&saveptr);
        item = strlib_tok_r(ps,",",&saveptr);
      
        while (item) {
          size_t len;

          if (i>=elements) {
            printf("FATAL strlib_list2stringarray: too many items (i=%ld,elements=%ld)\n",(long)i, (long)elements);
            goto strlib_list2stringarray_error;
          }
        
          // remove encapsulation between quotes
          len = strlen(item);
          if (len>1) {
            if ( ((item[0]=='\'')&&(item[len-1]=='\''))||((item[0]=='"')&&(item[len-1]=='"')) ) {
              item[len-1]='\0';
              item++;
            }
          }

          stringarray[i++] = strlib_newstr(item);

          // item=strtok_r(NULL,",",&saveptr);
          item=strlib_tok_r(NULL,",",&saveptr);
        }

      }
    }
  }
  
  FREE(str_tmp);

  if (nelements) *nelements=elements;
  return(stringarray);

strlib_list2stringarray_error:

  FREE(str_tmp);
  stringarray=strlib_freestringarray( stringarray, elements );

  if (nelements) *nelements=-1;
  return(stringarray);

} /* strlib_list2stringarray */

  // replace substrings of type num((<expression>)) with the
  // calculated expression values
PUBLIC char *strlib_numberstring( char *buffer, size_t buflen, 
                        const char *string, int *pstatus );

# define NUMBLOCKSTART "num(("
# define NUMBLOCKEND "))"
# define NDIGITS 15

/*+++------------------------------------------------------------------------
NAME

   _numberstring --- copy input string to output string and calculate numbers

SYNOPSIS
   *ps points to the current position in the input string (updated)
   *pb points to the current position in the output string (updated)
       The output string must be initialized over its useful length with
       non-zero characters and must be terminated by character NULL.

   The pointer to the output string is returned. The pointers *pb and *ps are
   incremented. *pb is only incremented until **pb is character NULL.
   The parameters itself can be header values.

----------------------------------------------------------------------------*/
char *_numberstring(char **pb, const char **ps, int *pstatus)
{ int status=0;
  char *value=NULL;
  if ((ps)&&(pb)) {
    int        err=0;
    char       *ppb=*pb;
    const char *pps=*ps;
    const char *start=NULL;
    const char *tail=NULL;

    value = ppb;

    start=strstr(pps,NUMBLOCKSTART);

    while (start) {
      double number;
      // copy pps to ppb until start
      while ((*ppb)&&(pps<start)) {
        *ppb = *pps; (ppb)++; (pps)++;
      }
      // read number after NUMBLOCKSTART-1 (start with '(')
      number=num_str2double (start+strlen(NUMBLOCKSTART)-1, &tail, &err);
      if (!err) {
        if (!strncmp((tail-1),NUMBLOCKEND,strlen(NUMBLOCKEND))) {
          // NUMBLOCKEND found and number conversion OK
          char numbuf[1024];
          size_t numbuflen=1024;
          num_double2str(numbuf,numbuflen,number,NULL,NDIGITS,&err);
          if (!err) {
            char *pnumbuf=numbuf;
            numbuf[numbuflen-1]='\0'; // force NUL termination
            // copy number to output buffer
            while ((*ppb)&&(*pnumbuf)) {
              *ppb = *pnumbuf; (ppb)++; (pnumbuf)++;
            }
            // increment pointer to input buffer
            pps = tail+1;
            start=strstr(pps,NUMBLOCKSTART);
          } else start=NULL;
        } else start=NULL;
      } else {
        // could be a forgotten NUMBLOCKSTART,
        // copy pps to ppb until start of tail
        while ((*ppb)&&(pps<tail)) {
          *ppb = *pps; (ppb)++; (pps)++;
        }
        start=strstr(pps,NUMBLOCKSTART);
      }
    }
    // NUMBLOCKSTART not found, copy pps to ppb
    // conversion error, copy pps to ppb
    while ((*ppb)&&(*pps)) {
      *ppb = *pps; (ppb)++; (pps)++;
    }
    *ppb = '\0'; // terminate with NUL
    if (pb) *pb = ppb;
    if (ps) *ps = pps;
  }
  if (pstatus) *pstatus = status;
  return(value);
} // _numberstring

/*
 * replace substrings of type num((<expression>)) with the
 * calculated expression values.  If the
 * calculation fails nothing will be changed.
 */
/*---------------------------------------------------------------------------
NAME

  strlib_numberstring

SYNOPSIS

  char *strlib_numberstring( char *buffer, size_t buflen,
                                  const char *string, int *pstatus );

DESCRIPTION

  Replace substrings of type num((<expression>)) with the calculated 
  expression values. If the calculation fails nothing will be changed.
  Nesting of nums is not supported.

EXAMPLE

  "abcnum((1+2+3))+num((3+4))" => "abc6+7"
  "...num((1+num((3+4))))..."  => "...num((1+7))..."

RETURN VALUE

  The pointer to the output string in the output buffer is returned.

---------------------------------------------------------------------------*/
PUBLIC char *strlib_numberstring( char *buffer, size_t buflen,
                                  const char *string, int *pstatus )
{
  int status=0;
  char *pb, *value;
  const char *ps;
  size_t j;

  // prepare output string
  pb = buffer;
  for (j=1;j<buflen;j++)
    *pb++ = ' '; // fill with spaces
  *pb = '\0'; // terminate with NUL

  // copy
  pb = buffer;
  ps = string;
  value = _numberstring( &pb, &ps, &status );

  if (pstatus) *pstatus=status;

  return( value );

} // strlib_numberstring
