/*
 *   Project: The SPD Image correction and azimuthal regrouping
 *                      http://forge.epn-campus.eu/projects/show/azimuthal
 *
 *   Copyright (C) 2005-2010 European Synchrotron Radiation Facility
 *                           Grenoble, France
 *
 *   Principal authors: P. Boesecke (boesecke@esrf.fr)
 *                      R. Wilcke (wilcke@esrf.fr)
 *
 *   This program is free software: you can redistribute it and/or modify
 *   it under the terms of the GNU Lesser General Public License as published
 *   by the Free Software Foundation, either version 3 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU Lesser General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   and the GNU Lesser General Public License  along with this program.
 *   If not, see <http://www.gnu.org/licenses/>.
 */

# define IODBG_VERSION      "iodbg : V1.05 Peter Boesecke 2017-06-30"
/*+++***********************************************************************
NAME

   iodbg.c

SYNOPSIS

   const char *iodbg_version;
   int iodbg( long mask );
   const char *iodbg_usage2str( void );
   long iodbg_debug_set( long debug );
   long iodbg_debug_or( long debug );
   long iodbg_debug_and( long debug );
   long iodbg_debug ( void );
   long iodbg_level ( void );
   long iodbg_level_set( long level );

DESCRIPTION

   Utility for optional messages, e.g. for debugging.


HISTORY
    2017-02-15  PB V1.00
    2017-02-28  PB V1.01
    2017-05-22  PB V1.02 IODBG_MEMALLOC, IODBG_MEMSIZE, IODBG_MEMLIST,
                         IODBG_MEMFILEPOS
    2017-05-23  PB V1.03 Show ioalloc debug options only for IOALLOC==1
    2017-05-29  PB V1.04 IODBG_SHOWNODESEARCH
    2017-06-30  PB V1.05 IODBG_EDF, IODBG_H5

***************************************************************************/

# include "iodbg.h"

/***************************************************************************
* Private part                                                             *
***************************************************************************/

# define IODBG_BUFLEN (2048)
char IODBG_Usage[IODBG_BUFLEN];

PRIVATE long      IODbgDebug = 0;
PRIVATE long      IODbgLevel = LONG_MAX;

/*+++------------------------------------------------------------------------
NAME

   iodbg_version

SYNOPSIS

   const char *iodbg_version  ( void )

DESCRIPTION
   Returns a pointer to the version string of the module iodbg

RETURN VALUE
   Pointer to the version string
-------------------------------------------------------------------------+*/
const char *iodbg_version   ( void )                                 /*---*/
{
  return ( IODBG_VERSION );

} /* iodbg_version */

/*--------------------------------------------------------------------------
NAME

  iodbg --- returns 0 or 1 depending on flags

SYNOPSIS

  int iodbg( long mask );

DESCRPTION

  Returns 1 if all set bits of mask are also set in IODbgDebug,
  otherwise 0 is returned.

--------------------------------------------------------------------------*/
int iodbg( long mask )
{
  if ( (IODbgDebug&mask) == mask )
    return(1);
  else return(0);

} // iodbg

/*--------------------------------------------------------------------------
NAME

  iodbg_usage2str --- return debug mode usage string

SYNOPSIS

  const char *iodbg_usage2str( void );

DESCRPTION

  Return debug mode usage string.

--------------------------------------------------------------------------*/
const char *iodbg_usage2str( void )
{ 
#if IOALLOC == 1
    snprintf(IODBG_Usage,IODBG_BUFLEN,
    "\n  verbose:0x%lx,showatt:0x%lx,showattval:0x%lx,showdata:0x%lx,showdataval:0x%lx,\
     \n  showheader:0x%lx,debug1:0x%lx,debug2:0x%lx,hierarchy:0x%lx,compression:0x%lx,\
     \n  raster:0x%lx,history:0x%lx,header:0x%lx,units:0x%lx,constants:0x%lx,programs:0x%lx,\
     \n  shownodesearch:0x%lx,edf:0x%lx,h5:0x%lx,\
     \n  memoryallocation:0x%lx,memorysize:0x%lx,memorylist:0x%lx,fileposition:0x%lx",
    IODBG_VERBOSE, IODBG_SHOWATT, IODBG_SHOWATTVAL, IODBG_SHOWDATA,
    IODBG_SHOWDATAVAL, IODBG_SHOWHEADER, IODBG_DEBUG1, IODBG_DEBUG2,
    IODBG_SHOWHIERARCHY,IODBG_COMPRESSION,IODBG_RASTER,IODBG_HISTORY,
    IODBG_HEADER,IODBG_NUMUNITS,IODBG_NUMCONSTANTS,IODBG_NUMPROGRAMS,
    IODBG_SHOWNODESEARCH,IODBG_EDF,IODBG_H5,
    IODBG_MEMALLOC,IODBG_MEMSIZE,IODBG_MEMLIST,IODBG_MEMFILEPOS);
#else
    snprintf(IODBG_Usage,IODBG_BUFLEN,
    "\n  verbose:0x%lx,showatt:0x%lx,showattval:0x%lx,showdata:0x%lx,showdataval:0x%lx,\
     \n  showheader:0x%lx,debug1:0x%lx,debug2:0x%lx,hierarchy:0x%lx,compression:0x%lx,\
     \n  raster:0x%lx,history:0x%lx,header:0x%lx,units:0x%lx,constants:0x%lx,programs:0x%lx,\
     \n  shownodesearch:0x%lx,edf:0x%lx,h5:0x%lx",
    IODBG_VERBOSE, IODBG_SHOWATT, IODBG_SHOWATTVAL, IODBG_SHOWDATA,
    IODBG_SHOWDATAVAL, IODBG_SHOWHEADER, IODBG_DEBUG1, IODBG_DEBUG2,
    IODBG_SHOWHIERARCHY,IODBG_COMPRESSION,IODBG_RASTER,IODBG_HISTORY,
    IODBG_HEADER,IODBG_NUMUNITS,IODBG_NUMCONSTANTS,IODBG_NUMPROGRAMS,
    IODBG_SHOWNODESEARCH,IODBG_EDF,IODBG_H5);
#endif // IOALLOC == 1
 
  IODBG_Usage[IODBG_BUFLEN-1]='\0'; // force termination
  return(IODBG_Usage);
} // iodbg_usage2str

int iodbg_fprint_debug( FILE *out )
{ fprintf(out,"debug                           = 0x%lx\n", IODbgDebug);
  fprintf(out,"print verbose messages          = %d\n", IODbgDebug&IODBG_VERBOSE?1:0);
  fprintf(out,"show attribute names            = %d\n", IODbgDebug&IODBG_SHOWATT?1:0);
  fprintf(out,"show attribute values           = %d\n", IODbgDebug&IODBG_SHOWATTVAL?1:0);
  fprintf(out,"show data names                 = %d\n", IODbgDebug&IODBG_SHOWDATA?1:0);
  fprintf(out,"show data values                = %d\n", IODbgDebug&IODBG_SHOWDATAVAL?1:0);
  fprintf(out,"show header contents            = %d\n", IODbgDebug&IODBG_SHOWHEADER?1:0);
  fprintf(out,"print debug level1 messages     = %d\n", IODbgDebug&IODBG_DEBUG1?1:0);
  fprintf(out,"print debug level2 messages     = %d\n", IODbgDebug&IODBG_DEBUG2?1:0);
  fprintf(out,"show file hierarchy             = %d\n", IODbgDebug&IODBG_SHOWHIERARCHY?1:0);
  fprintf(out,"show data compression           = %d\n", IODbgDebug&IODBG_COMPRESSION?1:0);
  fprintf(out,"show raster configuration       = %d\n", IODbgDebug&IODBG_RASTER?1:0);
  fprintf(out,"show history                    = %d\n", IODbgDebug&IODBG_HISTORY?1:0);
  fprintf(out,"show header                     = %d\n", IODbgDebug&IODBG_HEADER?1:0);
  fprintf(out,"show units                      = %d\n", IODbgDebug&IODBG_NUMUNITS?1:0);
  fprintf(out,"show constants                  = %d\n", IODbgDebug&IODBG_NUMCONSTANTS?1:0);
  fprintf(out,"show programs                   = %d\n", IODbgDebug&IODBG_NUMPROGRAMS?1:0);
  fprintf(out,"show node search                = %d\n", IODbgDebug&IODBG_SHOWNODESEARCH?1:0);
#if IOALLOC == 1
  fprintf(out,"show memory allocation and free = %d\n", IODbgDebug&IODBG_MEMALLOC?1:0);
  fprintf(out,"show used memory size           = %d\n", IODbgDebug&IODBG_MEMSIZE?1:0);
  fprintf(out,"show memory allocation list     = %d\n", IODbgDebug&IODBG_MEMLIST?1:0);
  fprintf(out,"show file position              = %d\n", IODbgDebug&IODBG_MEMFILEPOS?1:0);
#endif // IOALLOC == 1
  fprintf(out,"level                           = %ld\n", IODbgLevel);
  return(0);
} // iodbg_fprint_debug

/*--------------------------------------------------------------------------
NAME

  iodbg_debug_set --- set / reset module iodbg to debug mode

SYNOPSIS

  long iodbg_debug_set( long debug );

DESCRPTION

  Sets/resets module iodbg and sub-modules to debug mode
  Returns the new DEBUG mode

--------------------------------------------------------------------------*/
long iodbg_debug_set( long debug )
{ IODbgDebug = debug;

  if (IODbgDebug&IODBG_DEBUG1) iodbg_fprint_debug( stdout );

  return(IODbgDebug);

} // iodbg_debug_set

/*--------------------------------------------------------------------------
NAME

  iodbg_debug_or --- ors debug to the debug flags

SYNOPSIS

  long iodbg_debug_or( long debug );

DESCRPTION

  Sets the debug flags. If they are already set, nothing happens.
  Returns the new DEBUG mode

--------------------------------------------------------------------------*/
long iodbg_debug_or( long debug )
{ 
  IODbgDebug |= debug;

  if (IODbgDebug&IODBG_DEBUG1) iodbg_fprint_debug( stdout );

  return(IODbgDebug);

} // iodbg_debug_or

/*--------------------------------------------------------------------------
NAME

  iodbg_debug_and --- ands debug to the debug flags

SYNOPSIS

  long iodbg_debug_and( long debug );

DESCRPTION

  Sets the debug flags. If they are already set, nothing happens.
  Returns the new DEBUG mode

--------------------------------------------------------------------------*/
long iodbg_debug_and( long debug )
{
  IODbgDebug &= debug;

  if (IODbgDebug&IODBG_DEBUG1) iodbg_fprint_debug( stdout );

  return(IODbgDebug);

} // iodbg_debug_and


/*--------------------------------------------------------------------------
NAME

  iodbg_debug --- return debug value

SYNOPSIS

  long iodbg_debug ( void );

--------------------------------------------------------------------------*/
long iodbg_debug ( void )
{ return( IODbgDebug );
} // iodbg_debug

/*--------------------------------------------------------------------------
NAME

  iodbg_level --- return debug level

SYNOPSIS

  long iodbg_level ( void );

--------------------------------------------------------------------------*/
long iodbg_level ( void )
{ return( IODbgLevel );
} // iodbg_level

/*--------------------------------------------------------------------------
NAME

  iodbg_level_set --- set level

SYNOPSIS

  long iodbg_level_set( long level );

DESCRPTION

  Set iodbg level.

--------------------------------------------------------------------------*/
long iodbg_level_set( long level )
{ IODbgLevel = level;
  return(IODbgLevel);
} // iodbg_level_set

/*****************************************************************************/
