/*
 *   Project: The SPD Image correction and azimuthal regrouping
 *                      http://forge.epn-campus.eu/projects/show/azimuthal
 *
 *   Copyright (C) 2005-2010 European Synchrotron Radiation Facility
 *                           Grenoble, France
 *
 *   Principal authors: P. Boesecke (boesecke@esrf.fr)
 *                      R. Wilcke (wilcke@esrf.fr)
 *
 *   This program is free software: you can redistribute it and/or modify
 *   it under the terms of the GNU Lesser General Public License as published
 *   by the Free Software Foundation, either version 3 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU Lesser General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   and the GNU Lesser General Public License  along with this program.
 *   If not, see <http://www.gnu.org/licenses/>.
 */

# define IOALLOC_VERSION  "ioalloc : V0.7 Peter Boesecke 2017-09-27"
/*+++***********************************************************************
NAME

   ioalloc.c

DESCRIPTION

 If IOALLOC is defined, an alternative set of routines for memory
 management is used. The following routines are defined:
 
   void *iomalloc(size_t size);
   void *iocalloc(size_t nmemb, size_t size);
   void *iorealloc(void *ptr, size_t size);
   void iofree(void *ptr);
   void iomemsize(const char *prompt);
   void iomemlist(const char *prompt);

 - iomalloc(), iocalloc(), iorealloc(), iofree() replace the standard routines
    malloc(), calloc(), realloc() and free();
 - iomemsize() prints the total allocated memory size on the current line (no line feed).
 - iomemlist() prints the list of all allocated data segments on several lines

 The differences to the standard routines are:
 - the alternative set provides error printout indicating the current file
   and line number where the error occurred;
 - the alternative set operates on a linked list of data structures of type
   "iomem". Each structure contains the size of its data segment, a pointer to
   the next element in the list, and a pointer to its own data segment. This
   allows to keep track of the overall memory usage by the program with the
   routine iomemsize().
 - if the debug flag IODBG_MEMALLOC is set each function prints
   a debug line.

 Otherwise, the input parameters and return values of the routines in the
 alternative set are the same as the ones in the standard set.

 If IOALLOC is not defined, iomalloc(), iocalloc(), iorealloc() and 
 iofree() are defined to be the standard set of memory management routines, 
 and iomemsize() and iomemlist are dummies.

HISTORY

    2017-05-23  PB V0.6 SIZE_F
    2017-09-27  PB V0.7 the routines are quite slow,
                        using fill128 in structure definition

***************************************************************************/

/***************************************************************************
* Private part                                                             *
***************************************************************************/

#include "ioalloc.h"

# define SIZE_F (unsigned long)

/*---------------------------------------------------------------------------
NAME

   ioalloc_version --- returns the current version of ioalloc

SYNOPSIS

   const char *ioalloc_version ( void );

RETURN VALUE

const char *version string
---------------------------------------------------------------------------*/
const char *ioalloc_version ( void )
{ return ( IOALLOC_VERSION );
} /* ioalloc_version */

#if IOALLOC == 1

  #include <string.h> // for memset
  #include "iodbg.h"

  struct iomem {
    void *data;            /* data segment 4x */
    size_t size;           /* size of the data segment 4x */
    struct iomem *next;    /* pointer to next structure in linked list 4x */
    void *fill128;         /* dummy 4x */
  };

  static struct iomem *iomem_head = NULL;

  /*
   * dummy struct containing 4 pointer elements
   */
  struct palign1 {
    void *fill[4]; 
  };

  /*
   * dummy struct containing 4 size_t elements
   */
  struct palign2 {
    size_t fill[4];
  };

  /*
   * iomemalign is a dummy union with a size that is the maximum size 
   * of struct palign1 (4*sizeof(void*)) and struct palign2 
   * (4*sizeof(size_t)), but with at least the size of struct iomem. 
   * It is used for allocating the return ptr on a boundary which is 
   * a multiple of the size of iomemalign, on 32 bit machines usually 
   * 16 bytes, on 64 bit machines usually 32 bytes.
   */
  union iomemalign {
    struct iomem   mem;
    struct palign1 align1;
    struct palign2 align2;
  };

  void *_iomalloc(size_t size,char *file,int line)
  {
    struct iomem *ptr;
    /*
     * Allocate the new memory segment, put the address of the last memory
     * segment in the link pointer, and save the address of the new memory
     * segment for use in the link pointer at the next call.
     */
    if ( iodbg( IODBG_MEMALLOC )|| iodbg( IODBG_MEMSIZE ) ) printf("ioalloc->malloc(%lu)",SIZE_F size);
    if((ptr = (struct iomem *)malloc(size + sizeof(union iomemalign))) == NULL) {
      if ( iodbg( IODBG_MEMALLOC )|| iodbg( IODBG_MEMSIZE ) ) printf("\n");
      fprintf(stderr,"ERROR: ioalloc->malloc(%lu) Memory allocation failed (line=%d, file=%s).\n",
        SIZE_F size,line,file);
      return(NULL);
    }
    ptr->next = iomem_head;
    iomem_head = ptr;
    ptr->data = (void *)(((char *)ptr) + sizeof(union iomemalign));
    ptr->size = size;

    if ( iodbg( IODBG_MEMALLOC ) ) printf("->data=%p",ptr->data);
    if ( iodbg( IODBG_MEMSIZE   ) ) _iomemsize(", ");
    if ( iodbg( IODBG_MEMALLOC | IODBG_MEMFILEPOS ) ) printf(" (line=%d, file=%s))", line,file);
    if ( iodbg( IODBG_MEMALLOC ) ) printf(" done");
    if ( iodbg( IODBG_MEMALLOC )|| iodbg( IODBG_MEMSIZE ) ) printf("\n");

    return(ptr->data);
  } // _iomalloc

  void *_iocalloc(size_t nmemb, size_t size,char *file,int line)
  {
    struct iomem *ptr;
    size_t allocsize;
    /*
     * Allocate the new memory segment, put the address of the last memory
     * segment in the link pointer, and save the address of the new memory
     * segment for use in the link pointer at the next call.
     */
    if ( iodbg( IODBG_MEMALLOC )|| iodbg( IODBG_MEMSIZE ) ) printf("ioalloc->calloc(%lu,%lu)",SIZE_F nmemb, SIZE_F size);

    allocsize = nmemb*size;
    if((ptr = (struct iomem *)malloc(sizeof(union iomemalign)+allocsize)) == NULL) {
      if ( iodbg( IODBG_MEMALLOC )|| iodbg( IODBG_MEMSIZE ) ) printf("\n");
      fprintf(stderr,"ERROR: ioalloc->calloc(%lu) Memory allocation failed (line=%d, file=%s).\n",
        SIZE_F allocsize,line,file);
      return(NULL);
    }
  
    /*
     * clear allocated memory
     */
    memset((void *)ptr,0,sizeof(union iomemalign)+allocsize);
  
    ptr->next = iomem_head;
    iomem_head = ptr;
    ptr->data = (void *)(((char *)ptr) + sizeof(union iomemalign));
    ptr->size = allocsize;
  
    if ( iodbg( IODBG_MEMALLOC ) ) printf("->data=%p",ptr->data);
    if ( iodbg( IODBG_MEMSIZE   ) ) _iomemsize(", ");
    if ( iodbg( IODBG_MEMALLOC | IODBG_MEMFILEPOS ) ) printf(" (line=%d, file=%s))", line,file);
    if ( iodbg( IODBG_MEMALLOC ) ) printf(" done");
    if ( iodbg( IODBG_MEMALLOC )|| iodbg( IODBG_MEMSIZE ) ) printf("\n");

    return ( ptr->data );
  } // _iocalloc

  void *_iorealloc(void *ptr,size_t size,char *file,int line)
  {
    struct iomem *new_ptr;

    if (ptr) {

      struct iomem *loop_ptr=NULL,**ref_ptr;
      /*
       * Loop throuth the linked list until the requested data segment is found or
       * the end of the list is reached (then loop_ptr == NULL).
       */
      if ( iodbg( IODBG_MEMALLOC )|| iodbg( IODBG_MEMSIZE ) ) printf("ioalloc->realloc(%p,%lu)",ptr, SIZE_F size);
      for(loop_ptr = iomem_head, ref_ptr = &iomem_head; loop_ptr;
        loop_ptr = *(ref_ptr = &loop_ptr->next))
        if(ptr == loop_ptr->data)
          break;
      if(loop_ptr == NULL) {
        if ( iodbg( IODBG_MEMALLOC )|| iodbg( IODBG_MEMSIZE ) ) printf("\n");
        fprintf(stderr,"ERROR: ioalloc->realloc(%p,%lu) Pointer was never allocated (line=%d, file=%s).\n",
          ptr, SIZE_F size,line,file);
        return(NULL);
      }

      /*
       * Reallocate the requested data segment.
       */
      if((new_ptr = (struct iomem *) realloc(loop_ptr,sizeof(union iomemalign)+size)) == NULL) {
        fprintf(stderr,"ERROR: ioalloc->realloc(%p,%lu) Memory reallocation failed (line=%d, file=%s).\n",
          ptr, SIZE_F size,line,file);
        return(NULL);
      }

      *ref_ptr = new_ptr;
      new_ptr->data  = (void*)(((char*)new_ptr) + sizeof(union iomemalign));
      new_ptr->size = size;

      if ( iodbg( IODBG_MEMALLOC ) ) printf("->data=%p",new_ptr->data);
      if ( iodbg( IODBG_MEMSIZE   ) ) _iomemsize(", ");
      if ( iodbg( IODBG_MEMALLOC | IODBG_MEMFILEPOS ) ) printf(" (line=%d, file=%s))", line,file);
      if ( iodbg( IODBG_MEMALLOC ) ) printf(" done");
      if ( iodbg( IODBG_MEMALLOC )|| iodbg( IODBG_MEMSIZE ) ) printf("\n");

    } else {

      /*
       * Allocate a new memory segment, put the address of the last memory
       * segment in the link pointer, and save the address of the new memory
       * segment for use in the link pointer at the next call.
       */
      if ( iodbg( IODBG_MEMALLOC )|| iodbg( IODBG_MEMSIZE ) ) printf("ioalloc->realloc(NULL,%lu)",SIZE_F size);
      if((new_ptr = (struct iomem *)realloc(NULL,size + sizeof(union iomemalign))) == NULL) {
        if ( iodbg( IODBG_MEMALLOC )|| iodbg( IODBG_MEMSIZE ) ) printf("\n");
        fprintf(stderr,"ERROR: ioalloc->realloc(%lu) Memory allocation failed (line=%d, file=%s).\n",
          SIZE_F size,line,file);
        return(NULL);
      }
      new_ptr->next = iomem_head;
      iomem_head = new_ptr;
      new_ptr->data = (void *)(((char *)new_ptr) + sizeof(union iomemalign));
      new_ptr->size = size;

      if ( iodbg( IODBG_MEMALLOC ) ) printf("->data=%p",new_ptr->data);
      if ( iodbg( IODBG_MEMSIZE   ) ) _iomemsize(", ");
      if ( iodbg( IODBG_MEMALLOC | IODBG_MEMFILEPOS ) ) printf(" (line=%d, file=%s))", line,file);
      if ( iodbg( IODBG_MEMALLOC ) ) printf(" done");
      if ( iodbg( IODBG_MEMALLOC )|| iodbg( IODBG_MEMSIZE ) ) printf("\n");

    }

    return(new_ptr->data);

  } // _iorealloc

  void _iofree(void *ptr,char *file,int line)
  {
    struct iomem *loop_ptr, **ref_ptr;
    /*
     * Loop through the linked list until the requested data segment is found or
     * the end of the list is reached (then loop_ptr == NULL).
     */
    if ( iodbg( IODBG_MEMALLOC )|| iodbg( IODBG_MEMSIZE ) ) printf("ioalloc->free(%p)",ptr);

    for(loop_ptr = iomem_head, ref_ptr = &iomem_head; loop_ptr;
      loop_ptr = *(ref_ptr = &loop_ptr->next)) {
      if(ptr == loop_ptr->data)
        break;
    }
    if(loop_ptr == NULL) {
      if ( iodbg( IODBG_MEMALLOC )|| iodbg( IODBG_MEMSIZE ) ) printf("\n");
      fprintf(stderr,"ERROR: ioalloc->free(%p) Pointer was never allocated (line=%d, file=%s).\n",
        ptr,line,file);
      return;
    }
    /*
     * Free the requested data segment, and re-adjust the link pointer.
     */
    *ref_ptr = loop_ptr->next;
    free(loop_ptr);

    if ( iodbg( IODBG_MEMSIZE   ) ) _iomemsize(", ");
    if ( iodbg( IODBG_MEMALLOC | IODBG_MEMFILEPOS ) ) printf(" (line=%d, file=%s))", line,file);
    if ( iodbg( IODBG_MEMALLOC ) ) printf(" done");
    if ( iodbg( IODBG_MEMALLOC )|| iodbg( IODBG_MEMSIZE ) ) printf("\n");


    return;
  } // _iofree

  /*
   * Print the total memory size of all allocated data segments on the current line
   */
  void _iomemsize(const char *prompt)
  {
    struct iomem *loop_ptr;
    size_t sum = 0;
    if (!prompt) prompt="";

    for(loop_ptr = iomem_head; loop_ptr; loop_ptr = loop_ptr->next)
      sum += loop_ptr->size;
    
    printf("%smemory use=%7.3f MiB (%lu byte%s)",prompt,(float) (sum / 1024 / 1024), SIZE_F sum, sum==1?"":"s");
    return;
  } // _iomemsize

  /*
   * Print the list of all allocated data segments.
   */
  void _iomemlist(const char *prompt)
  {
    struct iomem *loop_ptr;
    long loop_cnt;

    if (!prompt) prompt="";

    printf("================================================================================\n");

    for(loop_ptr=iomem_head,loop_cnt=1; loop_ptr; loop_ptr=loop_ptr->next,loop_cnt++)
    {
      printf("%s\n",prompt);
      printf(" memory segment %3ld: %p\n",loop_cnt,loop_ptr);
      printf("           data %3s: %p\n"," ",loop_ptr->data);
      printf("           size %3s: %lu\n"," ",SIZE_F loop_ptr->size);
      printf("           next %3s: %p\n"," ",loop_ptr->next);
      printf("--------------------------------------------------------------------------------\n");
    }
    return;
  } // _iomemlist

#else

  void _iomemsize(const char *prompt)
  {
    return;
  }
  void _iomemlist(const char *prompt)
  {
    return;
  }

#endif // IOALLOC
