/*+++------------------------------------------------------------------------
NAME
  hdfattr.c --- number expressions

PURPOSE
  Looping through an hdf file and analyze the attributes. 

CALL
 hdfattr [debug=<debug>] [--help] <filename>
  verbose:0x1,showatt:0x2,showattval:0x4,showdata:0x8,showdataval:0x10,showlinktype:0x20,hdfattrdebug:0x40

AUTHOR
  2016-11-16 Peter Boesecke (PB)

HISTORY
  2016-11-16 V0.0 PB ready for scalar or 1 dim string attributes
  2016-11-16 V1.1 PB attr_value: wrong reference attr_data corrected to 
                     attr_rdata[0]

--------------------------------------------------------------------------*/
/***************************************************************************
* Version                                                                  *
***************************************************************************/
# define HDFATTRIB_VERSION  "hdfattr : $Revision: 1.5 $ / $Date: 2016/12/12 16:43:11 $ Peter Boesecke"

/****************************************************************************
*  Include                                                                  *
****************************************************************************/

#include "hdf5.h"
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <errno.h>
#include <fcntl.h>
#include "numio.h"
#include "strlib.h"

/***************************************************************************
* Definitions                                                              *
***************************************************************************/

# define STRINGVSIZE 2048
# define HDFATTRIB_BUFLEN 1024

# define HDFATTRIB_VERBOSE      0x1
# define HDFATTRIB_SHOWATT      0x2 
# define HDFATTRIB_SHOWATTVAL   0x4
# define HDFATTRIB_SHOWDATA     0x8
# define HDFATTRIB_SHOWDATAVAL  0x10
# define HDFATTRIB_LINKTYPE     0x20
# define HDFATTRIB_DEBUG        0x40

//#define FREE( a) { if ( a) { free( a); a = NULL; } }

int HdfAttribDebug = 0;
int HdfAttribLevel = 0;

char HDFATTRIB_Usage[HDFATTRIB_BUFLEN];

# define HDFATTRIB_USAGE \
  "[debug=<debug>] [--help] <filename>"

/*--------------------------------------------------------------------------
NAME

  hdfattr_version --- returns pointer to the version string

SYNOPSIS

  const char *hdfattr_version ( void );

DESCRPTION

  Returns pointer to the version string.

--------------------------------------------------------------------------*/
const char *hdfattr_version ( void )
{
  return ( HDFATTRIB_VERSION );
} /* hdfattr_version */

/*--------------------------------------------------------------------------
NAME

  hdfattr_usage2str --- return debug mode usage string

SYNOPSIS

  const char *hdfattr_usage2str( void );

DESCRPTION

  Return debug mode usage string.

--------------------------------------------------------------------------*/
const char *hdfattr_usage2str( void )
{ sprintf(HDFATTRIB_Usage,
    "  %s\n  verbose:0x%x,showatt:0x%x,showattval:0x%x,showdata:0x%x,showdataval:0x%x,showlinktype:0x%x,hdfattrdebug:0x%x",
    HDFATTRIB_USAGE, HDFATTRIB_VERBOSE, HDFATTRIB_SHOWATT, HDFATTRIB_SHOWATTVAL, 
    HDFATTRIB_SHOWDATA, HDFATTRIB_SHOWDATAVAL, HDFATTRIB_LINKTYPE, HDFATTRIB_DEBUG);
  return(HDFATTRIB_Usage);
} // hdfattr_usage2str 

int fprint_debug( FILE *out )
{ fprintf(out,"debug            = 0x%x\n", HdfAttribDebug);
  fprintf(out,"verbose          = %d\n", HdfAttribDebug&HDFATTRIB_VERBOSE?1:0);
  fprintf(out,"showatt          = %d\n", HdfAttribDebug&HDFATTRIB_SHOWATT?1:0);
  fprintf(out,"showattval       = %d\n", HdfAttribDebug&HDFATTRIB_SHOWATTVAL?1:0);
  fprintf(out,"showdata         = %d\n", HdfAttribDebug&HDFATTRIB_SHOWDATA?1:0);
  fprintf(out,"showdataval      = %d\n", HdfAttribDebug&HDFATTRIB_SHOWDATAVAL?1:0);
  fprintf(out,"showlinktype     = %d\n", HdfAttribDebug&HDFATTRIB_LINKTYPE?1:0);
  fprintf(out,"hdfattrdebug = %d\n", HdfAttribDebug&HDFATTRIB_DEBUG?1:0);
  fprintf(out,"level            = %d\n", HdfAttribLevel);
  return(0);
} // fprint_debug

/*--------------------------------------------------------------------------
NAME

  hdfattr_debug_set --- set / reset module hdfattr to debug mode

SYNOPSIS

  int hdfattr_debug_set( int debug );

DESCRPTION

  Set / reset module hdfattr to debug mode.

--------------------------------------------------------------------------*/
int hdfattr_debug_set( int debug )
{ HdfAttribDebug = debug;

  if (HdfAttribDebug&HDFATTRIB_DEBUG) fprint_debug( stdout );
  return(0);

} // hdfattr_debug_set

/*--------------------------------------------------------------------------
NAME

  hdfattr_level --- return debug level

SYNOPSIS

  int hdfattr_level ( void );

--------------------------------------------------------------------------*/
int hdfattr_level ( void )
{ return( HdfAttribLevel );
} // hdfattr_Level

/*--------------------------------------------------------------------------
NAME

  hdfattr_level_set --- set level

SYNOPSIS

  int hdfattr_level_set( int level );

DESCRPTION

  Set hdfattr level.

--------------------------------------------------------------------------*/
int hdfattr_level_set( int level )
{ HdfAttribLevel = level;
  return(0);
} // hdfattr_level_set

/*--------------------------------------------------------------------------
NAME

  hdfattr_debug --- return debug value

SYNOPSIS

  int hdfattr_debug ( void );

--------------------------------------------------------------------------*/
int hdfattr_debug ( void )

{ return( HdfAttribDebug );
} // hdfattr_debug

/*
 * Operator function.
 */

/*
 * Return string value of the attribute of loc_id with name attr_name 
 * (only scalar string value or first element of a string array)
 * returns NULL in case of an error
 * Could be extended for converting numbers to strings
 */
const char *attr_value( char buffer[], size_t buflen, hid_t  loc_id, const char *attr_name ) {
  hid_t attr_id=-1, attr_type=-1, attr_space=-1;
  htri_t attr_is_vls=-1; // htri_t: TRUE, FALSE or negative in case of failure
  H5T_cset_t attr_cset=-1;
  int attr_rank=-1;
  hsize_t *attr_dims=NULL, attr_size=0;
  hsize_t *attr_maxdims=NULL;
  hsize_t attr_storage_size=0;

  hid_t attr_memtype=-1;
  size_t attr_memsize=0;
  hsize_t attr_stringsize=0;

  H5T_class_t attr_class;

  herr_t h5status=0;

  char **attr_rdata=NULL;
  char *attr_data=NULL;
  char *value=NULL;

  if (!buffer) goto attr_value_error;
  if (buflen<1) goto attr_value_error;

  value = buffer;
  value[0]='\0'; 

  /*
   * Read attribute by name
   */
  if (HdfAttribDebug&HDFATTRIB_DEBUG) 
    printf(">> attr_id = H5Aopen_name(%d,%s)",loc_id,attr_name);
  attr_id = H5Aopen_name(loc_id,attr_name);
  if (HdfAttribDebug&HDFATTRIB_DEBUG) 
    printf(" = %d\n",attr_id);
  if (attr_id<0) goto attr_value_error;

 /*
  * Get datatype of the attribute.
  */
  if (HdfAttribDebug&HDFATTRIB_DEBUG) 
    printf(">> attr_type = H5Aget_type(%d)",attr_id);
  attr_type = H5Aget_type (attr_id);
  if (HdfAttribDebug&HDFATTRIB_DEBUG) 
    printf(" = %d\n", attr_type);

 /*
  * Get class of the attribute.
  * H5T_class_t H5Tget_class( hid_t dtype_id )
  */
  if (HdfAttribDebug&HDFATTRIB_DEBUG) 
    printf(">> H5Tget_class(%d)",attr_type);
  switch (attr_class=H5Tget_class(attr_type)) {
    case H5T_STRING:
      if (HdfAttribDebug&HDFATTRIB_DEBUG) 
        printf(" = H5T_STRING\n");
    
     /*
      * What is the character set of the attribute value?
      */
      if (HdfAttribDebug&HDFATTRIB_DEBUG) 
        printf(">> H5Tget_cset(%d)",attr_type);
      // H5T_cset_t H5Tget_cset( hid_t dtype_id )
      attr_cset = H5Tget_cset( attr_type );
      if (HdfAttribDebug&HDFATTRIB_DEBUG) {
        switch(attr_cset) {
          case H5T_CSET_ASCII: printf(" = H5T_CSET_ASCII\n");
                               break;
          case H5T_CSET_UTF8 : printf(" = H5T_CSET_UTF8\n");
                               break;
          default: printf(" = unknown (%d)\n",attr_cset);
        }
      }

     /*
      * Is the attribute a variable length string?
      */
      if (HdfAttribDebug&HDFATTRIB_DEBUG) 
        printf(">> H5Tis_variable_str(%d)",attr_type);
      attr_is_vls = H5Tis_variable_str( attr_type );
      if (HdfAttribDebug&HDFATTRIB_DEBUG) 
        printf(" = %s (%d)\n",(attr_is_vls<0)?"Error":(attr_is_vls?"Yes":"No"),attr_is_vls);

     /*
      * Get dataspace
      */
      if (HdfAttribDebug&HDFATTRIB_DEBUG) 
        printf(">> H5Aget_space(%d)",attr_id);
      attr_space = H5Aget_space (attr_id);
      if (HdfAttribDebug&HDFATTRIB_DEBUG) 
        printf(" = %d\n", attr_space);
  
     /*
      * Get rank of attribute, allocate attr_dims and attr_maxdims
      * By convention, the rank of a scalar dataspace is 0 (zero)
      */
      if (HdfAttribDebug&HDFATTRIB_DEBUG) 
        printf(">> H5Sget_simple_extent_ndims(%d)",attr_space);
      attr_rank = H5Sget_simple_extent_ndims( attr_space );
      if (HdfAttribDebug&HDFATTRIB_DEBUG)
        printf(" = %ld\n",(long) attr_rank);
      if (attr_rank<0) {
        printf("ERROR: H5Sget_simple_extent_ndims\n");
        goto attr_value_error;
      }

      if (attr_rank>0) {
        if (HdfAttribDebug&HDFATTRIB_DEBUG)
          printf(">>  (hsize_t*) CALLOC(%ld,%ld)",(long) attr_rank,(long) sizeof(hsize_t));
        attr_dims = (hsize_t*) CALLOC(attr_rank,sizeof(hsize_t));
        if (HdfAttribDebug&HDFATTRIB_DEBUG) {
          printf(" = %p\n",attr_dims);
        }
        if (HdfAttribDebug&HDFATTRIB_DEBUG)
          printf(">>  (hsize_t*) CALLOC(%ld,%ld)",(long) attr_rank,(long) sizeof(hsize_t));
        attr_maxdims = (hsize_t*) CALLOC(attr_rank,sizeof(hsize_t));
        if (HdfAttribDebug&HDFATTRIB_DEBUG) {
          printf(" = %p\n",attr_maxdims);
        }

      } else {
        // scalar dataspace
        if (HdfAttribDebug&HDFATTRIB_DEBUG)
          printf(">>  (hsize_t*) CALLOC(%ld,%ld)",(long) 1,(long) sizeof(hsize_t));
        attr_dims = (hsize_t*) CALLOC(1,sizeof(hsize_t));
        if (HdfAttribDebug&HDFATTRIB_DEBUG) {
          printf(" = %p\n",attr_dims);
        }
        if (HdfAttribDebug&HDFATTRIB_DEBUG)
          printf(">>  (hsize_t*) CALLOC(%ld,%ld)",(long) 1,(long) sizeof(hsize_t));
        attr_maxdims = (hsize_t*) CALLOC(1,sizeof(hsize_t));
        if (HdfAttribDebug&HDFATTRIB_DEBUG) {
          printf(" = %p\n",attr_maxdims);
        }
      }
  
      if (HdfAttribDebug&HDFATTRIB_DEBUG) 
        printf(">> H5Sget_simple_extent_dims(%d,%p,%p)",attr_space,attr_dims,attr_maxdims);
      attr_rank=H5Sget_simple_extent_dims (attr_space, attr_dims, attr_maxdims);
      if (HdfAttribDebug&HDFATTRIB_DEBUG)
        printf(" = %ld\n",(long) attr_rank);
      if (HdfAttribDebug&HDFATTRIB_DEBUG) {
        if (attr_rank>0) {
          long idim;
          for (idim=0;idim<attr_rank;idim++) {
            printf(">> attr_dims[%ld] = %ld; attr_maxdims[%ld] = %ld;\n",idim,attr_dims[idim],idim,attr_maxdims[idim]);
          }
        } else {
          printf(">> attr_dims = %ld; attr_maxdims = %ld\n",(long) attr_dims[0], (long) attr_maxdims[0]);
        }
      }

      if (attr_rank<0) {
        printf("ERROR: H5Sget_simple_extent_ndims\n");
        goto attr_value_error;
      }

     /*
      * HDF routines do not convert automatically between variable length strings
      * and fixed length strings. It must be done manually.
      */
      switch (attr_is_vls) {
        case 1: // variable length strings
          if (HdfAttribDebug&HDFATTRIB_DEBUG)
            printf(">> Reading variable length string\n");

          if (HdfAttribDebug&HDFATTRIB_DEBUG)
            printf(">>  (char **) CALLOC(%ld,%ld)",((long) (attr_dims[0]>0)?attr_dims[0]:1),(long) sizeof(char *));
          attr_rdata = (char **) CALLOC (((attr_dims[0]>0)?attr_dims[0]:1),sizeof (char *));
          if (HdfAttribDebug&HDFATTRIB_DEBUG)
            printf(" = %p\n",attr_rdata);
          if (!attr_rdata) goto attr_value_error;

         /*
          * Create the memory datatype.
          */
          if (HdfAttribDebug&HDFATTRIB_DEBUG) 
            printf(">> H5Tcopy(%d)",H5T_C_S1);
          attr_memtype = H5Tcopy (H5T_C_S1);
          if (HdfAttribDebug&HDFATTRIB_DEBUG) 
            printf(" = %d\n", attr_memtype);

          if (HdfAttribDebug&HDFATTRIB_DEBUG)
            printf(">> H5Tset_cset(%d,H5T_CSET_ASCII)\n",attr_memtype);
          h5status=H5Tset_cset(attr_memtype,H5T_CSET_ASCII);
          if (h5status<0) {
            printf("ERROR: H5Tset_cset\n");
            goto attr_value_error;
          }

          if (HdfAttribDebug&HDFATTRIB_DEBUG) 
            printf(">> H5Tset_size(%d,%s)\n",attr_memtype,"H5T_VARIABLE");
          h5status = H5Tset_size (attr_memtype, H5T_VARIABLE);
          if (h5status<0) {
            printf("ERROR: H5Tset_size\n");
            goto attr_value_error;
          }

         /*
          * Read variable length strings. 
          * attr_rdata is a pointer to a pointer array
          */
          if (HdfAttribDebug&HDFATTRIB_DEBUG) 
            printf(">> H5Aread(%d,%d,%p)\n",attr_id,attr_memtype,attr_rdata);
          h5status = H5Aread (attr_id, attr_memtype, attr_rdata);
          if (h5status<0) {
            printf("ERROR: H5Aread\n");
            goto attr_value_error;
          }

          if (HdfAttribDebug&HDFATTRIB_DEBUG) {
            if (attr_rank>0) {
              char **pps;
              long irank;

              pps=attr_rdata;
              for (irank=0;irank<attr_rank;irank++) {
                long idim;
                printf(">> attr_data[%ld] = ",irank);
                for (idim=0;idim<attr_dims[irank];idim++) {
                  if (idim) printf(",\"%s\"",*pps);
                  else printf("\"%s\"",*pps);
                  pps++;
                }
                printf("\n");
              }
            } else {
              printf(">> attr_rdata[0] = \"%s\"\n",attr_rdata[0]);
            }
          }

         /*
          * Write all attribute values to the output buffer
          * (rank<=2)
          */
          if (attr_rank>0) {
            char **pps;
            char *pv;
            size_t rest;
            long irank=0;

            pps=attr_rdata;
            pv = value;
            rest = buflen-1;
            for (irank=0;irank<attr_rank;irank++) {
              long idim;

              if (irank) {
                snprintf(pv,rest,";");
                rest--; pv++;
              }
              for (idim=0;idim<attr_dims[irank];idim++) {
                if (idim) {
                  snprintf(pv,rest,",%s",*pps);
                  rest-=strlen(*pps)+1;
                  pv+=strlen(*pps)+1;
                } else {
                  snprintf(pv,rest,"%s",*pps);
                  rest-=strlen(*pps);
                  pv+=strlen(*pps);
                }
                pps++; // For rank>1 the order of items is probably not as expected 
              }
            }
          } else {
            strncpy(value,attr_rdata[0],buflen-1);
          }
          value[buflen-1]='\0'; // force the last element to null

          if (HdfAttribDebug&HDFATTRIB_DEBUG) 
            printf(">> H5Dvlen_reclaim(%d,%d,%s,%p)\n",attr_memtype, attr_space, "H5P_DEFAULT", attr_rdata);
          h5status=H5Dvlen_reclaim (attr_memtype, attr_space, H5P_DEFAULT, attr_rdata);
          if (h5status<0) {
            printf("ERROR: H5Dvlen_reclaim\n");
            goto attr_value_error;
          }

          if (HdfAttribDebug&HDFATTRIB_DEBUG)
            printf(">> FREE(%p)\n",attr_rdata);
          FREE(attr_rdata);

          if (attr_memtype>=0) {
            if (HdfAttribDebug&HDFATTRIB_DEBUG)
              printf(">> H5Tclose(%d)\n",attr_memtype);
            H5Tclose(attr_memtype); attr_memtype=-1;
          }

          break;

        case 0: // fixed length string
          if (HdfAttribDebug&HDFATTRIB_DEBUG)
            printf(">> Reading fixed length string\n");

          if (HdfAttribDebug&HDFATTRIB_DEBUG) 
            printf(">> H5Aget_storage_size(%d)",attr_id);
          attr_storage_size = H5Aget_storage_size(attr_id);
          if (HdfAttribDebug&HDFATTRIB_DEBUG) 
            printf(" = %ld\n", (long) attr_storage_size);

          if (HdfAttribDebug&HDFATTRIB_DEBUG) 
            printf(">> H5Tget_size(%d)",attr_type);
          attr_stringsize = H5Tget_size (attr_type);
          if (HdfAttribDebug&HDFATTRIB_DEBUG)
            printf(" = %ld + 1\n", (long) attr_stringsize);
          attr_stringsize++;

          attr_memsize=attr_stringsize;
          if (attr_rank>0) {
            long irank;
            for (irank=0;irank<attr_rank;irank++) {
              attr_memsize*=attr_dims[irank];
            }
          }

          if (HdfAttribDebug&HDFATTRIB_DEBUG)
            printf(">>  (char *) CALLOC(%ld,%ld)",(long)attr_memsize,(long) sizeof(char));
          attr_data = (char *) CALLOC (attr_memsize,sizeof(char));
          if (HdfAttribDebug&HDFATTRIB_DEBUG)
            printf(" = %p\n",attr_data);
          if (!attr_data) goto attr_value_error;

         /*
          * Create the memory datatype.
          */
          if (HdfAttribDebug&HDFATTRIB_DEBUG)
            printf(">> H5Tcopy(%d)",H5T_C_S1);
          attr_memtype = H5Tcopy (H5T_C_S1);
          if (HdfAttribDebug&HDFATTRIB_DEBUG) 
            printf(" = %d\n", attr_memtype);

          if (HdfAttribDebug&HDFATTRIB_DEBUG)
            printf(">> H5Tset_cset(%d,H5T_CSET_ASCII)\n",attr_memtype);
          h5status=H5Tset_cset(attr_memtype,H5T_CSET_ASCII);
          if (h5status<0) {
            printf("ERROR: H5Tset_cset\n");
            goto attr_value_error;
          }

          if (HdfAttribDebug&HDFATTRIB_DEBUG)
            printf(">> H5Tset_size(%d,%ld)\n",attr_memtype,(long) attr_stringsize);
          h5status = H5Tset_size (attr_memtype, attr_stringsize);
          if (h5status<0) {
            printf("ERROR: H5Tset_size\n");
            goto attr_value_error;
          }

         /*
          * Read fixed length strings. 
          * attr_data is a pointer to a data array
          */
          if (HdfAttribDebug&HDFATTRIB_DEBUG)
            printf(">> H5Aread(%d,%d,%p)\n",attr_id,attr_memtype,attr_data);
          h5status = H5Aread (attr_id, attr_memtype, attr_data);
          if (h5status<0) {
            printf("ERROR: H5Aread\n");
            goto attr_value_error;
          }

          if (HdfAttribDebug&HDFATTRIB_DEBUG) {
            if (attr_rank>0) {
              char *ps;
              long irank;
              ps=attr_data;
              for (irank=0;irank<attr_rank;irank++) {
                long idim;
                printf(">> attr_data[%ld] = ",irank);
                for (idim=0;idim<attr_dims[irank];idim++) {
                  if (idim) printf(",\"%s\"",ps);
                  else printf("\"%s\"",ps);
                  ps+=attr_stringsize;
                }
                printf("\n");
              }
            } else {
              printf(">> attr_data = \"%s\"\n",attr_data);
            }
          }

         /*
          * Write all attribute values to the output buffer
          * (rank<=2)
          */
          if (attr_rank>0) {
            char *ps;
            char *pv;
            size_t rest;
            long irank=0;

            ps=attr_data;
            pv = value;
            rest = buflen-1;
            for (irank=0;irank<attr_rank;irank++) {
              long idim;

              if (irank) {
                snprintf(pv,rest,";");
                rest--; pv++;
              }
              for (idim=0;idim<attr_dims[irank];idim++) {
                if (idim) {
                  snprintf(pv,rest,",%s",ps);
                  rest-=strlen(ps)+1;
                  pv+=strlen(ps)+1;
                } else {
                  snprintf(pv,rest,"%s",ps);
                  rest-=strlen(ps);
                  pv+=strlen(ps);
                }
                ps+=attr_stringsize;
              }
            }
          } else {
            strncpy(value,attr_data,buflen-1);
          }
          value[buflen-1]='\0'; // force the last element to null

          if (HdfAttribDebug&HDFATTRIB_DEBUG)
            printf(">> H5Dvlen_reclaim(%d,%d,%s,%p)\n",attr_memtype, attr_space, "H5P_DEFAULT", attr_data);
          h5status=H5Dvlen_reclaim (attr_memtype, attr_space, H5P_DEFAULT, attr_data);
          if (h5status<0) {
            printf("ERROR: H5Dvlen_reclaim\n");
            goto attr_value_error;
          }

          if (HdfAttribDebug&HDFATTRIB_DEBUG)
            printf(">> FREE(%p)\n",attr_data);
          FREE(attr_data);

          if (attr_memtype>=0) {
            if (HdfAttribDebug&HDFATTRIB_DEBUG)
              printf(">> H5Tclose(%d)\n",attr_memtype);
            H5Tclose(attr_memtype); attr_memtype=-1;
          }

          break;

        default: goto attr_value_error;
      }

      if (HdfAttribDebug&HDFATTRIB_DEBUG)
        printf(">> FREE(%p)\n",attr_dims);
      FREE(attr_dims);
      if (HdfAttribDebug&HDFATTRIB_DEBUG)
        printf(">> FREE(%p)\n",attr_maxdims);
      FREE(attr_maxdims);
      if (HdfAttribDebug&HDFATTRIB_DEBUG)
        printf(">> H5Sclose(%d)\n",attr_space);
      H5Sclose( attr_space ); attr_space=-1;
        break;

    case H5T_INTEGER:
      if (HdfAttribDebug&HDFATTRIB_DEBUG)
        printf(" = H5T_INTEGER\n");
      snprintf(buffer,buflen,"(H5T_INTEGER)");
      break;
    case H5T_FLOAT:
      if (HdfAttribDebug&HDFATTRIB_DEBUG)
        printf(" = H5T_FLOAT\n");
      snprintf(buffer,buflen,"(H5T_FLOAT)");
      break;
    case H5T_BITFIELD:
      if (HdfAttribDebug&HDFATTRIB_DEBUG)
        printf(" = H5T_BITFIELD\n");
      snprintf(buffer,buflen,"(H5T_BITFIELD)");
      break;
    case H5T_OPAQUE:
      if (HdfAttribDebug&HDFATTRIB_DEBUG)
        printf(" = H5T_OPAQUE\n");
      snprintf(buffer,buflen,"(H5T_OPAQUE)");
      break;
    case H5T_COMPOUND:
      if (HdfAttribDebug&HDFATTRIB_DEBUG)
        printf(" = H5T_COMPOUND\n");
      snprintf(buffer,buflen,"(H5T_COMPOUND)");
      break;
    case H5T_REFERENCE:
      if (HdfAttribDebug&HDFATTRIB_DEBUG)
        printf(" = H5T_REFERENCE\n");
      snprintf(buffer,buflen,"(H5T_REFERENCE)");
      break;
    case H5T_ENUM:
      if (HdfAttribDebug&HDFATTRIB_DEBUG)
        printf(" = H5T_ENUM\n");
      snprintf(buffer,buflen,"(H5T_ENUM)");
      break;
    case H5T_VLEN:
      if (HdfAttribDebug&HDFATTRIB_DEBUG)
        printf(" = H5T_VLEN\n");
      snprintf(buffer,buflen,"(H5T_VLEN)");
      break;
    case H5T_ARRAY :
      if (HdfAttribDebug&HDFATTRIB_DEBUG)
        printf(" = H5T_ARRAY\n");
      snprintf(buffer,buflen,"(H5T_ARRAY)");
      break;
    default:
      if (HdfAttribDebug&HDFATTRIB_DEBUG)
        printf(" = %d (unsupported attribute class)\n",attr_class);
      snprintf(buffer,buflen,"(unsupported attribute class)");
      break;
  }


  if (HdfAttribDebug&HDFATTRIB_DEBUG)
    printf(">> H5Tclose(%d)\n",attr_type);
  H5Tclose( attr_type ); attr_type=-1;
  if (HdfAttribDebug&HDFATTRIB_DEBUG)
    printf(">> H5Aclose(%d)\n",attr_id);
  H5Aclose( attr_id ); attr_id=-1;

  return(value);

attr_value_error:

  if (attr_rdata) {
    if (HdfAttribDebug&HDFATTRIB_DEBUG)
      printf(">> H5Dvlen_reclaim(%d,%d,%s,%p)\n",attr_memtype, attr_space, "H5P_DEFAULT", attr_rdata);
    H5Dvlen_reclaim (attr_memtype, attr_space, H5P_DEFAULT, attr_rdata);
    if (HdfAttribDebug&HDFATTRIB_DEBUG)
      printf(">> FREE(%p)\n",attr_rdata);
    FREE( attr_rdata );
  }
  if (attr_data) {
    if (HdfAttribDebug&HDFATTRIB_DEBUG)
      printf(">> H5Dvlen_reclaim(%d,%d,%s,%p)\n",attr_memtype, attr_space, "H5P_DEFAULT", attr_data);
    H5Dvlen_reclaim (attr_memtype, attr_space, H5P_DEFAULT, attr_data);
    if (HdfAttribDebug&HDFATTRIB_DEBUG)
      printf(">> FREE(%p)\n",attr_data);
    FREE( attr_data );
  }

  if (HdfAttribDebug&HDFATTRIB_DEBUG)
    printf(">> H5Tclose(%d)\n",attr_memtype);
  H5Tclose(attr_memtype); attr_memtype=-1;

  if (HdfAttribDebug&HDFATTRIB_DEBUG)
    printf(">> FREE(%p)\n",attr_dims);
  FREE( attr_dims );
  if (HdfAttribDebug&HDFATTRIB_DEBUG)
    printf(">> FREE(%p)\n",attr_maxdims);
  FREE( attr_maxdims );
  if (HdfAttribDebug&HDFATTRIB_DEBUG)
    printf(">> H5Sclose(%d)\n",attr_space);
  H5Sclose( attr_space ); attr_space=-1;
  if (HdfAttribDebug&HDFATTRIB_DEBUG)
    printf(">> H5Tclose(%d)\n",attr_type);
  H5Tclose( attr_type ); attr_type=-1;
  if (HdfAttribDebug&HDFATTRIB_DEBUG)
    printf(">> H5Aclose(%d)\n",attr_id);
  H5Aclose( attr_id ); attr_id=-1;

  return(NULL);

} // attr_value

herr_t print_H5Aiterate (hid_t loc_id, const char *attr_name, const H5A_info_t *ainfo, void *opdata)
{   /* 
     * typedef struct {
     *   hbool_t             corder_valid;   // Indicate if creation order is valid
     *   H5O_msg_crt_idx_t   corder;         // Creation order
     *   H5T_cset_t          cset;           // Character set of attribute name
     *   hsize_t             data_size;      // Size of raw data
     * } H5A_info_t;
     */

    char buffer[STRINGVSIZE];
    const char *value;

    if (HdfAttribDebug&HDFATTRIB_DEBUG) {
      printf(">> print_H5Aiterate (id=%d)",loc_id);
      switch(ainfo->cset) {
        case H5T_CSET_ASCII: printf(" (H5T_CSET_ASCII)\n");
                             break;
        case H5T_CSET_UTF8 : printf(" (H5T_CSET_UTF8)\n");
                             break;
        default: printf(" (unknown character set %d)\n",ainfo->cset);
      }
    }

    if (HdfAttribDebug&HDFATTRIB_SHOWATTVAL) {
      value = attr_value( buffer, STRINGVSIZE, loc_id, attr_name );
      printf("    \"%s\" = \"%s\"\n",attr_name,value);
    } else {
      printf("    \"%s\"\n",attr_name);
    }

    if (value) return(0);
    else return(-1);
}

herr_t print_H5Lvisit (hid_t loc_id, const char *name, const H5L_info_t *linfo, void *opdata)
{ herr_t h5status=0;
  /*
   * typedef struct {
   *   H5L_type_t          type;           // Type of link
   *   hbool_t             corder_valid;   // Indicate if creation order is valid
   *   int64_t             corder;         // Creation order
   *   H5T_cset_t          cset;           // Character set of link name
   *   union {
   *     haddr_t         address;          // Address hard link points to
   *     size_t          val_size;         // Size of a soft link or UD link value
   *   } u;
   * } H5L_info_t;
   */

  switch (linfo->type) {
       //hid_t obj_id;
  case H5L_TYPE_ERROR:
       if (HdfAttribDebug&HDFATTRIB_LINKTYPE)
         printf(">>  Link type of \'%s\' is H5L_TYPE_ERROR.\n", name);
       printf("  \'%s\'\n", name);
       break;
  case H5L_TYPE_HARD:
       if (HdfAttribDebug&HDFATTRIB_LINKTYPE)
         printf(">>  Link type of \'%s\' is H5L_TYPE_HARD.\n", name);
       printf("  \'%s\'\n", name);
       if (HdfAttribDebug&HDFATTRIB_SHOWATT) {
         if (HdfAttribDebug&HDFATTRIB_DEBUG)
           printf(">> H5Aiterate_by_name(%d,%s,%d,%d,%p,%p,%p,%d)\n",
             loc_id, name, H5_INDEX_NAME, H5_ITER_INC, NULL, print_H5Aiterate, NULL, H5P_DEFAULT);
         h5status=H5Aiterate_by_name(loc_id, name, H5_INDEX_NAME, H5_ITER_INC, NULL, print_H5Aiterate, NULL, H5P_DEFAULT);
         if (h5status<0) {
           printf("ERROR: H5Aiterate_by_name\n");
           goto print_H5Lvisit_error;
         }
       }
       break;
  case H5L_TYPE_SOFT:
       if (HdfAttribDebug&HDFATTRIB_LINKTYPE)
         printf(">>  Link type of \'%s\' is H5L_TYPE_SOFT.\n", name);
       printf("  \'%s\'\n", name);

       break;
  case H5L_TYPE_EXTERNAL:
       if (HdfAttribDebug&HDFATTRIB_LINKTYPE)
         printf(">>  Link type of \'%s\' is H5L_TYPE_EXTERNAL.\n", name);
       printf("  \'%s\'\n", name);

       break;
  case H5L_TYPE_MAX:
       if (HdfAttribDebug&HDFATTRIB_LINKTYPE)
         printf(">>  Link type of \'%s\' is H5L_TYPE_MAX.\n", name);
       printf("  \'%s\'\n", name);

       break;
  default:
       if (HdfAttribDebug&HDFATTRIB_LINKTYPE)
         printf(">>  The link type %d of \'%s\' is unknown.\n", linfo->type,name);
       printf("  \'%s\'\n", name);
  }

  return (h5status);

print_H5Lvisit_error:

  return (h5status);

}

/*---------------------------------------------------------------------------
main
---------------------------------------------------------------------------*/
    
#if MAKE_FUNCTION
# define MAIN main_fileanalyze
#else
# define MAIN main
#endif
    
# define NULL_SPACE( s) ( ( s)? ( s) : ("" ) )


int MAIN (int argc, char *argv[])
{ 
  int errval=0;
  int status=0;
  int argcnt=1;
  int debug = 2;
  int level = 1;

  char *pstring[2];

  hid_t fid;
  herr_t h5status;

  #define BUFLEN 1024
  char buffer[BUFLEN];

  if (argc<=1) {
    printf("%s\n",hdfattr_usage2str());
    return(1);
  }

  // skip all options
  while (argc > argcnt) {
    if ( strncmp( argv[argcnt],"debug=", 6 ) == 0 ) {
      // sscanf(&argv[argcnt][6],"%d",&debug);
      debug = (int) num_str2double( &argv[argcnt][6], NULL, NULL);
      hdfattr_debug_set( debug );
      argcnt++; 
    } else if ( strncmp( argv[argcnt],"level=", 6 ) == 0 ) {
      sscanf(&argv[argcnt][6],"%d",&level);
      hdfattr_level_set( level );
      argcnt++; 
    } else if ( strncmp( argv[argcnt],"--version", 9 ) == 0 ) {
      printf("\n %s version: %s\n\n", argv[0], HDFATTRIB_VERSION);
      status=1;
      argcnt++;
      break; 
    } else if ( strncmp( argv[argcnt],"--help", 6 ) == 0 ) {
      printf("\n %s version: %s\n", argv[0], HDFATTRIB_VERSION);
      printf("%s\n",hdfattr_usage2str());
      status=1;

      argcnt++; 
      break; 
    } else break;
  }

  pstring[1] = (char *) NULL;
  if (!status) {
    if (argc>argcnt) {
      while (argc>argcnt) {
        pstring[0] = argv[argcnt++];
        if ( pstring[0] ) {
         /*
          * Open file readonly
          */
          if (HdfAttribDebug&HDFATTRIB_DEBUG)
            printf(">> H5Fopen(%s,%d,%d)",pstring[0], H5F_ACC_RDONLY, H5P_DEFAULT);
          fid = H5Fopen (pstring[0], H5F_ACC_RDONLY, H5P_DEFAULT);
          if (HdfAttribDebug&HDFATTRIB_DEBUG)
            printf(" = %d\n",fid);

          printf("\'%s\'\n", pstring[0]);
          if (HdfAttribDebug&HDFATTRIB_SHOWATT) {
            // show the file attributes
            if (HdfAttribDebug&HDFATTRIB_DEBUG)
              printf(">> H5Aiterate_by_name(%d,%s,%d,%d,%p,%p,%p,%d)\n",
                fid, "/", H5_INDEX_NAME, H5_ITER_INC, NULL, print_H5Aiterate, NULL, H5P_DEFAULT);
            h5status=H5Aiterate_by_name(fid, "/", H5_INDEX_NAME, H5_ITER_INC, NULL, print_H5Aiterate, NULL, H5P_DEFAULT);
            if (h5status<0) {
              printf("ERROR: H5Aiterate_by_name\n");
              goto main_error;
            }
          }

          if (HdfAttribDebug&HDFATTRIB_DEBUG)
            printf(">> H5Lvisit(%d,%d,%d,%p,%p)\n",
              fid, H5_INDEX_NAME, H5_ITER_INC, print_H5Lvisit, NULL);
          h5status=H5Lvisit( fid, H5_INDEX_NAME, H5_ITER_INC, print_H5Lvisit, NULL);
          if (h5status<0) {
            printf("ERROR: H5Lvisit\n");
            goto main_error;
          }

          if (HdfAttribDebug&HDFATTRIB_DEBUG)
            printf(">> H5Fclose(%d)\n",fid);
          h5status = H5Fclose (fid); fid=-1;
          if (h5status<0) {
            printf("ERROR: H5Fclose\n");
            goto main_error;
          }
        }
      }
    }
  }
  return ( status );

main_error:

  printf("ERROR: h5status = %d\n",h5status);

  if (HdfAttribDebug&HDFATTRIB_DEBUG)
    printf(">> H5Fclose(%d)\n",fid);
  H5Fclose (fid); fid=-1;

  return ( status );

} /* MAIN */
