/************************************************************

  The data is arranged in a way that allows series processing. A single data set uses one entry group
  (e.g. a time-resolved experiment using several detectors in parallel) containing several subgroups
  as shown below. 

  An entry group consists of one or more series groups. Each series group contains one or more memory
  groups that contain data volumes ("array") from different sources and their descriptions ("header").    
  Each memory contains the data of a single detector. The data in each memory group is described by 
  the values in the header. A data source is for example a 2D detector and a multi-channel scaler.

  Data volumes originating from different acquisition cycles are arranged in series groups.

  If requested, subsequent analysis stesp could be saved in different entries in a single h5 file.
  Saving everything in a single file is not necessary but possible.

  Series of data from different sources ("detector") can be stored in different ways:
  - as multi frame arrays (e.g. stacked images), e.g. for raw data
  - as single frame arrays (e.g. single images), e.g. for processed data or raw data of step scans
  But also series of multi frame acquisitions are possible.

  It is expected that multi-frame raw data is stored in a single series-group, e.g. /<entry>/<series>/...
  using one memory for each used detector and one memory for the multi-channel scaler data.

  Series of multi-frame raw data can be stored in subsequent series groups. A special case are single frame
  step scan data.

  Data originating from different independent experiments should either be stored in different files or
  in different entries.

  file:<file> (<SX_class SXfile>)
    group:<entry>  -> /<entry>  (SX_class SXentry)
      dataset: <group_number> -> /<entry>/<group_number> (SX_class SXnumber)
      group:<series>  -> /<entry>/<series>  (SX_class SXseries)
        dataset: <group_number> -> /<entry>/<series>/<group_number> (SX_class SXnumber)
        group: <memory> -> /<entry>/<series>/<memory> (SX_class SXmemory)
          dataset: <group_number> -> /<entry>/<series>/<memory>/<group_number> (SX_class SXnumber)
          [group: <datablock> -> /<entry>/<series>/<memory>/<datablock> (SX_class SXdata)]
            dataset: <array> [dim1, dim2, 1..Imax] -> /<entry>/<series>/<memory>/<datablock>/<array> (SX_class SXarray)
            group:<header>  -> /<entry>/<series>/<memory>/<datablock>/<header> (SX_class SXheader)
               dataset: <symbols>[1..Imax] -> /<entry>/<series>/<memory>/<datablock>/<header>/<symbols>/<value>[1..Imax]
          [group: <errorblock> -> /<entry>/<series>/<memory>/<errorblock> (SX_class SXerror)]
            dataset: <array> [dim1, dim2, 1..Imax] -> /<entry>/<series>/<memory>/<errorblock>/<array> (SX_class SXarray)
            group:<header>  -> /<entry>/<series>/<memory>/<errorblock>/<header> (SX_class SXheader)
               dataset: <symbols>[1..Imax] -> /<entry>/<series>/<memory>/<errorblock>/<header>/<symbols>/<value>[1..Imax]
        {group: <nextmemory> -> /<entry>/<series>/<nextmemory> (SX_class SXmemory)}
           ...
      {group:<nextseries>  -> /<entry>/<nextseries>  (SX_class SXseries)}
         ...
    {group:<nextentry>  -> /<nextentry>  (SX_class SXentry)}
      ...


  HISTORY
  2016-12-10 PB entry numbers and memory numbers
  2016-12-11 PB fopenmode
  2016-12-12 PB sx_link_exists, 
                add datasets to existing h5-group in existing h5-files
                added: FileOpenMode FileOpenAny
  2016-12-16 PB series group added
  2017-02-08 PB dset functions with prefix bsl2hdf_
  2017-03-03 PB if available, use dummy of first read frame as fill value
                Because the dummy value can change from frame to frame
                it is only a workaround

 ************************************************************/

/***************************************************************************
* Version                                                                  *
***************************************************************************/
# define BSL2HDF_VERSION  "bsl2hdf : $Revision: 1.3 $ / $Date: 2016/12/12 09:32:02 $ Peter Boesecke"

/****************************************************************************
*  Include                                                                  *
****************************************************************************/

#include "hdf5.h"
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <errno.h>
#include <fcntl.h>
#include "numio.h"
#include "strlib.h"

#include "edfio.h"
#include "raster.h"

/***************************************************************************
* Definitions                                                              *
***************************************************************************/

#define RANK 3
#define PRINTRANK( L, B ) { long i; printf("%s: ",L);for (i=0;i<RANK;i++) printf("%d ",B[i]); printf("\n"); }

# define BSL2HDF_BUFLEN (EdfMaxLinLen+1)
# define STRINGVSIZE (BSL2HDF_BUFLEN+BSL2HDF_BUFLEN)

# define BSL2HDF_VERBOSE      0x1
# define BSL2HDF_SHOWATT      0x2
# define BSL2HDF_SHOWATTVAL   0x4
# define BSL2HDF_SHOWDATA     0x8
# define BSL2HDF_SHOWDATAVAL  0x10
# define BSL2HDF_SHOWHEADER     0x20
# define BSL2HDF_DEBUG1       0x40
# define BSL2HDF_DEBUG2       0x80

#define MAX( a, b)  ((a)<(b)?(b):(a))
#define MIN( a, b)  ((a)>(b)?(b):(a))
//#define FREE( a) { if ( a) { free( a); a = NULL; } }

/***************************************************************************
* EDF array access definitions                                             *
***************************************************************************/

#define ABSPTR(A,D1,D2,I1,I2) (A)+((I1)+((I2)*(D1)))
#define NEXTCOL(pA,D1,D2) (pA)+1
#define NEXTROW(pA,D1,D2) (pA)+(D1)
#define NEXTCOLROW(pA,D1,D2) (pA)+(1+(D1))

#define SXCLASS "SX_class"
#define SXCLASS_GROUPNUMBER "SXgroupnumber"
#define SXCLASS_FILE   "SXfile"
#define SXCLASS_ENTRY   "SXentry"
#define SXCLASS_SERIES   "SXseries"
#define SXCLASS_MEMORY    "SXmemory"
#define SXCLASS_DATABLOCK  "SXdata"
#define SXCLASS_ERRORBLOCK "SXerror"
#define SXCLASS_ARRAY       "SXarray"
#define SXCLASS_HEADER      "SXheader"
#define SXCLASS_SYMBOL       "SXsymbol"

#define SYMBOLPREFIX ""
#define GROUPNUMBERNAME SXCLASS_GROUPNUMBER // "groupnumber"
#define ENTRYNAME       SXCLASS_ENTRY       // "entry"
#define SERIESNAME      SXCLASS_SERIES      // "series"
#define MEMORYNAME      SXCLASS_MEMORY      // "memory"
#define DATABLOCKNAME   SXCLASS_DATABLOCK   // "datablock"
#define ERRORBLOCKNAME  SXCLASS_ERRORBLOCK  // "errorblock"
#define ARRAYNAME       SXCLASS_ARRAY       // "array"  // array data set name inside datablock/errorblock
#define HEADERNAME      SXCLASS_HEADER      // "header"  // header data group name inside datablock/errorblock

#define SXTMP   "tmp"

/***************************************************************************
* File access mode                                                         *
***************************************************************************/

enum FileOpenMode { InValidFileOpenMode,
                    FileOpenNew,
                    FileOpenOld,
                    FileOpenRead,
                    FileOpenAny,
                    EndFileOpenMode };

const char *FileOpenModeStrings[] = { "invalid",
                                      "new",
                                      "old",
                                      "read",
                                      "any",
                                      (const char *) NULL };

/***************************************************************************
* Group search mode                                                        *
***************************************************************************/

enum GroupSearchMode { InValidGroupSearchMode,
                       EntrySearch,
                       SeriesSearch,
                       MemorySearch,
                       DataBlockSearch,
                       ErrorBlockSearch,
                       EndGroupSearchMode };

const char *GroupSearchModeStrings[] = { "invalid",
                                         "entrysearch",
                                         "seriessearch",
                                         "memorysearch",
                                         "datablocksearch"
                                         "errorblocksearch",
                                         (const char *) NULL };

/***************************************************************************
* Bsl2Hdf                                                                  *
***************************************************************************/
int Bsl2HdfDebug = 0;
int Bsl2HdfLevel = 0;

char BSL2HDF_Usage[BSL2HDF_BUFLEN];

# define BSL2HDF_USAGE \
  "[debug=<debug>] [--help] <filename>"

/*--------------------------------------------------------------------------
NAME

  bsl2hdf_version --- returns pointer to the version string

SYNOPSIS

  const char *bsl2hdf_version ( void );

DESCRPTION

  Returns pointer to the version string.

--------------------------------------------------------------------------*/
const char *bsl2hdf_version ( void )
{
  return ( BSL2HDF_VERSION );
} /* bsl2hdf_version */

/*--------------------------------------------------------------------------
NAME

  bsl2hdf_usage2str --- return debug mode usage string

SYNOPSIS

  const char *bsl2hdf_usage2str( void );

DESCRPTION

  Return debug mode usage string.

--------------------------------------------------------------------------*/
const char *bsl2hdf_usage2str( void )
{ sprintf(BSL2HDF_Usage,
    "  %s\n  verbose:0x%x,showatt:0x%x,showattval:0x%x,showdata:0x%x,showdataval:0x%x,showheader:0x%x,debug1:0x%x,debug2:0x%x",
    BSL2HDF_USAGE, BSL2HDF_VERBOSE, BSL2HDF_SHOWATT, BSL2HDF_SHOWATTVAL,
    BSL2HDF_SHOWDATA, BSL2HDF_SHOWDATAVAL, BSL2HDF_SHOWHEADER, BSL2HDF_DEBUG1, BSL2HDF_DEBUG2);
  return(BSL2HDF_Usage);
} // bsl2hdf_usage2str

int fprint_debug( FILE *out )
{ fprintf(out,"debug            = 0x%x\n", Bsl2HdfDebug);
  fprintf(out,"verbose          = %d\n", Bsl2HdfDebug&BSL2HDF_VERBOSE?1:0);
  fprintf(out,"showatt          = %d\n", Bsl2HdfDebug&BSL2HDF_SHOWATT?1:0);
  fprintf(out,"showattval       = %d\n", Bsl2HdfDebug&BSL2HDF_SHOWATTVAL?1:0);
  fprintf(out,"showdata         = %d\n", Bsl2HdfDebug&BSL2HDF_SHOWDATA?1:0);
  fprintf(out,"showdataval      = %d\n", Bsl2HdfDebug&BSL2HDF_SHOWDATAVAL?1:0);
  fprintf(out,"showheader       = %d\n", Bsl2HdfDebug&BSL2HDF_SHOWHEADER?1:0);
  fprintf(out,"debug1           = %d\n", Bsl2HdfDebug&BSL2HDF_DEBUG1?1:0);
  fprintf(out,"debug2           = %d\n", Bsl2HdfDebug&BSL2HDF_DEBUG2?1:0);
  fprintf(out,"level            = %d\n", Bsl2HdfLevel);
  return(0);
} // fprint_debug

/*--------------------------------------------------------------------------
NAME

  bsl2hdf_debug_set --- set / reset module bsl2hdf to debug mode

SYNOPSIS

  int bsl2hdf_debug_set( int debug );

DESCRPTION

  Set / reset module bsl2hdf to debug mode.

--------------------------------------------------------------------------*/
int bsl2hdf_debug_set( int debug )
{ Bsl2HdfDebug = debug;

  if (Bsl2HdfDebug&BSL2HDF_DEBUG1) fprint_debug( stdout );
  return(0);

} // bsl2hdf_debug_set

/*--------------------------------------------------------------------------
NAME

  bsl2hdf_level --- return debug level

SYNOPSIS

  int bsl2hdf_level ( void );

--------------------------------------------------------------------------*/
int bsl2hdf_level ( void )
{ return( Bsl2HdfLevel );
} // bsl2hdf_Level

/*--------------------------------------------------------------------------
NAME

  bsl2hdf_level_set --- set level

SYNOPSIS

  int bsl2hdf_level_set( int level );

DESCRPTION

  Set bsl2hdf level.

--------------------------------------------------------------------------*/
int bsl2hdf_level_set( int level )
{ Bsl2HdfLevel = level;
  return(0);
} // bsl2hdf_level_set

/*--------------------------------------------------------------------------
NAME

  bsl2hdf_debug --- return debug value

SYNOPSIS

  int bsl2hdf_debug ( void );

--------------------------------------------------------------------------*/
int bsl2hdf_debug ( void )

{ return( Bsl2HdfDebug );
} // bsl2hdf_debug

/*---------------------------------------------------------------------------
NAME
  fopenmode2str ---  returns openmode as string

SYNOPSIS
  const char * fopenmode2str ( int openmode )

DESCRIPTION
  shows internal data types
---------------------------------------------------------------------------*/
const char * fopenmode2str ( int openmode )
{ if ((openmode<0)||(openmode>=EndFileOpenMode))
    openmode = InValidFileOpenMode;
  return( FileOpenModeStrings[openmode] );
} /* fopenmode2str */

/*---------------------------------------------------------------------------
NAME
  fopenmode2str ---  returns mtype as string

SYNOPSIS
  const char * fopenmode2str ( int openmode )

DESCRIPTION
  shows internal data types
---------------------------------------------------------------------------*/
int str2fopenmode( const char * string )
{ int  openmode=-1;

  while ( FileOpenModeStrings[++openmode] ) {
    if (!strlib_casecmp(FileOpenModeStrings[openmode], string)) break;
  }
  if (openmode>=EndFileOpenMode)
    openmode=InValidFileOpenMode;

  return(openmode);

} /* str2fopenmode */

/*---------------------------------------------------------------------------
NAME
  groupsearchmode2str ---  returns group search mode as string

SYNOPSIS
  const char *groupsearchmode2str ( int groupsearchmode )
---------------------------------------------------------------------------*/
const char *groupsearchmode2str ( int groupsearchmode )
{ if ((groupsearchmode<0)||(groupsearchmode>=EndGroupSearchMode))
    groupsearchmode = InValidGroupSearchMode;
  return( GroupSearchModeStrings[groupsearchmode] );
} /* groupsearchmode2str */

/*---------------------------------------------------------------------------
NAME
  str2groupsearchmode ---  converts string to groupsearchmode

SYNOPSIS
  int str2groupsearchmode ( const char * string )
---------------------------------------------------------------------------*/
int str2groupsearchmode( const char *string )
{ int  groupsearchmode=-1;

  while ( GroupSearchModeStrings[++groupsearchmode] ) {
    if (!strlib_casecmp(GroupSearchModeStrings[groupsearchmode], string)) break;
  }
  if (groupsearchmode>=EndGroupSearchMode)
    groupsearchmode=InValidGroupSearchMode;

  return(groupsearchmode);

} /* str2fopenmode */

/*******************************************************************************
* Create, define, read and release a dictionary of dataset variables (BEGIN)   *
*                                                                              *
* DictOfDSets *bsl2hdf_new_dictofdsets( void );                                        *
* DictOfDSets *bsl2hdf_free_dictofdsets(DictOfDSets *dictofdsets);                     *
* long bsl2hdf_length_dictofdsets(DictOfDSets *dictofdsets);                           *
* Dset *bsl2hdf_append_dset( DictOfDSets *dictofdsets,                                 *
*                    const char *key, const char *value,                       *
*                    unsigned long maxvallen, unsigned long no_of_frames,      *
*                    hid_t dataset );                                          *
* const char *bsl2hdf_write_str_dset(Dset *dset, unsigned long frame_no,               *
*                            const char *value)                                *
* Dset *bsl2hdf_get_first_dset( DictOfDSets *dictofdsets, const char **pkey,           *
*                        const char **pvalue, hid_t *pdataset );               *
* Dset *bsl2hdf_get_next_dset( Dset *dset, const char **pkey,                          *
*                      const char **pvalue, hid_t *pdataset );                 *
* void bsl2hdf_print_dictofdsets(FILE *out, DictOfDSets * dictofdsets);                *
*******************************************************************************/

typedef struct Dset_ {
  char *key; // key 
  unsigned long maxvallen; // maximum length of string (without terminating 0)
  unsigned long no_of_frames; // length of char *value array[]
  char *value; // buffer containing no_of_frames strings with maximum maxvallen
               // characters + 1 terminating zero
  hid_t dataset; // dataset variable
  struct Dset_ * nextdset;
} Dset;

typedef struct {
  long  ndsets;    // number of dsets in dict
  Dset * firstdset; // first dset of dict
  Dset * lastdset;  // last dset of dict
} DictOfDSets;

/*---------------------------------------------------------------------------
NAME

  bsl2hdf_append_dset --- Writes a dset to the end of the dict

PURPOSE

  Creates a new dset at the end of dict. If dictofdsets is NULL
  nothing happens.

ARGUMENTS

  DictOfDSets * dictofdsets
  const char *key
  const char *value
  unsigned long no_of_frames : length of all char *string arrays 
  hid_t dataset

RETURN VALUE

  Pointer to the new dset or NULL in case of error.

---------------------------------------------------------------------------*/
Dset *bsl2hdf_append_dset( DictOfDSets * dictofdsets, 
                   const char *key, const char *value,
                   unsigned long maxvallen, unsigned long no_of_frames,
                   hid_t dataset )
{ Dset *new=NULL;

  if (dictofdsets&&key) {
    Dset * current;
    size_t size_dset;

    size_dset = sizeof ( Dset );
    new = MALLOC ( size_dset ); 
    if (!new)
      goto bsl2hdf_append_dset_error;

    new->key = strlib_newstr(key);
    new->maxvallen = maxvallen;
    new->no_of_frames = no_of_frames;
    if (value) new->value = strlib_newstr(value);
    else new->value = NULL;
    new->dataset = dataset;
    new->nextdset=NULL;

    current = dictofdsets->lastdset;
    if (current) {
      current->nextdset = new;
    } else {
      dictofdsets->firstdset = new;
    }
    dictofdsets->lastdset = new;
    dictofdsets->ndsets+=1l;
  }

  return(new);

bsl2hdf_append_dset_error:

  FREE(new);

  return(NULL);

} // bsl2hdf_append_dset

/*---------------------------------------------------------------------------
NAME

  bsl2hdf_write_str_dset --- Copies string to element frame_no of dset

PURPOSE

  Copies contents of value to element frame_no of dset.
  If dset is NULL nothing happens.

RETURN VALUE

  Pointer to the copy or NULL in case of error.

---------------------------------------------------------------------------*/
const char *bsl2hdf_write_str_dset(Dset *dset, unsigned long frame_no, const char *string)
{
  char *new=NULL, *pn, *dest=NULL;
  unsigned long no_of_frames, maxvallen, frame;

  if (dset) {
    no_of_frames = dset->no_of_frames;
    maxvallen = dset->maxvallen;
    if (!(dset->value)) { // allocate pointers and initialize first char with 0 
      // char new[no_of_frames][maxvallen+1]
      if (!(new = (char*) MALLOC ( sizeof( char ) * (no_of_frames*(maxvallen+1)))))
        goto bsl2hdf_write_str_dset_error;
      dset->value = new;
      pn=new;
      for (frame=1;frame<=no_of_frames;frame++) {
        *pn='\0'; // terminating 0
        pn+=maxvallen+1;
      }
    }
    dest=(dset->value)+(frame_no-1)*(maxvallen+1);
    if (string) 
      strncpy(dest,string,dset->maxvallen);
    dest[maxvallen]='\0'; // terminating 0
  }

  return(dest);

bsl2hdf_write_str_dset_error:

  if (new) {
    FREE(new);
    if (dset) dset->value=NULL;
  }

  return(dest);

} // bsl2hdf_write_str_dset

/*---------------------------------------------------------------------------
NAME

  bsl2hdf_get_first_dset --- Returns pointer to the first dset

PURPOSE

  If dictofdsets is not NULL goes to the first dset and 
  returns a pointer to it. In this case the dataset key, 
  the dataset value array and the dataset id are returned in
  **pkey, ***pvalue and *pdataset (if they are not NULL).

RETURN VALUE

  Pointer to the first dset or NULL

---------------------------------------------------------------------------*/
Dset *bsl2hdf_get_first_dset( DictOfDSets *dictofdsets, const char **pkey, 
                      const char **pvalue, hid_t *pdataset )
{ Dset *dset=NULL;

  if (dictofdsets) {
    dset = dictofdsets->firstdset;
    if (dset) {
      if (pkey) *pkey = dset->key;
      if (pvalue) *pvalue = (const char *) dset->value;
      if (pdataset) *pdataset = dset->dataset;
    }

  }

  return(dset);

} // bsl2hdf_get_first_dset

/*---------------------------------------------------------------------------
NAME

  bsl2hdf_get_next_dset --- return the next dset

PURPOSE
  
  Goes to the next dset and returns a pointer to it.
  If the input dset is not NULL the dataset key,
  the dataset value array and the dataset id are returned in
  **pkey, **pvalue and *pdataset (if they are not NULL).

RETURN VALUE

  Pointer to the next dset or NULL

---------------------------------------------------------------------------*/
Dset * bsl2hdf_get_next_dset( Dset * dset, const char **pkey, 
                      const char **pvalue, hid_t *pdataset )
{ Dset * next=NULL;

  if (dset) {
      next = dset->nextdset;
      if (next) {
        if (pkey) *pkey = next->key;
        if (pvalue) *pvalue = (const char *) next->value;
        if (pdataset) *pdataset = next->dataset;
      }
  }

  return( next );

} // bsl2hdf_get_next_dset

/*---------------------------------------------------------------------------
NAME

  bsl2hdf_new_dictofdsets( void ) --- Creates a dict of dsets

PURPOSE

  Creates a new dict of dsets

RETURN VALUE

  Pointer to root of new dset dict or NULL in case of error.

---------------------------------------------------------------------------*/
DictOfDSets * bsl2hdf_new_dictofdsets( void ) {
  size_t size_dictofdsets;
  DictOfDSets * new=NULL;

  size_dictofdsets = sizeof(DictOfDSets);
  new = MALLOC(size_dictofdsets);
  if (!new) 
    goto bsl2hdf_new_dictofdsets_error;

  new->ndsets = 0l;
  new->firstdset = NULL;
  new->lastdset = NULL;

  return(new);

bsl2hdf_new_dictofdsets_error:

  FREE(new);

  return(NULL);

} // bsl2hdf_new_dictofdsets

/*---------------------------------------------------------------------------
NAME

  bsl2hdf_free_dictofdsets --- Releases a dict of dsets

PURPOSE

  Releases the memory of a dict of dsets: 
    const char *key, hid_t dataset, Dset

RETURN VALUE

  NULL, otherwise error

---------------------------------------------------------------------------*/
DictOfDSets * bsl2hdf_free_dictofdsets( DictOfDSets * dictofdsets ) 
{ if (dictofdsets) {
    Dset *current, *next;

    // release all dsets 
    next = dictofdsets->firstdset;
    while (next) {
      current = next;
      next = current->nextdset;
      FREE(current->key);

      // free dsets
      FREE(current->value);
      if (current->dataset>=0)
        H5Dclose(current->dataset);
      FREE(current);
    }
    
    FREE(dictofdsets); 
  }

  return( NULL );

} // bsl2hdf_free_dictofdsets

/*---------------------------------------------------------------------------
NAME

  bsl2hdf_length_dictofdsets --- return the number of dsets in the dict

PURPOSE

  Return the number of dsets in the dict

RETURN VALUE

  Number of dsets.

---------------------------------------------------------------------------*/
long bsl2hdf_length_dictofdsets( DictOfDSets * dictofdsets )
{ long ndsets = 0l;

  if ( dictofdsets ) {
    ndsets = dictofdsets->ndsets;
  }

  return( ndsets );

} // bsl2hdf_length_dictofdsets

/*---------------------------------------------------------------------------
NAME
  
  bsl2hdf_print_dictofdsets --- print dict of dsets 
  
PURPOSE
  
  Print dict of dsets

---------------------------------------------------------------------------*/
void bsl2hdf_print_dictofdsets( FILE *out, DictOfDSets *dictofdsets, const char *info )
{
    Dset * dset;
    const char *key;
    const char *value;
    hid_t dataset=-1;
    long i;

    fprintf( out, "\n>>%s<<: dictionary of %ld dsets\n",info,bsl2hdf_length_dictofdsets(dictofdsets));

    i=0;
    dset = bsl2hdf_get_first_dset( dictofdsets,  &key, &value, &dataset );

    while ( dset ) {
      unsigned long j;
      const char *string=NULL;
      if ((dataset<0)&&(dset->no_of_frames<=1)) {
        if (value) {
          string = value;
          fprintf( out, "%ld: >>%s<<:>>%s<<\n",i,key,string);
        } else {
          fprintf( out, "%ld: >>%s<<\n",i,key);
        }
      } else {
        fprintf( out, "%ld: (>>%s<<,%d)\n",i,key,dataset);
        if (value) {
          for (j=0;j<dset->no_of_frames;j++) {
            string = value+j*(dset->maxvallen+1);
            fprintf( out, "     %ld: >>%s<<\n", j, string);
          }
        }
      }

      i++;
      dset = bsl2hdf_get_next_dset( dset, &key, &value, &dataset );

    } // while dset

    fprintf( out, "\n");

} // bsl2hdf_print_dictofdsets

/*******************************************************************************
* Create, define, read and release a dictionary of dataset variables (END)     *
*******************************************************************************/

char *newsxkey( const char * prefix, const char * key )
{ char * newkey=NULL;
  size_t prelen, keylen;

  keylen = strlen(key);
  prelen = strlen(prefix);

  newkey = (char*) MALLOC ( sizeof(char)*(keylen+prelen+1) );
  strncpy(newkey, prefix, prelen);
  strncpy(newkey+prelen, key, keylen+1);

  return(newkey);

} // newsxkey

/*
 * Return string value of the attribute of loc_id with name attr_key 
 * (only scalar string value or first element of a string array)
 * returns NULL in case of an error
 * Could be extended for converting numbers to strings
 * sx_attrib_read
 */
const char *sx_attrib_read( char buffer[], size_t buflen, hid_t  loc_id, const char *attr_key ) {
  hid_t attr_id=-1, attr_type=-1, attr_space=-1;
  htri_t attr_is_vls=-1; // htri_t: TRUE, FALSE or negative in case of failure
  H5T_cset_t attr_cset=-1;
  int attr_rank=-1;
  hsize_t *attr_dims=NULL, attr_size=0;
  hsize_t *attr_maxdims=NULL;
  hsize_t attr_storage_size=0;

  hid_t attr_memtype=-1;
  size_t attr_memsize=0;
  hsize_t attr_stringsize=0;

  H5T_class_t attr_class;

  herr_t h5status=0;

  char **attr_rdata=NULL;
  char *attr_data=NULL;
  char *value=NULL;

  if (Bsl2HdfDebug&BSL2HDF_DEBUG1) 
    printf("sx_attrib_read(%p,buflen=%ld,loc_id=%d,attr_key=%s) BEGIN\n",buffer, (long) buflen, loc_id, attr_key );

  if (!buffer) goto sx_attrib_read_error;
  if (buflen<1) goto sx_attrib_read_error;

  value = buffer;
  value[0]='\0'; 

  /*
   * Read attribute by name
   */
  if (Bsl2HdfDebug&BSL2HDF_DEBUG2) 
    printf(">> attr_id = H5Aopen_by_name(%d,%s,%s,%d,%d)",loc_id,".",attr_key,H5P_DEFAULT,H5P_DEFAULT);
  attr_id = H5Aopen_by_name(loc_id,".",attr_key,H5P_DEFAULT,H5P_DEFAULT ); 
  if (Bsl2HdfDebug&BSL2HDF_DEBUG2) 
    printf(" = %d\n",attr_id);
  if (attr_id<0) goto sx_attrib_read_error;

 /*
  * Get datatype of the attribute.
  */
  if (Bsl2HdfDebug&BSL2HDF_DEBUG2) 
    printf(">> attr_type = H5Aget_type(%d)",attr_id);
  attr_type = H5Aget_type (attr_id);
  if (Bsl2HdfDebug&BSL2HDF_DEBUG2) 
    printf(" = %d\n", attr_type);

 /*
  * Get class of the attribute.
  * H5T_class_t H5Tget_class( hid_t dtype_id )
  */
  if (Bsl2HdfDebug&BSL2HDF_DEBUG2) 
    printf(">> H5Tget_class(%d)",attr_type);
  switch (attr_class=H5Tget_class(attr_type)) {
    case H5T_STRING:
      if (Bsl2HdfDebug&BSL2HDF_DEBUG2) 
        printf(" = H5T_STRING\n");
    
     /*
      * What is the character set of the attribute value?
      */
      if (Bsl2HdfDebug&BSL2HDF_DEBUG2) 
        printf(">> H5Tget_cset(%d)",attr_type);
      // H5T_cset_t H5Tget_cset( hid_t dtype_id )
      attr_cset = H5Tget_cset( attr_type );
      if (Bsl2HdfDebug&BSL2HDF_DEBUG2) {
        switch(attr_cset) {
          case H5T_CSET_ASCII: printf(" = H5T_CSET_ASCII\n");
                               break;
          case H5T_CSET_UTF8 : printf(" = H5T_CSET_UTF8\n");
                               break;
          default: printf(" = unknown (%d)\n",attr_cset);
        }
      }

     /*
      * Is the attribute a variable length string?
      */
      if (Bsl2HdfDebug&BSL2HDF_DEBUG2) 
        printf(">> H5Tis_variable_str(%d)",attr_type);
      attr_is_vls = H5Tis_variable_str( attr_type );
      if (Bsl2HdfDebug&BSL2HDF_DEBUG2) 
        printf(" = %s (%d)\n",(attr_is_vls<0)?"Error":(attr_is_vls?"Yes":"No"),attr_is_vls);

     /*
      * Get dataspace
      */
      if (Bsl2HdfDebug&BSL2HDF_DEBUG2) 
        printf(">> H5Aget_space(%d)",attr_id);
      attr_space = H5Aget_space (attr_id);
      if (Bsl2HdfDebug&BSL2HDF_DEBUG2) 
        printf(" = %d\n", attr_space);
  
     /*
      * Get rank of attribute, allocate attr_dims and attr_maxdims
      * By convention, the rank of a scalar dataspace is 0 (zero)
      */
      if (Bsl2HdfDebug&BSL2HDF_DEBUG2) 
        printf(">> H5Sget_simple_extent_ndims(%d)",attr_space);
      attr_rank = H5Sget_simple_extent_ndims( attr_space );
      if (Bsl2HdfDebug&BSL2HDF_DEBUG2)
        printf(" = %ld\n",(long) attr_rank);
      if (attr_rank<0) {
        printf("ERROR: H5Sget_simple_extent_ndims\n");
        goto sx_attrib_read_error;
      }

      if (attr_rank>0) {
        if (Bsl2HdfDebug&BSL2HDF_DEBUG2)
          printf(">>  (hsize_t*) CALLOC(%ld,%ld)",(long) attr_rank,(long) sizeof(hsize_t));
        attr_dims = (hsize_t*) CALLOC(attr_rank,sizeof(hsize_t));
        if (Bsl2HdfDebug&BSL2HDF_DEBUG2) {
          printf(" = %p\n",attr_dims);
        }
        if (Bsl2HdfDebug&BSL2HDF_DEBUG2)
          printf(">>  (hsize_t*) CALLOC(%ld,%ld)",(long) attr_rank,(long) sizeof(hsize_t));
        attr_maxdims = (hsize_t*) CALLOC(attr_rank,sizeof(hsize_t));
        if (Bsl2HdfDebug&BSL2HDF_DEBUG2) {
          printf(" = %p\n",attr_maxdims);
        }

      } else {
        // scalar dataspace
        if (Bsl2HdfDebug&BSL2HDF_DEBUG2)
          printf(">>  (hsize_t*) CALLOC(%ld,%ld)",(long) 1,(long) sizeof(hsize_t));
        attr_dims = (hsize_t*) CALLOC(1,sizeof(hsize_t));
        if (Bsl2HdfDebug&BSL2HDF_DEBUG2) {
          printf(" = %p\n",attr_dims);
        }
        if (Bsl2HdfDebug&BSL2HDF_DEBUG2)
          printf(">>  (hsize_t*) CALLOC(%ld,%ld)",(long) 1,(long) sizeof(hsize_t));
        attr_maxdims = (hsize_t*) CALLOC(1,sizeof(hsize_t));
        if (Bsl2HdfDebug&BSL2HDF_DEBUG2) {
          printf(" = %p\n",attr_maxdims);
        }
      }
  
      if (Bsl2HdfDebug&BSL2HDF_DEBUG2) 
        printf(">> H5Sget_simple_extent_dims(%d,%p,%p)",attr_space,attr_dims,attr_maxdims);
      attr_rank=H5Sget_simple_extent_dims (attr_space, attr_dims, attr_maxdims);
      if (Bsl2HdfDebug&BSL2HDF_DEBUG2)
        printf(" = %ld\n",(long) attr_rank);
      if (Bsl2HdfDebug&BSL2HDF_DEBUG2) {
        if (attr_rank>0) {
          long idim;
          for (idim=0;idim<attr_rank;idim++) {
            printf(">> attr_dims[%ld] = %ld; attr_maxdims[%ld] = %ld;\n",idim,attr_dims[idim],idim,attr_maxdims[idim]);
          }
        } else {
          printf(">> attr_dims = %ld; attr_maxdims = %ld\n",(long) attr_dims[0], (long) attr_maxdims[0]);
        }
      }

      if (attr_rank<0) {
        printf("ERROR: H5Sget_simple_extent_ndims\n");
        goto sx_attrib_read_error;
      }

     /*
      * HDF routines do not convert automatically between variable length strings
      * and fixed length strings. It must be done manually.
      */
      switch (attr_is_vls) {
        case 1: // variable length strings
          if (Bsl2HdfDebug&BSL2HDF_DEBUG2)
            printf(">> Reading variable length string\n");

          if (Bsl2HdfDebug&BSL2HDF_DEBUG2)
            printf(">>  (char **) CALLOC(%ld,%ld)",((long) (attr_dims[0]>0)?attr_dims[0]:1),(long) sizeof(char *));
          attr_rdata = (char **) CALLOC (((attr_dims[0]>0)?attr_dims[0]:1),sizeof (char *));
          if (Bsl2HdfDebug&BSL2HDF_DEBUG2)
            printf(" = %p\n",attr_rdata);
          if (!attr_rdata) goto sx_attrib_read_error;

         /*
          * Create the memory datatype.
          */
          if (Bsl2HdfDebug&BSL2HDF_DEBUG2) 
            printf(">> H5Tcopy(%d)",H5T_C_S1);
          attr_memtype = H5Tcopy (H5T_C_S1);
          if (Bsl2HdfDebug&BSL2HDF_DEBUG2) 
            printf(" = %d\n", attr_memtype);

          if (Bsl2HdfDebug&BSL2HDF_DEBUG2)
            printf(">> H5Tset_cset(%d,H5T_CSET_ASCII)\n",attr_memtype);
          h5status=H5Tset_cset(attr_memtype,H5T_CSET_ASCII);
          if (h5status<0) {
            printf("ERROR: H5Tset_cset\n");
            goto sx_attrib_read_error;
          }

          if (Bsl2HdfDebug&BSL2HDF_DEBUG2) 
            printf(">> H5Tset_size(%d,%s)\n",attr_memtype,"H5T_VARIABLE");
          h5status = H5Tset_size (attr_memtype, H5T_VARIABLE);
          if (h5status<0) {
            printf("ERROR: H5Tset_size\n");
            goto sx_attrib_read_error;
          }

         /*
          * Read variable length strings. 
          * attr_rdata is a pointer to a pointer array
          */
          if (Bsl2HdfDebug&BSL2HDF_DEBUG2) 
            printf(">> H5Aread(%d,%d,%p)\n",attr_id,attr_memtype,attr_rdata);
          h5status = H5Aread (attr_id, attr_memtype, attr_rdata);
          if (h5status<0) {
            printf("ERROR: H5Aread\n");
            goto sx_attrib_read_error;
          }

          if (Bsl2HdfDebug&BSL2HDF_DEBUG2) {
            if (attr_rank>0) {
              char **pps;
              long irank;

              pps=attr_rdata;
              for (irank=0;irank<attr_rank;irank++) {
                long idim;
                printf(">> attr_data[%ld] = ",irank);
                for (idim=0;idim<attr_dims[irank];idim++) {
                  if (idim) printf(",\"%s\"",*pps);
                  else printf("\"%s\"",*pps);
                  pps++;
                }
                printf("\n");
              }
            } else {
              printf(">> attr_rdata[0] = \"%s\"\n",attr_rdata[0]);
            }
          }

         /*
          * Write all attribute values to the output buffer
          * (rank<=2)
          */
          if (attr_rank>0) {
            char **pps;
            char *pv;
            size_t rest;
            long irank=0;

            pps=attr_rdata;
            pv = value;
            rest = buflen-1;
            for (irank=0;irank<attr_rank;irank++) {
              long idim;

              if (irank) {
                snprintf(pv,rest,";");
                rest--; pv++;
              }
              for (idim=0;idim<attr_dims[irank];idim++) {
                if (idim) {
                  snprintf(pv,rest,",%s",*pps);
                  rest-=strlen(*pps)+1;
                  pv+=strlen(*pps)+1;
                } else {
                  snprintf(pv,rest,"%s",*pps);
                  rest-=strlen(*pps);
                  pv+=strlen(*pps);
                }
                pps++; // For rank>1 the order of items is probably not as expected 
              }
            }
          } else {
            strncpy(value,attr_rdata[0],buflen-1);
          }
          value[buflen-1]='\0'; // force the last element to null

          if (Bsl2HdfDebug&BSL2HDF_DEBUG2) 
            printf(">> H5Dvlen_reclaim(%d,%d,%s,%p)\n",attr_memtype, attr_space, "H5P_DEFAULT", attr_rdata);
          h5status=H5Dvlen_reclaim (attr_memtype, attr_space, H5P_DEFAULT, attr_rdata);
          if (h5status<0) {
            printf("ERROR: H5Dvlen_reclaim\n");
            goto sx_attrib_read_error;
          }

          if (Bsl2HdfDebug&BSL2HDF_DEBUG2)
            printf(">> FREE(%p)\n",attr_rdata);
          FREE(attr_rdata);

          if (attr_memtype>=0) {
            if (Bsl2HdfDebug&BSL2HDF_DEBUG2)
              printf(">> H5Tclose(%d)\n",attr_memtype);
            H5Tclose(attr_memtype); attr_memtype=-1;
          }

          break;

        case 0: // fixed length string
          if (Bsl2HdfDebug&BSL2HDF_DEBUG2)
            printf(">> Reading fixed length string\n");

          if (Bsl2HdfDebug&BSL2HDF_DEBUG2) 
            printf(">> H5Aget_storage_size(%d)",attr_id);
          attr_storage_size = H5Aget_storage_size(attr_id);
          if (Bsl2HdfDebug&BSL2HDF_DEBUG2) 
            printf(" = %ld\n", (long) attr_storage_size);

          if (Bsl2HdfDebug&BSL2HDF_DEBUG2) 
            printf(">> H5Tget_size(%d)",attr_type);
          attr_stringsize = H5Tget_size (attr_type);
          if (Bsl2HdfDebug&BSL2HDF_DEBUG2)
            printf(" = %ld + 1\n", (long) attr_stringsize);
          attr_stringsize++;

          attr_memsize=attr_stringsize;
          if (attr_rank>0) {
            long irank;
            for (irank=0;irank<attr_rank;irank++) {
              attr_memsize*=attr_dims[irank];
            }
          }

          if (Bsl2HdfDebug&BSL2HDF_DEBUG2)
            printf(">>  (char *) CALLOC(%ld,%ld)",(long)attr_memsize,(long) sizeof(char));
          attr_data = (char *) CALLOC (attr_memsize,sizeof(char));
          if (Bsl2HdfDebug&BSL2HDF_DEBUG2)
            printf(" = %p\n",attr_data);
          if (!attr_data) goto sx_attrib_read_error;

         /*
          * Create the memory datatype.
          */
          if (Bsl2HdfDebug&BSL2HDF_DEBUG2)
            printf(">> H5Tcopy(%d)",H5T_C_S1);
          attr_memtype = H5Tcopy (H5T_C_S1);
          if (Bsl2HdfDebug&BSL2HDF_DEBUG2) 
            printf(" = %d\n", attr_memtype);

          if (Bsl2HdfDebug&BSL2HDF_DEBUG2)
            printf(">> H5Tset_cset(%d,H5T_CSET_ASCII)\n",attr_memtype);
          h5status=H5Tset_cset(attr_memtype,H5T_CSET_ASCII);
          if (h5status<0) {
            printf("ERROR: H5Tset_cset\n");
            goto sx_attrib_read_error;
          }

          if (Bsl2HdfDebug&BSL2HDF_DEBUG2)
            printf(">> H5Tset_size(%d,%ld)\n",attr_memtype,(long) attr_stringsize);
          h5status = H5Tset_size (attr_memtype, attr_stringsize);
          if (h5status<0) {
            printf("ERROR: H5Tset_size\n");
            goto sx_attrib_read_error;
          }

         /*
          * Read fixed length strings. 
          * attr_data is a pointer to a data array
          */
          if (Bsl2HdfDebug&BSL2HDF_DEBUG2)
            printf(">> H5Aread(%d,%d,%p)\n",attr_id,attr_memtype,attr_data);
          h5status = H5Aread (attr_id, attr_memtype, attr_data);
          if (h5status<0) {
            printf("ERROR: H5Aread\n");
            goto sx_attrib_read_error;
          }

          if (Bsl2HdfDebug&BSL2HDF_DEBUG2) {
            if (attr_rank>0) {
              char *ps;
              long irank;
              ps=attr_data;
              for (irank=0;irank<attr_rank;irank++) {
                long idim;
                printf(">> attr_data[%ld] = ",irank);
                for (idim=0;idim<attr_dims[irank];idim++) {
                  if (idim) printf(",\"%s\"",ps);
                  else printf("\"%s\"",ps);
                  ps+=attr_stringsize;
                }
                printf("\n");
              }
            } else {
              printf(">> attr_data = \"%s\"\n",attr_data);
            }
          }

         /*
          * Write all attribute values to the output buffer
          * (rank<=2)
          */
          if (attr_rank>0) {
            char *ps;
            char *pv;
            size_t rest;
            long irank=0;

            ps=attr_data;
            pv = value;
            rest = buflen-1;
            for (irank=0;irank<attr_rank;irank++) {
              long idim;

              if (irank) {
                snprintf(pv,rest,";");
                rest--; pv++;
              }
              for (idim=0;idim<attr_dims[irank];idim++) {
                if (idim) {
                  snprintf(pv,rest,",%s",ps);
                  rest-=strlen(ps)+1;
                  pv+=strlen(ps)+1;
                } else {
                  snprintf(pv,rest,"%s",ps);
                  rest-=strlen(ps);
                  pv+=strlen(ps);
                }
                ps+=attr_stringsize;
              }
            }
          } else {
            strncpy(value,attr_data,buflen-1);
          }
          value[buflen-1]='\0'; // force the last element to null

          if (Bsl2HdfDebug&BSL2HDF_DEBUG2)
            printf(">> H5Dvlen_reclaim(%d,%d,%s,%p)\n",attr_memtype, attr_space, "H5P_DEFAULT", attr_data);
          h5status=H5Dvlen_reclaim (attr_memtype, attr_space, H5P_DEFAULT, attr_data);
          if (h5status<0) {
            printf("ERROR: H5Dvlen_reclaim\n");
            goto sx_attrib_read_error;
          }

          if (Bsl2HdfDebug&BSL2HDF_DEBUG2)
            printf(">> FREE(%p)\n",attr_data);
          FREE(attr_data);

          if (attr_memtype>=0) {
            if (Bsl2HdfDebug&BSL2HDF_DEBUG2)
              printf(">> H5Tclose(%d)\n",attr_memtype);
            H5Tclose(attr_memtype); attr_memtype=-1;
          }

          break;

        default: goto sx_attrib_read_error;
      }

      if (Bsl2HdfDebug&BSL2HDF_DEBUG2)
        printf(">> FREE(%p)\n",attr_dims);
      FREE(attr_dims);
      if (Bsl2HdfDebug&BSL2HDF_DEBUG2)
        printf(">> FREE(%p)\n",attr_maxdims);
      FREE(attr_maxdims);
      if (Bsl2HdfDebug&BSL2HDF_DEBUG2)
        printf(">> H5Sclose(%d)\n",attr_space);
      H5Sclose( attr_space ); attr_space=-1;
        break;

    case H5T_INTEGER:
      if (Bsl2HdfDebug&BSL2HDF_DEBUG2)
        printf(" = H5T_INTEGER\n");
      snprintf(buffer,buflen,"(H5T_INTEGER)");
      break;
    case H5T_FLOAT:
      if (Bsl2HdfDebug&BSL2HDF_DEBUG2)
        printf(" = H5T_FLOAT\n");
      snprintf(buffer,buflen,"(H5T_FLOAT)");
      break;
    case H5T_BITFIELD:
      if (Bsl2HdfDebug&BSL2HDF_DEBUG2)
        printf(" = H5T_BITFIELD\n");
      snprintf(buffer,buflen,"(H5T_BITFIELD)");
      break;
    case H5T_OPAQUE:
      if (Bsl2HdfDebug&BSL2HDF_DEBUG2)
        printf(" = H5T_OPAQUE\n");
      snprintf(buffer,buflen,"(H5T_OPAQUE)");
      break;
    case H5T_COMPOUND:
      if (Bsl2HdfDebug&BSL2HDF_DEBUG2)
        printf(" = H5T_COMPOUND\n");
      snprintf(buffer,buflen,"(H5T_COMPOUND)");
      break;
    case H5T_REFERENCE:
      if (Bsl2HdfDebug&BSL2HDF_DEBUG2)
        printf(" = H5T_REFERENCE\n");
      snprintf(buffer,buflen,"(H5T_REFERENCE)");
      break;
    case H5T_ENUM:
      if (Bsl2HdfDebug&BSL2HDF_DEBUG2)
        printf(" = H5T_ENUM\n");
      snprintf(buffer,buflen,"(H5T_ENUM)");
      break;
    case H5T_VLEN:
      if (Bsl2HdfDebug&BSL2HDF_DEBUG2)
        printf(" = H5T_VLEN\n");
      snprintf(buffer,buflen,"(H5T_VLEN)");
      break;
    case H5T_ARRAY :
      if (Bsl2HdfDebug&BSL2HDF_DEBUG2)
        printf(" = H5T_ARRAY\n");
      snprintf(buffer,buflen,"(H5T_ARRAY)");
      break;
    default:
      if (Bsl2HdfDebug&BSL2HDF_DEBUG2)
        printf(" = %d (unsupported attribute class)\n",attr_class);
      snprintf(buffer,buflen,"(unsupported attribute class)");
      break;
  }


  if (Bsl2HdfDebug&BSL2HDF_DEBUG2)
    printf(">> H5Tclose(%d)\n",attr_type);
  H5Tclose( attr_type ); attr_type=-1;
  if (Bsl2HdfDebug&BSL2HDF_DEBUG2)
    printf(">> H5Aclose(%d)\n",attr_id);
  H5Aclose( attr_id ); attr_id=-1;

  if (Bsl2HdfDebug&BSL2HDF_DEBUG1) 
    printf("sx_attrib_read(%p,buflen=%ld,loc_id=%d,attr_key=%s) (%s) END\n",buffer, (long) buflen, loc_id, attr_key, value );

  return(value);

sx_attrib_read_error:

  if (attr_rdata) {
    if (Bsl2HdfDebug&BSL2HDF_DEBUG2)
      printf(">> H5Dvlen_reclaim(%d,%d,%s,%p)\n",attr_memtype, attr_space, "H5P_DEFAULT", attr_rdata);
    H5Dvlen_reclaim (attr_memtype, attr_space, H5P_DEFAULT, attr_rdata);
    if (Bsl2HdfDebug&BSL2HDF_DEBUG2)
      printf(">> FREE(%p)\n",attr_rdata);
    FREE( attr_rdata );
  }
  if (attr_data) {
    if (Bsl2HdfDebug&BSL2HDF_DEBUG2)
      printf(">> H5Dvlen_reclaim(%d,%d,%s,%p)\n",attr_memtype, attr_space, "H5P_DEFAULT", attr_data);
    H5Dvlen_reclaim (attr_memtype, attr_space, H5P_DEFAULT, attr_data);
    if (Bsl2HdfDebug&BSL2HDF_DEBUG2)
      printf(">> FREE(%p)\n",attr_data);
    FREE( attr_data );
  }

  if (Bsl2HdfDebug&BSL2HDF_DEBUG2)
    printf(">> H5Tclose(%d)\n",attr_memtype);
  H5Tclose(attr_memtype); attr_memtype=-1;

  if (Bsl2HdfDebug&BSL2HDF_DEBUG2)
    printf(">> FREE(%p)\n",attr_dims);
  FREE( attr_dims );
  if (Bsl2HdfDebug&BSL2HDF_DEBUG2)
    printf(">> FREE(%p)\n",attr_maxdims);
  FREE( attr_maxdims );
  if (Bsl2HdfDebug&BSL2HDF_DEBUG2)
    printf(">> H5Sclose(%d)\n",attr_space);
  H5Sclose( attr_space ); attr_space=-1;
  if (Bsl2HdfDebug&BSL2HDF_DEBUG2)
    printf(">> H5Tclose(%d)\n",attr_type);
  H5Tclose( attr_type ); attr_type=-1;
  if (Bsl2HdfDebug&BSL2HDF_DEBUG2)
    printf(">> H5Aclose(%d)\n",attr_id);
  H5Aclose( attr_id ); attr_id=-1;

  if (Bsl2HdfDebug&BSL2HDF_DEBUG1) 
    printf("sx_attrib_read(%p,buflen=%ld,loc_id=%d,attr_key=%s) (%p) ERROR\n",buffer, (long) buflen, loc_id, attr_key, NULL );

  return(NULL);

} // sx_attrib_read

/*
 * write scalar string attributes to loc_id: attr_key - attr_val
 */
herr_t sx_attrib_write( hid_t loc_id, const char *attr_key, const char *attr_val ) 
{
  hid_t attr_space=-1, attr_type=-1, attr_id=-1;
  herr_t h5status=-1;

  if (Bsl2HdfDebug&BSL2HDF_DEBUG2) printf("\'%s\' = \'%s\'\n",attr_key,attr_val);

  attr_space = H5Screate(H5S_SCALAR);
  if (attr_space<0) goto sx_attrib_write_error;
    attr_type = H5Tcopy(H5T_C_S1);
      H5Tset_size(attr_type, strlen(attr_val));
      attr_id = H5Acreate( loc_id,attr_key, attr_type, attr_space, H5P_DEFAULT, H5P_DEFAULT );
      if (attr_id<0) goto sx_attrib_write_error;

        h5status = H5Awrite(attr_id, attr_type, attr_val);
        if (h5status<0) goto sx_attrib_write_error;

      if (attr_id>=0) H5Aclose(attr_id);

    if (attr_type>=0) H5Tclose(attr_type);

  if (attr_space>=0) H5Sclose(attr_space);

  return( h5status );

sx_attrib_write_error:

  if (attr_id>=0) H5Aclose(attr_id);
  if (attr_type>=0) H5Tclose(attr_type);
  if (attr_space>=0) H5Sclose(attr_space);

  return( h5status );

} // sx_attrib_write

/*
 * Write dictionary of scalar string attributes to loc_id.
 * DictOfDSets *dictofstringds=NULL;
 * dictofstringds=bsl2hdf_new_dictofdsets();
 * bsl2hdf_append_dset( dictofdsets, attr_key, attr_val, 0, 1, -1 );
 * bsl2hdf_append_dset( dictofdsets, SXCLASS, sxclassname, 0, 1, -1 );
 * ...
 * bsl2hdf_free_dictofdsets(dictofdsets);
 */
herr_t sx_attribdict_write( hid_t loc_id, DictOfDSets *dictofstringds )
{
  herr_t h5status=-1;
  const char *attr_key=NULL;
  const char *attr_val=NULL;
  Dset *stringdset;

  // run through stringds of all variable header elements
  stringdset = bsl2hdf_get_first_dset(dictofstringds,&attr_key, &attr_val, NULL);

  while(stringdset) {

    if (!attr_key) {
      h5status=-1;
      printf("ERROR: get_dset_and_increment\n");
      goto sx_attribdict_write_error;
    }

    h5status = sx_attrib_write( loc_id, attr_key, attr_val );

    if (h5status<0) goto sx_attribdict_write_error;

    stringdset = bsl2hdf_get_next_dset(stringdset, &attr_key, &attr_val, NULL);

  } // while

  return( 0 );

sx_attribdict_write_error:

  return( h5status );

} // sx_attribdict_write

herr_t sx_iterate_attrib (hid_t loc_id, const char *attr_key, const H5A_info_t *ainfo, void *op_data)
{   /*
     * typedef struct {
     *   hbool_t             corder_valid;   // Indicate if creation order is valid
     *   H5O_msg_crt_idx_t   corder;         // Creation order
     *   H5T_cset_t          cset;           // Character set of attribute name
     *   hsize_t             data_size;      // Size of raw data
     * } H5A_info_t;
     */

    const char *attr_val=NULL;
    DictOfDSets *dictofstringds=op_data;

    if (dictofstringds) {

      char buffer[STRINGVSIZE];

      if (Bsl2HdfDebug&BSL2HDF_DEBUG2) {
        printf(">> sx_iterate_attrib (id=%d,name=%s)",loc_id,attr_key);
        switch(ainfo->cset) {
          case H5T_CSET_ASCII: printf(" (H5T_CSET_ASCII)\n");
                               break;
          case H5T_CSET_UTF8 : printf(" (H5T_CSET_UTF8)\n");
                               break;
          default: printf(" (unknown character set %d)\n",ainfo->cset);
        }
      }

      attr_val = sx_attrib_read( buffer, STRINGVSIZE, loc_id, attr_key );

      if (attr_val) {
        if (Bsl2HdfDebug&BSL2HDF_DEBUG2) {
          printf("    \"%s\" = \"%s\"\n",attr_key,attr_val);
        }
        bsl2hdf_append_dset( dictofstringds, attr_key, attr_val, 0, 1, -1 );
      }
    }

    if (attr_val) return(0);
    else return(-1);

} // sx_iterate_attrib

/*
 * Read scalar string attributes of loc_id and write to a dictionary.
 * DictOfDSets *attributes=NULL;
 * ...
 * bsl2hdf_free_dictofdsets(attributes);
 */
herr_t sx_attribdict_read( hid_t loc_id, DictOfDSets *attributes )
{ 
  herr_t h5status=-1;

  if (attributes) {
    if (Bsl2HdfDebug&BSL2HDF_DEBUG2)
      printf(">> H5Aiterate_by_name(%d,%s,%d,%d,%p,%p,%p,%d)\n",
        loc_id, "/", H5_INDEX_NAME, H5_ITER_INC, NULL, sx_iterate_attrib, (void*) attributes, H5P_DEFAULT);
    h5status=H5Aiterate_by_name(loc_id, "/", H5_INDEX_NAME, H5_ITER_INC, NULL, sx_iterate_attrib, (void*) attributes, H5P_DEFAULT);
    if (h5status<0) {
      printf("ERROR: H5Aiterate_by_name\n");
      goto sx_attribdict_read_error;
    }

    if (Bsl2HdfDebug&BSL2HDF_SHOWATTVAL) {
      bsl2hdf_print_dictofdsets(stdout, attributes, "sx_attribdict_read");
    }
  } else goto sx_attribdict_read_error;

  return(0);

sx_attribdict_read_error:

  return(-1);

} // sx_attribdict_read

/*
 * Read string dataset key with single value and attributes
 */
const char *sx_stringds_read( char buffer[], size_t buflen, hid_t loc_id, const char *key, DictOfDSets *attributes ) {
  hid_t ds_id=-1, ds_type=-1, ds_space=-1;
  htri_t ds_is_vls=-1; // htri_t: TRUE, FALSE or negative in case of failure
  H5T_cset_t ds_cset=-1;
  int ds_rank=-1;
  hsize_t *ds_dims=NULL, ds_size=0;
  hsize_t *ds_maxdims=NULL;
  hsize_t ds_storage_size=0;

  hid_t ds_memtype=-1;
  size_t ds_memsize=0;
  hsize_t ds_stringsize=0;

  H5T_class_t ds_class;

  herr_t h5status=0;

  char **ds_rdata=NULL;
  char *ds_data=NULL;
  char *value=NULL;

  if (Bsl2HdfDebug&BSL2HDF_DEBUG1) 
    printf("sx_stringds_read(%p,buflen=%ld,loc_id=%d,key=%s) BEGIN\n",buffer, (long) buflen, loc_id, key );

  if (!buffer) goto sx_stringds_read_error;
  if (buflen<1) goto sx_stringds_read_error;

  value = buffer;
  value[0]='\0'; 

  /*
   * Read dataset by name
   */
  // hid_t H5Dopen( hid_t loc_id, const char *name, hid_t dapl_id )
  if (Bsl2HdfDebug&BSL2HDF_DEBUG2) 
    printf(">> ds_id = H5Dopen(%d,%s,%d)",loc_id,key,H5P_DEFAULT);
  ds_id = H5Dopen(loc_id,key,H5P_DEFAULT);
  if (Bsl2HdfDebug&BSL2HDF_DEBUG2) 
    printf(" = %d\n",ds_id);
  if (ds_id<0) goto sx_stringds_read_error;

  /*
   * Read the attributes
   */
  if (attributes) {
    if (sx_attribdict_read( ds_id, attributes )<0) goto sx_stringds_read_error;
  }

 /*
  * Get datatype of the dataset
  */
  if (Bsl2HdfDebug&BSL2HDF_DEBUG2) 
    printf(">> ds_type = H5Dget_type(%d)",ds_id);
  ds_type = H5Dget_type (ds_id);
  if (Bsl2HdfDebug&BSL2HDF_DEBUG2) 
    printf(" = %d\n", ds_type);

 /*
  * Get class of the dataset.
  * H5T_class_t H5Tget_class( hid_t dtype_id )
  */
  if (Bsl2HdfDebug&BSL2HDF_DEBUG2) 
    printf(">> H5Tget_class(%d)",ds_type);
  switch (ds_class=H5Tget_class(ds_type)) {
    case H5T_STRING:
      if (Bsl2HdfDebug&BSL2HDF_DEBUG2) 
        printf(" = H5T_STRING\n");
    
     /*
      * What is the character set of the dataset value?
      */
      if (Bsl2HdfDebug&BSL2HDF_DEBUG2) 
        printf(">> H5Tget_cset(%d)",ds_type);
      // H5T_cset_t H5Tget_cset( hid_t dtype_id )
      ds_cset = H5Tget_cset( ds_type );
      if (Bsl2HdfDebug&BSL2HDF_DEBUG2) {
        switch(ds_cset) {
          case H5T_CSET_ASCII: printf(" = H5T_CSET_ASCII\n");
                               break;
          case H5T_CSET_UTF8 : printf(" = H5T_CSET_UTF8\n");
                               break;
          default: printf(" = unknown (%d)\n",ds_cset);
        }
      }

     /*
      * Is the dataset a variable length string?
      */
      if (Bsl2HdfDebug&BSL2HDF_DEBUG2) 
        printf(">> H5Tis_variable_str(%d)",ds_type);
      ds_is_vls = H5Tis_variable_str( ds_type );
      if (Bsl2HdfDebug&BSL2HDF_DEBUG2) 
        printf(" = %s (%d)\n",(ds_is_vls<0)?"Error":(ds_is_vls?"Yes":"No"),ds_is_vls);

     /*
      * Get dataspace
      */
      if (Bsl2HdfDebug&BSL2HDF_DEBUG2) 
        printf(">> H5Dget_space(%d)",ds_id);
      ds_space = H5Dget_space (ds_id);
      if (Bsl2HdfDebug&BSL2HDF_DEBUG2) 
        printf(" = %d\n", ds_space);
  
     /*
      * Get rank of dataset, allocate ds_dims and ds_maxdims
      * By convention, the rank of a scalar dataspace is 0 (zero)
      */
      if (Bsl2HdfDebug&BSL2HDF_DEBUG2) 
        printf(">> H5Sget_simple_extent_ndims(%d)",ds_space);
      ds_rank = H5Sget_simple_extent_ndims( ds_space );
      if (Bsl2HdfDebug&BSL2HDF_DEBUG2)
        printf(" = %ld\n",(long) ds_rank);
      if (ds_rank<0) {
        printf("ERROR: H5Sget_simple_extent_ndims\n");
        goto sx_stringds_read_error;
      }

      if (ds_rank>0) {
        if (Bsl2HdfDebug&BSL2HDF_DEBUG2)
          printf(">>  (hsize_t*) CALLOC(%ld,%ld)",(long) ds_rank,(long) sizeof(hsize_t));
        ds_dims = (hsize_t*) CALLOC(ds_rank,sizeof(hsize_t));
        if (Bsl2HdfDebug&BSL2HDF_DEBUG2) {
          printf(" = %p\n",ds_dims);
        }
        if (Bsl2HdfDebug&BSL2HDF_DEBUG2)
          printf(">>  (hsize_t*) CALLOC(%ld,%ld)",(long) ds_rank,(long) sizeof(hsize_t));
        ds_maxdims = (hsize_t*) CALLOC(ds_rank,sizeof(hsize_t));
        if (Bsl2HdfDebug&BSL2HDF_DEBUG2) {
          printf(" = %p\n",ds_maxdims);
        }

      } else {
        // scalar dataspace
        if (Bsl2HdfDebug&BSL2HDF_DEBUG2)
          printf(">>  (hsize_t*) CALLOC(%ld,%ld)",(long) 1,(long) sizeof(hsize_t));
        ds_dims = (hsize_t*) CALLOC(1,sizeof(hsize_t));
        if (Bsl2HdfDebug&BSL2HDF_DEBUG2) {
          printf(" = %p\n",ds_dims);
        }
        if (Bsl2HdfDebug&BSL2HDF_DEBUG2)
          printf(">>  (hsize_t*) CALLOC(%ld,%ld)",(long) 1,(long) sizeof(hsize_t));
        ds_maxdims = (hsize_t*) CALLOC(1,sizeof(hsize_t));
        if (Bsl2HdfDebug&BSL2HDF_DEBUG2) {
          printf(" = %p\n",ds_maxdims);
        }
      }
  
      if (Bsl2HdfDebug&BSL2HDF_DEBUG2) 
        printf(">> H5Sget_simple_extent_dims(%d,%p,%p)",ds_space,ds_dims,ds_maxdims);
      ds_rank=H5Sget_simple_extent_dims (ds_space, ds_dims, ds_maxdims);
      if (Bsl2HdfDebug&BSL2HDF_DEBUG2)
        printf(" = %ld\n",(long) ds_rank);
      if (Bsl2HdfDebug&BSL2HDF_DEBUG2) {
        if (ds_rank>0) {
          long idim;
          for (idim=0;idim<ds_rank;idim++) {
            printf(">> ds_dims[%ld] = %ld; ds_maxdims[%ld] = %ld;\n",idim,ds_dims[idim],idim,ds_maxdims[idim]);
          }
        } else {
          printf(">> ds_dims = %ld; ds_maxdims = %ld\n",(long) ds_dims[0], (long) ds_maxdims[0]);
        }
      }

      if (ds_rank<0) {
        printf("ERROR: H5Sget_simple_extent_ndims\n");
        goto sx_stringds_read_error;
      }

     /*
      * HDF routines do not convert automatically between variable length strings
      * and fixed length strings. It must be done manually.
      */
      switch (ds_is_vls) {
        case 1: // variable length strings
          if (Bsl2HdfDebug&BSL2HDF_DEBUG2)
            printf(">> Reading variable length string\n");

          if (Bsl2HdfDebug&BSL2HDF_DEBUG2)
            printf(">>  (char **) CALLOC(%ld,%ld)",((long) (ds_dims[0]>0)?ds_dims[0]:1),(long) sizeof(char *));
          ds_rdata = (char **) CALLOC (((ds_dims[0]>0)?ds_dims[0]:1),sizeof (char *));
          if (Bsl2HdfDebug&BSL2HDF_DEBUG2)
            printf(" = %p\n",ds_rdata);
          if (!ds_rdata) goto sx_stringds_read_error;

         /*
          * Create the memory datatype.
          */
          if (Bsl2HdfDebug&BSL2HDF_DEBUG2) 
            printf(">> H5Tcopy(%d)",H5T_C_S1);
          ds_memtype = H5Tcopy (H5T_C_S1);
          if (Bsl2HdfDebug&BSL2HDF_DEBUG2) 
            printf(" = %d\n", ds_memtype);

          if (Bsl2HdfDebug&BSL2HDF_DEBUG2)
            printf(">> H5Tset_cset(%d,H5T_CSET_ASCII)\n",ds_memtype);
          h5status=H5Tset_cset(ds_memtype,H5T_CSET_ASCII);
          if (h5status<0) {
            printf("ERROR: H5Tset_cset\n");
            goto sx_stringds_read_error;
          }

          if (Bsl2HdfDebug&BSL2HDF_DEBUG2) 
            printf(">> H5Tset_size(%d,%s)\n",ds_memtype,"H5T_VARIABLE");
          h5status = H5Tset_size (ds_memtype, H5T_VARIABLE);
          if (h5status<0) {
            printf("ERROR: H5Tset_size\n");
            goto sx_stringds_read_error;
          }

         /*
          * Read variable length strings. 
          * ds_rdata is a pointer to a pointer array
          */
          if (Bsl2HdfDebug&BSL2HDF_DEBUG2) 
            printf(">> H5Dread(%d,%d,%d,%d,%d,%p)\n",ds_id,ds_memtype,H5S_ALL,H5S_ALL,H5P_DEFAULT,ds_rdata);
          h5status = H5Dread(ds_id,ds_memtype,H5S_ALL,H5S_ALL,H5P_DEFAULT,ds_rdata);
          if (h5status<0) {
            printf("ERROR: H5Rread\n");
            goto sx_stringds_read_error;
          }

          if (Bsl2HdfDebug&BSL2HDF_DEBUG2) {
            if (ds_rank>0) {
              char **pps;
              long irank;

              pps=ds_rdata;
              for (irank=0;irank<ds_rank;irank++) {
                long idim;
                printf(">> ds_data[%ld] = ",irank);
                for (idim=0;idim<ds_dims[irank];idim++) {
                  if (idim) printf(",\"%s\"",*pps);
                  else printf("\"%s\"",*pps);
                  pps++;
                }
                printf("\n");
              }
            } else {
              printf(">> ds_rdata[0] = \"%s\"\n",ds_rdata[0]);
            }
          }

         /*
          * Write all dataset values to the output buffer
          * (rank<=2)
          */
          if (ds_rank>0) {
            char **pps;
            char *pv;
            size_t rest;
            long irank=0;

            pps=ds_rdata;
            pv = value;
            rest = buflen-1;
            for (irank=0;irank<ds_rank;irank++) {
              long idim;

              if (irank) {
                snprintf(pv,rest,";");
                rest--; pv++;
              }
              for (idim=0;idim<ds_dims[irank];idim++) {
                if (idim) {
                  snprintf(pv,rest,",%s",*pps);
                  rest-=strlen(*pps)+1;
                  pv+=strlen(*pps)+1;
                } else {
                  snprintf(pv,rest,"%s",*pps);
                  rest-=strlen(*pps);
                  pv+=strlen(*pps);
                }
                pps++; // For rank>1 the order of items is probably not as expected 
              }
            }
          } else {
            strncpy(value,ds_rdata[0],buflen-1);
          }
          value[buflen-1]='\0'; // force the last element to null

          if (Bsl2HdfDebug&BSL2HDF_DEBUG2) 
            printf(">> H5Dvlen_reclaim(%d,%d,%s,%p)\n",ds_memtype, ds_space, "H5P_DEFAULT", ds_rdata);
          h5status=H5Dvlen_reclaim (ds_memtype, ds_space, H5P_DEFAULT, ds_rdata);
          if (h5status<0) {
            printf("ERROR: H5Dvlen_reclaim\n");
            goto sx_stringds_read_error;
          }

          if (Bsl2HdfDebug&BSL2HDF_DEBUG2)
            printf(">> FREE(%p)\n",ds_rdata);
          FREE(ds_rdata);

          if (ds_memtype>=0) {
            if (Bsl2HdfDebug&BSL2HDF_DEBUG2)
              printf(">> H5Tclose(%d)\n",ds_memtype);
            H5Tclose(ds_memtype); ds_memtype=-1;
          }

          break;

        case 0: // fixed length string
          if (Bsl2HdfDebug&BSL2HDF_DEBUG2)
            printf(">> Reading fixed length string\n");

          if (Bsl2HdfDebug&BSL2HDF_DEBUG2) 
            printf(">> H5Dget_storage_size(%d)",ds_id);
          ds_storage_size = H5Dget_storage_size(ds_id);
          if (Bsl2HdfDebug&BSL2HDF_DEBUG2) 
            printf(" = %ld\n", (long) ds_storage_size);

          if (Bsl2HdfDebug&BSL2HDF_DEBUG2) 
            printf(">> H5Tget_size(%d)",ds_type);
          ds_stringsize = H5Tget_size (ds_type);
          if (Bsl2HdfDebug&BSL2HDF_DEBUG2)
            printf(" = %ld + 1\n", (long) ds_stringsize);
          ds_stringsize++;

          ds_memsize=ds_stringsize;
          if (ds_rank>0) {
            long irank;
            for (irank=0;irank<ds_rank;irank++) {
              ds_memsize*=ds_dims[irank];
            }
          }

          if (Bsl2HdfDebug&BSL2HDF_DEBUG2)
            printf(">>  (char *) CALLOC(%ld,%ld)",(long)ds_memsize,(long) sizeof(char));
          ds_data = (char *) CALLOC (ds_memsize,sizeof(char));
          if (Bsl2HdfDebug&BSL2HDF_DEBUG2)
            printf(" = %p\n",ds_data);
          if (!ds_data) goto sx_stringds_read_error;

         /*
          * Create the memory datatype.
          */
          if (Bsl2HdfDebug&BSL2HDF_DEBUG2)
            printf(">> H5Tcopy(%d)",H5T_C_S1);
          ds_memtype = H5Tcopy (H5T_C_S1);
          if (Bsl2HdfDebug&BSL2HDF_DEBUG2) 
            printf(" = %d\n", ds_memtype);

          if (Bsl2HdfDebug&BSL2HDF_DEBUG2)
            printf(">> H5Tset_cset(%d,H5T_CSET_ASCII)\n",ds_memtype);
          h5status=H5Tset_cset(ds_memtype,H5T_CSET_ASCII);
          if (h5status<0) {
            printf("ERROR: H5Tset_cset\n");
            goto sx_stringds_read_error;
          }

          if (Bsl2HdfDebug&BSL2HDF_DEBUG2)
            printf(">> H5Tset_size(%d,%ld)\n",ds_memtype,(long) ds_stringsize);
          h5status = H5Tset_size (ds_memtype, ds_stringsize);
          if (h5status<0) {
            printf("ERROR: H5Tset_size\n");
            goto sx_stringds_read_error;
          }

         /*
          * Read fixed length strings. 
          * ds_data is a pointer to a data array
          */
          if (Bsl2HdfDebug&BSL2HDF_DEBUG2)
            printf(">> H5Dread(%d,%d,%d,%d,%d,%p)\n",ds_id,ds_memtype,H5S_ALL,H5S_ALL,H5P_DEFAULT,ds_data);
          h5status = H5Dread(ds_id,ds_memtype,H5S_ALL,H5S_ALL,H5P_DEFAULT,ds_data);

          if (h5status<0) {
            printf("ERROR: H5Dread\n");
            goto sx_stringds_read_error;
          }

          if (Bsl2HdfDebug&BSL2HDF_DEBUG2) {
            if (ds_rank>0) {
              char *ps;
              long irank;
              ps=ds_data;
              for (irank=0;irank<ds_rank;irank++) {
                long idim;
                printf(">> ds_data[%ld] = ",irank);
                for (idim=0;idim<ds_dims[irank];idim++) {
                  if (idim) printf(",\"%s\"",ps);
                  else printf("\"%s\"",ps);
                  ps+=ds_stringsize;
                }
                printf("\n");
              }
            } else {
              printf(">> ds_data = \"%s\"\n",ds_data);
            }
          }

         /*
          * Write all dataset values to the output buffer
          * (rank<=2)
          */
          if (ds_rank>0) {
            char *ps;
            char *pv;
            size_t rest;
            long irank=0;

            ps=ds_data;
            pv = value;
            rest = buflen-1;
            for (irank=0;irank<ds_rank;irank++) {
              long idim;

              if (irank) {
                snprintf(pv,rest,";");
                rest--; pv++;
              }
              for (idim=0;idim<ds_dims[irank];idim++) {
                if (idim) {
                  snprintf(pv,rest,",%s",ps);
                  rest-=strlen(ps)+1;
                  pv+=strlen(ps)+1;
                } else {
                  snprintf(pv,rest,"%s",ps);
                  rest-=strlen(ps);
                  pv+=strlen(ps);
                }
                ps+=ds_stringsize;
              }
            }
          } else {
            strncpy(value,ds_data,buflen-1);
          }
          value[buflen-1]='\0'; // force the last element to null

          if (Bsl2HdfDebug&BSL2HDF_DEBUG2)
            printf(">> H5Dvlen_reclaim(%d,%d,%s,%p)\n",ds_memtype, ds_space, "H5P_DEFAULT", ds_data);
          h5status=H5Dvlen_reclaim (ds_memtype, ds_space, H5P_DEFAULT, ds_data);
          if (h5status<0) {
            printf("ERROR: H5Dvlen_reclaim\n");
            goto sx_stringds_read_error;
          }

          if (Bsl2HdfDebug&BSL2HDF_DEBUG2)
            printf(">> FREE(%p)\n",ds_data);
          FREE(ds_data);

          if (ds_memtype>=0) {
            if (Bsl2HdfDebug&BSL2HDF_DEBUG2)
              printf(">> H5Tclose(%d)\n",ds_memtype);
            H5Tclose(ds_memtype); ds_memtype=-1;
          }

          break;

        default: goto sx_stringds_read_error;
      }

      if (Bsl2HdfDebug&BSL2HDF_DEBUG2)
        printf(">> FREE(%p)\n",ds_dims);
      FREE(ds_dims);
      if (Bsl2HdfDebug&BSL2HDF_DEBUG2)
        printf(">> FREE(%p)\n",ds_maxdims);
      FREE(ds_maxdims);
      if (Bsl2HdfDebug&BSL2HDF_DEBUG2)
        printf(">> H5Sclose(%d)\n",ds_space);
      H5Sclose( ds_space ); ds_space=-1;
        break;

    case H5T_INTEGER:
      if (Bsl2HdfDebug&BSL2HDF_DEBUG2)
        printf(" = H5T_INTEGER\n");
      snprintf(buffer,buflen,"(H5T_INTEGER)");
      break;
    case H5T_FLOAT:
      if (Bsl2HdfDebug&BSL2HDF_DEBUG2)
        printf(" = H5T_FLOAT\n");
      snprintf(buffer,buflen,"(H5T_FLOAT)");
      break;
    case H5T_BITFIELD:
      if (Bsl2HdfDebug&BSL2HDF_DEBUG2)
        printf(" = H5T_BITFIELD\n");
      snprintf(buffer,buflen,"(H5T_BITFIELD)");
      break;
    case H5T_OPAQUE:
      if (Bsl2HdfDebug&BSL2HDF_DEBUG2)
        printf(" = H5T_OPAQUE\n");
      snprintf(buffer,buflen,"(H5T_OPAQUE)");
      break;
    case H5T_COMPOUND:
      if (Bsl2HdfDebug&BSL2HDF_DEBUG2)
        printf(" = H5T_COMPOUND\n");
      snprintf(buffer,buflen,"(H5T_COMPOUND)");
      break;
    case H5T_REFERENCE:
      if (Bsl2HdfDebug&BSL2HDF_DEBUG2)
        printf(" = H5T_REFERENCE\n");
      snprintf(buffer,buflen,"(H5T_REFERENCE)");
      break;
    case H5T_ENUM:
      if (Bsl2HdfDebug&BSL2HDF_DEBUG2)
        printf(" = H5T_ENUM\n");
      snprintf(buffer,buflen,"(H5T_ENUM)");
      break;
    case H5T_VLEN:
      if (Bsl2HdfDebug&BSL2HDF_DEBUG2)
        printf(" = H5T_VLEN\n");
      snprintf(buffer,buflen,"(H5T_VLEN)");
      break;
    case H5T_ARRAY :
      if (Bsl2HdfDebug&BSL2HDF_DEBUG2)
        printf(" = H5T_ARRAY\n");
      snprintf(buffer,buflen,"(H5T_ARRAY)");
      break;
    default:
      if (Bsl2HdfDebug&BSL2HDF_DEBUG2)
        printf(" = %d (unsupported dataset class)\n",ds_class);
      snprintf(buffer,buflen,"(unsupported dataset class)");
      break;
  }


  if (Bsl2HdfDebug&BSL2HDF_DEBUG2)
    printf(">> H5Tclose(%d)\n",ds_type);
  H5Tclose( ds_type ); ds_type=-1;
  if (Bsl2HdfDebug&BSL2HDF_DEBUG2)
    printf(">> H5Dclose(%d)\n",ds_id);
  H5Dclose( ds_id ); ds_id=-1;
  if (Bsl2HdfDebug&BSL2HDF_DEBUG2)

  if (Bsl2HdfDebug&BSL2HDF_DEBUG1) 
    printf("sx_stringds_read(%p,buflen=%ld,loc_id=%d,key=%s) (%s) END\n",buffer, (long) buflen, loc_id, key, value );

  return(value);

sx_stringds_read_error:

  if (ds_rdata) {
    if (Bsl2HdfDebug&BSL2HDF_DEBUG2)
      printf(">> H5Dvlen_reclaim(%d,%d,%s,%p)\n",ds_memtype, ds_space, "H5P_DEFAULT", ds_rdata);
    H5Dvlen_reclaim (ds_memtype, ds_space, H5P_DEFAULT, ds_rdata);
    if (Bsl2HdfDebug&BSL2HDF_DEBUG2)
      printf(">> FREE(%p)\n",ds_rdata);
    FREE( ds_rdata );
  }
  if (ds_data) {
    if (Bsl2HdfDebug&BSL2HDF_DEBUG2)
      printf(">> H5Dvlen_reclaim(%d,%d,%s,%p)\n",ds_memtype, ds_space, "H5P_DEFAULT", ds_data);
    H5Dvlen_reclaim (ds_memtype, ds_space, H5P_DEFAULT, ds_data);
    if (Bsl2HdfDebug&BSL2HDF_DEBUG2)
      printf(">> FREE(%p)\n",ds_data);
    FREE( ds_data );
  }

  if (Bsl2HdfDebug&BSL2HDF_DEBUG2)
    printf(">> H5Tclose(%d)\n",ds_memtype);
  H5Tclose(ds_memtype); ds_memtype=-1;

  if (Bsl2HdfDebug&BSL2HDF_DEBUG2)
    printf(">> FREE(%p)\n",ds_dims);
  FREE( ds_dims );
  if (Bsl2HdfDebug&BSL2HDF_DEBUG2)
    printf(">> FREE(%p)\n",ds_maxdims);
  FREE( ds_maxdims );
  if (Bsl2HdfDebug&BSL2HDF_DEBUG2)
    printf(">> H5Sclose(%d)\n",ds_space);
  H5Sclose( ds_space ); ds_space=-1;
  if (Bsl2HdfDebug&BSL2HDF_DEBUG2)
    printf(">> H5Tclose(%d)\n",ds_type);
  H5Tclose( ds_type ); ds_type=-1;
  if (Bsl2HdfDebug&BSL2HDF_DEBUG2)
    printf(">> H5Dclose(%d)\n",ds_id);
  H5Dclose( ds_id ); ds_id=-1;

  if (Bsl2HdfDebug&BSL2HDF_DEBUG1) 
    printf("sx_stringds_read(%p,buflen=%ld,loc_id=%d,key=%s) (%p) ERROR\n",buffer, (long) buflen, loc_id, key, NULL );

  return(NULL);

} // sx_stringds_read

/*
 * Create string dataset key with single value and attributes
 */
herr_t sx_stringds_write(hid_t loc_id, const char *key, const char *val, DictOfDSets *attributes)
{
  herr_t h5status=-1;
  hid_t val_type;

  hid_t   stringds=-1, stringds_type=-1, stringds_space=-1;
  hsize_t stringds_dims[1] = {1}; // single string only

  if (Bsl2HdfDebug&BSL2HDF_DEBUG1) printf("sx_stringds_write( %d ) BEGIN\n",loc_id);

  if (Bsl2HdfDebug&BSL2HDF_DEBUG2) printf("\'%s\' = \'%s\'\n",key,val);

  /*
   * Create output dataspace (stringds)
   * NO AUTOMATIC TRANSLATION BETWEEN FIXED LENGTH AND VARIABLE LENGTH DATATYPES!
   */

  stringds_type = H5Tcopy (H5T_C_S1);
  if (H5Tset_size (stringds_type, strlen(val)+1)<0) goto sx_stringds_write_error;

  stringds_space = H5Screate_simple (1, stringds_dims, NULL);
  if (stringds_space<0) goto sx_stringds_write_error;

  stringds = H5Dcreate (loc_id, key, stringds_type, stringds_space, H5P_DEFAULT, H5P_DEFAULT, H5P_DEFAULT);
  if (stringds<0) goto sx_stringds_write_error;

  /*
   * Write attributes
   */
  if (sx_attribdict_write( stringds, attributes )<0) goto sx_stringds_write_error;

  /*
   * Write val to dataset
   */
  val_type = H5Tcopy (H5T_C_S1);
  if (H5Tset_size (val_type, strlen(val)+1)<0) goto sx_stringds_write_error;

  if (H5Dwrite (stringds, val_type, H5S_ALL, H5S_ALL, H5P_DEFAULT, val)<0) goto sx_stringds_write_error;

  /*
   * Close and release resources.
   */
  if (H5Dclose(stringds)<0) goto sx_stringds_write_error;
  if (H5Tclose(stringds_type)<0) goto sx_stringds_write_error;
  if (H5Sclose (stringds_space)<0) goto sx_stringds_write_error;
  if (H5Tclose (val_type)<0) goto sx_stringds_write_error;

  if (Bsl2HdfDebug&BSL2HDF_DEBUG1) printf("sx_stringds_write( %d ) END\n",loc_id);

  return( 0 );

sx_stringds_write_error:

  h5status=-1;

  if (stringds>=0) H5Dclose (stringds);
  if (stringds_type>=0) H5Tclose (stringds_type);
  if (stringds_space>=0) H5Sclose (stringds_space);
  if (val_type>=0) H5Tclose (val_type);

  printf("sx_stringds_write( %d ) (h5status=%d) ERROR\n",loc_id,h5status);

  return( h5status );

} // sx_stringds_write

herr_t sx_file_close( hid_t fid )
{ herr_t h5status=0;
  if (fid>=0) h5status = H5Fclose(fid);
  return ( h5status );
} // sx_file_close

/*
 * hid_t sx_file_create( const char *name, DictOfDSets *attributes )
 * Create HDF5 file name:classname
 *
 * DictOfDSets *attributes=NULL;
 * attributes=bsl2hdf_new_dictofdsets();
 * bsl2hdf_append_dset( dictofdsets, attr_key, attr_val, 0, 1, -1 );
 * bsl2hdf_append_dset( dictofdsets, SXCLASS, sxclassname, 0, 1, -1 );
 * bsl2hdf_free_dictofdsets(dictofdsets);
 */
hid_t sx_file_create( const char *name,  DictOfDSets *attributes )
{
  hid_t fid=-1;
  herr_t h5status=-1;

  if (Bsl2HdfDebug&BSL2HDF_DEBUG1) printf("sx_file_create( %s , ... ) BEGIN\n",name);
  /*
   * Create a new hdf5 file using the default properties.
   */
  fid = H5Fcreate (name, H5F_ACC_TRUNC, H5P_DEFAULT, H5P_DEFAULT);
  if (fid<0) goto sx_file_create_error;

  h5status = sx_attribdict_write( fid, attributes );
  if (h5status<0) goto sx_file_create_error;

  if (Bsl2HdfDebug&BSL2HDF_DEBUG1) printf("sx_file_create( %s , ... ) END\n",name);

  return(fid);

sx_file_create_error:

  printf("sx_file_create( %s , ... ) ERROR\n",name);
  sx_file_close(fid);

  return(-1);

} // sx_file_create

/*
 * hid_t sx_file_open( const char *name, DictOfDSets *attributes )
 * Open HDF5 file name for reading and writing.
 * The returned attributes must be released.
 */
hid_t sx_file_open( const char *name,  DictOfDSets *attributes )
{
  hid_t fid=-1;
  herr_t h5status=-1;

  if (Bsl2HdfDebug&BSL2HDF_DEBUG1) printf("sx_file_open( %s , ... ) BEGIN\n",name);
  /*
   * Open and existing hdf5 file for read and write
   */
  // hid_t H5Fcreate( const char *name, unsigned flags, hid_t fcpl_id, hid_t fapl_id )
  // fid = H5Fcreate (name, H5F_ACC_TRUNC, H5P_DEFAULT, H5P_DEFAULT);
  // hid_t H5Fopen( const char *name, unsigned flags, hid_t fapl_id );
  fid = H5Fopen( name, H5F_ACC_RDWR, H5P_DEFAULT );
  if (fid<0) goto sx_file_open_error;

  if (attributes) {
    h5status = sx_attribdict_read( fid, attributes );
    if (h5status<0) goto sx_file_open_error;
  }

  if (Bsl2HdfDebug&BSL2HDF_DEBUG1) printf("sx_file_open( %s , ... ) END\n",name);

  return(fid);

sx_file_open_error:

  printf("sx_file_open( %s , ... ) ERROR\n",name);
  sx_file_close(fid);

  return(-1);

} // sx_file_open

/*
 * htri_t sx_link_exists( hid_t loc_id, const char *name )
 * Checks, wheter name exists in loc_id
 * htri_t retval;
 * if ((retval)>0) {
 *   printf("data type is committed\n");
 * } else if (!retval) {
 *  printf("data type is not committed\n");
 * } else {
 *  printf("error determining whether data type is committed\n");
 * }
 */
htri_t sx_link_exists( hid_t loc_id, const char *name )
{
  htri_t exists=-1;

  if (Bsl2HdfDebug&BSL2HDF_DEBUG1) printf("sx_link_exists( %d, %s ) BEGIN\n",loc_id,name);
  // htri_t H5Lexists( hid_t loc_id, const char *name, hid_t lapl_id )
  exists = H5Lexists( loc_id, name, H5P_DEFAULT);
  if (Bsl2HdfDebug&BSL2HDF_DEBUG1) printf("sx_link_exists( %d, %s ) (exists=%d) END\n",loc_id,name,exists);

  return(exists);

} // sx_link_exists

herr_t sx_group_close( hid_t gid )
{ herr_t h5status=0;
  if (Bsl2HdfDebug&BSL2HDF_DEBUG1) printf("sx_group_close( %d ) BEGIN\n",gid);
  if (gid>=0) h5status = H5Gclose(gid);
  if (Bsl2HdfDebug&BSL2HDF_DEBUG1) printf("sx_group_close( %d ) (h5status=%d) END\n",gid,h5status);
  return ( h5status );
} // sx_group_close

/*
 * hid_t sx_group_create( hid_t loc_id, const char *name, DictOfDSets *attributes )
 * Create HDF5 group name:classname in loc_id
 *
 * DictOfDSets *attributes=NULL;
 * attributes=bsl2hdf_new_dictofdsets();
 * bsl2hdf_append_dset( dictofdsets, attr_key, attr_val, 0, 1, -1 );
 * bsl2hdf_append_dset( dictofdsets, SXCLASS, sxclassname, 0, 1, -1 );
 * bsl2hdf_free_dictofdsets(dictofdsets);
 */
hid_t sx_group_create( hid_t loc_id, const char *name,  DictOfDSets *attributes )
{
  hid_t gid=-1;
  herr_t h5status=-1;

  if (Bsl2HdfDebug&BSL2HDF_DEBUG1) printf("sx_group_create( ..., %s , ... ) BEGIN\n",name);

  gid = H5Gcreate(loc_id, name,H5P_DEFAULT, H5P_DEFAULT, H5P_DEFAULT);
  if (gid<0) goto sx_group_create_error;

  if ( attributes ) {
    h5status = sx_attribdict_write( gid, attributes );
    if (h5status<0) goto sx_group_create_error;
  }

  if (Bsl2HdfDebug&BSL2HDF_DEBUG1) printf("sx_group_create( ..., %s , ... ) (gid=%d) END\n",name,gid);

  return(gid);

sx_group_create_error:

  printf("sx_group_create( ..., %s , ... ) (gid=%d) ERROR\n",name,gid);
  sx_group_close(gid);

  return(-1);

} // sx_group_create

/*
 * hid_t sx_group_open( hid_t loc_id, const char *name, DictOfDSets *attributes )
 * Open HDF5 group name:classname in loc_id
 */
hid_t sx_group_open( hid_t loc_id, const char *name,  DictOfDSets *attributes )
{
  hid_t gid=-1;
  herr_t h5status=-1;

  if (Bsl2HdfDebug&BSL2HDF_DEBUG1) printf("sx_group_open( ..., %s , ... ) BEGIN\n",name);

  // gid = H5Gcreate(loc_id, name,H5P_DEFAULT, H5P_DEFAULT, H5P_DEFAULT);
  // hid_t H5Gopen( hid_t loc_id, const char * name, hid_t gapl_id ) 
  gid = H5Gopen(loc_id, name,H5P_DEFAULT);
  if (gid<0) goto sx_group_open_error;

  if (attributes) {
    h5status = sx_attribdict_read( gid, attributes );
    if (h5status<0) goto sx_group_open_error;
  }

  if (Bsl2HdfDebug&BSL2HDF_DEBUG1) printf("sx_group_open( ..., %s , ... ) (gid=%d) END\n",name,gid);

  return(gid);

sx_group_open_error:

  printf("sx_group_open( ..., %s , ... ) (gid=%d) ERROR\n",name,gid);
  sx_group_close(gid);

  return(-1);

} // sx_group_open

/*
 * hid_t sx_dataset_create(hid_t loc_id, const char *name, hsize_t rank, hsize_t maxdims[], hid_t datatype_id,
 *                         hid_t filltype_id, const void *pfillvalue, DictOfDSets *attributes );
 *
 * DictOfDSets *attributes=NULL;
 * attributes=bsl2hdf_new_dictofdsets();
 * bsl2hdf_append_dset( attributes, attr_key, attr_val, 0, 1, -1 );
 * bsl2hdf_append_dset( attributes, SXCLASS, SXCLASS_ARRAY, 0, 1, -1 );
 * bsl2hdf_free_dictofdsets(attributes);
 */
hid_t sx_dataset_create(hid_t loc_id, const char *name, hsize_t rank, hsize_t maxdims[], hid_t datatype_id, 
                        hid_t filltype_id, const void *pfillvalue, DictOfDSets *attributes )
{
  const char *attr_key=NULL;
  const char *attr_val=NULL;
  Dset *stringdset; 

  hid_t dataprop=-1, dataspace=-1, dataset=-1, datatype=-1;
  hsize_t *dataspacesize=NULL, *chunk=NULL;
  herr_t h5status=-1;

  dataspacesize = (hsize_t *) MALLOC(sizeof(hsize_t)*rank);
  if ( !dataspacesize ) goto sx_dataset_create_error;

  chunk = (hsize_t *) MALLOC(sizeof(hsize_t)*rank);
  if ( !chunk ) goto sx_dataset_create_error;

  datatype = H5Tcopy(datatype_id);
  if ( datatype<0 ) goto sx_dataset_create_error;

  // Create dataspace (initially empty)
  dataspacesize[0] = 0;
  dataspacesize[1] = 0;
  dataspacesize[2] = 0;

  if (Bsl2HdfDebug&BSL2HDF_DEBUG2) PRINTRANK( "dataspacesize", dataspacesize );
  if (Bsl2HdfDebug&BSL2HDF_DEBUG2) PRINTRANK( "maxdims", maxdims );

  dataspace = H5Screate_simple(rank, dataspacesize, maxdims);
  if (dataspace<0) goto sx_dataset_create_error;

  // Create chunked dataset
  dataprop = H5Pcreate(H5P_DATASET_CREATE);
  if (dataprop<0) goto sx_dataset_create_error;

  // herr_t H5Pset_layout( hid_t plist, H5D_layout_t layout )
  // herr_t H5Pset_chunk(hid_t plist, int ndims, const hsize_t * dim )
  chunk[0] = 1; // dims[0];
  chunk[1] = maxdims[1];
  chunk[2] = maxdims[2];
  if (Bsl2HdfDebug&BSL2HDF_DEBUG2) PRINTRANK( "chunk", chunk );
  h5status = H5Pset_chunk(dataprop, rank, chunk);
  if (h5status<0) goto sx_dataset_create_error;

  if ((pfillvalue)&&(filltype_id>=0)) {
    if (Bsl2HdfDebug&BSL2HDF_DEBUG2) printf("H5Pset_fill_value(...)\n");
    // herr_t H5Pset_fill_value(hid_t dcpl_id, hid_t type_id, const void *value);
    h5status = H5Pset_fill_value(dataprop,filltype_id,pfillvalue);
    if (h5status<0) goto sx_dataset_create_error;
  }

  // hid_t H5Dcreate( hid_t loc_id, const char *name, hid_t dtype_id, hid_t space_id, hid_t lcpl_id, hid_t dcpl_id, hid_t dapl_id  )
  dataset = H5Dcreate (loc_id, name, datatype, dataspace, H5P_DEFAULT, dataprop, H5P_DEFAULT);
  if (dataset<0) goto sx_dataset_create_error;

  // run through stringds of all variable header elements
  stringdset = bsl2hdf_get_first_dset(attributes,&attr_key, &attr_val, NULL);

  while(stringdset) {

    if (!attr_key) {
      h5status=-1;
      printf("ERROR: get_dset_and_increment\n");
      goto sx_dataset_create_error;
    }

    h5status = sx_attrib_write( dataset, attr_key, attr_val );

    if (h5status<0) goto sx_dataset_create_error;

    stringdset = bsl2hdf_get_next_dset(stringdset, &attr_key, &attr_val, NULL);

  } // while


  H5Pclose( dataprop );

  H5Sclose( dataspace );

  H5Tclose( datatype );

  FREE(chunk);

  FREE(dataspacesize);

  return( dataset );

sx_dataset_create_error:

  printf("sx_dataset_create_error\n");

  if (dataset>=0) H5Dclose( dataset );
  if (dataprop>=0) H5Pclose( dataprop );
  if (dataspace>=0) H5Sclose( dataspace );
  if (datatype>=0) H5Tclose( datatype );
  FREE(chunk);
  FREE(dataspacesize);

  return( -1 );

} // sx_dataset_create

/* write 2D array data dimensions dims[3], which has chunk size (maxdims[0]=1,maxdims[1]=maxdim1,maxdims[2]=maxdim2 */
herr_t sx_dataset_write(hid_t dataset, long frame_no, hid_t mem_type_id,
                            hsize_t rank, hsize_t maxdims[], hsize_t dims[], void *data )
{
  hid_t filespace=-1, memtype=-1, memspace=-1;
  hsize_t *dataspacesize=NULL, *start=NULL, *count=NULL;
  hsize_t *memstart=NULL;
  herr_t  h5status=-1;

  const char *error_info=NULL;

  dataspacesize = (hsize_t *) MALLOC(sizeof(hsize_t)*rank);
  if ( !dataspacesize ) goto sx_dataset_write_error;

  start = (hsize_t *) MALLOC(sizeof(hsize_t)*rank);
  if ( !start ) goto sx_dataset_write_error;

  count = (hsize_t *) MALLOC(sizeof(hsize_t)*rank);
  if ( !count ) goto sx_dataset_write_error;

  memstart = (hsize_t *) MALLOC(sizeof(hsize_t)*rank);
  if ( !memstart ) goto sx_dataset_write_error;

  dims[0] = 1;  // must always be 1

  // Extend the dataset
  dataspacesize[0] = frame_no;
  dataspacesize[1] = maxdims[1];
  dataspacesize[2] = maxdims[2];

  if (Bsl2HdfDebug&BSL2HDF_DEBUG2) PRINTRANK( "dataspacesize", dataspacesize ); 

  // herr_t H5Dset_extent( hid_t dset_id, const hsize_t size[] ) 
  h5status = H5Dset_extent (dataset, dataspacesize);
  if (h5status<0) {
    goto sx_dataset_write_error;
  }

  // Get (extended) filespace for writing data (must be called AFTER H5Dset_extent!)
  // hid_t H5Dget_space( hid_t dataset_id )
  filespace = H5Dget_space( dataset );
  if (filespace<0) {
    goto sx_dataset_write_error;
  }

  // Define and select the first part of the hyperslab selection.

  // Start offset of each dimension 
  start[0] = frame_no-1;
  start[1] = 0;
  start[2] = 0;

  // Restrict number of elements in each dimension to maxdims
  count[0] = 1;
  count[1] = MIN(dims[1],maxdims[1]);
  count[2] = MIN(dims[2],maxdims[2]);

  if (Bsl2HdfDebug&BSL2HDF_DEBUG2) PRINTRANK( "start", start ); 
  if (Bsl2HdfDebug&BSL2HDF_DEBUG2) PRINTRANK( "count", count );
  if (Bsl2HdfDebug&BSL2HDF_DEBUG2) PRINTRANK( "dims", dims ); 

//  h5status = H5Sselect_hyperslab (filespace, H5S_SELECT_SET, start, stride, count, block);
  h5status = H5Sselect_hyperslab (filespace, H5S_SELECT_SET, start, NULL, count, NULL);
  if (h5status<0) {
    error_info="H5Sselect_hyperslab (filespace, ...)";
    goto sx_dataset_write_error;
  }

  // Get memory type 
  memtype = H5Tcopy( mem_type_id );
  if (memtype<0) {
    error_info="H5Tcopy";
    goto sx_dataset_write_error;
  }

  // Define the memory space and a hyperslab with count[] elements in each direction
  // starting at {0,0,0}
  memspace = H5Screate_simple (RANK, dims, NULL);
  if (memspace<0) {
    error_info="H5Screate_simple:...memspace...";
    goto sx_dataset_write_error;
  }
  if (Bsl2HdfDebug&BSL2HDF_DEBUG2) PRINTRANK( "memspacesize", dims );
  
  memstart[0] = 0;
  memstart[1] = 0;
  memstart[2] = 0;

  if (Bsl2HdfDebug&BSL2HDF_DEBUG2) PRINTRANK( "memstart", memstart );

  // h5status = H5Sselect_hyperslab (memspace, H5S_SELECT_SET, memstart, memstride, memcount, memblock);
  h5status = H5Sselect_hyperslab (memspace, H5S_SELECT_SET, memstart, NULL, count, NULL);
  if (h5status<0) {
    error_info="H5Sselect_hyperslab (memspace, ...)";
    goto sx_dataset_write_error;
  }

  // herr_t H5Dwrite(hid_t dataset, hid_t mem_type_id, hid_t mem_space_id, hid_t file_space_id, hid_t xfer_plist_id, const void * buf  )
  h5status = H5Dwrite (dataset, memtype, memspace, filespace, H5P_DEFAULT, data);
  if (h5status<0) {
    error_info="H5Dwrite";
    goto sx_dataset_write_error;
  }

  H5Sclose( memspace );
  H5Tclose( memtype );
  H5Sclose( filespace );

  FREE(memstart);
  FREE(dataspacesize);
  FREE(count);
  FREE(start);

  return(0);

sx_dataset_write_error:

  printf("sx_dataset_write_error:%s\n",error_info);
  printf("h5status = %d\n",h5status);
  if (memspace>=0) H5Sclose( memspace );
  if (memtype>=0) H5Tclose( memtype );
  if (filespace>=0) H5Sclose( filespace );

  FREE(memstart);
  FREE(dataspacesize);
  FREE(count);
  FREE(start);

  return(-1);

} // sx_dataset_write

herr_t sx_dataset_close(hid_t dataset)
{ herr_t h5status=0;
  if (dataset>=0)
    h5status = H5Dclose (dataset);
  return( h5status );
} // sx_dataset_close

herr_t sx_header_close( hid_t gid )
{ herr_t h5status=0;
  if (gid>=0) h5status = H5Gclose(gid);
  return( h5status );
} // sx_header_close

/*
 * Creates a string array dataset with name key and number_of_frames elements. 
 * Maxvallen is the maximum valid character length of each element.
 */
hid_t sxheader_stringds_create(hid_t loc_id, const char *key, 
  unsigned long maxvallen, unsigned long number_of_frames, DictOfDSets *attributes )
{
  hid_t dataprop=-1, dataspace=-1, dataset=-1, datatype=-1;
  hsize_t maxsize=-1;
  hsize_t datadim[1];
  herr_t h5status=-1;

  maxsize=(hsize_t) (maxvallen+1);
  datadim[0]=(hsize_t) number_of_frames;

  datatype = H5Tcopy (H5T_C_S1);
  // herr_t H5Tset_size( hid_t dtype_id, size_tsize )
  h5status = H5Tset_size (datatype, maxsize);
  if (h5status<0) goto sxheader_stringds_create_error;

// herr_t H5Tset_strpad( hid_t dtype_id, H5T_str_t strpad ) with strpad=H5T_STR_NULLTERM (C)
   h5status = H5Tset_strpad( datatype,H5T_STR_NULLTERM);
   if (h5status<0) goto sxheader_stringds_create_error;

  dataspace =  H5Screate_simple (1, datadim, NULL); // create with maximum size
  if (dataspace<0) goto sxheader_stringds_create_error;

  // Create dataset properties
  dataprop = H5Pcreate(H5P_DATASET_CREATE);
  if (dataprop<0) goto sxheader_stringds_create_error;

  //herr_t H5Pset_alloc_time(hid_t plist_id, H5D_alloc_time_t alloc_time )
  h5status = H5Pset_alloc_time(dataprop, H5D_ALLOC_TIME_EARLY ); 
  if (h5status<0) goto sxheader_stringds_create_error;

  //create
  // hid_t H5Dcreate( hid_t loc_id, const char *name, hid_t dtype_id, hid_t space_id, hid_t lcpl_id, hid_t dcpl_id, hid_t dapl_id  )
  dataset = H5Dcreate (loc_id, key, datatype, dataspace, H5P_DEFAULT, dataprop, H5P_DEFAULT);
  if (dataset<0) goto sxheader_stringds_create_error;

  /*
   * Write attributes
   */
  if (sx_attribdict_write( dataset, attributes )<0) goto sxheader_stringds_create_error;

  //close
  if ( H5Pclose(dataprop) < 0) goto sxheader_stringds_create_error;
  if ( H5Sclose(dataspace) < 0) goto sxheader_stringds_create_error;
  if ( H5Tclose(datatype) < 0) goto sxheader_stringds_create_error;

  return( dataset );

sxheader_stringds_create_error:

  printf("sxheader_stringds_create_error\n");

  if (dataprop>=0) H5Pclose( dataprop );
  if (dataset>=0) H5Dclose( dataset );
  if (dataspace>=0) H5Sclose( dataspace );
  if (datatype>=0) H5Tclose( datatype );

  return( -1 );

} // sxheader_stringds_create

herr_t sxheader_stringds_close( hid_t dataset )
{ herr_t h5status=0;
  if (dataset>=0) h5status=H5Dclose(dataset);
  return(h5status);
} // sxheader_stringds_close

/*
 * Writes value to string dataset
 */
herr_t sxheader_stringds_write(hid_t dataset, unsigned long maxvallen, 
                               unsigned long no_of_frames,
                               const char *value )
{
  hid_t memtype=-1, memspace=-1;
  hsize_t maxsize=-1;
  hid_t dataspace=-1;

  hsize_t memdims[1];
  hsize_t start[1], count[1];

  char *error_info="";
  herr_t h5status=-1;

  maxsize=(hsize_t) (maxvallen+1);

  // Get dataspace for writing data
  // hid_t H5Dget_space( hid_t dataset_id )
  dataspace = H5Dget_space( dataset );
  if (dataspace<0) {
    error_info="H5Dget_space ( dataset )";
    goto sxheader_stringds_write_error;
  }

  // Get memory type
  memtype = H5Tcopy( H5T_C_S1 ); //
  if (memtype<0) {
    error_info="H5Tcopy";
    goto sxheader_stringds_write_error;
  }

  // herr_t H5Tset_size( hid_t dtype_id, size_tsize )
  h5status = H5Tset_size (memtype, maxsize);
  if (h5status<0) {
    error_info="H5Tset_size(memtype,...)";
    goto sxheader_stringds_write_error;
  }

  // Define the memory space and a hyperslab with count[] elements in each direction
  memdims[0] = no_of_frames;
  memspace = H5Screate_simple (1, memdims, NULL);
  if (memspace<0) {
    error_info="H5Screate_simple:...memspace...";
    goto sxheader_stringds_write_error;
  }

  // herr_t H5Dwrite(hid_t dataset, hid_t mem_type_id, hid_t mem_space_id, hid_t file_space_id, hid_t xfer_plist_id, const void * buf  )
  h5status = H5Dwrite (dataset, memtype, H5S_ALL, dataspace, H5P_DEFAULT, value);

  if (h5status<0) {
    error_info="H5Dwrite";
    goto sxheader_stringds_write_error;
  }

  H5Sclose( memspace );
  H5Tclose( memtype );
  H5Sclose( dataspace );

  return(h5status);

sxheader_stringds_write_error:

  printf("sxheader_stringds_write_error:%s\n",error_info);
  printf("h5status = %d\n",h5status);

  if (memspace>=0) H5Sclose( memspace );
  if (memtype>=0) H5Tclose( memtype );
  if (dataspace>=0) H5Sclose( dataspace );

  return(-1);

} // sxheader_stringds_write

/*---------------------------------------------------------------------------
NAME

  sx_header_write --- Writes sx headers as parameter group to loc_id

PURPOSE

  Creates a parameter group gp_name in loc_id and fills it with datasets
  consisting of key-value-pairs of the edf image headers DataNumbers[] of 
  chain Memory in edf stream.

  sx_header_write can only be used after a first loop through all key-value 
  pairs that have to be written. During this loop the header 'header_key' 
  must have been created. It must have been filled with all header keys 
  that have to be written. It contains automatically the information what
  values are constant and need only to be written once.

ARGUMENTS

  hid_t loc_id         : location in which the group gp_name is created
  const char *gp_name  : name of the group
  int stream           : edf stream
  int Chain            : edf chain number (>0 for data,<0 for variances,
                         0 is reserved and cannot be used
  const long DataNumbers[] : array containing all image numbers, 
                             DataNumber[0] contains the number of images
                             (frames) 
  const char *header_key : header key which has been filled in a first loop
                           with key-value pairs.
  int *pErrorValue      : edf error value
  int *pstatus          : status value, set in case of an error

RETURN VALUE

  herr_t hdf error status, negative in case of an error

---------------------------------------------------------------------------*/
herr_t sx_header_write(hid_t loc_id, const char *gp_name,
                            int stream, int Chain, const long DataNumbers[], 
                            const char *header_key, 
                            int *pErrorValue, int *pstatus)
{ int ErrorValue=0, status=0;
  hid_t sxheader_gid=-1;
  herr_t h5status=-1;
  htri_t gp_exists=-1, ds_exists=-1;
  char *sxkey=NULL;
  const char *key, *val;


  DictOfDSets *header=NULL;
  DictOfDSets *headerattributes=NULL;
  DictOfDSets *symbolattributes=NULL;

  headerattributes=bsl2hdf_new_dictofdsets();
  bsl2hdf_append_dset( headerattributes, SXCLASS, SXCLASS_HEADER, 0, 1, -1 );

  symbolattributes=bsl2hdf_new_dictofdsets();
  bsl2hdf_append_dset( symbolattributes, SXCLASS, SXCLASS_SYMBOL, 0, 1, -1 );

  // Get first key. There is no need to create an empty group.
  edf_first_header_element( header_key, &key, &val, &ErrorValue, &status);
  if (status) {
    printf("%s\n",edf_report_data_error(ErrorValue));
    goto sx_header_write_error;
  }

  if ((key) && (DataNumbers) && (DataNumbers[0] > 0)) {
    int variable_header_values=0;
    unsigned long no_of_frames;
    long DataNumber;

    /*
     * Create data spaces of header value strings
     */
    no_of_frames = DataNumbers[0];

    /*
     * Create all sxheader keys and write constant values
     */
    DataNumber = DataNumbers[1];

    /*
     * print sx-parameters of images (global)
     */
    edf_print_header_stat ( stdout, header_key, Bsl2HdfLevel-1, Bsl2HdfDebug&BSL2HDF_VERBOSE );

    /*
     * create string datasets for each header value:
     * - a single string of length MaxValLen if the value is constant for all frames,
     * - an array with no_of_frames strings of length MaxValLen if the header value 
     *   changes from frame to frame (ValueUpdateCount>1);
     */

    // htri_t H5Lexists( hid_t loc_id, const char *name, hid_t lapl_id )
    if (gp_exists=sx_link_exists( loc_id, gp_name )<0) goto sx_header_write_error;

    if (gp_exists) {
      if ( (sxheader_gid = sx_group_open( loc_id, gp_name, headerattributes )) <0 ) goto sx_header_write_error;
    } else {
      /*
       * hid_t sx_group_create( hid_t loc_id, const char *name, DictOfDSets *headerattributes )
       * Create HDF5 group key_%d:classname in loc_id
       */
      if ( (sxheader_gid = sx_group_create( loc_id, gp_name, headerattributes ))<0 ) goto sx_header_write_error;
    }

    /*
     * Writing parameters to sxheader_gid BEGIN
     */
    unsigned long KeyUpdateCount, ValueUpdateCount, MaxValLen;
    Dset *stringdset; 
    hid_t stringds;

    /*
     * Create dictionary of datasets
     */
    header = bsl2hdf_new_dictofdsets();

    // run through all header elements of header_key
    while(key != NULL) {

      sxkey = newsxkey( SYMBOLPREFIX, key );

      // "MinValLen", "MaxValLen", "KeyUpdateCount", "ValueUpdateCount"
      // unsigned long edf_stat_header_element  ( const char *header_key,
      //       const char *key, const char *statkey, int *pErrorValue, int *pstatus );

      KeyUpdateCount = edf_stat_header_element(header_key, key, "KeyUpdateCount", 
        &ErrorValue, &status);
      if (status) { 
        printf("%s\n",edf_report_data_error(ErrorValue)); 
        goto sx_header_write_error;
      }
      ValueUpdateCount = edf_stat_header_element(header_key, key, "ValueUpdateCount", 
        &ErrorValue, &status);
      if (status) { 
        printf("%s\n",edf_report_data_error(ErrorValue)); 
        goto sx_header_write_error;
      }
      MaxValLen = edf_stat_header_element(header_key, key, "MaxValLen", 
        &ErrorValue, &status);
      if (status) { 
        printf("%s\n",edf_report_data_error(ErrorValue)); 
        goto sx_header_write_error;
      }

      if ( (ValueUpdateCount == 1 ) && (KeyUpdateCount >= no_of_frames) ) {
        // single string value, use values from header header_key (length=MaxValLen)
        if (Bsl2HdfDebug&BSL2HDF_SHOWHEADER) printf("\'%s\' = \'%s\'  ->  \'%s\' (%lu x %lu)\n",key,val,sxkey,1,MaxValLen);

        if (ds_exists=sx_link_exists( sxheader_gid, sxkey )<0) goto sx_header_write_error;

          if (!ds_exists) {
            // create sxheader key dataset
            if (( stringds = sxheader_stringds_create( sxheader_gid, sxkey, MaxValLen, 1, symbolattributes ) )<0) {
              status=-1;
              goto sx_header_write_error;
            }

            //  write value to stringds
            if ( (h5status=sxheader_stringds_write( stringds, MaxValLen, 1, val)) < 0 ) {
              status=-1;
              goto sx_header_write_error;
            }

            // close dataset
            sxheader_stringds_close(stringds);
            stringds=-1;
          }

        // delete key from header header_key (to keep only variable symbols)
        edf_delete_header_element   ( header_key, key, &ErrorValue, &status);
        if (status) {
          printf("%s\n",edf_report_data_error(ErrorValue));
          goto sx_header_write_error;
        }

      } else {
        if (Bsl2HdfDebug&BSL2HDF_SHOWHEADER) printf("\'%s\' = \'%s\'  ->  \'%s\' (%lu x %lu)\n",
          key,"(variable)",sxkey,no_of_frames,MaxValLen);

        if (ds_exists=sx_link_exists( sxheader_gid, sxkey )<0) goto sx_header_write_error;

        if (!ds_exists) {
          // create variable datasets and keep them in a dictionary (filled in a consecutive loop)
          // need no_of_frames elements of fixed length strings with MaxValLen characters
          variable_header_values=1; // there are variable values 

          // create sxheader key dataset
          if (( stringds = sxheader_stringds_create( sxheader_gid, sxkey, MaxValLen, no_of_frames, symbolattributes ) )<0) {
            status=-1;
            goto sx_header_write_error;
          }

          // bsl2hdf_append_dset(DictOfDSets *dictofdsets, const char *key, const char *value, unsigned long no_of_frames, hid_t dataset);
          if (!(bsl2hdf_append_dset(header, key, NULL, MaxValLen, no_of_frames, stringds))) {
            sxheader_stringds_close(stringds);
            stringds=-1;
            status=-1;
            goto sx_header_write_error;
          }
        }
      }

      if (sxkey) { FREE(sxkey); sxkey=NULL; }

      edf_next_header_element( header_key, &key, &val, &ErrorValue, &status);
      if (status) { 
        printf("%s\n",edf_report_data_error(ErrorValue)); 
        goto sx_header_write_error;
      }

    } // while

    // update variable header values for all frames ( in header )
    if (variable_header_values) {
      long frame_no;
      const char *value=NULL; // array with all strings
      if (Bsl2HdfDebug&BSL2HDF_SHOWHEADER) {
        bsl2hdf_print_dictofdsets(stdout, header,"sx_header_write");
      }
      no_of_frames = DataNumbers[0];
      for (frame_no=1;frame_no<=no_of_frames;frame_no++) {

        DataNumber = DataNumbers[frame_no];

        if ( edf_test_header ( stream, DataNumber, Chain, NULL, NULL ) ) {

          // read full edf header
          edf_read_header         ( stream, DataNumber, Chain, SXTMP,
                                    &ErrorValue, &status );
          if (status) {
            printf("%s\n",edf_report_data_error(ErrorValue));
            goto sx_header_write_error;
          }

          // run through stringds of all variable header elements
          stringdset = bsl2hdf_get_first_dset(header,&key, NULL, NULL);
          while(stringdset) {

            if (!key) { 
              status=-1;
              printf("ERROR: get_dset_and_increment\n"); 
              goto sx_header_write_error;
            }

            // use variable values from header SXTMP
            edf_search_header_element(SXTMP, key, &val, &ErrorValue, &status);
            if (status) {
              printf("%s\n",edf_report_data_error(ErrorValue));
              goto sx_header_write_error;
            }


            if (Bsl2HdfDebug&BSL2HDF_SHOWHEADER) printf("\'%s\' = \'%s\' [frame %lu]\n",key,val,DataNumber);

            // const char *bsl2hdf_write_str_dset(Dset *dset, unsigned long frame_no, const char *value);*
            if ( bsl2hdf_write_str_dset(stringdset, frame_no, val) < 0 ) {
              status=-1;
              goto sx_header_write_error;
            }

            stringdset = bsl2hdf_get_next_dset(stringdset, &key, NULL, NULL);

          } // while

          edf_free_header(SXTMP);

        }

      } // for DataNumber

      if (Bsl2HdfDebug&BSL2HDF_SHOWHEADER) {
        bsl2hdf_print_dictofdsets(stdout, header,"sx_header_write");
      }

      // run through stringds of all variable header elements
      stringdset = bsl2hdf_get_first_dset(header,&key, &value, &stringds);
      while(stringdset) {

        if (!key) {
          status=-1;
          printf("ERROR: get_dset_and_increment, no key: stringds=%ld\n",stringds);
          goto sx_header_write_error;
        }

        //  write value to stringds
        // herr_t sxheader_stringds_write(hid_t dataset, unsigned long no_of_frames,
        //                                unsigned long maxvallen,
        //                                const char *value )
        if ( (h5status=sxheader_stringds_write( stringds, stringdset->maxvallen, 
              no_of_frames, value )) < 0 ) {
          status=-1;
          goto sx_header_write_error;
        }

        stringdset = bsl2hdf_get_next_dset(stringdset, &key, &value, &stringds);
      }


    } // variable_header_values

    /*
     * Writing parameters to sxheader_gid END
     */

    if (( h5status = sx_group_close(sxheader_gid) ) < 0) {
      sxheader_gid=-1;
      status=-1;
      goto sx_header_write_error;
    }

    // closes also all datasets
    header = bsl2hdf_free_dictofdsets(header);


  } // DataNumbers[0] > 0

  headerattributes=bsl2hdf_free_dictofdsets(headerattributes);
  symbolattributes=bsl2hdf_free_dictofdsets(symbolattributes);

  if (pErrorValue) *pErrorValue=ErrorValue;
  if (pstatus) *pstatus=status;

  if (sxkey) FREE(sxkey);
  edf_free_header(SXTMP);

  return(h5status);

sx_header_write_error:

  if (pErrorValue) *pErrorValue=ErrorValue;
  if (pstatus) *pstatus=status;

  bsl2hdf_free_dictofdsets(header);
  bsl2hdf_free_dictofdsets(headerattributes);
  bsl2hdf_free_dictofdsets(symbolattributes);
  sx_group_close(sxheader_gid);

  if (h5status>=0) h5status=-1;

  if (sxkey) FREE(sxkey);
  edf_free_header(SXTMP);

  return(h5status);

} // sx_header_write

typedef struct {
  DictOfDSets *searchstrings;
  DictOfDSets *foundstrings;
  int groupsearchmode; // GroupSearchMode enum type
} SXLinkSearch_data;

/*---------------------------------------------------------------------------
NAME

  sx_dataset_iteration -- iteration function called by H5Literate in sx_search_dataset


PURPOSE

  Tests, whether object name is a dataset and whether one of its attributes matches
  an attributes and value contained in searchstrings.
  If yes, add the dataset name to foundstrings.
  ...

ARGUMENTS

  herr_t sx_dataset_iteration (hid_t loc_id, const char *name, const H5L_info_t *info, void *op_data)

  hid_t loc_id           : location in which the iteration is done.
  const char *name       : name of the link
  const H5L_info_t *info : info of the link
  void *op_data          : pointer to a structure of type SXLinkSearch_data:
                           typedef struct {
                             DictOfDSets *searchstrings;
                             DictOfDSets *foundstrings;
                           } SXLinkSearch_data;

RETURN VALUE

  herr_t hdf error value, negative in case of an error

---------------------------------------------------------------------------*/
herr_t sx_dataset_iteration (hid_t loc_id, const char *dataset_name, const H5L_info_t *info, void *op_data)
{/* typedef struct {
  *   H5L_type_t     type;         // Type of link
  *   hbool_t        corder_valid; // Indicates whether creation
  *                                // order is valid
  *   int64_t        corder;       // Creation order
  *   H5T_cset_t     cset;         // Character set of link name
  *   union {
  *       haddr_t    address;      // Address hard link points to
  *       size_t     val_size;     // Size of soft link or
  *                                // user-defined link value
  *   } u;
  * } H5L_info_t;
  */

  H5O_info_t object_info;

  SXLinkSearch_data *plink_data=op_data;

  DictOfDSets *searchstrings=NULL;
  DictOfDSets *foundstrings=NULL;

  int attribute_match=0; // no match so far

  if (plink_data) {
    searchstrings = plink_data->searchstrings;
    foundstrings = plink_data->foundstrings;
  }

  if (Bsl2HdfDebug&BSL2HDF_DEBUG1) {
    printf(">> sx_dataset_iteration (id=%d,dataset_name=%s,info,op_data)",loc_id,dataset_name);
    if (Bsl2HdfDebug&BSL2HDF_DEBUG2) {
      switch(info->cset) {
        case H5T_CSET_ASCII: printf(" (H5T_CSET_ASCII)");
                             break;
        case H5T_CSET_UTF8 : printf(" (H5T_CSET_UTF8)");
                             break;
        default: printf(" (unknown character set %d)",info->cset);
      }
    }
    printf(" BEGIN\n");
  }

  /*
   * open dataset_id and read attribute sxclassname
   */
  hid_t dataset_id=-1;
  // hid_t H5Oopen( hid_t loc_id, const char *name, hid_t lapl_id );

  dataset_id = H5Oopen( loc_id, dataset_name, H5P_DEFAULT );
  if (dataset_id<0) goto sx_dataset_iteration_error;

  /*
   * Verify, that object dataset_id is a dataset, return if not
   * H5O_TYPE_GROUP, H5O_TYPE_DATASET, H5O_TYPE_NAMED_DATATYPE, H5O_TYPE_NTYPES, H5O_TYPE_UNKNOWN
   */
  // herr_t H5Oget_info( hid_t object_id, H5O_info_t *object_info )
  if ( H5Oget_info( dataset_id, &object_info ) <0 ) goto sx_dataset_iteration_error;

  if (object_info.type == H5O_TYPE_DATASET) {

    /*
     * Search a match between a dataset attribute (of dataset 'dataset_name')
     * and attributes contained in searchstrings
     */
    const char *search_key=NULL, *search_value=NULL;
    Dset *stringds=NULL;
    char buffer[STRINGVSIZE];
    const char *attr_value=NULL;

    /*
     * Compare searchstrings with dataset attributes (of dataset 'dataset_name')
     * Stop on first match.
     */
    attribute_match=0;
    stringds = bsl2hdf_get_first_dset(searchstrings,&search_key,&search_value, NULL);
    while(stringds) {
      htri_t attr_exists=-1;
      if (search_key) {
        attr_exists=H5Aexists(dataset_id,search_key);
        if (attr_exists<0) goto sx_dataset_iteration_error;
        if (attr_exists) {
          attr_value = sx_attrib_read( buffer, STRINGVSIZE,dataset_id,search_key);
          if (!strlib_casecmp(search_value,attr_value)) {
            /* matching attribute found, stop loop */
            attribute_match=+1;
            // printf("\"%s\": %s = %s\n",dataset_name,search_key,search_value); //+++++++++++++=
            break; // stop loop over searchstrings
          }
        } // if attr_exists
      }
      stringds = bsl2hdf_get_next_dset(stringds, &search_key, &search_value, NULL);
    } // while

    if (attribute_match>0) {
      if (foundstrings) {
        /* write search_key and dataset_name to foundstrings */
        bsl2hdf_append_dset( foundstrings, search_key, dataset_name, 0, 1, -1 );
      }
    }

  } // if H5O_TYPE_DATASET

  // close dataset_id
  H5Oclose( dataset_id );

  if (Bsl2HdfDebug&BSL2HDF_DEBUG1)
    printf(">> sx_dataset_iteration (id=%d,dataset_name=%s,info,op_data) (%d) END\n",loc_id,dataset_name,attribute_match);

  //  0 to continue  iteration
  // >0 to interrupt iteration (H5Literate returns this value)

  return(attribute_match); // stop, if a match has been found

sx_dataset_iteration_error:

  if (dataset_id>=0) H5Oclose( dataset_id );

  if (Bsl2HdfDebug&BSL2HDF_DEBUG1)
    printf(">> sx_dataset_iteration (id=%d,dataset_name=%s,info,op_data) (-1) ERROR\n",loc_id,dataset_name);

  return(-1);

} // sx_dataset_iteration

/*---------------------------------------------------------------------------
NAME

  sx_search_dataset ---

PURPOSE

 sx_search_dataset iterates through all datasets in loc_id. If one of the attributes
 of a dataset matches one of the attributes given in searchstrings, its name 
 is added to foundstrings.   

ARGUMENTS

  herr_t sx_search_dataset( hid_t loc_id,
                            DictOfDSets *searchstrings, DictOfDSets *foundstrings );

  hid_t loc_id         : location in which the group gp_name is created

  DictOfDSets *searchstrings (input) : search string dictionary
  DictOfDSets *foundstrings (output) : dictionary containing the names
                                       of matching datasets.

RETURN VALUE

  herr_t h5 error, negative in case of an error

---------------------------------------------------------------------------*/
herr_t sx_search_dataset( hid_t loc_id, DictOfDSets *searchstrings, DictOfDSets *foundstrings )
{ herr_t h5status=-1;

  if (Bsl2HdfDebug&BSL2HDF_DEBUG1) printf("sx_search_dataset( %d, %p, %p ) BEGIN\n",loc_id,searchstrings,foundstrings);

  if (foundstrings) {
    SXLinkSearch_data link_data;

    link_data.searchstrings=searchstrings;
    link_data.foundstrings=foundstrings;

    hsize_t idx=0;
    h5status=H5Literate( loc_id, H5_INDEX_NAME, H5_ITER_INC, &idx, sx_dataset_iteration, (void *) &link_data); 
    if (Bsl2HdfDebug&BSL2HDF_DEBUG1) printf("sx_search_dataset idx=%ld\n",(long)idx);
  }

  if (Bsl2HdfDebug&BSL2HDF_DEBUG1) printf("sx_search_dataset( %d, %p, %p ) (h5status=%d) END\n",loc_id,searchstrings,foundstrings,h5status);

  return(h5status);

} // sx_search_dataset

/*---------------------------------------------------------------------------
NAME

  sx_group_iteration -- iteration function called by H5Literate in sx_search_group


PURPOSE

  Tests, whether object name is a group and whether one of its attributes matches 
  an attributes and value contained in searchstrings.
  If yes, determine the group number and add number and name to foundstrings.
  ...

ARGUMENTS

  herr_t sx_group_iteration (hid_t loc_id, const char *name, const H5L_info_t *info, void *op_data)

  hid_t loc_id           : location in which the iteration is done.
  const char *name       : name of the link
  const H5L_info_t *info : info of the link
  void *op_data          : pointer to a structure of type SXLinkSearch_data:
                           typedef struct {
                             DictOfDSets *searchstrings;
                             DictOfDSets *foundstrings;
                           } SXLinkSearch_data;

RETURN VALUE

  herr_t hdf error value, negative in case of an error

---------------------------------------------------------------------------*/
herr_t sx_group_iteration (hid_t loc_id, const char *group_name, const H5L_info_t *info, void *op_data)
{/* typedef struct {
  *   H5L_type_t     type;         // Type of link
  *   hbool_t        corder_valid; // Indicates whether creation
  *                                // order is valid
  *   int64_t        corder;       // Creation order
  *   H5T_cset_t     cset;         // Character set of link name
  *   union {
  *       haddr_t    address;      // Address hard link points to
  *       size_t     val_size;     // Size of soft link or
  *                                // user-defined link value
  *   } u;
  * } H5L_info_t;
  */

  H5O_info_t object_info;

  SXLinkSearch_data *plink_data=op_data;

  DictOfDSets *searchstrings=NULL;
  DictOfDSets *foundstrings=NULL;
  int groupsearchmode=0;

  DictOfDSets *search_attributes=NULL;
  DictOfDSets *found_datasets=NULL;

  char group_name_with_path[BSL2HDF_BUFLEN]="";

  if (plink_data) {
    searchstrings = plink_data->searchstrings;
    foundstrings = plink_data->foundstrings; 
    groupsearchmode = plink_data->groupsearchmode; 
  }

  if (Bsl2HdfDebug&BSL2HDF_DEBUG1) {
    printf(">> sx_group_iteration (id=%d,group_name=%s,info,op_data)",loc_id,group_name);
    if (Bsl2HdfDebug&BSL2HDF_DEBUG2) {
      switch(info->cset) {
        case H5T_CSET_ASCII: printf(" (H5T_CSET_ASCII)");
                             break;
        case H5T_CSET_UTF8 : printf(" (H5T_CSET_UTF8)");
                             break;
        default: printf(" (unknown character set %d)",info->cset);
      }
    }
    printf(" BEGIN\n");
  }

  /*
   * Get the full name of the link (<path>/<group_name>)
   */
  // ssize_t H5Iget_name( hid_t obj_id, char *name, size_t size )
  if (H5Iget_name(loc_id,group_name_with_path,BSL2HDF_BUFLEN )<0) goto sx_group_iteration_error;
  if (strcmp("/",group_name_with_path))
    strlib_concat( group_name_with_path, BSL2HDF_BUFLEN, group_name_with_path, "/" );
  strlib_concat( group_name_with_path, BSL2HDF_BUFLEN, group_name_with_path, group_name );

  /* 
   * open group_id and read attribute sxclassname
   */
  hid_t group_id=-1;
  // hid_t H5Oopen( hid_t loc_id, const char *name, hid_t lapl_id );

  group_id = H5Oopen( loc_id, group_name, H5P_DEFAULT );
  if (group_id<0) goto sx_group_iteration_error;

  /*
   * Verify, that object group_id is a group, return if not
   * H5O_TYPE_GROUP, H5O_TYPE_DATASET, H5O_TYPE_NAMED_DATATYPE, H5O_TYPE_NTYPES, H5O_TYPE_UNKNOWN
   */
  // herr_t H5Oget_info( hid_t object_id, H5O_info_t *object_info )
  if ( H5Oget_info( group_id, &object_info ) <0 ) goto sx_group_iteration_error;

  if (object_info.type==H5O_TYPE_GROUP) {

    /*
     * Search a match between a group attribute (of group 'group_name')
     * and attributes contained in searchstrings
     */
    const char *search_key=NULL, *search_value=NULL;
    Dset *stringds=NULL;
    int attribute_match=-1;
    char buffer[STRINGVSIZE];
    const char *attr_value=NULL;

    /* 
     * Compare searchstrings with group attributes (of group 'group_name')
     * Stop on first match.
     */
    attribute_match=0;
    stringds = bsl2hdf_get_first_dset(searchstrings,&search_key,&search_value, NULL);
    while(stringds) {
      htri_t attr_exists=-1;
      if (search_key) {
        attr_exists=H5Aexists(group_id,search_key);
        if (attr_exists<0) goto sx_group_iteration_error;
        if (attr_exists) {
          attr_value = sx_attrib_read( buffer, STRINGVSIZE,group_id,search_key);
          if (!strlib_casecmp(search_value,attr_value)) {
            /* matching attribute found, stop loop */
            attribute_match=1;
            // printf("\"%s\": %s = %s\n",group_name,search_key,attr_value); //+++++++++++++=
            break; // stop loop over searchstrings
          } 
        } // if attr_exists
      }
      stringds = bsl2hdf_get_next_dset(stringds, &search_key, &search_value, NULL);
    }

    if (attribute_match) {
      const char *ds_name=NULL;
      const char *group_number_string=NULL;
      htri_t number_exists=-1;

      switch (groupsearchmode) {

      case DataBlockSearch:

        if (foundstrings) {
          // write group_number and group_name_with_path
          bsl2hdf_append_dset( foundstrings, "1", group_name_with_path, 0, 1, -1 );
        }

        break;

      case ErrorBlockSearch:

        if (foundstrings) {
          // write group_number and group_name_with_path
          bsl2hdf_append_dset( foundstrings, "-1", group_name_with_path, 0, 1, -1 );
        }

        break;

      default: // case EntrySearch, MemorySearch:

        /*
         * Determine group number:
         * id : group_id
         * key : group_name
         * a) try reading the group number from the dataset GROUPNUMBERNAME
         * b) search for a dataset in group_id with attribute SXCLASS_GROUPNUMBER
         *    and take its value as group number
         * c) converting group group_name to a number, e.g. group_name=<blabla><group_number>.
         */

        // a)
        ds_name = GROUPNUMBERNAME;
        if ((number_exists=sx_link_exists( group_id, ds_name ))<0) goto sx_group_iteration_error;
        if (!number_exists) ds_name=NULL;

        // b) 
        if (!ds_name) {
          // herr_t sx_search_dataset( hid_t group_id, DictOfDSets *search_attributes, DictOfDSets *found_datasets )
          found_datasets=bsl2hdf_new_dictofdsets();
          search_attributes=bsl2hdf_new_dictofdsets();
          bsl2hdf_append_dset( search_attributes, SXCLASS, SXCLASS_GROUPNUMBER, 0, 1, -1 );
          if ( sx_search_dataset( group_id, search_attributes, found_datasets )<0 ) goto sx_group_iteration_error;
          bsl2hdf_get_first_dset(found_datasets,NULL,&ds_name,NULL);
        }

        if (ds_name) {
          // read group number from dataset ds_name
          group_number_string=sx_stringds_read(buffer,STRINGVSIZE,group_id,ds_name,NULL);
        } // if ds_name

        // c)
        if (!group_number_string) {
          const char *pg;
          long number=0;

          // skip non-digit characters
          for (pg=group_name;*pg;pg++) {
            if ( isdigit(*pg)||(*pg=='-')||(*pg=='+') ) break;
          }

          number = (long) num_str2double( pg, NULL, NULL);
          group_number_string=num_long2str ( buffer, STRINGVSIZE, number, NULL );
        }

        if (!group_number_string)
          group_number_string = "1"; // just for now to be safe //++++++++++++++++++

        if (foundstrings) {
          // write group_number and group_name_with_path
          bsl2hdf_append_dset( foundstrings, group_number_string, group_name_with_path, 0, 1, -1 );
        }
      
        break;

      } // switch groupsearchmode

    } // attribute_match

  } // H5O_TYPE_GROUP

  found_datasets=bsl2hdf_free_dictofdsets(found_datasets);
  search_attributes=bsl2hdf_free_dictofdsets(search_attributes);

  // close group_id
  H5Oclose( group_id ); 

  if (Bsl2HdfDebug&BSL2HDF_DEBUG1)
    printf(">> sx_group_iteration (id=%d,group_name=%s,info,op_data) (0) END\n",loc_id,group_name);

  //  0 to continue  iteration
  // >0 to interrupt iteration (H5Literate returns this value)
  return(0);

sx_group_iteration_error:

  found_datasets=bsl2hdf_free_dictofdsets(found_datasets);
  search_attributes=bsl2hdf_free_dictofdsets(search_attributes);

  if (group_id>=0) H5Oclose( group_id ); 

  if (Bsl2HdfDebug&BSL2HDF_DEBUG1)
    printf(">> sx_group_iteration (id=%d,group_name=%s,info,op_data) (-1) ERROR\n",loc_id,group_name);

  return(-1);

} // sx_group_iteration

/*---------------------------------------------------------------------------
NAME

  sx_search_group --- 


PURPOSE

 sx_search_group iterates through all groups in loc_id. If one of the attributes
 of a group matches one of the attributes given in searchstrings, the corresponding
 groupnumber and the group name is added to foundstrings. 

ARGUMENTS

  herr_t sx_search_group( hid_t loc_id, 
                         DictOfDSets *searchstrings, DictOfDSets *foundstrings );

  hid_t loc_id         : location in which the group gp_name is created

  int groupsearchmode  : EntrySearch : search an entry group and determine its group number
                         MemorySearch: search a memory group and determine its group number 
                      DataBlockSearch: search a data block group
                     ErrorBlockSearch: search an error block group

  DictOfDSets *searchstrings (input) : search string dictionary
  DictOfDSets *foundstrings (output) : dictionary containing names of found groups 
                                       and their group numbers


RETURN VALUE

  herr_t h5 error, negative in case of an error

---------------------------------------------------------------------------*/
herr_t sx_search_group( hid_t loc_id, int groupsearchmode, DictOfDSets *searchstrings, DictOfDSets *foundstrings ) 
{ herr_t h5status=-1;

  if (Bsl2HdfDebug&BSL2HDF_DEBUG1) printf("sx_search_group( %d, %s, %p, %p ) BEGIN\n",loc_id,groupsearchmode2str(groupsearchmode),searchstrings,foundstrings);


  if (foundstrings) {
    SXLinkSearch_data link_data;
    link_data.searchstrings = searchstrings;
    link_data.foundstrings = foundstrings;
    link_data.groupsearchmode = groupsearchmode;

    // herr_t H5Literate( hid_t group_id, H5_index_t index_type, H5_iter_order_t order, hsize_t *idx, H5L_iterate_t op, void *op_data ) 
    // index_type=H5_INDEX_NAME, by name
    // order=H5_ITER_INC, incremental
    hsize_t idx=0;
    h5status=H5Literate( loc_id, H5_INDEX_NAME, H5_ITER_INC, &idx, sx_group_iteration, (void *) &link_data);
    if (h5status<0) goto sx_search_group_error;
  }

  if (Bsl2HdfDebug&BSL2HDF_DEBUG1) 
    printf("sx_search_group( %d, %s, %p, %p ) (h5status=%d) END\n",loc_id,groupsearchmode2str(groupsearchmode),searchstrings,foundstrings,h5status);

  return(h5status);

sx_search_group_error:

  if (Bsl2HdfDebug&BSL2HDF_DEBUG1) printf("sx_search_group( %d, %s, %p, %p ) (h5status=%d) ERROR\n",loc_id,groupsearchmode2str(groupsearchmode),searchstrings,foundstrings,h5status);
  return(-1);

} // sx_search_group

herr_t sx_search_group_by_name( hid_t loc_id,  const char *name, int groupsearchmode, DictOfDSets *searchstrings, DictOfDSets *foundstrings )
{ herr_t h5status=-1;

  if (Bsl2HdfDebug&BSL2HDF_DEBUG1) printf("sx_search_group_by_name( %d, %s, %s, %p, %p ) BEGIN\n",loc_id,name,groupsearchmode2str(groupsearchmode),searchstrings,foundstrings);

  if (foundstrings) {
    SXLinkSearch_data link_data;
    link_data.searchstrings = searchstrings;
    link_data.foundstrings = foundstrings;
    link_data.groupsearchmode = groupsearchmode;

    // herr_t H5Literate_by_name( hid_t loc_id, const char *name, H5_index_t index_type, H5_iter_order_t order, hsize_t *idx, H5L_iterate_t op, void *op_data, hid_t lapl_id ) 
    // index_type=H5_INDEX_NAME, by name
    // order=H5_ITER_INC, incremental
    hsize_t idx=0;
    h5status=H5Literate_by_name( loc_id, name, H5_INDEX_NAME, H5_ITER_INC, &idx, sx_group_iteration, (void *) &link_data, H5P_DEFAULT);
    if (h5status<0) goto sx_search_group_by_name_error;
  }

  if (Bsl2HdfDebug&BSL2HDF_DEBUG1) printf("sx_search_group_by_name( %d, %s, %s, %p, %p ) (h5status=%d) END\n",loc_id,name,groupsearchmode2str(groupsearchmode),searchstrings,foundstrings,h5status);

  return(h5status);

sx_search_group_by_name_error:

  if (Bsl2HdfDebug&BSL2HDF_DEBUG1) printf("sx_search_group_by_name( %d, %s, %s, %p, %p ) (h5status=%d) ERROR\n",loc_id,name,groupsearchmode2str(groupsearchmode),searchstrings,foundstrings,h5status);
  return(-1);

} // sx_search_group_by_name

/*---------------------------------------------------------------------------
main
---------------------------------------------------------------------------*/

#if MAKE_FUNCTION
# define MAIN main_bsl2hdf
#else
# define MAIN main
#endif

# define NULL_SPACE( s) ( ( s)? ( s) : ("" ) )

int MAIN (int argc, char *argv[])
{
  int errval=0;
  int status=0;
  int argcnt=1;
  int debug = 2;
  int level = 1;

  int omode = FileOpenAny;

  char *pstring[2];

  // output file file name
  const char *oname="output.h5";
  hid_t  fid=-1;          /* Handles */
  herr_t h5status=-1;

  // hdf file
  hsize_t     dims[RANK], maxdims[RANK];
  long        frame_no, no_of_frames=0, chain_no, no_of_chains=0;

  hid_t sxentry_gid=-1, sxseries_gid=-1, sxmemory_gid=-1; 
  hid_t sxdatablock_gid=-1, sxerrorblock_gid=-1, sxheader_gid=-1;
  htri_t sxentry_exists=-1, sxseries_exists=-1, sxmemory_exists=-1;
  htri_t sxdatablock_exists=-1, sxerrorblock_exists=-1, sxheader_exists=-1;
  hid_t sxdataset=-1;
  htri_t sxdataset_exists=-1;
  long entry_number=1;
  long series_number=1;

  DictOfDSets *file_attributes=NULL;

  DictOfDSets *entry_attributes=NULL;
  DictOfDSets *entry_number_attributes=NULL;
  DictOfDSets *series_attributes=NULL;
  DictOfDSets *series_number_attributes=NULL;
  DictOfDSets *memory_attributes=NULL;
  DictOfDSets *memory_number_attributes=NULL;
  DictOfDSets *datablock_attributes=NULL;
  DictOfDSets *errorblock_attributes=NULL;
  DictOfDSets *dataset_attributes=NULL;

  DictOfDSets *entries=NULL;
  DictOfDSets *entry_search_attributes=NULL;
  DictOfDSets *series=NULL;
  DictOfDSets *series_search_attributes=NULL;
  DictOfDSets *memories=NULL;
  DictOfDSets *memory_search_attributes=NULL;
  DictOfDSets *datablocks=NULL;
  DictOfDSets *datablock_search_attributes=NULL;
  DictOfDSets *errorblocks=NULL;
  DictOfDSets *errorblock_search_attributes=NULL;

  DictOfDSets *memories_withblocks=NULL;

  char *fname="file.edf";

  char entry_name[BSL2HDF_BUFLEN];
  char entry_number_string[BSL2HDF_BUFLEN];  
  char series_name[BSL2HDF_BUFLEN];
  char series_number_string[BSL2HDF_BUFLEN];  
  char memory_name[BSL2HDF_BUFLEN];
  char memory_number_string[BSL2HDF_BUFLEN];  

  int variable_header_values=0;

  long ChainNumber=1;
  long DataNumber=1;
  const long *DataNumbers=NULL, *ChainNumbers=NULL;
  int ErrorValue;

  int in1=-1;
  long  *Dim1=NULL;
  float *Array1 = NULL;
  size_t Size1;
  float IDummy1=-999.0, EDummy1=-1; // VarDummy in ipol.h


  if (argc<=1) {
    printf("%s\n",bsl2hdf_usage2str());
    return(1);
  }

  // skip all options
  while (argc > argcnt) {
    if ( strncmp( argv[argcnt],"debug=", 6 ) == 0 ) {
      // sscanf(&argv[argcnt][6],"%d",&debug);
      // long num_str2long ( const char *str, const char **tail, int *perrval);
      debug = (int) num_str2double( &argv[argcnt][6], NULL, NULL);
      bsl2hdf_debug_set( debug );
      argcnt++;
    } else if ( strncmp( argv[argcnt],"level=", 6 ) == 0 ) {
      sscanf(&argv[argcnt][6],"%d",&level);
      bsl2hdf_level_set( level );
      argcnt++;
    } else if ( strncmp( argv[argcnt],"--version", 9 ) == 0 ) {
      printf("\n %s version: %s\n\n", argv[0], BSL2HDF_VERSION);
      status=1;
      argcnt++;
      break;
    } else if ( strncmp( argv[argcnt],"oname=", 6 ) == 0 ) {
      oname=argv[argcnt]+6;
      argcnt++;
    } else if ( strncmp( argv[argcnt],"omode=", 6 ) == 0 ) {
      omode=str2fopenmode ( argv[argcnt]+6 );
      argcnt++;
    } else if ( strncmp( argv[argcnt],"series=", 7 ) == 0 ) {
      series_number = (long) num_str2double( &argv[argcnt][7], NULL, NULL);
      argcnt++;
    } else if ( strncmp( argv[argcnt],"entry=", 6 ) == 0 ) {
      entry_number = (long) num_str2double( &argv[argcnt][6], NULL, NULL);
      argcnt++;
    } else if ( strncmp( argv[argcnt],"--help", 6 ) == 0 ) {
      printf("\n %s version: %s\n", argv[0], BSL2HDF_VERSION);
      printf("%s\n",bsl2hdf_usage2str());
      status=1;

      argcnt++;
      break;
    } else break;
  }

  edfio_debug_set(bsl2hdf_level());

  entry_search_attributes = bsl2hdf_new_dictofdsets();
  bsl2hdf_append_dset( entry_search_attributes, SXCLASS, SXCLASS_ENTRY, 0, 1, -1 );
  series_search_attributes = bsl2hdf_new_dictofdsets();
  bsl2hdf_append_dset( series_search_attributes, SXCLASS, SXCLASS_SERIES, 0, 1, -1 );
  memory_search_attributes = bsl2hdf_new_dictofdsets();
  bsl2hdf_append_dset( memory_search_attributes, SXCLASS, SXCLASS_MEMORY, 0, 1, -1 );
  datablock_search_attributes = bsl2hdf_new_dictofdsets();
  bsl2hdf_append_dset( datablock_search_attributes, SXCLASS, SXCLASS_DATABLOCK, 0, 1, -1 );
  errorblock_search_attributes = bsl2hdf_new_dictofdsets();
  bsl2hdf_append_dset( errorblock_search_attributes, SXCLASS, SXCLASS_ERRORBLOCK, 0, 1, -1 );

  pstring[1] = (char *) NULL;
  if (!status) {
    if (argc>argcnt) {
      int actual_omode;

      variable_header_values=0;

      edfio_debug_set(bsl2hdf_level());

      if (omode==FileOpenAny) {
        /*
         * Open old file if it exists, otherwise create new file
         */
        if ( filename_exists ( oname ) ) actual_omode=FileOpenOld;
        else actual_omode=FileOpenNew;
      } else actual_omode=omode;

      if (actual_omode==FileOpenNew) {

        /*
         * Create a new hdf5 file using the default properties.
         */
        file_attributes=bsl2hdf_new_dictofdsets();
        bsl2hdf_append_dset( file_attributes, SXCLASS, SXCLASS_FILE, 0, 1, -1 );

        if (Bsl2HdfDebug&BSL2HDF_VERBOSE) printf("sx_file_create( %s , ... )\n",oname);
        fid = sx_file_create( oname, file_attributes );
        if (fid<0) goto main_error;

      } else if (actual_omode==FileOpenOld) {

        /*
         * Open an existing hdf5 file using the default properties.
         */
        // get file_attributes

        file_attributes=bsl2hdf_new_dictofdsets();

        if (Bsl2HdfDebug&BSL2HDF_VERBOSE) printf("sx_file_open( %s , ... )\n",oname);
        fid = sx_file_open( oname, file_attributes );
        if (fid<0) goto main_error;

        // search existing entries
        entries = bsl2hdf_new_dictofdsets();

        h5status=sx_search_group_by_name( fid, "/", EntrySearch, entry_search_attributes, entries );
        if (h5status<0) goto main_error;

        bsl2hdf_print_dictofdsets( stdout, entries, "/" ); //++++++++++++

        // in each entry look for existing SXseries

        Dset *entry=NULL;

        const char *entry_order, *entry_path;
        entry = bsl2hdf_get_first_dset(entries,&entry_order,&entry_path, NULL);

        while(entry) {

          // in each entry look for existing SXseries
          series = bsl2hdf_new_dictofdsets();

          h5status=sx_search_group_by_name( fid,  entry_path, SeriesSearch, series_search_attributes, series );
          if (h5status<0) goto main_error;

         //++++++++++++++++++ 
         // Needed: A function that determines as a function of memory number and entry number
         // the full path to the data block.
         // A positive memory number specifies a data block, 
         // a negative memory number specifies an error block.
         // The memory number zero specifies the general (default) block (optional).
         //++++++++++++++++++ 

          bsl2hdf_print_dictofdsets( stdout, series, entry_path );  //++++++++++++++++++++
        
          Dset *sery=NULL;
          const char *sery_order, *sery_path;
          sery = bsl2hdf_get_first_dset(series,&sery_order,&sery_path, NULL);
          while(sery) {

            // in each series look for existing SXmemory

            memories = bsl2hdf_new_dictofdsets();
            memories_withblocks = bsl2hdf_new_dictofdsets();

            h5status=sx_search_group_by_name( fid,  sery_path, MemorySearch, memory_search_attributes, memories );
            if (h5status<0) goto main_error;

            // bsl2hdf_print_dictofdsets( stdout, memories, sery_path ); //++++++++++++

            Dset *memory=NULL;
            const char *memory_order, *memory_path;
            memory = bsl2hdf_get_first_dset(memories,&memory_order,&memory_path, NULL);
            while(memory) {

              datablocks = bsl2hdf_new_dictofdsets();
              errorblocks = bsl2hdf_new_dictofdsets();

              h5status=sx_search_group_by_name( fid,  memory_path, DataBlockSearch, datablock_search_attributes, datablocks );
              if (h5status<0) goto main_error;

              // bsl2hdf_print_dictofdsets( stdout, datablocks, memory_path ); //++++++++++++

              h5status=sx_search_group_by_name( fid,  memory_path, ErrorBlockSearch, errorblock_search_attributes, errorblocks );
              if (h5status<0) goto main_error;

              // bsl2hdf_print_dictofdsets( stdout, errorblocks, memory_path ); //++++++++++++

              // update memories_withblocks
              Dset *datablock=NULL, *errorblock=NULL;
              const char *datablock_path, *errorblock_path;

              datablock = bsl2hdf_get_first_dset(datablocks,NULL,&datablock_path, NULL);
              if (datablock_path)
                bsl2hdf_append_dset( memories_withblocks, memory_order, datablock_path, 0, 1, -1 );

              errorblock = bsl2hdf_get_first_dset(errorblocks,NULL,&errorblock_path, NULL);
              if (errorblock_path) {
                char memory_errorblock_order[BSL2HDF_BUFLEN]="";
                strlib_concat( memory_errorblock_order, BSL2HDF_BUFLEN, "-", memory_order );
                bsl2hdf_append_dset( memories_withblocks, memory_errorblock_order, errorblock_path, 0, 1, -1 );
              }

              datablocks = bsl2hdf_free_dictofdsets(datablocks);
              errorblocks = bsl2hdf_free_dictofdsets(errorblocks);

              memory = bsl2hdf_get_next_dset(memory, &memory_order, &memory_path, NULL);
            }

            bsl2hdf_print_dictofdsets( stdout, memories_withblocks, sery_order ); //++++++++++++

            memories=bsl2hdf_free_dictofdsets(memories);
            memories_withblocks=bsl2hdf_free_dictofdsets(memories_withblocks);
       
            sery = bsl2hdf_get_next_dset(sery, &sery_order, &sery_path, NULL);
          } // while sery

          series = bsl2hdf_free_dictofdsets(series);

          entry = bsl2hdf_get_next_dset(entry, &entry_order, &entry_path, NULL);

        } // while entry

        entries = bsl2hdf_free_dictofdsets(entries);

      } else goto main_error;



//++++++++++++++++++ 

      /*
       * hid_t sx_group_create( hid_t loc_id, const char *name, DictOfDSets *attributes )
       * Create HDF5 group key_%d:classname in loc_id
       */
      entry_attributes=bsl2hdf_new_dictofdsets();
      bsl2hdf_append_dset( entry_attributes, SXCLASS, SXCLASS_ENTRY, 0, 1, -1 );
      snprintf(entry_name,BSL2HDF_BUFLEN,"%s_%d",ENTRYNAME,entry_number);

      sxentry_exists=sx_link_exists( fid, entry_name );
      if (sxentry_exists<0) goto main_error;

      if (sxentry_exists) {

        sxentry_gid = sx_group_open( fid, entry_name, entry_attributes );
        if (sxentry_gid<0) goto main_error;

      } else {

        sxentry_gid = sx_group_create( fid, entry_name, entry_attributes );
        if (sxentry_gid<0) goto main_error;

        entry_number_attributes=bsl2hdf_new_dictofdsets();
        bsl2hdf_append_dset( entry_number_attributes, SXCLASS, SXCLASS_GROUPNUMBER, 0, 1, -1 );

        num_long2str ( entry_number_string, BSL2HDF_BUFLEN, entry_number++, NULL);
        if (sx_stringds_write(sxentry_gid, GROUPNUMBERNAME, entry_number_string, entry_number_attributes)<0) goto main_error;
      }

      while (argc>argcnt) {
        pstring[0] = argv[argcnt++];
        if ( fname=pstring[0] ) {

          /* Open the edf file */
          if (Bsl2HdfDebug&BSL2HDF_VERBOSE) printf("edf_open_data_file( %s , ... )\n",fname);
          in1 = edf_open_data_file  ( fname, "read", &ErrorValue, &status );
          if (status) { printf("%s\n",edf_report_data_error(ErrorValue)); goto main_error; }

            /*
             * hid_t sx_group_create( hid_t loc_id, const char *name, DictOfDSets *attributes )
             * Create HDF5 group key_%d:classname in loc_id
             */
            series_attributes=bsl2hdf_new_dictofdsets();
            bsl2hdf_append_dset( series_attributes, SXCLASS, SXCLASS_SERIES, 0, 1, -1 );
            snprintf(series_name,BSL2HDF_BUFLEN,"%s_%d",SERIESNAME,series_number);

            sxseries_exists=sx_link_exists( sxentry_gid, series_name );
            if (sxseries_exists<0) goto main_error;

            if (sxseries_exists) {

              sxseries_gid = sx_group_open( sxentry_gid, series_name, series_attributes );
              if (sxseries_gid<0) goto main_error;

            } else {

              sxseries_gid = sx_group_create( sxentry_gid, series_name, series_attributes );
              if (sxseries_gid<0) goto main_error;

              series_number_attributes=bsl2hdf_new_dictofdsets();
              bsl2hdf_append_dset( series_number_attributes, SXCLASS, SXCLASS_GROUPNUMBER, 0, 1, -1 );

              num_long2str ( series_number_string, BSL2HDF_BUFLEN, series_number++, NULL);
              if (sx_stringds_write(sxseries_gid, GROUPNUMBERNAME, series_number_string, series_number_attributes)<0) goto main_error;
            }

            ChainNumbers = edf_chain_numbers ( in1, &ErrorValue, &status);
            if (status) {
              printf("%s\n",edf_report_data_error(ErrorValue));
              goto main_error;
            }

            no_of_chains = ChainNumbers[0];

            for (chain_no=1;chain_no<=no_of_chains;chain_no++) {

              ChainNumber = ChainNumbers[chain_no];

              DataNumbers = edf_block_numbers ( in1, ChainNumber, &ErrorValue, &status);
              if (status) {
                printf("%s\n",edf_report_data_error(ErrorValue));
                goto main_error;
              }

              // write images ______________________________________________________
              no_of_frames = DataNumbers[0];

              memory_attributes=bsl2hdf_new_dictofdsets();

              // use only positive data chains, negative are errors and are used when necessary
              if ( ( ChainNumber >= 1) && ( no_of_frames >= 1 ) ) {

                snprintf(memory_name,BSL2HDF_BUFLEN,"%s_%d",MEMORYNAME,ChainNumber);

                sxmemory_exists=sx_link_exists( sxseries_gid, memory_name );
                if (sxmemory_exists<0) goto main_error;

                if (sxmemory_exists) {

                  sxmemory_gid = sx_group_open( sxseries_gid, memory_name, memory_attributes );
                  if (sxmemory_gid<0) goto main_error;

                } else {

                  /*
                   * hid_t sx_group_create( hid_t loc_id, const char *name, DictOfDSets *attributes )
                   * Create HDF5 group key_%d:classname in loc_id
                   */
                  bsl2hdf_append_dset( memory_attributes, SXCLASS, SXCLASS_MEMORY, 0, 1, -1 );

                  sxmemory_gid = sx_group_create( sxseries_gid, memory_name, memory_attributes );
                  if (sxmemory_gid<0) goto main_error;

                  /*
                   * create data blocks (SX_class SXdata) and error blocks (SX_class SXerror)
                   * group: <memory> -> /<entry>/<series>/<memory> (SX_class SXmemory)
                   *   dataset: <group_number> -> /<entry>/<series>/<memory>/<group_number> (SX_class SXgroupnumber)
                   *   [group: <datablock> -> /<entry>/<series>/<memory>/<datablock> (SX_class SXdata)
                   *     dataset:array [dim1, dim2, 1..Imax] chunking maxdim1, maxdim2, infinity (SX_class SXarray)
                   *     group:header (SX_class SXheader)
                   *       dataset:<symbols>[1..Imax] chunking infinit (1 element for each image number or 1 for all)]
                   *   [group: <errorblock> -> /<entry>/<series>/<memory>/<errorblock> (SX_class SXerror)
                   *     dataset:array [dim1, dim2, 1..Imax] chunking maxdim1, maxdim2, infinity (SX_class SXarray)
                   *     group:header (SX_class SXheader)
                   *       dataset:<symbols>[1..Imax] chunking infinit (1 element for each image number or 1 for all)]
                   * {group: <nextmemory> -> /<entry>/<series>/<nextmemory> (SX_class SXmemory)}
                   */

                  memory_number_attributes=bsl2hdf_new_dictofdsets();
                  bsl2hdf_append_dset( memory_number_attributes, SXCLASS, SXCLASS_GROUPNUMBER, 0, 1, -1 );

                  num_long2str ( memory_number_string, BSL2HDF_BUFLEN, ChainNumber, NULL);
                  if (sx_stringds_write(sxmemory_gid, GROUPNUMBERNAME, memory_number_string, memory_number_attributes)<0) goto main_error;

                }

//++++++++++++++++++++++++++++++++++++++ DATABLOCKNAME BEGIN

                sxdataset=-1;

                datablock_attributes=bsl2hdf_new_dictofdsets();
                bsl2hdf_append_dset( datablock_attributes, SXCLASS, SXCLASS_DATABLOCK, 0, 1, -1 );

                for (frame_no=1;frame_no<=no_of_frames;frame_no++) {

                  DataNumber = DataNumbers[frame_no];

                  if (frame_no == 1) { // create image header
                    edf_new_header          ( DATABLOCKNAME );

                   sxdatablock_exists = sx_link_exists( sxmemory_gid, DATABLOCKNAME );
                   if (sxdatablock_exists<0) goto main_error;

                   if (sxdatablock_exists) {
                     /*
                      * hid_t sx_group_open( hid_t loc_id, const char *name,  DictOfDSets *dictofstringds )
                      * Open HDF5 group name in loc_id
                      */
                      sxdatablock_gid = sx_group_open( sxmemory_gid, DATABLOCKNAME,  datablock_attributes );
                   } else {
                     /*
                      * hid_t sx_group_create( hid_t loc_id, const char *name, DictOfDSets *attributes )
                      * Create HDF5 group name in loc_id
                      */
                      sxdatablock_gid = sx_group_create( sxmemory_gid, DATABLOCKNAME, datablock_attributes );
			      if (sxdatablock_gid<0) goto main_error;
                    }

                  }

                  /* Read the edf file as float (MFloat) */
                  if (Bsl2HdfDebug&BSL2HDF_VERBOSE) printf("edf_read_data( %d , %d ... )\n",in1,DataNumber);
                  Array1=NULL; Dim1=NULL; // otherwise the released ones will be used
                  edf_read_data (in1, DataNumber, ChainNumber, &Dim1, &Size1,
                                 (void**) &Array1, MFloat, &ErrorValue, &status );
                  if (status) { printf("%s\n",edf_report_data_error(ErrorValue)); goto main_error; }

                  // read full edf header
                  edf_read_header         ( in1, DataNumber, ChainNumber, DATABLOCKNAME,
                                        &ErrorValue, &status );
                  if (status) { printf("%s\n",edf_report_data_error(ErrorValue)); goto main_error; }

                  // Use dummy value of first frame as a more adequate h5 fill value 
                  { 
                    const char *dummyvalue=NULL;
                    edf_search_header_element(DATABLOCKNAME, "Dummy", &dummyvalue, &ErrorValue, &status);
                    if (dummyvalue) {
                      IDummy1 = ( float ) num_str2double( dummyvalue, NULL, &errval );
                    }
                  }
  
                  dims[0] = 1;  // dims[0] must be 1
                  dims[1] = Dim1[2];
                  dims[2] = Dim1[1];
  
                  if (frame_no == 1) { // create dataset
                    maxdims[0] = H5S_UNLIMITED; // unlimited
                    maxdims[1] = dims[1]; // should be the largest dim in the cycle, or given as option
                    maxdims[2] = dims[2]; // should be the largest dim in the cycle, or given as option

                    if (Bsl2HdfDebug&BSL2HDF_VERBOSE) printf("sx_dataset_create(...,%s,...)\n",SXCLASS_ARRAY);
                    // hid_t sx_dataset_create(hid_t loc_id, const char *name, hsize_t rank, 
                    //   hsize_t maxdims[], hid_t datatype_id, hid_t filltype_id, const void *fillvalue )

                    dataset_attributes=bsl2hdf_new_dictofdsets();
                    bsl2hdf_append_dset( dataset_attributes, SXCLASS, SXCLASS_ARRAY, 0, 1, -1 );

                    sxdataset_exists=sx_link_exists( sxdatablock_gid, ARRAYNAME );
                    if (sxdataset_exists<0) goto main_error;

                    if (sxdataset_exists) {
                      printf("ERROR: Cannot overwrite existing dataset %s\n",ARRAYNAME);
                      goto main_error;
                    }

                    sxdataset = sx_dataset_create(sxdatablock_gid, ARRAYNAME, RANK, maxdims, H5T_IEEE_F32LE, 
                                                 H5T_NATIVE_FLOAT, &IDummy1, dataset_attributes );
                    if (sxdataset<0) goto main_error;

                  } // if frame_no == 1

                  // write data array
                  if (Bsl2HdfDebug&BSL2HDF_VERBOSE) printf("sx_dataset_write(...,%d,...)\n",frame_no);
                  // herr_t sx_dataset_write(hid_t dataset, long frame_no, hid_t mem_type_id,
                  //                             hsize_t rank, hsize_t maxdims[], hsize_t dims[], void *data)
                  h5status = sx_dataset_write(sxdataset, frame_no, H5T_NATIVE_FLOAT,
                                                  RANK, maxdims, dims, Array1 );
                  if (h5status<0) goto main_error;

/*+++++++++++++++ old position of edf_read_header
                  // read full edf header
                  edf_read_header         ( in1, DataNumber, ChainNumber, DATABLOCKNAME,
                                        &ErrorValue, &status );
                  if (status) { printf("%s\n",edf_report_data_error(ErrorValue)); goto main_error; }
++++++++++++++++++++ */

                } // for DataNumber
                datablock_attributes=bsl2hdf_free_dictofdsets(datablock_attributes);

                /*
                 * Close dataset.
                 */
                h5status = sx_dataset_close(sxdataset);
                sxdataset=-1;

                dataset_attributes=bsl2hdf_free_dictofdsets(dataset_attributes);

                /*
                 * write sx-parameters of images
                 */

                h5status = sx_header_write(sxdatablock_gid, HEADERNAME,
                             in1, ChainNumber, DataNumbers, DATABLOCKNAME,
                             &ErrorValue, &status);
                if (status) { printf("%s\n",edf_report_data_error(ErrorValue)); goto main_error; }

                edf_free_header(DATABLOCKNAME);

                sx_group_close(sxdatablock_gid);
                sxdatablock_gid=-1;


//+++++++++++++++++++------------------- DATABLOCKNAME END

//++++++++++++++++++++++++++++++++++++++ ERRORBLOCKNAME BEGIN

                // write errors (variances) ________________________________________
                no_of_frames = DataNumbers[0];
                sxdataset=-1;

                errorblock_attributes=bsl2hdf_new_dictofdsets();
                bsl2hdf_append_dset( errorblock_attributes, SXCLASS, SXCLASS_ERRORBLOCK, 0, 1, -1 );

                for (frame_no=1;frame_no<=no_of_frames;frame_no++) {

                  DataNumber = DataNumbers[frame_no];

                  if (frame_no == 1) { // create image header
                    edf_new_header          ( ERRORBLOCKNAME );
                  }

                  // Does an error block exist?
                  if ( edf_test_header ( in1, DataNumber, -ChainNumber, NULL, NULL ) ) {

                    if (frame_no == 1) { // create image header

                     sxerrorblock_exists = sx_link_exists( sxmemory_gid, ERRORBLOCKNAME );
                     if (sxerrorblock_exists<0) goto main_error;

                     if (sxerrorblock_exists) {
                       /*
                        * hid_t sx_group_open( hid_t loc_id, const char *name,  DictOfDSets *dictofstringds )
                        * Open HDF5 group name in loc_id
                        */
                        sxerrorblock_gid = sx_group_open( sxmemory_gid, ERRORBLOCKNAME,  errorblock_attributes );
                     } else {
                       /*
                        * hid_t sx_group_create( hid_t loc_id, const char *name, DictOfDSets *attributes )
                        * Create HDF5 group key_%d:classname in loc_id
                        */
                        sxerrorblock_gid = sx_group_create( sxmemory_gid, ERRORBLOCKNAME, errorblock_attributes );
                        if (sxerrorblock_gid<0) goto main_error;
                      }
                    }

                    /* Read the edf file as float (MFloat) */
                    if (Bsl2HdfDebug&BSL2HDF_VERBOSE) printf("edf_read_data( %d , %d ... )\n",in1,DataNumber);
                    Array1=NULL; Dim1=NULL; // otherwise the released ones will be used
                    edf_read_data (in1, DataNumber, -ChainNumber, &Dim1, &Size1,
                                   (void**) &Array1, MFloat, &ErrorValue, &status );
                    if (status) { printf("%s\n",edf_report_data_error(ErrorValue)); goto main_error; }

                    // read full edf header
                    edf_read_header         ( in1, DataNumber, -ChainNumber, ERRORBLOCKNAME,
                                              &ErrorValue, &status );
                    if (status) { printf("%s\n",edf_report_data_error(ErrorValue)); goto main_error; }


                    dims[0] = 1;  // dims[0] must be 1
                    dims[1] = Dim1[2];
                    dims[2] = Dim1[1];

                    if (frame_no == 1) { // create dataset
                      maxdims[0] = H5S_UNLIMITED; // unlimited
                      maxdims[1] = dims[1]; // should be the largest dim in the cycle, or given as option
                      maxdims[2] = dims[2]; // should be the largest dim in the cycle, or given as option

                      dataset_attributes=bsl2hdf_new_dictofdsets();
                      bsl2hdf_append_dset( dataset_attributes, SXCLASS, SXCLASS_ARRAY, 0, 1, -1 );

                      sxdataset_exists=sx_link_exists( sxerrorblock_gid, ARRAYNAME );
                      if (sxdataset_exists<0) goto main_error;

                      if (sxdataset_exists) {
                        printf("ERROR: Cannot overwrite existing dataset %s\n",ARRAYNAME);
                        goto main_error;
                      }

                      if (Bsl2HdfDebug&BSL2HDF_VERBOSE) printf("sx_dataset_create(...,%s,...)\n",SXCLASS_ARRAY);
                      // hid_t sx_dataset_create(hid_t loc_id, const char *name, hsize_t rank,
                      //   hsize_t maxdims[], hid_t datatype_id, hid_t filltype_id, const void *fillvalue )
                      sxdataset = sx_dataset_create(sxerrorblock_gid, ARRAYNAME, RANK, maxdims, H5T_IEEE_F32LE,
                                                     H5T_NATIVE_FLOAT, &EDummy1, dataset_attributes );
                      if (sxdataset<0) goto main_error;
                    } // if frame_no == 1

                    // write error array
                    if (Bsl2HdfDebug&BSL2HDF_VERBOSE) printf("sx_dataset_write(...,%d,...)\n",frame_no);
                    // herr_t sx_dataset_write(hid_t dataset, long frame_no, hid_t mem_type_id,
                    //                             hsize_t rank, hsize_t maxdims[], hsize_t dims[], void *data)
                    h5status = sx_dataset_write(sxdataset, frame_no, H5T_NATIVE_FLOAT,
                                                      RANK, maxdims, dims, Array1 );
                    if (h5status<0) goto main_error;

/* +++++++++++++++++++ old position of edf_read_header
                    // read full edf header
                    edf_read_header         ( in1, DataNumber, -ChainNumber, ERRORBLOCKNAME,
                                              &ErrorValue, &status );
                    if (status) { printf("%s\n",edf_report_data_error(ErrorValue)); goto main_error; }
++++++++++++++++++ */

                  } // if edf_test_header

                } // for DataNumber
                errorblock_attributes=bsl2hdf_free_dictofdsets(errorblock_attributes);

                /*
                 * Close dataset.
                 */
                h5status = sx_dataset_close(sxdataset);
                sxdataset=-1;

                dataset_attributes=bsl2hdf_free_dictofdsets(dataset_attributes);

                /*
                 * write sx-parameters of errors (variances)
                 */

                h5status = sx_header_write(sxerrorblock_gid, HEADERNAME,
                             in1, -ChainNumber, DataNumbers, ERRORBLOCKNAME,
                             &ErrorValue, &status);
                if (status) { printf("%s\n",edf_report_data_error(ErrorValue)); goto main_error; }

                edf_free_header(ERRORBLOCKNAME);

                h5status = sx_group_close(sxerrorblock_gid);
                sxerrorblock_gid=-1;

//+++++++++++++++++++------------------- ERRORBLOCKNAME END

                /*
                 * Close data group
                 */
                // herr_t sx_group_close( hid_t gid );
                h5status = sx_group_close(sxmemory_gid);
                sxmemory_gid=-1;
                memory_number_attributes=bsl2hdf_free_dictofdsets(memory_number_attributes);
                memory_attributes=bsl2hdf_free_dictofdsets(memory_attributes);

              } // if (no_of_frames >= 1) ...

            } // for chain_no

            /*
             * Close series group
             */
            // herr_t sx_group_close( hid_t gid );
            h5status = sx_group_close(sxseries_gid);
            sxseries_gid=-1;
            series_number_attributes=bsl2hdf_free_dictofdsets(series_number_attributes);
            series_attributes=bsl2hdf_free_dictofdsets(series_attributes);

          /* Close edf data file */
          if (Bsl2HdfDebug&BSL2HDF_VERBOSE) printf("edf_close_data_file( %d , ... )\n",in1);
          edf_close_data_file( in1, &ErrorValue, &status );
          if (status) { printf("%s\n",edf_report_data_error(ErrorValue)); goto main_error; }
          in1=-1;

        }
      }

      /*
       * Close entry group
       */
      // herr_t sx_group_close( hid_t gid );
      h5status = sx_group_close(sxentry_gid);
      sxentry_gid=-1;
      entry_number_attributes=bsl2hdf_free_dictofdsets(entry_number_attributes);
      entry_attributes=bsl2hdf_free_dictofdsets(entry_attributes);

      memories_withblocks=bsl2hdf_free_dictofdsets(memories_withblocks);
      memories = bsl2hdf_free_dictofdsets(memories);
      entries = bsl2hdf_free_dictofdsets(entries);

      /*
       * Close hdf file.
       */
      if (Bsl2HdfDebug&BSL2HDF_VERBOSE) printf("sx_file_close( %d )\n",fid);
      sx_file_close( fid );
      fid=-1;

      if (Bsl2HdfDebug&BSL2HDF_VERBOSE) printf("bsl2hdf_free_dictofdsets( file_attributes )\n");
      file_attributes = bsl2hdf_free_dictofdsets(file_attributes);


    }
  }

  entry_search_attributes = bsl2hdf_free_dictofdsets(entry_search_attributes);
  series_search_attributes = bsl2hdf_free_dictofdsets(series_search_attributes);
  memory_search_attributes = bsl2hdf_free_dictofdsets(memory_search_attributes);
  datablock_search_attributes = bsl2hdf_free_dictofdsets(datablock_search_attributes);
  errorblock_search_attributes = bsl2hdf_free_dictofdsets(errorblock_search_attributes);

  return( status );

main_error:
  printf("Error exit\n");

  /*
   * Free dictionary of string datasets
   */
  bsl2hdf_free_dictofdsets(dataset_attributes);
  bsl2hdf_free_dictofdsets(errorblock_attributes);
  bsl2hdf_free_dictofdsets(datablock_attributes);
  bsl2hdf_free_dictofdsets(memory_number_attributes);
  bsl2hdf_free_dictofdsets(memory_attributes);
  bsl2hdf_free_dictofdsets(series_number_attributes);
  bsl2hdf_free_dictofdsets(series_attributes);
  bsl2hdf_free_dictofdsets(entry_number_attributes);
  bsl2hdf_free_dictofdsets(entry_attributes);
  bsl2hdf_free_dictofdsets(file_attributes);

  datablock_search_attributes=bsl2hdf_free_dictofdsets(datablock_search_attributes);
  datablocks = bsl2hdf_free_dictofdsets(datablocks);
  errorblock_search_attributes=bsl2hdf_free_dictofdsets(errorblock_search_attributes);
  errorblocks = bsl2hdf_free_dictofdsets(errorblocks);

  bsl2hdf_free_dictofdsets(memories_withblocks);
  bsl2hdf_free_dictofdsets(memory_search_attributes);
  bsl2hdf_free_dictofdsets(memories);
  bsl2hdf_free_dictofdsets(series_search_attributes);
  bsl2hdf_free_dictofdsets(series);
  bsl2hdf_free_dictofdsets(entry_search_attributes);
  bsl2hdf_free_dictofdsets(entries);

  /*
   * Close groups
   */
  sx_dataset_close(sxdataset);
  sx_header_close(sxheader_gid);
  sx_group_close(sxerrorblock_gid);
  sx_group_close(sxdatablock_gid);
  sx_group_close(sxmemory_gid);
  sx_group_close(sxseries_gid);
  sx_group_close(sxentry_gid);

  /*
   * Close edf headers
   */
  edf_free_header(ERRORBLOCKNAME);
  edf_free_header(DATABLOCKNAME);

  /*
   * Close hdf file.
   */
  sx_file_close( fid );

  /*
   * Close edf file.
   */

  edf_close_data_file( in1, NULL, NULL );
  return(status);

} // MAIN
