/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 * Copyright by The HDF Group.                                               * 
 * All rights reserved.                                                      *
 *                                                                           * 
 * This file is part of the HDF5 BITSHUFFLE filter plugin source.  The full  * 
 * copyright notice, including terms governing use, modification, and        * 
 * terms governing use, modification, and redistribution, is contained in    * 
 * the file COPYING, which can be found at the root of the BITSHUFFLE source * 
 * code distribution tree.  If you do not have access to this file, you may  * 
 * request a copy from help@hdfgroup.org.                                    * 
 * 2017-10-02 boesecke@esrf.fr movified for using static filter library      *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

/************************************************************

  This example shows how to write data and read it from a dataset
  using bitshuffle compression. 
  bitshuffle filter is not available in HDF5. 
  The example uses a new feature available in HDF5 version 1.8.11 
  to discover, load and register filters at run time.  

 ************************************************************/

#include "hdf5.h"
#include <stdio.h>
#include <stdlib.h>
#include "bshuf_h5filter.h"

/*
 * BSHUF_H5FILTER is defined in bshuf_h5filter.h
 " It can be defined for debugging when using the filter plugin
 *#define BSHUF_H5FILTER        32008
 */

#define FILE            "H5Z_filter_bitshuffle.h5"
#define DATASET         "data"
#define DIM0            32
#define DIM1            64
#define CHUNK0          4
#define CHUNK1          8

/************************************************************
Filter identifiers for the filters distributed with the HDF5 Library:
H5Z_FILTER_DEFLATE	The gzip compression, or deflation, filter
H5Z_FILTER_SZIP	The SZIP compression filter
H5Z_FILTER_NBIT	The N-bit compression filter
H5Z_FILTER_SCALEOFFSET  	The scale-offset compression filter
H5Z_FILTER_SHUFFLE	The shuffle algorithm filter
H5Z_FILTER_FLETCHER32	The Fletcher32 checksum, or error checking, filter 
************************************************************/

int main (void)
{
    hid_t           file_id = -1;    /* Handles */
    hid_t           space_id = -1;    /* Handles */
    hid_t           dset_id = -1;    /* Handles */
    hid_t           dcpl_id = -1;    /* Handles */
    herr_t          status;
    htri_t          avail;
    H5Z_filter_t    filter_id = 0;
    char            filter_name[80];
    hsize_t         dims[2] = {DIM0, DIM1},
                    chunk[2] = {CHUNK0, CHUNK1};
    unsigned int    flags;
    unsigned        filter_config;
    //const unsigned int cd_values[1] = {2};     /* bitshuffle default level is 2 */
    const unsigned int cd_values[2] = {0};     /* bitshuffle default level is 2 */
    size_t          cd_nelmts = 1;                /* number of elements in cd_values */
    //unsigned int       values_out[1] = {99};          
    unsigned int       values_out[2] = {99,99};          
    size_t          nelmts = 2;                /* number of elements in out_values */
    int             wdata[DIM0][DIM1],          /* Write buffer */
                    rdata[DIM0][DIM1],          /* Read buffer */
                    max;
    hsize_t         i, j;
    int             ret_value = 1, r;

    /* Register the filter with the library */
    printf ("Registering bitshuffle filter (not needed for plugin).\n");
    r = bshuf_register_h5filter();
    if (r < 0) printf ("failed to register.\n");

    /*
     * Initialize data.
     */
    for (i=0; i<DIM0; i++)
        for (j=0; j<DIM1; j++)
            wdata[i][j] = i * j - j;

    /*
     * Create a new file using the default properties.
     */
    file_id = H5Fcreate (FILE, H5F_ACC_TRUNC, H5P_DEFAULT, H5P_DEFAULT);
    if (file_id < 0) goto done;

    /*
     * Create dataspace.  Setting maximum size to NULL sets the maximum
     * size to be the current size.
     */
    space_id = H5Screate_simple (2, dims, NULL);
    if (space_id < 0) goto done;

    /*
     * Create the dataset creation property list, add the gzip
     * compression filter and set the chunk size.
     */
    dcpl_id = H5Pcreate (H5P_DATASET_CREATE);
    if (dcpl_id < 0) goto done;

    /* 
     * Check that filter is registered with the library now.
     */
    avail = H5Zfilter_avail(BSHUF_H5FILTER);
    if (avail)
        printf ("bitshuffle filter is available.\n");

    status = H5Pset_chunk (dcpl_id, 2, chunk);
    if (status < 0) printf ("failed to set chunk.\n");

    /* Use of the shuffle filter VASTLY improves performance of this
       and other block-oriented compression filters.  Be sure to add
       this before the compression filter!
    */
//+++++++++    status = H5Pset_shuffle(dcpl_id);
//+++++++++    if(status<0) printf ("failed to set shuffle.\n");


//    status = H5Pset_filter (dcpl_id, BSHUF_H5FILTER, H5Z_FLAG_MANDATORY, (size_t)1, cd_values);
    status = H5Pset_filter(dcpl_id, BSHUF_H5FILTER, H5Z_FLAG_MANDATORY, cd_nelmts, cd_values);
    if (status < 0) goto done;

    /*
     * Create the dataset.
     */
    printf ("....Writing bitshuffle compressed data ................\n");
    // unsigned int  
    
    dset_id = H5Dcreate (file_id, DATASET, H5T_STD_I32LE, space_id, H5P_DEFAULT, dcpl_id, H5P_DEFAULT);
    if (dset_id < 0) goto done;

    /*
     * Write the data to the dataset.
     */
    status = H5Dwrite (dset_id, H5T_NATIVE_INT, H5S_ALL, H5S_ALL, H5P_DEFAULT, wdata[0]);
    if (status < 0) printf ("failed to write data.\n");

    /*
     * Close and release resources.
     */
    H5Dclose (dset_id);
    dset_id = -1;
    H5Pclose (dcpl_id);
    dcpl_id = -1;
    H5Sclose (space_id);
    space_id = -1;
    H5Fclose (file_id);
    file_id = -1;
    status = H5close();
    if (status < 0) {
        printf ("/nFAILED to close library/n");
        goto done;
    }

    printf ("....Close the file and reopen for reading ........\n");
    /*
     * Now we begin the read section of this example.
     */

    /* 
     * Check that filter is registered with the library now.
     */
    avail = H5Zfilter_avail(BSHUF_H5FILTER);
    if (avail)
        printf ("bitshuffle filter is available.\n");
    else printf ("bitshuffle filter is not available.\n");


    /* Register the filter with the library, seems to be lost after closing */
    printf ("Registering bitshuffle filter (not needed for plugin).\n");
    r = bshuf_register_h5filter();
    if (r < 0) printf ("failed to register.\n");

    /*
     * Open file and dataset using the default properties.
     */
    file_id = H5Fopen (FILE, H5F_ACC_RDONLY, H5P_DEFAULT);
    if (file_id < 0) goto done;



    dset_id = H5Dopen (file_id, DATASET, H5P_DEFAULT);
    if (dset_id < 0) goto done;

    /*
     * Retrieve dataset creation property list.
     */
    dcpl_id = H5Dget_create_plist (dset_id);
    if (dcpl_id < 0) goto done;

    /*
     * Retrieve and print the filter id, compression level and filter's name for bitshuffle.
     */
    filter_id = H5Pget_filter (dcpl_id, (unsigned) 0, &flags, &nelmts, values_out, sizeof(filter_name), filter_name, NULL);
    printf ("Filter info is available from the dataset creation property \n ");
    printf ("  Filter identifier is ");
    size_t elem;
      switch (filter_id) {
        case BSHUF_H5FILTER:
          printf ("%d (BSHUF_H5FILTER)\n", filter_id);
          printf ("   Number of parameters is %d with the value ", nelmts);
          for (elem=0;elem<nelmts;elem++) {
            printf (" %u", values_out[elem]);
          }
          printf("\n");
          printf ("   To find more about the filter check %s\n", filter_name);
          break;
        case H5Z_FILTER_SHUFFLE:
          printf ("%d (H5Z_FILTER_SHUFFLE)\n", filter_id);
          printf ("   Number of parameters is %d with the value ", nelmts);
          for (elem=0;elem<nelmts;elem++) {
            printf (" %u", values_out[elem]);
          }
          printf("\n");
          printf ("   To find more about the filter check %s\n", filter_name); 
          break;
        default:
          printf ("not expected filter %d\n",filter_id);
          break;
      }
    
    /*
     * Read the data using the default properties.
     */
    printf ("....Reading bitshuffle compressed data ................\n");
    status = H5Dread (dset_id, H5T_NATIVE_INT, H5S_ALL, H5S_ALL, H5P_DEFAULT, rdata[0]);
    if (status < 0) printf ("failed to read data.\n");

    /*
     * Find the maximum value in the dataset, to verify that it was
     * read correctly.
     */
    max = rdata[0][0];
    for (i=0; i<DIM0; i++)
        for (j=0; j<DIM1; j++) {
            /*printf("%d \n", rdata[i][j]); */
            if (max < rdata[i][j])
                max = rdata[i][j];
        }
    /*
     * Print the maximum value.
     */
    printf ("Maximum value in %s is %d\n", DATASET, max);
    /* 
     * Check that filter is registered with the library now.
     */
    avail = H5Zfilter_avail(BSHUF_H5FILTER);
    if (avail)  
        printf ("bitshuffle filter is available.\n");
         
    ret_value = 0;

done:
    /*
     * Close and release resources.
     */
    if (dcpl_id >= 0) H5Pclose (dcpl_id);
    if (dset_id >= 0) H5Dclose (dset_id);
    if (space_id >= 0) H5Sclose (space_id);
    if (file_id >= 0) H5Fclose (file_id);

    return ret_value;
}
