/*
 *   Project: The SPD Image correction and azimuthal regrouping
 *                      http://forge.epn-campus.eu/projects/show/azimuthal
 *
 *   Copyright (C) 2005-2010 European Synchrotron Radiation Facility
 *                           Grenoble, France
 *
 *   Principal authors: P. Boesecke (boesecke@esrf.fr)
 *                      R. Wilcke (wilcke@esrf.fr)
 *
 *   This program is free software: you can redistribute it and/or modify
 *   it under the terms of the GNU Lesser General Public License as published
 *   by the Free Software Foundation, either version 3 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU Lesser General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   and the GNU Lesser General Public License  along with this program.
 *   If not, see <http://www.gnu.org/licenses/>.
 */

/*+++***********************************************************************
NAME

   h5sx.h

DESCRIPTION

   Include file for h5sx.c

HISTORY
    2017-01-10  PB V0.00 from h5io

***************************************************************************/

#ifndef _H5SX_
#  define _H5SX_

/***************************************************************************
* Includes                                                                 *
***************************************************************************/

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <errno.h>
#include <fcntl.h>

#include <hdf5.h>

#include "h5io.h"
#if LIBLZ4 == 1
#  include "bshuf_h5filter.h"
#endif // LIBLZ4

#ifndef WIN32
# include <unistd.h>
#endif

#include <ctype.h>

#ifdef sun      /* sun specific stuff */
#  include <sys/time.h>
#  include <floatingpoint.h>
#else
#  include <time.h>
#endif

/***************************************************************************
* General Definitions                                                      *
***************************************************************************/

#ifndef PRIVATE
# define PRIVATE      static /* used to declare variables of private type */
# define PUBLIC               /* used to declare variables of public type */
#endif

#define SXCLASS "SX_class"
#define SXCLASS_GROUPNUMBER "SXgroupnumber"
#define SXCLASS_BLOCKNUMBERING "SXblocknumbering"
#define SXCLASS_FILE   "SXfile"
#define SXCLASS_ENTRY   "SXentry"
#define SXCLASS_SERIES   "SXseries"
#define SXCLASS_MEMORY    "SXmemory"
#define SXCLASS_DATABLOCK  "SXdata"
#define SXCLASS_ERRORBLOCK "SXerror"
#define SXCLASS_ARRAY       "SXarray"
#define SXCLASS_HEADER      "SXheader"
#define SXCLASS_SYMBOL       "SXsymbol"

/*
 * Use the class names as default group and dataset names
 */
#define SYMBOLPREFIX ""
#define GROUPNUMBERNAME SXCLASS_GROUPNUMBER // "groupnumber"
#define BLOCKNUMBERINGNAME SXCLASS_BLOCKNUMBERING // "blocknumbering"
#define ENTRYNAME       SXCLASS_ENTRY       // "entry"
#define SERIESNAME      SXCLASS_SERIES      // "series"
#define MEMORYNAME      SXCLASS_MEMORY      // "memory"
#define DATABLOCKNAME   SXCLASS_DATABLOCK   // "datablock"
#define ERRORBLOCKNAME  SXCLASS_ERRORBLOCK  // "errorblock"
#define ARRAYNAME       SXCLASS_ARRAY       // "array"  // array data set name inside datablock/errorblock
#define HEADERNAME      SXCLASS_HEADER      // "header" // header data group name inside datablock/errorblock

#define SXTMP   "tmp"

/***************************************************************************
* Structure Definitions                                                    *
***************************************************************************/

PUBLIC typedef struct H5sxDset_ {
  char *key; // key
  unsigned long maxvallen; // maximum length of string (without terminating 0)
  unsigned long no_of_frames; // length of char *value array[]
  char *value; // buffer containing no_of_frames strings with maximum maxvallen
               // characters + 1 terminating zero
  hid_t dataset; // dataset variable
  struct H5sxDset_ *nextdset;
} H5sxDset;

PUBLIC typedef struct {
  long  ndsets;    // number of dsets in list
  H5sxDset *firstdset; // first dset of list
  H5sxDset *lastdset;  // last dset of list
} H5sxListOfDSets;

PUBLIC H5sxListOfDSets 
   *h5sx_new_listofdsets(H5sxListOfDSets *listofdsets),
   *h5sx_free_listofdsets(H5sxListOfDSets *listofdsets);

PUBLIC long 
    h5sx_length_listofdsets(H5sxListOfDSets *listofdsets);

PUBLIC H5sxDset 
    *h5sx_append_dset( H5sxListOfDSets *listofdsets,
                   const char *key, const char *value,
                   unsigned long maxvallen, unsigned long no_of_frames,
                   hid_t dataset ),
    *h5sx_get_first_dset(H5sxListOfDSets *listofdsets,const char **pkey,
                       const char **pvalue, hid_t *pdataset ),
    *h5sx_get_next_dset( H5sxDset *dset, const char **pkey,
                     const char **pvalue, hid_t *pdataset ),
    *h5sx_search_dset( H5sxListOfDSets *listofdsets,
                       const char *key, const char *value );


PUBLIC const char 
    *h5sxsearchlists( void ),
    *h5sx_write_str_dset(H5sxDset *dset, unsigned long frame_no,
                           const char *string);

PUBLIC void 
     h5sx_print_listofdsets(FILE *out, H5sxListOfDSets *listofdsets, 
                              const char *info, long indent);

/***************************************************************************
* Enum                                                                     *
***************************************************************************/

PUBLIC enum H5sxSearchMode { H5sxInValidSearchMode,
                             H5sxEntrySearch,
                             H5sxSeriesSearch,
                             H5sxMemorySearch,
                             H5sxDataBlockSearch,
                             H5sxErrorBlockSearch,
                             H5sxArraySearch,
                             H5sxHeaderSearch,
                             H5sxSymbolSearch,
                             H5sxGroupNumberSearch,
                             H5sxEndSearchMode };

/***************************************************************************
* Debug routines                                                           *
***************************************************************************/

/***************************************************************************
* Functions                                                                *
***************************************************************************/

PUBLIC extern const char
    *h5sx_version ( void );

PUBLIC extern int
     h5sx_open_stream ( const char *name, int mode, const char *filetype ),
     h5sx_close_stream ( int stream );

PUBLIC extern const char
     *h5sxsearchmode2str ( int h5sxsearchmode );

PUBLIC extern int
     h5sxstr2searchmode( const char *string );

PUBLIC extern long
     h5sx_highest_entrynumber( int stream, int *pstatus ),
     h5sx_highest_nodenumber( H5ioNode *node, int *pstatus );

PUBLIC extern H5ioData 
     *h5sx_read_array_value_native ( H5ioData *data, int mode ),
     *h5sx_read_symbol_value ( H5ioData *data ),
     *h5sx_empty_data_value ( H5ioData *data ),
     *h5sx_write_symbol ( H5ioNode *h5io_header, const char *Key, const char *Value, int *pstatus );

PUBLIC H5ioNode
     *h5sx_sync_image( int stream, int blocknumbering,
                       int DataType, int ByteOrder, int DataCompression, float Dummy,
                       long *MaxDim, long *DataDim,
                       long entry_number, long memory_number, long block_number );

PUBLIC extern herr_t 
     h5sx_flush_image( H5ioData *h5io_array, long *DataDim, void *Data ),
     h5sx_close_image ( H5ioData *h5io_array );

#endif
#ifndef _H5SX_
#  define _H5SX_
#endif
/***************************************************************************/
