/*
 *   Project: The SPD Image correction and azimuthal regrouping
 *                      http://forge.epn-campus.eu/projects/show/azimuthal
 *
 *   Copyright (C) 2005-2010 European Synchrotron Radiation Facility
 *                           Grenoble, France
 *
 *   Principal authors: P. Boesecke (boesecke@esrf.fr)
 *                      R. Wilcke (wilcke@esrf.fr)
 *
 *   This program is free software: you can redistribute it and/or modify
 *   it under the terms of the GNU Lesser General Public License as published
 *   by the Free Software Foundation, either version 3 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU Lesser General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   and the GNU Lesser General Public License  along with this program.
 *   If not, see <http://www.gnu.org/licenses/>.
 */

# define H5SX_VERSION      "h5sx : V0.993 Peter Boesecke 2018-02-08"
/*+++***********************************************************************
NAME

   h5sx.c

SYNOPSIS

DESCRIPTION

node type  : name filter (set of wildcards) : attribute filter (dictionary of attributes)

`SXCLASS_FILE` (default):

file       : None                           : {'SX_class':'SXfile'}
entry      : None                           : {'SX_class':'SXentry'}
series     : None                           : {'SX_class':'SXseries'}
memory     : None                           : {'SX_class':'SXmemory'}

datablock  : None                           : {'SX_class':'SXdata'}
errorblock : None                           : {'SX_class':'SXerror'}
array      : None                           : {'SX_class':'SXarray'}
header     : None                           : {'SX_class':'SXheader'}
symbols    : None                           : {'SX_class':'SXsymbol'}

other files:
"ID02PyFAI" (ID02 PyFAI output file),
"ID02PyFAIscaler" (ID02 PyFAI scaler output file),
"ID02LIMA" (ID02 LIMA raw data file).

HISTORY
    2017-03-03  PB V0.5
    2017-03-13  PB V0.6  search names and attributes are parameters of
                         locate functions, filetype
    2017-03-21  PB V0.7  inside NXdata look for dataset 'data' 
    2017-03-29  PB V0.8  h5sx_read_value_as_string: up to rank 2
                         warning for wrong scaler value dimension and
                         correction
                         h5sx_search_group_by_name: paths can contain
                         several paths separated by a colon.
    2017-03-31  PB V0.9  defaultsearchlist "default" 
                         opens current ID02LIMA and ID02PyFAI* files.
    2017-05-22  PB V0.91 FREE, MALLOC, CALLOC, REALLOC added
                         size_t is unsigned, initialization with -1
                         corrected to 0.
    2017-05-23  PB V0.92 strtok_r => strlib_tok_r
    2017-06-07  PB V0.93 debug option IODBG_SHOWNODESEARCH added.
    2017-06-28  PB V0.94 h5sx_open_stream: mode H5ioOpenNew added.
    2017-07-20  PB V0.95 h5sx_open_stream: create SXCLASS_FILE
                         all iodbg flags must include IODBG_H5
                         h5sx_sync_image: starting writing h5-files
                         h5sx_flush_image is currently missing
                         only the header is written, no data
    2017-08-28  PB V0.96 h5sx_dataset_open, size_t => hsize_t,
                         h5sx_close_image
    2017-09-04  PB V0.97 h5sx_dataset_create: deflate_level parameter added.
    2017-09-27  PB V0.98 h5sx_read_dataset_as_native_array: write debug
                         info about used dataset filter.
                         H5Dread segfault 
                         when reading data needing bitshuffle filter plugin
                         should be a low level problem
    2017-10-02  PB V0.99 bitshuffle filter statically linked and 
                         registered with bshuf_register_h5filter().
                         (added include "bshuf_h5filter.h" in h5sx.h)
    2018-01-10  PB V0.991 h5sx_open_stream: setting file->H5Path to "/"
                          h5sx_locate_symbols after h5sx_locate_arrays
                          Located arrays are excluded as symbols.
                          H5sxLinkSearch_data: foundstrings, vetostrings added
                          and included in h5sx_dataset_iteration.
                          h5sx_search_dataset_by_name splits datasets in
                          vetostring and foundstrings.
                          dataset attribute "interpretation":
                          "image":    return a table of images,
                          "spectrum": return a table of spectra,
                          "scalar":   return a table of scalars,
                          h5sx_new_array: write attribute interpretation=image
                          to dataset when creating an array.
    2018-02-07  PB V0.992 h5sxdefaultsearchlists: search lists added for
                          file types SLFF and SLMF.
    2018-02-08  PB V0.993 h5sxdefaultsearchlists: no default symbols 
                          for SLFF/SLMF

***************************************************************************/

/***************************************************************************
* Private part                                                             *
***************************************************************************/

#include "h5sx.h"

/***************************************************************************
* Defines                                                                  *
***************************************************************************/

#ifdef sun
# include <sys/param.h>
# define GETCWD(x,y) getwd (x)
#else
# ifdef WIN32
#  include <direct.h>
#  define GETCWD(x,y) _getcwd(x, y)
# else
#  define GETCWD(x,y) getcwd (x, y)
# endif
#endif

# define ISGRAPH(c) isgraph((int)(c))
# define ISSPACE(c) isspace((int)(c))
# define TOLOWER(s) (char)tolower((int)(s))
# define TOUPPER(s) (char)toupper((int)(s))

# define MAX( a, b) ((a)<(b)?(b):(a))
# define MIN( a, b) ((a)>(b)?(b):(a))

# define H5SX_BUFLEN (EdfMaxLinLen+1)
# define STRINGVSIZE (H5SX_BUFLEN+H5SX_BUFLEN)

# define H5SX_INFOBUFLEN 128

# define H5SXPRINT( R, L, B ) { long i; printf("%s: ",(L));for (i=0;i<(R);i++) printf("%ld ",(long)(B)[i]); printf("\n"); }

/*---------------------------------------------------------------------------
NAME

   h5sx_version --- returns the current version of h5sx

SYNOPSIS

   const char *h5sx_version ( void );

RETURN VALUE

const char *version string
---------------------------------------------------------------------------*/
const char *h5sx_version ( void )
{ return ( H5SX_VERSION );
} /* h5sx_version */

/***************************************************************************
* Strings for Enum H5sxSearchMode                                          *
***************************************************************************/
PUBLIC const char *H5sxSearchModeStrings[] = { "invalid",
                             "entrysearch",
                             "seriessearch",
                             "memorysearch",
                             "datablocksearch",
                             "errorblocksearch",
                             "arraysearch",
                             "headersearch",
                             "symbolsearch",
                             "groupnumbersearch",
                             (const char *) NULL };

/*---------------------------------------------------------------------------
NAME
  h5sxsearchmode2str ---  returns h5sx search mode as string

SYNOPSIS
  const char *h5sxsearchmode2str ( int h5sxsearchmode )
---------------------------------------------------------------------------*/
const char *h5sxsearchmode2str ( int h5sxsearchmode )
{ if ((h5sxsearchmode<0)||(h5sxsearchmode>=H5sxEndSearchMode))
    h5sxsearchmode = H5sxInValidSearchMode;
  return( H5sxSearchModeStrings[h5sxsearchmode] );
} /* h5sxsearchmode2str */

/*---------------------------------------------------------------------------
NAME
  h5sxstr2searchmode ---  converts string to h5sxsearchmode

SYNOPSIS
  int h5sxstr2searchmode ( const char * string )
---------------------------------------------------------------------------*/
int h5sxstr2searchmode( const char *string )
{ int  h5sxsearchmode=-1;

  while ( H5sxSearchModeStrings[++h5sxsearchmode] ) {
    if (!strlib_casecmp(H5sxSearchModeStrings[h5sxsearchmode], string)) break;
  }
  if (h5sxsearchmode>=H5sxEndSearchMode)
    h5sxsearchmode=H5sxInValidSearchMode;

  return(h5sxsearchmode);

} /* h5sxstr2searchmode */

/***************************************************************************
* Datatypes
***************************************************************************/
/* from edfio.h
enum DType { InValidDType,
             Unsigned8=1, Signed8,      Unsigned16,    Signed16,
             Unsigned32,  Signed32,     Unsigned64,    Signed64,
             FloatIEEE32, FloatIEEE64,  FloatIEEE128,  Unused12,
             EndDType };

enum MType { InValidMType,
             MUnsignedChar=1,    MChar,       MUnsignedShort, MShort,
             MUnsignedInteger,   MInteger,    MUnsignedLong,  MLong,
             MFloat,             MDouble,     MLongDouble,
             EndMType };
*/


/*---------------------------------------------------------------------------
NAME
  h5sxdtype2classstr ---  converts dtype_id to info string

SYNOPSIS
  const char *h5sxdtype2classstr( char *buffer, hsize_t buflen, hid_t dtype_id );

  enum H5T_class_t

    H5T_NO_CLASS         = -1,  //error
    H5T_INTEGER          = 0,   //integer types
    H5T_FLOAT            = 1,   //floating-point types
    H5T_TIME             = 2,   //date and time types
    H5T_STRING           = 3,   //character string types
    H5T_BITFIELD         = 4,   //bit field types
    H5T_OPAQUE           = 5,   //opaque types
    H5T_COMPOUND         = 6,   //compound types
    H5T_REFERENCE        = 7,   //reference types
    H5T_ENUM             = 8,   //enumeration types
    H5T_VLEN             = 9,   //Variable-Length types
    H5T_ARRAY            = 10,  //Array types

    H5T_NCLASSES                //this must be last

---------------------------------------------------------------------------*/
const char *h5sxdtype2classstr( char *buffer, hsize_t buflen, hid_t dtype_id )
{
  const char *value=NULL;

  if ((buffer)&&(buflen>0)) {
    H5T_class_t class_id=-1;
    long bits=-1, bytes=-1;

    hsize_t n=buflen-1;
    const char *bitsfx, *bytesfx;

    class_id = H5Tget_class(dtype_id);
    bits = (long) H5Tget_precision(dtype_id);
    bytes = (long) H5Tget_size(dtype_id);

    if (bits==1) bitsfx=""; else bitsfx="s";
    if (bytes==1) bytesfx=""; else bytesfx="s";

    switch (class_id) {
      case H5T_INTEGER: snprintf(buffer,n,"H5T_INTEGER: %ld byte%s (%ld bit%s)",bytes,bytesfx,bits,bitsfx); break;
      case H5T_FLOAT: snprintf(buffer,n,"H5T_FLOAT: %ld byte%s (%ld bit%s)",bytes,bytesfx,bits,bitsfx); break;
      case H5T_STRING: snprintf(buffer,n,"H5T_STRING: %ld byte%s (%ld bit%s)",bytes,bytesfx,bits,bitsfx); break;
      case H5T_BITFIELD: snprintf(buffer,n,"H5T_BITFIELD: %ld byte%s (%ld bit%s)",bytes,bytesfx,bits,bitsfx); break;
      case H5T_OPAQUE: snprintf(buffer,n,"H5T_OPAQUE: %ld byte%s (%ld bit%s)",bytes,bytesfx,bits,bitsfx); break;
      case H5T_COMPOUND: snprintf(buffer,n,"H5T_COMPOUND: %ld byte%s (%ld bit%s)",bytes,bytesfx,bits,bitsfx); break;
      case H5T_REFERENCE: snprintf(buffer,n,"H5T_REFERENCE: %ld byte%s (%ld bit%s)",bytes,bytesfx,bits,bitsfx); break;
      case H5T_ENUM: snprintf(buffer,n,"H5T_ENUM: %ld byte%s (%ld bit%s)",bytes,bytesfx,bits,bitsfx); break;
      case H5T_VLEN: snprintf(buffer,n,"H5T_VLEN: %ld byte%s (%ld bit%s)",bytes,bytesfx,bits,bitsfx); break;
      case H5T_ARRAY: snprintf(buffer,n,"H5T_ARRAY: %ld byte%s (%ld bit%s)",bytes,bytesfx,bits,bitsfx); break;
      default: snprintf(buffer,n,"(UNKNOWN): %ld byte%s (%ld bit%s)",bytes,bytesfx,bits,bitsfx); break;
    }
    buffer[n]='\0'; // force null termination
    value = buffer;
  }

  return( value );

} /* h5sxdtype2classstr */

/*---------------------------------------------------------------------------
NAME

  h5sxdtype2machinetype ---  converts dtype_id to edf machine type

SYNOPSIS

  int h5sxdtype2machinetype( hid_t dtype_id );

---------------------------------------------------------------------------*/
int h5sxdtype2machinetype( hid_t dtype_id )
{
  int mtype = InValidMType;

       if (H5Tequal( dtype_id, H5T_NATIVE_CHAR)>0   ) mtype = MChar;            // "char"
  else if (H5Tequal( dtype_id, H5T_NATIVE_SCHAR)>0  ) mtype = MChar;            // "signed char"
  else if (H5Tequal( dtype_id, H5T_NATIVE_UCHAR)>0  ) mtype = MUnsignedChar;    // "unsigned char"
  else if (H5Tequal( dtype_id, H5T_NATIVE_SHORT)>0  ) mtype = MShort;           // "short"; 
  else if (H5Tequal( dtype_id, H5T_NATIVE_USHORT)>0 ) mtype = MUnsignedShort;   // "unsigned short"; 
  else if (H5Tequal( dtype_id, H5T_NATIVE_INT)>0    ) mtype = MInteger;         // "int"; 
  else if (H5Tequal( dtype_id, H5T_NATIVE_UINT)>0   ) mtype = MUnsignedInteger; // "unsigned"; 
  else if (H5Tequal( dtype_id, H5T_NATIVE_LONG)>0   ) mtype = MLong;            // "long"; 
  else if (H5Tequal( dtype_id, H5T_NATIVE_ULONG)>0  ) mtype = MUnsignedLong;    // "unsigned long"; 
  else if (H5Tequal( dtype_id, H5T_NATIVE_LLONG)>0  ) mtype = InValidMType;     // "long long";
  else if (H5Tequal( dtype_id, H5T_NATIVE_ULLONG)>0 ) mtype = InValidMType;     // "unsigned long long";
  else if (H5Tequal( dtype_id, H5T_NATIVE_FLOAT)>0  ) mtype = MFloat;           // "float"; 
  else if (H5Tequal( dtype_id, H5T_NATIVE_DOUBLE)>0 ) mtype = MDouble;          // "double"; 
  else if (H5Tequal( dtype_id, H5T_NATIVE_LDOUBLE)>0) mtype = MLongDouble;      // "long double"; 
  else if (H5Tequal( dtype_id, H5T_NATIVE_HSIZE)>0  ) mtype = InValidMType;     // "hsize_t";
  else if (H5Tequal( dtype_id, H5T_NATIVE_HSSIZE)>0 ) mtype = InValidMType;     // "hssize_t";
  else if (H5Tequal( dtype_id, H5T_NATIVE_HERR)>0   ) mtype = InValidMType;     // "herr_t";
  else if (H5Tequal( dtype_id, H5T_NATIVE_HBOOL)>0  ) mtype = InValidMType;     // "hbool_t";
  else if (H5Tequal( dtype_id, H5T_NATIVE_B8)>0     ) mtype = edf_datatype2machinetype(Unsigned8);  // "8-bit";
  else if (H5Tequal( dtype_id, H5T_NATIVE_B16)>0    ) mtype = edf_datatype2machinetype(Unsigned16); // "16-bit";
  else if (H5Tequal( dtype_id, H5T_NATIVE_B32)>0    ) mtype = edf_datatype2machinetype(Unsigned32); // "32-bit";
  else if (H5Tequal( dtype_id, H5T_NATIVE_B64)>0    ) mtype = edf_datatype2machinetype(Unsigned64); // "64-bit";
  else mtype = InValidMType; // "(unknown)"

  return(mtype);

} /* h5sxdtype2machinetype */

/*---------------------------------------------------------------------------
NAME

  h5sxdtype2datatype ---  converts dtype_id to edf datatype

SYNOPSIS

  int h5sxdtype2datatype( hid_t dtype_id );

---------------------------------------------------------------------------*/
int h5sxdtype2datatype( hid_t dtype_id )
{
  return( edf_machinetype2datatype( h5sxdtype2machinetype( dtype_id ) ) );
} /* h5sxdtype2datatype */

/*---------------------------------------------------------------------------
NAME

  h5sxmachinetype2dtype ---  converts edf machine tpye to dtype_id

SYNOPSIS

  hid_t h5sxmachinetype2dtype( int machinetype );

DESCRIPTION

  The returned datatype identifier must not be modified. 
  Use H5Tcopy if necessary.

---------------------------------------------------------------------------*/
hid_t h5sxmachinetype2dtype( int machinetype, int byteorder )
{
  hid_t dtype_id = -1;

  if (byteorder == edf_byteorder()) {
    switch ( machinetype ) {
      case MUnsignedChar:    dtype_id = H5T_NATIVE_UCHAR; break;
      case MChar:            dtype_id = H5T_NATIVE_CHAR; break;
      case MUnsignedShort:   dtype_id = H5T_NATIVE_USHORT; break;
      case MShort:           dtype_id = H5T_NATIVE_SHORT; break;
      case MUnsignedInteger: dtype_id = H5T_NATIVE_UINT; break;
      case MInteger:         dtype_id = H5T_NATIVE_INT; break;
      case MUnsignedLong:    dtype_id = H5T_NATIVE_ULONG; break;
      case MLong:            dtype_id = H5T_NATIVE_LONG; break;
      case MFloat:           dtype_id = H5T_NATIVE_FLOAT; break;
      case MDouble:          dtype_id = H5T_NATIVE_DOUBLE; break;
      case MLongDouble:      dtype_id = H5T_NATIVE_LDOUBLE; break;
      // default: InValidMType
    }
  } else {
    printf("ERROR: h5sxmachinetype2dtype: cannot convert byteorder\n");
  }

  return( dtype_id );

} /* h5sxmachinetype2dtype */

/*---------------------------------------------------------------------------
NAME

  h5sxdatatype2dtype ---  converts edf datatype to dtype_id

SYNOPSIS

  hid_t h5sxdatatype2dtype( int datatype );

DESCRIPTION

  The returned datatype identifier must not be modified.
  Use H5Tcopy if necessary.

---------------------------------------------------------------------------*/
hid_t h5sxdatatype2dtype( int datatype, int byteorder )
{
  return( h5sxmachinetype2dtype( edf_datatype2machinetype( datatype ), byteorder ) );
} /* h5sxdatatype2dtype */


/*******************************************************************************
* Create, define, read and release a list of dataset variables (BEGIN)         *
*                                                                              *
* H5sxListOfDSets *h5sx_new_listofdsets( H5sxListOfDSets *listofdsets );       *
* H5sxListOfDSets *h5sx_move_listofdsets(H5sxListOfDSets *listofdsets);        *
* H5sxListOfDSets *h5sx_empty_listofdsets(H5sxListOfDSets *listofdsets);        *
* H5sxListOfDSets *h5sx_free_listofdsets(H5sxListOfDSets *listofdsets);        *
* long h5sx_length_listofdsets(H5sxListOfDSets *listofdsets);                  *
* H5sxDset *h5sx_append_dset( H5sxListOfDSets *listofdsets,                    *
*                    const char *key, const char *value,                       *
*                    unsigned long maxvallen, unsigned long no_of_frames,      *
*                    hid_t dataset );                                          *
* const char *h5sx_write_str_dset(H5sxDset *dset, unsigned long frame_no,      *
*                            const char *string);                              *
* H5sxDset *h5sx_get_first_dset(H5sxListOfDSets *listofdsets,const char **pkey,*
*                        const char **pvalue, hid_t *pdataset );               *
* H5sxDset *h5sx_get_next_dset( H5sxDset *dset, const char **pkey,             *
*                      const char **pvalue, hid_t *pdataset );                 *
* void h5sx_print_listofdsets(FILE *out, H5sxListOfDSets *listofdsets,         *
*                                        long indent);                         *
*******************************************************************************/

/*---------------------------------------------------------------------------
NAME

  h5sx_append_dset --- Writes a dset to the end of the list

PURPOSE

  Creates a new dset at the end of list. If listofdsets is NULL
  nothing happens.

ARGUMENTS

  H5sxListOfDSets * listofdsets
  const char *key
  const char *value
  unsigned long no_of_frames : length of all char *string arrays 
  hid_t dataset

RETURN VALUE

  Pointer to the new dset or NULL in case of error.

---------------------------------------------------------------------------*/
H5sxDset *h5sx_append_dset( H5sxListOfDSets *listofdsets, 
                   const char *key, const char *value,
                   unsigned long maxvallen, unsigned long no_of_frames,
                   hid_t dataset )
{ H5sxDset *new=NULL;

  if (listofdsets&&key) {
    H5sxDset * current;
    hsize_t size_dset;

    size_dset = sizeof ( H5sxDset );
    new = MALLOC ( size_dset ); 
    if (!new)
      goto h5sx_append_dset_error;

    new->key = strlib_newstr(key);
    new->maxvallen = maxvallen;
    new->no_of_frames = no_of_frames;
    if (value) new->value = strlib_newstr(value);
    else new->value = NULL;
    new->dataset = dataset;
    new->nextdset=NULL;

    current = listofdsets->lastdset;
    if (current) {
      current->nextdset = new;
    } else {
      listofdsets->firstdset = new;
    }
    listofdsets->lastdset = new;
    listofdsets->ndsets+=1l;
  }

  return(new);

h5sx_append_dset_error:

  FREE(new);

  return(NULL);

} // h5sx_append_dset

/*---------------------------------------------------------------------------
NAME

  h5sx_write_str_dset --- Copies string to element frame_no of dset

PURPOSE

  Copies contents of value to element frame_no of dset.
  If dset is NULL nothing happens.

RETURN VALUE

  Pointer to the copy or NULL in case of error.

---------------------------------------------------------------------------*/
const char *h5sx_write_str_dset(H5sxDset *dset, unsigned long frame_no, const char *string)
{
  char *new=NULL, *pn, *dest=NULL;
  unsigned long no_of_frames, maxvallen, frame;

  if (dset) {
    no_of_frames = dset->no_of_frames;
    maxvallen = dset->maxvallen;
    if (!(dset->value)) { // allocate pointers and initialize first char with 0 
      // char new[no_of_frames][maxvallen+1]
      if (!(new = (char*) MALLOC ( sizeof( char ) * (no_of_frames*(maxvallen+1)))))
        goto h5sx_write_str_dset_error;
      dset->value = new;
      pn=new;
      for (frame=1;frame<=no_of_frames;frame++) {
        *pn='\0'; // terminating 0
        pn+=maxvallen+1;
      }
    }
    dest=(dset->value)+(frame_no-1)*(maxvallen+1);
    if (string) 
      strncpy(dest,string,dset->maxvallen);
    dest[maxvallen]='\0'; // terminating 0
  }

  return(dest);

h5sx_write_str_dset_error:

  if (new) {
    FREE(new);
    if (dset) dset->value=NULL;
  }

  return(dest);

} // h5sx_write_str_dset

/*---------------------------------------------------------------------------
NAME

  h5sx_get_first_dset --- Returns pointer to the first dset

PURPOSE

  If listofdsets is not NULL goes to the first dset and 
  returns a pointer to it. In this case the dataset key, 
  the dataset value array and the dataset id are returned in
  **pkey, ***pvalue and *pdataset (if they are not NULL).

RETURN VALUE

  Pointer to the first dset or NULL

---------------------------------------------------------------------------*/
H5sxDset *h5sx_get_first_dset( H5sxListOfDSets *listofdsets, const char **pkey, 
                      const char **pvalue, hid_t *pdataset )
{ H5sxDset *dset=NULL;

  if (listofdsets) {
    dset = listofdsets->firstdset;
    if (dset) {
      if (pkey) *pkey = dset->key;
      if (pvalue) *pvalue = (const char *) dset->value;
      if (pdataset) *pdataset = dset->dataset;
    }

  }

  return(dset);

} // h5sx_get_first_dset

/*---------------------------------------------------------------------------
NAME

  h5sx_get_next_dset --- return the next dset

PURPOSE
  
  Goes to the next dset and returns a pointer to it.
  If the input dset is not NULL the dataset key,
  the dataset value array and the dataset id are returned in
  **pkey, **pvalue and *pdataset (if they are not NULL).

RETURN VALUE

  Pointer to the next dset or NULL

---------------------------------------------------------------------------*/
H5sxDset * h5sx_get_next_dset( H5sxDset * dset, const char **pkey, 
                      const char **pvalue, hid_t *pdataset )
{ H5sxDset * next=NULL;

  if (dset) {
      next = dset->nextdset;
      if (next) {
        if (pkey) *pkey = next->key;
        if (pvalue) *pvalue = (const char *) next->value;
        if (pdataset) *pdataset = next->dataset;
      }
  }

  return( next );

} // h5sx_get_next_dset

/*---------------------------------------------------------------------------
NAME

  h5sx_new_listofdsets( H5sxListOfDSets *listofdsets ) --- New list of dsets

PURPOSE

  Creates a new list of dsets and copies listofdsets in it.
  If listofdsets is NULL or empty an empty list of dsets is returned.
  The returned list must be released with h5sx_free_listofdsets
  for avoiding resource leaks.

RETURN VALUE

  Pointer to the new list of dsets or NULL in case of error.

---------------------------------------------------------------------------*/
H5sxListOfDSets *_h5sx_new_listofdsets( void ) {
  hsize_t size_listofdsets;
  H5sxListOfDSets *new=NULL;

  size_listofdsets = sizeof(H5sxListOfDSets);
  new = MALLOC(size_listofdsets);
  if (!new) goto _h5sx_new_listofdsets_error;

  // init list
  new->ndsets = 0l;
  new->firstdset = NULL;
  new->lastdset = NULL;

  return(new);

_h5sx_new_listofdsets_error:

  FREE(new);

  return(NULL);

} // _h5sx_new_listofdsets

H5sxListOfDSets *h5sx_new_listofdsets( H5sxListOfDSets *listofdsets )
{ H5sxListOfDSets *new=_h5sx_new_listofdsets();

  if (listofdsets) {
    H5sxDset *dset=listofdsets->firstdset;

    while (dset) {
      if (!h5sx_append_dset( new, dset->key, dset->value, dset->maxvallen, 
                             dset->no_of_frames, dset->dataset ))
        goto h5sx_new_listofdsets_error;
      dset=dset->nextdset;
    }
  }

  return(new);

h5sx_new_listofdsets_error:

  new=h5sx_free_listofdsets(new);

  return(NULL);

} // h5sx_new_listofdsets

/*---------------------------------------------------------------------------
NAME

  h5sx_move_listofdsets --- Moves a list of dsets to a new list

PURPOSE

  Creates a new output list and moves all items from
  the input list to the created list. The input list emptied.
  The returned list must be released with h5sx_free_listofdsets
  for avoiding resource leaks.

RETURN VALUE

  pointer to new list
  NULL in case of an error

---------------------------------------------------------------------------*/
H5sxListOfDSets *h5sx_move_listofdsets( H5sxListOfDSets *listofdsets )
{
  H5sxListOfDSets *new=_h5sx_new_listofdsets();

  if ((listofdsets&&new)) {
    // move content
    new->ndsets = listofdsets->ndsets;
    new->firstdset = listofdsets->firstdset;
    new->lastdset = listofdsets->lastdset;
    // reinitialize list
    listofdsets->ndsets = 0l;
    listofdsets->firstdset = NULL;
    listofdsets->lastdset = NULL;
  }

  return(new);

} // h5sx_move_listofdsets

/*---------------------------------------------------------------------------
NAME

  h5sx_empty_listofdsets --- Empties a list of dsets

PURPOSE

  Removes and releases the memories of all items of 
  listofdsets: const char *key, hid_t dataset, H5sxDset
  It returns an empty list.

RETURN VALUE

  pointer to listofdsets
  NULL in case of an error

---------------------------------------------------------------------------*/
H5sxListOfDSets *h5sx_empty_listofdsets( H5sxListOfDSets *listofdsets )
{

  if (listofdsets) {
    H5sxDset *current, *next;

    // release all dsets
    next = listofdsets->firstdset;
    while (next) {
      current = next;
      next = current->nextdset;
      FREE(current->key);

      // free dsets
      FREE(current->value);
      if (current->dataset>=0)
        H5Dclose(current->dataset);
      FREE(current);
    }
    // init list
    listofdsets->ndsets = 0l;
    listofdsets->firstdset = NULL;
    listofdsets->lastdset = NULL;
  }

  return(listofdsets);

} // h5sx_empty_listofdsets

/*---------------------------------------------------------------------------
NAME

  h5sx_free_listofdsets --- Releases a list of dsets

PURPOSE

  Releases the memory of a list of dsets: 
    const char *key, hid_t dataset, H5sxDset

RETURN VALUE

  NULL, otherwise error

---------------------------------------------------------------------------*/
H5sxListOfDSets *h5sx_free_listofdsets( H5sxListOfDSets *listofdsets ) 
{ 
  h5sx_empty_listofdsets( listofdsets );
  FREE(listofdsets); 
  return( NULL );

} // h5sx_free_listofdsets

/*---------------------------------------------------------------------------
NAME

  h5sx_length_listofdsets --- return the number of dsets in the list

PURPOSE

  Return the number of dsets in the list

RETURN VALUE

  Number of dsets.

---------------------------------------------------------------------------*/
long h5sx_length_listofdsets( H5sxListOfDSets * listofdsets )
{ long ndsets = 0l;

  if ( listofdsets ) {
    ndsets = listofdsets->ndsets;
  }

  return( ndsets );

} // h5sx_length_listofdsets

/*---------------------------------------------------------------------------
NAME
  
  h5sx_print_listofdsets --- print list of dsets 
  
PURPOSE
  
  Print list of dsets

---------------------------------------------------------------------------*/
void h5sx_print_listofdsets( FILE *out, H5sxListOfDSets *listofdsets, const char *info, long indent )
{
    H5sxDset * dset;
    const char *key;
    const char *value;
    hid_t dataset=-1;
    long i;

    fprintf( out, "\n%*s%s: %ld element%s.\n",(int)indent,"",info,h5sx_length_listofdsets(listofdsets),h5sx_length_listofdsets(listofdsets)==1?"":"s");

    i=0;
    dset = h5sx_get_first_dset( listofdsets,  &key, &value, &dataset );

    while ( dset ) {
      unsigned long j;
      const char *string=NULL;
      if ((dataset<0)&&(dset->no_of_frames<=1)) {
        if (value) {
          string = value;
          fprintf( out, " %*s%ld: >>%s<<:>>%s<<\n",(int)indent,"",i,key,string);
        } else {
          fprintf( out, " %*s%ld: >>%s<<\n",(int)indent,"",i,key);
        }
      } else {
        fprintf( out, " %*s%ld: (>>%s<<,%d)\n",(int)indent,"",i,key,dataset);
        if (value) {
          for (j=0;j<dset->no_of_frames;j++) {
            string = value+j*(dset->maxvallen+1);
            fprintf( out, " %*s     %ld: >>%s<<\n", (int)indent,"",j, string);
          }
        }
      }

      i++;
      dset = h5sx_get_next_dset( dset, &key, &value, &dataset );

    } // while dset

    fprintf( out, "\n");

} // h5sx_print_listofdsets

/*---------------------------------------------------------------------------
NAME

  h5sx_search_dset --- returns dset with key and value

PURPOSE

  Return the first dset with matching key and matching value, otherwise NULL.

  If key or value are NULL, it is not taken into account.
  If key and value are NULL, everything matches.
  The comparisons are case insensitive.

RETURN VALUE

  Address to matching dset or NULL, if not found.

---------------------------------------------------------------------------*/
H5sxDset *h5sx_search_dset( H5sxListOfDSets *listofdsets, 
                            const char *key, const char *value )
{
  H5sxDset *current_set=NULL;
  const char *current_key=NULL, *current_value=NULL;
  int matching=0;

  current_set=h5sx_get_first_dset(listofdsets,&current_key,&current_value,NULL);

  while (current_set) {
    if (!key) matching=1; 
    else matching=strlib_casecmp(current_key,key)?0:1;
    if (value) matching=strlib_casecmp(current_value,value)?0:matching;
    if (matching) break;
    current_set = current_set->nextdset;
  }

  return(current_set);

} /* h5sx_search_dset */

/*******************************************************************************
* Create, define, read and release a list of dataset variables (END)           *
*******************************************************************************/

const char *h5sxsearchlists( void ) 
{
  static const char *searchlists="\
  SXfile            : SX program output file\n\
  ID02PyFAI         : ID02 PyFAI output file (\"dahu\")\n\
  ID02PyFAIscaler   : ID02 PYFAI scaler file (\"dahu\")\n\
  ID02LIMA          : ID02 LIMA data file\n\
  SLFF              : ID01 detector flatfield file (version SL)\n\
  SLMF              : ID01 detector mask file (version SL)\n\
  default           : Try some possibilities";

  return(searchlists);
} /* h5sxsearchlists */

/*---------------------------------------------------------------------------
NAME
  h5sxdefaultsearchlists ---  returns h5sx search mode as string

SYNOPSIS
  int h5sxdefaultsearchlists ( const char *filetype,
                               H5sxListOfDSets *entry_search_names,
                               H5sxListOfDSets *entry_search_attributes,
                               H5sxListOfDSets *series_search_names,
                               H5sxListOfDSets *series_search_attributes,
                               H5sxListOfDSets *memory_search_names,
                               H5sxListOfDSets *memory_search_attributes,
                               H5sxListOfDSets *datablock_search_names,
                               H5sxListOfDSets *datablock_search_attributes,
                               H5sxListOfDSets *errorblock_search_names,
                               H5sxListOfDSets *errorblock_search_attributes,
                               H5sxListOfDSets *header_search_names,
                               H5sxListOfDSets *header_search_attributes,
                               H5sxListOfDSets *symbol_search_names,
                               H5sxListOfDSets *symbol_search_attributes,
                               H5sxListOfDSets *array_search_names,
                               H5sxListOfDSets *array_search_attributes,
                               H5sxListOfDSets *error_array_search_names,
                               H5sxListOfDSets *error_array_search_attributes
                             );

DESCRIPTION
  This function returns search attributes for the following filetypes:
  "SXfile"            : SX program output file
  "ID02PyFAI"         : ID02 PyFAI output file ("dahu")
  "ID02PyFAIscaler"   : ID02 PYFAI scaler file ("dahu")
  "ID02LIMA"          : ID02 LIMA data file
  "SLFF"              : ID01 detector flatfield file (version SL)
  "SLMF"              : ID01 detector mask file (version SL)
  "default"           : Try some possibilities

If a search name or search attribute is NULL or empty it is ignored.
If a node (entry, series, memory, datablock, errorblock, header)
cannot be found with the supplied search conditions, a dummy node 
is inserted in the owning node and the search is continued with the
next hierarchy level. If a dataset cannot be found it is skipped.

A special error array search condition can be defined. 

SXfile:

file       : *                              : {'SX_class':'SXfile'}
entry      : None                           : {'SX_class':'SXentry'}
series     : None                           : {'SX_class':'SXseries'}
memory     : None                           : {'SX_class':'SXmemory'}

datablock  : None                           : {'SX_class':'SXdata'}
errorblock : None                           : {'SX_class':'SXerror'}
array      : None                           : {'SX_class':'SXarray'}
header     : *                              : None // {'SX_class':'SXheader'}
symbols    : *                              : None // {'SX_class':'SXsymbol'}

ID02PyFAI (ID02 PyFAI output file):

file       : *                              : None
entry      : None                           : {'NX_class':'NXentry'}
series     : None                           : None
memory     : None                           : {'NX_class':'NXprocess'}
datablock  : None                           : {'NX_class':'NXdata'}
array      : {'data'}                       : None
header     : {'MCS','TFG','parameters'}     : None
symbols    : *                              : None

ID02PyFAIscaler (ID02 PYFAI scaler file):

file       : *                              : None
entry      : None                           : {'NX_class':'NXentry'}
series     : None                           : None
memory     : None                           : None
datablock  : None                           : {'NX_class':'NXcollection'}
array      : None                           : None
header     : {'MCS','TFG'}                  : {'NX_class':'NXcollection'}
symbols    : *                              : None

ID02LIMA (ID02 LIMA data file):

file       : *                              : None
entry      : None                           : {'NX_class':'NXentry'}
series     : {'measurement'}                : {'NX_class':'NXcollection'}
memory     : None                           : None
datablock  : None                           : {'NX_class':'NXdetector'}
array      : {'data'}                       : {'interpretation':'image',signal:1}
header     : {'parameters'}                 : None
symbols    : *                              : None

RETURN VALUE
0: success

---------------------------------------------------------------------------*/
int h5sxdefaultsearchlists ( const char *filetype,
                             H5sxListOfDSets *entry_search_names,
                             H5sxListOfDSets *entry_search_attributes,
                             H5sxListOfDSets *series_search_names,
                             H5sxListOfDSets *series_search_attributes,
                             H5sxListOfDSets *memory_search_names,
                             H5sxListOfDSets *memory_search_attributes,
                             H5sxListOfDSets *datablock_search_names,
                             H5sxListOfDSets *datablock_search_attributes,
                             H5sxListOfDSets *errorblock_search_names,
                             H5sxListOfDSets *errorblock_search_attributes,
                             H5sxListOfDSets *header_search_names,
                             H5sxListOfDSets *header_search_attributes,
                             H5sxListOfDSets *symbol_search_names,
                             H5sxListOfDSets *symbol_search_attributes,
                             H5sxListOfDSets *array_search_names,
                             H5sxListOfDSets *array_search_attributes,
                             H5sxListOfDSets *error_array_search_names,
                             H5sxListOfDSets *error_array_search_attributes
                           )
{
  if ((!filetype)||(!strlen(filetype))) filetype="default"; // filetype="ID02PyFAI";

  if (!strlib_casecmp(filetype,SXCLASS_FILE)) {
    // SX
    h5sx_append_dset( entry_search_attributes, SXCLASS, SXCLASS_ENTRY, 0, 1, -1 );
    h5sx_append_dset( series_search_attributes, SXCLASS, SXCLASS_SERIES, 0, 1, -1 );
    h5sx_append_dset( memory_search_attributes, SXCLASS, SXCLASS_MEMORY, 0, 1, -1 );
    h5sx_append_dset( datablock_search_attributes, SXCLASS, SXCLASS_DATABLOCK, 0, 1, -1 );
    h5sx_append_dset( errorblock_search_attributes, SXCLASS, SXCLASS_ERRORBLOCK, 0, 1, -1 );
    h5sx_append_dset( header_search_attributes, SXCLASS, SXCLASS_HEADER, 0, 1, -1 );
    h5sx_append_dset( symbol_search_names, "*", NULL, 0, 1, -1 );
    // symbol_search_attributes
    h5sx_append_dset( array_search_names, "*", NULL, 0, 1, -1 );
    // array_search_attributes
    h5sx_append_dset( error_array_search_names, "*", NULL, 0, 1, -1 );
    // array_search_attributes

  } else if (!strlib_casecmp(filetype,"ID02PyFAI")) { // default
    // PyFAI
    h5sx_append_dset( entry_search_attributes, "NX_class", "NXentry", 0, 1, -1 );
    h5sx_append_dset( memory_search_attributes, "NX_class", "NXprocess", 0, 1, -1 );
    h5sx_append_dset( datablock_search_attributes, "NX_class", "NXdata", 0, 1, -1 );
    // h5sx_append_dset( array_search_attributes, "interpretation", "image", 0, 1, -1 ); 
    h5sx_append_dset( array_search_names, "data", NULL, 0, 1, -1 );
    h5sx_append_dset( header_search_names, "MCS", NULL, 0, 1, -1 );
    h5sx_append_dset( header_search_names, "TFG", NULL, 0, 1, -1 );
    h5sx_append_dset( header_search_names, "parameters", NULL, 0, 1, -1 );
    // header_search_attributes
    h5sx_append_dset( symbol_search_names, "*", NULL, 0, 1, -1 );
    // symbol_search_attributes

  } else if (!strlib_casecmp(filetype,"ID02PyFAIscaler")) {
    // ID02PyFAIscaler
    h5sx_append_dset( entry_search_attributes, "NX_class", "NXentry", 0, 1, -1 );
    h5sx_append_dset( datablock_search_attributes, "NX_class", "NXinstrument", 0, 1, -1 );
    h5sx_append_dset( header_search_names, "MCS", NULL, 0, 1, -1 );
    h5sx_append_dset( header_search_names, "TFG", NULL, 0, 1, -1 );
    // h5sx_append_dset( header_search_names, "MCS", "AND", 0, 1, -1 );
    // h5sx_append_dset( header_search_names, "TFG", "AND", 0, 1, -1 );
    // h5sx_append_dset( header_search_attributes, "NX_class", "NXcollection", 0, 1, -1 );
    h5sx_append_dset( symbol_search_names, "*", NULL, 0, 1, -1 );

  } else if (!strlib_casecmp(filetype,"ID02LIMA")) {
    // ID02LIMA
    h5sx_append_dset( entry_search_attributes, "NX_class", "NXentry", 0, 1, -1 );
    h5sx_append_dset( series_search_names, "measurement", NULL, 0, 1, -1 );
    h5sx_append_dset( series_search_attributes, "NX_class", "NXcollection", 0, 1, -1 );
    h5sx_append_dset( datablock_search_attributes, "NX_class", "NXdetector", 0, 1, -1 );
    h5sx_append_dset( array_search_names, "data", NULL, 0, 1, -1 );
    h5sx_append_dset( array_search_attributes, "interpretation", "image", 0, 1, -1 );
    h5sx_append_dset( header_search_names, "parameters", NULL, 0, 1, -1 );
    h5sx_append_dset( symbol_search_names, "*", NULL, 0, 1, -1 );

  } else if (!strlib_casecmp(filetype,"SLFF")) {
    // SLFF (steven leake's flatfield format)
    h5sx_append_dset( entry_search_attributes, "NX_class", "NXentry", 0, 1, -1 );
    h5sx_append_dset( series_search_names, "measurement", NULL, 0, 1, -1 );
    h5sx_append_dset( series_search_attributes, "NX_class", "NXcollection", 0, 1, -1 );

    h5sx_append_dset( memory_search_attributes, "NX_class", "NXdetector", 0, 1, -1 );

    h5sx_append_dset( datablock_search_names, "ff", NULL, 0, 1, -1 );
    h5sx_append_dset( array_search_names, "flatfield", NULL, 0, 1, -1 );
    h5sx_append_dset( array_search_names, "ff", NULL, 0, 1, -1 );
    h5sx_append_dset( header_search_names, "parameters", NULL, 0, 1, -1 );
    //h5sx_append_dset( symbol_search_names, "*", NULL, 0, 1, -1 );

    //h5sx_append_dset( errorblock_search_names, "ff_rel_unc", NULL, 0, 1, -1 );
    h5sx_append_dset( errorblock_search_names, "ff", NULL, 0, 1, -1 );
    h5sx_append_dset( error_array_search_names, "flatfield_error", NULL, 0, 1, -1 );
    h5sx_append_dset( error_array_search_names, "ff_rel_unc", NULL, 0, 1, -1 );
    //h5sx_append_dset( header_search_names, "parameters", NULL, 0, 1, -1 );
    //h5sx_append_dset( symbol_search_names, "*", NULL, 0, 1, -1 );  // must require a header group

  } else if (!strlib_casecmp(filetype,"SLMF")) {
    // SLFF (steven leake's maskfile format)
    h5sx_append_dset( entry_search_attributes, "NX_class", "NXentry", 0, 1, -1 );
    h5sx_append_dset( series_search_names, "measurement", NULL, 0, 1, -1 );
    h5sx_append_dset( series_search_attributes, "NX_class", "NXcollection", 0, 1, -1 );

    h5sx_append_dset( memory_search_attributes, "NX_class", "NXdetector", 0, 1, -1 );

    h5sx_append_dset( datablock_search_names, "ff", NULL, 0, 1, -1 );
    h5sx_append_dset( array_search_names, "pixel_mask", NULL, 0, 1, -1 );
    h5sx_append_dset( array_search_names, "bad_pix_mask", NULL, 0, 1, -1 );
    h5sx_append_dset( header_search_names, "parameters", NULL, 0, 1, -1 );
    //h5sx_append_dset( symbol_search_names, "*", NULL, 0, 1, -1 );  // must require a header group

  } else if (!strlib_casecmp(filetype,"default")) {
    // default
    h5sx_append_dset( entry_search_attributes, "NX_class", "NXentry", 0, 1, -1 );
    h5sx_append_dset( series_search_names, "measurement", NULL, 0, 1, -1 );
    h5sx_append_dset( series_search_attributes, "NX_class", "NXcollection", 0, 1, -1 );
    h5sx_append_dset( memory_search_attributes, "NX_class", "NXprocess", 0, 1, -1 );

    h5sx_append_dset( datablock_search_attributes, "NX_class", "NXdetector", 0, 1, -1 );
    h5sx_append_dset( datablock_search_attributes, "NX_class", "NXdata", 0, 1, -1 );
    h5sx_append_dset( datablock_search_attributes, "NX_class", "NXinstrument", 0, 1, -1 );

    h5sx_append_dset( array_search_names, "data", NULL, 0, 1, -1 );
    h5sx_append_dset( array_search_names, "image_data", NULL, 0, 1, -1 );
    h5sx_append_dset( array_search_names, "imag", NULL, 0, 1, -1 );
    h5sx_append_dset( array_search_names, "Image1", NULL, 0, 1, -1 );
    h5sx_append_dset( array_search_attributes, "interpretation", "image", 0, 1, -1 );

    h5sx_append_dset( header_search_names, "MCS", NULL, 0, 1, -1 );
    h5sx_append_dset( header_search_names, "TFG", NULL, 0, 1, -1 );
    h5sx_append_dset( header_search_names, "parameters", NULL, 0, 1, -1 );

    h5sx_append_dset( symbol_search_names, "*", NULL, 0, 1, -1 );

  }

  return(0);

} /* h5sxdefaultsearchlists */

/*******************************************************************************
* Accessing the h5 file (BEGIN)                                                *
*******************************************************************************/

/*
 * htri_t h5sx_link_exists( hid_t loc_id, const char *name )
 * >0 link exists
 *  0 link does not exist
 * <0 error
 */
htri_t h5sx_link_exists( hid_t loc_id, const char *name )
{
  htri_t exists=-1;

  if (iodbg(IODBG_H5|IODBG_DEBUG1)) printf("h5sx_link_exists( %d, %s ) BEGIN\n",loc_id,name);
  // htri_t H5Lexists( hid_t loc_id, const char *name, hid_t lapl_id )
  exists = H5Lexists( loc_id, name, H5P_DEFAULT);
  if (iodbg(IODBG_H5|IODBG_DEBUG1)) printf("h5sx_link_exists( %d, %s ) (exists=%d) END\n",loc_id,name,exists);

  return(exists);

} // h5sx_link_exists

/*
 * int h5sx_group_exists( hid_t loc_id, const char *name )
 *  1 group exists
 *  0 group does not exist
 * -1 error or link exists, but it is not a group
 */
int h5sx_group_exists( hid_t loc_id, const char *name )
{ htri_t exists=-1;

  exists=h5sx_link_exists( loc_id, name );

  if (exists<0) goto h5sx_group_exists_error;

  if (exists>0) {
    H5O_info_t object_info;

    if ( H5Oget_info( loc_id, &object_info ) < 0 ) goto h5sx_group_exists_error;

    if ( object_info.type!=H5O_TYPE_GROUP ) goto h5sx_group_exists_error;

    return(1);
  } 

  return(0);

h5sx_group_exists_error:

  return(-1);

} // h5sx_group_exists

/*
 * Return string value of the attribute of loc_id with name attr_key 
 * (only scalar string value or first element of a string array)
 * returns NULL in case of an error
 * Could be extended for converting numbers to strings
 * h5sx_attrib_read
 */
const char *h5sx_attrib_read( char buffer[], hsize_t buflen, hid_t  loc_id, const char *attr_key ) 
{
  char infobuf[H5SX_INFOBUFLEN];

  hid_t attr_id=-1, attr_type=-1, attr_space=-1;
  htri_t attr_is_vls=-1; // htri_t: TRUE, FALSE or negative in case of failure
  H5T_cset_t attr_cset=-1;
  int attr_rank=-1;
  hsize_t *attr_dims=NULL;
  hsize_t *attr_maxdims=NULL;
  hsize_t attr_storage_size=0;

  hid_t attr_memtype=-1;
  hsize_t attr_memsize=0;
  hsize_t attr_stringsize=0;

  H5T_class_t attr_class;

  herr_t h5status=0;

  char **attr_rdata=NULL;
  char *attr_data=NULL;
  char *value=NULL;

  if (iodbg(IODBG_H5|IODBG_DEBUG1)) 
    printf("h5sx_attrib_read(%p,buflen=%ld,loc_id=%d,attr_key=%s) BEGIN\n",buffer, (long) buflen, loc_id, attr_key );

  if (!buffer) goto h5sx_attrib_read_error;
  if (buflen<1) goto h5sx_attrib_read_error;

  value = buffer;
  value[0]='\0'; 

  /*
   * Read attribute by name
   */
  if (iodbg(IODBG_H5|IODBG_DEBUG2)) 
    printf(">> attr_id = H5Aopen_by_name(%d,%s,%s,%d,%d)",loc_id,".",attr_key,H5P_DEFAULT,H5P_DEFAULT);
  attr_id = H5Aopen_by_name(loc_id,".",attr_key,H5P_DEFAULT,H5P_DEFAULT ); 
  if (iodbg(IODBG_H5|IODBG_DEBUG2)) 
    printf(" = %d\n",attr_id);
  if (attr_id<0) goto h5sx_attrib_read_error;

 /*
  * Get datatype of the attribute.
  */
  if (iodbg(IODBG_H5|IODBG_DEBUG2)) 
    printf(">> attr_type = H5Aget_type(%d)",attr_id);
  attr_type = H5Aget_type (attr_id);
  if (iodbg(IODBG_H5|IODBG_DEBUG2)) 
    printf(" = %d (%s, %s)\n", attr_type, h5sxdtype2classstr( infobuf, H5SX_INFOBUFLEN, attr_type ),
      edf_datatype2string( h5sxdtype2datatype( attr_type ))); 
  if (attr_type<0) goto h5sx_attrib_read_error;

 /*
  * Get class of the attribute.
  * H5T_class_t H5Tget_class( hid_t dtype_id )
  */
  if (iodbg(IODBG_H5|IODBG_DEBUG2)) 
    printf(">> H5Tget_class(%d)",attr_type);
  switch (attr_class=H5Tget_class(attr_type)) {
    case H5T_STRING:
      if (iodbg(IODBG_H5|IODBG_DEBUG2)) 
        printf(" = H5T_STRING\n");
    
     /*
      * What is the character set of the attribute value?
      */
      if (iodbg(IODBG_H5|IODBG_DEBUG2)) 
        printf(">> H5Tget_cset(%d)",attr_type);
      // H5T_cset_t H5Tget_cset( hid_t dtype_id )
      attr_cset = H5Tget_cset( attr_type );
      if (iodbg(IODBG_H5|IODBG_DEBUG2)) {
        switch(attr_cset) {
          case H5T_CSET_ASCII: printf(" = H5T_CSET_ASCII\n");
                               break;
          case H5T_CSET_UTF8 : printf(" = H5T_CSET_UTF8\n");
                               break;
          default: printf(" = unknown (%d)\n",attr_cset);
        }
      }

     /*
      * Is the attribute a variable length string?
      */
      if (iodbg(IODBG_H5|IODBG_DEBUG2)) 
        printf(">> H5Tis_variable_str(%d)",attr_type);
      attr_is_vls = H5Tis_variable_str( attr_type );
      if (iodbg(IODBG_H5|IODBG_DEBUG2)) 
        printf(" = %s (%d)\n",(attr_is_vls<0)?"Error":(attr_is_vls?"Yes":"No"),attr_is_vls);

     /*
      * Get dataspace
      */
      if (iodbg(IODBG_H5|IODBG_DEBUG2)) 
        printf(">> H5Aget_space(%d)",attr_id);
      attr_space = H5Aget_space (attr_id);
      if (iodbg(IODBG_H5|IODBG_DEBUG2)) 
        printf(" = %d\n", attr_space);
  
     /*
      * Get rank of attribute, allocate attr_dims and attr_maxdims
      * By convention, the rank of a scalar dataspace is 0 (zero)
      */
      if (iodbg(IODBG_H5|IODBG_DEBUG2)) 
        printf(">> H5Sget_simple_extent_ndims(%d)",attr_space);
      attr_rank = H5Sget_simple_extent_ndims( attr_space );
      if (iodbg(IODBG_H5|IODBG_DEBUG2))
        printf(" = %ld\n",(long) attr_rank);
      if (attr_rank<0) {
        printf("ERROR: h5sx_attrib_read->H5Sget_simple_extent_ndims: attr_key=%s\n",attr_key);
        goto h5sx_attrib_read_error;
      }

      if (attr_rank>0) {
        if (iodbg(IODBG_H5|IODBG_DEBUG2))
          printf(">>  attr_dims = (hsize_t*) CALLOC(%ld,%ld)",(long) attr_rank,(long) sizeof(hsize_t));
        attr_dims = (hsize_t*) CALLOC(attr_rank,sizeof(hsize_t));
        if (iodbg(IODBG_H5|IODBG_DEBUG2)) {
          printf(" = %p\n",attr_dims);
        }
        if (iodbg(IODBG_H5|IODBG_DEBUG2))
          printf(">>  attr_maxdims = (hsize_t*) CALLOC(%ld,%ld)",(long) attr_rank,(long) sizeof(hsize_t));
        attr_maxdims = (hsize_t*) CALLOC(attr_rank,sizeof(hsize_t));
        if (iodbg(IODBG_H5|IODBG_DEBUG2)) {
          printf(" = %p\n",attr_maxdims);
        }

      } else {
        // scalar dataspace
        if (iodbg(IODBG_H5|IODBG_DEBUG2))
          printf(">>  attr_dims = (hsize_t*) CALLOC(%ld,%ld)",(long) 1,(long) sizeof(hsize_t));
        attr_dims = (hsize_t*) CALLOC(1,sizeof(hsize_t));
        if (iodbg(IODBG_H5|IODBG_DEBUG2)) {
          printf(" = %p\n",attr_dims);
        }
        if (iodbg(IODBG_H5|IODBG_DEBUG2))
          printf(">>  attr_maxdims = (hsize_t*) CALLOC(%ld,%ld)",(long) 1,(long) sizeof(hsize_t));
        attr_maxdims = (hsize_t*) CALLOC(1,sizeof(hsize_t));
        if (iodbg(IODBG_H5|IODBG_DEBUG2)) {
          printf(" = %p\n",attr_maxdims);
        }
      }
  
      if (iodbg(IODBG_H5|IODBG_DEBUG2)) 
        printf(">> H5Sget_simple_extent_dims(%d,%p,%p)",attr_space,attr_dims,attr_maxdims);
      attr_rank=H5Sget_simple_extent_dims (attr_space, attr_dims, attr_maxdims);
      if (iodbg(IODBG_H5|IODBG_DEBUG2))
        printf(" = %ld\n",(long) attr_rank);
      if (iodbg(IODBG_H5|IODBG_DEBUG2)) {
        if (attr_rank>0) {
          long idim;
          for (idim=0;idim<attr_rank;idim++) {
            printf(">> attr_dims[%ld] = %ld; attr_maxdims[%ld] = %ld;\n",(long) idim,(long) attr_dims[idim],(long) idim,(long) attr_maxdims[idim]);
          }
        } else {
          printf(">> attr_dims = %ld; attr_maxdims = %ld\n",(long) attr_dims[0], (long) attr_maxdims[0]);
        }
      }

      if (attr_rank<0) {
        printf("ERROR: h5sx_attrib_read->H5Sget_simple_extent_ndims: attr_rank=%ld, attr_key=%s\n",(long) attr_rank, attr_key);
        goto h5sx_attrib_read_error;
      }

     /*
      * HDF routines do not convert automatically between variable length strings
      * and fixed length strings. It must be done manually.
      */
      switch (attr_is_vls) {
        case 1: // variable length strings
          if (iodbg(IODBG_H5|IODBG_DEBUG2))
            printf(">> Reading variable length string\n");

          if (iodbg(IODBG_H5|IODBG_DEBUG2))
            printf(">>  attr_rdata = (char **) CALLOC(%ld,%ld)",((long) ((attr_dims[0]>0)?attr_dims[0]:1)),(long) sizeof(char *));
          attr_rdata = (char **) CALLOC (((attr_dims[0]>0)?attr_dims[0]:1),sizeof (char *));
          if (iodbg(IODBG_H5|IODBG_DEBUG2))
            printf(" = %p\n",attr_rdata);
          if (!attr_rdata) goto h5sx_attrib_read_error;

         /*
          * Create the memory datatype.
          */
          if (iodbg(IODBG_H5|IODBG_DEBUG2)) 
            printf(">> H5Tcopy(%d)",H5T_C_S1);
          attr_memtype = H5Tcopy (H5T_C_S1);
          if (iodbg(IODBG_H5|IODBG_DEBUG2)) 
            printf(" = %d\n", attr_memtype);

          if (iodbg(IODBG_H5|IODBG_DEBUG2))
            printf(">> H5Tset_cset(%d,H5T_CSET_ASCII)\n",attr_memtype);
          h5status=H5Tset_cset(attr_memtype,H5T_CSET_ASCII);
          if (h5status<0) {
            printf("ERROR: h5sx_attrib_read->H5Tset_cset: attr_key=%s\n",attr_key);
            goto h5sx_attrib_read_error;
          }

          if (iodbg(IODBG_H5|IODBG_DEBUG2)) 
            printf(">> H5Tset_size(%d,%s)\n",attr_memtype,"H5T_VARIABLE");
          h5status = H5Tset_size (attr_memtype, H5T_VARIABLE);
          if (h5status<0) {
            printf("ERROR: h5sx_attrib_read->H5Tset_size: attr_key=%s\n",attr_key);
            goto h5sx_attrib_read_error;
          }

         /*
          * Read variable length strings. 
          * attr_rdata is a pointer to a pointer array
          */
          if (iodbg(IODBG_H5|IODBG_DEBUG2)) 
            printf(">> H5Aread(%d,%d,%p)\n",attr_id,attr_memtype,attr_rdata);
          h5status = H5Aread (attr_id, attr_memtype, attr_rdata);
          if (h5status<0) {
            printf("ERROR: h5sx_attrib_read->H5Aread: attr_key=%s\n",attr_key);
            goto h5sx_attrib_read_error;
          }

          if (iodbg(IODBG_H5|IODBG_DEBUG2)) {
            if (attr_rank>0) {
              char **pps;
              long irank;

              pps=attr_rdata;
              for (irank=0;irank<attr_rank;irank++) {
                long idim;
                printf(">> attr_data[%ld] = ",irank);
                for (idim=0;idim<attr_dims[irank];idim++) {
                  if (idim) printf(",\"%s\"",*pps);
                  else printf("\"%s\"",*pps);
                  pps++;
                }
                printf("\n");
              }
            } else {
              printf(">> attr_rdata[0] = \"%s\"\n",attr_rdata[0]);
            }
          }

         /*
          * Write all attribute values to the output buffer
          * (rank<=2)
          */
          if (attr_rank>0) {
            char **pps;
            char *pv;
            hsize_t rest;
            long irank=0;

            pps=attr_rdata;
            pv = value;
            rest = buflen-1;
            for (irank=0;irank<attr_rank;irank++) {
              long idim;

              if (irank) {
                snprintf(pv,rest,";");
                rest--; pv++;
              }
              for (idim=0;idim<attr_dims[irank];idim++) {
                if (idim) {
                  snprintf(pv,rest,",%s",*pps);
                  rest-=strlen(*pps)+1;
                  pv+=strlen(*pps)+1;
                } else {
                  snprintf(pv,rest,"%s",*pps);
                  rest-=strlen(*pps);
                  pv+=strlen(*pps);
                }
                pps++; // For rank>1 the order of items is probably not as expected 
              }
            }
          } else {
            strncpy(value,attr_rdata[0],buflen-1);
          }
          value[buflen-1]='\0'; // force the last element to null

          if (iodbg(IODBG_H5|IODBG_DEBUG2)) 
            printf(">> H5Dvlen_reclaim(%d,%d,%s,%p)\n",attr_memtype, attr_space, "H5P_DEFAULT", attr_rdata);
          h5status=H5Dvlen_reclaim (attr_memtype, attr_space, H5P_DEFAULT, attr_rdata);
          if (h5status<0) {
            printf("ERROR: h5sx_attrib_read->H5Dvlen_reclaim: attr_key=%s\n",attr_key);
            goto h5sx_attrib_read_error;
          }

          if (iodbg(IODBG_H5|IODBG_DEBUG2))
            printf(">> FREE(%p)\n",attr_rdata);
          FREE(attr_rdata);

          if (attr_memtype>=0) {
            if (iodbg(IODBG_H5|IODBG_DEBUG2))
              printf(">> H5Tclose(%d)\n",attr_memtype);
            H5Tclose(attr_memtype); attr_memtype=-1;
          }

          break;

        case 0: // fixed length string
          if (iodbg(IODBG_H5|IODBG_DEBUG2))
            printf(">> Reading fixed length string\n");

          if (iodbg(IODBG_H5|IODBG_DEBUG2)) 
            printf(">> H5Aget_storage_size(%d)",attr_id);
          attr_storage_size = H5Aget_storage_size(attr_id);
          if (iodbg(IODBG_H5|IODBG_DEBUG2)) 
            printf(" = %ld\n", (long) attr_storage_size);

          if (iodbg(IODBG_H5|IODBG_DEBUG2)) 
            printf(">> H5Tget_size(%d)",attr_type);
          attr_stringsize = H5Tget_size (attr_type);
          if (iodbg(IODBG_H5|IODBG_DEBUG2))
            printf(" = %ld + 1\n", (long) attr_stringsize);
          attr_stringsize++;

          attr_memsize=attr_stringsize;
          if (attr_rank>0) {
            long irank;
            for (irank=0;irank<attr_rank;irank++) {
              attr_memsize*=attr_dims[irank];
            }
          }

          if (iodbg(IODBG_H5|IODBG_DEBUG2))
            printf(">>  attr_data = (char *) CALLOC(%ld,%ld)",(long)attr_memsize,(long) sizeof(char));
          attr_data = (char *) CALLOC (attr_memsize,sizeof(char));
          if (iodbg(IODBG_H5|IODBG_DEBUG2))
            printf(" = %p\n",attr_data);
          if (!attr_data) goto h5sx_attrib_read_error;

         /*
          * Create the memory datatype.
          */
          if (iodbg(IODBG_H5|IODBG_DEBUG2))
            printf(">> H5Tcopy(%d)",H5T_C_S1);
          attr_memtype = H5Tcopy (H5T_C_S1);
          if (iodbg(IODBG_H5|IODBG_DEBUG2)) 
            printf(" = %d\n", attr_memtype);

          if (iodbg(IODBG_H5|IODBG_DEBUG2))
            printf(">> H5Tset_cset(%d,H5T_CSET_ASCII)\n",attr_memtype);
          h5status=H5Tset_cset(attr_memtype,H5T_CSET_ASCII);
          if (h5status<0) {
            printf("ERROR: h5sx_attrib_read->H5Tset_cset: attr_key=%s\n",attr_key);
            goto h5sx_attrib_read_error;
          }

          if (iodbg(IODBG_H5|IODBG_DEBUG2))
            printf(">> H5Tset_size(%d,%ld)\n",attr_memtype,(long) attr_stringsize);
          h5status = H5Tset_size (attr_memtype, attr_stringsize);
          if (h5status<0) {
            printf("ERROR: h5sx_attrib_read->H5Tset_size: attr_key=%s\n",attr_key);
            goto h5sx_attrib_read_error;
          }

         /*
          * Read fixed length strings. 
          * attr_data is a pointer to a data array
          */
          if (iodbg(IODBG_H5|IODBG_DEBUG2))
            printf(">> H5Aread(%d,%d,%p)\n",attr_id,attr_memtype,attr_data);
          h5status = H5Aread (attr_id, attr_memtype, attr_data);
          if (h5status<0) {
            printf("ERROR: h5sx_attrib_read->H5Aread: attr_key=%s\n",attr_key);
            goto h5sx_attrib_read_error;
          }

          if (iodbg(IODBG_H5|IODBG_DEBUG2)) {
            if (attr_rank>0) {
              char *ps;
              long irank;
              ps=attr_data;
              for (irank=0;irank<attr_rank;irank++) {
                long idim;
                printf(">> attr_data[%ld] = ",irank);
                for (idim=0;idim<attr_dims[irank];idim++) {
                  if (idim) printf(",\"%s\"",ps);
                  else printf("\"%s\"",ps);
                  ps+=attr_stringsize;
                }
                printf("\n");
              }
            } else {
              printf(">> attr_data = \"%s\"\n",attr_data);
            }
          }

         /*
          * Write all attribute values to the output buffer
          * (rank<=2)
          */
          if (attr_rank>0) {
            char *ps;
            char *pv;
            hsize_t rest;
            long irank=0;

            ps=attr_data;
            pv = value;
            rest = buflen-1;
            for (irank=0;irank<attr_rank;irank++) {
              long idim;

              if (irank) {
                snprintf(pv,rest,";");
                rest--; pv++;
              }
              for (idim=0;idim<attr_dims[irank];idim++) {
                if (idim) {
                  snprintf(pv,rest,",%s",ps);
                  rest-=strlen(ps)+1;
                  pv+=strlen(ps)+1;
                } else {
                  snprintf(pv,rest,"%s",ps);
                  rest-=strlen(ps);
                  pv+=strlen(ps);
                }
                ps+=attr_stringsize;
              }
            }
          } else {
            strncpy(value,attr_data,buflen-1);
          }
          value[buflen-1]='\0'; // force the last element to null

          if (iodbg(IODBG_H5|IODBG_DEBUG2))
            printf(">> H5Dvlen_reclaim(%d,%d,%s,%p)\n",attr_memtype, attr_space, "H5P_DEFAULT", attr_data);
          h5status=H5Dvlen_reclaim (attr_memtype, attr_space, H5P_DEFAULT, attr_data);
          if (h5status<0) {
            printf("ERROR: h5sx_attrib_read->H5Dvlen_reclaim: attr_key=%s\n",attr_key);
            goto h5sx_attrib_read_error;
          }

          if (iodbg(IODBG_H5|IODBG_DEBUG2))
            printf(">> FREE(%p)\n",attr_data);
          FREE(attr_data);

          if (attr_memtype>=0) {
            if (iodbg(IODBG_H5|IODBG_DEBUG2))
              printf(">> H5Tclose(%d)\n",attr_memtype);
            H5Tclose(attr_memtype); attr_memtype=-1;
          }

          break;

        default: goto h5sx_attrib_read_error;
      }

      if (iodbg(IODBG_H5|IODBG_DEBUG2))
        printf(">> FREE(%p)\n",attr_dims);
      FREE(attr_dims);
      if (iodbg(IODBG_H5|IODBG_DEBUG2))
        printf(">> FREE(%p)\n",attr_maxdims);
      FREE(attr_maxdims);
      if (iodbg(IODBG_H5|IODBG_DEBUG2))
        printf(">> H5Sclose(%d)\n",attr_space);
      H5Sclose( attr_space ); attr_space=-1;
        break;

    case H5T_INTEGER:
      if (iodbg(IODBG_H5|IODBG_DEBUG2))
        printf(" = H5T_INTEGER\n");
      snprintf(buffer,buflen,"(H5T_INTEGER)");
      break;
    case H5T_FLOAT:
      if (iodbg(IODBG_H5|IODBG_DEBUG2))
        printf(" = H5T_FLOAT\n");
      snprintf(buffer,buflen,"(H5T_FLOAT)");
      break;
    case H5T_BITFIELD:
      if (iodbg(IODBG_H5|IODBG_DEBUG2))
        printf(" = H5T_BITFIELD\n");
      snprintf(buffer,buflen,"(H5T_BITFIELD)");
      break;
    case H5T_OPAQUE:
      if (iodbg(IODBG_H5|IODBG_DEBUG2))
        printf(" = H5T_OPAQUE\n");
      snprintf(buffer,buflen,"(H5T_OPAQUE)");
      break;
    case H5T_COMPOUND:
      if (iodbg(IODBG_H5|IODBG_DEBUG2))
        printf(" = H5T_COMPOUND\n");
      snprintf(buffer,buflen,"(H5T_COMPOUND)");
      break;
    case H5T_REFERENCE:
      if (iodbg(IODBG_H5|IODBG_DEBUG2))
        printf(" = H5T_REFERENCE\n");
      snprintf(buffer,buflen,"(H5T_REFERENCE)");
      break;
    case H5T_ENUM:
      if (iodbg(IODBG_H5|IODBG_DEBUG2))
        printf(" = H5T_ENUM\n");
      snprintf(buffer,buflen,"(H5T_ENUM)");
      break;
    case H5T_VLEN:
      if (iodbg(IODBG_H5|IODBG_DEBUG2))
        printf(" = H5T_VLEN\n");
      snprintf(buffer,buflen,"(H5T_VLEN)");
      break;
    case H5T_ARRAY :
      if (iodbg(IODBG_H5|IODBG_DEBUG2))
        printf(" = H5T_ARRAY\n");
      snprintf(buffer,buflen,"(H5T_ARRAY)");
      break;
    default:
      if (iodbg(IODBG_H5|IODBG_DEBUG2))
        printf(" = %d (unsupported attribute class)\n",attr_class);
      snprintf(buffer,buflen,"(unsupported attribute class)");
      break;
  }

  if (iodbg(IODBG_H5|IODBG_DEBUG2))
    printf(">> H5Tclose(%d)\n",attr_type);
  H5Tclose( attr_type ); attr_type=-1;
  if (iodbg(IODBG_H5|IODBG_DEBUG2))
    printf(">> H5Aclose(%d)\n",attr_id);
  H5Aclose( attr_id ); attr_id=-1;

  if (iodbg(IODBG_H5|IODBG_DEBUG1)) 
    printf("h5sx_attrib_read(%p,buflen=%ld,loc_id=%d,attr_key=%s) (%s) END\n",buffer, (long) buflen, loc_id, attr_key, value );

  return(value);

h5sx_attrib_read_error:

  if (attr_rdata) {
    if (iodbg(IODBG_H5|IODBG_DEBUG2))
      printf(">> H5Dvlen_reclaim(%d,%d,%s,%p)\n",attr_memtype, attr_space, "H5P_DEFAULT", attr_rdata);
    H5Dvlen_reclaim (attr_memtype, attr_space, H5P_DEFAULT, attr_rdata);
    if (iodbg(IODBG_H5|IODBG_DEBUG2))
      printf(">> FREE(%p)\n",attr_rdata);
    FREE( attr_rdata );
  }
  if (attr_data) {
    if (iodbg(IODBG_H5|IODBG_DEBUG2))
      printf(">> H5Dvlen_reclaim(%d,%d,%s,%p)\n",attr_memtype, attr_space, "H5P_DEFAULT", attr_data);
    H5Dvlen_reclaim (attr_memtype, attr_space, H5P_DEFAULT, attr_data);
    if (iodbg(IODBG_H5|IODBG_DEBUG2))
      printf(">> FREE(%p)\n",attr_data);
    FREE( attr_data );
  }

  if (iodbg(IODBG_H5|IODBG_DEBUG2))
    printf(">> H5Tclose(%d)\n",attr_memtype);
  H5Tclose(attr_memtype); attr_memtype=-1;

  if (iodbg(IODBG_H5|IODBG_DEBUG2))
    printf(">> FREE(%p)\n",attr_dims);
  FREE( attr_dims );
  if (iodbg(IODBG_H5|IODBG_DEBUG2))
    printf(">> FREE(%p)\n",attr_maxdims);
  FREE( attr_maxdims );
  if (iodbg(IODBG_H5|IODBG_DEBUG2))
    printf(">> H5Sclose(%d)\n",attr_space);
  H5Sclose( attr_space ); attr_space=-1;
  if (iodbg(IODBG_H5|IODBG_DEBUG2))
    printf(">> H5Tclose(%d)\n",attr_type);
  H5Tclose( attr_type ); attr_type=-1;
  if (iodbg(IODBG_H5|IODBG_DEBUG2))
    printf(">> H5Aclose(%d)\n",attr_id);
  H5Aclose( attr_id ); attr_id=-1;

  printf("h5sx_attrib_read(%p,buflen=%ld,loc_id=%d,attr_key=%s) (%p) ERROR\n",buffer, (long) buflen, loc_id, attr_key, NULL );

  return(NULL);

} // h5sx_attrib_read

/*
 * Write scalar string attributes to loc_id: attr_key - attr_val
 * Returns a negative number in case of an error.
 */
int h5sx_attrib_write( hid_t loc_id, const char *attr_key, const char *attr_val )
{
  hid_t attr_space=-1, attr_type=-1, attr_id=-1;
  herr_t h5status=-1;

  if (iodbg(IODBG_H5|IODBG_DEBUG1))
    printf("h5sx_attrib_write(loc_id=%d,attr_key=%s,attr_val=%s) BEGIN\n", loc_id, attr_key, attr_val );

  attr_space = H5Screate(H5S_SCALAR);
  if (attr_space<0) goto h5sx_attrib_write_error;
    attr_type = H5Tcopy(H5T_C_S1);
      H5Tset_size(attr_type, strlen(attr_val));
      attr_id = H5Acreate( loc_id,attr_key, attr_type, attr_space, H5P_DEFAULT, H5P_DEFAULT );
      if (attr_id<0) goto h5sx_attrib_write_error;

        h5status = H5Awrite(attr_id, attr_type, attr_val);
        if (h5status<0) goto h5sx_attrib_write_error;

      if (attr_id>=0) H5Aclose(attr_id);

    if (attr_type>=0) H5Tclose(attr_type);

  if (attr_space>=0) H5Sclose(attr_space);

  if (iodbg(IODBG_H5|IODBG_DEBUG1))
    printf("h5sx_attrib_write(loc_id=%d,attr_key=%s,attr_val=%s) END\n", loc_id, attr_key, attr_val );

  return( 0 );

h5sx_attrib_write_error:

  printf("h5sx_attrib_write(loc_id=%d,attr_key=%s,attr_val=%s) ERROR\n", loc_id, attr_key, attr_val );

  if (attr_id>=0) H5Aclose(attr_id);
  if (attr_type>=0) H5Tclose(attr_type);
  if (attr_space>=0) H5Sclose(attr_space);

  return( -1 );

} // h5sx_attrib_write

herr_t h5sx_iterate_attrib (hid_t loc_id, const char *attr_key, const H5A_info_t *ainfo, void *op_data)
{   /*
     * typedef struct {
     *   hbool_t             corder_valid;   // Indicate if creation order is valid
     *   H5O_msg_crt_idx_t   corder;         // Creation order
     *   H5T_cset_t          cset;           // Character set of attribute name
     *   hsize_t             data_size;      // Size of raw data
     * } H5A_info_t;
     */

    const char *attr_val=NULL;
    H5sxListOfDSets *listofstringds=op_data;

    if (listofstringds) {

      char buffer[STRINGVSIZE];

      if (iodbg(IODBG_H5|IODBG_DEBUG2)) {
        printf(">> h5sx_iterate_attrib (id=%d,name=%s)",loc_id,attr_key);
        switch(ainfo->cset) {
          case H5T_CSET_ASCII: printf(" (H5T_CSET_ASCII)\n");
                               break;
          case H5T_CSET_UTF8 : printf(" (H5T_CSET_UTF8)\n");
                               break;
          default: printf(" (unknown character set %d)\n",ainfo->cset);
        }
      }

      attr_val = h5sx_attrib_read( buffer, STRINGVSIZE, loc_id, attr_key );

      if (attr_val) {
        if (iodbg(IODBG_H5|IODBG_DEBUG2)) {
          printf("    \"%s\" = \"%s\"\n",attr_key,attr_val);
        }
        h5sx_append_dset( listofstringds, attr_key, attr_val, 0, 1, -1 );
      }
    }

    if (attr_val) return(0);
    else return(-1);

} // h5sx_iterate_attrib

/*
 * Read scalar string attributes of loc_id and write to a list.
 * H5sxListOfDSets *attributes=NULL;
 * ...
 * h5sx_free_listofdsets(attributes);
 */
herr_t h5sx_attriblist_read( hid_t loc_id, H5sxListOfDSets *attributes )
{ 
  herr_t h5status=-1;

  if (attributes) {
    if (iodbg(IODBG_H5|IODBG_DEBUG2))
      printf(">> H5Aiterate_by_name(%d,%s,%d,%d,%p,%p,%p,%d)\n",
        loc_id, "/", H5_INDEX_NAME, H5_ITER_INC, NULL, h5sx_iterate_attrib, (void*) attributes, H5P_DEFAULT);
    h5status=H5Aiterate_by_name(loc_id, "/", H5_INDEX_NAME, H5_ITER_INC, NULL, h5sx_iterate_attrib, (void*) attributes, H5P_DEFAULT);
    if (h5status<0) {
      printf("ERROR: h5sx_attriblist_read->H5Aiterate_by_name\n");
      goto h5sx_attriblist_read_error;
    }

    if (iodbg(IODBG_H5|IODBG_SHOWATTVAL)) {
      h5sx_print_listofdsets(stdout, attributes, "h5sx_attriblist_read",1);
    }
  } else goto h5sx_attriblist_read_error;

  return(0);

h5sx_attriblist_read_error:

  return(-1);

} // h5sx_attriblist_read

/*
 * Write a list of scalar string attributes to loc_id.
 */
int h5sx_attriblist_write( hid_t loc_id, H5sxListOfDSets *attributes )
{
  const char *attr_key=NULL;
  const char *attr_val=NULL;
  H5sxDset *stringdset;

  // run through stringds of all variable header elements
  stringdset = h5sx_get_first_dset(attributes,&attr_key, &attr_val, NULL);

  while(stringdset) {

    if (!attr_key) {
      printf("ERROR: get_dset_and_increment\n");
      goto h5sx_attriblist_write_error;
    }

    if (h5sx_attrib_write( loc_id, attr_key, attr_val )<0) 
      goto h5sx_attriblist_write_error;

    stringdset = h5sx_get_next_dset(stringdset, &attr_key, &attr_val,NULL);

  } // while

  return( 0 );

h5sx_attriblist_write_error:

  return( -1 );

} // h5sx_attriblist_write

/*
 * Free array of pointers
 * Release nframes string elements of string_array.
 * If nframes is positive or zero, release all string elements of 
 * string_array until a NULL element is reached. 
 * Finally, release string_array.
 */
char **free_string_array( char **string_array, hsize_t nframes )
{
  if (iodbg(IODBG_H5|IODBG_DEBUG1))
    printf("free_string_array(%p) BEGIN\n",string_array);

  if (string_array) {
    char **pps;
    pps=string_array;

    if (nframes>0) {
      hsize_t i;
      for (i=0;i<nframes;i++) {
        FREE(*pps);
        pps++;
      }
    } else {
      while (*pps) {
        FREE(*pps);
        pps++; // string_array is terminated with NULL
      }
    }
    FREE(string_array);
  }

  if (iodbg(IODBG_H5|IODBG_DEBUG1))
    printf("free_string_array (%p) END\n",NULL);

  return(NULL);

} /* free_string_array */

herr_t h5sx_close_file( hid_t fid )
{ herr_t h5status=0;
  if (iodbg(IODBG_H5|IODBG_DEBUG1)) printf("h5sx_close_file(%d) BEGIN\n",fid);
  if (fid>=0) h5status = H5Fclose(fid);
  if (h5status<0) goto h5sx_close_file_error;

  if (iodbg(IODBG_H5|IODBG_DEBUG1)) printf("h5sx_close_file(%d) END (%d)\n",fid,h5status);
  return ( h5status );
h5sx_close_file_error:

  printf("h5sx_close_file(%d) ERROR (%d)\n",fid,h5status);
  return ( h5status );

} // h5sx_close_file

/*
 * hid_t h5sx_open_file( const char *name,  H5sxListOfDSets *attributes, unsigned flags )
 * Open HDF5 file name for reading and writing.
 * flags:
 *   H5F_ACC_RDWR        read/write access.
 *   H5F_ACC_RDONLY      read-only access.
 *   H5F_ACC_SWMR_WRITE  read/write access for a single-writer/multiple-reader (SWMR) scenario.
 *   H5F_ACC_SWMR_READ   read-only access for a single-writer/multiple-reader (SWMR) scenario. 
 * The returned attribute list must be released.
 */
hid_t h5sx_open_file( const char *name,  H5sxListOfDSets *attributes, unsigned flags )
{
  hid_t fid=-1;
  herr_t h5status=-1;

  if (iodbg(IODBG_H5|IODBG_DEBUG1)) printf("h5sx_open_file( %s , ... , %u ) BEGIN\n",name,flags);
  /*
   * Open and existing hdf5 file for read and write
   */   
  fid = H5Fopen( name, flags, H5P_DEFAULT );
  if (fid<0) goto h5sx_open_file_error;

  if (attributes) {
    h5status = h5sx_attriblist_read( fid, attributes );
    if (h5status<0) goto h5sx_open_file_error;
  }

  if (iodbg(IODBG_H5|IODBG_DEBUG1)) printf("h5sx_open_file( %s , ..., %u ) (%d) END\n",name,flags,fid);

  return(fid);

h5sx_open_file_error:

  printf("h5sx_open_file( %s , ..., %u ) ERROR\n",name,flags);
  h5sx_close_file( fid );

  return(-1);

} // h5sx_open_file

/*
 * hid_t h5sx_create_file( const char *name,  H5sxListOfDSets *attributes );
 * Create a new HDF5 file with the list of attributes.
 */
hid_t h5sx_create_file( const char *name,  H5sxListOfDSets *attributes )
{
  hid_t fid=-1;

  if (iodbg(IODBG_H5|IODBG_DEBUG1)) printf("h5sx_create_file( %s , ... ) BEGIN\n",name);

  /*
   * Create a new hdf5 file using the default properties.
   */
  fid = H5Fcreate (name, H5F_ACC_TRUNC, H5P_DEFAULT, H5P_DEFAULT);
  if (fid<0) goto h5sx_create_file_error;

  if (h5sx_attriblist_write(fid, attributes )<0) goto h5sx_create_file_error;

  if (iodbg(IODBG_H5|IODBG_DEBUG1)) printf("h5sx_create_file( %s , ... ) (%d) END\n",name,fid);

  return(fid);

h5sx_create_file_error:

  printf("h5sx_create_file( %s , ... ) ERROR\n",name);
  h5sx_close_file(fid);

  return(-1);

} // h5sx_create_file

herr_t h5sx_close_group( hid_t gid )
{ herr_t h5status=0;
  if (iodbg(IODBG_H5|IODBG_DEBUG1)) printf("h5sx_close_group( %d ) BEGIN\n",gid);
  if (gid>=0) h5status = H5Gclose(gid);
  if (iodbg(IODBG_H5|IODBG_DEBUG1)) printf("h5sx_close_group( %d ) (h5status=%d) END\n",gid,h5status);
  return ( h5status );
} // h5sx_close_group

/*
 * hid_t h5sx_insert_group( hid_t loc_id, const char *name, H5sxListOfDSets *attributes )
 * Open existing group or create HDF5 group with attributes in loc_id
 */
hid_t h5sx_insert_group( hid_t loc_id, const char *name,  H5sxListOfDSets *attributes )
{
  hid_t gid=-1;

  if (iodbg(IODBG_H5|IODBG_DEBUG1)) printf("h5sx_insert_group( %d, %s , ... ) BEGIN\n",loc_id,name);

  // if (h5sx_group_exists( hid_t loc_id, const char *name ))
  switch (h5sx_group_exists( loc_id, name )) {
    case 0: // create the group
      gid = H5Gcreate(loc_id, name,H5P_DEFAULT, H5P_DEFAULT, H5P_DEFAULT);
      if (gid<0) goto h5sx_insert_group_error;
      if ( attributes ) {
        if (h5sx_attriblist_write( gid, attributes )<0) goto h5sx_insert_group_error;
      }
      break;
    case 1: // open the existing group
      gid = H5Gopen(loc_id, name,H5P_DEFAULT);
      if (gid<0) goto h5sx_insert_group_error;
      // should check here attributes, otherwise error!
      break;
    default: // error, e.g. there could be already an object with the same name that is not a group
      goto h5sx_insert_group_error;
  }

  if (iodbg(IODBG_H5|IODBG_DEBUG1)) printf("h5sx_insert_group( %d, %s , ... ) (gid=%d) END\n",loc_id,name,gid);

  return(gid);

h5sx_insert_group_error:

  printf("h5sx_insert_group( %d, %s , ... ) ERROR\n",loc_id,name);
  h5sx_close_group(gid);

  return(-1);

} // h5sx_insert_group

/*
 * Read array dataset, its description and its attributes
 * If binary is NULL, only the description and the attributes are read
 * int h5sx_read_dataset_as_native_array( hid_t loc_id, const char *key, H5sxListOfDSets *attributes, 
 *                                  int *datatype, hsize_t *itemsize, hsize_t *rank, hsize_t **dims, hsize_t **maxdims, void **binary, hsize_t *binarysize )
 * The returned buffers (binary) and the returned dimensions (dims and maxdims) are allocated and must be 
 * released with FREE(array) and FREE(dims) for avoiding resource leaks.
 * In case of success the returned value is 0.
 */
int h5sx_read_dataset_as_native_array( hid_t loc_id, const char *key, H5sxListOfDSets *attributes, 
      int *datatype, hsize_t *itemsize, hsize_t *rank, hsize_t **dims, hsize_t **maxdims, void **binary, hsize_t *binarysize )
{
  char infobuf[H5SX_INFOBUFLEN];

  hid_t ds_id=-1, ds_type=-1, ds_space=-1;
  int ds_rank=-1;
  hsize_t *ds_dims=NULL;
  hsize_t *ds_maxdims=NULL;
  hsize_t ds_storage_size=-1;

  H5T_class_t ds_class;

  hid_t ds_memtype=-1, ds_nativetype=-1;
  hssize_t ds_npoints=-1;
  hsize_t ds_memitem=-1;
  hsize_t ds_memsize=0;

  int ds_datatype=InValidDType;

  void *ds_binary=NULL;

  herr_t h5status=-1;

  if (iodbg(IODBG_H5|IODBG_DEBUG1))
    printf("h5sx_read_dataset_as_native_array(loc_id=%d,key=%s,attributes=%p,itemsize=%p,rank=%p,dims=%p,binary=%p,binarysize=%p) BEGIN\n",
      loc_id, key, attributes, itemsize, rank, dims, binary, binarysize);

//----------------------------------------open begin

  /*
   * Read dataset by name
   */
  // hid_t H5Dopen( hid_t loc_id, const char *name, hid_t dapl_id )
  if (iodbg(IODBG_H5|IODBG_DEBUG2))
    printf(">> ds_id = H5Dopen(%d,%s,%d)",loc_id,key,H5P_DEFAULT);
  ds_id = H5Dopen(loc_id,key,H5P_DEFAULT);
  if (iodbg(IODBG_H5|IODBG_DEBUG2))
    printf(" = %d\n",ds_id);
  if (ds_id<0) goto h5sx_read_dataset_as_native_array_error;

  /*
   * Read the attributes
   */
  if (attributes) {
    if (h5sx_attriblist_read( ds_id, attributes )<0) goto h5sx_read_dataset_as_native_array_error;
  }

 /*
  * Get datatype of the dataset
  */
  if (iodbg(IODBG_H5|IODBG_DEBUG2))
    printf(">> ds_type = H5Dget_type(%d)",ds_id);
  ds_type = H5Dget_type (ds_id);
  if (iodbg(IODBG_H5|IODBG_DEBUG2))
    printf(" = %d (%s, %s)\n", ds_type, h5sxdtype2classstr( infobuf, H5SX_INFOBUFLEN, ds_type ),
      edf_datatype2string( h5sxdtype2datatype( ds_type )));
  if (ds_type<0) goto h5sx_read_dataset_as_native_array_error;

 /*
  * Get class of the dataset.
  * H5T_class_t H5Tget_class( hid_t dtype_id )
  */
  if (iodbg(IODBG_H5|IODBG_DEBUG2))
    printf(">> H5Tget_class(%d)",ds_type);
  ds_class=H5Tget_class(ds_type);
  if (iodbg(IODBG_H5|IODBG_DEBUG2))
    printf(" = %d\n",ds_class);
  if (ds_class<0) goto h5sx_read_dataset_as_native_array_error;

 /*
  * Get dataspace
  */
  if (iodbg(IODBG_H5|IODBG_DEBUG2))
    printf(">> H5Dget_space(%d)",ds_id);
  ds_space = H5Dget_space (ds_id);
  if (iodbg(IODBG_H5|IODBG_DEBUG2))
    printf(" = %d\n", ds_space);
  if (ds_space<0) goto h5sx_read_dataset_as_native_array_error;

 /*
  * Get rank of dataset, allocate ds_dims and ds_maxdims
  * By convention, the rank of a scalar dataspace is 0 (zero)
  */
  if (iodbg(IODBG_H5|IODBG_DEBUG2))
    printf(">> H5Sget_simple_extent_ndims(%d)",ds_space);
  ds_rank = H5Sget_simple_extent_ndims( ds_space );
  if (iodbg(IODBG_H5|IODBG_DEBUG2))
    printf(" = %ld\n",(long) ds_rank);
  if (ds_rank<0) goto h5sx_read_dataset_as_native_array_error;

  if (ds_rank>0) {
    if (iodbg(IODBG_H5|IODBG_DEBUG2))
      printf(">>  ds_dims = (hsize_t*) CALLOC(%ld,%ld)",(long) ds_rank,(long) sizeof(hsize_t));
    ds_dims = (hsize_t*) CALLOC(ds_rank,sizeof(hsize_t));
    if (iodbg(IODBG_H5|IODBG_DEBUG2)) {
      printf(" = %p\n",ds_dims);
    }
    if (iodbg(IODBG_H5|IODBG_DEBUG2))
      printf(">>  ds_maxdims = (hsize_t*) CALLOC(%ld,%ld)",(long) ds_rank,(long) sizeof(hsize_t));
    ds_maxdims = (hsize_t*) CALLOC(ds_rank,sizeof(hsize_t));
    if (iodbg(IODBG_H5|IODBG_DEBUG2)) {
      printf(" = %p\n",ds_maxdims);
    }

  } else {
    // scalar dataspace
    if (iodbg(IODBG_H5|IODBG_DEBUG2))
      printf(">>  ds_dims = (hsize_t*) CALLOC(%ld,%ld)",(long) 1,(long) sizeof(hsize_t));
    ds_dims = (hsize_t*) CALLOC(1,sizeof(hsize_t));
    if (iodbg(IODBG_H5|IODBG_DEBUG2)) {
      printf(" = %p\n",ds_dims);
    }
    if (iodbg(IODBG_H5|IODBG_DEBUG2))
      printf(">>  ds_maxdims = (hsize_t*) CALLOC(%ld,%ld)",(long) 1,(long) sizeof(hsize_t));
    ds_maxdims = (hsize_t*) CALLOC(1,sizeof(hsize_t));
    if (iodbg(IODBG_H5|IODBG_DEBUG2)) {
      printf(" = %p\n",ds_maxdims);
    }
  }

  if (iodbg(IODBG_H5|IODBG_DEBUG2))
    printf(">> H5Sget_simple_extent_dims(%d,%p,%p)",ds_space,ds_dims,ds_maxdims);
  ds_rank=H5Sget_simple_extent_dims (ds_space, ds_dims, ds_maxdims);
  if (iodbg(IODBG_H5|IODBG_DEBUG2)) {
    printf(" = %ld\n",(long) ds_rank);
    printf(">> ds_dims[%ld] = ",(long) ds_rank);
      h5io_fprint_dims(stdout, ds_rank, ds_dims);
    printf("\n");
    printf(">> ds_maxdims[%ld] = ",(long) ds_maxdims);
      h5io_fprint_dims(stdout, ds_rank, ds_maxdims);
    printf("\n");
  }

  if (ds_rank<0) goto h5sx_read_dataset_as_native_array_error;

  /*
   * Debug info about data set filters
   */
  if (iodbg(IODBG_H5|IODBG_DEBUG2)) {
 
    hid_t                dcpl_id=-1;
    int                  max_filter_number=0,filter_number=0;

   /*
    * Retrieve dataset creation property list.
    */
   dcpl_id = H5Dget_create_plist (ds_id);

   /*
    * Retrieve number of available filters 0 .. N-1
    * int H5Pget_nfilters(hid_t plist) 
    */
    max_filter_number = H5Pget_nfilters(dcpl_id);

    printf(">> Number of H5 dataset filters = %d\n",max_filter_number);

    for (filter_number=0;filter_number<max_filter_number;filter_number++) {
      char                 filter_name_buffer[H5SX_INFOBUFLEN];
      H5Z_filter_t         filter_id=-1;
      unsigned int         flags=0;
      size_t               cd_nelmts=0; /* Number of parameters */
      unsigned int         cd_values=0; /* value */

     /*
      * Retrieve and print the filter id, compression level and filter's name.
      * H5Z_filter_t H5Pget_filter(hid_t plist, unsigned int filter_number, 
      *                            unsigned int *flags, size_t *cd_nelmts, 
      *                            unsigned int *cd_values, size_t namelen, 
      *                            char name[], unsigned *filter_config );
      */
  
      filter_id = H5Pget_filter (dcpl_id, (unsigned) filter_number, &flags, &cd_nelmts,
                                 &cd_values, H5SX_INFOBUFLEN, filter_name_buffer, NULL);

      filter_name_buffer[H5SX_INFOBUFLEN-1]='\0'; // force null termination

      printf(">>   H5 dataset filter %d id = %d (%s)\n",filter_number,filter_id,filter_name_buffer);
    }

  } // IODBG_DEBUG2 (dataset filters)

  if (iodbg(IODBG_H5|IODBG_DEBUG2))
    printf(">> Reading array\n");

  if (iodbg(IODBG_H5|IODBG_DEBUG2))
    printf(">> H5Dget_storage_size(%d)",ds_id);
  ds_storage_size = H5Dget_storage_size(ds_id);
  if (iodbg(IODBG_H5|IODBG_DEBUG2))
    printf(" = %ld\n", (long) ds_storage_size);
  if (ds_storage_size<0) goto h5sx_read_dataset_as_native_array_error;

  if (iodbg(IODBG_H5|IODBG_DEBUG2))
    printf(">> H5Sget_simple_extent_npoints(%d)",ds_space);
  ds_npoints = H5Sget_simple_extent_npoints( ds_space );
  if (iodbg(IODBG_H5|IODBG_DEBUG2))
    printf(" = %ld\n", (long) ds_npoints);
  if (ds_npoints<0) goto h5sx_read_dataset_as_native_array_error;

  // determine native data type for reading
  if (iodbg(IODBG_H5|IODBG_DEBUG2))
    printf(">> H5Tget_native_type(%d,H5T_DIR_ASCEND)",ds_type);
  ds_nativetype = H5Tget_native_type( ds_type, H5T_DIR_ASCEND );
  if (iodbg(IODBG_H5|IODBG_DEBUG2))
    printf(" = %d (%s, %s)\n", ds_nativetype, h5sxdtype2classstr( infobuf, H5SX_INFOBUFLEN, ds_nativetype ),
      edf_datatype2string( h5sxdtype2datatype( ds_nativetype ))); 

  ds_memtype = H5Tcopy (ds_nativetype);
  // mem_space_id = H5S_ALL (later, read only a single frame)
  // file_space_id = H5S_ALL
  // xfer_plist_id = H5P_DEFAULT
  // buf for NATIVE data
  ds_datatype = h5sxdtype2datatype(ds_memtype);

  if (iodbg(IODBG_H5|IODBG_DEBUG2))
    printf(">> H5Tget_size(%d)",ds_memtype);
  ds_memitem = H5Tget_size (ds_memtype);
  if (iodbg(IODBG_H5|IODBG_DEBUG2))
    printf(" = %ld\n", (long) ds_memitem);

  ds_memsize=ds_memitem*ds_npoints; // memory size of array
  if (iodbg(IODBG_H5|IODBG_DEBUG2))
    printf(">> ds_memsize = %ld bytes\n",(long) ds_memsize);

// ds_memsize, ds_id, ds_memtype, 
//----------------------------------------open end

  // do not read binary data, if binary is NULL
  if (binary) {

   /*
    * Allocate memory for binary. This is an n-dimensional dataset.
    */

    ds_binary = (void *) MALLOC(ds_memsize);
    if (!ds_binary) goto h5sx_read_dataset_as_native_array_error;

   /*
    * Read the data.
    */
    // herr_t H5Dread( hid_t dataset_id, hid_t mem_type_id, hid_t mem_space_id, hid_t file_space_id, hid_t xfer_plist_id, void * buf )
    if (iodbg(IODBG_H5|IODBG_DEBUG2))
      printf(">>  H5Dread (%d, %d, H5S_ALL, H5S_ALL, H5P_DEFAULT, %p)",ds_id, ds_memtype, ds_binary);
    h5status = H5Dread (ds_id, ds_memtype, H5S_ALL, H5S_ALL, H5P_DEFAULT, ds_binary);
    if (iodbg(IODBG_H5|IODBG_DEBUG2))
      printf(" = %d\n",h5status);

  }

//----------------------------------------close begin
    if (iodbg(IODBG_H5|IODBG_DEBUG2))
      printf(">> H5Dclose(%d)\n",ds_id);
    H5Dclose(ds_id);  //==============

    if (iodbg(IODBG_H5|IODBG_DEBUG2))
      printf(">> H5Tclose(%d)\n",ds_type);
    H5Tclose(ds_type);

    if (iodbg(IODBG_H5|IODBG_DEBUG2))
      printf(">> H5Sclose(%d)\n",ds_space);
    H5Sclose(ds_space);

    if (iodbg(IODBG_H5|IODBG_DEBUG2))
      printf(">> H5Tclose(%d)\n",ds_memtype);
    H5Tclose(ds_memtype);

  if (datatype) *datatype=ds_datatype;
  if (itemsize) *itemsize=ds_memitem;
  if (rank) *rank=ds_rank;
  if (dims) *dims=ds_dims;
  if (maxdims) *maxdims=ds_maxdims;
  if (binary) *binary=(void *) ds_binary;
  if (binarysize) *binarysize=ds_memsize;

//----------------------------------------close end

  if (iodbg(IODBG_H5|IODBG_DEBUG1)) {
    printf("h5sx_read_dataset_as_native_array(loc_id=%d,key=%s,attributes=%p,datatype=%p,itemsize=%p,rank=%p,dims=%p,binary=%p,binarysize=%p)\n",
      loc_id, key, attributes, datatype, itemsize, rank, dims, binary, binarysize);
    printf("h5sx_read_dataset_as_native_array(...,datatype=%s,itemsize=%ld,rank=%ld,dims=",
      edf_datatype2string(ds_datatype),(long) ds_memitem,(long) ds_rank);
    h5io_fprint_dims(stdout, ds_rank, ds_dims);
    printf(",ds_binary=%p,binarysize=%ld (0) END\n",ds_binary,(long) ds_memsize);
  }

  return(0);

h5sx_read_dataset_as_native_array_error:

  if (iodbg(IODBG_H5|IODBG_DEBUG2))
    printf(">> H5Dclose(%d)\n",ds_id);
  H5Dclose(ds_id);

  if (iodbg(IODBG_H5|IODBG_DEBUG2))
    printf(">> H5Tclose(%d)\n",ds_type);
  H5Tclose(ds_type);

  if (iodbg(IODBG_H5|IODBG_DEBUG2))
    printf(">> H5Sclose(%d)\n",ds_space);
  H5Sclose(ds_space);

  if (iodbg(IODBG_H5|IODBG_DEBUG2))
    printf(">> H5Tclose(%d)\n",ds_memtype);
  H5Tclose(ds_memtype); 

  if (iodbg(IODBG_H5|IODBG_DEBUG2))
    printf(">> FREE(%p)\n",ds_dims);
  FREE(ds_dims);

  if (iodbg(IODBG_H5|IODBG_DEBUG2))
    printf(">> FREE(%p)\n",ds_maxdims);
  FREE(ds_maxdims);

  if (iodbg(IODBG_H5|IODBG_DEBUG2))
    printf(">> FREE(%p)\n",ds_binary);
  FREE(ds_binary);

  if (datatype) *datatype=InValidDType;
  if (itemsize) *itemsize=-1;
  if (rank) *rank=-1;
  if (dims) *dims=NULL;
  if (binarysize) *binarysize=(hsize_t) 0;

  printf("h5sx_read_dataset_as_native_array(loc_id=%d,key=%s,attributes=%p,datatype=%p,itemsize=%p,rank=%p,dims=%p,binary=%p,binarysize=%p) (-1) ERROR\n",
    loc_id, key, attributes, datatype, itemsize, rank, dims, binary, binarysize);

  return(-1);

} /* h5sx_read_dataset_as_native_array */

/*
 * Read dataset key as string array and its attributes
 * char *string[]
 * char **h5sx_read_number_as_string( hid_t loc_id, const char *key, H5sxListOfDSets *attributes, hsize_t *nframes )
 * The returned output value is allocated and must be released with h5io_free_values( value, nframes )
 * for avoiding resource leaks.
 */
char **h5sx_read_number_as_string( hid_t loc_id, const char *key, H5sxListOfDSets *attributes, hsize_t *nframes )
{
  char **string_array=NULL; // terminate with NULL!
  hsize_t string_array_len=1;

  int datatype;
  hsize_t itemsize, rank;
  hsize_t *dims=NULL, *maxdims=NULL;
  void *binary=NULL;
  hsize_t binarysize;
  hsize_t frames=0;

  hsize_t *actual_dims=NULL;
  hsize_t updated_dims[2];
  hsize_t actual_rank;

  int status=-1;

  if (iodbg(IODBG_H5|IODBG_DEBUG1))
    printf("h5sx_read_number_as_string(loc_id=%d,key=%s,attributes=%p,nframes=%p) BEGIN\n",loc_id, key, attributes, nframes);

  if ((status=h5sx_read_dataset_as_native_array( loc_id, key, NULL, &datatype, &itemsize, &rank, &dims, &maxdims, &binary, &binarysize)<0)) {
    goto h5sx_read_number_as_string_error;
  }

  if (rank>2) {
    if (iodbg(IODBG_H5|IODBG_VERBOSE)) {
      printf("WARNING: h5sx_read_number_as_string(key=%s): dataset ranks=%ld > 2 are not supported for strings, ignoring this key.\n",key,(long) rank);
    }
  } else {

    /*
     * Allocate array for string pointers
     */
    actual_dims = dims;
    actual_rank = rank;
    if (actual_rank>0) {
      frames=actual_dims[0]; // actual_dims[0] is the number of frames
  
      const char *name=NULL;
      if ((name=strrchr(key, (int) '/'))) name++; else name=key;
  
      if ( (!strncmp(name,"HS32",4))&&(actual_rank<2) ) {
        if (iodbg(IODBG_H5|IODBG_VERBOSE)) {
          printf("WARNING: Value of key %s has inconsistent array size for HS32 data.\n",key);
          printf("  Reading single values for %ld frames, need %ld values for a single frame\n",(long) frames,(long) frames);
        }
        updated_dims[0]=1;updated_dims[1]=dims[0];frames=1;actual_rank=2;
        actual_dims = updated_dims;
        if (iodbg(IODBG_H5|IODBG_VERBOSE))
          printf("  Corrected to %ld frame%s with %ld values.\n",(long) actual_dims[0],actual_dims[0]==1?"":"s",(long) actual_dims[1]);
      }
  
    } else {
      frames=1;
    }
    string_array_len=frames; // allocate 1 string pointer for each frame
  
    // initialize string_array elements with NULL and terminate with NULL
    if (iodbg(IODBG_H5|IODBG_DEBUG2))
      printf(">>  string_array = (char**) CALLOC(%ld,%ld)",(long)(string_array_len+1),(long) sizeof(char*));
    string_array = (char**) CALLOC (string_array_len+1,sizeof (char *));
    if (iodbg(IODBG_H5|IODBG_DEBUG2))
      printf(" = %p\n",string_array);
    if (!string_array) goto h5sx_read_number_as_string_error;
  
   /*
    * Write numbers to string_array[]
    * (linear filling of string_array)
    */
    if (actual_rank>0) {
      char **ppa;
      void *pnum;
      long iframe;
  
      ppa=string_array;
      pnum=binary;
      for (iframe=0;iframe<frames;iframe++) {
  
        if (actual_rank>1) {
          long idim;
          char buffer[H5SX_BUFLEN]={'\0'};
          hsize_t buflen=H5SX_BUFLEN;
  
          for (idim=0;idim<actual_dims[1];idim++) {
            char *value=NULL;
  
            value=(char *) h5io_number2string(datatype,pnum);
            if (idim>0) strlib_concat( buffer, buflen, buffer, "," );
            strlib_concat( buffer, buflen, buffer, value );
            FREE(value);
  
            pnum = (void *) ( (char*) pnum + itemsize );
          }
          if (actual_dims[1]>1) {
            strlib_concat( buffer, buflen, buffer, "]" );
            strlib_concat( buffer, buflen, "[", buffer );
          }
          *ppa=(char *) strlib_newstr(buffer);
        } else {
          *ppa=(char *) h5io_number2string(datatype,pnum);
          pnum = (void *) ( (char*) pnum + itemsize );
        }
        ppa++;
  
      }
  
    } else {
      *string_array=(char *) h5io_number2string(datatype,binary);
    }
  
  } // if rank ...

  FREE(dims); FREE(maxdims); FREE(binary);

  if (iodbg(IODBG_H5|IODBG_DEBUG1))
    printf("h5sx_read_number_as_string(loc_id=%d,key=%s,attributes=%p,frames=%ld) (%p) END\n",
      loc_id, key, attributes, (long) frames, string_array);

  if (nframes) *nframes=frames;

  return( string_array );

h5sx_read_number_as_string_error:

  FREE(dims); FREE(maxdims); FREE(binary);

  string_array = free_string_array( string_array, string_array_len );

  if (nframes) *nframes=0;

  printf("h5sx_read_number_as_string(loc_id=%d,key=%s,attributes=%p,frames=%ld) (NULL) ERROR\n",
    loc_id, key, attributes, (long) 0);

  return( NULL );

} /* h5sx_read_number_as_string */

/*
 * Read dataset key as string array and its attributes
 * char *string[]
 * char **h5sx_read_value_as_string( hid_t loc_id, const char *key, H5sxListOfDSets *attributes, hsize_t *nframes )
 * The output value is allocated and must be released with h5io_free_values( value, nframes ) 
 * for avoiding resource leaks.
 */
char **h5sx_read_value_as_string( hid_t loc_id, const char *key, H5sxListOfDSets *attributes, hsize_t *nframes )
{
  char infobuf[H5SX_INFOBUFLEN];

  hid_t ds_id=-1, ds_type=-1, ds_space=-1;
  htri_t ds_is_vls=-1; // htri_t: TRUE, FALSE or negative in case of failure
  H5T_cset_t ds_cset=-1;
  int ds_rank=-1;
  hsize_t *ds_dims=NULL;
  hsize_t *ds_maxdims=NULL;
  hsize_t ds_storage_size=0;

  hsize_t *actual_ds_dims=NULL;
  hsize_t updated_ds_dims[2];
  int actual_ds_rank=-1;

  char **string_array=NULL; // terminate with NULL!
  hsize_t string_array_len=1;

  hid_t ds_memtype=-1;
  hsize_t memsize=0;
  hsize_t ds_stringsize=0;

  H5T_class_t ds_class;

  herr_t h5status=0;

  char **ds_rdata=NULL;
  char *ds_data=NULL;
  hsize_t frames=0;

  if (iodbg(IODBG_H5|IODBG_DEBUG1)) 
    printf("h5sx_read_value_as_string(loc_id=%d,key=%s,attributes=%p,nframes=%p) BEGIN\n",loc_id, key, attributes, nframes);

  /*
   * Read dataset by name
   */
  // hid_t H5Dopen( hid_t loc_id, const char *name, hid_t dapl_id )
  if (iodbg(IODBG_H5|IODBG_DEBUG2)) 
    printf(">> ds_id = H5Dopen(%d,%s,%d)",loc_id,key,H5P_DEFAULT);
  ds_id = H5Dopen(loc_id,key,H5P_DEFAULT);
  if (iodbg(IODBG_H5|IODBG_DEBUG2)) 
    printf(" = %d\n",ds_id);
  if (ds_id<0) goto h5sx_read_value_as_string_error;

  /*
   * Read the attributes
   */
  if (attributes) {
    if (h5sx_attriblist_read( ds_id, attributes )<0) goto h5sx_read_value_as_string_error;
  }

 /*
  * Get datatype of the dataset
  */
  if (iodbg(IODBG_H5|IODBG_DEBUG2)) 
    printf(">> ds_type = H5Dget_type(%d)",ds_id);
  ds_type = H5Dget_type (ds_id);
  if (iodbg(IODBG_H5|IODBG_DEBUG2)) 
    printf(" = %d (%s, %s)\n", ds_type, h5sxdtype2classstr( infobuf, H5SX_INFOBUFLEN, ds_type ),
      edf_datatype2string( h5sxdtype2datatype( ds_type ))); 
  if (ds_type<0) goto h5sx_read_value_as_string_error;

 /*
  * Get class of the dataset.
  * H5T_class_t H5Tget_class( hid_t dtype_id )
  */
  if (iodbg(IODBG_H5|IODBG_DEBUG2)) 
    printf(">> H5Tget_class(%d)",ds_type);
  switch (ds_class=H5Tget_class(ds_type)) {

    case H5T_STRING:
      if (iodbg(IODBG_H5|IODBG_DEBUG2)) 
        printf(" = H5T_STRING\n");
    
     /*
      * What is the character set of the dataset value?
      */
      if (iodbg(IODBG_H5|IODBG_DEBUG2)) 
        printf(">> H5Tget_cset(%d)",ds_type);
      // H5T_cset_t H5Tget_cset( hid_t dtype_id )
      ds_cset = H5Tget_cset( ds_type );
      if (iodbg(IODBG_H5|IODBG_DEBUG2)) {
        switch(ds_cset) {
          case H5T_CSET_ASCII: printf(" = H5T_CSET_ASCII\n");
                               break;
          case H5T_CSET_UTF8 : printf(" = H5T_CSET_UTF8\n");
                               break;
          default: printf(" = unknown (%d)\n",ds_cset);
        }
      }

     /*
      * Is the dataset a variable length string?
      */
      if (iodbg(IODBG_H5|IODBG_DEBUG2)) 
        printf(">> H5Tis_variable_str(%d)",ds_type);
      ds_is_vls = H5Tis_variable_str( ds_type );
      if (iodbg(IODBG_H5|IODBG_DEBUG2)) 
        printf(" = %s (%d)\n",(ds_is_vls<0)?"Error":(ds_is_vls?"Yes":"No"),ds_is_vls);

     /*
      * Get dataspace
      */
      if (iodbg(IODBG_H5|IODBG_DEBUG2)) 
        printf(">> H5Dget_space(%d)",ds_id);
      ds_space = H5Dget_space (ds_id);
      if (iodbg(IODBG_H5|IODBG_DEBUG2)) 
        printf(" = %d\n", ds_space);
  
     /*
      * Get rank of dataset, allocate ds_dims and ds_maxdims
      * By convention, the rank of a scalar dataspace is 0 (zero)
      */
      if (iodbg(IODBG_H5|IODBG_DEBUG2)) 
        printf(">> H5Sget_simple_extent_ndims(%d)",ds_space);
      ds_rank = H5Sget_simple_extent_ndims( ds_space );
      if (iodbg(IODBG_H5|IODBG_DEBUG2))
        printf(" = %ld\n",(long) ds_rank);
      if (ds_rank<0) {
        printf("ERROR: h5sx_read_value_as_string->H5Sget_simple_extent_ndims: key=%s\n",key);
        goto h5sx_read_value_as_string_error;
      }

      if (ds_rank>2) {
        if (iodbg(IODBG_H5|IODBG_VERBOSE)) {
          printf("WARNING: h5sx_read_value_as_string(key=%s): dataset ranks=%ld > 2 are not supported for strings. Ignoring this key.\n",key,(long) ds_rank);
        }
      } else {

        if (ds_rank>0) {
          if (iodbg(IODBG_H5|IODBG_DEBUG2))
            printf(">>  ds_dims = (hsize_t*) CALLOC(%ld,%ld)",(long) ds_rank,(long) sizeof(hsize_t));
          ds_dims = (hsize_t*) CALLOC(ds_rank,sizeof(hsize_t));
          if (iodbg(IODBG_H5|IODBG_DEBUG2)) {
            printf(" = %p\n",ds_dims);
          }
          if (iodbg(IODBG_H5|IODBG_DEBUG2))
            printf(">>  ds_maxdims = (hsize_t*) CALLOC(%ld,%ld)",(long) ds_rank,(long) sizeof(hsize_t));
          ds_maxdims = (hsize_t*) CALLOC(ds_rank,sizeof(hsize_t));
          if (iodbg(IODBG_H5|IODBG_DEBUG2)) {
            printf(" = %p\n",ds_maxdims);
          }

        } else {
          // scalar dataspace
          if (iodbg(IODBG_H5|IODBG_DEBUG2))
            printf(">>  ds_dims = (hsize_t*) CALLOC(%ld,%ld)",(long) 1,(long) sizeof(hsize_t));
          ds_dims = (hsize_t*) CALLOC(1,sizeof(hsize_t));
          if (iodbg(IODBG_H5|IODBG_DEBUG2)) {
            printf(" = %p\n",ds_dims);
          }
          if (iodbg(IODBG_H5|IODBG_DEBUG2))
            printf(">>  ds_maxdims = (hsize_t*) CALLOC(%ld,%ld)",(long) 1,(long) sizeof(hsize_t));
          ds_maxdims = (hsize_t*) CALLOC(1,sizeof(hsize_t));
          if (iodbg(IODBG_H5|IODBG_DEBUG2)) {
            printf(" = %p\n",ds_maxdims);
          }
        }
  
        if (iodbg(IODBG_H5|IODBG_DEBUG2)) 
          printf(">> H5Sget_simple_extent_dims(%d,%p,%p)",ds_space,ds_dims,ds_maxdims);
        ds_rank=H5Sget_simple_extent_dims (ds_space, ds_dims, ds_maxdims);

        if (iodbg(IODBG_H5|IODBG_DEBUG2)) {
          printf(" = %ld\n",(long) ds_rank);
          printf(">> ds_dims[%ld] = ",(long) ds_rank);
            h5io_fprint_dims(stdout, ds_rank, (hsize_t *) ds_dims);
          printf("\n");
          printf(">> ds_maxdims[%ld] = ",(long) ds_rank);
            h5io_fprint_dims(stdout, ds_rank, (hsize_t *) ds_maxdims);
          printf("\n");
        }

        if (ds_rank<0) {
          printf("ERROR: h5sx_read_value_as_string->H5Sget_simple_extent_ndims(key=%s): ds_rank=%ld\n",key,(long) ds_rank);
          goto h5sx_read_value_as_string_error;
        }

        if (ds_rank>2) {
          printf("ERROR: h5sx_read_value_as_string->H5Sget_simple_extent_ndims(key=%s): dataset ranks=%ld > 2\n",key,(long) ds_rank);
          goto h5sx_read_value_as_string_error;
        }

       /*
        * HDF routines do not convert automatically between variable length strings
        * and fixed length strings. It must be done manually.
        */
        switch (ds_is_vls) {
          case 1: // variable length strings
            if (iodbg(IODBG_H5|IODBG_DEBUG2))
              printf(">> Reading variable length string\n");

            if (iodbg(IODBG_H5|IODBG_DEBUG2))
              printf(">>  ds_rdata = (char **) CALLOC(%ld,%ld)",(long) ((ds_dims[0]>0)?ds_dims[0]:1),(long) sizeof(char *));
            ds_rdata = (char **) CALLOC (((ds_dims[0]>0)?ds_dims[0]:1),sizeof (char *));
            if (iodbg(IODBG_H5|IODBG_DEBUG2))
              printf(" = %p\n",ds_rdata);
            if (!ds_rdata) goto h5sx_read_value_as_string_error;

           /*
            * Create the memory datatype.
            */
            if (iodbg(IODBG_H5|IODBG_DEBUG2)) 
              printf(">> H5Tcopy(%d)",H5T_C_S1);
            ds_memtype = H5Tcopy (H5T_C_S1);
            if (iodbg(IODBG_H5|IODBG_DEBUG2)) 
              printf(" = %d\n", ds_memtype);

            if (iodbg(IODBG_H5|IODBG_DEBUG2))
              printf(">> H5Tset_cset(%d,H5T_CSET_ASCII)\n",ds_memtype);
            h5status=H5Tset_cset(ds_memtype,H5T_CSET_ASCII);
            if (h5status<0) {
              printf("ERROR: h5sx_read_value_as_string->H5Tset_cset(key=%s)\n",key);
              goto h5sx_read_value_as_string_error;
            }

            if (iodbg(IODBG_H5|IODBG_DEBUG2)) 
              printf(">> H5Tset_size(%d,%s)\n",ds_memtype,"H5T_VARIABLE");
            h5status = H5Tset_size (ds_memtype, H5T_VARIABLE);
            if (h5status<0) {
              printf("ERROR: h5sx_read_value_as_string->H5Tset_size(key=%s)\n",key);
              goto h5sx_read_value_as_string_error;
            }

           /*
            * Read variable length strings. 
            * ds_rdata is a pointer to a pointer array
            */
            if (iodbg(IODBG_H5|IODBG_DEBUG2)) 
              printf(">> H5Dread(%d,%d,%d,%d,%d,%p)\n",ds_id,ds_memtype,H5S_ALL,H5S_ALL,H5P_DEFAULT,ds_rdata);
            h5status = H5Dread(ds_id,ds_memtype,H5S_ALL,H5S_ALL,H5P_DEFAULT,ds_rdata);
            if (h5status<0) {
              printf("ERROR: h5sx_read_value_as_string->H5Dread(key=%s)\n",key);
              goto h5sx_read_value_as_string_error;
            }

            // get ds_rdata

            /*
             * Allocate array for string pointers
             */
            actual_ds_dims = ds_dims;
            actual_ds_rank = ds_rank;
            if (actual_ds_rank>0) {
              frames=ds_dims[0]; // ds_dims[0] is the number of frames

              const char *name=NULL;
              if ((name=strrchr(key, (int) '/'))) name++; else name=key;

              if ( (!strncmp(name,"HS32",4))&&(actual_ds_rank<2) ) {
                if (iodbg(IODBG_H5|IODBG_VERBOSE)) {
                  printf("WARNING: Value of key %s has inconsistent array size for HS32 data.\n",key);
                  printf("  Reading single values for %ld frames, need %ld values for a single frame\n",(long) frames,(long) frames);
                }
                updated_ds_dims[0]=1;updated_ds_dims[1]=ds_dims[0];frames=1;actual_ds_rank=2;
                actual_ds_dims = updated_ds_dims;
                if (iodbg(IODBG_H5|IODBG_VERBOSE))
                  printf("  Corrected to %ld frame%s with %ld values.\n",(long) actual_ds_dims[0],actual_ds_dims[0]==1?"":"s",(long) actual_ds_dims[1]);
              }

            } else {
              frames=1;
            }
            string_array_len=frames; // allocate 1 string pointer for each frame

            // initialize string_array elements with NULL and terminate with NULL
            if (iodbg(IODBG_H5|IODBG_DEBUG2))
              printf(">>  string_array = (char**) CALLOC(%ld,%ld)",(long)(string_array_len+1),(long) sizeof(char*));
            string_array = (char**) CALLOC (string_array_len+1,sizeof (char *));
            if (iodbg(IODBG_H5|IODBG_DEBUG2))
              printf(" = %p\n",string_array);
            if (!string_array) goto h5sx_read_value_as_string_error;

           /*
            * Write numbers to string_array[]
            * (linear filling of string_array)
            */
            if (actual_ds_rank>0) {
              char **ppa;
              char **ppnum;
              long iframe;

              ppnum = ds_rdata;
              ppa=string_array;
              for (iframe=0;iframe<frames;iframe++) {

                if (actual_ds_rank>1) {
                  long idim;
                  char buffer[H5SX_BUFLEN]={'\0'};
                  hsize_t buflen=H5SX_BUFLEN;

                  for (idim=0;idim<actual_ds_dims[1];idim++) {

                    if (idim>0) strlib_concat( buffer, buflen, buffer, "','" );
                    strlib_concat( buffer, buflen, buffer, *ppnum );

                    ppnum++;
                  }
                  if (actual_ds_dims[1]>1) {
                    strlib_concat( buffer, buflen, buffer, "']" );
                    strlib_concat( buffer, buflen, "['", buffer );
                  }
                  *ppa=(char *) strlib_newstr(buffer);
                } else {
                  *ppa=(char *) strlib_newstr(*ppnum);
                   ppnum++;
                }
                if (iodbg(IODBG_H5|IODBG_DEBUG1)) {
                  printf(">> string_array[%ld] = \"%s\"\n",(long) iframe,*ppa);
                }
                ppa++;

              }

            } else {
              *string_array=(char *) ds_rdata[0];
              if (iodbg(IODBG_H5|IODBG_DEBUG1)) {
                printf(">> *string_array = \"%s\"\n",*string_array);
              }
            }

            if (iodbg(IODBG_H5|IODBG_DEBUG2)) 
              printf(">> H5Dvlen_reclaim(%d,%d,%s,%p)\n",ds_memtype, ds_space, "H5P_DEFAULT", ds_rdata);
            h5status=H5Dvlen_reclaim (ds_memtype, ds_space, H5P_DEFAULT, ds_rdata);
            if (h5status<0) {
              printf("ERROR: h5sx_read_value_as_string->H5Dvlen_reclaim(key=%s)\n",key);
              goto h5sx_read_value_as_string_error;
            }

            if (iodbg(IODBG_H5|IODBG_DEBUG2))
              printf(">> FREE(%p)\n",ds_rdata);
            FREE(ds_rdata);

            if (ds_memtype>=0) {
              if (iodbg(IODBG_H5|IODBG_DEBUG2))
                printf(">> H5Tclose(%d)\n",ds_memtype);
              H5Tclose(ds_memtype); ds_memtype=-1;
            }

            break;

          case 0: // fixed length string
            if (iodbg(IODBG_H5|IODBG_DEBUG2))
              printf(">> Reading fixed length string\n");

            if (iodbg(IODBG_H5|IODBG_DEBUG2)) 
              printf(">> H5Dget_storage_size(%d)",ds_id);
            ds_storage_size = H5Dget_storage_size(ds_id);
            if (iodbg(IODBG_H5|IODBG_DEBUG2)) 
              printf(" = %ld\n", (long) ds_storage_size);

            if (iodbg(IODBG_H5|IODBG_DEBUG2)) 
              printf(">> H5Tget_size(%d)",ds_type);
            ds_stringsize = H5Tget_size (ds_type);
            if (iodbg(IODBG_H5|IODBG_DEBUG2))
              printf(" = %ld + 1\n", (long) ds_stringsize);
            ds_stringsize++;

            memsize=ds_stringsize;
            if (ds_rank>0) {
              long irank;
              for (irank=0;irank<ds_rank;irank++) {
                memsize*=ds_dims[irank];
              }
            }

            if (iodbg(IODBG_H5|IODBG_DEBUG2))
              printf(">>  ds_data = (char *) CALLOC(%ld,%ld)",(long)memsize,(long) sizeof(char));
            ds_data = (char *) CALLOC (memsize,sizeof(char));
            if (iodbg(IODBG_H5|IODBG_DEBUG2))
              printf(" = %p\n",ds_data);
            if (!ds_data) goto h5sx_read_value_as_string_error;

           /*
            * Create the memory datatype.
            */
            if (iodbg(IODBG_H5|IODBG_DEBUG2))
              printf(">> H5Tcopy(%d)",H5T_C_S1);
            ds_memtype = H5Tcopy (H5T_C_S1);
            if (iodbg(IODBG_H5|IODBG_DEBUG2)) 
              printf(" = %d\n", ds_memtype);

            if (iodbg(IODBG_H5|IODBG_DEBUG2))
              printf(">> H5Tset_cset(%d,H5T_CSET_ASCII)\n",ds_memtype);
            h5status=H5Tset_cset(ds_memtype,H5T_CSET_ASCII);
            if (h5status<0) {
              printf("ERROR: h5sx_read_value_as_string->H5Tset_cset(key=%s)\n",key);
              goto h5sx_read_value_as_string_error;
            }

            if (iodbg(IODBG_H5|IODBG_DEBUG2))
              printf(">> H5Tset_size(%d,%ld)\n",ds_memtype,(long) ds_stringsize);
            h5status = H5Tset_size (ds_memtype, ds_stringsize);
            if (h5status<0) {
              printf("ERROR: h5sx_read_value_as_string->H5Tset_size(key=%s)\n",key);
              goto h5sx_read_value_as_string_error;
            }

           /*
            * Read fixed length strings. 
            * ds_data is a pointer to a data array
            */
            if (iodbg(IODBG_H5|IODBG_DEBUG2))
              printf(">> H5Dread(%d,%d,%d,%d,%d,%p)\n",ds_id,ds_memtype,H5S_ALL,H5S_ALL,H5P_DEFAULT,ds_data);
            h5status = H5Dread(ds_id,ds_memtype,H5S_ALL,H5S_ALL,H5P_DEFAULT,ds_data);
            if (h5status<0) {
              printf("ERROR: h5sx_read_value_as_string->H5Dread(key=%s)\n",key);
              goto h5sx_read_value_as_string_error;
            }

            // get ds_data

            /*
             * Allocate array for string pointers
             */
            actual_ds_dims = ds_dims;
            actual_ds_rank = ds_rank;
            if (actual_ds_rank>0) {
              frames=ds_dims[0]; // ds_dims[0] is the number of frames

              const char *name=NULL;
              if ((name=strrchr(key, (int) '/'))) name++; else name=key;

              if ( (!strncmp(name,"HS32",4))&&(actual_ds_rank<2) ) {
                if (iodbg(IODBG_H5|IODBG_VERBOSE)) {
                  printf("WARNING: Value of key %s has inconsistent array size for HS32 data.\n",key);
                  printf("  Reading single values for %ld frames, need %ld values for a single frame\n",(long) frames,(long) frames);
                }
                updated_ds_dims[0]=1;updated_ds_dims[1]=ds_dims[0];frames=1;actual_ds_rank=2;
                actual_ds_dims = updated_ds_dims;
                if (iodbg(IODBG_H5|IODBG_VERBOSE))
                  printf("  Corrected to %ld frame%s with %ld values.\n",(long) actual_ds_dims[0],actual_ds_dims[0]==1?"":"s",(long) actual_ds_dims[1]);
              }

            } else {
              frames=1;
            }
            string_array_len=frames; // allocate 1 string pointer for each frame

            // initialize string_array elements with NULL and terminate with NULL
            if (iodbg(IODBG_H5|IODBG_DEBUG2))
              printf(">>  string_array = (char**) CALLOC(%ld,%ld)",(long)(string_array_len+1),(long) sizeof(char*));
            string_array = (char**) CALLOC (string_array_len+1,sizeof (char *));
            if (iodbg(IODBG_H5|IODBG_DEBUG2))
              printf(" = %p\n",string_array);
            if (!string_array) goto h5sx_read_value_as_string_error;

           /*
            * Write numbers to string_array[]
            * (linear filling of string_array)
            */
            if (actual_ds_rank>0) {
              char **ppa;
              char *pnum;
              long iframe;

              pnum = ds_data;
              ppa=string_array;
              for (iframe=0;iframe<frames;iframe++) {

                if (actual_ds_rank>1) {
                  long idim;
                  char buffer[H5SX_BUFLEN]={'\0'};
                  hsize_t buflen=H5SX_BUFLEN;

                  for (idim=0;idim<actual_ds_dims[1];idim++) {

                    if (idim>0) strlib_concat( buffer, buflen, buffer, "','" );
                    strlib_concat( buffer, buflen, buffer, pnum );

                    pnum+=ds_stringsize;
                  }
                  if (actual_ds_dims[1]>1) {
                    strlib_concat( buffer, buflen, buffer, "']" );
                    strlib_concat( buffer, buflen, "['", buffer );
                  }
                  *ppa=(char *) strlib_newstr(buffer);
                } else {
                  *ppa=(char *) strlib_newstr(pnum);
                  pnum+=ds_stringsize;
                }
                if (iodbg(IODBG_H5|IODBG_DEBUG1)) {
                  printf(">> string_array[%ld] = \"%s\"\n",(long) iframe,*ppa);
                }
                ppa++;

              }

            } else {
              *string_array=(char *) strlib_newstr(ds_data);
              if (iodbg(IODBG_H5|IODBG_DEBUG1)) {
                printf(">> *string_array = \"%s\"\n",*string_array);
              }
            }

            if (iodbg(IODBG_H5|IODBG_DEBUG2))
              printf(">> H5Dvlen_reclaim(%d,%d,%s,%p)\n",ds_memtype, ds_space, "H5P_DEFAULT", ds_data);
            h5status=H5Dvlen_reclaim (ds_memtype, ds_space, H5P_DEFAULT, ds_data);
            if (h5status<0) {
              printf("ERROR: h5sx_read_value_as_string->H5Dvlen_reclaim(key=%s)\n",key);
              goto h5sx_read_value_as_string_error;
            }

            if (iodbg(IODBG_H5|IODBG_DEBUG2))
              printf(">> FREE(%p)\n",ds_data);
            FREE(ds_data);

            if (ds_memtype>=0) {
              if (iodbg(IODBG_H5|IODBG_DEBUG2))
                printf(">> H5Tclose(%d)\n",ds_memtype);
              H5Tclose(ds_memtype); ds_memtype=-1;
            }

            break;

          default: goto h5sx_read_value_as_string_error;
        }

        FREE(ds_dims);
        FREE(ds_maxdims);

      }

      if (iodbg(IODBG_H5|IODBG_DEBUG2))
        printf(">> H5Sclose(%d)\n",ds_space);
      H5Sclose( ds_space ); ds_space=-1;

      if (iodbg(IODBG_H5|IODBG_DEBUG2))
        printf(">> H5Tclose(%d)\n",ds_type);
      H5Tclose( ds_type ); ds_type=-1;
      if (iodbg(IODBG_H5|IODBG_DEBUG2))
        printf(">> H5Dclose(%d)\n",ds_id);
      H5Dclose( ds_id ); ds_id=-1;

      break;

    case H5T_INTEGER:
      if (iodbg(IODBG_H5|IODBG_DEBUG2))
        printf(" = H5T_INTEGER\n");

      if (iodbg(IODBG_H5|IODBG_DEBUG2))
        printf(">> H5Tclose(%d)\n",ds_type);
      H5Tclose( ds_type ); ds_type=-1;
      if (iodbg(IODBG_H5|IODBG_DEBUG2))
        printf(">> H5Dclose(%d)\n",ds_id);
      H5Dclose( ds_id ); ds_id=-1;

      string_array = h5sx_read_number_as_string( loc_id, key, NULL, &frames );

      break;
    case H5T_FLOAT:
      if (iodbg(IODBG_H5|IODBG_DEBUG2))
        printf(" = H5T_FLOAT\n");

      if (iodbg(IODBG_H5|IODBG_DEBUG2))
        printf(">> H5Tclose(%d)\n",ds_type);
      H5Tclose( ds_type ); ds_type=-1;
      if (iodbg(IODBG_H5|IODBG_DEBUG2))
        printf(">> H5Dclose(%d)\n",ds_id);
      H5Dclose( ds_id ); ds_id=-1;

      string_array = h5sx_read_number_as_string( loc_id, key, NULL, &frames );

      break;
    case H5T_BITFIELD:
      if (iodbg(IODBG_H5|IODBG_DEBUG2))
        printf(" = H5T_BITFIELD\n");

      if (iodbg(IODBG_H5|IODBG_DEBUG2))
        printf(">>  string_array = (char**) CALLOC(%ld,%ld)",(long)(frames+1),(long) sizeof(char*));
      string_array = (char**) CALLOC (frames+1,sizeof (char *));
      if (iodbg(IODBG_H5|IODBG_DEBUG2))
        printf(" = %p\n",string_array);
      if (!string_array) goto h5sx_read_value_as_string_error;

      if (iodbg(IODBG_H5|IODBG_DEBUG2))
        printf(">> H5Tclose(%d)\n",ds_type);
      H5Tclose( ds_type ); ds_type=-1;
      if (iodbg(IODBG_H5|IODBG_DEBUG2))
        printf(">> H5Dclose(%d)\n",ds_id);
      H5Dclose( ds_id ); ds_id=-1;

      *string_array = strlib_newstr("(H5T_BITFIELD)");
      break;
    case H5T_OPAQUE:
      if (iodbg(IODBG_H5|IODBG_DEBUG2))
        printf(" = H5T_OPAQUE\n");

      if (iodbg(IODBG_H5|IODBG_DEBUG2))
        printf(">>  string_array = (char**) CALLOC(%ld,%ld)",(long)(frames+1),(long) sizeof(char*));
      string_array = (char**) CALLOC (frames+1,sizeof (char *));
      if (iodbg(IODBG_H5|IODBG_DEBUG2))
        printf(" = %p\n",string_array);
      if (!string_array) goto h5sx_read_value_as_string_error;

      if (iodbg(IODBG_H5|IODBG_DEBUG2))
        printf(">> H5Tclose(%d)\n",ds_type);
      H5Tclose( ds_type ); ds_type=-1;
      if (iodbg(IODBG_H5|IODBG_DEBUG2))
        printf(">> H5Dclose(%d)\n",ds_id);
      H5Dclose( ds_id ); ds_id=-1;

      *string_array = strlib_newstr("(H5T_OPAQUE)");
      break;
    case H5T_COMPOUND:
      if (iodbg(IODBG_H5|IODBG_DEBUG2))
        printf(" = H5T_COMPOUND\n");

      if (iodbg(IODBG_H5|IODBG_DEBUG2))
        printf(">>  string_array = (char**) CALLOC(%ld,%ld)",(long)(frames+1),(long) sizeof(char*));
      string_array = (char**) CALLOC (frames+1,sizeof (char *));
      if (iodbg(IODBG_H5|IODBG_DEBUG2))
        printf(" = %p\n",string_array);
      if (!string_array) goto h5sx_read_value_as_string_error;

      if (iodbg(IODBG_H5|IODBG_DEBUG2))
        printf(">> H5Tclose(%d)\n",ds_type);
      H5Tclose( ds_type ); ds_type=-1;
      if (iodbg(IODBG_H5|IODBG_DEBUG2))
        printf(">> H5Dclose(%d)\n",ds_id);
      H5Dclose( ds_id ); ds_id=-1;

      *string_array = strlib_newstr("(H5T_COMPOUND)");
      break;
    case H5T_REFERENCE:
      if (iodbg(IODBG_H5|IODBG_DEBUG2))
        printf(" = H5T_REFERENCE\n");

      if (iodbg(IODBG_H5|IODBG_DEBUG2))
        printf(">>  string_array = (char**) CALLOC(%ld,%ld)",(long)(frames+1),(long) sizeof(char*));
      string_array = (char**) CALLOC (frames+1,sizeof (char *));
      if (iodbg(IODBG_H5|IODBG_DEBUG2))
        printf(" = %p\n",string_array);
      if (!string_array) goto h5sx_read_value_as_string_error;

      if (iodbg(IODBG_H5|IODBG_DEBUG2))
        printf(">> H5Tclose(%d)\n",ds_type);
      H5Tclose( ds_type ); ds_type=-1;
      if (iodbg(IODBG_H5|IODBG_DEBUG2))
        printf(">> H5Dclose(%d)\n",ds_id);
      H5Dclose( ds_id ); ds_id=-1;

      *string_array = strlib_newstr("(H5T_REFERENCE)");
      break;
    case H5T_ENUM:
      if (iodbg(IODBG_H5|IODBG_DEBUG2))
        printf(" = H5T_ENUM\n");

      if (iodbg(IODBG_H5|IODBG_DEBUG2))
        printf(">>  string_array = (char**) CALLOC(%ld,%ld)",(long)(frames+1),(long) sizeof(char*));
      string_array = (char**) CALLOC (frames+1,sizeof (char *));
      if (iodbg(IODBG_H5|IODBG_DEBUG2))
        printf(" = %p\n",string_array);
      if (!string_array) goto h5sx_read_value_as_string_error;

      if (iodbg(IODBG_H5|IODBG_DEBUG2))
        printf(">> H5Tclose(%d)\n",ds_type);
      H5Tclose( ds_type ); ds_type=-1;
      if (iodbg(IODBG_H5|IODBG_DEBUG2))
        printf(">> H5Dclose(%d)\n",ds_id);
      H5Dclose( ds_id ); ds_id=-1;

      *string_array = strlib_newstr("(H5T_ENUM)");
      break;
    case H5T_VLEN:
      if (iodbg(IODBG_H5|IODBG_DEBUG2))
        printf(" = H5T_VLEN\n");

      if (iodbg(IODBG_H5|IODBG_DEBUG2))
        printf(">>  string_array = (char**) CALLOC(%ld,%ld)",(long)(frames+1),(long) sizeof(char*));
      string_array = (char**) CALLOC (frames+1,sizeof (char *));
      if (iodbg(IODBG_H5|IODBG_DEBUG2))
        printf(" = %p\n",string_array);
      if (!string_array) goto h5sx_read_value_as_string_error;

      if (iodbg(IODBG_H5|IODBG_DEBUG2))
        printf(">> H5Tclose(%d)\n",ds_type);
      H5Tclose( ds_type ); ds_type=-1;
      if (iodbg(IODBG_H5|IODBG_DEBUG2))
        printf(">> H5Dclose(%d)\n",ds_id);
      H5Dclose( ds_id ); ds_id=-1;

      *string_array = strlib_newstr("(H5T_VLEN)");
      break;
    case H5T_ARRAY :
      if (iodbg(IODBG_H5|IODBG_DEBUG2))
        printf(" = H5T_ARRAY\n");

      if (iodbg(IODBG_H5|IODBG_DEBUG2))
        printf(">>  string_array = (char**) CALLOC(%ld,%ld)",(long)(frames+1),(long) sizeof(char*));
      string_array = (char**) CALLOC (frames+1,sizeof (char *));
      if (iodbg(IODBG_H5|IODBG_DEBUG2))
        printf(" = %p\n",string_array);
      if (!string_array) goto h5sx_read_value_as_string_error;

      if (iodbg(IODBG_H5|IODBG_DEBUG2))
        printf(">> H5Tclose(%d)\n",ds_type);
      H5Tclose( ds_type ); ds_type=-1;
      if (iodbg(IODBG_H5|IODBG_DEBUG2))
        printf(">> H5Dclose(%d)\n",ds_id);
      H5Dclose( ds_id ); ds_id=-1;

      *string_array = strlib_newstr("(H5T_ARRAY)");
      break;
    default:
      if (iodbg(IODBG_H5|IODBG_DEBUG2))
        printf(" = %d (unsupported dataset class)\n",ds_class);

      if (iodbg(IODBG_H5|IODBG_DEBUG2))
        printf(">>  string_array = (char**) CALLOC(%ld,%ld)",(long)(frames+1),(long) sizeof(char*));
      string_array = (char**) CALLOC (frames+1,sizeof (char *));
      if (iodbg(IODBG_H5|IODBG_DEBUG2))
        printf(" = %p\n",string_array);
      if (!string_array) goto h5sx_read_value_as_string_error;

      if (iodbg(IODBG_H5|IODBG_DEBUG2))
        printf(">> H5Tclose(%d)\n",ds_type);
      H5Tclose( ds_type ); ds_type=-1;
      if (iodbg(IODBG_H5|IODBG_DEBUG2))
        printf(">> H5Dclose(%d)\n",ds_id);
      H5Dclose( ds_id ); ds_id=-1;

      *string_array = strlib_newstr("(unsupported dataset class)");
      break;
  }

  if (nframes) *nframes=frames;

  if (iodbg(IODBG_H5|IODBG_DEBUG1)) 
    printf("h5sx_read_value_as_string(loc_id=%d,key='%s',attributes=%p,nframes=%ld) (%p) END\n",
      loc_id, key, attributes, (long) frames, string_array);

  return(string_array);

h5sx_read_value_as_string_error:

  if (ds_rdata) {
    if (iodbg(IODBG_H5|IODBG_DEBUG2))
      printf(">> H5Dvlen_reclaim(%d,%d,%s,%p)\n",ds_memtype, ds_space, "H5P_DEFAULT", ds_rdata);
    H5Dvlen_reclaim (ds_memtype, ds_space, H5P_DEFAULT, ds_rdata);
    if (iodbg(IODBG_H5|IODBG_DEBUG2))
      printf(">> FREE(%p)\n",ds_rdata);
    FREE( ds_rdata );
  }
  if (ds_data) {
    if (iodbg(IODBG_H5|IODBG_DEBUG2))
      printf(">> H5Dvlen_reclaim(%d,%d,%s,%p)\n",ds_memtype, ds_space, "H5P_DEFAULT", ds_data);
    H5Dvlen_reclaim (ds_memtype, ds_space, H5P_DEFAULT, ds_data);
    if (iodbg(IODBG_H5|IODBG_DEBUG2))
      printf(">> FREE(%p)\n",ds_data);
    FREE( ds_data );
  }

  if (iodbg(IODBG_H5|IODBG_DEBUG2))
    printf(">> H5Tclose(%d)\n",ds_memtype);
  H5Tclose(ds_memtype); ds_memtype=-1;

  if (iodbg(IODBG_H5|IODBG_DEBUG2))
    printf(">> FREE(%p)\n",ds_dims);
  FREE( ds_dims );
  if (iodbg(IODBG_H5|IODBG_DEBUG2))
    printf(">> FREE(%p)\n",ds_maxdims);
  FREE( ds_maxdims );
  if (iodbg(IODBG_H5|IODBG_DEBUG2))
    printf(">> H5Sclose(%d)\n",ds_space);
  H5Sclose( ds_space ); ds_space=-1;
  if (iodbg(IODBG_H5|IODBG_DEBUG2))
    printf(">> H5Tclose(%d)\n",ds_type);
  H5Tclose( ds_type ); ds_type=-1;
  if (iodbg(IODBG_H5|IODBG_DEBUG2))
    printf(">> H5Dclose(%d)\n",ds_id);
  H5Dclose( ds_id ); ds_id=-1;
  if (iodbg(IODBG_H5|IODBG_DEBUG2))
    printf(">> free_string_array(%p)\n",string_array);
  free_string_array(string_array,-1);

  if (nframes) *nframes=0;

  printf("h5sx_read_value_as_string(loc_id=%d,key=%s,attributes=%p,nframes=%ld) (%p) ERROR\n",
    loc_id, key, attributes, (long) 0, NULL);

  return(NULL);

} // h5sx_read_value_as_string

/*
 * Read dataset key as string and its attributes.
 */
const char *h5sx_read_dataset_as_string( char buffer[], hsize_t buflen, hid_t loc_id, const char *key, H5sxListOfDSets *attributes )
{
  hsize_t nframes=0;
  char **string_array=NULL;
  char* value=NULL;

  string_array=h5sx_read_value_as_string(loc_id,key,attributes,&nframes);

  if (nframes>0) {
    value=buffer;
    strncpy(buffer,string_array[0],buflen-1);
    buffer[buflen-1]='\0';
  } else {
    value=NULL;
  }

  free_string_array(string_array,nframes); 

  return( value );

} /* h5sx_read_dataset_as_string */

/*
 * hid_t h5sx_dataset_create(hid_t loc_id, const char *name, hsize_t rank, hsize_t maxdims[], hid_t datatype_id,
 *                         hid_t filltype_id, const void *pfillvalue, int deflate_level, H5sxListOfDSets *attributes );
 */
hid_t h5sx_dataset_create(hid_t loc_id, const char *name, hsize_t rank, hsize_t maxdims[], hid_t datatype_id, 
                        hid_t filltype_id, const void *pfillvalue, int deflate_level, H5sxListOfDSets *attributes )
{
  const char *attr_key=NULL;
  const char *attr_val=NULL;
  H5sxDset *stringdset; 

  hid_t dataprop=-1, dataspace=-1, dataset=-1, datatype=-1;
  hsize_t *dataspacesize=NULL, *chunk=NULL;
  herr_t h5status=-1;

  if (iodbg(IODBG_H5|IODBG_DEBUG1)) {
    printf("h5sx_dataset_create(loc_id=%d,name=%s,rank=%ld,maxdims=%p,datatype_id=%d,filltype_id=%d,pfillvalue=%p,deflate_level=%d,attributes=%p) BEGIN\n",
    loc_id,name,(long)rank,maxdims,datatype_id,filltype_id,pfillvalue,deflate_level,attributes);
  }

  dataspacesize = (hsize_t *) MALLOC(sizeof(hsize_t)*rank);
  if ( !dataspacesize ) goto h5sx_dataset_create_error;

  chunk = (hsize_t *) MALLOC(sizeof(hsize_t)*rank);
  if ( !chunk ) goto h5sx_dataset_create_error;

  datatype = H5Tcopy(datatype_id);
  if ( datatype<0 ) goto h5sx_dataset_create_error;

  // Create dataspace (initially empty)
  dataspacesize[0] = 0;
  dataspacesize[1] = 0;
  dataspacesize[2] = 0;

  if (iodbg(IODBG_H5|IODBG_DEBUG2)) H5SXPRINT( rank, ">> dataspacesize", dataspacesize );
  if (iodbg(IODBG_H5|IODBG_DEBUG2)) H5SXPRINT( rank, ">> maxdims", maxdims );

  dataspace = H5Screate_simple(rank, dataspacesize, maxdims);
  if (dataspace<0) goto h5sx_dataset_create_error;

  // Create chunked dataset
  dataprop = H5Pcreate(H5P_DATASET_CREATE);
  if (dataprop<0) goto h5sx_dataset_create_error;

  // herr_t H5Pset_layout( hid_t plist, H5D_layout_t layout )
  // herr_t H5Pset_chunk(hid_t plist, int ndims, const hsize_t * dim )
  chunk[0] = 1; // dims[0];
  chunk[1] = maxdims[1];
  chunk[2] = maxdims[2];
  if (iodbg(IODBG_H5|IODBG_DEBUG2)) H5SXPRINT( rank, ">> chunk", chunk );
  h5status = H5Pset_chunk(dataprop, rank, chunk);
  if (h5status<0) goto h5sx_dataset_create_error;

  if (deflate_level>0) {
   /*  
    * Set ZLIB / DEFLATE Compression using compression level deflate_level (e.g. 6).
    */
    if (iodbg(IODBG_H5|IODBG_DEBUG2)) printf(">> H5Pset_deflate(dataprop,%d)\n",deflate_level);
    h5status = H5Pset_deflate (dataprop, deflate_level);
    if (h5status<0) goto h5sx_dataset_create_error;
  }

  if ((pfillvalue)&&(filltype_id>=0)) {
    if (iodbg(IODBG_H5|IODBG_DEBUG2)) printf(">> H5Pset_fill_value(...)\n");
    // herr_t H5Pset_fill_value(hid_t dcpl_id, hid_t type_id, const void *value);
    h5status = H5Pset_fill_value(dataprop,filltype_id,pfillvalue);
    if (h5status<0) goto h5sx_dataset_create_error;
  }

  // hid_t H5Dcreate( hid_t loc_id, const char *name, hid_t dtype_id, hid_t space_id, hid_t lcpl_id, hid_t dcpl_id, hid_t dapl_id  )
  dataset = H5Dcreate (loc_id, name, datatype, dataspace, H5P_DEFAULT, dataprop, H5P_DEFAULT);
  if (dataset<0) goto h5sx_dataset_create_error;

  // write attributes
  stringdset = h5sx_get_first_dset(attributes,&attr_key, &attr_val, NULL);

  while(stringdset) {

    if (!attr_key) {
      h5status=-1;
      printf("ERROR: get_dset_and_increment\n");
      goto h5sx_dataset_create_error;
    }

    h5status = h5sx_attrib_write( dataset, attr_key, attr_val );

    if (h5status<0) goto h5sx_dataset_create_error;

    stringdset = h5sx_get_next_dset(stringdset, &attr_key, &attr_val, NULL);

  } // while

  H5Pclose( dataprop );
  H5Sclose( dataspace );
  H5Tclose( datatype );
  FREE(chunk);
  FREE(dataspacesize);

  if (iodbg(IODBG_H5|IODBG_DEBUG1)) {
    printf("h5sx_dataset_create(loc_id=%d,...) (%d) END\n",loc_id,dataset);
  }

  return( dataset );

h5sx_dataset_create_error:

  if (iodbg(IODBG_H5|IODBG_DEBUG1)) {
    printf("h5sx_dataset_create(loc_id=%d,...) (-1) ERROR\n",loc_id);
  }

  if (dataset>=0) H5Dclose( dataset );

  if (dataprop>=0) H5Pclose( dataprop );
  if (dataspace>=0) H5Sclose( dataspace );
  if (datatype>=0) H5Tclose( datatype );
  FREE(chunk);
  FREE(dataspacesize);

  return( -1 );

} // h5sx_dataset_create

/*
 * Open array dataset, its description and its attributes
 *
 * hid_t h5sx_dataset_open(hid_t loc_id, const char *name, 
 *                         hsize_t *rank, hsize_t *dims[], hsize_t *maxdims[], hsize_t *itemsize, hid_t *datatype_id, void **pfillvalue, 
 *                         H5sxListOfDSets *attributes );
 *
 * The returned dimensions (dims and maxdims) are allocated and must be released with FREE for avoiding resource leaks.
 * In case of success the returned value is 0.
 */
hid_t h5sx_dataset_open(hid_t loc_id, const char *name, 
                        hsize_t *rank, hsize_t **dims, hsize_t **maxdims, hsize_t *itemsize, hid_t *datatype_id, void **pfillvalue,                    
                        H5sxListOfDSets *attributes )
{
  const char *attr_key=NULL;
  const char *attr_val=NULL;
  H5sxDset *stringdset;

  hid_t dataprop=-1, dataspace=-1, dataset=-1, datatype=-1;
  hsize_t *dataspacesize=NULL, *chunk=NULL;
  herr_t h5status=-1;

  if (iodbg(IODBG_H5|IODBG_DEBUG1)) {
    printf("h5sx_dataset_open(loc_id=%d,name=%s,rank=%p,dims[]=%p,maxdims[]=%p,itemsize=%p,datatype_id=%p,pfillvalue=%p,attributes=%p) BEGIN\n",
    loc_id,name,rank,dims,maxdims,itemsize,datatype_id,pfillvalue,attributes);
  }

//+++++++++++++++++++++++

//+++++++++++++++++++++++

  if (iodbg(IODBG_H5|IODBG_DEBUG1)) {
    printf("h5sx_dataset_open(loc_id=%d,...) (%d) END\n",loc_id,dataset);
  }

  return( dataset );

h5sx_dataset_open_error:

  if (iodbg(IODBG_H5|IODBG_DEBUG1)) {
    printf("h5sx_dataset_open(loc_id=%d,...) (-1) ERROR\n",loc_id);
  }

  if (dataset>=0) H5Dclose( dataset );

  if (dataprop>=0) H5Pclose( dataprop );
  if (dataspace>=0) H5Sclose( dataspace );
  if (datatype>=0) H5Tclose( datatype );
  FREE(chunk);
  FREE(dataspacesize);

  return( -1 );

} // h5sx_dataset_open

/* write 2D array data dimensions dims[3], which has chunk size (maxdims[0]=1,maxdims[1]=maxdim1,maxdims[2]=maxdim2 */
herr_t h5sx_dataset_write(hid_t dataset, long frame_no, hid_t mem_type_id,
                            hsize_t rank, hsize_t maxdims[], hsize_t dims[], void *data )
{
  hid_t filespace=-1, memtype=-1, memspace=-1;
  hsize_t *dataspacesize=NULL, *start=NULL, *count=NULL;
  hsize_t *memstart=NULL;
  herr_t  h5status=-1;

  const char *error_info=NULL;

  if (iodbg(IODBG_H5|IODBG_DEBUG1)) {
    printf("h5sx_dataset_write(dataset=%d,frame_no=%ld,mem_type_id=%d,rank=%ld,maxdims=%p,dims=%p,data=%p) BEGIN\n",
    dataset,frame_no,mem_type_id,(long)rank,maxdims,dims,data);
  }

  dataspacesize = (hsize_t *) MALLOC(sizeof(hsize_t)*rank);
  if ( !dataspacesize ) goto h5sx_dataset_write_error;

  start = (hsize_t *) MALLOC(sizeof(hsize_t)*rank);
  if ( !start ) goto h5sx_dataset_write_error;

  count = (hsize_t *) MALLOC(sizeof(hsize_t)*rank);
  if ( !count ) goto h5sx_dataset_write_error;

  memstart = (hsize_t *) MALLOC(sizeof(hsize_t)*rank);
  if ( !memstart ) goto h5sx_dataset_write_error;

  dims[0] = 1;  // must always be 1

  // Extend the dataset
  dataspacesize[0] = frame_no;
  dataspacesize[1] = maxdims[1];
  dataspacesize[2] = maxdims[2];

  if (iodbg(IODBG_H5|IODBG_DEBUG2)) H5SXPRINT( rank, ">> dataspacesize", dataspacesize );

  // herr_t H5Dset_extent( hid_t dset_id, const hsize_t size[] ) 
  h5status = H5Dset_extent (dataset, dataspacesize);
  if (h5status<0) {
    goto h5sx_dataset_write_error;
  }

  // Get (extended) filespace for writing data (must be called AFTER H5Dset_extent!)
  // hid_t H5Dget_space( hid_t dataset_id )
  filespace = H5Dget_space( dataset );
  if (filespace<0) {
    goto h5sx_dataset_write_error;
  }

  // Define and select the first part of the hyperslab selection.

  // Start offset of each dimension 
  start[0] = frame_no-1;
  start[1] = 0;
  start[2] = 0;

  // Restrict number of elements in each dimension to maxdims
  count[0] = 1;
  count[1] = MIN(dims[1],maxdims[1]);
  count[2] = MIN(dims[2],maxdims[2]);

  if (iodbg(IODBG_H5|IODBG_DEBUG2)) H5SXPRINT( rank, ">> start", start );
  if (iodbg(IODBG_H5|IODBG_DEBUG2)) H5SXPRINT( rank, ">> count", count );
  if (iodbg(IODBG_H5|IODBG_DEBUG2)) H5SXPRINT( rank, ">> dims", dims );

//  h5status = H5Sselect_hyperslab (filespace, H5S_SELECT_SET, start, stride, count, block);
  h5status = H5Sselect_hyperslab (filespace, H5S_SELECT_SET, start, NULL, count, NULL);
  if (h5status<0) {
    error_info="H5Sselect_hyperslab (filespace, ...)";
    goto h5sx_dataset_write_error;
  }

  // Get memory type 
  memtype = H5Tcopy( mem_type_id );
  if (memtype<0) {
    error_info="H5Tcopy";
    goto h5sx_dataset_write_error;
  }

  // Define the memory space and a hyperslab with count[] elements in each direction
  // starting at {0,0,0}
  memspace = H5Screate_simple (rank, dims, NULL);
  if (memspace<0) {
    error_info="H5Screate_simple:...memspace...";
    goto h5sx_dataset_write_error;
  }
  if (iodbg(IODBG_H5|IODBG_DEBUG2)) H5SXPRINT( rank, ">> memspacesize", dims );
  
  memstart[0] = 0;
  memstart[1] = 0;
  memstart[2] = 0;

  if (iodbg(IODBG_H5|IODBG_DEBUG2)) H5SXPRINT( rank, ">> memstart", memstart );

  // h5status = H5Sselect_hyperslab (memspace, H5S_SELECT_SET, memstart, memstride, memcount, memblock);
  h5status = H5Sselect_hyperslab (memspace, H5S_SELECT_SET, memstart, NULL, count, NULL);
  if (h5status<0) {
    error_info="H5Sselect_hyperslab (memspace, ...)";
    goto h5sx_dataset_write_error;
  }

  // herr_t H5Dwrite(hid_t dataset, hid_t mem_type_id, hid_t mem_space_id, hid_t file_space_id, hid_t xfer_plist_id, const void * buf  )
  h5status = H5Dwrite (dataset, memtype, memspace, filespace, H5P_DEFAULT, data);
  if (h5status<0) {
    error_info="H5Dwrite";
    goto h5sx_dataset_write_error;
  }

  H5Sclose( memspace );
  H5Tclose( memtype );
  H5Sclose( filespace );

  FREE(memstart);
  FREE(dataspacesize);
  FREE(count);
  FREE(start);

  if (iodbg(IODBG_H5|IODBG_DEBUG1)) printf("h5sx_dataset_write(dataset=%d,...) (0) END\n", dataset);

  return(0);

h5sx_dataset_write_error:

  if (iodbg(IODBG_H5|IODBG_DEBUG1)) printf("h5sx_dataset_write(dataset=%d,...) (-1) ERROR\n", dataset);
  if (iodbg(IODBG_H5|IODBG_DEBUG2)) printf("h5status=%d,%s\n",h5status,error_info);

  if (memspace>=0) H5Sclose( memspace );
  if (memtype>=0) H5Tclose( memtype );
  if (filespace>=0) H5Sclose( filespace );

  FREE(memstart);
  FREE(dataspacesize);
  FREE(count);
  FREE(start);

  return(-1);

} // h5sx_dataset_write

herr_t h5sx_dataset_close(hid_t dataset)
{ herr_t h5status=0;
  if (iodbg(IODBG_H5|IODBG_DEBUG1)) printf("h5sx_dataset_close(dataset=%d) BEGIN\n", dataset);
  if (dataset>=0) h5status = H5Dclose (dataset);
  if (iodbg(IODBG_H5|IODBG_DEBUG1)) printf("h5sx_dataset_close(dataset=%d) (%d) END\n", dataset, h5status);
  return( h5status );
} // h5sx_dataset_close

/*
 * Create string dataset key with single value and attributes
 * (key can start with a path from loc_id)
 */
herr_t h5sx_write_dataset_as_string(hid_t loc_id, const char *key, const char *val, H5sxListOfDSets *attributes)
{
  herr_t h5status=-1;
  hid_t val_type=-1;

  hid_t   stringds=-1, stringds_type=-1, stringds_space=-1;
  hsize_t stringds_dims[1] = {1}; // single string only

  if (iodbg(IODBG_H5|IODBG_DEBUG1)) printf("h5sx_write_dataset_as_string( %d ) BEGIN\n",loc_id);

  if (iodbg(IODBG_H5|IODBG_DEBUG2)) printf("\'%s\' = \'%s\'\n",key?key:"(NULL)",val?val:"(NULL)");

  if (!(key&&val)) goto h5sx_write_dataset_as_string_error;

  /*
   * Create output dataspace (stringds)
   * NO AUTOMATIC TRANSLATION BETWEEN FIXED LENGTH AND VARIABLE LENGTH DATATYPES!
   */

  stringds_type = H5Tcopy (H5T_C_S1);
  if (H5Tset_size (stringds_type, strlen(val)+1)<0) goto h5sx_write_dataset_as_string_error;

  stringds_space = H5Screate_simple (1, stringds_dims, NULL);
  if (stringds_space<0) goto h5sx_write_dataset_as_string_error;

  stringds = H5Dcreate (loc_id, key, stringds_type, stringds_space, H5P_DEFAULT, H5P_DEFAULT, H5P_DEFAULT);
  if (stringds<0) goto h5sx_write_dataset_as_string_error;

  /*
   * Write attributes
   */
  if (h5sx_attriblist_write( stringds, attributes )<0) goto h5sx_write_dataset_as_string_error;

  /*
   * Write val to dataset
   */
  val_type = H5Tcopy (H5T_C_S1);
  if (H5Tset_size (val_type, strlen(val)+1)<0) goto h5sx_write_dataset_as_string_error;

  if (H5Dwrite (stringds, val_type, H5S_ALL, H5S_ALL, H5P_DEFAULT, val)<0) goto h5sx_write_dataset_as_string_error;

  /*
   * Close and release resources.
   */
  if (H5Dclose(stringds)<0) goto h5sx_write_dataset_as_string_error;
  if (H5Tclose(stringds_type)<0) goto h5sx_write_dataset_as_string_error;
  if (H5Sclose (stringds_space)<0) goto h5sx_write_dataset_as_string_error;
  if (H5Tclose (val_type)<0) goto h5sx_write_dataset_as_string_error;

  if (iodbg(IODBG_H5|IODBG_DEBUG1)) printf("h5sx_write_dataset_as_string( %d ) END\n",loc_id);

  return( 0 );

h5sx_write_dataset_as_string_error:

  h5status=-1;

  if (stringds>=0) H5Dclose (stringds);
  if (stringds_type>=0) H5Tclose (stringds_type);
  if (stringds_space>=0) H5Sclose (stringds_space);
  if (val_type>=0) H5Tclose (val_type);

  printf("h5sx_write_dataset_as_string( %d ) (h5status=%d) ERROR\n",loc_id,h5status);

  return( h5status );

} // h5sx_write_dataset_as_string

typedef struct {
  H5sxListOfDSets *searchnames;
  H5sxListOfDSets *searchattributes;
  H5sxListOfDSets *vetostrings;  // exclude these nodes
  H5sxListOfDSets *foundstrings; // matching nodes
  H5ioNode    *node;             // insert nodes here
  int searchmode;                // H5sxGroupSearchMode enum type
} H5sxLinkSearch_data;

/*---------------------------------------------------------------------------
NAME

  h5sx_dataset_iteration -- iteration function called by H5Literate in h5sx_search_dataset


PURPOSE

  Tests, whether object dataset_name is a dataset and whether one of its attributes matches
  an attributes and value contained in searchattributes.
  If yes, add the dataset name to foundstrings.
  ...

ARGUMENTS

  herr_t h5sx_dataset_iteration (hid_t loc_id, const char *dataset_name, const H5L_info_t *info, void *op_data)

  hid_t loc_id           : location in which the iteration is done.
  const char *dataset_name : name of the link
  const H5L_info_t *info : info of the link
  void *op_data          : pointer to a structure of type H5sxLinkSearch_data:
                           typedef struct {
                             H5sxListOfDSets *searchnames;
                             H5sxListOfDSets *searchattributes;
                             H5sxListOfDSets *vetostrings;
                             H5sxListOfDSets *foundstrings;
                             H5ioNode *node;
                             int searchmode;
                           } H5sxLinkSearch_data;

  For searchnames==NULL and searchattributes==NULL all datasets match,
  otherwise only datasets matching a searchname or a searchattribute.

  H5ioNode *node                     : if not NULL, the matching datasets 
                                       are inserted in node,

  searchmode:
    H5sxGroupNumberSearch: stop iteration with the first match.
    H5sxArraySearch:       
    H5sxSymbolSearch:      iterate through all (matching) datasets (default)

RETURN VALUE

  herr_t hdf error value, negative in case of an error

---------------------------------------------------------------------------*/
herr_t h5sx_dataset_iteration (hid_t loc_id, const char *dataset_name, const H5L_info_t *info, void *op_data)
{
 /* typedef struct {
  *   H5L_type_t     type;         // Type of link
  *   hbool_t        corder_valid; // Indicates whether creation
  *                                // order is valid
  *   int64_t        corder;       // Creation order
  *   H5T_cset_t     cset;         // Character set of link name
  *   union {
  *       haddr_t    address;      // Address hard link points to
  *       size_t     val_size;     // Size of soft link or
  *                                // user-defined link value
  *   } u;
  * } H5L_info_t;
  */

  H5O_info_t object_info;
  hid_t dataset_id=-1;

  H5sxLinkSearch_data *plink_data=op_data;

  H5sxListOfDSets *searchnames=NULL;
  H5sxListOfDSets *searchattributes=NULL;
  H5sxListOfDSets *vetostrings=NULL;
  H5sxListOfDSets *foundstrings=NULL;
  H5ioNode *node=NULL;
  int searchmode=0;

  int matchcount=0; // no match so far

  char dataset_name_with_path[H5SX_BUFLEN]="";

  if (plink_data) {
    searchnames = plink_data->searchnames;
    searchattributes = plink_data->searchattributes;
    vetostrings = plink_data->vetostrings;
    foundstrings = plink_data->foundstrings;
    node = plink_data->node;
    searchmode = plink_data->searchmode;
  }

  if (iodbg(IODBG_H5|IODBG_DEBUG1)) {
    printf(">> h5sx_dataset_iteration (id=%d,dataset_name=%s,info,op_data)",loc_id,dataset_name);
    if (iodbg(IODBG_H5|IODBG_DEBUG2)) {
      switch(info->cset) {
        case H5T_CSET_ASCII: printf(" (H5T_CSET_ASCII)");
                             break;
        case H5T_CSET_UTF8 : printf(" (H5T_CSET_UTF8)");
                             break;
        default: printf(" (unknown character set %d)",info->cset);
      }
    }
    printf(" BEGIN\n");
  }

  /*
   * Get the full name of the link (<path>/<dataset_name>)
   */
  // ssize_t H5Iget_name( hid_t obj_id, char *name, size_t size )
  if (H5Iget_name(loc_id,dataset_name_with_path,H5SX_BUFLEN )<0) goto h5sx_dataset_iteration_error;
  if (strcmp("/",dataset_name_with_path))
    strlib_concat( dataset_name_with_path, H5SX_BUFLEN, dataset_name_with_path, "/" );
  strlib_concat( dataset_name_with_path, H5SX_BUFLEN, dataset_name_with_path, dataset_name );

  /*
   * open dataset_id and read attribute sxclassname
   */
  // hid_t H5Oopen( hid_t loc_id, const char *name, hid_t lapl_id );

  dataset_id = H5Oopen( loc_id, dataset_name, H5P_DEFAULT );
  if (dataset_id<0) goto h5sx_dataset_iteration_error;

  /*
   * Verify, that object dataset_id is a dataset, return if not
   * H5O_TYPE_GROUP, H5O_TYPE_DATASET, H5O_TYPE_NAMED_DATATYPE, H5O_TYPE_NTYPES, H5O_TYPE_UNKNOWN
   */
  // herr_t H5Oget_info( hid_t object_id, H5O_info_t *object_info )
  if ( H5Oget_info( dataset_id, &object_info ) <0 ) goto h5sx_dataset_iteration_error;

  if (object_info.type == H5O_TYPE_DATASET) {

    /*
     * Accept only datasets that are not listed in vetostrings
     */
     if (!h5sx_search_dset(vetostrings,dataset_name,dataset_name_with_path)) {

      /*
       * Search a match between a dataset attribute (of group 'dataset_name')
       * and attributes listed in searchattributes
       */

      const char *search_key=NULL, *search_value=NULL;
      H5sxDset *stringds=NULL;
      char buffer[STRINGVSIZE];
      const char *attr_value=NULL;

      /*
       * Search a match between dataset_name and a key listed in searchnames
       */
      if (searchnames) {
        /*
         * Compare searchattributes with dataset attributes (of dataset 'dataset_name')
         * Stop on first match.
         */
        stringds = h5sx_get_first_dset(searchnames,&search_key,&search_value,NULL);
        while(stringds) {
          if (search_key) {
          if (!(strlib_wildcmp(dataset_name,search_key))) {
              /* matching key found, stop loop */
              if (strlib_casecmp(search_value,"AND")) {
                matchcount=+1;
              } // otherwise, find also a matching searchattribute
              // printf("\"%s\": %s (value=%s)\n",dataset_name,search_key,search_value); //+++++++++++++=
              break; // stop loop over searchnames
            } // if equal
          }
          stringds = h5sx_get_next_dset(stringds, &search_key, &search_value, NULL);
        } // while
      }

      if (searchattributes&&(matchcount<1)) {
        /*
         * Compare searchattributes with group attributes (of group 'dataset_name')
         * Stop on first match.
         */
        stringds = h5sx_get_first_dset(searchattributes,&search_key,&search_value, NULL);
        while(stringds) {
          htri_t attr_exists=-1;
          if (search_key) {
            attr_exists=H5Aexists(dataset_id,search_key);
            if (attr_exists<0) goto h5sx_dataset_iteration_error;
            if (attr_exists) {
              attr_value = h5sx_attrib_read( buffer, STRINGVSIZE,dataset_id,search_key);
              if (!strlib_casecmp(search_value,attr_value)) {
                /* matching attribute found, stop loop */
                matchcount+=1;
                // printf("\"%s\": %s = %s\n",dataset_name,search_key,attr_value); //+++++++++++++=
                break; // stop loop over searchattributes
              }
            } // if attr_exists
          }
          stringds = h5sx_get_next_dset(stringds, &search_key, &search_value, NULL);
        }
      }

      if (matchcount>0) {
        if (foundstrings) {
          h5sx_append_dset( foundstrings, dataset_name, dataset_name_with_path, 0, 1, -1 );
        }

        if (node) {
          H5ioData *inserted=NULL; // inserted dataset

         switch (searchmode) {

            case H5sxGroupNumberSearch:
              inserted = h5io_insert_data ( node, dataset_name, H5ioGroupNumberNode ); break;

            case H5sxArraySearch:
              // +++++++ inserted = h5io_insert_data ( node, dataset_name, H5ioArrayNode ); break;
              inserted = h5io_insert_data ( node, ARRAYNAME, H5ioArrayNode ); break;

            default:
            case H5sxSymbolSearch:
              inserted = h5io_insert_data ( node, dataset_name, H5ioSymbolNode ); break;
          }
          if (!inserted) goto h5sx_dataset_iteration_error;

          inserted->H5Path = strlib_newstr(dataset_name_with_path);
          inserted->H5id   = node->H5id; // root id of path
        }
      }
    } else {
      if (iodbg(IODBG_H5|IODBG_DEBUG2)) {
        printf(">> h5sx_dataset_iteration (id=%d,dataset_name=%s,info,op_data): dataset >>%s<< matches veto string",loc_id,dataset_name,dataset_name_with_path);
      }
    } // if not in vetostrings
  } // if H5O_TYPE_DATASET

  // close dataset_id
  H5Oclose( dataset_id );

  //  0 to continue  iteration
  // >0 to interrupt iteration (H5Literate returns this value)
  if (searchmode==H5sxGroupNumberSearch) {
    if (iodbg(IODBG_H5|IODBG_DEBUG1))
      printf(">> h5sx_dataset_iteration (id=%d,dataset_name=%s,info,op_data) (%d) END\n",loc_id,dataset_name,matchcount);
    return(matchcount); // interrupt iteration, if a match has been found
  } 

  if (iodbg(IODBG_H5|IODBG_DEBUG1))
    printf(">> h5sx_dataset_iteration (id=%d,dataset_name=%s,info,op_data) (0) END\n",loc_id,dataset_name);
  return(0); // continue iteration

h5sx_dataset_iteration_error:

  if (dataset_id>=0) H5Oclose( dataset_id );

  printf(">> h5sx_dataset_iteration (id=%d,dataset_name=%s,info,op_data) (-1) ERROR\n",loc_id,dataset_name);

  return(-1);

} // h5sx_dataset_iteration

/*---------------------------------------------------------------------------
NAME

  h5sx_search_dataset_by_name ---

PURPOSE

  h5sx_search_dataset_by_name iterates through all datasets in loc_id. If one 
  of the attributes of a dataset matches either one of the searchattributes
  or one of the searchnames (only key), its name is added to foundstrings. 

ARGUMENTS

  herr_t h5sx_search_dataset_by_name( hid_t loc_id,  const char *paths, int searchmode, 
                                H5sxListOfDSets *searchnames,  
                                H5sxListOfDSets *searchattributes,  
                                H5sxListOfDSets *datasets, H5ioNode *node );

  hid_t loc_id         : base of the 'paths'

  H5sxListOfDSets *searchnames (input)      : list of names to read (independent
                                          of searchattributes)
  H5sxListOfDSets *searchattributes (input) : list of search attributes
  H5sxListOfDSets *datasets (input)    : list of dataset names to exclude
  H5sxListOfDSets *datasets (output)   : list of matching dataset names
  H5ioNode *node                       : if not NULL, the matching datasets 
                                          are inserted in node.

RETURN VALUE

  herr_t h5 error, negative in case of an error

---------------------------------------------------------------------------*/
herr_t h5sx_search_dataset_by_name( hid_t loc_id,  const char *paths, int searchmode, 
  H5sxListOfDSets *searchnames, H5sxListOfDSets *searchattributes, H5sxListOfDSets *datasets, H5ioNode *node )
{ herr_t h5status=-1;
  char *paths_tmp=NULL;
  H5sxListOfDSets *vetostrings=NULL;

  if (iodbg(IODBG_H5|IODBG_DEBUG1)) printf("h5sx_search_dataset_by_name( %d, %s, %s, %p, %p, %p, %p ) BEGIN\n",
    loc_id,paths,h5sxsearchmode2str(searchmode),searchnames,searchattributes,datasets,node);

  if (h5sx_length_listofdsets(searchnames)+h5sx_length_listofdsets(searchattributes) > 0 ) {
    if ((node)||(datasets)) {
      H5sxLinkSearch_data link_data;

      // move datasets to vetostrings
      vetostrings=h5sx_move_listofdsets(datasets);

      link_data.searchnames = searchnames;
      link_data.searchattributes = searchattributes;
      link_data.vetostrings = vetostrings;
      link_data.foundstrings = datasets;
      link_data.node = node;
      link_data.searchmode = searchmode;

      // loop through all paths (separated with colons)
      char *saveptr=NULL;
      char *path=NULL;
      paths_tmp = strlib_newstr(paths);
      // path = strtok_r(paths_tmp,":",&saveptr);
      path = strlib_tok_r(paths_tmp,":",&saveptr);
      while (path) {

        hsize_t idx=0;
        h5status=H5Literate_by_name( loc_id, path, H5_INDEX_NAME, H5_ITER_INC, &idx, h5sx_dataset_iteration, (void *) &link_data, H5P_DEFAULT);
        if (h5status<0) goto h5sx_search_dataset_by_name_error;

        if (iodbg(IODBG_H5|IODBG_DEBUG1)) printf("h5sx_search_dataset_by_name idx=%ld\n",(long)idx);

        // path=strtok_r(NULL,":",&saveptr);
        path=strlib_tok_r(NULL,":",&saveptr);
      }

    }
  } // else cannot search for dataset, no search parameters given

  vetostrings=h5sx_free_listofdsets(vetostrings);

  FREE(paths_tmp);

  h5status = 0;

  if (iodbg(IODBG_H5|IODBG_DEBUG1)) printf("h5sx_search_dataset_by_name( %d, %s, %s, %p, %p, %p, %p ) (h5status=%d) END\n",
    loc_id,paths,h5sxsearchmode2str(searchmode),searchnames,searchattributes,datasets,node,h5status);

  return(h5status);

h5sx_search_dataset_by_name_error:

  vetostrings=h5sx_free_listofdsets(vetostrings);

  FREE(paths_tmp);

  if (iodbg(IODBG_H5|IODBG_DEBUG1)) printf("h5sx_search_dataset_by_name( %d, %s, %s, %p, %p, %p, %p ) (h5status=%d) ERROR\n",
    loc_id,paths,h5sxsearchmode2str(searchmode),searchnames,searchattributes,datasets,node,h5status);

  return(-1);

} /* h5sx_search_dataset_by_name */

/*---------------------------------------------------------------------------
NAME

  h5sx_group_iteration -- iteration function called by H5Literate in h5sx_search_group


PURPOSE

  If the object name is not a group the function returns.

  It is checked whether one of the attributes matches with name and value
  one of the searchattributes.

  It is checked whether the group_name matches one of the keys in searchnames.

  If the key value in searchnames is 'OR' for the matching key either the attribute 
  or the name must match, otherwise both.

  If both are true, the group is inserted in the owning node. If searchmode is
  H5sxDataBlockSearch, H5sxErrorBlockSearch or H5sxHeaderSearch the inserted group has the
  name DATABLOCKNAME, ERRORBLOCKNAME or HEADERNAME. In all other cases the group is
  inserted with their group number (default 1).

  The class of the inserted group depend on the class of the owning node (node->NodeClass).

  The group number and name are written to foundstrings.

  ...

ARGUMENTS

  herr_t h5sx_group_iteration (hid_t loc_id, const char *group_name, const H5L_info_t *info, void *op_data)

  hid_t loc_id           : location where the iteration is done.
  const char *group_name : name of the link
  const H5L_info_t *info : info of the link
  void *op_data          : pointer to a structure of type H5sxLinkSearch_data:
                           typedef struct {
                             H5sxListOfDSets *searchnames;
                             H5sxListOfDSets *searchattributes;
                             H5sxListOfDSets *foundstrings;
                             H5ioNode *node;
                             H5sxListOfDSets *searchmode;
                           } H5sxLinkSearch_data;

RETURN VALUE

  herr_t hdf error value, negative in case of an error

---------------------------------------------------------------------------*/
herr_t h5sx_group_iteration (hid_t loc_id, const char *group_name, const H5L_info_t *info, void *op_data)
{/* typedef struct {
  *   H5L_type_t     type;         // Type of link
  *   hbool_t        corder_valid; // Indicates whether creation
  *                                // order is valid
  *   int64_t        corder;       // Creation order
  *   H5T_cset_t     cset;         // Character set of link name
  *   union {
  *       haddr_t    address;      // Address hard link points to
  *       size_t     val_size;     // Size of soft link or
  *                                // user-defined link value
  *   } u;
  * } H5L_info_t;
  */

  H5O_info_t object_info;

  H5sxLinkSearch_data *plink_data=op_data;
  hid_t group_id=-1;

  H5sxListOfDSets *searchnames=NULL;
  H5sxListOfDSets *searchattributes=NULL;
  H5sxListOfDSets *foundstrings=NULL;
  H5ioNode        *node=NULL;
  int searchmode=0;

  H5sxListOfDSets *group_number_search_names=NULL;
  H5sxListOfDSets *group_number_search_attributes=NULL;
  H5sxListOfDSets *found_group_numbers=NULL;

  char group_name_with_path[H5SX_BUFLEN]="";

  if (plink_data) {
    searchnames = plink_data->searchnames;
    searchattributes = plink_data->searchattributes;
    foundstrings = plink_data->foundstrings; 
    node = plink_data->node;
    searchmode = plink_data->searchmode; 
  }

  if (iodbg(IODBG_H5|IODBG_DEBUG1)) {
    printf(">> h5sx_group_iteration (id=%d,group_name=%s,info,op_data)",loc_id,group_name);
    if (iodbg(IODBG_H5|IODBG_DEBUG2)) {
      switch(info->cset) {
        case H5T_CSET_ASCII: printf(" (H5T_CSET_ASCII)");
                             break;
        case H5T_CSET_UTF8 : printf(" (H5T_CSET_UTF8)");
                             break;
        default: printf(" (unknown character set %d)",info->cset);
      }
    }
    printf(" BEGIN\n");
  }

  /*
   * Get the full name of the link (<path>/<group_name>)
   */
  // ssize_t H5Iget_name( hid_t obj_id, char *name, size_t size )
  if (H5Iget_name(loc_id,group_name_with_path,H5SX_BUFLEN )<0) goto h5sx_group_iteration_error;
  if (strcmp("/",group_name_with_path))
    strlib_concat( group_name_with_path, H5SX_BUFLEN, group_name_with_path, "/" );
  strlib_concat( group_name_with_path, H5SX_BUFLEN, group_name_with_path, group_name );

  /* 
   * open group_id and read attribute h5sxclassname
   */
  // hid_t H5Oopen( hid_t loc_id, const char *name, hid_t lapl_id );

  group_id = H5Oopen( loc_id, group_name, H5P_DEFAULT );
  if (group_id<0) goto h5sx_group_iteration_error;

  /*
   * Verify, that object group_id is a group, return if not
   * H5O_TYPE_GROUP, H5O_TYPE_DATASET, H5O_TYPE_NAMED_DATATYPE, H5O_TYPE_NTYPES, H5O_TYPE_UNKNOWN
   */
  // herr_t H5Oget_info( hid_t object_id, H5O_info_t *object_info )
  if ( H5Oget_info( group_id, &object_info ) <0 ) goto h5sx_group_iteration_error;

  if (object_info.type==H5O_TYPE_GROUP) {

    /*
     * Search a match between a group attribute (of group 'group_name')
     * and attributes listed in searchattributes
     */
    const char *search_key=NULL, *search_value=NULL;
    H5sxDset *stringds=NULL;
    int matchcount=0;
    char buffer[STRINGVSIZE];
    const char *attr_value=NULL;

    /*
     * Search a match between group_name and a key listed in searchnames
     */
    if (searchnames) {
      /*
       * Compare searchattributes with dataset attributes (of dataset 'group_name')
       * Stop on first match.
       */
      stringds = h5sx_get_first_dset(searchnames,&search_key,&search_value,NULL);
      while(stringds) {
        if (search_key) {
          if (!(strlib_wildcmp(group_name,search_key))) {
            /* matching key found, stop loop */
            if (strlib_casecmp(search_value,"AND")) {
              matchcount=+1;
            } // otherwise, find also a matching searchattribute
            // printf("\"%s\": %s (value=%s)\n",group_name,search_key,search_value); //+++++++++++++=
            break; // stop loop over searchnames
          } // if equal
        }
        stringds = h5sx_get_next_dset(stringds, &search_key, &search_value, NULL);
      } // while
    }

    if (searchattributes&&(matchcount<1)) {
      /* 
       * Compare searchattributes with group attributes (of group 'group_name')
       * Stop on first match.
       */
      stringds = h5sx_get_first_dset(searchattributes,&search_key,&search_value, NULL);
      while(stringds) {
        htri_t attr_exists=-1;
        if (search_key) {
          attr_exists=H5Aexists(group_id,search_key);
          if (attr_exists<0) goto h5sx_group_iteration_error;
          if (attr_exists) {
            attr_value = h5sx_attrib_read( buffer, STRINGVSIZE,group_id,search_key);
            if (!strlib_casecmp(search_value,attr_value)) {
              /* matching attribute found, stop loop */
              matchcount+=1;
              // printf("\"%s\": %s = %s\n",group_name,search_key,attr_value); //+++++++++++++=
              break; // stop loop over searchattributes
            } 
          } // if attr_exists
        }
        stringds = h5sx_get_next_dset(stringds, &search_key, &search_value, NULL);
      }
    }

    if (matchcount>0) {
      const char *ds_name=NULL;
      const char *group_number_string=NULL;

      switch (searchmode) {

      case H5sxHeaderSearch:
        // use matching searchstring as name, maximum 1 item per search string

        if (foundstrings) {
          // write group_number and group_name_with_path
          h5sx_append_dset( foundstrings, HEADERNAME, group_name_with_path, 0, 1, -1 );
        }

        if (node) {
          H5ioNode *inserted=NULL; // inserted group node

          switch (node->NodeClass) {
            case H5ioBlockNode: 
              inserted = h5io_insert_node( node, HEADERNAME, H5ioHeaderNode ); break;
            case H5ioMemoryNode: 
              inserted = h5io_insert_node( node, HEADERNAME, H5ioBlockNode ); break;
            case H5ioSeriesNode: 
              inserted = h5io_insert_node( node, HEADERNAME, H5ioMemoryNode ); break; // Number
            case H5ioEntryNode: 
              inserted = h5io_insert_node( node, HEADERNAME, H5ioSeriesNode ); break; // Number
            case H5ioFileNode:
              inserted = h5io_insert_node( node, HEADERNAME, H5ioEntryNode ); break;  // Number
          }
          if (!inserted) goto h5sx_group_iteration_error;

          // append group_name_with_path to H5Path
          if (!strlib_is_empty(inserted->H5Path)) {
            char H5PathBuffer[H5SX_BUFLEN]={'\0'};
            strlib_concat(H5PathBuffer,H5SX_BUFLEN,inserted->H5Path,":");
            strlib_concat(H5PathBuffer,H5SX_BUFLEN,H5PathBuffer,group_name_with_path);
            FREE(inserted->H5Path);
            inserted->H5Path = strlib_newstr(H5PathBuffer);
          } else {
            inserted->H5Path = strlib_newstr(group_name_with_path);
          }
          inserted->H5id   = node->H5id; // root id of path
        }

        break;

      case H5sxDataBlockSearch:
        // use matching searchstring as name, maximum 1 item per search string

        if (foundstrings) {
          // write group_number and group_name_with_path
          h5sx_append_dset( foundstrings, DATABLOCKNAME, group_name_with_path, 0, 1, -1 );
        }

        if (node) {
          H5ioNode *inserted=NULL; // inserted group node

          switch (node->NodeClass) {
            case H5ioBlockNode:
              inserted = h5io_insert_node( node, DATABLOCKNAME, H5ioHeaderNode ); break;
            case H5ioMemoryNode:
              inserted = h5io_insert_node( node, DATABLOCKNAME, H5ioBlockNode ); break;
            case H5ioSeriesNode:
              inserted = h5io_insert_node( node, DATABLOCKNAME, H5ioMemoryNode ); break; // Number
            case H5ioEntryNode:
              inserted = h5io_insert_node( node, DATABLOCKNAME, H5ioSeriesNode ); break; // Number
            case H5ioFileNode:
              inserted = h5io_insert_node( node, DATABLOCKNAME, H5ioEntryNode ); break;  // Number
          }
          if (!inserted) goto h5sx_group_iteration_error;

          // append group_name_with_path to H5Path
          if (!strlib_is_empty(inserted->H5Path)) {
            char H5PathBuffer[H5SX_BUFLEN]={'\0'};
            strlib_concat(H5PathBuffer,H5SX_BUFLEN,inserted->H5Path,":");
            strlib_concat(H5PathBuffer,H5SX_BUFLEN,H5PathBuffer,group_name_with_path);
            FREE(inserted->H5Path);
            inserted->H5Path = strlib_newstr(H5PathBuffer);
          } else {
            inserted->H5Path = strlib_newstr(group_name_with_path);
          }
          inserted->H5id   = node->H5id; // root id of path
        }

        break;

      case H5sxErrorBlockSearch:
        // use matching searchstring as name, maximum 1 item per search string

        if (foundstrings) {
          // write group_number and group_name_with_path
          h5sx_append_dset( foundstrings, ERRORBLOCKNAME, group_name_with_path, 0, 1, -1 );
        }

        if (node) {
          H5ioNode *inserted=NULL; // inserted group node

          switch (node->NodeClass) {
            case H5ioBlockNode:
              inserted = h5io_insert_node( node, ERRORBLOCKNAME, H5ioHeaderNode ); break;
            case H5ioMemoryNode:
              inserted = h5io_insert_node( node, ERRORBLOCKNAME, H5ioBlockNode ); break;
            case H5ioSeriesNode:
              inserted = h5io_insert_node( node, ERRORBLOCKNAME, H5ioMemoryNode ); break; // Number
            case H5ioEntryNode:
              inserted = h5io_insert_node( node, ERRORBLOCKNAME, H5ioSeriesNode ); break; // Number
            case H5ioFileNode:
              inserted = h5io_insert_node( node, ERRORBLOCKNAME, H5ioEntryNode ); break;  // Number
          }
          if (!inserted) goto h5sx_group_iteration_error;

          // append group_name_with_path to H5Path
          if (!strlib_is_empty(inserted->H5Path)) {
            char H5PathBuffer[H5SX_BUFLEN]={'\0'};
            strlib_concat(H5PathBuffer,H5SX_BUFLEN,inserted->H5Path,":");
            strlib_concat(H5PathBuffer,H5SX_BUFLEN,H5PathBuffer,group_name_with_path);
            FREE(inserted->H5Path);
            inserted->H5Path = strlib_newstr(H5PathBuffer);
          } else {
            inserted->H5Path = strlib_newstr(group_name_with_path);
          }
          inserted->H5id   = node->H5id; // root id of path
        }

        break;

      default: // case H5sxEntrySearch, H5sxSeriesSearch, H5sxMemorySearch:
        // use group_number as name, maximum 1 item per group number

        /*
         * Determine group number:
         * id : group_id
         * key : group_name
         * a) search for a dataset in group_id with attribute SXCLASS_GROUPNUMBER
         *    or with the name GROUPNUMBERNAME and take its value as group number
         *    If more than a single dataset with this attribute was found, the result
         *    is undefined and depends on the search order
         * b) converting group group_name to a number, e.g. group_name=<blabla><group_number>.
         */

        ds_name=NULL;
        // a) 
        if (!ds_name) {
          // herr_t h5sx_search_dataset_by_name( hid_t loc_id, const char *name, int searchmode, 
          //   H5sxListOfDSets *searchnames, H5sxListOfDSets *searchattributes, H5sxListOfDSets *foundstrings, H5ioNode *node )
          found_group_numbers=h5sx_new_listofdsets(NULL);

          group_number_search_names=h5sx_new_listofdsets(NULL);
          h5sx_append_dset( group_number_search_names, GROUPNUMBERNAME, NULL, 0, 1, -1 );

          group_number_search_attributes=h5sx_new_listofdsets(NULL);
          h5sx_append_dset( group_number_search_attributes, SXCLASS, SXCLASS_GROUPNUMBER, 0, 1, -1 );

          if ( h5sx_search_dataset_by_name( group_id, "./", H5sxGroupNumberSearch, group_number_search_names,
               group_number_search_attributes, found_group_numbers, NULL )<0 ) 
            goto h5sx_group_iteration_error;
          h5sx_get_first_dset(found_group_numbers,NULL,&ds_name,NULL);
        }

        if (ds_name) {
          // read group number from dataset ds_name
          group_number_string=h5sx_read_dataset_as_string(buffer,STRINGVSIZE,group_id,ds_name,NULL);
        } // if ds_name

        // b)
        if (!group_number_string) {
          const char *pg;
          long number=0;

          // skip non-digit characters
          for (pg=group_name;*pg;pg++) {
            if ( isdigit(*pg)||(*pg=='-')||(*pg=='+') ) break;
          }

          if (strlen(pg)>0) {
            number = (long) num_str2double( pg, NULL, NULL);
            group_number_string=num_long2str ( buffer, STRINGVSIZE, number, NULL );
          } else {
            group_number_string = "1"; // just for now to be safe
          }
        }

        if (!group_number_string)
          group_number_string = "1"; // just for now to be safe

        if (foundstrings) {
          // write group_number and group_name_with_path
          h5sx_append_dset( foundstrings, group_number_string, group_name_with_path, 0, 1, -1 );
        }

        if (node) {
          H5ioNode *inserted=NULL; // inserted group node

          switch (node->NodeClass) {
            case H5ioBlockNode:
              inserted = h5io_insert_node( node, group_number_string, H5ioHeaderNode ); break;
            case H5ioMemoryNode:
              inserted = h5io_insert_node( node, group_number_string, H5ioBlockNode ); break;
            case H5ioSeriesNode:
              inserted = h5io_insert_node( node, group_number_string, H5ioMemoryNode ); break; // Number
            case H5ioEntryNode:
              inserted = h5io_insert_node( node, group_number_string, H5ioSeriesNode ); break; // Number
            case H5ioFileNode:
              inserted = h5io_insert_node( node, group_number_string, H5ioEntryNode ); break;  // Number
          }

          if (!inserted) goto h5sx_group_iteration_error;

          // append group_name_with_path to H5Path
          if (!strlib_is_empty(inserted->H5Path)) {
            char H5PathBuffer[H5SX_BUFLEN]={'\0'};
            strlib_concat(H5PathBuffer,H5SX_BUFLEN,inserted->H5Path,":");
            strlib_concat(H5PathBuffer,H5SX_BUFLEN,H5PathBuffer,group_name_with_path);
            FREE(inserted->H5Path);
            inserted->H5Path = strlib_newstr(H5PathBuffer);
          } else {
            inserted->H5Path = strlib_newstr(group_name_with_path);
          }
          inserted->H5Path = strlib_newstr(group_name_with_path);
          inserted->H5id   = node->H5id; // root id of path
        }

        break;

      } // switch searchmode

    } // matchcount

  } // H5O_TYPE_GROUP

  found_group_numbers=h5sx_free_listofdsets(found_group_numbers);
  group_number_search_attributes=h5sx_free_listofdsets(group_number_search_attributes);
  group_number_search_names=h5sx_free_listofdsets(group_number_search_names);

  // close group_id
  H5Oclose( group_id ); 

  if (iodbg(IODBG_H5|IODBG_DEBUG1))
    printf(">> h5sx_group_iteration (id=%d,group_name=%s,info,op_data) (0) END\n",loc_id,group_name);

  //  0 to continue  iteration
  // >0 to interrupt iteration (H5Literate returns this value)
  return(0);

h5sx_group_iteration_error:

  found_group_numbers=h5sx_free_listofdsets(found_group_numbers);
  group_number_search_attributes=h5sx_free_listofdsets(group_number_search_attributes);
  group_number_search_names=h5sx_free_listofdsets(group_number_search_names);

  if (group_id>=0) H5Oclose( group_id ); 

  printf(">> h5sx_group_iteration (id=%d,group_name=%s,info,op_data) (-1) ERROR\n",loc_id,group_name);

  return(-1);

} // h5sx_group_iteration

/*
 * If searchnames and searchattributes contain searchconditions 
 * groups are searched in the location specified with loc_id and paths and inserted to the group list of node.
 * If searchnames and searchattributes are empty
 * a dummy group with a name depending on searchmode and a class depending on NodeClass is inserted.
 * It refers to the same location as the owning group.
 */
herr_t h5sx_search_group_by_name( hid_t loc_id,  const char *paths, int searchmode, H5sxListOfDSets *searchnames,
  H5sxListOfDSets *searchattributes, H5sxListOfDSets *foundstrings, H5ioNode *node )
{ herr_t h5status=-1;
  char *paths_tmp=NULL;

  if (iodbg(IODBG_H5|IODBG_DEBUG1)) printf("h5sx_search_group_by_name( %d, %s, %s, %p, %p, %p, %p ) BEGIN\n",
    loc_id,paths,h5sxsearchmode2str(searchmode),searchnames,searchattributes,foundstrings,node);

  if (h5sx_length_listofdsets(searchnames)+h5sx_length_listofdsets(searchattributes) > 0 ) {
    if (foundstrings||node) {
      // search and insert the groups
      H5sxLinkSearch_data link_data;
      link_data.searchnames = searchnames;
      link_data.searchattributes = searchattributes;
      link_data.foundstrings = foundstrings;
      link_data.node = node;
      link_data.searchmode = searchmode;

      // loop through all paths (separated with colons)
      char *saveptr=NULL;
      char *path=NULL;
      paths_tmp = strlib_newstr(paths);
      // path = strtok_r(paths_tmp,":",&saveptr);
      path = strlib_tok_r(paths_tmp,":",&saveptr);
      while (path) {

        hsize_t idx=0;
        h5status=H5Literate_by_name( loc_id, path, H5_INDEX_NAME, H5_ITER_INC, &idx, h5sx_group_iteration, (void *) &link_data, H5P_DEFAULT);
        if (h5status<0) goto h5sx_search_group_by_name_error;

        // path=strtok_r(NULL,":",&saveptr);
        path=strlib_tok_r(NULL,":",&saveptr);
      }
    }
  } // else cannot search a group, no search parameters given

  if (node) {
    if (!(node->NodeList)) { // if NodeList is empty
      H5ioNode *inserted=NULL; // insert dummy node
      char *name=NULL;

      switch ( searchmode ) {
        case H5sxDataBlockSearch:
          name=DATABLOCKNAME; break;
        case H5sxErrorBlockSearch:
          name=ERRORBLOCKNAME; break;
        case H5sxHeaderSearch:
          name=HEADERNAME; break;
        default: 
          name="1";
      }

      switch (node->NodeClass) {
        case H5ioBlockNode:
          inserted = h5io_insert_node( node, name, H5ioHeaderNode ); break;
        case H5ioMemoryNode:
          inserted = h5io_insert_node( node, name, H5ioBlockNode ); break;
        case H5ioSeriesNode:
          inserted = h5io_insert_node( node, name, H5ioMemoryNode ); break;
        case H5ioEntryNode:
          inserted = h5io_insert_node( node, name, H5ioSeriesNode ); break;
        case H5ioFileNode:
          inserted = h5io_insert_node( node, name, H5ioEntryNode ); break;
      }
      if (!inserted) goto h5sx_search_group_by_name_error;

      // copy existing path of owning group
      inserted->H5Path = strlib_newstr(node->H5Path);
      inserted->H5id   = node->H5id;
    }
  }

  h5status=0; // Success

  FREE(paths_tmp);

  if (iodbg(IODBG_H5|IODBG_DEBUG1)) printf("h5sx_search_group_by_name( %d, %s, %s, %p, %p, %p, %p ) (h5status=%d) END\n",
    loc_id,paths,h5sxsearchmode2str(searchmode),searchnames,searchattributes,foundstrings,node,h5status);

  return(h5status);

h5sx_search_group_by_name_error:

  FREE(paths_tmp);

  printf("h5sx_search_group_by_name( %d, %s, %s, %p, %p, %p, %p ) (h5status=%d) ERROR\n",
    loc_id,paths,h5sxsearchmode2str(searchmode),searchnames,searchattributes,foundstrings,node,h5status);
  return(-1);

} // h5sx_search_group_by_name

/*******************************************************************************
* Accessing the h5 file (END)                                                *
*******************************************************************************/

/*---------------------------------------------------------------------------
NAME

  h5sx_new_entry --- open the entry corresponding to number or create it

SYNOPSIS

  H5ioNode *h5sx_new_entry ( H5ioNode *h5io_file, long number, int blocknumbering, int *pstatus )

DESCRIPTION

  The entry name is searched in the h5io_file->NodeList. If it does not 
  exist it is created and added to NodeList.

  H5ioNode *h5io_entry
  long entry_number
  int *pstatus

RETURN VALUE
success: DataList and *pstatus>=0
error:   NULL and *pstatus<0
---------------------------------------------------------------------------*/
H5ioNode *h5sx_new_entry ( H5ioNode *h5io_file, long number, int blocknumbering, int *pstatus )
{
  int status=-1;
  hid_t sxentry_gid=-1;
  H5ioNode *h5io_entry=NULL;
  H5sxListOfDSets *entry_attributes=NULL, *entry_number_attributes=NULL, *blocknumbering_attributes=NULL;
  char entry_name[H5SX_BUFLEN]={'\0'};

        // H5ioNode *h5sx_new_entry ( H5ioNode *h5io_file, long number, int blocknumbering, int *pstatus );
        // H5ioNode *h5sx_new_series ( H5ioNode *h5io_entry, long number, int *pstatus );
        // H5ioNode *h5sx_new_memory ( H5ioNode *h5io_series, long number, int *pstatus );
        // H5ioNode *h5sx_new_datablock ( H5ioNode *h5io_memory, const char *name, int *pstatus );
        // H5ioNode *h5sx_new_errorblock ( H5ioNode *h5io_memory, const char *name, int *pstatus );
        // H5ioNode *h5sx_new_header ( H5ioNode *h5io_block, const char *name, int *pstatus );
        // H5ioNode *h5sx_write_symbols ( H5ioNode *h5io_header, H5ioData *h5io_symbol, int *pstatus );
        // H5ioNode *h5sx_write_data ( H5ioNode *h5io_block, H5ioData *h5io_data, int *pstatus );

  if (iodbg(IODBG_H5|IODBG_DEBUG1|IODBG_DEBUG2)) printf(" h5sx_new_entry ( %p, %ld ... ) BEGIN\n",h5io_file,number);

  num_long2str ( entry_name, H5SX_BUFLEN, number, NULL);

  h5io_entry = h5io_search_node( h5io_file, entry_name );
  if (!h5io_entry) {
    char entry_name_with_path[H5SX_BUFLEN]={'\0'}; // name with path
    char entry_number[H5SX_BUFLEN]={'\0'}; // number

    if (strlib_is_empty( h5io_file->H5Path )) {
      // use name as it is
      snprintf(entry_name_with_path,H5SX_BUFLEN,"%s_%ld",ENTRYNAME,number);
    } else {
      // add separator before name
      snprintf(entry_name_with_path,H5SX_BUFLEN,"/%s_%ld",ENTRYNAME,number);
    }
    strlib_concat(entry_name_with_path,H5SX_BUFLEN,h5io_file->H5Path,entry_name_with_path);

    // create entry group in node physically 
    entry_attributes=h5sx_new_listofdsets(NULL);
    h5sx_append_dset( entry_attributes, SXCLASS, SXCLASS_ENTRY, 0, 1, -1 );

    sxentry_gid = h5sx_insert_group( h5io_file->H5id, entry_name_with_path, entry_attributes );
    if (sxentry_gid<0) goto h5sx_new_entry_error;

    // write group number pysically
    entry_number_attributes=h5sx_new_listofdsets(NULL);
    h5sx_append_dset( entry_number_attributes, SXCLASS, SXCLASS_GROUPNUMBER, 0, 1, -1 );

    num_long2str ( entry_number, H5SX_BUFLEN, number, NULL);
    if (h5sx_write_dataset_as_string(sxentry_gid, GROUPNUMBERNAME, entry_number, entry_number_attributes)<0) goto h5sx_new_entry_error;

    // write block numbering mode pysically
    blocknumbering_attributes=h5sx_new_listofdsets(NULL);
    h5sx_append_dset( blocknumbering_attributes, SXCLASS, SXCLASS_BLOCKNUMBERING, 0, 1, -1 );

    if (h5sx_write_dataset_as_string(sxentry_gid, BLOCKNUMBERINGNAME, h5io_blocknumbering2str(blocknumbering), blocknumbering_attributes)<0) goto h5sx_new_entry_error;

    // insert H5ioEntryNode
    h5io_entry = h5io_insert_node( h5io_file, entry_name, H5ioEntryNode );
    if (!h5io_entry) goto h5sx_new_entry_error;

    // write entry_name_with_path to H5Path
    h5io_entry->H5Path = strlib_newstr(entry_name_with_path);
    h5io_entry->H5id   = h5io_file->H5id; // root id of path (file)

    // attributes 
    h5sx_free_listofdsets( entry_attributes );
    h5sx_free_listofdsets( entry_number_attributes );
    h5sx_free_listofdsets( blocknumbering_attributes );

    h5sx_close_group( sxentry_gid );
  }

  status = 0;

  if (pstatus) *pstatus=status;

  if (iodbg(IODBG_H5|IODBG_DEBUG1|IODBG_DEBUG2)) printf(" h5sx_new_entry ( %p, %ld, %d  ) (%p) END\n",h5io_file,number,status,h5io_entry);

  return( h5io_entry );

h5sx_new_entry_error:

  if (iodbg(IODBG_H5|IODBG_DEBUG1|IODBG_DEBUG2)) printf(" h5sx_new_entry ( %p, %ld, %d ) (NULL) ERROR\n",h5io_file,number,status);

  // attributes 
  h5sx_free_listofdsets( entry_attributes );
  h5sx_free_listofdsets( entry_number_attributes );
  h5sx_free_listofdsets( blocknumbering_attributes );

  h5sx_close_group( sxentry_gid );

  if (pstatus) *pstatus=status;

  return( NULL );

} // h5sx_new_entry

/*---------------------------------------------------------------------------
NAME

  h5sx_new_series --- open the series corresponding to number or create it

SYNOPSIS

  H5ioNode *h5sx_new_series ( H5ioNode *h5io_entry, const char *name, int *pstatus );

DESCRIPTION

  The series name is searched in the h5io_entry->NodeList. If it does not 
  exist it is created and added to NodeList.

  H5ioNode *h5io_series
  long series_number
  int *pstatus

RETURN VALUE
success: DataList and *pstatus>=0
error:   NULL and *pstatus<0
---------------------------------------------------------------------------*/
H5ioNode *h5sx_new_series ( H5ioNode *h5io_entry, long number, int *pstatus )
{
  int status=-1;
  hid_t sxseries_gid=-1;
  H5ioNode *h5io_series=NULL;
  H5sxListOfDSets *series_attributes=NULL, *series_number_attributes=NULL;
  char series_name[H5SX_BUFLEN]={'\0'};

        // H5ioNode *h5sx_new_series ( H5ioNode *h5io_entry, long number, int *pstatus );
        // H5ioNode *h5sx_new_memory ( H5ioNode *h5io_series, long number, int *pstatus );
        // H5ioNode *h5sx_new_datablock ( H5ioNode *h5io_memory, int *pstatus );
        // H5ioNode *h5sx_new_errorblock ( H5ioNode *h5io_memory, int *pstatus );
        // H5ioNode *h5sx_new_header ( H5ioNode *h5io_block, const char *name, int *pstatus );
        // H5ioNode *h5sx_write_symbols ( H5ioNode *h5io_header, H5ioData *h5io_symbol, int *pstatus );
        // H5ioNode *h5sx_write_data ( H5ioNode *h5io_block, H5ioData *h5io_data, int *pstatus );

  if (iodbg(IODBG_H5|IODBG_DEBUG1|IODBG_DEBUG2)) printf(" h5sx_new_series ( %p, %ld, ...  ) BEGIN\n",h5io_entry,number);

  num_long2str ( series_name, H5SX_BUFLEN, number, NULL);

  h5io_series = h5io_search_node( h5io_entry, series_name );
  if (!h5io_series) {
    char series_name_with_path[H5SX_BUFLEN]={'\0'}; // name with path
    char series_number[H5SX_BUFLEN]={'\0'}; // number

    if (strlib_is_empty( h5io_entry->H5Path )) {
      // use name as it is
      snprintf(series_name_with_path,H5SX_BUFLEN,"%s_%ld",SERIESNAME,number);
    } else {
      // add separator before name
      snprintf(series_name_with_path,H5SX_BUFLEN,"/%s_%ld",SERIESNAME,number);
    }
    strlib_concat(series_name_with_path,H5SX_BUFLEN,h5io_entry->H5Path,series_name_with_path);

    // create series group in node physically 
    series_attributes=h5sx_new_listofdsets(NULL);
    h5sx_append_dset( series_attributes, SXCLASS, SXCLASS_SERIES, 0, 1, -1 );

    sxseries_gid = h5sx_insert_group( h5io_entry->H5id, series_name_with_path, series_attributes );
    if (sxseries_gid<0) goto h5sx_new_series_error;

    // write group number pysically
    series_number_attributes=h5sx_new_listofdsets(NULL);
    h5sx_append_dset( series_number_attributes, SXCLASS, SXCLASS_GROUPNUMBER, 0, 1, -1 );

    num_long2str ( series_number, H5SX_BUFLEN, number, NULL);
    if (h5sx_write_dataset_as_string(sxseries_gid, GROUPNUMBERNAME, series_number, series_number_attributes)<0) goto h5sx_new_series_error;

    // insert H5ioSeriesNode
    h5io_series = h5io_insert_node( h5io_entry, series_name, H5ioSeriesNode );
    if (!h5io_series) goto h5sx_new_series_error;

    // write series_name_with_path to H5Path
    h5io_series->H5Path = strlib_newstr(series_name_with_path);
    h5io_series->H5id   = h5io_entry->H5id; // root id of path (entry)

    // attributes 
    h5sx_free_listofdsets( series_attributes );
    h5sx_free_listofdsets( series_number_attributes );

    h5sx_close_group( sxseries_gid );
  }

  status = 0;

  if (pstatus) *pstatus=status;

  if (iodbg(IODBG_H5|IODBG_DEBUG1|IODBG_DEBUG2)) printf(" h5sx_new_series ( %p, %ld, %d  ) (%p) END\n",h5io_entry,number,status,h5io_series);

  return( h5io_series );

h5sx_new_series_error:

  if (iodbg(IODBG_H5|IODBG_DEBUG1|IODBG_DEBUG2)) printf(" h5sx_new_series ( %p, %ld, %d  ) (NULL) BEGIN\n",h5io_entry,number,status);

  // attributes 
  h5sx_free_listofdsets( series_attributes );
  h5sx_free_listofdsets( series_number_attributes );

  h5sx_close_group( sxseries_gid );

  if (pstatus) *pstatus=status;

  return( NULL );

} // h5sx_new_series

/*---------------------------------------------------------------------------
NAME

  h5sx_new_memory --- open the memory corresponding to number or create it

SYNOPSIS

  H5ioNode *h5sx_new_memory ( H5ioNode *h5io_series, const char *name, int *pstatus );

DESCRIPTION

  The memory name is searched in the h5io_series->NodeList. If it does not 
  exist it is created and added to NodeList.

  H5ioNode *h5io_memory
  long memory_number
  int *pstatus

RETURN VALUE
success: DataList and *pstatus>=0
error:   NULL and *pstatus<0
---------------------------------------------------------------------------*/
H5ioNode *h5sx_new_memory ( H5ioNode *h5io_series, long number, int *pstatus )
{
  int status=-1;
  hid_t sxmemory_gid=-1;
  H5ioNode *h5io_memory=NULL;
  H5sxListOfDSets *memory_attributes=NULL, *memory_number_attributes=NULL;
  char memory_name[H5SX_BUFLEN]={'\0'};

        // H5ioNode *h5sx_new_memory ( H5ioNode *h5io_series, long number, int *pstatus );
        // H5ioNode *h5sx_new_datablock ( H5ioNode *h5io_memory, const char *name, int *pstatus );
        // H5ioNode *h5sx_new_errorblock ( H5ioNode *h5io_memory, const char *name, int *pstatus );
        // H5ioNode *h5sx_new_header ( H5ioNode *h5io_block, const char *name, int *pstatus );
        // H5ioNode *h5sx_write_symbols ( H5ioNode *h5io_header, H5ioData *h5io_symbol, int *pstatus );
        // H5ioNode *h5sx_write_data ( H5ioNode *h5io_block, H5ioData *h5io_data, int *pstatus );

  if (iodbg(IODBG_H5|IODBG_DEBUG1|IODBG_DEBUG2)) printf(" h5sx_new_memory ( %p, %ld, ... ) BEGIN\n",h5io_series,number);

  num_long2str ( memory_name, H5SX_BUFLEN, number, NULL);

  h5io_memory = h5io_search_node( h5io_series, memory_name );
  if (!h5io_memory) {
    char memory_name_with_path[H5SX_BUFLEN]={'\0'}; // name with path
    char memory_number[H5SX_BUFLEN]={'\0'}; // number

    if (strlib_is_empty( h5io_series->H5Path )) {
      // use name as it is
      snprintf(memory_name_with_path,H5SX_BUFLEN,"%s_%ld",MEMORYNAME,number);
    } else {
      // add separator before name
      snprintf(memory_name_with_path,H5SX_BUFLEN,"/%s_%ld",MEMORYNAME,number);
    }
    strlib_concat(memory_name_with_path,H5SX_BUFLEN,h5io_series->H5Path,memory_name_with_path);

    // create memory group in node physically 
    memory_attributes=h5sx_new_listofdsets(NULL);
    h5sx_append_dset( memory_attributes, SXCLASS, SXCLASS_MEMORY, 0, 1, -1 );

    sxmemory_gid = h5sx_insert_group( h5io_series->H5id, memory_name_with_path, memory_attributes );
    if (sxmemory_gid<0) goto h5sx_new_memory_error;

    // write group number pysically
    memory_number_attributes=h5sx_new_listofdsets(NULL);
    h5sx_append_dset( memory_number_attributes, SXCLASS, SXCLASS_GROUPNUMBER, 0, 1, -1 );

    num_long2str ( memory_number, H5SX_BUFLEN, number, NULL);
    if (h5sx_write_dataset_as_string(sxmemory_gid, GROUPNUMBERNAME, memory_number, memory_number_attributes)<0) goto h5sx_new_memory_error;

    // insert H5ioMemoryNode
    h5io_memory = h5io_insert_node( h5io_series, memory_name, H5ioMemoryNode );
    if (!h5io_memory) goto h5sx_new_memory_error;

    // write memory_name_with_path to H5Path
    h5io_memory->H5Path = strlib_newstr(memory_name_with_path);
    h5io_memory->H5id   = h5io_series->H5id; // root id of path (series)

    // attributes 
    h5sx_free_listofdsets( memory_attributes );
    h5sx_free_listofdsets( memory_number_attributes );

    h5sx_close_group( sxmemory_gid );
  }

  status = 0;

  if (pstatus) *pstatus=status;

  if (iodbg(IODBG_H5|IODBG_DEBUG1|IODBG_DEBUG2)) printf(" h5sx_new_memory ( %p, %ld, %d  ) (%p) END\n",h5io_series,number,status,h5io_memory);

  return( h5io_memory );

h5sx_new_memory_error:

  if (iodbg(IODBG_H5|IODBG_DEBUG1|IODBG_DEBUG2)) printf(" h5sx_new_memory ( %p, %ld, %d  ) (NULL) ERROR\n",h5io_series,number,status);

  // attributes 
  h5sx_free_listofdsets( memory_attributes );
  h5sx_free_listofdsets( memory_number_attributes );

  h5sx_close_group( sxmemory_gid );

  if (pstatus) *pstatus=status;

  return( NULL );

} // h5sx_new_memory

/*---------------------------------------------------------------------------
NAME

  h5sx_new_datablock --- open the datablock in h5io_memory or create it

SYNOPSIS

  H5ioNode *h5sx_new_datablock ( H5ioNode *h5io_memory, int *pstatus );

DESCRIPTION

  The datablock name is searched in the h5io_memory->NodeList. If it does not 
  exist it is created and added to NodeList.

  H5ioNode *h5io_datablock
  int *pstatus

RETURN VALUE
success: DataList and *pstatus>=0
error:   NULL and *pstatus<0
---------------------------------------------------------------------------*/
H5ioNode *h5sx_new_datablock ( H5ioNode *h5io_memory, int *pstatus )
{
  int status=-1;
  hid_t sxdatablock_gid=-1;
  H5ioNode *h5io_datablock=NULL;
  H5sxListOfDSets *datablock_attributes=NULL;
  char datablock_name[H5SX_BUFLEN]={'\0'};

        // H5ioNode *h5sx_new_datablock ( H5ioNode *h5io_memory, int *pstatus );
        // H5ioNode *h5sx_new_errorblock ( H5ioNode *h5io_memory, int *pstatus );
        // H5ioNode *h5sx_new_header ( H5ioNode *h5io_block, int *pstatus );
        // H5ioNode *h5sx_write_symbols ( H5ioNode *h5io_header, H5ioData *h5io_symbol, int *pstatus );
        // H5ioNode *h5sx_write_data ( H5ioNode *h5io_block, H5ioData *h5io_data, int *pstatus );

  if (iodbg(IODBG_H5|IODBG_DEBUG1|IODBG_DEBUG2)) printf(" h5sx_new_datablock ( %p, ...  ) BEGIN\n",h5io_memory);

  snprintf( datablock_name, H5SX_BUFLEN, "%s", DATABLOCKNAME );

  h5io_datablock = h5io_search_node( h5io_memory, datablock_name );
  if (!h5io_datablock) {
    char datablock_name_with_path[H5SX_BUFLEN]={'\0'}; // name with path

    if (strlib_is_empty( h5io_memory->H5Path )) {
      // use name as it is
      snprintf(datablock_name_with_path,H5SX_BUFLEN,"%s",datablock_name);
    } else {
      // add separator before name 
      snprintf(datablock_name_with_path,H5SX_BUFLEN,"/%s",datablock_name);
    }
    strlib_concat(datablock_name_with_path,H5SX_BUFLEN,h5io_memory->H5Path,datablock_name_with_path);

    // create datablock group in node physically 
    datablock_attributes=h5sx_new_listofdsets(NULL);
    h5sx_append_dset( datablock_attributes, SXCLASS, SXCLASS_DATABLOCK, 0, 1, -1 );

    sxdatablock_gid = h5sx_insert_group( h5io_memory->H5id, datablock_name_with_path, datablock_attributes );
    if (sxdatablock_gid<0) goto h5sx_new_datablock_error;

    // insert H5ioBlockNode
    h5io_datablock = h5io_insert_node( h5io_memory, datablock_name, H5ioBlockNode );
    if (!h5io_datablock) goto h5sx_new_datablock_error;

    // write datablock_name_with_path to H5Path
    h5io_datablock->H5Path = strlib_newstr(datablock_name_with_path);
    h5io_datablock->H5id   = h5io_memory->H5id; // root id of path (series)

    // attributes 
    h5sx_free_listofdsets( datablock_attributes );

    h5sx_close_group( sxdatablock_gid );
  }

  status = 0;

  if (pstatus) *pstatus=status;

  if (iodbg(IODBG_H5|IODBG_DEBUG1|IODBG_DEBUG2)) printf(" h5sx_new_datablock ( %p, %d  ) (%p) END\n",h5io_memory,status,h5io_datablock);

  return( h5io_datablock );

h5sx_new_datablock_error:

  if (iodbg(IODBG_H5|IODBG_DEBUG1|IODBG_DEBUG2)) printf(" h5sx_new_datablock ( %p, %d  ) (NULL) ERROR\n",h5io_memory,status);

  // attributes 
  h5sx_free_listofdsets( datablock_attributes );

  h5sx_close_group( sxdatablock_gid );

  if (pstatus) *pstatus=status;

  return( NULL );

} // h5sx_new_datablock

/*---------------------------------------------------------------------------
NAME

  h5sx_new_errorblock --- open the errorblock in h5io_memory or create it

SYNOPSIS

  H5ioNode *h5sx_new_errorblock ( H5ioNode *h5io_memory, int *pstatus );

DESCRIPTION

  The errorblock name is searched in the h5io_memory->NodeList. If it does not 
  exist it is created and added to NodeList.

  H5ioNode *h5io_errorblock
  int *pstatus

RETURN VALUE
success: DataList and *pstatus>=0
error:   NULL and *pstatus<0
---------------------------------------------------------------------------*/
H5ioNode *h5sx_new_errorblock ( H5ioNode *h5io_memory, int *pstatus )
{
  int status=-1;
  hid_t sxerrorblock_gid=-1;
  H5ioNode *h5io_errorblock=NULL;
  H5sxListOfDSets *errorblock_attributes=NULL;
  char errorblock_name[H5SX_BUFLEN]={'\0'};

        // H5ioNode *h5sx_new_errorblock ( H5ioNode *h5io_memory, int *pstatus );
        // H5ioNode *h5sx_new_header ( H5ioNode *h5io_block, int *pstatus );
        // H5ioNode *h5sx_write_symbols ( H5ioNode *h5io_header, H5ioData *h5io_symbol, int *pstatus );
        // H5ioNode *h5sx_write_data ( H5ioNode *h5io_block, H5ioData *h5io_data, int *pstatus );

  if (iodbg(IODBG_H5|IODBG_DEBUG1|IODBG_DEBUG2)) printf(" h5sx_new_errorblock ( %p, ...  ) BEGIN\n",h5io_memory);

  snprintf( errorblock_name, H5SX_BUFLEN, "%s", ERRORBLOCKNAME );

  h5io_errorblock = h5io_search_node( h5io_memory, errorblock_name );
  if (!h5io_errorblock) {
    char errorblock_name_with_path[H5SX_BUFLEN]={'\0'}; // name with path

    if (strlib_is_empty( h5io_memory->H5Path )) {
      // use name as it is
      snprintf(errorblock_name_with_path,H5SX_BUFLEN,"%s",errorblock_name);
    } else { 
      // add separator before name 
      snprintf(errorblock_name_with_path,H5SX_BUFLEN,"/%s",errorblock_name);
    }
    strlib_concat(errorblock_name_with_path,H5SX_BUFLEN,h5io_memory->H5Path,errorblock_name_with_path);

    // create errorblock group in node physically 
    errorblock_attributes=h5sx_new_listofdsets(NULL);
    h5sx_append_dset( errorblock_attributes, SXCLASS, SXCLASS_ERRORBLOCK, 0, 1, -1 );

    sxerrorblock_gid = h5sx_insert_group( h5io_memory->H5id, errorblock_name_with_path, errorblock_attributes );
    if (sxerrorblock_gid<0) goto h5sx_new_errorblock_error;

    // insert H5ioBlockNode
    h5io_errorblock = h5io_insert_node( h5io_memory, errorblock_name, H5ioBlockNode );
    if (!h5io_errorblock) goto h5sx_new_errorblock_error;

    // write errorblock_name_with_path to H5Path
    h5io_errorblock->H5Path = strlib_newstr(errorblock_name_with_path);
    h5io_errorblock->H5id   = h5io_memory->H5id; // root id of path (series)

    // attributes 
    h5sx_free_listofdsets( errorblock_attributes );

    h5sx_close_group( sxerrorblock_gid );
  }

  status = 0;

  if (pstatus) *pstatus=status;

  if (iodbg(IODBG_H5|IODBG_DEBUG1|IODBG_DEBUG2)) printf(" h5sx_new_errorblock ( %p, %d ) (%p) END\n",h5io_memory,status,h5io_errorblock);

  return( h5io_errorblock );

h5sx_new_errorblock_error:

  if (iodbg(IODBG_H5|IODBG_DEBUG1|IODBG_DEBUG2)) printf(" h5sx_new_errorblock ( %p, %d ) (NULL) ERROR\n",h5io_memory,status);

  // attributes 
  h5sx_free_listofdsets( errorblock_attributes );

  h5sx_close_group( sxerrorblock_gid );

  if (pstatus) *pstatus=status;

  return( NULL );

} // h5sx_new_errorblock

/*---------------------------------------------------------------------------
NAME

  h5sx_new_header --- open the header in h5io_block or create it

SYNOPSIS

  H5ioNode *h5sx_new_header ( H5ioNode *h5io_block, int *pstatus );

DESCRIPTION

  The header name is searched in the h5io_block->NodeList. If it does not
  exist it is created and added to NodeList.

  H5ioNode *h5io_header
  int *pstatus

RETURN VALUE
success: NodeList and *pstatus>=0
error:   NULL and *pstatus<0
---------------------------------------------------------------------------*/
H5ioNode *h5sx_new_header ( H5ioNode *h5io_block, int *pstatus )
{
  int status=-1;
  hid_t header_gid=-1;
  H5ioNode *h5io_header=NULL;
  H5sxListOfDSets *header_attributes=NULL;
  char header_name[H5SX_BUFLEN]={'\0'};

        // H5ioNode *h5sx_new_header ( H5ioNode *h5io_block, int *pstatus );
        // H5ioNode *h5sx_write_symbols ( H5ioNode *h5io_header, H5ioData *h5io_symbol, int *pstatus );
        // H5ioNode *h5sx_write_data ( H5ioNode *h5io_block, H5ioData *h5io_data, int *pstatus );

  if (iodbg(IODBG_H5|IODBG_DEBUG1|IODBG_DEBUG2)) printf(" h5sx_new_header ( %p, ...  ) BEGIN\n",h5io_block);

  snprintf( header_name, H5SX_BUFLEN, "%s", HEADERNAME );

  h5io_header = h5io_search_node( h5io_block, header_name );
  if (!h5io_header) {
    char header_name_with_path[H5SX_BUFLEN]={'\0'}; // name with path

    if (strlib_is_empty( h5io_block->H5Path )) {
      // use name as it is
      snprintf(header_name_with_path,H5SX_BUFLEN,"%s",header_name);
    } else {
      // add separator before name
      snprintf(header_name_with_path,H5SX_BUFLEN,"/%s",header_name);
    }
    strlib_concat(header_name_with_path,H5SX_BUFLEN,h5io_block->H5Path,header_name_with_path);

    // create header group in node physically
    header_attributes=h5sx_new_listofdsets(NULL);
    h5sx_append_dset( header_attributes, SXCLASS, SXCLASS_HEADER, 0, 1, -1 );

    header_gid = h5sx_insert_group( h5io_block->H5id, header_name_with_path, header_attributes );
    if (header_gid<0) goto h5sx_new_header_error;

    // insert H5ioHeaderNode
    h5io_header = h5io_insert_node( h5io_block, header_name, H5ioHeaderNode );
    if (!h5io_header) goto h5sx_new_header_error;

    // write header_name_with_path to H5Path
    h5io_header->H5Path = strlib_newstr(header_name_with_path);
    h5io_header->H5id   = h5io_block->H5id; // root id of path (series)

    // attributes
    h5sx_free_listofdsets( header_attributes );

    h5sx_close_group( header_gid );
  }

  status = 0;

  if (pstatus) *pstatus=status;

  if (iodbg(IODBG_H5|IODBG_DEBUG1|IODBG_DEBUG2)) printf(" h5sx_new_header ( %p, %d  ) (%p) END\n",h5io_block,status,h5io_header);

  return( h5io_header );

h5sx_new_header_error:

  if (iodbg(IODBG_H5|IODBG_DEBUG1|IODBG_DEBUG2)) printf(" h5sx_new_header ( %p, %d  ) (NULL) ERROR\n",h5io_block,status);

  // attributes
  h5sx_free_listofdsets( header_attributes );

  h5sx_close_group( header_gid );

  if (pstatus) *pstatus=status;

  return( NULL );

} // h5sx_new_header

/*---------------------------------------------------------------------------
NAME

  h5sx_new_array --- open the array in h5io_block or create it

SYNOPSIS

  H5ioData *h5sx_new_array ( H5ioNode *h5io_block, int DataType, int ByteOrder, int DataCompression,
                             float Dummy, long *MaxDim, long *DataDim, int *pstatus );

DESCRIPTION

  The array name is searched in the h5io_block->DataList. If it does not
  exist it is created and added to DataList.
  DataCompression is currently a flag, indicating that the dataset should be compressed
  during writing.

  H5ioData *h5io_array
  int *pstatus

RETURN VALUE
success: DataList and *pstatus>=0
error:   NULL and *pstatus<0
---------------------------------------------------------------------------*/
H5ioData *h5sx_new_array ( H5ioNode *h5io_block, int DataType, int ByteOrder, int DataCompression,
                           float Dummy, long *MaxDim, long *DataDim, int *pstatus )
{
  int status=-1;
  hid_t array_id=-1;

  void *pFillValue=NULL;
  hid_t datatype_id=-1;
  hsize_t rank=-1, *dims=NULL, *maxdims=NULL;
  hsize_t itemsize=-1;
  int deflate_level=0;

  void *opFillValue=NULL;
  hid_t odatatype_id=-1;
  hsize_t orank=-1, *odims=NULL, *omaxdims=NULL;
  hsize_t oitemsize=-1;

  H5ioData *h5io_array=NULL;
  H5sxListOfDSets *array_attributes=NULL;
  char array_name[H5SX_BUFLEN]={'\0'};
  char array_name_with_path[H5SX_BUFLEN]={'\0'}; // array_name with path

  if (iodbg(IODBG_H5|IODBG_DEBUG1|IODBG_DEBUG2)) printf(" h5sx_new_array ( %p, ...  ) BEGIN\n",h5io_block);

  // convert Dummy to DataType, ByteOrder
  if (!(pFillValue = MALLOC( edf_data_sizeof(DataType) )))
    goto h5sx_new_array_error;

  if (edf_machine2machine ( pFillValue, edf_datatype2machinetype(DataType),
                            (void *) &Dummy, 0l, MFloat, 1)) {
    goto h5sx_new_array_error;
  }

  if ( ByteOrder != edf_byteorder() )
    edf_bswap ( pFillValue, pFillValue, edf_data_sizeof(DataType), 1 );

  snprintf( array_name, H5SX_BUFLEN, "%s", ARRAYNAME );

  h5io_array = h5io_search_data( h5io_block, array_name );

  if (strlib_is_empty( h5io_block->H5Path )) {
    // use name as it is
    snprintf(array_name_with_path,H5SX_BUFLEN,"%s",array_name);
  } else {
    // add separator before name
    snprintf(array_name_with_path,H5SX_BUFLEN,"/%s",array_name);
  }
  strlib_concat(array_name_with_path,H5SX_BUFLEN,h5io_block->H5Path,array_name_with_path);

  datatype_id = h5sxdatatype2dtype( DataType, ByteOrder );

  if (!h5io_array) {

    // create array dataset in node physically
    array_attributes=h5sx_new_listofdsets(NULL);
    h5sx_append_dset( array_attributes, SXCLASS, SXCLASS_ARRAY, 0, 1, -1 );
    h5sx_append_dset( array_attributes, "interpretation", "image", 0, 1, -1 );

    if (!MaxDim) goto h5sx_new_array_error;

    rank = (hsize_t) MaxDim[0]+1;
    if (!(maxdims=(hsize_t*) MALLOC( sizeof(hsize_t)*rank )))
      goto h5sx_new_array_error;

    { 
      long irank; 
      maxdims[0] = H5S_UNLIMITED; // unlimited
      for (irank=1;irank<rank;irank++) {
        maxdims[rank-irank] = MaxDim[irank]; // should be the largest dim in the cycle, or given as option
      }
    }

    if (DataCompression>1) {
      deflate_level = 6;
    }

    array_id = h5sx_dataset_create(h5io_block->H5id, array_name_with_path, rank, maxdims, datatype_id,
                      datatype_id, pFillValue, deflate_level, array_attributes );
    if (array_id<0) goto h5sx_new_array_error;

    // insert H5ioArrayNode
    h5io_array = h5io_insert_data( h5io_block, array_name, H5ioArrayNode );
    if (!h5io_array) goto h5sx_new_array_error;

  } else {
    // use existing array dataset for appending data
    // check attribute for SXCLASS, SXCLASS_ARRAY
    // check rank, maxdims, dims, DataType, ByteOrder, fillvalue

    // close open data set, this must never happen
    if (h5sx_dataset_close(h5io_array->H5data_id)) 
      goto h5sx_new_array_error;
    h5io_array->H5data_id=-1;

    array_id = h5sx_dataset_open(h5io_block->H5id, array_name_with_path,
                                 &orank, &odims, &omaxdims, &oitemsize, &odatatype_id, &opFillValue,
                                 array_attributes );
    if (array_id<0) goto h5sx_new_array_error;

    // check attribute SXCLASS, SXCLASS_ARRAY of opened dataset
    if (!h5sx_search_dset( array_attributes, SXCLASS, SXCLASS_ARRAY ))
      goto h5sx_new_array_error;

    // check rank, maxdims, itemsize, datatype_id, *pFillvalue
    // rank == orank
    // datatype_id == odatatype_id
    // *pFillvalue == *opFillvalue

  }

  // write array_name_with_path to H5Path
  h5io_array->H5Path = strlib_newstr(array_name_with_path);
  h5io_array->H5id   = h5io_block->H5id; // root id of path (series)

  // FREE(pFillValue); //++++++++++ linked to h5io_array

  // FREE(maxdims);  //++++++++++ linked to h5io_array
  // FREE(dims);  //++++++++++ linked to h5io_array

  // keep rank, dims, maxdims, itemsize, datatype_id, fillvalue , array_id

  h5io_array->DataType = DataType;
  h5io_array->ByteOrder = ByteOrder;
   h5io_array->ItemSize = edf_data_sizeof(DataType);
   h5io_array->Rank = rank;
   FREE(h5io_array->Dims); h5io_array->Dims = dims;
   FREE(h5io_array->MaxDims); h5io_array->MaxDims = maxdims;
  FREE(h5io_array->FillValue); h5io_array->FillValue = pFillValue;
  h5io_array->H5data_id = array_id;

  // attributes
  h5sx_free_listofdsets( array_attributes );

  status = 0;

  if (pstatus) *pstatus=status;

  if (iodbg(IODBG_H5|IODBG_DEBUG1|IODBG_DEBUG2)) printf(" h5sx_new_array ( %p, %d  ) (%p) END\n",h5io_block,status,h5io_array);

  return( h5io_array );

h5sx_new_array_error:

  if (iodbg(IODBG_H5|IODBG_DEBUG1|IODBG_DEBUG2)) printf(" h5sx_new_array ( %p, %d  ) (NULL) ERROR\n",h5io_block,status);

  FREE(pFillValue);
  FREE(maxdims);
  FREE(dims);

  // attributes
  h5sx_free_listofdsets( array_attributes );

  h5sx_dataset_close(array_id);

  if (pstatus) *pstatus=status;

  return( NULL );

} // h5sx_new_array

/*---------------------------------------------------------------------------
NAME

  h5sx_write_symbol --- write a key/value pair to the symbol list of h5io_header

SYNOPSIS

  H5ioData *h5sx_write_symbol ( H5ioNode *h5io_header, const char *Key, const char *Value, int *pstatus );

DESCRIPTION

  A key/value pair is inserted as symbol in h5io_header.

  H5ioNode *h5io_header
  const char *Key, *Value : key/value pair
  int *pstatus

RETURN VALUE
success: DataList and *pstatus>=0
error:   NULL and *pstatus<0
---------------------------------------------------------------------------*/
H5ioData *h5sx_write_symbol ( H5ioNode *h5io_header, const char *Key, const char *Value, int *pstatus )
{ int status=-1;

  hsize_t nframes=1;
  H5ioData *h5io_symbol=NULL;
  char symbol_name_with_path[H5SX_BUFLEN]={'\0'}; // name with path
  void **value=NULL;

  H5sxListOfDSets *symbol_attributes=NULL;

  if (iodbg(IODBG_H5|IODBG_DEBUG1)) printf("h5sx_write_symbol(h5io_header=%p,Key=%s,Value=%s,pstatus=%p) BEGIN\n",h5io_header,Key,Value,pstatus);

  if (h5io_header) {

    h5io_symbol = h5io_insert_data ( h5io_header, Key, H5ioSymbolNode );
    if (!h5io_symbol) goto h5sx_write_symbol_error;
  
    h5io_empty_data ( h5io_symbol );

    if (iodbg(IODBG_H5|IODBG_DEBUG2))
      printf(">>  value = (void **) CALLOC(%ld,%ld)",(long) nframes,(long) sizeof(void *));
    value=(void **) CALLOC(nframes,sizeof(void *));
    if (iodbg(IODBG_H5|IODBG_DEBUG2)) {
      printf(" = %p\n",value);
    }

    value[0] = (void *) strlib_newstr(Value);

    if (strlib_is_empty( h5io_header->H5Path )) {
      // use name as it is
      snprintf(symbol_name_with_path,H5SX_BUFLEN,"%s",Key);
    } else {
      // add separator before Key
      snprintf(symbol_name_with_path,H5SX_BUFLEN,"/%s",Key);
    }
    strlib_concat(symbol_name_with_path,H5SX_BUFLEN,h5io_header->H5Path,symbol_name_with_path);

    symbol_attributes=h5sx_new_listofdsets(NULL);
    h5sx_append_dset( symbol_attributes, SXCLASS, SXCLASS_SYMBOL, 0, 1, -1 );

    // herr_t h5sx_write_dataset_as_string(hid_t loc_id, const char *key, const char *val, H5sxListOfDSets *attributes)
    if (h5sx_write_dataset_as_string(h5io_header->H5id,symbol_name_with_path,Value,symbol_attributes)<0) goto h5sx_write_symbol_error;

    h5sx_free_listofdsets(symbol_attributes);

    h5io_symbol->Value = value;
    h5io_symbol->NValues = nframes;
    h5io_symbol->H5id = h5io_header->H5id;
    h5io_symbol->H5Path = strlib_newstr(symbol_name_with_path);
    h5io_symbol->Flags |= H5IO_DATA_VALUE_READ;

  }

  status = 0;

  if (pstatus) *pstatus=status;

  if (iodbg(IODBG_H5|IODBG_DEBUG1)) printf("h5sx_write_symbol(h5io_header=%p,Key=%s,Value=%s,status=%d) (h5io_symbol=%p) END\n",h5io_header,Key,Value,status,h5io_symbol);

  return( h5io_symbol );

h5sx_write_symbol_error:

  if (iodbg(IODBG_H5|IODBG_DEBUG1)) printf("h5sx_write_symbol(h5io_header=%p,Key=%s,Value=%s,status=%d) ERROR\n",h5io_header,Key,Value,status);

  h5sx_free_listofdsets(symbol_attributes);

  h5io_remove_data ( h5io_symbol );

  if (pstatus) *pstatus=status;

  return(NULL);

} // h5sx_write_symbol

/*---------------------------------------------------------------------------
NAME

  h5sx_locate_symbols --- locate all symbols and return the symbols list

SYNOPSIS

  H5ioData *h5sx_locate_symbols ( H5ioNode *h5io_header, H5sxListOfDSets *search_names, H5sxListOfDSets *search_attributes, H5sxListOfDSets *symbols, int *pstatus )

DESCRIPTION

  Searches for symbols in h5io_block starting in path and returns a pointer
  to the symbols list.

  The symbol list is released when removing the series list, i.e. when
  the stream is closed.

  H5ioNode *h5io_header
  H5sxListOfDSets *search_names
  H5sxListOfDSets *search_attributes
  H5sxListOfDSets *symbols: input: dataset names to avoid, output: names of found datasets
  int *pstatus

RETURN VALUE
success: DataList and *pstatus>=0
error:   NULL and *pstatus<0
---------------------------------------------------------------------------*/
H5ioData *h5sx_locate_symbols ( H5ioNode *h5io_header, H5sxListOfDSets *search_names, H5sxListOfDSets *search_attributes, H5sxListOfDSets *symbols, int *pstatus )
{ long level=7, indent=12;
  herr_t h5status=-1;
  int status=-1;

//++++++++  H5sxListOfDSets *symbols=NULL;

  if (iodbg(IODBG_H5|IODBG_DEBUG1)) printf("h5sx_locate_symbols( %p, %p, %p, %p ) BEGIN\n",h5io_header,search_names,search_attributes,pstatus);

  if (!h5io_header) goto h5sx_locate_symbols_error;

  if (iodbg_level()>level) {
    if (iodbg(IODBG_SHOWNODESEARCH)) {
      printf("%*sLocating symbols ...\n",(int)indent,"");
      //+++++++++++++ symbols=h5sx_new_listofdsets(NULL);
    }
  }

  // search existing symbols
  if (iodbg(IODBG_SHOWNODESEARCH||IODBG_VERBOSE)) {
    printf("%*sPath=%s\n",(int)indent+2,"",h5io_header->H5Path);
  }
  h5status=h5sx_search_dataset_by_name( h5io_header->H5id, h5io_header->H5Path, 
    H5sxSymbolSearch, search_names, search_attributes, symbols, h5io_header );
  if (h5status<0) goto h5sx_locate_symbols_error;

  if (iodbg_level()>level) {
    if (iodbg(IODBG_SHOWNODESEARCH|IODBG_VERBOSE)) h5sx_print_listofdsets( stdout, symbols, " Found symbols ...",indent);
    else if (iodbg(IODBG_SHOWNODESEARCH)) printf(" %*s%ld symbol%s found.\n",(int)indent,"",h5sx_length_listofdsets(symbols),h5sx_length_listofdsets(symbols)==1?"":"s");
  }

  //+++++++++++ symbols=h5sx_free_listofdsets(symbols);

  status=0;

  if (iodbg(IODBG_H5|IODBG_DEBUG1)) printf("h5sx_locate_symbols( %p, %p, %p, %d ) (%p) END\n",h5io_header,search_names,search_attributes,status,h5io_header->DataList);

  if (pstatus) *pstatus=status;
  return( h5io_header->DataList );

h5sx_locate_symbols_error:

  printf("h5sx_locate_symbols( %p, %p, %p, -1 ) (NULL) ERROR\n",h5io_header,search_names,search_attributes);

  //+++++++++++ h5sx_free_listofdsets(symbols);

  h5io_remove_data_list( h5io_header );

  if (pstatus) *pstatus=status;
  return( NULL );

} // h5sx_locate_symbols

/*---------------------------------------------------------------------------
NAME

  h5sx_read_symbol_value --- reads the value of the symbol

SYNOPSIS

  H5ioData *h5sx_read_symbol_value ( H5ioData *symbol );

DESCRIPTION

  Read the values of the data node if it is a symbol.
  Returns the pointer to the symbol or NONE if it is not a symbol.

RETURN VALUE
  Success: pointer to symbol structure
  Error: NONE 
---------------------------------------------------------------------------*/
H5ioData *h5sx_read_symbol_value ( H5ioData *data )
{
  if (iodbg(IODBG_H5|IODBG_DEBUG1)) printf("h5sx_read_symbol_value( %p ) BEGIN\n",data);

  if (data) {
    char **string_array=NULL;
    hsize_t nframes=0;

    if (data->DataClass == H5ioSymbolNode) {

      // char **h5sx_read_value_as_string( hid_t loc_id, const char *key, H5sxListOfDSets *attributes, hsize_t *nframes );
      string_array = h5sx_read_value_as_string( data->H5id, data->H5Path, NULL, &nframes );

      if (string_array) {
        data->Value=(void **) string_array;
        data->NValues=nframes;
        data->Flags |= H5IO_DATA_VALUE_READ;
      }
    }
  }
  if (iodbg(IODBG_H5|IODBG_DEBUG1)) printf("h5sx_read_symbol_value( %p ) (%p) END\n",data,data);

  return( data );

} /* h5sx_read_symbol_value */

/*---------------------------------------------------------------------------
NAME

  h5sx_read_symbols --- reads the values of all symbols in h5io_block

SYNOPSIS

  H5ioData *h5sx_read_symbols ( H5ioNode *node );

DESCRIPTION

  Reads the values of all data sets in node with class symbols and
  returns the pointer to the data list of node.

RETURN VALUE
  DataList
---------------------------------------------------------------------------*/
H5ioData *h5sx_read_symbols ( H5ioNode *node )
{ 
  H5ioData *datalist=NULL;

  if (node) {
    H5ioData *h5io_symbol=NULL;

    datalist = node->DataList;

    h5io_symbol = datalist;
    while (h5io_symbol) {
      h5sx_read_symbol_value ( h5io_symbol );
      h5io_symbol = h5io_symbol->Next;
    }
  }

  return( datalist );

} /* h5sx_read_symbols */

/*---------------------------------------------------------------------------
NAME

  h5sx_locate_headers --- locate all headers and return the headers list

SYNOPSIS

  H5ioNode   *h5sx_locate_headers ( H5ioNode  *h5io_block, H5sxListOfDSets *search_names, H5sxListOfDSets *search_attributes, int *pstatus );

DESCRIPTION

  Searches for headers in h5io_block starting in path and returns a pointer
  to the header.

  The header list is released when removing the series list, i.e. when
  the stream is closed.

  H5ioNode *h5io_block
  H5sxListOfDSets *search_names
  H5sxListOfDSets *search_attributes
  int *pstatus

RETURN VALUE
success: NodeList and *pstatus>=0
error:   NULL and *pstatus<0
---------------------------------------------------------------------------*/
H5ioNode   *h5sx_locate_headers ( H5ioNode  *h5io_block, H5sxListOfDSets *search_names, H5sxListOfDSets *search_attributes, int *pstatus )
{ long level=6, indent=10;
  herr_t h5status=-1;
  int status=-1;

  H5sxListOfDSets *headers=NULL;

  if (iodbg(IODBG_H5|IODBG_DEBUG1)) printf("h5sx_locate_headers( %p, %p, %p, %p ) BEGIN\n",h5io_block,search_names,search_attributes,pstatus);

  if (!h5io_block) goto h5sx_locate_headers_error;

  if (iodbg_level()>level) {
    if (iodbg(IODBG_SHOWNODESEARCH)) {
      printf("%*sLocating headers ...\n",(int)indent,"");
      if (iodbg(IODBG_VERBOSE)) {
        h5sx_print_listofdsets( stdout, search_names, "search_names", indent+2 );
        h5sx_print_listofdsets( stdout, search_attributes, "search_attributes", indent+2 );
      }
      headers=h5sx_new_listofdsets(NULL);
    }
  }

  h5status=h5sx_search_group_by_name( h5io_block->H5id, h5io_block->H5Path, H5sxHeaderSearch, 
    search_names, search_attributes, headers, h5io_block );
  if (h5status<0) goto h5sx_locate_headers_error;

  if (iodbg_level()>level) {
    if (iodbg(IODBG_SHOWNODESEARCH|IODBG_VERBOSE)) h5sx_print_listofdsets( stdout, headers, " Found headers ...",indent);
    else if (iodbg(IODBG_SHOWNODESEARCH)) printf(" %*s%ld header%s found.\n",(int)indent,"",h5sx_length_listofdsets(headers), h5sx_length_listofdsets(headers)==1?"":"s");
  }

  headers=h5sx_free_listofdsets(headers);

  status=0;

  if (iodbg(IODBG_H5|IODBG_DEBUG1)) printf("h5sx_locate_headers( %p, %p, %p, %d ) (%p) END\n",h5io_block,search_names,search_attributes,status,h5io_block->NodeList);

  if (pstatus) *pstatus=status;

  return( h5io_block->NodeList );

h5sx_locate_headers_error:

  printf("h5sx_locate_headers( %p, %p, %p, %d ) (NULL) ERROR\n",h5io_block,search_names,search_attributes,status);

  h5sx_free_listofdsets(headers);

  h5io_remove_node_list( h5io_block );

  if (pstatus) *pstatus=status;
  return( NULL );

} // h5sx_locate_headers

/*---------------------------------------------------------------------------
NAME

  h5sx_locate_arrays --- locate data arrays and return pointer

SYNOPSIS

  H5ioData *h5sx_locate_arrays ( H5ioNode  *h5io_block, 
                                 H5sxListOfDSets *search_names, 
                                 H5sxListOfDSets *search_attributes, 
                                 H5sxListOfDSets *arrays, int *pstatus )

DESCRIPTION

  Searches for a data array in h5io_block starting in path and returning
  a pointer to the data array.

  The data array is released when removing the series list, i.e. when
  the stream is closed.

  H5ioNode *h5io_block
  H5sxListOfDSets *search_names
  H5sxListOfDSets *search_attributes
  H5sxListOfDSets *arrays: input: dataset names to avoid, output: names of found datasets
  int *pstatus

RETURN VALUE
success: DataList and *pstatus>=0
error:   NULL and *pstatus<0
---------------------------------------------------------------------------*/
H5ioData *h5sx_locate_arrays ( H5ioNode  *h5io_block, H5sxListOfDSets *search_names, H5sxListOfDSets *search_attributes, H5sxListOfDSets *arrays, int *pstatus )
{ long level=6, indent=10;
  herr_t h5status=-1;
  int status=-1;

  // +++++++++++H5sxListOfDSets *arrays=NULL;

  if (iodbg(IODBG_H5|IODBG_DEBUG1)) printf("h5sx_locate_arrays( %p, %p, %p, %p ) BEGIN\n",h5io_block,search_names,search_attributes,pstatus);

  if (!h5io_block) goto h5sx_locate_arrays_error;

  if (iodbg_level()>level) {
    if (iodbg(IODBG_SHOWNODESEARCH)) {
      printf("%*sLocating arrays ...\n",(int)indent,"");
      if (iodbg(IODBG_VERBOSE)) {
        h5sx_print_listofdsets( stdout, search_names, "search_names", indent+2 );
        h5sx_print_listofdsets( stdout, search_attributes, "search_attributes", indent+2 );
      }
      //+++++++++++ arrays=h5sx_new_listofdsets(NULL);
    }
  }

  if ( h5sx_search_dset( search_names, NULL, "Skip" )) { 
    if (iodbg(IODBG_H5|IODBG_DEBUG2)) printf("h5sx_locate_arrays: Skip arrays\n");
  } else {
    // search existing arrays
    if (iodbg(IODBG_SHOWNODESEARCH||IODBG_VERBOSE)) {
      printf("%*sPath=%s\n",(int)indent+2,"",h5io_block->H5Path);
    }
    h5status=h5sx_search_dataset_by_name( h5io_block->H5id, h5io_block->H5Path, H5sxArraySearch, 
      search_names, search_attributes, arrays, h5io_block );
    if (h5status<0) goto h5sx_locate_arrays_error;
  }

  if (iodbg_level()>level) {
    if (iodbg(IODBG_SHOWNODESEARCH|IODBG_VERBOSE)) h5sx_print_listofdsets( stdout, arrays, " Found arrays ...",indent);
    else if (iodbg(IODBG_SHOWNODESEARCH)) printf(" %*s%ld array%s found.\n",(int)indent,"",h5sx_length_listofdsets(arrays), h5sx_length_listofdsets(arrays)==1?"":"s");
  }

  //++++++++++ arrays=h5sx_free_listofdsets(arrays);

  status=0;

  if (iodbg(IODBG_H5|IODBG_DEBUG1)) printf("h5sx_locate_arrays( %p, %p, %p, %d ) (%p) END\n",h5io_block,search_names,search_attributes,status,h5io_block->DataList);

  if (pstatus) *pstatus=status;
  return( h5io_block->DataList );

h5sx_locate_arrays_error:

  //++++++++++++h5sx_free_listofdsets(arrays);

  h5io_remove_data_list( h5io_block );

  if (iodbg(IODBG_H5|IODBG_DEBUG1)) printf("h5sx_locate_arrays( %p, %p, %p, %d ) (NULL) END\n",h5io_block,search_names,search_attributes,status);

  if (pstatus) *pstatus=status;
  return(NULL);

} // h5sx_locate_arrays

/*---------------------------------------------------------------------------
NAME

  h5sx_read_array_value_native --- reads the value of the array

SYNOPSIS

  H5ioData *h5sx_read_array_value_native ( H5ioData *array, int mode );

DESCRIPTION

  Reads the values of the data node if it is an array.
  Returns the pointer to the array or NULL if it is not a array.
  If mode is 0, only the array data descriptions are read and updated. The
  array values are not read.

  The array values can be released with h5sx_empty_data_value and 
  h5sx_read_array_value_native can be used again.

  The function can be applied several times on the same data block.
  At each call the pointers to Binary, Values, Dims and MaxDims will 
  be updated. The previous ones are not valid any more.
  Each time, previously allocated memory will be released.

  The function returns the binary array and interpretated values in
  the table Value[] with the length NValues.
  For ValueRank==0 each Value[] points to a single scalar value.
  For ValueRank>0 each Value[] points to an array with the rank
  ValueRank-1 and with dimensions specified in ValueDims[1], ValueDims[2],
  etc. ValueDims[0] is always set to NValues.

  If binary is not NULL the elements of Value[] are pointing to
  subarrays inside binary. If binary is NULL, the destination of each 
  element in Value[0..NValues-1] must be released individually.

  If the dataset attribute interpretation is "image" each element of 
  Value[] points to a 2-dimensional image with the dimension ValueDims[1]
  and ValueDims[2].
  If the dataset attribute interpretation is "spectrum" each element of 
  Value[] points to a 1-dimensional spectrum with the dimension ValueDims[1].
  If the dataset attribute interpretation is "scalar" each element of 
  Value[] points to a single scalar value.

  The minimum length of ValueDims is 1. The element ValueDims[0] exists 
  always. A single scalar value is written to a (linear) array containing
  a single element.

  Currently, the default interpretation is set to "image". This behaviour
  could be changed by adding the parameter "interpretation" to the parameter
  list of this function.

RETURN VALUE
  Success: pointer to array structure
  Error: NONE
---------------------------------------------------------------------------*/
H5ioData *h5sx_read_array_value_native ( H5ioData *data, int mode )
{ int status = -1;

  if (iodbg(IODBG_H5|IODBG_DEBUG1)) printf("h5sx_read_array_value_native( %p, mode=%d ) BEGIN\n",data,mode);

  void   **Value=NULL;
  hsize_t *ValueDims=NULL;
  hsize_t  ValueRank=0;
  H5sxListOfDSets *attributes=h5sx_new_listofdsets(NULL);

  if (data) {
    int datatype=InValidDType;
    void *binary=NULL;
    hsize_t binarysize=0;
    hsize_t itemsize=0, rank=0, *dims=NULL, *maxdims=NULL;
    hsize_t pitch=0;

    if (data->DataClass == H5ioArrayNode) {

      // void *h5sx_read_dataset_as_native_array( hid_t loc_id, const char *key, H5sxListOfDSets *attributes, 
      //    DType *datatype, hsize_t *itemsize, hsize_t *rank, hsize_t **dims, hsize_t **maxdims, void **binary, hsize_t *binarysize )

      if (mode) {
        status = h5sx_read_dataset_as_native_array( data->H5id, data->H5Path, attributes, &datatype, &itemsize, &rank, &dims, &maxdims, &binary, &binarysize );
      } else {
        status = h5sx_read_dataset_as_native_array( data->H5id, data->H5Path, attributes, &datatype, &itemsize, &rank, &dims, &maxdims, NULL, &binarysize );
      } 

      if (!status) {
        hsize_t nframes=0;
        int interpretation=H5ioInValidInterpretation;

        if ( data->Flags&H5IO_DATA_DESCRIPTION_READ ) {
          /* 
           * release previously allocated dims
           */
          if (dims) FREE(data->Dims);
          if (maxdims) FREE(data->MaxDims);
        }

        data->DataType    = datatype;  // use DataType, item type is only defined while object is open
        data->ItemSize    = itemsize;
        data->ByteOrder   = edf_byteorder(); // byte order of the machine
        data->Rank        = rank;
        data->Dims        = dims;
        data->MaxDims     = maxdims;

        if ( h5sx_search_dset( attributes, "interpretation", "image" ) )  
          interpretation=h5io_str2interpretation("image");
        else if ( h5sx_search_dset( attributes, "interpretation", "spectrum" ) )
          interpretation=h5io_str2interpretation("spectrum");
        else if ( h5sx_search_dset( attributes, "interpretation", "scalar" ) )
          interpretation=h5io_str2interpretation("scalar");
        else interpretation=H5ioImage; // H5ioInValidInterpretation

        nframes = 1;      // start value
        pitch = itemsize; // start value

        switch (interpretation) {
        case H5ioImage :
          /*
           * Value[] contains nframes elements with pointers to images
           */
          if (iodbg(IODBG_H5|IODBG_DEBUG2)) printf(">>  interpretation = image\n");

          ValueRank=3;
          if (iodbg(IODBG_H5|IODBG_DEBUG2))
            printf(">>  ValueDims = (hsize_t*) MALLOC(%ld*%ld)",(long) ValueRank, (long) sizeof(hsize_t));
          ValueDims=(hsize_t*) MALLOC( sizeof(hsize_t)*ValueRank );
          if (iodbg(IODBG_H5|IODBG_DEBUG2)) {
            printf(" = %p\n",ValueDims);
          }
          if (!ValueDims) goto h5sx_read_array_value_native_error;

          if (rank>2) {
            hsize_t idim;
            for (idim=0;idim<rank-2;idim++) {
              nframes *= dims[idim];
            }
            ValueDims[1]=dims[rank-2]; ValueDims[2]=dims[rank-1];
            pitch *= maxdims[rank-2]*maxdims[rank-1];
          } else if (rank==2) {
            ValueDims[1]=dims[0]; ValueDims[2]=dims[1];
            pitch *= maxdims[0]*maxdims[1];
          } else if (rank==1) {
            ValueDims[1]=1; ValueDims[2]=dims[0];
            pitch *= maxdims[0];
          } else {
            ValueDims[1]=1; ValueDims[2]=1;
          }
          ValueDims[0]=nframes;
          break;

        case H5ioSpectrum:
          /*
           * Value[] contains nframes elements with pointers to spectra
           */
          if (iodbg(IODBG_H5|IODBG_DEBUG2)) printf(">>  interpretation = spectrum\n");

          ValueRank=2;
          if (iodbg(IODBG_H5|IODBG_DEBUG2))
            printf(">>  ValueDims = (hsize_t*) MALLOC(%ld*%ld)",(long) ValueRank, (long) sizeof(hsize_t));
          ValueDims=(hsize_t*) MALLOC( sizeof(hsize_t)*ValueRank );
          if (iodbg(IODBG_H5|IODBG_DEBUG2)) {
            printf(" = %p\n",ValueDims);
          }
          if (!ValueDims) goto h5sx_read_array_value_native_error;

          if (rank>1) {
            hsize_t idim;
            for (idim=0;idim<rank-1;idim++) {
              nframes *= dims[idim];
            }
            ValueDims[1]=dims[rank-1];
            pitch *= maxdims[rank-1];
          } else if (rank==1) {
            ValueDims[1]=dims[0];
            pitch *= maxdims[0];
          } else {
            ValueDims[1]=1; 
          }
          ValueDims[0]=nframes;
          break;

        case H5ioScalar:
          /*
           * Value[] contains nframes elements with pointers to scalars
           */
          if (iodbg(IODBG_H5|IODBG_DEBUG2)) printf(">>  interpretation = scalar\n");

          ValueRank=1;
          if (iodbg(IODBG_H5|IODBG_DEBUG2))
            printf(">>  ValueDims = (hsize_t*) MALLOC(%ld*%ld)",(long) ValueRank, (long) sizeof(hsize_t));
          ValueDims=(hsize_t*) MALLOC( sizeof(hsize_t)*ValueRank );
          if (iodbg(IODBG_H5|IODBG_DEBUG2)) {
            printf(" = %p\n",ValueDims);
          }
          if (!ValueDims) goto h5sx_read_array_value_native_error;

          if (rank>0) {
            hsize_t idim;
            for (idim=0;idim<rank;idim++) {
              nframes *= dims[idim];
            }
          }
          ValueDims[0]=nframes;
          break;

        default:
          if (iodbg(IODBG_H5|IODBG_DEBUG2)) printf(">>  no interpretation attribute\n");
          nframes=rank>0?dims[0]:1;
          ValueRank=rank;
          if (ValueRank==0) ValueRank=1; // at least on element in Value
          if (ValueRank>0) {
            hsize_t idim;
            if (iodbg(IODBG_H5|IODBG_DEBUG2))
              printf(">>  ValueDims = (hsize_t*) MALLOC(%ld*%ld)",(long) ValueRank, (long) sizeof(hsize_t));
            ValueDims=(hsize_t*) MALLOC( sizeof(hsize_t)*ValueRank );
            if (iodbg(IODBG_H5|IODBG_DEBUG2)) {
              printf(" = %p\n",ValueDims);
            }
            if (!ValueDims) goto h5sx_read_array_value_native_error;

            for (idim=1;idim<rank;idim++) {
              ValueDims[idim] = dims[idim];
            }
            ValueDims[0]=nframes;
          }
          pitch = itemsize*h5io_pitch0(rank, maxdims);
        } // switch

        data->NValues     = nframes;
        data->ValueRank   = ValueRank;
        FREE(data->ValueDims);
        data->ValueDims   = ValueDims;
        
        data->Flags      |= H5IO_DATA_DESCRIPTION_READ;

        if (binary) {
          if ( data->Flags&H5IO_DATA_VALUE_READ ) {
            h5io_empty_data ( data );
          }

          data->Binary      = binary;
          data->Flags      |= H5IO_DATA_VALUE_READ;
          data->BinarySize  = binarysize;

          hsize_t i;

          if (iodbg(IODBG_H5|IODBG_DEBUG2))
            printf(">>  Value = (void **) CALLOC(%ld,%ld)",(long) nframes,(long) sizeof(void *));
          Value=(void **) CALLOC(nframes,sizeof(void *));
          if (iodbg(IODBG_H5|IODBG_DEBUG2)) {
            printf(" = %p\n",Value);
          }
          if (!Value) goto h5sx_read_array_value_native_error;

          for (i=0;i<nframes;i++) {
            Value[i] = (void *) ( (char*) binary + (pitch*i) );
          }
          data->Value = Value; // previous allocations have been released with h5io_empty_data
        }

      }
    }
  }

  attributes = h5sx_free_listofdsets(attributes);

  if (iodbg(IODBG_H5|IODBG_DEBUG1)) printf("h5sx_read_array_value_native( %p, mode=%d ) (%p) END\n",data,mode,data);

  return( data );

h5sx_read_array_value_native_error:

  attributes = h5sx_free_listofdsets(attributes);

  if (iodbg(IODBG_H5|IODBG_DEBUG1)) printf("h5sx_read_array_value_native( %p, mode=%d ) (%p) ERROR\n",data,mode,data);

  FREE(ValueDims);
  FREE(Value);

  return( NULL );

} /* h5sx_read_array_value_native */

/*---------------------------------------------------------------------------
NAME

  h5sx_empty_data_value --- empties the data value

SYNOPSIS

  H5ioData *h5sx_empty_data_value ( H5ioData *data );

DESCRIPTION

  Releases the memory used for binary data and values. It resets the
  H5IO_DATA_DESCRIPTION_READ flag and sets BinarySize to -1. Binary
  and Value are set to NULL.

RETURN VALUE
  Success: pointer to array structure
  Error: NULL 
---------------------------------------------------------------------------*/
H5ioData *h5sx_empty_data_value ( H5ioData *data ) 
{
  if (iodbg(IODBG_H5|IODBG_DEBUG1)) printf("h5sx_empty_data_value( %p ) BEGIN\n",data);

  h5io_empty_data  ( data );

  if (iodbg(IODBG_H5|IODBG_DEBUG1)) printf("h5sx_empty_data_value( %p ) (%p) END\n",data,data);

  return(data);
} /* h5sx_empty_data_value */

/*---------------------------------------------------------------------------
NAME

  h5sx_locate_datablocks --- locate all data blocks and return the block list

SYNOPSIS

  H5ioNode  *h5sx_locate_datablocks ( H5ioNode   *h5io_memory, H5sxListOfDSets *search_names, H5sxListOfDSets *search_attributes, int *pstatus );

DESCRIPTION

  Searches for blocks in h5io_memory starting in path and returning
  a pointer to the blocks list.

  The block list is released when removing the series list, i.e. when
  the stream is closed.

  H5ioNode *h5io_block
  H5sxListOfDSets *search_names
  H5sxListOfDSets *search_attributes
  int *pstatus

RETURN VALUE
success: NodeList and *pstatus>=0
error:   NULL and *pstatus<0
---------------------------------------------------------------------------*/
H5ioNode  *h5sx_locate_datablocks ( H5ioNode *h5io_memory, H5sxListOfDSets *search_names, H5sxListOfDSets *search_attributes, int *pstatus )
{ long level=5, indent=8;
  herr_t h5status=-1;
  int status=-1;

  H5sxListOfDSets *blocks=NULL;

  if (iodbg(IODBG_H5|IODBG_DEBUG1)) printf("h5sx_locate_datablocks( %p, %p, %p, %p ) BEGIN\n",h5io_memory,search_names,search_attributes,pstatus);

  if (!h5io_memory) goto h5sx_locate_datablocks_error;

  if (iodbg_level()>level) {
    if (iodbg(IODBG_SHOWNODESEARCH)) {
      printf("%*sLocating data blocks ...\n",(int)indent,"");
      if (iodbg(IODBG_VERBOSE)) {
        h5sx_print_listofdsets( stdout, search_names, "search_names", indent+2 );
        h5sx_print_listofdsets( stdout, search_attributes, "search_attributes", indent+2 );
      }
      blocks=h5sx_new_listofdsets(NULL);
    }
  }

  h5status=h5sx_search_group_by_name( h5io_memory->H5id, h5io_memory->H5Path, 
    H5sxDataBlockSearch, search_names, search_attributes, blocks, h5io_memory );
  if (h5status<0) goto h5sx_locate_datablocks_error;

  if (iodbg_level()>level) {
    if (iodbg(IODBG_SHOWNODESEARCH|IODBG_VERBOSE)) h5sx_print_listofdsets( stdout, blocks, " Found data blocks ...",indent);
    else if (iodbg(IODBG_SHOWNODESEARCH)) printf(" %*s%ld data block%s found.\n",(int)indent,"",h5sx_length_listofdsets(blocks), h5sx_length_listofdsets(blocks)==1?"":"s");
  }

  blocks=h5sx_free_listofdsets(blocks);

  status=0;

  if (iodbg(IODBG_H5|IODBG_DEBUG1)) printf("h5sx_locate_datablocks( %p, %p, %p, %d ) (%p) END\n",h5io_memory,search_names,search_attributes,status,h5io_memory->NodeList);

  if (pstatus) *pstatus=status;
  return( h5io_memory->NodeList );

h5sx_locate_datablocks_error:

  h5sx_free_listofdsets(blocks);

  h5io_remove_node_list( h5io_memory );

  printf("h5sx_locate_datablocks( %p, %p, %p, %d ) (NULL) ERROR\n",h5io_memory,search_names,search_attributes,status);

  if (pstatus) *pstatus=status;
  return( NULL );

} // h5sx_locate_datablocks

/*---------------------------------------------------------------------------
NAME

  h5sx_locate_errorblocks --- locate all error blocks and return the block list

SYNOPSIS

  H5ioNode  *h5sx_locate_errorblocks ( H5ioNode   *h5io_memory, H5sxListOfDSets *search_names, H5sxListOfDSets *search_attributes, int *pstatus );

DESCRIPTION

  Searches for blocks in h5io_memory starting in path and returning
  a pointer to the blocks list.

  The block list is released when removing the series list, i.e. when
  the stream is closed.

  H5ioNode *h5io_block
  H5sxListOfDSets *search_names
  H5sxListOfDSets *search_attributes
  int *pstatus

RETURN VALUE
success: NodeList and *pstatus>=0
error:   NULL and *pstatus<0
---------------------------------------------------------------------------*/
H5ioNode  *h5sx_locate_errorblocks ( H5ioNode *h5io_memory, H5sxListOfDSets *search_names, H5sxListOfDSets *search_attributes, int *pstatus )
{ long level=5, indent=8;
  herr_t h5status=-1;
  int status=-1;

  H5sxListOfDSets *blocks=NULL;

  if (iodbg(IODBG_H5|IODBG_DEBUG1)) printf("h5sx_locate_errorblocks( %p, %p, %p, %p ) BEGIN\n",h5io_memory,search_names,search_attributes,pstatus);

  if (!h5io_memory) goto h5sx_locate_errorblocks_error;

  if (iodbg_level()>level) {
    if (iodbg(IODBG_SHOWNODESEARCH)) {
      printf("%*sLocating error blocks ...\n",(int)indent,"");
      if (iodbg(IODBG_VERBOSE)) {
        h5sx_print_listofdsets( stdout, search_names, "search_names", indent+2 );
        h5sx_print_listofdsets( stdout, search_attributes, "search_attributes", indent+2 );
      }
      blocks=h5sx_new_listofdsets(NULL);
    }
  }

  h5status=h5sx_search_group_by_name( h5io_memory->H5id, h5io_memory->H5Path,
    H5sxErrorBlockSearch, search_names, search_attributes, blocks, h5io_memory );
  if (h5status<0) goto h5sx_locate_errorblocks_error;

  if (iodbg_level()>level) {
    if (iodbg(IODBG_SHOWNODESEARCH|IODBG_VERBOSE)) h5sx_print_listofdsets( stdout, blocks, " Found error blocks ...",indent);
    else if (iodbg(IODBG_SHOWNODESEARCH)) printf(" %*s%ld error block%s found.\n",(int)indent,"",h5sx_length_listofdsets(blocks), h5sx_length_listofdsets(blocks)==1?"":"s");
  }

  blocks=h5sx_free_listofdsets(blocks);

  status=0;

  if (iodbg(IODBG_H5|IODBG_DEBUG1)) printf("h5sx_locate_errorblocks( %p, %p, %p, %d ) (%p) END\n",h5io_memory,search_names,search_attributes,status,h5io_memory->NodeList);

  return( h5io_memory->NodeList );

h5sx_locate_errorblocks_error:

  printf("h5sx_locate_errorblocks( %p, %p, %p, %d ) (NULL) ERROR\n",h5io_memory,search_names,search_attributes,status);

  h5sx_free_listofdsets(blocks);

  h5io_remove_node_list( h5io_memory );

  return( NULL );

} // h5sx_locate_errorblocks

/*---------------------------------------------------------------------------
NAME

  h5sx_locate_memories --- locate all memories and return the memories list

SYNOPSIS

  H5ioNode   *h5sx_locate_memories ( H5ioNode   *h5io_series, H5sxListOfDSets *search_names, H5sxListOfDSets *search_attributes, int *pstatus );

DESCRIPTION

  Searches for memories in h5io_series starting in path and returns a pointer
  to the memories list.

  The memory list is released when removing the series list, i.e. when
  the stream is closed.

  H5ioNode *h5io_block
  H5sxListOfDSets *search_names
  H5sxListOfDSets *search_attributes
  int *pstatus

RETURN VALUE
success: NodeList and *pstatus>=0
error:   NULL and *pstatus<0
---------------------------------------------------------------------------*/
H5ioNode *h5sx_locate_memories ( H5ioNode *h5io_series, H5sxListOfDSets *search_names, H5sxListOfDSets *search_attributes, int *pstatus )
{ long level=4, indent=6;
  herr_t h5status=-1;
  int status=-1;

  H5sxListOfDSets *memories=NULL;

  if (iodbg(IODBG_H5|IODBG_DEBUG1)) printf("h5sx_locate_memories( %p, %p, %p, %p ) BEGIN\n",h5io_series,search_names,search_attributes,pstatus);

  if (!h5io_series) goto h5sx_locate_memories_error;

  if (iodbg_level()>level) {
    if (iodbg(IODBG_SHOWNODESEARCH)) {
      printf("%*sLocating memories ...\n",(int)indent,"");
      if (iodbg(IODBG_VERBOSE)) {
        h5sx_print_listofdsets( stdout, search_names, "search_names", indent+2 );
        h5sx_print_listofdsets( stdout, search_attributes, "search_attributes", indent+2 );
      }
      memories=h5sx_new_listofdsets(NULL);
    }
  }

  // search existing memories
  if (iodbg(IODBG_SHOWNODESEARCH||IODBG_VERBOSE)) {
    printf("%*sPath=%s\n",(int)indent+2,"",h5io_series->H5Path);
  }
  h5status=h5sx_search_group_by_name( h5io_series->H5id, h5io_series->H5Path, 
    H5sxMemorySearch, search_names, search_attributes, memories, h5io_series );
  if (h5status<0) goto h5sx_locate_memories_error;

  if (iodbg_level()>level) {
    if (iodbg(IODBG_SHOWNODESEARCH|IODBG_VERBOSE)) h5sx_print_listofdsets( stdout, memories, " Found memories ...",indent);
    else if (iodbg(IODBG_SHOWNODESEARCH)) printf(" %*s%ld memor%s found.\n",(int)indent,"",h5sx_length_listofdsets(memories), h5sx_length_listofdsets(memories)==1?"y":"ies");
  }

  memories=h5sx_free_listofdsets(memories);

  status=0;

  if (iodbg(IODBG_H5|IODBG_DEBUG1)) printf("h5sx_locate_memories( %p, %p, %p, %d ) (%p) END\n",h5io_series,search_names,search_attributes,status,h5io_series->NodeList);

  if (pstatus) *pstatus=status;
  return( h5io_series->NodeList );

h5sx_locate_memories_error:

  printf("h5sx_locate_memories( %p, %p, %p, %d ) (NULL) ERROR\n",h5io_series,search_names,search_attributes,status);

  h5sx_free_listofdsets(memories);

  h5io_remove_node_list( h5io_series );

  if (pstatus) *pstatus=status;
  return( NULL );

} // h5sx_locate_memories

/*---------------------------------------------------------------------------
NAME

  h5sx_locate_series --- locate all series and return the series list

SYNOPSIS

  H5ioNode *h5sx_locate_series ( H5ioNode *h5io_entry, H5sxListOfDSets *search_names, H5sxListOfDSets *search_attributes, int *pstatus );

DESCRIPTION

  Searches for series in h5io_entry in path and returns a pointer
  to the series list.

  The series list is released when removing the entry list, i.e. the stream 
  is closed.

  H5ioNode *h5io_block
  H5sxListOfDSets *search_names
  H5sxListOfDSets *search_attributes
  int *pstatus

RETURN VALUE
success: NodeList and *pstatus>=0
error:   NULL and *pstatus<0
---------------------------------------------------------------------------*/
H5ioNode *h5sx_locate_series ( H5ioNode *h5io_entry, H5sxListOfDSets *search_names, H5sxListOfDSets *search_attributes, int *pstatus )
{ long level=3, indent=4;
  herr_t h5status=-1;
  int status=-1;

  H5sxListOfDSets *series=NULL;

  if (iodbg(IODBG_H5|IODBG_DEBUG1)) printf("h5sx_locate_series( %p ) BEGIN\n",h5io_entry);

  if (!h5io_entry) goto h5sx_locate_series_error;

  if (iodbg_level()>level) {
    if (iodbg(IODBG_SHOWNODESEARCH)) {
      printf("%*sLocating series ...\n",(int)indent,"");
      if (iodbg(IODBG_VERBOSE)) {
        h5sx_print_listofdsets( stdout, search_names, "search_names", indent+2 );
        h5sx_print_listofdsets( stdout, search_attributes, "search_attributes", indent+2 );
      }
      series=h5sx_new_listofdsets(NULL);
    }
  }

  // search existing series
  if (iodbg(IODBG_SHOWNODESEARCH||IODBG_VERBOSE)) {
    printf("%*sPath=%s\n",(int)indent+2,"",h5io_entry->H5Path);
  }
  h5status=h5sx_search_group_by_name( h5io_entry->H5id, h5io_entry->H5Path, 
    H5sxSeriesSearch, search_names, search_attributes, series, h5io_entry );
  if (h5status<0) goto h5sx_locate_series_error;

  if (iodbg_level()>level) {
    if (iodbg(IODBG_SHOWNODESEARCH|IODBG_VERBOSE)) h5sx_print_listofdsets( stdout, series, " Found series ...",indent);
    else if (iodbg(IODBG_SHOWNODESEARCH)) printf(" %*s%ld series found.\n",(int)indent,"",h5sx_length_listofdsets(series));
  }

  series=h5sx_free_listofdsets(series);

  status=0;

  if (iodbg(IODBG_H5|IODBG_DEBUG1)) printf("h5sx_locate_series( %p, %p, %p, %d ) (%p) END\n",h5io_entry,search_names,search_attributes,status,h5io_entry->NodeList);

  if (pstatus) *pstatus=status;
  return( h5io_entry->NodeList );

h5sx_locate_series_error:

  printf("h5sx_locate_series( %p, %p, %p, %d ) (NULL) ERROR\n",h5io_entry,search_names,search_attributes,status);

  h5sx_free_listofdsets(series);

  h5io_remove_node_list( h5io_entry );

  if (pstatus) *pstatus=status;
  return( NULL );

} // h5sx_locate_series

/*---------------------------------------------------------------------------
NAME

  h5sx_locate_entries --- locates all entries and return the entry list

SYNOPSIS

  H5ioNode  *h5sx_locate_entries ( H5ioNode *h5io_file, H5sxListOfDSets *search_names, H5sxListOfDSets *search_attributes, int *pstatus );

DESCRIPTION

  Searches for entries in file starting in path and returns a pointer
  to the entry list.

  H5ioNode *h5io_block
  H5sxListOfDSets *search_names
  H5sxListOfDSets *search_attributes
  int *pstatus

RETURN VALUE
success: NodeList and *pstatus>=0
error:   NULL and *pstatus<0
---------------------------------------------------------------------------*/
H5ioNode  *h5sx_locate_entries ( H5ioNode *h5io_file, H5sxListOfDSets *search_names, H5sxListOfDSets *search_attributes, int *pstatus )
{ long level=2, indent=2;
  herr_t h5status=-1;
  int status=-1;
  const char *path="/";

  H5sxListOfDSets *entries=NULL;

  if (iodbg(IODBG_H5|IODBG_DEBUG1)) printf("h5sx_locate_entries( %p, %p, %p, %p ) BEGIN\n",h5io_file,search_names,search_attributes,pstatus);

  if (!h5io_file) goto h5sx_locate_entries_error;

  if (iodbg_level()>level) {
    if (iodbg(IODBG_SHOWNODESEARCH)) {
      printf("%*sLocating entries ...\n",(int)indent,"");
      if (iodbg(IODBG_VERBOSE)) {
        h5sx_print_listofdsets( stdout, search_names, "search_names", indent+2 );
        h5sx_print_listofdsets( stdout, search_attributes, "search_attributes", indent+2 );
      }
      entries=h5sx_new_listofdsets(NULL);
    }
  }

  // search existing entries
  if (iodbg(IODBG_SHOWNODESEARCH||IODBG_VERBOSE)) {
    printf("%*sPath=%s\n",(int)indent+2,"",path);
  }
  h5status=h5sx_search_group_by_name( h5io_file->H5id, path, H5sxEntrySearch, 
   search_names, search_attributes, entries, h5io_file );
  if (h5status<0) goto h5sx_locate_entries_error;

  if (iodbg_level()>level) {
    if (iodbg(IODBG_SHOWNODESEARCH|IODBG_VERBOSE)) h5sx_print_listofdsets( stdout, entries, " Found entries ...",indent);
    else if (iodbg(IODBG_SHOWNODESEARCH)) printf(" %*s%ld entr%s found.\n",(int)indent,"",h5sx_length_listofdsets(entries), h5sx_length_listofdsets(entries)==1?"y":"ies");
  }

  entries=h5sx_free_listofdsets(entries);

  status=0;

  if (iodbg(IODBG_H5|IODBG_DEBUG1)) printf("h5sx_locate_entries( %p, %p, %p, %d ) (%p) END\n",h5io_file,search_names,search_attributes,status,h5io_file->NodeList);

  if (pstatus) *pstatus=status;
  return( h5io_file->NodeList );

h5sx_locate_entries_error:

  printf("h5sx_locate_entries( %p, %p, %p, %d ) (NULL) ERROR\n",h5io_file,search_names,search_attributes,status);

  h5sx_free_listofdsets(entries);

  h5io_remove_node_list( h5io_file );

  if (pstatus) *pstatus=status;
  return( NULL );

} // h5sx_locate_entries

long h5sx_guess_number(const char *string, long defval)
{ int errval=0;
  long number=0;
  const char *pg=string;

  // skip non-digit characters
  for (pg=string;*pg;pg++) {
    if ( isdigit(*pg)||(*pg=='-')||(*pg=='+') ) break;
  }

  if (strlen(pg)>0) {
    number = (long) num_str2double( pg, NULL, &errval);
    if (errval) number=defval;
  } else number=defval;

  return(number);
} // h5sx_guess_number

/*---------------------------------------------------------------------------
NAME

  h5sx_highest_nodenumber --- returns the hightest node number in NodeList

SYNOPSIS

  long h5sx_highest_nodenumber( H5ioNode *node, int *pstatus );

DESCRIPTION

  Returns the highest number in node->NodeList for
  H5ioNode *node: File-, Entry- or SeriesNode
  The names of these nodes are numbers.

  If nodelist is empty the return value is 1.

  In case of success *pstatus is 0, otherwise negative.

RETURN VALUE

  highest nodenumber in node->NodeList

---------------------------------------------------------------------------*/
long h5sx_highest_nodenumber( H5ioNode *node, int *pstatus ) 
{ int status=0;
  long highest_nodenumber=1;
  H5ioNode *nodelist=NULL;
  if (node) {
    switch (node->NodeClass) {
      case H5ioFileNode:
      case H5ioEntryNode:
      case H5ioSeriesNode:
        nodelist = node->NodeList;
        while ( nodelist ) {
          long nodenumber;
          switch (nodelist->NodeClass) {
            case H5ioEntryNode:
            case H5ioSeriesNode:
            case H5ioMemoryNode:
              nodenumber=h5sx_guess_number(nodelist->Name, highest_nodenumber);
              if (nodenumber>highest_nodenumber) highest_nodenumber=nodenumber;
              break;
          }
          nodelist = nodelist->NodeList;
        }
        break;
      default: status=-1;
    }
  }
  if (pstatus) *pstatus=status;
  return(highest_nodenumber);
} // h5sx_highest_nodenumber

/*---------------------------------------------------------------------------
NAME

  h5sx_highest_entrynumber --- returns the hightest entry number in stream

SYNOPSIS

  long h5sx_highest_entrynumber( int stream, int *pstatus );

DESCRIPTION

  Returns the highest entry number of stream.

  In case of success *pstatus is 0, otherwise negative.

---------------------------------------------------------------------------*/
long h5sx_highest_entrynumber( int stream, int *pstatus ) 
{ H5ioNode *h5io_file=NULL;
  int status=-1;
  long entry_number=1;

  h5io_file = h5io_stream_info( stream );
  if (!h5io_file) goto h5sx_highest_entrynumber_error;

  entry_number=h5sx_highest_nodenumber( h5io_file, &status );
  if (status) goto h5sx_highest_entrynumber_error;

  if (pstatus) *pstatus=status;
  return(entry_number);

h5sx_highest_entrynumber_error:

  if (pstatus) *pstatus=status;
  return(1);

} // h5sx_highest_entrynumber

/*---------------------------------------------------------------------------
NAME

  h5sx_sync_image --- inserts/searches h5io_block

SYNOPSIS

  H5ioNode *h5sx_sync_image( int stream, int blocknumbering,
                       int DataType, int ByteOrder, int DataCompression, float Dummy,
                       long *MaxDim, long *DataDim,
                       long entry_number, long memory_number, long block_number );

DESCRIPTION

  Searches for an h5io_block that is specified by entry_number, memory_number
  and block_number. If it does not exist, it is inserted.

  If DataCompression >1 the written data will be compressed if possible.

  Depending on blocknumbering the destination block will be located in the 
  current series (H5ioArrayIndex) or in a new series (H5ioSeriesNumber).

  For avoiding resource leaks stream must be released after use with
  h5sx_close_stream.

  int blocknumbering:
  H5ioArrayIndex:     write to the current series (highest number)
  H5ioSeriesNumber:   create a new series for each image.

RETURN VALUE
  NULL error, otherwise H5ioNode *h5io_block with h5io_array=h5io_block->DataList 
---------------------------------------------------------------------------*/
H5ioNode *h5sx_sync_image( int stream, int blocknumbering,
                     int DataType, int ByteOrder, int DataCompression, float Dummy, 
                     long *MaxDim, long *DataDim,
                     long entry_number, long memory_number, long block_number )
{
  // H5ioEntryNode, H5ioSeriesNode, H5ioMemoryNode, H5ioBlockNode, H5ioHeaderNode
  H5ioNode *h5io_file=NULL;
  H5ioNode *h5io_entry=NULL, *h5io_series=NULL, *h5io_memory=NULL, *h5io_block=NULL;
  H5ioNode *h5io_header=NULL;
  H5ioData *h5io_array=NULL;

  int status=0;
  long series_number=0;

  if (iodbg(IODBG_H5|IODBG_DEBUG1|IODBG_DEBUG2)) printf("h5sx_sync_image( %d, ... ) BEGIN\n",stream);

  h5io_file = h5io_stream_info( stream );
  if (!h5io_file) goto h5sx_sync_image_error;

  h5io_entry=h5sx_new_entry ( h5io_file, entry_number, blocknumbering, &status );
  if (!h5io_entry) goto h5sx_sync_image_error;

  if (!h5io_entry->NodeList) {
    series_number=1;
  } else {
    series_number = h5sx_highest_nodenumber( h5io_entry, &status );
    if (status<0) goto h5sx_sync_image_error;
  }

  // check whether block_number is unique, otherwise sequential numbering or error
  //++++

  switch ( blocknumbering ) {
    case H5ioArrayIndex:
      // use series with highest number
      break;
    default:
    case H5ioSeriesNumber:
      series_number = block_number;
      break;
  }

  h5io_series=h5sx_new_series ( h5io_entry, series_number, &status );
  if (!h5io_series) goto h5sx_sync_image_error;

  h5io_memory=h5sx_new_memory ( h5io_series, labs(memory_number), &status );
  if (!h5io_memory) goto h5sx_sync_image_error;

  if (memory_number>0) {
    h5io_block=h5sx_new_datablock ( h5io_memory, &status );
  } else if (memory_number<0) {
    h5io_block=h5sx_new_errorblock ( h5io_memory, &status );
  }  else {
    /* generalblock */
  }
  if (!h5io_block) goto h5sx_sync_image_error;

  h5io_header=h5sx_new_header ( h5io_block, &status );
  if (!h5io_header) goto h5sx_sync_image_error;

  h5io_array=h5sx_new_array ( h5io_block, DataType, ByteOrder, DataCompression, Dummy, 
                              MaxDim, DataDim, &status );
  if (!h5io_array) goto h5sx_sync_image_error;

  if (iodbg(IODBG_H5|IODBG_DEBUG1|IODBG_DEBUG2)) printf("h5sx_sync_image( %d, ... ) END (%p) \n",stream,h5io_block);

  return(h5io_block);

h5sx_sync_image_error:

  if (iodbg(IODBG_H5|IODBG_DEBUG1|IODBG_DEBUG2)) printf("h5sx_sync_image( %d, ... ) ERROR (NULL) \n",stream);

  return(NULL);

} // h5sx_sync_image

herr_t h5sx_flush_image( H5ioData *h5io_array, long *DataDim, void *Data )
{
  hid_t array_id=-1;
  hid_t datatype_id=-1;
  hsize_t rank=-1;
  hsize_t *maxdims=NULL;
  hsize_t *dims=NULL;
  hsize_t nframes=-1;

  hid_t h5status=-1;

  if (iodbg(IODBG_H5|IODBG_DEBUG1|IODBG_DEBUG2)) printf("h5sx_flush_image( %p, ... ) BEGIN\n",h5io_array);

  if (!h5io_array) goto h5sx_flush_image_error;
  if (!DataDim) goto h5sx_flush_image_error;

  /*  int DataType, 
      hsize_t ItemSize, 
      hsize_t Rank, 
      hsize_t *Dims, 
      hsize_t *MaxDims
   */

  array_id = h5io_array->H5data_id;
  datatype_id = h5sxdatatype2dtype( h5io_array->DataType, h5io_array->ByteOrder );

  rank = h5io_array->Rank;
  maxdims = h5io_array->MaxDims;
  dims = h5io_array->Dims;
  nframes = h5io_array->NValues;

    if ( rank != (hsize_t) (DataDim[0]+1) ) {
      printf("ERROR: h5sx_flush_image rank=%ld, DataDim[0]=%ld: rank != DataDim[0]+1",(long) rank, DataDim[0]);
      goto h5sx_flush_image_error;
    }

    if (!dims) {
      if (!(dims=(hsize_t*) MALLOC( sizeof(hsize_t)*rank )))
        goto h5sx_flush_image_error;
    }

    {
      hsize_t irank;
      dims[0] = 1; // only 1 frame to add
      for (irank=1;irank<rank;irank++) {
        dims[rank-irank] = DataDim[irank];
      }
    }
    h5io_array->Dims = dims;

  // herr_t h5sx_dataset_write(hid_t data_id, long frame_no, hid_t datatype_id,
  //                           hsize_t rank, hsize_t maxdims[], hsize_t dims[], void *data)
  h5status = h5sx_dataset_write(array_id, nframes+1, datatype_id,
                                rank, maxdims, dims, Data );
  if (h5status<0) goto h5sx_flush_image_error;

  if (iodbg(IODBG_H5|IODBG_DEBUG1|IODBG_DEBUG2)) printf("h5sx_flush_image( %p, ... ) END\n",h5io_array);

  return(0);

h5sx_flush_image_error:

  if (iodbg(IODBG_H5|IODBG_DEBUG1|IODBG_DEBUG2)) printf("h5sx_flush_image( %p, ... ) (-1) ERROR\n",h5io_array);

  return(-1);

} // h5sx_flush_image

/*---------------------------------------------------------------------------
NAME

  h5sx_close_image --- closes data for writing

SYNOPSIS

  herr_t h5sx_close_image ( H5ioData *h5io_array );

DESCRIPTION

  Closes the h5 data set of h5io_array.
  If the H5data_id value is >=0 the data set is open for writing.
  It is closed using h5sx_dataset_close. The dataset value H5data_id 
  is reset to -1.
  If the dataset does not exist the function returns 0.

  H5ioData *h5io_array : dataset

RETURN VALUE
success: 0
error:   <0
---------------------------------------------------------------------------*/
herr_t h5sx_close_image ( H5ioData *h5io_array )
{
  herr_t h5status=0;

  if (h5io_array) {
    h5status=h5sx_dataset_close(h5io_array->H5data_id);
    h5io_array->H5data_id = -1;

    /*more buffers to FREE */
    FREE(h5io_array->Dims); 
    FREE(h5io_array->MaxDims);
    FREE(h5io_array->FillValue);

  }

  return(h5status);

} // h5sx_close_image

/*---------------------------------------------------------------------------
NAME

  h5sx_close_stream --- closes stream

SYNOPSIS

  int h5sx_close_stream ( int stream )

DESCRIPTION

  Close stream, close HDF5 file.

RETURN VALUE
success: int status >= 0
error:              <0
---------------------------------------------------------------------------*/
int h5sx_close_stream ( int stream )
{ int status=-1;
  H5ioNode *info;

  if (iodbg(IODBG_H5|IODBG_DEBUG1)) printf("h5sx_close_stream( %d ) BEGIN\n",stream);

  info = h5io_stream_info( stream );

  if (info) {
    hid_t fid;
    fid = info->H5id;
    h5sx_close_file(fid);
    info->H5id=-1;
  }

  status = h5io_free_stream ( stream );
  if (status<0) goto h5sx_close_stream_error;

  if (iodbg(IODBG_H5|IODBG_DEBUG1)) printf("h5sx_close_stream( %d ) END\n",stream);

  return( status );

h5sx_close_stream_error:

  if (iodbg(IODBG_H5|IODBG_DEBUG1)) printf("h5sx_close_stream( %d ) ERROR (%d)\n",stream,status);

  return( status );

} /* h5sx_close_stream */

/*---------------------------------------------------------------------------
NAME

  h5sx_open_stream --- opens a stream for name and returns its id

SYNOPSIS

  int h5sx_open_stream ( const char *name, int mode, const char *filetype );

DESCRIPTION

  Opens an h5 file by name and returns a stream. Depending on mode
  the data structure is initialized with the content of the file.
  For avoiding resource leaks the stream must be released after use with
  h5sx_close_stream.

  int mode :
  H5ioOpenNew : create new file (delete existing file)
  H5ioOpenOld : open old file (read write access)
  H5ioOpenRead: open old file (read-only access)
  H5ioOpenAny:  if file exists: H5ioOpenOld, otherwise H5ioOpenNew


RETURN VALUE
success: int stream >= 0
error:              <0
---------------------------------------------------------------------------*/
int h5sx_open_stream ( const char *name, int mode, const char *filetype )
{
  /*
   * iterate through file and get
   * entry list
   * series list
   * memory list
   * frame list ("block numbers")
   */

  int status=-1;

  int stream=-1;
  hid_t fid=-1;
  unsigned flags=0;

  H5ioNode    *h5io_file=NULL;
  H5ioNode    *h5io_entry=NULL;
  H5ioNode    *h5io_series=NULL;
  H5ioNode    *h5io_memory=NULL;
  H5ioNode    *h5io_block=NULL;
  H5ioNode    *h5io_header=NULL;
  H5ioData    *h5io_symbol=NULL;
  H5ioData    *h5io_array=NULL;

  H5sxListOfDSets *file_attributes=NULL;

  H5sxListOfDSets *entry_search_names=h5sx_new_listofdsets(NULL);
  H5sxListOfDSets *entry_search_attributes=h5sx_new_listofdsets(NULL);

  H5sxListOfDSets *series_search_names=h5sx_new_listofdsets(NULL);
  H5sxListOfDSets *series_search_attributes=h5sx_new_listofdsets(NULL);

  H5sxListOfDSets *memory_search_names=h5sx_new_listofdsets(NULL);
  H5sxListOfDSets *memory_search_attributes=h5sx_new_listofdsets(NULL);

  H5sxListOfDSets *datablock_search_names=h5sx_new_listofdsets(NULL);
  H5sxListOfDSets *datablock_search_attributes=h5sx_new_listofdsets(NULL);

  H5sxListOfDSets *errorblock_search_names=h5sx_new_listofdsets(NULL);
  H5sxListOfDSets *errorblock_search_attributes=h5sx_new_listofdsets(NULL);

  H5sxListOfDSets *header_search_names=h5sx_new_listofdsets(NULL);
  H5sxListOfDSets *header_search_attributes=h5sx_new_listofdsets(NULL);

  H5sxListOfDSets *symbol_search_names=h5sx_new_listofdsets(NULL);
  H5sxListOfDSets *symbol_search_attributes=h5sx_new_listofdsets(NULL);

  H5sxListOfDSets *array_search_names=h5sx_new_listofdsets(NULL);
  H5sxListOfDSets *array_search_attributes=h5sx_new_listofdsets(NULL);

  H5sxListOfDSets *error_array_search_names=h5sx_new_listofdsets(NULL);
  H5sxListOfDSets *error_array_search_attributes=h5sx_new_listofdsets(NULL);

  H5sxListOfDSets *found_arrays=h5sx_new_listofdsets(NULL);

  if (iodbg(IODBG_H5|IODBG_DEBUG1)) printf("h5sx_open_stream( %s, %s, %s ) BEGIN\n",name,h5io_openmode2str ( mode ),filetype);

  stream = h5io_new_stream( name );

  if (stream>=0) {

    file_attributes=h5sx_new_listofdsets(NULL);

#if LIBLZ4 == 1
    /* Register the bitshuffle filter */
    if (iodbg(IODBG_H5|IODBG_DEBUG2)) printf (">>  Registering bitshuffle filter.\n");
    if (bshuf_register_h5filter() < 0 ) printf ("WARNING: Failed to register bitshuffle filter.\n");
#endif // LIBLZ4

    /*
     * H5ioOpenAny becomes either H5ioOpenOld or H5ioOpenNew
     * with write access, depending whether file exists or not.
     */
    if (mode == H5ioOpenAny) {
      // if file exists: H5ioOpenOld, otherwise H5ioOpenNew
      FILE *tryit;
      if ((tryit=fopen(name,"rb"))) {
        fclose(tryit);
        mode = H5ioOpenOld;
      } else {
        mode = H5ioOpenNew;
      }
    }

    switch (mode) {
      case H5ioOpenNew : // create new file (delete existing file)
        flags = H5F_ACC_RDWR; // everything else would be ignored
        break;
      case H5ioOpenOld : // open old file (read write access)
        flags = H5F_ACC_RDWR;
        break;
      default: 
      case H5ioOpenRead: // open old file (read-only access)
        flags = H5F_ACC_RDONLY;
        break;
    }

    switch (mode) {
      case H5ioOpenOld:
      case H5ioOpenRead:

       /*
        * Open an existing hdf5 file using the default properties
        * and reading file attributes.
        */

        if (iodbg_level()>1) {
          if (iodbg(IODBG_SHOWNODESEARCH|IODBG_VERBOSE)) printf("  Opening file %s ... \n",name);
          else if (iodbg(IODBG_SHOWNODESEARCH)) printf("  Opening file ... \n");
        }

        if (iodbg(IODBG_H5|IODBG_VERBOSE)) printf("h5sx_open_file( %s , ... , %u)\n",name,flags);
        fid = h5sx_open_file( name, file_attributes, flags );
        if (fid<0) goto h5sx_open_stream_error;

       /*
        * Save H5id in h5io_file info.
        */
        h5io_file = h5io_stream_info( stream );
        if (!h5io_file) goto h5sx_open_stream_error;
        h5io_file->H5id= fid;
        h5io_file->H5Path=strlib_newstr("/"); //+++++++++++++  2018-01-10: Attention, everything could match, checks needed

       /*
        * Get the file search lists
        */
        if ( h5sx_search_dset( file_attributes, SXCLASS, SXCLASS_FILE ) ) {
          h5sxdefaultsearchlists ( SXCLASS_FILE,
             entry_search_names, entry_search_attributes,
             series_search_names, series_search_attributes,
             memory_search_names, memory_search_attributes,
             datablock_search_names, datablock_search_attributes,
             errorblock_search_names, errorblock_search_attributes,
             header_search_names, header_search_attributes,
             symbol_search_names, symbol_search_attributes,
             array_search_names, array_search_attributes,
             error_array_search_names, error_array_search_attributes );
        } else {
          h5sxdefaultsearchlists ( filetype,
             entry_search_names, entry_search_attributes,
             series_search_names, series_search_attributes,
             memory_search_names, memory_search_attributes,
             datablock_search_names, datablock_search_attributes,
             errorblock_search_names, errorblock_search_attributes,
             header_search_names, header_search_attributes,
             symbol_search_names, symbol_search_attributes,
             array_search_names, array_search_attributes,
             error_array_search_names, error_array_search_attributes );
        }

       /*
        * Locate all entries in h5io_file
        */
        h5io_entry = h5sx_locate_entries ( h5io_file, entry_search_names, entry_search_attributes, &status );
        if (status<0) goto h5sx_open_stream_error;

        while (h5io_entry) {

         /*
          * Locate all series in h5io_entry
          */
          h5io_series = h5sx_locate_series( h5io_entry, series_search_names, series_search_attributes, &status );
          if (status<0) goto h5sx_open_stream_error;

          while (h5io_series) {

           /*
            * Locate all memories in h5io_series
            */
            h5io_memory = h5sx_locate_memories( h5io_series, memory_search_names, memory_search_attributes, &status );
            if (status<0) goto h5sx_open_stream_error;

            while (h5io_memory) {

             /*
              * Locate datablocks in h5io_memory
              */
              h5io_block = h5sx_locate_datablocks ( h5io_memory, datablock_search_names, datablock_search_attributes, &status );
              if (status<0) goto h5sx_open_stream_error;

             /*
              * Locate errorblocks in h5io_memory
              */
              h5io_block = h5sx_locate_errorblocks ( h5io_memory, errorblock_search_names, errorblock_search_attributes, &status );
              if (status<0) goto h5sx_open_stream_error;

              while (h5io_block) {

                found_arrays=h5sx_new_listofdsets(NULL);

                /*
                 * Get data array in h5io_block
                 */
//+++++++++++++ TOCHECK
                if ( strlib_casecmp(h5io_node_name( h5io_block ),ERRORBLOCKNAME) == 0) {
                  h5io_array = h5sx_locate_arrays ( h5io_block, error_array_search_names, error_array_search_attributes, found_arrays, &status );
                  if (status<0) goto h5sx_open_stream_error;
                } else {
                  h5io_array = h5sx_locate_arrays ( h5io_block, array_search_names, error_array_search_attributes, found_arrays, &status );
                  if (status<0) goto h5sx_open_stream_error;
                }

                while (h5io_array) {

                  // H5ioData *h5sx_read_array_value_native ( h5io_array, mode );
                  h5sx_read_array_value_native( h5io_array, 0 );

                  h5io_array = h5io_array->Next;

                }

               /*
                * Get header in h5io_block
                */
                h5io_header = h5sx_locate_headers ( h5io_block, header_search_names, header_search_attributes, &status );
                if (status<0) goto h5sx_open_stream_error;

                while (h5io_header) {

                 /*
                  * Get symbols in h5io_headers
                  */
                  h5io_symbol = h5sx_locate_symbols ( h5io_header, symbol_search_names, symbol_search_attributes, found_arrays, &status );
                  if (status<0) goto h5sx_open_stream_error;

                  while (h5io_symbol) {

                    h5sx_read_symbol_value ( h5io_symbol );
                  
                    h5io_symbol = h5io_symbol->Next;
                  }

                  h5io_header = h5io_header->Next;
                }

                found_arrays=h5sx_free_listofdsets(found_arrays);

                h5io_block = h5io_block->Next;
              }

              h5io_memory = h5io_memory->Next;
            }

            h5io_series = h5io_series->Next;
          }

          h5io_entry = h5io_entry->Next;
        }

        break;

      case H5ioOpenNew : // create new SXCLASS_FILE, truncate any existing file
        // H5ioNode *h5sx_new_file ( const char *name, int *pstatus );

        h5sx_append_dset( file_attributes, SXCLASS, SXCLASS_FILE, 0, 1, -1 );

        if (iodbg(IODBG_H5|IODBG_VERBOSE)) printf("h5sx_create_file( %s , ... )\n",name);
        fid = h5sx_create_file( name, file_attributes );
        if (fid<0) goto h5sx_open_stream_error;

       /*
        * Save H5id in h5io_file info.
        */
        h5io_file = h5io_stream_info( stream );
        if (!h5io_file) goto h5sx_open_stream_error;
        h5io_file->H5id= fid;

        // BlockNumbering = H5ioSeriesNumber | H5ioArrayIndex

        break;

      default:
        goto h5sx_open_stream_error;

    }

   /*
    * free search names and attributes
    */
    found_arrays = h5sx_free_listofdsets(found_arrays);

    error_array_search_names = h5sx_free_listofdsets(error_array_search_names);
    error_array_search_attributes = h5sx_free_listofdsets(error_array_search_attributes);

    array_search_names = h5sx_free_listofdsets(array_search_names);
    array_search_attributes = h5sx_free_listofdsets(array_search_attributes);

    symbol_search_names = h5sx_free_listofdsets(symbol_search_names);
    symbol_search_attributes = h5sx_free_listofdsets(symbol_search_attributes);

    header_search_names = h5sx_free_listofdsets(header_search_names);
    header_search_attributes = h5sx_free_listofdsets(header_search_attributes);

    datablock_search_names = h5sx_free_listofdsets(datablock_search_names);
    datablock_search_attributes = h5sx_free_listofdsets(datablock_search_attributes);

    errorblock_search_names = h5sx_free_listofdsets(errorblock_search_names);
    errorblock_search_attributes = h5sx_free_listofdsets(errorblock_search_attributes);

    memory_search_names = h5sx_free_listofdsets(memory_search_names);
    memory_search_attributes = h5sx_free_listofdsets(memory_search_attributes);

    series_search_names = h5sx_free_listofdsets(series_search_names);
    series_search_attributes = h5sx_free_listofdsets(series_search_attributes);

    entry_search_names = h5sx_free_listofdsets(entry_search_names);
    entry_search_attributes = h5sx_free_listofdsets(entry_search_attributes);

    file_attributes = h5sx_free_listofdsets(file_attributes);

  }

  if (iodbg(IODBG_SHOWHIERARCHY)) h5io_fprint_streams ( stdout, iodbg_level () );

  if (iodbg(IODBG_H5|IODBG_DEBUG1)) printf("h5sx_open_stream( %s, %s, %s ) END\n",name,h5io_openmode2str ( mode ),filetype);

  return(stream);

h5sx_open_stream_error:

  printf("h5sx_open_stream( %s, %s, %s ) (-1) ERROR\n",name,h5io_openmode2str ( mode ),filetype);

  found_arrays = h5sx_free_listofdsets(found_arrays);

  array_search_attributes = h5sx_free_listofdsets(array_search_attributes);
  array_search_names = h5sx_free_listofdsets(array_search_names);

  symbol_search_attributes = h5sx_free_listofdsets(symbol_search_attributes);
  symbol_search_names = h5sx_free_listofdsets(symbol_search_names);

  header_search_names = h5sx_free_listofdsets(header_search_names);
  header_search_attributes = h5sx_free_listofdsets(header_search_attributes);

  datablock_search_names = h5sx_free_listofdsets(datablock_search_names);
  datablock_search_attributes = h5sx_free_listofdsets(datablock_search_attributes);

  errorblock_search_names = h5sx_free_listofdsets(errorblock_search_names);
  errorblock_search_attributes = h5sx_free_listofdsets(errorblock_search_attributes);

  memory_search_names = h5sx_free_listofdsets(memory_search_names);
  memory_search_attributes = h5sx_free_listofdsets(memory_search_attributes);

  series_search_names = h5sx_free_listofdsets(series_search_names);
  series_search_attributes = h5sx_free_listofdsets(series_search_attributes);

  entry_search_names = h5sx_free_listofdsets(entry_search_names);
  entry_search_attributes = h5sx_free_listofdsets(entry_search_attributes);

  h5sx_free_listofdsets(file_attributes);
  h5sx_close_stream( stream );
  return(-1);

} /* h5sx_open_stream */
