/*
 *   Project: The SPD Image correction and azimuthal regrouping
 *                      http://forge.epn-campus.eu/projects/show/azimuthal
 *
 *   Copyright (C) 2005-2010 European Synchrotron Radiation Facility
 *                           Grenoble, France
 *
 *   Principal authors: P. Boesecke (boesecke@esrf.fr)
 *                      R. Wilcke (wilcke@esrf.fr)
 *
 *   This program is free software: you can redistribute it and/or modify
 *   it under the terms of the GNU Lesser General Public License as published
 *   by the Free Software Foundation, either version 3 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU Lesser General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   and the GNU Lesser General Public License  along with this program.
 *   If not, see <http://www.gnu.org/licenses/>.
 */

/*+++***********************************************************************
NAME

   h5io.h

DESCRIPTION

   Include file for h5io.c

HISTORY
    2017-01-10  PB V0.00 from bslio

***************************************************************************/

#ifndef _H5IO_
#  define _H5IO_

/***************************************************************************
* Includes                                                                 *
***************************************************************************/

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <errno.h>
#include <fcntl.h>
#include <limits.h>

#include <hdf5.h>

#include "iodbg.h"
#include "numio.h"
#include "strlib.h"
#include "edfio.h"

#ifndef WIN32
# include <unistd.h>
#endif

#include <ctype.h>

#ifdef sun      /* sun specific stuff */
#  include <sys/time.h>
#  include <floatingpoint.h>
#else
#  include <time.h>
#endif

/***************************************************************************
* General Definitions                                                      *
***************************************************************************/

#ifndef PRIVATE
# define PRIVATE      static /* used to declare variables of private type */
# define PUBLIC               /* used to declare variables of public type */
#endif

/***************************************************************************
* Enum                                                                     *
***************************************************************************/
PUBLIC enum H5ioOpenMode { H5ioInValidOpenMode,
                           H5ioOpenNew,
                           H5ioOpenOld,
                           H5ioOpenRead,
                           H5ioOpenAny,
                           H5ioEndOpenMode };

PUBLIC enum H5ioNodeClass { H5ioInValidNodeClass,
                            H5ioFileNode,
                            H5ioEntryNode,
                            H5ioSeriesNode,
                            H5ioMemoryNode,
                            H5ioBlockNode,
                            H5ioHeaderNode,
                            H5ioEndNodeClass };

PUBLIC enum H5ioDataClass { H5ioInValidDataClass,
                            H5ioSymbolNode,
                            H5ioArrayNode,
                            H5ioGroupNumberNode,
                            H5ioEndDataClass };

PUBLIC enum H5ioBlockNumbering { H5ioInValidBlockNumbering,
                                 H5ioArrayIndex,
                                 H5ioSeriesNumber,
                                 H5ioEndBlockNumbering };

PUBLIC enum H5ioInterpretation { H5ioInValidInterpretation,
                                 H5ioScalar,
                                 H5ioSpectrum,
                                 H5ioImage,
                                 H5ioEndInterpretation };

/***************************************************************************
* H5io_Data Flags                                                          *
***************************************************************************/

# define H5IO_DATA_INSERTED          0x1
# define H5IO_DATA_VALUE_READ        0x2
# define H5IO_DATA_DESCRIPTION_READ  0x4

/***************************************************************************
* Structure Definitions                                                    *
***************************************************************************/

  typedef struct H5io_Data {
    char         *Name;                                        /* the key */
    char         *H5Path;
    hid_t         H5id;                                /* h5 path root id */
    int           DataClass;

    void         *Binary;            /* allocated binary buffer for array */
    hsize_t        BinarySize;                   /* size of binary buffer */
    int           DataType;        /* data type of a single array element */
    hsize_t        ItemSize;     /* size of a single binary array element */
    int           ByteOrder;      /* byte order of a single array element */
    hsize_t        Rank;                             /* rank of the array */
    hsize_t       *Dims;                       /* dimensions of the array */
    hsize_t       *MaxDims;            /* maximum dimensions of the array */
    void        **Value;          /* Linear array with NValues elements.
                                     Each element points to a value of 
                                     type ItemType either in Binaries, if
                                     Binaries is not NULL or otherwise
                                     to separately allocated data. If 
                                     Binaries is NULL, the memory to which
                                     each element is pointing must be 
                                     released individually. In all cases,
                                     the allocated array Value must be 
                                     released after use. If ItemType is
                                     H5T_STRING, Binaries must be a 
                                     character array or NULL and each
                                     element of Value is a string.        */
    hsize_t        NValues;                  /* length of the Value array */
    hsize_t        ValueRank;    /* rank+1 of the values, copy of Rank if 
                                    interpretation attribute is not used  */
    hsize_t       *ValueDims;    /* dimensions of the values, 
                                    ValueDims[0] is set to NValues, 
                                    copy of Dims if interpretation
                                    attribute is not used */
    void         *FillValue;     /* pointer to an allocated FillValue with 
                                    data type of a single array element   */
    hid_t         H5data_id;     /* if >=0, current dataset id, must be 
                                    set to -1 after closing dataset       */
    int           Flags;                               /* H5io_Data Flags */
    struct H5io_Data *Previous, *Next;         /* previous and next array */
    struct H5io_Node *Owner;                          /* the owning block */
  } H5ioData;

  typedef struct H5io_Node {
    char        *Name;
    hid_t        H5id;                              /* root id of h5 path */
    int          NodeClass;

    struct H5io_Node *NodeList;                              /* node list */
    H5ioData    *DataList;                                /* dataset list */

    char        *H5Path;         /* H5 path relative to H5id or file path */
    struct H5io_Node *Previous, *Next;          /* previous and next node */
    struct H5io_Node *Owner;                           /* the owning node */

    long *NodeNumbers;    /* info array with sequential list of node numbers in NodeList */
  } H5ioNode;

/***************************************************************************
* Debug                                                                    *
***************************************************************************/

PUBLIC extern const char
    *h5io_version ( void ),
    *h5io_usage2str( void );

PUBLIC extern int
     h5io_fprint_debug( FILE *out );

/***************************************************************************
* Functions                                                                *
***************************************************************************/

PUBLIC extern hsize_t
      h5io_pitch0( hsize_t rank, hsize_t *dim ),
      h5io_numberofelements( hsize_t rank, hsize_t *dim );

PUBLIC extern void
      h5io_fprint_dims( FILE *out, hsize_t rank, hsize_t *dims );

PUBLIC extern void
    **h5io_free_values( void **value, hsize_t nframes );

PUBLIC extern const char
     *h5io_node_name ( H5ioNode *node ),
     *h5io_blocknumbering2str ( int blocknumbering ),
     *h5io_number2string ( int datatype, void *pdata ),
     *h5io_interpretation2str ( int interpretation );

PUBLIC extern const char
     *h5io_nodeclass2str ( int nodeclass ),
     *h5io_dataclass2str ( int dataclass ),
     *h5io_imagesorting2str ( int imagesorting );

PUBLIC extern long
      h5io_node_number ( H5ioNode *node );

PUBLIC extern int
      h5io_str2blocknumbering( const char *string ),
      h5io_str2interpretation( const char *string ),
      h5io_str2nodeclass( const char *string ),
      h5io_str2dataclass( const char *string ),
      h5io_str2imagesorting( const char *string );

PUBLIC extern const char
     *h5io_openmode2str ( int openmode );

PUBLIC extern int
      h5io_str2openmode( const char *string );

PUBLIC extern H5ioData
     *h5io_data_list   ( H5ioNode *node ),
     *h5io_search_data ( H5ioNode *block, const char *name ),
     *h5io_insert_data ( H5ioNode *block, const char *name, int dataclass );

PUBLIC extern void
      h5io_empty_data       ( H5ioData *data ),
      h5io_remove_data      ( H5ioData *data ),
      h5io_remove_data_list ( H5ioNode *block ),
      h5io_fprint_data_list ( FILE *out, H5ioNode *block, long level );

PUBLIC extern H5ioNode
    *h5io_node_list   ( H5ioNode *node ),
    *h5io_search_node ( H5ioNode *owner, const char *name ),
    *h5io_insert_node ( H5ioNode *owner, const char *name, int nodeclass );

PUBLIC extern void
     h5io_remove_node      ( H5ioNode *entry ),
     h5io_remove_node_list ( H5ioNode *file ),
     h5io_fprint_node_list ( FILE *out, H5ioNode *file, long level );

PUBLIC extern int
     h5io_new_stream    ( const char *name ),
     h5io_search_stream ( const char *name ),
     h5io_free_stream   ( int stream );

PUBLIC extern void
     h5io_fprint_streams ( FILE *out, long level );

PUBLIC extern H5ioNode 
    *h5io_stream_info( int stream );

#endif
#ifndef _H5IO_
#  define _H5IO_
#endif
/***************************************************************************/
