/*
 *   Project: The SPD Image correction and azimuthal regrouping
 *                      http://forge.epn-campus.eu/projects/show/azimuthal
 *
 *   Copyright (C) 2005-2010 European Synchrotron Radiation Facility
 *                           Grenoble, France
 *
 *   Principal authors: P. Boesecke (boesecke@esrf.fr)
 *                      R. Wilcke (wilcke@esrf.fr)
 *
 *   This program is free software: you can redistribute it and/or modify
 *   it under the terms of the GNU Lesser General Public License as published
 *   by the Free Software Foundation, either version 3 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU Lesser General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   and the GNU Lesser General Public License  along with this program.
 *   If not, see <http://www.gnu.org/licenses/>.
 */

# define H5IO_VERSION      "h5io : V0.93 Peter Boesecke 2018-01-10"
/*+++***********************************************************************
NAME

   h5io.c

SYNOPSIS

DESCRIPTION


HISTORY
    2017-03-12  PB V0.6 
    2017-03-21  PB V0.7 
    2017-05-22  PB V0.8 ioalloc, size_t is unsigned, otherwise use ssize_t,
                        initializations with -1 corrected to 0
    2017-07-20  PB V0.9 debug info is only written when IODBG_H5 is set in
                        iodbg,
                        h5io_blocknumbering2str, h5io_str2blocknumbering,
                        starting adapting for writing h5-files
    2017-08-31  PB V0.91 removing h5io_debug, h5io_level,
                         H5ioData structure updated, size_t => hsize_t
    2017-09-01  PB V0.92 h5io_node_name, h5io_node_number
    2018-01-10  PB V0.93 ValueRank and ValueDims added,
                         h5io_interpretation2str, h5io_str2interpretation added

***************************************************************************/

/***************************************************************************
* Private part                                                             *
***************************************************************************/

#include "h5io.h"

/***************************************************************************
* Defines                                                                  *
***************************************************************************/

#ifdef sun
# include <sys/param.h>
# define GETCWD(x,y) getwd (x)
#else
# ifdef WIN32
#  include <direct.h>
#  define GETCWD(x,y) _getcwd(x, y)
# else
#  define GETCWD(x,y) getcwd (x, y)
# endif
#endif

# define ISGRAPH(c) isgraph((int)(c))
# define ISSPACE(c) isspace((int)(c))
# define TOLOWER(s) (char)tolower((int)(s))
# define TOUPPER(s) (char)toupper((int)(s))

# define MAX( a, b)  ((a)<(b)?(b):(a))
# define MIN( a, b)  ((a)>(b)?(b):(a))

# define H5IO_BUFLEN (EdfMaxLinLen+1)

/*****************************************************************************/
# define MaxH5ioFiles           20              /* maximum opened h5io files */
/*****************************************************************************/

/***************************************************************************
* Strings for Enum H5ioOpenMode                                            *
***************************************************************************/
PUBLIC const char *H5ioOpenModeStrings[] = { "invalid",
                           "new",
                           "old",
                           "read",
                           "any",
                           (const char *) NULL };

PUBLIC const char *H5ioNodeClassStrings[] = { "invalid",
                           "file",
                           "entry",
                           "series",
                           "memory",
                           "block",
                           "header",
                           (const char *) NULL };

PUBLIC const char *H5ioDataClassStrings[] = { "invalid",
                           "symbol",
                           "array",
                           "groupnumber",
                           (const char *) NULL };

PUBLIC const char *H5ioBlockNumberingStrings[] = { "invalid",
                            "arrayindex",
                            "seriesnumber",
                            (const char *) NULL };

PUBLIC const char *H5ioInterpretationStrings[] = { "invalid",
                            "scalar",
                            "spectrum",
                            "image",
                            (const char *) NULL };

/***************************************************************************
* H5io                                                                     *
***************************************************************************/

const char *SeparationLine[] = {
    "=========================================================",
    "- - - - - - - - - - - - - - - - - - - - - - - - - - - - -",
    " -  -  -  -  -  -  -  -  -  -  -  -  -  -  -  -  -  -  - ",
    "-   -   -   -   -   -   -   -   -   -   -   -   -   -   -",
    " -     -     -     -     -     -     -     -     -     - ",
    "-       -       -       -       -       -       -       -",
    "-           -           -           -           -        "
};

const int nSeparationLines = 7;
const int H5ioColumnStart =  5;
const int H5ioColumnWidth = 15;

static H5ioNode *H5ioTable[MaxH5ioFiles];
static int H5ioInit = 0;

/*****************************************************************************/

/*---------------------------------------------------------------------------
NAME

   h5io_version --- returns the current version of h5io

SYNOPSIS

   const char *h5io_version ( void );

RETURN VALUE

const char *version string
---------------------------------------------------------------------------*/
const char *h5io_version ( void )
{ return ( H5IO_VERSION );
} /* h5io_version */

/*--------------------------------------------------------------------------
NAME

  h5io_usage2str --- return debug mode usage string

SYNOPSIS

  const char *h5io_usage2str( void );

DESCRPTION

  Return debug mode usage string.

--------------------------------------------------------------------------*/
const char *h5io_usage2str( void )
{ return(iodbg_usage2str());
} // h5io_usage2str

int h5io_fprint_debug( FILE *out )
{ return(iodbg_fprint_debug(out));
} // h5io_fprint_debug

/*--------------------------------------------------------------------------
NAME

  h5io_separation --- returns a separation line

SYNOPSIS

  const char *h5io_separation( int d )

DESCRPTION

  Return separation line of depth d.

--------------------------------------------------------------------------*/
const char *h5io_separation( int d )
{
  if ((0<=d)&&(d<nSeparationLines)) {
    return(SeparationLine[d]);
  }
  return("");
} /* h5io_separation */

/*---------------------------------------------------------------------------
NAME
  h5io_nodeclass2str ---  convert H5ioNodeClass to string

SYNOPSIS
  const char *h5io_nodeclass2str ( int nodeclass )

DESCRIPTION
  convert H5ioNodeClass to string
---------------------------------------------------------------------------*/
const char *h5io_nodeclass2str ( int nodeclass )
{ if ((nodeclass<0)||(nodeclass>=H5ioEndNodeClass))
    nodeclass = H5ioInValidNodeClass;
  return( H5ioNodeClassStrings[nodeclass] );
} /* h5io_nodeclass2str */

/*---------------------------------------------------------------------------
NAME
  h5io_str2nodeclass ---  convert string to H5ioNodeClass

SYNOPSIS
  int h5io_str2nodeclass( const char *string );

DESCRIPTION
  convert string to H5ioNodeClass
---------------------------------------------------------------------------*/
int h5io_str2nodeclass( const char *string )
{ int  nodeclass=-1;

  while ( H5ioNodeClassStrings[++nodeclass] ) {
    if (!strlib_casecmp(H5ioNodeClassStrings[nodeclass], string)) break;
  }
  if (nodeclass>=H5ioEndNodeClass)
    nodeclass=H5ioInValidNodeClass;

  return(nodeclass);

} /* h5io_str2nodeclass */

/*---------------------------------------------------------------------------
NAME
  h5io_dataclass2str ---  convert H5ioDataClass to string

SYNOPSIS
  const char *h5io_dataclass2str ( int dataclass )

DESCRIPTION
  convert H5ioDataClass to string
---------------------------------------------------------------------------*/
const char *h5io_dataclass2str ( int dataclass )
{ if ((dataclass<0)||(dataclass>=H5ioEndDataClass))
    dataclass = H5ioInValidDataClass;
  return( H5ioDataClassStrings[dataclass] );
} /* h5io_dataclass2str */

/*---------------------------------------------------------------------------
NAME
  h5io_str2dataclass ---  convert string to H5ioDataClass

SYNOPSIS
  int h5io_str2dataclass( const char *string );

DESCRIPTION
  convert string to H5ioDataClass
---------------------------------------------------------------------------*/
int h5io_str2dataclass( const char *string )
{ int  dataclass=-1;

  while ( H5ioDataClassStrings[++dataclass] ) {
    if (!strlib_casecmp(H5ioDataClassStrings[dataclass], string)) break;
  }
  if (dataclass>=H5ioEndDataClass)
    dataclass=H5ioInValidDataClass;

  return(dataclass);

} /* h5io_str2dataclass */

/*---------------------------------------------------------------------------
NAME
  h5io_blocknumbering2str ---  convert H5ioBlockNumbering to string

SYNOPSIS
  const char *h5io_blocknumbering2str ( int blocknumbering )

DESCRIPTION
  convert H5ioBlockNumbering to string
---------------------------------------------------------------------------*/
const char *h5io_blocknumbering2str ( int blocknumbering )
{ if ((blocknumbering<0)||(blocknumbering>=H5ioEndBlockNumbering))
    blocknumbering = H5ioInValidBlockNumbering;
  return( H5ioBlockNumberingStrings[blocknumbering] );
} /* h5io_blocknumbering2str */

/*---------------------------------------------------------------------------
NAME
  h5io_str2blocknumbering ---  convert string to H5ioBlockNumbering

SYNOPSIS
  int h5io_str2dataclass( const char *string );

DESCRIPTION
  convert string to H5ioBlockNumbering
---------------------------------------------------------------------------*/
int h5io_str2blocknumbering( const char *string )
{ int  blocknumbering=-1;

  while ( H5ioBlockNumberingStrings[++blocknumbering] ) {
    if (!strlib_casecmp(H5ioBlockNumberingStrings[blocknumbering], string)) break;
  }
  if (blocknumbering>=H5ioEndBlockNumbering)
    blocknumbering=H5ioInValidBlockNumbering;

  return(blocknumbering);

} /* h5io_str2blocknumbering */

/*---------------------------------------------------------------------------
NAME
  h5io_interpretation2str ---  convert H5ioInterpretation to string

SYNOPSIS
  const char *h5io_interpretation2str ( int interpretation )

DESCRIPTION
  convert H5ioInterpretation to string
---------------------------------------------------------------------------*/
const char *h5io_interpretation2str ( int interpretation )
{ if ((interpretation<0)||(interpretation>=H5ioEndInterpretation))
    interpretation = H5ioInValidInterpretation;
  return( H5ioInterpretationStrings[interpretation] );
} /* h5io_interpretation2str */

/*---------------------------------------------------------------------------
NAME
  h5io_str2interpretation ---  convert string to H5ioInterpretation

SYNOPSIS
  int h5io_str2dataclass( const char *string );

DESCRIPTION
  convert string to H5ioInterpretation
---------------------------------------------------------------------------*/
int h5io_str2interpretation( const char *string )
{ int  interpretation=-1;

  while ( H5ioInterpretationStrings[++interpretation] ) {
    if (!strlib_casecmp(H5ioInterpretationStrings[interpretation], string)) break;
  }
  if (interpretation>=H5ioEndBlockNumbering)
    interpretation=H5ioInValidBlockNumbering;

  return(interpretation);

} /* h5io_str2interpretation */

/*---------------------------------------------------------------------------
NAME
  h5io_openmode2str ---  convert H5ioOpenMode to string

SYNOPSIS
  const char *h5io_openmode2str ( int openmode )

DESCRIPTION
  convert H5ioOpenMode to string
---------------------------------------------------------------------------*/
const char *h5io_openmode2str ( int openmode )
{ if ((openmode<0)||(openmode>=H5ioEndOpenMode))
    openmode = H5ioInValidOpenMode;
  return( H5ioOpenModeStrings[openmode] );
} /* h5io_openmode2str */

/*---------------------------------------------------------------------------
NAME
  h5io_str2openmode ---  convert string to H5ioOpenMode

SYNOPSIS
  int h5io_str2openmode( const char *string );

DESCRIPTION
  convert string to H5ioOpenMode
---------------------------------------------------------------------------*/
int h5io_str2openmode( const char *string )
{ int  openmode=-1;

  while ( H5ioOpenModeStrings[++openmode] ) {
    if (!strlib_casecmp(H5ioOpenModeStrings[openmode], string)) break;
  }
  if (openmode>=H5ioEndOpenMode)
    openmode=H5ioInValidOpenMode;

  return(openmode);

} /* h5io_str2openmode */

/*==================== some useful internal functions ====================*/
/*
 * Returns the number of elements of an array with rank and dim[],
 * dim is a linear array with rank elements keeping the size of each dimension.
 */
hsize_t h5io_numberofelements(hsize_t rank,hsize_t *dims)
{ hsize_t nof=0;

  if (dims) {
    hsize_t i;
    nof=1;
    for (i=0;i<rank;i++) {
      nof*=*dims;
      dims++;
    }
  }

  return(nof);

} /* h5io_numberofelements */

/*
 * Returns for a C-type array with rank and dims[] the number of elements 
 * to skip when incrementing the index 0 by 1, dims is a linear array with 
 * rank elements keeping the size of each dimension.
 */
hsize_t h5io_pitch0(hsize_t rank,hsize_t *dims)
{ hsize_t pitch0=0;

  if (dims) {
    hsize_t i;
    pitch0=1;
    for (i=1;i<rank;i++) {
      dims++;
      pitch0*=*dims;
    }
  }

  return(pitch0);

} /* h5io_pitch0 */

/*
 * Prints a comma separated list of the array sizes starting with the
 * lowest index. Prints 1 if rank is zero, irrespective of dims.
 * If rank is negative, nothing is printed.
 */
void h5io_fprint_dims(FILE *out, hsize_t rank,hsize_t *dims) 
{
  if (rank==0) {
    fprintf(out,"[1]");
  } else if (rank>0) {
    if (dims) {
      hsize_t idim;
      fprintf(out,"[");
      for (idim=0;idim<rank;idim++) {
        if (idim==0) {
          fprintf(out,"%ld",(long) dims[idim]);
        } else {
          fprintf(out,", %ld",(long) dims[idim]);
        }
      }
      fprintf(out,"]");
    } else {
      fprintf(out,"(unset)");
    }
  }

} /* h5io_fprint_dims */

/*
 * Releases nframes pointers in pointer array 'value',
 * and releases the array 'value'. If nframes is less
 * than 1, only the array 'value' is released
 */
void **h5io_free_values( void **value, hsize_t nframes )
{
  if (value) {
    void **pps;
    pps=value;

    if (nframes>0) {
      hsize_t i;
      for (i=0;i<nframes;i++) {
        FREE(*pps);
        pps++;
      }
    }
    FREE(value);
  }

  return(NULL);

} /* h5io_free_values */

/*
 * Interpretes data at pdata as datatype and converts it to a string.
 * The pointer to the resulting string is returned or NULL in case of 
 * a conversion error.
 * For avoiding resource leaks the returned pointer must be released with
 * FREE(pointer).
 */
const char *h5io_number2string ( int datatype, void *pdata )
{
  char *string=NULL;
  int converted=0;
  char buffer[H5IO_BUFLEN];

  if (pdata) {
    switch (edf_datatype2machinetype(datatype)) {
      case MUnsignedChar: {
        unsigned char *tmp = pdata;
        if (snprintf(buffer,H5IO_BUFLEN-1,"%hhu",*tmp )>0) converted=1; break;
      }
      case MChar: {
        char *tmp = pdata;
        if (snprintf(buffer,H5IO_BUFLEN-1,"%hhd",*tmp )>0) converted=1; break;
      }
      case MUnsignedShort: {
        unsigned short *tmp = pdata;
        if (snprintf(buffer,H5IO_BUFLEN-1,"%hu", *tmp )>0) converted=1; break;
      }
      case MShort: {
        short *tmp = pdata;
        if (snprintf(buffer,H5IO_BUFLEN-1,"%hd", *tmp )>0) converted=1; break;
      }
      case MUnsignedInteger: {
        unsigned int *tmp = pdata;
        if (snprintf(buffer,H5IO_BUFLEN-1,"%u",  *tmp )>0) converted=1; break;
      }
      case MInteger: {
        int *tmp = pdata;
        if (snprintf(buffer,H5IO_BUFLEN-1,"%d",  *tmp )>0) converted=1; break;
      }
      case MUnsignedLong: {
        unsigned long *tmp = pdata;
        if (snprintf(buffer,H5IO_BUFLEN-1,"%lu", *tmp )>0) converted=1; break;
      }
      case MLong: {
        long *tmp = pdata;
        if (snprintf(buffer,H5IO_BUFLEN-1,"%ld", *tmp )>0) converted=1; break;
      }
      case MFloat: {
        float *tmp = pdata;
        if (snprintf(buffer,H5IO_BUFLEN-1,"%g", *tmp )>0) converted=1; break;
      }
      case MDouble: {
        double *tmp = pdata;
        if (snprintf(buffer,H5IO_BUFLEN-1,"%g", *tmp )>0) converted=1; break;
      }
  # if __WORDSIZE >= 64
      case MLongDouble: {
        long double *tmp = pdata;
        if (snprintf(buffer,H5IO_BUFLEN-1,"%Lg", *tmp )>0) converted=1; break;
      }
  # endif
      // default: (unknown)
    }
  }
  if (converted) {
    buffer[H5IO_BUFLEN-1]='\0';
    string = strlib_newstr(buffer);
  } else {
    fprintf(stdout,"WARNING (h5io_number2string): Cannot convert %s data at %p to a string\n",
      edf_datatype2string(datatype),pdata);
  }

  return(string);

} /* h5io_number2string */

/*==========================data structure BEGIN ==========================*/
/*---------------------------------------------------------------------------
NAME

  _h5io_init --- initialization of this module

---------------------------------------------------------------------------*/
void _h5io_init ( void )
{ hsize_t stream;

  for (stream=0;stream<MaxH5ioFiles;stream++)
    H5ioTable[stream]=NULL;

  H5ioInit = 1;

} /* _h5io_init */

/*---------------------------------------------------------------------------
NAME

  _h5io_init_data --- initialization of the h5io data set structure

---------------------------------------------------------------------------*/
void _h5io_init_data ( H5ioData *data, int dataclass )
{ if (data) {
    data->Name       = (char *) NULL;                             // the key
    data->H5Path     = (char *) NULL;       // path from H5id to the dataset
    data->H5id       = -1;                                // h5 path root id
    data->DataClass  = dataclass;                           // H5ioDataClass
    data->Binary     = (void *) NULL;// the binary buffer for the data array
    data->BinarySize = (hsize_t) 0;         // the size of the binary buffer
    data->DataType   = InValidDType;    // edf data type of an array element
    data->ItemSize   = (hsize_t) 0;   // size of a single data array element
    data->ByteOrder  = InValidBOrder;      // byte order of a single element
    data->Rank       = (hsize_t) 0;                // rank of the data array
    data->Dims       = (hsize_t *) NULL;     // dimensions of the data array
    data->MaxDims    = (hsize_t *) NULL;               // maximum dimensions
    data->Value      = (void **) NULL;        // pointer array to data value
    data->NValues    = (hsize_t) 0;             // length of the Value array 
    data->ValueRank  = (hsize_t) 0;             // rank after interpretation
    data->ValueDims  = (hsize_t *) NULL;  // dimensions after interpretation
    data->FillValue  = NULL;       // pointer to FillValue, NULL if not used
    data->H5data_id  = -1;              // if >=0, currently open dataset id
    data->Flags      = 0;                                 // H5io_Data Flags
    data->Previous   = (H5ioData *) NULL;                // previous dataset
    data->Next       = (H5ioData *) NULL;                    // next dataset
    data->Owner      = (H5ioNode *) NULL;                // the owning block
  }
} /* _h5io_init_array */

/*---------------------------------------------------------------------------
NAME

  _h5io_new_data --- creates new h5io data set 

RETURN VALUE
success:  H5ioData * pointer to new h5io data set
error:    NULL
---------------------------------------------------------------------------*/
H5ioData *_h5io_new_data ( const char *name, int dataclass )
{ H5ioData *newdata=NULL;

  if (name) {
    if ( (newdata = (H5ioData*) CALLOC(1,sizeof(H5ioData))) ) {
      _h5io_init_data ( newdata, dataclass );
      newdata->Name = strlib_newstr(name);
    }
  }
  return(newdata);

} /* _h5io_new_data */

/*---------------------------------------------------------------------------
NAME

  h5io_search_data --- searches the h5io data set in the data list of block

SYNOPSIS

  H5ioData *h5io_search_data ( H5ioNode *block, const char *name );

DESCRIPTION

RETURN VALUE
success H5ioData * found data set
error   NULL
---------------------------------------------------------------------------*/
H5ioData *h5io_search_data ( H5ioNode *block, const char *name )
{ H5ioData *current = NULL;

  if ((block)&&(name)) {

    /* search data set */
    current = block->DataList;
    if (current!=(H5ioData *) NULL )
    while( ( current!=(H5ioData *) NULL ) &&
           ( strlib_casecmp(current->Name,name)!=0 ) ) {
      current = current->Next;
    }
  }

  return( current );

} /* h5io_search_data */

/*---------------------------------------------------------------------------
NAME

  h5io_insert_data --- insert a h5io data set in the data list of block

SYNOPSIS

  H5ioData  *h5io_insert_data ( H5ioNode *block, const char *name,
                                H5ioDataClass dataclass )

DESCRIPTION

RETURN VALUE
success:            pointer to inserted data set
error:              NULL pointer
---------------------------------------------------------------------------*/
H5ioData  *h5io_insert_data ( H5ioNode *block, const char *name,
                              int dataclass )
{ H5ioData *data=NULL, *previous=NULL, *next=NULL, *newdata=NULL;
  int notfound = -1;

  if (iodbg(IODBG_H5|IODBG_DEBUG2)) printf("h5io_insert_data(%p,%s,%s) BEGIN\n",
    block, name, h5io_nodeclass2str(dataclass));

  if ((block)&&(name)) {

    /* search insertion point (insertion before next) */
    next = block->DataList;
    while( (next)&&(notfound<0) ) {
      notfound = strlib_numcasecmp(next->Name,name);
      if (notfound<0) { previous = next; next = next->Next; }
    }

    /* create new data set, if data set was not found */
    if ( notfound ) {
      /* create new data set */
      newdata = _h5io_new_data ( name, dataclass );
      if (!newdata) goto h5io_insert_data_error;

      /* insert new data set before next */
      if (next) next->Previous = newdata;
      newdata->Next=next;
      newdata->Previous=previous;
      if (previous) previous->Next=newdata;
        else block->DataList=newdata;

      /* link to owning block */
      newdata->Owner     = block;

      /* update flag */
      newdata->Flags    |= H5IO_DATA_INSERTED;

      next = newdata;
    }

    data = next;

    if (dataclass!=data->DataClass) 
      goto h5io_insert_data_error;

  }

  return( data );

h5io_insert_data_error:

  FREE( newdata );
  return( NULL );

} /* h5io_insert_data */

/*---------------------------------------------------------------------------
NAME

   h5io_empty_data --- empties data node from allocated data

SYNOPSIS

   void h5io_empty_data ( H5ioData *data );

DESCRIPTION
   All binary data in data data node is released. The links H5Path and
   H5id remain valid. The dimension array is not removed.
   The data can be read again. 

---------------------------------------------------------------------------*/
void h5io_empty_data ( H5ioData *data )
{
  if (data) {
    if (!data->Binary) {
      // if Binary is empty release each Value
      data->Value=h5io_free_values( data->Value, data->NValues );
    }
    FREE(data->Value);
    FREE(data->Binary);
    data->BinarySize = (hsize_t) 0; // (size_t is unsiged)
    data->Flags     &= ~H5IO_DATA_VALUE_READ; 
  }

} /* h5io_empty_data */

/*---------------------------------------------------------------------------
NAME

   h5io_remove_data --- removes data set from the data list

SYNOPSIS

   void h5io_remove_data ( H5ioData *data );

DESCRIPTION
The data set is removed from the data list of the owning block. The 
internally allocated data is released. If data was the only data set in 
block->DataList it is set to NULL.  If data is NULL, nothing is done.
---------------------------------------------------------------------------*/
void h5io_remove_data ( H5ioData *data )
{
  if (data) {
    H5ioData *previous=NULL, *next=NULL;

    previous   = data->Previous;
    next       = data->Next;

    if (next)       next->Previous = previous;
    if (previous)   previous->Next = next;
    else data->Owner->DataList = next;

    FREE(data->Name);
    FREE(data->H5Path);
    FREE(data->Dims);
    FREE(data->MaxDims);

    if (!data->Binary) {
      // if Binary is empty release each Value
      data->Value=h5io_free_values( data->Value, data->NValues );
    }
    FREE(data->Value);
    FREE(data->ValueDims);
    FREE(data->Binary);

    FREE(data->FillValue);
    if (data->H5data_id>=0) {
      printf("WARNING: h5io_remove_data(%p): H5data_id=%d seems to remain open\n",data,data->H5data_id);
    }

    FREE(data);
  }

  return;

} /* h5io_remove_data */

/*---------------------------------------------------------------------------
NAME

   h5io_remove_data_list --- empty the data list of block

SYNOPSIS

   void h5io_remove_data_list ( H5ioNode *block )

---------------------------------------------------------------------------*/
void h5io_remove_data_list ( H5ioNode *block )
{
  H5ioData *data=NULL, *next=NULL;

  if (block) {

    next = block->DataList;
    while ( next!=(H5ioData*) NULL ) {
      data = next;
      next=next->Next;
      h5io_remove_data ( data );
    }
    block->DataList = NULL;
  }

  return;

} /* h5io_remove_data_list */

/*---------------------------------------------------------------------------
NAME

   h5io_fprint_data_list --- prints h5io data contents

SYNOPSIS

   void h5io_fprint_data_list ( FILE* out, H5ioNode *block, int level );

---------------------------------------------------------------------------*/
void h5io_fprint_data_list ( FILE *out, H5ioNode *block, long level )
{
  int indent=(int) (iodbg_level()-level)+H5ioColumnStart;

  H5ioData *data=NULL;
  hsize_t nframe;

  if ((level>0)&&(block)) {
    data = block->DataList;

    while(data) {

      if (iodbg(IODBG_H5|IODBG_VERBOSE)) {
        fprintf(out,"%*s%s\n",indent,"",h5io_separation( iodbg_level()-level ));
        fprintf(out,"%*s%s%*s = '%s'\n",indent,"",h5io_dataclass2str(data->DataClass),\
          H5ioColumnWidth-(int) strlen(h5io_dataclass2str(data->DataClass)),"",data->Name);
        fprintf(out,"%*sH5Path          = '%s'\n",indent,"",data->H5Path);
        fprintf(out,"%*sH5id            = %d\n",indent,"",data->H5id);
        fprintf(out,"%*sDataClass       = %s\n",indent,"",h5io_dataclass2str(data->DataClass));
        fprintf(out,"%*sPrevious Data   = ",indent,"");
        if (data->Previous)
          fprintf(out,"'%s'\n", data->Previous->Name);
          else fprintf(out,"(no previous data)\n");
        fprintf(out,"%*sNext Data       = ",indent,"");
        if ((data->Next)!=(H5ioData*) NULL)
          fprintf(out,"'%s'\n", data->Next->Name);
          else fprintf(out,"(no next data)\n");
        fprintf(out,"%*sOwning Block    = %p\n",indent,"",data->Owner);
        fprintf(out,"%*sBinary          = %p\n",indent,"",data->Binary);
        fprintf(out,"%*sBinarySize      = %ld byte%s\n",indent,"",\
          (long) data->BinarySize,(data->BinarySize==1)?"":"s");
        fprintf(out,"%*sDataType        = %s\n",indent,"",edf_datatype2string(data->DataType));
        fprintf(out,"%*sItemSize        = %ld byte%s\n",indent,"",\
          (long) data->ItemSize,(data->ItemSize==1)?"":"s");
        fprintf(out,"%*sByteOrder       = %s\n",indent,"",edf_byteorder2string(data->ByteOrder));
        fprintf(out,"%*sRank            = %ld\n",indent,"",(long) data->Rank);
        fprintf(out,"%*sDims            = ",indent,"");
          h5io_fprint_dims(out,data->Rank,data->Dims) ;fprintf(out,"\n");
        fprintf(out,"%*sMaxDims         = ",indent,"");
          h5io_fprint_dims(out,data->Rank,data->MaxDims) ;fprintf(out,"\n");
        fprintf(out,"%*sValue           = %p\n",indent,"",data->Value);
        fprintf(out,"%*sNValues         = %ld\n",indent,"",(long) data->NValues);
        fprintf(out,"%*sValueRank       = %ld\n",indent,"",(long) data->ValueRank);
        fprintf(out,"%*sValueDims       = ",indent,"");
          h5io_fprint_dims(out,data->Rank,data->ValueDims) ;fprintf(out,"\n");
        fprintf(out,"%*sFillValue       = %p\n",indent,"",data->FillValue);
        fprintf(out,"%*sH5data_id       = %d\n",indent,"",data->H5data_id);
        fprintf(out,"%*sFlags           = 0x%x\n",indent,"",data->Flags);
        if (data->Flags&H5IO_DATA_VALUE_READ) {
          if (iodbg(IODBG_H5|IODBG_SHOWDATAVAL)) {
            switch (data->DataClass) {
              case H5ioSymbolNode:
                for (nframe=0;nframe<data->NValues;nframe++) {
                  fprintf(out,"%*s Value[%04ld]     = %p ('%s')\n",indent,"",(long) nframe+1,data->Value[nframe],(char *) data->Value[nframe]);
                }
                break;
              default:
                for (nframe=0;nframe<data->NValues;nframe++) {
                  fprintf(out,"%*s Value[%04ld]     = %p\n",indent,"",(long) nframe+1,data->Value[nframe]);
                }
            }
          }
        } else {
          fprintf(out,"%*s (value not read)\n",indent,"");
        }

        fprintf(out,"%*s%s\n",indent,"",h5io_separation( iodbg_level()-level ));
      } else { /* short */
        fprintf(out,"%*s%s%*s = '%s'\n",indent,"",h5io_dataclass2str(data->DataClass),\
          H5ioColumnWidth-(int) strlen(h5io_dataclass2str(data->DataClass)),"",data->Name);
        if (iodbg(IODBG_H5|IODBG_SHOWDATAVAL)) {
          switch (data->DataClass) {
            case H5ioSymbolNode:
              for (nframe=0;nframe<data->NValues;nframe++) {
                fprintf(out,"%*s Value[%04ld]     = '%s'\n",indent,"",(long) nframe+1,(char *) data->Value[nframe]);
              }
          }
        }

      }
      data=data->Next;
    }
  }

  return;

} /* h5io_fprint_data_list */

/*---------------------------------------------------------------------------
NAME

  h5io_node_name --- returns the name of node

SYNOPSIS

  const char *h5io_node_name ( H5ioNode *node );

DESCRIPTION

  Returns the name the current node or NULL if node or name does not exist.

---------------------------------------------------------------------------*/
const char *h5io_node_name ( H5ioNode *node )
{ const char *name=NULL;
  
  if (node) {
    name = node->Name;
  }
  
  return( name );

} /* h5io_node_name */

/*---------------------------------------------------------------------------
NAME

  h5io_node_number --- returns the name to a number (default 1)

SYNOPSIS

  long h5io_node_number ( H5ioNode *node );

DESCRIPTION

  Converts the name of the current node to a long integer number. If it is 
  not possible 1 is returned.

---------------------------------------------------------------------------*/
long h5io_node_number ( H5ioNode *node )
{ long number;
  int errval;

  number = num_str2long ( h5io_node_name ( node ), NULL, &errval);
  if (errval) number=1;

  return( number );

} /* h5io_node_number */

/*---------------------------------------------------------------------------
NAME

  h5io_data_list --- returns the data list of the current node

SYNOPSIS

  H5ioData *h5io_data_list ( H5ioNode *node );

DESCRIPTION

  Returns the data list of the current node or NULL if node or the data list
  is NULL.

---------------------------------------------------------------------------*/
H5ioData *h5io_data_list ( H5ioNode *node )
{ H5ioData *datalist=NULL;

  if (node) {
    datalist = node->DataList;
  }

  return( datalist );

} /* h5io_data_list */

/*---------------------------------------------------------------------------
NAME

  h5io_node_list --- returns the node list of the current node

SYNOPSIS

  H5ioNode *h5io_node_list ( H5ioNode *node );

DESCRIPTION

  Returns the node list of the current node or NULL if node or the node list
  is NULL.

---------------------------------------------------------------------------*/
H5ioNode *h5io_node_list ( H5ioNode *node )
{ H5ioNode *nodelist=NULL;

  if (node) {
    nodelist = node->NodeList;
  }

  return( nodelist );

} /* h5io_node_list */

/*---------------------------------------------------------------------------
NAME

  _h5io_init_node --- initialization of the h5io node structure

SYNOPSIS

  void _h5io_init_node ( H5ioNode *node, H5ioNodeClass nodeclass );

---------------------------------------------------------------------------*/
void _h5io_init_node ( H5ioNode *node, int nodeclass )
{ if (node) {
    node->Name       = NULL;
    node->H5id       = -1;
    node->NodeClass  = nodeclass;
    node->NodeList   = NULL;
    node->DataList   = NULL;
    node->H5Path     = NULL;
    node->Previous   = NULL;
    node->Next       = NULL;
    node->Owner      = NULL;
    node->NodeNumbers = NULL;
  }
  return;
} /* _h5io_init_node */

/*---------------------------------------------------------------------------
NAME

  _h5io_new_node --- creates a new h5io node

SYNOPSIS

  H5ioNode *_h5io_new_node ( const char *name, H5ioNodeClass nodeclass );

RETURN VALUE
success:  H5ioNode * pointer to new h5io node
error:    NULL
---------------------------------------------------------------------------*/
H5ioNode *_h5io_new_node ( const char *name, int nodeclass )
{ H5ioNode *newnode=NULL;

  if (name) {
    if ( ( newnode = (H5ioNode *) CALLOC(1,sizeof(H5ioNode)) ) ) {
      _h5io_init_node ( newnode, nodeclass );
      newnode->Name = strlib_newstr(name);
    }
  }

  return(newnode);

} /* _h5io_new_node */

/*---------------------------------------------------------------------------
NAME

  h5io_search_node --- searches the h5io node by 'name' in 'owner'

SYNOPSIS

  H5ioNode *h5io_search_node ( H5ioNode *owner, const char *name )

DESCRIPTION

RETURN VALUE
success H5ioNode *found node
error   NULL
---------------------------------------------------------------------------*/
H5ioNode *h5io_search_node ( H5ioNode *owner, const char *name )
{ H5ioNode *current = NULL;

  if ((owner)&&(name)) {

    /* search node */
    current = owner->NodeList;
    while( ( current ) &&
           ( strlib_casecmp(current->Name,name) ) ) {
      current = current->Next;
    }

  }
  return( current );

} /* h5io_search_node */

/*---------------------------------------------------------------------------
NAME

  h5io_insert_node --- insert a h5io node to the node list of owner

SYNOPSIS

  H5ioNode  *h5io_insert_node ( H5ioNode *owner, const char *name,
                                H5ioNodeClass nodeclass );

DESCRIPTION

RETURN VALUE
success:            pointer to inserted node
error:              NULL
---------------------------------------------------------------------------*/
H5ioNode *h5io_insert_node ( H5ioNode *owner, const char *name, 
                             int nodeclass )
{ H5ioNode *node=NULL, *previous=NULL, *next=NULL, *newnode=NULL;
  int notfound = -1;

  if (iodbg(IODBG_H5|IODBG_DEBUG2)) printf("h5io_insert_node(%p,%s,%s) BEGIN\n",
    owner,name,h5io_nodeclass2str(nodeclass));

  if ((owner)&&(name)) {

    /* search insertion point (insertion before *pnext) */
    next = owner->NodeList;
    while( (next)&&(notfound<0) ) {
      notfound = strlib_numcasecmp(next->Name,name);
      if (notfound<0) { previous = next; next = next->Next; }
    }

    /* create new node, if no node found */
    if ( notfound  ) {
      /* create new node */
      newnode = _h5io_new_node ( name, nodeclass );
      if ( !newnode ) goto h5io_insert_node_error;

      /* insert new node before *pnext */
      if (next) next->Previous = newnode;
      newnode->Next=next;
      newnode->Previous=previous;
      if (previous) previous->Next=newnode;
        else owner->NodeList=newnode;

      /* link to owner */
      newnode->Owner   = owner;

      next = newnode;
    }

    node = next;

    if (node->NodeClass != nodeclass) 
      goto h5io_insert_node_error;

  }

  if (iodbg(IODBG_H5|IODBG_DEBUG2)) printf("h5io_insert_node(%p,%s,%s) (%p) END\n",
    owner,name,h5io_nodeclass2str(nodeclass),node);

  return( node );

h5io_insert_node_error:

  if (iodbg(IODBG_H5|IODBG_DEBUG2)) printf("h5io_insert_node(%p,%s,%s) (NULL) ERROR\n",
    owner,name,h5io_nodeclass2str(nodeclass));

  return( NULL );

} /* h5io_insert_node */

/*---------------------------------------------------------------------------
NAME

   h5io_remove_node --- removes node from the node list

SYNOPSIS

   void h5io_remove_node ( H5ioNode *node );

DESCRIPTION
All allocations of node and all links to it are removed. If node is the
only node in 'node->Owner->NodeList' 'node->Owner->NodeList' will
be set to NULL. If 'node' is NULL, nothing will be done.
---------------------------------------------------------------------------*/
void h5io_remove_node ( H5ioNode *node )
{
  H5ioNode *previous=NULL, *next=NULL;

  if (node) {

    H5ioNode *owner;

    owner = node->Owner;

    previous   = node->Previous;
    next       = node->Next;

    h5io_remove_data_list ( node );
    h5io_remove_node_list ( node );

    if (next)       next->Previous = previous;
    if (previous)   previous->Next = next;
    else if (owner) owner->NodeList = next;

    FREE(node->NodeNumbers);
    FREE(node->Name);
    FREE(node->H5Path);
    FREE(node);

  }

  return;

} /* h5io_remove_node */

/*---------------------------------------------------------------------------
NAME

   h5io_remove_node_list --- empty the node list of owner

SYNOPSIS

   void h5io_remove_node_list ( H5ioNode *owner )

---------------------------------------------------------------------------*/
void h5io_remove_node_list ( H5ioNode *owner )
{
  H5ioNode *node=NULL, *next=NULL;
  H5ioNode *nodelist;

  nodelist = h5io_node_list ( owner );

  if (nodelist) {

    next = nodelist;
    while ( next ) {
      node = next;
      next=next->Next;
      h5io_remove_node ( node );
    }
    owner->NodeList = NULL;

  }

  return;

} /* h5io_remove_node_list */

void h5io_fprint_node_list ( FILE *out, H5ioNode *owner, long level );

/*---------------------------------------------------------------------------
NAME

   h5io_fprint_node --- prints h5io node contents

SYNOPSIS

   void h5io_fprint_node ( FILE *out, H5ioNode *node, long level );

---------------------------------------------------------------------------*/
void h5io_fprint_node ( FILE *out, H5ioNode *node, long level )
{
  int indent=(int) (iodbg_level()-level)+H5ioColumnStart;

  if ((node)&&(level>0)) {
    if (iodbg(IODBG_H5|IODBG_VERBOSE)) {
      fprintf(out,"%*s%s\n",indent,"",h5io_separation( iodbg_level()-level ));
      fprintf(out,"%*s%s%*s = '%s'\n",indent,"",h5io_nodeclass2str(node->NodeClass),\
        H5ioColumnWidth-(int) strlen(h5io_nodeclass2str(node->NodeClass)),"",node->Name);
      fprintf(out,"%*sH5Path          = '%s'\n",indent,"",node->H5Path);
      fprintf(out,"%*sH5id            = %d\n",indent,"",node->H5id);
      fprintf(out,"%*sNodeClass       = '%s'\n",indent,"",h5io_nodeclass2str(node->NodeClass));
      fprintf(out,"%*sPrevious Node   = ",indent,"");
      if (node->Previous)
        fprintf(out,"'%s'\n", node->Previous->Name);
        else fprintf(out,"%s\n","(no previous node)");
      fprintf(out,"%*sNext Node       = ",indent,"");
      if (node->Next)
        fprintf(out,"'%s'\n", node->Next->Name);
        else fprintf(out,"(no next node)\n");
      if (node->Owner)
        fprintf(out,"%*sOwning Node     = '%s'\n",indent,"",node->Owner->Name);
        else fprintf(out,"%*s (no owning node)\n",indent,"");
      fprintf(out,"%*sDataList        = %p\n",indent,"",h5io_data_list(node));
      fprintf(out,"%*sNodeList        = %p\n",indent,"",h5io_node_list(node));
      h5io_fprint_data_list( out, node, level-1 );
      h5io_fprint_node_list( out, node, level-1 );
      fprintf(out,"%*sNodeNumbers     = %p\n",indent,"",node->NodeNumbers);
      fprintf(out,"%*s%s\n",indent,"",h5io_separation( iodbg_level()-level ));
    } else { /* short */
      fprintf(out,"%*s%s%*s = '%s'\n",indent,"",h5io_nodeclass2str(node->NodeClass),\
        H5ioColumnWidth-(int) strlen(h5io_nodeclass2str(node->NodeClass)),"",node->Name);
      h5io_fprint_data_list ( out, node, level-1 );
      h5io_fprint_node_list( out, node, level-1 );
    }
  }

  return;

} /* h5io_fprint_node */

/*---------------------------------------------------------------------------
NAME

   h5io_fprint_node_list --- prints h5io node list

SYNOPSIS

   void h5io_fprint_node_list ( FILE *out, const H5ioNode *node,
                                long level );

---------------------------------------------------------------------------*/
void h5io_fprint_node_list ( FILE *out, H5ioNode *owner, long level )
{ 
  if ((level>0)&&(owner)) {
    H5ioNode *node=NULL;

    node = h5io_node_list( owner );
    while(node) {
      h5io_fprint_node ( out, node, level );
      node=node->Next;
    }
  }

  return;

} /* h5io_fprint_node_list */

/*---------------------------------------------------------------------------
NAME

   h5io_fprint_streams --- prints h5io file contents

SYNOPSIS

   void h5io_fprint_streams ( FILE *out, long level )

---------------------------------------------------------------------------*/
void h5io_fprint_streams ( FILE *out, long level )
{
  int indent=(int) ((iodbg_level()-level))+H5ioColumnStart;

  H5ioNode *file=NULL;

  if (!H5ioInit) _h5io_init ();

  if (level>0) {
    int stream=-1;
    for (stream=0;stream<MaxH5ioFiles;stream++) {
      if ((file=H5ioTable[stream])) {
        fprintf(out,"%*sStream          = %d\n",indent,"",stream);
        h5io_fprint_node ( out, file, level-1 );
      } // else unused
    }
  }

  return;

} /* h5io_fprint_streams */

/*---------------------------------------------------------------------------
NAME

  h5io_search_stream --- searches the stream with name and returns its id

SYNOPSIS

  int h5io_search_stream ( const char *name );

DESCRIPTION

  Searches a stream with name and returns its stream number. If the stream
  cannot been found the returned value is negative.

RETURN VALUE
name found:      int stream >= 0
name not found:             <0
---------------------------------------------------------------------------*/
int h5io_search_stream ( const char *name ) 
{ int stream=-1, istream;
  H5ioNode *file=NULL;

  if (!H5ioInit) _h5io_init ();

  for (istream=0;istream<MaxH5ioFiles;istream++) {
    if ((file=H5ioTable[istream])) {
      if (strlib_casecmp(file->Name,name)==0) {
        stream=istream;
        break;
      }
    }
  }

  return( stream );

} /* h5io_search_stream */

/*---------------------------------------------------------------------------
NAME

  h5io_new_stream --- Gets a new stream for name and returns the stream number

SYNOPSIS

  int h5io_new_stream ( const char *name );

DESCRIPTION

  Creates a file structure and returns the stream index.
  The file structure is initialized with name.
  For avoiding resource leaks the stream must be released after use with
  h5io_free_stream.

RETURN VALUE
success: int stream >= 0
error:              <0
---------------------------------------------------------------------------*/
int h5io_new_stream ( const char *name )
{ int stream=-1, istream;
  H5ioNode *file=NULL;

  if (!H5ioInit) _h5io_init ();

  for (istream=0;istream<MaxH5ioFiles;istream++) {
    if (!H5ioTable[istream]) {
      file =  _h5io_new_node ( name, H5ioFileNode );
      if (file) {
        stream=istream;
        H5ioTable[stream]=file;
      }
      break;
    }
  }

  return( stream );

} /* h5io_new_stream */

/*---------------------------------------------------------------------------
NAME

   h5io_free_stream --- releases all entries of the specified stream

SYNOPSIS

   int h5io_free_stream ( int stream );

DESCRIPTION

This function releases all buffers and structures used for the stream.
This function does not close the file. This must be done separately before
calling this function.

RETURN VALUE
success:   0
error:     <0 
---------------------------------------------------------------------------*/
int h5io_free_stream ( int stream )
{ H5ioNode *file=NULL;
  int status=-1;

  if (!H5ioInit) _h5io_init ();

  if ((stream>=0)&&(stream<MaxH5ioFiles)) {
    file = H5ioTable[stream];
    if (file)  {
      /* remove data list (if there is any) */
      h5io_remove_data_list ( file );
      /* remove node list */
      h5io_remove_node_list( file );
      /* release file name */
      FREE(file->Name);
      /* release file structure */
      FREE(H5ioTable[stream]);
    }
    status=0;
  }

  return(status);
  
} /* h5io_free_stream */

/*---------------------------------------------------------------------------
NAME

   h5io_stream_info --- returns pointer to file information

SYNOPSIS

   H5ioNode *h5io_stream_info( int stream );

DESCRIPTION

   This function returns a pointer to the H5ioNode structure.

RETURN VALUE
   On success H5ioTable[stream] or NULL in case of an error.
---------------------------------------------------------------------------*/
H5ioNode *h5io_stream_info( int stream )
{ H5ioNode *info=NULL;

  if (!H5ioInit) _h5io_init ();

  if ((stream>=0)&&(stream<MaxH5ioFiles)) {
    info = H5ioTable[stream];
  }

  return(info);

} /* h5io_stream_info */

/*==========================data structure END ============================*/
