/*
 *   Project: The SPD Image correction and azimuthal regrouping
 *                      http://forge.epn-campus.eu/projects/show/azimuthal
 *
 *   Copyright (C) 2005-2010 European Synchrotron Radiation Facility
 *                           Grenoble, France
 *
 *   Principal authors: P. Boesecke (boesecke@esrf.fr)
 *                      R. Wilcke (wilcke@esrf.fr)
 *
 *   This program is free software: you can redistribute it and/or modify
 *   it under the terms of the GNU Lesser General Public License as published
 *   by the Free Software Foundation, either version 3 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU Lesser General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   and the GNU Lesser General Public License  along with this program.
 *   If not, see <http://www.gnu.org/licenses/>.
 */

# define AVG_VERSION      "avg : V1.04 Peter Boesecke 2017-06-16"

/*+++------------------------------------------------------------------------
NAME
  avg.c --- frame averaging using statistical functions

INCLUDE FILES
  # include avg.h

PURPOSE
  Creates averages and variances from stacked images (3d volume).

AUTHOR
  2012 Peter Boesecke (PB)

HISTORY
  2016-05-19 V1.00 PB extracted from saxs_average.c V4.07
  2016-08-11 V1.01 PB string2avgmode, fill_empty_avgpoints:
                      unused variables removed and format corrected
  2016-11-29 V1.02 PB fill_empty_avgpoints: initialized j_1 and j_2 with 0
                      for avoiding compilation warnings.
  2017-05-19 V1.03 PB ioalloc
  2017-06-16 V1.04 PB new_avg_data: allocates and resizes AvgData buffers.

--------------------------------------------------------------------------*/

/****************************************************************************
*  Include                                                                  *
****************************************************************************/

# include "avg.h"

/*--------------------------------------------------------------------------
NAME

  avg_version --- returns pointer to the version string

SYNOPSIS

  const char *avg_version ( void );

DESCRPTION

  Returns pointer to the version string.

--------------------------------------------------------------------------*/
PUBLIC const char *avg_version ( void )
{
  return ( AVG_VERSION );
} /* avg_version */

/***************************************************************************
* Averaging Mode Translation Strings                                       *
* enum AvgMode { InValidAvgMode, Mean=1, Median, Quantil, EndAvgMode };    *
***************************************************************************/
static const char * AvgModeStrings[] = { "Invalid", "mean", "median",
                                       "quantil", (const char *) NULL };

/*---------------------------------------------------------------------------
NAME
  String2AvgMode --- converts a string to an averaging mode

SYNOPSIS
  (AvgMode) int String2AvgMode( const char * string );

DESCRIPTION

RETURN VALUE
  AvgMode == 0 : error, e.g. cannot convert
  AvgMode >  0 : valid averaging mode value

  -------------------------------------------------------------------------*/
int string2avgmode( const char *string )
{ int i, value=InValidAvgMode;
  size_t n;

  n = strlen(string);

  for (i=0;i<EndAvgMode;i++) {
    if (!(strlib_ncasecmp( string, AvgModeStrings[i], n ))) {
      value = i;
      break;
    };
  }

  return( value );

} // string2avgmode

/*---------------------------------------------------------------------------
NAME
  avgmode2string --- converts an averaging mode to a string

SYNOPSIS
  const char *avgmode2string( (AvgMode) int amode );

DESCRIPTION

RETURN VALUE
  "Invalid" : error, e.g. cannot convert
  averaging mode strings (in case of no errors)

  -------------------------------------------------------------------------*/
const char *avgmode2string( int amode )
{ if ((amode<0)||(amode>=EndAvgMode)) amode = InValidAvgMode;
   return( AvgModeStrings[amode] );
} // avgmode2string

/*---------------------------------------------------------------------------
NAME
  fminval --- returns the minimum value of float array a with length n

SYNOPSIS
  float fminval ( float *a, long n )

DESCRIPTION
  Returns the minimum value of float array a with length n.

RETURN VALUE
  Minimum value of float array a for n>0 and a!=NULL

  -------------------------------------------------------------------------*/
float fminval ( float *a, long n )
{ float value=0.0;

  if ( a ) {
    float *pa;
    pa = a;
    if (n>0)
      value = *pa;
    for (pa=a+1;pa<a+n;pa++) {
      if (*pa<value)
        value = *pa;
    }
  }

  return(value);

} // fminval

/**********************************************************************************
* Create, define, read and release a list of points (BEGIN)                       *
*                                                                                 *
* AvgListOfPoints *new_avglistofpoints( void );                                   *
* AvgListOfPoints *free_avglistofpoints( AvgListOfPoints *listofpoints );         *
* long length_avglistofpoints( AvgListOfPoints *listofpoints );                   *
* AvgPoint *append_avgpoint( AvgListOfPoints *listofpoints, long i_1, long i_2 ); *
* AvgPoint *first_avgpoint( AvgListOfPoints *listofpoints );                      * 
* AvgPoint *get_avgpoint_and_increment( AvgPoint *point, long *pi_1, long *pi_2 ) *
* void print_avglistofpoints( FILE *out, AvgListOfPoints *listofpoints )          *
**********************************************************************************/

/*---------------------------------------------------------------------------
NAME

  append_avgpoint --- Writes a point to the end of the list

PURPOSE

  Writes a new point to the end of the list. If listofpoints is NULL
  nothing happens.

RETURN VALUE

  Pointer to the new point or NULL in case of error.

---------------------------------------------------------------------------*/
AvgPoint * append_avgpoint( AvgListOfPoints * listofpoints, long i_1, long i_2 )
{ AvgPoint * new=NULL;

  if (listofpoints) {
    AvgPoint * current;
    size_t size_point;

    size_point = sizeof ( AvgPoint );
    new = MALLOC ( size_point ); 
    if (!new)
      goto append_avgpoint_error;

    new->i_1 = i_1;
    new->i_2 = i_2;
    new->nextpoint=NULL;

    current = listofpoints->lastpoint;
    if (current) {
      current->nextpoint = new;
    } else {
      listofpoints->firstpoint = new;
    }
    listofpoints->lastpoint = new;
    listofpoints->npoints+=1l;
  }

  return(new);

append_avgpoint_error:

  FREE(new);

  return(NULL);

} // append_avgpoint

/*---------------------------------------------------------------------------
NAME

  first_avgpoint --- Returns pointer to the first point

PURPOSE

  Returns pointer to the first point

RETURN VALUE

  Pointer to the first point or NULL

---------------------------------------------------------------------------*/
AvgPoint * first_avgpoint( AvgListOfPoints * listofpoints )
{ AvgPoint * first=NULL;

  if (listofpoints) {
    first = listofpoints->firstpoint;
  }

  return(first);

} // first_avgpoint

/*---------------------------------------------------------------------------
NAME

  get_avgpoint_and_increment --- Returns the current point and goes to the next

PURPOSE
  
  Returns the indices of the current point and 
  returns pointer to the next point.
  If the input point is not NULL the indices are returned in
  *pi_1 and *pi_2.

RETURN VALUE

  Pointer to the next point or NULL

---------------------------------------------------------------------------*/
AvgPoint * get_avgpoint_and_increment( AvgPoint * point, long *pi_1, long *pi_2 )
{ AvgPoint * next=NULL;

  if (point) {
      if (pi_1) *pi_1 = point->i_1;
      if (pi_2) *pi_2 = point->i_2;
    next = point->nextpoint;
  }

  return( next );

} // get_avgpoint_and_increment 

/*---------------------------------------------------------------------------
NAME

  new_avglistofpoints( ) --- Allocates a new list of points

PURPOSE

  Creates a new list of points

RETURN VALUE

  Pointer to root of new point list or NULL in case of error.

---------------------------------------------------------------------------*/
AvgListOfPoints * new_avglistofpoints( void ) {
  size_t size_listofpoints;
  AvgListOfPoints * new=NULL;

  size_listofpoints = sizeof(AvgListOfPoints);
  new = MALLOC(size_listofpoints);
  if (!new) 
    goto new_avglistofpoints_error;

  new->npoints = 0l;
  new->firstpoint = NULL;
  new->lastpoint = NULL;

  return(new);

new_avglistofpoints_error:

  FREE(new);

  return(NULL);

} // new_avglistofpoints

/*---------------------------------------------------------------------------
NAME

  free_avglistofpoints --- Releases a list of points

PURPOSE

  Releases the memory of a list of points.

RETURN VALUE

  NULL, otherwise error

---------------------------------------------------------------------------*/
AvgListOfPoints * free_avglistofpoints( AvgListOfPoints * listofpoints ) 
{ if (listofpoints) {
    AvgPoint * current, * next;

    // release all points 
    next = listofpoints->firstpoint;
    while (next) {
      current = next;
      next = current->nextpoint;
      FREE(current);
    }
    
    FREE(listofpoints); 
  }

  return( NULL );

} // free_avglistofpoints

/*---------------------------------------------------------------------------
NAME

  length_avglistofpoints --- return the number of points in the list

PURPOSE

  Return the number of points in the list

RETURN VALUE

  Number of points.

---------------------------------------------------------------------------*/
long length_avglistofpoints( AvgListOfPoints * listofpoints )
{ long npoints = 0l;

  if ( listofpoints ) {
    npoints = listofpoints->npoints;
  }

  return( npoints );

} // length_avglistofpoints

/*---------------------------------------------------------------------------
NAME
  
  print_avglistofpoints --- print list of points 
  
PURPOSE
  
  Print list of points

---------------------------------------------------------------------------*/
void print_avglistofpoints( FILE * out, AvgListOfPoints * listofpoints )
{ AvgPoint * point;
  long i;

  fprintf( out, "\nList of %ld points\n", length_avglistofpoints(listofpoints));

  i=0;
  point = first_avgpoint( listofpoints );
  while ( point ) {
    long i_1, i_2;

    point = get_avgpoint_and_increment( point, &i_1, &i_2 );

    fprintf( out, "%ld: (%ld,%ld)\n", i++, i_1, i_2);

  }

  fprintf( out, "\n");

} // print_avglistofpoints

/*---------------------------------------------------------------------------
NAME
  
  print_listofdatapoints --- print list of points and values
  
PURPOSE
  
  Print list of points

---------------------------------------------------------------------------*/
void print_listofdatapoints( FILE * out, AvgListOfPoints * listofpoints,
                             float *data, long dim_1, long dim_2,
                             const char *name )
{ AvgPoint * point; 
  long i;

  fprintf( out, "\nList of %ld data points\n", length_avglistofpoints(listofpoints));
    
  i=0;
  point = first_avgpoint( listofpoints );
  while ( point ) {
    long i_1, i_2;
    float *pdata;
  
    point = get_avgpoint_and_increment( point, &i_1, &i_2 );

    pdata = ABSPTR(data,dim_1,dim_2,i_1,i_2);
    fprintf( out, "%ld: %s(%ld,%ld) = %g\n", i++, name, i_1, i_2, *pdata);

  }

  fprintf( out, "\n");

} // print_avglistofpoints

/***************************************************************************
* Create, define, read and release a list of points (END)                  *
***************************************************************************/

/*---------------------------------------------------------------------------
NAME
  fill_empty_avgpoints --- Replaces all empty points with the local average

PURPOSE
  Replaces all empty points in data with the average value in a pixel radius 
  around it. The variance array is not used. The number of updated points
  is returned.
  
ARGUMENTS
  float * data    (i/o) : input/output array
  long    dim_1   (i)   : data array dimension 1
  long    dim_2   (i)   : data array dimension 2
  float   dummy   (i)   : dummy value
  float   ddummy  (i)   : ddummy value
  float   fil_1   (i)   : length 1 of the averaging area 
  float   fil_2   (i)   : length 2 of the averaging area
  int     debug   (i)   : 1 print some debug information
                        : 0 do not print debug information

  int *pstatus    (o)   : return status
                           0 : Success
                          -1 : Failed
                          -2 : memory allocation error

RETURN VALUE

  long fillcnt : number of updated empty points values

  
---------------------------------------------------------------------------*/
long fill_empty_avgpoints( float *data, long dim_1, long dim_2, 
                      AvgListOfPoints * emptypoints,
                      float dummy, float ddummy, float fil_1, float fil_2, 
                      int debug, int *pstatus )
{

  long int j_1=0, j_2=0;
  float *pdata;

  float *copy=NULL, *pcopy;
  size_t copy_size;

  int status;

  AvgPoint * emptypoint=NULL;
  float f1_1, f1_2, f3_1, f3_2; // index coordinates
  float sum, weight;
  long fillcnt;

  status = 0;

  fil_1 = MAX2(0.0,fil_1);
  fil_2 = MAX2(0.0,fil_2);

  fillcnt = 0;

  if ( (length_avglistofpoints( emptypoints )>0) && ((fil_1>0.0) || (fil_2>0.0)) ) {

    // allocate memory for copy
    copy_size = sizeof(float)*dim_1*dim_2;
    copy = (float *) MALLOC (copy_size);
    if (!copy) {
      status=-2; // memory allocation error
      goto fill_empty_avgpoints_error;
    }

    if (debug>2) 
      print_listofdatapoints( stdout, emptypoints, data, dim_1, dim_2, "data");

    // make a copy of the data array
    memmove(copy,data,copy_size);

    if (debug>2) 
      print_listofdatapoints( stdout, emptypoints, copy, dim_1, dim_2, "copy");

    // It must be excluded that averages use empty points. If the dummy value is
    // not set the empty points in copy must be set to a temporary dummy value.
    // The dummy value must be outside the range of values of the array copy.
    if ( NODUMMYDEFINED(dummy, ddummy) ) {
      // define temporary dummy and ddummy
      dummy = FLOORF ( MIN2( fminval ( copy, dim_1*dim_2 ),0.0 ) ) - 10 ;
      ddummy = DDSET(dummy);
      // set all empty points in copy to dummy
      emptypoint = first_avgpoint( emptypoints );
      while ( emptypoint ) {
        emptypoint = get_avgpoint_and_increment( emptypoint, &j_1, &j_2 );
        pcopy = ABSPTR(copy,dim_1,dim_2,j_1,j_2);
        *pcopy = dummy;
      }
    } // else already defined as dummy

    if (debug) {
      printf("fill_empty_avgpoints: internal dummy is %g+-%g\n",dummy,ddummy);
    }

    // replace empty points in data

    emptypoint = first_avgpoint( emptypoints );

    while ( emptypoint ) {

      emptypoint = get_avgpoint_and_increment( emptypoint, &j_1, &j_2 );
      pdata = ABSPTR(data,dim_1,dim_2,j_1,j_2);

      // average a rectangular region in copy with length (fil_1,fil_2) 
      // centered at pixel (j_1, j_2) (array coordinate of center is j_1+0.5)
      f1_1 = A2INDEX(j_1 + 0.5 - fil_1*0.5);
      f1_2 = A2INDEX(j_2 + 0.5 - fil_2*0.5);

      f3_1 = A2INDEX(j_1 + 0.5 + fil_1*0.5);
      f3_2 = A2INDEX(j_2 + 0.5 + fil_2*0.5);

      if (Isum2ldw( copy, (int) dim_1, (int) dim_2, dummy, ddummy,
                    f1_1, f1_2, f3_1, f3_2, &sum, &weight ) ) {
        *pdata = sum/weight;
        fillcnt++;
      } else {
        if (debug) {
          printf("fill_empty_avgpoints: Bad pixel (%ld,%ld) could not be filled.\n",j_1,j_2);
        }
      }

    }

    FREE(copy);

  } // else nothing to do

  if (pstatus) *pstatus = status; 

  return(fillcnt);

fill_empty_avgpoints_error:

  FREE(copy);

  if (pstatus) *pstatus = status; 

  return(fillcnt);

} // fill_empty_avgpoints

/*---------------------------------------------------------------------------
NAME

  new_avg_data --- Allocates all avg_data buffers frame data, dummy and ddummy

SYNOPSIS

  AvgData *new_avg_data(AvgData *avg_data, long dim_1, long dim_2, long dim_3);

DESCRIPTION

  If avg_data is NULL the AvgData structure is allocated, otherwise it is 
  reused. The data buffer sizes of avg_data are resized according to dim_1, 
  dim_2 and dim_3. If dim_3 is zero all buffer memories are released.

    avg_data sizeof(AvgData);
    avg_data->data sizeof(float)*dim_1*dim_2*dim_3
    avg_data->dummy sizeof(float)*dim_1
    avg_data->ddummy sizeof(float)*dim_1
  
  In case of success the pointer to avg_data is returned. In case of an
  error avg_data is released and NULL is returned.

PURPOSE

  If the input value of avg_data is NULL a new AvgData structure is allocated 
  and used, otherwise the input avg_data structure is used. In case of success,
  the pointer to the used avg_data structure is returned and must be released with
  free_avg_data for avoiding memory leaks. In case of an error, the AvgData
  structure is released and the return values is NULL. If dim_1, dim_2 and dim_3 are 
  positive or zero, the buffers of data, dummy and ddummy are increased in size
  if necessary, otherwise they are kept.

  If dim_1, dim_2 and dim_3 are negative, the buffers of data, dummy and ddummy 
  are released and the frame counter is reset to 0.

RETURN VALUE

  In case of success: pointer to avg_data
  in case of error: NULL

---------------------------------------------------------------------------*/
AvgData *new_avg_data(AvgData *avg_data, long dim_1, long dim_2, long dim_3)
{
  size_t size;

  if (avg_data==NULL) {
    if (!(avg_data =  (AvgData *) CALLOC( 1, sizeof ( AvgData ) ) ) ) {
      goto new_avg_data_error;
    }
  }

  if ( (dim_1>=0) && (dim_2>=0) && (dim_3>=0) ) {

    /* allocate frame averaging buffers with dim_1*dim_2*dim_3 pixels */
    size = sizeof(float)*dim_1*dim_2*dim_3;
    //++++++++ if (avg_data->data_size != size) {
    //++++++++   FREE(avg_data->data); 
    //++++++++   avg_data->data_size = 0;
    //++++++++ }
    if (avg_data->data_size < size) {
      if (!(avg_data->data= (float*) REALLOC( avg_data->data, size ))) {
        fprintf(stderr,"ERROR: Cannot allocate %zu bytes for data (%ldx%ldx%ld floats)\n",\
          size, dim_1,dim_2,dim_3);
        avg_data->data_size = 0;
        goto new_avg_data_error;
      }
      avg_data->data_size = size;
    }

    size = sizeof(float)*dim_3;
    //++++++++ if (avg_data->dummy_size != size) {
    //++++++++   FREE(avg_data->dummy);
    //++++++++   avg_data->dummy_size = 0;
    //++++++++ }
    if (avg_data->dummy_size < size) {
      if (!(avg_data->dummy= (float*) REALLOC( avg_data->dummy, size ))) {
        fprintf(stderr,"ERROR: Cannot allocate %zu bytes for dummies (%ld floats)\n",
          size, dim_3);
        avg_data->dummy_size = 0;
        goto new_avg_data_error;
      }
      avg_data->dummy_size = size;
    }

    size = sizeof(float)*dim_3;
    //++++++++ if (avg_data->ddummy_size != size) {
    //++++++++   FREE(avg_data->ddummy);
    //++++++++   avg_data->ddummy_size = 0;
    //++++++++ }
    if (avg_data->ddummy_size < size) {
      if (!(avg_data->ddummy= (float*) REALLOC( avg_data->ddummy, size ))) {
        fprintf(stderr,"ERROR: Cannot allocate %zu bytes for ddummies (%ld floats)\n",
          size, dim_3);
        avg_data->ddummy_size = 0;
        goto new_avg_data_error;
      }
      avg_data->ddummy_size = size;
    }

  } else {
    // Reiinitialize buffers and counters
    FREE(avg_data->data);
    avg_data->data_size=0;
    FREE(avg_data->dummy);
    avg_data->dummy_size=0;
    FREE(avg_data->ddummy);
    avg_data->ddummy_size=0;

    avg_data->counter=0;

  }

  return(avg_data);

new_avg_data_error:

  free_avg_data(avg_data);

  return(NULL);

} // new_avg_data

/*---------------------------------------------------------------------------
NAME

  free_avg_data --- Release AvgData structure

PURPOSE

  Release AvgData structure

RETURN VALUE

  NULL

---------------------------------------------------------------------------*/
AvgData *free_avg_data ( AvgData *avg_data )
{
  if (avg_data) {
    FREE (avg_data->data);
    FREE (avg_data->dummy);
    FREE (avg_data->ddummy);
    FREE (avg_data);
  }
  return( NULL );
} // free_avg_data

/*---------------------------------------------------------------------------
NAME
  average --- Calculate average and variance of pixel (i_1,i_2)

PURPOSE
  Calculate average and variance for pixel (i_1,i_2). The type of average
  (mean, median) can be chosen. The routine takes into account dummy values.

  In filter mode "mean", "median" the sigma values are used to calculated
  the lower and upper limit values: lower value = sigma * lfac, upper value
  = sigma * ufac.
  In filter mode "quantil" all values outside the lower and upper quantil
  value are rejected.
  In average mode "mean", "median" and "quantil" the mean value, median 
  value and pq-quantil values is calculated.

ARGUMENTS
  const long dim[]        : input dimension dim = { 3, dim_1, dim_2, dim_3 }
  const float *frames (i) : input frames
  const float *dummy  (i) : input dummies, one value for each frame
  const float *ddummy (i) : input ddummies, one value for each frame
  int i_1, int i_2 (i) : pixel indices 
                         (i_1 from 0 to dim_1-1, i_2 from 0 to dim_2-1)
  (AvgMode) int amode   (i) : 1: mean, 2: median, 3:quantil
  (AvgMode) int fmode   (i) : 1: mean, 2: median, 3:quantil
  int vmode           (i) : 0: calculate variance using n
                            1: calculate variance using n-1
  int dovar           (i) : 0: do not return variance
                            1: return variance
  float lfac          (i) : sigma factor for lower value 
  float ufac          (i) : sigma factor for upper value 
  float mins          (i) : minimum sigma
  float loq           (i) : lower quantil (only for fmode==quantil)
  float upq           (i) : upper quantil (only for fmode==quantil)
  float pq            (i) : quantil (only for amode==quantil)

  int *pstatus    (o)   : return status
                           0 : Success
                          -1 : Failed
                          -2 : memory allocation error

RETURN VALUE

  AvgValue ave : average value

---------------------------------------------------------------------------*/
AvgValue average( const long dim[], const float *frames, 
                  const float *dummy, const float *ddummy,
                  long i_1, long i_2, int amode, int fmode,
                  int vmode, int dovar, float lfac, float ufac,
                  float mins, float loq, float upq, float pq,
                  int *pstatus )
{ AvgValue ave;
  double *values=NULL;  // table of valid values (dim_3 elements) 

  const float *src;
  long k, pitch;
  double *dest;
  long nvalues=0, ivalues=0 ;    // number of valid values 
  double lower, upper;
  double dlfac, dufac, dmins, dloq, dupq, dpq;

  int status=-1; // Failed; 
  
  dlfac = (double) lfac;
  dufac = (double) ufac;
  dmins = (double) mins;
  dloq   = (double) loq;
  dupq   = (double) upq;
  dpq   = (double) pq;

  ave.val = 0.0;
  ave.var = VarDummy;
  ave.n = 0;

  // 1) extract pixel values (i_1, i_2) from frames and copy them to values
  if (!( values = (double *) MALLOC( sizeof(double)*dim[3] ) )) {
    status = -2; // memory allocation error
    goto average_error;
  }

  src     = frames+i_1+i_2*dim[1];
  dest    = values;
  pitch   = dim[1]*dim[2];
  nvalues = 0;

  for (k=0;k<dim[3];k++) {
    double value;
    value = (double) *src;

    // UPDATE( *dest, value, dummy[k], ddummy[k] );
    if ( NODUMMY(value, dummy[k], ddummy[k]) ) {
      *dest = value;
      dest++;
      nvalues++;
    }

    src+=pitch; // next frame
  }

  // 2) filter values
  if (fmode) {
    double sigma, val, var;
    double upqval, loqval, hdif;

    // 2a) calculate average and deviation from average
    switch (fmode ) {
      case Mean:      
        val = mean( values, nvalues );
        var = variance ( values, nvalues, val );
        sigma = sqrt(var);
        break;

      case Quantil:
        upqval = dquantil ( values, nvalues, dupq );
        loqval = dquantil ( values, nvalues, dloq );
        hdif = fabs((upqval-loqval)*0.5); // half distance beween quantils
        val = (upqval+loqval)*0.5; // average of upper and lower quantil
        sigma = hdif;
        break;

      case Median:
      default:
        val = dmedian( values, nvalues);
        var = dmediance ( values, nvalues, val);
        sigma = sqrt(var);
    }

    if (sigma<dmins) sigma=dmins;

    // 2b) calculate lower and upper limits
    switch (fmode ) {
      case Quantil:
        lower = val - sigma; // lower quantil value
        upper = val + sigma; // upper quantil value
        break;  
      case Mean:
      case Median:
      default:
        if (lfac<0.0)
          lower = minimum ( values, nvalues ); // accept all below
        else
          lower = val - sigma * dlfac;
    
        if (ufac<0.0)
          upper = maximum ( values, nvalues ); // accept all above
        else
          upper = val + sigma * dufac;
    }

    // 2c) remove table values outside lower and upper limits
    ivalues = nvalues;
    nvalues = minmaxfilter ( values, nvalues, lower, upper );
    ivalues -= nvalues; // ignored values
  }

  // 3) calculate average and variance

  switch (amode) {
    case Median: 
      ave.val = dmedian ( values, nvalues );
      if (dovar) 
        ave.var = dmediance( values, nvalues, ave.val );
      break;
     case Quantil:
        ave.val = dquantil ( values, nvalues, dpq );
        if (dovar)
          ave.var = dquantilance(  values, nvalues, dpq, ave.val );
        break;
    case Mean: 
    default: 
      ave.val = mean ( values, nvalues );
      if (dovar)
        ave.var = variance( values, nvalues, ave.val );
  }
  ave.n   = nvalues;
  ave.i   = ivalues;

  // 4) calculate variance of the average
  
  if (dovar) {
    if (vmode==0) { // calculate with n values
       if (nvalues>0) {
         ave.var /= nvalues;
       } else ave.var = VarDummy; // variance cannot be calculated
     }  else { // calculate with n-1 values
       if (nvalues>1) {
         ave.var /= nvalues-1;
       } else ave.var = VarDummy; // variance cannot be calculated
     }
  }

  FREE(values);

  status = 0; // Success;
  if (pstatus) *pstatus=status;
  return( ave ); 

average_error:

  FREE(values);

  if (pstatus) *pstatus=status;
  return( ave ); 
  
} // average 

