/***************************************************************************/
/* Written 1994++ by Peter Boesecke                                        */
/* Copyright (C) 2011 European Synchrotron Radiation Facility              */
/*                       Grenoble, France                                  */
/*                                                                         */
/*    Principal authors: Peter Boesecke  (boesecke@esrf.eu)                */
/*                                                                         */
/*    This program is free software: you can redistribute it and/or modify */
/*    it under the terms of the GNU General Public License as published by */
/*    the Free Software Foundation, either version 3 of the License, or    */
/*    (at your option) any later version.                                  */
/*                                                                         */
/*    This program is distributed in the hope that it will be useful,      */
/*    but WITHOUT ANY WARRANTY; without even the implied warranty of       */
/*    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the        */
/*    GNU General Public License for more details.                         */
/*                                                                         */
/*    You should have received a copy of the GNU General Public License    */
/*    along with this program.  If not, see <http://www.gnu.org/licenses/>.*/
/***************************************************************************/
/*+++
1 saxs_new
  Generates a new image with pixel values expression(_x1,_x2)

2 PURPOSE
  Generates a new image with pixel values expression(_x1,_x2)
  _x1 and _x2 are the coordinates of the chosen reference system.

  Arguments:
  saxs_new [options] <onam> <odum> <dim 1> <dim 2> <val>"

  Defaults:
  output file name    : <onam>              : output.edf
  output dummy        : <odum>              : 0.0
  dimension 1         : <odim 1>            : 1
  dimension 2         : <odim 2>            : 1
  fill  value         : <val>               : 0.0

  Wildcard:
  A "=" in the argument list chooses the default.

2 HISTORY
  2000-11-24  PB V3.0
  2001-01-10  PB V3.1 GetReference with output image
  2001-07-09  PB V4.00 new loop, no repetition
  2002-06-31  PB V4.50 ave
  2003-02-04  PB V4.51 line args for Factor and Const
  2003-02-09  PB V4.52 line args for Dummy, DDummy, Dim
  2003-05-30  PB V4.53 ImageLoop: Function call changed
  2003-11-30  PB V4.54 UpdateImageParameters: (dummy function)
  2004-04-17  PB V4.55 new options: val for value, m1 for slope1, m2 for slope2
  2004-10-31  PB V4.56 MAKE_FUNCTION
  2005-06-30  PB V4.57 num_str2prog: val defined as line parameter to accept 
                                     variables
                       m1, m2 removed, use -val _x1*m1+_x2*m2
  2005-07-03  PB V4.58 some comments changed
  2005-07-04  PB V4.59 ReportSaxsStatus parameter list updated
  2005-08-11  PB V4.60 num_str2prog: err defined as line parameter to accept
                                     variables
  2007-04-23  PB V4.61 saxs_new: valprog, errprog initialized with NULL
  2007-06-11  PB V4.62 ImageLoop parameters updated
  2012-01-31  PB V4.63 ImageLoop parameters updated
  2015-10-07  PB V4.64 Usage string updated

---*/
# define Version  "saxs_new V4.64 2015-10-07, Peter Boesecke"

# include <errno.h>
# include <stdio.h>
# include <unistd.h>
# include <fcntl.h>

# include "SaxsPrograms.h"
# include "numio.h"

# define Usage "[options] \n\
                <onam> <odum> <odim1> <odim2> <val>\n\
                Generates an image from an expression(_x1,_x2)\n\
                  -oval <expression(_x1,_x2)>\n\
                Generates a variance image from an expression(_x1,_x2,_I)\n\
                  +var -oerr <expression(_x1,_x2,_I)>\n\
                _x1,_x2: coordinates of the chosen reference system\n\
                     _I: image intensity at (_x1,_x2) (in variance calculation)\n\n\
   Ex: saxs_new +var -err \"_I*_I\" -odim 7 11 -val \"(_x1-2.3)*(_x2-7)\" new.edf"

# define BlockNum 1       /* 1 output sequence */

/* Line Options */
# define SVal "val"
# define SErr "err"

/*---------------------------------------------------------------------------
1 saxs_new

2 PURPOSE
  Generates a new image with constant pixel values
---------------------------------------------------------------------------*/
void saxs_new (CmdBlk * pcb, long num[], ImgHeadBlk ihb[], int * pstatus)
{ int i,imax;
  int j;

  float *I0Data, *E0Data;
  int   I0Dim_1,I0Dim_2;
  float *pI0Data, *pE0Data;
  float I0Dummy, I0DDummy;
  float Value, Variance;

  NumProg *valprog=(NumProg *) NULL, *errprog=(NumProg *) NULL;
  int errval;
  int buflen=512;
  char buffer[512];

  int i_1, i_2;
  float f_1[BlockNum], f_2[BlockNum];
 
  float Off_1[BlockNum], Off_2[BlockNum];
  float Ps_1[BlockNum], Ps_2[BlockNum];
 
  float f1_1[BlockNum], f3_1[BlockNum], Df_1[BlockNum];
  float f1_2[BlockNum], f3_2[BlockNum], Df_2[BlockNum];
 
  int Imin_1, Imin_2, Imax_1, Imax_2;
 
  float Wmin_1, Wmax_1, W_1, DW_1;
  float Wmin_2, Wmax_2, W_2, DW_2;

  IO_line Val;
  IO_line Err;

  *pstatus = Success;

  imax = pcb->ImgBlkLen;
  printf("\n Calculating ihb[0,% d] = Function(",ihb[0].ImNum);
  for(i=1;i<imax;i++) {
    printf("ihb[% d,% d] ", i, ihb[i].ImNum); }

  printf(")\n\n");

 /* Check the number of images */
  if (pcb->ImgBlkLen!=1) {
     printf("%d images found, 1 output image required\n",
             pcb->ImgBlkLen); *pstatus=Failed; return; }

 /* Get option values */
  Val = option_line ( SVal, pcb, pcb->ib, num, 1, pstatus );
  if (*pstatus!=Success) return;

  if (ihb[0].VarDat) {
    Err = option_line ( SErr, pcb, pcb->ib, num, 1, pstatus );
    if (*pstatus!=Success) return;
  }

  GetReferenceParameters( pcb, ihb, 0, imax-1,
                          Off_1, Off_2, Ps_1,  Ps_2, pstatus );
  if (*pstatus!=Success) return;

  UpdateImageParameters ( pcb, ihb, 1, imax-1,
                          Off_1, Off_2, Ps_1,  Ps_2, pstatus );
  if (*pstatus!=Success) return;

 /* 1 output */
  I0Data  = ihb[0].Data;
  E0Data  = ihb[0].VarDat;
  I0Dummy = ihb[0].Dummy.V;
  I0DDummy = ihb[0].DDummy.V;
  I0Dim_1  = (int) ihb[0].Dim[1];
  I0Dim_2  = (int) ihb[0].Dim[2];

  GetImageRange         ( pcb, ihb, 1, imax-1,
                          Off_1, Off_2, Ps_1,  Ps_2,
                          f1_1, f3_1, Df_1, f1_2, f3_2, Df_2,
                          &Imin_1, &Imax_1, &Imin_2, &Imax_2,
                          &Wmin_1, &Wmax_1, &DW_1, &Wmin_2, &Wmax_2, &DW_2, 1);

  valprog = num_str2prog ( "Val", Val.V, NULL, &errval, 2, "x1", "x2" );
  if (errval) {
    fprintf(ERROUT,"ERROR: %s\n",num_errval2str( buffer,buflen,errval ));
    num_rmprog ( valprog, &errval );
    num_rmprog ( errprog, &errval );
    *pstatus=Failed; return;
  }

  if (E0Data) {
    errprog = num_str2prog ( "Err", Err.V, NULL, &errval, 3, "x1", "x2", "I" );
    if (errval) {
      fprintf(ERROUT,"ERROR: %s\n",num_errval2str( buffer,buflen,errval ));
      num_rmprog ( errprog, &errval );
      num_rmprog ( valprog, &errval );
      *pstatus=Failed; return;
    }
  }

  /* loop over the output array */
  W_2 = Wmin_2; for (i=0;i<imax;i++) f_2[i]=f1_2[i];
  for (i_2=Imin_2;i_2<=Imax_2;i_2++) {
 
    pI0Data = ABSPTR(I0Data,I0Dim_1,I0Dim_2,Imin_1,i_2);
    pE0Data = E0Data-I0Data+pI0Data;
    W_1 = Wmin_1; for (i=0;i<imax;i++) f_1[i]=f1_1[i];
    for (i_1=Imin_1;i_1<=Imax_1;i_1++) {
      /* do something with the data */
 
      Value = num_runprog ( valprog, &errval, W_1, W_2 );
      if (!errval) {
        UPDATE( *pI0Data, Value, I0Dummy, I0DDummy );
      }

      if (E0Data) {
        Variance = num_runprog ( errprog, &errval, W_1, W_2, Value );
        if (!errval) {
          UPDATE( *pE0Data, Variance, -1.0, 0.1 );
        }
      }
 
      pI0Data++;
      pE0Data++;
      W_1+=DW_1; for (i=0;i<imax;i++) { f_1[i]+=Df_1[i]; }
    } /* for i_1 ... */
 
    W_2+=DW_2; for (i=0;i<imax;i++) { f_2[i]+=Df_2[i]; }
  } /* for i_2 ... */

  num_rmprog ( errprog, &errval );
  num_rmprog ( valprog, &errval );

} /* saxs_new */

/*---------------------------------------------------------------------------
user_io
Do all the keyboard io and return cb, and ib
---------------------------------------------------------------------------*/

void user_io(CmdBlk * pcb, ImgBlk * ib, int * pstatus)
{
  char  progfn[InputLineLength];
  ImgHeadBlk ihb[BlockNum];

  float ROff_1, RPs_1, UOff_1, UPs_1;
  float ROff_2, RPs_2, UOff_2, UPs_2;

  IO_line *pVal, *pErr;

 /* Determine program name without directory */
   (void) RightFR((char *) pcb->argv[0],DirSeparator,progfn,InputLineLength);

  /* show usage if no arguments are given */
  if (pcb->argc<=1) printf("Usage: %s %s\n",progfn,Usage);

  /*--- Write name of program ---*/
  printf("\n");
  printf("   %s %s\n",progfn,Version);
  printf("\n");

  ArgvFilenames ( pcb, ib, ihb, 0, BlockNum-1, pstatus);
  if (*pstatus!=Success) return;
  GetReference(pcb->RSys.V,0,ihb,&ROff_1,&ROff_2,&RPs_1,&RPs_2,pstatus );
  if (*pstatus!=Success) return;
  GetReference(pcb->USys.V,0,ihb,&UOff_1,&UOff_2,&UPs_1,&UPs_2,pstatus );
  if (*pstatus!=Success) return;

  /*--- Argument  : value val */
  pVal = (IO_line*) option_parameter_search( SVal, pcb, pstatus );
  if (*pstatus!=Success) return;
  argv_line(pcb,"Image value (expression(_x1,_x2))", pVal,pVal->V,pstatus);
  if (*pstatus!=Success) return;

  /*--- Argument  : value err */
  pErr = (IO_line*) option_parameter_search( SErr, pcb, pstatus );
  if (*pstatus!=Success) return;
  argv_line(pcb,"Variance value (expression(_x1,_x2,_I))", 
    pErr,pErr->V,pstatus);
  if (*pstatus!=Success) return;

  printf("\n");
  if (ib[0].Name.I)    printf("o/p file            : %s\n",ib[0].Name.V);
  if (ib[0].Dummy.I)   printf("output dummy        : %s\n",ib[0].Dummy.V);
  if (ib[0].Dim[1].I)  printf("output dimension 1  : %s\n",ib[0].Dim[1].V);
  if (ib[0].Dim[2].I)  printf("output dimension 2  : %s\n",ib[0].Dim[2].V);
  if (pVal->I)         printf("image value         : %s\n",pVal->V);
  if (pErr->I)         printf("variance value      : %s\n",pErr->V);

  printf("\n");

  if (pcb->TestBit) {
    PrintBlocks ( pcb, ib );
    printf("\n"); }

  return;
}

/*---------------------------------------------------------------------------
main
---------------------------------------------------------------------------*/

#if MAKE_FUNCTION
# define MAIN main_saxs_new
#else
# define MAIN main
#endif

int MAIN (int argc, char *argv[])
{
  CmdBlk cb;                /* command block  */
  ImgBlk ib[BlockNum];      /* image blocks */

  int status;
  int arg_no = 0;

 /* Init options, control block and image blocks */
  InitOptions( Usage, Version, NULL, TRUE, &cb, ib, BlockNum );

 /* Additional options */
  option_define ( IO_tpline, SVal, "0.0", &cb ); // value
  option_define ( IO_tpline, SErr, "0.0", &cb ); // value

 /* Read options from argument list */
  ReadOptions( argv, &arg_no, &cb, ib, &status);

  /* USER KEYBOARD I/O */
  if (status==Success) {
    argv_start ( &cb, 1 );
    user_io( &cb, ib, &status);
    argv_end( &cb ); /* must be called after user_io */
  }

  /* SEQUENCE CALCULATION */
  if (status==Success) 
    IMAGELOOP( &cb, ib, saxs_new, NULL, NULL, TRUE, &status );

  return( ReportSaxsStatus( status, 0 ) );

} /* MAIN */

