/***************************************************************************/
/* Written 1994++ by Peter Boesecke                                        */
/* Copyright (C) 2011 European Synchrotron Radiation Facility              */
/*                       Grenoble, France                                  */
/*                                                                         */
/*    Principal authors: Peter Boesecke  (boesecke@esrf.eu)                */
/*                                                                         */
/*    This program is free software: you can redistribute it and/or modify */
/*    it under the terms of the GNU General Public License as published by */
/*    the Free Software Foundation, either version 3 of the License, or    */
/*    (at your option) any later version.                                  */
/*                                                                         */
/*    This program is distributed in the hope that it will be useful,      */
/*    but WITHOUT ANY WARRANTY; without even the implied warranty of       */
/*    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the        */
/*    GNU General Public License for more details.                         */
/*                                                                         */
/*    You should have received a copy of the GNU General Public License    */
/*    along with this program.  If not, see <http://www.gnu.org/licenses/>.*/
/***************************************************************************/
/*+++
1 rapid2saxs
  Copies rapid data to edf data files including scaler data

2 PURPOSE
  Copies rapid data to edf data files including scaler data

  Calculation:       I0 = I1
  Error propagation: V0 = V1

  Input file 1 is copied to the output file. A second and third
  input file is open.

  -i1nam <rapid header file> : frame data
    -i1mem 1
    -i1fst <first frame>
    -i1inc <last frame> 
    -i1lst <number of frames> 
    -i2nam <i1nam> : scaler data
      -i2mem 2
      -i2fst 1
      -i2inc 0
    -i3nam <i1nam> : timing data
      -i3mem 3
      -i3fst 1
      -i3inc 0

  Arguments:
  rapid2saxs [options] <i1nam> <onam> <i2mem> <i3mem> <bibo>"

  Defaults:
  <input file name 1> : input.edf 
  <output file name>  : output.edf
  <memory 2>          : memory of scaler data
  <memory 3>          : memory of timing data
  <bibo>              : bsl input byte order

  Wildcard:
  A "=" in the argument list chooses the default.

2 HISTORY
  2003-04-06 PB V4.50 based on saxs_mac
  2003-04-13 PB V4.51 loop, HighByteFirst is default
  2003-04-13 PB V4.52 deltatime, exposuretime
  2003-04-15 PB V4.53 defaults i2inc=0, i3inc=0 
  2003-05-30 PB V4.54 ImageLoop: new function call 
  2003-07-07 PB V4.55 if ihb[2] is not available, header values
                      of scalers are read and incremented
                      if ihb[3] is not available, header value
                      of exposre time is read and incremented
                      header values of intensity0 and intensit1 
                      are read and incremented
                      if ihb[1] is not available, nothing is done
  2003-11-30 PB V4.56 BinSiz
  2004-10-30 PB V4.57 MAKE_FUNCTION
  2005-07-15 PB V4.58 ReportSaxsStatus parameter updated
  2005-09-17 PB V4.59 VarDat
  2007-04-23 PB V4.60 compatibility to lcc
  2007-06-11 PB V4.61 ImageLoop parameters updated
  2012-01-31 PB V4.62 ImageLoop parameters updated

---*/
# define Version  "rapid2saxs V4.62 2012-01-31, Peter Boesecke"

# include <errno.h>
# include <stdio.h>
# include <unistd.h>
# include <fcntl.h>

# include "SaxsPrograms.h"

# define Usage "[options] <i1nam> <onam> <odum> <i2mem> <i3mem> <bibo>"

# define BlockNum 4       /* 3 input sequences + 1 output sequence */

static const char * _BOrderStrings[4] = { "UnSpecified",
                                          "LowByteFirst", "HighByteFirst",
                                          (const char *) NULL };


/*---------------------------------------------------------------------------
NAME

  deltatime --- calculates delta time from timing data and returns deltatime
 
DESCRIPTION
  Calculates delta time for a given frame using the information contained in
  timing data. Delta time in seconds is returned. Delta time is the time
  between the start of the cycle and the start of a live frame.
  Delta time is calculated from the array float timing [4][framemax] 
  in the following way:
    deltatime( frameno ) = 
      timing[0][0] + Sum(i=1,i<=frameno-1:timing[i-1][1]+timing[i][0])

PARAMETERS
  float timing[dim[1],dim[2]] : timing array 
                                first array index i_1:
                                0: dead time (s)
                                1: live time (s)
                                2: accumulated dead time (s)
                                3: accumulated live time (s) (ExposureTime)
                                second array index i_2:
                                frameno
  long dim[3]
  long frameno                : frameno: 1, 2, ... dim[2]
  
RETURN VALUE
  Delta time since start of the cycle in seconds. Negative values in case
  of an error.

---------------------------------------------------------------------------*/
float deltatime( float * timing, int dim_1, int dim_2, long frameno )
{ float sum;
  int   i_1;
  int   Dim_1,Dim_2;
  float *pData;

  frameno=MIN2(dim_1,frameno);
  if (frameno<1) return(-1.0);

  if (!timing) return(-1.0);

  pData = ABSPTR(timing,dim_1,dim_2,0,0); // dead time
  sum = *pData;

  pData = ABSPTR(timing,dim_1,dim_2,0,1); // live time
  for (i_1=1;i_1<frameno;i_1++) {
    sum  += *pData; // live time i_1-1
    pData = ABSPTR(timing,dim_1,dim_2,i_1,0); // dead time
    sum  += *pData; // dead time i_1
    pData = ABSPTR(timing,dim_1,dim_2,i_1,1); // live time
  }

  return ( sum );
 
} // deltatime

/*---------------------------------------------------------------------------
NAME
 
  exposuretime --- calculates exposure time from timing data
 
DESCRIPTION
  Calculates exposure time for a given frame using the information contained in
  timing data. The exposure time in seconds is returned. 
  Exposure time is calculated from the array float timing [4][framemax]
  in the following way:
    exposuetime( frameno ) = timing[frameno-1][3]
 
PARAMETERS
  float timing[dim[1],dim[2]] : timing array
                                first array index i_1:
                                0: dead time (s)
                                1: live time (s)
                                2: accumulated dead time (s)
                                3: accumulated live time (s) (ExposureTime)
                                second array index i_2:
                                frameno
  int dim_1, dim_2
  long frameno                : frameno: 1, 2, ... dim[2]
 
RETURN VALUE
  Exposure time. Negative value in case of an error.
 
---------------------------------------------------------------------------*/
float exposuretime( float * timing, int dim_1, int dim_2, long frameno )
{ float exptim;
  int   i_1;
  int   Dim_1,Dim_2;
  float *pData;
 
  frameno=MIN2(dim_1,frameno);
  if (frameno<1) return(-1.0);
 
  if (!timing) return(-1.0);

  pData = ABSPTR(timing,dim_1,dim_2,frameno-1,3); // accumulated live time

  exptim = *pData;
 
  return ( exptim );
 
} // exposuretime

/*---------------------------------------------------------------------------
NAME
 
  starttime --- reads start time from title string and returns it as isotime 
 
DESCRIPTION
  The input string is of the form 

     "SRS Data recorded on Sun Mar 23 15:43:27 2003"

  The portion "Mar 23 15:43:27 2003" is extracted from the string and converted
  to an isotime string: "2003-03-23 15:43:27"
 
PARAMETERS
  char buffer[buflen] : output buffer
  size_t buflen       : size of output buffer
  const char * string : input string 
 
RETURN VALUE
  Start time, NULL in case of an error.
 
---------------------------------------------------------------------------*/
char * starttime(char *buffer, size_t buflen, const char * string)
{ 
  return( (char *) NULL );

//  return( buffer ); // not implemented

} // starttime

/*---------------------------------------------------------------------------
NAME

  rapid2saxs --- conversion of BSL rapid files to edf saxs files

DESCRIPTION

  memory 1, image 1 to N : image data
  memory 2, image 1 : scaler data
  memory 3, image 1 : timing data
  
---------------------------------------------------------------------------*/
void rapid2saxs (CmdBlk * pcb, long num[], ImgHeadBlk ihb[], int * pstatus)
{ const float eps=1e-32;
  int i,imax;
  int j;

  float *I0Data, *E0Data;
  int   I0Dim_1, I0Dim_2;
  float *pI0Data, *pE0Data; 
  float I0Dummy, I0DDummy;
  float VarDDummy=DDSET(VarDummy);

  int   I1Ave;
  float *I1Data, *E1Data;
  int   I1Dim_1,I1Dim_2;
  float I1Dummy, I1DDummy;
  float I1Value, I1Sum, I1Weight;
  float E1Value, E1Sum, E1Weight;

  float *I2Data;
  int   I2Dim_1,I2Dim_2;
  float *pI2Data;
  float I2Dummy, I2DDummy;
  float I2Value;

  float *I3Data;
  int   I3Dim_1,I3Dim_2;
  float *pI3Data;
  float I3Dummy, I3DDummy;
  float I3Value;

  int i_1, i_2;
  float f_1[BlockNum], f_2[BlockNum];

  float Off_1[BlockNum], Off_2[BlockNum];
  float Ps_1[BlockNum], Ps_2[BlockNum];

  float f1_1[BlockNum], f3_1[BlockNum], Df_1[BlockNum];
  float f1_2[BlockNum], f3_2[BlockNum], Df_2[BlockNum]; 

  int Imin_1, Imin_2, Imax_1, Imax_2;
 
  float Wmin_1, Wmax_1, DW_1;
  float Wmin_2, Wmax_2, DW_2;

  long  frameno=1;
  long  scalerno;
  char  scalerkey[IO_len];
  float value, scalervalue;
  int   scalerok;
  char  timinginfo[IO_len];
  char  buffer[IO_len];
  float hmdeltatime, hmexposuretime;
  float intensity0, intensity1;
  float tmp;

  *pstatus = Success;

  printf("\n Calculating ihb[0,% d] = Function(",ihb[0].ImNum);
  for(i=1;i<pcb->ImgBlkLen;i++) {
    if (ihb[i].I) printf("ihb[% d,% d] ", i, ihb[i].ImNum); }
 
  printf(")\n\n");

  /* copy image data */
  if (ihb[1].I) {
    imax = 2; // only input block 1 and output block used for copy 

    GetReferenceParameters( pcb, ihb, 0, imax-1,
                          Off_1, Off_2, Ps_1,  Ps_2, pstatus );
    if (*pstatus!=Success) return;

    UpdateImageParameters ( pcb, ihb, 1, imax-1,
                            Off_1, Off_2, Ps_1,  Ps_2, pstatus );
    if (*pstatus!=Success) return;

    GetImageRange         ( pcb, ihb, 1, imax-1,
                            Off_1, Off_2, Ps_1,  Ps_2,
                            f1_1, f3_1, Df_1, f1_2, f3_2, Df_2,
                            &Imin_1, &Imax_1, &Imin_2, &Imax_2,
                            &Wmin_1, &Wmax_1, &DW_1, &Wmin_2, &Wmax_2, &DW_2, 1);

   /* 1 input, 1 output */
    I0Data  = ihb[0].Data;
    E0Data = ihb[0].VarDat;
    I0Dummy = ihb[0].Dummy.V;
    I0DDummy = ihb[0].DDummy.V;
    I0Dim_1  = (int) ihb[0].Dim[1];
    I0Dim_2  = (int) ihb[0].Dim[2];

    I1Ave   = pcb->ib[1].Ave.V;
    I1Data  = ihb[1].Data;
    E1Data  = ihb[1].VarDat;
    I1Dummy = ihb[1].Dummy.V;
    I1DDummy = ihb[1].DDummy.V;
    I1Dim_1  = (int) ihb[1].Dim[1];
    I1Dim_2  = (int) ihb[1].Dim[2];
    frameno  = ihb[1].ImNum;

    /* loop over the output array  */
    for (i=0;i<imax;i++) f_2[i]=f1_2[i];
    for (i_2=Imin_2;i_2<=Imax_2;i_2++) {

      pI0Data = ABSPTR(I0Data,I0Dim_1,I0Dim_2,Imin_1,i_2);
      pE0Data = E0Data-I0Data+pI0Data;
      for (i=0;i<imax;i++) f_1[i]=f1_1[i];
      for (i_1=Imin_1;i_1<=Imax_1;i_1++) {

        if ( E0Data ) {
          // V0 = V1
          if ( Isum2ldwE(I1Data,E1Data,I1Dim_1,I1Dim_2,I1Dummy,I1DDummy,
                 f_1[1], f_2[1], f_1[1]+Df_1[1], f_2[1]+Df_2[1],
                 &I1Sum, &I1Weight, &E1Sum, &E1Weight) ) {
            I1Value = I1Sum; if (I1Ave) I1Value /= I1Weight;
            UPDATE( *pI0Data, I1Value, I0Dummy, I0DDummy ); 
            if (E1Sum >= 0.0) {
              E1Value = E1Sum; if (I1Ave) E1Value /= E1Weight*E1Weight;
              UPDATE( *pE0Data, E1Value, VarDummy, VarDDummy );
            }
          }
        } else {
          if (Isum2ldw (I1Data,I1Dim_1,I1Dim_2,I1Dummy,I1DDummy,
              f_1[1],f_2[1],f_1[1]+Df_1[1],f_2[1]+Df_2[1],&I1Sum,&I1Weight)) {
            I1Value = I1Sum; if (I1Ave) I1Value /= I1Weight;
            UPDATE( *pI0Data, I1Value, I0Dummy, I0DDummy );
          }
        }
        pI0Data++;
        pE0Data++;
        for (i=0;i<imax;i++) f_1[i]+=Df_1[i];
    
      } /* for i_1 ... */
  
      for (i=0;i<imax;i++) f_2[i]+=Df_2[i];
    } /* for i_2 ... */

    if (ImageIsNew( 0, ihb )) {
      // HMFrame 
      WriteHeaderLong(pcb,0,ihb,KHFrame,frameno, pstatus);
      if (*pstatus!=Success) return;

      // HMFileName
      WriteHeaderLine(pcb,0,ihb,KHFile,pcb->ib[1].FileName, pstatus);
      if (*pstatus!=Success) return;

      // HMStartTime
      if (starttime(buffer, IO_len, ihb[1].Title.V)) {
        CopyLine(buffer,ihb[0].Time.V,IO_len,0);
        ihb[0].Time.I = TRUE;
      }
    } // ImageIsNew
  } else return; // no further action, if image does not exist
  
  /* write scalers into header */
  if (ihb[2].I) {

    I2Data   = ihb[2].Data;
    I2Dummy  = ihb[2].Dummy.V;
    I2DDummy = ihb[2].DDummy.V;
    I2Dim_1  = (int) ihb[2].Dim[1];
    I2Dim_2  = (int) ihb[2].Dim[2];

    for (i_2=0;i_2<I2Dim_2;i_2++) { // read scalers from ihb[2]
      scalerno = i_2+1;
      i_1 = frameno-1;
      if ((i_1>=0)&&(i_1<I2Dim_1)) {
        sprintf(scalerkey,"%s%s%02u",KHScaler,KHC,scalerno);

        // update already existing scaler value
        I2Value = I2Dummy;
        if (!ImageIsNew( 0, ihb )) 
          if (ReadHeaderFloat ( pcb, 0, ihb, scalerkey, &I2Value, pstatus ))
            if (*pstatus!=Success) return;

        pI2Data = ABSPTR(I2Data,I2Dim_1,I2Dim_2,frameno-1,i_2);

        UPDATE( I2Value, *pI2Data, I2Dummy, I2DDummy );

        WriteHeaderFloat(pcb,0,ihb,scalerkey,I2Value, pstatus);
        if (*pstatus!=Success) return;
      }
    }
  } else { // read scaler values from header until a value is missing
    scalerok=1;
    for (scalerno=1;scalerok;scalerno++) {

      sprintf(scalerkey,"%s%s%02u",KHScaler,KHC,scalerno);

      // read scaler value
      if (ReadHeaderFloat ( pcb, 1, ihb, scalerkey, &scalervalue, pstatus)) {
        if (*pstatus!=Success) return;

        // update output scaler value
        value = 0.0;
        if (!ImageIsNew( 0, ihb ))
          if (ReadHeaderFloat ( pcb, 0, ihb, scalerkey, &value, pstatus ))
            if (*pstatus!=Success) return;

        value += scalervalue;

        WriteHeaderFloat(pcb,0,ihb,scalerkey,value, pstatus);
        if (*pstatus!=Success) return;
      } else scalerok = FALSE;
    }
  }

  /* write timing into header */
  /* only DeltaTime possible */
  if (ihb[3].I) {
    I3Data   = ihb[3].Data;
    I3Dummy  = ihb[3].Dummy.V;
    I3DDummy = ihb[3].DDummy.V;
    I3Dim_1  = (int) ihb[3].Dim[1];
    I3Dim_2  = (int) ihb[3].Dim[2];
 
    sprintf(timinginfo,"");
    for (i_2=0;i_2<I3Dim_2;i_2++) {
      i_1 = frameno-1;
      if ((i_1>=0)&&(i_1<I3Dim_1)) {
 
        pI3Data = ABSPTR(I3Data,I3Dim_1,I3Dim_2,frameno-1,i_2);
        if (strlen(timinginfo)<IO_len-30)
          sprintf(timinginfo,"%s%15.7g ",timinginfo,*pI3Data);
      }
    }
    WriteHeaderLine(pcb,0,ihb,"TimingInfo",timinginfo, pstatus);
    if (*pstatus!=Success) return;

    // HMDeltaTime
    if (ImageIsNew( 0, ihb )) {
      hmdeltatime=deltatime( I3Data, I3Dim_1, I3Dim_2, frameno );
      if (hmdeltatime>=0.0) {
        WriteHeaderFloat(pcb,0,ihb,KHDeltaTime,hmdeltatime, pstatus);
        if (*pstatus!=Success) return;
      }
    }

    // ExposureTime
    tmp=exposuretime( I3Data, I3Dim_1, I3Dim_2, frameno );
    if (tmp>=0.0) {
      hmexposuretime=0.0;
      if (!ImageIsNew( 0, ihb ))
        if (ReadHeaderFloat ( pcb, 0, ihb,
                              KValueTime, &hmexposuretime, pstatus ))
          if (*pstatus!=Success) return;
      hmexposuretime+=tmp;
      WriteHeaderFloat(pcb,0,ihb,KValueTime,hmexposuretime, pstatus);
      if (*pstatus!=Success) return;
    }

  } else { // read ExposureTime from header 
    // ExposureTime
    if (ReadHeaderFloat ( pcb, 1, ihb,
                          KValueTime, &tmp, pstatus )) {
      if (*pstatus!=Success) return;

      if (tmp>=0.0) {
        hmexposuretime=0.0;
        if (!ImageIsNew( 0, ihb ))
          if (ReadHeaderFloat ( pcb, 0, ihb,
                                KValueTime, &hmexposuretime, pstatus ))
            if (*pstatus!=Success) return;
        hmexposuretime+=tmp;
        WriteHeaderFloat(pcb,0,ihb,KValueTime,hmexposuretime, pstatus);
        if (*pstatus!=Success) return;
      }
    }
  }

  // Intensity0
  if (ReadHeaderFloat ( pcb, 1, ihb,
                        KValueI0, &tmp, pstatus )) {
    if (*pstatus!=Success) return;

    if (tmp>=0.0) {
      intensity0=0.0;
      if (!ImageIsNew( 0, ihb ))
        if (ReadHeaderFloat ( pcb, 0, ihb,
                              KValueI0, &intensity0, pstatus ))
          if (*pstatus!=Success) return;
      intensity0+=tmp;
      WriteHeaderFloat(pcb,0,ihb,KValueI0,intensity0, pstatus);
      if (*pstatus!=Success) return;
    }
  }

  // Intensity1
  if (ReadHeaderFloat ( pcb, 1, ihb,
                        KValueI1, &tmp, pstatus )) {
    if (*pstatus!=Success) return;

    if (tmp>=0.0) {
      intensity1=0.0;
      if (!ImageIsNew( 0, ihb ))
        if (ReadHeaderFloat ( pcb, 0, ihb,
                              KValueI1, &intensity1, pstatus ))
          if (*pstatus!=Success) return;
      intensity1+=tmp;
      WriteHeaderFloat(pcb,0,ihb,KValueI1,intensity1, pstatus);
      if (*pstatus!=Success) return;
    }
  }

  *pstatus = Success;

} /* rapid2saxs*/

/*---------------------------------------------------------------------------
NAME
 
  _ByteOrder2String --- converts byte order to a string
 
SYNOPSIS
 
  _BOrder byte_order;
  const char * _ByteOrder2String( int byte_order );
 
DESCRIPTION
 
GCC AND G++
  For compatibility between gcc and g++ the declaration of the variable
  with its enumerated type has been replaced by "int".
 
RETURN VALUE
  Pointer to a constant result string.
 
AUTHOR
03-Mar-1998 PB Specification
  -------------------------------------------------------------------------*/
const char * _ByteOrder2String( int byte_order )
{
   if ((byte_order<0)||(byte_order>=3)) byte_order = 0;
   return( _BOrderStrings[byte_order] );
 
} /* _ByteOrder2String */

/*---------------------------------------------------------------------------
NAME
 
  outputname_default --- get default output name 
 
SYNOPSIS
 
  char * outputname_default( char * buffer, size_t buflen, 
                             const char * pattern )
 
DESCRIPTION
 
RETURN VALUE
  Pointer to the output string.
 
AUTHOR
03-Mar-1998 PB Specification
  -------------------------------------------------------------------------*/
char * outputname_default( char *buffer, size_t buflen, const char *pattern )
{ char name[IO_len];

  // copy filename to name 
  filename_name ( name, IO_len, pattern );

  // default only for bsl-type header names
  if ( (strlen(name)==10) && (buflen>=7) &&
    (name[3]=='0') && (name[4]=='0') && (name[5]=='0') && (name[6]=='.') ) {
    buffer[0]=tolower(name[0]); // a
    buffer[1]=tolower(name[1]); // 0
    buffer[2]=tolower(name[2]); // 0
    buffer[3]=name[6]; // .
    buffer[4]=tolower(name[7]); // b
    buffer[5]=tolower(name[8]); // s
    buffer[6]=tolower(name[9]); // l
    buffer[7]='\0';
  } else {
    CopyLine( DefaultOutput, buffer, buflen, 0);
  }

  return( buffer );

} // outputname_default

/*---------------------------------------------------------------------------
user_io
Do all the keyboard io and return cb, and ib
---------------------------------------------------------------------------*/
void user_io(CmdBlk * pcb, ImgBlk * ib, int * pstatus)
{
  char  progfn[InputLineLength];
  ImgHeadBlk ihb[BlockNum];

  float ROff_1, RPs_1, UOff_1, UPs_1;
  float ROff_2, RPs_2, UOff_2, UPs_2;

  char outputname[IO_len];

 /* Determine program name without directory */
   (void) RightFR((char *) pcb->argv[0],DirSeparator,progfn,InputLineLength);

  /* show usage if no arguments are given */
  if (pcb->argc<=1) printf("Usage: %s %s\n",progfn,Usage);

  /*--- Write name of program ---*/
  printf("\n");
  printf("   %s %s\n",progfn,Version);
  printf("   Use option \"-i2nam (-i3nam) /dev/null\" to ignore scaler (timing) memory\n");
  printf("\n");

  /*--- Argument  : input file name */
  argv_filename( pcb, "Input file name", ib, 1, "A00000.BSL", pstatus);
  if (*pstatus!=Success) return;

  /*--- 2nd input file name (scalers) */
  if (!ib[2].Name.I) {
    CopyLine(ib[1].Name.V,ib[2].Name.V,IO_len,0);
    ib[2].Name.I = TRUE;
    if (!ib[2].First.I) { ib[2].First.V = 1; ib[2].First.I = TRUE; }
    if (!ib[2].Inc.I) { ib[2].Inc.V = 0; ib[2].Inc.I = TRUE; }
    extract_filenumbers( &ib[2], &(ib[2].Name), TRUE, pstatus );
    if (*pstatus!=Success) return;
  }

  /*--- 3rd input file name (timing) */
  if (!ib[3].Name.I) {
    CopyLine(ib[1].Name.V,ib[3].Name.V,IO_len,0);
    ib[3].Name.I = TRUE;
    if (!ib[3].First.I) { ib[3].First.V = 1; ib[3].First.I = TRUE; }
    if (!ib[3].Inc.I) { ib[3].Inc.V = 0; ib[3].Inc.I = TRUE; }
    extract_filenumbers( &ib[3], &(ib[3].Name), TRUE, pstatus );
    if (*pstatus!=Success) return;
  }

  /*--- Argument  : output file name */
  argv_filename(pcb,"Output file name",ib,0,
    outputname_default( outputname, IO_len, ib[1].Name.V ), pstatus);
  if (*pstatus!=Success) return; 

  /*--- OpenMode of input file */
  if (!ib[1].OpenMode.I) {
    ib[1].OpenMode.V = IO_Old | IO_FileProtect;
    ib[1].OpenMode.I = TRUE;
    } 

  /*--- OpenMode of 2nd input file */
  if (!ib[2].OpenMode.I) {
    ib[2].OpenMode.V = IO_Old | IO_FileProtect;
    ib[2].OpenMode.I = TRUE;
    }

  /*--- OpenMode of 3rd input file */
  if (!ib[3].OpenMode.I) {
    ib[3].OpenMode.V = IO_Old | IO_FileProtect;
    ib[3].OpenMode.I = TRUE;
    }

  /*--- OpenMode of output file */
  if (!ib[0].OpenMode.I) {
    ib[0].OpenMode.V = IO_ImageProtect;
    ib[0].OpenMode.I = TRUE;
    }

  /*--- Argument : memory of scaler data */
  argv_long( pcb,"Memory of scaler data (0 for none)", &ib[2].Memory, ib[2].Memory.V, pstatus);
  if (*pstatus!=Success) return;

  /*--- Argument : memory of timing data */
  argv_long( pcb,"Memory of timing data (0 for none)", &ib[3].Memory, ib[3].Memory.V, pstatus);
  if (*pstatus!=Success) return;

  /*--- Use dummy file for memory 0 */
  if (ib[2].Memory.V==0) sprintf(ib[2].Name.V,"/dev/null");
  if (ib[3].Memory.V==0) sprintf(ib[3].Name.V,"/dev/null");

  /*--- Argument : bsl input byte order */
  argv_long( pcb,"BSL input byte order (1: LowByteFirst, 2: HighByteFirst )", 
    &(pcb->BslInputByteOrder), pcb->BslInputByteOrder.V, pstatus);
  if (*pstatus!=Success) return;

//  GetReference(pcb->RSys.V,1,ihb,&ROff_1,&ROff_2,&RPs_1,&RPs_2,pstatus );
//  if (*pstatus!=Success) return;
//  GetReference(pcb->USys.V,1,ihb,&UOff_1,&UOff_2,&UPs_1,&UPs_2,pstatus );
//  if (*pstatus!=Success) return;

  printf("\n");
  if (ib[1].Name.I)    printf("i/p file             : %s\n",ib[1].Name.V);
  if (ib[0].Name.I)    printf("o/p file             : %s\n",ib[0].Name.V);
  if (ib[1].First.I)   printf("first image          : %d\n",ib[1].First.V);
  if (ib[1].Last.I)    printf("last image           : %d\n",ib[1].Last.V);
  if (ib[1].Inc.I)     printf("increment            : %d\n",ib[1].Inc.V);
  if (ib[0].Dummy.I)   printf("output dummy         : %s\n",ib[0].Dummy.V);
  if (ib[0].Dim[1].I)  printf("output dimension 1   : %s\n",ib[0].Dim[1].V);
  if (ib[0].Dim[2].I)  printf("output dimension 2   : %s\n",ib[0].Dim[2].V);
  if (ib[2].Memory.I)  printf("scaler memory        : %d\n",ib[2].Memory.V);
  if (ib[3].Memory.I)  printf("timing memory        : %d\n",ib[3].Memory.V);
  if (pcb->BslInputByteOrder.I) 
    printf("bsl input byte order : %s\n",
      _ByteOrder2String(pcb->BslInputByteOrder.V));
  printf("\n");

  if (pcb->TestBit) {
    PrintBlocks ( pcb, ib );
    printf("\n"); }

  return;
}

/*---------------------------------------------------------------------------
main
---------------------------------------------------------------------------*/

#if MAKE_FUNCTION
# define MAIN main_rapid2saxs
#else
# define MAIN main
#endif

int MAIN (int argc, char *argv[])
{
  CmdBlk cb;                /* command block  */
  ImgBlk ib[BlockNum];      /* image blocks */

  int status;
  int arg_no = 0;

 /* Init options, control block and image blocks */
  InitOptions( Usage, Version, NULL, TRUE, &cb, ib, BlockNum );

 /* defaults */
  ib[2].Memory.V = 2l;
  ib[3].Memory.V = 3l;
  cb.BslInputByteOrder.V = HighByteFirst;

 /* Read options from argument list */
  ReadOptions( argv, &arg_no, &cb, ib, &status);

  /* USER KEYBOARD I/O */
  if (status==Success) {
    argv_start ( &cb, 1 );
    user_io( &cb, ib, &status);
    argv_end( &cb ); /* must be called after user_io */
  }

  /* SEQUENCE CALCULATION */
  if (status==Success)
    IMAGELOOP( &cb, ib, rapid2saxs, NULL, NULL, FALSE, &status );

  return( ReportSaxsStatus( status, 0 ) );

} /* MAIN */
