/***************************************************************************/
/* Written 1994++ by Peter Boesecke                                        */
/* Copyright (C) 2011 European Synchrotron Radiation Facility              */
/*                       Grenoble, France                                  */
/*                                                                         */
/*    Principal authors: Peter Boesecke  (boesecke@esrf.eu)                */
/*                                                                         */
/*    This program is free software: you can redistribute it and/or modify */
/*    it under the terms of the GNU General Public License as published by */
/*    the Free Software Foundation, either version 3 of the License, or    */
/*    (at your option) any later version.                                  */
/*                                                                         */
/*    This program is distributed in the hope that it will be useful,      */
/*    but WITHOUT ANY WARRANTY; without even the implied warranty of       */
/*    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the        */
/*    GNU General Public License for more details.                         */
/*                                                                         */
/*    You should have received a copy of the GNU General Public License    */
/*    along with this program.  If not, see <http://www.gnu.org/licenses/>.*/
/***************************************************************************/
# define Version  "edfdump 1.15 2015-02-23, Peter Boesecke"
/*---------------------------------------------------------------------------
NAME
  edfdump - print parameters of an edf file

SYNOPSIS
  edfdump [-h] [-e <keyword>] <files...>

DESCRIPTION
  Edfdump opens the named input <files> as edf files, prints the data 
  block numbers and searches in each header for <keywords>. With the option
  -i only edf files are checked. 
  In case of success the return status is 0, otherwise different from 0.
  return status >0 : edfio ErrorValue
  return status  0 : Success
  return status -1 : general error
  return status -2 : back from help

  -h, --help    print help
  -b <block>    block key, e.g. 1 (default: all blocks)
  -c <chain>    chain key, e.g. Image.Psd (default: all chains)
  -e <keyword>  input of a header keyword to look for, by repeting this
                option several keywords can be defined.
  -s            print filename, chain key, block key and keywords in 
                a single line separated by <tab>, if no keywords are 
                given print file name, chain key and all block keys 
                in a single line
  -S            print keywords in a single line separated by <tab>,
                omit filename, chain key, block key
  -i            ignore non-edf files
  -I            show names of edf files only 
  -f            print file format
  -d <debug>    debug

HISTORY
  V1.0 2001-05-19 Peter Boesecke
  V1.2 2001-05-21 PB multiple files, non edf-files are skipped
  V1.4 2001-05-31 PB
  V1.5 2001-06-02 PB
  V1.6 2001-06-13 PB -I -> \r\n
  V1.7 2004-10-30 PB MAKE_FUNCTION, exit -> return
  V1.8 2005-02-12 PB stderr 
  V1.9 2005-10-27 PB return_status is the last edfio ErrorValue for status!=0
                     of a multiple selection of files. All selected files 
                     are tested.
  V1.10 2006-08-09 PB pipe option -p (default not set)
  V1.11 2007-04-23 PB <string.h>, <ctype.h> included
  V1.12 2007-06-15 PB option -f (FileFormat) added
  V1.13 2008-05-27 PB file name added for option -f,
                      -f included into help
---------------------------------------------------------------------------*/

# include <errno.h>
# include <stdio.h>
# include <unistd.h>
# include <fcntl.h>
# include <string.h>
# include <ctype.h>

# include "edfio.h"
# include "numio.h"

static const char * OptionStrings[] =
                     { "Invalid",
                       "-h",  "--help",  "-b",     "-c",     "-e",
                       "-s",  "-S",      "-x",     "-i",     "-I",     
                       "-p",  "-f", "-d", "--debug",
                       (const char *) NULL };

enum Option          { InValidOption,
                       Help1, Help2,     BlockKey, ChainKey, Keyword, 
                       Short, VeryShort, ShortNoLF, Ignore, IgnoreShort, 
                       Pipe, FileFormat, Debug1, Debug2,
                       EndOption };

/*---------------------------------------------------------------------------
NAME
  opt2num 

SYNOPSIS

  int opt2num( const char * string )

DESCRIPTION
  Returns the option number according to OptionStrings

RETURN VALUE
  0: not found, otherwise option number
---------------------------------------------------------------------------*/
int opt2num( const char * string )
{
  int NE=1,optno=0;
  while ( (NE && OptionStrings[optno]) )
    NE = strcmp( string , OptionStrings[optno++] );
  if (NE) optno = 0; else optno--;

  return(optno);

} // opt2num

void edfdumphelp( FILE * out, const char * progname, int verbose ) 
{ 
  fprintf(out,"Usage: %s [-h] [options] [-e <keyword>] <files...>\n",progname);
  fprintf(out,"Version: %s\n",Version);
  
  if (!verbose) return;
  fprintf(out,"Data format version: %s\n",edf_dataformat_version());
  fprintf(out,"   -h : this help\n");
  fprintf(out,"   -c <chainkey> -b <blockkey> -e <keyword>\n");
  fprintf(out,"   -s : short format, -S : print keywords only\n");
  fprintf(out,"   -i : ignore non-edf files, -I : show edf file names\n");
  fprintf(out,"   -p : read file names from stdin (stop with <ctrl-d>)\n");
  fprintf(out,"   -f : print format (block start and end addresses)\n");
  fprintf(out,"\n");
  fprintf(out,"Example:\n");
  fprintf(out,"   edfdump -c Image.Psd -b 1 -e Dim_1 sphere.edf\n");
  fprintf(out,"\n");

} // edfdumphelp

/*---------------------------------------------------------------------------
NAME
  edfdump

SYNOPSIS

  int edfdump( char * fname[], int ignore, int format, 
               char * chainkey[], char * blockkey[], char * keyword[] );

DESCRIPTION
  edfdump to stdout for all files in fname[], fname[] must stop with NULL

ARGUMENTS
  char * fname[]   : table of filenames, terminated with NULL
  int ignore       :  1 : ignore non-edf files
                      0 : normal
  int format       : output format 
  char * keyword[] : table of keywords, terminated with NULL

RETURN VALUE
  0, if OK, otherwise error
---------------------------------------------------------------------------*/
int edfdump( char * fname[], int ignore, int format, 
             char * chainkey[], char * blockkey[], char * keyword[] )
{int status, ErrorValue;
 int stream;
 int return_status=0; 

 for (;*fname!=(char*) NULL;fname++) {
   stream = edf_open_data_file ( *fname, "read", &ErrorValue, &status );
   if (status) {
      if (!ignore) {
        return_status=ErrorValue;
        fprintf(stderr,"%s: no edf file\r\n", *fname); 
      }
      // release allocated memory
      edf_free_all ( );
   } else {
      if (format==1001) fprintf(stdout,"%s\r\n", *fname);
        else edf_dump ( stdout, stream, format, chainkey, blockkey, keyword );

//     edf_print_filetable ( stdout, 4, 0 ); //++++++++++++++++++++

       edf_close_data_file( stream, &ErrorValue, &status );
       if (status) { 
         return_status=ErrorValue;
         fprintf(stderr,"%s\n",edf_report_data_error(ErrorValue));
         return(return_status); 
       }
     }
   }

 return(return_status);

} // edfdump

/* removes trailing line feed \n */
char * trimlf ( char * s )
{ size_t slen;
  if ( s ) {
    slen = strlen( s );
    if ( slen > 0 )
      if (s[slen-1]=='\n') s[slen-1]='\0';
  }
  return ( s );
} // trimlf

/* returns 1 if s is an empty string (only white space) otherwise 0. */
int isempty( const char *s )
{ const char *ps;
  int empty=1;

  if (s) {
    ps = s;
    while (*ps) { empty = empty && isspace(*ps); ps++; }
  }
  return( empty );

} //isempty

/*---------------------------------------------------------------------------
main
---------------------------------------------------------------------------*/

#if MAKE_FUNCTION
# define MAIN main_edfdump 
#else
# define MAIN main
#endif

int MAIN (int argc, char *argv[])
{ char *nul = (char *) NULL;
  char **ckeyv, **bkeyv, **keyv;
  char **pargv, **pckeyv, **pbkeyv, **pkeyv;
  int optno, ckeyc, bkeyc, keyc;
  int format=0, ignore=0, pipe=0;
  int status=-1;

  char *pstring[2];

  #define BUFLEN 1024
  char buffer[BUFLEN];

  // count options or help
  pargv = &argv[1];
  ckeyc = 1;
  bkeyc = 1;
  keyc  = 1;
  for (;*pargv!=nul && (**pargv=='-');*pargv++) {
    switch ( opt2num( *pargv ) ) {
       case   Help1  :
       case   Help2  : edfdumphelp( stdout, argv[0], 1 ); 
                       status=-2; return(status);
       case ChainKey : if (pargv[1]!=nul) { ckeyc++; *(++pargv); } break;
       case BlockKey : if (pargv[1]!=nul) { bkeyc++; *(++pargv); } break;
       case Keyword  : if (pargv[1]!=nul) { keyc++; *(++pargv); } break; 
       default       : break;
      }
    }

  // allocate memory for chainkeys
  if (!(ckeyv = (char **) malloc( sizeof(size_t) * ckeyc ))) {
    perror("malloc( sizeof(size_t) * ckeyc )");
    return(status);
    }
  // allocate memory for blockkeys
  if (!(bkeyv = (char **) malloc( sizeof(size_t) * bkeyc ))) {
    perror("malloc( sizeof(size_t) * bkeyc )");
    return(status);
    }
  // allocate memory for keywords 
  if (!(keyv = (char **) malloc( sizeof(size_t) * keyc ))) {
    perror("malloc( sizeof(size_t) * keyc )");
    return(status);
    }

  // get options
  pargv  = &argv[1];
  pckeyv = ckeyv;
  pbkeyv = bkeyv;
  pkeyv  = keyv;
  for (;*pargv!=nul && (**pargv=='-');*pargv++) {
    optno = opt2num( *pargv );
    switch ( optno ) {
       case Help1     :
       case Help2     : break;
       case Debug1    :
       case   Debug2 : if (pargv[1]!=nul) {
           edfio_debug( num_str2long ( *(++pargv), NULL, NULL) );
         } break;
       case BlockKey  : if (pargv[1]!=nul) *pbkeyv++ = *(++pargv); break;
       case Keyword   : if (pargv[1]!=nul) *pkeyv++  = *(++pargv); break;
       case Short     : format=1; break; 
       case VeryShort : format=2; break;
       case ShortNoLF : format=6; break;
       case Ignore    : ignore=1; break;
       case IgnoreShort : ignore=1; format=1001; break;
       case Pipe      : pipe=1; break;
       case FileFormat: format=10001; break;
       default        : break;
      }
    }
  *pckeyv = nul;
  *pbkeyv = nul;
  *pkeyv  = nul;

  if (*pargv) status=edfdump( pargv, ignore, format, ckeyv, bkeyv, keyv );
  else if (pipe) {
    // read from standard input
    pstring[1] = (char *) NULL;
    while ( !feof( stdin ) ) {
      pstring[0] = fgets ( buffer, BUFLEN, stdin );
      pstring[0] = trimlf ( pstring[0] );

      if ( pstring[0] ) {
        if ( (!strncmp(pstring[0],"exit",4)) || (!strncmp(pstring[0],"quit",4)) )
          return ( status );
        // edfdump 

        if (!isempty (pstring[0])) {
          status=edfdump( pstring, ignore, format, ckeyv, bkeyv, keyv );
        } else fprintf( stdout,"\n");
      }
    }
  } else {
    edfdumphelp( stdout, argv[0], 0 );
    status=-2;
  }

  free( ckeyv );
  free( bkeyv );
  free( keyv );

  return( status );

} // main

