/***************************************************************************/
/* Written 1994++ by Peter Boesecke                                        */
/* Copyright (C) 2011 European Synchrotron Radiation Facility              */
/*                       Grenoble, France                                  */
/*                                                                         */
/*    Principal authors: Peter Boesecke  (boesecke@esrf.eu)                */
/*                                                                         */
/*    This program is free software: you can redistribute it and/or modify */
/*    it under the terms of the GNU General Public License as published by */
/*    the Free Software Foundation, either version 3 of the License, or    */
/*    (at your option) any later version.                                  */
/*                                                                         */
/*    This program is distributed in the hope that it will be useful,      */
/*    but WITHOUT ANY WARRANTY; without even the implied warranty of       */
/*    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the        */
/*    GNU General Public License for more details.                         */
/*                                                                         */
/*    You should have received a copy of the GNU General Public License    */
/*    along with this program.  If not, see <http://www.gnu.org/licenses/>.*/
/***************************************************************************/
/*+++
1 sphere2saxs
  Generates the specific scattering cross section of diluted spheres 

2 PURPOSE
  Generates either the X-ray scattering cross section 1/A dSigma/dOmega 
  (+norm) or creates the expectation values (DeltaIs) of photons for each
  detector pixel (-norm). The latter one is the default.
  The scattering pattern can either be calculated for a solution of diluted 
  monodisperse spheres or diluted spheres following a Schulz distribution or
  can be read from a file. The file must contain in its first column the
  scattering vector s (s=2*sin(Theta)/Lambda) in units of 1/nm and in the 
  second column the normalized scattering cross section in 1/A dSigma/dOmega 
  in units of 1/m. If the generated pattern must not be multiplied with the
  polarization factor P sphere2saxs must be started with the option -dpol. The
  default is +dpol. In the case of -norm the number of incoming photons and 
  the sample transmission are required. 
  To simulate a detector pattern the not normalized image (-norm) should 
  additionally be multiplied by the detector sensitivity (flat field). 
  The result can be fed into a poissonian noise generator (saxs_poisson) to 
  generate a detector pattern. 

  METHOD
  This approach is only valid for diluted particles. 
  
  +norm (default):

  (1/A * dSigma/dOmega) = 1/V * (dSigma/dOmega) * d 
                        = 1/V * (N * re^2 * DeltaNe^2 * |f|^2) * d

   -norm:
   DeltaIs = I0 * (1/A * dSigma/dOmega) * DeltaOmega

   V                             : scattering volume
   dSigma/dOmega                 : differential scattering cross section
   d                             : thickness
   re                            : classical electron radius
   P = P ( Pol, PChi, PPsi )     : polarization factor
   N = V * RhoS                  : total number of particles in the 
                                   scattering volume V
   DeltaNe = Vs*DeltaRhoElektron : number of excess electrons in the 
                                   scattering particles compared to 
                                   the density of the surrounding 
                                   matrix. 
   For spheres:

   Vs = (4/3)*pi*R^3 : volume of a spherical particle with radius R

   |f|^2 = [ 3 * (sin(q*R) - (q*R)*cos(q*R))/(q*R)^3 ]^2

   All together:
   
   1/A*(dSigma/dOmega) = n * re^2 * P * DeltaRhoElektron^2 * Vs^2 * |f|^2 * d
      with Vs and |f|^2 as defined above 

   Parameters:

   DeltaRhoElektron :      electron volume density difference 
                      -dre <DeltaRhoElektron [#electrons/m^3]>
   R                :      radius of spheres
      mono disperse         -rad <R [m]>   
      Schulz distribution   -rad Schulz,<R [m]>,<Z>  with Z>-1
      from file             -rad File,<name[xy.txt]>,<skipl[0]>,<skipc[0]>,
                                      <swap[1]>,<colx[1]>,<coly[2]>
        The parameter "File" reads a table that must contain in colx the 
        scattering vectors s in 1/nm and in coly the differential scattering 
        cross sections. All lines starting with # are automatically skipped.
        If the option -d <thickness> is given the values are multiplied by 
        <thickness>.
             colx: s=2sin(Theta)/lambda [1/nm]
             coly: 1/V*(dSigma/dOmega) or
                   1/A*(dSigma/dOmega) for sample thickness d=1 (default)
      from function         -rad Function,function(_s,_q)
        The variables _s and _q are the scattering vectors in 1/nm, where
        q=2*pi*s with s=2*sin(Theta)/lambda [1/nm]
        If the option -d <thickness> is given the values are multiplied by 
        <thickness>.

   vfr = n * Vs 
       = (N/V) * Vs :      volume fraction of particles 
                           (N: number of particles, V: scattering volume,
                            Vs: average volume of particles)
                      -vfr <vfr>
   d                :      thickness of scattering volume
                      -d   <d [m]>

   Hint: To give the numbers in nanometers type <number>/_nm, e.g.
         -rad 100_nm sets the radius to 100 nm
         -dre 10/_nm/_nm/_nm

   The generated pattern is multiplied with the electron polarization
   factor P that depends on Pol, PChi and PPsi:

   Pol   : degree of polarization (0<=P<=1), default 1.0
   PChi  : ellipticity (after Poincar) (-pi/4<=PChi<=+pi/4), default 0.0
           PChi=-pi/4 left hand (cw) circular polarization
           PChi<0 left hand polarization
           PChi==0 linear polarization
           PChi>0 right hand polarization
           PChi=pi/4 right hand (ccw) circular polarization
   PPsi  : polarization direction (after Poincar) (0<=PPsi<pi), default 0.0
           PPsi is the angle between axis x_1 and the polarization direction
           measured ccw around axis x_1

  Arguments:
  sphere2saxs [options] <onam> <odum> <odim 1> <odim 2> 
                        <opix 1> <opix 2> <odis> <owvl>
                        <d> <dre> <rad> <vfr>
  Defaults:

  output file name              : <onam>              : output.edf
  output dummy                  : <odum>              : 0.0
  dimension 1                   : <odim 1>            : 512
  dimension 2                   : <odim 2>            : 512
  pixel size 1                  : <opix 1>            : 100_um
  pixel size 2                  : <opix 2>            : 100_um
  center 1                      : <ocen 1>            : half of dim 1 
  center 2                      : <ocen 2>            : half of dim 2 
  sample distance               : <odis>              : 10_m
  wavelength                    : <owvl>              : 0.1_nm
  scattering volume thickness   : <d>                 : 100_um

  radius of spheres or radial
      distribution function     : <rad>               : 100_nm
    electron density difference : <dre>               : 10/nm/nm/nm
    volume fraction             : <vfr>               : 0.01

2 HISTORY
  2000-11-25  PB from saxs_new
  2000-12-04  PB V3.1 -rad <radius or radius distribution function>
  2000-12-22  PB V3.2 f2_schulz
  2000-12-27  PB V3.3 normalization to volume fraction -vfr
  2001-01-10  PB V3.1 GetReference with output image after geometric setup
  2001-07-09  PB V4.00 new loop, no repetition
  2001-08-19  PB V4.01 normalization keywords are written
  2001-09-16  PB V4.02 no default for dummy
  2002-04-07  PB V4.03 radial distribution function from file,
                       calculation of theta corrected, 
                       normalization to vfr changed to vfr*(1-vfr)
                       paramter list changed
  2002-04-21  PB V4.04 
  2002-06-01  PB V4.50 ave
  2003-02-09  PB V4.51 line args for Dummy, DDummy, Dim
  2003-05-25  PB V4.52 line args for Center, PixSiz, SampleDistance, WaveLength
  2003-05-30  PB V4.53 new parameterlist for longkeyvalue function 
                       ImageLoop: Function call changed
  2003-08-14  PB V4.54 option_define, all option numbers removed
  2004-10-30  PB V4.55 MAKE_FUNCTION
  2005-02-11  PB V4.56 unnormalize2d, waxs, polarization,
                       maximum_scattering_vector
  2005-02-26  PB V4.57 show table values only for TestBit>2
  2005-05-30  PB V4.58 distribution type Function
  2005-06-30  PB V4.59 function of _s and/or _q 
  2005-07-03  PB V4.60 comments corrected 
  2005-07-15  PB V4.61 ReportSaxsStatus parameter list updated
  2006-08-08  PB V4.62 from_table sets SaxsError
  2007-02-14  PB V4.63 Schulz, Schultz, Shulz
  2007-06-11  PB V4.64 ImageLoop parameters updated
  2008-11-29  PB V4.65 sphere2saxs_init (to set default dimensions)
                       ArgvFilenames -> ArgvBlocks
  2008-12-01  PB V4.66 default center definition corrected
  2010-05-31  PB V4.67 update for waxs.c V1.10, update for polarization.c V1.4
  2011-06-22  PB V4.68 update for polarization.c V1.6 (Orientation)
  2011-06-27  PB V4.69 waxs_Saxs2Waxs updated
  2011-07-23  PB V4.70 waxs_Saxs2Waxs -> waxs_S2Sp
  2012-01-31  PB V4.71 ImageLoop parameters updated
  2014-02-07  PB V4.72 sphere2saxs: waxs_Init parameters updated
  2016-04-02  PB V4.73 ceil -> ceilf (only float values)

---*/
# define Version  "sphere2saxs V4.73 2016-04-02, Peter Boesecke"

# include <errno.h>
# include <stdio.h>
# include <unistd.h>
# include <fcntl.h>

# include "SaxsPrograms.h"
# include "numio.h"
# include "schulz.h"
# include "table.h"
# include "waxs.h"
# include "polarization.h"

# define Usage "[options] \n\
             <onam> <odum> <odim 1> <odim 2> <opix 1> <opix 2> <odis> <owvl>\n\
             <rad> <dre> <vfr> <d>\n\
             <pol> <pchi> <ppsi>\n\
             <norm> <monv> <trm> <timv>\n\
             Creates 2d powder scattering pattern with different models.\n\
             Options: -dre, -rad,  -vfr, -d\n\
                      -pol, -pchi, -ppsi, +dpol"

# define BlockNum 1       /* 1 output sequence */

/* float options */
# define SDre  "dre"  // DeltaRhoElektron [1/m^3]
# define SVfr  "vfr"  // Volume fraction of particles
# define SD    "d"    // Thickness of scattering volume [m]
# define SI0   "monv" // Incoming number of photons [ph]
# define STrm  "trm"  // Transmission
# define STim  "timv" // Measurement time [s], info only, not used for calculations
# define SPol  "pol"  // degree of polarization (0<=P<=1) 
# define SPChi "pchi" // ellipticity (after Poincar) (-pi/4<=PChi<=+pi/4) 
# define SPPsi "ppsi" // polarization direction (after Poincar) (0<=PPsi<pi) 

/* line options */
# define SRad  "rad"  // Radius Distribution Function

/* flag options */
# define SNorm  "norm" // +norm/-norm : output of normalized/unnormalized data
# define SDoPol "dpol" // +dpol/-dpol : multiply with polarization factor P 

enum DistributionType { InValidDistributionType,
                        MonoDisperse=1, Schulz, Schultz, Shultz, 
                        File, Function, EndDistributionType };

/*---------------------------------------------------------------------------
 Private Definitions 
---------------------------------------------------------------------------*/
PRIVATE const char * ModuleName = "sphere2saxs";

PRIVATE const char * DistributionTypeStrings[8] =
                      { "Invalid", 
                        "MonoDisperse", "Schulz", "Schultz", "Shultz", 
                        "File", "Function", (const char *) NULL };

/*---------------------------------------------------------------------------
renorm_flexp : tries to reduce the float expressoin in expr->V 
---------------------------------------------------------------------------*/
void renorm_flexp( IO_flexp * expr )
{ double tmp;
  int status;
  tmp=doubleexpr( expr->V, &status );
  if ( status == Success ) sprintf(expr->V,"%g",tmp);
} // renorm_flexp

/*---------------------------------------------------------------------------
renorm_lvexp : tries to reduce the long expressoin in expr->V 
---------------------------------------------------------------------------*/
void renorm_lvexp( IO_lvexp * expr )
{ long tmp;
  int status;
  tmp=longexpr( expr->V, &status );
  if ( status == Success ) sprintf(expr->V,"%d",tmp);
} // renorm_lvexp

/*---------------------------------------------------------------------------
calc_len
---------------------------------------------------------------------------*/
float calc_len( WaxsCoord edge )
{ float len;
  len = sqrt(edge.s_1*edge.s_1+edge.s_2*edge.s_2);
  return( len );
} /* calc_len */

/*---------------------------------------------------------------------------
maximum_scattering_vector

   Returns the maximum reciprocal distance (in [1/m]).
---------------------------------------------------------------------------*/
float maximum_scattering_vector( CmdBlk * pcb, ImgHeadBlk ihb[], int blkno,
                               int * pstatus )
{ ImgHeadBlk * pihb;
  double K;

  *pstatus = Success;

  pihb = &ihb[blkno];

  if (!(pihb->WaveLength.I) ) {
    *pstatus = Failed; 
    fprintf(ERROUT,"Missing wavelength definition\n"); return( 0l );
  }

  K    = (double) WAVENUMBER(ihb[0].WaveLength.V);

  return ( 2*K );

} /* maximum_scattering_vector */

/*---------------------------------------------------------------------------
 
   from_table 
 
   Reads 1/A dSigma/dOmega from a two column table. The columns are delimited
   by the character limc (default: tabulator) and terminated by the character
   liml (default: carriage return):

   <skiplines> lines (terminated by liml) to skip (default: 0)
   <skipchars> characters to skip (default: 0)
   lines starting with <comment> (default: "#") are skipped

   x-column    y-column
   <s*nm> limc <I*sterad> liml 

   If <swap> is 1 x- and y-columns are swapped (default: 0). 
   
 
   1/A dSigma/dOmega : differential scattering cross section 
                       per beam cross section A [1/sterad]
     A = V/d   with

     V               : scattering volume
     d               : thickness of scattering volume 
 
 
 PARAMETERS
   float radial[]        : radial output buffer
   long dim              : length of output buffer
   float off             : offset
   float dummy           : dummy value
   float ddummy          : ddummy value
   float pix             : radial pixel size [m]
   float cen             : center
   float dis             : sample distance [m]
   float wvl             : wavelength [m]
   float d               : thickness of scattering volume [m]
   float weight          : weight factor, all values are multiplied by weight

   xy-table parameters
   const char * filename, long skiplines, long skipchars,
   int swap, long colx, long coly, long readlines,
   const char * comment, const char * liml, const char * limc,

   int initialize        : 1: initializes radial[] with dummy and adds values
                           0: adds values to radial[] without initializing
   int debug             : !=0: show simple debug info
                            >2: show input table values
                            >3: show interpolated table values
                             0: do not show debug info

   int * pstatus              return status
 
---------------------------------------------------------------------------*/
void from_table(float radial[], long dim, float off,
                float dummy, float ddummy, float pix,
                float cen, float dis, float wvl, 
                float d, float weight, 

                const char * filename, long skiplines, long skipchars,
                int swap, long colx, long coly, long readlines, 
                const char * comment, const char * liml, const char * limc,

                int initialize, int debug, int * pstatus )
{
   const float eps = 1e-10;
   const double pi = 3.1415926535897932384626;
   const double re = 2.817940325e-15; /* classical electron radius (m) */

   const char * RoutineName = "from_table";
 
   long i;
   float Off, Ps;
   float s;
   float radius; // radius around origin
   float f, f2;
   float n;
   float factor1;
   float value;

   long lines;
   TableXYData * xydata;

   *pstatus = Success;

   SetSaxsError( ModuleName, RoutineName, (const char *) NULL );
 
   if (debug) {
     printf("dim              = %d\n", dim);
     printf("off              = %g\n", off);
     printf("dummy            = %g\n", dummy);
     printf("ddummy           = %g\n", ddummy);
     printf("pix              = %g\n", pix);
     printf("cen              = %g\n", cen);
     printf("dis              = %g\n", dis);
     printf("wvl              = %g\n", wvl);
     printf("d                = %g\n", d);
     printf("weight           = %g\n", weight);
     printf("filename         = %s\n", filename);
     printf(" skiplines       = %d\n", skiplines);
     printf(" skipchars       = %d\n", skipchars);
     printf(" swap            = %d\n", swap);
     printf(" colx            = %d\n", colx);
     printf(" coly            = %d\n", coly);
     printf(" readlines       = %d\n", readlines);
     printf(" comment         = char(%d)\n", comment[0]);
     printf(" liml            = char(%d)\n", liml[0]);
     printf(" limc            = char(%d)\n", limc[0]);
     printf("initialize       = %s\n", initialize?"yes":"no");
     }

   /* init table */
   table_Init( (double) dummy , comment, liml, limc );

   /* read table */
   //     get number of valid xy-pairs
   lines =  table_ReadXYDouble( (TableXYData *) NULL, filename,
                           skiplines, skipchars, swap, colx, coly, readlines );
   if ( lines < 0 ) {
     perror("from_table : table_table_ReadXYDouble ");
     SetSaxsErrorMessage( filename );
     *pstatus = Failed;
     return;
   }
   //    allocate table
   if (debug) {
     printf("%s has %d xy-pairs\n",filename,lines);
     printf("\n");
   }
   if (!(xydata = table_newXYData( lines ))) {
     perror("from_table : table_newXYData ");
     SetSaxsErrorMessage( filename );
     *pstatus = Failed;
     return;
   }
   //    read xy-pairs */
   if ( table_ReadXYDouble( xydata, filename, skiplines, skipchars,
                            swap, colx, coly, readlines ) < 0 ) {
     perror("from_table : table_table_ReadXYDouble ");
     SetSaxsErrorMessage( filename );
     *pstatus = Failed;
     table_freeXYData( xydata );
     return; 
   }
   if (debug>2) 
     for (i=0;i<xydata->len;i++) 
       printf("%d : %g : %g \n",i+1,xydata->X[i],xydata->Y[i]);
 
   /* create radial array */
   factor1 = weight * d;

   /* Set the radial reference system to saxs */
     SAXSREF(Off,Ps,off,pix,cen,dis,wvl);

   if (debug)
     printf("Off=%g, Ps=%g\n",Off,Ps);
 
   if (initialize) for (i=0;i<dim;i++) radial[i] = dummy;

   for (i=0;i<dim;i++) {
     radius = WORLD(i,Off,Ps);
     s = radius; // Waxs coordinate

     // interpolate xy-table
     value = factor1 * table_LinIntXYDouble( xydata, s ); 
 
     UPDATE( radial[i], value, dummy, ddummy );
     }
 
   if (debug>3) {
     printf("%15s %15s %15s\n","i","radius","radial");
     for (i=0;i<dim;i++) {
       radius = WORLD(i,Off,Ps);
       printf("%15d %15g %15g\n",i,radius,radial[i]);
       }
     }
 
   table_freeXYData( xydata );

   *pstatus = Success;

   return;
 
} /* from_table */

/*---------------------------------------------------------------------------
 
   from_function
 
   Calculates 1/A dSigma/dOmega as a function of the scattering vector _s
   _s is the scattering vector s=2*sin(Theta)/Lambda in 1/nm.

   1/A dSigma/dOmega = weight*d*function(_s)
 
   1/A dSigma/dOmega : differential scattering cross section
                       per beam cross section A [1/sterad]
     A = V/d   with
 
     V               : scattering volume
     d               : thickness of scattering volume
 
 
 PARAMETERS
   float radial[]        : radial output buffer
   long dim              : length of output buffer
   float off             : offset
   float dummy           : dummy value
   float ddummy          : ddummy value
   float pix             : radial pixel size [m]
   float cen             : center
   float dis             : sample distance [m]
   float wvl             : wavelength [m]
   float d               : thickness of scattering volume [m]
   float weight          : weight factor, all values are multiplied by weight
 
   function parameters
   const char * expression
 
   int initialize        : 1: initializes radial[] with dummy and adds values
                           0: adds values to radial[] without initializing
   int debug             : !=0: show simple debug info
                             0: do not show debug info
 
   int * pstatus              return status
 
---------------------------------------------------------------------------*/
void from_function( float radial[], long dim, float off,
                    float dummy, float ddummy, float pix,
                    float cen, float dis, float wvl,
                    float d, float weight,
                    const char * expression,
                    int initialize, int debug, int * pstatus )
{
   const float eps = 1e-10;
   const double pi = 3.1415926535897932384626;
   const double re = 2.817940325e-15; /* classical electron radius (m) */
 
   long i;
   float Off, Ps;
   float s;
   float radius; // radius around origin
   float factor1;
   float value;

   NumProg *program;
   int errval;
   int buflen=512;
   char buffer[512];
 
   *pstatus = Success;
 
   if (debug) {
     printf("dim              = %d\n", dim);
     printf("off              = %g\n", off);
     printf("dummy            = %g\n", dummy);
     printf("ddummy           = %g\n", ddummy);
     printf("pix              = %g\n", pix);
     printf("cen              = %g\n", cen);
     printf("dis              = %g\n", dis);
     printf("wvl              = %g\n", wvl);
     printf("d                = %g\n", d);
     printf("weight           = %g\n", weight);
     printf("expression       = %s\n", expression);
   }

   /* create radial array */
   factor1 = weight * d;
 
   /* Set the radial reference system to saxs */
     SAXSREF(Off,Ps,off,pix,cen,dis,wvl);
 
   if (debug)
     printf("Off=%g, Ps=%g\n",Off,Ps);
 
   if (initialize) for (i=0;i<dim;i++) radial[i] = dummy;

   program = num_str2prog ("sphere2saxs",expression,NULL,&errval,2,"s","q");
   if (errval) {
     fprintf(ERROUT,"ERROR: %s\n",num_errval2str( buffer,buflen,errval ));
     num_rmprog ( program, &errval );
     *pstatus=Failed; return;
   } 
 
   for (i=0;i<dim;i++) {
     radius = WORLD(i,Off,Ps);
     s = radius; // Waxs coordinate
 
     // calculate function value
     value = factor1 * num_runprog ( program, &errval, s, 2*pi*s );
     if (!errval) {
       UPDATE( radial[i], value, dummy, ddummy );
     } 
   }

   num_rmprog ( program, &errval );
 
   if (debug>3) {
     printf("%15s %15s %15s\n","i","radius","radial");
     for (i=0;i<dim;i++) {
       radius = WORLD(i,Off,Ps);
       printf("%15d %15g %15g\n",i,radius,radial[i]);
       }
     }
 
   *pstatus = Success;
 
   return; 

} /* from_function */

/*---------------------------------------------------------------------------

   diluted_spheres

   Creates 1/A dSigma/dOmega of a system of diluted spheres 

   d/V*(dSigma/dOmega) = n * re^2 * DeltaRhoElektron^2 * Vs^2 * |f|^2 * d
   
   with 

   Vs = (4/3)*pi*R^3 : volume of a spherical particle with radius R

   |f|^2 = [ 3 * (sin(q*R) - (q*R)*cos(q*R))/(q*R)^3 ]^2

   dSigma/dOmega     : differential scattering cross section
   V                 : scattering volume 
   re                : classical electron radius

   n = N/V           : density of particles


 PARAMETERS
   float radial[]        : radial output buffer
   long dim              : length of output buffer
   float off             : offset
   float dummy           : dummy value
   float ddummy          : ddummy value
   float pix             : radial pixel size [m]
   float cen             : center
   float dis             : sample distance [m]
   float wvl             : wavelength [m]
   float d               : thickness of scattering volume [m]
   float weight          : weight factor, all values are multiplied by weight

   float R               : radius of spheres [m]
   float DeltaRhoElektron:electron volume density difference [#electrons/m^3]
   float vfr             : volume fraction of particles

   int initialize        : 1: initializes radial[] with 0 and adds values
                           0: adds values to radial[] without initializing

 RETURN VALUE
   0: success
   1: failed 
---------------------------------------------------------------------------*/
int diluted_spheres( float radial[], long dim, float off, 
                     float dummy, float ddummy, float pix,
                     float cen, float dis, float wvl, 
                     float d, float weight, 
                     float R, float DeltaRhoElektron, float vfr, 
                     int initialize, int debug )
{ 
   const float eps = 1e-4;
   const double pi = 3.1415926535897932384626;
   const double re = 2.817940325e-15; /* classical electron radius (m) */

   long i;
   float Off, Ps;
   float qR;
   float radius; // radius around origin 
   float f, f2;
   float n;
   float factor1, factor2;
   float value;

   if (debug) {
     printf("dim               = %d\n", dim);
     printf("off               = %g\n", off);
     printf("dummy             = %g\n", dummy);
     printf("ddummy            = %g\n", ddummy);
     printf("pix               = %g\n", pix);
     printf("cen               = %g\n", cen);
     printf("dis               = %g\n", dis);
     printf("wvl               = %g\n", wvl);
     printf("d                 = %g\n", d);
     printf("weight            = %g\n", weight);
     printf("R                 = %g\n", R);
     printf(" DeltaRhoElektron = %g\n", DeltaRhoElektron);
     printf(" vfr              = %g\n", vfr);
     printf("initialize        = %s\n", initialize?"yes":"no");
     }

   n = (vfr*(1.0-vfr))/((4.0/3.0)*pi*R*R*R);

   factor1 = weight *
             n * 
             re*re * 
             DeltaRhoElektron*DeltaRhoElektron * 
             ((4.0/3.0)*pi*R*R*R)*((4.0/3.0)*pi*R*R*R) *
             d;
   factor2 = 2.0*pi*R/WAVENUMBER(1);

   /* Set the radial reference system to saxs */
     SAXSREF(Off,Ps,off,pix,cen,dis,wvl);

   if (initialize) for (i=0;i<dim;i++) radial[i] = dummy;

   for (i=0;i<dim;i++) {
     radius = WORLD(i,Off,Ps);
     qR = factor2*radius; // radius in Waxs coordinates
     if (fabs(qR)<eps) f2 = 1.0;
       else { 
       f  = 3.0 * ( sin(qR) - (qR)*cos(qR) ) / ( qR*qR*qR );
       f2 = f*f;
     }
     value = factor1 * f2;

     UPDATE( radial[i], value, dummy, ddummy );
     }

   if (debug>3) {
     printf("%15s %15s %15s\n","i","radius","radial");
     for (i=0;i<dim;i++) {
       radius = WORLD(i,Off,Ps);
       printf("%15d %15g %15g\n",i,radius,radial[i]);
       }
     }

   return(0);

} /* diluted_spheres */

/*---------------------------------------------------------------------------

   diluted_schulz

   Creates 1/A dSigma/dOmega of a system of diluted schulz-distributed spheres

   d/V*(dSigma/dOmega) = n * re^2 * DeltaRhoElektron^2 * Vs^2 * f2_schulz * d

   with

   Vs = (4/3)*pi*R^3 : volume of a spherical particle with radius R

   R                 : average radius
   Z                 : width parameter of Schulz distribution
                       sigma = <R>/sqrt(Z+1)

   f2_schulz        : schulz averaged square of form factor of spheres with 
                       average radius R

   dSigma/dOmega     : differential scattering cross section
   V                 : scattering volume
   re                : classical electron radius

   vfr               : volume fraction of particles

 PARAMETERS
   float radial[]        : radial output buffer
   long dim              : length of output buffer
   float off             : offset
   float dummy           : dummy value
   float ddummy          : ddummy value
   float pix             : radial pixel size [m]
   float cen             : center
   float dis             : sample distance [m]
   float wvl             : wavelength [m]
   float d               : thickness scattering volume [m]
   float weight          : weight factor, all values are multiplied by weight

   float R               : average radius of spheres [m]
   float Z               : width parameter of Schulz distribution
   float DeltaRhoElektron:electron volume density difference [#electrons/m^3]
   float vfr             : volume fraction of particles

   int initialize        : 1: initializes radial[] with 0 and adds values
                           0: adds values to radial[] without initializing

 RETURN VALUE
   0: success
   1: failed
---------------------------------------------------------------------------*/
int diluted_schulz( float radial[], long dim, float off, 
                     float dummy, float ddummy, float pix,
                     float cen, float dis, float wvl, 
                     float d, float weight, 
                     float R, float Z, float DeltaRhoElektron, float vfr, 
                     int initialize, int debug )
{
   const float eps = 1e-10;
   const double pi = 3.1415926535897932384626;
   const double re = 2.817940325e-15; /* classical electron radius (m) */

   long i;
   float Off, Ps;
   float qR;
   float radius; // radius around origin
   float f2;
   float n;
   float factor1, factor2;
   float value;

   if (debug) {
     printf("dim               = %d\n", dim);
     printf("off               = %g\n", off);
     printf("dummy             = %g\n", dummy);
     printf("ddummy            = %g\n", ddummy);
     printf("pix               = %g\n", pix);
     printf("cen               = %g\n", cen);
     printf("dis               = %g\n", dis);
     printf("wvl               = %g\n", wvl);
     printf("d                 = %g\n", d);
     printf("weight            = %g\n", weight);
     printf("R                 = %g\n", R);
     printf(" Z                = %g\n", Z);
     printf(" DeltaRhoElektron = %g\n", DeltaRhoElektron);
     printf(" vfr              = %g\n", vfr);
     printf("initialize        = %s\n", initialize?"yes":"no");
     }

   n = (vfr*(1.0-vfr))/( (4.0/3.0)*pi*R3_schulz(R,Z) );

   factor1 = weight *
             n *
             re*re *
             DeltaRhoElektron*DeltaRhoElektron *
             ((4.0/3.0)*pi*R*R*R)*((4.0/3.0)*pi*R*R*R) *
             d;

   factor2 = 2.0*pi*R/WAVENUMBER(1);

   /* Set the radial reference system to saxs */
     SAXSREF(Off,Ps,off,pix,cen,dis,wvl);

   if (initialize) for (i=0;i<dim;i++) radial[i] = dummy;

   for (i=0;i<dim;i++) {
     radius = WORLD(i,Off,Ps);
     if (fabs(dis)>eps) {
       qR = factor2*radius; // radius in Waxs coordinates
       f2 = f2_schulz( Z, qR );
       if (f2>=0.0) {
         value = factor1 * f2;

         UPDATE( radial[i], value, dummy, ddummy );
         }
       }
     }

   if (debug>3) {
     printf("%15s %15s %15s\n","i","radius","radial");
     for (i=0;i<dim;i++) {
       radius = WORLD(i,Off,Ps);
       printf("%15d %15g %15g\n",i,radius,radial[i]);
       }
     }

   return(0);

} /* diluted_schulz */

/*---------------------------------------------------------------------------
1 unnormalize2d

2 PURPOSE
  Unnormalizes the absolute intensities of a 2d scattering pattern

      1/A * dSigma/dOmega

  in data[] by multiplicating each point with the number of transmitted
  photons (I1) and by multiplicating each point with the covered spherical
  angle DeltaOmega:

      I1 * (pix1*pix2)/(L^2*cos(2Theta)) = I1 * pix1*pix2*dis/L^3.

  The result is the average number DeltaIs of scattered photons per pixel:

      DeltaIs = I0 * (1/A * dSigma/dOmega) * DeltaOmega

 PARAMETERS
   float data[]          : image buffer
   int dim1              : dim 1 of data
   int dim2              : dim 2 of data 
   float off1            : offset 1
   float off2            : offset 2
   float dummy
   float ddummy
   float pix1            : pixel size 1 [m]
   float pix2            : pixel size 2 [m]
   float cen1            : center 1
   float cen2            : center 2
   float dis             : sample distance [m]
   float I1              : transmitted number of photons [ph]

 RETURN VALUE
   0: success
   1: failed

---------------------------------------------------------------------------*/
int unnormalize2d( float data[], long dim1, long dim2, 
                   float off1, float off2, float dummy, float ddummy,
                   float pix1, float pix2, float cen1, float cen2, 
                   float dis, float I1, int debug )
{
  int i_1, i_2;
  float factor1, factor2;
  float Off_1, Ps_1; 
  float Off_2, Ps_2; 
  float radius2; // square of the distance from center
  float L, L2; // distance of pixel from detector
  float *pdata;
  float W_1, W_2, W_22;
  
  if (debug) {
    printf("dim1             = %d\n", dim1);
    printf("dim2             = %d\n", dim2);
    printf("off1             = %g\n", off1);
    printf("off2             = %g\n", off2);
    printf("dummy            = %g\n", dummy);
    printf("ddummy           = %g\n", ddummy);
    printf("pix1             = %g\n", pix1);
    printf("pix2             = %g\n", pix2);
    printf("cen1             = %g\n", cen1);
    printf("cen2             = %g\n", cen2);
    printf("dis              = %g\n", dis);
    printf("I1               = %g\n", I1);
  }

  /* Set the reference system to normal */
  NORMALREF(Off_1,Ps_1,off1,pix1,cen1);
  NORMALREF(Off_2,Ps_2,off2,pix2,cen2);
  
  factor1 = I1*pix1*pix2*dis;
  factor2 = dis*dis;
  
  pdata = data;
  for (i_2=0;i_2<dim2;i_2++) {
    W_2 = WORLD(i_2,Off_2,Ps_2);
    W_22 = W_2*W_2;
    for (i_1=0;i_1<dim1;i_1++) {
      if ( NODUMMY( *pdata, dummy, ddummy ) ) {
        W_1 = WORLD(i_1,Off_1,Ps_1);
        radius2 = W_22+W_1*W_1;
        L2 = radius2 + factor2;
        L  = sqrt(L2);
        *pdata *= factor1/(L*L2);
      }
      pdata++;
    }
  }
    
  return(0);

} /* unnormalize2d */

/*---------------------------------------------------------------------------
NAME

  string2distributiontype --- converts a string to DistributionType 

SYNOPSIS

  (DistributionType) int string2distributiontype( char * string );

DESCRIPTION
  The input string might be changed (blanks removed etc)

RETURN VALUE
  DType == 0 : error, e.g. cannot convert
  DType >  0 : valid distribution type value

AUTHOR
2000-12-04  PB
  --------------------------------------------------------------------------*/
int string2distributiontype( char * string )
{ int  NE=1;
  int  i = 0;
  int  status;

  while ( (NE && DistributionTypeStrings[i]) ) {
    NE = strcmp(string,DistributionTypeStrings[i++]);
  }

  i = MAX2(0,i-1);

  if (NE) {
    (void) floatexpr( string, &status ); 
    if (status==Success) NE=0;
    i = MonoDisperse;
  }

  if (NE) i = InValidDistributionType;

  return( i );

} /* string2distributiontype */

/*---------------------------------------------------------------------------
sphere2saxs_init

Initializes the output header of each output file before the output
data array is allocated.

The number of available blocks is defined by the program and does not need
to be checked here. All input images are open in SaxsImageLoop when a new
output image is created.

sphere2saxs:
---------------------------------------------------------------------------*/
void sphere2saxs_init( CmdBlk *pcb, long num[], ImgHeadBlk ihb[], int *pstatus )
{ ImgBlk *ib;

  ib = pcb->ib;

  /*--- Set output header */
  if (!ib[0].Dim[1].I) ihb[0].Dim[1] = 512;
  if (!ib[0].Dim[2].I) ihb[0].Dim[2] = 512;

} // sphere2saxs_init

/*---------------------------------------------------------------------------
sphere2saxs

PURPOSE
  Generates a new image with constant pixel values
---------------------------------------------------------------------------*/
void sphere2saxs (CmdBlk * pcb, long num[], ImgHeadBlk ihb[], int * pstatus)
{ int i,imax;

  float *I0Data;
  int   I0Dim_1,I0Dim_2;
  float *pI0Data;
  float I0Dummy, I0DDummy;
  float Value;

  int i_1, i_2;
  float f_1[BlockNum], f_2[BlockNum];
 
  float Off_1[BlockNum], Off_2[BlockNum];
  float Ps_1[BlockNum], Ps_2[BlockNum];
 
  float f1_1[BlockNum], f3_1[BlockNum], Df_1[BlockNum];
  float f1_2[BlockNum], f3_2[BlockNum], Df_2[BlockNum];
 
  int Imin_1, Imin_2, Imax_1, Imax_2;
 
  float Wmin_1, Wmax_1, DW_1;
  float Wmin_2, Wmax_2, DW_2;

  double K, rot1, rot2, rot3;
  WaxsCoord W, W0;

  WParams wparams;
  PParams pparams;

  long  ori;
  double pol, pchi, ppsi;
  double ff;
  long   dopol;

  float dre, rad, vfr, d;
  float arg1, arg2;
  char  *arg;
  long  skiplines=0l, skipchars=0l, readlines=-1l;
  long  colx=1l, coly=2l;
  int   swap=1;
  int   distribution_type;
  float i0, trm, time;
  long  norm;
  float pixsiz, center = ARRAYSTART, offset = 0.0;
  float f_r;
  long radial_dim;
  float * radial;
  float radius;

  float dummy = -1.0;
  float ddummy = DDSET(dummy);
  float SampleDistance, WaveLength;
  char  line1[InputLineLength],
        line2[InputLineLength],
        filename[InputLineLength],
        comment[InputLineLength],
        liml[InputLineLength],
        limc[InputLineLength];

  IO_long  Norm;
  IO_line  Rad;
  IO_float I0, Trm, Tim, Dre, Vfr, D;

  IO_long  DoPol;
  IO_float Pol, PChi, PPsi;

  *pstatus = Success;

  imax = pcb->ImgBlkLen;
  printf("\n Calculating ihb[0,% d] = Function(",ihb[0].ImNum);
  for(i=1;i<imax;i++) {
    printf("ihb[% d,% d] ", i, ihb[i].ImNum); }

  printf(")\n\n");

 /* Check the number of images */
  if (pcb->ImgBlkLen!=1) {
     fprintf(ERROUT,"%d images found, 1 output image required\n",
             pcb->ImgBlkLen); *pstatus=Failed; return; }

 /* --- Get option values */
  Norm = option_flag( SNorm, pcb, pstatus);
  if (*pstatus!=Success) return;
  Rad = option_line ( SRad, pcb, pcb->ib, num, 1, pstatus );
  if (*pstatus!=Success) return;
  I0 = option_float ( SI0, pcb, pcb->ib, num, 1, pstatus );
  if (*pstatus!=Success) return;
  Trm = option_float ( STrm, pcb, pcb->ib, num, 1, pstatus );
  if (*pstatus!=Success) return;
  Tim = option_float ( STim, pcb, pcb->ib, num, 1, pstatus );
  if (*pstatus!=Success) return;
  Dre = option_float ( SDre, pcb, pcb->ib, num, 1, pstatus );
  if (*pstatus!=Success) return;
  Vfr = option_float ( SVfr, pcb, pcb->ib, num, 1, pstatus );
  if (*pstatus!=Success) return;
  D = option_float ( SD, pcb, pcb->ib, num, 1, pstatus );
  if (*pstatus!=Success) return;
  Pol = option_float ( SPol, pcb, pcb->ib, num, 1, pstatus );
  if (*pstatus!=Success) return;
  PChi = option_float ( SPChi, pcb, pcb->ib, num, 1, pstatus );
  if (*pstatus!=Success) return;
  PPsi = option_float ( SPPsi, pcb, pcb->ib, num, 1, pstatus );
  if (*pstatus!=Success) return;
  DoPol = option_flag( SDoPol, pcb, pstatus);
  if (*pstatus!=Success) return;

  /* Check the reference system: Saxs required */
  if (pcb->RSys.V != IO_Saxs) {
    fprintf(ERROUT,"ERROR: Reference system %s required\n",
      reftostr(IO_Saxs));
    fprintf(ERROUT,"       Start program with option -rsys %s\n",
      reftostr(IO_Saxs));
    *pstatus=Failed; return;
    }

   SAXSREF(Off_1[0],Ps_1[0],ihb[0].Offset[1].V,ihb[0].PixSiz[1].V,\
           ihb[0].Center[1].V,ihb[0].SampleDistance.V,ihb[0].WaveLength.V);
   SAXSREF(Off_2[0],Ps_2[0],ihb[0].Offset[2].V,ihb[0].PixSiz[2].V,\
           ihb[0].Center[2].V,ihb[0].SampleDistance.V,ihb[0].WaveLength.V);

 /* 1 output */
  I0Data  = ihb[0].Data;
  I0Dummy = ihb[0].Dummy.V;
  I0DDummy = ihb[0].DDummy.V;
  I0Dim_1  = (int) ihb[0].Dim[1];
  I0Dim_2  = (int) ihb[0].Dim[2];

  GetImageRange         ( pcb, ihb, 1, imax-1,
                          Off_1, Off_2, Ps_1,  Ps_2,
                          f1_1, f3_1, Df_1, f1_2, f3_2, Df_2,
                          &Imin_1, &Imax_1, &Imin_2, &Imax_2,
                          &Wmin_1, &Wmax_1, &DW_1, &Wmin_2, &Wmax_2, &DW_2, 1);

  pixsiz         = MIN2(ihb[0].PixSiz[1].V,ihb[0].PixSiz[2].V);
  SampleDistance = ihb[0].SampleDistance.V;
  WaveLength     = ihb[0].WaveLength.V;
  dre            = Dre.V;
  arg            = Rad.V;
  vfr            = Vfr.V;
  if (D.I) d=D.V; else d=1.0;
  i0             = I0.V;
  trm            = Trm.V;
  time           = Tim.V;
  norm           = Norm.V;

  radial_dim = CEILF( S2INDEX(maximum_scattering_vector(pcb,ihb,0,pstatus),\
    offset,pixsiz,center,SampleDistance,WaveLength) ) + 1;

  if (!(radial = malloc( sizeof(float) * radial_dim ))) {
    *pstatus=NotEnoughMemoryAvailable; return;
  }

  LeftFL(arg,",",line1,InputLineLength);
  distribution_type = string2distributiontype( line1 );

  switch (distribution_type) {
    case MonoDisperse : // R
      rad  = floatexpr( line1, pstatus ); if (*pstatus!=Success) return; 

      if (!ihb[0].Title.I) {
        sprintf(ihb[0].Title.V,
              "Spheres: R=%g nm, vfr=%g, dre=%g e/nm^3, d=%g mm",
              rad*1e9, vfr, dre*1e-27, d*1e3 ); 
        ihb[0].Title.I = TRUE;
        }

      diluted_spheres( radial, radial_dim, offset, dummy, ddummy,
                       pixsiz, center, SampleDistance, WaveLength,
                       d, 1.0, rad, dre, vfr, TRUE, pcb->TestBit); 
      break;

    case Schulz: Schultz: Shultz : // "Schulz", R, Z
      RightFL(arg,",",line1,InputLineLength);
      LeftFL(line1,",",line2,InputLineLength);
      arg1  = floatexpr( line2, pstatus ); if (*pstatus!=Success) return;

      RightFL(line1,",",line1,InputLineLength);
      LeftFL(line1,",",line2,InputLineLength);
      arg2  = floatexpr( line2, pstatus ); if (*pstatus!=Success) return;

      if (!ihb[0].Title.I) {
        sprintf(ihb[0].Title.V,
               "Schulz: R=%g nm, Z=%g, vfr=%g, dre=%g e/nm^3, d=%g mm",
               arg1*1e9, arg2, vfr, dre*1e-27, d*1e3 );
        ihb[0].Title.I = TRUE;
        } 

      diluted_schulz( radial, radial_dim, offset, dummy, ddummy,
                       pixsiz, center, SampleDistance, WaveLength,
                       d, 1.0, arg1, arg2, dre, vfr, TRUE, pcb->TestBit);
      break;

    case File : 
      // File,filename,skipl,skipc,swap,colx,coly,readl,commentx,liml,limc
      // filename
      RightFL(arg,",",line1,InputLineLength);
      LeftFL(line1,",",filename,InputLineLength);
      EditLine (Trim, filename);
      if (strlen(filename)==0) sprintf(filename,"xy.txt");

      // skiplines
      RightFL(line1,",",line1,InputLineLength);
      LeftFL(line1,",",line2,InputLineLength);
      EditLine (Trim, line2);
      if (strlen(line2)>0)
        skiplines = longexpr( line2, pstatus ); if (*pstatus!=Success) return;

      // skipchars
      RightFL(line1,",",line1,InputLineLength);
      LeftFL(line1,",",line2,InputLineLength);
      EditLine (Trim, line2);
      if (strlen(line2)>0)
        skipchars = longexpr( line2, pstatus ); if (*pstatus!=Success) return;

      // swap
      RightFL(line1,",",line1,InputLineLength);
      LeftFL(line1,",",line2,InputLineLength);
      EditLine (Trim, line2);
      if (strlen(line2)>0)
        swap = (int) longexpr( line2, pstatus ); if (*pstatus!=Success) return;

      // colx 
      RightFL(line1,",",line1,InputLineLength);
      LeftFL(line1,",",line2,InputLineLength);
      EditLine (Trim, line2);
      if (strlen(line2)>0)
        colx = longexpr( line2, pstatus ); if (*pstatus!=Success) return;

      // coly
      RightFL(line1,",",line1,InputLineLength);
      LeftFL(line1,",",line2,InputLineLength);
      EditLine (Trim, line2);
      if (strlen(line2)>0)
        coly = longexpr( line2, pstatus ); if (*pstatus!=Success) return;

      // readlines 
      RightFL(line1,",",line1,InputLineLength);
      LeftFL(line1,",",line2,InputLineLength);
      EditLine (Trim, line2);
      if (strlen(line2)>0)
        readlines = longexpr( line2, pstatus ); if (*pstatus!=Success) return;

      // comment 
      RightFL(line1,",",line1,InputLineLength);
      LeftFL(line1,",",comment,InputLineLength);

      // liml 
      RightFL(line1,",",line1,InputLineLength);
      LeftFL(line1,",",liml,InputLineLength);

      // limc
      RightFL(line1,",",line1,InputLineLength);
      LeftFL(line1,",",limc,InputLineLength);

      if (!ihb[0].Title.I) {
        sprintf(ihb[0].Title.V, "from : %s",filename );
        ihb[0].Title.I = TRUE;
        }

      from_table( radial, radial_dim, offset, dummy, ddummy, 
                  pixsiz, center, SampleDistance, WaveLength,
                  d, 1.0,
  
                  filename, skiplines, skipchars,
                  swap, colx, coly, readlines,
                  comment, liml, limc,
 
                  TRUE, pcb->TestBit, pstatus );

      if (*pstatus!=Success) return;

      break;

    case Function :
      // Function,expression
      RightFL(arg,",",line1,InputLineLength);
      EditLine (Trim, line1);
      if (strlen(line1)==0) sprintf(line1,"%g",dummy);

      from_function( radial, radial_dim, offset, dummy, ddummy,
                     pixsiz, center, SampleDistance, WaveLength,
                     d, 1.0,
                     line1,
                     TRUE, pcb->TestBit, pstatus );

      break;

    default :
       fprintf(ERROUT,"Unknown distribution function : %s\n", arg); 
       *pstatus=Failed; return;

  }

  ori   = ihb[0].Orientation.V;
  K     = (double) WAVENUMBER(ihb[0].WaveLength.V);
  rot1  = (double) ihb[0].DetRot[1].V;
  rot2  = (double) ihb[0].DetRot[2].V;
  rot3  = (double) ihb[0].DetRot[3].V;
  pol   = (double) Pol.V;
  pchi  = (double) PChi.V;
  ppsi  = (double) PPsi.V;
  ff    = (double) 1.0;
  dopol = DoPol.V;

  // waxs init
  if (waxs_Init ( &wparams, ori, K, rot1, rot2, rot3 )) {
    fprintf(ERROUT,"Waxs initialization error\n") ;
    *pstatus=OutOfRange; return;
  }

  // polarization init
  if (polarization_Init( &pparams, ori,
      K, rot1, rot2, rot3, pol, pchi, -ppsi, 1.0, 0)) {
    fprintf(ERROUT,"Polarization initialization error\n") ;
    *pstatus=OutOfRange; return;
  }

  if (pcb->TestBit) {
    polarization_PrintParams ( stdout, pparams );
    printf("\n"); }

  /* loop over the output array */
  W0.s_2 = Wmin_2; for (i=0;i<imax;i++) f_2[i]=f1_2[i];
  for (i_2=Imin_2;i_2<=Imax_2;i_2++) {
    pI0Data = ABSPTR(I0Data,I0Dim_1,I0Dim_2,Imin_1,i_2);
    W0.s_1 = Wmin_1; for (i=0;i<imax;i++) f_1[i]=f1_1[i];
    for (i_1=Imin_1;i_1<=Imax_1;i_1++) {

      if (ihb[0].Projection.V == IO_ProSaxs)
        W = waxs_S2Sp( &wparams, NULL, W0 ); // calculate Waxs-coordinate
      else W = W0;

      /* do something with the data */
      radius = sqrt(W.s_1*W.s_1+W.s_2*W.s_2);
      f_r = S2INDEX(radius, offset, pixsiz, center, SampleDistance, WaveLength);
 
      if (Ipol2ld    ( radial, radial_dim, 1,
                       dummy, ddummy, f_r, 0, &Value) ) {

        if (dopol)
          Value = Value * (float) polarization_factor ( &pparams, W, IO_ProWaxs );

        UPDATE( *pI0Data, Value, I0Dummy, I0DDummy );

      }

      pI0Data++;
      W0.s_1+=DW_1; for (i=0;i<imax;i++) { f_1[i]+=Df_1[i]; }
    } /* for i_1 ... */
 
    W0.s_2+=DW_2; for (i=0;i<imax;i++) { f_2[i]+=Df_2[i]; }
  } /* for i_2 ... */ 

  if (radial) free(radial);

  if (!norm) {
    // unnormalize 2d pattern 
    unnormalize2d( ihb[0].Data, ihb[0].Dim[1], ihb[0].Dim[2],
                   ihb[0].Offset[1].V, ihb[0].Offset[2].V, 
                   ihb[0].Dummy.V, ihb[0].DDummy.V,
                   ihb[0].PixSiz[1].V, ihb[0].PixSiz[2].V, 
                   ihb[0].Center[1].V, ihb[0].Center[2].V,
                   ihb[0].SampleDistance.V, i0*trm,
                   pcb->TestBit );
    // write intensity and time into header
    WriteHeaderFloat(pcb,0,ihb,KValueI0,i0, pstatus); 
    if (*pstatus!=Success) return; 
    WriteHeaderFloat(pcb,0,ihb,KValueI1,i0*trm, pstatus);
    if (*pstatus!=Success) return;
    WriteHeaderFloat(pcb,0,ihb,KValueTime,time, pstatus);
    if (*pstatus!=Success) return;
  }

} /* sphere2saxs*/

/*---------------------------------------------------------------------------
user_io
Do all the keyboard io and return cb, and ib
---------------------------------------------------------------------------*/
void user_io(CmdBlk * pcb, ImgBlk * ib, int * pstatus)
{
  char  progfn[IO_len];
  char  tmp[InputLineLength];
  ImgHeadBlk ihb[BlockNum];

  float ROff_1, RPs_1, UOff_1, UPs_1;
  float ROff_2, RPs_2, UOff_2, UPs_2;

  int distribution_type;
  char line1[InputLineLength];

  long num[BlockNum];
  long tmp_long;

  IO_long  *pNorm;
  IO_line  *pRad;
  IO_flexp *pI0, *pTrm, *pDre, *pVfr, *pD;

  IO_long  *pDoPol;
  IO_flexp *pPol, *pPChi, *pPPsi;

  /*--- Determine program name without directory ---*/
  filename_name ( progfn, IO_len,  pcb->argv[0] );

  /* show usage if no arguments are given */
  if (pcb->argc<=1) printf("Usage: %s %s\n",progfn,Usage);

  /*--- Write name of program ---*/
  printf("\n");
  printf("   %s %s\n",progfn,Version);
  printf("\n");

  ArgvBlocks ( pcb, ib, ihb, sphere2saxs_init, NULL, 0, BlockNum-1, pstatus);
  if (*pstatus!=Success) return;

  num[0] = ib[0].First.V;

  /*--- Argument  : pixel size 1 */
  argv_flexp(pcb,"Pixel size 1 [m]",&ib[0].PixSiz[1],"100e-6",pstatus);
  if (*pstatus!=Success) return;

  /*--- Argument  : pixel size 2 */
  argv_flexp(pcb,"Pixel size 2 [m]",&ib[0].PixSiz[2],"100e-6",pstatus);
  if (*pstatus!=Success) return;

  /*--- Argument  : center 1 */
//  tmp_long = longkeyvalue(ib[0].Dim[1].V,pcb,ib,num,0,pstatus);
//  sprintf(tmp,"%g",SETCTRDEF(tmp_long));
//  if (*pstatus!=Success) return; 
  sprintf(tmp,"%g",SETCTRDEF(ihb[0].Dim[1]));
  argv_flexp(pcb,"Center 1 [pixel coordinate]",
                 &ib[0].Center[1],tmp,pstatus);
  if (*pstatus!=Success) return;

  /*--- Argument  : center 2 */
//  tmp_long = longkeyvalue(ib[0].Dim[2].V,pcb,ib,num,0,pstatus);
//  sprintf(tmp,"%g",SETCTRDEF(tmp_long));
//  if (*pstatus!=Success) return; 
  sprintf(tmp,"%g",SETCTRDEF(ihb[0].Dim[2]));
  argv_flexp(pcb,"Center 2 [pixel coordinate]",
                 &ib[0].Center[2],tmp,pstatus);
  if (*pstatus!=Success) return;

  /*--- Argument  : sample distance */
  argv_flexp(pcb,"Sample distance [m]",&ib[0].SampleDistance,"10.0",pstatus);
  if (*pstatus!=Success) return;

  /*--- Argument  : wavelength */
  argv_flexp(pcb,"Wavelength [m]",&ib[0].WaveLength,"0.1e-9",pstatus);
  if (*pstatus!=Success) return;

  /* --- End of geometric setup */
  BalanceHeaderBlock ( pcb, ib, 0, num, ihb, pstatus );
  if (*pstatus!=Success) return;
  GetReference(pcb->RSys.V,0,ihb,&ROff_1,&ROff_2,&RPs_1,&RPs_2,pstatus );
  if (*pstatus!=Success) return;
  GetReference(pcb->USys.V,0,ihb,&UOff_1,&UOff_2,&UPs_1,&UPs_2,pstatus );
  if (*pstatus!=Success) return;

  /*--- Argument  : Radius of particles */
  if (pcb->Prompt.V) {
    printf("Particle model:\n");
    printf("  mono disperse        : radius [m]\n");
    printf("  Schulz distribution  : \"Schulz, radius [m], Z\" with Z>-1\n");
    printf("  from File            : \"File,name,skipl,skipc,swap,colx,coly\"\n");
    printf("              defaults : name[\"xy.txt\"], skipl[0], skipc[0]\n");
    printf("                         swap[1], colx[1], coly[2]\n");
    printf("  from Function        : \"Function,expression(_s,_q)\"\n");
    printf("                         where _s (_q=2*pi*_s) is the\n");
    printf("                         scattering vector\n");
    printf("                         s=2*sin(Theta)/lambda [1/nm]\n");
  }

  pRad = (IO_line*) option_parameter_search( SRad, pcb, pstatus );
  if (*pstatus!=Success) return;
  argv_line(pcb," Particle radius [m] or distribution function",
    pRad,pRad->V,pstatus);
  if (*pstatus!=Success) return;

  LeftFL( pRad->V,",",line1,InputLineLength);
  distribution_type = string2distributiontype( line1 );

  pDre = (IO_flexp*) option_parameter_search( SDre, pcb, pstatus );
  if (*pstatus!=Success) return;
  renorm_flexp( pDre );
  pVfr = (IO_flexp*) option_parameter_search( SVfr, pcb, pstatus );
  if (*pstatus!=Success) return;
  renorm_flexp( pVfr );
  pD = (IO_flexp*) option_parameter_search( SD, pcb, pstatus );
  if (*pstatus!=Success) return;
  renorm_flexp( pD );

  if (distribution_type != File) {
    /*--- Argument  : DeltaRhoElektron */
    argv_flexp(pcb, " Electron density diff. between particles and matrix [#/m^3]",
      pDre,pDre->V,pstatus);
    if (*pstatus!=Success) return;

    /*--- Argument  : Volume fraction */
    argv_flexp(pcb," Volume fraction",
      pVfr,pVfr->V,pstatus);
    if (*pstatus!=Success) return;

    /*--- Argument  : Sample thickness */
    argv_flexp(pcb," Sample thickness [m]",
      pD,pD->V,pstatus);
    if (*pstatus!=Success) return; 
  }

  /*---            Polarization */
  pDoPol = (IO_long*) option_parameter_search( SDoPol, pcb, pstatus );
  if (*pstatus!=Success) return;

  if (pDoPol->V) {
  /*---             Description of polarization */
    if (pcb->Prompt.V) {
      printf("Polarization:\n");
      printf(" Description of the polarization by the Poincar sphere.\n");
      printf(" The polarization state of the primary beam is defined by \n");
      printf(" three parameters\n");
      printf("  Pol : degree of polarization (0<=Pol<=1)\n");
      printf("  PChi: ellipticity (-45_deg<=PChi<=+45_deg)\n");
      printf("        PChi=-45_deg left hand (cw) circular polarization\n");
      printf("        PChi<0 left hand polarization\n");
      printf("        PChi==0 linear polarization\n");
      printf("        PChi>0 right hand polarization\n");
      printf("        PChi=45_deg right hand (ccw) circular polarization\n");
      printf("  PPsi: rotation of the polarization ellipse/axis (0<=PPsi<180_deg)\n");
      printf("        The angle PPsi is measured ccw around axis x_3 starting\n");
      printf("        at axis x_1.\n");
    }

    /*--- Argument  : Polarization */
    pPol = (IO_flexp*) option_parameter_search( SPol, pcb, pstatus );
    if (*pstatus!=Success) return;
    argv_flexp(pcb," Degree of polarization (0<=Pol<=1)",
               pPol,pPol->V,pstatus);
    if (*pstatus!=Success) return;

    /*--- Argument  : Ellipticity */
    pPChi = (IO_flexp*) option_parameter_search( SPChi, pcb, pstatus );
    if (*pstatus!=Success) return;
    argv_flexp(pcb," Ellipticity of polarization ellipse (-45_deg<=PChi<=+45_deg)",
               pPChi,pPChi->V,pstatus);
    if (*pstatus!=Success) return;

    /*--- Argument  : Rotation angle of polarization ellipse around axis 3 */
    pPPsi = (IO_flexp*) option_parameter_search( SPPsi, pcb, pstatus );
    if (*pstatus!=Success) return;
    argv_flexp(pcb," Rotation angle of polarization ellipse (0<=PPsi<180_deg)",
               pPPsi,pPPsi->V,pstatus);
    if (*pstatus!=Success) return;
  }

  /*--- Argument  : Normalization? */
  pNorm = (IO_long*) option_parameter_search( SNorm, pcb, pstatus );
  if (*pstatus!=Success) return;
  argv_flag(pcb,"Should the data be normalized?",
    pNorm,pNorm->V,pstatus);
  if (*pstatus!=Success) return;

  pI0 = (IO_flexp*) option_parameter_search( SI0, pcb, pstatus );
  if (*pstatus!=Success) return;
  renorm_flexp( pI0 );
  pTrm = (IO_flexp*) option_parameter_search( STrm, pcb, pstatus );
  if (*pstatus!=Success) return;
  renorm_flexp( pTrm );

  if (!pNorm->V) {
    /*--- Argument  : I0 */
    argv_flexp(pcb,"Incoming number of photons [ph]",
      pI0,pI0->V,pstatus);
    if (*pstatus!=Success) return;

    /*--- Argument  : Trm */
    argv_flexp(pcb,"Sample transmission",
      pTrm,pTrm->V,pstatus);
    if (*pstatus!=Success) return;
    }

  printf("\n");
  if (ib[0].Name.I)           printf("o/p file               : %s\n",
    ib[0].Name.V);
  if (ib[0].Dummy.I)          printf("output dummy           : %s\n",
    ib[0].Dummy.V);
  if (ib[0].Dim[1].I)         printf("output dimension 1     : %s\n",
    ib[0].Dim[1].V);
  if (ib[0].Dim[2].I)         printf("output dimension 2     : %s\n",
    ib[0].Dim[2].V);
  if (ib[0].PixSiz[1].I)      printf("pixel size 1           : %s m\n",
    ib[0].PixSiz[1].V);
  if (ib[0].PixSiz[2].I)      printf("pixel size 2           : %s m\n",
    ib[0].PixSiz[2].V);
  if (ib[0].Center[1].I)      printf("center 1               : %s\n",
    ib[0].Center[1].V);
  if (ib[0].Center[2].I)      printf("center 1               : %s\n",
    ib[0].Center[2].V);
  if (ib[0].SampleDistance.I) printf("sample distance        : %s m\n",
    ib[0].SampleDistance.V);
  if (ib[0].WaveLength.I)     printf("wavelength             : %s m\n",
    ib[0].WaveLength.V);
  if (pRad->I)                printf("particle model         : %s\n",
    pRad->V);
  if (pDre->I)                printf(" electr. dens. diff.   : %s /m^3 %s\n",
    pDre->V,(distribution_type == File)?"(ignored)":"" );
  if (pVfr->I)                printf(" volume fraction       : %s %s\n",
    pVfr->V,(distribution_type == File)?"(ignored)":"" );
  if (pD->I)                  printf("sample thickness       : %s m\n",
    pD->V);
  if (pDoPol->V) {
    if (pPol->I)                printf("degree of polarization : %s\n",
      pPol->V);
    if (pPChi->I)               printf("ellipticity angle pchi : %s\n",
      pPChi->V);
    if (pPPsi->I)               printf("orientation angle ppsi : %s\n",
      pPPsi->V);
  }
  if (pNorm->I)               printf("normalization          : %s\n",
    (pNorm->V)?"yes":"no");
  if (pI0->I)                 printf("incoming photons       : %s ph\n",
    pI0->V);
  if (pTrm->I)                printf("transmission           : %s\n",
    pTrm->V);

  printf("\n");

  if (pcb->TestBit) {
    PrintBlocks ( pcb, ib );
    printf("\n"); }

  return;
}

/*---------------------------------------------------------------------------
main
---------------------------------------------------------------------------*/

#if MAKE_FUNCTION
# define MAIN main_sphere2saxs
#else
# define MAIN main
#endif

int MAIN (int argc, char *argv[])
{
  CmdBlk cb;                /* command block  */
  ImgBlk ib[BlockNum];      /* image blocks */

  int status;
  int arg_no = 0;

 /* Init options, control block and image blocks */
  InitOptions( Usage, Version, NULL, TRUE, &cb, ib, BlockNum );

 /* Define special options */
  option_define ( IO_tpflexp,  SDre,            "10.0/nm3", &cb );
  option_define ( IO_tpflexp,  SVfr,                "0.01", &cb );
  option_define ( IO_tpflexp,  SD,                  "1e-4", &cb );
  option_define ( IO_tpflexp,  SI0,                 "1e13", &cb );
  option_define ( IO_tpflexp,  STrm,                 "1.0", &cb );
  option_define ( IO_tpflexp,  STim,                 "1.0", &cb );
  option_define ( IO_tpline,   SRad,                "1e-7", &cb );
  option_define ( IO_tpflag,   SNorm,               "TRUE", &cb );

  option_define ( IO_tpflag,   SDoPol,              "TRUE", &cb );
  option_define ( IO_tpflexp,  SPol ,                "1.0", &cb );
  option_define ( IO_tpflexp,  SPChi,            "0.0_deg", &cb );
  option_define ( IO_tpflexp,  SPPsi,            "0.0_deg", &cb );

 /* Default reference system is SAXS */
  cb.RSys.V = IO_Saxs; /* default */ 

 /* Read options from argument list */
  ReadOptions( argv, &arg_no, &cb, ib, &status);

  /* USER KEYBOARD I/O */
  if (status==Success) {
    argv_start ( &cb, 1 );
    user_io( &cb, ib, &status);
    argv_end( &cb ); /* must be called after user_io */
  }

  /* SEQUENCE CALCULATION */
  if (status==Success)
    IMAGELOOP( &cb, ib, sphere2saxs, sphere2saxs_init, NULL, TRUE, &status );

  return( ReportSaxsStatus( status, 0 ) );

} /* MAIN */

