/***************************************************************************/
/* Written 1994++ by Peter Boesecke                                        */
/* Copyright (C) 2011 European Synchrotron Radiation Facility              */
/*                       Grenoble, France                                  */
/*                                                                         */
/*    Principal authors: Peter Boesecke  (boesecke@esrf.eu)                */
/*                                                                         */
/*    This program is free software: you can redistribute it and/or modify */
/*    it under the terms of the GNU General Public License as published by */
/*    the Free Software Foundation, either version 3 of the License, or    */
/*    (at your option) any later version.                                  */
/*                                                                         */
/*    This program is distributed in the hope that it will be useful,      */
/*    but WITHOUT ANY WARRANTY; without even the implied warranty of       */
/*    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the        */
/*    GNU General Public License for more details.                         */
/*                                                                         */
/*    You should have received a copy of the GNU General Public License    */
/*    along with this program.  If not, see <http://www.gnu.org/licenses/>.*/
/***************************************************************************/
/*+++
1 saxs_curve
  Projects all rows between alf1 and alf2 to a single line and
  writes scattering vector, intensity and sigma to an ascii output file.
  alf1 and alf2 must be given in rad.

2 PURPOSE
  Projects all rows between alf1 and alf2 to a single line and
  writes intensity and sigma against the scattering vector or angle, 
  depending on the header key AxisType_1.

  If AxisType_1 is Angle (option -i1type Angle) the default reference 
  system is Normal (option -rsys Normal). The option -waxs/+waxs is ignored. 
  The default column label is angle/rad. If the scale factor is 180/2/pi 
  (option -scf 180/2/pi) the default x-column label is angle/deg. 
  If the reference system is different from Normal the default x-column 
  label is x or x/<scalefactor>, if scalefactor is different from 1. 
  The x-coordinate is calculated according to the chosen reference system and
  is multiplied with the scalefactor (option -scf <scalefactor>).

  If AxisType_1 is Numerator (option -i1type Numerator) the default reference
  system is Normal (option -rsys Normal). The option -waxs/+waxs is ignored.
  The default x-column label is x or x/<scalefactor>, if scalefactor is 
  different from 1. The x-coordinate is calculated according to the chosen 
  reference system and is multiplied with the scalefactor 
  (option -scf <scalefactor>).

  If AxisType_1 is Distance (option -i1type Distance) the default reference    
  system is Saxs (option -rsys Saxs). The projection type can either be 
  Saxs or Waxs.  The option +waxs (default) writes the exact scattering 
  vector into the x-column. The option -waxs ignores the Projection Type 
  (option -i1pro Saxs|Waxs). In the case of ProjectionType Saxs 
  (option -i1pro Saxs|Waxs) the x-column is calculated in small angle 
  approximation (sin x ~= x ~= tan x).

  Each line of the input image must correspond to an azimuthal 
  average/projection, i.e. the scattering vectors of the n-th element in 
  each line must be identical like for files created with saxs_angle, 
  saxs_arc, saxs_row, saxs_col.

  All lines of the input image between alf1 and alf2 are averarged. Only
  a single line per input image will be written. The angle is calculated
  using the Normal reference system. The default is the maximum range.

  Calculation:       I0 = I1
  Error propagation: V0 = V1

  Arguments:
  saxs_curve [options] <i1nam> <onam> <i1fst> <i1lst> <i1inc> 
             <odum> <odim1> <odim2> <ofac> <ocon>"

  Defaults:
  <input file name>   : input.edf 
  <output file name>  : output.edf
  <first image>       : <first image number in input file>
  <last image>                              : <last image number in input file>
     if argument list ends with first image : <first image>
  <increment>         : 1
  <dummy>             : <dummy value in first image of input file>
  <dimension 1>       : <horizontal dimension of first image in input file>
  <dimension 2>       : <vertical dimension of first image in input file>
  <output const>      : 0.0
  <output factor>     : 1.0

  Specific options:
  -alf1 <start angle/rad> : default is minimum angle
  -alf2 <stop angle/rad>  : default is maximum angle
  -ext 'extension'        : extension of the output file (default: ".txt")
  -hdmk 'marker'          : first character in a header line (default: "#")
  -spc 'spacer'           : spacer string between columns (default: "  \t")
  -hedl 'labels'          : column labels, = is wildcard (default: s I stddev)

  +-head                  : enables/disables the header printing (default: +head)
  +-hd1                   : enables/disables column labels (default: +hd1)
  +-gpl                   : enables/disables gnuplot compatibility (default: +gpl)
  -scf <factor>           : multiplication of scattering vector s (default: 1),
                            -scf 2_pi converts s*nm to q*nm
  +-hist                  : writes/do not write history into header (default: +hist)
  +-waxs                  : calculate exact scattering vector length from 
                            saxs-coordinate (default: +waxs) 

  Wildcard:
  A "=" in the argument list chooses the default.

2 HISTORY
  2016-03-25  PB V4.76 from saxs_curves V4.75
  2016-04-06  PB V4.77 Integration range of input axis 2 is split modulo 2 pi: 
                       -i1axis = Angle (default is read from header key AxisType_2)
                       For other axis types the integration is done from alf1 to alf2,
                       e.g. -i1axis = Distance.
                       In averaging mode (+i1ave, default) not more than a single turn 
                       is contributing to the variance.
                       If alf1 and alf2 are identical only dummies are written.
                       Inverting the values of alf1 and alf2 changes the sign of
                       the output. 
  2016-04-07  PB V4.78 option gpl added (plotting multi-scan files with gnuplot)
                       Example 1: Plotting spec scan #S 1 to #S 5
                       > saxs_curve +spec sphere_arc.edf
                       gnuplot> plot 'sphere_arc.txt' i "S 1" with  errorlines
                       gnuplot> replot 'sphere_arc.txt' i "S 2" with  errorlines
                       ...
                       gnuplot> replot 'sphere_arc.txt' i "S 5" with  errorlines

                       Example 2: Plotting scan block 0 to 4 (not in spec format)
                       > saxs_curve -spec -head sphere_arc.edf
                       gnuplot> plot 'sphere_arc.txt' i 0 with  errorlines
                       gnuplot> replot 'sphere_arc.txt' i 1 with  errorlines
                       ...
                       gnuplot> replot 'sphere_arc.txt' i 4 with  errorlines
  2016-04-08  PB V4.79 +gpl adds always separator lines between scans, independent
                       of +-hd1.

---*/
# define Version  "saxs_curve V4.79 2016-04-08, Peter Boesecke"

# include <errno.h>
# include <stdio.h>
# include <unistd.h>
# include <fcntl.h>

# include "SaxsPrograms.h"
# include "strlib.h"

# define FREE( a) { if ( a) { free( a); a = NULL; } }

/* Defaults */
# define Usage "[options] \n\
                <i1nam> <onam> <i1fst> <i1lst> <i1inc> \n\
                <odum> <odim1> <odim2> <ofac> <ocon>"

// line
# define SHdmk      "hdmk"
# define SSpacer    "spc"
# define SHeadL     "hedl"
# define SExtension "ext"
# define SKey       "key"  /* header key to write */

// flag
# define SHist  "hist" // write history
# define SHead  "head" // write header
# define SAll   "all"  // write all values
# define SWaxs  "waxs" // calculate 2*sin(Theta) from tan(2*Theta)
# define SSpec  "spec" // write spec type header 
# define Si1Head    "hd1"  // write labels 
# define SGnuPlot   "gpl"  // compatibility with gnuplot
// flexp
# define SScf   "scf"  // scale factor for header column (default 1.0)
// lvexp
# define SHlen      "len"  // minimum number of header lines (if +head)

# define BlockNum 2       /* 1 input sequence + 1 output sequence */

const float twopi=(float) (2.0*SAXS_PI);
const char *delim = " ";
const char *speclabel = "#L";

/*----------------------------------------------------------------------------
NAME
  write_history

DESCRIPTION
  Writes the full history into the output file 'out'

RETURN VALUE
  Number of written lines

HISTORY
  2007-11-12 PB from saxs_ascii
----------------------------------------------------------------------------*/
long write_history( FILE * out, CmdBlk * pcb, long num[], ImgHeadBlk ihb[],
                    int spec, int * pstatus )
{
  long cnt=0;
  IO_line Hdmk;

  const char * header_key="header_tmp";
  const char * key, * value;
  const char * hdmk;

  int ErrorValue;

  *pstatus = Failed;

  /* write history to a separate header */

  if ( !edf_new_header( header_key ) ) return( cnt );
  edf_history_write_header ( header_key, SaxsHistoryKey,
                             &ErrorValue, pstatus );
  SetSaxsErrorExternal( ErrorValue, ReportSaxsImageError );
  if (*pstatus!=Success) { edf_free_header( header_key ); return( cnt ); }

  Hdmk = option_line ( SHdmk, pcb, pcb->ib, num, 1, pstatus );
  if (*pstatus!=Success) { edf_free_header( header_key ); return( cnt ); }

  edf_first_header_element ( header_key, &key, &value, &ErrorValue, pstatus );
  SetSaxsErrorExternal( ErrorValue, ReportSaxsImageError );
  if (*pstatus!=Success) { edf_free_header( header_key ); return( cnt ); }

  if (spec) hdmk="# "; else hdmk=Hdmk.V;

  while (key) {
    fprintf(out,"%s%s = %s \r\n", hdmk, key, value);
    cnt++;
    edf_next_header_element ( header_key, &key, &value, &ErrorValue, pstatus );
    SetSaxsErrorExternal( ErrorValue, ReportSaxsImageError );
    if (*pstatus!=Success) { edf_free_header( header_key ); return( cnt ); }
  }

  if ( !edf_free_header( header_key ) ) return( cnt );

  *pstatus = Success;
  return(cnt);

} /* write_history */

/*----------------------------------------------------------------------------
NAME
  write_spechead

DESCRIPTION
  Writes spec header values of block 0

RETURN VALUE
  Number of written lines

HISTORY
  2008-11-30 PB
----------------------------------------------------------------------------*/
long write_spechead( FILE * out, CmdBlk *pcb, long num[], ImgHeadBlk ihb[],
                    int *pstatus )
{ long cnt=0;
  int blkno=0;
  fprintf(out,"\r\n");cnt++;
  fprintf(out,"#S %d %s %s \r\n", ihb[blkno].ImNum, 
    ihb[blkno].Time.I?ihb[blkno].Time.V:"",
    ihb[blkno].Title.I?ihb[blkno].Title.V:"");cnt++;
  *pstatus = Success;
  return(cnt);
} // write_spechead

/*----------------------------------------------------------------------------
NAME
  write_head

DESCRIPTION
  Writes header values of block 0

RETURN VALUE
  Number of written lines

HISTORY
  2007-11-12 PB from saxs_ascii
----------------------------------------------------------------------------*/
long write_head( FILE * out, CmdBlk *pcb, long num[], ImgHeadBlk ihb[],
                 int spec, int *pstatus )
{ long cnt=0;
  IO_long cntmax, Hist;
  IO_line Hdmk, KeyList;

  char Key[IO_len], KeyValue[IO_len];
  int blkno=0, parno;

  const char *hdmk;

  Hdmk = option_line ( SHdmk, pcb, pcb->ib, num, 1, pstatus );
  if (*pstatus!=Success) return( cnt );

  KeyList = option_line ( SKey, pcb, pcb->ib, num, 1, pstatus );
  if (*pstatus!=Success) return( cnt );

  Hist = option_flag ( SHist, pcb, pstatus );
  if (*pstatus!=Success) return( cnt );

  cntmax = option_long ( SHlen, pcb, pcb->ib, num, 1, pstatus ); // header lines
  if (*pstatus!=Success) return( cnt );

  if (spec) hdmk="# "; else hdmk=Hdmk.V;

  fprintf(out,"%s%s = % d \r\n", hdmk, KImage, ihb->ImNum); cnt++;
  fprintf(out,"%s%s = % d \r\n", hdmk, "Dim_1", ihb->Dim[1]); cnt++;
  if (ihb->Dim[2]>1)
    fprintf(out,"%s%s = % d \r\n", hdmk, "Dim_2", ihb->Dim[2]); cnt++;
  if (ihb->Dummy.I) { 
    if (DUMMYDEFINED( ihb->Dummy.V, ihb->DDummy.V)) {
      fprintf(out,"%s%s = % e \r\n",
        hdmk, KDummy, ihb->Dummy.V); cnt++; 
    if (ihb->DDummy.I)       { fprintf(out,"%s%s = % e \r\n",
      hdmk, KDDummy, ihb->DDummy.V); cnt++; }
    }
  }
  if (ihb->SampleDistance.I) { fprintf(out,"%s%s = % e \r\n",
    hdmk, KSampleDistance, ihb->SampleDistance.V); cnt++; }
  if (ihb->WaveLength.I)   { fprintf(out,"%s%s = % e \r\n",
    hdmk, KWaveLength, ihb->WaveLength.V); cnt++; }
  if (ihb->Title.I)        { fprintf(out,"%s%s = % s \r\n",
    hdmk, KTitle, ihb->Title.V); cnt++; }
  if (ihb->Time.I)         { fprintf(out,"%s%s = % s \r\n",
    hdmk, KTime, ihb->Time.V); cnt++; }

  if (strlen(KeyList.V)) {
    parno = 1;
    while (strlib_param ( Key, IO_len, KeyList.V, ',', parno++ )) {
      if (strlen(Key)) {
        ReadHeaderLine          ( pcb, 1, ihb, Key, KeyValue, pstatus );
        if (*pstatus==Success)fprintf(out,"%s%s = %s \r\n",hdmk,Key,KeyValue);
      }
    }
  }

  if (Hist.V)
    // write history
    cnt += write_history( out, pcb, num, ihb, spec, pstatus );
  if (*pstatus != Success) return( cnt );

  // write empty lines
  for (;cnt<cntmax.V-1;cnt++) {
    fprintf(out,"%s\r\n", hdmk);
  }

  return( cnt );

} // write_head

/*----------------------------------------------------------------------------
NAME
  write_labels

DESCRIPTION
  Writes column labels

HISTORY
  2007-11-12 PB from saxs_ascii
----------------------------------------------------------------------------*/
void write_labels ( FILE * out, float scf, char *hdmk,
                   const char * spacer, const char * headline, int spec,
                   const char * firstlabel,
                   float *data, float *vardat, int dim_1, int dim_2,
                   float off_2, float ps_2, int * pstatus)
{   int i_2;
    char buffer[1024];
    char *label=(char *) NULL, *nextlabel=(char *) NULL;
    char *headl=(char *) NULL;

    if (!hdmk) hdmk="#";

    if (headline) {
      headl = malloc(strlen(headline)+1);
      nextlabel = strcpy(headl,headline);
    }

    // write scattering vector label
    if (nextlabel) label = strlib_tok(nextlabel, delim);
    else label=(char *) NULL;

    if (spec) {
      // print number of columns
      fprintf(out,"#N %d \r\n",1+dim_2*2); // x-axis + dim_2 * ( I + sigma )
      // print speclabel with trailing space
      fprintf(out,"%s ",speclabel);
      // if necessary, remove speclabel from label
      if (label) {
        if (!strncmp(label,speclabel,strlen(speclabel))) {
          // remove it from label and search first label
          label = label+strlen(speclabel);
        } else if (!strncmp(label,hdmk,strlen(hdmk))) {
          // remove hdmk from label
          label = label+strlen(hdmk);
        }
        if (!(strlen(label))) label = strlib_tok((char *) NULL, delim);
      }
    } else if ((label) && (!strncmp(label,hdmk,strlen(hdmk)))) {
      // print hdmk and remove it from label
      label = label+strlen(hdmk);
      fprintf(out,"%s",hdmk);
      if (!(strlen(label))) label = strlib_tok((char *) NULL, delim);
    } else if (!label) fprintf(out,"%s",hdmk);

    if ((!label) || (!strncmp(label,WildCard,strlen(WildCard)))) {
      sprintf(buffer,firstlabel);
      label = &buffer[0];
    }

    fprintf(out,"%s",label);

    // write all data labels
    for (i_2=0; i_2<dim_2; i_2++) {

      // write Intensity label
      if (nextlabel) label = strlib_tok((char *) NULL, delim);
      else label=(char *) NULL;
      if ((!label) || (!strncmp(label,WildCard,strlen(WildCard)))) {
        if (dim_2==1) sprintf(buffer,"I");
        else sprintf(buffer,"I_%d", i_2+1);
        label = &buffer[0];
      }
      fprintf(out,"%s%s",spacer, label); 

      // write sigma label

      if (nextlabel) label = strlib_tok((char *) NULL, delim);
      else label=(char *) NULL;
      if ((!label) || (!strncmp(label,WildCard,strlen(WildCard)))) {
        if (vardat) { // if available, write square root of variance
          if (dim_2==1) sprintf(buffer,"stddev");
          else sprintf(buffer,"stddev_%d",i_2+1);
        } else { // otherwise, write square root of data
          if (dim_2==1) sprintf(buffer,"sqrt(|I|)");
          else sprintf(buffer,"sqrt(|I_%d|)",i_2+1);
        }
        label = &buffer[0];
      }
      fprintf(out,"%s%s",spacer, label); 

    }

    fprintf(out," \r\n");

    if (headl) free (headl);

    *pstatus = Success;

} // write_labels

/*----------------------------------------------------------------------------
NAME
  write_data

DESCRIPTION
  Writes data columns to the output file 'out'

HISTORY
  2007-11-12 PB from saxs_ascii
----------------------------------------------------------------------------*/
void write_data ( FILE * out, int waxs, int all, float K, float scf,
                  const char * spacer,
                  float *data, float *vardat, int dim_1, int dim_2,
                  float off_1, float off_2, float ps_1, float ps_2,
                  float dummy, float ddummy, int * pstatus)
{ int i_1, i_2;
  float O, V, W;

  float *pdata, *pvardat;
  float vardummy;

  int good;

  if ( (DUMMYDEFINED( dummy, ddummy) && (dummy<0.0) ) ) vardummy=dummy;
  else vardummy=VarDummy;

  // data
  for (i_1=0; i_1<dim_1; i_1++) {
    W=WORLD( i_1, off_1, ps_1 );
    if (waxs) { V = 2*K*sin(0.5*atan(W/K)); } else { V = W; }
    O = scf*V;

    // suppress dummy values
    pdata = ABSPTR(data,dim_1,dim_2,i_1,0);
    good = NODUMMY(*pdata,dummy,ddummy);
    if ((all)||(good)) {
      // write s
      fprintf(out,"%g",O);

      // write all rows
      for (i_2=0; i_2<dim_2; i_2++) {
        pdata = ABSPTR(data,dim_1,dim_2,i_1,i_2);
        if (vardat) pvardat = vardat-data+pdata;

        // write I
        fprintf(out,"%s%g",spacer, *pdata);
        // write sigma
        if (vardat) // if available, write square root of variance
         fprintf(out,"%s%g",spacer,(*pvardat<0)?vardummy:sqrt(*pvardat));
        else { // otherwise, write square root of data
          if (good) {
            fprintf(out,"%s%g",spacer,(*pdata<-1)?vardummy:sqrt(fabs(*pdata)));
          } else fprintf(out,"%s%g",spacer,vardummy);
        }
      }

      fprintf(out," \r\n");
    }
  }

  *pstatus = Success;

} // write_data

/*----------------------------------------------------------------------------
NAME
  write_saxs_s

SYNOPSIS

  void write_saxs_s ( CmdBlk * pcb, long num[], ImgHeadBlk ihb[],
                      const char* fname_full,
                      float data[], float vardat[], int dim_1, int dim_2,
                      float off_1, float off_2, float ps_1, float ps_2,
                      float dummy, float ddummy, int * pstatus)

DESCRIPTION
  Write data against scattering vector in first column
----------------------------------------------------------------------------*/
void write_saxs_s ( CmdBlk * pcb, long num[], ImgHeadBlk ihb[], 
                    const char* fname_full, 
                    float data[], float vardat[], int dim_1, int dim_2,
                    float off_1, float off_2, float ps_1, float ps_2,
                    float dummy, float ddummy, int * pstatus)
{ FILE *OutputFileId;

  IO_line Hdmk, Spacer, HeadL;
  IO_float Scf;
  IO_long All, Waxs, Spec, i1Head, Head, GnuPlot;

  int WaxsI, SpecI;
  float K;

  char buffer[1024];

  Head = option_flag ( SHead, pcb, pstatus );
  if (*pstatus!=Success) return;
  Hdmk = option_line ( SHdmk, pcb, pcb->ib, num, 1, pstatus );
  if (*pstatus!=Success) return;
  Spacer = option_line ( SSpacer, pcb, pcb->ib, num, 1, pstatus );
  if (*pstatus!=Success) return;
  HeadL = option_line ( SHeadL, pcb, pcb->ib, num, 1, pstatus );
  if (*pstatus!=Success) return;
  All = option_flag ( SAll, pcb, pstatus );
  if (*pstatus!=Success) return;
  Waxs = option_flag ( SWaxs, pcb, pstatus );
  if (*pstatus!=Success) return;
  Spec = option_flag ( SSpec, pcb, pstatus );
  if (*pstatus!=Success) return;
  Scf = option_float ( SScf, pcb, pcb->ib, num, 1, pstatus );
  if (*pstatus!=Success) return;
  i1Head = option_flag ( Si1Head, pcb, pstatus );
  if (*pstatus!=Success) return;
  GnuPlot = option_flag ( SGnuPlot, pcb, pstatus );
  if (*pstatus!=Success) return;

  /*--- Open output file
    If this is the first image of the sequence, an existing file
    with the same name must be deleted (open mode = "w"),
    in all other cases it must be open to append a line (open mode "a"). */

  if (pcb->TestBit) printf("Opening output file %s\n",fname_full);
  *pstatus = FileOpeningError;
  if (ImageIsFirst( pcb, 0, ihb )) {
    /* 1st image: new write */
    if ((OutputFileId=fopen(fname_full,"wb")) == NULL) return;
  } else {
    /* appending */
    if ((OutputFileId=fopen(fname_full,"ab")) == NULL) return;
  }

  // if HeadL starts with <speclabel> default of specfile is TRUE
  if ((!Spec.I)&&(HeadL.I)) {
    if (!strncmp(HeadL.V,speclabel,strlen(speclabel))) SpecI = TRUE;
    else SpecI = FALSE;
  } else SpecI = Spec.V;

  if ((!i1Head.V)||(Hdmk.I)) SpecI = FALSE;

  WaxsI = Waxs.V;
  /* Do not set WaxsI if the image is already in Waxs Projection */
  if (ihb[0].Projection.V==IO_ProWaxs) WaxsI=0;

  if (WaxsI)
    if (ihb[0].WaveLength.I) K = WAVENUMBER(ihb[0].WaveLength.V);
      else {
      *pstatus=Failed;
      fprintf(ERROUT,
        "Missing parameter WaveLength for scattering vector calculation\n");
      return;
    }

  /* Write spec head */
  if (SpecI) {
    write_spechead( OutputFileId, pcb, num, ihb, pstatus );
    if (*pstatus!=Success) return;
  }

  /* Write head */
  if (Head.V) {
    write_head( OutputFileId, pcb, num, ihb, SpecI, pstatus );
    if (*pstatus!=Success) return;
  }

  /*--- Write column labels */
  if ((SpecI)||(ImageIsFirst( pcb, 0, ihb ))) {
    /* 1st image: write header line */
    if (i1Head.V) {

      if (fabs(Scf.V-1.0)<1e-32) sprintf(buffer,"s*nm");
      else if (fabs(Scf.V-twopi)<0.00253) sprintf(buffer,"q*nm");
      else sprintf(buffer,"(s*%g)*nm",Scf.V);

      write_labels ( OutputFileId, Scf.V, Hdmk.V, Spacer.V, HeadL.V, SpecI,
                     &buffer[0], data, vardat, dim_1, dim_2,
                     off_2, ps_2, pstatus);
      if (*pstatus) return;
    }
  }

  /*--- Write data columns */
  printf("Writing data to %s\n",fname_full);
  write_data( OutputFileId, WaxsI, All.V, K, Scf.V, Spacer.V,
              data, vardat, dim_1, dim_2,
              off_1, off_2, ps_1, ps_2,
              dummy, ddummy, pstatus );
  if (*pstatus) return;

  /*--- Write separating line feeds between scans for gnuplot */
  if (GnuPlot.V) {
    fprintf(OutputFileId,"\r\n");
    if (!SpecI) fprintf(OutputFileId,"\r\n");
  }

// close output text file
  if (pcb->TestBit) printf("Closing file %s\n",fname_full);
  *pstatus = FileClosingError;
  if (fclose(OutputFileId)!=0) return;

  *pstatus = Success;

} // write_saxs_s

/*----------------------------------------------------------------------------
NAME
  write_saxs_a

SYNOPSIS

  void write_saxs_a ( CmdBlk * pcb, long num[], ImgHeadBlk ihb[],
                      const char* fname_full,
                      float data[], float vardat[], int dim_1, int dim_2,
                      float off_1, float off_2, float ps_1, float ps_2,
                      float dummy, float ddummy, int * pstatus)

DESCRIPTION
  Write data against Normal coordinate in first column, e.g. radial angle
----------------------------------------------------------------------------*/
void write_saxs_a ( CmdBlk * pcb, long num[], ImgHeadBlk ihb[],
                    const char* fname_full,
                    float data[], float vardat[], int dim_1, int dim_2,
                    float off_1, float off_2, float ps_1, float ps_2,
                    float dummy, float ddummy, int * pstatus)
{ FILE *OutputFileId;

  IO_line Hdmk, Spacer, HeadL;
  IO_float Scf;
  IO_long All, Waxs, Spec, i1Head, Head, GnuPlot;

  int WaxsI, SpecI;
  float K;

  char buffer[1024];

  Head = option_flag ( SHead, pcb, pstatus );
  if (*pstatus!=Success) return;
  Hdmk = option_line ( SHdmk, pcb, pcb->ib, num, 1, pstatus );
  if (*pstatus!=Success) return;
  Spacer = option_line ( SSpacer, pcb, pcb->ib, num, 1, pstatus );
  if (*pstatus!=Success) return;
  HeadL = option_line ( SHeadL, pcb, pcb->ib, num, 1, pstatus );
  if (*pstatus!=Success) return;
  All = option_flag ( SAll, pcb, pstatus );
  if (*pstatus!=Success) return;
  Waxs = option_flag ( SWaxs, pcb, pstatus );
  if (*pstatus!=Success) return;
  Spec = option_flag ( SSpec, pcb, pstatus );
  if (*pstatus!=Success) return;
  Scf = option_float ( SScf, pcb, pcb->ib, num, 1, pstatus );
  if (*pstatus!=Success) return;
  i1Head = option_flag ( Si1Head, pcb, pstatus );
  if (*pstatus!=Success) return;
  GnuPlot = option_flag ( SGnuPlot, pcb, pstatus );
  if (*pstatus!=Success) return;

  /*--- Open output file
    If this is the first image of the sequence, an existing file
    with the same name must be deleted (open mode = "w"),
    in all other cases it must be open to append a line (open mode "a"). */

  if (pcb->TestBit) printf("Opening output file %s\n",fname_full);
  *pstatus = FileOpeningError;
  if (ImageIsFirst( pcb, 0, ihb )) {
    /* 1st image: new write */
    if ((OutputFileId=fopen(fname_full,"wb")) == NULL) return;
  } else {
    /* appending */
    if ((OutputFileId=fopen(fname_full,"ab")) == NULL) return;
  }

  // if HeadL starts with <speclabel> default of specfile is TRUE
  if ((!Spec.I)&&(HeadL.I)) {
    if (!strncmp(HeadL.V,speclabel,strlen(speclabel))) SpecI = TRUE;
    else SpecI = FALSE;
  } else SpecI = Spec.V;

  if ((!i1Head.V)||(Hdmk.I)) SpecI = FALSE;

  /* Write spec head */
  if (SpecI) {
    write_spechead( OutputFileId, pcb, num, ihb, pstatus );
    if (*pstatus!=Success) return;
  }

  /* Write head */
  if (Head.V) {
    write_head( OutputFileId, pcb, num, ihb, SpecI, pstatus );
    if (*pstatus!=Success) return;
  }

  /*--- Write column labels */
  if ((SpecI)||(ImageIsFirst( pcb, 0, ihb ))) {
    /* 1st image: write header line */
    if (i1Head.V) {

      switch ((int) ihb[1].AxisType[1].V ) {
        case IO_AxisTypeAngle     :
 
            if (pcb->RSys.V == IO_Normal) {
              if (fabs(Scf.V-1.0)<1e-32) sprintf(buffer,"angle/rad"); // angle in rad
              else if (fabs(Scf.V-360.0/twopi)<0.00253) sprintf(buffer,"angle/deg"); // angle in deg
              else sprintf(buffer,"(angle*%g)/rad",Scf.V); // angle in rad multiplied
            } else { // angle in something else
              if (fabs(Scf.V-1.0)<1e-32) sprintf(buffer,"angle");
                else sprintf(buffer,"angle*%g",Scf.V);
            }
          break;

        case IO_AxisTypeNumerator : 
        default : if (fabs(Scf.V-1.0)<1e-32) sprintf(buffer,"x"); // x-axis
                  else sprintf(buffer,"x*%g",Scf.V); // x-axis multiplied 
      }

      write_labels ( OutputFileId, Scf.V, Hdmk.V, Spacer.V, HeadL.V, SpecI,
                     &buffer[0], data, vardat, dim_1, dim_2,
                     off_2, ps_2, pstatus);
      if (*pstatus) return;
    }
  }

  /*--- Write data columns */
  printf("Writing data to %s\n",fname_full);
//  write_data( OutputFileId, WaxsI, All.V, K, Scf.V, Spacer.V,
  write_data( OutputFileId, FALSE, All.V, 1.0, Scf.V, Spacer.V,
              data, vardat, dim_1, dim_2,
              off_1, off_2, ps_1, ps_2,
              dummy, ddummy, pstatus );
  if (*pstatus) return;

  /*--- Write separating line feeds between scans for gnuplot */
  if (GnuPlot.V) {
    fprintf(OutputFileId,"\r\n");
    if (!SpecI) fprintf(OutputFileId,"\r\n");
  }

// close output text file
  if (pcb->TestBit) printf("Closing file %s\n",fname_full);
  *pstatus = FileClosingError;
  if (fclose(OutputFileId)!=0) return;

  *pstatus = Success;

} // write_saxs_a

/*---------------------------------------------------------------------------
NAME
 saxs_w
PURPOSE
  Writes scattering vector/radial angle, intensity and sigma of a row by row file
---------------------------------------------------------------------------*/
void saxs_w (CmdBlk * pcb, long num[], ImgHeadBlk ihb[], 
             const char *what, int * pstatus)
{ const float eps=1e-32;
  int i,imax;
  int j;

  float *I0Data, *E0Data; 
  int   I0Dim_1, I0Dim_2;
  float *pI0Data, *pE0Data;
  float I0Dummy, I0DDummy;
  float VarDDummy=DDSET(VarDummy);

  int   I1Ave;
  float *I1Data, *E1Data;
  int   I1Dim_1, I1Dim_2;
  float I1Dummy, I1DDummy;
  float I1Value, I1Sum, I1Weight;
  float E1Value, E1Sum, E1Weight;

  int i_1, i_2;
  float f_1[BlockNum], f_2[BlockNum];

  float Off_1[BlockNum], Off_2[BlockNum];
  float Ps_1[BlockNum], Ps_2[BlockNum];

  float f1_1[BlockNum], f3_1[BlockNum], Df_1[BlockNum];
  float f1_2[BlockNum], f3_2[BlockNum], Df_2[BlockNum]; 

  int Imin_1, Imin_2, Imax_1, Imax_2;
 
  float Wmin_1, Wmax_1, DW_1;
  float Wmin_2, Wmax_2, DW_2;
  float W2_1, W2_3;

  float *I0Weight=NULL, *E0Weight=NULL;
  int mode=0;

  int InImage=1;
  float alf11,alf12,alf21,alf22,alf31,alf32;
  int alf11I=FALSE,alf12I=FALSE,alf21I=FALSE,alf22I=FALSE;
  int alf31I=FALSE,alf32I=FALSE;
  int sign=1; 
  long fullcircles=0, circles;

  /* output file */
  char fname_full[IO_len];

  IO_line Extension;

  float alf1, alf2;
  int alf1I, alf2I;
  float W1_2, W3_2;
  IO_float Alf1, Alf2;

  float ipolmin_cur;

  char buffer[1024];

  *pstatus = Success;

  imax = pcb->ImgBlkLen;
  printf("\n Calculating ihb[0,% d] = Function(",ihb[0].ImNum);
  for(i=1;i<imax;i++) {
    printf("ihb[% d,% d] ", i, ihb[i].ImNum); 
  }

  printf(")\n\n");

  if (strcmp(what,"a")) {
    // if not "a" use scattering vector
    if (!(pcb->RSys.I)) pcb->RSys.V = IO_Saxs;

    /* check the reference system: Saxs required */
    if (pcb->RSys.V != IO_Saxs) {
      printf("ERROR: Reference system %s required\n",reftostr(IO_Saxs));
      printf("       Start program with option -rsys %s\n",reftostr(IO_Saxs));
    }
  } else {
    // if "a" use IO_Normal coordinate
    if (!(pcb->RSys.I)) pcb->RSys.V = IO_Normal;
  }

  // reference system of coordinate 1 is pcb->RSys.V
  GetReferenceParametersC( pcb->RSys.V, pcb, ihb, 1, 0, imax-1,
                           Off_1, Ps_1, pstatus );
  if (*pstatus!=Success) goto saxs_w_error;

  // reference system of coordinate 2 is IO_Normal
  GetReferenceParametersC( IO_Normal, pcb, ihb, 2, 0, imax-1,
                           Off_2, Ps_2, pstatus );
  if (*pstatus!=Success) goto saxs_w_error;

  // update output image parameters of axis 1
  UpdateImageParametersC ( pcb->RSys.V, pcb, ihb, 1, 1, 0, 1, imax-1,
                          Off_1, Off_2, Ps_1,  Ps_2, pstatus );
  if (*pstatus!=Success) goto saxs_w_error;

  // update output image parameters of axis 2
  UpdateImageParametersC ( IO_Normal, pcb, ihb, 2, 2, 0, 1, imax-1,
                           Off_1, Off_2, Ps_1,  Ps_2, pstatus );
  if (*pstatus!=Success) goto saxs_w_error;

 /* Get options */
  Extension = option_line ( SExtension, pcb, pcb->ib, num, 1, pstatus );
  if (*pstatus!=Success) goto saxs_w_error;
  Alf1 = option_float ( "alf1", pcb, pcb->ib, num, 1, pstatus );
  if (*pstatus!=Success) goto saxs_w_error;
  Alf2 = option_float ( "alf2", pcb, pcb->ib, num, 1, pstatus );
  if (*pstatus!=Success) goto saxs_w_error;

  // Get input image ranges (using Off and Ps)
  GetLinearRange  ( pcb, ihb, 1, 1, 0,  1, imax-1,
                    FALSE, 0.0,  FALSE, 0.0,
                    Off_1, Ps_1, Off_2, Ps_2,
                    f1_1, f3_1, Df_1,
                    &Imin_1, &Imax_1,
                    &Wmin_1, &Wmax_1, &DW_1, 1 ); // coordinate 1

  // Get input world coordinate range
  GetSmallRectangle( pcb, ihb, 2, 2, -1, 1, 1,
                     Off_1, Ps_1, Off_2, Ps_2,
                     &W1_2, &W3_2 );

  // keep original Ipolmin in ipolmin_cur
  ipolmin_cur = Ipolmin(-1);
  Ipolmin(0); // do not use Ipolmin for Projection_1U

  if (pcb->TestBit) {
    printf("Ipolmin has been set from %g to 0\n",ipolmin_cur);
  }

  // Order Alf1 and Alf2
  if (Alf1.I) alf1 = Alf1.V;
  else alf1 = W1_2;
  alf1I = Alf1.I;
  if (Alf2.I) alf2 = Alf2.V;
  else alf2 = W3_2;
  alf2I = Alf2.I;

  if (alf2<alf1) {
    float tmp;
    int tmpI;
    sign=-1;
    tmp=alf1;alf1=alf2;alf2=tmp;
    tmpI=alf1I;alf1I=alf2I;alf2I=tmpI;
  } else sign=1;

  if (pcb->TestBit) {
    printf(" Input Image coordinate 2\n");
    printf("   W1_2 = %15g,   W3_2 = %15g\n",W1_2,W3_2);
    printf("   alf1 = %15g,   alf2 = %15g\n",alf1,alf2);
  }

  /* Axis 2 coordinates are calculated in the IO_Normal reference system. If axis 2 is of
     type angle, it must be periodic with 2 pi. If this is not the case the calculation
     will not be correct because the integration range between alf1 and alf2 is calculated 
     modulo 2 pi.
     If axis 2 is not of type angle, the integration will be done as usual between alf1 
     and alf2. If alf1 and alf2 are not given, the full axis 2 range is the integration range.
     Up to three integration ranges are necessary.
     First alf1 and alf2 are ordered so that alf1<=alf2. The original order is kept 
     as a sign. The angular range between alf1 and alf2 is splitted in additional full
     circles and in a range less or equal to a single circle (alf2-alf1<=2 pi). If 
     alf1 is less than the minimum axis coordinate (W1) the integration range is splitted
     in two sections from W1 to alf2 and from alf1+2_PI to the maximum coordinate (W3).

     a1) alf11..alf12, alf21..alf22: two sections from W1 (minimum) to alf2 
         and alf1+2_PI to W3 (maximum) or 
     a2) alf11..alf12: one section from alf1 (>=W1) to alf2 (<=W3) or
     b) full circles > 1: N>=0 full range from W1 to W1+2_PI
  */
  if ( ((int) ihb[1].AxisType[2].V )==IO_AxisTypeAngle ) {
    // Angular type axis (Split in up to 3 integration ranges)

    // Calculate number of full circles
    fullcircles = (long) ( CEILF((alf2-alf1)/twopi) ) - 1; 
    // the following condition will be TRUE if alf2==alf1
    if (fullcircles<0) fullcircles=0;

    if (fullcircles>0) {
      alf31 = W1_2; alf31I = TRUE;
      alf32 = W1_2+twopi; alf32I = TRUE;
    }

    alf2 -= twopi * fullcircles;

    while (alf2<=W1_2) {
      alf1+=twopi;
      alf2+=twopi;
    }

    if (alf1<W1_2) {
      alf11 = W1_2; alf11I = TRUE;
      alf12 = alf2; alf12I = TRUE;

      alf21 = alf1 + twopi; alf21I = TRUE;
      alf22 = W3_2; alf22I = TRUE;
    } else {
      if (alf2>W1_2+twopi) {
        alf11 = alf1; alf11I = TRUE;
        alf12 = W3_2; alf12I = TRUE; // W1_2+twopi

        alf21 = W1_2; alf21I = TRUE;
        alf22 = alf2 - twopi; alf22I = TRUE;
      } else {
        alf11 = alf1; alf11I = TRUE;
        alf12 = alf2; alf12I = TRUE;
      }
    }
  } else {
    // Not angular type axis (single integration range)
    alf11 = alf1; alf11I = TRUE;
    alf12 = alf2; alf12I = TRUE;
  }

  if (pcb->TestBit) {
    printf(" Integration ranges (fullcircles = %15ld, sign=%15d)\n",fullcircles,sign);
    if (alf11I) printf("   alf11 = %15g\n",alf11);
    else printf("   alf11 = %15s\n","(None)");
    if (alf12I) printf("   alf12 = %15g\n",alf12);
    else printf("   alf12 = %15s\n","(None)");
    if (alf21I) printf("   alf21 = %15g\n",alf21);
    else printf("   alf21 = %15s\n","(None)");
    if (alf22I) printf("   alf22 = %15g\n",alf22);
    else printf("   alf22 = %15s\n","(None)");
    if (alf31I) printf("   alf31 = %15g\n",alf31);
    else printf("   alf31 = %15s\n","(None)");
    if (alf32I) printf("   alf32 = %15g\n",alf32);
    else printf("   alf32 = %15s\n","(None)");
  }

 /* 1 input, 1 output */
  I0Data  = ihb[0].Data;
  E0Data = ihb[0].VarDat;
  I0Dummy = ihb[0].Dummy.V;
  I0DDummy = ihb[0].DDummy.V;
  I0Dim_1  = (int) ihb[0].Dim[1];
  I0Dim_2  = (int) ihb[0].Dim[2];

  I1Ave   = pcb->ib[1].Ave.V;
  I1Data  = ihb[1].Data;
  E1Data  = ihb[1].VarDat;
  I1Dummy = ihb[1].Dummy.V;
  I1DDummy = ihb[1].DDummy.V;
  I1Dim_1  = (int) ihb[1].Dim[1];
  I1Dim_2  = (int) ihb[1].Dim[2];

  /* write output line by output line (usually I0Dim_2 equals 1, but larger if required) */
  for (i_2=0;i_2<I0Dim_2;i_2++) {

    mode = 0; // init Projection_1U

    // a1) Second range (alf21, alf22)

    if (alf21I&alf22I) {
      f1_2[InImage]=INDEX(alf21,Off_2[InImage],Ps_2[InImage]);
      f3_2[InImage]=INDEX(alf22,Off_2[InImage],Ps_2[InImage]);

      if (pcb->TestBit) {
        printf(" a1: Input Image [%d] range (axis 2)\n",InImage);
        printf("   alf21 = %15g => f1_2 = %15g\n",alf21,f1_2[InImage]);
        printf("   alf22 = %15g => f3_2 = %15g\n",alf22,f3_2[InImage]);
      }

      // I0Weight, E0Weight
      if (!I0Weight) {
        I0Weight=(float*) malloc(sizeof(float)*I0Dim_1);
        if (!I0Weight) {
          *pstatus = NotEnoughMemoryAvailable;
          goto saxs_w_error;
        }
      }
      if ((E0Data)&&(!E0Weight)) {
        E0Weight=(float*) malloc(sizeof(float)*I0Dim_1);
      if (!E0Weight) {
          *pstatus = NotEnoughMemoryAvailable;
          goto saxs_w_error;
        }
      }

      /* write projection (usually I0Dim_2 equals 1, but larger if required) */
      pI0Data = ABSPTR(I0Data,I0Dim_1,I0Dim_2,Imin_1,i_2);
      pE0Data = E0Data?E0Data-I0Data+pI0Data:NULL;
      Projection_1U ( pI0Data, pE0Data,
                      I0Weight, E0Weight,
                      I0Dim_1,
                      Imin_1, Imax_1, I0Dummy, 1.0,
                      I1Data, E1Data, I1Dim_1, I1Dim_2,
                      f1_1[1], f3_1[1], Df_1[1], f1_2[1], f3_2[1],
                      I1Dummy, I1DDummy, mode, pcb->TestBit );

      mode=2; // next time add to Projection_1U
    }

    // a2) Principal range (alf11, alf12) (always used)

    f1_2[InImage]=INDEX(alf11,Off_2[InImage],Ps_2[InImage]);
    f3_2[InImage]=INDEX(alf12,Off_2[InImage],Ps_2[InImage]);

    if (pcb->TestBit) {
      printf(" a2: Input Image [%d] range (axis 2)\n",InImage);
      printf("   alf11 = %15g => f1_2 = %15g\n",alf11,f1_2[InImage]);
      printf("   alf12 = %15g => f3_2 = %15g\n",alf12,f3_2[InImage]);
    }

    if (fullcircles>0) {
      // I0Weight, E0Weight
      if (!I0Weight) {
        I0Weight=(float*) malloc(sizeof(float)*I0Dim_1);
        if (!I0Weight) {
          *pstatus = NotEnoughMemoryAvailable;
          goto saxs_w_error;
        }
      }
      if ((E0Data)&&(!E0Weight)) {
        E0Weight=(float*) malloc(sizeof(float)*I0Dim_1);
        if (!E0Weight) {
          *pstatus = NotEnoughMemoryAvailable;
          goto saxs_w_error;
        }
      }
    } else {
      // average here
      if (I1Ave) mode=1;
    }

    /* write projection (usually I0Dim_2 equals 1, but larger if required) */
    pI0Data = ABSPTR(I0Data,I0Dim_1,I0Dim_2,Imin_1,i_2);
    pE0Data = E0Data?E0Data-I0Data+pI0Data:NULL;
    Projection_1U ( pI0Data, pE0Data, 
                    I0Weight, E0Weight,
                    I0Dim_1, 
                    Imin_1, Imax_1, I0Dummy, 1.0,
                    I1Data, E1Data, I1Dim_1, I1Dim_2,
                    f1_1[1], f3_1[1], Df_1[1], f1_2[1], f3_2[1],
                    I1Dummy, I1DDummy, mode, pcb->TestBit );

    mode=2; // next time add to Projection_1U

    // b) Full circles
    if (fullcircles>0) {

      f1_2[InImage]=INDEX(alf31,Off_2[InImage],Ps_2[InImage]);
      f3_2[InImage]=INDEX(alf32,Off_2[InImage],Ps_2[InImage]);

      if (pcb->TestBit) {
        printf(" b: Input Image [%d] range (axis 2) (fullcircles=%15ld)\n",InImage,fullcircles);
        printf("   alf31 = %15g => f1_2 = %15g\n",alf31,f1_2[InImage]);
        printf("   alf32 = %15g => f3_2 = %15g\n",alf32,f3_2[InImage]);
      }

      for (circles=1;circles<fullcircles;circles++) {
        /* write projection (usually I0Dim_2 equals 1, but larger if required) */
        pI0Data = ABSPTR(I0Data,I0Dim_1,I0Dim_2,Imin_1,i_2);
        pE0Data = E0Data?E0Data-I0Data+pI0Data:NULL;
        Projection_1U ( pI0Data, pE0Data,
                        I0Weight, E0Weight,
                        I0Dim_1,
                        Imin_1, Imax_1, I0Dummy, 1.0,
                        I1Data, E1Data, I1Dim_1, I1Dim_2,
                        f1_1[1], f3_1[1], Df_1[1], f1_2[1], f3_2[1],
                        I1Dummy, I1DDummy, mode, pcb->TestBit );
      }

      if (I1Ave) {
        mode=1;
        /*
           reinitialize existing pE0Data and E0Weight,
           not more than a full circle must contribute to variance
         */
        if (pE0Data) {
          for (i_1=0;i_1<I0Dim_1;i_1++) pE0Data[i_1] = VarDummy;
        }
        if (E0Weight) {
          for (i_1=0;i_1<I0Dim_1;i_1++) E0Weight[i_1] = 0.0;
        }
      }

      /* write projection (usually I0Dim_2 equals 1, but larger if required) */
      pI0Data = ABSPTR(I0Data,I0Dim_1,I0Dim_2,Imin_1,i_2);
      pE0Data = E0Data?E0Data-I0Data+pI0Data:NULL;
      Projection_1U ( pI0Data, pE0Data,
                      I0Weight, E0Weight,
                      I0Dim_1,
                      Imin_1, Imax_1, I0Dummy,  1.0,
                      I1Data, E1Data, I1Dim_1, I1Dim_2,
                      f1_1[1], f3_1[1], Df_1[1], f1_2[1], f3_2[1],
                      I1Dummy, I1DDummy, mode, pcb->TestBit );
    } // fullcircles

    // d) Apply sign
    if (sign!=1) {
      float factor=(float) sign;
      pI0Data = ABSPTR(I0Data,I0Dim_1,I0Dim_2,Imin_1,i_2);
      for (i_1=0;i_1<I0Dim_1;i_1++) {
        *pI0Data *= factor;
        pI0Data++;
      }
    }

  } // for i_2

  if (pcb->TestBit) {
    printf("Resetting Ipolmin to %g\n",ipolmin_cur);
  }

  // resetting Ipolmin to original value
  Ipolmin(ipolmin_cur);

  /* Generate filename of ASCII file */

  /* Check for .gz extension (use fname_full as temporary buffer) */
  filename_extension ( fname_full, IO_len, pcb->ib[0].FileName );
  if (!strcmp(fname_full,".gz")) {
    // Remove .gz extension
    filename_body ( fname_full, IO_len,  pcb->ib[0].FileName ); // remove .gz
    filename_body ( fname_full, IO_len,  fname_full ); // remove ext.
  } else {
    /* Generate filename of ASCII file */
    filename_body ( fname_full, IO_len,  pcb->ib[0].FileName ); // remove ext.
  }

  if ( filename_compare ( fname_full, DUMMYFILENAME ) )
    sprintf( fname_full,"%s%s",fname_full,Extension.V); // add extension

  if (strcmp(what,"a")) {
    write_saxs_s ( pcb, num, ihb, 
                   fname_full,
                   I0Data, E0Data, I0Dim_1, I0Dim_2,
                   Off_1[0], Off_2[0], Ps_1[0], Ps_2[0],
                   I0Dummy, I0DDummy, pstatus);
  } else {
    write_saxs_a ( pcb, num, ihb,
                   fname_full,
                   I0Data, E0Data, I0Dim_1, I0Dim_2,
                   Off_1[0], Off_2[0], Ps_1[0], Ps_2[0],
                   I0Dummy, I0DDummy, pstatus);
  }
  if (*pstatus!=Success) goto saxs_w_error;

  FREE(I0Weight);
  FREE(E0Weight);

  return;

saxs_w_error:

  FREE(I0Weight);
  FREE(E0Weight);

  return;

} // saxs_w

/*---------------------------------------------------------------------------
NAME
 saxs_curve
PURPOSE
  Writes intensity and standard error against scattering vector or angle
  of a row by row file into a text file.
---------------------------------------------------------------------------*/
void saxs_curve (CmdBlk * pcb, long num[], ImgHeadBlk ihb[], int * pstatus)
{
  *pstatus = Success;

 /* Check the number of images */
  if (pcb->ImgBlkLen!=2) {
     printf("%d images found, 1 input and 1 output image required\n",
             pcb->ImgBlkLen); *pstatus=Failed; return; }

  switch ((int) ihb[1].AxisType[1].V ) {
    case IO_AxisTypeAngle     : saxs_w (pcb, num, ihb, "a", pstatus); break;
    case IO_AxisTypeNumerator : saxs_w (pcb, num, ihb, "a", pstatus); break;
    case IO_AxisTypeDistance  : saxs_w (pcb, num, ihb, "s", pstatus); break;
    default                   : saxs_w (pcb, num, ihb, "s", pstatus);
  } /* switch */

  return;

} // saxs_curve

/*---------------------------------------------------------------------------
saxs_curve_init

Initializes the output header of each output file before the output
data array is allocated.

The output header is already initialized with the header of the first
image of the first input file sequence. This routine modifies this default.

The number of available blocks is defined by the program and does not need
to be checked here. All input images are open in SaxsImageLoop when a new
output image is created.

saxs_curve:
  Axis 1 output: Same geometry as input image
  Axis 2 output: Change parameters to numerator. Modify geometrical
                 parameters only, if not already set by image block (.I<1)
                 Offset, Center, BinSiz, PixSiz, AxisType. The parameters
                 SampleDistance, WaveLength, DetRot, Orientation, Projection
                 are not changed.
---------------------------------------------------------------------------*/
void saxs_curve_init ( CmdBlk *pcb, long num[], ImgHeadBlk ihb[], int *pstatus )
{ ImgBlk *ib;
  long I1Inc;

  ib = pcb->ib;

  /*--- Set output header */

  /* output offset 2 */
  if (ihb[0].Offset[2].I>=0) {
    ihb[0].Offset[2].V = 0.0;
    ihb[0].Offset[2].I = TRUE;
  }

  /* output center 2 */
  if (ihb[0].Center[2].I>=0) {
    // INDEX2I(I,O) ((I)+(O)+DAI)
    ihb[0].Center[2].V = INDEX2I(0,ihb[0].Offset[2].V);
    ihb[0].Center[2].I = TRUE;
  }

  /* output BinSiz 2 */
  if (ihb[0].BinSiz[2].I>=0) {
    ihb[0].BinSiz[2].V = 1;
  }

  /* output pixel size 2 */
  if (ihb[0].PixSiz[2].I>=0) {
    ihb[0].PixSiz[2].V =  1;
    ihb[0].PixSiz[2].I = TRUE;
  }

  /* Set AxisType_2 to Numerator */
  if (ihb[0].AxisType[2].I>=0) {
    ihb[0].AxisType[2].V = IO_AxisTypeNumerator;
    ihb[0].AxisType[2].I = TRUE;
  }

  /* output dimension 2 */
  if (!ib[0].Dim[2].I) {
    ihb[0].Dim[2] = 1;
  }

} // saxs_curve_init

/*---------------------------------------------------------------------------
user_io
Do all the keyboard io and return cb, and ib
---------------------------------------------------------------------------*/
void user_io(CmdBlk * pcb, ImgBlk * ib, int * pstatus)
{
  char  progfn[InputLineLength];
  ImgHeadBlk ihb[BlockNum];

  float ROff_1, RPs_1, UOff_1, UPs_1;
  float ROff_2, RPs_2, UOff_2, UPs_2;

 /* Determine program name without directory */
   (void) RightFR((char *) pcb->argv[0],DirSeparator,progfn,InputLineLength);

  /* show usage if no arguments are given */
  if (pcb->argc<=1) printf("Usage: %s %s\n",progfn,Usage);

  /*--- Write name of program ---*/
  printf("\n");
  printf("   %s %s\n",progfn,Version);
  printf("\n");

  ArgvBlocks ( pcb, ib, ihb, NULL, DefaultExtension, 0, BlockNum-1, pstatus);
  if (*pstatus!=Success) return;
  GetReference(pcb->RSys.V,1,ihb,&ROff_1,&ROff_2,&RPs_1,&RPs_2,pstatus );
  if (*pstatus!=Success) return;
  GetReference(pcb->USys.V,1,ihb,&UOff_1,&UOff_2,&UPs_1,&UPs_2,pstatus );
  if (*pstatus!=Success) return;

  /*--- Argument  : multiplication factor */
  printf("<output image> = <input image> * Factor + Const\n");
  argv_flexp(pcb,"Multiplication factor",&ib[0].Factor,"1.0",pstatus);
  if (*pstatus!=Success) return;

  /*--- Argument  : additive constant */
  printf("<output image> = <input image> * Factor + Const\n");
  argv_flexp(pcb,"Addition constant",&ib[0].Const,"0.0",pstatus);
  if (*pstatus!=Success) return;

  printf("\n");
  if (ib[1].Name.I)    printf("i/p file           : %s\n",ib[1].Name.V);
  if (ib[0].Name.I)    printf("o/p file           : %s\n",ib[0].Name.V);
  if (ib[1].First.I)   printf("first image        : %d\n",ib[1].First.V);
  if (ib[1].Last.I)    printf("last image         : %d\n",ib[1].Last.V);
  if (ib[1].Inc.I)     printf("increment          : %d\n",ib[1].Inc.V);
  if (ib[0].Dummy.I)   printf("output dummy       : %s\n",ib[0].Dummy.V);
  if (ib[0].Dim[1].I)  printf("output dimension 1 : %s\n",ib[0].Dim[1].V);
  if (ib[0].Dim[2].I)  printf("output dimension 2 : %s\n",ib[0].Dim[2].V);
  if (ib[0].Factor.I)  printf("factor             : %s\n",ib[0].Factor.V);
  if (ib[0].Const.I)   printf("constant           : %s\n",ib[0].Const.V);
  printf("\n");

  if (pcb->TestBit) {
    PrintBlocks ( pcb, ib );
    printf("\n"); }

  return;
} // user_io

/*---------------------------------------------------------------------------
main
---------------------------------------------------------------------------*/

#if MAKE_FUNCTION
# define MAIN main_saxs_curve
#else
# define MAIN main
#endif

int MAIN (int argc, char *argv[])
{
  CmdBlk cb;                /* command block  */
  ImgBlk ib[BlockNum];      /* image blocks */

  int status;
  int arg_no = 0;

 /* Init options, control block and image blocks */
  InitOptions( Usage, Version, NULL, TRUE, &cb, ib, BlockNum );

  option_define ( IO_tpflag, SHead, "TRUE", &cb ); /* default on */
  option_define ( IO_tpline, SKey,"Intensity0,Intensity1,SampleThickness", &cb);
  option_define ( IO_tpflag, SHist, "TRUE", &cb );
  option_define ( IO_tpline, SHdmk, "#", &cb ); /* default is # */
  option_define ( IO_tpline, SHeadL, "#= I_[Title,,1]", &cb); /*#L s+Title++*/
  option_define ( IO_tpline, SSpacer, "  \t", &cb ); /* default spc,spc,tab */
  option_define ( IO_tpline, SExtension, ".txt", &cb );
  option_define ( IO_tpflag, SAll, "FALSE", &cb );
  option_define ( IO_tpflag, SWaxs, "TRUE", &cb );
  option_define ( IO_tpflag, SSpec, "TRUE", &cb );
  option_define ( IO_tpflexp, SScf, "1.0", &cb );
  option_define ( IO_tpflag, Si1Head, "TRUE", &cb ); /* default on */
  option_define ( IO_tpflag, SGnuPlot, "TRUE", &cb ); /* default on */
  option_define ( IO_tplvexp, SHlen, "25", &cb ); /* at least 25 header lines */
  option_define ( IO_tpflexp, "alf1", "0.0", &cb );
  option_define ( IO_tpflexp, "alf2", "0.0", &cb );

 /* Read options from argument list */
  ReadOptions( argv, &arg_no, &cb, ib, &status);

  /*--- OpenMode of output file */
  if (!ib[0].OpenMode.I) {
    ib[0].OpenMode.V = IO_ImageProtect | IO_DontReadWrite | IO_DontOpen;
    ib[0].OpenMode.I = TRUE;
  }

  /* USER KEYBOARD I/O */
  if (status==Success) {
    argv_start ( &cb, 1 );
    user_io( &cb, ib, &status);
    argv_end( &cb ); /* must be called after user_io */
  }

  /* SEQUENCE CALCULATION */
  if (status==Success) 
    IMAGELOOP( &cb, ib, saxs_curve, saxs_curve_init, NULL, TRUE, &status );

  return( ReportSaxsStatus( status, 0 ) );

} /* MAIN */
