/***************************************************************************/
/* Written 1994++ by Peter Boesecke                                        */
/* Copyright (C) 2011 European Synchrotron Radiation Facility              */
/*                       Grenoble, France                                  */
/*                                                                         */
/*    Principal authors: Peter Boesecke  (boesecke@esrf.eu)                */
/*                                                                         */
/*    This program is free software: you can redistribute it and/or modify */
/*    it under the terms of the GNU General Public License as published by */
/*    the Free Software Foundation, either version 3 of the License, or    */
/*    (at your option) any later version.                                  */
/*                                                                         */
/*    This program is distributed in the hope that it will be useful,      */
/*    but WITHOUT ANY WARRANTY; without even the implied warranty of       */
/*    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the        */
/*    GNU General Public License for more details.                         */
/*                                                                         */
/*    You should have received a copy of the GNU General Public License    */
/*    along with this program.  If not, see <http://www.gnu.org/licenses/>.*/
/***************************************************************************/
/*+++
1 saxs_col
  Projections of cols.

2 PURPOSE
  Projections of cols in a sequence on a single col.
  The col is written to a line of the output image.

  Calculation:       I0 = Projection(I1)
  Error propagation: V0 = Projection(V1)

  Arguments:
  saxs_col [options] <i1nam> <onam> <i1fst> <i1lst> <i1inc> 
             <odum> <odim1> <odim2> <col1> <col2>"

  Defaults:
  <input file name>   : input.edf 
  <output file name>  : output.edf
  <first image>       : <first image number in input file>
  <last image>                              : <last image number in input file>
     if argument list ends with first image : <first image>
  <increment>         : 1
  <dummy>             : <dummy value in first image of input file>
  <dimension 1>       : <horizontal dimension of first image in input file>
  <dimension 2>       : <vertical dimension of first image in input file>
  <first col>         : start of averaging
  <last col>          : end of averaging

  Wildcard:
  A "=" in the argument list chooses the default.

2 HISTORY
  08-Oct-1995 PB 
  04-Dec-1995 PB 3.0
  30-Aug-1996 PB 3.11
  30-Aug-1996 PB 3.12 Option +-vsum (default -vsum)
                      -vsum : switch off sum
                      +vsum : sum of values
                      Option +-vint (default -vint)
                      -vint : switch off vint
                      +vint : integration of values
                      Default : averaging
  27-Sep-1996 PB 3.13 Output line number is always calculated with REALREF
  26-Jun-1999 PB DevSeparator, DirSeparator, ExtSeparator defined in SaxsInput
  2000-08-04  PB %f->%g
  2001-07-08  PB V3.14 FFirst
  2001-07-09  PB V4.00 new loop, no repetition
  2002-06-02  PB V4.5
  2002-06-09  PB V4.51 i_2<I0Dim_2
  2003-02-03  PB V4.52 num[BlockNum]
  2003-02-09  PB V4.53 line args for Dummy, DDummy, Dim
  2003-04-19  PB V4.54 ArgvColFilenames -> ArgvColFilenames,
                       default of col2 corrected
  2003-04-25  PB V4.55 vsum is no option any more (removed from Usage)
  2003-05-25  PB V4.56 Offset -> flexp, Center -> flexp, PixSiz -> flexp
  2003-05-30  PB V4.57 ImageLoop: Function call changed
  2003-08-06  PB V4.58 offset, pixel size, center of output coordinate 1 
                       are now calculated from input coordinate 2. These
                       parameters were not correctly set.
                       ib[0].Offset[2].V calculated as float
  2003-08-12  PB V4.59 option_define, all option numbers removed
  2003-11-30  PB V4.60 BinSiz
  2004-10-31  PB V4.61 MAKE_FUNCTION
  2005-07-15  PB V4.62 ReportSaxsStatus parameter list updated
  2005-08-04  PB V4.63 PrintImageHeaderBlock paramters updated
  2005-08-19  PB V4.64 VarDat
  2005-10-24  PB V4.65 Call of Projection_2 with pE0Data=NULL if E0Data is NULL
  2007-06-05  PB V4.66 row1, row2, col1, col2 are not automatically set
  2007-06-10  PB V4.67 ImageLoop parameters updated
  2007-12-01  PB V4.68
  2008-08-05  PB V4.69 init function, ArgvBlocks replaces ArgvColFilenames
  2008-11-29  PB V4.70 saxs_col_init inserted into ImageLoop, Single2Multi
  2012-01-31  PB V4.71 ImageLoop parameters updated
  2016-03-09  PB V4.72 UpdateImageParametersC
  2016-04-02  PB V4.73 floor -> floorf

---*/

# define Version  "saxs_col V4.73 2016-04-02, Peter Boesecke"

# include <errno.h>
# include <stdio.h>
# include <unistd.h>
# include <fcntl.h>

# include "SaxsPrograms.h"

# define Usage "[options] \n\
                <i1nam> <onam> <i1fst> <i1lst> <i1inc> \n\
                <odum> <odim1> <odim2> <col1> <col2>\n\
Purpose : Projections of cols in a sequence, output to a single image\n\
Options : -col1, -col2, -row1, -row2, +vint"

# define BlockNum 2       /* 1 input sequence + 1 output sequence */

/* Flexp Options */
# define SCol1 "col1"
# define SCol2 "col2"
# define SRow1 "row1"
# define SRow2 "row2"

/* Flag Options */
# define SFlgInt "vint"

/*---------------------------------------------------------------------------
1 saxs_col

2 PURPOSE
  Projections of a range of cols on a single line of the output image.
  output col number : filled line by line, determined by ImNum[1].

  The interesting area in the input image is defined by
  W21 : upper left corner, W22 : upper right corner,
  W11 : lower left corner, W12 : lower right corner.

  W21 = {row2,col1}, WP22 = {row2, col2}.
  W11 = {row1,col1}, WP12 = {row1, col2},

  Col1, Col2 etc.

  array: averaging of col1 to col2. resulting column to line of output
  image: averaging of col1 to col2. resulting column to line of output
  real:
  saxs:
---------------------------------------------------------------------------*/
void saxs_col (CmdBlk * pcb, long num[], ImgHeadBlk ihb[], int * pstatus)
{ int i,imax;
  int j;

  float *I0Data, *E0Data;
  int   I0Dim_1,I0Dim_2;
  float *pI0Data, *pE0Data;
  float I0Dummy, I0DDummy;

  int   I1Ave;
  float *I1Data, *E1Data;
  int   I1Dim_1, I1Dim_2;
  float I1Dummy, I1DDummy;

  int i_2;
  float f_1[BlockNum], f_2[BlockNum];
 
  float Off_1[BlockNum], Off_2[BlockNum];
  float Ps_1[BlockNum], Ps_2[BlockNum];
 
  float f1_1[BlockNum], f3_1[BlockNum], Df_1[BlockNum];
  float f1_2[BlockNum], f3_2[BlockNum], Df_2[BlockNum];
 
  int Imin_1, Imin_2, Imax_1, Imax_2;
 
  float Wmin_1, Wmax_1, W_1, DW_1;
  float Wmin_2, Wmax_2, W_2, DW_2;

  float W1_1, W3_1, W1_2, W3_2;

  float col1, col2, row1, row2;

  float factor;

  IO_float Col1, Col2, Row1, Row2;
  IO_long  FlgInt;

  *pstatus = Success;

  imax = pcb->ImgBlkLen;
  printf("\n Calculating ihb[0,% d] = Function(",ihb[0].ImNum);
  for(i=1;i<imax;i++) {
    printf("ihb[% d,% d] ", i, ihb[i].ImNum); }

  printf(")\n\n");

  for(i=1;i<imax;i++) {
    if (pcb->TestBit) PrintImageHeaderBlock(&ihb[i]);}

 /* Check the number of images */
  if (pcb->ImgBlkLen!=2) {
     printf("%d images found, 1 input and 1 output image required\n",
             pcb->ImgBlkLen); *pstatus=Failed; return; }

  Col1 = option_float ( SCol1, pcb, pcb->ib, num, 1, pstatus );
  if (*pstatus!=Success) return;
  Col2 = option_float ( SCol2, pcb, pcb->ib, num, 1, pstatus );
  if (*pstatus!=Success) return;
  Row1 = option_float ( SRow1, pcb, pcb->ib, num, 1, pstatus );
  if (*pstatus!=Success) return;
  Row2 = option_float ( SRow2, pcb, pcb->ib, num, 1, pstatus );
  if (*pstatus!=Success) return;
  FlgInt = option_flag ( SFlgInt, pcb, pstatus );
  if (*pstatus!=Success) return;

 /* Get input reference system parameters */
  GetReferenceParameters( pcb, ihb, 1, imax-1,
                          Off_1, Off_2, Ps_1,  Ps_2, pstatus );
  if (*pstatus!=Success) return;

 /* Get output reference system parameters of coordinate 1 */
  GetReferenceParametersC(pcb->RSys.V, pcb, ihb,
                          1, 0, 0, Off_1, Ps_1, pstatus );
 /* Set output row reference system of coordinate 2 to real */
  GetReferenceParametersC(IO_Real, pcb, ihb,
                          2, 0, 0, Off_2, Ps_2, pstatus );

 /* Update parameters of output coordinate 1 */
  UpdateImageParametersC( pcb->RSys.V, pcb, ihb, 2, 2, 0, 1, imax-1,
                          Off_1, Off_2, Ps_1, Ps_2, pstatus );
  if (*pstatus!=Success) return;

 /* Set AxisType_1 to AxisType_2 of input */
  if (ihb[0].AxisType[1].I>=0) {
    ihb[0].AxisType[1].V = ihb[1].AxisType[2].V; 
    ihb[0].AxisType[1].I = ihb[1].AxisType[2].I;
  }
 
 /* BinSiz is no longer well defined for output coordinate 2, set it to 1 */
  if (ihb[0].BinSiz[2].I>=0) ihb[0].BinSiz[2].V = 1.0;

 /* Set AxisType_2 to Numerator */
  if (ihb[0].AxisType[2].I>=0) {
    ihb[0].AxisType[2].V = IO_AxisTypeNumerator; ihb[0].AxisType[2].I = TRUE;
  }

 /* Get region of interest of the input image */
  GetSmallRectangle( pcb,ihb,1,1,-1,1,1,Off_1,Ps_1,Off_2,Ps_2,&W1_1,&W3_1 );
  GetSmallRectangle( pcb,ihb,2,2,-1,1,1,Off_1,Ps_1,Off_2,Ps_2,&W1_2,&W3_2 );

  if (Col1.I) col1=Col1.V; else col1=W1_1;
  if (Col2.I) col2=Col2.V; else col2=W3_1;
  if (Row1.I) row1=Row1.V; else row1=W1_2;
  if (Row2.I) row2=Row2.V; else row2=W3_2;

 /* 1 input, 1 output */
  I0Data  = ihb[0].Data;
  E0Data  = ihb[0].VarDat;
  I0Dummy = ihb[0].Dummy.V;
  I0DDummy = ihb[0].DDummy.V;
  I0Dim_1  = (int) ihb[0].Dim[1];
  I0Dim_2  = (int) ihb[0].Dim[2];
 
  I1Ave   = pcb->ib[1].Ave.V;
  I1Data  = ihb[1].Data;
  E1Data  = ihb[1].VarDat;
  I1Dummy = ihb[1].Dummy.V;
  I1DDummy = ihb[1].DDummy.V;
  I1Dim_1  = (int) ihb[1].Dim[1];
  I1Dim_2  = (int) ihb[1].Dim[2];

  // get range of input coordinate 1 (OutImage=-1)
  GetLinearRange      ( pcb, ihb,
                        -1, 1, -1, 1, imax-1,
                        1, col1, 1, col2,
                        Off_1, Ps_1, Off_2, Ps_2,
                        f1_1, f3_1, Df_1,
                        &Imin_2, &Imax_2, &Wmin_1, &Wmax_1, &DW_1, 1 );

  // get range of output coordinate 1 and input coordinate 2
  GetLinearRange      ( pcb, ihb,
                        1, 2, 0, 1, imax-1,
                        1, row1, 1, row2,
                        Off_1, Ps_1, Off_2, Ps_2,
                        f1_2, f3_2, Df_2,
                        &Imin_1, &Imax_1, &Wmin_2, &Wmax_2, &DW_2, 1 );

  // use input image number as world coordinate of output rows
  i_2 = INDEX(ihb[1].ImNum,Off_2[0],Ps_2[0]); /* input image number */
  if (pcb->TestBit)  {
    printf("Output row number = %d (%9.5g)\n",\
      i_2,INDEX(ihb[1].ImNum,Off_2[0],Ps_2[0]));
    } 

  if (FlgInt.V) factor = Ps_1[1]; else factor = 1.0;
  if ( (0<=i_2) && (i_2<I0Dim_2) ) {

    pI0Data = ABSPTR(I0Data,I0Dim_1,I0Dim_2,0,i_2);
    pE0Data = E0Data?E0Data-I0Data+pI0Data:NULL;
    Projection_2  ( pI0Data, pE0Data, I0Dim_1, Imin_1, Imax_1, I0Dummy, factor,
                    I1Data, E1Data, I1Dim_1, I1Dim_2,
                    f1_1[1], f3_1[1], f1_2[1], f3_2[1], Df_2[1],
                    I1Dummy, I1DDummy, I1Ave, pcb->TestBit );

  }

} /* saxs_col */

/*---------------------------------------------------------------------------
saxs_col_init

Initializes the output header of each output file before the output 
data array is allocated.

The output header is already initialized with the header of the first
image of the first input file sequence. This routine modifies this default.

The number of available blocks is defined by the program and does not need 
to be checked here. All input images are open in SaxsImageLoop when a new 
output image is created.

saxs_row: 
  Axis 1 output: Same geometry as input image Axis 2
  Axis 2 output: Change parameters to numerator. Modify geometrical 
                 parameters only, if not already set by image block (.I<1)
                 Offset, Center, BinSiz, PixSiz, AxisType. The parameters 
                 SampleDistance, WaveLength, DetRot, Orientation, Projection 
                 are not changed.
---------------------------------------------------------------------------*/
void saxs_col_init ( CmdBlk *pcb, long num[], ImgHeadBlk ihb[], int *pstatus ) 
{ ImgBlk *ib;
  long I1Inc;

  ib = pcb->ib;

  if (Single2Multi()) I1Inc = ib[1].FInc.V;
  else I1Inc = ib[1].Inc.V;

  /*--- Set output header */

  /* Axis 1 */

  /* output offset 1 */
  if (ihb[0].Offset[1].I>=0) {
    ihb[0].Offset[1].V = ihb[1].Offset[2].V;
    ihb[0].Offset[1].I = ihb[1].Offset[2].I;
  }

  /* output center 1 */
  if (ihb[0].Center[1].I>=0) {
    ihb[0].Center[1].V = ihb[1].Center[2].V ;
    ihb[0].Center[1].I = ihb[1].Center[2].I ;
  }

  /* output BinSiz 1 */
  if (ihb[0].BinSiz[1].I>=0) {
    ihb[0].BinSiz[1].V = ihb[1].BinSiz[2].V;
    ihb[0].BinSiz[1].I = ihb[1].BinSiz[2].I;
  }

  /* output pixel size 1 */
  if (ihb[0].PixSiz[1].I>=0) {
    ihb[0].PixSiz[1].V = ihb[1].PixSiz[2].V;
    ihb[0].PixSiz[1].I = ihb[1].PixSiz[2].I;
  }

  /* output AxisType 1 */
  if (ihb[0].AxisType[1].I>=0) {
    ihb[0].AxisType[1].V = ihb[1].AxisType[2].V; 
    ihb[0].AxisType[1].I = ihb[1].AxisType[2].I; 
  }

  /* output dimension 1 */
  if (!ib[0].Dim[1].I) ihb[0].Dim[1] = ihb[1].Dim[2];

  /* Axis 2 */

  /* output offset 2 */
  if (ihb[0].Offset[2].I>=0) {
    ihb[0].Offset[2].V = ((float) ib[1].First.V/I1Inc) - 1.0;
    ihb[0].Offset[2].I = TRUE;
  }

  /* output center 2 */
  if (ihb[0].Center[2].I>=0) {
    // INDEX2I(I,O) ((I)+(O)+DAI)
    ihb[0].Center[2].V = INDEX2I(0,ihb[0].Offset[2].V);
    ihb[0].Center[2].I = TRUE;
  }

  /* output BinSiz 2 */
  if (ihb[0].BinSiz[2].I>=0) ihb[0].BinSiz[2].V = I1Inc;

  /* output pixel size 2 */
  if (ihb[0].PixSiz[2].I>=0) {
    ihb[0].PixSiz[2].V = I1Inc;
    ihb[0].PixSiz[2].I = TRUE;
  }

  /* Set AxisType_2 to Numerator */
  if (ihb[0].AxisType[2].I>=0) {
    ihb[0].AxisType[2].V = IO_AxisTypeNumerator; ihb[0].AxisType[2].I = TRUE;
  }

  /* output dimension 2 */
  if (!ib[0].Dim[2].I) {
    if (Single2Multi()) ihb[0].Dim[2] = 1;
    else {
      if (ihb[0].PixSiz[2].V==0) {
        ihb[0].Dim[2] = 1 + (MAX2(ib[1].First.V,ib[1].Last.V) -
                 MIN2(ib[1].First.V,ib[1].Last.V));
      } else {
        ihb[0].Dim[2] = (int) FLOORF(1.5+((MAX2(ib[1].First.V,ib[1].Last.V) -
                 MIN2(ib[1].First.V,ib[1].Last.V))/ihb[0].PixSiz[2].V));
      }
    }
  }

} // saxs_col_init

/*---------------------------------------------------------------------------
user_io
Do all the keyboard io and return cb, and ib
---------------------------------------------------------------------------*/

void user_io(CmdBlk * pcb, ImgBlk * ib, int * pstatus)
{
  char  progfn[InputLineLength];
  ImgHeadBlk ihb[BlockNum];

  float ROff_1, RPs_1, UOff_1, UPs_1;
  float ROff_2, RPs_2, UOff_2, UPs_2;

  char prompt_line[IO_len];

  IO_flexp *pCol1, *pCol2; 

 /* Determine program name without directory */
   (void) RightFR((char *) pcb->argv[0],DirSeparator,progfn,InputLineLength);

  /* show usage if no arguments are given */
  if (pcb->argc<=1) printf("Usage: %s %s\n",progfn,Usage);

  /*--- Write name of program ---*/
  printf("\n");
  printf("   %s %s\n",progfn,Version);
  printf("\n");

  /* --- output into first image */
  if (!ib[0].First.I) { ib[0].First.V = 1; ib[0].First.I = TRUE; }
  /* --- only 1 output image */
  if (!ib[0].Inc.I) { ib[0].Inc.V = 0; ib[0].Inc.I = TRUE; }

  ArgvBlocks ( pcb, ib, ihb, saxs_col_init, NULL, 0, BlockNum-1, pstatus);
  if (*pstatus!=Success) return;
  GetReference(pcb->RSys.V,1,ihb,&ROff_1,&ROff_2,&RPs_1,&RPs_2,pstatus );
  if (*pstatus!=Success) return;
  GetReference(pcb->USys.V,1,ihb,&UOff_1,&UOff_2,&UPs_1,&UPs_2,pstatus );
  if (*pstatus!=Success) return;

  /*--- Argument 5 : col 1 */
  printf("Column boundaries in %s coordinates\n",reftostr(pcb->RSys.V));
  pCol1 = (IO_flexp*) option_parameter_search( SCol1, pcb, pstatus );
  if (*pstatus!=Success) return;
  sprintf(prompt_line,"boundary (col1), e.g. %g",
    WORLD(INDEXSTART+LOWERBORDER,ROff_1,RPs_1));
  argv_flexp(pcb,prompt_line,pCol1,"",pstatus);
  if (*pstatus!=Success) return; 

  /*--- Argument 6 : col 2 */
  printf("Column boundaries in %s coordinates\n",reftostr(pcb->RSys.V));
  pCol2 = (IO_flexp*) option_parameter_search( SCol2, pcb, pstatus );
  if (*pstatus!=Success) return;
  sprintf(prompt_line,"boundary (col2), e.g. %g",
    WORLD(INDEXSTART+ihb[1].Dim[1]+LOWERBORDER,ROff_1,RPs_1));
  argv_flexp(pcb,prompt_line,pCol2,"",pstatus);
  if (*pstatus!=Success) return; 

  printf("\n");
  if (ib[1].Name.I)    printf("i/p file           : %s\n",ib[1].Name.V);
  if (ib[0].Name.I)    printf("o/p file           : %s\n",ib[0].Name.V);
  if (ib[1].First.I)   printf("first image        : %d\n",ib[1].First.V);
  if (ib[1].Last.I)    printf("last image         : %d\n",ib[1].Last.V);
  if (ib[1].Inc.I)     printf("increment          : %d\n",ib[1].Inc.V);
  if (ib[0].Dummy.I)   printf("output dummy       : %s\n",ib[0].Dummy.V);
  if (ib[0].Dim[1].I)  printf("output dimension 1 : %s\n",ib[0].Dim[1].V);
  if (ib[0].Dim[2].I)  printf("output dimension 2 : %s\n",ib[0].Dim[2].V);
  if (pCol1->I)        printf("boundary (col1)    : %s\n",pCol1->V);
  if (pCol2->I)        printf("boundary (col2)    : %s\n",pCol2->V);
  printf("\n");

  if (pcb->TestBit) {
    PrintBlocks ( pcb, ib );
    printf("\n"); }

  return;
}

/*---------------------------------------------------------------------------
main
---------------------------------------------------------------------------*/

#if MAKE_FUNCTION
# define MAIN main_saxs_col
#else
# define MAIN main
#endif

int MAIN (int argc, char *argv[])
{
  CmdBlk cb;                /* command block  */
  ImgBlk ib[BlockNum];      /* image blocks */

  int status;
  int arg_no = 0;

 /* Init options, control block and image blocks */
  InitOptions( Usage, Version, NULL, TRUE, &cb, ib, BlockNum );

 /* Define special options */
  option_define ( IO_tpflexp, SCol1,   "-inf", &cb );
  option_define ( IO_tpflexp, SCol2,   "+inf", &cb );
  option_define ( IO_tpflexp, SRow1,   "-inf", &cb );
  option_define ( IO_tpflexp, SRow2,   "+inf", &cb );
  option_define ( IO_tpflag,  SFlgInt, "FALSE", &cb );

 /* Read options from argument list */
  ReadOptions( argv, &arg_no, &cb, ib, &status);

  /* USER KEYBOARD I/O */
  if (status==Success) {
    argv_start ( &cb, 1 );
    user_io( &cb, ib, &status);
    argv_end( &cb ); /* must be called after user_io */
  }

  /* SEQUENCE CALCULATION */
  if (status==Success) 
    IMAGELOOP(&cb,ib,saxs_col,saxs_col_init,NULL,TRUE,&status);

  return( ReportSaxsStatus( status, 0 ) );

} /* MAIN */

