/***************************************************************************/
/* Written 1994++ by Peter Boesecke                                        */
/* Copyright (C) 2011 European Synchrotron Radiation Facility              */
/*                       Grenoble, France                                  */
/*                                                                         */
/*    Principal authors: Peter Boesecke  (boesecke@esrf.eu)                */
/*                                                                         */
/*    This program is free software: you can redistribute it and/or modify */
/*    it under the terms of the GNU General Public License as published by */
/*    the Free Software Foundation, either version 3 of the License, or    */
/*    (at your option) any later version.                                  */
/*                                                                         */
/*    This program is distributed in the hope that it will be useful,      */
/*    but WITHOUT ANY WARRANTY; without even the implied warranty of       */
/*    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the        */
/*    GNU General Public License for more details.                         */
/*                                                                         */
/*    You should have received a copy of the GNU General Public License    */
/*    along with this program.  If not, see <http://www.gnu.org/licenses/>.*/
/***************************************************************************/
/*+++
1 saxs_bsl
  Transformation of an edf image into bsl format (daresbury otoko)

2 PURPOSE
  Transformation of an edf image into bsl format (daresbury otoko)

  Calculation:       I0 = I1
  Error propagation: V0 = V1 

  Arguments:
  saxs_bsl [options] 
                <i1nam> <output header name> <i1fst> <i1lst> <i1inc> 
                <odum> <odim1> <odim2>
  Defaults:
  <input file name>   : input.edf 
  <output header name>: I54000.EDF (default conversion from input filename)
  <first image>       : <first image number in input file>
  <last image>                              : <last image number in input file>
     if argument list ends with first image : <first image>
  <increment>         : 1
  <dummy>             : <dummy value in first image of input file>
  <dimension 1>       : <horizontal dimension of first image in input file>
  <dimension 2>       : <vertical dimension of first image in input file>

  Specific arguments:
  -pref 'prefix'      : specify the prefix of the ouptput file

  Wildcard:
  A "=" in the argument list chooses the default.

2 HISTORY
  28-Nov-1995 PB
  21-May-1996 PB  option +bsl outputs into a single bsl binary file
  26-Jun-1999 PB  DevSeparator, DirSeparator, ExtSeparator defined in SaxsInput
  2000-08-04  PB  %f->%g
  2001-07-07  PB  IO_DontReadWrite
  2001-07-08  PB  V3.11 FFirst
  2001-07-09  PB  V3.12 (pcb->ib)[0].FileNumber
  2001-07-09  PB  V4.00 new loop, no repetition
  2001-08-30  PB  V4.01 ImageIsFirst, ImageIsLast
  2002-06-01  PB  V4.50 Ave
  2003-02-03  PB  V4.51 num[BlockNum]
  2003-02-09  PB  V4.52 Dummy, DDummy, Dim -> IO_*exp
  2003-05-30  PB  V4.53 ImageLoop: Function call changed
  2003-08-12  PB  V4.54 option_define, all option numbers removed
  2003-11-30  PB  V4.55 BinSiz
  2004-10-31  PB  V4.56 MAKE_FUNCTION
  2005-07-15  PB  V4.57 ReportSaxsStatus parameter list updated
  2005-08-04  PB  V4.58 PrintImageHeaderBlock paramters updated
  2005-09-17  PB  V4.59 VarDat, the variance data (E0Data) is currently
                        not written to a bsl file because a good 
                        bsl file name is missing. Should it be called
                        Enn000.STD ?
  2007-06-11  PB  V4.60 ImageLoop parameters updated
  2008-12-05  PB  V4.61 open all files as binary file to avoid
                        translation of \r\n to \n when compiled on windows.
  2012-01-31  PB  V4.62 ImageLoop parameters updated
  2014-03-05  PB  V4.63 Bsl output byte order added, option -bobo 1|2

---*/
# define Version  "saxs_bsl V4.63 2014-03-05, Peter Boesecke"

# include <errno.h>
# include <stdio.h>
# include <unistd.h>
# include <fcntl.h>

# include "SaxsPrograms.h"

# define SPrefix "hnam"
# define SBobo   "bobo"
# define SSingle "sngl"
# define SBsl    "bsl"

# define Usage "[options] \n\
                <i1nam> <output header name> <i1fst> <i1lst> <i1inc> \n\
                <odum> <odim1> <odim2>"

# define BlockNum 2       /* 1 input sequence + 1 output sequence */

long FrameCounter=0l;

/*---------------------------------------------------------------------------
ToBslName
Conversion of string to a bsl name of the type BNN000.BSL
All characters are converted to capital letters.  The first three characters
are used for BNN. The first three characters after '.' are taken as extension.
char * sout  output string
char * sin   input string
in NNFlg     0: NN taken from second and third character
             1: NN taken from last two characters before '.'
---------------------------------------------------------------------------*/
void ToBslName ( char * sout, char * sin, int NNFlg )
{ register int i, k;
  char extension[InputLineLength];
  char name[InputLineLength];
  char NN[3];

  /* conversion to capital letters */
  CopyLine( sin, sout, InputLineLength, 1);
  /* extraction of extension */
  RightFL(sout,".",extension,InputLineLength);
  for (i=strlen(extension);i<3;i++) extension[i] = 'B';
  extension[3] = (char) 0;
  /* extraction of number NN*/
  LeftFL(sout,".",name,InputLineLength);
  for (i=0;i<2;i++) NN[i]='0';NN[2]=(char) 0;
  for (k=1,i=strlen(name)-1;(k>=0) && (i>=0);k--,i--)
    if ( (i>0) && 
         ( ((sout[i]>=(int)'0') && (sout[i]<=(int)'9'))||(sout[i]==(int)'%') ) 
       )
      NN[k] = sout[i]; else NN[k] = '0'+ (char) (sout[i]%10);
  /* creation of bsl name */
  sout[0] = sout[0];
  if (NNFlg) {
    sout[1] = NN[0];
    sout[2] = NN[1];
    } else {
    if (((sout[1]<(int)'0')||(sout[1]>(int)'9'))&&(sout[1]!=(int)'%'))
      sout[1]='0';
    if (((sout[2]<(int)'0')||(sout[2]>(int)'9'))&&(sout[2]!=(int)'%'))
      sout[2]='0';
    }
  sout[3] = '0';
  sout[4] = '0';
  sout[5] = '0';
  sout[6] = '.';
  sout[7] = extension[0];
  sout[8] = extension[1];
  sout[9] = extension[2];
  sout[10] = (char) 0;

} /* ToBslName */

/*---------------------------------------------------------------------------
BslFileName
Replacement of nnn in the string BNNnnn.BSL
---------------------------------------------------------------------------*/
void BslFileName ( char * sout, const char * sin, int nnn )
{ int ddd;
  char Temp1[IO_len], Temp2[IO_len];

  filename_parameter(Temp1,IO_len,sin,0);
  filename_name(Temp2, IO_len,Temp1);
  CopyLine( Temp2, Temp2, 11, 1);

  nnn = nnn%1000; if (nnn<0) nnn = nnn+1000;

  ddd = nnn%10; nnn = nnn / 10; Temp2[5] = '0'+ (char) ddd;
  ddd = nnn%10; nnn = nnn / 10; Temp2[4] = '0'+ (char) ddd;
  ddd = nnn%10; nnn = nnn / 10; Temp2[3] = '0'+ (char) ddd;

  sprintf(sout,"%s%s", filename_path(Temp1,IO_len,Temp1),Temp2);

} /* BslFileName */

/*---------------------------------------------------------------------------
Saxs2BslFilenames

Ask for all file names in the following way:
  input file 1 [input.edf] :
  input file 2 [<input file 1>] :
  ...
  output file [output.edf] :

  first image [<minimum image number>] :
  last image [<maximum image number>] :
  increment [1] :

  output dummy [<dummy in first image>] :

If blkno >= 1 the header block of the first image is read into ihb[1].
The routine must be used for all images at once.
---------------------------------------------------------------------------*/
void Saxs2BslFilenames ( CmdBlk * pcb, ImgBlk ib[], ImgHeadBlk ihb[],
                     int block_1st, int block_lst, int * pstatus)
{ int blkno, first_block;
  long image_1st, image_lst;
  char linebuffer[IO_len];
  char Path[IO_len], Name[IO_len];
  long num[BlockNum];

  IO_line * pPrefix;

  first_block = MAX2(1,block_1st);

  /* ---  File names and open modes --- */

  for (blkno = first_block;blkno <= block_lst;blkno++) {

    /*--- Open mode ib[blkno].OpenMode of input files */
    if (!ib[blkno].OpenMode.I) {
      ib[blkno].OpenMode.V = IO_Old | IO_FileProtect;
      ib[blkno].OpenMode.I = TRUE;
      }

    /*--- File name ib[blkno].Name of input files */
    (void) sprintf(linebuffer,"Input sequence %d",blkno);
    if (blkno==1)
      argv_filename( pcb,linebuffer, ib, 1, DefaultInput, pstatus);
    else argv_filename( pcb,linebuffer,ib, blkno,ib[blkno-1].Name.V,pstatus);
    /* Exit in all cases after an abort from the first argument */
    if (*pstatus!=Success) return;

    OpenImageFile( pcb,ib,blkno,ib[blkno].Name.V,ib[1].FFirst.V,
                   IO_Old|IO_FileProtect,pstatus);
    if (*pstatus!=Success) return;

    if (blkno==1) {
      /* Search for minimum and maximum image number in 1st input sequence */
      (void) SearchMinMaxImage ( pcb, ib, 1, &image_1st, &image_lst, pstatus);
      if (*pstatus!=Success) return;
    }

  } /* for input files */

  if (block_1st<=0) {
    /*--- OpenMode of output file */
    if (!ib[0].OpenMode.I) {
      ib[0].OpenMode.V = IO_ImageProtect | IO_DontReadWrite | IO_DontOpen;
      ib[0].OpenMode.I = TRUE;
      }

    /*--- File name ib[0].Name of output (not open) */
    if (!ib[0].Name.I) {
      (void) sprintf(ib[0].Name.V,"%s",DefaultOutput);
      }

    /*--- Argument  : output bsl name B%%nnn.%%%, fixed length 11 characters */
    pPrefix = (IO_line*) option_parameter_search( SPrefix, pcb, pstatus );
    if (*pstatus!=Success) return;

    if (!pPrefix->I) {
      /* Determine prefix and extension from input file name */
      filename_name(pPrefix->V,IO_len,ib[1].Name.V);
      pPrefix->I = TRUE;
      }

    /* extract output filenumbers from BSL file name */
    extract_filenumbers( &ib[0], pPrefix, TRUE, pstatus );
    if (*pstatus!=Success) return;

    // filter bsl file name
    filename_parameter ( Name,IO_len,pPrefix->V,0);
    filename_path ( Path, IO_len-10, Name ); //bsl name<10 char
    filename_name ( Name, IO_len, Name );
    ToBslName(Name,Name,TRUE);
    if (strcmp(Path,"./"))
      sprintf(pPrefix->V,"%s%s",Path,Name);
     else sprintf(pPrefix->V,"%s",Name);

    argv_line( pcb,"Output header name",pPrefix,pPrefix->V,pstatus);
    if (*pstatus!=Success) return;

    /* extract output filenumbers from BSL file name */
    extract_filenumbers( &ib[0], pPrefix, TRUE, pstatus );
    if (*pstatus!=Success) return;

    // filter bsl file name
    filename_parameter ( Name,IO_len,pPrefix->V,0);
    filename_path ( Path, IO_len-10, Name ); //bsl name<10 char
    filename_name ( Name, IO_len, Name );
    ToBslName(Name,Name,FALSE);
    if (strcmp(Path,"./"))
      sprintf(pPrefix->V,"%s%s",Path,Name);
     else sprintf(pPrefix->V,"%s",Name);

    /*--- Take prefix as default for output filename */
    if (!ib[0].Name.I) {
      (void) sprintf(ib[0].Name.V,"%s",pPrefix->V);
      ib[0].Name.I = TRUE;
      }

  } /* output file */

  /* ---  Image numbers --- */

  if (first_block==1) {

    /*--- Argument : number of first image */
    argv_long( pcb,"First image of input sequence 1", &ib[1].First, image_1st,
               pstatus);
    if (*pstatus!=Success) return;

    /*--- Argument : number of last image */
    if (pcb->argc==*pcb->parg_no+1) image_lst = ib[1].First.V;
    argv_long( pcb,"Last image of input sequence 1", &ib[1].Last, image_lst,
               pstatus);
    if (*pstatus!=Success) return;

    /*--- Argument : increment (default = 1) */
    argv_long( pcb,"Increment of input sequence 1", &ib[1].Inc, 1 , pstatus);
    if (*pstatus!=Success) return;

    num[1] = ib[1].First.V;
    (void) ReadImageHeader( pcb, ib, 1, num, ihb, pstatus);
    if (*pstatus!=Success) return;
    if (pcb->TestBit) PrintImageHeaderBlock(&ihb[1]);

    /* Defaults for output file */
    if (!ib[0].Dummy.I) sprintf(ib[0].Dummy.V,"%g",ihb[1].Dummy.V);
    if (!ib[0].Dim[1].I) sprintf(ib[0].Dim[1].V,"%d",ihb[1].Dim[1]);
    if (!ib[0].Dim[2].I) sprintf(ib[0].Dim[2].V,"%d",ihb[1].Dim[2]);                                                                          

  } /* first input file */

  for (blkno = MAX2(2,first_block);blkno<=block_lst;blkno++) {
    /*--- image numbers of all other images */
    (void) sprintf(linebuffer,"First image of input sequence %d",blkno);
    argv_long( pcb,linebuffer, &ib[blkno].First, ib[1].First.V, pstatus);
    if (*pstatus!=Success) return;
    num[blkno] = ib[1].First.V;

  } /* all other input files */

  if (block_1st <= 0) {
    /*--- Argument : output dummy */
    argv_flexp( pcb, "Output dummy", &ib[0].Dummy, ib[0].Dummy.V, pstatus);
    if (*pstatus!=Success) return;

    /*--- Argument : output dimension 1 */
    argv_lvexp(pcb,"Output dimension 1", &ib[0].Dim[1],ib[0].Dim[1].V,pstatus);
    if (*pstatus!=Success) return;

    /*--- Argument : output dimension 2 */
    argv_lvexp(pcb,"Output dimension 2", &ib[0].Dim[2],ib[0].Dim[2].V, pstatus);
    if (*pstatus!=Success) return;

    }

  for (blkno = blkno = MAX2(1,first_block);blkno<=block_lst;blkno++) {
    CloseImageFile( pcb, ib, blkno, pstatus) ;
    if (*pstatus!=Success) return;
  } /* close all input files */

} /* Saxs2BslFilenames */

/*---------------------------------------------------------------------------
1 saxs_bsl

2 PURPOSE
  Multiplicates an image with a factor and adds a constant
---------------------------------------------------------------------------*/
void saxs_bsl (CmdBlk * pcb, long num[], ImgHeadBlk ihb[], int * pstatus)
{ int i,imax;
  int j;

  long  bsl_output_byteorder=0; // machine byte order
  float *I0Data, *E0Data;
  int   I0Dim_1,I0Dim_2;
  float *pI0Data, *pE0Data;
  float I0Dummy, I0DDummy;
  float VarDDummy=DDSET(VarDummy);

  int   I1Ave;
  float *I1Data, *E1Data;
  int   I1Dim_1,I1Dim_2;
  float I1Dummy, I1DDummy;
  float I1Value, I1Sum, I1Weight;
  float E1Value, E1Sum, E1Weight;

  int i_1, i_2;
  float f_1[BlockNum], f_2[BlockNum];
 
  float Off_1[BlockNum], Off_2[BlockNum];
  float Ps_1[BlockNum], Ps_2[BlockNum];
 
  float f1_1[BlockNum], f3_1[BlockNum], Df_1[BlockNum];
  float f1_2[BlockNum], f3_2[BlockNum], Df_2[BlockNum];
 
  int Imin_1, Imin_2, Imax_1, Imax_2;
 
  float Wmin_1, Wmax_1, W_1, DW_1;
  float Wmin_2, Wmax_2, W_2, DW_2;

  char OutputHeaderName[InputLineLength];
  char OutputBinaryName[InputLineLength];
  FILE *OutputHeaderId;
  FILE *OutputBinaryId;
  int  nn;
  char CC;
  char NN[3];
  size_t t_length;
  char fname_full[IO_len], Temp[IO_len];

  IO_line Prefix;
  IO_long Bobo;
  IO_long Single;

  *pstatus = Success;

  imax = pcb->ImgBlkLen;
  printf("\nCalculating ihb[0,% d] = Function(",ihb[0].ImNum);
  for(i=1;i<imax;i++) {
    printf("ihb[% d,% d] ", i, ihb[i].ImNum); }

  printf(")\n\n");

  for(i=1;i<imax;i++) {
    if (pcb->TestBit) PrintImageHeaderBlock(&ihb[i]);}

 /* Check the number of images */
  if (pcb->ImgBlkLen!=2) {
     printf("%d images found, 1 input and 1 output image required\n",
             pcb->ImgBlkLen); *pstatus=Failed; return; }

  Prefix = option_line ( SPrefix, pcb, pcb->ib, num, 1, pstatus );
  if (*pstatus!=Success) return;

  Bobo = option_long ( SBobo, pcb, pcb->ib, num, 1, pstatus );
  if (*pstatus!=Success) return;

  /* set byte order of all bsl input files */
  switch (Bobo.V ) {
    case 1: bsl_output_byteorder = LowByteFirst;  break;
    case 2: bsl_output_byteorder = HighByteFirst; break;
    default: // use machine byte order
            bsl_output_byteorder = edf_byteorder(); break;
  }

  Single = option_flag( SSingle, pcb, pstatus);
  if (*pstatus!=Success) return;

  GetReferenceParameters( pcb, ihb, 0, imax-1,
                          Off_1, Off_2, Ps_1,  Ps_2, pstatus );
  if (*pstatus!=Success) return;

  UpdateImageParameters ( pcb, ihb, 1, imax-1,
                          Off_1, Off_2, Ps_1,  Ps_2, pstatus );
  if (*pstatus!=Success) return;

  GetImageRange         ( pcb, ihb, 1, imax-1,
                          Off_1, Off_2, Ps_1,  Ps_2,
                          f1_1, f3_1, Df_1, f1_2, f3_2, Df_2,
                          &Imin_1, &Imax_1, &Imin_2, &Imax_2,
                          &Wmin_1, &Wmax_1, &DW_1, &Wmin_2, &Wmax_2, &DW_2, 1);

 /* 1 input, 1 output */
  I0Data  = ihb[0].Data;
  E0Data = ihb[0].VarDat;
  I0Dummy = ihb[0].Dummy.V;
  I0DDummy = ihb[0].DDummy.V;
  I0Dim_1  = (int) ihb[0].Dim[1];
  I0Dim_2  = (int) ihb[0].Dim[2];

  I1Ave = pcb->ib[1].Ave.V;
  I1Data  = ihb[1].Data;
  E1Data  = ihb[1].VarDat;
  I1Dummy = ihb[1].Dummy.V;
  I1DDummy = ihb[1].DDummy.V;
  I1Dim_1  = (int) ihb[1].Dim[1];
  I1Dim_2  = (int) ihb[1].Dim[2];

  /* loop over the output array */
  W_2 = Wmin_2; for (i=0;i<imax;i++) f_2[i]=f1_2[i];
  for (i_2=Imin_2;i_2<=Imax_2;i_2++) {
 
    pI0Data = ABSPTR(I0Data,I0Dim_1,I0Dim_2,Imin_1,i_2);
    pE0Data = E0Data-I0Data+pI0Data;
    W_1 = Wmin_1; for (i=0;i<imax;i++) f_1[i]=f1_1[i];
    for (i_1=Imin_1;i_1<=Imax_1;i_1++) {
 
      if ( E0Data ) {
        // V0 = V1
        if ( Isum2ldwE(I1Data,E1Data,I1Dim_1,I1Dim_2,I1Dummy,I1DDummy,
               f_1[1], f_2[1], f_1[1]+Df_1[1], f_2[1]+Df_2[1],
               &I1Sum, &I1Weight, &E1Sum, &E1Weight) ) {
          /* then do something with the data */
          I1Value = I1Sum; if (I1Ave) I1Value /= I1Weight;
          UPDATE( *pI0Data, I1Value, I0Dummy, I0DDummy );
 
          if (E1Sum >= 0.0) {
            E1Value = E1Sum; if (I1Ave) E1Value /= E1Weight*E1Weight;
            UPDATE( *pE0Data, E1Value, VarDummy, VarDDummy );
          }
        }
      } else {
        if (Isum2ldw (I1Data,I1Dim_1,I1Dim_2,I1Dummy,I1DDummy,
                        f_1[1],f_2[1],f_1[1]+Df_1[1],f_2[1]+Df_2[1],
                        &I1Sum,&I1Weight)) {
          I1Value = I1Sum; if (I1Ave) I1Value /= I1Weight;
          UPDATE( *pI0Data, I1Value, I0Dummy, I0DDummy );
        } /* if (Ipol2d ... */
      }
 
      pI0Data++;
      pE0Data++;
      W_1+=DW_1; for (i=0;i<imax;i++) { f_1[i]+=Df_1[i]; }
      } /* for i_1 ... */
 
    W_2+=DW_2; for (i=0;i<imax;i++) { f_2[i]+=Df_2[i]; }
    } /* for i_2 ... */

  if (!Single.I) Single.V = FALSE;

  // Update frame counter 
  if  (ImageIsFirst( pcb, 0, ihb )) FrameCounter=1l;
   else FrameCounter+=1l;

  // Generate BSL filename
  filename_parameter (fname_full,IO_len,Prefix.V,0);
  filename_pattern (fname_full,IO_len,fname_full,(pcb->ib)[0].FileNumber);
  filename_full (fname_full,IO_len,fname_full);
 
  /* Reset frame counter */
  if  (ImageIsFirst( pcb, 0, ihb )) FrameCounter=1l;
 
  if (!Single.V) {
    if (pcb->TestBit) printf("Output into multi bsl file\n");
 
    /* Output into multi bsl file (otoko) format */
 
    BslFileName ( OutputHeaderName, fname_full, 0 );
    BslFileName ( OutputBinaryName, fname_full, ihb[0].ImNum );
 
    if (pcb->TestBit) printf("OutputHeaderName = %s\n",OutputHeaderName);
    if (pcb->TestBit) printf("OutputBinaryName = %s\n",OutputBinaryName);
 
    /* write header file */
    if  (ImageIsFirst( pcb, 0, ihb )) {
      if ((OutputHeaderId=fopen(OutputHeaderName,"wb")) == NULL) {
        printf("ERROR : Cannot create % s\n",OutputHeaderName); return; }
      if (ihb[0].Title.I) {
        fprintf(OutputHeaderId,"%80s\n",ihb[0].Title.V);}
        else fprintf(OutputHeaderId,"first header\n");
      fprintf(OutputHeaderId,"second header\n");
    } else 
      if ((OutputHeaderId=fopen(OutputHeaderName,"ab")) == NULL) {
        printf("ERROR : Cannot open % s\n",OutputHeaderName); return; }
 
    if  ( !ImageIsLast( pcb, 0, ihb ) )
      fprintf(OutputHeaderId,"%8d%8d%8d%8d%8d%8d%8d%8d%8d%8d\n",
              I0Dim_1,I0Dim_2,1,0,0,0,0,0,0,1);
      else fprintf(OutputHeaderId,"%8d%8d%8d%8d%8d%8d%8d%8d%8d%8d\n",
                   I0Dim_1,I0Dim_2,1,0,0,0,0,0,0,0);
    fprintf(OutputHeaderId,"% s\n",
            filename_name(Temp,IO_len,OutputBinaryName));
 
    fclose(OutputHeaderId);
 
    /* write binary file */
 
    t_length = sizeof(float)*(I0Dim_1*I0Dim_2);
    if (pcb->TestBit) printf("Length of output binary file = %d\n",t_length);
   
    printf("Writing %s\n",OutputBinaryName);
    if ((OutputBinaryId=fopen(OutputBinaryName,"wb")) == NULL) return;
 
    if ( (bsl_output_byteorder) == edf_byteorder() ) {
      fwrite(I0Data,t_length,1,OutputBinaryId);
    } else {
      // swap byte order
      void* dest=NULL;
      if (!( dest = (void*) malloc( t_length ))) return;
      edf_bswap( dest, I0Data, sizeof(float), I0Dim_1*I0Dim_2 );
      fwrite(dest,t_length,1,OutputBinaryId);
      if (dest) free(dest); dest=NULL;
    }
 
    fclose(OutputBinaryId);
 
  } else {         
    if (pcb->TestBit) printf("Output into multi bsl file\n");
 
    /* Output into multi bsl file (otoko) format */
 
    BslFileName ( OutputHeaderName, fname_full, 0 );
    BslFileName ( OutputBinaryName, fname_full, 1 );
 
    if (pcb->TestBit) printf("OutputHeaderName = %s\n",OutputHeaderName);
    if (pcb->TestBit) printf("OutputBinaryName = %s\n",OutputBinaryName);
 
    /* write header file */
    if ((OutputHeaderId=fopen(OutputHeaderName,"wb")) == NULL) {
      printf("ERROR : Cannot create % s\n",OutputHeaderName); return; }
    if (ihb[0].Title.I) {
      fprintf(OutputHeaderId,"%80s\n",ihb[0].Title.V);}
      else fprintf(OutputHeaderId,"first header\n");
    fprintf(OutputHeaderId,"second header\n");
 
    fprintf(OutputHeaderId,"%8d%8d%8d%8d%8d%8d%8d%8d%8d%8d\n",
            I0Dim_1,I0Dim_2,FrameCounter,0,0,0,0,0,0,0);
    fprintf(OutputHeaderId,"% s\n",filename_name(Temp,IO_len,OutputBinaryName));
 
    fclose(OutputHeaderId);
 
    /* write binary file */
 
    t_length = sizeof(float)*(I0Dim_1*I0Dim_2);
    if (pcb->TestBit) printf("Length of output binary file = %d\n",t_length);
 
    if  (ImageIsFirst( pcb, 0, ihb )) {
      printf("Writing %s\n",OutputBinaryName);
 
      if ((OutputBinaryId=fopen(OutputBinaryName,"wb")) == NULL) return;
 
      if ( (bsl_output_byteorder) == edf_byteorder() ) {
        fwrite(I0Data,t_length,1,OutputBinaryId);
      } else {
        // swap byte order
        void* dest=NULL;
        if (!( dest = (void*) malloc( t_length ))) return;
        edf_bswap( dest, I0Data, sizeof(float), I0Dim_1*I0Dim_2 );
        fwrite(dest,t_length,1,OutputBinaryId);
        if (dest) free(dest); dest=NULL;
      }
 
      fclose(OutputBinaryId);
    } else {
      printf("Appending data to %s\n",OutputBinaryName);
 
      if ((OutputBinaryId=fopen(OutputBinaryName,"ab")) == NULL) return;
 
      if ( (bsl_output_byteorder) == edf_byteorder() ) {
        fwrite(I0Data,t_length,1,OutputBinaryId);
      } else {
        // swap byte order
        void* dest=NULL;
        if (!( dest = (void*) malloc( t_length ))) return;
        edf_bswap( dest, I0Data, sizeof(float), I0Dim_1*I0Dim_2 );
        fwrite(dest,t_length,1,OutputBinaryId);
        if (dest) free(dest); dest=NULL;

      }
 
      fclose(OutputBinaryId);
 
    }
 
  }

  printf("\n");

  printf("Input file         : % s\n",
          filename_full(fname_full,IO_len,pcb->ib[1].FileName));
  printf("Input image number : % d\n",ihb[1].ImNum);
  printf("Output header file : % s\n",OutputHeaderName);
  printf("Output binary file : % s\n",OutputBinaryName);

} /* saxs_bsl*/

/*---------------------------------------------------------------------------
user_io
Do all the keyboard io and return cb, and ib
---------------------------------------------------------------------------*/
void user_io(CmdBlk * pcb, ImgBlk * ib, int * pstatus)
{
  char  progfn[InputLineLength];
  ImgHeadBlk ihb[BlockNum];

  float ROff_1, RPs_1, UOff_1, UPs_1;
  float ROff_2, RPs_2, UOff_2, UPs_2;

  IO_line * pPrefix;

 /* Determine program name without directory */
   (void) RightFR((char *) pcb->argv[0],DirSeparator,progfn,InputLineLength);

  /* show usage if no arguments are given */
  if (pcb->argc<=1) printf("Usage: %s %s\n",progfn,Usage);

  /*--- Write name of program ---*/
  printf("\n");
  printf("   %s %s\n",progfn,Version);
  printf("\n");

  pPrefix = (IO_line*) option_parameter_search( SPrefix, pcb, pstatus );

  Saxs2BslFilenames ( pcb, ib, ihb, 0, BlockNum-1, pstatus);
  if (*pstatus!=Success) return;
  GetReference(pcb->RSys.V,1,ihb,&ROff_1,&ROff_2,&RPs_1,&RPs_2,pstatus );
  if (*pstatus!=Success) return;
  GetReference(pcb->USys.V,1,ihb,&UOff_1,&UOff_2,&UPs_1,&UPs_2,pstatus );
  if (*pstatus!=Success) return;

  printf("\n");
  if (ib[1].Name.I)    printf("input file         : %s\n",ib[1].Name.V);
  if (pPrefix->I)  
                       printf("output header      : %s\n",pPrefix->V);
  if (ib[1].First.I)   printf("first image        : %d\n",ib[1].First.V);
  if (ib[1].Last.I)    printf("last image         : %d\n",ib[1].Last.V);
  if (ib[1].Inc.I)     printf("increment          : %d\n",ib[1].Inc.V);
  if (ib[0].Dummy.I)   printf("output dummy       : %s\n",ib[0].Dummy.V);
  if (ib[0].Dim[1].I)  printf("output dimension 1 : %s\n",ib[0].Dim[1].V);
  if (ib[0].Dim[2].I)  printf("output dimension 2 : %s\n",ib[0].Dim[2].V);
  printf("\n");

  if (pcb->TestBit) {
    PrintBlocks ( pcb, ib );
    printf("\n"); }

  return;
} /* user_io */

/*---------------------------------------------------------------------------
main
---------------------------------------------------------------------------*/

#if MAKE_FUNCTION
# define MAIN main_saxs_bsl
#else
# define MAIN main
#endif

int MAIN (int argc, char *argv[])
{
  CmdBlk cb;                /* command block  */
  ImgBlk ib[BlockNum];      /* image blocks */

  int status;
  int arg_no = 0;

  IO_long Bsl, Single;

 /* Init options, control block and image blocks */
  InitOptions( Usage, Version, NULL, TRUE, &cb, ib, BlockNum );

  option_define ( IO_tpline, SPrefix, "A01000", &cb );
  option_define ( IO_tplong, SBobo, "0", &cb );
  option_define ( IO_tpflag, SSingle, "FALSE", &cb );
  option_define ( IO_tpflag, SBsl, "FALSE", &cb );

 /* Read options from argument list */
  ReadOptions( argv, &arg_no, &cb, ib, &status);

 /* copy Bsl to Single */
 /* Force reference system to SAXS */
  if (status==Success) Bsl    = option_flag( SBsl, &cb, &status );
  if (status==Success) Single = option_flag( SSingle, &cb, &status );

  if (status==Success)
    if (Bsl.I) {
      printf("Please use option +/-%s instead of +/-%s\n", SSingle, SBsl);
      if (!Single.I) {
        option_flag_update( Bsl.V, TRUE, SSingle, &cb, &status );
        }
      }

  /* USER KEYBOARD I/O */
  if (status==Success) {
    argv_start ( &cb, 1 );
    user_io( &cb, ib, &status);
    argv_end( &cb ); /* must be called after user_io */
  }

  /* RESET FRAME COUNTER */
  FrameCounter=0l;

  /* SEQUENCE CALCULATION */
  if (status==Success) 
    IMAGELOOP( &cb, ib, saxs_bsl, NULL, NULL, TRUE, &status );

  return( ReportSaxsStatus( status, 0 ) );

} /* MAIN */
