/***************************************************************************/
/* Written 1994++ by Peter Boesecke                                        */
/* Copyright (C) 2011 European Synchrotron Radiation Facility              */
/*                       Grenoble, France                                  */
/*                                                                         */
/*    Principal authors: Peter Boesecke  (boesecke@esrf.eu)                */
/*                                                                         */
/*    This program is free software: you can redistribute it and/or modify */
/*    it under the terms of the GNU General Public License as published by */
/*    the Free Software Foundation, either version 3 of the License, or    */
/*    (at your option) any later version.                                  */
/*                                                                         */
/*    This program is distributed in the hope that it will be useful,      */
/*    but WITHOUT ANY WARRANTY; without even the implied warranty of       */
/*    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the        */
/*    GNU General Public License for more details.                         */
/*                                                                         */
/*    You should have received a copy of the GNU General Public License    */
/*    along with this program.  If not, see <http://www.gnu.org/licenses/>.*/
/***************************************************************************/
/*+++
1 SaxsRoutine

2 DESCRIPTION
  Some general routines for coordinate system determination (GetReference...),
  region of interest determination (GetLinearRange, GetImageRange), image data 
  filters (TransformFloat2d, ClipFloat2d, RebinFloat2d, GnawFloat2d). See 
  example routines for details.

2 CALL
     ArgvBlocks()
     ArgvFilenames()
     GetLinearRange()
     UpdateImageParametersC()
     UpdateImageParameters()
     GetImageRange()
     GetReferenceParameters()
     GetReference()
     GetReferenceParametersC()
     GetReferenceC()
     GetSmallRectangle()
     GetLargeRectangle()
     TransformFloat2d()
     ClipFloat2d()
     RebinFloat2d()
     GnawFloat2d()
     SaxsRoutineVersion()
     Projection_1()
     Projection_2()
     copy_amat()
     invert_amat()
     AffineFloat2d()
     Affine()
     arc_lfactor()
     

2 HISTORY
25-Dec-1995 PB test_routines.h
26-Jan-1996 PB ArgvFilenames: test of pcb->ImgBlkLen
28-Jan-1996 PB RebinFloat2d TransformFloat2d
               Transform replaced by TransformFloat2d (long int Dim_1 Dim_2!!!)
29-Mar-1996 PB ClipFloat2d
22-Apr-1996 PB Limit error in Ipol2ld corrected
19-May-1996 PB GnawFloat2d
20-May-1996 PB Error in GnawFloat2d for 0 Gnaws corrected
15-Oct-1996 PB V1.0 test_routines renamed to SaxsRoutine, SaxsRoutineVersion
11-Nov-1999 PB V1.01 DUMMYDEFINED updated
26-Nov-1999 PB V1.02 cc on dec alpha: statements of the type *pd++=*pd+xyz;
                     changed to { *pd=*pd+xyz; pd++; }
21-Dec-1999 PB V1.03 Ipol2, Ipol2d, Ipol2ld: check for NULL pointer
25-Dec-1999 PB V1.04 GetLinearOverlap
27-Dec-1999 PB V1.05 RowProjection and ColProjection
27-Dec-1999 PB V1.06 AffineFloat2d
31-Dec-1999 PB V1.07 GetReference: new reference system BEAM 
                     invert_amat, copy_amat
31-Dec-1999 PB V1.08 Affine
10-Jan-2000 PB V1.09 new reference systems CENTER and NORMAL,
                     NORMAL replaces BEAM
25-Feb-2000 PB V1.10 Error in i_2 loop range in RowProjection and 
                     ColProjection corrected
2000-08-04  PB V1.10 %f changed to %g
2000-11-16  PB V1.11 Ipol2ldw
2000-11-18  PB V1.12 weight in Ipol2ldw is also set without interpolation
2001-01-11  PB V1.13 ihb[0] is created in ArgvFilename
2001-07-07  PB V1.14 GetLinearOverlap (DoNotScale)
2001-07-08  PB V1.15 file name patterns
2001-07-09  PB V1.16 repeat command removed
2002-05-18  PB V1.17 Ipol2ldw description improved and unused lines removed
                     Isum2ldw added, Ipol2d replaced by Ipol2ld
                     GetLinearOverlap
2002-05-25  PB V1.18 GetSmallRectangle, GetLinearOverlap, GetImageOverlap
2002-05-25  PB V1.19 GetLargeRectangle, ERROUT, Isum2ldw corrected
2002-05-28  PB V1.20 Isum2ldw corrected
2002-05-29  PB V1.21 GetLinear/ImageRange replaces GetLinear/ImageOverlap
2002-05-30  PB V1.22 ROUND1, ROUND2 
2002-06-01  PB V1.23 Projection_1 
2002-06-02  PB V1.24 GetReferenceC, GetReferenceParametersC, Projection_2 
                     GetSmallRectangle, GetLargeRectangle
2002-06-02  PB V1.25 GetLinarOverlap, GetImageOverlap removed
2003-02-02  PB V1.26 ArgvFilenames: num[]
2003-02-04  PB V1.27 ArgvFilenames: input files closed at the end
2003-02-08  PB V1.28 ArgvFilenames: Dummy, DDummy -> IO_flexp,
                                    Bin -> IO_lvexp
2003-02-10  PB V1.29 Shift -> IO_flexp
2003-04-19  PB V1.30 GetReferenceC: bug in check of header values corrected
2003-11-30  PB V1.31 BinSiz, UpdateImageParametersC, UpdateImageParameters
2004-04-12  PB V1.32 ClipFloat2d: comparison for maxclip without a dummy
                                  corrected.
                                  Values exceeding the limits of minclip or 
                                  maxclip are set to minclip or maxclip if 
                                  no dummy is set. They used to be set to 0.
2004-07-08  PB V1.33 RebinFloat2d: copy always if input and output buffers 
                                   are different, parameter ave added
2005-03-31  PB V1.34 UpdateImageParametersC: 
                       rsys region updates PixSiz and Center
2005-08-04  PB V1.35 PrintImageHeaderBlock parameters updated
2005-08-07  PB V1.36 Isum2ldwE added, obvious bug in Isum2ldw corrected
                     (*pval*=vsign was replaced by *sum*=vsign. because
                      *pval is an input pixel value )
2005-09-20  PB V1.37 Parameter varline added to Projection_1 and Projection_2,
                     Isum2ldwE can be called with VarDat=NULL, default 0.0
                     for non-dummy pixels, 
                     AffineFloat2d, Affine modified for VarDat, number of
                     parameters increased,
                     GnawFloat2d modified for VarDat,
                     RowProjection and ColProjection removed (use Isum2ldwE),
                     Isum2ldwE: calculation of varweight corrected
2005-10-13  PB V1.38 invert_amat: eps defined as double (for cc)
2005-12-04  PB V1.39 All interpolation routines moved to ipol.c and ipol.h
2005-12-05  PB V1.40 
2007-04-19  PB V1.41 -Wall compiler warnings resolved
2007-04-20           %zu formats complemented by %lu
2007-06-10  PB V1.42 ArgvFilenames: NewImageHeader call updated, InitFunction
                                    should be passed
2008-08-05  PB V1.43 ArgvBlocks replaces ArgvFilenames
2008-11-19  PB V1.44 filename.h included
2009-10-02  PB V1.45 arc_lfactor extracted from SaxsArc.c and added here
2009-10-06  PB V1.46 Projection routines included from project.h
2009-11-11  PB V1.47 Projection_1&2: unused variables removed
2010-03-10  PB V1.48 ArgvBlocks: print format corrected
2011-07-24  PB V1.49 reference system IO_Tangens added.
2015-10-03  PB V1.49a cosmetics
2015-12-07  PB V1.50 Reducing warnings in print statements when switching
                     between 32-bit and 64-bit system by transforming
                     size_t variables with SIZE_T to unsigned long
                     before printing as %lu. %lz is not known by all
                     compilers.
2015-12-08  PB V1.51 UpdateImageParametersC:
                     Off (unused-but-set-variable) commented
2016-02-17  PB V1.52 ArgvBlocks: Create default filename without .gz suffix
2016-03-09  PB V1.53 UpdateLinearParameters => UpdateImageParametersC and
                     parameter RSys added
2016-04-02  PB V1.54 V1.54 Projection_1U added,
                     floor -> floorf (float), ceil -> ceilf (float), 
                     no double values are used here

---*/

/****************************************************************************
* SaxsRoutine                                                               *
****************************************************************************/
# include "SaxsImage.h"
# include "SaxsRoutine.h"
# include "SaxsError.h"
# include "filename.h"
# include "project.h"

#ifndef ROUND1
# define ROUND1( x ) FLOORF( ( x ) + 0.5 )
#endif

#ifndef ROUND3
# define ROUND3( x ) CEILF( ( x ) - 0.5 )
#endif

/***************************************************************************
* Portables Print Format for size_t variable (to be printed with %lu)      *
***************************************************************************/
# define SIZE_T (unsigned long)

/*----------------------------------------------------------------------------
 Constants
-----------------------------------------------------------------------------*/
const char * SRName = "SaxsRoutine";

/*----------------------------------------------------------------------------
1 SaxsRoutineVersion

2 DESCRIPTION
  Returns a pointer to the version string.
  char * SaxsRoutineVersion ( void )

2 HISTORY
  15-Oct-1996 PB
----------------------------------------------------------------------------*/
char * SaxsRoutineVersion ( void )
{ static char * Version = SR_Version;
  return ( Version );
} /* SaxsRoutineVersion */

/*----------------------------------------------------------------------------
RebinFloat2d
  Two dimensional rebinning of a float array. The result is written into the
  input array and the new dimensions are returned.
Parameters
  float * dataIn  (i)     pointer to input array
  float * dataOut (o)     pointer to output array
  long * dim_1    (i/o)   pointer to input/output dimension 1
  long * dim_2    (i/o)   pointer to input/output dimension 2
  float dummy     (i)     dummy value
  float ddummy    (i)     ddummy value
  long bin_1      (i)     rebin factor 1 (>=1)
  long bin_2      (i)     rebin factor 2 (>=1)
  long ave        (i)     average!=0/sum==0 pixel values
-----------------------------------------------------------------------------*/
void RebinFloat2d ( float * dataIn, float * dataOut,
                    long int * dim_1, long int * dim_2,
                    float dummy, float ddummy,
                    long int bin_1, long int bin_2,
                    long int ave )
{ register long int j_1, j_2, i_1, i_2;
  float * pin, * pout;
  float value, sum, count;

  bin_1 = MAX2(1,bin_1);
  bin_2 = MAX2(1,bin_2);

  if ((bin_1>1) || (bin_2>1) || (dataOut!=dataIn)) {
    pout = dataOut;
    for (j_2=0;j_2<=*dim_2-bin_2;j_2+=bin_2)
      for (j_1=0;j_1<=*dim_1-bin_1;j_1+=bin_1) {
        sum = 0.0; count = 0.0;
        for (i_2=j_2;i_2<j_2+bin_2;i_2++) {
          pin = ABSPTR(dataIn,*dim_1,*dim_2,j_1,i_2);
          for (i_1=j_1;i_1<j_1+bin_1;i_1++) {
            value = *(pin++);
            if NODUMMY(value,dummy,ddummy)
              { sum += value; count += 1.0; }
          }
        }
        if (count>0.0) *(pout++) = ave?sum/count:sum;
        else *(pout++) = dummy;
      }
    *dim_1 /= bin_1; *dim_2 /= bin_2;
  }

} /* RebinFloat2d */

/*---------------------------------------------------------------------------
  ClipFloat2d
  Sets all values of the data field outside minclip and maxclip to dummy.
  ---------------------------------------------------------------------------*/
void ClipFloat2d( float *data, long int dim_1, long int dim_2,
                         float dummy, float ddummy,
                         float minclip, int minclipI,
                         float maxclip, int maxclipI )
{ register long int i;
  long int datanum;
  float *pdata;

  datanum = dim_1 * dim_2;
  pdata = data;
  if (minclipI) {
    if (DUMMYDEFINED(dummy,ddummy)) {
      for (i=0;i<datanum;i++) {
        if NODUMMY(*pdata,dummy,ddummy) {
           if (minclip>*pdata) *pdata = dummy;
           }
        pdata +=1;
        }
      } else for (i=0;i<datanum;i++) {
               if (minclip>*pdata) *pdata = minclip;
               pdata +=1;
               }
    } /* if (minclipI) */

  pdata = data;
  if (maxclipI) {
    if (DUMMYDEFINED(dummy,ddummy)) {
      for (i=0;i<datanum;i++) {
        if NODUMMY(*pdata,dummy,ddummy) {
           if (maxclip<*pdata) *pdata = dummy;
           }
        pdata +=1;
        }
      } else for (i=0;i<datanum;i++) {
               if (maxclip<*pdata) *pdata = maxclip;
               pdata +=1;
               }
    } /* if (maxclipI) */

} /* ClipFloat2d */

/*---------------------------------------------------------------------------
  TransformFloat2d
  Multiplies the data field with a factor and adds a constant.
  ---------------------------------------------------------------------------*/
void TransformFloat2d( float *data, long int dim_1, long int dim_2,
                       float dummy, float ddummy,
                       float factor, float constant )
{ register long int i;
  long int datanum;
  const float eps1 = 1e-9, eps2 = 1e-30;
  float *pdata;

  datanum = dim_1 * dim_2;
  pdata = data;
  if ((ABS(factor-1.0)>eps1) || (ABS(constant)>eps2)) {
    if (DUMMYDEFINED(dummy,ddummy)) {
      for (i=0;i<datanum;i++) {
        if NODUMMY(*pdata,dummy,ddummy) *pdata = *pdata * factor + constant;
        pdata +=1;
        }
      } else for (i=0;i<datanum;i++) {
               *pdata = *pdata * factor + constant; pdata++;
               }
    } /* if ((ABS ... */

} /* TransformFloat2d */

/*----------------------------------------------------------------------------
GnawFloat2d
  Sets all pixels inside an ellipse around a dummy to dummy.
  The result is written into the input array. The routine
  allocates a short integer array with the same dimensions as the
  data array for intermediate results. Its memory is released after
  successful termination of GnawFloat2d.
Parameters
  float * data    (i)     input/output array
  float * vardat  (i)     input/output variance array (NULL if unused)
  long    dim_1   (i)     input/output/tmp dimension 1
  long    dim_2   (i)     input/output/tmp dimension 2
  float   dummy   (i)     dummy value
  float   ddummy  (i)     ddummy value
  long    gnaw_1  (i)     axis 1 of the ellipse (>=0)
  long    gnaw_2  (i)     axis 2 of the ellipse (>=0)
  int   * pstatus (o)     exit status
-----------------------------------------------------------------------------*/
void GnawFloat2d ( float * data, float * vardat,
                   long int dim_1, long int dim_2,
                   float dummy, float ddummy,
                   long int gnaw_1, long int gnaw_2,
                   int * pstatus )
{ register long int i, j_1, j_2, i_1, i_2;
  long int i_1min;
  float * pdata, *pvardat;
  short int * tmp, * ptmp;
  size_t t_tmp;

  float value;
  long int gnaw2_1, gnaw2_2, gnaw2;
  long int dist2_1, dist2_2;

  *pstatus = Success;

  gnaw_1 = MAX2(0,gnaw_1);
  gnaw_2 = MAX2(0,gnaw_2);

  if ((gnaw_1>0) || (gnaw_2>0)) {

    /* allocate memory for tmp array*/
    t_tmp = sizeof(short int)*dim_1*dim_2;
    tmp = (short int *) malloc (t_tmp);
    if (tmp == (short int *) NULL) {
      *pstatus=NotEnoughMemoryAvailable; return; }

    /* clear tmp array */
    ptmp = tmp;
    for (i=0;i<dim_1*dim_2;i++) *ptmp++=0;

    /* search dummies in data array */
    gnaw2_1 = gnaw_1 * gnaw_1 + 1;
    gnaw2_2 = gnaw_2 * gnaw_2 + 1;
    gnaw2   = gnaw2_1*gnaw2_2;
    pdata = data;
    for (j_2=0;j_2<dim_2;j_2++)
      for (j_1=0;j_1<dim_1;j_1++) {
        value = *(pdata++);
        if DUMMY(value,dummy,ddummy) {
          /* create dummy ellipse around j_1, j_2 in tmp */
          for (i_2=MAX2(0,j_2-gnaw_2);i_2<MIN2(dim_2,j_2+gnaw_2+1);i_2++) {
            dist2_2 = i_2 - j_2; dist2_2 *= dist2_2;
            i_1min = MAX2(0,j_1-gnaw_1);
            ptmp = ABSPTR(tmp,dim_1,dim_2,i_1min,i_2);
            for (i_1=i_1min;i_1<MIN2(dim_1,j_1+gnaw_1+1);i_1++) {
              dist2_1 = i_1 - j_1; dist2_1 *= dist2_1;
              if (dist2_1*gnaw2_2+dist2_2*gnaw2_1<=gnaw2) *ptmp = 1;
              ptmp++;
            }
          }
        }
      }

    /* copy dummies from tmp to data */
    ptmp = tmp;
    pdata = data;
    if (vardat) { 
      pvardat = vardat;
      for (i=0;i<dim_1*dim_2;i++) {
        if (*ptmp++) {
          *pdata=dummy;
          *pvardat=VarDummy;
        }
        pdata++;
        pvardat++;
      }
    } else {
      for (i=0;i<dim_1*dim_2;i++) {
        if (*ptmp++) *pdata=dummy;
        pdata++;
      }
    }

    /* free tmp array */
    free(tmp);

  }

} /* GnawFloat2d */

/*---------------------------------------------------------------------------
1 GetReferenceC
  Determine the reference system and set Off and Ps for a single coordinate
 
2 PURPOSE
  Determine the reference system and set Off and Ps for a single image and a
  single coordinate. It should be used to determine default values from the 
  first image for user input. Inside image loops only GetReferenceParameters 
  should be used.
 
2 CALL
  void GetReferenceC  ( long RSys, ImgHeadBlk ihb[],
                          int coordinate, int blockno,
                          float *Off, float *Ps, int * pstatus );

  return value                    (o)   : void

                     long RSys    (i)   : reference system
  ImgHeadBlk ihb[blockno]->Dim    (i)
             ihb[blockno]->Offset
             ihb[blockno]->PixSiz
             ihb[blockno]->Center, ->Wavelength, ->SampleDistance
          float *Off              (o)   : offset
          float *Ps               (o)   : pixel size

          int * pstatus           (o)   : SAXS status (Success and Failed)
 
2 HISTORY
  2002-06-02  PB GetReferenceC
  2003-04-19  PB IO_Center, IO_Real: parameter checks corrected
-----------------------------------------------------------------------------*/
void GetReferenceC  ( long RSys, ImgHeadBlk ihb[],
                      int coordinate, int blockno,
                      float *Off, float *Ps, int * pstatus )
{ const char * RoutineName = "GetReferenceC";
  ImgHeadBlk * pihb;
 
  *pstatus=Success;
 
  pihb = &ihb[blockno];
 
  switch (RSys) {
    case IO_Array :
      ARRAYREF((*Off),(*Ps));
 
      break;
    case IO_Image :
      if (!(pihb->Offset[coordinate].I)) {
        *pstatus = Failed; break; }

      IMAGEREF(*Off,*Ps,pihb->Offset[coordinate].V);

      break;

    case IO_Center :
      if ( !((pihb->Offset[coordinate].I) && (pihb->Center[coordinate].I)) ) {
        *pstatus = Failed; break; }

      CENTERREF(*Off,*Ps,
                pihb->Offset[coordinate].V,pihb->Center[coordinate].V);

      break;

    case IO_Region  :
      if ( !(pihb->Offset[coordinate].I) ) { // BinSiz is always set
        *pstatus = Failed; break; }
 
      REALREF(*Off,*Ps,
              pihb->Offset[coordinate].V,pihb->BinSiz[coordinate].V);
 
      break;

    case IO_Real  :
      if ( !((pihb->Offset[coordinate].I) && (pihb->PixSiz[coordinate].I)) ) { 
        *pstatus = Failed; break; }

      REALREF(*Off,*Ps,
              pihb->Offset[coordinate].V,pihb->PixSiz[coordinate].V);

      break;

    case IO_Normal :
      if (!( (pihb->Offset[coordinate].I) && (pihb->PixSiz[coordinate].I) &&
             (pihb->Center[coordinate].I) )) { 
        *pstatus = Failed; break; }

      NORMALREF(*Off,*Ps,pihb->Offset[coordinate].V,
                pihb->PixSiz[coordinate].V,pihb->Center[coordinate].V);

      break;

    case IO_Tangens  :
      if (!( (pihb->Offset[coordinate].I) && (pihb->PixSiz[coordinate].I) &&
             (pihb->Center[coordinate].I) &&
             (pihb->SampleDistance.I) )) {
        *pstatus = Failed; break; }

      TANGENSREF(*Off,*Ps,pihb->Offset[coordinate].V,
                 pihb->PixSiz[coordinate].V,pihb->Center[coordinate].V,
                 pihb->SampleDistance.V);

    case IO_Saxs  :
      if (!( (pihb->Offset[coordinate].I) && (pihb->PixSiz[coordinate].I) &&
             (pihb->Center[coordinate].I) &&
             (pihb->SampleDistance.I) && (pihb->WaveLength.I) )) {
        *pstatus = Failed; break; }

      SAXSREF(*Off,*Ps,pihb->Offset[coordinate].V,
              pihb->PixSiz[coordinate].V,pihb->Center[coordinate].V,
              pihb->SampleDistance.V,pihb->WaveLength.V);

      break;

    default       :
      fprintf(ERROUT,"%s : Unknown reference system %ld\n",RoutineName,RSys);
      *pstatus = Failed; return;

  } /* switch */

  if (*pstatus!=Success) {
    fprintf(ERROUT,
      "%s : Missing parameters for the chosen reference system (%s)\n",
      RoutineName,reftostr(RSys)); return; }
 
} /* GetReferenceC */

/*---------------------------------------------------------------------------
1 GetReference
  Determine the reference system and set Off and Ps
 
2 PURPOSE
  Determine the reference system and set Off and Ps for a single image.
  Should be used to determine default values from the first image for
  user input. Inside image loops only GetReferenceParameters should be used.
 
2 CALL
  void GetReference( CmdBlk * pcb, int blockno, ImgHeadBlk ihb[],
                     float * Off_1, float * Off_2,
                     float * Ps_1,  float * Ps_2,
                     int * pstatus );
  return value            (o)   : void
  CmdBlk * pcb            (i)   : command block
             pcb->RSys    (i)   : reference system
  ImgHeadBlk pihb->Dim   (i)
             pihb->Offset
             pihb->PixSiz
             pihb->Center, ->Wavelength, ->SampleDistance
  float * Off_1, * Off_2  (o)   : offset
  float * Ps_1, * Ps_2    (o)   : pixel size
  int * pstatus           (o)   : SAXS status (Success and Failed)
 
2 HISTORY
  24-Apr-1995 Peter Boesecke from GetReferenceParameters
  27-Nov-1995 PB GetReference
  2002-06-02  PB using GetReferenceC
-----------------------------------------------------------------------------*/
void GetReference     ( long int RSys, int blockno, ImgHeadBlk ihb[],
                        float * Off_1, float * Off_2,
                        float * Ps_1,  float * Ps_2,
                        int * pstatus )
{
  GetReferenceC  ( RSys, ihb, 1, blockno, Off_1, Ps_1, pstatus );
  if (*pstatus!=Success) return;
 
  GetReferenceC  ( RSys, ihb, 2, blockno, Off_2, Ps_2, pstatus );
  if (*pstatus!=Success) return;
 
} /* GetReference */

/*---------------------------------------------------------------------------
1 GetReferenceParametersC
  Determine the reference system and set Off and Ps for a single coordinate

2 PURPOSE
  Determine the reference system, set Off, Ps and shift for a single
  coordinate. Only to be used inside image loops.

2 CALL
  void GetReferenceParametersC( long RSys, CmdBlk * pcb, ImgHeadBlk ihb[],
                                int coordinate, int MinImage, int MaxImage,
                                float Off[], float Ps[],
                                int * pstatus ); 
  return value            (o)   : void
              int RSys    (i)   : reference system
  CmdBlk * pcb		  (i)   : command block
  ImgHeadBlk ihb[MaxImage+1](i) : image header block 
             ihb[].Dim 	
             ihb[].Offset 
             ihb[].PixSiz
             ihb[].Center, .Wavelength, .SampleDistance
             ihb[0].Shift (i)   : shift of output image (image 0)
  int coordinate          (i)   : coordinate 1 or 2
  int MinImage            (i)   : smallest image number that should be used
  int MaxImage            (i)   : largest images, that should be used
                                  0 : output image, 1 1st input image etc.
                                  MinImage>MaxImage : no image
  float Off[]             (o)   : offset (index: 0, MinImage to MaxImage)
  float Ps[]              (o)   : pixel size (index: 0, MinImage to MaxImage)
  int *pstatus	  	  (o)   : SAXS status (Success and Failed)

2 HISTORY
  2002-06-02 Peter Boesecke using GetReferenceParameters
---------------------------------------------------------------------------*/
void GetReferenceParametersC( long RSys, CmdBlk * pcb, ImgHeadBlk ihb[],
                              int coordinate, int MinImage, int MaxImage,
                              float Off[], float Ps[], int * pstatus )
{ register int i;
  const char * RoutineName = "GetReferenceParametersC";

  if (pcb->TestBit) printf("%s : %s\n", RoutineName, reftostr(RSys));
  SetSaxsError( SRName, RoutineName, NULL );

  *pstatus = Success;

  for (i=MinImage;i<=MaxImage;i++) {

    GetReferenceC  ( RSys, ihb, coordinate, i, &Off[i], &Ps[i], pstatus );

    if (*pstatus!=Success) return; 

  } /* for (i=MinImage;i<=MaxImage ... */

  /* Subtract output shift for calculation */
  if (MinImage==0) {
    if (ihb[0].Shift[coordinate].I) Off[0] = Off[0]-ihb[0].Shift[coordinate].V;
  }

  if (pcb->TestBit) for (i=MinImage;i<=MaxImage;i++) {
    printf("Off_%u[%d] = % g, Ps_%u[%d] = % g\n", 
      coordinate,i,Off[i],coordinate,i,Ps[i]); 
  }

} /* GetReferenceParametersC */

/*---------------------------------------------------------------------------
1 GetReferenceParameters
  Determine the reference system and set Off and Ps
 
2 PURPOSE
  Determine the reference system and set Off and Ps and shift the output
  image. Only to be used inside image loops.
 
2 CALL
  void GetReferenceParameters( CmdBlk * pcb, ImgHeadBlk ihb[],
                               int MinImage, int MaxImage,
                               float Off_1[], float Off_2[],
                               float Ps_1[],  float Ps_2[],
                               int * pstatus );
  return value            (o)   : void
  CmdBlk * pcb            (i)   : command block
             pcb->RSys    (i)   : reference system
  ImgHeadBlk ihb[MaxImage+1](i) : image header block
             ihb[].Dim
             ihb[].Offset
             ihb[].PixSiz
             ihb[].Center, .Wavelength, .SampleDistance
             ihb[0].Shift (i)   : shift of output image (image 0)
  int MinImage            (i)   : smallest image number that should be used
  int MaxImage            (i)   : largest images, that should be used
                                  0 : output image, 1 1st input image etc.
                                  MinImage>MaxImage : no image
  float Off_1[], Off_2[]  (o)   : offset
  float Ps_1[], Ps_[]     (o)   : pixel size
  int *pstatus            (o)   : SAXS status (Success and Failed)
 
2 HISTORY
  26-Feb-1995 Peter Boesecke creation
---------------------------------------------------------------------------*/
void GetReferenceParameters( CmdBlk * pcb, ImgHeadBlk ihb[],
                             int MinImage, int MaxImage,
                             float Off_1[], float Off_2[],
                             float Ps_1[],  float Ps_2[],
                             int * pstatus )
{ 
  GetReferenceParametersC( pcb->RSys.V, pcb, ihb, 1, MinImage, MaxImage,
                           Off_1, Ps_1, pstatus );
  if (*pstatus!=Success) return;

  GetReferenceParametersC( pcb->RSys.V, pcb, ihb, 2, MinImage, MaxImage,
                           Off_2, Ps_2, pstatus );
  if (*pstatus!=Success) return;
 
} /* GetReferenceParameters */

/*---------------------------------------------------------------------------
NAME
 
   GetLargeRectangle
 
SYNOPSIS
 
  void GetLargeRectangle  ( CmdBlk *pcb,      ImgHeadBlk ihb[],
                            int   OutCoordinate, int InCoordinate,
                            int   OutImage, int   MinImage,   int   MaxImage,
                            float Off_1[],    float Ps_1[],
                            float Off_2[],    float Ps_2[],
                            float *W1,        float *W3 );
 
DESCRIPTION
 
   This routine calculates the common area of all images. It is the largest
   rectangular area that is contained in all images. The coordinates of
   the input and output images can be swapped (OutCoordinate, InCoordinate).
 
   *pcb          (i) : image header blocks
   OutCoordinate (i) : 1 or 2
   InCoordinate  (i) : 1 or 2
   OutImage      (i) : used if >= 0
   MinImage .. MaxImage (i) : >= 0 (range)
   Off_1, Ps_1   (i) : WORLD coordinate parameters of coordinate 1
   Off_2, Ps_2   (i) : WORLD coordinate parameters of coordinate 2
   *W1, *W3      (o) : Lower left and upper right corner of common region
                       for the selected coordinate. There is no overlap
                       if *W1>*W2.
 
HISTORY
   2002-06-02 Peter Boesecke
 
---------------------------------------------------------------------------*/
void GetLargeRectangle  ( CmdBlk *pcb,      ImgHeadBlk ihb[],
                          int   OutCoordinate, int InCoordinate,
                          int   OutImage, int   MinImage,   int   MaxImage,
                          float Off_1[],    float Ps_1[],
                          float Off_2[],    float Ps_2[],
                          float *W1,        float *W3 )
{ const char * RoutineName = "GetLargeRectangle";
  float w1, w3;
  float *OffOut, *PsOut, *Off, *Ps;
  int i;
 
  switch (InCoordinate) {
    case 1: Off = Off_1; Ps = Ps_1; break;
    case 2: Off = Off_2; Ps = Ps_2; break;
    default: fprintf(ERROUT,"ERROR: Undefined input coordinate %d in %s\n",
             InCoordinate, RoutineName); exit(-1);
  }
 
  /* region of output image OutImage or MinImage*/
  if (OutImage>=0) {
     switch (OutCoordinate) {
         case 1: OffOut = Off_1; PsOut = Ps_1; break;
         case 2: OffOut = Off_2; PsOut = Ps_2; break;
         default: fprintf(ERROUT,"ERROR: Undefined output coordinate %d in %s\n",
                  OutCoordinate, RoutineName); exit(-1);
     }
     w1=WORLD(LOWERBORDER,OffOut[OutImage],PsOut[OutImage]);
     w3=WORLD(ihb[OutImage].Dim[OutCoordinate]+LOWERBORDER,
              OffOut[OutImage],PsOut[OutImage]);
    } else {
     w1=WORLD(LOWERBORDER,Off[MinImage],Ps[MinImage]);
     w3=WORLD(ihb[MinImage].Dim[InCoordinate]+LOWERBORDER,
              Off[MinImage],Ps[MinImage]);
    }

  // to ascending order
  *W1=MIN2(w1,w3); *W3=MAX2(w1,w3);
 
  /* area that contains all images */
  for (i=MAX2(0,MinImage);i<=MaxImage;i++) {
    if ( (i!=OutImage) && (!ihb[i].DoNotScale) ) {
      w1=WORLD(LOWERBORDER,Off[i],Ps[i]);
      w3=WORLD(ihb[i].Dim[InCoordinate]+LOWERBORDER,Off[i],Ps[i]);
      *W1=MIN2(*W1,MIN2(w1,w3));
      *W3=MAX2(*W3,MAX2(w1,w3));
      } // if (i!=OutImage)
    } /* for (i */

} /* GetLargeRectangle */

/*---------------------------------------------------------------------------
NAME

   GetSmallRectangle

SYNOPSIS

  void GetSmallRectangle  ( CmdBlk *pcb,      ImgHeadBlk ihb[],
                            int   OutCoordinate, int InCoordinate,
                            int   OutImage, int   MinImage,   int   MaxImage,
                            float Off_1[],    float Ps_1[],
                            float Off_2[],    float Ps_2[],
                            float *W1,        float *W3 );

DESCRIPTION

   This routine calculates the common area of all images. It is the largest 
   rectangular area that is contained in all images. The coordinates of
   the input and output images can be swapped (OutCoordinate, InCoordinate).

   *pcb          (i) : image header blocks
   OutCoordinate (i) : 1 or 2
   InCoordinate  (i) : 1 or 2
   OutImage      (i) : used if >= 0
   MinImage .. MaxImage (i) : >= 0 (range)
   Off_1, Ps_1   (i) : WORLD coordinate parameters of coordinate 1
   Off_2, Ps_2   (i) : WORLD coordinate parameters of coordinate 2
   *W1, *W3      (o) : Lower left and upper right corner of common region
                       for the selected coordinate. There is no overlap 
                       if *W1>*W2.
   
HISTORY
   2002-06-02 Peter Boesecke

---------------------------------------------------------------------------*/
void GetSmallRectangle  ( CmdBlk *pcb,      ImgHeadBlk ihb[],
                          int   OutCoordinate, int InCoordinate,
                          int   OutImage, int   MinImage,   int   MaxImage,
                          float Off_1[],    float Ps_1[],
                          float Off_2[],    float Ps_2[],
                          float *W1,        float *W3 )
{ const char * RoutineName = "GetSmallRectangle";
  float w1, w3;
  float *OffOut, *PsOut, *Off, *Ps;
  int i;

  switch (InCoordinate) {
    case 1: Off = Off_1; Ps = Ps_1; break;
    case 2: Off = Off_2; Ps = Ps_2; break; 
    default: fprintf(ERROUT,"ERROR: Undefined input coordinate %d in %s\n",
             InCoordinate, RoutineName); exit(-1);
  }

  /* region of output image OutImage or MinImage*/
  if (OutImage>=0) {
     switch (OutCoordinate) {
         case 1: OffOut = Off_1; PsOut = Ps_1; break;
         case 2: OffOut = Off_2; PsOut = Ps_2; break;
         default: fprintf(ERROUT,"ERROR: Undefined output coordinate %d in %s\n",
                  OutCoordinate, RoutineName); exit(-1);
     }
     w1=WORLD(LOWERBORDER,OffOut[OutImage],PsOut[OutImage]);
     w3=WORLD(ihb[OutImage].Dim[OutCoordinate]+LOWERBORDER,
              OffOut[OutImage],PsOut[OutImage]);
    } else {
     w1=WORLD(LOWERBORDER,Off[MinImage],Ps[MinImage]);
     w3=WORLD(ihb[MinImage].Dim[InCoordinate]+LOWERBORDER,
              Off[MinImage],Ps[MinImage]);
    }
 
  // to ascending order
  *W1=MIN2(w1,w3); *W3=MAX2(w1,w3);
 
  /* overlap area that is common to all input images */
  for (i=MAX2(0,MinImage);i<=MaxImage;i++) {
    if ( (i!=OutImage) && (!ihb[i].DoNotScale) ) {
      w1=WORLD(LOWERBORDER,Off[i],Ps[i]);
      w3=WORLD(ihb[i].Dim[InCoordinate]+LOWERBORDER,Off[i],Ps[i]);
      *W1=MAX2(*W1,MIN2(w1,w3));
      *W3=MIN2(*W3,MAX2(w1,w3));
      } // if (i!=OutImage)
    } /* for (i */

} /* GetSmallRectangle */ 

/*---------------------------------------------------------------------------
NAME
 
   UpdateImageParametersC --- Update header 
 
SYNOPSIS                                                                                                                                      

   void UpdateImageParametersC( long RSys, CmdBlk *pcb, ImgHeadBlk ihb[],
                                int OutCoordinate, int InCoordinate,
                                int OutImage, int MinImage, int MaxImage,
                                float Off_1[],    float Off_2[],
                                float Ps_1[],     float Ps_2[],
                                int * pstatus );

DESCRIPTION                                                                                                                                   

    Updates header values of the output image for a single coordinate
    that must be changed after call of GetLinearRange. Currently, only 
    BinSize and PixSize must be updated.
    BinSize is always updated for rsys != IO_Region
    Center is updated for rsys == IO_Region and rsys == IO_Real

ARGUMENTS
long  RSys         (i) : reference system to use
CmdBlk *pcb
ImgHeadBlk ihb[]
int   OutCoordinate(i) : coordinate of the output image (1 or 2)
int   InCoordinate (i) : coordinate of the input image (1 or 2)
int   OutImage     (i) : block number of the output image (not used if <0)
int   MinImage     (i) : minimum input block number (not used if <0)
int   MaxImage     (i) : maximum input block number (not used if <MinImage)
float Off_1[], Ps_1[]  : program array offsets and pixel sizes of coordinate 1
float Off_2[], Ps_2[]  : program array offsets and pixel sizes of coordinate 2
int * pstatus      (o) : output status

HISTORY
  2003-11-30 PB
---------------------------------------------------------------------------*/
void UpdateImageParametersC( long RSys, CmdBlk *pcb, ImgHeadBlk ihb[],
                             int OutCoordinate, int InCoordinate,
                             int OutImage, int MinImage, int MaxImage,
                             float Off_1[],    float Off_2[],
                             float Ps_1[],     float Ps_2[],
                             int * pstatus )
{ const char * RoutineName = "UpdateImageParametersC";
  // float *Off; // unused-but-set-variable
  float *Ps;
  float Bin;
  float eps = 1e-30;

  *pstatus = Failed;

  if (pcb->TestBit) printf("%s : (OutCoordinate %d, InCoordinate %d)\n",
    RoutineName,OutCoordinate,InCoordinate);

  if ((OutImage>=0)&&(MinImage>=0)&&(MinImage<=MaxImage)) {
    switch (InCoordinate) {
      case 1: /* Off = Off_1; unused-but-set-variable */ Ps = Ps_1; break;
      case 2: /* Off = Off_2; unused-but-set-variable */ Ps = Ps_2; break;
      default: fprintf(ERROUT,"ERROR: Undefined input coordinate %d in %s\n",
               InCoordinate, RoutineName); exit(-1);
    }

    if (OutImage!=MinImage) {
      if ( fabs(Ps[MinImage]) < eps ) {
        *pstatus = DivByZero; return;
      }
      Bin = Ps[OutImage]/Ps[MinImage];
      if (RSys == IO_Region) { 
        /* Update output PixSiz if it was not set by image block (.I>=0) */
        if (ihb[OutImage].PixSiz[OutCoordinate].I>=0)
          ihb[OutImage].PixSiz[OutCoordinate].V *= Bin;
        /* Update output Center if it was not set by image block (.I>=0) */
        if (ihb[OutImage].Center[OutCoordinate].I>=0) {
          if ( fabs(Bin) < eps ) {
            *pstatus = DivByZero; return;
          }
          ihb[OutImage].Center[OutCoordinate].V /= Bin; 
        }
      } else if (RSys == IO_Real) {
        /* Update output BinSiz if it was not set by image block (.I>=0) */
        if (ihb[OutImage].BinSiz[OutCoordinate].I>=0)
          ihb[OutImage].BinSiz[OutCoordinate].V *= Bin;
        /* Update output Center if it was not set by image block (.I>=0) */
        if (ihb[OutImage].Center[OutCoordinate].I>=0) {
          if ( fabs(Bin) < eps ) {
            *pstatus = DivByZero; return;
          }
          ihb[OutImage].Center[OutCoordinate].V /= Bin;
        }
      } else {
        /* Update output BinSiz if it was not set by image block (.I>=0) */
        if (ihb[OutImage].BinSiz[OutCoordinate].I>=0)
          ihb[OutImage].BinSiz[OutCoordinate].V *= Bin;
      }
    }

  }

  *pstatus = Success;

} // UpdateImageParametersC

/*---------------------------------------------------------------------------
NAME
 
   UpdateImageParameters --- Update image header after GetReferenceParameters 
 
SYNOPSIS
 
   void UpdateImageParameters ( CmdBlk *pcb, ImgHeadBlk ihb[],
                                int MinImage, int MaxImage,
                                float Off_1[],    float Off_2[],
                                float Ps_1[],     float Ps_2[],
                                int *pstatus );
 
DESCRIPTION
    Updates header values of the output image that must be changed after
    call of GetReferenceParamters. Currently, only BinSize and PixSize must be
    updated.
    if (rsys == IO_Region): 
      PixSize must be updated (not implemented 2003-11-29)
    else BinSize must be updated
 
ARGUMENTS
CmdBlk *pcb
ImgHeadBlk ihb[]
int   MinImage     (i) : minimum input block number (not used if <0)
int   MaxImage     (i) : maximum input block number (not used if <MinImage)
float Off_1[], Ps_1[]  : program array offsets and pixel sizes of coordinate 1
float Off_2[], Ps_2[]  : program array offsets and pixel sizes of coordinate 2
int * pstatus      (o) :  output status
The output image block number is always 0.
 
HISTORY
  2003-11-29 PB
---------------------------------------------------------------------------*/
void UpdateImageParameters ( CmdBlk *pcb, ImgHeadBlk ihb[],
                             int MinImage, int MaxImage,
                             float Off_1[],    float Off_2[],
                             float Ps_1[],     float Ps_2[],
                             int *pstatus )
{ 
  UpdateImageParametersC ( pcb->RSys.V, pcb, ihb, 1, 1,
                           0, MinImage, MaxImage,
                           Off_1, Off_2, Ps_1, Ps_2, pstatus );
  if(*pstatus!=Success) return;

  UpdateImageParametersC ( pcb->RSys.V, pcb, ihb, 2, 2,
                           0, MinImage, MaxImage,
                           Off_1, Off_2, Ps_1, Ps_2, pstatus );
  //if(*pstatus!=Success) return;
 
} // UpdateImageParameters


/*---------------------------------------------------------------------------
NAME

   GetLinearRange --- linear overlap in world and program array indices 

SYNOPSIS

   void GetLinearRange      ( CmdBlk *pcb,      ImgHeadBlk ihb[],
                              int   OutCoordinate, int InCoordinate,
                              int   OutImage,   int   MinImage, int MaxImage,
                              int   W1InI,      float W1In,
                              int   W3InI,      float W3In,
                              float Off_1[],    float Ps_1[],
                              float Off_2[],    float Ps_2[],
                              float f1[],       float f3[],     float Df[],
                              int   *Imin,      int   *Imax,
                              float *Wmin,      float *Wmax,    float *DW,
                              int    mode );

   There must be at least MaxImage+1 image blocks defined.

DESCRIPTION
  Calculates the overlap region of the output image (BlockNo=OutImage) 
  and all input images (BlockNo=[MinImage..MaxImage]) in world 
  coordinates (*Wmin, *Wmax) and in program array indices (fmin[],fmax[]). 
  If MaxImage<MinImage only the region of block OutImage is returned.

  The values of OutImage, MinImage and MaxImage must be in the range 
  [0..L-1], where L is the allocated length of the arrays Off[], Ps[], fmin[] 
  and fmax[]. The arrays Off[] and Ps[] must have been initialized with
  GetReferenceParameters for all used blocks (usually 0 to L). 

  if swapout is set, the coordinates of the output image are swapped.

  The block OutImage determines the position of the calculation grid. Its 
  outer border is defined by the centers of the pixels in OutImage that 
  are still inside the overlap region. In this case, fmin and fmax of the
  output image are restricted to integer values.
  If OutImage<0, no output block is chosen and this option is switched off.

  If the flag DoNotScale is set in an input image block the image is 
  skipped.

ARGUMENTS
CmdBlk *pcb
ImgHeadBlk ihb[]
int   OutCoordinate: coordinate of the output image (1 or 2)
int   InCoordinate : coordinate of the input image (1 or 2)
int   OutImage     : block number of the output image (not used if <0)
int   MinImage     : block number of the first used input image (>=0)
int   MaxImage     : block number of the last used input image 
int   W1InI        : 0: world coordinate of lower left corner (WminIn) not set
float W1In         : world coordinate of lower left corner
int   W3InI        : 0: maximum world coordinate (WmaxIn) not set, otherwise set 
float W3In         : world coordinate of upper right corner
float Off_1[], Ps_1[]   : program array offsets and pixel sizes of coordinate 1
float Off_2[], Ps_2[]   : program array offsets and pixel sizes of coordinate 2
float f1[], f3[]     (o): range of program array coordinates corresponding to 
                          the lower left and upper right corners of the images 
float Df[]           (o): delta values of program array coordinates that
                          correspond to the delta values of the world
                          coordinate of the output image.
int                  (i): swap coordinate of output image
int   *Imin, *Imax   (o): range of program array indices in the output image 
                          (used for loops, only if OutImage>=0)
float *Wmin, *Wmax   (o): range of world coordinate corresponding to the 
                          pixel centers of the output image
float *DW            (o): delta value of world coordinate

Without output image (OutImage<0) Imin, Wmin, Wmax and DW are set to 0
and Imax is set to -1, Df[] is not accessed 


HISTORY
  1999-12-25 Peter Boesecke from GetImageOverlap (version 27-Apr-1995)
  2001-07-07 DoNotScale
  2002-05-20 f1, f3, W1, W3;
  2002-05-25 Imin, Imax, Df, DW
  2002-06-02 OutCoordinate, InCoordinate
---------------------------------------------------------------------------*/
void GetLinearRange      ( CmdBlk *pcb,      ImgHeadBlk ihb[],
                           int   OutCoordinate, int InCoordinate,
                           int   OutImage,   int   MinImage,   int   MaxImage,
                           int   W1InI,      float W1In,
                           int   W3InI,      float W3In,
                           float Off_1[],    float Ps_1[],
                           float Off_2[],    float Ps_2[],
                           float f1[],       float f3[],     float Df[],
                           int   *Imin,      int   *Imax,
                           float *Wmin,      float *Wmax,    float *DW,
                           int    mode )
{ const char * RoutineName = "GetLinearRange";
  register int i;
  float W1, W3;
  float w1, w3;
  float fmin, fmax;
  float *OffOut, *PsOut, *Off, *Ps;

  int maxloop=1;

  if (pcb->TestBit) printf("%s : mode=%d (OutCoordinate %d, InCoordinate %d)\n",
    RoutineName,mode,OutCoordinate,InCoordinate);

  switch (InCoordinate) {
    case 1: Off = Off_1; Ps = Ps_1; break;
    case 2: Off = Off_2; Ps = Ps_2; break;
    default: fprintf(ERROUT,"ERROR: Undefined input coordinate %d in %s\n",
             InCoordinate, RoutineName); exit(-1);
  }

  /* --- initialize output values */
  *Imin = 0;   *Imax = -1;
  *Wmin = 0.0; *Wmax = 0.0; *DW = 0.0;

  /* --- calculate the overlap area in world coordinates *W1, *W3 */
  switch (mode) {
    case 1  : GetSmallRectangle  ( pcb, ihb,
                                   OutCoordinate, InCoordinate,
                                   OutImage, MinImage, MaxImage,
                                   Off_1, Ps_1, Off_2, Ps_2, &W1, &W3 );

              /* restrict area to external region */
              if (W1InI) w1=W1In; else w1 = W1;
              if (W3InI) w3=W3In; else w3 = W3;
              W1=MAX2(W1,MIN2(w1,w3));
              W3=MIN2(W3,MAX2(w1,w3));
              break;
    case 2  : GetLargeRectangle  ( pcb, ihb,
                                   OutCoordinate, InCoordinate,
                                   OutImage, MinImage, MaxImage,
                                   Off_1, Ps_1, Off_2, Ps_2, &W1, &W3 );
              /* enlarge area by external region */
              if (W1InI) w1=W1In; else w1 = W1;
              if (W3InI) w3=W3In; else w3 = W3;
              W1=MIN2(W1,MIN2(w1,w3));
              W3=MAX2(W3,MAX2(w1,w3));
              break;
    default : fprintf(ERROUT,"%s : undefined mode %d\n",RoutineName,mode);
              exit(-1);
  }

  if (pcb->TestBit) {
    if (W1InI||W3InI) {
       printf("W1In = ");
       if (W1InI) printf("%g, ", W1In); else printf("(none), ");
       printf("W3In = ");
       if (W3InI) printf("%g, ", W3In); else printf("(none) ");
       printf("\n");
    }
  } // if

  if (OutImage>=0) {
     /* --- recalculate the overlap area from the size of the output array,
            the output image determines the calculation grid and
            only discrete array points are allowed */ 
     switch (OutCoordinate) {
         case 1: OffOut = Off_1; PsOut = Ps_1; break;
         case 2: OffOut = Off_2; PsOut = Ps_2; break;
         default: fprintf(ERROUT,"ERROR: Undefined output coordinate %d in %s\n",
                  OutCoordinate, RoutineName); exit(-1);
     }

     f1[OutImage]=INDEX(W1,OffOut[OutImage],PsOut[OutImage]);
     f3[OutImage]=INDEX(W3,OffOut[OutImage],PsOut[OutImage]); 

    /* find the minimum and maximum integer index inside the overlap area */ 
    if (f1[OutImage]<=f3[OutImage]) fmin=CEILF(f1[OutImage]); 
      else fmin=FLOORF(f1[OutImage]);

    if (pcb->TestBit>1) {
      printf(" Begin intermediate results\n");
      printf("       W1 = %g,      W3 = %g\n",W1,W3);
      printf("  f1[%3u] = %g, f3[%3u] = %g\n",
               OutImage,f1[OutImage],OutImage,f3[OutImage]);
      printf(" End intermediate results\n");
      }

    /* Looking for integer indices less than 0.5 pixels away from the edges */
    if (f1[OutImage]<=f3[OutImage]) {
      fmin=ROUND1(f1[OutImage]);fmax=ROUND3(f3[OutImage]);
      } else {
      fmin=ROUND1(f3[OutImage]);fmax=ROUND3(f1[OutImage]);
      }

    /* The absolute minimum for fmin is 0.0,
       the absolute maximum for fmax is 
       ihb[OutImage].Dim[OutCoordinate]-1, PB 05-Dec-1995 */
    fmin = MAX2(fmin,0.0);
    fmax = MIN2(fmax,(float)(ihb[OutImage].Dim[OutCoordinate]-1));

    /* recalculate f1, f3, W1 and W3 for the modified region */
    f1[OutImage]=fmin-0.5;
    f3[OutImage]=fmax+0.5;

    W1 = WORLD(f1[OutImage],OffOut[OutImage],PsOut[OutImage]);
    W3 = WORLD(f3[OutImage],OffOut[OutImage],PsOut[OutImage]);

    /* calculate loop parameters */
    *Imin = (int) FLOORF(fmin);
    *Imax = (int) FLOORF(fmax);
    *Wmin = WORLD(fmin,OffOut[OutImage],PsOut[OutImage]);
    *Wmax = WORLD(fmax,OffOut[OutImage],PsOut[OutImage]);

    maxloop = *Imax-*Imin+1; 
    *DW=W3-W1; if (maxloop>1) *DW /= maxloop; 

    // delta value of program array coordinate
    Df[OutImage]=f3[OutImage]-f1[OutImage]; if (maxloop>1) Df[OutImage]/=maxloop;

    if (pcb->TestBit) {
      printf(" Output image\n");
      printf("         W1 = %g,      W3 = %g\n",W1,W3);
      printf("    f1[%3u] = %g, f3[%3u] = %g, Df[%3u] = %g\n",
               OutImage,f1[OutImage],OutImage,f3[OutImage], 
               OutImage,Df[OutImage]);
      printf("  fmin      = %g, fmax    = %g\n",fmin,fmax);
      printf("  Imin      = %d, Imax    = %d\n",*Imin,*Imax);
      printf("       Wmin = %g,    Wmax = %g ,   DW = %g\n",*Wmin,*Wmax,*DW);
      }

  } // if (OutImage>=0)

  /* --- calculate the program array indices */
  for (i=MAX2(0,MinImage);i<=MaxImage;i++) {
    if ( (i!=OutImage) && (!ihb[i].DoNotScale) ) {
      f1[i]=INDEX(W1,Off[i],Ps[i]);
      f3[i]=INDEX(W3,Off[i],Ps[i]);
      if (OutImage>=0)
        Df[i]=f3[i]-f1[i]; if (maxloop>1) Df[i]/=maxloop; 
    } // if
  } // for

  /* print the program array indices */
  if (pcb->TestBit) {
    printf(" Input images\n");
    printf("         W1 = %g,      W3 = %g\n",W1,W3);
    for (i=MAX2(0,MinImage);i<=MaxImage;i++) {
      if ( (i!=OutImage) && (!ihb[i].DoNotScale) ) {
        if (OutImage>=0) {
          printf("    f1[%3u] = %g, f3[%3u] = %g, Df[%3u] = %g\n",
            i,f1[i],i,f3[i],i,Df[i]);
        } else {
          printf("    f1[%3u] = %g, f3[%3u] = %g\n",
            i,f1[i],i,f3[i]);
        }
      } // if (i!=OutImage)
    } /* for (i */
  } /* if (pcb-> ... */

} /* GetLinearRange */

/*---------------------------------------------------------------------------
1 GetImageRange
  Calculate the overlap region in the world, array boundaries [0..Dim-1]
 
2 PURPOSE
  Calculates the overlap region of the output image (0) with the input
  images (i>0) in the world, array boundaries [0..Dim-1]
  if imax<imin only the region of the output image is returned.
 
2 CALL
   void GetImageRange    ( CmdBlk * pcb, ImgHeadBlk ihb[],
                             int MinImage, int MaxImage,
                             float Off_1[], float Off_2[],
                             float Ps_1[],  float Ps_2[],
                             float f1_1[], float f3_1[], float Df_1[],
                             float f1_2[], float f3_2[], float Df_2[],
                             int   * Imin_1, int * Imax_1,
                             int   * Imin_2, int * Imax_2,
                             float * Wmin_1, float * Wmax_1, float *DW_1,
                             float * Wmin_2, float * Wmax_2, float *DW_2,
                             int    mode)
 
  Dimension of all arrays must be at least MaxImage+1.
 
  return value              (o): void
  CmdBlk * pcb              (i): command block
           pcb->TestBit     (i)
  ImgHeadBlk ihb[MaxImage+1](i): image header block
             ihb[]->Dim
  int MinImage              (i): smallest input image number that
                                 should be considered > 0
                                 The output image (0) is always
                                 taken into account
  int MaxImage              (i): largest input image number, that
                                 should be used
                                 1 1st input image etc.
                                 MinImage>MaxImage : no image
  float Off_1[], Off_2[]    (i): offset
  float Ps_1[], Ps_[]       (i): pixel size
  float f1_1[], f3_1[]      (o): range of program array coordinates 1
  float Df_1[]              (o): Delta values of program array coordinates 1
  float f1_2[], f3_2[]      (o): range of program array coordinates 2
  float Df_2[]              (o): Delta values of program array coordinates 2
  int   *Imin_1, *Imax_1    (o): range of program array indices of coordinate 1
                                 in the output image
  int   *Imin_2, *Imax_2    (o): range of program array indices of coordinate 2
                                 in the output image
  float * Wmin_1, * Wmax_1  (o): range of world coordinate 1 in the pixel 
                                 centers of the output image
  float * DW_1              (o): Delta value of world coordinate 1 in the output
                                 image
  float * Wmin_2, * Wmax_2  (o): range of world coordinate 2 in the pixel
                                 centers of the output image
  float * DW_2              (o): Delta value of world coordinate 2 in the output
                                 image
  int mode                  (i): 1 : common area only, 2: overall area
 
 
2 HISTORY
  26-Feb-1995 Peter Boesecke creation
  14-Apr-1995 Peter Boesecke Call error of Wmin_1, Wmin_2, Wmax_1, Wmax_1
                             corrected. The addresses of these arguments
                             are now passed.
  27-Apr-1995 Peter Boesecke Checked that the output image is always
                             taken into account.
  25-Dec-1999 Peter Boesecke splitted into two parts with GetLinearOverlap
  2002-05-20  Peter Boesecke f1, f3, W1 and W3
  2002-05-25  Peter Boesecke
---------------------------------------------------------------------------*/                                                              
void GetImageRange       ( CmdBlk * pcb, ImgHeadBlk ihb[],
                             int MinImage, int MaxImage,
                             float Off_1[], float Off_2[],
                             float Ps_1[],  float Ps_2[],
                             float f1_1[], float f3_1[], float Df_1[],
                             float f1_2[], float f3_2[], float Df_2[],
                             int   * Imin_1, int * Imax_1,
                             int   * Imin_2, int * Imax_2,
                             float * Wmin_1, float * Wmax_1, float *DW_1,
                             float * Wmin_2, float * Wmax_2, float *DW_2,
                             int   mode)
{ GetLinearRange  ( pcb, ihb, 1, 1, 0,   MinImage,   MaxImage,
                    FALSE, 0.0,  FALSE, 0.0,
                    Off_1, Ps_1, Off_2, Ps_2,
                    f1_1, f3_1, Df_1,
                    Imin_1, Imax_1,
                    Wmin_1, Wmax_1, DW_1, mode ); // coordinate 1

  GetLinearRange  ( pcb, ihb, 2, 2, 0,   MinImage,   MaxImage,
                    FALSE, 0.0,  FALSE, 0.0,
                    Off_1, Ps_1, Off_2, Ps_2,
                    f1_2, f3_2, Df_2,
                    Imin_2, Imax_2,
                    Wmin_2, Wmax_2, DW_2, mode ); // coordinate 2

} /* GetImageRange */

/*---------------------------------------------------------------------------
NAME

   Projection_1U --- (updated) projection of image rows to a line (including variance)

SYNOPSIS

    void Projection_1U     ( float *line, float *varline, 
                             float *lineweight, float *varlineweight, int dim,
                             int imin, int imax, float initvalue, float factor,
                             float *data, float *vardat, int dim_1, int dim_2,
                             float f1_1, float f3_1, float Df_1,
                             float f1_2, float f3_2,
                             float dummy, float ddummy, int ave, int testbit )

DESCRIPTION
Averaging/Integration of rows between f1_2 and f3_2 in the 2d array data
(float data[dim_1,dim_2]). The result is written to the 1d array line.
If mode==0 the output arrays line, lineweight, varline and varlineweight are
initialized (line with initvalue, varline with VarDummy, lineweight and 
varlineweight with 0). If lineweight and varlineweight are NULL pointers the
function works like Projection_1. In this case only a single call is possible
and mode can be 0 (integration) or 1 (averaging).

The first call to Projection_1U must be with mode=0, for adding new
integration ranges the next calls must be with mode=2. For averaging,
the last call must be with mode=1, otherwise mode=2 keeps the integrated
values in line and varline.

Each element i between imin and imax of line is filled with the 
average/integral of the rectangle between the edges with the index 
coordinates (fi_1, f1_2) and (fi_1+Df_1, f3_2) in the input data array, 
where fi_1 starts with f1_1 for imin and is incremented by Df_1 for each element.

f1_1, f3_1, f1_2 and f3_2 are index coordinates of the input data array.

ARGUMENTS
float *line          : 1d output/updated array float line[dim]
float *varline       : 1d output/updated variance array float varline[dim]
float *lineweight    : 1d output/updated array with weights float lineweight[dim]
float *varlineweight : 1d output/updated array with weights float varlineweight[dim]
long  dim            : dimensions of output arrays (dim=dim_1!)
long  imin, imax     : range of indices in output arrays
float initvalue      : all elements in line that could not be determined
                       are set to initvalue
                       (variance array values are set to VarDummy)
float factor         : multiplication factor for output values
float *data          : 2d input array float data[dim_1,dim_2]
float *vardat        : 2d input variance array float vardat[dim_1,dim_2]
long  dim_1, dim_2   : dimensions of the input array
float f1_1, f3_1     : lowest and highest index 1 in the input array
float Df_1           : increment of index coordinate 1
float f1_2, f3_2     : indices of integration limits
float dummy, ddummy  : dummy and ddummy of input array
                       (negative values are dummy values of vardat)
int   mode           : 0: init and integration, 1: update and average
                          (if lineweight==NULL and varlineweight==NULL
                           init and average, like Projection_1), 
                       2: integration (update all output arrays)
int   testbit        : 1/0: do/dont print debug information

HISTORY
2016-04-01 Peter Boesecke
----------------------------------------------------------------------------*/
void Projection_1U         ( float *line, float *varline,
                             float *lineweight, float *varlineweight, int dim,
                             int imin, int imax, float initvalue, float factor,
                             float *data, float *vardat, int dim_1, int dim_2,
                             float f1_1, float f3_1, float Df_1,
                             float f1_2, float f3_2,
                             float dummy, float ddummy, int mode, int testbit )
{ const char * RoutineName = "Projection_1U";

  if (testbit) printf("+++ %s\n",RoutineName);

  project_1u ( line, varline, 
              lineweight,varlineweight,dim,
              imin, imax, initvalue, factor,
              data, vardat, dim_1, dim_2,
              f1_1, f3_1, Df_1,
              f1_2, f3_2,
              dummy, ddummy, mode );

} /* Projection_1U */

/*---------------------------------------------------------------------------
NAME
 
   Projection_1 --- projection of image rows to a line (including variance)
 
SYNOPSIS
 
    void Projection_1      ( float *line, float *varline, int dim,
                             int imin, int imax, float initvalue, float factor,
                             float *data, float *vardat, int dim_1, int dim_2,
                             float f1_1, float f3_1, float Df_1,
                             float f1_2, float f3_2,
                             float dummy, float ddummy, int ave, int testbit )
 
DESCRIPTION
Averaging/Integration of rows between f1_2 and f3_2 in the 2d array data
(float data[dim_1,dim_2]). The result is written to the 1d array line.
The output array line is initialized with initvalue. Each element i between
imin and imax of line is filled with the average/integral of the rectangle
between the edges with the index coordinates (fi_1, f1_2) and (fi_1+Df_1, f3_2)
in the input data array, where fi_1 starts with f1_1 for imin and is 
incremented by Df_1 for each element.
 
f1_1, f3_1, f1_2 and f3_2 are index coordinates of the input data array.

ARGUMENTS
float *line          : 1d output array float line[dim]
float *varline       : 1d output variance array float varline[dim]
long  dim            : dimensions of output arrays (dim=dim_1!)
long  imin, imax     : range of indices in output arrays
float initvalue      : all elements in line that could not be determined
                       are set to initvalue
                       (variance array values are set to VarDummy)
float factor         : multiplication factor for output values
float *data          : 2d input array float data[dim_1,dim_2]
float *vardat        : 2d input variance array float vardat[dim_1,dim_2]
long  dim_1, dim_2   : dimensions of the input array
float f1_1, f3_1     : lowest and highest index 1 in the input array
float Df_1           : increment of index coordinate 1
float f1_2, f3_2     : indices of integration limits
float dummy, ddummy  : dummy and ddummy of input array 
                       (negative values are dummy values of vardat)
int   ave            : 0: integration, 1: average
int   testbit        : 1/0: do/dont print debug information
 
HISTORY
2005-09-17 Peter Boesecke
----------------------------------------------------------------------------*/
void Projection_1          ( float *line, float *varline, int dim,
                             int imin, int imax, float initvalue, float factor,
                             float *data, float *vardat, int dim_1, int dim_2,
                             float f1_1, float f3_1, float Df_1,
                             float f1_2, float f3_2,
                             float dummy, float ddummy, int ave, int testbit )
{ const char * RoutineName = "Projection_1";
 
  if (testbit) printf("+++ %s\n",RoutineName);

  project_1 ( line, varline, dim,
              imin, imax, initvalue, factor,
              data, vardat, dim_1, dim_2,
              f1_1, f3_1, Df_1,
              f1_2, f3_2,
              dummy, ddummy, ave );
 
} /* Projection_1 */

/*---------------------------------------------------------------------------
NAME
 
   Projection_2 --- projection of image columns to a line (including variance)
 
SYNOPSIS

   void Projection_2       ( float *line, float *varline, int dim,
                             int imin, int imax, float initvalue, float factor,
                             float *data, float *vardat, int dim_1, int dim_2,
                             float f1_1, float f3_1,
                             float f1_2, float f3_2, float Df_2,
                             float dummy, float ddummy, int ave, int testbit );
 
DESCRIPTION
Averaging/Integration of columns between f1_1 and f3_1 in the 2d array data
(float data[dim_1,dim_2]). The result is written to the 1d array line.
The output array line is initialized with initvalue. Each element i between
imin and imax of line is filled with the average/integral of the rectangle
between the edges with the index coordinates (f1_1, fi_2) and (f3_1, fi_2+Df_2)
in the input data array, where fi_2 starts with f1_2 for imin and is incremented
by Df_2 for each element.
 
f1_1, f3_1, f1_2 and f3_2 are index coordinates of the input data array.

ARGUMENTS
float *line          : 1d output array float line[dim]
float *varline       : 1d output variance array float varline[dim]
long  dim            : dimension of output array (dim=dim_1!)
long  imin, imax     : range of indices in output array
float initvalue      : all elements in line that could not be determined
                       are set to initvalue,
                       (variance array values are set to VarDummy)
float factor         : multiplication factor for output values
float *data          : 2d input array float data[dim_1,dim_2]
float *vardat        : 2d input variance array float vardat[dim_1,dim_2]
long  dim_1, dim_2   : dimensions of the input array
float f1_1, f3_1     : lowest and highest index 1 in the input array
float f1_2, f3_2     : indices of integration limits
float Df_2           : increment of index coordinate 2
float dummy, ddummy  : dummy and ddummy of input array 
                       (negative values are dummy values of vardat)
int   ave            : 0: integration, 1: average
int   testbit        : 1/0: do/dont print debug information
 
HISTORY
2005-09-16 Peter Boesecke
----------------------------------------------------------------------------*/
void Projection_2          ( float *line, float *varline, int dim,
                             int imin, int imax, float initvalue, float factor,
                             float *data, float *vardat, int dim_1, int dim_2,
                             float f1_1, float f3_1,
                             float f1_2, float f3_2, float Df_2,
                             float dummy, float ddummy, int ave, int testbit )
{ const char * RoutineName = "Projection_2";
 
  if (testbit) printf("+++ %s\n",RoutineName);

  project_2             ( line, varline, dim,
                          imin, imax, initvalue, factor,
                          data, vardat, dim_1, dim_2,
                          f1_1, f3_1,
                          f1_2, f3_2, Df_2,
                          dummy, ddummy, ave );
 
} /* Projection_2 */

/*---------------------------------------------------------------------------
NAME

  copy_amat --- copies an affine transformation matrix

SYNOPSIS
  
  int copy_amat(double t[3][2], double p[3][2]);

DESCRIPTION
    Copies the affine transformation matrix t to p.

RETURN VALUE
    0: Success

-----------------------------------------------------------------------------*/
int copy_amat(double t[3][2], double p[3][2])
{ register int i,j;
  for(j=0;j<2;j++) for (i=0;i<3;i++) p[i][j] = t[i][j];
  return(0);
} /* copy_amat */

/*---------------------------------------------------------------------------
NAME

  invert_amat --- inverts an affine transformation matrix

SYNOPSIS

  int invert_amat(double t[3][2], double p[3][2]);

DESCRIPTION

        The affine transformation T: (k,l) -> (i,j) is defined as:

             i = t[0][0]*k+t[1][0]*l + t[2][0];
             j = t[0][1]*k+t[1][1]*l + t[2][1];

        The inverse transformation matrix P: (i,j) -> (k,l) is defined as:

             k = p[0][0]*i+p[1][0]*j + p[2][0];
             l = p[0][1]*i+p[1][1]*j + p[2][1];

        If the determinante of the inverse transformation matrix 
        is smaller than 1e-6, invert_amat returns with error.

ARGUMENTS

   double t[3][2] : transformation matrix T
   double p[3][2] : inverted matrix P (can be identical to T)

RETURN VALUE
    0: Success
   -1: matrix inversion error (determinante too small)

-----------------------------------------------------------------------------*/
int invert_amat(double t[3][2], double p[3][2])
{ const double eps = 1e-6;
  double dett;
  double tmp[3][2];

  /* calculation of inverse transformation matrix */

  dett = t[0][0]*t[1][1]-t[1][0]*t[0][1];

  if (fabs(dett)<eps) return(-1); // matrix inversion error

  tmp[0][0] = t[1][1]/dett; tmp[1][0] = -t[1][0]/dett;
    tmp[2][0] = (t[1][0]*t[2][1]-t[2][0]*t[1][1])/dett;
  tmp[0][1] = -t[0][1]/dett; tmp[1][1] = t[0][0]/dett;
    tmp[2][1] = (t[0][1]*t[2][0]-t[0][0]*t[2][1])/dett;

  copy_amat(tmp,p);

  return(0);

} /* invert_amat */

/*----------------------------------------------------------------------------
NAME

   AffineFloat2d --- Affine transformation of a 2d float image

SYNOPSIS

   AffineFloat2d         ( float *data, float * vardat, long dim_1, long dim_2,
                           float dummy, float ddummy,
                           float *out, float *varout, long odim_1, long odim_2,
                           float odummy, float oddummy,
                           double t[3][2], int isw, initialize, int *pstatus );

DESCRIPTION

        float t[3][2] : Transformation matrix data[i,j] -- out[k,l]
        
        if (isw > 0) {
             i=t[0][0]*k+t[1][0]*l+t[2][0];
             j=t[0][1]*k+t[1][1]*l+t[2][1];
           } else {
             k=t[0][0]*i+t[1][0]*j+t[2][0];
             l=t[0][1]*i+t[1][1]*j+t[2][1];
           }

        If, in the case of isw<=0, the determinante of the inverse 
        transformation matrix is smaller than 1e-6:
        (fabs(t[0][0]*t[1][1]-t[1][0]*t[0][1]) <  1e-6)
        AffineFloat2d returns without transformation and sets 
        status to FloatingPointError. 

        If vardat and varout are not NULL pointers, the variances
        of the output values are calculated.

        In case of success AffineFloat2d sets status to Success.

ARGUMENTS

 const float * data      (i) :  input image 
 const float * vardat    (i) :  input variance (or NULL if no variance 
                                calculation should be done)
 long int idim_1, idim_2 (i) :  dimensions of input image
 float dummy, ddummy     (i) :  input dummy and ddummy
 float * out             (o) :  output image
 float * varout          (o) :  output variance
 long int odim_1, odim_2 (i) :  dimensions of output image
 float odummy, oddummy   (i) :  output dummy and ddummy
 float t[3][2]           (i) :  transformation matrix 
 int isw                 (i) :  parameter specifying processing contents
 int initialize          (i) :  if not 0, the output arrays are initialized 
                                with dummies
 int * pstatus           (o) :  output status

RETURN VALUE
 void 

HISTORY
 1999-12-27 V1.0 Peter Boesecke
 2005-09-15 V1.1 PB variance calculation
----------------------------------------------------------------------------*/
void AffineFloat2d( float *data, float *vardat, long int dim_1, long int dim_2,
                    float dummy, float ddummy,
                    float *out, float *varout, long int odim_1, long int odim_2,
                    float odummy, float oddummy, double t[3][2], int isw, 
                    int initialize, int *pstatus )
{ double p00,p10,p20,p01,p11,p21;

  register long int i_1, i_2;
  float f_1, f_2;
  float sum, weight, value;
  float varsum, varweight, varval;
  float *pout, *pvarout;
  float VarDDummy=DDSET(VarDummy);

  *pstatus = Failed;

  if (isw<=0) {
    /* invert matrix */
    if (invert_amat(t,t)) { *pstatus = FloatingPointError; return; }
   } // if (isw<=0)

  p00 = t[0][0]; p10 = t[1][0];   p20 = t[2][0];
  p01 = t[0][1]; p11 = t[1][1];   p21 = t[2][1];

  if ( initialize ) {
    pout = out;
    pvarout = varout;
    for (i_2=0;i_2<odim_2;i_2++) {
      for (i_1=0;i_1<odim_1;i_1++) {
        *pout++ = odummy; 
        if ( varout ) *pvarout++ = VarDummy;
      } // i_1
    } // i_2
  } // initialize

  pout = out;
  pvarout = varout;
  for (i_2=0;i_2<odim_2;i_2++) {
    for (i_1=0;i_1<odim_1;i_1++) {
      f_1= (float) ( (p00*i_1)+(p10*i_2)+ p20 );
      f_2= (float) ( (p01*i_1)+(p11*i_2)+ p21 );

      if ( varout ) {
        if ( Isum2ldwE (data, vardat,(int) dim_1, (int) dim_2,
               dummy, ddummy, f_1, f_2, f_1+1.0, f_2+1.0,
               &sum, &weight, &varsum, &varweight) ) {
          /* then do something with the data */
          value = sum/weight;
          UPDATE( *pout, value, odummy, oddummy );
          if (varsum>=0.0) {
            // The scaling of varweight with varweight/(p00*p11-p10*p01)
            // does not seem to be correct because the value of 1 contributing
            // input pixel is transformed to 1 output pixel
            // varweight *= fabs(varweight/(p00*p11-p10*p01));
            varval = varsum/(varweight*varweight);
            UPDATE( *pvarout, varval, VarDummy, VarDDummy );
          } 
        } // Isum2ldwE
      } else {
        if ( Ipol2ld (data, (int) dim_1, (int) dim_2, 
               dummy, ddummy, f_1, f_2, &value) ) {
          UPDATE( *pout, value, odummy, oddummy );
        } // Ipol2ld
      } /* if varout */
      pout++;
      pvarout++;
    } // i_1
  } // i_2
  
  *pstatus = Success;

} /* AffineFloat2d */

/*-----------------------------------------------------------------------
NAME

   Affine --- Affine transformation of an image block

SYNOPSIS

   void Affine (  CmdBlk * pcb, ImgHeadBlk ihb[],
                  int InputBlock, int OutputBlock,
                  double t[3][2], int initialize, int * pstatus );

DESCRIPTION
Affine changes the planar orientation of the data using the 
subroutine AffineFloat2d. The transformation is applied to world
coordinates that are defined by the reference system (pcb->RSys.V). 
If required, the values of pixel size and offset of the output 
image are updated after the transformation. The input image block
is block 1 and the output image block is 0. The output shift 
parameters are taken into account.

   Transformation of world coordinates:

      W' = A * W + B;

      W  = WORLD( I , O , P  ) = P  * (O +I );
      W' = WORLD( I', O', P' ) = P' * (O'+I');

      Transformation of indices:

      P' * (O'+I') = A * ( P  * (O +I ) ) + B

            <=> I' = A' * I + B';


           ( i0 )         ( o0 )       ( p0,  0 )
      I  = |    |  ; O  = |    | ; P = |        |
           ( i1 )         ( o1 )       ( 0 , p1 )


           ( i0' )          ( op0 )        ( pp0,  0 )
      I' = |     |  ; O'  = |     | ; P' = |         |
           ( i1' )          ( op1 )        ( 0 , pp1 )


                       ( T[0,0], T[1,0] )       ( T[2,0] )
      T = (A,B) :  A = |                | ; B = |        |
                       ( T[0,1], T[1,1] )       ( T[2,1] )


      Tp = (A',B') :

                   ( Tp[0,0], Tp[1,0] )
             A' =  |                  |
                   ( Tp[0,1], Tp[1,1] )

             A' =  inv(P') * A * P

                   ( 1/pp0 ,   0  )   ( T[0,0], T[1,0] )   ( p0,      0  )
                =  |              | * |                | * |             |
                   (   0  , 1/pp1 )   ( T[0,1], T[1,1] )   (   0    , p1 )

                   ( Tp[2,0] )
             B' =  |         |
                   ( Tp[2,1] )

                =  inv(P') * B + A' * O - O'

                   ( 1/pp0,  0 )   ( T[2,0] )          ( o0 )    ( op0 )
                =  |           | * |        | + A'  *  |    |  - |     |
                   ( 0,  1/pp1 )   ( T[2,1] )          ( o1 )    ( op1 )

ARGUMENTS
  CmdBlk * pcb            (i)   : command block
             pcb->RSys    (i)   : reference system
  ImgHeadBlk ihb[block]   (i)   : image header block 
                                  (block=OutputBlock, InputBlock)
             ihb[].Dim
             ihb[].Offset
             ihb[].PixSiz
             ihb[].Center, .Wavelength, .SampleDistance
                    ihb[1].Data : input data array
                    ihb[0].Data : output data array
             ihb[0].Shift (i)   : shift of output image (image 0)
  int InputBlock                : input block number
  int OutputBlock               : output block number
  double t[3][2]          (i)   : world coordinates transformation matrix
  int initialize          (i)   : 0: only valid pixels are updated
                                  1: invalid pixels are set to dummy
  int * pstatus           (o)   : output status

HISTORY
1999-12-31 Peter Boesecke
2003-11-30 PB UpdateImageParameters
2005-07-30 PB VarDat
-----------------------------------------------------------------------*/
void Affine (  CmdBlk * pcb, ImgHeadBlk ihb[],
               int InputBlock, int OutputBlock,
               double t[3][2], int initialize, int * pstatus )
{ int j,k;
  int imax = pcb->ImgBlkLen;
  ImgHeadBlk *pin  = &ihb[InputBlock], 
             *pout = &ihb[OutputBlock];

  int isw=-1;
  double tp[3][2];
  double p0, p1, o0, o1;
  double pp0, pp1, op0, op1;

  double IndexCenIn_1, IndexCenIn_2, IndexCenOut_1, IndexCenOut_2;
  double pp[3][2];

  float *Off_1, *Off_2, *Ps_1, *Ps_2;
  Off_1 = (float *) malloc(sizeof(float)*imax);
  if (!Off_1) { *pstatus = NotEnoughMemoryAvailable; return;}
  Off_2 = (float *) malloc(sizeof(float)*imax);
  if (!Off_2) { *pstatus = NotEnoughMemoryAvailable; 
                 free(Off_1); return;}
  Ps_1 = (float *) malloc(sizeof(float)*imax);
  if (!Ps_1) { *pstatus = NotEnoughMemoryAvailable; 
               free(Off_1); free(Off_2); return;}
  Ps_2 = (float *) malloc(sizeof(float)*imax);
  if (!Ps_2) { *pstatus = NotEnoughMemoryAvailable; 
               free(Off_1); free(Off_2); free(Ps_1); return;}

  *pstatus = Success;

  if (imax<2) {
    fprintf(ERROUT,
      " ERROR in Affine: Not enough image blocks (%d / 2) found\n",
      imax);
    free(Off_1); free(Off_2); free(Ps_1); free(Ps_2);
    *pstatus=Failed; return; }

  GetReferenceParameters( pcb, ihb, 0, imax-1,
                          Off_1, Off_2, Ps_1,  Ps_2, pstatus );
  if (*pstatus!=Success) {
     free(Off_1); free(Off_2); free(Ps_1); free(Ps_2); 
     return; }

  p0  = (double) Ps_1[InputBlock];   p1 = (double) Ps_2[InputBlock];
  o0  = (double) Off_1[InputBlock];  o1 = (double) Off_2[InputBlock]; 
  pp0 = (double) Ps_1[OutputBlock];  pp1 = (double) Ps_2[OutputBlock];
  op0 = (double) Off_1[OutputBlock]; op1 = (double) Off_2[OutputBlock];

  free(Off_1); free(Off_2); free(Ps_1); free(Ps_2);

  /* include world coordinates in transformation matrix tp */
  tp[0][0] = t[0][0] * (p0/pp0);
  tp[1][0] = t[1][0] * (p1/pp0);
  tp[2][0] = t[2][0] / pp0 + tp[0][0] * o0 + tp[1][0] * o1 - op0;
  tp[0][1] = t[0][1] * (p0/pp1);
  tp[1][1] = t[1][1] * (p1/pp1);
  tp[2][1] = t[2][1] / pp1 + tp[0][1] * o0 + tp[1][1] * o1 - op1;

  if (pcb->TestBit) {
    for (k=0;k<2;k++) for (j=0;j<3;j++)
        printf("tp[%d][%d] = %g\n",j,k,tp[j][k]);
    }

  AffineFloat2d ( pin->Data, pin->VarDat, pin->Dim[1], pin->Dim[2],
                  pin->Dummy.V, pin->DDummy.V,
                  pout->Data, pout->VarDat, pout->Dim[1], pout->Dim[2],
                  pout->Dummy.V, pout->DDummy.V, tp, isw, initialize,
                  pstatus );
  if (*pstatus!=Success) return; 

  /* transformation of unused image parameters */

     if (isw>0) { copy_amat(tp,pp); } else {
       if (invert_amat(tp,pp)) { *pstatus=FloatingPointError; return; }
       } // if (isw>0)

     if (pcb->TestBit) {
       for (k=0;k<2;k++) for (j=0;j<3;j++)
           printf("pp[%d][%d] = %g\n",j,k,pp[j][k]);
       }

     if (pcb->RSys.V != IO_Region) { // update always bin size
          pout->BinSiz[1].V =
            pin->BinSiz[1].V/sqrt(pp[0][0]*pp[0][0]+pp[0][1]*pp[0][1]);
          pout->BinSiz[2].V =
            pin->BinSiz[2].V/sqrt(pp[1][0]*pp[1][0]+pp[1][1]*pp[1][1]);
      }

      switch (pcb->RSys.V) {
        case IO_Center: // update pixel size

          if ((pin->PixSiz[1].I)&&(pin->PixSiz[2].I)) {
            pout->PixSiz[1].V =
              pin->PixSiz[1].V/sqrt(pp[0][0]*pp[0][0]+pp[0][1]*pp[0][1]);
            pout->PixSiz[2].V =
              pin->PixSiz[2].V/sqrt(pp[1][0]*pp[1][0]+pp[1][1]*pp[1][1]);
            }
          break;
        case IO_Array: // like IO_Image
        case IO_Image: // like IO_Region
        case IO_Region: // update pixel size and position of center like IO_Real

          if ((pin->PixSiz[1].I)&&(pin->PixSiz[2].I)) {
            pout->PixSiz[1].V =
              pin->PixSiz[1].V/sqrt(pp[0][0]*pp[0][0]+pp[0][1]*pp[0][1]);
            pout->PixSiz[2].V =
              pin->PixSiz[2].V/sqrt(pp[1][0]*pp[1][0]+pp[1][1]*pp[1][1]);
            }
        case IO_Real: // update bin size and position of center

          if ((pin->Center[1].I)&&(pin->Center[2].I)) {

                IndexCenIn_1 = I2INDEX(pin->Center[1].V,pin->Offset[1].V);
                IndexCenIn_2 = I2INDEX(pin->Center[2].V,pin->Offset[2].V);

            //       W' = A * W + B
            IndexCenOut_1=pp[0][0]*IndexCenIn_1+pp[1][0]*IndexCenIn_2+pp[2][0];
            IndexCenOut_2=pp[0][1]*IndexCenIn_1+pp[1][1]*IndexCenIn_2+pp[2][1];

            pout->Center[1].V = INDEX2I(IndexCenOut_1,pout->Offset[1].V);
            pout->Center[2].V = INDEX2I(IndexCenOut_2,pout->Offset[2].V);
            }

           if (OutputBlock==0) {
             /* Off_1[0], Off_2[0] includes shift. It must be subtracted. */
             if (pout->Shift[1].I) pout->Center[1].V -= pout->Shift[1].V;
             if (pout->Shift[2].I) pout->Center[2].V -= pout->Shift[2].V;
             }

        } // switch

    return;

} /* Affine */

/*---------------------------------------------------------------------------
ArgvBlocks

Ask for all file names in the following way:
  input file 1 [input.edf] :
  input file 2 [<input file 1>] :
  ...
  output file [output.edf] :
  if (Extension!=NULL)
  output file [<input file 1 (without extension)><Extension>] :

  first image [<minimum image number>] :
  last image [<maximum image number>] :
  increment [1] :

  output dummy, e.g. <dummy in first image> :

The output image block is initialized with InitFunction before it is used.

If block_lst >= 1, all input images are open for read and closed 
afterwards.  The header block of the first image is read into ihb[1].
The routine must be used for all images at once.

PARAMETERS
 CmdBlk * pcb, ImgBlk ib[]: roots of command block and image blocks
 ImgHeadBlk ihb[]: root of image header block
 void (*InitFunction) (CmdBlk *,long [],ImgHeadBlk [],int * ): 
   used defined initialization function of image header block, 
   ignored if NULL 
 const char *Extension: string used as extension for the output file. 
   The extension is added to the filename body (filename without 
   extension) of the first input file. The resulting string is used as 
   default file name of the output file. If Extension is NULL the global
   string DefaultOutput is used as default output filename.
   Examples:
 int block_1st, int block_lst : number of first last block
   if block_1st is <= 0, an output block is used,
   if block_lst is >= 1, input block(s) are used,
   if block_lst is <=1, no input block is used, 
   if block_lst is < block_fst, no block is used.
 int * pstatus: return status, 0 if OK
---------------------------------------------------------------------------*/
void ArgvBlocks ( CmdBlk * pcb, ImgBlk ib[], ImgHeadBlk ihb[],
               void (*InitFunction) (CmdBlk *,long [],ImgHeadBlk [],int * ),
               const char *Extension, int block_1st, int block_lst, 
               int * pstatus)
{ int blkno, first_input, last_input;
  size_t blocks; // number of blocks to allocate; 
  long image_1st=1, image_lst=1;
  char linebuffer[IO_len];
  const char *RoutineName = "ArgvBlocks";
  long *num;

  char fname[IO_len];

  SetSaxsError( SR_Version, RoutineName, NULL );

  *pstatus = Success;

  first_input = MAX2(1,block_1st);
  last_input  = block_lst;
  blocks      = MAX2(first_input,block_lst); // at least 1 block

  num = (long *) malloc(sizeof(long)*blocks);
  if (!num) { 
    printf("ERROR: Failed to allocate %lu bytes\n",SIZE_T (sizeof(long)*blocks));
    exit(-1);
  }

  /* ---  File names and open modes --- */

  for (blkno = first_input;blkno <= last_input;blkno++) {

    /*--- Open mode ib[blkno].OpenMode of input files */
    if (!ib[blkno].OpenMode.I) {
      ib[blkno].OpenMode.V = IO_Old | IO_FileProtect;
      ib[blkno].OpenMode.I = TRUE;
    }

    /*--- File name ib[blkno].Name of input files */
    (void) sprintf(linebuffer,"Input sequence %d",blkno);
    if (blkno==1)
      argv_filename( pcb, linebuffer, ib, 1, DefaultInput, pstatus);
    else argv_filename( pcb,linebuffer,ib,blkno,ib[blkno-1].Name.V,pstatus );

    /* Exit in all cases after an abort from the first argument */
    if (*pstatus!=Success) { free(num); return; }

    OpenImageFile(pcb,ib,blkno,ib[blkno].Name.V,ib[blkno].FFirst.V,
                  IO_Old|IO_FileProtect,pstatus);
    if (*pstatus!=Success) { free(num); return; } 

    /* Search for minimum and maximum image number */
    (void) SearchMinMaxImage (pcb,ib,blkno,&image_1st,&image_lst,pstatus);
    if (*pstatus!=Success) { free(num); return; } 
    if (!ib[blkno].First.I) ib[blkno].First.V = image_1st;
    if (!ib[blkno].Last.I)  ib[blkno].Last.V  = image_lst;
    num[blkno] = ib[blkno].First.V;

  } /* for input files */

  if (block_1st<=0) {
    /*--- OpenMode of output file */
    if (!ib[0].OpenMode.I) {
      ib[0].OpenMode.V = IO_ImageProtect;
      ib[0].OpenMode.I = TRUE;
    }

    if (!ib[0].Name.I) { 
      if (Extension) { // create output file name from input file name
        char Suffix[IO_len];
        filename_name ( fname, IO_len, ib[1].Name.V ); // remove path
        // remove compression extensions first
        filename_extension( Suffix, IO_len, fname);
        if (!strcmp(Suffix,".gz")) {
          filename_body ( fname, IO_len, fname ); // remove .gz
        }
        filename_body ( ib[0].Name.V, IO_len, fname ); // remove extension
        sprintf(ib[0].Name.V,"%s%s",ib[0].Name.V,Extension);
      } else sprintf(ib[0].Name.V,"%s",DefaultOutput);
    }

    /*--- File name ib[0].Name of output */
    argv_filename( pcb, "Output sequence", ib, 0, ib[0].Name.V, pstatus);
    if (*pstatus!=Success) { free(num); return; } 

  } /* output file */

  /* ---  Image numbers --- */

  if ((first_input==1) && (pcb->ImgBlkLen>1)) {

    /*--- Argument : number of first image */
    argv_long(pcb,"First image of input sequence 1",&ib[1].First,ib[1].First.V,
               pstatus);
    if (*pstatus!=Success) { free(num); return; } 

    /*--- Argument : number of last image */
    if (pcb->argc==*pcb->parg_no+1) image_lst = ib[1].First.V;
      else image_lst = ib[1].Last.V;
    argv_long( pcb,"Last image of input sequence 1", &ib[1].Last,image_lst,
               pstatus);
    if (*pstatus!=Success) { free(num); return; } 

    /*--- Argument : increment (default = 1) */
    argv_long( pcb,"Increment of input sequence 1", &ib[1].Inc, 1 , pstatus);
    if (*pstatus!=Success) { free(num); return; } 

    num[1]=ib[1].First.V;
    (void) ReadImageHeader( pcb, ib, 1, num, ihb, pstatus);
    if (*pstatus!=Success) { free(num); return; } 
    if (pcb->TestBit) PrintImageHeaderBlock(&ihb[1]);

  } /* first input file */

  for (blkno = MAX2(2,first_input);blkno<=last_input;blkno++) {
    /*--- image numbers of all other images */
    (void) sprintf(linebuffer,"First image of input sequence %d",blkno);
    argv_long( pcb,linebuffer, &ib[blkno].First, ib[blkno].First.V, pstatus);
    if (*pstatus!=Success) { free(num); return; } 

  } /* all other input files */

  if (block_1st <= 0) {

    if (ib[0].First.I) num[0] = ib[0].First.V;
    else num[0] = image_1st;

    if ( (pcb->ImgBlkLen>1) && (ihb[1].I) ) {
      NewImageHeader(pcb,ib,0,num,ihb,&(ihb[1]),InitFunction,pstatus);
    } else {
      NewImageHeader(pcb,ib,0,num,ihb,NULL,InitFunction,pstatus);
    }
    if (*pstatus!=Success) { free(num); return; } 

    /*--- Argument : output dummy */
    sprintf(linebuffer,"Output dummy, e.g. %g",ihb[0].Dummy.V);
    argv_flexp( pcb, linebuffer, &ib[0].Dummy, "", pstatus);
    if (*pstatus!=Success) return;

    /*--- Argument : output dimension 1 */
    sprintf(linebuffer,"Output dimension 1, e.g. %ld",ihb[0].Dim[1]);
    argv_lvexp(pcb,linebuffer,
      &ib[0].Dim[1],ib[0].Dim[1].I?ib[0].Dim[1].V:"",pstatus);
    if (*pstatus!=Success) return;

    /*--- Argument : output dimension 2 */
    sprintf(linebuffer,"Output dimension 2, e.g. %ld",ihb[0].Dim[2]);
    argv_lvexp(pcb,linebuffer,
      &ib[0].Dim[2],ib[0].Dim[2].I?ib[0].Dim[2].V:"", pstatus);
    if (*pstatus!=Success) return;

  }

  for (blkno=first_input;blkno<=last_input;blkno++) {
    CloseImageFile( pcb, ib, blkno, pstatus) ;
    if (*pstatus!=Success) { free(num); return; }
  } /* close all input files */

  free(num);

} /* ArgvBlocks */

/*---------------------------------------------------------------------------
ArgvFilenames

  Should be replaced with ArgvBlocks.
---------------------------------------------------------------------------*/
void ArgvFilenames ( CmdBlk * pcb, ImgBlk ib[], ImgHeadBlk ihb[],
                     int block_1st, int block_lst, int * pstatus)
{ ArgvBlocks ( pcb, ib, ihb, NULL, NULL, block_1st, block_lst, pstatus );
} // ArgvFilenames

/*---------------------------------------------------------------------------
NAME

  arc_lfactor --- Multiplication of each pixel with a symmetry factor

PURPOSE

  Multiplication of each pixel with a symmetry factor.
  Primary usage in saxs_arc.

DESCRIPTION

ARGUMENTS


---------------------------------------------------------------------------*/
void arc_lfactor ( float * Data,   float * EData,
                   int   Dim_1,    int   Dim_2,
                   float Offset_1, float PSize_1, float Center_1,
                   float Offset_2, float PSize_2, float Center_2,
                   float SampleDistance, float WaveLength,
                   float Dummy,    float DDummy,
                   float sangle,   int   rsys,
                   int csym, int s2, int testbit, int * pstatus )
{ 
  const double arc_csym_fac = SAXS_PI/2.0;
  const double arc_radius_eps = 1e-32;

  float *pData, *pEData;
  float Off_1, Ps_1, Off_2, Ps_2;

  float Radius, Radius2, W_1, W_2;
  float SinDiffAngle, SinSAngle, CosSAngle, SinAngle, CosAngle;

  float factor;

  int i_1, i_2;

  *pstatus = Success;

  switch ( rsys ) {
    case IO_Center  : CENTERREF(Off_1,Ps_1,Offset_1,Center_1);
                      CENTERREF(Off_2,Ps_2,Offset_2,Center_2); 
                      break;
    case IO_Normal  : NORMALREF(Off_1,Ps_1,Offset_1,PSize_1,Center_1);
                      NORMALREF(Off_2,Ps_2,Offset_2,PSize_2,Center_2);
                      break;
    case IO_Tangens : TANGENSREF(Off_1,Ps_1,Offset_1,PSize_1,Center_1,\
                                 SampleDistance);
                      TANGENSREF(Off_2,Ps_2,Offset_2,PSize_2,Center_2,\
                                 SampleDistance);
                      break;
 
    case IO_Saxs    : SAXSREF(Off_1,Ps_1,Offset_1,PSize_1,Center_1,\
                              SampleDistance, WaveLength);
                      SAXSREF(Off_2,Ps_2,Offset_2,PSize_2,Center_2,\
                              SampleDistance, WaveLength);
                      break;
    default: 
      fprintf(ERROUT,"Only the following reference systems can be chosen:\n");
      fprintf(ERROUT,"%s, %s, %s, %s \n",
        reftostr(IO_Center),reftostr(IO_Normal),
        reftostr(IO_Tangens),reftostr(IO_Saxs));
      *pstatus = Failed;
      return;
  }

  if ((csym)||(s2)) {

    SinSAngle = sin(sangle);
    CosSAngle = cos(sangle);
    pData   = ABSPTR(Data,Dim_1,Dim_2,0,0);
    pEData   = EData-Data+pData;
    for (i_2=0;i_2<Dim_2;i_2++) {
      W_2     = WORLD(i_2,Off_2,Ps_2);
      for (i_1=0;i_1<Dim_1;i_1++) {
        W_1     = WORLD(i_1,Off_1,Ps_1);
        Radius2      = W_1*W_1+W_2*W_2;
        Radius       = sqrt(Radius2);

        if (fabs(Radius)>arc_radius_eps) {
          CosAngle     = W_1/Radius;
          SinAngle     = W_2/Radius;
          SinDiffAngle = SinAngle*CosSAngle-CosAngle*SinSAngle;
        } else {
          SinDiffAngle = 0.0;
        }

        if NODUMMY(*pData, Dummy, DDummy) {
          if (csym) factor = arc_csym_fac*fabs(SinDiffAngle);
          else factor = 1.0;
          /* Multiplication with Radius^2*dr */
          if (s2) factor *= Radius2;

          *pData    *= factor;
          if (EData) *pEData *= factor*factor;
        }
        pData++;
        pEData++;
      }
    }
  }

} // arc_lfactor

/*****************************************************************************/
