/***************************************************************************/
/* Written 1994++ by Peter Boesecke                                        */
/* Copyright (C) 2011 European Synchrotron Radiation Facility              */
/*                       Grenoble, France                                  */
/*                                                                         */
/*    Principal authors: Peter Boesecke  (boesecke@esrf.eu)                */
/*                                                                         */
/*    This program is free software: you can redistribute it and/or modify */
/*    it under the terms of the GNU General Public License as published by */
/*    the Free Software Foundation, either version 3 of the License, or    */
/*    (at your option) any later version.                                  */
/*                                                                         */
/*    This program is distributed in the hope that it will be useful,      */
/*    but WITHOUT ANY WARRANTY; without even the implied warranty of       */
/*    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the        */
/*    GNU General Public License for more details.                         */
/*                                                                         */
/*    You should have received a copy of the GNU General Public License    */
/*    along with this program.  If not, see <http://www.gnu.org/licenses/>.*/
/***************************************************************************/
# define SAXSRPK_VERSION "SaxsReplacekey V1.23 Peter Boesecke 2015-12-08"
/*+++------------------------------------------------------------------------
NAME
   SaxsReplacekey --- Replaces keys in strings with values

SYNOPSIS

   # include SaxsReplacekey.h

INCLUDE FILES
   filename.h

RESTRICTIONS

DESCRIPTION
  FNAME  file name (as given)
  <file name> = [FPATH][FBASE][FEXT]
  FPATH  file path (file name path, extracted from file name)
  FBASE  base name (file name without path, without extension)
  FEXT   extension (including separator, extracted from file name)
  FNUM   file number, e.g. value of %%%% in name%%%%.edf 
  MNUM   memory number (positiv image data, negativ variance data, default 1)
  INUM   image number
  LNUM   loop number
  CYCLES number of cycles before output image is written

  NCNT (ihb[isblock].DataCtrl) and ACNT (ihb[isblock].ActualLoopCount) must
  be read from the image header block, which is currently not supplied in the
  parameter list of the routines.

  (The description in SAXSRPK_SHOW should always be updated.)

HISTORY
2003-02-03 V1.0 Peter Boesecke should work, to be tested with files
2003-02-06 V1.1 floatkeyvalue, longkeyvalue, linekeyvalue
2003-02-07 V1.2 Check for StreamOpenMode
2003-05-29 V1.3 ihb is removed from all parameter lists,
                include SaxsOption.h instead of SaxsImage.h
2003-08-03 V1.4 replacekey: no replacement, if pcb, ib or Num is 0
                TestBit prints added
2003-08-11 V1.5 doublekeyvalue
2005-07-11 V1.6 SetSaxsErrorMessage
2005-07-14 V1.7 seb removed, SetSaxsErrorExternal 
2005-11-30 V1.8 linekeyvalue: parameter flag added
2006-08-02 V1.9 spkey: additional parameters ipar+delimeter, flag->npar2
2006-08-08 V1.10 recursive use of spkey for all parameters
2006-08-10 V1.11 special case sensitive keys FNUM, MNUM, INUM, LNUM, FNAME
                 range of iblock checked 
2006-09-17 V1.12 specials keys can be read from all blocks, header keys
                 only from input blocks
2007-04-19 V1.13 -Wall compiler warnings resolved
2007-04-23 V1.14 replacekey: int j -> size_t j
2007-06-05       Description added
2007-06-17 V1.15 spkey: special key CYCLES added,
                 SaxsFilename.h -> filename.h
2010-11-08 V1.16 spkey: NUM==0 -> ihblock=isblock=1
2010-11-18 V1.17 FPATH
2013-05-28 V1.18 replacekey_show added
2015-10-04 V1.19 hasreplacekey added
2015-10-05 V1.20 spkey: special keys FBASE, FEXT, FBODY added.
2015-10-12 V1.21 definition of FBASE and FBODY adapted
2015-10-13 V1.22 FBODY is deprecated, but still supported
2015-12-08 V1.23 spkey: pkend (unused-but-set-variable) commented

--------------------------------------------------------------------------*/

/***************************************************************************
* Private part                                                             *
***************************************************************************/

/***************************************************************************
* Include files                                                            *
***************************************************************************/
# include <stdio.h>
# include <stdlib.h>
# include <string.h>
# include <stdlib.h>
 
# include "SaxsDefinition.h"
# include "SaxsError.h"
# include "SaxsInput.h"
# include "SaxsOption.h"
# include "SaxsImage.h"
# include "filename.h"
# include "edfio.h"                      /* data format specific routines */

# include "SaxsReplacekey.h"

#ifndef WIN32
# include <unistd.h>
#endif

/***************************************************************************
 * Defines                                                                 *
 ***************************************************************************/

# define SAXSRPK_BUFLEN                  1024

# define SAXSRPK_START                   '['
# define SAXSRPK_END                     ']'
# define SAXSRPK_SEPARATOR               ','

# define SAXSRPK_SHOW  "\
    Header keys (case insensitive)\n\
     [<key>] or [<key>,<blkno>] or [<key>,<blkno>,<npar>]\n\
    Special keys (case sensitive)\n\
     Key    Description\n\
     FNAME  file name (as given)\n\
       <file name> = [FPATH][FBASE][FEXT]\n\
     FPATH  file path (filename path, extracted from file name)\n\
     FBASE  base name (file name without path, without extension)\n\
     FEXT   extension (including separator, extracted from file name)\n\
     FNUM   file number, e.g. value of %%%% in name%%%%.edf\n\
     MNUM   memory number (positiv image data, negativ variance data, default 1)\n\
     INUM   image number\n\
     LNUM   loop number\n\
     CYCLES number of cycles before output image is written\
"

/******************************************************************************
* Private Constants                                                           *
******************************************************************************/
PRIVATE char white_spaces[7] = { ' ', '\t', '\r', '\n', '\f', '\v', '\0' };

/******************************************************************************
* Private Type Defs                                                           *
******************************************************************************/

/******************************************************************************
* Private Variables                                                           *
******************************************************************************/

/******************************************************************************
* Routines                                                                    *
******************************************************************************/
char * spstring( char ** pb, const char **ps, long npar,
                 CmdBlk *pcb, ImgBlk ib[], long Num[],
                 long blkno, int level, int * pstatus );
char * spkey( char ** pb, const char **ps, long npar,
              CmdBlk *pcb, ImgBlk ib[], long Num[],
              long blkno, int level, int * pstatus );
void spnum( long *value, long defval, 
            char ** pb, const char **ps, long npar,
            CmdBlk *pcb, ImgBlk ib[], long Num[],
            long blkno, int level );

/*+++------------------------------------------------------------------------
NAME

   replacekey_version --- return version string

SYNOPSIS

   const char * replacekey_version ( void );

DESCRIPTION

RETURN VALUE
   Pointer to a constant character string containing the version of
   this module.

----------------------------------------------------------------------------*/
const char * replacekey_version ( void )
{ return(SAXSRPK_VERSION);
} // replacekey_version

/*+++------------------------------------------------------------------------
NAME

   replacekey_show --- return a string showing all keys

SYNOPSIS

   const char * replacekey_show ( void );

DESCRIPTION

RETURN VALUE
   Pointer to a constant character string containing the version of
   this module.

----------------------------------------------------------------------------*/
const char * replacekey_show ( void )
{ return(SAXSRPK_SHOW);
} // replacekey_show

/*+++------------------------------------------------------------------------
NAME

    hasreplacekey

SYNOPSIS

    int hasreplacekey( const char * string );

DESCRIPTION

Returns 1 string has a key between [ and ], otherwise 0.

HISTORY
04-Oct-2015 Peter Boesecke
---------------------------------------------------------------------------*/
int hasreplacekey( const char * string )
{ int found=0;
  char *ps=NULL, *pe=NULL;

  ps = strchr( string, (int) SAXSRPK_START );
  if (ps) {
    pe = strchr( ps, (int) SAXSRPK_END );
  }

  if (pe) found=1;

  return(found);

} /* hasreplacekey */

/*---------------------------------------------------------------------------
NAME
 
    sp_is_delimeter
 
SYNOPSIS
 
    int sp_is_delimeter ( char *delimeter, char c );
 
DESCRIPTION
 
Returns 1 if c is a character in the delimeter string, 0, if not.
 
HISTORY
18-Jan-1998 Peter Boesecke
---------------------------------------------------------------------------*/
int sp_is_delimeter ( char *delimeter, char c )
{
   if ( strchr( delimeter, (int) c ) )
     return(1); /* delimeter */ else return(0); /* no delimeter */
 
} /* sp_is_delimeter */

/*+++------------------------------------------------------------------------
NAME

   replacekey --- copy input string to output and insert header values

SYNOPSIS

   char * replacekey ( char * buffer, size_t buflen,const char *string, long npar,
                       CmdBlk *pcb, ImgBlk ib[], long Num[],
                       long blkno, int * pstatus );

DESCRIPTION
    Copies the input string to the output buffer. If possible, all keywords 
    descriptors are replaced by the corresponding header values. The header 
    values are read with edf_read_header_line(...) from the data file headers. 
    If the keyword cannot be found the keyword descriptor is copied from the
    input string to the output string without a specific error message.

SYNTAX
    Keywords are enclosed in brackets, e.g. "[Title]". It is replaced by the 
    header value of "Title" that is found in the image header of the current
    image block blkno (stream = ib[blkno].Stream, image number = Num[blkno]). 
    The full syntax is 
       <header value> := '[' <key> [',' <blkno> [',' <npar>]] ']'. 
    <blkno> is the number of an input image block, <npar> is the number of 
    the parameter in the header string. A header value cannot 
    be read from the output image (<blkno>=0), but a special key value can.
    If the image block number of the current image block is smaller than 1 
    the default of <blkno> for header keys is 1 for specials keys 0.
    If a requested header value cannot be read the input string is copied 
    to the output string.

    If (Num == NULL) Num[i]==1 is used

RESTRICTIONS
    All image blocks must be open. Only values from open image blocks can 
    be read. Otherwise the keys are not replaced.
    If any of the pointers pcb or ib are NULL the keys are not replaced. The 
    input is copied to the output. 

EXAMPLE
    blkno = n, npar = 0
    Header of block n:  Dim_1 = 512 ;
    Input string:       3*[Dim_1] 
    Output string:      3*512

    blkno = n, npar = 0
    Header of block n:  Dim_1 = 512 ;
    Header of block 2:  Dim_1 = 1024 ;
    Input string:       3*[Dim_1,2]
    Output string:      3*1024

    blkno = n, npar = 0
    Header of block n:  Title = NNN pixels;
    Header of block m:  Title = MMM pixels;
    Input string:       "AAA [Title,m] [Title,n]"
    Output string:      "AAA MMM pixels NNN pixels"

    blkno = n, npar = 1
    Header of block n:  Dim_1 = 512 pixels;
    Input string:       3*[Dim_1]
    Output string:      3*512

    blkno = n, npar = 2
    Header of block n:  Title = parn1 parn2 parn3;
    Input string:       This is [Title,n,2]
    Output string:      This is parn2 


RETURN VALUE
   Pointer to the output buffer or NULL in case of an error. 

----------------------------------------------------------------------------*/
char * replacekey ( char * buffer, size_t buflen, const char *string, long npar,
                    CmdBlk *pcb, ImgBlk ib[], long Num[],
                    long blkno, int * pstatus )

{ char *pb, *value;
  const char *ps;
  size_t j;

  *pstatus = Success;

  if (pcb->TestBit>1) printf(" -- replacekey\n");

  if (!buflen) return ( (char*) NULL );

  if ((!pcb)||(!ib)) {
    // Do not replace keys, copy string to buffer 
    strncpy(buffer,string,buflen-1);
    buffer[buflen-1]='\0'; // terminate with 0
    return ( buffer ); 
  }

  // prepare output string 
  pb = buffer;
  for (j=1;j<buflen;j++)
    *pb++ = ' '; // fill with spaces
  *pb = '\0'; // terminate with NUL

  // copy
  pb = buffer;
  ps = string;
  value = spstring( &pb, &ps, npar, pcb, ib, Num, blkno, 0, pstatus);
  if (*pstatus!=Success) return ( value );

  if (pcb->TestBit>2) printf(" -- replacekey: value = %s\n",value);

  return( value );

} // replacekey 

/*+++------------------------------------------------------------------------
NAME
 
   spstring --- copy input string to output string and insert header values
 
SYNOPSIS
   *ps points to the current position in the input string (updated)
   *pb points to the current position in the output string (updated)
       The output string must be initialized over its useful length with
       non-zero characters and must be terminated by character NULL.
   if npar is 0, the full value string is copied.
   If npar is >=1, only the substring between the npar-th-1 and the
   npar-th delimeter is copied.
   Default delimeters are white-spaces.
   The pointer to the output string is returned. The pointers *pb and *ps are
   incremented. *pb is only incremented until **pb is character NULL. 
   The parameters itself can be header values.

    If (Num == NULL) Num[i]==1 is used
 
----------------------------------------------------------------------------*/
char * spstring( char ** pb, const char **ps, long npar,
                 CmdBlk *pcb, ImgBlk ib[], long Num[],
                 long blkno, int level, int * pstatus)
{  char *value; 

   if (pcb->TestBit>1) printf(" -- spstring (level = %d)\n",level);
   if (pcb->TestBit>2) printf("    in = \"%s\"\n",*ps);

   *pstatus = Success;

   value = *pb;
   while ((**pb)&&(**ps))
     switch (**ps) {
       case SAXSRPK_START:
         (*ps)++; spkey(pb,ps,npar,pcb,ib,Num,blkno,level+1,pstatus); break;
       default :
         **pb = **ps; (*pb)++; (*ps)++; break;
     } /* switch */

   if (*pstatus!=Success) return ( value );

   **pb = '\0'; // terminate with NUL

   if (pcb->TestBit>2) printf("    next = \"%s\"\n",*ps);
 
   return( value );
 
} /* spstring */

/*+++------------------------------------------------------------------------
NAME

   spkey --- extract key from input string and insert value into output string

SYNOPSIS
   *ps points to the current position in the input string (updated)
   *pb points to the current position in the output string (updated)
       The output string must be initialized over its useful length with
       non-zero characters and must be terminated by character NULL.
   if npar is 0, the full value string is copied.
   If npar is >=1, only the substring between the npar-th-1 and the
   npar-th delimeter is copied.
   Default delimeters are white-spaces.
   The pointer to the output string is returned. The pointers *pb and *ps are
   incremented. *pb is only incremented until **pb is character NULL.

    If (Num == NULL) Num[i]==1 is used

USAGE
   input string  "[<key>,<blkno>,<npar>]"
          <key>: header key (string), no default
        <blkno>: block number (long integer), default blkno
         <npar>: parameter number (long integer), default npar

----------------------------------------------------------------------------*/
char * spkey( char ** pb, const char **ps, long npar, 
              CmdBlk *pcb, ImgBlk ib[], long Num[],
              long blkno, int level, int * pstatus)
{  const char *psstart, *psstop;
   char *value;
   char key[EdfMaxKeyLen+1], *pk; //, *pkend; unused-but-set-variable
   char Line[EdfMaxValLen+1], *pL;
   int stream, DataChain;
   long DataNumber, StreamOpenMode;
   int ErrorValue=0;
   long isblock, ihblock, imax;
   long i, ipar;
   register int j;
   int keystatus;
   char *delimeter;
 
   if (pcb->TestBit>1) printf(" -- spkey (level = %d)\n",level);
   if (pcb->TestBit>2) printf("    in = \"%s\"\n",*ps);
 
   *pstatus = Success;
   keystatus = Success;

   // remember start position
   psstart = *ps;
 
   // prepare key string 
   pk = key;
   for (j=0;j<EdfMaxKeyLen;j++)
     *pk++ = ' '; // fill with spaces
   *pk = '\0'; // terminate with NUL

   imax      = pcb->ImgBlkLen;
   // defaults
   // isblock is the block number used for special keys (0 .. imax-1)
   // ihblock is the block number used for header keys (1 .. imax-1)
   isblock = MAX2(0,blkno);
   isblock = MIN2(isblock,imax-1);
   ihblock = MAX2(1,isblock);
   ihblock = MIN2(ihblock,imax-1);

   ipar      = npar;
   delimeter = white_spaces;
 
   value = *pb;
   pk = key;
   // pkend = key+EdfMaxKeyLen; unused-but-set-variable
   while ((*pk)&&(**ps))
     switch (**ps) {
       case SAXSRPK_START:
         // in levels > 1 copy only the first parameter
         (*ps)++; spkey(&pk,ps,1,pcb,ib,Num,blkno,level+1,pstatus); break;
       case SAXSRPK_SEPARATOR: 
         (*ps)++; 
         if (pcb->TestBit>1) printf(" -- spnum\n");
         if (pcb->TestBit>2) printf("    in = \"%s\"\n",*ps);
         spnum(&isblock,isblock,&pk,ps,1,pcb,ib,Num,blkno,level+1);
         // restrict range
         isblock = MAX2(0,isblock);
         isblock = MIN2(isblock,imax-1);
         ihblock = MAX2(1,isblock);
         ihblock = MIN2(ihblock,imax-1);

         if (pcb->TestBit>2) {
           printf("    isblock = %ld, ihblock = %ld, next = \"%s\"\n",
             isblock,ihblock,*ps);
         }
         if (**ps == SAXSRPK_SEPARATOR ) {
           (*ps)++; 
           if (pcb->TestBit>1) printf(" -- spnum\n");
           if (pcb->TestBit>2) printf("    in = \"%s\"\n",*ps);
           spnum(&ipar,ipar,&pk,ps,1,pcb,ib,Num,blkno,level+1);

           if (pcb->TestBit>2) 
             printf("    ipar = %ld, next = \"%s\"\n",ipar,*ps);
         }
         // search SAXSRPK_END:
         while (**ps == SAXSRPK_SEPARATOR ) {
           (*ps)++;
           spnum(NULL,0,&pk,ps,1,pcb,ib,Num,blkno,level+1);

         }
         break;
       case SAXSRPK_END:
         *pk = '\0'; (*ps)++; break;
       default :
         *(pk++) = **ps; (*ps)++; break;
     } /* switch */

   // get end position
   psstop = *ps;

   if (*pstatus!=Success) return ( value );

   if (pcb->TestBit>1) {
     printf("    in spkey: key = %s, isblock = %ld, ihblock = %ld, ipar = %ld\n",
       key,isblock,ihblock,ipar);
   }
   if (pcb->TestBit>2) printf("    next = \"%s\"\n",*ps);

   // copy keys to *pb
   // first check for special keys
   stream = ib[isblock].Stream;
   StreamOpenMode = ib[isblock].StreamOpenMode;
   if (Num) DataNumber = Num[isblock];
   else DataNumber = 1l;
   DataChain = ib[isblock].Memory.V;

   if ( strncmp( key,"FNUM", 4 ) == 0 ) {
     //FileNumber
     num_long2str ( Line, EdfMaxValLen+1,ib[isblock].FileNumber,&ErrorValue);
     SetSaxsErrorExternal ( ErrorValue, ReportSaxsImageError );
   } else if ( strncmp( key,"LNUM", 4 ) == 0 ) {
     //LoopNumber
     num_long2str ( Line, EdfMaxValLen+1,ib[isblock].LoopNumber,&ErrorValue);
     SetSaxsErrorExternal ( ErrorValue, ReportSaxsImageError );
   } else if ( strncmp( key,"INUM", 4 ) == 0 ) {
     //ImageNumber
     num_long2str ( Line, EdfMaxValLen+1,DataNumber,&ErrorValue);
     SetSaxsErrorExternal ( ErrorValue, ReportSaxsImageError );
   } else if ( strncmp( key,"MNUM", 4 ) == 0 ) {
     //Memory
     num_long2str ( Line, EdfMaxValLen+1,DataChain,&ErrorValue);
     SetSaxsErrorExternal ( ErrorValue, ReportSaxsImageError );
   } else if ( strncmp( key,"FNAME", 5 ) == 0 ) {
     //FileName
     strncpy(Line,ib[isblock].FileName,EdfMaxValLen);
     Line[EdfMaxValLen]=(char) 0;
   } else if ( strncmp( key,"FPATH", 5 ) == 0 ) {
     //FilePath
     filename_path ( Line, EdfMaxValLen+1, ib[isblock].FileName );
   } else if ( strncmp( key,"FBASE", 5 ) == 0 ) {
     //Basename (file name without path, without extension)
     filename_base ( Line, EdfMaxValLen+1, ib[isblock].FileName );
   } else if ( strncmp( key,"FEXT", 4 ) == 0 ) {
     //Extension of basename
     filename_extension ( Line, EdfMaxValLen+1, ib[isblock].FileName );
   } else if ( strncmp( key,"FBODY", 5 ) == 0 ) {
     //Bodyname (filename without extension)
     filename_body ( Line, EdfMaxValLen+1, ib[isblock].FileName );
   } else if ( strncmp( key,"CYCLES", 6 ) == 0 ) {
     //Cycles
     num_long2str ( Line, EdfMaxValLen+1,pcb->Cycles,&ErrorValue);
     SetSaxsErrorExternal ( ErrorValue, ReportSaxsImageError );
   } else {
     // read header value and copy it to *pb
     stream = ib[ihblock].Stream;
     StreamOpenMode = ib[ihblock].StreamOpenMode;
     if (Num) DataNumber = Num[ihblock];
     else DataNumber = 1l;
     DataChain = ib[ihblock].Memory.V;

     if ( !(( IO_DontReadWrite | IO_Dummy ) & StreamOpenMode ) ) {
       // now search header keys
       edf_read_header_line ( stream, DataNumber, DataChain,
                              key, Line, &ErrorValue, &keystatus );
       SetSaxsErrorExternal ( ErrorValue, ReportSaxsImageError );
     } else keystatus = KeywordMissing;
     if (pcb->TestBit>1) {
       printf("    in spkey: key cannot be read from file (block=%ld)\n",ihblock);
       printf("              stream=%d, DataNumber=%ld, DataChain=%d, key=%s\n",
                             stream, DataNumber, DataChain, key);
     }
   }

   if (keystatus==Success) {
     // copy Line to output buffer
     pL = Line;
     if ( ipar == 0 ) { // copy full string
        while ( (**pb) && (*pL) ) *(*pb)++ = *pL++;
     } else { // copy ipar-th substring
       // search start position
       i=1;
       while ( (i<ipar)&&(*pL) ) 
         if (sp_is_delimeter(delimeter,*pL++)) i++;
       // copy substring 
       while ( (**pb) && (*pL) && !(sp_is_delimeter(delimeter,*pL)) )
         *(*pb)++ = *pL++;
     }
   } else {
     // cannot replace key, copy input string to output buffer
     *ps = psstart;
     if (**pb) *(*pb)++ = SAXSRPK_START;
     while ((**pb)&&(*ps<psstop))
       *(*pb)++ = *(*ps)++;
   }
 
   return( value );
 
} /* spkey */

/*+++------------------------------------------------------------------------
NAME
 
   spnum --- scans input string for a long integer expression 
 
SYNOPSIS
   *ps points to the current position in the input string (updated). It is 
   always incremented either until a string terminator '\0' is found 
   or until the characters SAXSRPK_SEPARATOR or SAXSRPK_END are found. In 
   both cases *ps points to the termination character.
   In case of success value is updated with the read integer number,
   otherwise it contains the default.
   If value is NULL, only ps is updated.
----------------------------------------------------------------------------*/
void spnum( long *value, long defval,
            char ** pb, const char **ps, long npar,
            CmdBlk *pcb, ImgBlk ib[], long Num[],
            long blkno, int level )
{  char num[EdfMaxValLen+1], *pn, *pnend;
   register int j;

   int status = Success;
 
   // prepare num string
   pn = num;
   for (j=0;j<EdfMaxValLen;j++)
     *pn++ = ' '; // fill with spaces
   *pn = '\0'; // terminate with NUL

   // copy number string
   pn = num;
   pnend = num+EdfMaxValLen;
   while ((*pn)&&(**ps))
     switch (**ps) {
       case SAXSRPK_START:
         // in levels > 1 copy only the first parameter (default)
         (*ps)++; spkey(&pn,ps,1,pcb,ib,Num,blkno,level+1,&status); break;
       case SAXSRPK_SEPARATOR:
       case SAXSRPK_END:
         *pn = '\0'; break;
       default :
         if (pn<pnend) *(pn++) = **ps; (*ps)++; break;
     } /* switch */

   if ( value ) {
     *value = (int) longexpr( num, &status);
     if (status!=Success) *value = defval;
   }

   return;
 
} /* spnum */

/*+++------------------------------------------------------------------------
NAME
 
   doublekeyvalue --- returns a double value
 
SYNOPSIS
   All keys in string are replaced by their values. The float value of
   the expression is returned.
----------------------------------------------------------------------------*/
double doublekeyvalue( const char *string,
                       CmdBlk *pcb, ImgBlk ib[], long Num[],
                       long blkno, int * pstatus )
{ static char line[IO_len];
  double value=0.0;
 
  *pstatus = Success;
 
  replacekey ( line, IO_len, string, 1,
               pcb, ib, Num, blkno, pstatus );
  if (*pstatus!=Success) return( value );
 
  value = doubleexpr(line,pstatus);
  if (*pstatus!=Success) {
    SetSaxsErrorMessage( line );
    return( value );
  }
 
  if (pcb->TestBit)
    printf("doublekeyvalue: string=%s, line=%s, value=%g\n",string,line,value);
 
  return( value );
 
} // doublekeyvalue

/*+++------------------------------------------------------------------------
NAME
 
   floatkeyvalue --- returns a float value, either from ib 
 
SYNOPSIS
   All keys in string are replaced by their values. The float value of
   the expression is returned.
----------------------------------------------------------------------------*/
float floatkeyvalue( const char *string,
                     CmdBlk *pcb, ImgBlk ib[], long Num[],
                     long blkno, int * pstatus ) 
{ static char line[IO_len];
  float value=0.0;

  *pstatus = Success;

  replacekey ( line, IO_len, string, 1,
               pcb, ib, Num, blkno, pstatus );
  if (*pstatus!=Success) return( value );

  value = floatexpr(line,pstatus);
  if (*pstatus!=Success) {
    SetSaxsErrorMessage( line );
    return( value );
  }

  if (pcb->TestBit)
    printf("floatkeyvalue: string=%s, line=%s, value=%g\n",string,line,value);

  return( value );

} // floatkeyvalue

/*+++------------------------------------------------------------------------
NAME
 
   longkeyvalue --- returns a long value, either from ib
 
SYNOPSIS
   All keys in string are replaced by their values. The long value of
   the expression is returned.
----------------------------------------------------------------------------*/
long longkeyvalue( const char *string,
                   CmdBlk *pcb, ImgBlk ib[], long Num[],
                   long blkno, int * pstatus )
{ static char line[IO_len];
  long value;
 
  *pstatus = Success;
  value = 0l;
 
  replacekey ( line, IO_len, string, 1,
               pcb, ib, Num, blkno, pstatus );
  if (*pstatus!=Success) return( value );
 
  value = longexpr(line,pstatus);
  if (*pstatus!=Success) {
    SetSaxsErrorMessage( line );
    return( value );
  }
 
  if (pcb->TestBit)
    printf("longkeyvalue: string=%s, line=%s, value=%ld\n",string,line,value);

  return( value );
 
} // longkeyvalue

/*+++------------------------------------------------------------------------
NAME
 
   linekeyvalue --- returns a line, either from ib
 
SYNOPSIS
   All keys in string are replaced by their values. The pointer to the output 
   line is returned. If npar is 0, the keys are replaced by by the full value,
   if npar is 1, the keys are replaced with the first parameter of value
   until the first white space.

   If (Num == NULL) Num[i]==1 is used
----------------------------------------------------------------------------*/
char * linekeyvalue (char * buffer, size_t buflen, const char *string, long npar,
                     CmdBlk *pcb, ImgBlk ib[], long Num[],
                     long blkno, int * pstatus )
{ char * value;
 
  *pstatus = Success;
 
  value = replacekey ( buffer, buflen, string, npar,
                       pcb, ib, Num, blkno, pstatus );
  if (*pstatus!=Success) return( value );

  if (pcb->TestBit)
    printf("linekeyvalue: string=%s, npar=%ld, value=%s\n",string,npar,value);
 
  return( value );
 
} // linekeyvalue

