/***************************************************************************/
/* Written 1994++ by Peter Boesecke                                        */
/* Copyright (C) 2011 European Synchrotron Radiation Facility              */
/*                       Grenoble, France                                  */
/*                                                                         */
/*    Principal authors: Peter Boesecke  (boesecke@esrf.eu)                */
/*                                                                         */
/*    This program is free software: you can redistribute it and/or modify */
/*    it under the terms of the GNU General Public License as published by */
/*    the Free Software Foundation, either version 3 of the License, or    */
/*    (at your option) any later version.                                  */
/*                                                                         */
/*    This program is distributed in the hope that it will be useful,      */
/*    but WITHOUT ANY WARRANTY; without even the implied warranty of       */
/*    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the        */
/*    GNU General Public License for more details.                         */
/*                                                                         */
/*    You should have received a copy of the GNU General Public License    */
/*    along with this program.  If not, see <http://www.gnu.org/licenses/>.*/
/***************************************************************************/
/*+++
1 saxs_scal
 Read the value of a keyword from the headers of each picture 
 and write it to a text file.

2 PURPOSE

  Arguments:
  saxs_scal [options] <i1nam> <output prefix> <i1fst> <i1lst> <i1inc> <keyword>
  defaults:
  <input file name>   : input.edf 
  <output prefix>     : input 
  <first image>       : <first image number in input file>
  <last image>                              : <last image number in input file>
     if argument list ends with first image : <first image>
  <increment>         : 1
  <scaler keyword>    : "Image"  

  Specific options:
  +key <keyword>                : keyword 
  +pref <output file prefix>    : prefix of the output file name
  +ext  <output file extension> : extension of the output file name
  +fnam <output file name>      : output file name

  The values are written to the file <prefix>_<keyword><extension> .

  Wildcard:
  A "=" in the argument list chooses the default.

2 HISTORY
  1995-04-28  PJE      creation from saxs_norm
  1995-12-04  PB
  1999-06-26  PB  V3.0  DevSeparator, DirSeparator, ExtSeparator 
                        defined in SaxsInput
  1999-12-31  PB  V3.01 cr\lf in output file "\r\n"
  2000-08-04  PB        %f->%g
  2001-07-03  PB  V3.02 filename_full
  2001-07-07  PB  V3.03 IO_DontReadWrite, IO_DontReadData
  2001-07-08  PB  V3.04 FFirst
  2001-07-09  PB  V3.05 (pcb->ib)[0].FileNumber
  2001-07-09  PB  V4.00 new loop, no repetition
  2003-02-03  PB  V4.01 num[BlockNum]
  2003-02-09  PB  V4.02 line args for Dummy, DDummy, Dim 
  2003-05-30  PB  V4.03 ImageLoop: Function call changed
  2003-08-14  PB  V4.04 option_define, all option numbers removed
  2004-10-31  PB  V4.05 MAKE_FUNCTION
  2005-07-15  PB  V4.06 ReportSaxsStatus parameter list updated
  2005-08-04  PB  V4.07 PrintImageHeaderBlock paramters updated
  2007-06-11  PB  V4.08 ImageLoop parameters updated
  2008-12-05  PB  V4.09 open output files as binary files to avoid
                        translation of \r\n to \n when compiled on windows.
  2011-05-10  PB  V4.10 possibility to write several keys in a line 
  2011-05-14  PB  V4.11 split.h -> strlib.h
  2012-01-31  PB  V4.12 ImageLoop parameters updated
  2016-02-17  PB  V4.13 remove .gz suffix when creating output filename

---*/
# define Version  "saxs_scal V4.13 2016-02-17, Peter Boesecke"

# include <errno.h>
# include <stdio.h>
# include <unistd.h>
# include <fcntl.h>

# include "SaxsPrograms.h"
# include "strlib.h"

/* Option definitions */
# define SKey              "key"   /* string */
# define SPrefix           "pref"
# define SExtension        "ext"    
# define SFName            "fnam"

# define Default_extension ".txt"
# define Default_prefix    "input"
# define Default_key       KImage

# define Usage "[options] \n\
                <i1nam> <onam> <i1fst> <i1lst> <i1inc> <key>{,<key>}\n\
Purpose: Extract keyword values from image headers and write them into a file\n\
Options: -key, -pref, -ext, -fnam"

# define BlockNum 2       /* 1 input sequence + 1 output sequence */

/*---------------------------------------------------------------------------
SaxsScalArgvFilenames

Ask for all file names in the following way:
  input file 1 [input.edf] :
  input file 2 [<input file 1>] :
  ...
  output file [output.edf] :

  first image [<minimum image number>] :
  last image [<maximum image number>] :
  increment [1] :

  output dummy [<dummy in first image>] :

If blkno >= 1 the header block of the first image is read into ihb[1].
The routine must be used for all images at once.
---------------------------------------------------------------------------*/
void SaxsScalArgvFilenames ( CmdBlk * pcb, ImgBlk ib[], ImgHeadBlk ihb[],
                     int block_1st, int block_lst, int * pstatus)
{ int blkno, first_block;
  long image_1st, image_lst;
  char linebuffer[1000];
  int nameblk;
  long num[BlockNum];

  IO_line *pPrefix;

  first_block = MAX2(1,block_1st);

  /* ---  File names and open modes --- */

  for (blkno = first_block;blkno <= block_lst;blkno++) {

    /*--- Open mode ib[blkno].OpenMode of input files */
    if (!ib[blkno].OpenMode.I) {
      ib[blkno].OpenMode.V = IO_Old | IO_FileProtect | IO_DontReadData;
      ib[blkno].OpenMode.I = TRUE;
      }

    /*--- File name ib[blkno].Name of input files */
    (void) sprintf(linebuffer,"Input sequence %d",blkno);
    if (blkno==1)
      argv_filename( pcb,linebuffer, ib, 1, DefaultInput, pstatus);
    else argv_filename( pcb,linebuffer,ib,blkno,ib[blkno-1].Name.V,pstatus);
    /* Exit in all cases after an abort from the first argument */
    if (*pstatus!=Success) return;

    OpenImageFile( pcb,ib,blkno,ib[blkno].Name.V,ib[1].FFirst.V,
                   IO_Old|IO_FileProtect,pstatus);
    if (*pstatus!=Success) return;

    if (blkno==1) {
      /* Search for minimum and maximum image number in 1st input sequence */
      (void) SearchMinMaxImage ( pcb, ib, 1, &image_1st, &image_lst, pstatus);
      if (*pstatus!=Success) return;
    }

  } /* for input files */

  if (block_1st<=0) {

    /*--- OpenMode of output file */
    if (!ib[0].OpenMode.I) {
      ib[0].OpenMode.V = IO_ImageProtect | IO_DontReadData | IO_DontReadWrite |\
                         IO_DontOpen;
      ib[0].OpenMode.I = TRUE;
      }

    /*--- Argument  : text file prefix */
    if (ib[0].Name.I) nameblk = 0; else nameblk = 1;
    pPrefix = (IO_line*) option_parameter_search( SPrefix, pcb, pstatus );
    if (*pstatus!=Success) return;

    if (!pPrefix->I) {
      /* Determine prefix from input file name */
      /* Check for .gz extension (use pPrefix->V as temporary buffer) */
      filename_extension ( pPrefix->V, IO_len, ib[nameblk].Name.V );
      if (!strcmp(pPrefix->V,".gz")) {
        // Remove .gz extension
        filename_body ( pPrefix->V, IO_len, ib[nameblk].Name.V ); // remove .gz
        filename_body ( pPrefix->V, IO_len,  pPrefix->V ); // remove ext.
      } else {
        filename_body ( pPrefix->V, IO_len, ib[nameblk].Name.V ); // remove ext.
      }

      // remove path
      filename_name ( pPrefix->V, IO_len, pPrefix->V );
      pPrefix->I = TRUE;
    }

    /*--- Prefix of output file name */
    argv_line( pcb,"Ascii file prefix",pPrefix,pPrefix->V,
               pstatus);
    if (*pstatus!=Success) return;
 
    /* extract output filenumbers from ASCII file name */
    extract_filenumbers( &ib[0], pPrefix, TRUE, pstatus );
    if (*pstatus!=Success) return;
 
    /* Expand Prefix to the full file name */
    if (!filename_full (pPrefix->V,IO_len,pPrefix->V)) {
      fprintf(ERROUT,"Input file name or path not allowed\n");
      *pstatus = FileNotFound;
      return;
    }
 
    /*--- Take prefix as default for output filename */
    if (!ib[0].Name.I) {
      (void) sprintf(ib[0].Name.V,"%s",pPrefix->V);
    }
        
  } /* output file */

  /* ---  Image numbers --- */

  if (first_block==1) {

    /*--- Argument : number of first image */
    argv_long( pcb,"First image of input sequence 1", &ib[1].First, image_1st,
               pstatus);
    if (*pstatus!=Success) return;

    /*--- Argument : number of last image */
    if (pcb->argc==*pcb->parg_no+1) image_lst = ib[1].First.V;
    argv_long( pcb,"Last image of input sequence 1", &ib[1].Last, image_lst,
               pstatus);
    if (*pstatus!=Success) return;

    /*--- Argument : increment (default = 1) */
    argv_long( pcb,"Increment of input sequence 1", &ib[1].Inc, 1 , pstatus);
    if (*pstatus!=Success) return;

    num[1] = ib[1].First.V;
    (void) ReadImageHeader( pcb, ib, 1, num, ihb, pstatus);
    if (*pstatus!=Success) return;
    if (pcb->TestBit) PrintImageHeaderBlock(&ihb[1]);

    /* Defaults for output file */
    if (!ib[0].Dummy.I) sprintf(ib[0].Dummy.V,"%g",ihb[1].Dummy.V);
    if (!ib[0].Dim[1].I) sprintf(ib[0].Dim[1].V,"%d",ihb[1].Dim[1]);
    if (!ib[0].Dim[2].I) sprintf(ib[0].Dim[2].V,"%d",ihb[1].Dim[2]);

  } /* first input file */

  for (blkno = MAX2(2,first_block);blkno<=block_lst;blkno++) {
    /*--- image numbers of all other images */
    (void) sprintf(linebuffer,"First image of input sequence %d",blkno);
    argv_long( pcb,linebuffer, &ib[blkno].First, ib[1].First.V, pstatus);
    if (*pstatus!=Success) return;

  } /* all other input files */

  for (blkno = MAX2(1,first_block);blkno<=block_lst;blkno++) {
    CloseImageFile( pcb, ib, blkno, pstatus) ;
    if (*pstatus!=Success) return;
  } /* close all input files */

} /* SaxsScalArgvFilenames */

/*---------------------------------------------------------------------------
1 saxs_scal

2 PURPOSE
  Reading of the value of an scaler from an esrf data file and storage in
  an output ascii file.
---------------------------------------------------------------------------*/
void saxs_scal (CmdBlk * pcb, long num[], ImgHeadBlk ihb[], int * pstatus)
{ const int NumberOfImages=2;
  const float eps=1e-30;
  int i,imax;
  int j;
  int ErrorValue;

  char fname_full[IO_len];

  FILE *OutputFileId;

  IO_line KeyList, FName, Extension, Prefix;
  char Key[IO_len], KeyValue[IO_len];
  int parno;

  *pstatus = Success;

  imax = pcb->ImgBlkLen;

  for(i=1;i<imax;i++) {
    if (pcb->TestBit) PrintImageHeaderBlock(&ihb[i]);}

  /* Check number of images */
  if (pcb->ImgBlkLen!=NumberOfImages) {
    fprintf(ERROUT,"%d images found, 1 input and 1 output image required\n",
            pcb->ImgBlkLen); *pstatus=Failed; return; 
  }

  /* --- Get option values */
  KeyList = option_line ( SKey, pcb, pcb->ib, num, 1, pstatus );
  if (*pstatus!=Success) return;
  Extension = option_line ( SExtension, pcb, pcb->ib, num, 1, pstatus );
  if (*pstatus!=Success) return;
  Prefix = option_line ( SPrefix, pcb, pcb->ib, num, 1, pstatus );
  if (*pstatus!=Success) return;
  FName = option_line ( SFName, pcb, pcb->ib, num, 1, pstatus );
  if (*pstatus!=Success) return;

  /*--- Get output filename */
  if (!FName.I) {
    // first key is used for filename
    strlib_param ( Key, IO_len, KeyList.V, ',', 1 );
    sprintf(FName.V,"%s_%s%s",Prefix.V,Key,Extension.V);
  }

  /* Generate filename of ASCII file */
  filename_parameter (fname_full,IO_len,FName.V,0);
  filename_pattern (fname_full,IO_len,fname_full,(pcb->ib)[0].FileNumber);
  filename_full (fname_full,IO_len,fname_full);

  /*--- Open output file
    If this is the first image of the sequence, an existing file 
    with the same name must be deleted (open mode = "w"),
    in all other cases it must be open to append a line (open mode "a"). */

  if (pcb->TestBit) printf("Opening output file %s\n",fname_full);
  *pstatus = FileOpeningError;
  if (ihb[1].ImNum==pcb->ib[1].First.V) {
    /* 1st image: new write */
    if ((OutputFileId=fopen(fname_full,"wb")) == NULL) return;
  } else {
    /* appending */
    if ((OutputFileId=fopen(fname_full,"ab")) == NULL) return;
  }

  /*--- Write to the output file 
    If this is the first image of the sequence, a header line is written */

  if (pcb->TestBit) printf("Writing keyword to file %s\n",fname_full);
  if (ihb[1].ImNum==pcb->ib[1].First.V) {
    /* 1st image: write header line */
    printf("\n");
    printf("Writing to file : %s\n\n", fname_full);
    fprintf(OutputFileId,"Image");

    /* --- Write key names */
    if (KeyList.I) {
      parno = 1;
      while (strlib_param ( Key, IO_len, KeyList.V, ',', parno++ )) {
        if (strlen(Key)) {
          fprintf(OutputFileId,"\t%s",Key);
        }
      }
    }
    fprintf(OutputFileId,"\r\n");
  } else
    printf("Appending to file : %s\n\n", fname_full);

  /*--- Write Image Number */
  fprintf(OutputFileId,"%d",ihb[1].ImNum);

  /*--- Read and write keyword values */
  if (KeyList.I) {
    parno = 1;
    while (strlib_param ( Key, IO_len, KeyList.V, ',', parno++ )) {
      if (strlen(Key)) {
        if (pcb->TestBit) printf("Reading keyword %s\n",Key);
        ReadHeaderLine          ( pcb, 1, ihb, Key, KeyValue, pstatus );
        if (*pstatus) fprintf(OutputFileId,"\t%s","(not found)");
        else fprintf(OutputFileId,"\t%s",KeyValue);
      }
    }
  }

  /*--- Terminate line with crlf */
  fprintf(OutputFileId,"\r\n");

  /*--- Close output file */

  if (pcb->TestBit) printf("Closing file %s\n",fname_full);
  *pstatus = FileClosingError;
  if (fclose(OutputFileId)!=0) return;

  *pstatus = Success;

 } /* saxs_scal*/

/*---------------------------------------------------------------------------
user_io
Do all the keyboard io and return cb, and ib
---------------------------------------------------------------------------*/

void user_io(CmdBlk * pcb, ImgBlk * ib, int * pstatus)
{
  char  progfn[InputLineLength];
  ImgHeadBlk ihb[BlockNum];

  float ROff_1, RPs_1, UOff_1, UPs_1;
  float ROff_2, RPs_2, UOff_2, UPs_2;

  IO_line *pKey, *pPrefix;

 /* Determine program name without directory */
   (void) RightFR((char *) pcb->argv[0],DirSeparator,progfn,
           InputLineLength);

  /* show usage if no arguments are given */
  if (pcb->argc<=1) printf("Usage: %s %s\n",progfn,Usage);

  /*--- Write name of program ---*/
  printf("\n");
  printf("   %s %s\n",progfn,Version);
  printf("\n");

  SaxsScalArgvFilenames ( pcb, ib, ihb, 0, BlockNum-1, pstatus);
  if (*pstatus!=Success) return;
  GetReference(pcb->RSys.V,1,ihb,&ROff_1,&ROff_2,&RPs_1,&RPs_2,pstatus );
  if (*pstatus!=Success) return;
  GetReference(pcb->USys.V,1,ihb,&UOff_1,&UOff_2,&UPs_1,&UPs_2,pstatus );
  if (*pstatus!=Success) return;

  /*--- Argument  : keyword */
  pKey = (IO_line*) option_parameter_search( SKey, pcb, pstatus );
  if (*pstatus!=Success) return;
  argv_line(pcb,"keyword",pKey,pKey->V,pstatus);
  if (*pstatus!=Success) return;

  pPrefix = (IO_line*) option_parameter_search( SPrefix, pcb, pstatus );
  if (*pstatus!=Success) return;

  printf("\n");
  if (ib[1].Name.I)   printf("i/p file           : %s\n",ib[1].Name.V);
  if (pPrefix->I)     printf("output file prefix : %s\n",pPrefix->V);
  if (ib[1].First.I)  printf("first image        : %d\n",ib[1].First.V);
  if (ib[1].Last.I)   printf("last image         : %d\n",ib[1].Last.V);
  if (ib[1].Inc.I)    printf("increment          : %d\n",ib[1].Inc.V);
  if (pKey->I)        printf("keyword            : %s\n",pKey->V);
  printf("\n");

  if (pcb->TestBit) {
    PrintBlocks ( pcb, ib );
    printf("\n"); }

  return;
} /* user_io */

/*---------------------------------------------------------------------------
main
---------------------------------------------------------------------------*/

#if MAKE_FUNCTION
# define MAIN main_saxs_scal
#else
# define MAIN main
#endif

int MAIN (int argc, char *argv[])
{
  CmdBlk cb;                /* command block  */
  ImgBlk ib[BlockNum];      /* image blocks */

  int status;
  int arg_no = 0;

 /* Init options, control block and image blocks */
  InitOptions( Usage, Version, NULL, TRUE, &cb, ib, BlockNum );

  option_define ( IO_tpline, SKey,Default_key, &cb );
  option_define ( IO_tpline, SPrefix,Default_prefix, &cb );
  option_define ( IO_tpline, SExtension, Default_extension, &cb );
  option_define ( IO_tpline, SFName,"output.txt", &cb );

 /* Read options from argument list */
  ReadOptions( argv, &arg_no, &cb, ib, &status);

  /* USER KEYBOARD I/O */
  if (status==Success) {
    argv_start ( &cb, 1 );
    user_io( &cb, ib, &status);
    argv_end( &cb ); /* must be called after user_io */
  }

  /* SEQUENCE CALCULATION */
  if (status==Success) 
    IMAGELOOP( &cb, ib, saxs_scal, NULL, NULL, TRUE, &status );

  return( ReportSaxsStatus( status, 0 ) );

} /* MAIN */

