/***************************************************************************/
/* Written 1994++ by Peter Boesecke                                        */
/* Copyright (C) 2011 European Synchrotron Radiation Facility              */
/*                       Grenoble, France                                  */
/*                                                                         */
/*    Principal authors: Peter Boesecke  (boesecke@esrf.eu)                */
/*                                                                         */
/*    This program is free software: you can redistribute it and/or modify */
/*    it under the terms of the GNU General Public License as published by */
/*    the Free Software Foundation, either version 3 of the License, or    */
/*    (at your option) any later version.                                  */
/*                                                                         */
/*    This program is distributed in the hope that it will be useful,      */
/*    but WITHOUT ANY WARRANTY; without even the implied warranty of       */
/*    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the        */
/*    GNU General Public License for more details.                         */
/*                                                                         */
/*    You should have received a copy of the GNU General Public License    */
/*    along with this program.  If not, see <http://www.gnu.org/licenses/>.*/
/***************************************************************************/
/*+++
1 saxs_collapserow
  Projects all rows of image sequence 1

2 PURPOSE
  Projects all rows of image sequence 1

  Calculation:       I0 = Projection(I1)
  Error propagation: V0 = Projection(V1)

  Arguments:
  saxs_collapserow [options] <i1nam> <onam> <i1fst> <i1lst> <i1inc>
             <odum> <odim1> <odim2> <i1fac> <i1con>       
             <ofac> <ocon>

  Defaults:
  <input sequence 1>  : input.edf 
  <input sequence 2>  : <input sequence 1>
  <output sequence>   : output.edf
  <first image>       : <first image number in sequence 1>
  <last image>                              : <last image number in input file>
     if argument list ends with first image : <first image>
  <increment>         : 1
  <first image>       : <first image number in sequence 2>
  <output dummy>      : <dummy value in first image of input file>
  <output dimension 1>: <horizontal dimension of first image in input file>
  <output dimension 2>: <vertical dimension of first image in input file>
  <i1fac>             : 1.0 input 1 multiplication factor
  <i1con>             : 0.0 input 1 addition constant
  <ofac>              : 1.0
  <ocon>              : 0.0

  Wildcard:
  A "=" in the argument list chooses the default.

2 HISTORY
  2016-03-09 PB V4.64 from saxs_addrow V4.63
 
---*/
# define Version  "saxs_collapserow V4.64 2016-03-09, Peter Boesecke"

# include <errno.h>
# include <stdio.h>
# include <unistd.h>
# include <fcntl.h>

# include "SaxsPrograms.h"

# define Usage "[options] \n\
             <i1nam> <onam> <i1fst> <i1lst> <i1inc>\n\
             <odum> <odim1> <odim2> <i1fac> <i1con>\n\
             <ofac> <ocon>\n\
         --- Average rows of i1  ---"

# define BlockNum 2       /* 1 input sequence + 1 output sequence */

/*---------------------------------------------------------------------------
1 saxs_collapserow

2 PURPOSE
  
---------------------------------------------------------------------------*/

void saxs_collapserow (CmdBlk * pcb, long num[], ImgHeadBlk ihb[], int * pstatus)
{ const int NumberOfImages=2;
  const float eps=1e-30;
  int i,imax;
  int j;

  float *I0Data, *E0Data;
  int   I0Dim_1, I0Dim_2;
  float *pI0Data, *pE0Data;
  float I0Dummy, I0DDummy;
  float VarDDummy=DDSET(VarDummy);

  int   I1Ave;
  float *I1Data, *E1Data;
  int   I1Dim_1, I1Dim_2;
  float I1Dummy, I1DDummy;
  float I1Value, I1Sum, I1Weight;
  float E1Value, E1Sum, E1Weight;

  int i_1, i_2;
  float f_1[BlockNum], f_2[BlockNum];
 
  float Off_1[BlockNum], Off_2[BlockNum];
  float Ps_1[BlockNum], Ps_2[BlockNum];
 
  float f1_1[BlockNum], f3_1[BlockNum], Df_1[BlockNum];
  float f1_2[BlockNum], f3_2[BlockNum], Df_2[BlockNum];
 
  int Imin_1, Imin_2, Imax_1, Imax_2;
 
  float Wmin_1, Wmax_1, W_1, DW_1;
  float Wmin_2, Wmax_2, W_2, DW_2;

  float row1, row2;
 
  float W1_2, W3_2;

  IO_float Row1, Row2;

  *pstatus = Success;

  imax = pcb->ImgBlkLen;
  printf("\n Calculating ihb[0,% d] = Function(",ihb[0].ImNum);
  for(i=1;i<imax;i++) {
    printf("ihb[% d,% d] ", i, ihb[i].ImNum); }

  printf(")\n\n");

  for(i=1;i<imax;i++) {
    if (pcb->TestBit) PrintImageHeaderBlock(&ihb[i]);}

 /* Check the number of images */
  if (pcb->ImgBlkLen!=NumberOfImages) {
     printf("%d images found, 2 input and 1 output image required\n",
             imax); *pstatus=Failed; return; }

  GetReferenceParameters( pcb, ihb, 0, imax-1,
                          Off_1, Off_2, Ps_1,  Ps_2, pstatus );
  if (*pstatus!=Success) return;

  UpdateImageParameters ( pcb, ihb, 1, imax-1,
                          Off_1, Off_2, Ps_1,  Ps_2, pstatus );
  if (*pstatus!=Success) return;

 /* Get option values */
  Row1 = option_float ( "row1", pcb, pcb->ib, num, 1, pstatus );
  if (*pstatus!=Success) return;
  Row2 = option_float ( "row2", pcb, pcb->ib, num, 1, pstatus );
  if (*pstatus!=Success) return; 

 /* 2 input, 1 output */
  I0Data  = ihb[0].Data;
  E0Data  = ihb[0].VarDat;
  I0Dummy = ihb[0].Dummy.V;
  I0DDummy = ihb[0].DDummy.V;
  I0Dim_1  = (int) ihb[0].Dim[1];
  I0Dim_2  = (int) ihb[0].Dim[2];

  I1Ave    = pcb->ib[1].Ave.V;
  I1Data   = ihb[1].Data;
  E1Data   = ihb[1].VarDat;
  I1Dummy  = ihb[1].Dummy.V;
  I1DDummy = ihb[1].DDummy.V;
  I1Dim_1  = (int) ihb[1].Dim[1];
  I1Dim_2  = (int) ihb[1].Dim[2];

 /* averaging of rows in input array 1 */

  // Get maximum world coordinate range W1_2 .. W3_2 of input array 1
  GetSmallRectangle( pcb,ihb,2,2,-1,1,1,Off_1,Ps_1,Off_2,Ps_2,&W1_2,&W3_2 );

  if (Row1.I) row1=Row1.V; else row1=W1_2;
  if (Row2.I) row2=Row2.V; else row2=W3_2; 

  // get range of coordinate 1 for input and output image 
  GetLinearRange      ( pcb, ihb,
                        1, 1, 0, 1, imax-1,
                        FALSE, 0.0, FALSE, 0.0,
                        Off_1, Ps_1, Off_2, Ps_2,
                        f1_1, f3_1, Df_1,
                        &Imin_1, &Imax_1, &Wmin_1, &Wmax_1, &DW_1, 1 );

  // get range of coordinate 2 without output image (OutImage=-1)
  GetLinearRange      ( pcb, ihb,
                        2, 2, 1, 1, imax-1,
                        1, row1, 1, row2,
                        Off_1, Ps_1, Off_2, Ps_2,
                        f1_2, f3_2, Df_2,
                        &Imin_2, &Imax_2, &Wmin_2, &Wmax_2, &DW_2, 1 );

  // write projection (usually I0Dim_2 equals 1, but larger if required)
  for (i_2=0;i_2<I0Dim_2;i_2++) {
    pI0Data = ABSPTR(I0Data,I0Dim_1,I0Dim_2,Imin_1,i_2);
    pE0Data = E0Data?E0Data-I0Data+pI0Data:NULL;
    Projection_1  ( pI0Data, pE0Data, I0Dim_1, Imin_1, Imax_1, I0Dummy, 1.0,
                    I1Data, E1Data, I1Dim_1, I1Dim_2,
                    f1_1[1], f3_1[1], Df_1[1], f1_2[1], f3_2[1],
                    I1Dummy, I1DDummy, I1Ave, pcb->TestBit );
  }

} /* saxs_collapserow*/

/*---------------------------------------------------------------------------
saxs_collapserow_init

Initializes the output header of each output file before the output
data array is allocated.

The output header is already initialized with the header of the first
image of the first input file sequence. This routine modifies this default.

The number of available blocks is defined by the program and does not need
to be checked here. All input images are open in SaxsImageLoop when a new
output image is created.

saxs_collapserow:
  Axis 1 output: Same geometry as input image
  Axis 2 output: Change parameters to numerator. Modify geometrical
                 parameters only, if not already set by image block (.I<1)
                 Offset, Center, BinSiz, PixSiz, AxisType. The parameters
                 SampleDistance, WaveLength, DetRot, Orientation, Projection
                 are not changed.
---------------------------------------------------------------------------*/
void saxs_collapserow_init ( CmdBlk *pcb, long num[], ImgHeadBlk ihb[], int *pstatus )
{ ImgBlk *ib;
  long I1Inc;

  ib = pcb->ib;

  //++++++++++ if (Single2Multi()) I1Inc = ib[1].FInc.V;
  //++++++++++ else I1Inc = ib[1].Inc.V;

  /*--- Set output header */

  /* output offset 2 */
  if (ihb[0].Offset[2].I>=0) {
    ihb[0].Offset[2].V = 0.0;
    ihb[0].Offset[2].I = TRUE;
  }

  /* output center 2 */
  if (ihb[0].Center[2].I>=0) {
    // INDEX2I(I,O) ((I)+(O)+DAI)
    ihb[0].Center[2].V = INDEX2I(0,ihb[0].Offset[2].V);
    ihb[0].Center[2].I = TRUE;
  }

  /* output BinSiz 2 */
  if (ihb[0].BinSiz[2].I>=0) {
    ihb[0].BinSiz[2].V = 1; //++++++++++I1Inc;
  }

  /* output pixel size 2 */
  if (ihb[0].PixSiz[2].I>=0) {
    ihb[0].PixSiz[2].V =  1; //++++++++++ I1Inc;
    ihb[0].PixSiz[2].I = TRUE;
  }

  /* Set AxisType_2 to Numerator */
  if (ihb[0].AxisType[2].I>=0) {
    ihb[0].AxisType[2].V = IO_AxisTypeNumerator; 
    ihb[0].AxisType[2].I = TRUE;
  }

  /* output dimension 2 */
  if (!ib[0].Dim[2].I) {
    ihb[0].Dim[2] = 1;
  }

} // saxs_collapserow_init

/*---------------------------------------------------------------------------
user_io
Do all the keyboard io and return cb, and ib
---------------------------------------------------------------------------*/

void user_io(CmdBlk * pcb, ImgBlk * ib, int * pstatus)
{
  char  progfn[InputLineLength];
  ImgHeadBlk ihb[BlockNum];

  float ROff_1, RPs_1, UOff_1, UPs_1;
  float ROff_2, RPs_2, UOff_2, UPs_2;

 /* Determine program name without directory */
   (void) RightFR((char *) pcb->argv[0],DirSeparator,progfn,InputLineLength);

  /* show usage if no arguments are given */
  if (pcb->argc<=1) printf("Usage: %s %s\n",progfn,Usage);

  /*--- Write name of program ---*/
  printf("\n");
  printf("   %s %s\n",progfn,Version);
  printf("\n");

  ArgvFilenames ( pcb, ib, ihb, 0, BlockNum-1, pstatus);
  if (*pstatus!=Success) return;
  GetReference(pcb->RSys.V,1,ihb,&ROff_1,&ROff_2,&RPs_1,&RPs_2,pstatus );
  if (*pstatus!=Success) return;
  GetReference(pcb->USys.V,1,ihb,&UOff_1,&UOff_2,&UPs_1,&UPs_2,pstatus );
  if (*pstatus!=Success) return;

  /*--- Argument  : multiplication factor */
  printf("<1st sequence> = <read 1st sequence> * Factor + Const\n");
  argv_flexp(pcb,"Multiplication factor",&ib[1].Factor,"1.0",pstatus);
  if (*pstatus!=Success) return;
 
  /*--- Argument  : additive constant */
  argv_flexp(pcb,"Addition constant",&ib[1].Const,"0.0",pstatus);
  if (*pstatus!=Success) return;
 
  /*--- Argument  : multiplication factor */
  printf("<output sequence> = (<1st sequence>+<2nd sequence>)*Factor+Const\n");
  argv_flexp(pcb,"Multiplication factor",&ib[0].Factor,"1.0",pstatus);
  if (*pstatus!=Success) return;
 
  /*--- Argument  : additive constant */
  argv_flexp(pcb,"Addition constant",&ib[0].Const,"0.0",pstatus);
  if (*pstatus!=Success) return;

  printf("\n");
  if (ib[1].Name.I)    printf("i/p file 1         : %s\n",ib[1].Name.V);
  if (ib[0].Name.I)    printf("o/p file           : %s\n",ib[0].Name.V);
  if (ib[1].First.I)   printf("first image in 1   : %d\n",ib[1].First.V);
  if (ib[1].Last.I)    printf("last image         : %d\n",ib[1].Last.V);
  if (ib[1].Inc.I)     printf("increment          : %d\n",ib[1].Inc.V);
  if (ib[0].Dummy.I)   printf("output dummy       : %s\n",ib[0].Dummy.V);
  if (ib[0].Dim[1].I)  printf("output dimension 1 : %s\n",ib[0].Dim[1].V);
  if (ib[0].Dim[2].I)  printf("output dimension 2 : %s\n",ib[0].Dim[2].V);
  if (ib[1].Factor.I)  printf("i/p 1 factor       : %s\n",ib[1].Factor.V);
  if (ib[1].Const.I)   printf("i/p 1 constant     : %s\n",ib[1].Const.V);
  if (ib[0].Factor.I)  printf("output factor      : %s\n",ib[0].Factor.V);
  if (ib[0].Const.I)   printf("output constant    : %s\n",ib[0].Const.V);
  printf("\n");

  if (pcb->TestBit) {
    PrintBlocks ( pcb, ib );
    printf("\n"); }

  return;
} /* user_io */

/*---------------------------------------------------------------------------
main
---------------------------------------------------------------------------*/

#if MAKE_FUNCTION
# define MAIN main_saxs_collapserow
#else
# define MAIN main
#endif

int MAIN (int argc, char *argv[])
{
  CmdBlk cb;                /* command block  */
  ImgBlk ib[BlockNum];      /* image blocks */

  int status;
  int arg_no = 0;

 /* Init options, control block and image blocks */
  InitOptions( Usage, Version, NULL, TRUE, &cb, ib, BlockNum );

  option_define ( IO_tpflexp, "row1", "0.0", &cb );
  option_define ( IO_tpflexp, "row2", "0.0", &cb );

 /* Read options from argument list */
  ReadOptions( argv, &arg_no, &cb, ib, &status);

  /* USER KEYBOARD I/O */
  if (status==Success) {
    argv_start ( &cb, 1 );
    user_io( &cb, ib, &status);
    argv_end( &cb ); /* must be called after user_io */
  }

  /* SEQUENCE CALCULATION */
  if (status==Success)
    IMAGELOOP( &cb, ib, saxs_collapserow, saxs_collapserow_init, NULL, TRUE, &status );

  return( ReportSaxsStatus( status, 0 ) );

} /* MAIN */

