/***************************************************************************/
/* Written 1994++ by Peter Boesecke                                        */
/* Copyright (C) 2011 European Synchrotron Radiation Facility              */
/*                       Grenoble, France                                  */
/*                                                                         */
/*    Principal authors: Peter Boesecke  (boesecke@esrf.eu)                */
/*                                                                         */
/*    This program is free software: you can redistribute it and/or modify */
/*    it under the terms of the GNU General Public License as published by */
/*    the Free Software Foundation, either version 3 of the License, or    */
/*    (at your option) any later version.                                  */
/*                                                                         */
/*    This program is distributed in the hope that it will be useful,      */
/*    but WITHOUT ANY WARRANTY; without even the implied warranty of       */
/*    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the        */
/*    GNU General Public License for more details.                         */
/*                                                                         */
/*    You should have received a copy of the GNU General Public License    */
/*    along with this program.  If not, see <http://www.gnu.org/licenses/>.*/
/***************************************************************************/
/*+++
1 saxs_angle
  Transformation from cartesian coordinates to polar coordinates 

2 PURPOSE
  Transformation from cartesian coordinates to polar coordinates
  with pixel interpolation and azimuthal averaging.

  Calculation:       I0 = I1
  Error propagation: V0 = V1

  x_1 of the output file corresponds to the radius
  x_2 of the output file corresponds to the angle

  The angle coordinate (x_2) uses always the real reference system (Offset, 
  PixSiz). Center, WaveLength and SampleDistance are not used.

  Arguments:
  saxs_angle [options] <i1nam> <onam> <i1fst> <i1lst> <i1inc> <odum> 
                <i1cen_1> <i1cen_2> <r0> <dr> <odim_1> <a0> <da> <odim_2>"

  Defaults:
  <input file name>   : input.edf 
  <output file name>  : output.edf
  <first image>       : <first image number in input file>
  <last image>                              : <last image number in input file>
     if argument list ends with first image : <first image>
  <increment>         : 1
  <dummy>             : <dummy value in first image of input file>
  <i1cen_1> (image)   : <image center coordinate along dimension 1>
  <i1cen_2> (image)   : <image center coordinate along dimension 2>
  <r0>      (rsys)    : <minimum radius coordinate>
  <dr>      (rsys)    : <radial interval>
  <odim_1>            : <number of radial intervals>
  <a0>      (rsys)    : <minimum angle coordinate>
  <da>      (rsys)    : <angular interval>
  <odim_2>            : <number of angular intervals>

  The image coordinate of the center is (<i1cen_1>, <i1cen_2>). The input 
  data array is radially cut into <odim_1> rings with the thicknesses <dr>.  
  The rings are defined in the following way:
  0 <= o_1 <= (odim_1-1) from <minimum radius> + o_1 * <radial interval> to
  <minimum radius> + (o_1+1) * <radial interval>
  The default of <dr> is chosen in that way that not more than one full pixel
  falls between two subsequent rings. 

  The input data array is azimuthally averaged along the center of the odim_1
  numbers of rings which are defined by 

  0 <= o_1 <= (odim_1-1),  
  <minimum radius> + (o_1+0.5) * <radial interval>.

  The azimuthal integration is done in odim_2 angular sections which are 
  defined by:

  0 <= o_2 <= (odim_2-1),
  <angle minimum> + o_2 * <angular interval>, 
  <angle minimum> + (o_2+1) * <angular interval>. 

  All pixel values are linearly interpolated between the four neighbours.

  The averaged value for ring o_1 and angle o_2 is written to element 
  (o_1,o_2) of the output array.

  Wildcard:
  A "=" in the argument list chooses the default.

2 HISTORY
  13-Mar-1996 PB  V3.0
  21-Aug-1996 PB  V3.1 
                  option -s2 for multiplication of the result with s2 and a
                  factor (-s2 <factor>)
                  (-rsys array, -rsys image : multiplication with radius^2,
                   -rsys real : multiplication with radius^2 [m],
                   -rsys saxs : mulitplication with s^2 [1/nm] )
  30-Aug-1996 PB  V3.22
                  option +s2 multiplication with radius^2.
                  This allows the direct calculation of re^2*Q/V from the 
                  data by summation over i_1.
                  (-rsys array, -rsys image : multiplication with radius^2,
                   -rsys real : multiplication with radius^2 [m^2],
                   -rsys saxs : mulitplication with s^2 [1/nm^2])
  21-Sep-1996 PB  V3.3
                  option +csym : multiplication of the input pattern with
                                 Pi/2*sin(Alpha-ASym), where Alpha is the 
                                 angle of the data point from the w_1 axis 
                                 and ASym the angle of the symmetry axis from
                                 the w_1 axis (default: no multiplication)
                  option -asym : specification of the symmetry axis in degrees
                                 from the w_1 axis (default: 0.0_deg)
  31-Dec-1999 PB  V3.4 new reference system BEAM 
  10-Jan-2000 PB  V3.5 new reference systems CENTER and NORMAL, 
                  NORMAL replaces BEAM
  22-Mar-2000 PB  V3.6 optional output of azimuthal sum 
                  option: +vsum
  2000-08-04  PB  V3.6 %f->%g
  2000-11-16  PB  V3.7 azimuthal sum calculated with weight
  2001-07-08  PB  V3.8 FFirst
  2001-07-09  PB  V4.00 new loop, no repetition
  2003-02-02  PB  V4.01 num[BlockNum]
  2003-02-07  PB  V4.02 ib[].Bin -> ihb[].Bin
  2003-02-09  PB  V4.03 line args for Dummy, DDummy, Dim
  2003-02-10  PB  V4.04 pcb->Shift -> ihb[0].Shift
  2003-05-25  PB  V4.05 line args for Center
  2003-05-30  PB  V4.05 new parameter list for longkeyvalue (unused)
                        ImageLoop: function call changed
  2003-08-12  PB  V4.06 option_define, all option numbers removed
  2003-11-30  PB  V4.07 BinSiz, IO_Region
  2004-10-14  PB  V4.08 all angles in radian or with '_' and unit
                        default of Asym corrected from 1.0 deg to 0.0_deg 
                        (typing error?)
  2004-10-31  PB  V4.09 MAKE_FUNCTION
  2004-12-03  PB  V4.10 saxs_angle_init: Angular reference system is Real,
                        -> Offset_2[0] is calculated with R2OFFSET, independent
                        of the reference system. PixSiz, BinSiz, Offset are
                        calculated for all reference systems
  2005-07-15  PB  V4.11 ReportSaxsStatus parameter list updated
  2005-08-04  PB  V4.12 PrintImageHeaderBlock paramters updated
  2005-09-20  PB  V4.13 VarDat, saxs_angle_init creates VarDat,
                        GetReferenceParametersC, S2PSIZE for IO_SAXS,
                        integration corrected
  2005-10-08  PB  V4.14 ImageIsNew added
  2005-10-08  PB  V4.15 angle_sum
  2005-10-13  PB  V4.16 Xy
  2005-10-14  PB  V4.17 write DisplacedDim_... into displacement files
  2005-10-28  PB  V4.18 ERROUT
  2007-04-23  PB  V4.19 void SetOutputDimension, 
                        bug in function concat correction (avoids bufferovfl)
  2007-06-02  PB  V4.20 No default for i1cen
  2007-06-11  PB  V4.21 ImageLoop parameters updated
  2007-12-01  PB  V4.22 AxisType
  2008-08-07  PB  V4.23 saxs_angle_init, no global defaults for dummy and Dim_1
  2008-12-05  PB  V4.24 open all files as binary file to avoid
                        translation of \r\n to \n when compiled on windows.
  2009-10-02  PB  V4.25 SaxsAngle -> angle
  2011-06-07  PB  V4.26 strncat -> strncpy
  2011-06-30  PB  V4.27 concat->strlib_concat
  2011-07-08  PB  V4.28 include strlib.h
  2011-07-24  PB  V4.29 reference system IO_Tangens
  2012-01-31  PB  V4.30 ImageLoop parameters updated
  2012-06-27  PB  V4.31 lcc workaround
  2015-10-03  PB  V4.32 FileAlreadyOpen

---*/
# define Version  "saxs_angle V4.32 2015-10-03, Peter Boesecke"

# include <errno.h>
# include <stdio.h>
# include <unistd.h>
# include <fcntl.h>
# include <string.h>

# include "SaxsPrograms.h"
# include "angle.h"
# include "strlib.h"

# define FREE( a) { if ( a) { free( a); a = NULL; } }

/* Defaults */
# define Default_dim1  300
# define Default_dim2  360

/* Special options */
# define SR0       "r0"    /* float */
# define SDr       "dr"    /* float */
# define SA0       "a0"    /* float */
# define SDa       "da"    /* float */
# define SAsym     "asym"  /* float */

# define SXyname   "xyna"  /* line */  // xy-file name
# define SXy       "xy"    /* flag */  // xy-file flag

# define SS2       "s2"    /* flag */
# define SCsym     "csym"  /* flag */
# define SVsum     "vsum"  /* flag */

# define Usage "[options] \n\
                <i1nam> <onam> <i1fst> <i1lst> <i1inc> <odum> \n\
                <i1cen1> <i1cen2> <r0> <dr> <dim1> <a0> <da> <dim2> "

# define BlockNum 2       /* 1 input sequence + 1 output sequence */

/*---------------------------------------------------------------------------
DefaultRadialInterval

 Returns a default interval for the radius depending on the chosen
 reference system.
---------------------------------------------------------------------------*/
float DefaultRadialInterval( CmdBlk * pcb, ImgHeadBlk ihb[], int blkno, 
                             int * pstatus )
{ float dr;
  ImgHeadBlk * pihb;

   *pstatus = Success;
   dr = 1.0;
   pihb = &ihb[blkno];

   switch (pcb->RSys.V) {
     case IO_Array : dr = 1.0; break;
     case IO_Image : dr = 1.0; break;
     case IO_Center: dr = 1.0; break;
     case IO_Region: dr = PSIZE2R( MIN2( fabs(pihb->BinSiz[1].V),
                                   fabs(pihb->BinSiz[2].V) ) ); break;
     case IO_Real  : if (!( (pihb->PixSiz[1].I) && (pihb->PixSiz[2].I) ) ) {
                       *pstatus = Failed; break; }
                     dr = PSIZE2R( MIN2( fabs(pihb->PixSiz[1].V),
                                   fabs(pihb->PixSiz[2].V) ) ); break;
     case IO_Normal: if (!( (pihb->PixSiz[1].I) && (pihb->PixSiz[2].I) ) ) {
                       *pstatus = Failed; break; }
                     dr = PSIZE2N( MIN2( fabs(pihb->PixSiz[1].V),
                                   fabs(pihb->PixSiz[2].V) ) ); break;
     case IO_Tangens : if (!( (pihb->PixSiz[1].I) && (pihb->PixSiz[2].I) &&
                       (pihb->SampleDistance.I) ) ) {
                       *pstatus = Failed; break; }
                     dr = PSIZE2T( MIN2( fabs(pihb->PixSiz[1].V),
                                   fabs(pihb->PixSiz[2].V) ),
                                   pihb->SampleDistance.V ); break;
     case IO_Saxs  : if (!( (pihb->PixSiz[1].I) && (pihb->PixSiz[2].I) &&
                       (pihb->SampleDistance.I) && (pihb->WaveLength.I) ) ) {
                       *pstatus = Failed; break; }
                     dr = PSIZE2S( MIN2( fabs(pihb->PixSiz[1].V),
                                   fabs(pihb->PixSiz[2].V) ),
                                   pihb->SampleDistance.V,
                                   pihb->WaveLength.V ); break;
     default       : *pstatus = Failed; break;
     }

  if (*pstatus!=Success) {
    fprintf(ERROUT,"Missing parameters for the chosen reference system (% s)\n",
            reftostr(pcb->RSys.V)); return( dr ); }

  return ( dr );

} /* DefaultRadialInterval */

/*---------------------------------------------------------------------------
calc_dist
---------------------------------------------------------------------------*/
float calc_dist( float A[], float B[] )
{ return(sqrt((A[1]-B[1])*(A[1]-B[1])+(A[2]-B[2])*(A[2]-B[2])));
} /* calc_dist */

/*---------------------------------------------------------------------------
SetOutputDimension

 Sets the output dimension ihb[0].Dim[1] and ihb[0].Dim[2] to a default.

 If the flags ib[0].Dim[1].I or ib[0].Dim[2].I are set these values are used.

 If the flag ib[0].Dim[1].I is not set the maximum image distance 
 between the center and the 4 edges of the image ihb[blkno] is used as 
 default for Dim[1].

 If the flag ib[0].Dim[2].I is not set the default for Dim[2] is 
 Default_dim2.
---------------------------------------------------------------------------*/
void SetOutputDimension( CmdBlk * pcb, ImgHeadBlk ihb[], int blkno,
                          int * pstatus )
{ ImgHeadBlk * pihb;
  float Off_1, Off_2;
  float Ps_1, Ps_2;
  float edge1[3], edge2[3], edge3[3], edge4[3];
  float center[3];
  float distance;
  float i_10, i_11, i_20, i_21;
  ImgBlk *ib = pcb->ib;
 
  *pstatus = Success;

  pihb = &ihb[blkno];

  if (!( (pihb->Center[1].I) && (pihb->Center[2].I) ) ) {
    *pstatus = Failed; 
    fprintf(ERROUT,"Missing center definitions\n"); return;
  } else {
    center[1] = pihb->Center[1].V; center[2] = pihb->Center[2].V;
  }
 
  if (!(ib[0].Dim[1].I)) {
    IMAGEREF(Off_1,Ps_1,ihb[1].Offset[1].V);
    IMAGEREF(Off_2,Ps_2,ihb[1].Offset[2].V);

    i_10 = i_20 = A2INDEX(ARRAYSTART+LOWERBORDER);
    i_11 = A2INDEX(ARRAYSTART+LOWERBORDER+ihb[1].Dim[1]);
    i_21 = A2INDEX(ARRAYSTART+LOWERBORDER+ihb[1].Dim[2]);

    /* edge 1 */
    edge1[1] = WORLD(i_10,Off_1,Ps_1); 
    edge1[2] = WORLD(i_20,Off_2,Ps_2);

    edge2[1] = WORLD(i_11,Off_1,Ps_1);
    edge2[2] = edge1[2];

    edge3[1] = edge2[1];
    edge3[2] = WORLD(i_21,Off_2,Ps_2);

    edge4[1] = edge1[1];
    edge4[2] = edge3[2];

    distance = MAX4( calc_dist(edge1,center), calc_dist(edge2,center), 
                     calc_dist(edge3,center), calc_dist(edge4,center) );
 
    ihb[0].Dim[1] = (long int) distance; 

  }

  if (!(ib[0].Dim[2].I)) ihb[0].Dim[2] = Default_dim2;
    
} /* SetOutputDimension */

/*---------------------------------------------------------------------------
 saxs_angle_init

 Prepare output header parameters for polar coordinates (x_1 = radius, 
 x_2 = angle) and allocate memory for output block ihb[0].

 All geometrical parameters that have not been set by the image block, i.e.
 with .I flags >= 0, are changed according to the following list.

  Defaults:

  r0 = 0.0
  dr = DefaultRadialInterval;

  i1cen_1 no default
  i1cen_2 no default

  ocsys = "polar"

  ocen_1 = 0.0;
  ocen_2 = 0.0;

  array,
  image : if ((i1pix_1.I) && (i2pix_2.I)) {
          opix_1   (Alpha1 != Alpha0) :
                 = sqrt( i1pix_1^2 + (i1pix_2^2 - i1pix_1^2) * 0.5 *
                     ( 1 - 0.5 * (sin(2 Alpha1) - sin(2 Alpha0)) /
                                 (Alpha1 - Alpha0) ) )
                   Alpha1 = Alpha0 :
                 = sqrt( i1pix_1^2 + (i1pix_2^2 - i1pix_1^2) * 0.5 *
                     ( 1 - cos(2 Alpha0) ) );
          }
          opix_2 = R2PSIZE(da);
          ooff_1 = I2OFFSET(r0);
          ooff_2 = R2OFFSET(a0,opix_2);

  region: ...

  real :  opix_1 = R2PSIZE(dr);
          opix_2 = R2PSIZE(da);
          ooff_1 = R2OFFSET(r0,opix_1);
          ooff_2 = R2OFFSET(a0,opix_2);

  center: ...

  normal: ...

  saxs :  opix_1 = S2PSIZE(dr,SampleDistance,WaveLength);
          opix_2 = R2PSIZE(da);
          ooff_1 = S2OFFSET(r0,opix_1,ocen_1,SampleDistance,WaveLength);
          ooff_2 = R2OFFSET(a0,opix_2);

 return value            :	void
 CmdBlk * pcb	         :   
 ImgHeadBlk ihb[0] (i/o) :
 ImgHeadBlk ihb[1] (i)   :
 int * pstatus     (o)   : return status

---------------------------------------------------------------------------*/
void saxs_angle_init( CmdBlk * pcb, long num [], ImgHeadBlk ihb[], 
                      int * pstatus )
{ ImgBlk *ib;

  const float alpha_eps = 1e-30;
  const float deg2rad = SAXS_PI/180.0, rad2deg = 180.0/SAXS_PI;
  register long int i;
  ImgHeadBlk * pihb;
  
  float Alpha0, Alpha1;
  float r0, dr, a0, da;
  float i1pix_1, i1pix_2;
  float i1bis_1, i1bis_2;

  IO_float R0, Dr, A0, Da, Asym;
  IO_long  S2;

  *pstatus = Success;

  if (pcb->TestBit) printf("saxs_angle_init\n");

  ib = pcb->ib;
  pihb = &ihb[0];

  /* calculate dimension for ihb[0] from ihb[1] or take from image block */
  SetOutputDimension( pcb, ihb, 1, pstatus );
  if (*pstatus!=Success) return;

  R0 = option_float ( SR0, pcb, pcb->ib, num, 1, pstatus );
  if (*pstatus!=Success) return;
  Dr = option_float ( SDr, pcb, pcb->ib, num, 1, pstatus );
  if (*pstatus!=Success) return;
  A0 = option_float ( SA0, pcb, pcb->ib, num, 1, pstatus );
  if (*pstatus!=Success) return;
  Da = option_float ( SDa, pcb, pcb->ib, num, 1, pstatus );
  if (*pstatus!=Success) return;

  S2 = option_flag ( SS2, pcb, pstatus );
  if (*pstatus!=Success) return;

  /* --- Set transformation parameters */
  if (!(Dr.I)) {
    Dr.V = DefaultRadialInterval(pcb,ihb,1,pstatus);
    if (*pstatus!=Success) return;    
  }

  /*--- Set output header to use polar coordinates */

  /* calculate geometrical header values for polar coordinates,
     replace all parameters that have not been set with an option, i.e.
     ....I>=0 */

  r0 = R0.V;
  dr = Dr.V;
  a0 = A0.V;
  da = Da.V;

  Alpha0 = a0;
  Alpha1 = (a0 + da * pihb->Dim[2]);
  i1pix_1 = ihb[1].PixSiz[1].V; i1pix_2 = ihb[1].PixSiz[2].V;
  i1bis_1 = ihb[1].BinSiz[1].V; i1bis_2 = ihb[1].BinSiz[2].V;

  /* Center is undefined for regrouped image, zero if not explicitely set */
  if (pihb->Center[1].I>=0) pihb->Center[1].V = 0.0; pihb->Center[1].I = TRUE;
  if (pihb->Center[2].I>=0) pihb->Center[2].V = 0.0; pihb->Center[2].I = TRUE;

  if (pihb->AxisType[2].I>=0) {
    pihb->AxisType[2].V = IO_AxisTypeAngle; pihb->AxisType[2].I = TRUE;
  }

  switch (pcb->RSys.V) {
    case IO_Array :  /* the same as IO_Image */;

    case IO_Image :  
         if (pihb->PixSiz[1].I>=0) {
           if ((ihb[1].PixSiz[1].I) && (ihb[1].PixSiz[2].I)) {
             if (fabs(Alpha1-Alpha0)<alpha_eps) {
               pihb->PixSiz[1].V = sqrt( i1pix_1*i1pix_1 +
                (i1pix_2*i1pix_2 - i1pix_1*i1pix_1) * 0.5 *
                 (1 - cos(2*Alpha0)) );
               } else {
               pihb->PixSiz[1].V = sqrt( i1pix_1*i1pix_1 +
                (i1pix_2*i1pix_2 - i1pix_1*i1pix_1) * 0.5 *
                 (1 - 0.5*(sin(2*Alpha1) - sin(2*Alpha0)) / (Alpha1-Alpha0)) );
               }
             pihb->PixSiz[1].I = TRUE;
             }
           }
         if (pihb->BinSiz[1].I>=0) {
           if ((ihb[1].BinSiz[1].I) && (ihb[1].BinSiz[2].I)) {
             if (fabs(Alpha1-Alpha0)<alpha_eps) {
               pihb->BinSiz[1].V = sqrt( i1bis_1*i1bis_1 +
                (i1bis_2*i1bis_2 - i1bis_1*i1bis_1) * 0.5 *
                 (1 - cos(2*Alpha0)) );
               } else {
               pihb->BinSiz[1].V = sqrt( i1bis_1*i1bis_1 +
                (i1bis_2*i1bis_2 - i1bis_1*i1bis_1) * 0.5 *
                 (1 - 0.5*(sin(2*Alpha1) - sin(2*Alpha0)) / (Alpha1-Alpha0)) );
               }
             pihb->BinSiz[1].I = TRUE;
             }
           }
         if (pihb->Offset[1].I>=0) {
           pihb->Offset[1].V = I2OFFSET(r0); pihb->Offset[1].I = TRUE;
           }
         /* The angular reference system is always Real */
         if (pihb->PixSiz[2].I>=0) {
           pihb->PixSiz[2].V = R2PSIZE(da); pihb->PixSiz[2].I = TRUE;
           }
         if (pihb->BinSiz[2].I>=0) {
          pihb->BinSiz[2].V = 1.0; pihb->BinSiz[2].I = TRUE;
          }
         if (pihb->Offset[2].I>=0) {
           pihb->Offset[2].V = R2OFFSET(a0,pihb->PixSiz[2].V);
           pihb->Offset[2].I = TRUE;
           }
         break;

    case IO_Center :
         if (pihb->PixSiz[1].I>=0) {
           if ((ihb[1].PixSiz[1].I) && (ihb[1].PixSiz[2].I)) {
             if (fabs(Alpha1-Alpha0)<alpha_eps) {
               pihb->PixSiz[1].V = sqrt( i1pix_1*i1pix_1 +
                (i1pix_2*i1pix_2 - i1pix_1*i1pix_1) * 0.5 *
                 (1 - cos(2*Alpha0)) );
               } else {
               pihb->PixSiz[1].V = sqrt( i1pix_1*i1pix_1 +
                (i1pix_2*i1pix_2 - i1pix_1*i1pix_1) * 0.5 *
                 (1 - 0.5*(sin(2*Alpha1) - sin(2*Alpha0)) / (Alpha1-Alpha0)) );
               }
             pihb->PixSiz[1].I = TRUE;
             }
           }
         if (pihb->BinSiz[1].I>=0) {
           if ((ihb[1].BinSiz[1].I) && (ihb[1].BinSiz[2].I)) {
             if (fabs(Alpha1-Alpha0)<alpha_eps) {
               pihb->BinSiz[1].V = sqrt( i1bis_1*i1bis_1 +
                (i1bis_2*i1bis_2 - i1bis_1*i1bis_1) * 0.5 *
                 (1 - cos(2*Alpha0)) );
               } else {
               pihb->BinSiz[1].V = sqrt( i1bis_1*i1bis_1 +
                (i1bis_2*i1bis_2 - i1bis_1*i1bis_1) * 0.5 *
                 (1 - 0.5*(sin(2*Alpha1) - sin(2*Alpha0)) / (Alpha1-Alpha0)) );
               }
             pihb->BinSiz[1].I = TRUE;
             }
           }
         if (pihb->Offset[1].I>=0) {
           pihb->Offset[1].V =
             C2OFFSET(r0,pihb->Center[1].V);
           pihb->Offset[1].I = TRUE;
           }
         /* The angular reference system is always Real */
         if (pihb->PixSiz[2].I>=0) {
           pihb->PixSiz[2].V = R2PSIZE(da); pihb->PixSiz[2].I = TRUE;
           }
         if (pihb->BinSiz[2].I>=0) {
          pihb->BinSiz[2].V = 1.0; pihb->BinSiz[2].I = TRUE;
          }
         if (pihb->Offset[2].I>=0) {
           pihb->Offset[2].V = R2OFFSET(a0,pihb->PixSiz[2].V);
           pihb->Offset[2].I = TRUE;
           }
         break;

    case IO_Region  :
         if (pihb->BinSiz[1].I>=0) {
          pihb->BinSiz[1].V = dr; pihb->BinSiz[1].I = TRUE;
          }
         if (pihb->PixSiz[1].I>=0) {
          pihb->PixSiz[1].V = i1pix_1*pihb->BinSiz[1].V/i1bis_1; 
          pihb->PixSiz[1].I = TRUE;
          }
         if (pihb->Offset[1].I>=0) {
           pihb->Offset[1].V = R2OFFSET(r0,pihb->BinSiz[1].V);
           pihb->Offset[1].I = TRUE;
           }
         /* The angular reference system is always Real */
         if (pihb->PixSiz[2].I>=0) {
           pihb->PixSiz[2].V = R2PSIZE(da); pihb->PixSiz[2].I = TRUE;
           }
         if (pihb->BinSiz[2].I>=0) {
          pihb->BinSiz[2].V = 1.0; pihb->BinSiz[2].I = TRUE;
          }
         if (pihb->Offset[2].I>=0) {
           pihb->Offset[2].V = R2OFFSET(a0,pihb->PixSiz[2].V);
           pihb->Offset[2].I = TRUE;
           }
         break;

    case IO_Real  :
         if (pihb->PixSiz[1].I>=0) {
          pihb->PixSiz[1].V = dr; pihb->PixSiz[1].I = TRUE;
          }
         if (pihb->BinSiz[1].I>=0) {
          pihb->BinSiz[1].V = i1bis_1*pihb->PixSiz[1].V/i1pix_1;
          pihb->BinSiz[1].I = TRUE;
          }
         if (pihb->Offset[1].I>=0) {
           pihb->Offset[1].V = R2OFFSET(r0,pihb->PixSiz[1].V); 
           pihb->Offset[1].I = TRUE;
           }
         if (pihb->PixSiz[2].I>=0) {
          pihb->PixSiz[2].V = R2PSIZE(da); pihb->PixSiz[2].I = TRUE;
          }
         if (pihb->BinSiz[2].I>=0) {
          pihb->BinSiz[2].V = 1.0; pihb->BinSiz[2].I = TRUE;
          }
         if (pihb->Offset[2].I>=0) {
           pihb->Offset[2].V = R2OFFSET(a0,pihb->PixSiz[2].V); 
           pihb->Offset[2].I = TRUE;
           }
         break;

    case IO_Normal :
         if (pihb->PixSiz[1].I>=0) {
          pihb->PixSiz[1].V = N2PSIZE(dr); pihb->PixSiz[1].I = TRUE;
          }
         if (pihb->BinSiz[1].I>=0) {
          pihb->BinSiz[1].V = i1bis_1*pihb->PixSiz[1].V/i1pix_1;
          pihb->BinSiz[1].I = TRUE;
          }
         if (pihb->Offset[1].I>=0) {
           pihb->Offset[1].V =
             N2OFFSET(r0,pihb->PixSiz[1].V,pihb->Center[1].V);
           pihb->Offset[1].I = TRUE;
           }
         /* The angular reference system is always Real */
         if (pihb->PixSiz[2].I>=0) {
           pihb->PixSiz[2].V = R2PSIZE(da); pihb->PixSiz[2].I = TRUE;
           }
         if (pihb->BinSiz[2].I>=0) {
          pihb->BinSiz[2].V = 1.0; pihb->BinSiz[2].I = TRUE;
          }
         if (pihb->Offset[2].I>=0) {
           pihb->Offset[2].V = R2OFFSET(a0,pihb->PixSiz[2].V);
           pihb->Offset[2].I = TRUE;
           }
         break;

    case IO_Tangens  :
         if (pihb->PixSiz[1].I>=0) {
           pihb->PixSiz[1].V =
             T2PSIZE(dr,pihb->SampleDistance.V);
           pihb->PixSiz[1].I = TRUE;
           }
         if (pihb->BinSiz[1].I>=0) {
           pihb->BinSiz[1].V = i1bis_1*pihb->PixSiz[1].V/i1pix_1;
           pihb->BinSiz[1].I = TRUE;
           }
         if (pihb->Offset[1].I>=0) {
           pihb->Offset[1].V =
             T2OFFSET(r0,pihb->PixSiz[1].V,pihb->Center[1].V,
                      pihb->SampleDistance.V);
           pihb->Offset[1].I = TRUE;
           }
         /* The angular reference system is always Real */
         if (pihb->PixSiz[2].I>=0) {
           pihb->PixSiz[2].V = R2PSIZE(da); pihb->PixSiz[2].I = TRUE;
           }
         if (pihb->BinSiz[2].I>=0) {
           pihb->BinSiz[2].V = 1.0; pihb->BinSiz[2].I = TRUE;
           }
         if (pihb->Offset[2].I>=0) {
           pihb->Offset[2].V = R2OFFSET(a0,pihb->PixSiz[2].V);
           pihb->Offset[2].I = TRUE;
           }
         break;

    case IO_Saxs  :
         if (pihb->PixSiz[1].I>=0) {
           pihb->PixSiz[1].V = 
             S2PSIZE(dr,pihb->SampleDistance.V,pihb->WaveLength.V);
           pihb->PixSiz[1].I = TRUE;
           }
         if (pihb->BinSiz[1].I>=0) {
          pihb->BinSiz[1].V = i1bis_1*pihb->PixSiz[1].V/i1pix_1;
          pihb->BinSiz[1].I = TRUE;
          }
         if (pihb->Offset[1].I>=0) {
           pihb->Offset[1].V = 
             S2OFFSET(r0,pihb->PixSiz[1].V,pihb->Center[1].V,
                      pihb->SampleDistance.V,pihb->WaveLength.V);
           pihb->Offset[1].I = TRUE;
           }
         /* The angular reference system is always Real */
         if (pihb->PixSiz[2].I>=0) {
           pihb->PixSiz[2].V = R2PSIZE(da); pihb->PixSiz[2].I = TRUE;
           }
         if (pihb->BinSiz[2].I>=0) {
           pihb->BinSiz[2].V = 1.0; pihb->BinSiz[2].I = TRUE;
           }
         if (pihb->Offset[2].I>=0) {
           // write R2OFFSET into two lines, lcc workaround
           pihb->Offset[2].V = R2OFFSET(a0,\
             pihb->PixSiz[2].V);
           pihb->Offset[2].I = TRUE;
           }
         break;

    default : 
         fprintf(ERROUT,"Impossible reference system : %s\n",
           reftostr(pcb->RSys.V));
         *pstatus = Failed; return;

    } /* switch */

  if (pcb->TestBit) {
    printf("r0 = %g\n",r0);
    printf("dr = %g\n",dr);
    printf("a0 = %g\n",a0);
    printf("da = %g\n",da);
    PrintImageHeaderBlock(&ihb[0]);
  }

  if (*pstatus!=Success) {
    fprintf(ERROUT,"Missing parameters for the chosen reference system (% s)\n",
            reftostr(pcb->RSys.V)); return; }

} /* saxs_angle_init */

/*---------------------------------------------------------------------------
1 saxs_angle

2 PURPOSE
  Transformation from cartesian coordinates to polar coordinates
---------------------------------------------------------------------------*/
void saxs_angle (CmdBlk * pcb, long num [], ImgHeadBlk ihb[], int * pstatus)
{ const float deg2rad = SAXS_PI/180.0, rad2deg = 180.0/SAXS_PI;
  const float csym_fac = SAXS_PI/2.0, radius_eps = 1e-30;

  const char * RoutineName = "saxs_angle";

  const char * OutputBinaryName = "i1test.bin";
  FILE * OutputBinaryId;

  int i,imax;
  int j;

  float *I0Data, *E0Data;
  int   I0Dim_1,I0Dim_2;
  float *pI0Data, *pE0Data;
  float I0Dummy, I0DDummy;
  float VarDDummy=DDSET(VarDummy);

  int   I1Ave;
  float *I1Data, *E1Data;
  int   I1Dim_1,I1Dim_2;
  float *pI1Data, *pE1Data;
  float I1Dummy, I1DDummy;
  float I1Value, I1Sum, I1Weight;
  float E1Value, E1Sum, E1Weight;

  float I1CircleSum, I1CircleSumWeight;
  float E1CircleSum, E1CircleSumWeight;

  float I1Cen_1,I1Cen_2;
  float Wcenter_1, Wcenter_2;
  float Radius2, Radius, Angle, DAngle, DDAngle, Angle0;

  float Radius_1, Radius_2;
  float SinDiffAngle, SAngle, SinSAngle, CosSAngle, SinAngle, CosAngle; 

  float factor;

  int i_1, i_2;

  float f0_1, f0_2;
  float W0_1, W0_2;
  float W1_1, W1_2;

  float Off_1[BlockNum], Off_2[BlockNum];
  float Ps_1[BlockNum], Ps_2[BlockNum];

  IO_float Asym;
  IO_long  S2, Csym, Vsum;

 /* xyfile parameters */
  char xypath[IO_len], xyname[IO_len];
  char xy_pattern[IO_len], xyfull[IO_len];
  char xfile[IO_len], yfile[IO_len];
  int xstream, ystream;
  float *xdata, *ydata;
  float *pxdata, *pydata;
  int idata;
  long xynum;
  size_t xysize;
  int ErrorValue;

  IO_long Xy;
  IO_line Xyname;

  char key[IO_len];

  float I0Off_1, I0Ps_1;
  float I0Off_2, I0Ps_2;
  float I1Off_1, I1Ps_1;
  float I1Off_2, I1Ps_2;

  float xybis_1, xybis_2;
  float xypix_1, xypix_2;
  float xycen_1, xycen_2;
  float xydis, xywvl;
  float xyrot_1, xyrot_2, xyrot_3;

  SetSaxsError( Version, RoutineName, NULL );

  *pstatus = Success;

  imax = pcb->ImgBlkLen;
  printf("\n Calculating ihb[0,% d] = Function(",ihb[0].ImNum);
  for(i=1;i<imax;i++) {
    printf("ihb[% d,% d] ", i, ihb[i].ImNum); }

  printf(")\n\n");

 /* Check the number of images */
  if (pcb->ImgBlkLen!=2) {
    fprintf(ERROUT,"%d images found, 1 input and 1 output image required\n",
            pcb->ImgBlkLen); *pstatus=Failed; return; }

 /* --- Get option values */
  Asym = option_float ( SAsym, pcb, pcb->ib, num, 1, pstatus );
  if (*pstatus!=Success) return;
  S2   = option_flag ( SS2, pcb, pstatus );
  if (*pstatus!=Success) return;
  Csym = option_flag ( SCsym, pcb, pstatus );
  if (*pstatus!=Success) return;
  Vsum = option_flag ( SVsum, pcb, pstatus );
  if (*pstatus!=Success) return;

  Xy = option_flag( SXy, pcb, pstatus);
  if (*pstatus!=Success) return;
  Xyname = option_line ( SXyname, pcb, pcb->ib, num, 1, pstatus );
  if (*pstatus!=Success) return;

  /* input block */
  GetReferenceParameters( pcb, ihb, 1, imax-1,
                          Off_1, Off_2, Ps_1,  Ps_2, pstatus );
  if (*pstatus!=Success) return;

  /* output block */
  GetReferenceParametersC( pcb->RSys.V, pcb, ihb, 1, 0, 0,
                           Off_1, Ps_1, pstatus );
  /* Set the angular reference system to real */
  GetReferenceParametersC( IO_Real, pcb, ihb, 2, 0, 0,
                           Off_2, Ps_2, pstatus );

/* Subtract output shift for calculation */
  if (ihb[0].Shift[2].I) Off_2[0] = Off_2[0]-ihb[0].Shift[2].V;
/* Center for output angle axis is always 0.0 */
  ihb[0].Center[2].V = 0.0;

  if (pcb->TestBit) {
    printf("Off_1[%d] = % f, Ps_1[%d] = % f\n", 0,Off_1[0],0,Ps_1[0]);
    printf("Off_2[%d] = % f, Ps_2[%d] = % f\n", 0,Off_2[0],0,Ps_2[0]); 
  }

/*  BinSiz is no longer well defined, set it to 1 */
  if (ihb[0].BinSiz[1].I>=0) ihb[0].BinSiz[1].V = 1.0; // start again from 1
  if (ihb[0].BinSiz[2].I>=0) ihb[0].BinSiz[2].V = 1.0;

  /* 1 input, 1 output */
  I0Data  = ihb[0].Data;
  E0Data = ihb[0].VarDat;
  I0Dummy = ihb[0].Dummy.V;
  I0DDummy = ihb[0].DDummy.V;
  I0Dim_1  = (int) ihb[0].Dim[1];
  I0Dim_2  = (int) ihb[0].Dim[2];

  I1Ave   = pcb->ib[1].Ave.V;
  I1Data  = ihb[1].Data;
  E1Data  = ihb[1].VarDat;
  I1Dummy = ihb[1].Dummy.V;
  I1DDummy = ihb[1].DDummy.V;
  I1Dim_1  = (int) ihb[1].Dim[1];
  I1Dim_2  = (int) ihb[1].Dim[2];

  I1Off_1  = ihb[1].Offset[1].V;
  I1Off_2  = ihb[1].Offset[2].V;
  I1Cen_1  = ihb[1].Center[1].V;
  I1Cen_2  = ihb[1].Center[2].V;

  /* calculate actual center */
  Wcenter_1 = WORLD(I2INDEX(I1Cen_1,I1Off_1),Off_1[1],Ps_1[1]); 
  Wcenter_2 = WORLD(I2INDEX(I1Cen_2,I1Off_2),Off_2[1],Ps_2[1]);
  
  SAngle  = Asym.V;

  if (pcb->TestBit) {
    printf("Wcenter_1 = % g,     Wcenter_2 = % g\n",Wcenter_1,Wcenter_2); 
    printf("SAngle    = % g_deg\n", SAngle * rad2deg);
  }

  if ( ((Csym.V)||(S2.V)) && (ImageIsNew(1,ihb)) ) {
    if (Csym.V)
      printf(" Multiplication with (Pi/2)*sin(Alpha-% g_deg)\n",SAngle*rad2deg);
    if (S2.V)
      printf(" Multiplication with radius^2\n");

    SinSAngle = sin(SAngle); 
    CosSAngle = cos(SAngle); 
    pI1Data   = ABSPTR(I1Data,I1Dim_1,I1Dim_2,0,0);
    pE1Data   = E1Data-I1Data+pI1Data;
    for (i_2=0;i_2<I1Dim_2;i_2++) {
      Radius_2     = WORLD(i_2,Off_2[1],Ps_2[1])-Wcenter_2;
      for (i_1=0;i_1<I1Dim_1;i_1++) {
        Radius_1     = WORLD(i_1,Off_1[1],Ps_1[1])-Wcenter_1;
        Radius2      = Radius_1*Radius_1+Radius_2*Radius_2;
        Radius       = sqrt(Radius2);

        if (fabs(Radius)>radius_eps) {
          CosAngle     = Radius_1/Radius;
          SinAngle     = Radius_2/Radius; 
          SinDiffAngle = SinAngle*CosSAngle-CosAngle*SinSAngle;
        } else {
          SinDiffAngle = 0.0;
        }

        if NODUMMY(*pI1Data, I1Dummy, I1DDummy) {
          if (Csym.V) factor = csym_fac*fabs(SinDiffAngle);
          else factor = 1.0;
          /* Multiplication with Radius^2*dr */
          if (S2.V) factor *= Radius2;

          *pI1Data    *= factor;  
          if (E1Data) *pE1Data *= factor*factor;
        }
        pI1Data++;
        pE1Data++;
      }
    }

    if (pcb->TestBit>3) {
      /* write test file */
      printf("Writing %s\n",OutputBinaryName);
      if ((OutputBinaryId=fopen(OutputBinaryName,"wb")) != NULL) {
        fwrite(I1Data,sizeof(float)*(I1Dim_1*I1Dim_2),1,OutputBinaryId);
        fclose(OutputBinaryId);
      }
    }

  }

  /* angular averaging */
  angle_sum ( I0Data, E0Data, I0Dim_1, I0Dim_2,
              Off_1[0], Ps_1[0], Off_2[0], Ps_2[0],
              I0Dummy, I0DDummy,
              I1Data, E1Data, I1Dim_1, I1Dim_2,
              Off_1[1], Ps_1[1], Off_2[1], Ps_2[1],
              I1Dummy, I1DDummy,
              Wcenter_1, Wcenter_2,
              Vsum.V, I1Ave, pcb->TestBit, pstatus );
  if (*pstatus!=Success) return;

  /* open xy-files */
  xyname[0] = xypath[0] = '\0';
  xfile[0]  = yfile[0]  = '\0';
  xstream   = ystream   = -1;
  xdata     = ydata     = (float *) NULL;
  if (Xy.V) {
    filename_parameter (xy_pattern,IO_len,Xyname.V,0);
    filename_pattern (xy_pattern,IO_len,xy_pattern,pcb->ib[0].FileNumber);
    filename_full ( xyfull, IO_len, xy_pattern );
    filename_path ( xypath, IO_len, xyfull );
    filename_name ( xyname, IO_len, xyfull );
    xysize = edf_dim_product ( ihb[1].Dim ); // identical to input array
    /* open x-file */
    sprintf(xfile,"%s%s%s",xypath,"x",xyname);
    xstream = edf_search_stream( xfile,  "new", &ErrorValue, pstatus );
    SetSaxsErrorExternal( ErrorValue, ReportSaxsImageError );
    if (*pstatus!=Success) return;
    if (xstream>=0) { *pstatus = FileAlreadyOpen; return; }
    xstream = edf_open_data_file ( xfile, "new", &ErrorValue, pstatus );
    SetSaxsErrorExternal( ErrorValue, ReportSaxsImageError );
    if (*pstatus!=Success) return;
    if (xysize>0) {
      if (!(xdata = (float *) malloc(sizeof(float)*xysize))) {
        *pstatus = NotEnoughMemoryAvailable; return; }
      pxdata = xdata;
      for (idata=0;idata<xysize;idata++) *(pxdata++)=0.0;
    }
    /* open y-file */
    sprintf(yfile,"%s%s%s",xypath,"y",xyname);
    ystream = edf_search_stream( yfile,  "new", &ErrorValue, pstatus );
    SetSaxsErrorExternal( ErrorValue, ReportSaxsImageError );
    if (*pstatus!=Success) {  FREE(xdata); return; }
    if (ystream>=0) { FREE(xdata); *pstatus = FileAlreadyOpen; return; }
    ystream = edf_open_data_file ( yfile, "new", &ErrorValue, pstatus );
    SetSaxsErrorExternal( ErrorValue, ReportSaxsImageError );
    if (*pstatus!=Success) {  FREE(xdata); return; }
    if (xysize>0) {
      if (!(ydata = (float *) malloc(sizeof(float)*xysize))) {
        FREE(xdata); *pstatus = NotEnoughMemoryAvailable; return; }
      pydata = ydata;
      for (idata=0;idata<xysize;idata++) *(pydata++)=0.0;
    }
  }

  /* loop over the input array  */
  if (Xy.V) {

    /* The displacements are given in array coordinates */
    /* calculate I0 and I1 array coordinates */
    ARRAYREF(I0Off_1,I0Ps_1);
    ARRAYREF(I0Off_2,I0Ps_2);
    ARRAYREF(I1Off_1,I1Ps_1);
    ARRAYREF(I1Off_2,I1Ps_2);

    for (i_2=0;i_2<I1Dim_2;i_2++) {
      W1_2 =  WORLD( i_2, Off_2[1], Ps_2[1] ); 
      Radius_2 = W1_2 - Wcenter_2;
      pI1Data = ABSPTR(I1Data,I1Dim_1,I1Dim_2,0,i_2);
      for (i_1=0;i_1<I1Dim_1;i_1++) {
        W1_1 =  WORLD( i_1, Off_1[1], Ps_1[1] );
        Radius_1 = W1_1 - Wcenter_1;

        Radius2      = Radius_1*Radius_1+Radius_2*Radius_2;
        Radius       = sqrt(Radius2);

        if (fabs(Radius)>radius_eps) {
          CosAngle     = Radius_1/Radius;
          SinAngle     = Radius_2/Radius;
        } else {
          CosAngle = 1.0;
          SinAngle = 0.0;
        }

        Angle = (float) atan2( (double) SinAngle, (double) CosAngle );
        if (Angle<0.0) Angle += 2*SAXS_PI;

        W0_1 = Radius;
        W0_2 = Angle;

        /* calculate I0 indices */
        f0_1 = INDEX(W0_1, Off_1[0], Ps_1[0]);
        f0_2 = INDEX(W0_2, Off_2[0], Ps_2[0]);

        /* calculate x- and y-shifts */
        pxdata = xdata + (pI1Data - I1Data);
        *pxdata = WORLD(f0_1, I0Off_1, I0Ps_1) - WORLD(i_1, I1Off_1, I1Ps_1);
        pydata = ydata + (pI1Data - I1Data);
        *pydata = WORLD(f0_2, I0Off_2, I0Ps_2) - WORLD(i_2, I1Off_2, I1Ps_2);

        pI1Data++;
      } /* for i_1 ... */
    } /* for i_2 ... */
  } /* if (Xy.V) */

  /* write and close xy-files */
  if (Xy.V) {
    /* xfile */
    xynum = ihb[0].ImNum;
    printf("Writing : %s%s, Image :  %d\n","x",xyname,xynum);
    edf_write_header_history ( xstream, xynum, 1l,
                               SaxsHistoryKey, &ErrorValue, pstatus );
    SetSaxsErrorExternal( ErrorValue, ReportSaxsImageError );
    if (*pstatus!=Success) { FREE(xdata); FREE(ydata); return; }
    /* The xy-displacement files cover the same region as the input image,
       i.e. dimensions, offsets and binning sizes are the same. */
    edf_write_header_float(xstream,xynum,1l,KOffset_1,ihb[1].Offset[1].V,
                          &ErrorValue, pstatus );
    SetSaxsErrorExternal( ErrorValue, ReportSaxsImageError );
    if (*pstatus!=Success) { FREE(xdata); FREE(ydata); return; }
    edf_write_header_float(xstream,xynum,1l,KOffset_2,ihb[1].Offset[2].V,
                           &ErrorValue, pstatus );
    SetSaxsErrorExternal( ErrorValue, ReportSaxsImageError );
    if (*pstatus!=Success) { FREE(xdata); FREE(ydata); return; }
    if ( (ihb[1].BinSiz[1].I)||(ihb[1].BinSiz[2].I)||\
         (ihb[1].BinSiz[1].V!=1.0)||(ihb[1].BinSiz[2].V!=1.0) ) {
      edf_write_header_float(xstream,xynum,1l,KBinSiz_1,ihb[1].BinSiz[1].V,
                             &ErrorValue, pstatus );
      SetSaxsErrorExternal( ErrorValue, ReportSaxsImageError );
      if (*pstatus!=Success) { FREE(xdata); FREE(ydata); return; }
      edf_write_header_float(xstream,xynum,1l,KBinSiz_2,ihb[1].BinSiz[2].V,
                             &ErrorValue, pstatus );
      SetSaxsErrorExternal( ErrorValue, ReportSaxsImageError );
      if (*pstatus!=Success) { FREE(xdata); FREE(ydata); return; }
    }
    xypix_1 = ihb[0].PixSiz[1].V; xypix_2 = ihb[0].PixSiz[2].V;
    xycen_1 = ihb[0].Center[1].V; xycen_2 = ihb[0].Center[2].V;
    xydis   = ihb[0].SampleDistance.V;
    xywvl   = ihb[0].WaveLength.V;
    xyrot_1 = ihb[0].DetRot[1].V; xyrot_2 = ihb[0].DetRot[2].V;
    xyrot_3 = ihb[0].DetRot[3].V;
    edf_write_header_float(xstream,xynum,1l,KPixSiz_1,xypix_1,
                           &ErrorValue, pstatus );
    SetSaxsErrorExternal( ErrorValue, ReportSaxsImageError );
    if (*pstatus!=Success) { FREE(xdata); FREE(ydata); return; }
    edf_write_header_float(xstream,xynum,1l,KPixSiz_2,xypix_2,
                           &ErrorValue, pstatus );
    SetSaxsErrorExternal( ErrorValue, ReportSaxsImageError );
    if (*pstatus!=Success) { FREE(xdata); FREE(ydata); return; }
    edf_write_header_float(xstream,xynum,1l,KCenter_1,xycen_1,
                           &ErrorValue, pstatus );
    SetSaxsErrorExternal( ErrorValue, ReportSaxsImageError );
    if (*pstatus!=Success) { FREE(xdata); FREE(ydata); return; }
    edf_write_header_float(xstream,xynum,1l,KCenter_2,xycen_2,
                           &ErrorValue, pstatus );
    SetSaxsErrorExternal( ErrorValue, ReportSaxsImageError );
    if (*pstatus!=Success) { FREE(xdata); FREE(ydata); return; }
    edf_write_header_float(xstream,xynum,1l,KSampleDistance,xydis,
                           &ErrorValue, pstatus );
    SetSaxsErrorExternal( ErrorValue, ReportSaxsImageError );
    if (*pstatus!=Success) { FREE(xdata); FREE(ydata); return; }
    edf_write_header_float(xstream,xynum,1l,KWaveLength,xywvl,
                           &ErrorValue, pstatus );
    SetSaxsErrorExternal( ErrorValue, ReportSaxsImageError );
    if (*pstatus!=Success) { FREE(xdata); FREE(ydata); return; }

    edf_write_header_unit(xstream,xynum,1l,KDetRot_1,xyrot_1,"deg",
                          &ErrorValue, pstatus );
    SetSaxsErrorExternal( ErrorValue, ReportSaxsImageError );
    if (*pstatus!=Success) { FREE(xdata); FREE(ydata); return; }
    edf_write_header_unit(xstream,xynum,1l,KDetRot_2,xyrot_2,"deg",
                          &ErrorValue, pstatus );
    SetSaxsErrorExternal( ErrorValue, ReportSaxsImageError );
    if (*pstatus!=Success) { FREE(xdata); FREE(ydata); return; }
    edf_write_header_unit(xstream,xynum,1l,KDetRot_3,xyrot_3,"deg",
                          &ErrorValue, pstatus );
    SetSaxsErrorExternal( ErrorValue, ReportSaxsImageError );
    if (*pstatus!=Success) { FREE(xdata); FREE(ydata); return; }

   /* Write parameters of transformed image to x-displacement file */
    edf_write_header_float(xstream,xynum,1l,
                strlib_concat(key,IO_len,KDisplaced,"Dim_1"),ihb[0].Dim[1],
                &ErrorValue, pstatus );
    SetSaxsErrorExternal( ErrorValue, ReportSaxsImageError );
    if (*pstatus!=Success) { FREE(xdata); FREE(ydata); return; }
    edf_write_header_float(xstream,xynum,1l,
                strlib_concat(key,IO_len,KDisplaced,"Dim_2"),ihb[0].Dim[2],
                &ErrorValue, pstatus );
    SetSaxsErrorExternal( ErrorValue, ReportSaxsImageError );
    if (*pstatus!=Success) { FREE(xdata); FREE(ydata); return; }
    edf_write_header_float(xstream,xynum,1l,
                strlib_concat(key,IO_len,KDisplaced,KOffset_1),ihb[0].Offset[1].V,
                &ErrorValue, pstatus );
    SetSaxsErrorExternal( ErrorValue, ReportSaxsImageError );
    if (*pstatus!=Success) { FREE(xdata); FREE(ydata); return; }
    edf_write_header_float(xstream,xynum,1l,
                strlib_concat(key,IO_len,KDisplaced,KOffset_2),ihb[0].Offset[2].V,
                &ErrorValue, pstatus );
    SetSaxsErrorExternal( ErrorValue, ReportSaxsImageError );
    if (*pstatus!=Success) { FREE(xdata); FREE(ydata); return; }
    edf_write_header_float(xstream,xynum,1l,
                 strlib_concat(key,IO_len,KDisplaced,KBinSiz_1),ihb[0].BinSiz[1].V,
                 &ErrorValue, pstatus );
    SetSaxsErrorExternal( ErrorValue, ReportSaxsImageError );
    if (*pstatus!=Success) { FREE(xdata); FREE(ydata); return; }
    edf_write_header_float(xstream,xynum,1l,
                 strlib_concat(key,IO_len,KDisplaced,KBinSiz_2),ihb[0].BinSiz[2].V,
                 &ErrorValue, pstatus );
    SetSaxsErrorExternal( ErrorValue, ReportSaxsImageError );
    if (*pstatus!=Success) { FREE(xdata); FREE(ydata); return; }
    edf_write_header_float(xstream,xynum,1l,
                 strlib_concat(key,IO_len,KDisplaced,KPixSiz_1),ihb[0].PixSiz[1].V,
                 &ErrorValue, pstatus );
    SetSaxsErrorExternal( ErrorValue, ReportSaxsImageError );
    if (*pstatus!=Success) { FREE(xdata); FREE(ydata); return; }
    edf_write_header_float(xstream,xynum,1l,
                 strlib_concat(key,IO_len,KDisplaced,KPixSiz_2),ihb[0].PixSiz[2].V,
                 &ErrorValue, pstatus );
    SetSaxsErrorExternal( ErrorValue, ReportSaxsImageError );
    if (*pstatus!=Success) { FREE(xdata); FREE(ydata); return; }
    edf_write_header_float(xstream,xynum,1l,
                 strlib_concat(key,IO_len,KDisplaced,KCenter_1),ihb[0].Center[1].V,
                 &ErrorValue, pstatus );
    SetSaxsErrorExternal( ErrorValue, ReportSaxsImageError );
    if (*pstatus!=Success) { FREE(xdata); FREE(ydata); return; }
    edf_write_header_float(xstream,xynum,1l,
                 strlib_concat(key,IO_len,KDisplaced,KCenter_2),ihb[0].Center[2].V,
                 &ErrorValue, pstatus );
    SetSaxsErrorExternal( ErrorValue, ReportSaxsImageError );
    if (*pstatus!=Success) { FREE(xdata); FREE(ydata); return; }

   if ( (pcb->RSys.V == IO_Tangens) || (pcb->RSys.V == IO_Saxs) ) {
      edf_write_header_float(xstream,xynum,1l,
        strlib_concat(key,IO_len,KDisplaced,KSampleDistance),ihb[0].SampleDistance.V,
        &ErrorValue, pstatus );
      SetSaxsErrorExternal( ErrorValue, ReportSaxsImageError );
      if (*pstatus!=Success) { FREE(xdata); FREE(ydata); return; }
   }

   if ( pcb->RSys.V == IO_Saxs ) {
      edf_write_header_float(xstream,xynum,1l,
        strlib_concat(key,IO_len,KDisplaced,KWaveLength),ihb[0].WaveLength.V,
        &ErrorValue, pstatus );
      SetSaxsErrorExternal( ErrorValue, ReportSaxsImageError );
      if (*pstatus!=Success) { FREE(xdata); FREE(ydata); return; }
    }

    edf_write_data( xstream,xynum,1l,ihb[1].Dim,xdata,MFloat,
                    &ErrorValue, pstatus );
    SetSaxsErrorExternal( ErrorValue, ReportSaxsImageError );
    if (*pstatus!=Success) { FREE(xdata); FREE(ydata); return; }
    FREE(xdata);
    edf_close_data_file ( xstream, &ErrorValue, pstatus );
    SetSaxsErrorExternal( ErrorValue, ReportSaxsImageError );
    if (*pstatus!=Success) { FREE(ydata); return; }
   /* yfile */
    printf("Writing : %s%s, Image :  %d\n","y",xyname,xynum);
    edf_write_header_history ( ystream, xynum, 1l,
                               SaxsHistoryKey, &ErrorValue, pstatus );
    SetSaxsErrorExternal( ErrorValue, ReportSaxsImageError );
    if (*pstatus!=Success) { FREE(ydata); return; }
    edf_write_header_float(ystream,xynum,1l,KOffset_1,ihb[1].Offset[1].V,
                           &ErrorValue, pstatus );
    SetSaxsErrorExternal( ErrorValue, ReportSaxsImageError );
    if (*pstatus!=Success) { FREE(ydata); return; }
    edf_write_header_float(ystream,xynum,1l,KOffset_2,ihb[1].Offset[2].V,
                           &ErrorValue, pstatus );
    SetSaxsErrorExternal( ErrorValue, ReportSaxsImageError );
    if (*pstatus!=Success) { FREE(ydata); return; }
    if ( (ihb[1].BinSiz[1].I)||(ihb[1].BinSiz[2].I)||\
         (ihb[1].BinSiz[1].V!=1.0)||(ihb[1].BinSiz[2].V!=1.0) ) {
      edf_write_header_float(ystream,xynum,1l,KBinSiz_1,ihb[1].BinSiz[1].V,
                             &ErrorValue, pstatus );
    SetSaxsErrorExternal( ErrorValue, ReportSaxsImageError );
      if (*pstatus!=Success) { FREE(ydata); return; }
      edf_write_header_float(ystream,xynum,1l,KBinSiz_2,ihb[1].BinSiz[2].V,
                             &ErrorValue, pstatus );
    SetSaxsErrorExternal( ErrorValue, ReportSaxsImageError );
      if (*pstatus!=Success) { FREE(ydata); return; }
    }
    edf_write_header_float(ystream,xynum,1l,KPixSiz_1,xypix_1,
                           &ErrorValue, pstatus );
    SetSaxsErrorExternal( ErrorValue, ReportSaxsImageError );
    if (*pstatus!=Success) { FREE(ydata); return; }
    edf_write_header_float(ystream,xynum,1l,KPixSiz_2,xypix_2,
                           &ErrorValue, pstatus );
    SetSaxsErrorExternal( ErrorValue, ReportSaxsImageError );
    if (*pstatus!=Success) { FREE(ydata); return; }
    edf_write_header_float(ystream,xynum,1l,KCenter_1,xycen_1,
                           &ErrorValue, pstatus );
    SetSaxsErrorExternal( ErrorValue, ReportSaxsImageError );
    if (*pstatus!=Success) { FREE(ydata); return; }
    edf_write_header_float(ystream,xynum,1l,KCenter_2,xycen_2,
                           &ErrorValue, pstatus );
    SetSaxsErrorExternal( ErrorValue, ReportSaxsImageError );
    if (*pstatus!=Success) { FREE(ydata); return; }
    edf_write_header_float(ystream,xynum,1l,KSampleDistance,xydis,
                           &ErrorValue, pstatus );
    SetSaxsErrorExternal( ErrorValue, ReportSaxsImageError );
    if (*pstatus!=Success) { FREE(ydata); return; }
    edf_write_header_float(ystream,xynum,1l,KWaveLength,xywvl,
                           &ErrorValue, pstatus );
    SetSaxsErrorExternal( ErrorValue, ReportSaxsImageError );
    if (*pstatus!=Success) { FREE(ydata); return; }

    edf_write_header_unit(ystream,xynum,1l,KDetRot_1,xyrot_1,"deg",
                          &ErrorValue, pstatus );
    SetSaxsErrorExternal( ErrorValue, ReportSaxsImageError );
    if (*pstatus!=Success) { FREE(ydata); return; }
    edf_write_header_unit(ystream,xynum,1l,KDetRot_2,xyrot_2,"deg",
                          &ErrorValue, pstatus );
    SetSaxsErrorExternal( ErrorValue, ReportSaxsImageError );
    if (*pstatus!=Success) { FREE(ydata); return; }
    edf_write_header_unit(ystream,xynum,1l,KDetRot_3,xyrot_3,"deg",
                          &ErrorValue, pstatus );
    SetSaxsErrorExternal( ErrorValue, ReportSaxsImageError );
    if (*pstatus!=Success) { FREE(ydata); return; }

   /* Write parameters of transformed image to y-displacement file */
    edf_write_header_float(ystream,xynum,1l,
                strlib_concat(key,IO_len,KDisplaced,"Dim_1"),ihb[0].Dim[1],
                &ErrorValue, pstatus );
    SetSaxsErrorExternal( ErrorValue, ReportSaxsImageError );
    if (*pstatus!=Success) { FREE(ydata); return; }
    edf_write_header_float(ystream,xynum,1l,
                strlib_concat(key,IO_len,KDisplaced,"Dim_2"),ihb[0].Dim[2],
                &ErrorValue, pstatus );
    SetSaxsErrorExternal( ErrorValue, ReportSaxsImageError );
    if (*pstatus!=Success) { FREE(ydata); return; }
    edf_write_header_float(ystream,xynum,1l,
                strlib_concat(key,IO_len,KDisplaced,KOffset_1),ihb[0].Offset[1].V,
                &ErrorValue, pstatus );
    SetSaxsErrorExternal( ErrorValue, ReportSaxsImageError );
    if (*pstatus!=Success) { FREE(ydata); return; }
    edf_write_header_float(ystream,xynum,1l,
                strlib_concat(key,IO_len,KDisplaced,KOffset_2),ihb[0].Offset[2].V,
                &ErrorValue, pstatus );
    SetSaxsErrorExternal( ErrorValue, ReportSaxsImageError );
    if (*pstatus!=Success) { FREE(ydata); return; }
    edf_write_header_float(ystream,xynum,1l,
                strlib_concat(key,IO_len,KDisplaced,KBinSiz_1),ihb[0].BinSiz[1].V,
                &ErrorValue, pstatus );
    SetSaxsErrorExternal( ErrorValue, ReportSaxsImageError );
    if (*pstatus!=Success) { FREE(ydata); return; }
    edf_write_header_float(ystream,xynum,1l,
                strlib_concat(key,IO_len,KDisplaced,KBinSiz_2),ihb[0].BinSiz[2].V,
                &ErrorValue, pstatus );
    SetSaxsErrorExternal( ErrorValue, ReportSaxsImageError );
    if (*pstatus!=Success) { FREE(ydata); return; }
    edf_write_header_float(ystream,xynum,1l,
                 strlib_concat(key,IO_len,KDisplaced,KPixSiz_1),ihb[0].PixSiz[1].V,
                 &ErrorValue, pstatus );
    SetSaxsErrorExternal( ErrorValue, ReportSaxsImageError );
    if (*pstatus!=Success) { FREE(ydata); return; }
    edf_write_header_float(ystream,xynum,1l,
                 strlib_concat(key,IO_len,KDisplaced,KPixSiz_2),ihb[0].PixSiz[2].V,
                 &ErrorValue, pstatus );
    SetSaxsErrorExternal( ErrorValue, ReportSaxsImageError );
    if (*pstatus!=Success) { FREE(ydata); return; }
    edf_write_header_float(ystream,xynum,1l,
                 strlib_concat(key,IO_len,KDisplaced,KCenter_1),ihb[0].Center[1].V,
                 &ErrorValue, pstatus );
    SetSaxsErrorExternal( ErrorValue, ReportSaxsImageError );
    if (*pstatus!=Success) { FREE(ydata); return; }
    edf_write_header_float(ystream,xynum,1l,
                 strlib_concat(key,IO_len,KDisplaced,KCenter_2),ihb[0].Center[2].V,
                 &ErrorValue, pstatus );
    SetSaxsErrorExternal( ErrorValue, ReportSaxsImageError );
    if (*pstatus!=Success) { FREE(ydata); return; }

    if ( (pcb->RSys.V == IO_Tangens) || (pcb->RSys.V == IO_Saxs) ) {
      edf_write_header_float(ystream,xynum,1l,
        strlib_concat(key,IO_len,KDisplaced,KSampleDistance),ihb[0].SampleDistance.V,
        &ErrorValue, pstatus );
      SetSaxsErrorExternal( ErrorValue, ReportSaxsImageError );
      if (*pstatus!=Success) { FREE(ydata); return; }
    }

    if ( pcb->RSys.V == IO_Saxs ) {
      edf_write_header_float(ystream,xynum,1l,
        strlib_concat(key,IO_len,KDisplaced,KWaveLength),ihb[0].WaveLength.V,
        &ErrorValue, pstatus );
      SetSaxsErrorExternal( ErrorValue, ReportSaxsImageError );
      if (*pstatus!=Success) { FREE(ydata); return; }
    }

    edf_write_data( ystream, xynum, 1l, ihb[1].Dim, ydata, MFloat,
                    &ErrorValue, pstatus );
    SetSaxsErrorExternal( ErrorValue, ReportSaxsImageError );
    if (*pstatus!=Success) { FREE(ydata); return; }
    edf_close_data_file ( ystream, &ErrorValue, pstatus );
    SetSaxsErrorExternal( ErrorValue, ReportSaxsImageError );
    if (*pstatus!=Success) { FREE(ydata); return; }
    FREE(ydata);
  }

} /* saxs_angle*/

/*---------------------------------------------------------------------------
ArgvAngleFilenames

Ask for all file names in the following way:
  input file 1 [input.edf] :
  output file [output.edf] :

  first image [<minimum image number>] :
  last image [<maximum image number>] :
  increment [1] :

  output dummy [<dummy in first image>] :

The header block of the first image is read into ihb[1].
---------------------------------------------------------------------------*/
void ArgvAngleFilenames ( CmdBlk * pcb, ImgBlk ib[], ImgHeadBlk ihb[],
                          int * pstatus)
{ int blkno;
  long int image_1st, image_lst;
  char tmp_line[IO_len];
  float Bin_1, Bin_2;
  long num[BlockNum];
  
  /* ---  File names and open modes --- */

    blkno = 1;
    /*--- Open mode ib[blkno].OpenMode of input files */
    if (!ib[blkno].OpenMode.I) {
      ib[blkno].OpenMode.V = IO_Old | IO_FileProtect;
      ib[blkno].OpenMode.I = TRUE;
    }

    /*--- File name ib[blkno].Name of input files */
    (void) sprintf(tmp_line,"Input sequence %d",blkno);
    if (blkno==1)
      argv_filename( pcb,tmp_line, ib, 1, DefaultInput, pstatus);
    else argv_filename( pcb,tmp_line,ib,blkno,ib[blkno-1].Name.V,pstatus);
    /* Exit in all cases after an abort from the first argument */
    if (*pstatus!=Success) return;

    OpenImageFile(pcb,ib,blkno,ib[blkno].Name.V,ib[1].FFirst.V,
                  IO_Old|IO_FileProtect,pstatus);
    if (*pstatus!=Success) return;

    /* Search for minimum and maximum image number */
    (void) SearchMinMaxImage (pcb,ib,blkno,&image_1st,&image_lst,pstatus);
    if (*pstatus!=Success) return;
    if (!ib[blkno].First.I) ib[blkno].First.V = image_1st;
    if (!ib[blkno].Last.I)  ib[blkno].Last.V  = image_lst;
    num[blkno] = ib[blkno].First.V;

    /*--- OpenMode of output file */
    if (!ib[0].OpenMode.I) {
      ib[0].OpenMode.V = IO_ImageProtect;
      ib[0].OpenMode.I = TRUE;
    }
 
    /*--- File name ib[0].Name of output */
    argv_filename( pcb,"Output sequence",ib,0,DefaultOutput,pstatus);
    if (*pstatus!=Success) return;

  /* ---  Image numbers --- */

    /*--- Argument : number of first image */
    argv_long(pcb,"First image of input sequence 1",&ib[1].First,ib[1].First.V,
               pstatus);
    if (*pstatus!=Success) return;

    /*--- Argument : number of last image */
    if (pcb->argc==*pcb->parg_no+1) image_lst = ib[1].First.V;
      else image_lst = ib[1].Last.V;
    argv_long( pcb,"Last image of input sequence 1", &ib[1].Last,image_lst,
               pstatus);
    if (*pstatus!=Success) return;

    /*--- Argument : increment (default = 1) */
    argv_long( pcb,"Increment of input sequence 1", &ib[1].Inc, 1 , pstatus);
    if (*pstatus!=Success) return;

    num[1] = ib[1].First.V;
    (void) ReadImageHeader( pcb, ib, 1, num, ihb, pstatus);
    if (*pstatus!=Success) return;
    if (pcb->TestBit) PrintImageHeaderBlock(&ihb[1]);

    /*--- Argument : image coordinate 1 of center */
    Bin_1 = MAX2( 1,ihb[1].Bin[1].V);
    sprintf(tmp_line,"Center 1 (image coordinate), e.g. %g",
      Bin_1*ihb[1].Center[1].V);
    argv_flexp(pcb,tmp_line,&ib[1].Center[1],"",pstatus);
    if (*pstatus!=Success) return;

    /*--- Argument : image coordinate 2 of center */
    Bin_2 = MAX2( 1,ihb[1].Bin[2].V);
    sprintf(tmp_line,"Center 2 (image coordinate), e.g. %g",
      Bin_2*ihb[1].Center[2].V);
    argv_flexp(pcb,tmp_line,&ib[1].Center[2],"",pstatus);
    if (*pstatus!=Success) return;

    // re-read image header
    num[1] = ib[1].First.V;
    (void) ReadImageHeader( pcb, ib, 1, num, ihb, pstatus);
    if (*pstatus!=Success) return;
    if (pcb->TestBit) PrintImageHeaderBlock(&ihb[1]);

    /* Defaults for output file */
    sprintf(tmp_line,"%g",ihb[1].Dummy.V);
    if (!ib[0].Dummy.I) CopyLine(tmp_line,ib[0].Dummy.V,IO_len,0);

    /*--- Argument : output dummy */
    sprintf(tmp_line,"Output dummy, e.g. %s",ib[0].Dummy.V);
    argv_flexp( pcb, tmp_line, &ib[0].Dummy, "", pstatus);
    if (*pstatus!=Success) return;

    CloseImageFile( pcb, ib, 1, pstatus) ;
    if (*pstatus!=Success) return;

} /* ArgvAngleFilenames */

/*---------------------------------------------------------------------------
user_io
Do all the keyboard io and return cb, and ib
---------------------------------------------------------------------------*/
void user_io(CmdBlk * pcb, ImgBlk * ib, int * pstatus)
{
  const char * InputRefSys  = "Input reference system is";

  char  progfn[IO_len];
  ImgHeadBlk ihb[BlockNum];

  char tmp_line[IO_len];

  float ROff_1, RPs_1, UOff_1, UPs_1;
  float ROff_2, RPs_2, UOff_2, UPs_2;

  IO_flexp *pR0, *pDr, *pA0, *pDa;

 /* Determine program name without directory */
   (void) RightFR((char *) pcb->argv[0],DirSeparator,progfn,InputLineLength);

  /* show usage if no arguments are given */
  if (pcb->argc<=1) printf("Usage: %s %s\n",progfn,Usage);

  /*--- Write name of program ---*/
  printf("\n");
  printf("   %s %s\n",progfn,Version);
  printf("\n");

  ArgvAngleFilenames ( pcb, ib, ihb, pstatus);
  if (*pstatus!=Success) return;
  GetReference(pcb->RSys.V,1,ihb,&ROff_1,&ROff_2,&RPs_1,&RPs_2,pstatus );
  if (*pstatus!=Success) return;
  GetReference(pcb->USys.V,1,ihb,&UOff_1,&UOff_2,&UPs_1,&UPs_2,pstatus );
  if (*pstatus!=Success) return;

  pR0 = (IO_flexp*) option_parameter_search( SR0, pcb, pstatus );
  if (*pstatus!=Success) return;
  pDr = (IO_flexp*) option_parameter_search( SDr, pcb, pstatus );
  if (*pstatus!=Success) return;
  pA0 = (IO_flexp*) option_parameter_search( SA0, pcb, pstatus );
  if (*pstatus!=Success) return;
  pDa = (IO_flexp*) option_parameter_search( SDa, pcb, pstatus );
  if (*pstatus!=Success) return;

  /*--- Argument : start radius */
  printf("%s %s\n",InputRefSys,reftostr(pcb->RSys.V));
  argv_flexp(pcb,"Start radius (RSys) ", pR0,pR0->V,pstatus);
  if (*pstatus!=Success) return;

  /*--- Argument : radial interval */
  printf("%s %s\n",InputRefSys,reftostr(pcb->RSys.V));
  if (!(pDr->I) ) {
    sprintf(pDr->V,"%g", DefaultRadialInterval(pcb,ihb,1,pstatus));
    if (*pstatus!=Success) return;
  }
  sprintf(tmp_line,"Radial interval (%s), e.g. %s",
    reftostr(pcb->RSys.V), pDr->V);
  argv_flexp(pcb,tmp_line,pDr,"",pstatus);
  if (*pstatus!=Success) return;

  /*--- Argument : output dimension 1 */
  SetOutputDimension( pcb, ihb, 1, pstatus ); /* no check of pstatus */
  sprintf(tmp_line,"Number of radial intervals, e.g. %ld",ihb[0].Dim[1]);
  argv_lvexp(pcb,tmp_line, &ib[0].Dim[1],"",pstatus);
  if (*pstatus!=Success) return;

  /*--- Argument : minimum angle (default 0.0) */
  argv_flexp(pcb,"Start angle", pA0, pA0->V, pstatus);
  if (*pstatus!=Success) return;

  /*--- Argument : angular interval */
  argv_flexp(pcb,"Angular interval", pDa,pDa->V,pstatus);
  if (*pstatus!=Success) return;

  /*--- Argument : output dimension 2 */
  sprintf(tmp_line,"%ld",ihb[0].Dim[2]);
  argv_lvexp(pcb,"Number of angular intervals", 
              &ib[0].Dim[2],tmp_line, pstatus);
  if (*pstatus!=Success) return;

  printf("\n");
  if (ib[1].Name.I)    
     printf("i/p file                : %s\n",ib[1].Name.V);
  if (ib[0].Name.I)    
     printf("o/p file                : %s\n",ib[0].Name.V);
  if (ib[1].First.I)   
     printf("first image             : %d\n",ib[1].First.V);
  if (ib[1].Last.I)    
     printf("last image              : %d\n",ib[1].Last.V);
  if (ib[1].Inc.I)     
     printf("increment               : %d\n",ib[1].Inc.V);
  if (ib[0].Dummy.I)   
     printf("output dummy            : %s\n",ib[0].Dummy.V);
  if (ib[1].Center[1].I) 
     printf("center 1 (image coord.) : %s\n",ib[1].Center[1].V);
  if (ib[1].Center[2].I)
     printf("center 2 (image coord.) : %s\n",ib[1].Center[2].V);

  if (pR0->I) 
     printf("minimum radius          : %s\n",pR0->V);
  if (pDr->I) 
     printf("radial interval         : %s\n",pDr->V);
  if (ib[0].Dim[1].I)  
     printf("num. radial intervals   : %s\n",ib[0].Dim[1].V);
  if (pA0->I) 
     printf("minimum angle           : %s\n",pA0->V);
  if (pDa->I) 
     printf("angular interval        : %s\n",pDa->V);
  if (ib[0].Dim[2].I)  
     printf("num. angular intervals  : %s\n",ib[0].Dim[2].V);
  printf("\n");

  if (pcb->TestBit) {
    PrintBlocks ( pcb, ib );
    printf("\n"); }

  return;
}

/*---------------------------------------------------------------------------
main
---------------------------------------------------------------------------*/

#if MAKE_FUNCTION
# define MAIN main_saxs_angle
#else
# define MAIN main
#endif

int MAIN (int argc, char *argv[])
{
  CmdBlk cb;                /* command block  */
  ImgBlk ib[BlockNum];      /* image blocks */

  int status;
  int arg_no = 0;

  IO_long Xy;
  IO_line Xyname;

 /* Init options, control block and image blocks */
  InitOptions( Usage, Version, NULL, TRUE, &cb, ib, BlockNum );

  option_define ( IO_tpflexp, SR0,  "0.0", &cb );
  option_define ( IO_tpflexp, SDr,  "1.0", &cb );
  option_define ( IO_tpflexp, SA0,  "0.0_deg", &cb );
  option_define ( IO_tpflexp, SDa,  "1.0_deg", &cb );
  option_define ( IO_tpflexp, SAsym,"0.0_deg", &cb );

  option_define ( IO_tpflag, SS2,   "FALSE", &cb );
  option_define ( IO_tpflag, SCsym, "FALSE", &cb );
  option_define ( IO_tpflag, SVsum, "FALSE", &cb );

  option_define ( IO_tpline,  SXyname, "file.edf", &cb );
  option_define ( IO_tpflag,  SXy,     "FALSE",    &cb );

 /* Read options from argument list */
  ReadOptions( argv, &arg_no, &cb, ib, &status);

 /* Set xy-flag if xy-file name is given */
  if (status==Success)
    Xy = option_flag(SXy,&cb,&status );
  if (status==Success)
    if (!Xy.I) Xyname = option_line(SXyname,&cb,NULL,NULL,0,&status);
  if (status==Success)
    if (!Xy.I) option_flag_update(Xyname.I?TRUE:FALSE,Xy.I,SXy,&cb,&status);

  /* USER KEYBOARD I/O */
  if (status==Success) {
    argv_start ( &cb, 1 );
    user_io( &cb, ib, &status);
    argv_end( &cb ); /* must be called after user_io */
  }

  /* SEQUENCE CALCULATION */
  if (status==Success)
    IMAGELOOP( &cb, ib, saxs_angle, saxs_angle_init, NULL, TRUE, &status );

  return( ReportSaxsStatus( status, 0 ) );

} /* MAIN */

