/***************************************************************************/
/* Written 1994++ by Peter Boesecke                                        */
/* Copyright (C) 2011 European Synchrotron Radiation Facility              */
/*                       Grenoble, France                                  */
/*                                                                         */
/*    Principal authors: Peter Boesecke  (boesecke@esrf.eu)                */
/*                                                                         */
/*    This program is free software: you can redistribute it and/or modify */
/*    it under the terms of the GNU General Public License as published by */
/*    the Free Software Foundation, either version 3 of the License, or    */
/*    (at your option) any later version.                                  */
/*                                                                         */
/*    This program is distributed in the hope that it will be useful,      */
/*    but WITHOUT ANY WARRANTY; without even the implied warranty of       */
/*    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the        */
/*    GNU General Public License for more details.                         */
/*                                                                         */
/*    You should have received a copy of the GNU General Public License    */
/*    along with this program.  If not, see <http://www.gnu.org/licenses/>.*/
/***************************************************************************/
/****************************************************************************
*                                                                           *
* History                                                                   *
* 2010-08-19 PB running version                                             *
* 2011-07-21 PB minor adaptations, strlib functions used                    *
* 2015-12-08 PB search_list, remove_list, search_element:                   *
*               previous (unused-but-set-variable) commented                *
*               print_lists:                                                *
*               Reducing warnings in print statements when switching        *
*               between 32-bit and 64-bit system by transforming            *
*               size_t variables with SIZE_T to unsigned long               *
*               before printing as %lu. %lz is not known by all             *
*               compilers.                                                  *
*               tolower and isgraph replaced with the macros                *
*               TOLOWER and ISGRAPH.                                        *
****************************************************************************/

/****************************************************************************
*  Include                                                                  *
****************************************************************************/
# include "roca_list.h"

/***************************************************************************
* Private Macros                                                           *
***************************************************************************/

# define ISGRAPH(c) isgraph((int)(c))
# define ISSPACE(c) isspace((int)(c))
# define ISDIGIT(c) isdigit((int)(c))
# define TOLOWER(s) (char)tolower((int)(s))
# define TOUPPER(s) (char)toupper((int)(s))

# define STRLEN(s) (s == (char *) NULL ? 0 : strlen ( (char *) s))
# define STRCMP(s1,s2) strcmp ( (char *) s1, (char *) s2 )
# define STRCPY(s1,s2) strcpy ( (char *) s1, (char *) s2 )

# define FREE(s) if (s) { free(s); s=NULL; }

# define MAXLEN 1024

/***************************************************************************
* Portables Print Format for size_t variable (to be printed with %lu)      *
***************************************************************************/
# define SIZE_T (unsigned long)

/***************************************************************************
* Static Variables                                                         *
***************************************************************************/

static int ListDebug=0;
static LList *ListRoot=NULL;
static int ListInit=0;

/******************************************************************************
* Routines                                                                    *
******************************************************************************/

/*---------------------------------------------------------------------------
NAME 
    
   list_debug_set
  
SYNOPSIS

   int list_debug_set ( int debug );

DESCRIPTION

   Sets the debug mode of the module
   0x1  show routine names

RETURN VALUE

   0 in case of success

---------------------------------------------------------------------------*/
int list_debug_set ( int debug ) {
  if (ListDebug) printf("list_debug_set:\n");
  ListDebug = debug;
  return(0);
} // list_debug_set

/*---------------------------------------------------------------------------
NAME

   newstr --- allocate memory and copy a character string into it

SYNOPSIS

   char * newstr( const char * string );

DESCRIPTION
  Allocates strlen(´string´)+1 bytes of memory and copies ´string´ into it.
  In case of success the pointer to the allocated memory is returned. The
  null pointer is returned in case of an error.
  If ´string´ is the NULL pointer the NULL pointer is returned.
  
RETURN VALUE
  Returns the pointer to the allocated string or (char *) NULL in case
  of an error.
---------------------------------------------------------------------------*/
char * newstr( const char * string )
{ char * value;
  if (ListDebug) printf("newstr:\n");

  if (!string) return( (char *) NULL );
  if (!(value = (char *) malloc(STRLEN(string)+1))) return((char *) NULL);
  (void) STRCPY(value,string);

  return( value );

} /* newstr */

/*---------------------------------------------------------------------------
NAME

  numcmp --- number sensitive sort of two strings (<0, 0, >0)

SYNOPSIS

  int numcmp( const char * key1, const char * key2 );

DESCRIPTION
  All characters are converted to uppercase, leading '+' or '-' signs are
  taken into account and then removed. Before comparison, the length of the 
  shorter string is padded with leading '0's to the same length as the 
  other string.

HISTORY
Peter Boesecke
---------------------------------------------------------------------------*/
int numcmp( const char * key1, const char * key2 )
{ register int i,j;
  char buf1[MAXLEN+1], buf2[MAXLEN+1];
  char *pc1, *pc1o, *pc2, *pc2o;
  char stop1='\0', stop2='\0';
  int l1,l2;
  int vz1=1, vz2=1;
  int comparison;

   pc1 = &buf1[0]; pc2 = &buf2[0];

      if (key1) for (i=0;(i<MAXLEN)&&(*key1)&&(*key1!=stop1);i++) {
        if (ISGRAPH(*key1)) *pc1++=TOUPPER(*key1++); else key1++;
      }
      *pc1='\0';

      if (key2) for (i=0;(i<MAXLEN)&&(*key2)&&(*key2!=stop2);i++) {
        if (ISGRAPH(*key2)) *pc2++=TOUPPER(*key2++); else key2++;
      }
      *pc2='\0';

      /* remove a leading '+' or a leading '-' and
         remove afterwards multiple leading '0's */
      pc1o = pc1 = &buf1[0];
      if (*pc1=='+') pc1++; else if (*pc1=='-') {vz1=-1;pc1++;}

      pc2o = pc2 = &buf2[0];
      if (*pc2=='+') pc2++; else if (*pc2=='-') {vz2=-1;pc2++;}

      if (vz1<vz2) return(-1); else if (vz2<vz1) return(1);

      while (*pc1 == '0') pc1++; if (*pc1 == '\0') *pc1o++='0';
      while (*pc1) *pc1o++ = *pc1++; *pc1o='\0';

      while (*pc2 == '0') pc2++; if (*pc2 == '\0') *pc2o++='0';
      while (*pc2) *pc2o++ = *pc2++; *pc2o='\0';

      /* pad strings with leading '0's to same lengths */
      l1=STRLEN(buf1); l2=STRLEN(buf2);
      if (l1!=l2) {
        if ( l1<l2) {
          for (i=l2,j=l1;j>=0;i--,j--) buf1[i]=buf1[j];
          for (;i>=0;i--)              buf1[i]='0';
        } else {
          for (i=l1,j=l2;j>=0;i--,j--) buf2[i]=buf2[j];
          for (i=i;i>=0;i--)           buf2[i]='0';
        }
      }


  comparison = vz1*STRCMP(buf1,buf2);

  return( comparison );

} /* numcmp */

/*---------------------------------------------------------------------------
NAME 
    
   list_init
  
SYNOPSIS

   int list_init( void  )

DESCRIPTION

   Initialization of the module

RETURN VALUE

   0 in case of success

---------------------------------------------------------------------------*/
int list_init( void  )
{ 
  if (ListDebug) printf("list_init:\n");

  ListRoot=NULL;
  ListInit=1;
  return( 0 );
} // list_init

/*---------------------------------------------------------------------------
NAME 
    
   new_list --- Creates an empty list 
  
SYNOPSIS

   int new_list( const char *Key, LList **plist );

DESCRIPTION

   Creates and inserts an empty list with the name Key and returns the pointer 
   to the new list in *plist. An existing list with the same name is deleted.

ARGUMENTS

   *Key    : name of the list
   **plist : *plist is the returned pointer to the new list

RETURN VALUE

   0 in case of success

---------------------------------------------------------------------------*/
int new_list( const char *Key, LList **plist )
{ LList *list;

  if (ListDebug) printf("new_list: >>%s<<\n",Key?Key:"(none)");

  if (!ListInit) list_init( );

  if (plist) *plist=NULL;

  if (search_list( Key, &list )) return(-1);
  if (delete_list( list,  NULL )) return(-1);
  if (insert_list( Key, &list )) return(-1);

  if (plist) *plist=list;

  return(0);

} // new_list

/*---------------------------------------------------------------------------
NAME 
    
  insert_list --- Creates a list with the name Key
  
SYNOPSIS

   int insert_list( const char *Key, LList **plist )

DESCRIPTION
  
   Creates and inserts a list with the name Key and returns the pointer to the 
   new list in *plist. If it already exists only the pointer is returned.
  
ARGUMENTS

   *Key    : name of the list
   **plist : *plist is the returned pointer to the inserted list
    
RETURN VALUE
  
   0 in case of success
  
---------------------------------------------------------------------------*/
int insert_list( const char *Key, LList **plist )
{ LList *list, *next, *previous;
  int notfound = -1;
  
  if (ListDebug) printf("insert_list: >>%s<<\n",Key?Key:"(null)");

  if (!ListInit) list_init( );

  if (plist) *plist = NULL;
  if (!Key) return(-1);

  previous = (LList *) NULL;
  list = (LList *) NULL;
  next = ListRoot;

  /* search insertion point (insertion before *next) */
  while (( next!=(LList *) NULL ) && ( notfound<0 )) {
    notfound=numcmp(next->Key,Key);
    if (notfound<0) { previous = next; next = next->Next; }
  }

  /* create list, if not found, otherwise keep existing */
  if ( notfound ) {
    /* create new list */
    if (!(list = (LList *) malloc( sizeof(LList) ) )) return(-1);
    if (!(list->Key = newstr( Key ))) { free(list); return(-1); }
    list->Elements = NULL;
    list->ElementsLen = (size_t) 0;

    /* insert list before next */
    if (next) next->Previous = list;
    list->Next=next;
    list->Previous=previous;
    if (previous) previous->Next=list;
       else ListRoot = list;

    next = list;
  }

  if (plist) *plist = next;

  return(0);

} // insert_list

/*---------------------------------------------------------------------------
search_list (success:0, error:-1)
---------------------------------------------------------------------------*/
int search_list( const char *Key, LList **plist )
{ LList *list, *next;
  // LList *previous; // unused-but-set-variable
  int notfound = -1;

  if (ListDebug) printf("search_list: >>%s<<\n",Key?Key:"(null)");

  if (!ListInit) list_init( );

  if (plist) *plist = NULL;
  if (!Key) return(-1);

  // previous = (LList *) NULL; unused-but-set-variable
  list = (LList *) NULL;
  next = ListRoot;

  /* search insertion point (insertion before *next) */
  while (( next ) && ( notfound<0 )) {
    notfound=numcmp(next->Key,Key);
    if (notfound<0) { 
      // previous = next; unused-but-set-variable
      next = next->Next;
    }
  } 

  /* found */
  if (!notfound) list=next;

  if (plist) *plist = list;

  return( 0 );

} // search_list

/*---------------------------------------------------------------------------
delete_list (success:0, error:-1)
---------------------------------------------------------------------------*/
int delete_list( LList *list,  LList **pnext )
{ LList *previous, *next;
  if (!ListInit) list_init( );

  if (ListDebug) printf("delete_list:\n");

  if (pnext) *pnext=NULL;
  if (!list) return(0); // nothing to do

  /* unlink list */
  previous = list->Previous;
  next = list->Next;

  if ( next ) next->Previous = previous;
  if ( previous ) previous->Next = next;
    else ListRoot = next;

  /* free list */
  if (list->Key) free(list->Key);
  free_elements( list );
  free(list);
  
  return(0);
} // delete_list

/*---------------------------------------------------------------------------
remove_list (success:0, error:-1)
---------------------------------------------------------------------------*/
int remove_list( const char *Key, LList **pnext )
{ LList *next;
  // LList *previous; // unused-but-set-variable
  int notfound = -1;

  if (ListDebug) printf("remove_list: >>%s<<\n",Key?Key:"(null)");

  if (!ListInit) list_init( );

  // previous = (LList *) NULL; unused-but-set-variable
  next = ListRoot;

  if (pnext) *pnext = NULL;
  if (!Key) return(-1);

  /* search list */
  while (( next!=(LList *) NULL ) && ( notfound<0 )) {
    notfound=numcmp(next->Key,Key);
    if (notfound<0) { 
      // previous = next; unused-but-set-variable
      next = next->Next; 
    }
  }

  /* remove list next, if found */
  if ( !notfound ) {
    if ( delete_list( next, &next ) ) return(-1);
  }

  if (pnext) *pnext = next;

  return( 0 );

} // remove_list

/*---------------------------------------------------------------------------
free_lists (success:0, error:-1)
---------------------------------------------------------------------------*/
int free_lists( void )
{ LList *next;

  if (ListDebug) printf("free_lists:\n");

  if (!ListInit) list_init( );

  next=ListRoot;

  while (next) {
    if ( delete_list( next,  &next ) ) return(-1);
  }

  ListRoot=NULL;

  return(0);

} // free_lists

/*---------------------------------------------------------------------------
insert_element (success:0, error:-1)
---------------------------------------------------------------------------*/
int insert_element( LList *list, const char *Key, LElement **pelement )
{ LElement *element, *next, *previous;
  int notfound = -1;

  if (ListDebug) printf("insert_element: >>%s<<\n",Key?Key:"(null)");

  if (!ListInit) list_init( );

  if (pelement) *pelement = NULL;
  if (!Key) return(-1);
  if (!list) return(-1); // nothing can be inserted

  previous = NULL;
  element = NULL;
  next = list->Elements;

  /* search insertion point (insertion before *next) */
  while (( next ) && ( notfound<0 )) {
    notfound=numcmp(next->Key,Key);
    if (notfound<0) { previous = next; next = next->Next; }
  }

  /* create element, if not found */
  if ( notfound ) {
    /* create new element */
    if (!(element = (LElement *) malloc( sizeof(LElement) ) )) return(-1);
    if (!(element->Key = newstr( Key ))) { free(element); return(-1); }
    element->Value = NULL;
    element->List = list;

    /* insert element before next */
    if (next) next->Previous = element;
    element->Next=next;
    element->Previous=previous;
    if (previous) previous->Next=element;
       else list->Elements = element;

    list->ElementsLen++; // increment element counter

    next = element;
  }

  if (pelement) *pelement = next;

  if (ListDebug) printf("insert_element: >>%s<< inserted into list >>%s<<\n",
    (*pelement)->Key, list->Key);

  return(0);

} // insert_element

/*---------------------------------------------------------------------------
search_element (success:0, error:-1)
---------------------------------------------------------------------------*/
int search_element( LList *list, const char *Key, LElement **pelement )
{ LElement *element, *next;
  // LElement *previous; // unused-but-set-variable
  int notfound = -1;

  if (ListDebug) printf("search_element: >>%s<<\n",Key?Key:"(null)");

  if (!ListInit) list_init( );

  if (!list) return(0); // nothing to look for, list is empty
  if (pelement) *pelement = NULL;
  if (!Key) return(-1);

  // previous = NULL; unused-but-set-variable
  element = NULL;
  next = list->Elements;

  /* search insertion point (insertion before *next) */
  while (( next ) && ( notfound<0 )) {
    notfound=numcmp(next->Key,Key);
    if (notfound<0) { 
      // previous = next; unused-but-set-variable
      next = next->Next; 
    }
  }

  /* found */
  if (!notfound) element=next;

  if (pelement) *pelement = element;

  return( 0 );

} // search_element

/*---------------------------------------------------------------------------
write_element_value (success:0, error:-1)
---------------------------------------------------------------------------*/
int write_element_value( LList *list, const char *Key, const char * Value)
{ LElement *element;

  if (ListDebug) printf("write_element_value: >>%s<< <- >>%s<<\n",
    Key?Key:"(null)",Value?Value:"(null)");

  if (!ListInit) list_init( );

  if (insert_element( list, Key, &element )) return(-1);
  if (put_element_value( element, Value )) return(-1);

  return(0);

} // write_element_value

/*---------------------------------------------------------------------------
read_element_value (success:0, error:-1)
---------------------------------------------------------------------------*/
int read_element_value( LList *list, const char *Key, const char **pValue)
{ LElement *element;

  if (ListDebug) printf("read_element_value: >>%s<<\n", Key?Key:"(null)");

  if (!ListInit) list_init( );

  if (pValue) *pValue=NULL;

  if (search_element( list, Key, &element )) return(-1);

  if (get_element_value( element, pValue)) return(-1);

  return(0);

} // read_element_value

/*---------------------------------------------------------------------------
remove_element (success:0, error:-1)
---------------------------------------------------------------------------*/
int remove_element( LList *list, const char *Key, LElement **pnext )
{ LElement *element, *next;

  if (ListDebug) printf("remove_element: >>%s<<\n",Key?Key:"(null)");

  if (!ListInit) list_init( );

  if (pnext) *pnext=NULL;

  if (search_element( list, Key, &element )) return(-1);
  if (delete_element( element,  &next )) return(-1);

  if (pnext) *pnext=next;
  return(0); 

} // remove_element

/*---------------------------------------------------------------------------
put_element_value (success:0, error:-1)
---------------------------------------------------------------------------*/
int put_element_value( LElement *element, const char *Value) 
{
  if (ListDebug) printf("put_element_value: >>%s<<\n",Value?Value:"(null)");

  if (!ListInit) list_init( );

  if (!element) return(-1);
  if (element->Value) free(element->Value); element->Value=newstr(Value);
  return(0);
} // put_element_value

/*---------------------------------------------------------------------------
get_element_value (success:0, error:-1)
---------------------------------------------------------------------------*/
int get_element_value( LElement *element, const char **pValue)
{
  if (ListDebug) printf("get_element_value:\n");

  if (!ListInit) list_init();

  if (element) {
     if (pValue) *pValue=element->Value;
  } else  if (pValue) *pValue=NULL;

  return(0);

} // get_element_value

/*---------------------------------------------------------------------------
delete_element (success:0, error:-1)
---------------------------------------------------------------------------*/
int delete_element( LElement *element,  LElement **pnext )
{ LElement *previous, *next;

  if (ListDebug) printf("delete_element:\n");

  if (!ListInit) list_init( );

  if (!element) return(0); // nothing to do
  if (pnext) *pnext = NULL;

  /* unlink element */
  previous = element->Previous;
  next = element->Next;

  if ( next ) next->Previous = previous;
  if ( previous ) previous->Next = next;
    else (element->List->Elements) = next;
  element->List->ElementsLen--;

  /* free element */
  if (element->Key) free(element->Key);
  if (element->Value) free(element->Value);
  free(element);

  return(0);
  
} // delete_element

/*---------------------------------------------------------------------------
free_elements (success:0, error:-1)
---------------------------------------------------------------------------*/
int free_elements( LList *list )
{ LElement *next;

  if (ListDebug) printf("free_elements:\n");

  if (!ListInit) list_init( );

  if (!list) return(0); // nothing to do

  next=list->Elements;

  while (next) {
    if ( delete_element( next,  &next ) ) return(-1);
  }

  list->Elements=NULL;
  list->ElementsLen=(size_t) 0;

  return(0);

} // free_elements

/*---------------------------------------------------------------------------
NAME

   first_element --- returns a pointer to the first element of the element list

SYNOPSIS

   int first_element( LList *list, LElement **pelement );

DESCRIPTION

   Returns a pointer to the first element of the element list in *pelement.

ARGUMENT

  LList *list: pointer to the list
  **pelement: *pelement is replaced with the pointer to the first element.

RETURN VALUE

   0: success, 1: error

---------------------------------------------------------------------------*/
int first_element( LList *list, LElement **pelement )
{ LElement *element=NULL;

  if (pelement) *pelement=element;

  if (!ListInit) list_init();

  if (!list) return(0);  

  element = list->Elements;

  if (pelement) *pelement=element;

  return(0);

} // first_element

/*---------------------------------------------------------------------------
NAME

   next_element --- returns a pointer to the next element after *pelement 

SYNOPSIS

   int next_element( LList *list, LElement **pelement );

DESCRIPTION

   Returns a pointer to the first element of the element list in *pelement.
   During call *pelement must contain the pointer to the previous
   element.

ARGUMENT

  LList *list: pointer to the list
  **pelement: *pelement is replaced with the pointer to the first element.

RETURN VALUE

   0: success, 1: error

---------------------------------------------------------------------------*/
int next_element( LList *list, LElement **pelement )
{ LElement *element=NULL;

  if (!pelement) return(-1); // *pelement must exist

  element = *pelement;

  if (!ListInit) list_init();

  if (!list) return(0);

  // check *pelement
  if (!is_member(list, element)) return(-1); // element must exist;

  *pelement = element->Next;

  return(0);

} // next_element

/*---------------------------------------------------------------------------
NAME

   is_element --- returns 1 if an element with the name Key exists

SYNOPSIS

   int is_element( LList *list, const char *Key );

DESCRIPTION

   Returns 1 if an element with the name Key exists.

RETURN VALUE

   0: success, 1: error

---------------------------------------------------------------------------*/
int is_element( LList *list, const char *Key )
{ LElement *element; 
  int value=0;

  if (!ListInit) list_init();

  if (!list) return(value); 
  if (search_element( list, Key, &element )) return(value); 

  if (element) value=1;

  return(value);

} // is_element

/*---------------------------------------------------------------------------
NAME

   is_member --- checks, whether element is a member of list 

SYNOPSIS

   int is_member( LList *list, LElement *element );

RETURN VALUE

   Returns 1 if element is a member of list.

---------------------------------------------------------------------------*/
int is_member( LList *list, LElement *element )
{ int value=0;
  LElement *pe;

  if (!ListInit) list_init();

  if (!list) return(value);

  if (!element) return(value);

  pe = list->Elements; 

  while (pe) {
    if (pe == element) { value=1; break; }
    pe=pe->Next;
  }

  return( value );
   
} // is_member

/*---------------------------------------------------------------------------
NAME

   number_of_elements --- returns the number of elements in the list

SYNOPSIS

   size_t number_of_elements( LList *list );

DESCRIPTION

   Returns the number of elements in the list

RETURN VALUE

   number of elements in the list

---------------------------------------------------------------------------*/
size_t number_of_elements( LList *list )
{ if (!list) return((size_t) 0);
  return(list->ElementsLen);
} // number_of_elements

/*---------------------------------------------------------------------------
NAME

   cnt_elements --- counts the number of already defined specific elements

SYNOPSIS

   int cnt_elements( LList *list, const char *Keys );

DESCRIPTION

   Counts how many element names listed in Keys are already defined in the list.
   It returns the number of already defined elements. The keys must be separated
   by spaces, e.g. Keys="pix1 pix2 cen1 cen2 dis rot1 rot2 rot3".

RETURN VALUE

  Number of elements already defined.

---------------------------------------------------------------------------*/
int cnt_elements( LList *list, const char *Keys )
{ int cnt=0l;
  char *pc, *Key, *KeyBuf=NULL;
  size_t Keyslen;

  if (!list) return(cnt); 
  if (!Keys) return(cnt);

  // copy Keys 
  Keyslen = STRLEN(Keys);
  if ( !(KeyBuf = (char *) malloc(Keyslen+1)) ) goto cnt_elements_error;
  strncpy( KeyBuf, Keys, Keyslen ); 
  KeyBuf[Keyslen]='\0';

  // split KeyString into Keys
  pc = KeyBuf;
  while (pc) {
    Key = pc;
    if ( (pc=strchr(pc,(int) ' ')) ) {
      *pc='\0'; pc++; // terminate Key
    }
    cnt+=is_element( list, Key );
  }
 
  if (KeyBuf) free(KeyBuf);

  return( cnt );

cnt_elements_error:

  if (KeyBuf) free(KeyBuf);

  return(-1);


} // cnt_elements

int cnt_params( LList *list, const char *KeyNames[] )
{ int cnt=0l, num=0;
  const char *Key;

  if (!list) return(cnt);
  if (!KeyNames) return(cnt);

  // split KeyString into Keys
  while ( (Key=KeyNames[num++]) ) cnt+=is_element( list, Key );

  return( cnt );

} // cnt_params

/*---------------------------------------------------------------------------
print_elements (success:0, error:-1)
---------------------------------------------------------------------------*/
int print_elements( FILE * out, LList * list, int level, int verbose )
{ const char * SeparationLine =
    "-       -       -       -       -       -       -       -";

  LElement * element;

  if (ListDebug) printf("print_elements:\n");

  if (level<1) return(0);

  if (!ListInit) list_init( );

  element = list->Elements;

  while (element) {
    if (verbose) {
      fprintf(out,"   %s\n",SeparationLine);
      fprintf(out,"   Key               = %s\n",element->Key);
      fprintf(out,"   Value             = %s\n",element->Value?element->Value:"(empty)");
      fprintf(out,"   Previous Key      = ");
      if (element->Previous)
        fprintf(out,"%s\n", element->Previous->Key);
        else fprintf(out,"(no previous element)\n");
      fprintf(out,"   Next Key          = ");
      if (element->Next)
        fprintf(out,"%s\n", element->Next->Key);
        else fprintf(out,"(no next element)\n");
      fprintf(out,"   List              = %s\n",element->List?element->List->Key:"(empty)");
    } else {
      fprintf(out,"   '%s' = '%s'\n",element->Key,element->Value?element->Value:"(empty)");
    }
    element=element->Next;
  }

  if (verbose) fprintf(out,"   %s\n",SeparationLine);

  return(0);

} /* print_elements */

/*---------------------------------------------------------------------------
print_lists (success:0, error:-1)
---------------------------------------------------------------------------*/
int print_lists( FILE * out, int level, int verbose )
{ const char * SeparationLine =
    "-   -   -   -   -   -   -   -   -   -   -   -   -   -   -";

  LList * list;

  if (ListDebug) printf("print_lists:\n");

  if (level<1) return(0);

  if (!ListInit) list_init( );

  list = ListRoot;

  while(list) {
    if (verbose) {
      fprintf(out,"  %s\n",SeparationLine);
      fprintf(out,"  Key                = %s\n",list->Key);
      fprintf(out,"  ElementsLen        = '%lu'\n",SIZE_T list->ElementsLen);
      print_elements( out, list, level-1, verbose );
      fprintf(out,"  Previous Key       = ");
      if (list->Previous)
        fprintf(out,"%s\n", list->Previous->Key);
        else fprintf(out,"(no previous list)\n");
      fprintf(out,"  Next Key           = ");
      if (list->Next)
        fprintf(out,"%s\n", list->Next->Key);
        else fprintf(out,"(no next list)\n");
    } else {
      fprintf(out,"  Key                = '%s'\n",list->Key);
      fprintf(out,"  ElementsLen        = '%lu'\n",SIZE_T list->ElementsLen);
      print_elements( out, list, level-1, verbose );
    }
    list=list->Next;
  }

  if (verbose) fprintf(out,"  %s\n",SeparationLine);

  return(0);

} /* print_lists */
