/***************************************************************************/
/* Written 1994++ by Peter Boesecke                                        */
/* Copyright (C) 2011 European Synchrotron Radiation Facility              */
/*                       Grenoble, France                                  */
/*                                                                         */
/*    Principal authors: Peter Boesecke  (boesecke@esrf.eu)                */
/*                                                                         */
/*    This program is free software: you can redistribute it and/or modify */
/*    it under the terms of the GNU General Public License as published by */
/*    the Free Software Foundation, either version 3 of the License, or    */
/*    (at your option) any later version.                                  */
/*                                                                         */
/*    This program is distributed in the hope that it will be useful,      */
/*    but WITHOUT ANY WARRANTY; without even the implied warranty of       */
/*    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the        */
/*    GNU General Public License for more details.                         */
/*                                                                         */
/*    You should have received a copy of the GNU General Public License    */
/*    along with this program.  If not, see <http://www.gnu.org/licenses/>.*/
/***************************************************************************/
/*+++
1 saxs_average
  Creates an average and variances from a series of images.

2 PURPOSE
  Creates an average and variances from a series of images.
  All images must have the same geometries:

  Dim_1, Dim_2, 
  Offset_1, Offset_2
  Dummy, Ddummy

  No reference system other than array is possible.

  -add <#naverage> chooses the number of images to average 
                   (default 0, all)

  Calculation:       I0 = average(I1)  == average of Add.V images
  Error propagation: V0 = variance(I1) == variance of Add.V images
                                          (input variances are ignored)

  Arguments:
  saxs_average [options] <i1nam> <onam> <i1fst> <i1lst> <i1inc> 
             <odum> <odim1> <odim2> <ofac> <ocon>"

  Defaults:
  <input file name>   : input.edf 
  <output file name>  : output.edf
  <first image>       : <first image number in input file>
  <last image>                              : <last image number in input file>
     if argument list ends with first image : <first image>
  <increment>         : 1
  <dummy>             : <dummy value in first image of input file>
  <dimension 1>       : <horizontal dimension of first image in input file>
  <dimension 2>       : <vertical dimension of first image in input file>
  <output const>      : 0.0
  <output factor>     : 1.0

  Wildcard:
  A "=" in the argument list chooses the default.

2 HISTORY
  2012-02-09  PB 4.00
  2012-06-26  PB 4.01 strlib.h included, 
                       4) calculate variance of the average
                         --> variance calculation corrected
  2012-06-27  PB 4.02 percentchar added to avoid lcc format string bug
  2012-09-12  PB 4.03 pq splitted into loq, upq with defaults loq=0.1, upq=0.9
  2012-09-12  PB 4.04 average: 
                      amode: average value corrected in variance calculation,
                             quantil is now pq-quantil of the option pq
                      fmode: quantil filters between lpq and upq.
  2013-04-25  PB 4.05 variance of the mean is set to VarDummy, if it cannot be
                      calculated (n<1 for -n-1 (vmode=0), 
                                  n<2 for +n-1 (vmode=1))
                      Until now it was set to 0.0.
  2013-05-23  PB 4.06 average, saxs_average_term: to standarize indices,
                      i_1, i_2 run now from 0 to dim-1, instead from 1 to dim 
                      as previously defined.
                      To avoid empty points when using a too strong filter 
                      condition, the option +fill has been added. It is 
                      intended to be used for averaging and filtering of a
                      series of dark images, when dummy values cannot be used, 
                      e.g. when a negative dummy value is defined for unsigned
                      values or when not dummy value is set.
                      The fill option creates an internal list of all unset
                      pixels (either removed by filters or already defined as 
                      dummies) and replaces each pixel value by the average 
                      in a region of interest (roi) with lengths (fil_1, fil_2) 
                      centered at the unset pixel. The variances
                      of these pixels will not be updated and remain VarDummy.

---*/
# define Version  "saxs_average V4.06 2013-05-27, Peter Boesecke"

# include <errno.h>
# include <stdio.h>
# include <unistd.h>
# include <fcntl.h>

# include "statfunc.h"
# include "strlib.h"
# include "SaxsPrograms.h"

# define FREE( a) { if ( a) { free( a); a = NULL; } }

# define Usage "[options] \n\
                <i1nam> <onam> <i1fst> <i1lst> <i1inc> \n\
                <odum> <odim1> <odim2> <ofac> <ocon> \n\n\
                Calculation of the average and the variance of a series of\n\
                images. All images that will be averaged are first read\n\
                and kept in memory. If the option +var is set (default) also\n\
                the variance with respect to the calculated average is \n\
                calculated and written into the error image (-omem -1).\n\
                The variance is only well defined in averaging mode mean\n\
                which is the default.\n\
                Flatfield:\n\
                To remove cosmics when averaging a series of images the\n\
                filter mode median can be used.\n\
                Hot pixel mask:\n\
                To create a hot pixel mask a series of dark images should\n\
                be averaged with averaging mode median or quantil. Only\n\
                pixel values that are in more than 50% of the cases hot\n\
                (median) will get the hot value, all others will have \n\
                the usual dark value. The averaging mode quantil allows\n\
                to choose other levels than 50%.\n\
                options: -add <number of images to average>\n\
                              (default 0 to average all)\n\
                         -amod   <mean|median|quantil> average function\n\
                                   (default: mean)\n\
                         -fmod   <mean|median|quantil> filter function\n\
                                   (default: none)\n\
                         +/-n-1  <calculate empiric variance>\n\
                                   (default +n-1)\n\
                         -sig-   <lowfac> to filter values < <lowfac>*sigma\n\
                                   (default 5)\n\
                         -sig+   <upfac> to filter values > <upfac>*sigma\n\
                                   (default 5)\n\
                         -mins   <minimum sigma> for filter\n\
                                   (default 0.5)\n\
                         -loq    <lower quantil (fmode)> [0..1] (0<=loq<1)\n\
                                   (default 0.1)\n\
                         -upq    <upper quantil (fmode)> [0..1] (0<=upq<1)\n\
                                   (default 0.1)\n\
                         -pq     <quantil (amode)> [0..1] (0<=pq<=1)\n\
                                   (default 0.5 == median calculation)\n\
                         +/-fill <try to fill filtered out pixels with average>\n\
                                   (default -fill)\n\
                         -fil1   <pixel length 1 of averaging region>\n\
                                   (default 3)\n\
                         -fil2   <pixel length 2 of averaging region>\n\
                                   (default 3)\n\
                In averaging mode quantil the pq-quantil is returned. For\n\
                pq==0.5 (default) the median is calculated.\n\
                In filter mode quantil only values inside the interval\n\
                [<lower pq-quantil value>..<upper pq-quantil value>] are\n\
                accepted for averaging. Values outside this interval are\n\
                ignored."

# define BlockNum 2       /* 1 input sequence + 1 output sequence */

# define SFramename   "fnam"  /* line */    // frame-file name
# define SAMode       "amod"  /* line */    // averaging mode 
# define SFMode       "fmod"  /* line */    // filter mode 
# define SNM1         "n-1"   /* flag */    // calculate variance with n-1
# define SSIGM        "sig-"  /* double */  // lower sigma limit factor
# define SSIGP        "sig+"  /* double */  // upper sigma limit factor
# define SMINS        "mins"  /* double */  // minimum sigma for filter 
# define SLOQ         "loq"   /* double */  // lower quantil value (0=<loq<1) 
# define SUPQ         "upq"   /* double */  // upper quantil value (0<upq<=1) 
# define SPQ          "pq"    /* double */  // quantil value (0<=pq<=1) 
# define SFILL        "fill"  /* flag */    // fill filtered out pixels
# define SFIL1        "fil1"  /* double */  // averaging length 1 
# define SFIL2        "fil2"  /* double */  // averaging length 2

const float eps=1e-32;
const char percentchar='%';

/***************************************************************************
* Averaging Mode Translation Tables                                        *
***************************************************************************/
enum AMode { InValidAMode, Mean=1, Median, Quantil, EndAMode };

static const char * AModeStrings[] = { "Invalid", "mean", "median",
                                       "quantil", (const char *) NULL };

/***************************************************************************
* Create, define, read and release a list of points (BEGIN)                *
*                                                                          *
* ListOfPoints * new_listofpoints( void );                                 *
* ListOfPoints * free_listofpoints( ListOfPoints * listofpoints );         *
* long length_listofpoints( ListOfPoints * listofpoints );                 *
* Point * append_point( ListOfPoints * listofpoints, long i_1, long i_2 ); *
* Point * first_point( ListOfPoints * listofpoints );                      * 
* Point * get_point_and_increment( Point * point, long *pi_1, long *pi_2 ) *
* void print_listofpoints( FILE * out, ListOfPoints * listofpoints )       *
***************************************************************************/

typedef struct Point_ {
  long i_1; // index 1
  long i_2; // index 2
  struct Point_ * nextpoint;
} Point;

typedef struct {
  long  npoints;    // number of points in list
  Point * firstpoint; // first point of list
  Point * lastpoint;  // last point of list
} ListOfPoints;

/*---------------------------------------------------------------------------
NAME

  append_point --- Writes a point to the end of the list

PURPOSE

  Writes a new point to the end of the list. If listofpoints is NULL
  nothing happens.

RETURN VALUE

  Pointer to the new point or NULL in case of error.

---------------------------------------------------------------------------*/
Point * append_point( ListOfPoints * listofpoints, long i_1, long i_2 )
{ Point * new=NULL;

  if (listofpoints) {
    Point * current;
    size_t size_point;

    size_point = sizeof ( Point );
    new = malloc ( size_point ); 
    if (!new)
      goto append_point_error;

    new->i_1 = i_1;
    new->i_2 = i_2;
    new->nextpoint=NULL;

    current = listofpoints->lastpoint;
    if (current) {
      current->nextpoint = new;
    } else {
      listofpoints->firstpoint = new;
    }
    listofpoints->lastpoint = new;
    listofpoints->npoints+=1l;
  }

  return(new);

append_point_error:

  FREE(new);

  return(NULL);

} // append_point

/*---------------------------------------------------------------------------
NAME

  first_point --- Returns pointer to the first point

PURPOSE

  Returns pointer to the first point

RETURN VALUE

  Pointer to the first point or NULL

---------------------------------------------------------------------------*/
Point * first_point( ListOfPoints * listofpoints )
{ Point * first=NULL;

  if (listofpoints) {
    first = listofpoints->firstpoint;
  }

  return(first);

} // first_point

/*---------------------------------------------------------------------------
NAME

  get_point_and_increment --- Returns the current point and goes to the next

PURPOSE
  
  Returns the indices of the current point and 
  returns pointer to the next point.
  If the input point is not NULL the indices are returned in
  *pi_1 and *pi_2.

RETURN VALUE

  Pointer to the next point or NULL

---------------------------------------------------------------------------*/
Point * get_point_and_increment( Point * point, long *pi_1, long *pi_2 )
{ Point * next=NULL;

  if (point) {
      if (pi_1) *pi_1 = point->i_1;
      if (pi_2) *pi_2 = point->i_2;
    next = point->nextpoint;
  }

  return( next );

} // get_point_and_increment 

/*---------------------------------------------------------------------------
NAME

  new_listofpoints( ) --- Allocates a new list of points

PURPOSE

  Creates a new list of points

RETURN VALUE

  Pointer to root of new point list or NULL in case of error.

---------------------------------------------------------------------------*/
ListOfPoints * new_listofpoints( void ) {
  size_t size_listofpoints;
  ListOfPoints * new=NULL;

  size_listofpoints = sizeof(ListOfPoints);
  new = malloc(size_listofpoints);
  if (!new) 
    goto new_listofpoints_error;

  new->npoints = 0l;
  new->firstpoint = NULL;
  new->lastpoint = NULL;

  return(new);

new_listofpoints_error:

  FREE(new);

  return(NULL);

} // new_listofpoints

/*---------------------------------------------------------------------------
NAME

  free_listofpoints --- Releases a list of points

PURPOSE

  Releases the memory of a list of points.

RETURN VALUE

  NULL, otherwise error

---------------------------------------------------------------------------*/
ListOfPoints * free_listofpoints( ListOfPoints * listofpoints ) 
{ if (listofpoints) {
    Point * current, * next;

    // release all points 
    next = listofpoints->firstpoint;
    while (next) {
      current = next;
      next = current->nextpoint;
      FREE(current);
    }
    
    FREE(listofpoints); 
  }

  return( NULL );

} // free_listofpoints

/*---------------------------------------------------------------------------
NAME

  length_listofpoints --- return the number of points in the list

PURPOSE

  Return the number of points in the list

RETURN VALUE

  Number of points.

---------------------------------------------------------------------------*/
long length_listofpoints( ListOfPoints * listofpoints )
{ long npoints = 0l;

  if ( listofpoints ) {
    npoints = listofpoints->npoints;
  }

  return( npoints );

} // length_listofpoints

/*---------------------------------------------------------------------------
NAME
  
  print_listofpoints --- print list of points 
  
PURPOSE
  
  Print list of points

---------------------------------------------------------------------------*/
void print_listofpoints( FILE * out, ListOfPoints * listofpoints )
{ Point * point;
  long i;

  fprintf( out, "\nList of %ld points\n", length_listofpoints(listofpoints));

  i=0;
  point = first_point( listofpoints );
  while ( point ) {
    long i_1, i_2;

    point = get_point_and_increment( point, &i_1, &i_2 );

    fprintf( out, "%ld: (%ld,%ld)\n", i++, i_1, i_2);

  }

  fprintf( out, "\n");

} // print_listofpoints

/*---------------------------------------------------------------------------
NAME
  
  print_listofdatapoints --- print list of points and values
  
PURPOSE
  
  Print list of points

---------------------------------------------------------------------------*/
void print_listofdatapoints( FILE * out, ListOfPoints * listofpoints,
                             float *data, long dim_1, long dim_2,
                             const char *name )
{ Point * point; 
  long i;

  fprintf( out, "\nList of %ld data points\n", length_listofpoints(listofpoints));
    
  i=0;
  point = first_point( listofpoints );
  while ( point ) {
    long i_1, i_2;
    float *pdata;
  
    point = get_point_and_increment( point, &i_1, &i_2 );

    pdata = ABSPTR(data,dim_1,dim_2,i_1,i_2);
    fprintf( out, "%ld: %s(%ld,%ld) = %g\n", i++, name, i_1, i_2, *pdata);

  }

  fprintf( out, "\n");

} // print_listofpoints

/***************************************************************************
* Create, define, read and release a list of points (END)                  *
***************************************************************************/

/*---------------------------------------------------------------------------
NAME
  fminval --- returns the minimum value of float array a with length n

SYNOPSIS
  float fminval ( float *a, long n )

DESCRIPTION
  Returns the minimum value of float array a with length n.

RETURN VALUE
  Minimum value of float array a for n>0 and a!=NULL

  -------------------------------------------------------------------------*/
float fminval ( float *a, long n )
{ float value=0.0;

  if ( a ) {
    float *pa;
    pa = a;
    if (n>0)
      value = *pa;
    for (pa=a+1;pa<a+n;pa++) {
      if (*pa<value)
        value = *pa;
    }
  }

  return(value);

} // fminval

/*---------------------------------------------------------------------------
NAME
  String2AMode --- converts a string to an averaging mode 

SYNOPSIS
  (AMode) int String2AMode( const char * string );

DESCRIPTION

RETURN VALUE
  AMode == 0 : error, e.g. cannot convert
  AMode >  0 : valid averaging mode value 

  -------------------------------------------------------------------------*/
int string2amode( const char *string )
{ int  NE=TRUE;
  int i, value=InValidAMode;
  size_t n;

  n = strlen(string);

  for (i=0;i<EndAMode;i++) {
    if (!(strlib_ncasecmp( string, AModeStrings[i], n ))) {
      value = i;
      break;
    };
  }

  return( value );

} // string2amode

/*---------------------------------------------------------------------------
NAME
  amode2string --- converts an averaging mode to a string

SYNOPSIS
  const char *amode2string( (AMode) int amode );

DESCRIPTION

RETURN VALUE
  "Invalid" : error, e.g. cannot convert
  averaging mode strings (in case of no errors)

  -------------------------------------------------------------------------*/
const char *amode2string( int amode )
{ if ((amode<0)||(amode>=EndAMode)) amode = InValidAMode;
   return( AModeStrings[amode] );
} // amode2string

/*---------------------------------------------------------------------------
NAME

  new_main_data, free_main_data --- Creates and releases common variables 

PURPOSE
  Creates and releases variables used by all functions.
  Must be declared in Command Block with InitOptions.

---------------------------------------------------------------------------*/
typedef struct {
  float *data;        // stacked input images
  size_t data_size;   // size of data
  float *dummy;       // stacked dummy values, one per frame 
  size_t dummy_size;  // size of ddummy
  float *ddummy;      // stacked dummy values, one per frame 
  size_t ddummy_size; // size of ddummy
  long   counter;     // counter of stacked frames

} MainData;

MainData * new_main_data( void )
{ return( (MainData *) calloc( 1, sizeof ( MainData ) ) );
} // new_main_data

MainData * free_main_data ( MainData * main_data )
{
  if (main_data) {
    if (main_data->data) free( main_data->data );
    if (main_data->dummy) free (main_data->dummy);
    if (main_data->ddummy) free (main_data->ddummy);
    free (main_data);
    main_data = NULL;
  }
  return( main_data );
} // free_main_data

/*---------------------------------------------------------------------------
NAME
  average --- Calculate average and variance of pixel (i_1,i_2)

PURPOSE
  Calculate average and variance for pixel (i_1,i_2). The type of average
  (mean, median) can be chosen. The routine takes into account dummy values.

  In filter mode "mean", "median" the sigma values are used to calculated
  the lower and upper limit values: lower value = sigma * lfac, upper value
  = sigma * ufac.
  In filter mode "quantil" all values outside the lower and upper quantil
  value are rejected.
  In average mode "mean", "median" and "quantil" the mean value, median 
  value and pq-quantil values is calculated.

ARGUMENTS
  const long dim[]        : input dimension dim = { 3, dim_1, dim_2, dim_3 }
  const float *frames (i) : input frames
  const float *dummy  (i) : input dummies, one value for each frame
  const float *ddummy (i) : input ddummies, one value for each frame
  int i_1, int i_2 (i) : pixel indices 
                         (i_1 from 0 to dim_1-1, i_2 from 0 to dim_2-1)
  (AMode) int amode   (i) : 1: mean, 2: median, 3:quantil
  (AMode) int fmode   (i) : 1: mean, 2: median, 3:quantil
  int vmode           (i) : 0: calculate variance using n
                            1: calculate variance using n-1
  int dovar           (i) : 0: do not return variance
                            1: return variance
  float lfac          (i) : sigma factor for lower value 
  float ufac          (i) : sigma factor for upper value 
  float mins          (i) : minimum sigma
  float loq           (i) : lower quantil (only for fmode==quantil)
  float upq           (i) : upper quantil (only for fmode==quantil)
  float pq            (i) : quantil (only for amode==quantil)

  int *pstatus     (o) : return status 0==OK, otherwise error

RETURN VALUE

  AveValue ave : average value

---------------------------------------------------------------------------*/
typedef struct {
  float val;  // average value
  float var;  // variance
  long  n;    // number of averaged values
  long  i;    // number of ignored values
} AveValue;

AveValue average( const long dim[], const float *frames, 
                  const float *dummy, const float *ddummy,
                  long i_1, long i_2, int amode, int fmode,
                  int vmode, int dovar, float lfac, float ufac,
                  float mins, float loq, float upq, float pq,
                  int *pstatus )
{ AveValue ave;
  double *values=NULL;  // table of valid values (dim_3 elements) 

  const float *src;
  long k, pitch;
  double *dest;
  long nvalues=0, ivalues=0 ;    // number of valid values 
  double lower, upper;
  double dlfac, dufac, dmins, dloq, dupq, dpq;

  int status=Failed; 
  
  dlfac = (double) lfac;
  dufac = (double) ufac;
  dmins = (double) mins;
  dloq   = (double) loq;
  dupq   = (double) upq;
  dpq   = (double) pq;

  ave.val = 0.0;
  ave.var = VarDummy;
  ave.n = 0;

  // 1) extract pixel values (i_1, i_2) from frames and copy them to values
  if (!( values = (double *) malloc( sizeof(double)*dim[3] ) )) {
    status = NotEnoughMemoryAvailable; 
    goto average_error;
  }

  src     = frames+i_1+i_2*dim[1];
  dest    = values;
  pitch   = dim[1]*dim[2];
  nvalues = 0;

  for (k=0;k<dim[3];k++) {
    double value;
    value = (double) *src;

    // UPDATE( *dest, value, dummy[k], ddummy[k] );
    if ( NODUMMY(value, dummy[k], ddummy[k]) ) {
      *dest = value;
      dest++;
      nvalues++;
    }

    src+=pitch; // next frame
  }

  // 2) filter values
  if (fmode) {
    double sigma, val, var;
    double upqval, loqval, hdif;

    // 2a) calculate average and deviation from average
    switch (fmode ) {
      case Mean:      
        val = mean( values, nvalues );
        var = variance ( values, nvalues, val );
        sigma = sqrt(var);
        break;

      case Quantil:
        upqval = dquantil ( values, nvalues, dupq );
        loqval = dquantil ( values, nvalues, dloq );
        hdif = fabs((upqval-loqval)*0.5); // half distance beween quantils
        val = (upqval+loqval)*0.5; // average of upper and lower quantil
        sigma = hdif;
        break;

      case Median:
      default:
        val = dmedian( values, nvalues);
        var = dmediance ( values, nvalues, val);
        sigma = sqrt(var);
    }

    if (sigma<dmins) sigma=dmins;

    // 2b) calculate lower and upper limits
    switch (fmode ) {
      case Quantil:
        lower = val - sigma; // lower quantil value
        upper = val + sigma; // upper quantil value
        break;  
      case Mean:
      case Median:
      default:
        if (lfac<0.0)
          lower = minimum ( values, nvalues ); // accept all below
        else
          lower = val - sigma * dlfac;
    
        if (ufac<0.0)
          upper = maximum ( values, nvalues ); // accept all above
        else
          upper = val + sigma * dufac;
    }

    // 2c) remove table values outside lower and upper limits
    ivalues = nvalues;
    nvalues = minmaxfilter ( values, nvalues, lower, upper );
    ivalues -= nvalues; // ignored values
  }

  // 3) calculate average and variance

  switch (amode) {
    case Median: 
      ave.val = dmedian ( values, nvalues );
      if (dovar) 
        ave.var = dmediance( values, nvalues, ave.val );
      break;
     case Quantil:
        ave.val = dquantil ( values, nvalues, dpq );
        if (dovar)
          ave.var = dquantilance(  values, nvalues, dpq, ave.val );
        break;
    case Mean: 
    default: 
      ave.val = mean ( values, nvalues );
      if (dovar)
        ave.var = variance( values, nvalues, ave.val );
  }
  ave.n   = nvalues;
  ave.i   = ivalues;

  // 4) calculate variance of the average
  
  if (dovar) {
    if (vmode==0) { // calculate with n values
       if (nvalues>0) {
         ave.var /= nvalues;
       } else ave.var = VarDummy; // variance cannot be calculated
     }  else { // calculate with n-1 values
       if (nvalues>1) {
         ave.var /= nvalues-1;
       } else ave.var = VarDummy; // variance cannot be calculated
     }
  }

  FREE(values);

  status = Success;
  if (pstatus) *pstatus=status;
  return( ave ); 

average_error:

  FREE(values);

  if (pstatus) *pstatus=status;
  return( ave ); 
  
} // average 

/*---------------------------------------------------------------------------
NAME
  fill_empty_points --- Replaces all empty points with the local average

PURPOSE
  Replaces all empty points in data with the average value in a pixel radius 
  around it. The variance array is not used. The number of updated points
  is returned.
  
ARGUMENTS
  float * data    (i/o) : input/output array
  long    dim_1   (i)   : data array dimension 1
  long    dim_2   (i)   : data array dimension 2
  float   dummy   (i)   : dummy value
  float   ddummy  (i)   : ddummy value
  float   fil_1   (i)   : length 1 of the averaging area 
  float   fil_2   (i)   : length 2 of the averaging area
  int     debug   (i)   : 1 print some debug information
                        : 0 do not print debug information

  int *pstatus    (o)   : return status 0==OK, otherwise error

RETURN VALUE

  long fillcnt : number of updated empty points values

  
---------------------------------------------------------------------------*/
long fill_empty_points( float *data, long dim_1, long dim_2, 
                      ListOfPoints * emptypoints,
                      float dummy, float ddummy, float fil_1, float fil_2, 
                      int debug, int *pstatus )
{

  long int i, j_1, j_2, i_1, i_2;
  long int i_1min;
  float *pdata;

  float *copy=NULL, *pcopy;
  size_t copy_size;

  int status;

  float value;

  Point * emptypoint=NULL;
  float f1_1, f1_2, f3_1, f3_2; // index coordinates
  float sum, weight;
  long fillcnt;

  status = Success;

  fil_1 = MAX2(0.0,fil_1);
  fil_2 = MAX2(0.0,fil_2);

  fillcnt = 0;

  if ( (length_listofpoints( emptypoints )>0) && ((fil_1>0.0) || (fil_2>0.0)) ) {

    // allocate memory for copy
    copy_size = sizeof(float)*dim_1*dim_2;
    copy = (float *) malloc (copy_size);
    if (!copy) {
      status=NotEnoughMemoryAvailable; 
      goto fill_empty_points_error;
    }

    if (debug>2) 
      print_listofdatapoints( stdout, emptypoints, data, dim_1, dim_2, "data");

    // make a copy of the data array
    memmove(copy,data,copy_size);

    if (debug>2) 
      print_listofdatapoints( stdout, emptypoints, copy, dim_1, dim_2, "copy");

    // It must be excluded that averages use empty points. If the dummy value is
    // not set the empty points in copy must be set to a temporary dummy value.
    // The dummy value must be outside the range of values of the array copy.
    if ( NODUMMYDEFINED(dummy, ddummy) ) {
      // define temporary dummy and ddummy
      dummy = (floor) ( MIN2( fminval ( copy, dim_1*dim_2 ),0.0 ) ) - 10;
      ddummy = DDSET(dummy);
      // set all empty points in copy to dummy
      emptypoint = first_point( emptypoints );
      while ( emptypoint ) {
        emptypoint = get_point_and_increment( emptypoint, &j_1, &j_2 );
        pcopy = ABSPTR(copy,dim_1,dim_2,j_1,j_2);
        *pcopy = dummy;
      }
    } // else already defined as dummy

    if (debug) {
      printf("fill_empty_points: internal dummy is %g\n",dummy,ddummy);
    }

    // replace empty points in data

    emptypoint = first_point( emptypoints );

    while ( emptypoint ) {

      emptypoint = get_point_and_increment( emptypoint, &j_1, &j_2 );
      pdata = ABSPTR(data,dim_1,dim_2,j_1,j_2);

      // average a rectangular region in copy with length (fil_1,fil_2) 
      // centered at pixel (j_1, j_2) (array coordinate of center is j_1+0.5)
      f1_1 = A2INDEX(j_1 + 0.5 - fil_1*0.5);
      f1_2 = A2INDEX(j_2 + 0.5 - fil_2*0.5);

      f3_1 = A2INDEX(j_1 + 0.5 + fil_1*0.5);
      f3_2 = A2INDEX(j_2 + 0.5 + fil_2*0.5);

      if (Isum2ldw( copy, (int) dim_1, (int) dim_2, dummy, ddummy,
                    f1_1, f1_2, f3_1, f3_2, &sum, &weight ) ) {
        *pdata = sum/weight;
        fillcnt++;
      } else {
        if (debug) {
          printf("fill_empty_points: Bad pixel (%ld,%ld) could not be filled.\n",j_1,j_2);
        }
      }

    }

    FREE(copy);

  } // else nothing to do

  if (pstatus) *pstatus = status; 

  return(fillcnt);

fill_empty_points_error:

  FREE(copy);

  if (pstatus) *pstatus = status; 

  return(fillcnt);

} // fill_empty_points

/*---------------------------------------------------------------------------
NAME
  saxs_average --- Creates average and variance from a series of images.

PURPOSE
  Creates average and variance from a series of images.

saxs_average_init : initialization of ihb variables in first image
saxs_average      : called every loop
saxs_average_term : finalization (final calculation, releasing variables ...)
---------------------------------------------------------------------------*/
void saxs_average_init (CmdBlk * pcb, long num[], ImgHeadBlk ihb[], 
                        int * pstatus)
{ MainData * frames = pcb->MainData;

  long  dim_1=0, dim_2=1, dim_3=1;
  int status=Failed;

  if (pcb->TestBit>1)
    printf("saxs_average_init\n");

  SetSaxsError( "saxs_average", "saxs_average_init", NULL );

  FREE(frames->data);
  frames->data_size=0;
  FREE(frames->dummy);
  frames->dummy_size=0;
  FREE(frames->ddummy);
  frames->ddummy_size=0;
  frames->counter=0l;

  // I0Dim is initialized with I1Dim
  dim_1 = ihb[0].Dim[1];
  dim_2 = ihb[0].Dim[2];

  /* avoid realloc, if possible allocate all frames indicated by Add.V */
  if ( (pcb->Add.I)&&(pcb->Add.V>0) )
    dim_3=pcb->Add.V;
  else dim_3=1;

  frames->data_size = dim_1*dim_2*dim_3*sizeof(float);
  if (pcb->TestBit>1)
    printf("alloc %zu bytes data\n",frames->data_size);
  if (!(frames->data= (float*) malloc( frames->data_size ))) {
    fprintf(ERROUT,
      "ERROR: Cannot allocate %zu bytes for data (%dx%dx%d floats)\n",
      frames->data_size, dim_1,dim_2,dim_3);
    goto saxs_average_init_error;
  }

  frames->dummy_size = dim_3*sizeof(float);
  if (pcb->TestBit>1)
    printf("alloc %zu bytes data\n",frames->dummy_size);
  if (!(frames->dummy= (float*) malloc( frames->dummy_size ))) {
    fprintf(ERROUT,
      "ERROR: Cannot allocate %zu bytes for dummies (%d floats)\n",
      frames->dummy_size, dim_3);
    goto saxs_average_init_error;
  }

  frames->ddummy_size = dim_3*sizeof(float);
  if (pcb->TestBit>1)
    printf("alloc %zu bytes data\n",frames->ddummy_size);
  if (!(frames->ddummy= (float*) malloc( frames->ddummy_size ))) {
    fprintf(ERROUT,
      "ERROR: Cannot allocate %zu bytes for ddummies (%d floats)\n",
      frames->ddummy_size, dim_3);
    goto saxs_average_init_error;
  }

  if (pstatus) *pstatus = Success;

  return;

saxs_average_init_error:

  FREE(frames->data);
  frames->data_size=0;
  FREE(frames->dummy);
  frames->dummy_size=0;
  FREE(frames->ddummy);
  frames->ddummy_size=0;
  frames->counter=0;

  if (pstatus) *pstatus = status;

  return;

} /* saxs_average_init */

void saxs_average (CmdBlk * pcb, long num[], ImgHeadBlk ihb[], int * pstatus)
{
  MainData * frames = pcb->MainData;

  int i,imax;

  float *I0Data, *E0Data;
  int   I0Dim_1, I0Dim_2;
  float *pI0Data, *pE0Data;
  float I0Dummy, I0DDummy;
  float VarDDummy=DDSET(VarDummy);

  float *I1Data, *E1Data;
  long  I1Dim_1, I1Dim_2;
  float I1Dummy, I1DDummy;
  float I1Value, I1Sum, I1Weight;
  float E1Value, E1Sum, E1Weight;

  float *dest;
  long  nelements=0;
  size_t required_data_size;
  float *pdummy;
  size_t required_dummy_size;
  float *pddummy;
  size_t required_ddummy_size;

  float Off_1[BlockNum], Off_2[BlockNum];
  float Ps_1[BlockNum], Ps_2[BlockNum];

  long  dim_1=0, dim_2=1, dim_3=1;
  float dummy = 0.0, ddummy = DDSET(dummy);

  int status=Failed;

  // Increment frame counter 
  frames->counter+=1l;

  imax = pcb->ImgBlkLen;
  if (!ihb[0].ActualLoopCount) {
    printf("\n Calculating ihb[0,%ld] = Function(",ihb[0].ImNum);
    for(i=1;i<imax;i++) {
      printf("ihb[%d,%ld] ", i, ihb[i].ImNum); }
    printf(")\n\n");
  }

 /* Check the number of images */
  if (pcb->ImgBlkLen!=2) {
    fprintf(ERROUT,"%d images found, 1 input and 1 output image required\n",
             pcb->ImgBlkLen); 
    goto saxs_average_error; 
  }

  GetReferenceParameters( pcb, ihb, 0, imax-1,
                          Off_1, Off_2, Ps_1,  Ps_2, &status );
  if (status!=Success)
    goto saxs_average_error;

  SetSaxsError( "saxs_average", "saxs_average", NULL );

  if ( ( fabs(Off_1[0]-Off_1[1]) > eps ) ||
       ( fabs(Off_2[0]-Off_2[1]) > eps ) ||
       ( fabs(Ps_1[0]-Ps_1[1]) > eps ) ||
       ( fabs(Ps_2[0]-Ps_2[1]) > eps ) ) {
    
    status=ReferenceNotSupported;
    fprintf(ERROUT,"ERROR: Input and output images do not have the same geometry. To avoid\n");
    fprintf(ERROUT,"       this error choose the reference system array (option -rsys array).\n");
    goto saxs_average_error;
  }

 /* 1 input, 1 output */
  I0Data  = ihb[0].Data;
  E0Data = ihb[0].VarDat;
  I0Dummy = ihb[0].Dummy.V;
  I0DDummy = ihb[0].DDummy.V;
  I0Dim_1  = (int) ihb[0].Dim[1];
  I0Dim_2  = (int) ihb[0].Dim[2];

  I1Data  = ihb[1].Data;
  E1Data  = ihb[1].VarDat;
  I1Dummy = ihb[1].Dummy.V;
  I1DDummy = ihb[1].DDummy.V;
  I1Dim_1  = ihb[1].Dim[1];
  I1Dim_2  = ihb[1].Dim[2];

  dim_1 = I0Dim_1;
  dim_2 = I0Dim_2;
  dim_3 = frames->counter;

  // check memory size and realloc if necessary 
  nelements = dim_1*dim_2;
  required_data_size = nelements*dim_3*sizeof(float);
  if ( frames->data_size < required_data_size ) {
    float * new;
    if (pcb->TestBit>1)
      printf("realloc %zu bytes data\n",required_data_size);
    if (!(new=
        (float *) realloc(frames->data, required_data_size ))) {
      fprintf(ERROUT,
        "ERROR: Failed to resize volume to %zu bytes (%dx%d%d floats)\n",
        required_data_size, dim_1, dim_2, dim_3);
      goto saxs_average_error;
    }
    frames->data_size = required_data_size;
    frames->data = new;
  }
  dest = frames->data + (dim_3-1)*nelements;

  required_dummy_size = dim_3*sizeof(float);
  if ( frames->dummy_size < required_dummy_size ) {
    float * new;
    if (pcb->TestBit>1)
      printf("realloc %zu bytes for dummies\n",required_dummy_size);
    if (!(new=
        (float *) realloc(frames->dummy, required_dummy_size ))) {
      fprintf(ERROUT,
        "ERROR: Failed to resize dummies to %zu bytes (%d floats)\n",
        required_dummy_size, dim_3);
      goto saxs_average_error;
    }
    frames->dummy_size = required_dummy_size;
    frames->dummy = new;
  }
  pdummy = frames->dummy + (dim_3-1);

  required_ddummy_size = dim_3*sizeof(float);
  if ( frames->ddummy_size < required_ddummy_size ) {
    float * new;
    if (pcb->TestBit>1)
      printf("realloc %zu bytes for ddummies\n",required_ddummy_size);
    if (!(new=
        (float *) realloc(frames->ddummy, required_ddummy_size ))) {
      fprintf(ERROUT,
        "ERROR: Failed to resize ddummies to %zu bytes (%d floats)\n",
        required_ddummy_size, dim_3);
      goto saxs_average_error;
    }
    frames->ddummy_size = required_ddummy_size;
    frames->ddummy = new;
  }
  pddummy = frames->ddummy + (dim_3-1);

  /*
   * Keep input images in 3d frame buffer until end of loop
   */
  if ( ( dim_1 == I1Dim_1 ) && ( dim_2 == I1Dim_2  ) ) {
    // copy I1Data to new frame
    memmove(dest,I1Data,nelements*sizeof(float));
    *pdummy  = I1Dummy;
    *pddummy = I1DDummy;
  } else {
    status = Failed;
    fprintf(ERROUT,"ERROR: The dimensions %ldx%ld of %s:%d\n",
      I1Dim_1,I1Dim_2,pcb->ib[1].FileName,ihb[1].ImNum);
    fprintf(ERROUT,"       are different from the output dimensions %ldx%ld.\n",
      dim_1,dim_2);
    goto saxs_average_error;
  }

  if (pstatus) *pstatus = Success;

  return;

saxs_average_error:

  FREE(frames->data);
  frames->data_size=0;
  FREE(frames->dummy);
  frames->dummy_size=0;
  FREE(frames->ddummy);
  frames->ddummy_size=0;
  frames->counter=0l;

  if (pstatus) *pstatus = status;

  return;

} /* saxs_average */

void saxs_average_term (CmdBlk * pcb, long num[], ImgHeadBlk ihb[],     
                        int * pstatus)
{ MainData * frames = pcb->MainData;

  float *I0Data, *E0Data;
  int   I0Dim_1, I0Dim_2;
  float *pI0Data, *pE0Data;
  float I0Dummy, I0DDummy;
  float VarDDummy=DDSET(VarDummy);

  int  errval;
  int  stream;
  IO_line Framename;
  IO_line AMode;
  IO_line FMode;
  IO_long NM1;
  IO_float SIGM;
  IO_float SIGP;
  IO_float MINS;
  IO_float LOQ;
  IO_float UPQ;
  IO_float PQ;
  IO_long FILL;
  IO_float FIL1;
  IO_float FIL2;
  ListOfPoints * emptypoints=NULL;
  int  amode;
  int  fmode;

  long  dim_1=0, dim_2=1, dim_3=1;

  long used=0, ignored=0, filled=0;

  int i_1, i_2;
  int status=Failed;

  if (pcb->TestBit>1)
    printf("saxs_average_term\n");

  SetSaxsError( "saxs_average", "saxs_average_term", NULL );

  Framename = option_line ( SFramename, pcb, pcb->ib, num, 0, &status );
  if (status!=Success)
    goto saxs_average_term_error;
  AMode = option_line ( SAMode, pcb, pcb->ib, num, 0, &status );
  if (status!=Success)
    goto saxs_average_term_error;
  FMode = option_line ( SFMode, pcb, pcb->ib, num, 0, &status );
  if (status!=Success)
    goto saxs_average_term_error;
  NM1 = option_flag(SNM1,pcb,pstatus);
  if (*pstatus!=Success)
    goto saxs_average_term_error;
  SIGM = option_float ( SSIGM, pcb, pcb->ib, num, 1, pstatus );
  if (*pstatus!=Success)
    goto saxs_average_term_error;
  SIGP = option_float ( SSIGP, pcb, pcb->ib, num, 1, pstatus );
  if (*pstatus!=Success)
    goto saxs_average_term_error;
  MINS = option_float ( SMINS, pcb, pcb->ib, num, 1, pstatus );
  if (*pstatus!=Success)
    goto saxs_average_term_error;
  LOQ = option_float ( SLOQ, pcb, pcb->ib, num, 1, pstatus );
  if (*pstatus!=Success)
    goto saxs_average_term_error;
  UPQ = option_float ( SUPQ, pcb, pcb->ib, num, 1, pstatus );
  if (*pstatus!=Success)
    goto saxs_average_term_error;
  PQ = option_float ( SPQ, pcb, pcb->ib, num, 1, pstatus );
  if (*pstatus!=Success)
    goto saxs_average_term_error;
  FILL = option_flag(SFILL,pcb,pstatus);
  if (*pstatus!=Success)
    goto saxs_average_term_error;
  FIL1 = option_float( SFIL1, pcb, pcb->ib, num, 1, pstatus );
  if (*pstatus!=Success)
    goto saxs_average_term_error;
  FIL2 = option_float( SFIL2, pcb, pcb->ib, num, 1, pstatus );
  if (*pstatus!=Success)
    goto saxs_average_term_error;

  amode = string2amode( AMode.V );
  fmode = string2amode( FMode.V );
  if (pcb->TestBit>1) {
    printf("AMode.V = %s; amode %d = %s\n",AMode.V,amode,amode2string(amode));
    printf("FMode.V = %s; fmode %d = %s\n",FMode.V,fmode,amode2string(fmode));
    printf("  NM.V = %d\n",NM1.V);
    printf("SIGM.V = %g\n",SIGM.V);
    printf("SIGP.V = %g\n",SIGP.V);
    printf("MINS.V = %g\n",MINS.V);
    printf(" LOQ.V = %g\n",LOQ.V);
    printf(" UPQ.V = %g\n",UPQ.V);
    printf("  PQ.V = %g\n",PQ.V);
    printf("FILL.V = %d\n",FILL.V);
    printf("FIL1.V = %g\n",FIL1.V);
    printf("FIL2.V = %g\n",FIL2.V);
  }

  printf("\n Averaging %d images (%s)\n",frames->counter,amode2string(amode));

  I0Data  = ihb[0].Data;
  E0Data = ihb[0].VarDat;
  I0Dummy = ihb[0].Dummy.V;
  I0DDummy = ihb[0].DDummy.V;
  I0Dim_1  = (int) ihb[0].Dim[1];
  I0Dim_2  = (int) ihb[0].Dim[2];

  dim_1  = (int) ihb[0].Dim[1];
  dim_2  = (int) ihb[0].Dim[2]; 
  dim_3  = frames->counter;

  /* The following check does not seem to be necessary. TermFunction is 
     only called when the output image has been updated */
  if  (!(ImageIsNew( 0, ihb ))) {
    long dim[4];
    dim[0]=3;dim[1]=dim_1;dim[2]=dim_2;dim[3]=dim_3;
    if (Framename.I) {
      /* 
       * write 3d frame buffer
       */

      if (pcb->TestBit)
        printf("    Writing file \'%s\'\n", Framename.V);

      // open edf file
      stream=edf_open_data_file  ( Framename.V, "new", &errval, &status );
      if (status) {
        fprintf(ERROUT,"Error opening \'%s\'\n",Framename.V);
        goto saxs_average_term_error;
      }

      // write edf file
      edf_write_data ( stream, 1, 1, dim, frames->data, MFloat,
                       &errval, &status );
      if (status) {
        fprintf(ERROUT,"Error writing data to \'%s\'\n",Framename.V);
        goto saxs_average_term_error;
      }

      // close edf file
      edf_close_data_file ( stream, &errval, &status );
      if (status) {
        fprintf(ERROUT,"Error closing \'%s\'\n",Framename.V);
        goto saxs_average_term_error;
      }
    }

    /* 
     * Calculate averages, variances and 
     * copy to I0Data and E0Data
     */

    emptypoints = new_listofpoints();

    pI0Data = I0Data;
    pE0Data = E0Data;
    for (i_2=0;i_2<I0Dim_2;i_2++) {
      AveValue ave;
      for (i_1=0;i_1<I0Dim_1;i_1++) {

        if ( E0Data ) {
          // value and variance (NM1.V==0: n values, NM1.V!=0: n-1 values)
          ave = average( dim, frames->data, frames->dummy, frames->ddummy,
                         i_1, i_2, amode, fmode, NM1.V, 1,
                         SIGM.V, SIGP.V, MINS.V, LOQ.V, UPQ.V, PQ.V, &status );
          if (status) {
            fprintf(ERROUT,"Error averaging frames at pixel %dx%d\n",i_1,i_2);
            goto saxs_average_term_error;
          }
          if (ave.n>0) {
            *pI0Data = ave.val;
            *pE0Data = ave.var;
          } else {
            if (!(append_point( emptypoints, i_1, i_2 ))) {
              fprintf(ERROUT,"Error appending pixel %dx%d to list of empty points (%p)\n",
                      i_1,i_2,emptypoints);
              goto saxs_average_term_error;
            }
          }
          used+=ave.n;
          ignored+=ave.i;
        } else {
          // value
          ave = average( dim, frames->data, frames->dummy, frames->ddummy,
                         i_1, i_2, amode, fmode, NM1.V, 0,
                         SIGM.V, SIGP.V, MINS.V, LOQ.V, UPQ.V, PQ.V, &status );
          if (status) {
            fprintf(ERROUT,"Error averaging frames at pixel %dx%d\n",i_1,i_2);
            goto saxs_average_term_error;
          }
          if (ave.n>0) {
            *pI0Data = ave.val;
          } else {
            if (!(append_point( emptypoints, i_1, i_2 ))) {
              fprintf(ERROUT,"Error appending pixel %dx%d to list of empty points (%p)\n",
                      i_1,i_2,emptypoints);
              goto saxs_average_term_error;
            }
          }
          used+=ave.n;
          ignored+=ave.i;
        }
  
        pI0Data++;
        pE0Data++;
  
      } /* for i_1 ... */

    } /* for i_2 ... */
 
    if (FILL.V) {
      if (pcb->TestBit>2)  print_listofpoints( stdout, emptypoints );

      filled =  fill_empty_points( I0Data, I0Dim_1, I0Dim_2, emptypoints,
                                   I0Dummy, I0DDummy, FIL1.V, FIL2.V, 
                                   pcb->TestBit, &status );
      if (status)
        goto saxs_average_term_error;
    }

  } /* if !ImageIsNew */

  if (fmode) {
    long total = used+ignored;
    double percent=0.0;
    if (total>0)
      percent = 100.0 * (((double) ignored)/((double) total));
    else percent = 100.0;
    printf("   %ld of %ld input pixels ignored (%3.1lg%c) \n",
      ignored,total,percent,percentchar);
    if (FILL.V) {
      long nempty;
      nempty = length_listofpoints( emptypoints );
      printf("   %ld of %ld empty points filled (%gx%g) ",
        filled,nempty,FIL1.V, FIL2.V);
      if (nempty!=filled) {
        printf("(%ld empty points could not be filled) ",
          nempty-filled);
      }
      printf("\n");
    }
  }
  printf("\n");

  FREE(frames->data);
  frames->data_size=0;
  FREE(frames->dummy);
  frames->dummy_size=0;
  FREE(frames->ddummy);
  frames->ddummy_size=0;
  frames->counter=0l;
  free_listofpoints( emptypoints ); 

  if (pstatus) *pstatus = Success;

  return;

saxs_average_term_error:

  FREE(frames->data);
  frames->data_size=0;
  FREE(frames->dummy);
  frames->dummy_size=0;
  FREE(frames->ddummy);
  frames->ddummy_size=0;
  frames->counter=0l;

  free_listofpoints( emptypoints ); 

  if (pstatus) *pstatus = status;

  return;

} /* saxs_average_term */

/*---------------------------------------------------------------------------
user_io
Do all the keyboard io and return cb, and ib
---------------------------------------------------------------------------*/
void user_io(CmdBlk * pcb, ImgBlk * ib, int * pstatus)
{
  char  progfn[InputLineLength];
  ImgHeadBlk ihb[BlockNum];

  IO_line AMode;
  IO_line FMode;
  IO_long NM1;
  IO_flexp *pSIGM;
  IO_flexp *pSIGP;
  IO_flexp *pMINS;
  IO_flexp *pLOQ;
  IO_flexp *pUPQ;
  IO_flexp *pPQ;
  IO_long FILL;
  IO_flexp *pFIL1;
  IO_flexp *pFIL2;

  float ROff_1, RPs_1, UOff_1, UPs_1;
  float ROff_2, RPs_2, UOff_2, UPs_2;

 /* Determine program name without directory */
   (void) RightFR((char *) pcb->argv[0],DirSeparator,progfn,InputLineLength);

  /* show usage if no arguments are given */
  if (pcb->argc<=1) printf("Usage: %s %s\n",progfn,Usage);

  /*--- Write name of program ---*/
  printf("\n");
  printf("   %s %s\n",progfn,Version);
  printf("\n");

  ArgvFilenames ( pcb, ib, ihb, 0, BlockNum-1, pstatus);
  if (*pstatus!=Success) return;
  GetReference(pcb->RSys.V,1,ihb,&ROff_1,&ROff_2,&RPs_1,&RPs_2,pstatus );
  if (*pstatus!=Success) return;
  GetReference(pcb->USys.V,1,ihb,&UOff_1,&UOff_2,&UPs_1,&UPs_2,pstatus );
  if (*pstatus!=Success) return;

  /*--- Argument  : multiplication factor */
  printf("<output image> = <input image> * Factor + Const\n");
  argv_flexp(pcb,"Multiplication factor",&ib[0].Factor,"1.0",pstatus);
  if (*pstatus!=Success) return;

  /*--- Argument  : additive constant */
  printf("<output image> = <input image> * Factor + Const\n");
  argv_flexp(pcb,"Addition constant",&ib[0].Const,"0.0",pstatus);
  if (*pstatus!=Success) return;

  AMode = option_line(SAMode,pcb,NULL,NULL,0,pstatus);
  if (*pstatus!=Success) return;

  FMode = option_line(SFMode,pcb,NULL,NULL,0,pstatus);
  if (*pstatus!=Success) return;

  NM1 = option_flag(SNM1,pcb,pstatus);
  if (*pstatus!=Success) return;

  pSIGM = (IO_flexp*) option_parameter_search( SSIGM, pcb, pstatus );
  if (*pstatus!=Success) return;

  pSIGP = (IO_flexp*) option_parameter_search( SSIGP, pcb, pstatus );
  if (*pstatus!=Success) return;

  pMINS = (IO_flexp*) option_parameter_search( SMINS, pcb, pstatus );
  if (*pstatus!=Success) return;

  pLOQ = (IO_flexp*) option_parameter_search( SLOQ, pcb, pstatus );
  if (*pstatus!=Success) return;

  pUPQ = (IO_flexp*) option_parameter_search( SUPQ, pcb, pstatus );
  if (*pstatus!=Success) return;

  pPQ = (IO_flexp*) option_parameter_search( SPQ, pcb, pstatus );
  if (*pstatus!=Success) return;

  FILL = option_flag(SFILL,pcb,pstatus);
  if (*pstatus!=Success) return;

  pFIL1 = (IO_flexp*) option_parameter_search( SFIL1, pcb, pstatus );
  if (*pstatus!=Success) return;
 
  pFIL2 = (IO_flexp*) option_parameter_search( SFIL2, pcb, pstatus );
  if (*pstatus!=Success) return;
 
  printf("\n");
  if (ib[1].Name.I)    printf("i/p file                : %s\n",ib[1].Name.V);
  if (ib[0].Name.I)    printf("o/p file                : %s\n",ib[0].Name.V);
  if (ib[1].First.I)   printf("first image             : %ld\n",ib[1].First.V);
  if (ib[1].Last.I)    printf("last image              : %ld\n",ib[1].Last.V);
  if (ib[1].Inc.I)     printf("increment               : %ld\n",ib[1].Inc.V);
  if (ib[0].Dummy.I)   printf("output dummy            : %s\n",ib[0].Dummy.V);
  if (ib[0].Dim[1].I)  printf("output dimension 1      : %s\n",ib[0].Dim[1].V);
  if (ib[0].Dim[2].I)  printf("output dimension 2      : %s\n",ib[0].Dim[2].V);
  if (ib[0].Factor.I)  printf("factor                  : %s\n",ib[0].Factor.V);
  if (ib[0].Const.I)   printf("constant                : %s\n",ib[0].Const.V);
                       printf("average mode            : %s\n", amode2string(string2amode(AMode.V)));
                       printf("filter mode             : %s\n", amode2string(string2amode(FMode.V)));
                       printf("create variances        : %s\n", pcb->CreateVariance.V?"yes":"no");
                       printf("variance calculation    : %s\n", NM1.V?"n-1":"n");
                       printf("sigma-                  : %s\n", pSIGM->V);
                       printf("sigma+                  : %s\n", pSIGP->V);
                       printf("minimum sigma           : %s\n", pMINS->V);
                       printf("lower quantil loq       : %s\n", pLOQ->V);
                       printf("upper quantil upq       : %s\n", pUPQ->V);
                       printf("quantil (amode)         : %s\n", pPQ->V);
                       printf("fill empty points       : %s\n", FILL.V?"n-1":"n");
  if (FILL.V) {
                       printf("fill averaging length 1 : %s\n", pFIL1->V);
                       printf("fill averaging length 2 : %s\n", pFIL2->V);
  } 
  printf("\n");

  if (pcb->TestBit) {
    PrintBlocks ( pcb, ib );
    printf("\n"); }

  return;
}

/*---------------------------------------------------------------------------
main
---------------------------------------------------------------------------*/

#if MAKE_FUNCTION
# define MAIN main_saxs_average
#else
# define MAIN main
#endif

int MAIN (int argc, char *argv[])
{
  CmdBlk cb;                /* command block  */
  ImgBlk ib[BlockNum];      /* image blocks */

  int status;
  int arg_no = 0;

 /* Init options, control block and image blocks */
  InitOptions( Usage, Version, new_main_data(), TRUE, &cb, ib, BlockNum );

  option_define ( IO_tpline,  SFramename, "framebuffer.edf", &cb );
  option_define ( IO_tpline,  SAMode, "mean", &cb );
  option_define ( IO_tpline,  SFMode, "none", &cb );
  option_define ( IO_tpflag,  SNM1, "TRUE", &cb ); // get variance from n-measurements
  option_define ( IO_tpflexp, SSIGM, "5", &cb ); // lower sigma factor
  option_define ( IO_tpflexp, SSIGP, "5", &cb ); // upper sigma factor
  option_define ( IO_tpflexp, SMINS, "0.5", &cb ); // minimum sigma per pixel
  option_define ( IO_tpflexp, SLOQ, "0.1", &cb ); // lower quantil (fmode)
  option_define ( IO_tpflexp, SUPQ, "0.9", &cb ); // upper quantil (fmode)
  option_define ( IO_tpflexp, SPQ, "0.5", &cb ); // quantil (amode)

  option_define ( IO_tpflag,  SFILL, "FALSE", &cb ); // replace dummies with average 
  option_define ( IO_tpflexp,   SFIL1, "3.0", &cb ); // averaging length 1
  option_define ( IO_tpflexp,   SFIL2, "3.0", &cb ); // averaging length 2
 
 /* Read options from argument list */
  ReadOptions( argv, &arg_no, &cb, ib, &status);

 /* Default: Combine all images of the sequence (Add.V=0)*/
  if (status==Success) {
    if (!cb.Add.I) {
      cb.Add.V = 0; cb.Add.I = TRUE;
    }
  }

 /* Default: Create variance array (Var.V=1)*/
  if (status==Success) {
    if (!cb.CreateVariance.I) {
      cb.CreateVariance.V = 1; cb.CreateVariance.I = TRUE;
    }
  }

  /* USER KEYBOARD I/O */
  if (status==Success) {
    argv_start ( &cb, 1 );
    user_io( &cb, ib, &status);
    argv_end( &cb ); /* must be called after user_io */
  }

  /* SEQUENCE CALCULATION */
  if (status==Success) 
    IMAGELOOP( &cb, ib, saxs_average, saxs_average_init, saxs_average_term, TRUE, &status );

  cb.MainData=free_main_data(cb.MainData);

  return( ReportSaxsStatus( status, 0 ) );

} /* MAIN */
