/**
 * 
 */
package fable.framework.xmlparser.object;

import java.util.HashMap;
import java.util.Map;

/**
 * This class represents a node (<option> tag) in a PeakSearch xml file <option
 * option="format"> <short_name>F</short_name> <dest>format</dest>
 * <action>store</action> (...) </option><br>
 * <br>
 * 
 * Comment: There are problems with the parser adding white space depending on
 * how the XML file is formatted. Some of the getters have been fixed to trim.
 * Probably all of them should be.
 * 
 * @author suchet
 */
public class Xmloption {
	private HashMap<String, String> xmlNames; // <tag, value>
	private final static String NOT_AVAILABLE = "NA";

	/**
	 * Constructor. Only makes the HashMap.
	 */
	public Xmloption() {
		xmlNames = new HashMap<String, String>();
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see java.lang.Object#clone()
	 */
	public Xmloption clone() {
		Xmloption opt = new Xmloption();
		String key, value;
		for (Map.Entry<String, String> entry : xmlNames.entrySet()) {
			key = entry.getKey();
			value = entry.getValue();
			opt.getXmlNames().put(key, value);
		}
		return opt;
	}

	/**
	 * 
	 * @return
	 */
	public HashMap<String, String> getNode() {
		return xmlNames;
	}

	/**
	 * 
	 * @param key
	 * @param value
	 */
	public void putTag(String key, String value) {
		xmlNames.put(key, value);
	}

	public boolean isRequired() {
		return xmlNames.get("required") != null ? xmlNames.get("required")
				.toLowerCase().equals("true") : false;
	}

	public boolean isEnabled() {
		return xmlNames.get("enabled") != null ? xmlNames.get("enabled")
				.toLowerCase().equals("true") : true;
	}

	// in xml file, condition is like <enabled>myoption="10"<enabled>.
	// So this function return the key of the condition to enable the field
	public String getEnabledCondition() {
		String keyCondition = "";
		String condition = xmlNames.get("enabled");
		if (xmlNames.get("enabled") != null) {
			String[] split = condition.split("=");
			keyCondition = (split != null && split.length == 2) ? split[0] : "";
		}
		return keyCondition;// return myoption
	}

	public String getEnabledValue() {
		String keyValue = "";
		String condition = xmlNames.get("enabled");
		if (xmlNames.get("enabled") != null) {
			String[] split = condition.split("=");
			keyValue = (split != null && split.length == 2) ? split[1] : "";
		}
		return keyValue;// return 10;
	}

	/**
	 * @return the option
	 */
	public String getOption() {
		// System.out.println("Options" + xmlNames.get("option"));
		return xmlNames.get("option") != null ? xmlNames.get("option")
				: NOT_AVAILABLE;
	}

	/**
	 * @return the label of the option
	 */
	public String getLabel() {
		// System.out.println("Options" + xmlNames.get("option"));
		return xmlNames.get("label") != null ? xmlNames.get("label")
				: NOT_AVAILABLE;
	}

	/**
	 * @return the short_name
	 */
	public String getShortName() {
		return xmlNames.get("short_name") != null ? xmlNames.get("short_name")
				: NOT_AVAILABLE;
	}

	/**
	 * @return the action
	 */
	public String getAction() {
		return xmlNames.get("action") != null ? xmlNames.get("action")
				: NOT_AVAILABLE;
	}

	/**
	 * @return the dest
	 */
	public String getDest() {
		return xmlNames.get("dest") != null ? xmlNames.get("dest")
				: NOT_AVAILABLE;
	}

	/**
	 * @return the type
	 */
	public String getType() {
		return xmlNames.get("type") != null ? xmlNames.get("type")
				: NOT_AVAILABLE;
	}

	/**
	 * @return the help
	 */
	public String getHelp() {
		// Trim the parsed value
		// TODO It would be good to format the returned value here, to avoid
		// problems when the XML file is auto formatted. That would require
		// eliminating excess white space and putting \n in at the right places
		// if the line is long. It should be possible to find a code snippet
		// that does this.
		return xmlNames.get("help") != null ? xmlNames.get("help").trim()
				: NOT_AVAILABLE;
	}

	/**
	 * @return the default_value
	 */
	public String getDefaultValue() {
		// TODO It seems to parse <default></default> as "\n\t\t"
		// So trim the parsed value
		return xmlNames.get("default") != null ? xmlNames.get("default").trim()
				: "";
	}

	/**
	 * @return the default_value
	 */
	public String getGroup() {
		return xmlNames.get("group") != null ? xmlNames.get("group") : "";
	}

	public String getChoices() {
		return xmlNames.get("choices") != null ? xmlNames.get("choices") : "";
	}

	public void setValue(String val) {
		xmlNames.put("value", val);
	}

	public String getValue() {
		return xmlNames.get("value");
	}

	public boolean getVary() {
		String vary = xmlNames.get("vary");
		boolean ret = false;
		if (vary != null) {
			ret = vary.toLowerCase().equals("true");
		}
		return ret;
	}

	/**
	 * For file path, in windows, change all "\" with "/", otherwhise python
	 * can't find the path
	 * 
	 * @return the string representation in python code, not for humans
	 */
	public String toString() {
		String type = getType();
		String value = getValue();
//		value = value.replaceAll("\\\\", "/");
		String argtopython = getDest() + "=";
		// Please discuss with Jon - this is buggy
		// In python you cannot have a type of outfile or outstem
		// as those are specific option names.
		if (type.equals("string") || type.equals("outfile")
				|| type.equals("outstem") || type.equals("file")
				|| type.equals("choice")) {
			value = "\"" + value + "\"";
		}
		String action = getAction();
		if (action.equals("append")) {
			value = "[" + value + "]";
		}
		// If you are writing to python, you mean dest
		argtopython += value;
		return argtopython;
	}

	/**
	 * @return the xmlNames
	 */
	public HashMap<String, String> getXmlNames() {
		return xmlNames;
	}

	/**
	 * Utility for printing out the option in a readable format. Can be used for
	 * debugging.
	 */
	public void printOption() {
		printOption(this);
	}

	/**
	 * Utility for printing out the given option in a readable format. Can be
	 * used for debugging.
	 * 
	 * @param optionName
	 *            The name of an option.
	 */
	public static void printOption(Xmloption option) {
		boolean abort = false;
		String info = "";
		if (option == null) {
			info += "Option is null\n";
			abort = true;
		}
		if (abort) {
			System.out.println(info);
			return;
		}
		info += option.getOption() + "\n";
		info += String.format("  [%08x]", option.hashCode()) + "\n";
		info += "  Value: " + option.getValue() + "\n";
		info += "  Default: " + option.getDefaultValue() + "\n";
		info += "  Label: " + option.getLabel() + "\n";
		info += "  Short Name: " + option.getShortName() + "\n";
		info += "  Type: " + option.getType() + "\n";
		info += "  Choices: " + option.getChoices() + "\n";
		info += "  Enabled: " + option.isEnabled() + "\n";
		info += "  Enabled Condition: " + option.getEnabledCondition() + "\n";
		info += "  Required: " + option.isRequired() + "\n";
		info += "  Vary: " + option.getVary() + "\n";
		info += "  Action: " + option.getAction() + "\n";
		info += "  Dest: " + option.getDest() + "\n";
		System.out.println(info);
	}

}
