/* StatisticalTools.c
 * 
 * A set of statistical tools based on the GSL library exported to Python.
 * Maintained by G.Winter
 * 19th January 2005
 * 
 * Implements
 * mean(list) => float
 * sd(list) => float
 * t(N, alpha) => float
 * 
 * 
 * 
 * 
 * $Id: StatisticalTools.c,v 1.3 2005/11/22 13:38:15 svensson Exp $
 */

#include <gsl/gsl_cdf.h>
#include <gsl/gsl_statistics_double.h>
#include <math.h>
#include "Python.h"

static PyObject * Mean(PyObject * self,
		       PyObject * args)
{
  PyObject * list;

  list = PyTuple_GetItem(args, 0);

  if (PyList_Check(list))
    {
      int i, length;
      double * values, mean;

      length = PyList_Size(list);
      
      values = (double *) malloc (length * sizeof(double));

      for (i = 0; i < length; i++)
	{
	  values[i] = PyFloat_AsDouble(PyList_GetItem(list, i));
	}

      mean = gsl_stats_mean(values, 1, length);

      free((void *) values);

      return PyFloat_FromDouble(mean);
    }
  else
    {
      PyErr_SetString(PyExc_TypeError, "Argument should be a list");
      return NULL;
    }

}

static PyObject * Sd(PyObject * self,
		     PyObject * args)
{
  PyObject * list;

  list = PyTuple_GetItem(args, 0);

  if (PyList_Check(list))
    {
      int i, length;
      double * values, sd;

      length = PyList_Size(list);
      
      values = (double *) malloc (length * sizeof(double));

      for (i = 0; i < length; i++)
	{
	  values[i] = PyFloat_AsDouble(PyList_GetItem(list, i));
	}

      sd = gsl_stats_sd(values, 1, length);

      free((void *) values);

      return PyFloat_FromDouble(sd);
    }
  else
    {
      PyErr_SetString(PyExc_TypeError, "Argument should be a list");
      return NULL;
    }

}

static PyObject * T(PyObject * self,
		    PyObject * args)
{
  double N, alpha, t;
  if (!PyArg_ParseTuple(args, "dd", &N, &alpha))
    {
      return NULL;
    }

  t = gsl_cdf_tdist_Qinv(alpha, N);

  return PyFloat_FromDouble(t);
}


static PyMethodDef StatisticalToolsMethods[] = {
  {"mean", Mean, METH_VARARGS, "Calculate the mean value."},
  {"sd", Sd, METH_VARARGS, "Calculate the standard deviation."},
  {"t", T, METH_VARARGS, "Calculate the critical T value for N degrees "
   "of freedom to significance alpha."},
  {NULL, NULL, 0, NULL}
};

PyMODINIT_FUNC
initStatisticalTools(void)
{
  (void) Py_InitModule("StatisticalTools", StatisticalToolsMethods);
}

