#!/usr/bin/env python
# Pointless.py
# Maintained by G.Winter
# 5th July 2005
# 
# An interface to the program "pointless" developed by Phil Evans
# (pre@mrc-lmb.cam.ac.uk), program available from
# 
# ftp://ftp.mrc-lmb.cam.ac.uk/pub/pre/pointless.linux
# 
# $Id: Pointless.py,v 1.4 2006/01/25 16:22:41 gwin Exp $

import os, sys

if not os.environ.has_key('XIAHOME'):
    raise RuntimeError, 'XIAHOME not defined'

if not os.environ.has_key('CCP4'):
    raise RuntimeError, 'CCP4 not defined'

# add the Driver class to the path

sys.path.append(os.path.join(os.environ['XIAHOME'],
                             'scheduler', 'Scheduler', 'Driver'))

sys.path.append(os.path.join(os.environ['XIAHOME'],
                             'XIA-Lib', 'CCP4'))
sys.path.append(os.path.join(os.environ['XIAHOME'],
                             'XIA-DP', 'Indexers', 'XDS'))

# import the symmetry and mtz classes
from MtzLib import Mtz
from SymmetryLib import SymInfoLib
from XDSCrystallography import constrain_cell_by_spacegroup
import PointlessHelper

import Driver

class PointlessWrapper(Driver.Driver):
    '''A wrapper for the program pointless developed by Phil Evans
    for determining Laue group from unmerged intensities.'''

    def __init__(self):
        Driver.Driver.__init__(self)

        self.setExecutable('pointless')
        self.hklin = None

    def setHklin(self, hklin):
        '''Set the input reflection file'''
        
        # check that the file exists
        if not os.path.exists(hklin):
            raise RuntimeError, 'file not found: %s' % hklin

        hklout = '%s.pls' % hklin

        # rebatch the file to just include 15 degrees of data
        rebatch = PointlessHelper.LimitBatches15Deg(hklin, hklout)

        if not rebatch:
            self.hklin = hklin
        else:
            self.hklin = hklout

        return None

    def run(self):
        return self.pointless()

    def pointless(self):
        '''Actually find the correct Laue group'''

        if not self.hklin:
            raise RuntimeError, 'HKLIN not defined'

        self.start('HKLIN %s' % self.hklin)

        # kill the standard input
        self.close()

        spacegroup = None
        found_solution = False
        reindex_operator = None

        collect_Z = False
        Z_scores = { } 
        conf = { }

        while 1:
            line = self.output()

            if not line:
                break

            # keep this so that the character positions are known
            line_orig = line
            
            line = line.strip()
            list = line.split()

            # now grep the output for the information I want

            # this is all about finding the right point group

            if found_solution and len(line) > 0 and line[0] == '<':
                # get the first spacegroup from the line
                spacegroup = line.split('> <')[0].replace('<', '').replace(
                    '>', '')

            if list[:2] == ['>>>>>', '1']:
                reindex_operator = line.split(
                    '[')[1].split(']')[0].replace(',', ', ')
                found_solution = True
            if list[:2] == ['>>>>>', '2']:
                found_solution = False

            # this is all about establishing the Z scores

            if list[:3] == ['More', 'information', 'about']:
                collect_Z = False

            if collect_Z:
                if len(line):
                    index = int(line_orig[:2])
                    Z = float(line_orig[19:25])
                    Z_scores[index] = Z
                    conf[index] = line[14:24].count('*')

            # mod to work with pointless 0.5.2 - the DNA 1.1 version
            if list[:3] == ['Laue', 'Group', 'NetZcc'] or \
                   list[:3] == ['Laue', 'Group', 'NetZc']:
                collect_Z = True


        self.kill()

        keys = Z_scores.keys()
        keys.sort()

        if False:

            print 'Z scores    confidence'
            
            for k in keys:
                print '%d   %5.2f  %1d' % (k, Z_scores[k], conf[k])
                
            print

        # there is some more important information here - the confidence
        # if this is 0 for the best solution (e.g. conf[1] == 0)
        # then more data will be needed

        return spacegroup, reindex_operator, conf[1]

class ReindexWrapper(Driver.Driver):
    '''A wrapper for the CCP4 program reindex'''

    def __init__(self):
        Driver.Driver.__init__(self)

	self.setExecutable(os.path.join(os.environ['CCP4'], 'bin',
                                        'reindex'))
        self.hklin = None
        self.hklout = None
        self.spacegroup = None
        self.reindex_operator = None

    def setHklin(self, hklin):
        # first should check that hklin is ok
        self.hklin = hklin

    def setHklout(self, hklout):
        self.hklout = hklout

    def setSpacegroup(self, spacegroup):
        self.spacegroup = spacegroup

    def setReindex_operator(self, reindex_operator):
        self.reindex_operator = reindex_operator

    def run(self):
        self.reindex()

    def reindex(self):
        '''Actually perform the reindexing operation'''

        if not self.hklin:
            raise RuntimeError, 'HKLIN not defined'

        if not self.hklout:
            raise RuntimeError, 'HKLOUT not defined'

        if not self.spacegroup:
            raise RuntimeError, 'spacegroup not defined'

        self.start('HKLIN %s HKLOUT %s' % \
                   (self.hklin, self.hklout))

        self.input('SYMMETRY \'%s\'' % self.spacegroup)
        if self.reindex_operator:
            self.input('REINDEX %s' % self.reindex_operator)

        self.close()

        while 1:

            line = self.output()

            if not line:
                break

        self.kill()

        return None

def Reindex(hklin, hklout, spacegroup, reindex_operator):
    '''Reindex from HKLIN to HKLOUT assigning spacegroup
    and reindexing operator'''

    rw = ReindexWrapper()

    rw.setHklin(hklin)
    rw.setHklout(hklout)
    rw.setSpacegroup(spacegroup)
    rw.setReindex_operator(reindex_operator)

    return rw.reindex()

def RegulariseCell(spacegroup, cell):
    '''Taking a spacegroup OBJECT and a unit cell as a list of 6 numbers
    return a regularised list of cell constants, that is constrained
    by the symmetry rules for that lattice.'''

    return constrain_cell_by_spacegroup(spacegroup.get_number(), cell)

def CorrectPointgroup(hklin, hklout):
    '''Determine the correct point group from hklin, and write
    to hklout if necessary (that is, if the correct point group is
    different to the one recorded). If this is the case, the True
    will be returned. If hklout is not used, False will be returned.
    In either case both the initial and selected spacegroups will be
    returned for information, and if the lattice is correct but the
    pointgroup incorrect the data will be nevertheless reindexed.'''

    finished = True

    syminfo = SymInfoLib()
    mtz = Mtz(hklin)

    if not finished:
        raise RuntimeError, 'this routine is not finished'
    
    p = PointlessWrapper()
    p.setHklin(hklin)
    spacegroup, reindex_operator, confidence = p.run()

    if confidence == 0:
        # raise RuntimeError, 'correct pointgroup uncertain'
        print 'Warning: point group is uncertain (confidence = 0)'

    # get and compare the spacegroup operators
    spag_in = mtz.get_spacegroup()
    spag_out = syminfo.get_spacegroup(spacegroup)

    if reindex_operator == 'h, k, l' and \
           spag_in.get_pointgroup() == spag_out.get_pointgroup():
        return False, spag_in.get_symbol('old')[0], \
               spag_out.get_symbol('old')[0]
    elif reindex_operator == 'h, k, l':
        # lattice is assigned correctly but spacegroup needs
        # assignment correctly in the header - likely
        # case is P4 -> P422

        pass

    Reindex(hklin, hklout, spacegroup, reindex_operator)

    new_unit_cell = Mtz(hklout).get_cell().get()
    new_unit_cell = constrain_cell_by_spacegroup(spag_out.get_number(),
                                                 list(new_unit_cell))

    print 'Constrained cell should be:'
    print '%6.2f %6.2f %6.2f %6.2f %6.2f %6.2f' % \
          tuple(new_unit_cell)
    print 'This will need to be implemented later on by CAD'
    print 'cad HKLIN1 foo HKLOUT bar << eof'
    print 'labin file 1 all'
    print 'dcell 1 1 %6.2f %6.2f %6.2f %6.2f %6.2f %6.2f' % \
          tuple(new_unit_cell)
    print 'eof'
    
    return True, spag_in.get_symbol('old')[0], \
           spag_out.get_symbol('old')[0]

if __name__ == '__main__':

    if len(sys.argv) < 2:
        raise RuntimeError, '%s hklin [hklout]' % sys.argv[0]

    if len(sys.argv) == 2:

        hklin = sys.argv[1]
    
        p = PointlessWrapper()
        p.setHklin(hklin)
        spacegroup, reindex_key, confidence = p.run()

        print '%s (%d)' % (spacegroup, confidence)

    elif len(sys.argv) == 3:

        hklin = sys.argv[1]
        hklout = sys.argv[2]

        reindex, initial, correct = CorrectPointgroup(hklin, hklout)

        print '%s %s %s' % (str(reindex), initial, correct)
