#!/usr/bin/env python
# BeamCentreAcuracy.py
# Maintained by G.Winter
# 24th January 2006
# 
# A jiffy to compute the difference between the header beam centre and the
# beam centre refined by labelit.

import os, sys, math
import re, string
import exceptions

environ_key = 'DNAHOME'

if not os.environ.has_key(environ_key):
    raise RuntimeError, '%s not defined' % environ_key

environ = os.environ[environ_key]

# Provide access to the Driver module - this should be all we need for this
# case, since this is designed to be a closed module.
sys.path.append(os.path.join(environ, 'scheduler', 'Scheduler', 'Driver'))
from Driver import Driver, DriverException

# also need DiffractionImage
sys.path.append(os.path.join(environ, 'scheduler', 'DiffractionImage',
                             'lib'))
from DiffractionImage import DiffractionImage

def image_to_template_and_directory(image):
    '''Take the name of an image (optionally full path) and return a
    suitable template and directory for use elsewhere'''

    # looking for foo_bar_1_nnn.exten

    regexp = re.compile(r'(.*)/(.*)_([0-9]*)\.(.*)')
    match = regexp.match(image)
    if match:
        directory = match.group(1)
        prefix = match.group(2)
        number = match.group(3)
        extension = match.group(4)
        for d in string.digits:
            number = number.replace(d, '#')
        template = prefix + '_' + number + '.' + extension
        return template, directory
    else:
        # perhaps there was no directory in the expression - try without
        directory = os.getcwd()
        regexp = re.compile(r'(.*)_([0-9]*)\.(.*)')
        match = regexp.match(image)
        prefix = match.group(1)
        number = match.group(2)
        extension = match.group(3)
        for d in string.digits:
            number = number.replace(d, '#')
        template = prefix + '_' + number + '.' + extension
        return template, directory    

def get_image_list(template, directory):
    '''Get a list of images (as integers) which match template in
    directory'''

    files = os.listdir(directory)

    # compose a regular expression to find these files

    expression = template.replace('#', '([0-9]*)', 1)
    expression = expression.replace('#', '')
    regexp = re.compile(expression)

    images = []
    
    for file in files:
        match = regexp.match(file)
        if match and len(template) == len(file):
            images.append(int(match.group(1)))

    # always tidy to return things in order
    
    images.sort()
    return images

def template_and_directory_and_number_to_image(template, directory, image):
    '''Given a template, directory and a number create a full path to an
    image'''

    expression = r'(.*)_(\#*)\.(.*)'
    regexp = re.compile(expression)
    match = regexp.match(template)
    prefix = match.group(1)
    extension = match.group(3)
    length = len(match.group(2))
    
    format = '%0' + str(length) + 'd'
    
    number = format % image
    filename = directory + '/' + prefix + '_' + number + '.' + extension

    return filename

def get_header(image):
    '''Read the header information from this image to a dictionary.'''
    try:
        # I think that the 0 means "just open the header"
        # this is correct (checked!)
        d = DiffractionImage(image, 0)
    except exceptions.Exception, e:
        raise RuntimeError, 'Error %s opening image %s' % \
              (e, image)

    header = { }
    header['time'] = d.getExposureTime()
    header['beam_x'] = d.getBeamX()
    header['beam_y'] = d.getBeamY()
    header['pix_x'] = d.getPixelX()
    header['pix_y'] = d.getPixelY()
    header['width'] = d.getWidth()
    header['height'] = d.getHeight()

    # not reading the date information - you have been warned! The
    # canonical implementation is not this one...

    # check to see if these are in mm or in pixels - assume
    # that this is somewhere near the middle of the
    # detector - so if value > detector width in mm then * pixel
    # size

    if d.getBeamX() > (d.getWidth() * d.getPixelX()) and \
           d.getBeamY() > (d.getHeight() * d.getPixelY()):
        header['beam_x'] = d.getBeamX() * d.getPixelX()
        header['beam_y'] = d.getBeamY() * d.getPixelY()
    elif d.getBeamX() > (d.getWidth() * d.getPixelX()) or \
             d.getBeamY() > (d.getHeight() * d.getPixelY()):
        raise RuntimeError, 'inconsistent beam centre values: %f %f' % \
              (d.getBeamX(), d.getBeamY())

    header['dist'] = d.getDistance()
    header['wave'] = d.getWavelength()
    messages = d.getMessage()
    header['message'] = messages.split(';')
    header['start'] = d.getPhiStart()
    header['end'] = d.getPhiEnd()
    header['osc'] = d.getPhiEnd() - d.getPhiStart()

    # not including the squareness of the diffraction image here

    bytes = open(image, 'r').read(4)

    if bytes == '{\nHE':
        header['format'] = 'smv'
    elif bytes[:2] == 'II':
        header['format'] = 'tiff'
    elif bytes[:2] == 'RA':
        header['format'] = 'raxis'
    elif bytes == '\xd2\x04\x00\x00':
        header['format'] = 'marip'
        
    del(d)

    return header

def nint(a):
    b = int(a)
    if a - b > 0.5:
        return b + 1
    return b

def dynamicUnitTest(image):
    # this one can be run with just a pointer to an image from
    # within the set e.g. /tmp/frames/postref-bert_1_001.img

    # parse image to template and directory
    template, directory = image_to_template_and_directory(image)
    
    # get list of available images
    images = get_image_list(template, directory)

    # construct the image name for the first image
    first_image = template_and_directory_and_number_to_image(template,
                                                             directory,
                                                             images[0])

    # get the header information for the first image in the list
    header = get_header(first_image)

    last = images[0] + nint(90.0 / header['osc'])
    if not last in images:
        last = images[-1]

    last_image = template_and_directory_and_number_to_image(template,
                                                            directory,
                                                            last)

    beam = LabelitBeam(first_image, last_image)

    # print out the beam and offset now

    diff_sq = ((header['beam_x'] - beam[0]) * (header['beam_x'] - beam[0])) + \
              ((header['beam_y'] - beam[1]) * (header['beam_y'] - beam[1]))

    offset = math.sqrt(diff_sq)

    print 'header: %5.2f %5.2f refined: %5.2f %5.2f difference %5.2f' % \
          (header['beam_x'], header['beam_y'],
           beam[0], beam[1], offset)

    return

# use labelit to get the correct direct beam position using the
# --index_only option - optionally!

class LabelitBeamWrapper(Driver):
    '''A really lightweight driver class to use Labelit to get the
    correct beam centre ONLY - this will not actually be used to provide
    anything useful in terms of the actual indexing.'''

    def __init__(self):
        Driver.__init__(self)
        self.setExecutable('labelit.screen')

        self.images = None

    def setImages(self, first, second):
        self.images = (first, second)

    def run(self):
        '''Actually run labelit to get the "correct" beam centre out...'''

        if not self.images:
            raise RuntimeError, 'images not assigned'

        self.start('--index_only %s %s' % self.images)
        self.close()

        beam = (0.0, 0.0)

        while 1:
            line = self.output()

            if not line:
                break

            if line[:13] == 'Beam center x':
                list = line.split()
                x = float(list[3].replace('mm,', ''))
                y = float(list[5].replace('mm,', ''))
                beam = (x, y)

                

        self.kill()

        # print 'Correct beam found to be: %6.2f %6.2f' % beam

        return beam

def LabelitBeam(first, second):
    '''Use labelit to compute the direct beam from these images.'''
    lbw = LabelitBeamWrapper()
    lbw.setImages(first, second)
    return lbw.run()

if __name__ == '__main__':
    
    dynamicUnitTest(sys.argv[1])
