#!/usr/bin/env python
# BESTWrapper.py
# Maintained by G.Winter
# 6th February 2006
# 
# A new wrapper for BEST version 3.0.
# 
# This is to provide strategies for data collection for DNA 1.1 and for
# the e-HTPX strategy service - note well that this is very different to
# how things worked before, where the strategies were computed by Mosflm
# then verified and fiddled with by BEST.
# 
# Inputs:
# Desired I/sigma at the inscribed circle detector resolution.
# Redundancy   [0,inf]
# Completeness [0,1]
# Anomalous    [boolean]
# Images2Use   [image 1]
# Strategy     [input strategy - for statistics computation]
# Outputs:
# Phi start, end
# Statistics
# 
# New inputs to BEST (interesting first):
# -i2s I    I/sigma value (defaults to 2)
# -e none   anisotropy correction off
# -R r -C c redundancy (multiplicity) and completeness - default to minimum 
#           and maximum respectively
# -a        anomalous on
#
# Boring ones:
# -T/-S/-M  hardware limitations - total exposure/rotation
#           speed/min exposure
# 
# Old ones:
# -r r      resolution/A
# -f str    detector name
# -t exp    exposure time for measurement
# -dna stf  .stf output file
# -mos      mosflm on
#
# Input files:
# .dat .par .hkl .hkl ... -> examples are test.* in this directory.
#
# Input files created by Mosflm - note well that the spacegroup may need to
# be changed in some of these to get the strategy right for higher spacegroups
# for a given lattice

# things we need from Python

import os
import sys
import math
import copy

# check the environment, set up paths &c.

if not os.environ.has_key('DNAHOME'):
    raise RuntimeError, 'DNAHOME not defined'

sys.path.append(os.path.join(os.environ['DNAHOME'],
                             'scheduler',
                             'Scheduler',
                             'Driver'))
sys.path.append(os.path.join(os.environ['DNAHOME'],
                             'scheduler',
                             'Scheduler',
                             'Mosflm'))
sys.path.append(os.path.join(os.environ['DNAHOME'],
                             'xsd',
                             'python'))

# things we need from DNA

from Driver import Driver
from Output import Output
from Somewhere import Somewhere
from Messenger import Messenger

import XSD

# helper functions

# note well, that this requires some analysis of the detector-inf.dat file
# which comes with BEST - critically the type of detector used needs to
# be available to the program - at the moment the supported detectors are
# 
# ADSC Q4 Q210 Q315
# Mar 165, 225 [ccd detectors]
# 
# Named q4, q210, q312, mar165, mar225 in the detector-inf.dat file
# along with their 2x binned counterparts
# note that the detector-inf.dat entries should look like the entries in
# the best detector infom file.

def guess_detector_type(header):
    '''Determine a detector type from the list given above, based on an
    image header dictionary.'''

    if not header.has_key('Format'):
        raise RuntimeError, 'image format unknown'

    Messenger.log_write('Format: %s Width: %d Pixel: %f' % \
                        (header['Format'], header['Width'],
                         header['PixelX']), 2)

    if header['Format'] == 'smv':
        # then this is an ADSC style detector
        if header['Width'] == 2304 and \
           math.fabs(header['PixelX'] - 0.0816) < 0.001:
            return 'q4'
        if header['Width'] == 2048 and \
           math.fabs(header['PixelX'] - 0.1024) < 0.001:
            return 'q210-2x'
        if header['Width'] == 3072 and \
           math.fabs(header['PixelX'] - 0.1024) < 0.001:
            return 'q315-2x'
        if header['Width'] == 4096 and \
           math.fabs(header['PixelX'] - 0.0512) < 0.001:
            return 'q210'
        if header['Width'] == 6144 and \
           math.fabs(header['PixelX'] - 0.0512) < 0.001:
            return 'q315'
        raise RuntimeError, 'unknown detector format'

    if header['Format'] == 'tiff':
        if header['Width'] == 4096 and \
           math.fabs(header['PixelX'] - 0.040) < 0.002:
            return 'mar165'
        if header['Width'] == 2048 and \
           math.fabs(header['PixelX'] - 0.080) < 0.002:
            return 'mar165'
        if header['Width'] == 3072 and \
           math.fabs(header['PixelX'] - 0.073) < 0.001:
            return 'mar225'
        if header['Width'] == 4096 and \
           math.fabs(header['PixelX'] - 0.079) < 0.001:
            return 'mar325'

        raise RuntimeError, 'unknown detector format'

    if header['Format'] == 'raxis':
        return 'raxis'

    if header['Format'] == 'marip':
        if math.fabs(header['Width'] * header['PixelX'] - 345.0) < 1.0:
            return 'mar345'

    raise RuntimeError, 'unknown file format'

def Strategy_response(output):
    '''Convert an output best.stf file into DNA'''

    # check for overloaded reflections in the strategy - if there are 
    # overloaded reflectiosn then we need to reduce the exposure time
    # - maybe by a factor of two...

    bins = []
    for j in output.searchforlist('statistical_prediction', \
                                  'resolution_bin_'):
        i = int(j)
        bins.append(i)
        
    bins.sort()
        
    # take off the last bin
    bin = bins[-1]
    overload = float(output.get('statistical_prediction', \
                                'resolution_bin_%d' % bins[0], \
                                'fract_overload')[0])

    # this is in percent! this is also rather arbitrary! FIXME
    if overload > 1.00:
        strategy_response = XSD.Strategy_response()
        status = XSD.Status()
        status.setCode('error')
        status.setMessage('Detector overloaded %f' % overload)
        strategy_response.setStatus(status)
        return strategy_response

    # 24/APR/06 include predictions of the statistics of this set
    strategy_statistics = XSD.Strategy_statistics()

    #  <xsd:complexType name="strategy_statistics">
    #    <xsd:sequence>
    #      <xsd:element name="r_merge" type="xsd:double"/>
    #      <xsd:element name="i_over_sigma" type="xsd:double"/>
    #      <xsd:element name="overloads" type="xsd:double"/>
    #      <xsd:element name="max_overloads" type="xsd:double"/>
    #      <xsd:element name="multiplicity" type="xsd:double"/>
    #      <xsd:element name="completeness" type="xsd:double"/>
    #      <xsd:element name="resolution" type="xsd:double"/>
    #    </xsd:sequence>
    #  </xsd:complexType>
    
    # overall overloads
    
    strategy_statistics.setOverloads(
        float(output.get('statistical_prediction', \
                         'resolution_bin_%d' % bins[-1], \
                         'fract_overload')[0]))
    
    # maximum - assumed to be lowest resolution bin
    
    strategy_statistics.setMax_overloads(
         float(output.get('statistical_prediction', \
                          'resolution_bin_%d' % bins[0], \
                          'fract_overload')[0]))
    
    # r merge
    
    strategy_statistics.setR_merge(
         float(output.get('statistical_prediction', \
                          'resolution_bin_%d' % bins[-1], \
                          'R_factor')[0]))
    
    # completeness
    
    strategy_statistics.setCompleteness(
         float(output.get('data_collection_strategy', \
                          'summary_1', \
                          'completeness')[0]))
    
    # multiplicity
    
    strategy_statistics.setMultiplicity(
         float(output.get('data_collection_strategy', \
                          'summary_1', \
                          'redundancy')[0]))
    
    # resolution
    
    strategy_statistics.setResolution(
         float(output.get('data_collection_strategy', \
                          'summary_1', \
                          'resolution')[0]))
    
    # i / sigma
    
    strategy_statistics.setI_over_sigma(
         float(output.get('data_collection_strategy', \
                          'summary_1', \
                          'i_sigma')[0]))
    
    
    strategy_response = XSD.Strategy_response()

    strategy_response.setStrategy_statistics(strategy_statistics)
    
    status = XSD.Status()
    status.setCode('ok')
    strategy_response.setStatus(status)
    completeness = XSD.Completeness()
    completeness.setStandard(output.get('general_inform', \
                                        'rel_achievable_refl_1', \
                                        'fraction_achievable_%')[0])
    strategy_response.setCompleteness(completeness)

    runs = []

    for j in output.searchforlist('data_collection_strategy',
                                  'collection_run_'):
        i = int(j)
        runs.append(i)
            
    runs.sort()

    # we should have exactly one "run" at the moment!
    if len(runs) > 0:
        i = runs[0]
        start = float(output.get('data_collection_strategy',
                                 'collection_run_%d' % i, 'phi_start')[0])
        # this is really an int but...
        number = float(output.get('data_collection_strategy',
                                  'collection_run_%d' % i, 'number_of_images')[0])
        width = float(output.get('data_collection_strategy',
                                 'collection_run_%d' % i, 'phi_width')[0])
        time = float(output.get('data_collection_strategy',
                                'collection_run_%d' % i, 'exposure_time')[0])
    else:
        # we put in a strategy
        start = float(output.get('statistical_prediction',
                                 'user_data_collection_strategy_1', \
                                 'phi_start')[0])
        number = float(output.get('statistical_prediction',
                                  'user_data_collection_strategy_1', \
                                  'number_of_images')[0])
        end = float(output.get('statistical_prediction',
                               'user_data_collection_strategy_1', \
                               'phi_end')[0])
        width = (end - start) / number

    end = start + (number * width)

    strategy_summary = XSD.Strategy_summary()
    strategy_summary.setNumber_of_segments(1)
    segment = XSD.Segment()
    oscillation = XSD.Oscillation_sequence()
    prediction = XSD.Predicted_spots()

    # FIXME this isn't really good enough

    oscillation.setStart(start)
    oscillation.setRange(width)
    oscillation.setNumber_of_images(number)
    oscillation.setOverlap(0.0)
    oscillation.setExposure_time(time)
    oscillation.setStart_image_number(1)
    oscillation.setNumber_of_passes(1)
    
    # this is in the statistics... maybe I should get it - though from where?
    prediction.setFull(0.0)
    prediction.setOverlap(0.0)
    
    segment.setPredicted_spots(prediction)
    segment.setOscillation_sequence(oscillation)

    strategy_interpretation = XSD.Strategy_interpretation()
    strategy_interpretation.addOscillation_sequence(oscillation)

    strategy_response.setStrategy_interpretation(strategy_interpretation)
    
    strategy_response.addSegment(segment)
    strategy_summary.addSegment(segment)
    strategy_response.addStrategy_summary(strategy_summary)

    return strategy_response

class BESTWrapper(Driver):
    '''A class to wrap the latest version of BEST, for the calculation
    of data collection strategies and statistics from user defined
    strategies.'''

    def __init__(self):
        Driver.__init__(self)
        self.setExecutable('best')

        # input files

        self.dat_file = ''
        self.par_file = ''
        self.hkl_file_list = []

        # input parameters - default values which represent the likely
        # minimum

        self.completeness = 0.99
        self.multiplicity = 2.0
        self.i_over_sig = 3.0
        self.resolution = 0.0

        # input flags - do we want to use anisotropy correction
        # and/or anomalous data collection

        self.anisotropy = False
        self.anomalous = False

        # the detector ID & exposure time used for the reference frames
        self.detector_id = None
        self.exposure_time = 0.0

        # user defined phi ranges and so on - for testing of strategies
        self.user_phi_start = 0.0
        self.user_phi_end = 0.0
        self.user_phi_width = 0.0
        self.user_exposure_time = 0.0

        # debugging tools
        self.print_stats = False

        # beamline hardware limitations & user prejudices
        self.bl_min_exposure_time = 0.0
        self.bl_max_total_exposure = 0.0
        self.bl_min_oscillation = 0.0
        self.bl_max_phi_speed = 0.0
        self.bl_min_phi_speed = 0.0
        
        return

    # set methods for input files

    def set_parameter_files(self, dat_file, par_file):
        '''Set the parameter files for input - the data file (image
        background) and the parameter file (unit cell, spacegroup &c.)'''

        self.dat_file = dat_file
        self.par_file = par_file

        # check that the input files are the correct way aroud and
        # of the correct format
        self.verify_input_files()

        return

    def add_hkl_file(self, hkl_file):
        '''Add a hkl file to the list - these are used for intensity
        analysis.'''

        self.verify_hkl_file(hkl_file)
        self.hkl_file_list.append(hkl_file)

        return

    # set methods for the input parameters

    def set_completeness(self, completeness):
        '''Set the completeness - in the range [0.0, 1.0].'''
        
        if completeness > 1.00 or completeness < 0.00:
            raise RuntimeError, 'completeness not in range [0.0, 1.0]: %f' % \
                  completeness

        self.completeness = completeness

        # add a ceiling on the completeness value
        if self.completeness > 0.99:
            self.completeness = 0.99

        return

    def set_multiplicity(self, multiplicity):
        '''Set the multiplicity, in the range [0.0, inf]'''

        if multiplicity < 0.0:
            raise RuntimeError, 'multiplicity not > 0.0: %f' % multiplicity

        self.multiplicity = multiplicity

        return

    def set_i_over_sig(self, i_over_sig):
        '''Set the required I/sigma value at the edge.'''

        if i_over_sig < 0.0:
            raise RuntimeError, 'I/sigma < 0.0'

        self.i_over_sig = i_over_sig

        return

    def set_resolution(self, resolution):
        '''Set the resolution at which the above I/sigma is wanted.'''

        if resolution < 0.0:
            raise RuntimeError, 'resolution < 0.0'

        self.resolution = resolution

        return

    def set_anisotropy(self, anisotropy = True):
        '''Switch the anisotropy [default on].'''

        self.anisotropy = anisotropy

        return

    def set_anomalous(self, anomalous = True):
        '''Switch the anomalous [default on].'''

        self.anomalous = anomalous

        return

    # set the detector ID

    def set_detector(self, detector_id):
        '''Set the detector ID to a value from the detector info file.'''

        # check that this detector exists

        lines = open('%s/detector-inf.dat' % os.environ['besthome'],
                     'r').readlines()

        start = lines.index('----------------------------------------------------------------------------------------------\n') + 1
        end = lines.index('end\n')

        known_detectors = ''

        for l in lines[start:end]:
            detector = l.split()[0]
            if detector_id == detector:
                self.detector_id = detector_id
                return

            known_detectors = '%s %s' % (known_detectors, detector)

        raise RuntimeError, 'detector "%s" unknown (%s)' % \
              (detector_id, known_detectors)

    def guess_detector(self, header):
        self.set_detector(guess_detector_type(header))

    # and the exposure time used for the reference frames

    def set_exposure_time(self, exposure_time):
        '''Set the exposure time used for collecting the reference frames.'''

        if exposure_time < 0.0:
            raise RuntimeError, 'negative exposure time'

        self.exposure_time = exposure_time

        return

    # the user defined strategy input
    def set_user_strategy(self, phi_start, phi_end, phi_width, exposure_time):
        '''Set a user defined strategy for calculating the statistics of
        data collection with.'''

        # verify that these numbers are good
        if exposure_time <= 0.0:
            raise RuntimeError, 'bad exposure time'

        if phi_width <= 0.0:
            raise RuntimeError, 'bad oscillation width'

        if math.fabs(phi_end - phi_start) < phi_width:
            raise RuntimeError, 'no rotation range given'

        self.user_phi_start = phi_start
        self.user_phi_end = phi_end
        self.user_phi_width = phi_width
        self.user_exposure_time = exposure_time

        return

    # a couple of getter methods, for getting the results
    def get_strategy(self):
        '''Get a dictionary with the strategy in.'''
        return copy.deepcopy(self.strategy)

    def get_statistics(self):
        '''get a dictionary with the statistics in.'''
        return copy.deepcopy(self.statistics)

    # user prejudice and (baked) beanline limitations

    def setBl_min_exposure_time(self, bl_min_exposure_time):
        self.bl_min_exposure_time = bl_min_exposure_time
        return

    def setBl_max_total_exposure(self, bl_max_total_exposure):
        self.bl_max_total_exposure = bl_max_total_exposure
        return

    def setBl_min_oscillation(self, bl_min_oscillation):
        self.bl_min_oscillation = bl_min_oscillation
        return

    def getBl_min_oscillation(self):
        return self.bl_min_oscillation

    def setBl_max_phi_speed(self, bl_max_phi_speed):
        self.bl_max_phi_speed = bl_max_phi_speed
        return

    def setBl_min_phi_speed(self, bl_min_phi_speed):
        self.bl_min_phi_speed = bl_min_phi_speed
        return

    # that's all of the getter and setter methods - now some "real" work

    def verify_hkl_file(self, hkl_file):
        '''Check that the contents of a hkl file look ok.'''

        f = open(hkl_file, 'r')

        lines = f.readlines()

        # check the first and last lines of the file look like
        #   32   17  -30     101.90      75.54

        l = lines[0].split()
        if len(l) != 5:
            raise RuntimeError, 'bad hkl file: %s' % hkl_file

        l = lines[-1].split()
        if len(l) != 5:
            raise RuntimeError, 'bad hkl file: %s' % hkl_file

        # presume all is well...
        return

    def verify_input_files(self):
        '''Check that the input background and parameter files look ok.'''

        if self.dat_file == '':
            raise RuntimeError, 'no dat file (background) set'

        if self.par_file == '':
            raise RuntimeError, 'no par file (parameters) set'

        # check the first line of the dat file is three numbers
        lines =  open(self.dat_file, 'r').readlines()

        l = lines[0].split()
        
        try:
            floats = map(float, l)
            if len(l) != 3:
                raise RuntimeError, 'dat file incorrect format'
        except ValueError, ve:
            raise RuntimeError, 'dat file incorrect format'

        # next verify the parameter file format is correct
        lines = open(self.par_file, 'r').readlines()

        if lines[0].strip() != '# parameter file for BEST':
            raise RuntimeError, 'parameter file incorrect format'

        # assume that all is well...
        
        return
        
    def calculate_strategy(self):
        '''Compute a strategy based on all of the above information.'''

        # check that all of the input is ok
        self.verify_input_files()

        if len(self.hkl_file_list) == 0:
            raise RuntimeError, 'no hkl files'

        if self.detector_id is None:
            raise RuntimeError, 'no detector set'

        if self.exposure_time == 0.0:
            raise RuntimeError, 'no exposure time set'

        if self.resolution == 0.0:
            raise RuntimeError, 'no resolution set'

        # write some stuff to the screen to keep el user happy
        # see bug # 1655

        Messenger.log_write('.')
        Messenger.log_write('Strategy requirements:')
        Messenger.log_write('I/sigma:       %f' % self.i_over_sig)
        Messenger.log_write('Completeness:  %f' % self.completeness)
        Messenger.log_write('Multiplicity:  %f' % self.multiplicity)
        Messenger.log_write('Resolution:    %f' % self.resolution)
        Messenger.log_write('Min Osc:       %f' % self.bl_min_oscillation)
        Messenger.log_write('.')

        command_line = '-f %s -i2s %f -t %f -C %f -R %f -r %f' % \
                       (self.detector_id,
                        self.i_over_sig,
                        self.exposure_time,
                        self.completeness,
                        self.multiplicity,
                        self.resolution)

        if self.anisotropy:
            command_line += ' -e full'
        else:
            command_line += ' -e none'

        if self.anomalous:
            command_line += ' -a'

        # add on the user prejudices / beamline limitations

        if self.bl_min_exposure_time > 0.0:
            command_line += ' -M %f' % self.bl_min_exposure_time

        if self.bl_max_total_exposure > 0.0:
            command_line += ' -T %f' % self.bl_max_total_exposure

        if self.bl_min_oscillation > 0.0:
            command_line += ' -w %f' % self.bl_min_oscillation

        if self.bl_max_phi_speed > 0.0:
            command_line += ' -S %f' % self.bl_max_phi_speed

        command_line += ' -dna best.stf'
        command_line += ' -mos %s %s' % (self.dat_file, self.par_file)

        for hkl in self.hkl_file_list:
            command_line += ' %s' % hkl

        self.start(command_line)

        self.close_kill()

        # program is run, now collect the output

        response = Strategy_response(
            Output(os.path.join(self.getWorkingDirectory(),
                                'best.stf')))

        # write some stuff from the strategy response to the screen, again to
        # keep the computer "looking busy" c/f bug # 1655

        if response.getStatus().getCode() != "ok":
            Messenger.log_write("There was a problem with the strategy:")
            Messenger.log_write(response.getStatus().getmessage())
            return response

        stats = response.getStrategy_statistics()

        # if we get to here we know everything's ok
        Messenger.log_write('.')
        Messenger.log_write('Strategy results:')
        Messenger.log_write('Completeness:   %f' %
                            float(response.getCompleteness().getStandard()))
        Messenger.log_write('Multiplicity:   %f' % stats.getMultiplicity())
        strat = response.getStrategy_interpretation()
        strat = strat.getOscillation_sequence()[0]
        
        Messenger.log_write('Phi start:      %f' % float(strat.getStart()))
        Messenger.log_write('Phi end:        %f' %
                            (float(strat.getStart()) +
                            (float(strat.getRange()) *
                             float(strat.getNumber_of_images()))))
        Messenger.log_write('Phi width:      %f' %
                            float(strat.getRange()))
        Messenger.log_write('Exposure time:  %f' %
                            float(strat.getExposure_time()))
        Messenger.log_write('I/sigma:        %f' % stats.getI_over_sigma())
        Messenger.log_write('R merge:        %f' % stats.getR_merge())
        Messenger.log_write('.')
        
        return response

    def calculate_statistics(self):
        '''Compute the statistics for a user defined strategy.'''

        # check that all of the input is ok
        self.verify_input_files()

        if len(self.hkl_file_list) == 0:
            raise RuntimeError, 'no hkl files'

        if self.detector_id is None:
            raise RuntimeError, 'no detector set'

        if self.exposure_time == 0.0:
            raise RuntimeError, 'no exposure time set'

        if self.resolution == 0.0:
            raise RuntimeError, 'no resolution set'

        # verify that we also have a user defined strategy - flag this by
        # the fact that the user_phi_width, user_exposure_time are set

        if self.user_exposure_time == 0.0:
            raise RuntimeError, 'exposure time not set'

        if self.user_phi_width == 0.0:
            raise RuntimeError, 'oscillation width not set'

        if math.fabs(self.user_phi_end - self.user_phi_start) < \
           self.user_phi_width:
            raise RuntimeError, 'no oscillation set'

        command_line = '-f %s -r %f -t %f' % \
                       (self.detector_id,
                        self.resolution,
                        self.exposure_time)

        # add the user defined strategy
        command_line += ' -phi %f %f -c %f %f' % \
                        (self.user_phi_start,
                         self.user_phi_end - self.user_phi_start,
                         self.user_phi_width,
                         self.user_exposure_time)
        
        if self.anomalous:
            command_line += ' -a'

        command_line += ' -dna best.stf'
        command_line += ' -mos %s %s' % (self.dat_file, self.par_file)

        for hkl in self.hkl_file_list:
            command_line += ' %s' % hkl

        self.start(command_line)

        self.close_kill()

        # program is run, now collect the output

        response = Strategy_response(
            Output(os.path.join(self.getWorkingDirectory(),
                                'best.stf')))

        return response

    def parse_strategy_output(self, output):
        '''Parse the output object containing the results from BEST.'''

        # get the data collection strategy stuff out

        index = output.searchforlist('dc_optimal_time', \
                                     'conditions_')[0]

        completeness = 0.01 * float(output.get('dc_optimal_time', \
                                               'conditions_%s' % index, \
                                               'completeness')[0])
        multiplicity = float(output.get('dc_optimal_time', \
                                        'conditions_%s' % index, \
                                        'redundancy')[0])
        phi_start = float(output.get('dc_optimal_time', \
                                     'conditions_%s' % index, \
                                     'phi_start')[0])
        phi_end = float(output.get('dc_optimal_time', \
                                     'conditions_%s' % index, \
                                     'phi_end')[0])

        index = output.searchforlist('data_collection_strategy',
                           'collection_run_')[0]

        exposure_time = float(output.get('data_collection_strategy',
                                         'collection_run_%s' % index,
                                         'exposure_time')[0])
        phi_width = float(output.get('data_collection_strategy',
                                     'collection_run_%s' % index,
                                     'phi_width')[0])
        num_images = int(output.get('data_collection_strategy',
                                    'collection_run_%s' % index,
                                    'number_of_images')[0])
        distance = float(output.get('data_collection_strategy',
                                    'collection_run_%s' % index,
                                    'distance')[0])

        # print out these results - just for testing purposes

        if self.print_stats:

            print ''
            print 'BEST data collection strategy for %6.2f%% completeness' % \
                  (100.0 * completeness)
            print 'and %5.2f multiplicity' % multiplicity
            print ''
            print 'Start:    %6.2f   End: %6.2f         Width %6.2f' % \
                  (phi_start, phi_end, phi_width)
            print 'Exp. Time: %6.3f' % exposure_time
            print ''
        

        # next get the predictions of the statistics from this data
        # collection run - this is another method

        strategy = { }
        strategy['phi_start'] = phi_start
        strategy['phi_end'] = phi_end
        strategy['phi_width'] = phi_width
        strategy['exposure_time'] = exposure_time
        strategy['completeness'] = completeness
        strategy['multiplicity'] = multiplicity
        strategy['distance'] = distance

        self.strategy = strategy

        return strategy
    
    def parse_statistics_output(self, output):
        '''Pull the statistics out of the BEST output XML/STF file.'''

        indices = output.searchforlist('statistical_prediction',
                                       'resolution_bin_')
        indices = map(int, indices)
        indices.sort()

        if self.print_stats:
            
            print 'Statistics:'
            print ' Dmax    Dmin   I/sig   R  Overload'

        max_overload = 0.0

        for i in indices[:-1]:
            dmax = float(output.get('statistical_prediction',
                                    'resolution_bin_%d' % i,
                                    'min_resolution')[0])
            dmin = float(output.get('statistical_prediction',
                                    'resolution_bin_%d' % i,
                                    'max_resolution')[0])
            i_over_sig = float(output.get('statistical_prediction',
                                    'resolution_bin_%d' % i,
                                    'average_i_over_sigma')[0])
            r = 0.01 * float(output.get('statistical_prediction',
                                        'resolution_bin_%d' % i,
                                        'R_factor')[0])
            overload = float(output.get('statistical_prediction',
                                        'resolution_bin_%d' % i,
                                        'fract_overload')[0])
            
            if max_overload < overload:
                max_overload = overload

            if self.print_stats:
                print '%6.2f %6.2f %7.2f %4.2f  %4.2f' % \
                      (dmax, dmin, i_over_sig, r, overload)

        if self.print_stats:

            print ''
            print 'Overall:'
            print ' Dmax    Dmin   I/sig   R  Overload'

        for i in indices[-1:]:
            dmax = float(output.get('statistical_prediction',
                                    'resolution_bin_%d' % i,
                                    'min_resolution')[0])
            dmin = float(output.get('statistical_prediction',
                                    'resolution_bin_%d' % i,
                                    'max_resolution')[0])
            i_over_sig = float(output.get('statistical_prediction',
                                    'resolution_bin_%d' % i,
                                    'average_i_over_sigma')[0])
            r = 0.01 * float(output.get('statistical_prediction',
                                        'resolution_bin_%d' % i,
                                        'R_factor')[0])
            overload = float(output.get('statistical_prediction',
                                        'resolution_bin_%d' % i,
                                        'fract_overload')[0])

            if self.print_stats:

                print '%6.2f %6.2f %7.2f %4.2f  %4.2f' % \
                      (dmax, dmin, i_over_sig, r, overload)
                print ''

        results = { }
        results['dmax'] = dmax
        results['dmin'] = dmin
        results['i_over_sig'] = i_over_sig
        results['r_factor'] = r
        results['overloads'] = overload
        results['max_overload'] = max_overload

        self.statistics = results

        return results

    def print_strategy(self, strategy = None):
        '''Print out the dictionary resulting from the strategy
        calculation.'''

        if strategy is None:
            strategy = self.strategy

        print 'Phi:        %6.2f to %6.2f / %6.2f' % \
              (strategy['phi_start'],
               strategy['phi_end'],
               strategy['phi_width'])
        print 'Exposure:   %6.3f' % strategy['exposure_time']
        print ' => %3d%% completeness / %6.2f multiplicity' % \
              (int(100 * strategy['completeness']),
               strategy['multiplicity'])

        return

    def print_statistics(self, statistics = None):
        '''Print out the contents of the dictionary with the stats in.'''

        if statistics is None:
            statistics = self.statistics

        print '  Resolution     I/sigma    R    Overloads '
        print '%6.2f %6.2f   %6.2f   %5.2f   %5.2f  [%5.2f]' % \
              (statistics['dmax'],
               statistics['dmin'],
               statistics['i_over_sig'],
               statistics['r_factor'],
               statistics['overloads'],
               statistics['max_overload'])
        
        return

def do_strategy_rq(working, exposure, strategy_rq):
    '''Actually perform the strategy request with information from the
    input, Somewhere and the local file system.'''

    # Input requirements
    # 
    # completeness Multiplicity i_over_sigma resolution anomalous
    # exposure time
    # 
    # are these all in the strategy request?
    # here is an example
    # <?xml version="1.0" encoding="ISO-8859-1"?>
    # <strategy_request>
    #   <strategy_settings>  <!-- anomalous boolean goes in here -->
    #   <resolution>
    #     <upper>3.950000</upper>
    #   </resolution>
    #   <overlap_limit>2.000000</overlap_limit>
    #   </strategy_settings>
    # </strategy_request>
    # 
    # So most of what I want isn't in there - oh.
    #
    # Default values:
    # completeness 1.0
    # multiplicity 1.0
    # anomalous false
    # i_over_sigma 3.0
    # exposure_time => will need access to image to establish this -
    #                  is it anywhere in the input files?
    # Add it (sent message to dna-support asking if this was ok - did it
    # before I got a response!)
    # 
    # Added the following to strategy_settings
    # 
    # optional double completeness
    # optional double multiplicity
    # optional double i_over_sigma
    # 
    # defaults as above will be statically implemented.

    # Where does the information come from???
    # hkl files from integrating images come from
    # 
    # Somewhere.get('best_hkl')
    # 
    # The par and dat files are
    # 
    # ../integrate_1/bestfile.par ../autoindex_all/bestfile.dat
    #
    # this is probably not optimal. 

    bw = BESTWrapper()

    bw.setWorkingDirectory(working)
    
    bw.set_parameter_files(os.path.join(os.path.dirname(working),
                                        'autoindex_all/bestfile.dat'),
                           os.path.join(os.path.dirname(working),
                                        'integrate_1/bestfile.par'))
    for hkl in Somewhere.get('best_hkl'):
        bw.add_hkl_file(hkl)

    settings = strategy_rq.getStrategy_settings()

    beamline = settings.getBeamline_parameters()

    # update 20/JUN/06 add a check for the user defined min osc.
    # this is settings.user_defined_minimum_phi_oscillation.

    # expecting in here:
    # 
    # A maximum_exposure
    # B minimum_exposure_time
    # C minimum_phi_speed
    # D maximum_phi_speed
    # E minimum_phi_oscillation
    # 
    # will use as the following in BEST 3.0
    # -w => E
    # -T => A
    # -M => B
    # -S => D
    # => C is not used (that's probably ok, encoded in B,E)
    #

    if beamline:
        max_exposure = beamline.getMaximum_exposure()
        min_exposure_time = beamline.getMinimum_exposure_time()
        min_phi_speed = beamline.getMinimum_phi_speed()
        max_phi_speed = beamline.getMaximum_phi_speed()
        min_oscillation = beamline.getMinimum_phi_oscillation()

        # if these are set, pass the values on...

        if max_exposure:
            bw.setBl_max_total_exposure(max_exposure)

        if min_exposure_time:
            bw.setBl_min_exposure_time(min_exposure_time)

        if min_phi_speed:
            # write a comment to the user that this is set but probably
            # not useful?
            bw.setBl_min_phi_speed(min_phi_speed)

        if max_phi_speed:
            bw.setBl_max_phi_speed(max_phi_speed)

        if min_oscillation:
            bw.setBl_min_oscillation(min_oscillation)

    # here check for the user minimum oscillation
    user_minosc = settings.getUser_desired_minimum_phi_oscillation()

    if user_minosc:
        if user_minosc > bw.getBl_min_oscillation():
            bw.setBl_min_oscillation(user_minosc)

    if settings.getCompleteness():
        bw.set_completeness(settings.getCompleteness())
    else:
        bw.set_completeness(1.0)

    if settings.getMultiplicity():
        bw.set_multiplicity(settings.getMultiplicity())
    else:
        bw.set_multiplicity(1.0)

    if settings.getI_over_sigma():
        bw.set_i_over_sig(settings.getI_over_sigma())
    else:
        bw.set_i_over_sig(2.0)

    if settings.getAnomalous():
        bw.set_anomalous()
    
    bw.set_resolution(settings.getResolution().getUpper())

    # where is this going to come from???
    bw.set_detector('q4')
    bw.set_exposure_time(exposure)

    result = bw.calculate_strategy()

    return result

    

def unit_test():
    '''Run a unit test to see if everything is working. This uses the test*
    files in the CVS repository.'''

    bw = BESTWrapper()
    bw.set_parameter_files('../data/test.dat', '../data/test.par')
    bw.add_hkl_file('../data/test_1.dat')
    bw.add_hkl_file('../data/test_2.dat')

    bw.set_completeness(1.0)
    bw.set_multiplicity(4.0)
    bw.set_i_over_sig(2.0)
    bw.set_resolution(1.5)
    bw.set_anomalous()

    bw.set_detector('q4')
    bw.set_exposure_time(1.0)

    print bw.calculate_strategy().marshal()
    
    #best_strategy = bw.get_strategy()
    #best_stats = bw.get_statistics()

    #bw.set_user_strategy(10.0, 100.0, 0.5, 2.0)
    #bw.calculate_statistics()

    #user_strategy = bw.get_strategy()
    #user_stats = bw.get_statistics()

    # now print some results

    #print 'BEST said'
    #bw.print_strategy(best_strategy)
    #bw.print_statistics(best_stats)

    #print 'TEST said'
    #bw.print_strategy(user_strategy)
    #bw.print_statistics(user_stats)

    # now do an overload test - this told me that the overloads
    # are in percents...

    #overload_test = False
    #if overload_test:
        
    #bw.set_user_strategy(10.0, 100.0, 0.5, 200.0)
    #bw.calculate_statistics()
    
    #over_strategy = bw.get_strategy()
    #over_stats = bw.get_statistics()
    
    #print 'OVERLOAD said'
    #bw.print_strategy(over_strategy)
    #bw.print_statistics(over_stats)
    
    #multi_test = False

    #if multi_test:
    #for i in range(36):
    #start = 10.0 * i
    #end = start + 10.0
    #bw.set_user_strategy(start, end, 0.5, 2.0)
    #bw.calculate_statistics()
    #strategy = bw.get_strategy()
    #print '%6.2f %5.2f %5.2f' % \
    #      (start,
    #       100.0 * strategy['completeness'],
    #       strategy['multiplicity'])
            

if __name__ == '__main__':
    unit_test()

    
