#!/usr/bin/env python
# Translation.py
# Maintained by G.Winter
# A component in the revision of the scheduler code - version 1.1
# 19th January 2004
# 
# This component will create DNA result objects and populate them from 
# Output objects derived from 'mosflm.stf' files.
# 
# 
# $Id: Translation.py,v 1.36 2006/06/20 08:22:50 gwin Exp $

import os

if not os.environ.has_key('DNAHOME'):
    raise RuntimeError, 'DNA not found'

dna = os.environ['DNAHOME']

import sys
sys.path.append(dna + '/xsd/python')
sys.path.append(os.path.join(dna,
                             'scheduler',
                             'Scheduler',
                             'New'))

import string, exceptions

import XSD
import PGInfo

sys.path.append(dna + '/expertise/python/graeme/Expert')
import Expert
from Messenger import Messenger
from Somewhere import Somewhere

def Index_response_error(error_message):
    '''Create an index response with an error message and nothing
    else in'''
    index_response = XSD.Index_response()
    status = XSD.Status()
    status.setCode('error')
    status.setMessage(str(error_message))
    index_response.setStatus(status)
    return index_response
    

def Index_response(output):
    try:
        return Index_response_real(output)
    except exceptions.Exception, e:
        return Index_response_error('Error parsing output: %s' % str(e))

def Index_response_real(output):
    '''Convert an object of type Output to DNA Python class'''

    # this needs to return more helpful error messages
    # maybe it should but that isn't a problem really at this stage!
    # it should also perhaps include the results if processing succeeded,
    # since it may be being prejudicial...

    # check for specific errors
    try:
        warnings = output.get('error')
        index_response = XSD.Index_response()
        status = XSD.Status()
        status.setCode('error')
        if warnings.has_key('fatal_indexing_error'):
            key = warnings['fatal_indexing_error']['message'].keys()[0]
            message = 'Autoindexing failure in Mosflm: '
            message += str(warnings['fatal_indexing_error']['message'][key])
        else:
            # this needs to be a much more sophisticated interpretation!
            # implemented and tested a bit!
            message = ''
            for key in warnings.keys():
                if str(key)[0] != '_':
                    message += "ERROR [%s]" % str(key)
                    for k in warnings[key].keys():
                        if str(k)[0] != '_':
                            for i in warnings[key][k].keys():
                                message += '(image %d - %s)' % \
                                           (i, str(warnings[key][k][i]))
                    
        status.setMessage(message)
        index_response.setStatus(status)
        return index_response
    except exceptions.Exception, e:
        # no warnings found
        pass

    # check for specific warning signs before proceeding
    try:
        warnings = output.get('warning')
        index_response = XSD.Index_response()
        status = XSD.Status()
        status.setCode('warning')
        message = ''
        for w in warnings.keys():
            message = message + str(warnings[w]) 
        status.setMessage(message)
        index_response.setStatus(status)
    except:
        index_response = XSD.Index_response()
        status = XSD.Status()
        status.setCode('ok')
        index_response.setStatus(status)

    images = output.get('spot_finding', 'spots_found', 'total').keys()
    images.sort()

    # this should generate a pointer to the last image - which will be the
    # one for which all of the results should have been written out
    lastImage = images[-1]

    try:
        min_sep = output.getFloat('autoindexing_parameters',
                                  'parameter_list',
                                  'closest_separation',
                                  lastImage)

        Somewhere.store('minimum_separation', min_sep)

    except:
        pass

    lattice_character_response = XSD.Lattice_character_response()
    for i in range(1, 45):
        lattice_character = XSD.Lattice_character()
        lattice_character.setPenalty(output.getInt('autoindex_solutions', \
                                                   'solution_' + str(i), \
                                                   'penalty',
                                                   lastImage))
        lattice_character.setLattice(output.getString('autoindex_solutions', \
                                                      'solution_' + str(i), \
                                                      'lattice',
                                                      lastImage))
        lattice_character.setNumber(i)
        cell = XSD.Cell()

        for property in ['a', 'b', 'c', 'alpha', 'beta', 'gamma']:
            capitalProperty = string.upper(property[0]) + property[1:]
            method = getattr(cell, 'set' + capitalProperty)
            method(output.getFloat('autoindex_solutions', \
                                   'solution_' + str(i), \
                                   property,
                                   lastImage))
        lattice_character.setCell(cell)
        lattice_character_response.addLattice_character(lattice_character)

    index_response.setLattice_character_response(lattice_character_response)

    # tally the found, used and relected spots
    found = 0
    used = 0
    rejected = 0

    for key in output.get('spot_finding', 'spots_found', 'kept').keys():
        found += output.getInt('spot_finding', 'spots_found', 'total', key)
        used += output.getInt('spot_finding', 'spots_found', 'kept', key)

    rejected = found - used

    spot_search_response = XSD.Spot_search_response()

    spot_search_response.setFound(found)
    spot_search_response.setUsed(used)
    spot_search_response.setRejected(rejected)

    index_response.setSpot_search_response(spot_search_response)

    solution = XSD.Solution()

    try:
        warnings = output.get('warning')
        status.setCode('warning')
    except:
        warnings = { }

    # this section here might not work if the refinement failed - which
    # will be indicated by a warning of solution refinement

    if not warnings.has_key('solution_refinement'):

        new = XSD.Beam()

        new.setX(output.getFloat('beam_refinement', 'refined_beam', 'x',
                                 lastImage))
        new.setY(output.getFloat('beam_refinement', 'refined_beam', 'y',
                                 lastImage))
        
        shift = XSD.Beam()
        shift.setX(output.getFloat('beam_refinement', 'initial_beam', 'x',
                                   lastImage))
        shift.setY(output.getFloat('beam_refinement', 'initial_beam', 'y',
                                   lastImage))
        
        shift.setX(new.getX() - shift.getX())
        shift.setY(new.getY() - shift.getY())
        
        beam_shift = XSD.Beam_shift()
        beam_shift.setNew_beam(new)
        beam_shift.setShift(shift)
        
        refinement = XSD.Refinement()
        refinement.setBeam_shift(beam_shift)
        
        reflections = XSD.Reflections()
        reflections.setUsed(output.getInt('refinement', 'parameters', 'used',
                                          lastImage))
        reflections.setUsed_in_indexing(output.getInt('refinement',
                                                      'parameters',
                                                      'out_of',
                                                      lastImage))
        
        refinement.setReflections(reflections)
        
        solution.setRefinement(refinement)
        
        solution.setSpot_deviation(output.getFloat('refinement',
                                                   'deviations',
                                                   'positional',
                                                   lastImage))
        
    number = output.getInt('solution_refinement',
                           'selection',
                           'number',
                           lastImage)
    
    spacegroup_number = output.getInt('solution_refinement',
                                      'selection',
                                      'spacegroup_number',
                                      lastImage)
    
    symmetry = output.getString('solution_refinement',
                                'selection',
                                'spacegroup',
                                lastImage)

    initial_cell = XSD.Cell()
    initial_orientation_matrix = XSD.Matrix()
    refined_cell = XSD.Cell()

    for property in ['a', 'b', 'c', 'alpha', 'beta', 'gamma']:
        capitalProperty = string.upper(property[0]) + property[1:]
        method = getattr(initial_cell, 'set' + capitalProperty)
        method(output.getString('initial_cell', 'cell', property, lastImage))
        method = getattr(refined_cell, 'set' + capitalProperty)
        method(output.getString('refined_cell', 'cell', property, lastImage))

    for property in ['11', '12', '13', '21', '22', '23', '31', '32', '33']:
        method = getattr(initial_orientation_matrix, 'setE' + property)
        method(output.getFloat('unconstrained_indexing_result',
                               'a_matrix', property, lastImage))
    

    a_matrix = XSD.Matrix()
    u_matrix = XSD.Matrix()

    for property in ['11', '12', '13', '21', '22', '23', '31', '32', '33']:
        method = getattr(a_matrix, 'setE' + property)
        method(output.getFloat('refined_matrices', 'a_matrix', property,
                               lastImage))
        method = getattr(u_matrix, 'setE' + property)
        method(output.getFloat('refined_matrices', 'u_matrix', property,
                               lastImage))

    initial = XSD.Initial()

    initial.setCell(initial_cell)
    initial.setOrientation_matrix(initial_orientation_matrix)
    solution.setInitial(initial)

    orientation = XSD.Orientation()
    orientation.setCell(refined_cell)
    orientation.setU_matrix(u_matrix)
    orientation.setA_matrix(a_matrix)

    solution.setOrientation(orientation)

    solution.setNumber(number)
    solution.setSpacegroup_number(spacegroup_number)
    solution.setSymmetry(symmetry)

    spag, point, latt = PGInfo.store.get(spacegroup_number)
    
    solution.setLattice(latt)
    solution.setPointgroup(point)

    index_response.setSolution(solution)
    index_response.setMosaicity_value(output.getFloat('mosaicity_estimation',
                                                      'mosaicity',
                                                      'value',
                                                      lastImage))

    return index_response

                                                  
def Cell_refinement_response(output):
    '''Convert a mosflm.stf output object into DNA speak'''

    cell_refinement_response = XSD.Cell_refinement_response()

    # check for errors in the output

    if output.key('error') == 1:
        warnings = output.get('error')
        message = ''
        status = XSD.Status()
        status.setCode('error')
        message = ''
        for w in warnings.keys():
            message = message + str(warnings[w]) 
        status.setMessage(message)

        cell_refinement_response.setStatus(status)

        return cell_refinement_response

    # then everything is okay

    status = XSD.Status()
    status.setCode('ok')
    cell_refinement_response.setStatus(status)

    keys = output.get('cell_refinement', 'refined_cell', 'a').keys()
    keys.sort()

    lastImage = keys[0]

    refined_cell = XSD.Cell()
    for property in ['a', 'b', 'c', 'alpha', 'beta', 'gamma']:
        Property = string.upper(property[0]) + property[1:]
        method = getattr(refined_cell, 'set' + Property)
        method(output.getFloat('cell_refinement', 'refined_cell', property, \
                               lastImage))

    mosaic = output.getFloat('parameter_refinement_pass',
                             'beam_spread_angles',
                             'mosaicity', lastImage)

    cell_refinement_response.setRefined_cell(refined_cell)
    cell_refinement_response.setRefined_mosaic_spread(mosaic * 0.5)

    Messenger.log_write('Refined mosaic spread: %f' % (0.5 * mosaic))

    return cell_refinement_response

def Strategy_response(output):
    '''Convert an output mosflm.stf file into DNA'''

    strategy_response = XSD.Strategy_response()

    status = XSD.Status()
    status.setCode('ok')
    strategy_response.setStatus(status)

    keys = output.get('rotation_segments', 'information', \
                      'number_of_segments').keys()
    keys.sort()
    lastImage = keys[0]

    completeness = XSD.Completeness()
    completeness.setStandard(output.getFloat('strategy_segment_1', 'segment', 'standard_completeness', lastImage))
    
    strategy_response.setCompleteness(completeness)

    strategy_summary = XSD.Strategy_summary()

    segments = output.getInt('rotation_segments', 'information', 'number_of_segments', lastImage)

    min = 10000.0

    strategy_summary.setNumber_of_segments(segments)

    for i in range(1, segments + 1):
        segment = XSD.Segment()
        oscillation = XSD.Oscillation_sequence()
        prediction = XSD.Predicted_spots()

        oscillation.setStart(output.getFloat('rotation_segment_' + str(i),
                                             'segment', 'phi_start',
                                             lastImage))
        oscillation.setRange(output.getFloat('rotation_segment_' + str(i),
                                             'segment', 'angle',
                                             lastImage))
        r = output.getFloat('rotation_segment_' + str(i),
                            'segment', 'angle',
                            lastImage)

        # keep track of the minimum oscillation angle required
        if r < min:
            min = r

        oscillation.setNumber_of_images(output.getInt('rotation_segment_' + \
                                                      str(i), 
                                                      'segment', 'steps',
                                                      lastImage))
        prediction.setFull(output.getFloat('rotation_segment_' + str(i),
                                            'segment', 'full_percentage',
                                            lastImage))
        prediction.setOverlap(output.getFloat('rotation_segment_' + str(i),
                                               'segment', 'overlap_percentage',
                                               lastImage))

        segment.setOscillation_sequence(oscillation)
        segment.setPredicted_spots(prediction)
        strategy_summary.addSegment(segment)

    # now the overall report
    segment = XSD.Segment()
    oscillation = XSD.Oscillation_sequence()
    prediction = XSD.Predicted_spots()

    prediction.setFull(0.0)
    prediction.setOverlap(0.0)
    start = output.getFloat('strategy_segment_1', 'segment',
                            'phi_start', lastImage)
    end = output.getFloat('strategy_segment_1', 'segment',
                          'phi_end', lastImage)
    oscillation.setStart(start)
    oscillation.setRange(end - start)
    if min > 0:
        oscillation.setNumber_of_images((end - start) / min)
    else:
        oscillation.setNumber_of_images(end - start)

    segment.setPredicted_spots(prediction)
    segment.setOscillation_sequence(oscillation)
    strategy_response.addSegment(segment)

    strategy_response.addStrategy_summary(strategy_summary)

    return strategy_response

def Integrate_response_error(message):
    '''Turn an error message into an integrate response'''
    integrate_response = XSD.Integrate_response()
    status = XSD.Status()
    status.setCode('error')
    status.setMessage(message)
    integrate_response.setStatus(status)
    return integrate_response
    
    
def Integrate_response(output):
    '''Turn mosflm.stf into dna'''

    # first check that everything is ok

    integrate_response = XSD.Integrate_response()

    try:
        warnings = output.get('error')
        status = XSD.Status()
        status.setCode('error')
        message = ''
        if warnings.has_key('fatal_mosflm_error'):
            keys = warnings['fatal_mosflm_error'].keys()
            message = 'Integration failure in Mosflm: '
            for k in keys:
                message += str(k) + ':' + \
                           str(warnings['fatal_mosflm_error'][k])
        else:
            message = str(warnings)
        status.setMessage(message)

        integrate_response.setStatus(status)

        Messenger.log_write('Error found! %s' % message)

        return integrate_response

    except:
        pass

    # if we get this far then all be peachy

    try:

        status = XSD.Status()
        status.setCode('ok')
        integrate_response.setStatus(status)
        
        # now get a list of images in this output file
        
        images = output.get('processing_image', 'image', 'number').keys()
        images.sort()

        # Hmm... should check that all of the images are here at this stage

        if len(images) > 1:
            Messenger.log_write('Integration results for batch %d to %d' % (images[0], images[-1]))
            Messenger.log_write('Image I/sig I/sig(outer) %bad')

        for image in images:
            distance = output.getFloat('summary_information', 'summary', \
                                       'distance', image)
            isigall = output.getFloat('summary_information', 'summary', \
                                      'isigall', image)
            isigout = output.getFloat('summary_information', 'summary', \
                                      'isigout', image)
            yscale = output.getFloat('summary_information', 'summary', \
                                     'yscale', image)
            
            full = output.getInt('summary_information', 'spots', \
                                 'full', image)
            partial = output.getInt('summary_information', 'spots', \
                                    'partial', image)
            overlap = output.getInt('summary_information', 'spots', \
                                    'overlap', image)
            negative = output.getInt('summary_information', 'spots', \
                                     'negative', image)
            bad = output.getInt('summary_information', 'spots', \
                                'bad_spots', image)

            try:
                if not output.get('parameter_refinement_pass', 'cell', \
                                  'a').has_key(image):
                    a = 0
                    b = 0
                    c = 0
                    alpha = 0
                    beta = 0
                    gamma = 0
                    mosaic = output.getFloat('parameter_refinement_pass',
                                             'beam_spread_angles',
                                             'mosaicity', image)
                else:
                    a = output.getFloat('parameter_refinement_pass', 'cell', \
                                        'a', image)
                    b = output.getFloat('parameter_refinement_pass', 'cell', \
                                        'b', image)
                    c = output.getFloat('parameter_refinement_pass', 'cell', \
                                        'c', image)
                    alpha = output.getFloat('parameter_refinement_pass', 'cell', \
                                            'alpha', image)
                    beta = output.getFloat('parameter_refinement_pass', 'cell', \
                                           'beta', image)
                    gamma = output.getFloat('parameter_refinement_pass', 'cell', \
                                            'gamma', image)
                    mosaic = output.getFloat('parameter_refinement_pass',
                                             'beam_spread_angles',
                                             'mosaicity', image)
            except:
                a = 0
                b = 0
                c = 0
                alpha = 0
                beta = 0
                gamma = 0
                mosaic = 0

            if len(images) > 1:
                Messenger.log_write('%4d  %3.2f   %3.2f    %3.2f' % \
                                    (image, isigall, isigout,
                                     100.0 * float(bad) / \
                                     float(full + partial)))

            integrated_image = XSD.Integrated_image()
            integrated_image.setImage(image)

            integrated_image.setRefined_mosaic_spread(0.5 * mosaic)
            
            integration_summary = XSD.Integration_summary()
            integration_summary.setDistance(distance)
            integration_summary.setYscale(yscale)
            integration_summary.setOuter_signal(isigout)
            integration_summary.setOverall_signal(isigall)
            
            spot_information = XSD.Spot_information()
            spot_information.setFull_spots(full)
            spot_information.setPartial_spots(partial)
            spot_information.setNegative_spots(negative)
            spot_information.setOverlap_spots(overlap)
            spot_information.setBad_spots(bad)
            
            integration_summary.setSpot_information(spot_information)
            integrated_image.setIntegration_summary(integration_summary)
            
            for i in range(0, 9):
                bin = XSD.Integration_bin()
                resolution = XSD.Resolution()
                
                upper = output.get('profile_fitted_full', 'bin_' + \
                                   str(i), 'resolution', image)
                lower = output.get('profile_fitted_full', 'bin_' + \
                                   str(i), 'resolution', image)
                
                
                if upper == 'overall':
                    upper = 0
                if lower == 'overall':
                    lower = 0

                upper = float(upper)
                lower = float(lower)
            
                resolution.setUpper(upper)
                resolution.setLower(lower)

                fullProfile = XSD.Spot_summary_element()
                partialProfile = XSD.Spot_summary_element()
                fullSummation = XSD.Spot_summary_element()
                partialSummation = XSD.Spot_summary_element()

                fullProfile.setSignal(output.getFloat('profile_fitted_full', \
                                                      'bin_' + str(i), 'i', \
                                                      image))
                fullProfile.setSignal_to_noise(output.getFloat(
                    'profile_fitted_full', \
                    'bin_' + str(i), 'i_sig', \
                    image))
                fullProfile.setNoise(output.getFloat('profile_fitted_full', \
                                                     'bin_' + str(i), 'sig', \
                                                     image))
                fullProfile.setSpot_count(output.getFloat('profile_fitted_full', \
                                                          'bin_' + str(i), \
                                                          'count', image))

            

                partialProfile.setSignal(output.getFloat('profile_fitted_partial', \
                                                         'bin_' + str(i), 'i', \
                                                         image))
                partialProfile.setSignal_to_noise(output.getFloat(
                    'profile_fitted_partial', \
                    'bin_' + str(i), 'i_sig', \
                    image))
                partialProfile.setNoise(output.getFloat('profile_fitted_partial', \
                                                        'bin_' + str(i), 'sig', \
                                                        image))
                partialProfile.setSpot_count(output.getFloat('profile_fitted_partial', \
                                                             'bin_' + str(i), \
                                                             'count', image))
            
            
            
                fullSummation.setSignal(output.getFloat('summation_full', \
                                                        'bin_' + str(i), 'i', \
                                                        image))
                fullSummation.setSignal_to_noise(output.getFloat(
                    'summation_full', \
                    'bin_' + str(i), 'i_sig', \
                    image))
                fullSummation.setNoise(output.getFloat('summation_full', \
                                                       'bin_' + str(i), 'sig', \
                                                       image))
                fullSummation.setSpot_count(output.getFloat('summation_full', \
                                                            'bin_' + str(i), \
                                                            'count', image))
                
                
                
                partialSummation.setSignal(output.getFloat('summation_partial', \
                                                           'bin_' + str(i), 'i', \
                                                           image))
                partialSummation.setSignal_to_noise(output.getFloat(
                    'summation_partial', \
                    'bin_' + str(i), 'i_sig', \
                    image))
                partialSummation.setNoise(output.getFloat('summation_partial', \
                                                          'bin_' + str(i), 'sig', \
                                                          image))
                partialSummation.setSpot_count(output.getFloat('summation_partial', \
                                                               'bin_' + str(i), \
                                                           'count', image))
                
                
                profile = XSD.Measured_spot_summary()
                summation = XSD.Measured_spot_summary()
                
                profile.setFull(fullProfile)
                profile.setPartial(partialProfile)
                
                summation.setFull(fullSummation)
                summation.setPartial(partialSummation)
                
                bin.setNumber(i)
                
                bin.setResolution(resolution)
                bin.setMeasured_spots_profile(profile)
                bin.setMeasured_spots_summation(summation)
                
                integrated_image.addIntegration_bin(bin)
                
            integrate_response.addIntegrated_image(integrated_image)

    except exceptions.Exception, e:
        status.setCode('error')
        status.setMessage('Error getting something from the integration output (%s) - check this!' % str(e))
        
        integrate_response = XSD.Integrate_response()
        integrate_response.setStatus(status)
        return integrate_response

    bestResolution = 1000.0

    for i in integrate_response.getIntegrated_image():
        r = Expert.EstimateResolution(i)
        if r < bestResolution and r > 0.0:
            bestResolution = r

    resolution = XSD.Resolution()
    resolution.setLower(0.0)
    resolution.setUpper(bestResolution)

    if bestResolution < 100.0:

        Messenger.log_write('Highest resolution recorded as %3.2f' % \
                            bestResolution, 2)

    else:
        Messenger.log_write(
            'Warning!!! No resolution reported for these images')
        Messenger.log_write(
            'This indicates that I/sigma was lower than 3.0')
        

    integrate_response.setCalculated_resolution(resolution)

    return integrate_response


            
